//
//                     TxWin, Textmode Windowing Library
//
//   Original code Copyright (c) 1995-2021 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   TxLib, released under MIT License
//
//   Copyright (c) 1995-2021  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on TxWin licensing can be directed to: info@dfsee.com
//
// ==========================================================================
//
// Timer SET and EXPIRATION-test functions that works on ELAPSED time,
// not on spent-CPU time like clock() does on some platforms, causing
// time measurement to fail (time-stand-still when thread blocked)
// This caused problem with mouse-DBLCLK detect, and progress status updating
//
// Interface will use 'nanoseconds-since-programstart' as its base unit,
// and implements that with a resolution as high as supported by the OS
//
//
// Author: J. van Wijk
//
// JvW  28-02-2021 LICENSING: Changed from LGPL to the more liberal MIT license
// JvW  03-06-2017 Initial version, 'best' resolution ELAPSED time functions

#include <txlib.h>
#include <txtpriv.h>                            // for the TxTrTstamp flag


#if   defined (UNIX)                            // gettimeofday is used
   #include <sys/time.h>
#endif

#define TXTMR_NS_PER_SEC  ((TXTIMER) 1000000000)
#define TXTMR_US_PER_SEC  ((TXTIMER) 1000000)

static TXTIMER         txtmr_base;


// Get nanoseconds (elapsed) time value from OS, with best resolution possible
static TXTIMER TxTmrGetNanoSecStamp             // RET   nanosecond value
(
   void
);


//- Note: can NOT use any trace, since it is used for trace-timestamping too!


/*****************************************************************************/
// Determine and set the BASE timer-value, resulting in 0 nsec since startup
/*****************************************************************************/
void TxTmrInitializeBase                        // call once, at program start
(
   void
)
{
   #if   defined (UNIX)                        // gettimeofday is used
      TRACES(("Resolution nsec: unknown, minimum: 1000\n"));
   #elif defined (WIN32)
      TRACES(("Resolution nsec: unknown, minimum: 100\n"));
   #else                                        // probably DOS or OS/2, uses clock()
      TRACES(("Resolution nsec: %llu\n", TXTMR_NS_PER_SEC / CLOCKS_PER_SEC));
   #endif

   txtmr_base = TxTmrGetNanoSecStamp();
}                                               // end 'TxTmrInitializeBase'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// Set a timer variable to expire after the given duration in nanoseconds
/*****************************************************************************/
TXTIMER TxTmrSetTimer                           // RET   timer expiration value
(
   TXTIMER             duration                 // IN    Duration in nanoseconds
)
{
   TXTIMER             rc = 0;                  // function return
   #if defined (DUMP)
      BOOL             ts = TxTrTstamp;         // save the flag
      TxTrTstamp = FALSE;                       // avoid recursion (TraceLeader)
   #endif

   rc = TxTmrGetNanoSecStamp() - txtmr_base + duration;

   #if defined (DUMP)
      TxTrTstamp = ts;                          // restore flag
   #endif
   return (rc);
}                                               // end 'TxTmrSetTimer'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set a timer variable to expire after the given duration in nanoseconds
/*****************************************************************************/
BOOL TxTmrTimerExpired                          // RET   timer has expired
(
   TXTIMER             timer                    // IN    timer expiration value
)
{
   BOOL                rc;

   rc = ((TxTmrGetNanoSecStamp() - txtmr_base) > timer);

   return (rc);
}                                               // end 'TxTmrInitializeBase'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get nanoseconds (elapsed) time value from OS, with best resolution possible
/*****************************************************************************/
static TXTIMER TxTmrGetNanoSecStamp             // RET   nanosecond value
(
   void
)
{
   TXTIMER             rc = 0;                  // function return

   #if   defined (UNIX)                         // gettimeofday is used
      struct timeval   tv;
   #elif defined (WIN32)
      FILETIME         ft;
   #else                                        // probably DOS or OS/2, uses clock()
   #endif

   #if   defined (UNIX)                         // gettimeofday is used, 1 uSec resolution
      if (gettimeofday( &tv, NULL) == 0)        // hi part is seconds, low is uSec
      {
         rc = ((TXTIMER) tv.tv_sec * TXTMR_NS_PER_SEC) + tv.tv_usec * 1000;
         TRLEVX(789,("             TxTmr RAW sec:%8.8lx usec:%8.8lx  rc:%llx\n", tv.tv_sec, tv.tv_usec, rc));
      }
      else
      {
         TRLEVX(789,( "gettimeofday error: %s\n", strerror(errno)));
      }
   #elif defined (WIN32)
      GetSystemTimeAsFileTime( &ft);            // uses 100 nsec time units in 64-bits (hi+lo)
      rc = (((TXTIMER) ft.dwHighDateTime << 32) + ft.dwLowDateTime) * 100;
      TRLEVX(789,("             TxTmr RAW sec:%8.8lx 100n:%8.8lx  rc:%llx\n", ft.dwHighDateTime, ft.dwLowDateTime, rc));

   #else                                        // probably DOS or OS/2, uses clock()
      rc = ((TXTIMER) clock() * TXTMR_NS_PER_SEC) / CLOCKS_PER_SEC;
      TRLEVX(789,("             TxTmr RAW clock:%8.8lx  rc:%llx\n", clock(), rc));
   #endif

   return (rc);
}                                               // end 'TxTmrGetNanoSecStamp'
/*---------------------------------------------------------------------------*/

