//
//                     TxWin, Textmode Windowing Library
//
//   Original code Copyright (c) 1995-2021 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   TxLib, released under MIT License
//
//   Copyright (c) 1995-2021  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on TxWin licensing can be directed to: info@dfsee.com
//
// ==========================================================================
//
// TX standard command execution
//
// Author: J. van Wijk
//
// Developed for LPT/DFSEE utilities
//
// JvW  28-02-2021 LICENSING: Changed from LGPL to the more liberal MIT license
// JvW  20-11-2001 Use TXA command parser only
// JvW  11-10-2001 Reduced memory usage for SMALLMEM by removing commands
// JvW  15-02-2000 Fixed 'set ansi on/off' bug

#include <txlib.h>                              // TX library interface
#include <txtpriv.h>                            // TX library privates
#include <txwpriv.h>                            // TX windows privates

#if !defined (UNIX)
   #include <dos.h>
#endif



char         *txStdCmdHelp[] =
{
   "",
   "TX library generic commands",
   "===========================",
   "",
   " CC [#lin [last]] = Clipboard Copy, copy lines from output window to clipboard",
#if defined (WIN32)
   " CINFO            = Display Windows Console-info buffer and window size details",
#endif
   " CP    or  PASTE  = Clipboard Paste, copy clipboard text to the output window",
   " CD        [path] = Change current directory and current drive",
   " CWD      [drive] = Show current working directory, for specified driveletter",
   " CMD    [command] = Force execution of command by Operating System, not internal",
   " CONFIRM [-y] msg = Show a notification, or a confirmation message [-y]",
   " DISPLAY          = Show current display size in rows and columns",
   " DTV    [dt-spec] = Show converted date-time spec: days, years-, year-month-",
   " EXIST filename   = Show if the specified (path +) filename exists or not",
   " HELP [words|i|s] = Search all Help for 'words', list Items or loaded Sections",
#if defined (DOS32)
   " KEYB [ctry [cp]] = Show or set country with optional codepage for keyboard",
#endif
   " LOADHELP fn [ds] = Load additiona help section from file, with description",
   " LOG   [file | .] = Start, stop or resume logging, (appending) to a file",
   " MODE [c,r] [-w-] = Set columns and rows for the window, -w- is non-windowed",
   " RC    returncode = Set specified returncode, explain its meaning when known",
   " SAY   anything   = Display 'anything' on the output window AND the logfile!",
   " SCRFILE [fn] [l] = Save screen buffer to file [fn], last [l] lines",
   " SCREEN  on | off = Screen output on or off (deprecated, use 'set screen ...')",
   " SET [prop value] = Set config property to a value, 'SET' lists all properties",
   " SLEEP  [seconds] = Sleep for specified number of seconds, default is 1",
#if defined (DUMP)
   " TRACE [lvl-opts] = Set trace level/options for application functional tracing",
#endif
   " TXDIR fn [o] [a] = List files for wildcard selection 'fn' and several options",
   " UICCUBE          = Test UI display, Color-Cube 88/256 colors (Linux, macOS)",
   " UICTEST          = Test UI display Colors, Character-set, drawing-Characters",
   " UIKEYS           = Test user-interface interpretation of keyboard and mouse",
   " UITIME           = Test user-interface real-date-time and CPU-clock timing",
   " WINHWND          = Show window handles for program (implemented for OS/2 only)",
   " WINLIST          = Show window list with details   (implemented for OS/2 only)",
   "",
   "",
   NULL
};


#if defined (USEWINDOWING)
static  char       *cmd_help[] =
{
   "",
   "Give various help overviews, and search the builtin help system",
   "",
   " Usage: HELP  [search-phrase | f | i | s]",
   "",
   "   search-phrase : Phrase of one or more words to search for in all",
   "                   help-texts builtin or loaded from external files",
   "                   and build a search-result list to select from.",
   "",
   "   i             : Present list of help-items in (menu) help section",
   "",
   "   s             : Present list of builtin and loaded help sections",
   "",
   " To show help on valid commands, use the '\?', '\?\?' and '\?\?\?' commands.",
   "",
   " To show help on command syntx and options, use 'command  -\?'",
   "",
   NULL
};
#endif


static char txsetPrio = 'S';                    // default Std priority


/*****************************************************************************/
// Get array of strings containing the TxLib specific standard command help
/*****************************************************************************/
char **TxGetStdCmdHelp
(
   void
)
{
   return (txStdCmdHelp);
}                                               // end 'TxGetStdCmdHelp'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Execute standard commands
/*****************************************************************************/
ULONG TxStdCommand
(
   void                                         // parameters through TXA
)
{
   ULONG               rc  = NO_ERROR;
   LONG                nr  = 0;
   TXLN                dc;                      // local command string
   int                 cc  = 0;                 // command string count
   char               *c0, *c1, *c2, *c3;       // parsed command parts
   char               *c4, *c5, *c6, *c7;       // parsed command parts
   TXLN                s1;                      // temporary string space
   char               *pp;                      // parameter pointers

   ENTER();

   #if defined (USEWINDOWING)
      if (txwa->sbview)
      {
         TxaGetArgString( TXA_CUR, 0, TXA_OPT, TXMAXLN, dc);
         sprintf( s1, "TxWin executing: %s", dc);
         txwSendMsg( (TXWHANDLE) txwa->sbview, TXWM_STATUS, (TXWMPARAM) s1, (TXWMPARAM) cSchemeColor);
      }
   #endif                                       // USEWINDOWING

   pp = TxaGetArgString( TXA_CUR, 1, 0, TXMAXLN, dc); // dc => cmd from arg 1
   cc = TxaArgCount( );                         // number of parameters
   c0 = TxaArgValue(0);
   c1 = TxaArgValue(1);
   c2 = TxaArgValue(2);
   c3 = TxaArgValue(3);
   c4 = TxaArgValue(4);
   c5 = TxaArgValue(5);
   c6 = TxaArgValue(6);
   c7 = TxaArgValue(7);

   TRACES(("cc: %u c0:'%s' c1:'%s' c2:'%s' c3:'%s' c4:'%s' c5:'%s' c6:'%s' c7:'%s' pp:'%s'\n",
            cc,    c0,     c1,     c2,     c3,     c4,     c5,     c6,     c7,     pp));

   if (strcasecmp(c0, "log"      ) == 0)
   {
      char             *current = TxQueryLogName();

      if ((TxaOption('?')) || (c1[0] == '?'))
      {
         TxPrint( "\nStart or stop logging to a file\n\n"
                  " Usage:  log  [filename] [-r] [-f:ff] [-m:mm]\n\n"
                  "   -f[:ff]      retain ff numbered logfiles when cycling\n"
                  "   -m[:mm]      cycle to next logfile after size mm Kb\n"
                  "   -r           close and reopen log on each line (slow)\n"
                  "   filename     filename with optional path for logfile\n"
                  "                When not specified, logging is STOPPED\n\n");
         TxPrint( " Substitution of special characters in the filename:\n"
                  "   ~            At END of name only, date as: YYYYMMDD\n"
                  "   ~1           Day     2 digits (DD)\n"
                  "   ~2           Month   2 digits (MM)\n"
                  "   ~3           Year    4 digits (YYYY)\n"
                  "   ~4           Hour    2 digits (HH)\n"
                  "   ~5           Minute  2 digits (MM)\n"
                  "   ~6           Second  2 digits (SS)\n"
                  "   ~7           Full name of current weekday\n"
                  "   ~8           Full name of current month\n"
                  "   ~9           2 digits ISO 8601 weeknumber\n"
                  "   ^            At END of name only, append unique 3-digit number\n"
                  "                (can be combined with ~1..9 date/time fragments)\n\n");

         if (current != NULL)
         {
            TxPrint( "Logging to file   : %s\n", current);
         }
         else
         {
            TxPrint( "There is currently NO logfile active.\n");
         }
      }
      else
      {
         if (TxaOptSet('r'))
         {
            TxSetLogReOpen( TxaOption('r'));
         }
         if (TxaOptSet('m'))
         {
            TxSetLogMaxSize( TxaOptBkmg( 'm', 2047, 'k'));
         }
         if (TxaOptSet('f'))
         {
            TxSetLogRetain( TxaOptNum('f', NULL, 1));
         }
         TxAppendToLogFile( c1, TRUE);
      }
   }
   else if (strcasecmp(c0, "trace"    ) == 0)
   {
      if (TxaOption('?'))
      {
         TxPrint( "\nSet and check TRACE level, format and destinations\n\n"
                  "Usage: trace [level | filename] [-r] [-s] [-t] [-d] [-f] [-m]\n\n"
                  "   -d[:xx]      add xx ms delay for each trace line, slowdown\n"
                  "   -f[:ff]      retain ff numbered logfiles when cycling\n"
                  "   -m[:mm]      cycle to next logfile after size mm Kb\n"
                  "   -r           close and reopen log on each line (slow, default)\n"
                  "   -r-          open log just once (may be truncated on crash)\n"
                  "   -s           trace to the scoll-buffer (screen) too\n"
                  "   -s-          switch off trace to screen\n"
                  "   -t           add timestamps and force thread-id display\n"
                  "   -t-          switch off timestamp and thread-id\n"
                  "   level        0 = off, 1 = window-title, 10..999 = detail\n"
                  "   filename     filename with optional path for tracefile\n"
                  "                (this is the same file used as normal LOG)\n\n"
                  "   The (left) <Alt>+/ key cycles levels 0 -> 1 -> 100 -> 0\n\n");
      }
      else
      {
         if (TxaOptSet('d'))                    // delay xx ms per traceline
         {
            TxTrSlowDown = TxaOptNum( 'd', NULL, 100);
         }
         if (TxaOptSet('r'))
         {
            TxSetLogReOpen(  TxaOption('r'));
         }
         if (TxaOptSet('t'))                    // timestamping specified
         {
            TxTraceSetStamp( TxaOption('t'));
         }
         if (TxaOptSet('s'))
         {
            TxTrLogOnly =   !TxaOption('s');    // to screen with -s
         }
         if (TxaOptSet('m'))
         {
            TxSetLogMaxSize( TxaOptBkmg( 'm', 2047, 'k'));
         }
         if (TxaOptSet('f'))
         {
            TxSetLogRetain( TxaOptNum('f', NULL, 1));
         }
         if (isdigit( c1[0]))                   // setting a level
         {
            TxTrLevel = atol(c1);
         }
         else
         {
            if (strlen( c1))
            {
               TxAppendToLogFile( c1, TRUE);
               TxTrLevel = 88;                  // all except TRACEX
            }
         }
      }
      TxPrint("Funct trace level : %u  trace to %s\n",  TxTrLevel,
                        (TxTrLevel == 1)              ? "Window title-bar" :
        (TxTrLogOnly) ? (TxQueryLogFile( NULL, NULL)) ? "logfile only"
                                                      : "nowhere!"
                      : (TxQueryLogFile( NULL, NULL)) ? "log and screen"
                                                      : "screen only");
   }
   else if (strcasecmp(c0, "cmd"      ) == 0)
   {
      TxaOptSetItem( "-B");                     // no trusted command check
      if (cc > 1)
      {
         rc = TxExternalCommand( pp);           // execute parameter as cmd
      }
      else
      {
         if ((pp = getenv( "COMSPEC")) != NULL) // when defined
         {
            TxExternalCommand( pp);             // run command interpreter
         }
         else
         {
            TxPrint( "No command processor defined in 'COMSPEC' environment variable\n");
         }
      }
   }
   #if defined (USEWINDOWING)
   else if (strcasecmp(c0, "display") == 0)
   {
      TxPrint( "\nCurrent display is: %hu columns x %hu rows = %u cells\n",
                TxScreenCols(), TxScreenRows(), (ULONG) TxScreenCols() * (ULONG) TxScreenRows());
   }
   #endif
   else if (strcasecmp(c0, "help"     ) == 0)
   {
      #if defined (USEWINDOWING)
         if (TxaOption('?'))                 // explicit help request
         {
            TxShowTxt( cmd_help);            // give usage
            TxPrint("\nBased on TXlibrary: %s\n", txVersionString());
         }
         else
         {
            if (cc > 1)                      // at least one parameter
            {
               if (strlen( c1) == 1)
               {
                  switch (toupper(c1[0]))
                  {
                     case 'F':
                        txwHelpDialog( TXWH_REQUEST_ID, TXWH_REQ_FINDDIALOG, NULL);
                        break;

                     case 'I':
                        txwHelpDialog( TXWH_REQUEST_ID, TXWH_REQ_ITEMLIST, NULL);
                        break;

                     default:
                     case 'S':
                        txwHelpDialog( TXWH_REQUEST_ID, TXWH_REQ_SECTIONLIST, NULL);
                        break;
                  }
               }
               else                          // perform grep
               {
                  txwHelpDialog( TXWH_REQUEST_ID, TXWH_REQ_GREPSTRING, pp);
               }
            }
            else
            {
               txwHelpDialog( TXWH_REQUEST_ID, TXWH_REQ_GREPDIALOG, "");
            }
         }
      #else
         TxShowTxt( txStdCmdHelp);
         TxPrint("\nBased on TXlibrary: %s\n", txVersionString());
      #endif
   }
   else if (strcasecmp(c0, "loadhelp"     ) == 0)
   {
      #if defined (USEWINDOWING)
         if (TxaOption('?') || (cc <= 1))       // explicit help request or no param
         {
            TxPrint("\nLoad additional (text/script) file with help items from a file\n");
            TxPrint("\n Usage:  %s  fname [descr]  [-H:section] [-S]\n"
                    "\n   fname       = Filename for helpfile, path and extension optional\n"
                      "   descr       = Optional description for this new help section\n\n"
                      "   -S          = Filter help sections from SCRIPT source file\n"
                      "   -H:section  = Help section base value, a unique value, range 70..199\n"
                      "                 (70..99 reserved for system, 100..199 for user scripts)\n", c0);
         }
         else
         {
            ULONG   base = 0;

            if (TxaOptSet( 'H'))
            {
               base = TxaOptNum( 'H', NULL, 10) * 1000;
            }
            rc = txwLoadRegisterHelpFile( c1, c2, base, TxaOption( 'S'));
            if (rc == NO_ERROR)
            {
               TxPrint("\nNew help section successfully loaded from '%s'.\n", c1);
            }
         }
      #endif
   }
   #ifndef TXMIN
   else if (strncasecmp(c0, "txd", 3) == 0)
   {
      char            *filter[5] = {NULL,NULL,NULL,NULL,NULL};

      if (TxaOption('?'))                       // explicit help request
      {
         TxPrint("\nShow files and/or directories in selectable format\n");
         TxPrint("\n Usage:  %s  fspec  select  'attrs' [includes] [-b]\n\n"
                   "   fspec  = File or directory spec, including wildcards\n"
                   "   search = display selection criteria: SFDP\n"
                   "            S=subdirs, F=Files, D=Dirs, P=Parent ..\n"
                   "   attrs  = file-attributes to match  : '+A+R+S+H-A-R-S-H'\n"
                   "            +A  archive-bit  -A NO archive bit\n"
                   "            +R  read-only    -R NOT read-only\n"
                   "            +S  system-bit   -S NO system bit\n"
                   "            +H  hidden       -H NOT hidden\n"
                   "   inexcl = include/exclude wildcards\n", c0);
         TxPrint("\n   -b     = show basename only\n\n");
      }
      else
      {
         if (cc > 4)
         {
            filter[0] = c4;
         }
         if (cc > 5)
         {
            filter[1] = c5;
         }
         if (cc > 6)
         {
            filter[2] = c6;
         }
         if (cc > 7)
         {
            filter[3] = c7;
         }
         TxPrint( "\nTxDir for: %s, options: %s, attributes: %s\n\n",
                  (cc > 1) ? c1 : "current dir",
                  (cc > 2) ? c2 : "file+dirs",
                  (cc > 3) ? c3 : "all files");
         if (TxaOption('b'))                    // basename only
         {
            rc = TxFileTree( c1, c2, c3, filter, txFileTreeNoPath,  NULL);
         }
         else
         {
            rc = TxFileTree( c1, c2, c3, filter, txFileTreeOneFile, NULL);
         }
      }
   }
   else if (strncasecmp(c0, "uicc", 4    ) == 0) // 88/256 color cube display
   {
      int     bg;
      int     fg = (cc > 1) ? atoi( c1) : 8;    // default/set forground color
      #if defined (DARWIN)
         int  nc = TxaOptNum( 'c', NULL, 6);    // default 6x6x6
      #else
         int  nc = TxaOptNum( 'c', NULL, 4);    // default 4x4x4
      #endif
      int     r,g,b;                            // rgb cube indexes
      int     limit = (nc == 4) ? 88 : 256;     // maximum color index

      TxPrint("\nUsage: %s  [foreground-color]    [-c:6 | -c:4]\n", c0);
      TxPrint("In non-windowed mode ('mode -w-') shows all available BG colors in %d-color palette\n"
              "In the windowed mode ('mode -w' ) shows the ones mapped to std 16 background colors\n\n", limit);

      //- Show the first 16 (PC-style) color mappings, available for compatibility
      for (r = 0; r < 4; r++)
      {
         for (g = 0; g < 4; g++)
         {
            bg = (r * 4) + g;                   // 16 'PC-style' background colors

            TxPrint( "[48;5;%d;38;5;0m  %3d  [48;5;%d;38;5;%dm  %3d  [48;5;%d;38;5;15m  %3d  %s  ",
                             bg,          bg,        bg,     fg,   fg,        bg,           bg,  CNN);
         }
         TxPrint("%s\n", CNN);
      }
      TxPrint("\n");

      //- Show the cube colors, either 6x6x6 or 4x4x4
      for (r = 0; r < nc; r++)
      {
         for (g = 0; g < nc; g++)
         {
            for (b = 0; b < nc; b++)
            {
               bg = (((r * nc) + g) * nc) + b + 16;

               TxPrint( "[48;5;%d;38;5;0m  %3d  [48;5;%d;38;5;%dm  %3d  [48;5;%d;38;5;15m  %3d  %s  ",
                                bg,          bg,        bg,     fg,   fg,        bg,           bg,  CNN);
            }
            TxPrint("%s\n", CNN);
         }
         TxPrint("\n");
      }

      //- Show the greyscale at end of palette, with foreground color on top
      fg = TxaOptNum('f', NULL, 178);
      for (bg = nc * nc * nc + 16; bg < limit; bg++)
      {
         TxPrint( "[48;5;%d;38;5;%dm  %3d ", bg, fg, bg);
      }
      TxPrint("%s\n", CNN);
   }
   else if (strncasecmp(c0, "uict", 4    ) == 0)
   {
      int     bg,fg;

      TxPrint("%s", ansi[NORMAL]);
      for (bg = 0; bg < 16; bg++)
      {
         TxPrint("\n       ");
         for (fg = 0; fg < 16; fg++)
         {
            TxPrint( "%s %1X%1X %s", ansi[fg + (bg * 16)], bg, fg, ansi[NORMAL]);
         }
      }
      TxPrint("\n\n");
      TxPrint(" Ŀ\n" );
      TxPrint(" \\0123456789ABCDEF\n");
      TxPrint(" 0    \n" );
      TxPrint(" 1\n" );
      TxPrint(" 2 ! #$%%&'()*+,-./\n");
      TxPrint(" 30123456789:;<=>?   Ŀ %s ͻ %s\n", ansi[TXaBYnC], ansi[NORMAL]);
      TxPrint(" 4@ABCDEFGHIJKLMNO    Single line   default  %s  Double line   Yellow   %s\n", ansi[TXaBYnC], ansi[NORMAL]);
      TxPrint(" 5PQRSTUVWXYZ[ ]^_    box character Ĵ %s  box character ͹ %s\n", ansi[TXaBYnC], ansi[NORMAL]);
      TxPrint(" 6`abcdefghijklmno    with joints   colors   %s  with joints   on Cyan  %s\n", ansi[TXaBYnC], ansi[NORMAL]);
      TxPrint(" 7pqrstuvwxyz{|}~    %s ͼ %s\n", ansi[TXaBYnC], ansi[NORMAL]);
      TxPrint(" 8\n" );
      TxPrint(" 9\n" );
      TxPrint(" A  %s ͸ %s ķ %s\n", ansi[TXaBZnW], ansi[TXaNWnM], ansi[NORMAL]);
      TxPrint(" B  %s  Mixed lines     Grey   %s  Mixed lines   White on %s\n", ansi[TXaBZnW], ansi[TXaNWnM], ansi[NORMAL]);
      TxPrint(" Cϳ  %s  box character ͵ %s  box character Ķ %s\n", ansi[TXaBZnW], ansi[TXaNWnM], ansi[NORMAL]);
      TxPrint(" D߳  %s  with joints   on White %s  with joints   Magenta  %s\n", ansi[TXaBZnW], ansi[TXaNWnM], ansi[NORMAL]);
      TxPrint(" E  %s ; %s Ľ %s\n", ansi[TXaBZnW], ansi[TXaNWnM], ansi[NORMAL]);
      TxPrint(" F\n" );
      TxPrint(" /0123456789ABCDEF\n" );
      TxPrint(" \n" );
      TxPrint("%s", ansi[NORMAL]);
   }
   else if (strcasecmp(c0, "uikeys" ) == 0)
   {
      ULONG            input;
      BOOL             sameLine = TxaOption( 'u');
      TXW_INPUT_EVENT  mouse;
      BOOL             doubleClick;
      ULN64            dblClkTimer = 0;

      TxPrint( "\n"
      #if defined (UNIX)
               "Note: "
         #if defined (DUMP)
               "use startup switch '-K' for extra debug info (shows table while comparing)\nAlso note that"
         #endif
               "'^' can be another Escape, check hex value!        "
      #endif
               "Type <Esc> to abort the test ...\n\n");
      do
      {
         input = txwGetInputEvent( TRUE, &mouse);
         if (input == TXW_INPUT_MOUSE)          // mouse
         {
            doubleClick = FALSE;
            if ((mouse.value != 0) && ((mouse.value & TXm_DRAGGED) == 0)) // button DOWN, not dragged
            {
               if (TxTmrTimerExpired( dblClkTimer) == FALSE) // NOT expired
               {
                  doubleClick = TRUE;
               }
            }
            dblClkTimer = TxTmrSetTimer( TMRMSEC( 250));
            TxPrint("Mouse @: %3hu %3hu button:%4.4hx Kstate:%4.4hx seconds:%12.6lf %s\n",
                     mouse.row, mouse.col, mouse.value, mouse.state,
                     TxTmrGetSecondsFromStart(), (doubleClick) ? "(DBLCLK)" : "        ");
         }
         else if (input != 0)
         {
            TxPrint("Keyvalue: %3.3lX = %-*.*s seconds:%12.6lf        \n",
                     input, 24, 24, txwKeyDescription( input), TxTmrGetSecondsFromStart());
         }
         if (sameLine)                          // rewrite next line over this one
         {
            TxPrint( CU1);
         }
      } while (input != TXK_ESCAPE);
   }
   else if (strcasecmp(c0, "uitime"  ) == 0)    // compare clock() versus TXTIMER accuracy
   {
      clock_t          clockLast = clock();
      clock_t          clockNow;
      time_t           last = time( &last);     // last timestamp in seconds
      time_t           now;                     // current timestamp
      TXTT             datetime;

      TxPrint( "\nType <Esc> to abort the test ...\n");
      while (!TxAbort())
      {
         now = time( &now);
         if (now != last)                       // one second passed
         {
            clockNow = clock();
            strftime( datetime, TXMAXTT, "%Y-%m-%d %H:%M:%S", localtime( &now));
            TxPrint( "Now: %s ClockNow: %14llu Clocks-per-sec:%7llu  Seconds-UP: %15.9lf\n",
                      datetime, (ULN64) clockNow, (ULN64) clockNow - (ULN64) clockLast, TxTmrGetSecondsFromStart());
            last      = now;
            clockLast = clockNow;
         }
      }
   }
   else if (strcasecmp(c0, "cc" ) == 0)
   {
      if (TxaOption('?'))                       // explicit help request
      {
         TxPrint("\nCopy a number of lines from the Output-window to the clipboard\n");
         TxPrint("\n Usage:  %s  [#lines  [first]]\n\n"
                   "   #lines = Number of line to be copied to the clipboard, default screensize\n"
                   "   first  = Number of lines (from END) to start the copy, default is 3 lines\n\n", c0);
      }
      else
      {
         ULONG         lines = (cc > 1) ? (ULONG) atoi( c1) : 0; // use built-in defaults
         ULONG         first = (cc > 2) ? (ULONG) atoi( c2) : 0; // when not specified
         char         *text  = NULL;

         if ((((nr = (LONG) txwSaveToMemSB( lines, first, &text)) != 0)) && (text != NULL))
         {
            if ((rc = TxCopyClipBoardText( text)) == NO_ERROR)
            {
               TxPrint( "\nCopied %d lines to the clipboard ...\n", nr);
            }
            TxFreeMem( text);
         }
         else
         {
            TxPrint( "\nFailed to get Output-window text lines!\n\n");
         }
      }
   }
   else if ((strcasecmp(c0, "cp"    ) == 0) ||
            (strcasecmp(c0, "paste" ) == 0)  )
   {
      if (TxaOption('?'))                       // explicit help request
      {
         TxPrint("\nPaste text-contents of the clipboard into the (end of the ) Output-window\n");
         TxPrint("\n Usage:  %s\n\n", c0);
      }
      else
      {
         TxPasteClipBoardText();
      }
   }
   #if defined (WIN32)
   else if (strcasecmp(c0, "cinfo"    ) == 0)   // console info display
   {
      txwDisplayConsoleInfo();
   }
   #endif
   #if defined (USEWINDOWING)
   else if (strcasecmp(c0, "mode"    ) == 0)    // change display mode
   {
      #if defined (UNIX)                        // same as auto-detected resize
         TxScreenTerminate();
         TxScreenInitialize();                  // reinit at new windowsize
      #else
         if ((cc > 1) &&                        // resize screen using OS cmd
             (!TxaExeSwitch('S')))              // when not in shell-mode
         {
            USHORT        cols;
            USHORT        rows;

            if ((cc > 2) && (isdigit(c2[0])))
            {
               rows = (USHORT) atoi( c2);       // nr of rows
            }
            else if ((pp = strchr( c1, ',')) != NULL)
            {
               rows = (USHORT) atoi( pp+1);
            }
            else                                // single param, keep rows
            {
               rows = TxScreenRows();
            }
            cols = (USHORT) atoi( c1);          // nr of columns
            if (cols < 10)
            {
               cols = TxScreenCols();
            }
            sprintf( s1, "mode %hu,%hu", cols, rows);

            TxExternalCommand( s1);             // set mode, will cls too
         }
         TxSetBrightBgMode( TRUE);              // no blinking, use bright BG
      #endif
      rc = TX_DISPLAY_CHANGE;                   // signal display change to APP
   }
   #endif                                       // windowing
   #endif                                       // not TXMIN
   else if (strcasecmp(c0, "confirm"  ) == 0)
   {
      if (!TxaOption('?'))
      {
         if (cc > 1)
         {
            strcpy( s1, pp);
            TxRepl( s1, '~', '\n');
         }
         else                                   // no text specified
         {
            strcpy( s1, "Continue");
         }
         if (TxaOption('y'))                    // confirm Yes/No
         {
            strcat( s1, " ? [Y/N] : ");
            if (!TxConfirm( TxaOptNum( 'H', NULL, 0), s1))
            {
               rc = TX_ABORTED;                 // Will result in 'better'
            }                                   // message to user :-)
         }
         else
         {
            if (cc == 1)                        // no ? after custom text
            {
               strcat( s1, " ?");
            }
            if (!TxNamedMessage( !TxaOption('n'), TxaOptNum( 'H', NULL, 0), " Confirmation message ", s1))
            {
               rc = TX_ABORTED;                 // Will result in 'better'
            }                                   // message to user :-)
         }
         TxCancelAbort();                       // avoid aborting on Escape ...
      }
      else
      {
         TxPrint("\nPresent a popup message with either an OK, or Yes/No/Cancel buttons\n");
         TxPrint("\n Usage:   %s  message [-y]  [-help:ssiii]\n\n"
                   "   message   = The message text as a string or just a sequence of words\n"
                   "   -y        = Use Yes/No/Cancel buttons instead of just OK\n"
                   "   -n        = No acknowledge needed, just print to screen/log"
                   "   -H:ssiii  = Help-ID, two-digit section, three digits item\n"
                   "               'ss' should be same as value used on 'loadhelp'\n"
                   "               'iii' is the itemnumber on the text like '#001'\n\n"
                   " Example: confirm -y -H:11003 Really want to do this\n\n", c0);
      }
   }
#if defined (DOS32)
   else if (strcasecmp(c0, "keyb"    ) == 0)       // change keyboard mapping
   {
      if ((cc > 1) && (!TxaOption('?')))
      {
         if ((rc = TxSetNlsKeyboard( c1, c2)) != NO_ERROR)
         {
            TxPrint( "\nError %u setting keyboard mapping '%s'\n\n", rc, c1);
         }
      }
      else                                      // give help
      {
         TxPrint("\nSet country or codepage-specific keyboard mapping\n");
         TxPrint("\n Usage:   %s  def [cp]\n\n"
                   "   def  = Keyboard definition file (.kl) basename\n"
                   "   cp   = Codepage value valid for that language\n\n"
                   " Example: keyb nl 850\n\n", c0);

         TxExternalCommand( "keyb");            // show current keyb, if any
      }
   }
#endif
   else if ((strcasecmp(c0, "cd") == 0) || ((strlen(c0) == 2) && (c0[1] == ':') ))
   {
      if (strcasecmp(c0, "cd") == 0)
      {
         if (*pp == '~')                        // relative to HOME directory
         {
            #if defined (WIN32)
               char   *home = getenv("HOMEPATH");
            #else
               char   *home = getenv("HOME");   // start with home directory
            #endif

            strcpy( s1, (home) ? home : ".");
            strcat( s1, pp + 1);                // add relative path, if any
         }
         else                                   // explicit path given
         {
            strcpy( s1, pp);
         }
      }
      else                                      // absolute, with driveletter
      {
         strcpy( s1, c0);
      }
      TxStrip( s1, s1, ' ', ' ');
      if (strlen( s1))                          // only when specified
      {
         TxRepl( s1, FS_PALT_SEP, FS_PATH_SEP); // fixup ALT separators
         if (s1[ strlen(s1) - 1] == FS_PATH_SEP)
         {
             s1[ strlen(s1) - 1] = 0;           // remove trailing path seperator
         }
         #if defined (UNIX)
            rc = (ULONG) chdir( s1);
         #else
            if (s1[1] == ':')                   // set drive too, if specified
            {
               #if !defined (UNIX)
                  unsigned  drives;

                  _dos_setdrive(toupper( s1[0]) - 'A' +1, &drives);
               #endif
            }
            if ((strlen( s1) > 2) || (s1[1] != ':'))
            {
               rc = (ULONG) chdir( s1);
            }
         #endif
      }
      if (rc != NO_ERROR)
      {
         TxPrint( "\nError changing to : '%s'\n", s1);
         rc = NO_ERROR;
      }
      getcwd(s1, TXMAXLN);
      TxPrint("\nWorking directory : %s%s%s%s\n",
                 CBC, s1, (strlen(s1) > 3) ? FS_PATH_STR : "", CNN);
   }
   else if (strcasecmp(c0, "cwd"  ) == 0)
   {
      if (!TxaOption('?'))
      {
         strcpy( s1, c1);

         TxGetCwd( s1, TXMAXLN);

         TxPrint( "Current directory on drive %c: is '%s'\n", s1[0], s1);
      }
      else
      {
         TxPrint( "Show current working directory on specified dive\n\n"
                  " Usage: %s [drive]\n\n"
                  "        drive = drive letter, A .. Z\n", c0);
      }
   }
   else if (strcasecmp(c0, "dtv"  ) == 0)       // date-time string value test
   {
      time_t dt = TxaParseDateTime( pp);        // convert rest of line as parameter

      strftime( s1, TXMAXLN,  "%Y-%m-%d %H:%M:%S", localtime( &dt));

      TxPrint( "Converted date/time string: '%s' to '%s'\n", pp, s1);
   }
   else if (strcasecmp(c0, "exist"  ) == 0)
   {
      if (cc > 1)
      {
         TxPrint("File '%s' does%s exist\n", s1, (TxFileExists(c1)) ? "": " NOT");
      }
   }
   else if (strcasecmp(c0, "rc"  ) == 0)        // set/explain returncode value
   {
      rc = (ULONG) TxaParseNumber( c1, TX_RADIX_APP_CLASS * 4, NULL);
   }
   #if defined (USEWINDOWING)
   else if ((strcasecmp(c0, "scrfile") == 0))   // screen to file
   {
      ULONG            lines = -1;              // default all lines

      if (cc <= 1)
      {
         strcpy( s1, "screen");
      }
      else
      {
         strcpy( s1, c1);
      }
      TxFnameExtension( s1, "log");             // append default extension
      if (cc > 2)
      {
         lines = (ULONG) atol( c2);
      }
      lines = txwSavePrintfSB( s1, lines, (cc <= 3));
      TxPrint( "Saved %u lines from screen-buffer to %s\n", lines, s1);
   }
   #endif                                       // USEWINDOWING
   else if ((strcasecmp(c0, "screen"   ) == 0)) // backward compatibility!
   {
      if ((cc > 1) && (!TxaOption('?')))
      {
         if ((strcasecmp(c1, "on") == 0) || (c1[0] == '1'))
         {
            TxScreenState(DEVICE_ON);
         }
         else
         {
            TxScreenState(DEVICE_OFF);
         }
      }
      else
      {
         rc = (ULONG) TxScreenState( DEVICE_TEST);
         TxPrint("Screen output is switched %s.\n",
                 (rc == (ULONG) DEVICE_ON) ? "on" : "off");
      }
   }
   else if ((strcasecmp(c0, "set"      ) == 0))
   {
      if (cc > 1)
      {
         if      ((strcasecmp(c1, "screen"   ) == 0))
         {
            if (cc > 2)
            {
               #if defined (USEWINDOWING)
               if       (strncasecmp(c2, "i", 1)  == 0)
               {
                  if (txwa->sbview)
                  {
                     txwa->sbview->window->sb.altcol ^= TXSB_COLOR_INVERT;
                  }
               }
               else if  (strncasecmp(c2, "b", 1)  == 0)
               {
                  if (txwa->sbview)
                  {
                     txwa->sbview->window->sb.altcol ^= TXSB_COLOR_BRIGHT;
                  }
               }
               else if  (strncasecmp(c2, "s", 1)  == 0)
               {
                  if (txwa->sbview)
                  {
                     txwa->sbview->window->sb.altcol ^= TXSB_COLOR_B2BLUE;
                  }
               }
               else
               #endif                           // USEWINDOWING
               if ((strcasecmp(c2, "on") == 0) || (c2[0] == '1'))
               {
                  TxScreenState(DEVICE_ON);
               }
               else
               {
                  TxScreenState(DEVICE_OFF);
               }
            }
            else
            {
               rc = TxScreenState( DEVICE_TEST);
               TxPrint("\nSet screen text-output properties\n\n Usage: %s %s  on | off"
               #if defined (USEWINDOWING)
                                                        " | invert | bright | swapblue  (toggle!)"
               #endif
                       "\n\nScreen output is switched %s\n", c0, c1, (rc == DEVICE_ON) ? "on" : "off");
            }
         }
   #if defined (USEWINDOWING)
      #if defined (LINUX)
         else if ((strncasecmp(c1, "clip", 7 ) == 0))
         {
            if (cc > 2)
            {
               if ((strcasecmp(c2, "on") == 0) || (c2[0] == 'c') || (c2[0] == 'C'))
               {
                  txwa->xSelClip = TRUE;
               }
               else
               {
                  txwa->xSelClip = FALSE;
               }
            }
            else
            {
               TxPrint("\nSet xclip use X-selection-CLIPBOARD, not -PRIMARY\n\n"
                       " Usage: %s %s on | off\n", c0, c1);
            }
            TxPrint("\nxclip use X-selection-CLIPBOARD currently set to: %s\n",
                    (txwa->xSelClip) ? "on (use CLIPBOARD)" : "off (use PRIMARY)");
         }
      #endif
   #endif
         else if ((strncasecmp(c1, "logfile", 7 ) == 0))
         {

            if (cc > 2)
            {
               if ((strcasecmp(c2, "on") == 0) || (c2[0] == '1'))
               {
                  TxLogfileState(DEVICE_ON);
               }
               else
               {
                  TxLogfileState(DEVICE_OFF);
               }
            }
            else
            {
               rc = TxLogfileState( DEVICE_TEST);
               TxPrint("\nSet logfile output status\n\n"
                       " Usage: %s %s on | off\n\n"
                       "LOGFILE is currently %s\n", c0, c1,
                       (rc == DEVICE_ON) ? "on" : "off");
            }
         }
         else if ((strncasecmp(c1, "ansi", 4 ) == 0))
         {
            if (cc > 2)
            {
               if ((strcasecmp(c2, "on") == 0) || (c2[0] == '1'))
               {
                  TxSetAnsiMode( A_ON);
               }
               else
               {
                  TxSetAnsiMode( A_OFF);
               }
            }
            else
            {
               TxPrint("\nSet usage of ANSI color for text output\n\n"
                       " Usage: %s %s on | off\n\n"
                       "ANSI is currently %s\n", c0, c1,
                       (TxGetAnsiMode() == A_ON) ? "ON" : "OFF");
            }
         }
   #if defined (USEWINDOWING)
         else if ((strncasecmp(c1, "history", 4 ) == 0))
         {
            if (cc > 2)
            {
               if      ((strcasecmp(c2, "plain" ) == 0) || (c2[0] == '0'))
               {
                  txwa->historyMode = TXW_HIST_PLAIN;
               }
               else if ((strcasecmp(c2, "filter") == 0) || (c2[0] == '1'))
               {
                  txwa->historyMode = TXW_HIST_FILTER;
               }
               else if ((strcasecmp(c2, "unique") == 0) || (c2[0] == '2'))
               {
                  txwa->historyMode = TXW_HIST_UNIQUE;
               }
               else
               {
                  txwa->historyMode = TXW_HIST_UNIFIL;
               }
            }
            else
            {
               TxPrint("\nSet filter behaviour for the history buffer (commandline, <F11>)\n\n"
                       " Usage: %s %s plain | 0 | filter | 1 | unique | 2 | classic | 3\n", c0, c1);
            }
            TxPrint("\nHISTORY behaviour is currently: ");
            switch (txwa->historyMode)
            {
               case TXW_HIST_PLAIN:  TxPrint( "PLAIN   (no duplicate removal, no completion filter)\n"); break;
               case TXW_HIST_FILTER: TxPrint( "FILTER  (no duplicate removal, completion filtering)\n"); break;
               case TXW_HIST_UNIQUE: TxPrint( "UNIQUE  (duplicate removal, no completion filtering)\n"); break;
               default:              TxPrint( "CLASSIC (duplicate removal and completion filtering)\n"); break;
            }
         }
         else if ((strncasecmp(c1, "scheme", 6 ) == 0))
         {
            if (cc > 2)
            {
               txwColorScheme( c2[0], NULL);
            }
            else
            {
               TxPrint("\nSet color-scheme used for Windowing\n\n"
                       " Usage: %s %s  grey|m3d|nobl|cmdr|white|black|dfsee\n", c0, c1);
            }
            TxPrint("\nSCHEME is currently: '%s'\n", txwcs->name);
            if (txwa->desktop != NULL)
            {
               txwInvalidateAll();
            }
         }
         else if ((strncasecmp(c1, "style", 5 ) == 0))
         {
            if (cc > 2)
            {
               txwcs->linestyle = (atol(c2) % TXW_CS_LAST);
            }
            else
            {
               TxPrint("\nSet linestyle used for Windowing\n\n"
                       " Usage: %s %s  0..3\n\n"
                       "        0=double 1=3d 2=halfblock 3=fullblock\n", c0, c1);
            }
            TxPrint("\nSTYLE is currently: %u\n", txwcs->linestyle);
            if (txwa->desktop != NULL)
            {
               txwInvalidateAll();
            }
         }
         else if ((strncasecmp(c1, "color", 5 ) == 0))
         {
            if (txwa->sbview)
            {
               TXWINDOW  *sbwin = txwa->sbview->window;

               if (cc > 2)
               {
                  sbwin->sb.altcol = atol(c2) & TXSB_COLOR_MASK;
                  txwInvalidateWindow((TXWHANDLE) txwa->sbview, FALSE, FALSE);
               }
               else
               {
                  TxPrint("\nSet color-scheme used for Output\n\n"
                          " Usage: %s %s  numeric-value 0 .. 7\n\n"
                          "  0 = standard output colors\n"
                          "  1 = invert all color values\n"
                          "  2 = force bright foreground\n"
                          "  4 = use Blue/Brown background\n\n"
                          "  Values can be combined by adding them.\n", c0, c1);
               }
               TxPrint("\nCOLOR value is currently: '%u'\n", sbwin->sb.altcol);
            }
            else
            {
               TxPrint("\nSet color-scheme for output not supported.\n");
            }
         }
         else if ((strncasecmp(c1, "palet", 5 ) == 0))
         {
            if (cc > 2)
            {
               int     colors = atoi( c2);
               TxSwitchAnsiPalette( colors);    // Set the palette
               txwRefreshScreen();              // flush, redraw everything
            }
            else
            {
               TxPrint("\nSet number of colors to be used\n\n"
                       " Usage: %s %s  1 | 2 | 8 | 16"

                 #if defined (UNIX)
                                                        " | 88 | 256"
                 #endif
                   "\n\n    1 = no color, just standard set  1 FG +  1 BG\n"
                       "    2 = no color, NORMAL + INTENSE   2 FG +  1 BG\n"
                       "    8 = standard set of 'PC' colors 16 FG +  8 BG\n"
                       "   16 = extended set of 'PC' colors 16 FG + 16 BG\n"
                 #if defined (UNIX)
                       "   88 = 16 FG/BG colors taken from palette of  88\n"
                       "  256 = 16 FG/BG colors taken from palette of 256\n"
                 #endif
                       , c0, c1);
            }
            TxPrint("\nPALETTE number of colors is currently: %d\n", txwa->colors);
         }
   #endif                                 // USEWINDOWING
         else if ((strncasecmp(c1, "asc", 3) == 0))
         {
            if (cc > 2)
            {
               if ((strcasecmp(c2, "on") == 0) || (c2[0] == '7'))
               {
                  TxSetAscii7Mode( TRUE);
               }
               else
               {
                  TxSetAscii7Mode( FALSE);
               }
               #if defined (USEWINDOWING)
                  txwRefreshScreen();           // flush, redraw everything
               #endif
            }
            else
            {
               TxPrint("\nSet ASCII output to 7-bit only\n\n"
                       " Usage: %s %s 7 | 8\n\n"
                       "ASCII is currently %s-bit\n", c0, c1,
                       (TxGetAscii7Mode()) ? "7" : "8");
            }
         }
         else if (strncasecmp(c1, "hidden", 6) == 0) // show hidden dirs/files, ON/OFF
         {
            if (cc > 2)
            {
               if ((strcasecmp(c2, "on") == 0) || (c2[0] == '1'))
               {
                  txwa->dialogShowHidden = TRUE;
               }
               else
               {
                  txwa->dialogShowHidden = FALSE;
               }
            }
            else
            {
               TxPrint("\nSet SHOWHIDDEN directories/files in File Dialogs\n\n"
                       " Usage: %s %s on | off\n\n", c0, c1);
            }
            TxPrint( "\nShow hidden files : %s\n", (txwa->dialogShowHidden)
                      ? "ON,  SHOW hidden files or directories in File Dialogs"
                      : "OFF, HIDE hidden files or directories in File Dialogs");
         }
         else if ((strncasecmp(c1, "insert", 6 ) == 0))
         {
            if (cc > 2)
            {
               if (strcasecmp(c2, "on") == 0)
               {
                  txwa->autoInsert = 1;
                  txwa->insert  = TRUE;
               }
               else if (isdigit( c2[ 0]))       // AUTO value specified
               {
                  txwa->autoInsert = atoi( c2);
                  txwa->insert = TRUE;
               }
               else
               {
                  txwa->autoInsert = 0;
                  txwa->insert = FALSE;
               }
            }
            else
            {
               TxPrint("\nSet (AUTO) insert-mode for entryfields\n" " Usage: %s %s off | 0 | on | 1 | 2..N (field length)\n", c0, c1);
            }
            if (txwa->autoInsert > 1)
            {
               TxPrint("\nAUTO INSERT threshold set: %d (INSERT ON when entryfield larger)\n", txwa->autoInsert);
            }
            else                                // FIXED insert setting
            {
               TxPrint("\nFIXED INSERT mode now set: %s\n", (txwa->insert) ? "on" : "off");
            }
         }
   #if defined (DEV32) || defined (WIN32)
         else if ((strncasecmp(c1, "sysmouse", 8 ) == 0))
         {
            if (cc > 2)
            {
            #if defined (DEV32)
             #if defined (USEWINDOWING)
               ULONG   beginstate = txwa->sysmouse;
             #endif
            #endif

               if      ((strcasecmp(c2, "off"     ) == 0) || (c2[0] == '0'))
               {
                  txwa->sysmouse = 0;
               }
            #if defined (DEV32)
               else if ((strcasecmp(c2, "dragtext") == 0) || (c2[0] == '1'))
               {
                  txwa->sysmouse = 1;
               }
            #endif
               else
               {
                  txwa->sysmouse = 2;
               }

               #if defined (DEV32)
                #if defined (USEWINDOWING)
                  //- Now fix-up the actual menu disabling, based on old and new setting value
                  //- either restoring the status at library-init time, or disabling menus
                  switch (beginstate)
                  {
                     case 0:
                        switch (txwa->sysmouse)
                        {
                           case  2: (void) TxSysMenuMouseActions(  txwa->smMouseAction);
                           case  1: (void) TxDragTextMouseMarking( txwa->smDragTextMark);  break;
                           default: break;
                        }
                        break;

                     case 1:
                        switch (txwa->sysmouse)
                        {
                           case  2: (void) TxSysMenuMouseActions(  txwa->smMouseAction);   break;
                           case  0: (void) TxDragTextMouseMarking( TX_UNCHECKED);          break;
                           default: break;
                        }
                        break;

                     case  2:
                     default:
                        switch (txwa->sysmouse)
                        {
                           case  0: (void) TxDragTextMouseMarking( TX_UNCHECKED);
                           case  1: (void) TxSysMenuMouseActions(  TX_UNCHECKED);          break;
                           default: break;
                        }
                        break;
                  }
                #endif
               #endif
            }
            else
            {
               #if defined (DEV32)
                  TxPrint("\nSet automatic OS/2 'Mouse Actions' and DragText 'Mouse Marking' management\n\n"
                          " Usage: %s %s off | 0 | dragtext | 1 | on | 2\n\n"
                          "  off      | 0   = Disable std OS/2 + Dragtext menu items until program exit\n"
                          "  dragtext | 1   = Disable std OS/2 item, leave DragText fully operational\n"
                          "  on       | 2   = Leave std OS/2 and DragText menu items under user control\n", c0, c1);
               #else
                  TxPrint("\nSet automatic Windows 'Quick Edit Mode' handling for console mouse marking\n\n"
                          " Usage: %s %s off | 0 | on | 1\n\n"
                          "  off      | 0   = Disable Windows console 'Quick Edit Mode' for this session\n"
                          "  on       | 1   = Leave Windows 'Quick Edit Mode' under user control\n", c0, c1);
               #endif
            }
            TxPrint("\nSYSMOUSE currently: ");
            switch (txwa->sysmouse)
            {
               #if defined (DEV32)
                  case 0:  TxPrint( "OFF      ('Mouse Actions' and DragText 'Mouse Marking' disabled)\n"); break;
                  case 1:  TxPrint( "DRAGTEXT ('Mouse Actions' disabled, DragText menus unchanged)\n");    break;
                  case 2:
                  default: TxPrint( "ON       ('Mouse Actions' and DragText menus under user control)\n"); break;
               #else
                  case 0:  TxPrint( "OFF      ('Quick Edit Mode' disabled during current session)\n"); break;
                  default: TxPrint( "ON       ('Quick Edit Mode' under user control)\n"); break;
               #endif
            }
         }
   #endif
         else if ((strncasecmp(c1, "type", 4  ) == 0))
         {
            if (cc > 2)
            {
               if ((strcasecmp(c2, "on") == 0) || (c2[0] == '1'))
               {
                  txwa->typeahead = TRUE;
               }
               else
               {
                  txwa->typeahead = FALSE;
               }
            }
            else
            {
               TxPrint("\nSet keyboard type-ahead\n\n" " Usage: %s %s on | off\n", c0, c1);
            }
            TxPrint("\nTYPEahead is currently set: %s\n", (txwa->typeahead) ? "on" : "off");
         }
         else if ((strncasecmp(c1, "radix", 5  ) == 0))
         {
            if (cc > 2)
            {
               if (toupper(c2[0]) == 'H')
               {
                  txwa->radixclass = 0xFFFFFFFF;
               }
               else
               {
                  txwa->radixclass = (ULONG) TxaParseNumber( c2, TX_RADIX_STD_CLASS, NULL);
               }
            }
            else
            {
               TxPrint("\nSet mcs-number Radix mask, default 0 = all decimal\n\n"
                       " Usage: %s %s H | 0 | mask\n\n"
                       "     H = all classes set to HEX\n"
                       "     0 = all classes set to DECimal\n"
                       "  mask = classes with bit SET will be HEX, others DECimal\n", c0, c1);
               rc = TX_PENDING;
            }
            TxPrint("\nNumber Radix class mask currently set to: 0x%8.8x = %u\n", txwa->radixclass, txwa->radixclass );
         }
         else if ((strncasecmp(c1, "reset", 5  ) == 0))
         {
            if (cc > 2)
            {
               if ((strcasecmp(c2, "on") == 0) || (c2[0] == '1'))
               {
                  txwa->resetterm = TRUE;
               }
               else
               {
                  txwa->resetterm = FALSE;
               }
            }
            else
            {
               TxPrint("\nReset terminal on program exit\n\n" " Usage: %s %s on | off\n", c0, c1);
            }
            TxPrint("\nRESET terminal is currently set: %s\n", (txwa->resetterm) ? "on" : "off");
         }
         else if ((strncasecmp(c1, "rmbmenu", 7  ) == 0))
         {
            if (cc > 2)
            {
               if ((strcasecmp(c2, "on") == 0) || (c2[0] == '1'))
               {
                  txwa->rmb2Menu = TRUE;
               }
               else
               {
                  txwa->rmb2Menu = FALSE;
               }
            }
            else
            {
               TxPrint("\nRight-Mouse-Button click opens (popup) Menu\n\n" " Usage: %s %s on | off\n", c0, c1);
            }
            TxPrint("\nRMB to Menu is currently set: %s\n", (txwa->rmb2Menu) ? "on" : "off");
         }
         #ifndef DOS32
         else if (strncasecmp(c1, "prio", 4  ) == 0)
         {
            TxPrint("\nRelative priority : ");
            switch (TxSetPriority( c2[0]))
            {
               case 'M': TxPrint( "Minimum\n");  break;
               case 'L': TxPrint( "Low\n");      break;
               case 'S': TxPrint( "Standard\n"); break;
               case 'H': TxPrint( "High\n");     break;
               case 'X': TxPrint( "maXimum\n");  break;
               default:  TxPrint( "unknown!\n"); break;
            }
            TxPrint( "\n");
         }
         #endif
         else                                   // non TXLIB set property
         {
            rc = TX_PENDING;
         }
      }
      else
      {
         TxPrint( "\nSet program property to specified value, or show current\n\n"
                  "Usage:      %s   property     value\n\n", c0);
         TxPrint( "TxWindows   SET properties :  (capital part required as keyword only)\n"
                  "  ANSI-colors     : on     |   off\n"
                  "  ASCii-7bit      : on     |   off\n"
         #if defined (USEWINDOWING)
            #if defined (LINUX)
                  "  CLIPselect      : on     |   off | clipboard | primary\n"
            #endif
                  "  HIDDEN          : on     |   off              (show hidden items in File Dialogs)\n");
         TxPrint( "  HISTORY  mode   : plain  |   0   | filter    |    1   | unique | 2 | classic | 3\n"
                  "  INSERT   mode   : off    |   0   | on        |    1   | 2 .. N (field  length)\n"
                  "  PALET colors    : 1 | 2  |   8   | 16"
               #if defined (UNIX)
                                                         "        |   88   | 256"
               #endif
         #endif
                "\n  LOGFILE         : on     |   off\n"
         #ifndef DOS32
                  "  PRIOrity        : Min    |   Low | Standard  | High   | maX    | Query\n"
         #endif
                  "  Radix           : H      |   0   | mask\n"
                  "  RESET           : on     |   off\n"
                  "  RMBMENU         : on     |   off\n"
                  "  SCREEN          : on     |   off"
         #if defined (USEWINDOWING)
                                                    " | invert    | bright | swapblue\n"
                  "  SCHEME color    : dfsee  |   3d  | nobl      | cmdr   | white  | black | grey"
         #endif
                  "\n"
         #if defined (DEV32)
                  "  SYSMOUSE mode   : off    |   0   | dragtext  |   1    | on     | 2 \n"
         #endif
         #if defined (WIN32)
                  "  SYSMOUSE mode   : off    |   0   | on        |   1 \n"
         #endif
                  "  TYPEahead       : on     |   off\n"
         #if defined (USEWINDOWING)
                  "  STYLE  lines    : numeric-value, range 0 .. 3\n"
                  "  COLOR  output   : numeric-value, range 0 .. 8\n"
         #endif
                  "\n");
         rc = TX_PENDING;
      }
   }
   else if ((strcasecmp(c0, "say"      ) == 0))
   {
      TxPrint("%s\n", pp);
   }
   else if ((strcasecmp(c0, "sleep"    ) == 0))
   {
      nr = atol( c1);
      if (nr == 0)
      {
         nr = 1;                                // default 1 sec
      }
      if (!TxaOption('q'))
      {
         TxPrint( "\nSleeping for %d seconds ...\n", nr);
      }
      TxSleep( nr * 1000);
   }

   #if defined (USEWINDOWING)
   else if ((strcasecmp(c0, "winhwnd" ) == 0))
   {
      #if defined (DEV32)
         if (txwa->api && txwa->api->pfnWQW && txwa->api->pfnWWFID)
         {
            TxPrint( "\nFrame  hwnd : %8.8lx\n", txwa->api->hwndFrame);
            TxPrint(   "   SYSMENU  : %8.8lx\n", (txwa->api->pfnWWFID)( txwa->api->hwndFrame, FID_SYSMENU  ));
            TxPrint(   "   TITLEBAR : %8.8lx\n", (txwa->api->pfnWWFID)( txwa->api->hwndFrame, FID_TITLEBAR ));
            TxPrint(   "   MINMAX   : %8.8lx\n", (txwa->api->pfnWWFID)( txwa->api->hwndFrame, FID_MINMAX   ));
            TxPrint(   "   CLIENT   : %8.8lx\n", (txwa->api->pfnWWFID)( txwa->api->hwndFrame, FID_CLIENT   ));
         }
         else
         {
            TxPrint( "\nPM environment with HWND not available!\n\n");
         }
      #else
         TxPrint( "\n%s not implemented for this platform (yet)\n\n", c0);
      #endif
   }
   else if ((strcasecmp(c0, "winlist"    ) == 0))
   {
      #if defined (DEV32)
         USHORT             i;
         size_t             usSWBSize;
         USHORT             count;
         PSWBLOCK           ppswblk;

         if (txwa->api->pfnWQSE != NULL)
         {
            count = (txwa->api->pfnWQSL)( txwa->api->hHAB, NULL, 0);
            if (count > 0)
            {
                TxPrint( "\nWindowlist entries: %u\n\n", count);
                usSWBSize = (count * sizeof(SWBLOCK));
                if ((ppswblk = (SWBLOCK *) TxAlloc( 1, usSWBSize)) != NULL)
                {
                    TxPrint( "%4.4s %8.8s %8.8s %4.4s %3.3s %1.1s %1.1s %-s\n",
                            "hsw", "hwnd", "hwndIcon", "PID", "SID", "V", "J", "szSwtitle");

                    count = (txwa->api->pfnWQSL)( txwa->api->hHAB, ppswblk, usSWBSize);
                    for (i = 0; i < count; i++)
                    {
                        TxPrint( "%4.4X %8.8x %8.8x %4.4x %3.3x %1.1x %1.1x %-s\n",
                                ppswblk->aswentry[i].hswitch,
                                ppswblk->aswentry[i].swctl.hwnd,
                                ppswblk->aswentry[i].swctl.hwndIcon,
                                ppswblk->aswentry[i].swctl.idProcess,
                                ppswblk->aswentry[i].swctl.idSession,
                                ppswblk->aswentry[i].swctl.uchVisibility,
                                ppswblk->aswentry[i].swctl.fbJump,
                                ppswblk->aswentry[i].swctl.szSwtitle
                              );
                    }
                    TxFreeMem( ppswblk);
                }
            }
         }
         else
         {
            TxPrint( "\nPM environment not available!\n\n");
         }
      #else
         TxPrint( "\n%s not implemented for this platform (yet)\n\n", c0);
      #endif
   }
   #endif

   else
   {
      rc = TX_CMD_UNKNOWN;
   }
   #if defined (UNIX)
      if (rc == (ULONG) -1)
      {
         rc = TX_UNIX_ERROR;                    // specific 16-bit error value
      }
   #endif
   RETURN (rc & 0xffff);                        // 16-bit, preserve flag values
}                                               // end 'TxStdCommand'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set current thread priority Min/Low/Std/High/maX
/*****************************************************************************/
char TxSetPriority                              // RET   resulting priority
(
   char                prio                     // IN    priority M/L/S/H/X/Q
)
{
   ENTER();
   TRACES(( "Priority command: %2.2hu\n", prio));

   if (prio && strchr( "mlshxMLSHX", prio) != NULL)
   {
      txsetPrio = toupper(prio);
      switch (txsetPrio)
      {
         case 'M': TxThreadPrioMin();  break;
         case 'L': TxThreadPrioLow();  break;
         case 'S': TxThreadPrioStd();  break;
         case 'H': TxThreadPrioHigh(); break;
         case 'X': TxThreadPrioMax();  break;
      }
   }
   RETURN (txsetPrio);
}                                               // end 'TxSetPriority'
/*---------------------------------------------------------------------------*/

