//
//                     TxWin, Textmode Windowing Library
//
//   Original code Copyright (c) 1995-2021 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   TxLib, released under MIT License
//
//   Copyright (c) 1995-2021  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on TxWin licensing can be directed to: info@dfsee.com
//
// ==========================================================================
//
// TxLib filesystem functions, base section
//
// JvW  28-02-2021 LICENSING: Changed from LGPL to the more liberal MIT license

#include <txlib.h>                              // TxLib interface
#include <txwpriv.h>                            // private window interface

#include <sys/stat.h>                           // for low level stuff


#if defined (USEWINDOWING)
// Sorting criteria attached to each volume list
static TXSELSORT       txfd_vol_sort =
{
  {                                             // actual, initial sort order
     TXS_SORT_4,                                // desc ascending case-insensitive letter
     TXS_SORT_TEXT | TXS_SORT_IGNCASE | TXS_SORT_ASCEND  | 23,  //- (Vsize)
     0,
     0,
     TXS_SORT_DESC | TXS_SORT_IGNCASE | TXS_SORT_ASCEND  |  7,  //- (letter)
     TXS_SORT_DESC | TXS_SORT_IGNCASE | TXS_SORT_ASCEND  | 23,  //- (fsys)
     TXS_SORT_DESC | TXS_SORT_IGNCASE | TXS_SORT_ASCEND  | 10,  //- (label)
     TXS_SORT_DESC | TXS_SORT_IGNCASE | TXS_SORT_ASCEND  | 37,  //- (Fspace)
  },
  {                                             // reverse   (c-F8 / c-R)
     "unsorted   <->",                          // unsorted  (c-F7 / c-U)
     "total size <->",                          // text   1  (c-F2 / c-X)
     "",                                        // text   2  (c-F1 / c-D)
     "",                                        // text   3  (c-F9 / c-A)
     "dr. letter <->",                          // desc   4  (c-F5 / c-T)
     "filesystem <->",                          // desc   5  (c-F6 / c-B)
     "vol. label <->",                          // desc   6  (c-F3 / c-N)
     "free space <->",                          // desc   7  (c-F4 / c-E)
  },
  ""                                            // current description
};
#endif


#if   defined (WIN32)

// OpenWatcom older than 1.8 (1280) did not have these ...
#if __WATCOMC__ < 1280
//- stuff needed for (DDK) IOCTL (conflicts if including whole ntddstor.h)

#pragma pack(4)
typedef enum _STORAGE_PROPERTY_ID
{
  StorageDeviceProperty = 0,
  StorageAdapterProperty,
  StorageDeviceIdProperty
} STORAGE_PROPERTY_ID, *PSTORAGE_PROPERTY_ID;

typedef enum _STORAGE_QUERY_TYPE
{
  PropertyStandardQuery = 0,
  PropertyExistsQuery,
  PropertyMaskQuery,
  PropertyQueryMaxDefined
} STORAGE_QUERY_TYPE, *PSTORAGE_QUERY_TYPE;

typedef struct _STORAGE_PROPERTY_QUERY
{
  STORAGE_PROPERTY_ID  PropertyId;
  STORAGE_QUERY_TYPE  QueryType;
  UCHAR  AdditionalParameters[1];
} STORAGE_PROPERTY_QUERY, *PSTORAGE_PROPERTY_QUERY;


typedef enum _STORAGE_BUS_TYPE
{
        BusTypeUnknown = 0x00,
        BusTypeScsi,
        BusTypeAtapi,
        BusTypeAta,
        BusType1394,
        BusTypeSsa,
        BusTypeFibre,
        BusTypeUsb,
        BusTypeRAID,
        BusTypeMaxReserved = 0x7F
} STORAGE_BUS_TYPE, *PSTORAGE_BUS_TYPE;


typedef struct _STORAGE_DEVICE_DESCRIPTOR
{
  ULONG  Version;
  ULONG  Size;
  UCHAR  DeviceType;
  UCHAR  DeviceTypeModifier;
  BOOLEAN  RemovableMedia;
  BOOLEAN  CommandQueueing;
  ULONG  VendorIdOffset;
  ULONG  ProductIdOffset;
  ULONG  ProductRevisionOffset;
  ULONG  SerialNumberOffset;
  STORAGE_BUS_TYPE  BusType;
  ULONG  RawPropertiesLength;
  UCHAR  RawDeviceProperties[1];
} STORAGE_DEVICE_DESCRIPTOR, *PSTORAGE_DEVICE_DESCRIPTOR;


#define IOCTL_STORAGE_QUERY_PROPERTY \
  CTL_CODE(IOCTL_STORAGE_BASE, 0x0500, METHOD_BUFFERED, FILE_ANY_ACCESS)

#pragma pack()
#endif // watcom older than 1.8

#elif defined (DOS32)

#define DRIVETYPE_REMOTE  0x1000                // BIT 12 from Ioctl 4409 dx
#define DRIVETYPE_RAMDRV  0x0800                // BIT 11 from Ioctl 4409 dx

// Return data structure for DOS INT21 function 7303
typedef struct _dos_disksize_ex
{
   USHORT              size;                    // size of structure
   USHORT              version;                 // IN/OUT structure version (0)
   ULONG               Spc;                     // Sectors per cluster  (compr)
   ULONG               Bps;                     // Bytes per Sector
   ULONG               ClusAvail;               // Available clusters
   ULONG               ClusTotal;               // Total     clusters
   ULONG               SectAvail;               // Available sectors
   ULONG               SectTotal;               // Total     sectors
   ULONG               CcprAvail;               // Available clusters   (compr)
   ULONG               CcprTotal;               // Total     clusters   (compr)
   ULONG               Reserved1;
   ULONG               Reserved2;
} DOS_DISKSIZE_EX;                              // end of struct "_dos_disksize_ex"


#elif defined (DARWIN)
#elif defined (LINUX)

#define BLKGETSIZE        0x1260
#define BLKSSZGET         0x1268

#define ROOT_DEVICE       "/dev/root"

#else

#define TXFSDC_BLOCKR         0x00              // block device removable
#define TXFSDC_GETBPB         0x00              // get device bpb info

#define TXFSDC_UNLOCK         0x00              // unlock logical drive
#define TXFSDC_LOCK           0x01              // lock logical drive
#define TXFSDC_EJECT          0x02              // eject removable
#define TXFSDC_LOAD           0x03              // load removable

typedef struct drivecmd
{
   BYTE                cmd;                     // 0=unlock 1=lock 2=eject
   BYTE                drv;                     // 0=A, 1=B 2=C ...
} DRIVECMD;                                     // end of struct "drivecmd"

#define TXFSBPB_REMOVABLE     0x08              // BPB attribute for removable

typedef struct drivebpb
{
   TXFS_EBPB           ebpb;                    // extended BPB
   BYTE                reserved[6];
   USHORT              cyls;
   BYTE                type;
   USHORT              attributes;              // device attributes
   BYTE                fill[6];                 // documented for IOCtl
} DRIVEBPB;                                     // end of struct "drivebpb"

#endif


/*****************************************************************************/
// Build string with present volumes, optional FLOPPY and LAN drives included
/*****************************************************************************/
int TxFsVolumes                                 // RET   nr of drives listed
(
   ULONG               flags,                   // IN    FLOPPY/LAN/CD select
   char               *vols                     // OUT   Present volumes
)
{
   #if   defined (WIN32)
   #elif defined (DOS32)
      union  REGS      regs;
      char            *ep;
   #elif defined (UNIX)
      //- to be refined
   #else
      ULONG            current;
   #endif
   USHORT              first = 2;               // default start at C:
   USHORT              ml;
   ULONG               drivemap = 0;
   TXTM                drive;

   ENTER();
   TxFsAutoFailCriticalErrors( TRUE);           // avoid Not-ready pop-ups

   if (flags & TXFSV_FLOP)                      // check floppies too
   {
      #if defined (DOS32)
         ep = getenv( "FLOPPYDR");              // DFSee bootable CD/diskette ?
         if (ep && (!strncasecmp( ep, "B:", 2)))   // Booted from FSYS boot CDROM
         {                                      // bootimage is A: - VERY SLOW!
            first = 1;                          // so start at B:
         }
         else                                   // in all other cases
         {                                      // start at diskette A:
            first = 0;
         }
      #else
         first = 0;
      #endif
   }

   #if   defined (WIN32)
      drivemap = GetLogicalDrives();
   #elif defined (DOS32)
      for (ml = first; ml < 26; ml++)           // drives A/B/C to Z
      {
         TxxClearReg( regs);
         regs.h.al = 0x09;                      // block device, remote
         regs.h.bl = (BYTE) (ml+1);             // 1=A, 2=B etc
         TxxDosInt21( regs, TXDX_DOS_IOCTL);
         if (regs.x.cflag == 0)                 // drive exists
         {
            //- to be refined, might use bit 12 (remote) as LAN indicator
            //- and perhaps the MSCDEX calls for CDROM filtering

            TRACES(( "Block device drivetype: 0x%4.4lx for drive: %c\n",
                      TXWORD.dx, (char) (ml + 'A')));
            drivemap |= (1 << ml);

            // TRHEXS( 70, &gpar, sizeof(TXFS_GPARM), "TXFS_GPARM");
         }
      }
   #elif defined (UNIX)
      //- to be refined
   #else
      DosQCurDisk( &current, &drivemap);
   #endif
   strcpy( vols, "");
   for (ml = first; ml < 26; ml++)              // drives A/B/C to Z
   {
      if (drivemap & (1 << ml))
      {
         TXTT       fstype;

         sprintf(drive, "%c:", 'A' + ml);
         if (TxFsType( drive, fstype, NULL))
         {
            BOOL       removable = TxFsIsRemovable( drive);
            BOOL       incl = TRUE;

            //- Note, connected Win-NT drives use the REAL FS-name like NTFS!
            if      ((strncasecmp( fstype, "LAN",    3) == 0) || //- OS/2 LAN drives
                     (strncasecmp( fstype, "NDFS32", 5) == 0) || //- Netdrive drives
                     (strncasecmp( fstype, "REMOTE", 5) == 0)  ) //- NT disconnected
            {                                                    //- DOS REMOTE
               incl = (flags & TXFSV_LAN);
            }
            else if ((strncasecmp( fstype, "CD",  2) == 0)  || // OS2:CDFS, DOS/WIN:CDROM
                     (strncasecmp( fstype, "UDF", 3) == 0)   ) // OS2:UDF DVD's
            {
               removable = TRUE;                // overrule for any CDROM/DVD
               incl      = (flags & TXFSV_CD);
            }
            if ((flags & TXFSV_REM) && (removable == FALSE))
            {
               incl = FALSE;                    // just want removables ...
            }
            if (incl)
            {
               drive[1] = 0;                    // clip to just the letter
               strcat( vols, drive);            // and add to volume-list
            }
         }
      }
   }
   TxFsAutoFailCriticalErrors( FALSE);          // enable criterror handler
   TRACES(("Volumes: '%s'\n", vols));
   RETURN(strlen( vols));
}                                               // end 'TxFsVolumes'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get volumelabel for specified driveletter, or -not-ready- if not accessible
/*****************************************************************************/
char *TxFsVolumeLabel                           // RET   label or "-not-ready-"
(
   char               *drive,                   // IN    Drive specification
   char               *label                    // INOUT Label string (min 12)
)
{
   #if   defined (WIN32)
      TXTS             DriveRoot;
      ULONG            ulMaxCompLen;
      ULONG            ulFsFlags;
      TXTS             fsname;
   #elif defined (DOS32)
      union  REGS      regs;
      struct SREGS     sreg;
      TXDX_RMINFO      txdx_rmi;
      char            *fn;
      char            *dta;
   #elif defined (UNIX)
      //- to be refined
   #else
      FSINFO           fsi;
   #endif

   ENTER();

   strcpy( label, TXFS_NOTREADY);
   #if   defined (WIN32)
      sprintf(DriveRoot, "%c:\\", drive[0]);
      if (!GetVolumeInformation( DriveRoot,
                                 label,  TXMAXTS,
                                 NULL, &ulMaxCompLen, &ulFsFlags,
                                 fsname, TXMAXTS))
      {
         TRACES(("Get failed, label: '%s' error: %u\n", label, GetLastError()));
      }
   #elif defined (DOS32)
      if ((txwa->dpmi1 != NULL) && (txwa->dpmi2 != NULL))
      {
         memset( txwa->dpmi1, 0, 512);
         memset( txwa->dpmi2, 0, 512);
         fn    = txwa->dpmi1;                   // DPMI compatible spec
         dta   = txwa->dpmi2;                   // DPMI compatible DTA

         sprintf( fn, "%c:\\*.*", drive[0]);    // DPMI comp drive-spec

         //- first DOS-int, 211A = Set DTA
         memset( &regs,  0, sizeof(regs));
         memset( &sreg,  0, sizeof(sreg));

         memset( &txdx_rmi, 0, sizeof(txdx_rmi));
         txdx_rmi.eax = 0x1A00;                 // Set DTA address AH=1A
         txdx_rmi.ds  = txDpmiSegment(dta);     // in  DTA   (ds:dx, dx=0)

         TRHEXS( 100,  &txdx_rmi,  sizeof(txdx_rmi), "txdx_rmi");

         regs.w.ax    = TXDX_DPMI_RMINT;        // simulate realmode INT
         regs.h.bl    = TXDX_DOS;               // DOS interrupt 21
         regs.x.edi   = FP_OFF( &txdx_rmi);     // real mode register struct
         sreg.es      = FP_SEG( &txdx_rmi);

         txDpmiCall( &regs, &sreg);

         TRACES(("regs.x.cflag:%4.4hx  ax:%4.4hx\n", regs.x.cflag, TXWORD.ax));

         if (regs.x.cflag == 0)                 // DTA is set now
         {
            //- second DOS-int, 214E = FindFirst
            memset( &regs,  0, sizeof(regs));
            memset( &sreg,  0, sizeof(sreg));

            memset( &txdx_rmi, 0, sizeof(txdx_rmi));
            txdx_rmi.eax = 0x4E00;              // FindFirst       AH=4E
            txdx_rmi.ecx = FATTR_LABEL;         // in  attribute
            txdx_rmi.ds  = txDpmiSegment(fn);   // in  spec  (ds:dx, dx=0)

            TRHEXS( 100,  &txdx_rmi,  sizeof(txdx_rmi), "txdx_rmi");

            regs.w.ax    = TXDX_DPMI_RMINT;     // simulate realmode INT
            regs.h.bl    = TXDX_DOS;            // DOS interrupt 21
            regs.x.edi   = FP_OFF( &txdx_rmi);  // real mode register struct
            sreg.es      = FP_SEG( &txdx_rmi);

            txDpmiCall( &regs, &sreg);

            TRHEXS( 100,  txwa->dpmi2,  512, "dpmi2 (DTA)");
            TRACES(("regs.x.cflag:%4.4hx  ax:%4.4hx\n", regs.x.cflag, TXWORD.ax));

            if (regs.x.cflag == 0)
            {
               strcpy( label, dta + 0x1e);      // label OUT
            }
         }
      }
   #elif defined (UNIX)
      //- to be refined
   #else
      if (DosQueryFSInfo( *drive - 'A' +1, FSIL_VOLSER,
                          &fsi, sizeof(FSINFO)) == NO_ERROR)
      {
         strcpy( label, fsi.vol.szVolLabel);
      }
   #endif
   TRACES(("label: '%s'\n", label));
   RETURN( label);
}                                               // end 'TxFsVolumeLabel'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Show volume/filesystem info for all drive-letters (or devices on UNIX)
/*****************************************************************************/
void TxFsDrivemap
(
   char               *lead,                    // IN    Lead text, max 3 chars
   ULONG               flags                    // IN    FLOPPY/LAN/CD select
)
{
   #if defined   (LINUX)
      FILE            *mnt;
   #elif defined (DARWIN)                       // DARWIN MAC OS X (GCC)
   #else
      TXTS             volumes;
   #endif
   #ifndef DARWIN
      TXLN             line;
      char            *s = NULL;
   #endif

   ENTER();

   #if defined (LINUX)
      if ((mnt = fopen( "/proc/mounts", "rb")) != NULL)
      {
         while (!feof(mnt) && !ferror(mnt))
         {
            if (fgets( line, TXMAXLN, mnt) != NULL)
            {
               if (strncasecmp( line, "/dev/", 5) == 0)
               {
                  //- to be refined, honnor FLOPY/LAN/CD flags ?
                  //- could do additional tests here ...

                  if ((s = strchr( line, ' ')) != NULL)
                  {
                     *s = 0;
                     TxFsShow( lead, line);
                  }
               }
            }
         }
         fclose( mnt);
      }
   #elif defined (DARWIN)                       // DARWIN MAC OS X (GCC)
      //- to be refined, use getmntinfo() with statfs structure
      //- and perhaps some filtering (review Linux too!)
      TxFsShow( lead, "");
   #else
      TxFsVolumes( flags, volumes);             // get available volumes
      for (s = volumes; *s && !TxAbort(); s++)  // walk all volume letters
      {
         sprintf( line, "%c:", *s);
         TxFsShow( lead, line);
      }
   #endif
   VRETURN();
}                                               // end 'TxFsDrivemap'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Show volume/filesystem info on a single output line using TxPrint
/*****************************************************************************/
ULONG TxFsShow
(
   char               *lead,                    // IN    Lead text, max 3 chars
   char               *drive                    // IN    Drive spec or device
)
{
   ULONG               rc = TX_INVALID_DRIVE;
   #if defined (DARWIN)
      TXTM             device;
      char            *diskinfo = NULL;         // diskutil list output text
      TXLN             command;                 // diskutil command / output
   #else
      ULN64            frees;                   // Free sectors
      ULN64            total;                   // Total sectors
      USHORT           bps;                     // Sectorsize
      BOOL             fType = FALSE;
      TXTM             text;
      TXTM             detail;
      #if defined (UNIX)
         TXTM          device;
      #endif
   #endif
   TXTT                leader;

   ENTER();

   strcpy( leader, lead);                       // optional lead fragment
   #if defined (UNIX)
      strcpy( device, drive);                   // copy (partition) devicename
      if (isdigit( device[ 8]))                 // sdaX probing
      {
         device[ 8] = 0;                        // remove partition part of devname
      }
      if (TxFsIsRemovable( device))
      {
         strcat( leader, "Removable dev.");
      }
      else
      {
         strcat( leader, "Mounted device");
      }
   #else
      if (TxFsIsRemovable( drive))
      {
         strcat( leader, "RemovableDrive");
      }
      else
      {
         strcat( leader, "Volume info on");
      }
   #endif

   TxFsAutoFailCriticalErrors( TRUE);           // avoid Not-ready pop-ups
   #ifndef UNIX
   if (isupper(drive[0]))
   #endif

   #ifndef DARWIN
   {
      fType = TxFsType( drive, text, detail);
      TxStrip( text, text, 0, ' ');             // strip spaces from FS-name
      if (strcmp(text, "UNKNOWN") != 0)
      {
         TxStrToUpper( text);                   // uppercase for readability
         rc = TxFsSpace(drive, &frees, &total, &bps);
      }
   }
   #endif
   #if defined (LINUX)
      if (rc == NO_ERROR)
      {
         strcpy( device, drive);
         TxLinuxRootDevice(drive, device);      // translate root device
         TxPrint("%-18.18s: %s%-15.15s%s Size% 9.1lf MiB, fs: %s%-8.8s%s  %s%s\n",
                  leader, CBG, device, CNN,
                  (total == 0) ? 0 : TXSMIB(total,  bps),
                  CBM, text, CBY, detail, CNN);
      }
      else
      {
         TxPrint("%-18.18s: %s%-15.15s%s is not accessible, fs: %s%s%s\n",
                  leader, CBG, drive, CNN, CBM, (fType) ? text : "-", CNN);
      }
   #elif defined (DARWIN)                       // DARWIN MAC OS X (GCC)
      sprintf( command, "diskutil list | grep %s", (drive[0] == 'r') ? drive + 1: drive);
      rc = txcExecRedirectIO( command, NULL, &diskinfo);
      if (rc == NO_ERROR)
      {
         TxRepl( diskinfo, '\n', 0);            // terminate at first end-of-line
         strcpy( command, diskinfo + 5);        // copy part AFTER the item number
         TxStrip( command, command, ' ', ' ');  // and strip leading/trailing spaces
         TxPrint("%-18.18s: %s%-15.15s%s %s\n", leader, CBG, drive, CNN, command);
         TxFreeMem( diskinfo);
      }
      else
      {
         TxPrint("%-18.18s: %s%-15.15s%s is not listed by 'diskinfo list'\n", leader, CBG, drive, CNN);
      }
   #else
      if (rc == NO_ERROR)
      {
         TxPrint("%-18.18s: %s%c:%s Free% 9.1lf MiB of% 9.1lf MiB, fs: %s%-8.8s%s  %s%s\n",
                  leader, CBG, drive[0], CNN,
                  (frees == 0) ? 0 : TXSMIB(frees,  bps),
                  (total == 0) ? 0 : TXSMIB(total,  bps),
                  CBM, text, CBY, detail, CNN);
      }
      else
      {
         TxPrint("%-18.18s: %s%s%s           Not ready, no size info, fs: %s%s%s\n",
                  leader, CBG, drive, CNN, CBM, (fType) ? text : "-", CNN);
      }
   #endif
   TxFsAutoFailCriticalErrors( FALSE);          // enable criterror handler
   RETURN (rc);
}                                               // end 'TxFsShow'
/*---------------------------------------------------------------------------*/


#if defined (USEWINDOWING)
/*****************************************************************************/
// Build selection-list with volume/FS info for drive-letters or Linux-devices
/*****************************************************************************/
TXSELIST *TxFsDriveSelist                       // RET   selection list or NULL
(
   ULONG               flags,                   // IN    include FLOP/LAN/CDROM
   BOOL                flop                     // IN    runtime floppy test
)                                               //       (A: = 0, C: = 2 etc)
{
   TXSELIST           *list  = NULL;            // total list
   #ifndef DARWIN
   TXS_ITEM           *item;                    // single item
   ULONG               entries = 0;             // entries in list
   int                 i;
   #endif
   #if defined (LINUX)
      FILE            *mnt;
      TXLN             line;
      TXLN             device;
      TXLN             mount;
      TXLN             fsys;
      TXLN             rest;                    // need a LONG line here!
   #elif defined (DARWIN)                       // DARWIN MAC OS X (GCC)
   #else
      TXTM             volumes;
      char             curdrive;
      char            *s;
      char            *startvolume = volumes;
   #endif

   ENTER();

   #if defined (LINUX)
      if ((mnt = fopen( "/proc/mounts", "rb")) != NULL)
      {
         while (!feof(mnt) && !ferror(mnt))
         {
            if (fgets( line, TXMAXLN, mnt) != NULL)
            {
               //- to be refined, honor exclude-floppy and LAN flags ?
               if (strncasecmp( line, "/dev/", 5) == 0)
               {
                  entries++;                    // count valid device names
               }
            }
         }
         if (TxSelCreate( entries, entries, entries,
                          TXS_AS_NOSTATIC, FALSE, NULL, &list) == NO_ERROR)
         {
            char           *listdescr;          // list level description

            list->astatus = TXS_AS_NOSTATIC      | // all dynamic allocated
                            TXS_LST_DESC_PRESENT | // with list description
                            TXS_LST_DYN_CONTENTS;

            if ((listdescr  = TxAlloc( 1, TXMAXTM)) != NULL)
            {
               sprintf( listdescr, "mounted device to be used (from /proc/mount)");

               list->miscInfo = listdescr;
            }
            fseek( mnt, 0, SEEK_SET);           // rewind to start of file
            for (i = 0; (i < entries) && !feof(mnt) && !ferror(mnt);)
            {
               if (fgets( line, TXMAXLN, mnt) != NULL)
               {
                  if (strncasecmp( line, "/dev/", 5) == 0)
                  {
                     if ((item  = TxAlloc( 1, sizeof(TXS_ITEM))) != NULL)
                     {
                        list->count    = i +1;  // actual item count
                        list->items[i] = item;  // attach item to list

                        if (((item->text = TxAlloc( 1, TXMAXLN)) != NULL) &&
                            ((item->desc = TxAlloc( 1, TXMAXLN)) != NULL)  )
                        {
                           item->value  = TXDID_MAX +  i;
                           item->helpid = TXWH_USE_WIN_HELP; // from list-window itself

                           item->index  = 0;    // no quick-select

                           line[ TXMAXTM -1] = 0;

                           sscanf( line, "%s %s %s %s", device, mount, fsys, rest);

                           TxLinuxRootDevice( device, device); // translate root device
                           rest[ 2] = 0;
                           TxStrToUpper( rest);

                           sprintf( item->text, "%-20s %s", device, fsys);
                           sprintf( item->desc, "%-8.8s %-20.20s mounted %s at: %-35s",
                                                 fsys, device, rest, mount);

                           TRACES(("text: %d '%s'\n", strlen(item->text), item->text));
                           TRACES(("desc: %d '%s'\n", strlen(item->desc), item->desc));
                        }
                     }
                     i++;                       // to next entry
                  }
               }
            }
         }
         fclose( mnt);
      }
   #elif defined (DARWIN)                       // DARWIN MAC OS X (GCC)
   #else
      getcwd( volumes, TXMAXTM);                // get current drive/directory
      curdrive = toupper(volumes[0]);           // keep current driveletter

      TxFsVolumes( flags, volumes);             // get available volumes

      #if defined (DOS32)
         if (TxPhysDisketteDrives() == 1)       // just one drive really there
         {
            if (strncasecmp( volumes, "AB", 2) == 0) // A and B are listed
            {
               startvolume++;                   // quick hack to remove first
               if ((s = getenv("FLOPPYDR")) != NULL)
               {
                  startvolume[0] = s[0];        // env var used by program to
               }                                // indicate floppy driveletter
            }
         }
      #endif
      if ((entries = strlen(startvolume)) != 0)
      {
         TRACES(( "Driveletters to be added to selection: '%s'\n", startvolume));
         if (TxSelCreate( entries, entries, entries,
                          TXS_AS_NOSTATIC, FALSE, NULL, &list) == NO_ERROR)
         {
            char           *listdescr;          // list level description

            list->astatus = TXS_AS_NOSTATIC      | // all dynamic allocated
                            TXS_LST_DESC_PRESENT | // with list description
                            TXS_LST_DYN_CONTENTS;


            if ((listdescr  = TxAlloc( 1, TXMAXTM)) != NULL)
            {
               sprintf( listdescr, "%svolume (driveletter) to be used",
                        (flags & TXFSV_REM) ? "removable " : "");

               list->miscInfo = listdescr;
            }
            for (i = 0, s = startvolume; i < entries; i++, s++) // walk all volumes
            {
               if ((item  = TxAlloc( 1, sizeof(TXS_ITEM))) != NULL)
               {
                  list->count    = i +1;        // actual item count
                  list->items[i] = item;        // attach item to list
                  item->userdata = (void *) list->count; // assign item number for unsort

                  if (((item->text = TxAlloc( 1, TXMAXTT)) != NULL) &&
                      ((item->desc = TxAlloc( 1, TXMAXLN)) != NULL)  )
                  {
                     char    drive[3];

                     sprintf(drive, "%c:", *s);

                     item->value  = TXDID_MAX +  (*s - 'A');

                     item->helpid = TXWH_USE_WIN_HELP; // from list-window itself

                     item->index  = 1;          // drive-letter is quick-select
                     if ((!flop) && (*s < 'C')) // simulate floppy info ?
                     {
                        sprintf( item->text, "%s             FAT",  drive);
                        sprintf( item->desc, "Diskette drive %s   Status "
                                             "not determined yet.", drive);
                     }
                     else
                     {
                        ULONG   rc    = NO_ERROR;
                        ULN64   frees = 0;      // Free sectors
                        ULN64   total = 0;      // Total sectors
                        USHORT  bps;            // Sectorsize
                        TXTM    fsys;
                        TXTM    detail;
                        TXTS    label;

                        TxFsAutoFailCriticalErrors( TRUE); // avoid Not-ready pop-ups
                        TxFsType( drive, fsys, detail);

                        TxFsVolumeLabel( drive, label);
                        if (strcmp( label, TXFS_NOTREADY) != 0) // no label access
                        {
                           if (strcmp( fsys, "UNKNOWN") != 0) // unknown filesystem
                           {
                              rc = TxFsSpace( drive, &frees, &total, &bps);
                           }
                        }
                        else
                        {
                           rc = TX_INVALID_DRIVE;
                        }
                        TRACES(( "Drive %s  label: '%s' fsys: '%s' detail: '%s' RC:%u\n", drive, label, fsys, detail, rc));
                        if (rc == NO_ERROR)
                        {
                           sprintf( item->text, "%s %-11.11s %-8.8s %9.1lf MiB",
                                    drive, label, fsys,
                                   (total == 0) ? 0.0 : TXSMIB( total, bps));
                           TRACES(("text: %d '%s'\n", strlen(item->text), item->text));
                           TRACES(("t*:%8.8x  d*:%8.8x\n", item->text,  item->desc));
                           TRACES(("total:%u  frees:%u  bps:%hu\n", total, frees, bps));

                           sprintf( item->desc, "Drive %s %-11.11s %-8.8s Free: %9.1lf MiB  %s",
                                    drive, label, fsys,
                                    (frees == 0) ? 0.0 : TXSMIB( frees, bps), detail);
                        }
                        else
                        {
                           sprintf( item->text, "%s %-11.11s %-8.8s", drive, label, fsys);
                           TRACES(("text: %d '%s'\n", strlen(item->text), item->text));
                           sprintf( item->desc, "Drive %s   Removable not ready "
                                                "or unknown filesystem", drive);
                        }
                        TxFsAutoFailCriticalErrors( FALSE); // enable criterror handler
                     }
                     TRACES(("desc: %d '%s'\n", strlen(item->desc), item->desc));
                  }
               }
               if (*s == curdrive)
               {
                  list->selected = i;           // make this the selected one
               }
            }
         }
      }
   #endif
   if (list == NULL)                            // empty sofar ?
   {
      list = TxSelEmptyList( "- No volumes matching the criteria -",
            "No volumes found that match the criteria for this list", FALSE);
   }
   if (list != NULL)                            // attach sort information
   {
      TRACES(("Count:%4lu selected now: %u = '%s'\n", list->count, list->selected,
                                                       list->items[ list->selected]->text));
      list->sortinfo = &txfd_vol_sort;
      TxSelSortCurrent( list);
      list->selected = 0;                       // make sure FIRST is selected after sort!

      TRACES(("Count:%4lu selected now: %u = '%s'\n", list->count, list->selected,
                                                       list->items[ list->selected]->text));
   }
   RETURN( list);
}                                               // end 'TxFsDriveSelist'
/*---------------------------------------------------------------------------*/
#endif                                          // USEWINDOWING


/*****************************************************************************/
// Determine attached fstype, e.g. HPFS for specified drive
/*****************************************************************************/
BOOL TxFsType                                   // RET   FS type resolved
(
   char               *drive,                   // IN    Drive specification
   char               *fstype,                  // OUT   Attached FS type
   char               *details                  // OUT   details (UNC) or NULL
)
{
   BOOL                rc = FALSE;
   #if defined (WIN32)
      TXTS             DriveRoot;
      ULONG            ulMaxCompLen;
      ULONG            ulFsFlags;
   #elif defined (DOS32)
      union  REGS      regs;
      USHORT           drtype;
   #elif defined (LINUX)
      FILE            *mnt;
      TXLN             line;
      TXTM             mount;
      TXTM             rest;
      TXTM             device;
      TXTM             target;                  // target device name (drive)
      int              length;
   #elif defined (DARWIN)                       // DARWIN MAC OS X (GCC)
   #else
      FSQBUFFER2   *fsinfo;                     // Attached FS info
      ULONG         fsdlen = 2048;              // Fs info data length
   #endif

   ENTER();
   TRACES(("drive: '%s'\n", drive));

   strcpy(fstype, "none");
   if (details)
   {
      strcpy(details, "");
   }

   #if defined (WIN32)
      sprintf(DriveRoot, "%c:\\", drive[0]);
      rc = GetVolumeInformation( DriveRoot,
                                 NULL, 0,       // no volume name
                                 NULL,          // no volume serialnr
                                 &ulMaxCompLen,
                                 &ulFsFlags,
                                 fstype,
                                 TXMAXTS);
      if (rc == FALSE)                          // no mounted filesystem
      {
         rc = TRUE;
         switch (GetDriveType(DriveRoot))
         {
            case DRIVE_REMOTE : strcpy(fstype, "REMOTE "); break;
            case DRIVE_CDROM  : strcpy(fstype, "CDROM  "); break;
            case DRIVE_RAMDISK: strcpy(fstype, "RAMDISK"); break;

            default:
               rc = FALSE;
               break;
         }
      }
   #elif defined (DOS32)
      TxxClearReg( regs);
      regs.h.al = 0x09;                         // block device, remote
      regs.h.bl = drive[0] - 'A' +1;            // 1=A, 2=B etc
      TxxDosInt21( regs, TXDX_DOS_IOCTL);
      if (regs.x.cflag == 0)                    // drive exists
      {
         drtype = TXWORD.dx;                    // keep drivetype bits

         TxxClearReg( regs);
         TXWORD.bx = 0;
         TxxMplexInt( regs, TXDX_MPLEX_MSCDEX_INSTALL);
         if (TXWORD.bx != 0)                    // nr of CDROM driveletters
         {
            TxxClearReg( regs);
            TXWORD.cx = drive[0] - 'A';         // 0=A
            TxxMplexInt( regs, TXDX_MPLEX_MSCDEX_DRCHECK);
            if (TXWORD.ax != 0)                 // nonzero if a supported CDROM
            {
               strcpy(fstype, "CDROM  ");
               rc = TRUE;
            }
         }
         if (rc == FALSE)                       // not identified yet
         {
            if (drtype == DRIVETYPE_RAMDRV)     // MS or FreeDOS ramdisk
            {
               strcpy(fstype, "RAMDISK");
               rc = TRUE;
            }
            else if (drtype & DRIVETYPE_REMOTE) // drive is REMOTE
            {
               strcpy(fstype, "REMOTE ");
               rc = TRUE;
            }
            else                                // assume rest is local
            {
               if (drive[0] < 'C')
               {
                  strcpy(fstype, "FAT12  ");    // must be some FAT :-)
               }
               else
               {
                  strcpy(fstype, "FAT(32)");    // must be some FAT :-)
               }
               rc = TRUE;
            }
         }
      }
   #elif defined (LINUX)
      if ((mnt = fopen( "/proc/mounts", "rb")) != NULL)
      {
         strcpy( target, drive);
         strcat( target, " ");                  // add space to distinguish 1 from 11
         length = strlen( target);

         TRACES(("match fstab lines starting with: '%s'\n", target));

         while (!feof(mnt) && !ferror(mnt))
         {
            if (fgets( line, TXMAXLN, mnt) != NULL) // line HAS spaces after the devicename
            {
               strcpy( device, "");
               TxLinuxRootDevice(line, device); // translate root device
               strcat( device, " ");            // make sure there is a trailing space

               TRACES(("Match '%s' to line: '%s'\n", target, line));

               if ((strncasecmp( line,   target, length) == 0) || // direct match
                   (strncasecmp( device, target, length) == 0)  ) // translated match
               {
                  line[ TXMAXTM -1] = 0;
                  sscanf( line, "%*s %s %s %s", mount, fstype, rest);
                  if (details)
                  {
                     rest[ 2] = 0;
                     TxStrToUpper( rest);

                     TRACES(("rest: '%s'  mount: '%s'\n", rest, mount));
                     sprintf( details, "%s %s", rest, mount);
                  }
                  rc = TRUE;
                  break;                        // out of the while ...
               }
            }
         }
         fclose( mnt);
      }
   #elif defined (DARWIN)                       // DARWIN MAC OS X (GCC)
   #else
      if ((fsinfo = TxAlloc(1, fsdlen)) != NULL)
      {
         if (DosQFSAttach(drive, 0, 1, fsinfo, &fsdlen) == NO_ERROR)
         {
            strcpy(fstype,      fsinfo->szName + fsinfo->cbName +1);
            if (details && (fsinfo->cbFSAData != 0))
            {
               strcpy( details, fsinfo->szName + fsinfo->cbName +
                                                 fsinfo->cbFSDName +2);
            }
            TRACES(("FS item type  : %hu\n", fsinfo->iType));
            TRHEXS( 70, fsinfo, fsdlen, "FSQBUFFER2");
            rc = TRUE;
         }
         else
         {
            TRACES(("DosQFSAttach for drive '%s' failed\n", drive));
         }
         TxFreeMem(fsinfo);
      }
      else
      {
         TRACES(("TxAlloc buffer for drive '%s' failed\n", drive));
      }
   #endif
   TRACES(("Filesystem name : '%s' for drive: '%s' details: '%s'\n",
                        fstype, drive, (details) ? details : ""));
   BRETURN (rc);
}                                               // end 'TxFsType'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Eject a removable medium specified by driveletter
/*****************************************************************************/
ULONG TxFsEjectRemovable
(
   char               *drive                    // IN    Driveletter to eject
)
{
   ULONG               rc = NO_ERROR;           // DOS rc
   #if   defined (WIN32)
   #elif defined (DOS32)
   #elif defined (LINUX)
   #elif defined (DARWIN)                       // DARWIN MAC OS X (GCC)
   #else
      DRIVECMD         IOCtl;
      ULONG            DataLen;
      ULONG            ParmLen;
   #endif

   ENTER();
   TxFsAutoFailCriticalErrors( TRUE);           // avoid Not-ready pop-ups

   TRACES(("Drive: %s\n", drive));
   #if   defined (WIN32)
   #elif defined (DOS32)
   #elif defined (LINUX)
   #elif defined (DARWIN)                       // DARWIN MAC OS X (GCC)
   #else
      ParmLen   = sizeof(DRIVECMD);
      IOCtl.cmd = TXFSDC_EJECT;
      IOCtl.drv = toupper(drive[0]) - 'A';
      DataLen   = 0;

      rc = DosDevIOCtl((HFILE) -1, IOCTL_DISK,
                       DSK_UNLOCKEJECTMEDIA,
                       &IOCtl, ParmLen, &ParmLen,
                       NULL,   DataLen, &DataLen);
   #endif
   TxFsAutoFailCriticalErrors( FALSE);          // enable criterror handler
   RETURN (rc);
}                                               // end 'TxFsEjectRemovable'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Non-UNIX: Determine if a driveletter represents a removable medium/device
// Windows:  Determine if a numeric dsk represents a removable medium/device
// UnixLike: Determine if a disk device represents a removable medium/device
/*****************************************************************************/
BOOL TxFsIsRemovable                            // RET   drive is removable
(
   char               *drive                    // IN    Driveletter or device
)
{
   BOOL                rc = FALSE;
   #if   defined (WIN32)
      TXTS             dspec;
      TXHFILE          dh   = 0;                // Direct Filehandle
   #elif defined (DOS32)
   #elif defined (LINUX)
      FILE            *fh;
      char            *baredev;                 // bare device name
      TXTM             remindicator;            // full path to indicator file
      TXLN             line;
   #elif defined (DARWIN)                       // DARWIN MAC OS X (GCC)
      TXLN             command;                 // redirected command
   #else
      DRIVECMD         IOCtl;
      DRIVEBPB         RemAt;
      ULONG            DataLen;
      ULONG            ParmLen;
      BYTE             NoRem;
   #endif

   ENTER();
   TxFsAutoFailCriticalErrors( TRUE);           // avoid Not-ready pop-ups

   TRACES(("Drive: %s\n", drive));
   #if   defined (WIN32)
      sprintf(dspec, "%c:\\", drive[0]);
      switch (GetDriveType(dspec))
      {
         case DRIVE_REMOVABLE:
            TRACES(("GetDriveType reports REMOVABLE\n"));
            rc = TRUE;
            break;

         default:
            if (drive[0] < 'A')                 // must be numeric, disk level
            {
               sprintf( dspec, "\\\\.\\PHYSICALDRIVE%s", drive); // physical disk specification
            }
            else                                // alpha, must be a driveletter
            {
               sprintf( dspec, "\\\\.\\%c:", drive[0]); // logical volume specification
            }
            dh = CreateFile( dspec, 0,          // no real access needed!
                             FILE_SHARE_READ | FILE_SHARE_WRITE,
                             NULL, OPEN_EXISTING, 0, NULL);
            TRACES(( "CreateFile '%s' gets handle: %u\n", dspec, dh));
            if (dh != INVALID_HANDLE_VALUE)
            {
               ULONG         received;          // nr of bytes received
               STORAGE_PROPERTY_QUERY    query; // query request
               STORAGE_DEVICE_DESCRIPTOR sdd;   // device descriptor

               memset( &sdd,   0, sizeof( sdd  ));
               memset( &query, 0, sizeof( query));
               query.PropertyId = StorageDeviceProperty;
               query.QueryType  = PropertyStandardQuery;
               sdd.Size = sizeof( STORAGE_DEVICE_DESCRIPTOR);
               sdd.RemovableMedia = FALSE;
               sdd.BusType        = BusTypeUnknown;

               TRACES(("sdd.size on entry: %u\n", sdd.Size));

               if (DeviceIoControl(dh,
                                   IOCTL_STORAGE_QUERY_PROPERTY,
                                   &query, sizeof(query),
                                   &sdd,   sdd.Size,
                                   &received, NULL))
               {
               }
               else
               {
                  TRACES(( "IOCTL_STORAGE_QUERY_PROPERTY for '%s' error: %s\n", dspec, txNtLastError()));
               }
               TRACES(("sdd.size on exit : %u  RawProp size: %u\n", sdd.Size, sdd.RawPropertiesLength));
               TRACES(( "BusType: %u Rem:%u\n", sdd.BusType, sdd.RemovableMedia));
               if ((sdd.RemovableMedia)        ||
                   (sdd.BusType == BusTypeUsb) ||
                   (sdd.BusType == BusType1394) )
               {
                  rc = TRUE;
               }
               CloseHandle( dh);
            }
            break;
      }
   #elif defined (DOS32)
   #elif defined (LINUX)
     TxLinuxRootDevice( drive, line);           // Copy and translate root device
     TxStrip( line, line, 0, ' ');              // strip spaces from devicename
     baredev = strrchr( line, '/');
     if (baredev != NULL)
     {
        sprintf(  remindicator, "/sys/block%s/removable", baredev);
        TRACES(( "remindicator file: '%s'\n", remindicator));

        if ((fh = fopen( remindicator, "rb")) != NULL)
        {
           if (fgets( line, TXMAXLN, fh) != NULL)
           {
              if (line[0] == '1')               // indicator says REMOVABLE
              {
                 rc = TRUE;
              }
           }
           fclose( fh);
        }
     }
   #elif defined (DARWIN)                       // DARWIN MAC OS X (GCC)
      sprintf(    command, "diskutil list %s | grep external > /dev/null 2>&1", drive);
      TRACES(("command: '%s'\n", command));
      if (system( command) == 0)                // execute the command
      {
         rc = TRUE;
      }
   #else                                        // OS/2
      ParmLen   = sizeof(IOCtl);
      IOCtl.cmd = TXFSDC_BLOCKR;
      IOCtl.drv = toupper(drive[0]) - 'A';
      DataLen   = sizeof(NoRem);

      if (DosDevIOCtl((HFILE) -1, IOCTL_DISK,
                                  DSK_BLOCKREMOVABLE,
                                  &IOCtl, ParmLen, &ParmLen,
                                  &NoRem, DataLen, &DataLen) == NO_ERROR)
      {
         if (NoRem)                             // non-removable sofar, check
         {                                      // BPB as well (USB devices)
            ParmLen   = sizeof(IOCtl);
            IOCtl.cmd = TXFSDC_GETBPB;
            IOCtl.drv = toupper(drive[0]) - 'A';
            DataLen   = sizeof(RemAt);

            if (DosDevIOCtl((HFILE) -1, IOCTL_DISK,
                                        DSK_GETDEVICEPARAMS,
                                        &IOCtl, ParmLen, &ParmLen,
                                        &RemAt, DataLen, &DataLen) == NO_ERROR)

            {
               if (RemAt.attributes & TXFSBPB_REMOVABLE)
               {
                  TRACES(( "BPB removable, like USB\n"));
                  rc = TRUE;                    // removable, probably USB
               }
            }
         }
         else
         {
            TRACES(( "Block removable\n"));
            rc = TRUE;                          // removable block device
         }
      }
   #endif
   TxFsAutoFailCriticalErrors( FALSE);          // enable criterror handler
   BRETURN (rc);
}                                               // end 'TxFsIsRemovable'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Determine free and total-available space (sectors) on specified drive
/*****************************************************************************/
ULONG TxFsSpace
(
   char               *drive,                   // IN    Drive specification
   ULN64              *sfree,                   // OUT   Free sectors
   ULN64              *stotal,                  // OUT   Total sectors
   USHORT             *bpsect                   // OUT   Sectorsize
)
{
   ULONG               rc = NO_ERROR;
   #if defined (WIN32)
      TXTS             DriveRoot;
      ULONG            cSectorUnit;
      ULONG            cUnitAvail;
      ULONG            cUnit;
      ULONG            cbSector;
   #elif defined (DOS32)
      union  REGS      regs;
      struct SREGS     sreg;
      TXDX_RMINFO      txdx_rmi;
      char            *dRoot;
      DOS_DISKSIZE_EX *sizex;
   #elif defined (LINUX)
      TXHFILE          dh = 0;                  // Direct Filehandle
      TXTM             device;
   #elif defined (DARWIN)                       // DARWIN MAC OS X (GCC)
   #else
      FSALLOCATE       FSAllocate;
   #endif

   ENTER();

   *sfree  = 0;
   *stotal = 0;
   *bpsect = 512;

   #if defined (WIN32)
      sprintf(DriveRoot, "%c:\\", drive[0]);
      if (GetDiskFreeSpace(DriveRoot,
             &cSectorUnit, &cbSector, &cUnitAvail, &cUnit))
      {
         *sfree   = (ULN64) cSectorUnit * cUnitAvail;
         *stotal  = (ULN64) cSectorUnit * cUnit;
         *bpsect  = (USHORT)      cbSector;
      }
      else
      {
         rc = TX_INVALID_DRIVE;
      }
   #elif defined (DOS32)

      if ((txwa->dpmi1 != NULL) && (txwa->dpmi2 != NULL))
      {
         memset( txwa->dpmi1, 0, 512);
         memset( txwa->dpmi2, 0, 512);
         dRoot = (char *)            txwa->dpmi1; // DPMI compatible spec
         sizex = (DOS_DISKSIZE_EX *) txwa->dpmi2; // DPMI compatible data

         sprintf(dRoot, "%c:\\", drive[0]);     // path of drive root

         sizex->size    = sizeof(DOS_DISKSIZE_EX);
         sizex->version = 0;

         memset( &txdx_rmi, 0, sizeof(txdx_rmi));
         txdx_rmi.eax = TXDX_DOS_FREESPACE_EX;
         txdx_rmi.ecx = sizeof(DOS_DISKSIZE_EX);
         txdx_rmi.ds  = txDpmiSegment(dRoot);   // in  spec    (ds:dx, dx=0)
         txdx_rmi.es  = txDpmiSegment(sizex);   // return data (es:di, di=0)

         TRHEXS( 500,  &txdx_rmi,  sizeof(txdx_rmi), "txdx_rmi");

         memset( &regs,  0, sizeof(regs));
         memset( &sreg,  0, sizeof(sreg));
         regs.w.ax    = TXDX_DPMI_RMINT;        // simulate realmode INT
         regs.h.bl    = TXDX_DOS;               // DOS interrupt 21
         regs.x.edi   = FP_OFF( &txdx_rmi);
         sreg.es      = FP_SEG( &txdx_rmi);

         txDpmiCall( &regs, &sreg);

         if ((regs.x.cflag == 0) && (TXWORD.ax == 0x0300)) // FreeDOS result OK
         {
            *sfree   =          sizex->SectAvail;
            *stotal  =          sizex->SectTotal;
            *bpsect  = (USHORT) sizex->Bps;
         }
      }
      if (*sfree == 0)                          // not set, use classic call
      {
         TxxClearReg( regs);
         regs.h.dl = (BYTE) ((toupper(drive[0]) - 'A') +1);
         TxxDosInt21( regs, TXDX_DOS_FREESPACE);
         if (TXWORD.ax != 0xffff)               // drive exists
         {
            *sfree   = (ULN64)  TXWORD.ax * TXWORD.bx;
            *stotal  = (ULN64)  TXWORD.ax * TXWORD.dx;
            *bpsect  = (USHORT) TXWORD.cx;
         }
         else
         {
            rc = TX_INVALID_DRIVE;
         }
      }
   #elif defined (LINUX)
      strcpy( device, drive);
      TxLinuxRootDevice( device, device);       // translate root device
      if ((dh = open( device, O_RDONLY | O_LARGEFILE)) != -1)
      {
         ULONG      size = 0;                   // ioctl size return value

         //- Note: BLKSSZGET ioctl is NOT reliable on partition devices
         //-       As far as I can tell it will return 4096 on those ...
         //-       It does NOT report an error though.

         /*
         if (ioctl( dh, BLKSSZGET, &size) == 0)
         {
            *bpsect = (USHORT) size;
         }
         else
         {
            TRACES(( "BLKSSZGET ioctl handle %u: %s\n", dh, strerror(errno)));
         }
         */

         //- to be refined, check how to get volumes over 2TiB (64-bit)
         if (ioctl( dh, BLKGETSIZE, &size) == 0)
         {
            *stotal = size;                     // total size in sectors
            *sfree  = 0;                        // info not available ...
         }
         else
         {
            TRACES(( "BLKSSZGET ioctl handle %u: %s\n", dh, strerror(errno)));
         }
         close( dh);
      }
      else
      {
         rc = TX_INVALID_DRIVE;
      }
   #elif defined (DARWIN)                       // DARWIN MAC OS X (GCC)
      //- to be refined
   #else
      rc=DosQFSInfo((toupper(drive[0]) - 'A') +1, FSIL_ALLOC, &FSAllocate, sizeof(FSAllocate));
      if (rc == NO_ERROR)
      {
         *sfree   = (ULN64)  FSAllocate.cSectorUnit * FSAllocate.cUnitAvail;
         *stotal  = (ULN64)  FSAllocate.cSectorUnit * FSAllocate.cUnit;
         *bpsect  = (USHORT) FSAllocate.cbSector;
      }
   #endif
   TRACES(("Free: 0x%llx, Total: 0x%llx = %.1lf MiB, bps: %hu\n", *sfree, *stotal, TXSMIB( *stotal, *bpsect), *bpsect));
   RETURN (rc);
}                                               // end 'TxFsSpace'
/*---------------------------------------------------------------------------*/


#if defined (LINUX)
/*************************************************************************************************/
// Translate root-devicename to disk-partition device name using cached /etc/fstab info
/*************************************************************************************************/
BOOL TxLinuxRootDevice                          // RET   rootdevice translated
(
   char               *devname,                 // IN    Line with a device-name
   char               *root                     // OUT   Real root-device name
)
{
   BOOL                rc = FALSE;              // function return
   FILE               *fh;
   static TXTM         rootdev = "";            // cached root device string
   TXLN                line;
   TXTM                name;                    // device-name value
   TXTM                mount;                   // mount location

   ENTER();

   if (strncmp( devname, ROOT_DEVICE, strlen(ROOT_DEVICE)) == 0)
   {
      if (strlen(rootdev) != 0)
      {
         strcpy( root, rootdev);
         rc = TRUE;
      }
      else
      {
         if ((fh = fopen( "/etc/fstab", "rb")) != NULL)
         {
            while (!feof(fh) && !ferror(fh))
            {
               if (fgets( line, TXMAXLN, fh) != NULL)
               {
                  sscanf( line, "%s %s %*s", name, mount);
                  TRACES(( "mount '%s' for line: '%s'\n", mount, line));

                  if (strcmp( mount, "/") == 0) // mounted to root
                  {
                     strcpy( rootdev, name);
                     strcpy( root, rootdev);    // cache found value
                     rc = TRUE;
                  }
               }
            }
            fclose( fh);
         }
      }
   }
   else if (root != devname)                    // not the same buffer
   {
      strcpy( root, devname);
   }
   TRACES(("devname: '%s'  root: '%s'\n", devname, root));
   BRETURN (rc);
}                                               // end 'TxLinuxRootDevice'
/*-----------------------------------------------------------------------------------------------*/
#endif

