//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// SETBOOT.EXE replacement, based on DFSee functionality
//
// Author: J. van Wijk
//
// Avoid the bugs in IBM setboot with BootManager in other than first primary
//
// Fix for Bugtracker Project: eComStation 1.2 EN,  Bug ID: 883

#include <txlib.h>                              // TX library interface
                                                // also sets the WIN32 etc ...

#define SETB_VN "1.10"
#define SETB_VD "30-01-2017"                    // Copyright Fsys-software only, OEM ArcaOS

//efine SETB_VN "1.09"
//efine SETB_VD "10-10-2014"                    // Allow -disk:* and make that the default

//efine SETB_VN "1.08"
//efine SETB_VD "11-04-2013"                    // Minor update, allow numeric logfilename
//efine SETB_VD "10-04-2010"                    // Allow -disk:nr to force BMGR disk

//efine SETB_VN "1.07"
//efine SETB_VD "25-06-2008"                    // Allow colon on /IBD:C: syntax

//efine SETB_VN "1.06"
//efine SETB_VD "20-01-2008"                    // BldLevel info in OS/2 executable

#define SETB_V   SETB_VN " " SETB_VD
//efine SETB_V   "1.05 02-06-2007"              // Minor update to compile for darwin
//efine SETB_V   "1.04 21-04-2006"              // Rebuild with DFSee 8.xx source/TxWin
//efine SETB_V   "1.03 08-11-2005"              // Allow /query as alias for /q or /Q
//efine SETB_V   "1.02 25-08-2005"              // Fix quoting like -4:"Fedora Core 3"
//efine SETB_V   "1.01 24-08-2005"              // Fix Linux version, minor text update
//efine SETB_V   "1.00 21-08-2005"              // Initial version, internal test
#if   defined (WIN32)
   #define SETB_N "SETBOOT winNT"
#elif defined (DOS32)
   #define SETB_N "SETBOOT Dos32"
#elif defined (LINUX)
   #define SETB_N "SETBOOT Linux"
#elif defined (DARWIN)
   #define SETB_N "SETBOOT macOS"
#else
   #define SETB_N "SETBOOT  OS/2"
#endif
#define SETB_C "(c) 2005-2017: Fsys Software / Jan van Wijk"

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsupart.h>                           // FDISK partition functions
#include <dfsafdsk.h>                           // FDISK sector types
#include <dfsufdsk.h>                           // FDISK utility functions
#include <dfsetbt.h>                            // DFSee setboot function

#if defined (DUMP)
#define SETB_TRACE        "SETBTRACE"           // Trace startup values
#endif                                          // environment variable

#define SETB_O_QUIET       TXA_O_APP0           // Screen off, no output

#if defined (DEV32)
static char DfsBldLevel[] = "@#Fsys Software:" SETB_VN "#@##1##" __DATE__ "  " __TIME__
                            "    Merlin::EN::::@@Setboot; " SETB_C;
#endif

                                                // Global data-buffers
BYTE       *rbuf = NULL;                        // read buffer ptr, generic
BYTE       *brec = NULL;                        // read buffer ptr, bootrec

#if defined (DOS32)
BYTE       *xpar = NULL;                        // DPMI parameter  ptr,  512
BYTE       *xbuf = NULL;                        // DPMI big buffer ptr, 16KB
   short    xparSelector = 0;
   short    xbufSelector = 0;
   short    rbufSelector = 0;
   short    brecSelector = 0;
#endif

DFSINF      dfs_anchor =
{
   NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL,
   NULL, NULL, NULL, NULL, NULL, NULL, NULL,
   NULL, NULL, NULL, NULL, NULL, NULL, NULL,    // FS specific functions
   0,                                           // FS specific SLT size
   1,                                           // FS specific entry point
   0,                                           // FS specific truncation point
   0,                                           // FS last sector in-use
   0,                                           // FS maximum size for expand
   0,                                           // FS freespace, unallocated
   0,                                           // FS unallocated, Smart candidate
   DFSTAT_UNKNOWN,                              // FS dirty status unknown
   0,                                           // FS mode id (need some Init())
   NULL,                                        // FS specific cmdhelp text
   NULL,                                        // FS specific info
   NULL,                                        // FS specific sector types
   NULL,                                        // BASE sector type string
   NULL,                                        // ptr to boot record
   0,                                           // default snlist size
   NULL,                                        // default sn list
   NULL,                                        // default sn info
   0, 0,                                        // Browse, 'this'   sn+info
   "Unknown",                                   // list/browse description
   "-w",                                        // default value    list-show
   "-d",                                        // default contents list-show
   "",                                          // initial list FS mode
   "",                                          // FS specific command after select
   ".",                                         // default SaveTo path (current dir)
   {FALSE, FALSE, FALSE, "", "", NULL},         // persistent recovery parameters
   {0,NULL,0,0,0,0},                            // SPACE to be searched by 'find -o:S'
   NULL,                                        // sector area list
   NULL,                                        // info, store dependend
#if defined (DEV32)
   DfsBldLevel,
   {NULL,NULL},                                 // open & seek > 2GB files
   FALSE,                                       // use automatic sync of LVM
#endif
   "",                                          // no FS skip on correlate
   FALSE,                                       // no LVM info yet
   0,                                           // default boot-disk id
   0,                                           // Nr of possible WIN partitions
   TRUE,                                        // list active
   0,                                           // rexx value
   0,                                           // physical disk active BMGR
   15,                                          // current timeout value BMGR
   DFSI13X_NOTTHERE,                            // I13X none/optional/required
   FALSE,                                       // I13X capable MBR on disk 1
   1,                                           // partition-id for 'part 0'
   TRUE,                                        // batch-mode
   FALSE,                                       // dialogs will be used
   TXAO_NORMAL,                                 // normal verbosity
   TXAE_QUIT,                                   // quit on errors
   1,                                           // nr of sector read attempts
   DFS_CHSTYLE_IBM,                             // default CHS-dummy style IBM
   0,                                           // overall return-code
   TRUE,                                        // retc is explainable
   80,                                          // visible screen area characters
   22,                                          // visible screen area lines
   FALSE,                                       // OS/2 filesystems likely present
   TRUE,                                        // Ignore lock errors
   TRUE,                                        // auto quit on fdisk/setboot
   TRUE,                                        // reg confirm required
   FALSE,                                       // default store mode R/O
   TRUE,                                        // SLT build can show progress
   TRUE,                                        // running in classic mode
   FALSE,                                       // show every warning and more
   0,                                           // timestamp next status update
   0,                                           // secs between STDOUT progress
   FALSE,                                       // force (just) progress-bar to STDOUT
   FALSE,                                       // use Hex sector-nr fields
   FALSE,                                       // use Hex CHS-value fields
#if defined (OEM_BRANDED)
   FALSE,                                       // geoCalc default OFF (Vista)
   FALSE,                                       // gptAuto default OFF
#else
   TRUE,                                        // geoCalc default ON
   TRUE,                                        // gptAuto default ON
#endif
   FALSE,                                       // volFloppy default OFF
   FALSE,                                       // logAuto default OFF
   FALSE,                                       // logDate default OFF
   TRUE,                                        // var/expr ${ resolve on cmds
   FALSE,                                       // show part-type 0x00 as partitions
   0,                                           // scroll-buffer window
   0,                                           // command handling window
   0,                                           // commandline window, history
   TRUE,                                        // prompt after every manual command
   FALSE,                                       // BASIC menu/dialog active (not EXPERT)
   FALSE,                                       // no automatic menu at start
   TRUE,                                        // automatic pulldown drop
   0,                                           // default drop-down menu
   0,                                           // worklevel
   0,                                           // alignment (automatic Cyl/Mib)
   SECTORSIZE /2,                               // default (hex) display size
   0,                                           // length last Find argument
   TXMAXLN,                                     // hard-coded maximum
   NULL,                                        // sel-list, 1 volume
   NULL,                                        // sel-list, 1 removable
   NULL,                                        // sel-list, 1 disk
   NULL,                                        // sel-list, 1 disk or All
   NULL,                                        // sel-list, mounted disks
   NULL,                                        // sel-list, 1 partition
   NULL,                                        // sel-list, logicals enabled only
   NULL,                                        // sel-list, 1 part or All
   NULL,                                        // sel-list, filtered freespace
   0,                                           // #enabled  filtered freespace
   NULL,                                        // sel-list, filtered pri-waste
   0,                                           // #enabled  filtered pri-waste
   NULL,                                        // sel-list, GPT + Primary, no LOG-only
   0,                                           // #enabled  GPr
   NULL,                                        // sel-list, LOG, no PRI-only, no GPT
   0,                                           // #enabled  Log
   NULL,                                        // sel-list, All freespace
   NULL,                                        // sel-list, bootables
   NULL,                                        // sel-list, Color schemes
   #if defined (DOS32)
      FALSE,                                    // Windows-9x DosBox detected
   #endif
   "#",                                         // command separator string
   NULL                                         // DFSee executable directory
};

DFSINF       *dfsa  = &dfs_anchor;              // DFS anchor block

char *switchhelp[] =
{
   "  switches          (switches are case-insensitive)",
   "",
   "        /?          = help on SETBOOT commandline switches (this text)",
   "        /c          = confirm actions before writing to the disk",
   "        /d:d[:]     = set next-boot to system with driveletter 'd:'",
   "        /disk:nr    = Operate on Bootmanager on specified disk",
   "                      . = current, * = ALL, bmgr = BMGR-disk\n",
#if   defined (DEV32)
   "        /f          = flush disk buffers, inhibits PM popup message!",
#elif defined (WIN32)
#else
   "        /f          = force disk-buffer flush on rebooting the system",
#endif
   "        /list       = list Bootmanager names available for booting",
   "        /l:logfile  = log all screen output to a file: 'logfile.log'",
   "        /quiet      = quiet option, no output to the screen at all",
   "",
   NULL
};


// Set extra long names usable as switch or command option
static void setbSetLongSwitchNames
(
   void
);

int main (int argc, char *argv[]);

/*****************************************************************************/
/* Main function of the program, handle commandline-arguments                */
/*****************************************************************************/
int main (int argc, char *argv[])
{
   ULONG               rc = NO_ERROR;           // function return
   TXLN                stringBuffer;

   TxINITmain( SETB_TRACE, "SETB",
               TxaExeSwitch(SETB_O_QUIET),
               TRUE,                            // allow switches anywhwere
               setbSetLongSwitchNames);

   if (TxaExeSwitch('L'))                       // start logfile now ?
   {
      TxAppendToLogFile( TxaExeSwAsString( 'L', TXMAXLN, stringBuffer), TRUE);
   }
   else if (TxaExeSwitch('l'))
   {
      TxAppendToLogFile( TxaExeSwAsString( 'l', TXMAXLN, stringBuffer), TRUE);
   }
   if (TxaExeSwitch('?'))                       // switch help requested
   {
      TxPrint( "\n%s %s %s\n", SETB_N, SETB_V, SETB_C);
      TxPrint( "Interface to IBM BootManager, replaces IBM version (avoids NTLDR corruption)\n"
               "OEM version licensed to Arca Noae LLC (ArcaOS) and Serenity Systems Int (eCS).\n");
      TxPrint( "\nUsage: %s ", TxaExeArgv(0));
      TxShowTxt( switchhelp);                   // Module specific switches
      rc = setbCommand();                       // more usage from SETBOOT
   }
   else
   {
      dfstInitStoreInfo();
      if (TxaExeSwitch('R') || TxaExeSwitch('r'))
      {
         dfsa->ReadOnly = TRUE;                 // set stores to R/O by default
      }
      if (TxaExeSwitch('C') || TxaExeSwitch('c'))
      {
         dfsa->batch = FALSE;                   // use confirmation, no batch
      }
      if (_MAX_PATH < dfsa->maxPath)
      {
         dfsa->maxPath = _MAX_PATH;             // use smaller platform limit
      }

      TRACES(("Allocating generic read and parameter buffers ...\n"));

      #if defined (DOS32)
         xpar = txDpmiAlloc(     1, SECTORSIZE, &xparSelector);
         xbuf = txDpmiAlloc( XBUFS, SECTORSIZE, &xbufSelector);
         rbuf = txDpmiAlloc( RBUFS, SECTORSIZE, &rbufSelector);
         brec = txDpmiAlloc( BBUFS, SECTORSIZE, &brecSelector);
      #else
         rbuf = TxAlloc( RBUFS, SECTORSIZE);
         brec = TxAlloc( BBUFS, SECTORSIZE);
      #endif
      TRACES(("Allocated % 3hu sectors readbuffer at: %8.8lx\n", RBUFS, rbuf));
      TRACES(("Allocated % 3hu sectors bootbuffer at: %8.8lx\n", BBUFS, brec));

      if (    (rbuf != NULL) && (brec != NULL)
         #if defined (DOS32)
           && (xpar != NULL) && (xbuf != NULL)
         #endif
         )
      {
         #if defined (UNIX)                     // no 8-bit ASCII allowed
            TxSetAscii7Mode( TRUE);             // on standard terminals
         #endif

         dfsInitMediaMap( DFS_MAX_DISK);        // Init Media, read diskinfo
         if ((dfsa->bmgrDisk != 0) ||           // disks and BMGR are there
             TxaOption('b')  || TxaOption('B')) // or just want to boot ...
         {
            rc = setbCommand();                 // actual SETBOOT functionality
            if (rc != SETB_NO_DOS_SYS)
            {
               dfsExplainRC( rc);
            }
         }
         else                                   // IBM SETBOOT compatible ...
         {
            rc = SETB_NO_BMGR_FOUND;
            TxPrint( "Boot Manager is not installed.\n");
         }

         DFSFNCALL(dfsa->FsClose,0,0,NULL,NULL);  // FS specific handler, close
         dfstInitStoreInfo();                     // close all open stores
         dfsmDetachMedium( DFSM_UNMOUNT_ALL, FALSE); // quietly unmount
      }
      else
      {
         TxPrint( "\nError allocating minimum buffer memory\n");
         rc = DFS_ALLOC_ERROR;
      }
      dfsFreeDiskInfo();                        // free disk-partition info

      #if defined (DOS32)
         txDpmiFree(xparSelector);
         txDpmiFree(xbufSelector);
         txDpmiFree(rbufSelector);
         txDpmiFree(brecSelector);
      #else
         TxFreeMem( brec);
         TxFreeMem( rbuf);
      #endif
   }
   TxEXITmain(rc);                              // TX Exit code, incl tracing
}                                               // end 'main'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set extra long names usable as switch or command option
/*****************************************************************************/
static void setbSetLongSwitchNames
(
   void
)
{
   TxaOptionLongName( SETB_O_QUIET, "quiet");   // Screen off, no output
}                                               // end 'dfsSetLongSwitchNames'
/*---------------------------------------------------------------------------*/

