//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// DFSee major functions: wiping
//
// Author: J. van Wijk
//
// JvW  27-07-2006   Initial version, split off from DFSMAJOR.C
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfshpfs.h>                            // HPFS structure defs
#include <dfstore.h>                            // Store and sector I/O
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfs.h>                                // DFS  navigation and defs
#include <dfsver.h>                             // DFS  version and naming
#include <dfswipe.h>                            // wipe functions
#include <dfsutil.h>                            // DFS  utility functions
#include <dfsupart.h>                           // PART utility functions
#include <dfsufdsk.h>                           // FDSK utility functions


/*****************************************************************************/
// High level WIPE command logic
/*****************************************************************************/
ULONG dfsWipeCommand
(
   int                 cc,                      // IN    number of parameters
   char               *c1,                      // IN    parameter string 1
   char               *c2,                      // IN    parameter string 2
   char               *c3                       // IN    parameter string 3
)
{
   ULONG               rc = NO_ERROR;           // function return
   SEC512              ms;                      // mixed string buffer
   USHORT              ml  = 0;                 // mixed string 1 length
   ULN64               sn  = 0;                 // current sector number
   ULN64               sectors = 0;
   TXLN                s0;                      // temporary string space
   BOOL                ok = TRUE;
   TXTT                sizetext;                // text buffer
   BOOL                wipefree = TxaOption('f'); // wipe freespace only
   DFSPARTINFO        *p;

   ENTER();

   rc = dfsSetObjectCurrent( "DPVI", TRUE);     // open object when specified
   if ((rc == NO_ERROR) || (rc == DFS_BAD_STRUCTURE))
   {
      ULN64         maxSectors = dfstGetLogicalSize( DFSTORE);

      ok &= TxaOptMutEx((cc > 2), "F",  "if 'First' is specified!", NULL);
      ok &= TxaOptMutEx((cc > 3), "sS", "if 'size'  is specified!", NULL);
      if (ok)
      {
         ok &= dfsOptionSizeSectors( 'F', DFSTORE, 'm', 0,      &sn); // from
         ok &= dfsOptionSizeSectors( 's', DFSTORE, 'm', 0, &sectors); // size

         if (cc > 2)                            // regular parameters
         {
            sn = dfsGetSymbolicSN( c2, 0);
         }
         if (cc > 3)
         {
            sectors = dfsGetSymbolicSize( c3, sectors, DFSTORE, sn);
         }
         else if (sectors == 0)                 // default to FROM to end of object
         {
            sectors = dfsGetSymbolicSize( "$", sectors, DFSTORE, sn);
         }
         if ((sn + sectors) > maxSectors)
         {
            sectors = maxSectors - sn;          // limit to object size
         }

         p = SINF->p;
         if ((p != NULL) && TxaOption('L'))     // partition, exclude-LVM option
         {
            if (sectors > p->lvmReserved)       // valid reserved area
            {
               sectors -= p->lvmReserved;       // exclude signature-area
               TRACES(( "sectors -L: 0x%llx\n", sectors));
            }
         }
      }
      if (TxaOption('?'))
      {
         TxPrint(
           "\nWipe contents of selected item with a pattern (clear)\n"
           "\n Usage:  wipe  z | r | pattern    [. | First [size]] [options]\n"
           "\n    or:  wipe  -h:'xx' | -a:'str'  -F:n   -s:s\n"
           "\n    'z' is zeroes, 'r' is random pattern and '.' is current sector"
           "\n    Firstsector default is 0, size default is ALL sectors!\n\n");
         TxPrint(" Options:\n"
           "\n    -a:'asc'    ASCII wipe-pattern value, string"
           "\n    -b:sectors  Use a user-defined buffer size, in SECTORS"
           "\n    -f          Wipe ONLY freespace within a filesystem"
           "\n    -F:n        First sector as mcs-number, default 0"
           "\n    -h:'hex'    HEX wipe-pattern value, sequence of hex pairs"
           "\n    -m:'mix'    Mixed ASCII/HEX/UNICODE wipe-pattern"
           "\n    -O:q or -O- Minimal progress reporting (no dots),"
           "\n    -P          Prompt, allowing editing of the specified value"
           "\n                with one of the -a, -h, -u or -m options"
           "\n    -s:n        Size to be wiped, as mcs-number, default ALL"
           "\n    -t          Test only, no real wipe will be performed"
           "\n    -u:'uni'    UNICODE wipe-pattern value, string"
           "\n    -V          Verify each sector wiped by reading it back,"
           "\n                this can slow the process down considerably!");
         ok = FALSE;
      }
      else if (dfstStoreType( DFSTORE) != DFST_UNUSED)
      {
         if      (!strcasecmp(c1, "z") || TxaOption('z')) // zero pattern
         {
            ml = 0;
         }
         else if (!strcasecmp(c1, "r") || TxaOption('r')) // random pattern
         {
            int i;

            srand((unsigned int) clock());
            for (i = 0; i < SECTORSIZE; i++)
            {
               ms[i] = (BYTE) (rand() % 0xff);
            }
            ml = SECTORSIZE;
         }
         else                                   // get custom defined pattern
         {
            ml = dfsGetMixedString( c1, "wipe-pattern", FALSE, FALSE,
                                        (char *) &ms[0]);
         }
         if (ml == 0)
         {
            ml = 16;
            memset( ms, 0, ml);                 // default pattern, zeroes
         }
         cc = max(2, cc);                       // fake mandatory param
      }
      else                                      // no object open ...
      {
         TxPrint( "\nNo volume, partition or disk opened in default store (%hu), use the VOL, PART\n"
                  "or DISK cmd to select one before using the 'wipe' command.!\n", DFSTORE);
         rc = DFS_CMD_FAILED;
         ok = FALSE;
      }
      if (ok)
      {
         strcpy( sizetext, "");
         dfstrSz64( sizetext, "", sectors, "");
         if (sn == 0)
         {
            strcpy(  s0, "start of selected area");
         }
         else
         {
            strcpy( s0, "");
            dfstrSz64( s0, "", sn, " from the start");
         }
         if (wipefree)
         {
            if (dfsa->FsLsnAllocated)           // alloc info available ?
            {
               if (DFSTORE_WRITE_ALLOWED)
               {
                  TxNamedMessage( !dfsa->batch, 5019, " WARNING: about to WIPE sectors ",
                     "Wiping all FREE sectors should only be done when the filesystem is not in use "
                     "and CLEAN (chkdsk). It will also make UNDELETE of files impossible!");

                  TxPrint( "\nwipefree / 'wipe -f' will use a pattern with"
                           " length %u:\n", ml);
                  TxDisplHexDump( ms, ml);

                  if ((dfsa->batch) || (TxConfirm( 5013,
                     "%sWipe FREE sectors on store:\n%s\n%s\n\nWipe size is  : %s,\nStarting at: %s ? [Y/N] : ",
                      TxaOption('t') ? "TEST " : "", dfstStoreDesc1( DFSTORE), dfstStoreDesc2( DFSTORE), sizetext, s0)))
                  {
                     rc = dfsWipeFree( sn, sectors, (dfsa->verbosity) ? DFSP_BARS : DFSP_NONE, ms, ml);
                  }
                  else
                  {
                     rc = DFS_NO_CHANGE;
                  }
               }
               else
               {
                  rc = DFS_READ_ONLY;
               }
            }
            else
            {
               TxNamedMessage( !dfsa->batch, 0, " INFO: Not implemented ",
                          "Freespace wipe not supported for "
                          "filesystem/mode %s%s%s\n", CBM, SINF->afsys, CNN);
               rc = DFS_CMD_FAILED;
            }
         }
         else                                   // regular area wipe
         {
            TXLN       s1 = {0};

            if (DFSTORE_WRITE_ALLOWED)
            {
               if (!strcasecmp(c1, "r") || TxaOption('r'))
               {
                  TxPrint( "\nWipe uses a RANDOM pattern, repeated "
                           "in every sector, pattern start:\n");
                  TxDisplHexDump( ms, 32);
               }
               else
               {
                  TxPrint( "\nWipe uses the repeated pattern with"
                           " length %u:\n", ml);
                  TxDisplHexDump( ms, ml);
               }

               //- extra warning if partial wipe (not whole object), starting at sector 0
               if ((sn == 0) && (sectors < maxSectors)) // extra WARNING text, for MBR/Bootrec wipe
               {
                  if (SINF->p != NULL)          // it is a partition
                  {
                     strcpy( s1, "NOTE: Wipe starting at 0 includes CLEARING the bootsector,\n"
                                 "effectively deleting most filesystems present, if any!\n\n");
                  }
                  else
                  {
                     strcpy( s1, "NOTE: Wipe starting at 0 includes CLEARING the partition-tables,\n"
                                 "effectively removing ALL the existing partitions, if any!\n\n");
                  }
               }
               if ((dfsa->batch) || (TxConfirm( 5011,
                  "%sWipe sectors on store:\n%s\n%s\n\n%sWipe size  : %s,\nStarting at: %s ? [Y/N] : ",
                   TxaOption('t') ? "TEST " : "", dfstStoreDesc1( DFSTORE), dfstStoreDesc2( DFSTORE), s1, sizetext, s0)))
               {
                  ULONG  progressType = ((dfsa->verbosity != TXAO_QUIET) &&
                                         (sectors >= dfstGeoSectors(DFSTORE))) ? DFSP_BARS : DFSP_NONE;

                  rc = dfsWipeArea( sn, sectors, progressType, "Wiped (area)", TxaOption('V'), ms, ml);
                  if (progressType == DFSP_NONE)
                  {
                     TxPrint( "\nWipe completed %s\n", (rc == NO_ERROR) ? "successfully." : "with errors!");
                  }
               }
               else
               {
                  rc = DFS_NO_CHANGE;
               }
            }
            else
            {
               rc = DFS_READ_ONLY;
            }
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsWipeCommand'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Wipe sectors with pattern using (delete) sectornumbers in the SN-table
/*****************************************************************************/
ULONG dfsDelWipeList
(
   ULN64              *list,                    // IN    count + list of sn's
   BYTE               *pattern,                 // IN    pattern buffer
   USHORT              patsize,                 // IN    pattern size
   char               *select                   // IN    select wildcard or !
)
{
   ULONG               rc  = NO_ERROR;
   ULN64               sn;

   ENTER();
   if (*select == '!')                          // wipe any type of sector
   {
      TxPrint( "Wiping using list : ");
   }
   for (sn = 1; (sn <= *list) && (!TxAbort() && (rc == NO_ERROR)); sn++)
   {
      rc = dfsDelWipeSingle( list[sn], pattern, patsize, select);
   }
   if (*select == '!')                          // wipe any type of sector
   {
      TxPrint( "\nFinished wiping   : %u sectors\n", *list);
   }
   RETURN(rc);
}                                               // end 'dfsDelWipeList'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Wipe (file/fnode) sector(s)s with specified pattern
/*****************************************************************************/
ULONG dfsDelWipeSingle
(
   ULN64               fn,                      // IN    LSN
   BYTE               *pattern,                 // IN    pattern buffer
   USHORT              patsize,                 // IN    pattern size
   char               *accept                   // IN    select wildcard or !
)
{
   ULONG               rc = NO_ERROR;
   TXLN                fullpath;

   ENTER();
   if (*accept == '!')                          // wipe any type of sector
   {
      TxPrint( ".");                            // show '.' progress here
   }
   else                                         // validate and display
   {
      sprintf( fullpath, "Wipe sec 0x%12.12llX : ",     fn);
      rc = DFSFNCALL(dfsa->FsFileInformation, fn, 0, accept, fullpath);
   }
   switch (rc)
   {
      case NO_ERROR:                            // matched accept criteria
         //- to be refined
         //- Use DFSFNCALL to FS-specific function first, this might
         //- wipe more than one sector or all allocated-sectors
         //- generic wipe one sector when PENDING
         rc = dfsWipeArea( fn, 1, DFSP_NONE, NULL, FALSE, pattern, patsize);
         break;

      case DFS_PENDING:                         // skipped, wrong sector type
      case DFS_ST_MISMATCH:                     // skipped on accept criteria
         rc = NO_ERROR;
         break;

      default:                                  // other error, abort
         TxPrint("Abort on 0x%12.12llX : error %u\n", fn, rc);
         break;
   }
   RETURN(rc);
}                                               // end 'dfsDelWipeSingle'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DFS write sectors to volume/partition using a fixed pattern (wipe)
/*****************************************************************************/
ULONG dfsWipeArea
(
   ULN64               fn,                      // IN    start LSN
   ULN64               size,                    // IN    nr of sectors to write
   ULONG               progress,                // IN    display progress
   char               *reason,                  // IN    reason text for progress
   BOOL                verify,                  // IN    read-verify 1st sector
   BYTE               *pattern,                 // IN    pattern buffer
   USHORT              patsize                  // IN    pattern size
)
{
   ULONG               rc  = NO_ERROR;
   BYTE               *data;                    // sector buffer
   ULN64               i;                       // number of sectors
   ULN64               sn;                      // Data LSN
   BYTE               *p;
   USHORT              bps = dfsGetSectorSize();
   ULONG               spb;                     // sectors per buffer
   BYTE               *vbuf = NULL;
   TXLN                text;

   ENTER();
   TRARGS(( "start:0x%llx  size:0x%llx  progress:%u\n", fn, size, progress));

   #if defined (DOS32)                          // preallocated DPMI compatible
      spb  = dfsGetBufferSize( DFSOPTIMALBUF, (RBUFBYTES / bps) / 2);
      data = rbuf;                              // allocated buffer (no free!)
      if (verify)
      {
         vbuf = rbuf + (spb * bps);             // use 2nd half for the verify
      }
   #else
      spb  = dfsGetBufferSize( DFSOPTIMALBUF, DFSMAXBUFSIZE);
      if (verify)
      {
         vbuf = TxAlloc( spb, bps);
      }
      data = TxAlloc( spb, bps);                // default size is tracksize
   #endif
   TRACES(("Wipe  buffer size : %u sectors  at: %8.8x\n", spb, data));
   if (data != NULL)
   {
      ULONG            sects  = spb;            // size to wipe in one Write
      ULN64            lastw  = fn;
      BOOL             test   = TxaOption('t') || TxaOption(TXA_O_TEST);
      TXTIMER          stime  = TxTmrGetNanoSecFromStart();

      if (progress & DFSP_BAR)
      {
         TxPrint("Wipe method used  : %s whole area%s, buffer% 6u sect",
            (test)   ?  "TEST only" : "REAL wipe",
            (verify) ? " + VERIFY"  : "", spb);
         dfsSize( " = ", spb, "\n");
      }
      for (i = 0, p = (BYTE *) data; i < (bps * spb); i++)
      {
         *p++ = pattern[i % patsize];           // initialize Wipe sectors
      }
      if (progress)
      {
         dfsProgressInit( fn, size, 0, "Sector:", reason, progress, 0);
         dfsProgressShow( 0, 0, 0, NULL);       // 0 of 0 done (if first is bad :-)
      }

      for ( i  = 0,   sn = fn;
           (i < size) && (rc == NO_ERROR) && !TxAbort();
            i += spb, sn += spb)
      {
         if ((i + spb) >= size)                 // incomplete block
         {
            sects = size - i;
         }
         if (test == FALSE)
         {
            rc = dfsWrite( sn, sects, data);
            if (rc == NO_ERROR)
            {
               if (vbuf != NULL)                // perform optional verify
               {
                  memset( vbuf, 0xfe, (sects * bps));
                  rc = dfsRead( sn, sects, vbuf);
                  if (rc == NO_ERROR)
                  {
                     if (memcmp( vbuf, data, sects * bps) == 0)
                     {
                        TRACES(("Verified OK\n"));
                     }
                     else
                     {
                        rc = DFS_ST_MISMATCH;   // verify failed
                     }
                  }
               }
            }
            else
            {
               sprintf( text, "Write error on %5u sector(s) at sector 0x%16.16llx.", sects, sn);
               switch (dfsa->eStrategy)
               {
                  default:
                  case TXAE_QUIT:               // no action, keep bad rc
                     TxPrint( "%s  Aborting ...\n", text);
                     break;

                  case TXAE_CONFIRM:
                     if (dfsa->batch)           // CONFIRM not possible, but do NOT ignore errors!
                     {
                        TxPrint( "%s  Aborting (batch) ...\n", text);
                     }
                     else
                     {
                        dfsProgressSuspend();   // allow regular screen output
                        if (TxConfirm( 5322, "%s\n\nContinue the wipe ? [Y/N]: ", text))
                        {
                           if (TxConfirm( 5333, "Ignore errors from now on ? [Y/N]: "))
                           {
                              dfsa->eStrategy = TXAE_IGNORE;
                              sprintf( text, "Ignoring further errors, continue wiping");
                           }
                           else
                           {
                              sprintf( text, "Continue wiping");
                           }
                           dfsProgressResume( text);
                           rc = NO_ERROR;
                        }
                     }
                     break;

                  case TXAE_IGNORE:             // ignore the error
                     TxPrint( "%s  Ignore, continue ...\n", text);
                     rc = NO_ERROR;
                     break;
               }
            }
         }
         if (progress)
         {
            dfsProgressShow( sn + sects, 0, 0, NULL);
         }
         lastw = sn + sects -1;                 // last wiped
      }
      dfsProgressTerm();
      if (rc == DFS_PSN_LIMIT)                  // reached end of disk/vol/part
      {
         if (progress & DFSP_BAR)
         {
            TxPrint("End of disk/volume\n");
         }
         rc = NO_ERROR;
      }
      if (progress)
      {
         if (progress & DFSP_BAR)
         {
            TxPrint("Last wiped sector : 0x%llX", lastw);
            dfsSz64(" size ", lastw - fn +1, "\n");
            dfsDisplayThroughput( stime, lastw -fn +1, bps);
         }
      }
      if (data != rbuf)
      {
         TxFreeMem( data);
         TxFreeMem( vbuf);
      }
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN(rc);
}                                               // end 'dfsWipeArea'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DFS wipe area for FREESPACE only using a fixed pattern (wipefree)
/*****************************************************************************/
ULONG dfsWipeFree
(
   ULN64               fn,                      // IN    start LSN
   ULN64               size,                    // IN    nr of sectors to write
   ULONG               progress,                // IN    display progress
   BYTE               *pattern,                 // IN    pattern buffer
   USHORT              patsize                  // IN    pattern size
)
{
   ULONG               rc  = NO_ERROR;
   BYTE               *data;                    // sector buffer
   ULN64               i, j;                    // number of sectors
   ULN64               sn;                      // Data LSN
   ULN64               wf;                      // Wipe LSN
   ULN64               wfstart = DFS_MAX_PSN;
   ULN64               wfsize  = 0;             // wipefree start/size
   BYTE               *p;
   USHORT              bps = dfsGetSectorSize();
   ULONG               spb;                     // sectors per buffer
   TXLN                text;

   ENTER();
   TRARGS(( "start:0x%llx  size:0x%llx  progress:%u\n", fn, size, progress));

   spb  = dfsGetBufferSize( DFSOPTIMALBUF, DFSMAXBUFSIZE);
   #if defined (DOS32)                          // preallocated DPMI compatible
      data = rbuf;                              // allocated buffer (no free!)
   #else
      data = TxAlloc( bps, spb);
   #endif
   TRACES(("Wipefree buf size : %u sectors  at: %8.8x\n", spb, data));
   if (data != NULL)
   {
      ULONG            sects  = spb;            // size to wipe in one Write
      ULN64            lastw  = fn;             // last wiped sector
      ULN64            zeroed = 0;              // nr of wiped sectors
      ULN64            ssize  = size;           // net size in sectors
      ULONG            csize  = (ULONG) dfsGetClusterSize();
      ULN64            cl1st  = dfsSn2Cl( fn);
      ULN64            sn1st  = dfsCl2Sn( cl1st);
      BOOL             test   = TxaOption('t') || TxaOption(TXA_O_TEST);
      TXTIMER          stime  = TxTmrGetNanoSecFromStart();

      if (sn1st > fn)                           // adjusted first sector ?
      {
         ssize = size + fn - sn1st;             // adjust size too
      }

      TRACES(("fn  : 0x%llx  ==>  sn1st: 0x%llx\n", fn,   sn1st));
      TRACES(("size: 0x%llx  ==>  ssize: 0x%llx\n", size, ssize));

      TxPrint("Wipe method used  : %s freespace, buffer% 6u sect", (test) ? "TEST only" : "REAL wipe", spb);
      dfsSize( " = ", spb, "\n");

      for (i = 0, p = (BYTE *) data; i < (bps * spb); i++)
      {
         *p++ = pattern[i % patsize];           // initialize Wipe sectors
      }
      if (progress)
      {
         dfsProgressInit( sn1st, ssize, 0, "Sector:", "wiped (Free)", DFSP_BARS, 0);
         dfsProgressShow( 0, 0, 0, NULL);  // 0 of 0 done (if first is bad :-)
      }

      for ( j  =  0,      sn  = sn1st;
           (j <= (ssize + csize)) && (rc == NO_ERROR) && !TxAbort();
            j +=  csize,  sn += csize)
      {
         TRACES(("j:0x%llx sn:0x%llx ssize:0x%llx wfstart:0x%llx wfsize:0x%llx\n",
                  j, sn, ssize, wfstart, wfsize));
         if ((j >= ssize) || (DFSFNCALL( dfsa->FsLsnAllocated, sn, 0, NULL, NULL)))
         {
            if ((wfstart != DFS_MAX_PSN) &&     // searching for last free
                (wfsize  == 0))
            {
               if (j >= ssize)                  // just beyond given size
               {
                  sn = sn1st + ssize;           // use exact end boundary
               }
               wfsize = sn - wfstart;
            }
         }
         else                                   // not allocated
         {
            if (wfstart == DFS_MAX_PSN)         // searching for first free
            {
               wfstart = sn;
               if (progress)                    // progress while skipping ...
               {
                  dfsProgressShow( sn +1, 0, 0, NULL);
               }
            }
         }
         if ((wfstart != DFS_MAX_PSN) &&        // a free area has been found
             (wfsize  != 0))
         {
            for ( i  = 0,   wf = wfstart,  sects = spb;
                 (i < wfsize) && (rc == NO_ERROR) && !TxAbort();
                  i += spb, wf += spb)
            {
               if ((i + spb) >= wfsize)         // incomplete block
               {
                  sects = wfsize - i;
               }
               if (test == FALSE)               // if not testing ...
               {
                  rc = dfsWrite( wf, sects, data);
                  if (rc != NO_ERROR)
                  {
                     sprintf( text, "Write error on %5u sector(s) at sector 0x%16.16llx.", sects, wf);
                     switch (dfsa->eStrategy)
                     {
                        default:
                        case TXAE_QUIT:         // no action, keep bad rc
                           TxPrint( "%s  Aborting ...\n", text);
                           break;

                        case TXAE_CONFIRM:
                           if (dfsa->batch)     // CONFIRM not possible, but do NOT ignore errors!
                           {
                              TxPrint( "%s  Aborting (batch) ...\n", text);
                           }
                           else
                           {
                              dfsProgressSuspend(); // allow regular screen output
                              if (TxConfirm( 5322, "%s\n\nContinue the wipe ? [Y/N]: ", text))
                              {
                                 if (TxConfirm( 5333, "Ignore errors from now on ? [Y/N]: "))
                                 {
                                    dfsa->eStrategy = TXAE_IGNORE;
                                    sprintf( text, "Ignoring further errors, continue wiping");
                                 }
                                 else
                                 {
                                    sprintf( text, "Continue wiping");
                                 }
                                 dfsProgressResume( text);
                                 rc = NO_ERROR;
                              }
                           }
                           break;

                        case TXAE_IGNORE:       // ignore the error
                           TxPrint( "%s  Ignore, continue ...\n", text);
                           rc = NO_ERROR;
                           break;
                     }
                  }
               }
               if (progress)
               {
                  dfsProgressShow( wf + sects, 0, 0, NULL);
               }
               lastw = wf + sects -1;           // last wiped
            }
            zeroed += wfsize;
            wfstart = DFS_MAX_PSN;              // search next area ...
            wfsize  = 0;
         }
      }
      TRACES(("After loop; j:0x%llx sn:0x%llx ssize:0x%llx wfstart:0x%llx wfsize:0x%llx\n",
                           j, sn, ssize, wfstart, wfsize));

      dfsProgressTerm();
      if (rc == DFS_PSN_LIMIT)                  // reached end of disk/vol/part
      {
         if (progress)
         {
            TxPrint("End of disk/volume\n");
         }
         rc = NO_ERROR;
      }
      if ((progress) || (dfsa->verbosity != TXAO_QUIET))
      {
         dfsSz64( "Cleared freespace : ", zeroed, "  ");
         dfsSz64( "out of : ", lastw - fn +1, "\n");
         dfsDisplayThroughput( stime, lastw -fn +1, bps);
      }
      if (data != rbuf)
      {
         TxFreeMem( data);
      }
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN(rc);
}                                               // end 'dfsWipeFree'
/*---------------------------------------------------------------------------*/

