//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// DFSee SVISTA major functions: svgrab
//
// Author: J. van Wijk
//
// JvW  20-12-2004   Initial version, modelled after DFSMAJOR.C
//


#include <txlib.h>                              // ANSI controls and logging

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfs.h>                                // DFS  navigation and defs
#include <dfsver.h>                             // DFS  version and naming
#include <dfsvoem.h>                            // DFS  SVISTA major functions
#include <dfsutil.h>                            // DFS  utility functions
#include <dfsupart.h>                           // PART utility functions
#include <dfsafdsk.h>                           // FDSK command functions
#include <dfsufdsk.h>                           // FDSK utility functions
#include <dfsimage.h>                           // image create restore

#if defined (HAVE_CMD_GRAB)

#define DFSV_POST_FIXPBR       0x00000001       // fix hiddensector & PBR Geo
#define DFSV_POST_BOOTINI      0x00000002       // fix BOOT.INI index

typedef struct dfsv_grab_map
{
   USHORT              ppid;                    // PID on physical disk to grab
   USHORT              vpid;                    // PID on virtual disk
   BOOL                dummy;                   // 1 cylinder dummy, no contents
   ULONG               post;                    // post-processing flags
} DFSV_GRAB_MAP;                                // end of struct "dfsv_grab_map"

typedef struct dfsv_grabspec
{
   USHORT              parts;                   // total partitions in image
   USHORT              items;                   // real image items (non-dummy)
   USHORT              disknr;                  // source  disk number (1st)
   USHORT              virtnr;                  // virtual disk number
   DFSV_GRAB_MAP       map[DFS_MAX_PART];       // partition mapping info
} DFSV_GRABSPEC;                                // end of struct "dfsv_grabspec"

#define DFSGRAB_STANDARD      0x07              // no dummies, check order
#define DFSGRAB_DUMMIES       0x01              // insert dummy partitions
#define DFSGRAB_SAMEDISK      0x02              // don't allow multiple disks
#define DFSGRAB_ASCENDING     0x04              // require ascending order
#define DFSGRAB_ADVANCED      0x00              // most flexible
// DFSV grab partition(s) verify specification and create working structures
static ULONG dfsSvGrabVerify                    // RET   result code
(
   char               *plist,                   // IN    partitions to grab
   ULONG               features,                // IN    feature set
   BOOL                verbose,                 // IN    verbose progress
   DFSV_GRABSPEC     **spec                     // OUT   verified grab spec
);                                              //       TxFreeMem by caller!

// DFSV create virtual disk with partition-definitions for the image
static ULONG dfsSvGrabVirtual                   // RET   result code
(
   char               *fname,                   // IN    image filename
   BOOL                verbose,                 // IN    verbose progress
   BOOL                geo255h,                 // IN    force 255 head geo
   DFSV_GRABSPEC      *spec                     // INOUT verified grab spec
);


// DFSV write virtual partition-definitions plus physical contents to RAW image
static ULONG dfsSvGrabWrite                     // RET   result code
(
   char               *fname,                   // IN    image filename
   BOOL                verbose,                 // IN    verbose progress
   BOOL                testmode,                // IN    fake image-write
   DFSV_GRABSPEC      *spec                     // IN    verified grab spec
);

// DFSV post-process the created image to fix expected minor inconsistencies
static ULONG dfsSvGrabPostProcess               // RET   result code
(
   BOOL                verbose,                 // IN    verbose progress
   DFSV_GRABSPEC      *spec                     // IN    verified grab spec
);


static  char       *cmd_grab[] =
{
   "",
   "Grab one or more partitions from one disk to a (RAW) imagefile",
   "",
   " Usage: "
#if defined (OEMSV)
   "SV"
#else
   "IM"
#endif
   "GRAB  [filename]  -g:'PID1 PID2 .. PIDn'  [-D-] [-v-] [-m-] [-t]",
   "",
   "   filename  = fully qualified filename for the image to be created,",
   "               with a default file extension of '.hdd' (RAW IMAGE).",
   "",
   "   -g:'PIDs' : String with absolute partition numbers to be grabbed",
   "               PIDs must be in ascending order, from the same disk."
   "",
   "   -D-       : Do NOT insert dummy partitions for skipped partitions",
   "",
   "   -G-       : Do NOT force a 255 head, 63 sector geometry, keep original",
   "",
   "   -A        : Advanced, allow selecting partitions from multiple disks",
   "               and non-ascending partition order (implies -D-)",
   "",
   "   -v-       : No verbose progress reporting and no dfsee-command echo",
   "               (visible in logfile only, if global '-q' switch used)",
   "",
   "   -m-       : Suppress progress message lines indicating 'prepare' and",
   "               'finishing' stages (to STDOUT if global  '-G' switch used)",
   "",
   "   -t        : test modus, simulated writing of the imagefile only",
   "",
   NULL
};


/*****************************************************************************/
// Grab one or more partitions from a disk to a new RAW imagefile
/*****************************************************************************/
ULONG dfsvGrabPartitions
(
   void
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXLN                pstring;
   TXLN                fstring;
   TXLN                s1;
   BOOL                logging  = (TxQueryLogName() != NULL);
   BOOL                message  = !TxaOptUnSet('m');
   BOOL                verbose  = !TxaOptUnSet('v');
   BOOL                geo255h  = !TxaOptUnSet('G');
   BOOL                testonly =  TxaOption('t');
   ULONG               feature  = DFSGRAB_STANDARD;
   DFSV_GRABSPEC      *grabs    = NULL;

   ENTER();

   if (TxaOption('g'))
   {
      if (message)
      {
         dfsGuiStdMessage( "Analysing disks, prepare to create image");
      }
      if (TxaOption('A'))                       // advanced mode, implies -D-
      {
         feature = DFSGRAB_ADVANCED;            // no dummies, no checks
      }
      else
      {
         if (TxaOptUnSet('D'))                  // -D- specified
         {
            feature &= ~DFSGRAB_DUMMIES;        // reset grab-dummies
         }
      }
      strcpy( fstring, TxaArgValue(1));
      if (strlen(fstring) > 0)                  // we have an image to make
      {
         if (!logging)                          // force log with imagefile
         {                                      // when no regular logfile
            strcpy( pstring,   fstring);        // same path/name as image
            TxStripExtension(  pstring);
            TxAppendToLogFile( pstring, TRUE);  // append to new logfile
         }
         TxFnameExtension( fstring, "hdd");
      }
      TxaOptAsString( 'g', TXMAXLN, pstring);
      TRACES(( "fstring: '%s' pstring: '%s'\n", fstring, pstring));

      dfsReadDiskInfo( FDSK_QUIET);             // Read all partition info
      if ((rc = dfsSvGrabVerify( pstring, feature, verbose, &grabs)) == NO_ERROR)
      {
         if ((rc = dfsSvGrabVirtual( fstring, verbose, geo255h, grabs)) == NO_ERROR)
         {
            if (strlen(fstring))                // create real image
            {
               if ((rc = dfsSvGrabWrite( fstring, verbose, testonly, grabs)) == NO_ERROR)
               {
                  if (message)
                  {
                     dfsGuiStdMessage( "Partitions copied, finishing image");
                  }
                  if (!testonly)                // if image should be there ...
                  {
                     sprintf( s1, "attach -r:'%s'", fstring);
                     rc = dfsMultiCommand( s1, 0, verbose, FALSE, verbose);
                     if (rc == NO_ERROR)
                     {
                        //- remove virtual, so the mounted imagefile now gets
                        //- its disknumber, and all partitions on it get the
                        //- numbers as recorded in the map[index].vpid

                        sprintf( s1, "unmount -B %hu", grabs->virtnr);
                        dfsMultiCommand( s1, 0, verbose, FALSE, verbose);

                        dfsa->stdOutInterval = 0; // avoid further STDOUT progress

                        rc = dfsSvGrabPostProcess( verbose, grabs);
                     }
                  }
               }
            }
            dfsMultiCommand( "unmount -B -last", 0, verbose, FALSE, verbose);
         }
         TxFreeMem( grabs);                     // free GRAB structures
      }
      if (!logging)                             // not logging at entry ?
      {
         TxAppendToLogFile( "", TRUE);          // stop the log ...
      }
   }
   else
   {
      dfsGuiStdMessage( "List of partitions in -g: option is mandatory!");
      rc = DFS_VALUE_ERROR;
      dfsa->explain = FALSE;                    // avoid garbage explanation
   }
   if (rc == DFS_VALUE_ERROR)                   // some parameter problem
   {
      TxShowTxt( cmd_grab);                     // give usage
      TxPrint("\n Use the '-q -G' switches to suppress normal output, and enable\n"
                " error messages and a percentage progress on STDOUT, to drive\n"
                " a GUI frontend progress indicator.  Example invocation:\n\n"
                " %s -q -G "
                        #if defined (OEMSV)
                           "sv"
                        #else
                           "im"
                        #endif
                             "grab x:\\grabfile -g:'1 3' -D-\n", TxaExeArgv(0));
   }
   RETURN (rc);
}                                               // end 'dfsvGrabPartitions'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DFSV grab partition(s) verify specification and create working structures
/*****************************************************************************/
static ULONG dfsSvGrabVerify                    // RET   result code
(
   char               *plist,                   // IN    partitions to grab
   ULONG               fs,                      // IN    feature set
   BOOL                verbose,                 // IN    verbose progress
   DFSV_GRABSPEC     **spec                     // OUT   verified grab spec
)                                               //       TxFreeMem by caller!
{
   ULONG               rc = NO_ERROR;           // function return
   DFSV_GRABSPEC      *gs = NULL;
   char               *s  = plist;
   USHORT              part  = 0;               // partition ID
   USHORT              next  = 0;               // next valid PID
   int                 index = 0;
   DFSPARTINFO        *pi;
   DFSDISKINFO        *d;
   TXLN                error;

   ENTER();

   if ((gs = TxAlloc( 1, sizeof( DFSV_GRABSPEC))) != NULL)
   {
      while (*s == ' ') s++;                    // skip spaces
      do
      {
         TRACES(( "index:%hu part:%hu next:%hu  rest:'%s'\n", index, part, next, s));
         part = atoi( s);
         if ((pi = dfsGetPartInfo( part)) != NULL)
         {
            if (next == 0)                      // first one
            {
               gs->disknr = pi->disknr;         // copy disk number
               if ((d = dfsGetDiskInfo( gs->disknr)) != NULL)
               {
                  next = d->firstpart;          // start dummies from here
               }
               else                             // should not happen, but if so
               {                                // do not create any dummies
                  next = part -1;               // by closing the gap
               }
            }
            else if ((fs & DFSGRAB_SAMEDISK) &&
                     (gs->disknr != pi->disknr)) // on other disk!
            {
               sprintf( error, "Partition number %hu is not on the SAME disk!", part);
               dfsGuiStdMessage( error);
               rc = DFS_VALUE_ERROR;
               dfsa->explain = FALSE;           // avoid garbage explanation
            }
            else if ((fs & DFSGRAB_ASCENDING) &&
                     (pi->id < next))           // PID same or smaller
            {
               sprintf( error, "Partition number %hu is not in ascending order!", part);
               dfsGuiStdMessage( error);
               rc = DFS_VALUE_ERROR;
               dfsa->explain = FALSE;           // avoid garbage explanation
            }
            if (rc == NO_ERROR)                 // make dummies and real part
            {
               if (fs & DFSGRAB_DUMMIES)
               {
                  while (next < part)
                  {
                     gs->map[index  ].ppid  = next;
                     gs->map[index++].dummy = TRUE;
                     if (verbose)
                     {
                        TxPrint( "1cyl dummy for nr : %hu\n", next);
                     }
                     next++;
                  }
               }
               gs->items++;                     // on more 'real' item to do
               gs->map[index++].ppid  = part;
               next = part +1;                  // next valid one
               if (verbose)
               {
                  TxPrint( "Grab partition nr : %hu\n", part);
               }
            }
         }
         else                                   // invalid partition number
         {
            sprintf( error, "Partition number %hu does not exist!", part);
            dfsGuiStdMessage( error);
            rc = DFS_VALUE_ERROR;
            dfsa->explain = FALSE;              // avoid garbage explanation
         }
         while ((*s) && (isdigit(*s))) s++;     // skip this number
         while ( *s == ' ')            s++;     // skip spaces
         if ((*s) && (!isdigit(*s)))            // when followed by non-number:
         {
            sprintf( error, "Invalid character in partition number string: '%c'!", *s);
            dfsGuiStdMessage( error);
            rc = DFS_VALUE_ERROR;
            dfsa->explain = FALSE;              // avoid garbage explanation
         }
      } while ((*s) && (rc == NO_ERROR));

      if (rc == NO_ERROR)
      {
         gs->parts = index;                     // number of entries in map
      }
      else
      {
         TxFreeMem( gs);
      }
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   *spec = gs;
   RETURN (rc);
}                                               // end 'dfsSvGrabVerify'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DFSV create virtual disk with partition-definitions for the image
/*****************************************************************************/
static ULONG dfsSvGrabVirtual                   // RET   result code
(
   char               *fname,                   // IN    image filename
   BOOL                verbose,                 // IN    verbose progress
   BOOL                geo255h,                 // IN    force 255 head geo
   DFSV_GRABSPEC      *spec                     // INOUT verified grab spec
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXLN                s1;
   int                 i;
   DFSDISKINFO        *d;                       // diskinfo FIRST grab disk
   DFSPARTINFO        *p;
   DFSPARTINFO        *v;
   char               *basename;                // image filename, no path
   int                 actives = 0;             // nr of ACTIVE partitions

   ENTER();

   dfsSelectDisk( spec->disknr, FALSE, FALSE);  // quietly select real disk
   strcpy( s1, "attach -m");                    // create mem-disk of same size
   if (geo255h)
   {                                            // force 255 head geometry
      strcat( s1, " ? 255 63");                 // with same disk size
   }
   rc = dfsMultiCommand( s1, 0, verbose, FALSE, verbose);
   if (rc == NO_ERROR)
   {
      strcpy( fdsk->ShowCmd, "-");              // switch FDISK autoshow off
      spec->virtnr = SINF->disknr;              // record new disk number

      //- Note: -clean is not needed (Virtual is empty) so reduce the risk ...
      sprintf( s1, "newmbr -B -Q -disk:%hu -from:%hu", spec->virtnr, spec->disknr);
      if ((rc = dfsMultiCommand( s1, 0, verbose, FALSE, verbose)) == NO_ERROR)
      {
         for (i = 0; (i < spec->parts) && (rc == NO_ERROR); i++)
         {
            if ((p = dfsGetPartInfo( spec->map[i].ppid)) != NULL)
            {
               sprintf( s1, "cr -B -M -Q -I- -M -As:%hu", spec->map[i].ppid);
               if (spec->map[i].dummy)
               {
                  strcat( s1, " -s:1,c ;dummy");
               }
               else if (p->partent.Status & 0x80)
               {
                  strcat( s1, " -F");           // make grabbed one ACTIVE too
                  actives++;
               }
               if ((rc = dfsMultiCommand( s1, 0, verbose, FALSE, verbose)) == NO_ERROR)
               {
                  spec->map[i].vpid = dfsa->autoPid; // record new PID on virtual
                  if (verbose)
                  {
                     TxPrint( "Map Phys->Virtual : % 2hu -> % 2hu\n",
                               spec->map[i].ppid, spec->map[i].vpid);
                  }
                  dfsReadDiskInfo( FDSK_QUIET); // re-read disk info for new
               }
            }
         }
         if (actives == 0)                      // if none set ACTIVE
         {
            d = dfsGetDiskInfo( spec->virtnr);  // set first partition ACTIVE
            if (d && (d->primaries != 0))       // on the virtual disk
            {
               sprintf(  s1, "startable -B %hu", d->firstpart);
               rc = dfsMultiCommand( s1, 0, verbose, FALSE, FALSE); // echo, quiet
            }
         }
         d = dfsGetDiskInfo( spec->disknr);     // disk for FIRST partition
         if (d && (d->lvmPartitions != 0))      // LVM info on source disk
         {
            for (i = 0; (i < spec->parts) && (rc == NO_ERROR); i++)
            {
               if ((p = dfsGetPartInfo( spec->map[i].ppid)) != NULL)
               {
                  if ((fname != NULL) && (strlen(fname)))
                  {
                     basename = TxGetBaseName( fname);
                  }
                  else                          // no name specified, test
                  {
                     basename = "Virtual GRAB test"; // use static name
                  }
                  if (spec->map[i].dummy)
                  {
                     sprintf( s1, "lvm -B -Q %hu -D -s -n:'%s' "
                                  "-v:'Dummy:%s' -p:'%s' -l- -m- "
                                  "-lvmsnv:0x%8.8x -lvmsnp:0x%8.8x "
                                  "-lvmsnd:0x%8.8x -lvmsnb:0x%8.8x",
                              spec->map[i].vpid,    basename,
                              p->lvm.VoluName,      p->lvm.PartName,
                              p->lvm.VolumeId,      p->lvm.PartitId,
                              d->lvmDiskId,         d->lvmBootId);
                  }
                  else
                  {
                     if (p->lvmReserved != 0)   // LVM signature area needed
                     {                          // before executing the LVM cmd
                        if ((v = dfsGetPartInfo( spec->map[i].vpid)) != NULL)
                        {
                           TxPrint( "\nClone to virtual  : %hu, LVM-sig area\n", p->id);
                           dfsSelectDisk( spec->virtnr, FALSE, FALSE); // select virt disk
                           sprintf(  s1, "clone -B -d:%hu -f:0x%llx,s -a:0x%llx,s -s:0x%x,s",
                                     p->disknr,
                                     p->lastPsn - p->lvmReserved +1,
                                     v->lastPsn - p->lvmReserved +1,
                                     p->lvmReserved);
                           rc = dfsMultiCommand( s1, 0, verbose, FALSE, FALSE); // echo, quiet
                        }
                     }
                     sprintf( s1, "lvm -B -Q %hu -D -s -J -n:'%s' "
                                  "-v:'%s' -p:'%s' -l:%c -m%c "
                                  "-lvmsnv:0x%8.8x -lvmsnp:0x%8.8x "
                                  "-lvmsnd:0x%8.8x -lvmsnb:0x%8.8x",
                              spec->map[i].vpid,    basename,
                              p->lvm.VoluName,      p->lvm.PartName,
                             (p->lvm.letter != 0) ? p->lvm.letter : '-',
                             (p->lvm.OnBmMenu)    ? ' '           : '-',
                              p->lvm.VolumeId,      p->lvm.PartitId,
                              d->lvmDiskId,         d->lvmBootId);
                  }
                  rc = dfsMultiCommand( s1, 0, verbose, FALSE, verbose);
               }
            }
            dfsReadDiskInfo( FDSK_QUIET);       // re-read disk info for LVM
         }
         if (verbose)                           // show result as a table
         {
            dfsMultiCommand( "part -w- -d", 0, FALSE, FALSE, TRUE);
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsSvGrabVirtual'
/*---------------------------------------------------------------------------*/


#define SVSFIRST (DFSI_RAWIMAGE |  DFSI_NOPRITEM)
#define SVSILENT (DFSI_RAWIMAGE |  DFSI_APPENDTO |  DFSI_NOPRITEM)
#define SVPRSHOW (DFSI_RAWIMAGE |  DFSI_APPENDTO)
/*****************************************************************************/
// DFSV write virtual partition-definitions plus physical contents to RAW image
/*****************************************************************************/
static ULONG dfsSvGrabWrite                     // RET   result code
(
   char               *fname,                   // IN    image filename
   BOOL                verbose,                 // IN    verbose progress
   BOOL                testmode,                // IN    fake image-write
   DFSV_GRABSPEC      *spec                     // IN    verified grab spec
)
{
   ULONG               rc = NO_ERROR;           // function return
   int                 i;
   DFSDISKINFO        *d;                       // diskinfo FIRST grab disk
   DFSPARTINFO        *p;
   DFSPARTINFO        *v;
   ULONG               first;                   // first sector todo
   ULONG               isize;                   // image area size
   BYTE                NonHiddenType;

   ENTER();

   if ((d = dfsGetDiskInfo( spec->disknr)) != NULL)
   {
      if (!testmode)
      {
         TxPrint( "\nWrite MBR track-0 : first sector with partition table");
         dfsSelectDisk( spec->virtnr, FALSE, FALSE);
         rc = dfsCreateImage( 0, 1, fname, SVSFIRST, 0); // MBR from virt
         if (rc == NO_ERROR)
         {
            TxPrint( "\nWrite MBR track-0 : possible BMGR/overlay code");
            dfsSelectDisk( spec->disknr, FALSE, FALSE);
            rc = dfsCreateImage( 1, d->geoSecs -2, fname, SVSILENT, 0); // track-0
            if (rc == NO_ERROR)
            {
               TxPrint( "\nWrite MBR track-0 : last sector, possible LVM info");
               dfsSelectDisk( spec->virtnr, FALSE, FALSE);
               rc = dfsCreateImage( d->geoSecs -1, 1, fname, SVSILENT, 0); // pri LVM
            }
         }
      }
      dfsProgressItemsTodo( spec->items, "partition");
      for (i = 0; (i < spec->parts) && (rc == NO_ERROR); i++)
      {
         if (((p = dfsGetPartInfo( spec->map[i].ppid)) != NULL) &&
             ((v = dfsGetPartInfo( spec->map[i].vpid)) != NULL)  )
         {
            if (spec->map[i].dummy == FALSE)    // post-processing requirements
            {
               NonHiddenType = p->partent.PartitionType;

               if (dfsPartTypeHidable(NonHiddenType))  //- only allowed on hidable!
               {                                       //- otherwise values like 35
                  NonHiddenType &= ~DFS_P_PHIDDEN;     //- for LVM will be masked out
               }
               TRACES(( "Set post PID:%hu type:%hu p-ini:%hu v-ini:%hu\n",
                         v->id, NonHiddenType, p->diskPart, v->diskPart));
               if (dfsPartTypeHidable(NonHiddenType) ||
                   (NonHiddenType == DFS_P_WARP_LVM))
               {
                  TRACES(( "PID:%hu Set post-processing FIXPBR\n", v->id));
                  spec->map[i].post |= DFSV_POST_FIXPBR;
               }
               if ((p->primary) &&
                   ((NonHiddenType == DFS_P_FAT32)   ||
                    (NonHiddenType == DFS_P_FAT32X)  || // potential Windows
                    (NonHiddenType == DFS_P_FAT16)   || // system partitions
                    (NonHiddenType == DFS_P_FAT16X)  ||
                    (NonHiddenType == DFS_P_BIGDOS)  ||
                   ((NonHiddenType == DFS_P_INST_FS) &&
                     (strncasecmp(p->fsform, "NTFS", 4) == 0))))
               {
                  if (p->diskPart != v->diskPart) // BOOT.INI index is different
                  {
                     TRACES(( "PID:%hu Set post-processing FIXBOOT\n", v->id));
                     spec->map[i].post |= DFSV_POST_BOOTINI;
                  }
               }
            }
            if (!testmode)
            {
               if (spec->map[i].dummy)          // image from virtual
               {
                  TxPrint( "\nWrite partition   : %hu, 1-cylinder %s DUMMY",
                            p->id, (p->primary) ? "primary" : "logical");
                  dfsSelectDisk( spec->virtnr, FALSE, FALSE);
                  first = (v->primary) ? v->basePsn : v->partPsn;
                  rc = dfsCreateImage( first, v->lastPsn - first +1,
                                       fname, SVSILENT, 0);
               }
               else                             // image from real disk
               {

                  if (p->primary == FALSE)      // selected logical
                  {
                     TxPrint( "\nWrite partition   : %hu, logical EBR track", p->id);
                     dfsSelectDisk( spec->virtnr, FALSE, FALSE);
                     first = v->partPsn;
                     isize = v->basePsn - first;
                     rc = dfsCreateImage( first, isize, fname, SVSILENT, 0);
                  }
                  if (rc == NO_ERROR)
                  {
                     TxPrint( "\nWrite partition   : %hu, %s 0x%2.2hu partition contents",
                               p->id, (p->primary) ? "primary" : "logical",
                               p->partent.PartitionType);
                     dfsSelectDisk( p->disknr, FALSE, FALSE);
                     first = p->basePsn;
                     isize = p->lastPsn -first +1 - p->lvmReserved;
                     rc = dfsCreateImage( first, isize, fname, SVPRSHOW, 0);
                     if ((rc == NO_ERROR) && (p->lvmReserved != 0))
                     {
                        TxPrint( "\nWrite partition   : %hu, %s 0x%2.2hu LVM signature area",
                                  p->id, (p->primary) ? "primary" : "logical",
                                  p->partent.PartitionType);
                        dfsSelectDisk( spec->virtnr, FALSE, FALSE);
                        first = v->lastPsn - p->lvmReserved +1;
                        isize = p->lvmReserved;
                        rc = dfsCreateImage( first, isize, fname, SVSILENT, 0);
                     }
                  }
               }
            }
            else if (spec->map[i].dummy == FALSE)
            {
               TxPrint( "\nSimulated write   : partition %hu to imagefile %s\n",
                         p->id, fname);

               isize = p->lastPsn - p->basePsn;   // simulated size
               dfsProgressInit( 0, isize, 0, "Sector:", "svgrab test", DFSP_STAT, 0);
               for (first = 0; first < isize; first += 1000000)
               {
                  dfsProgressShow( first + min((isize - first), 1000000), 0, 0, NULL);
                  TxSleep( 777);
               }
               dfsProgressTerm();
            }
         }
         else                                   // should be impossible
         {
            rc = DFS_CMD_FAILED;
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsSvGrabWrite'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DFSV post-process the created image to fix expected minor inconsistencies
/*****************************************************************************/
static ULONG dfsSvGrabPostProcess               // RET   result code
(
   BOOL                verbose,                 // IN    verbose progress
   DFSV_GRABSPEC      *spec                     // IN    verified grab spec
)
{
   ULONG               rc = NO_ERROR;           // function return
   int                 i;
   TXLN                s1;

   ENTER();

   for (i = 0; i < spec->parts; i++)
   {
      if (spec->map[i].post != 0)               // any post-processing needed ?
      {
         TRACES(( "Process PID:%02.2hu, post:%8.8x\n", spec->map[i].vpid, spec->map[i].post));

         sprintf( s1, "part -r -q %hu", spec->map[i].vpid);
         dfsMultiCommand( s1, 0, verbose, FALSE, verbose);

         if (SINF->partid == spec->map[i].vpid) // partition in image selected
         {
            if ((spec->map[i].post & DFSV_POST_FIXPBR) != 0)
            {
               strcpy( s1, "fixpbr -B");
               dfsMultiCommand( s1, 0, verbose, FALSE, verbose);
            }
            if ((spec->map[i].post & DFSV_POST_BOOTINI) != 0)
            {
               strcpy( s1, "bootini -B fix -2");
               dfsMultiCommand( s1, 0, verbose, FALSE, verbose);
            }
         }
      }
   }
   sprintf( s1, "part -a -c -V -d:%hu", spec->virtnr);  //- is the IMAGE now!
   dfsMultiCommand( s1, 0, verbose, FALSE, FALSE);      //- echo, and log
   RETURN (rc);
}                                               // end 'dfsSvGrabPostProcess'
/*---------------------------------------------------------------------------*/

#endif                                          // have GRAB
