//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// DFS VDI image data cache and indexing
//
// Implements creating and using data structures to access VirtualBox static or
// dynamic disk image files (VDI) with an INDEX that maps LSN-areas to file-offsets
// and a data-cache that holds one or more recently accessed areas.
//
// The INDEX, cache and read/write functions support multiple VDI file images
// transparently, either for separate virtual disks, or as snapshots chained
// to a base disk VDI. All chained VDI's for a virtual disk are kept OPENED
// until this VboxDisk is closed.
//
// To accomodate adding snapshots to base-VDI's, an 'open-disk' administration
// is kept that allows guided attachement of snapshots to existing base disks.
//
// Author: J. van Wijk
//
// JvW  15-11-2016   Initial version, derived from DFSIMZCI
//
#ifndef    DFSVDICI
   #define DFSVDICI


// Get VDI type confirmation and optional information, from given filename
BOOL dfsImageTypeIsVdi                          // RET   Image is a VDI type
(
   char               *fname,                   // IN    Image filename
   DFS_GUID           *this,                    // OUT   GUID for this image
   DFS_GUID           *link,                    // OUT   GUID for LINK image (parent)
   ULONG              *extents,                 // OUT   used extents in this VDI
   ULONG              *tOffset,                 // OUT   table offset, in bytes
   ULONG              *dOffset,                 // OUT   data  offset, in bytes
   ULONG              *imSize,                  // OUT   Image size in sectors
   ULONG              *vdiBuf,                  // OUT   VDI extent size in bytes
   ULONG              *vdiBps                   // OUT   VDI bytes per sector
);


// Get VDI type confirmation and optional information, from OPEN filehandle
BOOL dfsGetVdiImageInfo                         // RET   Image is a VDI type
(
   char               *fname,                   // IN    Image filename
   TXHFILE             fhandle,                 // IN    Handle to OPEN Image file
   DFS_GUID           *this,                    // OUT   GUID for this image
   DFS_GUID           *link,                    // OUT   GUID for LINK image (parent)
   ULONG              *extents,                 // OUT   used extents in this VDI
   ULONG              *tOffset,                 // OUT   table offset, in bytes
   ULONG              *dOffset,                 // OUT   data  offset, in bytes
   ULONG              *imSize,                  // OUT   Image size in sectors
   ULONG              *vdiBuf,                  // OUT   VDI extent size in bytes
   ULONG              *vdiBps                   // OUT   VDI bytes per sector
);


// Determine if specified link GUID is the last 'this' GUID in existing chain
BOOL dfsVdiIsNextSnapShot                       // RET   GUID is snapshot for chain
(
   DFS_GUID           *link,                    // OUT   GUID for LINK image (parent)
   void              **ppVdi                    // OUT   access data for found chain
);


// Initialize Cache and Indexing structures to access (opened) VDI image
ULONG dfsVdiInitCacheIndex
(
   TXHFILE             fh,                      // IN    handle to opened VDI
   char               *fname,                   // IN    path+name, for .I02 etc
   BOOL                verbose,                 // IN    show info to screen/log
   void              **ppVdi                    // OUT   anonymised access data
);


// Add a (snapshot) VDI image to an existing VDI chain
ULONG dfsVdiAdd2CacheIndex
(
   TXHFILE             fh,                      // IN    handle to opened VDI
   char               *fname,                   // IN    path+name, for VDI image
   BOOL                verbose,                 // IN    show info to screen/log
   void               *pVdi,                    // IN    anonymised access data
   ULONG              *fcount                   // OUT   number of files after add
);


// Free Cache and Indexing structures associated with an VDI image
ULONG dfsVdiFreeCacheIndex
(
   void              **pVdi                     // INOUT anonymised access data
);


// Read ONE sector from an VDI image file, using supplied access data
ULONG dfsVdiReadSector
(
   ULN64               sn,                      // IN    wanted sector number
   ULONG               bps,                     // IN    bytes per sector (buffer)
   void               *pVdi,                    // INOUT anonymised access data
   BYTE               *data                     // OUT   sector data
);


// Write ONE sector to VDI image file (trough cache), using given access data
// Writing a single sector avoids having to 'map' sector-ranges on one or more
// of the possibly cached VDI areas, at the cost of a little more overhead
ULONG dfsVdiWriteSector
(
   ULN64               sn,                      // IN    sector number to write
   ULONG               bps,                     // IN    bytes per sector (buffer)
   void               *pVdi,                    // INOUT anonymised access data
   BYTE               *data                     // IN    sector data
);

#endif
