//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// DFS VDI image data cache and indexing
//
// Implements creating and using data structures to access VirtualBox static or
// dynamic disk image files (VDI) with an INDEX that maps LSN-areas to file-offsets
// and a data-cache that holds one or more recently accessed areas.
//
// The INDEX, cache and read/write functions support multiple VDI file images
// transparently, either for separate virtual disks, or as snapshots chained
// to a base disk VDI. All chained VDI's for a virtual disk are kept OPENED
// until this VboxDisk is closed.
//
// To accomodate adding snapshots to base-VDI's, an 'open-disk' administration
// is kept that allows guided attachement of snapshots to existing base disks.
//
// Writing to sectors will first bring in the extent in cache (when not there yet)
// then update that sector in cache, and update the 'dirty' count for this cache.
// Only when the cache needs to be discarded (LRU, re-use) a dirty one will be
// written back to the (latest) VDI imagefile to the location given by the index.
// When the extent is new (sparse -1 index or from a higher-up VDI in the chain),
// a new extent will be written at the end of the last VDI file in the chain and
// the BLAT as well as the hdr.usedBlocks are updated in-memory (so are 'cached')
// Dirty cache extents are written out as well whenever the disk-image is closed,
// together with the BLAT and HDR (when new extents have been added).
//
// Note: VDI disks are limited to 32bit sectornumbers by design (2TB, 512 bps)
//
// Author: J. van Wijk
//
// JvW  15-11-2016   Initial version, derived from DFSIMZCI
// JvW  18-11-2016   Added write capability, through cache
// JvW  19-04-2017   Made R/W interface use 64bit sectornumbers
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsver.h>                             // DFS version info
#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions

#include <dfsvdici.h>                           // DFS VDI cache/index functions

#define VDICI_BASE_FILE                 0       // File number for BASE file
#define VDICI_SPARSE_FNR           0xFFFF       // filenr for SPARSE index entries
#define VDICI_ZERO_BLOCK       0xFFFFFFFF       // Blocknr for sparse ZERO blocks
#define VDICI_MAGIC            0xdf5beed1       // magic identifier for vdici root
#define VDICI_CACHESIZE                 5       // to be increased ?

#define VDICI_HDRSIZE                 512       // fixed size of a VDI header
#define VDICI_VDI_SIGNATURE    0xbeda107f       // VDI unique signature value
#define VDICI_DYNAMIC                   1       // dynamic type VDI image
#define VDICI_STATIC                    2       // static type VDI image

typedef struct vdiheader
{
   BYTE                vboxDescr[ 0x40];        // 000 VirtualBox descriptive text
   ULONG               signature;               // 040 VDI signature
   USHORT              minorVers;               // 044 minor version number
   USHORT              majorVers;               // 046 major version number
   ULONG               headrSize;               // 048 size of (rest of) header
   ULONG               imageType;               // 04c static/dynamic type image
   ULONG               imageFlags;              // 050 image flags (unused ?)
   BYTE                imageDescr[ 0x100];      // 054 Image description  (unused?)
   ULONG               blatOffset;              // 154 Block alloc table offset
   ULONG               dataOffset;              // 158 Data blocks offset
   ULONG               geoCyls;                 // 15c VDI geo Cylinders  (unused?)
   ULONG               geoHeads;                // 160 VDI geo Heads      (unused?)
   ULONG               geoSecs;                 // 164 VDI geo Sect/track (unused?)
   ULONG               bpsector;                // 168 VDI geo Bytes/Sect (unused?)
   ULONG               unused1;                 // 16c unused
   ULN64               byteSize;                // 170 Disksize in bytes
   ULONG               blockSize;               // 178 Size each diskblock (1 Mib?)
   ULONG               blockExtra;              // 17c extra block data   (unused?)
   ULONG               totalBlocks;             // 180 #blocks total in the disk
   ULONG               usedBlocks;              // 184 #blocks used in the disk
   DFS_GUID            thisGuid;                // 188 UUID of this VDI image
   DFS_GUID            lastGuid;                // 198 UUID of last snapshot ?
   DFS_GUID            linkGuid;                // 1a8 UUID linked VDI (snapshot, 0 = base)
   DFS_GUID            parentGuid;              // 1b8 UUID of parent ?
   ULONG               garbage[ 6];             // 1c8 Unknown contents
   ULONG               unused2[ 8];             // 1e0 Zeroes to end of header (512)
} VDIHEADER;                                    // end of struct "vdiheader"


typedef struct vdiindex
{
   USHORT              filenr;                  // file nr, 1==BASE 2=1st snapshot etc
   ULONG               blockNr;                 // data block nr in file, or -1 (sparse)
} VDIINDEX;                                     // end of struct "vdiindex"


typedef struct vdicachedata                     // cache data element
{
   TXTIMER             lastUse;                 // last use timestamp or 0
   ULONG               ix;
   ULONG               dirtySects;              // number of sector writes to buffer
   BYTE               *Buffer;                  // cache data buffer
} VDICACHEDATA;                                 // end of struct "vdicachedata"


typedef struct vdicifile                        // BASE/snapshots, max is MAX_DISK
{
   ULONG               blatOffset;              // block-table offset, in bytes
   ULONG               dataOffset;              // Data blocks offset, in bytes
   DFS_GUID            thisGuid;                // GUID for this imagefile
   TXHFILE             vdiHandle;               // R/W filehandle first file  (IMZ)
   TXLN                vdiName;                 // path+filename  first file  (IMZ)
} VDICIFILE;                                    // end of struct "vdicifile"


typedef struct vdiciroot                        // ROOT info for VDI cache/index
{
   ULONG               magic;                   // magic signature
   ULONG               sectors;                 // nr of sectors in VDI, size
   ULONG               bpsector;                // bytes per buffer sector
   ULONG               bufSize;                 // number of sectors per buffer
   ULONG               files;                   // filecount for chained VDI files
   VDICIFILE          *vdiChain;                // info on all chained VDI files
   ULONG               lastIx;                  // last unit read (sequential test)
   ULONG               cacheCount;              // number of cachedata elements
   VDICACHEDATA       *cache;                   // allocated array of cache elements
   ULONG               indexCount;              // number of index elements
   VDIINDEX           *index;                   // allocated array of index elements
   ULONG               usedBlocks;              // #blocks used in LAST VDI in chain
   ULONG               newBlocks;               // #blocks NEW (hdr/blat update needed)
   VDIHEADER          *hdr;                     // cached HDR  of LAST VDI in chain
   ULONG              *blat;                    // cached BLAT of LAST VDI in chain
} VDICIROOT;                                    // end of struct "vdiciroot"


static VDICIROOT *vdiDiskDB[ DFS_MAX_DISK] = {NULL}; // overall VDI disk database


// Get a (pointer to) a cache instance for this data-block, flush if dirty
ULONG dfsVdiGetCache
(
   ULONG               ix,                      // IN    data-block index
   VDICIROOT          *ir,                      // INOUT VDI info, root structure
   VDICACHEDATA      **cacheInstance            // OUT   best-fit cache-instance
);


// Read data-block from VDI file into a cache element
static ULONG dfsVdiFile2Cache
(
   VDICACHEDATA       *cache,                   // INOUT cache structure to use
   VDICIROOT          *ir                       // INOUT VDI info, root structure
);


// Write data-block from cache element to the (last) VDI file
static ULONG dfsVdiCache2File
(
   VDICACHEDATA       *cache,                   // INOUT cache structure to use
   VDICIROOT          *ir                       // INOUT VDI info, root structure
);


// Create indexes for a (base or snapshot) VDI image file, reads HDR and BLAT
static ULONG dfsVdiIndexImageFile
(
   int                 filenr,                  // IN    sequence number for file
   TXHFILE             fh,                      // IN    handle to opened VDI/Inn
   VDICIROOT          *pVdi                     // INOUT VDI info, root structure
);


// Write back the (modified) HDR and BLAT to (base or snapshot) VDI image file
static ULONG dfsVdiWriteBlatHdr
(
   VDICIROOT          *ir                       // IN    VDI info, root structure
);


/*****************************************************************************/
// Get VDI type confirmation and optional information, from given filename
/*****************************************************************************/
BOOL dfsImageTypeIsVdi                          // RET   Image is a VDI type
(
   char               *fname,                   // IN    Image filename
   DFS_GUID           *this,                    // OUT   GUID for this image
   DFS_GUID           *link,                    // OUT   GUID for LINK image (parent)
   ULONG              *extents,                 // OUT   used extents in this VDI
   ULONG              *tOffset,                 // OUT   table offset, in bytes
   ULONG              *dOffset,                 // OUT   data  offset, in bytes
   ULONG              *imSize,                  // OUT   Image size in sectors
   ULONG              *vdiBuf,                  // OUT   VDI extent size in bytes
   ULONG              *vdiBps                   // OUT   VDI bytes per sector
)
{
   BOOL                rc  = FALSE;             // function return
   TXHFILE             fhandle;                 // Handle for Image file

   ENTER();

   if (TxFileOpenReadOnly( fname, &fhandle) == NO_ERROR)
   {
      rc = dfsGetVdiImageInfo( fname, fhandle,
                               this, link, extents, tOffset, dOffset,
                               imSize, vdiBuf, vdiBps);
      TxClose( fhandle);
   }
   RETURN (rc);
}                                               // end 'dfsImageTypeIsVdi'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get VDI type confirmation and optional information, from OPEN filehandle
/*****************************************************************************/
BOOL dfsGetVdiImageInfo                         // RET   Image is a VDI type
(
   char               *fname,                   // IN    Image filename
   TXHFILE             fhandle,                 // IN    Handle to OPEN Image file
   DFS_GUID           *this,                    // OUT   GUID for this image
   DFS_GUID           *link,                    // OUT   GUID for LINK image (parent)
   ULONG              *extents,                 // OUT   used extents in this VDI
   ULONG              *tOffset,                 // OUT   table offset, in bytes
   ULONG              *dOffset,                 // OUT   data  offset, in bytes
   ULONG              *imSize,                  // OUT   Image size in sectors
   ULONG              *vdiBuf,                  // OUT   VDI extent size in bytes
   ULONG              *vdiBps                   // OUT   VDI bytes per sector
)
{
   BOOL                rc  = FALSE;             // function return
   VDIHEADER           hdr;                     // VDI header, 1 sector
   ULONG               handled;                 // nr of bytes handled

   ENTER();

   TRACES(("fname: '%s'  handle: %lu\n", fname, fhandle));

   TxFileSeek( fhandle, 0, SEEK_SET);           // make sure we are at start of file
   if (TxRead( fhandle, &hdr, VDICI_HDRSIZE, &handled) == NO_ERROR)
   {
      TRACES(("sig:%8.8lx  Buf:%8.8lx Bps:%8.8lx blocks:%8.8lx usedBlocks:%8.8lx tOff:%8.8lx dOff:%8.8lx\n",
               hdr.signature, hdr.blockSize, hdr.bpsector, hdr.totalBlocks, hdr.usedBlocks, hdr.blatOffset, hdr.dataOffset));
      if (hdr.signature == VDICI_VDI_SIGNATURE)
      {
         if (this != NULL)
         {
            memcpy( this, &hdr.thisGuid, DFS_GUID_LENGTH);
         }
         if (link != NULL)
         {
            memcpy( link, &hdr.linkGuid, DFS_GUID_LENGTH);
         }
         if (extents != NULL)
         {
            *extents = hdr.usedBlocks;
         }
         if (vdiBuf != NULL)
         {
            *vdiBuf = hdr.blockSize;
         }
         if (vdiBps != NULL)
         {
            *vdiBps = hdr.bpsector;
         }
         if (imSize != NULL)
         {
            *imSize = (ULONG) (hdr.totalBlocks * (hdr.blockSize / hdr.bpsector));
         }
         if (tOffset != NULL)
         {
            *tOffset = hdr.blatOffset;
         }
         if (dOffset != NULL)
         {
            *dOffset = hdr.dataOffset;
         }
         rc = TRUE;
      }
   }
   BRETURN (rc);
}                                               // end 'dfsGetVdiImageInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Determine if specified link GUID is the last 'this' GUID in existing chain
/*****************************************************************************/
BOOL dfsVdiIsNextSnapShot                       // RET   GUID is snapshot for chain
(
   DFS_GUID           *link,                    // OUT   GUID for LINK image (parent)
   void              **ppVdi                    // OUT   anonymised access data
)
{
   BOOL                rc  = FALSE;             // function return
   int                 i;
   VDICIROOT          *ir = NULL;               // VDI info, root structure
   VDICIFILE          *vfile;                   // last imagefile in a chain

   ENTER();

   for (i = 0; i < DFS_MAX_DISK; i++)           // iterate over slots
   {
      ir = vdiDiskDB[ i];
      if (ir != NULL)                           // in-use handle/reference slot
      {
         vfile = &(ir->vdiChain[ ir->files - 1]);

         if (memcmp( vfile->thisGuid, link, DFS_GUID_LENGTH) == 0)
         {
            TRACES(("Matching VDI slot: %d\n", i));
            *ppVdi = ir;                        // return the found reference
            rc = TRUE;
            break;                              // should only be one ...
         }
      }
   }
   BRETURN (rc);
}                                               // end 'dfsVdiIsNextSnapShot'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Initialize Cache and Indexing structures to access (opened) VDI image
// This should be the BASE image file, with snapshots added/indexed later
/*****************************************************************************/
ULONG dfsVdiInitCacheIndex
(
   TXHFILE             fh,                      // IN    handle to opened VDI
   char               *fname,                   // IN    path+name, for VDI image
   BOOL                verbose,                 // IN    show info to screen/log
   void              **ppVdi                    // OUT   anonymised access data
)
{
   ULONG               rc = NO_ERROR;
   VDICIROOT          *ir = NULL;               // VDI info, root structure
   int                 i;                       // iterator
   ULONG               bOffset;                 // block table offset
   ULONG               dOffset;                 // data blocks offset
   ULONG               bufferBytes;
   DFS_GUID            thisGuid;                // GUID for this image
   VDICIFILE          *vfile;

   ENTER();

   if ((ir = TxAlloc( 1, sizeof( VDICIROOT))) != NULL)
   {
      for (i = 0; i < DFS_MAX_DISK; i++)        // find an available slot, and take it
      {
         if (vdiDiskDB[ i] == NULL)             // available handle/reference slot
         {
            TRACES(("Occupy  VDI slot: %d\n", i));
            vdiDiskDB[ i] = ir;                 // make it occupied, no free yet
            break;                              // need only one ...
         }
      }
      ir->magic = VDICI_MAGIC;

      if (dfsGetVdiImageInfo( fname, fh, &thisGuid, NULL, &ir->usedBlocks, &bOffset, &dOffset,
                                         &ir->sectors, &bufferBytes, &ir->bpsector))
      {
         ir->files      = 1;
         ir->bufSize    = bufferBytes / ir->bpsector;
         ir->cacheCount = VDICI_CACHESIZE;
         ir->indexCount = ((ir->sectors - 1) / ir->bufSize) + 1;

         if (verbose)
         {
            dfsSizeBps( "VDI defined  size : ", ir->sectors, ir->bpsector, "");
            TxPrint(    " for '%s'\n", fname);
            dfsSizeBps( "   allocated size : ", ir->usedBlocks * ir->bufSize, ir->bpsector, "");
            TxPrint(    " in %lu blocks\n",     ir->usedBlocks);
         }
         if (((ir->vdiChain = TxAlloc( DFS_MAX_DISK,   sizeof( VDICIFILE)   )) != NULL) &&
             ((ir->index    = TxAlloc( ir->indexCount, sizeof( VDIINDEX)    )) != NULL) &&
             ((ir->blat     = TxAlloc( ir->indexCount, sizeof( ULONG)       )) != NULL) &&
             ((ir->hdr      = TxAlloc(              1, VDICI_HDRSIZE        )) != NULL) &&
             ((ir->cache    = TxAlloc( ir->cacheCount, sizeof( VDICACHEDATA))) != NULL)  )
         {
            vfile = &ir->vdiChain[ 0];          // base is filenr 0 by definition
            vfile->vdiHandle  = fh;
            vfile->blatOffset = bOffset;
            vfile->dataOffset = dOffset;
            memcpy( vfile->thisGuid, thisGuid, DFS_GUID_LENGTH);
            strcpy( vfile->vdiName,  fname);

            for (i = 0; i < ir->cacheCount; i++)
            {
               if ((ir->cache[ i].Buffer = TxAlloc( ir->bufSize, ir->bpsector)) == NULL)
               {
                  rc = DFS_ALLOC_ERROR;
                  break;
               }
            }
            if (rc == NO_ERROR)
            {
               dfsProgressInit( 0, ir->sectors, 0, "Sector:", "indexed", DFSP_BARS, 0);

               rc = dfsVdiIndexImageFile( 0, fh, ir); // base VDI

               dfsProgressTerm();               // allow regular screen output
            }
         }
         else
         {
            rc = DFS_ALLOC_ERROR;
         }
      }
      else
      {
         rc = DFS_ST_MISMATCH;                  // unexpected, should be VDI
      }

   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   *ppVdi = ir;

   if (rc != NO_ERROR)                          // free (partly) allocated memory
   {
      dfsVdiFreeCacheIndex( ppVdi);
   }
   RETURN ( rc);
}                                               // end 'dfsVdiInitCacheIndex'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add a (snapshot) VDI image to an existing VDI chain
/*****************************************************************************/
ULONG dfsVdiAdd2CacheIndex
(
   TXHFILE             fh,                      // IN    handle to opened VDI
   char               *fname,                   // IN    path+name, for VDI image
   BOOL                verbose,                 // IN    show info to screen/log
   void               *pVdi,                    // IN    anonymised access data
   ULONG              *fcount                   // OUT   number of files after add
)
{
   ULONG               rc = NO_ERROR;
   VDICIROOT          *ir = (VDICIROOT *) pVdi; // VDI info, root structure
   ULONG               sectors;
   ULONG               bpsector;
   ULONG               bOffset;                 // block table offset
   ULONG               dOffset;                 // data blocks offset
   ULONG               bufferBytes;
   DFS_GUID            thisGuid;                // GUID for this image
   VDICIFILE          *vfile;

   ENTER();


   if ((ir != NULL) && (ir->magic == VDICI_MAGIC))
   {
      if (ir->newBlocks == 0)                   // not written to existing chain yet ?
      {
         if (dfsGetVdiImageInfo( fname, fh, &thisGuid, NULL, &ir->usedBlocks,  &bOffset, &dOffset,
                                            &sectors, &bufferBytes, &bpsector))
         {
            if (verbose)
            {
               dfsSizeBps( "VDI snapshot size : ", ir->sectors, ir->bpsector, "");
               TxPrint(    " for '%s'\n", fname);
               dfsSizeBps( "   allocated size : ", ir->usedBlocks * ir->bufSize, ir->bpsector, "");
               TxPrint(    " in %lu blocks\n",     ir->usedBlocks);
            }
            if (ir->files < DFS_MAX_DISK)
            {
               if ((bpsector    ==  ir->bpsector)          &&
                   (sectors     ==  ir->sectors )          &&
                   (bufferBytes == (ir->bufSize * bpsector)))
               {
                  int     i;

                  for (i = 1; i < ir->cacheCount; i++)
                  {
                     ir->cache[ i].lastUse = 0; // invalidate cache contents
                  }
                  vfile = &ir->vdiChain[ ir->files]; // take next file slot
                  vfile->vdiHandle  = fh;
                  vfile->blatOffset = bOffset;
                  vfile->dataOffset = dOffset;
                  memcpy( vfile->thisGuid, thisGuid, DFS_GUID_LENGTH);
                  strcpy( vfile->vdiName,  fname);

                  dfsProgressInit( 0, ir->sectors, 0, "Sector:", "indexed", DFSP_BARS, 0);

                  rc = dfsVdiIndexImageFile( ir->files, fh, ir); // snapshot

                  dfsProgressTerm();            // allow regular screen output
                  ir->files++;                  // update count
                  *fcount = ir->files;
               }
               else
               {
                  rc = DFS_ST_MISMATCH;         // unexpected, should match BASE
               }
            }
            else
            {
               TxPrint( "Max# of snapshots : %d has been reached, can NOT add this VDI\n", DFS_MAX_DISK);
               rc = DFS_NO_CHANGE;
            }
         }
         else
         {
            rc = DFS_ST_MISMATCH;               // unexpected, should be VDI
         }
      }
      else
      {
         TxPrint( "Existing VDI has  : %lu new allocated blocks, can NOT add this VDI\n", ir->newBlocks);
         rc = DFS_NO_CHANGE;
      }
   }
   else
   {
      rc = DFS_BAD_STRUCTURE;
   }
   RETURN ( rc);
}                                               // end 'dfsVdiAdd2CacheIndex'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Flush Cache, then free Cache and Indexing structures for a VDI image chain
/*****************************************************************************/
ULONG dfsVdiFreeCacheIndex
(
   void              **pVdi                     // INOUT anonymised access data
)
{
   ULONG               rc = NO_ERROR;
   VDICIROOT          *ir = (VDICIROOT *) *pVdi; // VDI info, root structure
   int                 i;

   ENTER();
   TRACES(("ir:%8.8lx Magic:%8.8lx  sectors:%8.8lx\n", ir, ir->magic, ir->sectors));

   if ((ir != NULL) && (ir->magic == VDICI_MAGIC))
   {
      for (i = 0; i < ir->cacheCount; i++)
      {
         if (ir->cache[ i].dirtySects != 0)     // there is unwritten data in cache
         {
            rc = dfsVdiCache2File( &ir->cache[ i], ir); // flush to VDI file
         }
         TRACES(("Free cache buffer: %d at 0x%8.8lx\n", i, ir->cache[ i].Buffer));
         TxFreeMem( ir->cache[ i].Buffer);
      }
      if (ir->newBlocks != 0)
      {
         ir->hdr->usedBlocks = ir->usedBlocks;  // update in-memory HDR
         rc = dfsVdiWriteBlatHdr( ir);          // and write back to last VDI
      }
      for (i = 1; i < ir->files; i++)           // close all snapshot files
      {                                         // (base closed by caller!)
         TRACES(("Closing snapshot %d, VDI: '%s'\n", i, ir->vdiChain[ i].vdiName));
         TxClose( ir->vdiChain[ i].vdiHandle);
      }
      for (i = 0; i < DFS_MAX_DISK; i++)
      {
         if (vdiDiskDB[ i] == ir)               // matching handle/reference slot
         {
            TRACES(("Release VDI slot: %d\n", i));
            vdiDiskDB[ i] = NULL;               // make it available, no free yet
            break;                              // there can only be one ...
         }
      }
      TRACES(("Free vdiChain at: 0x%8.8lx\n", ir->vdiChain));
      TxFreeMem( ir->vdiChain);
      TRACES(("Free cache    at: 0x%8.8lx\n", ir->cache));
      TxFreeMem( ir->cache);
      TRACES(("Free index    at: 0x%8.8lx\n", ir->index));
      TxFreeMem( ir->index);
      TRACES(("Free blat     at: 0x%8.8lx\n", ir->blat));
      TxFreeMem( ir->blat);
      TRACES(("Free header   at: 0x%8.8lx\n", ir->hdr));
      TxFreeMem( ir->hdr);
      TRACES(("Free root     at: 0x%8.8lx\n", *pVdi));
      TxFreeMem( *pVdi);                        // free ROOT structure, zero ref
   }
   else
   {
      rc = DFS_BAD_STRUCTURE;
   }
   RETURN ( rc);
}                                               // end 'dfsVdiFreeCacheIndex'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read ONE sector from an VDI image file, using supplied access data
// Reading a single sector avoids having to 'map' sector-ranges on one or more
// of the possibly cached VDI areas, at the cost of a little more overhead
/*****************************************************************************/
ULONG dfsVdiReadSector
(
   ULN64               sn,                      // IN    wanted sector number
   ULONG               bps,                     // IN    bytes per sector (buffer)
   void               *pVdi,                    // INOUT anonymised access data
   BYTE               *data                     // OUT   sector data
)
{
   ULONG               rc = NO_ERROR;
   VDICIROOT          *ir = (VDICIROOT *) pVdi; // VDI info, root structure
   ULONG               ix;                      // data-block index
   ULONG               rsn;                     // Relative sector in data-block
   VDICACHEDATA       *cache;                   // pointer to cache to be used

   ENTER();
   TRACES(("pVdi:%8.8lx sn: 0x%llx  bps:%lu\n", pVdi, sn, bps));

   if (ir && (ir->magic == VDICI_MAGIC) && (bps == ir->bpsector))
   {
      if ((ir->bufSize) && (sn < (ULN64) ir->sectors))
      {
         ix     = (ULONG) sn / ir->bufSize;
         rsn    = (ULONG) sn % ir->bufSize;

         TRACES(("Block nr: 0x%8.8lx  rsn:0x%8.8lx\n", ix, rsn));

         if (ir->index[ ix].blockNr != VDICI_ZERO_BLOCK) // block is present, get it
         {
            rc = dfsVdiGetCache( ix, ir, &cache); // try the cache first
            if (rc == NO_ERROR)
            {
               if (cache->lastUse == 0)         // no valid contents, need to read
               {
                  rc = dfsVdiFile2Cache( cache, ir);
               }
               if (rc == NO_ERROR)
               {
                  //- data is in the cache now, copy the single specified sector
                  memcpy( data, cache->Buffer + (rsn * ir->bpsector), ir->bpsector);
                  cache->lastUse = TxTmrGetNanoSecFromStart(); // update LastUse timestamp
               }
            }
         }
         else                                   // SPARSE extent, not in file
         {
            TRACES(("SPARSE extent, fill ZEROs\n"));
            memset( data, 0x00, bps);           // set whole buffer to ZERO
         }
      }
      else
      {
         rc = DFS_PSN_LIMIT;                    // outside of the VDI stored range
      }
   }
   else
   {
      rc = DFS_BAD_STRUCTURE;
   }
   RETURN ( rc);
}                                               // end 'dfsVdiReadSector'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Write ONE sector to VDI image file (trough cache), using given access data
// Writing a single sector avoids having to 'map' sector-ranges on one or more
// of the possibly cached VDI areas, at the cost of a little more overhead
/*****************************************************************************/
ULONG dfsVdiWriteSector
(
   ULN64               sn,                      // IN    sector number to write
   ULONG               bps,                     // IN    bytes per sector (buffer)
   void               *pVdi,                    // INOUT anonymised access data
   BYTE               *data                     // IN    sector data
)
{
   ULONG               rc = NO_ERROR;
   VDICIROOT          *ir = (VDICIROOT *) pVdi; // VDI info, root structure
   ULONG               ix;                      // data-block index
   ULONG               rsn;                     // Relative sector in data-block
   VDICACHEDATA       *cache;                   // pointer to cache to be used

   ENTER();
   TRACES(("pVdi:%8.8lx sn: 0x%llx  bps:%lu\n", pVdi, sn, bps));

   if (ir && (ir->magic == VDICI_MAGIC) && (bps == ir->bpsector))
   {
      if ((ir->bufSize) && (sn < (ULN64) ir->sectors))
      {
         ix     = (ULONG) sn / ir->bufSize;
         rsn    = (ULONG) sn % ir->bufSize;

         TRACES(("Block nr: 0x%lx  rsn:0x%lx\n", ix, rsn));

         dfsVdiGetCache( ix, ir, &cache);       // get best fitting cache element
         if (rc == NO_ERROR)
         {
            if (ir->index[ ix].blockNr != VDICI_ZERO_BLOCK) // block is present, get it
            {
               if (cache->lastUse == 0)         // no valid contents, need to read
               {
                  rc = dfsVdiFile2Cache( cache, ir);
               }
            }
            else                                // SPARSE, not in file yet
            {                                   // create ZEROED cache for it
               TRACES(("SPARSE extent, fill ZEROs\n"));
               memset( cache->Buffer, 0, ir->bufSize * ir->bpsector);
            }
            if (ir->index[ ix].filenr != (ir->files - 1))
            {
               //- block NOT in the last VDI, or was SPARSE, must move it (in index)
               ir->index[ ix].filenr  = ir->files - 1;
               ir->index[ ix].blockNr = ir->usedBlocks; // update our own index
               ir->blat[  ix]         = ir->usedBlocks; // update last VDI blat

               ir->usedBlocks++;                // increment, for next new one
               ir->newBlocks++;
            }
            if (rc == NO_ERROR)
            {
               BYTE   *cacheData = cache->Buffer + (rsn * ir->bpsector);

               //- current data is in the cache now, write/count only when DIFFERENT!
               if (memcmp( cacheData, data, ir->bpsector) != 0)
               {
                  memcpy(  cacheData, data, ir->bpsector);  //- update sector contents
                  cache->dirtySects++;                      //- count dirty sector
                  TRACES(("Write OK, dirtySects: %lu\n", cache->dirtySects));
               }
               else
               {
                  TRACES(("Write discarded, same contents as current.\n"));
               }
               cache->lastUse = TxTmrGetNanoSecFromStart(); // update LastUse timestamp
            }
         }
      }
      else
      {
         rc = DFS_PSN_LIMIT;                    // outside of the VDI stored range
      }
   }
   else
   {
      rc = DFS_BAD_STRUCTURE;
   }
   RETURN ( rc);
}                                               // end 'dfsVdiWriteSector'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get a (pointer to) a cache instance for this data-block, flush if dirty
/*****************************************************************************/
ULONG dfsVdiGetCache
(
   ULONG               ix,                      // IN    data-block index
   VDICIROOT          *ir,                      // INOUT VDI info, root structure
   VDICACHEDATA      **cacheInstance            // OUT   best-fit cache-instance
)
{
   ULONG               rc = NO_ERROR;
   VDICACHEDATA       *ci = NULL;
   int                 i;
   BOOL                inCache = FALSE;         // sector was already in cache

   //- First, see if there is a cache-element for this unit already
   for (i = 0; i < ir->cacheCount; i++)
   {
      if ((ir->cache[ i].ix == ix) && (ir->cache[ i].lastUse != 0))
      {
         TRACES(("VdiGetCache: BLOCK %lu already cached in nr: %d\n", ix, i));
         ci = &(ir->cache[ i]);                 // same unit already there
         inCache = TRUE;
         break;
      }
   }
   if (ci == NULL)                              // not present yet
   {
      //- Second, see if this unit is a sequential-access for a cache-element
      for (i = 0; i < ir->cacheCount; i++)
      {
         if ((ir->cache[ i].ix + 1 == ix) && (ir->cache[ i].lastUse != 0))
         {
            TRACES(("VdiGetCache: BLOCK %lu in sequential order for nr: %d\n", ix, i));
            ci = &(ir->cache[ i]);              // block is NEXT one for this element
            break;
         }
      }
   }
   if (ci == NULL)                              // find a free/discardable one
   {
      TXTIMER  oldest  = TXTMR_MAX;

      TRACES(("VdiGetCache: BLOCK %lu LRU candidates: ", ix));
      for (i = 0; i < ir->cacheCount; i++)
      {
         TXTIMER  lastUse = ir->cache[ i].lastUse;

         if      (lastUse == 0)                 // Free, use this
         {
            TRINTF(("=> Free nr: %d", i));
            ci = &(ir->cache[ i]);
            break;
         }
         else if (lastUse < oldest)             // oldest sofar
         {
            oldest = lastUse;
            TRINTF((" %d ", i));
            ci = &(ir->cache[ i]);
         }
      }
      TRINTF(("\n"));
   }
   if ((ci != NULL) && (inCache == FALSE))      // We are re-using a cache buffer
   {
      if (ci->dirtySects != 0)                  // there is unwritten data in cache
      {
         rc = dfsVdiCache2File( ci, ir);        // flush datablock to VDI file
         ci->dirtySects = 0;                    // mark it clean again
      }
      ci->lastUse = 0;                          // LastUse 0 indicates invalid data
      ci->ix      = ix;                         // set it to this ix value
   }
   *cacheInstance = ci;
   return ( rc);
}                                               // end 'dfsVdiGetCache'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read data-block from VDI file into a cache element
/*****************************************************************************/
static ULONG dfsVdiFile2Cache
(
   VDICACHEDATA       *cache,                   // INOUT cache structure to use
   VDICIROOT          *ir                       // INOUT VDI info, root structure
)
{
   ULONG               rc = NO_ERROR;
   ULONG               ix = cache->ix;
   USHORT              filenr;
   VDICIFILE          *vfile;

   ENTER();

   filenr =  ir->index[ ix].filenr;
   vfile  = &ir->vdiChain[ filenr];

   TRACES(("Read  Block#: 0x%8.8lx in file#: %lu = '%s'\n", ix, filenr, vfile->vdiName));

   if (rc == NO_ERROR)                          // file is open, read ...
   {
      ULONG            handled;                 // nr of bytes handled
      ULONG            sz = ir->bufSize * ir->bpsector; // size of buffers

      TxFileSeek(  vfile->vdiHandle, ir->index[ ix].blockNr * sz + vfile->dataOffset, SEEK_SET);
      rc = TxRead( vfile->vdiHandle, cache->Buffer, sz, &handled);
      TRACES(("TxRead rc: %ld, read: %lu\n", (LONG) rc, handled));
   }
   RETURN ( rc);
}                                               // end 'dfsVdiFile2Cache'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Write data-block from cache element to the (last) VDI file
/*****************************************************************************/
static ULONG dfsVdiCache2File
(
   VDICACHEDATA       *cache,                   // INOUT cache structure to use
   VDICIROOT          *ir                       // INOUT VDI info, root structure
)
{
   ULONG               rc = NO_ERROR;
   ULONG               ix = cache->ix;
   USHORT              filenr;
   VDICIFILE          *vfile;

   ENTER();

   filenr =  ir->index[ ix].filenr;
   vfile  = &ir->vdiChain[ filenr];

   TRACES(("Write Block#: 0x%8.8lx in file#: %lu = '%s'\n", ix, filenr, vfile->vdiName));

   if (rc == NO_ERROR)                          // file is open, write ...
   {
      ULONG            handled;                 // nr of bytes handled
      ULONG            sz = ir->bufSize * ir->bpsector; // size of buffers

      TxFileSeek(   vfile->vdiHandle, ir->index[ ix].blockNr * sz + vfile->dataOffset, SEEK_SET);
      rc = TxWrite( vfile->vdiHandle, cache->Buffer, sz, &handled);
   }
   RETURN ( rc);
}                                               // end 'dfsVdiCache2File'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create indexes for a (base or snapshot) VDI image file, reads HDR and BLAT
/*****************************************************************************/
static ULONG dfsVdiIndexImageFile
(
   int                 filenr,                  // IN    VDI sequence nr, 0 = base
   TXHFILE             fh,                      // IN    handle to opened VDI
   VDICIROOT          *ir                       // INOUT VDI info, root structure
)
{
   ULONG               rc = NO_ERROR;
   ULONG               handled;                 // nr of bytes handled
   VDIINDEX           *pix;                     // next index record to fill
   VDICIFILE          *vfile;                   // VDI file info
   int                 i;

   ENTER();

   TRACES(("ir:%8.8lx Magic:%8.8lx  sectors:%8.8lx\n", ir, ir->magic, ir->sectors));

   vfile = &ir->vdiChain[ filenr];              // base is filenr 0, others 1..n

   //- Read table in memory (worst case a few megabytes) cached for write updates
   TxFileSeek(  fh, 0, SEEK_SET);
   rc = TxRead( fh, (BYTE *) ir->hdr, VDICI_HDRSIZE, &handled); // read header
   TRACES(("TxRead rc: %ld, read: %lu\n", (LONG) rc, handled));
   if (rc == NO_ERROR)
   {
      TxFileSeek(  fh, vfile->blatOffset, SEEK_SET);
      rc = TxRead( fh, (BYTE *) ir->blat, ir->indexCount * sizeof( ULONG), &handled);
      TRACES(("TxRead rc: %ld, read: %lu\n", (LONG) rc, handled));
      if (rc == NO_ERROR)
      {
         for (i = 0; i < ir->indexCount; i++)
         {
            if ((filenr      == VDICI_BASE_FILE) || // BASE fills all slots
                (ir->blat[i] != VDICI_ZERO_BLOCK) ) // others only when 'in use'
            {
               pix = &ir->index[ i];                //- next record to fill
               if (ir->blat[i] != VDICI_ZERO_BLOCK) //- real entry, use real filenr
               {
                  pix->filenr = filenr;
               }
               else
               {
                  pix->filenr = VDICI_SPARSE_FNR;   //- fake sequence nr
               }
               pix->blockNr = ir->blat[i];          //- relative block in VDI file
               TRACES(( "Block#:%7lu at relBlock %ld\n", i, pix->blockNr));
            }
         }
      }
   }
   RETURN ( rc);
}                                               // end 'dfsVdiIndexImageFile'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Write back the (modified) HDR and BLAT to (base or snapshot) VDI image file
/*****************************************************************************/
static ULONG dfsVdiWriteBlatHdr
(
   VDICIROOT          *ir                       // IN    VDI info, root structure
)
{
   ULONG               rc = NO_ERROR;
   TXHFILE             fh;                      // handle to opened VDI
   ULONG               handled;                 // nr of bytes handled
   VDICIFILE          *vfile;                   // VDI file info

   ENTER();

   vfile = &ir->vdiChain[ ir->files -1];        // nr for LAST loaded VDI
   fh    = vfile->vdiHandle;

   TxFileSeek(   fh, 0, SEEK_SET);
   rc = TxWrite( fh, (BYTE *) ir->hdr, VDICI_HDRSIZE, &handled); // read header
   TRACES(("TxWrite rc: %ld, written: %lu\n", (LONG) rc, handled));
   if (rc == NO_ERROR)
   {
      //- Write table from memory, after write updates
      TxFileSeek(   fh, vfile->blatOffset, SEEK_SET);
      rc = TxWrite( fh, (BYTE *) ir->blat, ir->indexCount * sizeof( ULONG), &handled);
      TRACES(("TxWrite rc: %ld, written: %lu\n", (LONG) rc, handled));
   }
   RETURN ( rc);
}                                               // end 'dfsVdiWriteBlatHdr'
/*---------------------------------------------------------------------------*/


