//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// XFS utility functions
//
// Author: J. van Wijk
//
// JvW  15-07-2007 Initial version, derived from XFS
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsver.h>                             // DFS version info
#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfstable.h>                           // SLT utility functions

#include <dfsxfs.h>                             // XFS disk structures
#include <dfsaxfs.h>                            // XFS analysis functions
#include <dfsuxfs.h>                            // XFS utility functions
#include <dfslxfs.h>                            // XFS slt & error definitions


/*****************************************************************************/
// Identify if FS structures for XFS are present, set fsform and label
/*****************************************************************************/
ULONG dfsXfsFsIdentify
(
   S_BOOTR            *boot,                    // IN    Bootsector ref or NULL
   DFSPARTINFO        *p                        // INOUT partition info
)
{
   ULONG               rc = DFS_FS_UNKNOWN;     // function return
   S_XFS_SUPER        *sup;                     // superblock sector
   char                st;

   ENTER();

   if ((sup = TxAlloc(1, dfsGetSectorSize())) != NULL)
   {
      if (dfsRead( p->basePsn + XFS_LSNSUP1, 1, (BYTE *) sup) == NO_ERROR)
      {
         if ((dfsXfsIdent(XFS_LSNSUP1, 0, &st, sup) != DFS_PENDING) && (st == ST_XFSSUP))
         {
            strcpy( p->fsform, "XFS");
            TxCopy( p->plabel, sup->VolumeName, XFS_LEN_LBL +1);
            memcpy( p->fsUuid, sup->Uuid,       FS_UUID_LENGTH);
            p->uuidPresent = TRUE;
            rc = NO_ERROR;
         }
      }
      TxFreeMem( sup);
   }
   RETURN (rc);
}                                               // end 'dfsXfsFsIdentify'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate Block-nr to LSN
/*****************************************************************************/
ULN64 dfsXfsBlock2LSN                           // RET   LSN for this block
(
   ULONG               block                    // IN    block number
)
{
   ULN64               lsn = block;

   lsn = block * xfs->SectorsPerBlock;
   return (lsn);
}                                               // end 'dfsXfsBlock2LSN'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate LSN to Block-nr
/*****************************************************************************/
ULONG dfsXfsLSN2Block                           // RET   block for this LSN
(
   ULN64               lsn                      // IN    LSN
)
{
   ULONG               block = 0;

   block = lsn / xfs->SectorsPerBlock;
   return (block);
}                                               // end 'dfsXfsLSN2Block'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Write cached superblock to superblock location on disk
/*****************************************************************************/
ULONG dfsXfsWriteSuperBlocks
(
   void
)
{
   ULONG               rc  = NO_ERROR;

   ENTER();

   if (DFSTORE_WRITE_ALLOWED)
   {
      TxPrint("Superblock update : 0x%2.2lx ", XFS_LSNSUP1);
      rc = dfsWrite(   XFS_LSNSUP1, 1, (BYTE   *) xfs->sup);
      TxPrint( "%s\n", (rc == NO_ERROR) ? "written" : "failed");
   }
   else
   {
      rc = DFS_READ_ONLY;
   }
   RETURN(rc);
}                                               // end 'dfsXfsWriteSuperBlocks'
/*---------------------------------------------------------------------------*/

