//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// DFSee utility & display services, non setboot stuff
//
// Author: J. van Wijk
//
// JvW  22-07-95   Initial version, split off from DHPFS.C
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsupart.h>                           // FS partition utilities
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS  navigation and defs
#include <dfsver.h>                             // DFS  version and naming
#include <dfsutil.h>                            // DFS  utility functions
#include <dfsulzw.h>                            // DFSee compression interface

#if defined (UNIX)
       char mapchar[] = ".-+=*!$@#\0";          // Map usage indicators
#else
       char mapchar[] = "\0";          // Map usage indicators
#endif


/*****************************************************************************/
// Determine disknumber from spec and option value; support -r, -g and -i too
/*****************************************************************************/
USHORT dfsGetDiskNumber                         // RET   disknr or FDSK_ANY
(
   char                option,                  // IN    option char to use
   char               *defspec,                 // IN    default spec (* or .)
   char               *spec,                    // IN    disk spec, "" or NULL
   BOOL                reread                   // IN    force reread diskinfo
)
{
   USHORT              rc ;                      // function return
   TXA_OPTION         *opt;
   TXTM                s1;

   ENTER();
   TRACES(( "option: '%c' defspec: '%s' spec: '%s'\n", option, defspec, spec));

   if ((opt = TxaOptValue( option)) != NULL)    // disk option specified
   {
      switch (opt->type)
      {
         case TXA_NO_VAL: strcpy(  s1, ".");                         break;
         case TXA_STRING: strcpy(  s1,         opt->value.string);   break;
         default:         sprintf( s1, "%lld", opt->value.number);   break;
      }
   }
   else if (spec && strlen(spec))
   {
      strcpy( s1, spec);
   }
   else
   {
      strcpy( s1, defspec);                     // no spec, current disk
   }
   #ifndef OEMSB
   if (reread || (dfsGetPartInfo(1) == NULL))   // -r or no diskinfo yet
   {
      USHORT           diskgeo = FDSK_QUIET;

      if ((TxaOption('g')) || TxaOption('m'))
      {
         switch (s1[0])
         {
            case '.':   diskgeo = SINF->disknr;          break;
            case '*':   diskgeo = FDSK_ANY;              break;
            default:    diskgeo = (USHORT) (atol(s1));   break;
         }
      }
      rc = dfsReadDiskInfo( diskgeo);
   }
   #endif
   rc = dfsParseDiskSpec( s1, NULL);
   RETURN (rc);
}                                               // end 'dfsGetDiskNumber'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Return valid disk-id for disk-specification;  <empty> | . | * | nr
// or the range FIRST disk plus LAST disk todo, when 'lastDisk' is not NULL
/*****************************************************************************/
USHORT dfsParseDiskSpec                         // RET   disk/FDSK_ANY or FIRST
(
   char               *spec,                    // IN    disk select spec
   USHORT             *lastDisk                 // INOUT last in range, 0=invalid
)                                               //       or NULL for single disknr
{
   USHORT              rc;                      // function return

   ENTER();
   TRARGS(("spec='%s'\n", spec));
   switch (spec[0])
   {
      case '.':                                 // .
         rc = 0;
         break;

      case 'b':                                 // special value: BMGR disk
      case 'B':
         rc = dfsa->bmgrDisk;
         break;

      case '\0':                                // empty
      case '*':                                 // or wildcard
      case '0':                                 // or ZERO (from ListBox)
         if (strlen(spec) <= 1)                 // unless multi-digit
         {
            rc = FDSK_ANY;
            break;
         }
      default:
         rc = (USHORT) atoi(spec);
         break;
   }
   if (rc == 0)                                 // replace zero by current
   {
      rc = (SINF->disknr) ? SINF->disknr : 1;   // default current disk or 1
   }
   if (lastDisk != NULL)                        // range output requested
   {
      if (rc == FDSK_ANY)                       // all disks
      {
         rc = 1;
         *lastDisk = dfsPartitionableDisks();
      }
      else if ((rc != 0) && (rc <= dfsPartitionableDisks())) // single disk, return as range
      {
         *lastDisk = rc;
      }
      else                                      // single disk, invalid number
      {                                         // rc will hold the invalid value
         *lastDisk = 0;                         // invalid single disk, return empty range
      }
   }
   RETURN (rc);
}                                               // end 'dfsParseDiskSpec'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read sector and Identify type of the sector
/*****************************************************************************/
BYTE dfsReadIdentifySector                      // RET   type of sector
(
   ULN64               lsn                      // IN    Sector LSN
)
{
   BYTE                rc = ST_UDATA;           // rc, sector type
   BYTE               *sec;                     // sector data

   ENTER();

   if ((sec = TxAlloc( 1, dfsGetSectorSize())) != NULL)
   {
      if (dfsRead( lsn, 1, sec) == NO_ERROR)
      {
         rc = dfsIdentifySector( lsn, 0, sec);
      }
      TxFreeMem( sec);
   }
   RETURN (rc);
}                                               // end 'dfsReadIdentifySector'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set allocation bits for filesystem for a range of sectors
/*****************************************************************************/
ULONG dfsSetAllocForRange
(
   ULN64               start,                   // IN    start sector number
   ULONG               size,                    // IN    size of range to set
   ULONG               clsize,                  // IN    granularity, sect/cl
   BOOL                set                      // IN    SET alloc (or reset)
)
{
   ULONG               rc = NO_ERROR;           // function return
   ULONG               i;

   ENTER();

   TRACES(( "%s start:0x%llx size:%8.8x spc:%8.8x\n", (set) ? "ALLOC" : "FREE ", start, size, clsize));

   for (i = 0; (i < size) && (rc == NO_ERROR); i += clsize)
   {
      rc = DFSFNCALL( dfsa->FsLsnSetAlloc,  start + i, 0, (char *) ((set) ? &set : NULL), NULL);
   }
   RETURN (rc);
}                                               // end 'dfsSetAllocForRange'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Check allocation bits for filesystem for a range of sectors
/*****************************************************************************/
ULONG dfsCheckAllocForRange
(
   ULN64               start,                   // IN    start sector number
   ULN64               size,                    // IN    size of range to check
   ULONG               clsize,                  // IN    granularity, sect/cl
   BOOL                set,                     // IN    check if SET (or free)
   ULN64              *bads                     // INOUT Nr of failing sectors
)
{
   ULONG               rc = NO_ERROR;           // function return
   ULONG               al;
   ULN64               i;

   ENTER();

   TRACES(( "Check if %s,start:0x%llx size:0x%llx spc:%8.8x\n", (set) ? "ALLOC" : "FREE ", start, size, clsize));

   for (i = 0; i < size; i += clsize)
   {
      al = DFSFNCALL( dfsa->FsLsnAllocated,  start + i, 0, NULL, NULL);
      if ((al && !set) || (!al && set))
      {
         if (bads != NULL)
         {
            *bads += clsize;                    // count failing sectors
         }
         rc = DFS_VALUE_ERROR;
      }
   }
   RETURN (rc);
}                                               // end 'dfsCheckAllocForRange'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Assemble string of repeated characters with leading and trailing strings
/*****************************************************************************/
void dfsRstr
(
   char               *string,                  // OUT   resulting string
   char               *lead1,                   // IN    leading string 1
   char               *lead2,                   // IN    leading string 2
   char               *single,                  // IN    string to repeat
   USHORT              count,                   // IN    repeat count
   char               *trail1,                  // IN    trailing text 1
   char               *trail2                   // IN    trailing text 2
)
{
   TX4K                multi;                   // Must be as large as TxPrint buffer!

   strcpy( multi, lead1);
   strcat( multi, lead2);
   while ( count--)
   {
      strcat( multi, single);
   }
   strcat( multi, trail1);
   strcat( multi, trail2);
   strcat( string, multi);
}                                               // end 'dfsRstr'
/*---------------------------------------------------------------------------*/


/*************************************************************************************************/
// Mangle 32-bit LVM-ID value using 3-digit HEX mangle string-value
/*************************************************************************************************/
ULONG dfsLvmIdMangle                            // RET   Mangled Id value
(
   ULONG               id,                      // IN    base Id value
   char               *ms                       // IN    Hex mangle string[3]
)
{
   ULONG               rc = (id & 0xfffff000);  // clear last 3 digits
   ULONG               ml = 0;                  // mangle value, ulong

   sscanf( ms, "%x", &ml);

   return (rc | ml);
}                                               // end 'dfsLvmIdMangle'
/*-----------------------------------------------------------------------------------------------*/


/*****************************************************************************/
// Copy SIZE data sectors from SRC to DEST, allowing overlapping SRC and DEST
/*****************************************************************************/
ULONG dfsCopySectorData
(
   ULN64               fLsn,                    // IN    FROM LSN
   ULN64               dLsn,                    // IN    DEST LSN
   ULONG               size                     // IN    SIZE in sectors
)
{
   ULONG               rc  = NO_ERROR;          // function result
   ULN64               from;
   ULN64               dest;
   ULONG               done;
   USHORT              bps  = dfsGetSectorSize(); // bytes per sector
   BYTE               *data = NULL;

   ENTER();
   TRACES(("Copy 0x%x sectors from 0x%llx to 0x%llx\n", size, fLsn, dLsn));

   data = TxAlloc( 1, bps);
   if (data != NULL)
   {
      #if defined (USEWINDOWING)
         if (txwIsWindow( TXHWND_DESKTOP))
         {
            dfsa->statusTimer = TxTmrSetTimer( TMR_SEC( 2)); // only after 2 seconds ...
         }
      #endif
      if (fLsn < dLsn)                          // to higher SN, copy Hi-to-Lo
      {
         from = fLsn + size -1;
         dest = dLsn + size -1;
      }
      else                                      // to lower  SN, copy Lo-to-Hi
      {
         from = fLsn;
         dest = dLsn;
      }
      for (done = 0; (done < size) && (rc == NO_ERROR); done++)
      {
         if ((rc = dfsRead( from, 1, data)) == NO_ERROR)
         {
            rc = dfsWrite(  dest, 1, data);
            if (fLsn < dLsn)                    // to higher SN, copy Hi-to-Lo
            {
               from--;
               dest--;
            }
            else                                // to lower  SN, copy Lo-to-Hi
            {
               from++;
               dest++;
            }
         }
         #if defined (USEWINDOWING)
            if (txwIsWindow( TXHWND_DESKTOP))
            {
               if (TxTmrTimerExpired( dfsa->statusTimer))
               {
                  TXTM    status;
                  sprintf( status, "Copy data area, sector %6u of %u", done + 1, size);
                  txwSetSbviewStatus( status, cSchemeColor);
                  dfsa->statusTimer = TxTmrSetTimer( DFSP_STATUS_INTERVAL);
               }
            }
         #endif
      }
      TxFreeMem( data);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsCopySectorData'
/*---------------------------------------------------------------------------*/

