//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// DFSee utility & display services, sector manipulation and run section
//
// Author: J. van Wijk
//
// JvW  18-08-2005 Initial version, split off from DFSUTIL.C
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfshpfs.h>                            // HPFS structure defs
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS  navigation and defs
#include <dfsver.h>                             // DFS  version and naming
#include <dfsutil.h>                            // DFS  utility functions
#include <dfsupart.h>                           // PART utility functions
#include <dfsufdsk.h>                           // FDSK utility functions


/*****************************************************************************/
// Locate copy of specified sector, search forward (2nd FAT from 1st)
/*****************************************************************************/
ULONG dfsLocateDuplicateSector                  // RET   LSN found or 0
(
   ULONG               start,                   // IN    start sector lsn
   ULONG               limit,                   // IN    max sectors to search
   ULONG               length                   // IN    length to match
)
{
   ULONG               rc = 0;                  // function return
   BYTE               *ref;
   BYTE               *sec;
   size_t              bps = (size_t) dfsGetSectorSize();

   ENTER();

   if (((ref = TxAlloc( 1, bps)) != NULL) &&
       ((sec = TxAlloc( 1, bps)) != NULL)  )
   {
      if (dfsRead( start, 1, ref) == NO_ERROR)
      {
         ULONG         lsn;
         size_t        slen = (size_t) min((size_t) length, bps);
         ULONG         dr = NO_ERROR;

         for ( lsn  =  start +1;
              (lsn <= (start + limit)) && !TxAbort() && (rc == 0) && (dr == NO_ERROR);
               lsn++)
         {
            dr = dfsRead( lsn, 1, sec);
            if (memcmp( ref, sec, slen) == 0)
            {
               rc = lsn;
            }
         }
      }
      TxFreeMem( ref);
      TxFreeMem( sec);
   }
   RETURN (rc);
}                                             // end 'dfsLocateDuplicateSector'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Locate a sector with specified string/signature at specified offset
/*****************************************************************************/
ULONG dfsLocateSignatureSector                  // RET   LSN found or 0
(
   ULONG               start,                   // IN    start sector lsn
   ULONG               limit,                   // IN    lsn range to search
   char               *sign,                    // IN    signature data
   ULONG               slen,                    // IN    signature length
   ULONG               soff                     // IN    signature offset
)
{
   ULONG               rc  = 0;                 // function return
   BYTE               *sec = NULL;

   ENTER();

   TRACES(("start: %8.8x  end: %8.8x,  offset: %u\n", start, limit, soff));
   TRHEXS(  70, sign, slen, "Signature");

   if ((sec = TxAlloc( 1, dfsGetSectorSize())) != NULL)
   {
      ULONG            lsn;
      ULONG            dr = NO_ERROR;

      for ( lsn =  start;
           (lsn < (start + limit)) && (rc == 0) && (dr == NO_ERROR);
            lsn++)
      {
         dr = dfsRead( lsn, 1, sec);
         if (memcmp( sec + soff, sign, (size_t) slen) == 0)
         {
            rc = lsn;
         }
      }
      TxFreeMem( sec);
   }
   RETURN (rc);
}                                             // end 'dfsLocateSignatureSector'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// Read sector(s) with BOOT.INI contents, display default line, optional update
/*****************************************************************************/
ULONG dfsCheckFixNtBootIni
(
   ULN64               sn,                      // IN    sector number
   char               *param,                   // IN    modification parameter
   BOOL                update2nd                // IN    update 2nd occurence too
)                                               //       and force update
{
   ULONG               rc = NO_ERROR;           // function return
   char               *p, *s, *t;               // search pointers in data
   int                 ll;                      // line length
   USHORT              pn = 0;                  // calculated part number
   USHORT              dn = 0;                  // default-line part number
   USHORT              fn = 0;                  // fixed partition number
   TXLN                arcPath;                 // as found in default
   char               *defPath;                 // default path string
   BOOL                arcFits = TRUE;          // no need to expand digits

   ENTER();

   //- Note: we will limit display and update to the first 510 bytes to avoid
   //-       corrupting MFT fixup values for internal data attributes.
   //-       boot.ini typically starts at 0x120, well within that limit.
   //-       The "default=" line always ends before 200 bytes into the file.

   if ((rc = dfsRead( sn, 1, rbuf)) == NO_ERROR)
   {
      TxPrint( "Default boot entry line in boot.ini ");
      if (((p = TxMemStr( rbuf, "default=", SECTORSIZE)) != NULL) &&
          ((s = TxMemStr( p, ")partition(", SECTORSIZE)) != NULL)  )
      {
         memset( arcPath, 0, TXMAXLN);

         if (SINF->p != NULL)                   // do we have a partition id ?
         {
            pn = SINF->p->diskPart;             // ARC style partition index
         }
         s += 11;                               // points to partition nr now
         ll = s - p + 10;                       // calculated line length
         if (*(s+1) != ')')                     // old is double digit
         {
            ll++;
         }
         TxPrint( "is:\n\n%s%*.*s%s\n\n", CBC, ll, ll, p, CNN);

         dn = atoi( s);                         // current partition-number

         TxPrint( "The partition-number in the default line is currently : %s%hu%s\n"
                  "To boot Win-NT/W2K/XP in THIS partition, it should be : %s%hu%s\n",
                           (dn == pn) ? CBC : CBR, dn, CNN, CBG, pn, CNN);

         if ((param != NULL) && (strlen(param) != 0))
         {
            sscanf( param, "%hu", &fn);

            defPath = p +8;                     // points to path component
            memcpy( arcPath, defPath, (s - defPath)); // copy arc-path, except P-nr
            TRACES(("arcPath: %s", arcPath));

            if (fn == 0)                        // non-numeric or zero
            {
               fn = pn;                         // use calculated value
            }
            if (fn < 10)                        // new is single digit
            {
               if (*(s+1) != ')')               // old was double digit
               {
                  *s++ = '0';                   // write leading zero
               }
               *s = fn + '0';                   // write the digit
            }
            else                                // new is double digit
            {
               if (*(s+1) != ')')               // old was double digit
               {
                  *s++ = (fn / 10) + '0';       // write high order digit
                  *s   = (fn % 10) + '0';       // write low  order digit
               }
               else                             // old was single digit
               {
                  arcFits = FALSE;              // will not update 2nd
                  if (((t = TxMemStr((char *) rbuf, "timeout=", SECTORSIZE)) != NULL) &&
                      ((p - t) == 12))          // 2-digit timeout in prev line
                  {
                     t   +=  8;                 // first digit of timeout
                     *t++ = '9';                // set to 9 seconds (1 digit)
                     memcpy( t, p -2, ll -8);   // move start of line forward
                     *s-- = (fn % 10) + '0';    // write low  order digit
                     *s   = (fn / 10) + '0';    // write high order digit
                  }
                  else                          // update not possible
                  {
                     TxPrint( "\nUpdate in-place of the partition-index is "
                              "NOT possible!\nUse another method like MS "
                              "'bootcfg /rebuild' to fix boot.ini\n\n");
                     rc = DFS_BAD_STRUCTURE;
                  }
               }
            }
            if ( (rc == NO_ERROR) &&            // OK and have a valid update
                ((dn != fn) || (update2nd)))    // or want to force updating ?
            {
               if (update2nd)                   // try 2nd occurence too
               {
                  if ((p = TxMemStr( s, arcPath, SECTORSIZE)) != NULL)
                  {
                     if (arcFits && ((p + strlen(arcPath) +2 - (char *) rbuf) < 510))
                     {
                        TRACES(( "2nd occurence digits, before: '%2.2s\n", p+strlen( arcPath)));
                        memcpy( p, defPath, strlen( arcPath) +2); // 2 digits at most
                        TRACES(( "2nd occurence digits, after : '%2.2s\n", p+strlen( arcPath)));
                     }
                     else
                     {
                        TxPrint("\nBOOT.INI 2nd occurence of the ARC-path NOT updated yet!\n");
                     }
                  }
               }
               if ((dfsa->batch) ||             // forced or confirmed
                   (TxConfirm( 5058,
                  "Write BOOT.INI file back, with default partition "
                  "index changed from %hu to %hu ? [Y/N] : ", dn, fn)))
               {
                  if (DFSTORE_WRITE_ALLOWED)
                  {
                     if ((rc = dfsWrite( sn, 1, rbuf)) == NO_ERROR)
                     {
                        TxPrint( "\nBOOT.INI default partition index changed "
                                   "from %hu to %hu\n\n", dn, fn);
                        if (update2nd)
                        {
                           TxPrint( "2nd occurence of the index in "
                                    "other line synchronized too!\n");
                        }
                     }
                  }
                  else
                  {
                     rc = DFS_READ_ONLY;
                  }
               }
               else
               {
                  rc = DFS_NO_CHANGE;           // Abort without changes
               }
            }
            else if (dn == pn)                    // index already as specified
            {
               TxPrint( "\nThe requested index %hu is the same as "
                        "already present\nin the BOOT.INI file, no "
                        "change is needed!\n\n", fn);
            }
         }
         else
         {
            TxPrint( "\nTo fix the value in the default BOOT.INI line, use the command\n"
                     "  'bootini fix'  or the corresponding menu-item:\n\n"
                     "   Mode=%s -> Fix BOOT.INI partition index\n", SINF->afsys);
            TxPrint( "\nFor the values needed to boot to ANOTHER partition, use the\n"
                       "value displayed in the 'BI' column of the 'part -s' display.\n\n");

            if (dn != pn)                       // report no-match in RC
            {
               rc = dn;                         // use found value for query
               dfsa->explain = FALSE;           // and don't give incorrect
            }                                   // explanations about this :-)
         }
      }
      else
      {
         TxPrint( "not found, file has non-standard format!\n\n");
      }
   }
   RETURN (rc);
}                                               // end 'dfsCheckFixNtBootIni'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Run the DFSDPROC script on specified disk, with any options
/*****************************************************************************/
ULONG dfsRunDfsDiskAnalysis                     // RET   function result
(
   USHORT              disk,                    // IN    disk number
   char               *precision,               // IN    ALL, CYL, MIB, BOTH or ""
   char               *cyls,                    // IN    #cyl or "?"
   char               *head,                    // IN    #heads
   char               *sect,                    // IN    #sectors/track
   char               *superfs,                 // IN    Superblock filesystem
   char               *basefn                   // IN    Base filename
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXLN                diskproc;                // filename/command buffer
   TXLN                logname = {0};           // old log filename
   TXLN                basename;
   FILE               *df;                      // diskfile ptr

   ENTER();
   TRACES(( "disk:%hu preci:'%s' cyls:'%s' head:'%s' sect:'%s' sup:'%s' base:'%s'\n",
             disk,    precision, cyls,     head,     sect,     superfs, basefn));

   if (*basefn)
   {
      strcpy( basename, basefn);
      TxStripExtension( basename);              // remove extension, if any
   }
   else if (toupper(*precision == 'A'))         // search all sectors
   {
      strcpy(  basename, DFSDISKSHORT "all");
   }
   else if ((*head) && (*head  != '.'))         // head number specified
   {
      sprintf( basename, DFSDISKSHORT "%3.3ld", atol( head));
   }
   else                                         // regular, CYL boundaries
   {
      strcpy( basename, DFSDISK_BASE);
   }
   sprintf( diskproc, "%s.sk%hu", basename, disk);
   if ((df = fopen( diskproc, "w")) != NULL)    // create new logfile
   {                                            // overwriting existing ...
      fprintf( df, "DFSee FULL disk analysis for disk: %hu\n\n", disk);
      fclose(  df);

      if (TxQueryLogName() != NULL)
      {
         strcpy( logname, TxQueryLogName());    // remember current name
      }

      TxAppendToLogFile( diskproc, TRUE);       // append to new logfile
      sprintf( diskproc, "about -Q -P-#runscript -B -p- -E:i dfsdproc.dfs %hu "
                         "%s %s %s %s %s \"%s\"", disk,
                          (*precision) ? precision : "cylinder",
                          (*cyls)      ? cyls      : ".",
                          (*head)      ? head      : ".",
                          (*sect)      ? sect      : ".", superfs, basename);

      rc = dfsMultiCommand( diskproc, 0, TRUE, FALSE, TRUE);

      TxAppendToLogFile( logname, TRUE);        // close, reopen old ...

      TxPrint("DFSDISK procedure : finished for disk: %hu\n", disk);
   }
   else
   {
      TxPrint( "\nDFSDISK analysis result logfile '%s%s%s' cannot be created!\n"
               "The drive/path could be invalid, write-protected or full ...\n", CBR, diskproc, CNN);
      rc = ERROR_WRITE_PROTECT;
   }
   RETURN(rc);
}                                               // end 'dfsRunDfsDiskAnalysis'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Run the DFSDFAST script on specified disk, with any options
/*****************************************************************************/
ULONG dfsRunDfsFastAnalysis                     // RET   function result
(
   USHORT              disk,                    // IN    disk number
   char               *basefn                   // IN    Base filename
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXLN                diskproc;                // filename/command buffer
   TXLN                logname = {0};           // old log filename
   TXLN                basename;
   FILE               *df;                      // diskfile ptr

   ENTER();
   TRACES(( "disk:%hu base:'%s'\n", disk, basefn));

   if (*basefn)
   {
      strcpy( basename, basefn);
      TxStripExtension( basename);              // remove extension, if any
   }
   else                                         // regular, CYL boundaries
   {
      strcpy( basename, DFSFAST_BASE);
   }
   sprintf( diskproc, "%s.sk%hu", basename, disk);
   if ((df = fopen( diskproc, "w")) != NULL)    // create new logfile
   {                                            // overwriting existing ...
      fprintf( df, "DFSee FAST disk analysis for disk: %hu\n\n", disk);
      fclose(  df);

      if (TxQueryLogName() != NULL)
      {
         strcpy( logname, TxQueryLogName());    // remember current name
      }

      TxAppendToLogFile( diskproc, TRUE);       // append to new logfile
      sprintf( diskproc, "about -Q -P-#runscript -B -p- -E:i dfsdfast.dfs %hu \"%s\"", disk, basename);

      rc = dfsMultiCommand( diskproc, 0, TRUE, FALSE, TRUE);

      TxAppendToLogFile( logname, TRUE);        // close, reopen old ...

      TxPrint("DFSFAST procedure : finished for disk: %hu\n", disk);
   }
   else
   {
      TxPrint( "\nDFSFAST analysis result logfile '%s%s%s' cannot be created!\n"
               "The drive/path could be invalid, write-protected or full ...\n", CBR, diskproc, CNN);
      rc = ERROR_WRITE_PROTECT;
   }
   RETURN(rc);
}                                               // end 'dfsRunDfsFastAnalysis'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Run the DOxxxx script on specified partition and optional filesystem
/*****************************************************************************/
ULONG dfsRunChkPartAnalysis                     // RET   function result
(
   USHORT              part,                    // IN    partition-ID
   char               *fsname                   // IN    optional fsys name
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXLN                chkpart;                 // filename/command buffer
   TXTS                fsys    = {0};           // filesystem name
   TXLN                logname = {0};           // old log filename
   DFSPARTINFO        *p;                       // partition info
   FILE               *df;                      // diskfile ptr

   ENTER();

   if (fsname && strlen(fsname))
   {
      strcpy( fsys, fsname);                    // use explicit filesystem
   }
   else
   {
      if ((p = dfsGetPartInfo( part)) != NULL)
      {
         switch (toupper(p->fsform[0]))
         {
            case 'H':                           // HPFS or HFS
               if (toupper(p->fsform[1]) == 'F')
               {
                      strcpy( fsys, "HFS");        break;
               }
               else
               {
                      strcpy( fsys, "HPFS");       break;
               }
            case 'E':
               if (toupper(p->fsform[1]) == 'F')
               {
                      strcpy( fsys, "EFAT");       break;
               }
               else
               {
                      strcpy( fsys, "EXT");        break;
               }
            case 'F': strcpy( fsys, "FAT");        break;
            case 'A': strcpy( fsys, "APFS");       break;
            case 'J': strcpy( fsys, "JFS");        break;
            case 'N': strcpy( fsys, "NTFS");       break;
            case 'R': strcpy( fsys, "RSR");        break;
            case 'S': strcpy( fsys, "SWAP");       break;
            case 'X': strcpy( fsys, "XFS");        break;
            default:  strcpy( fsys, "AUX");        break;
         }
      }
   }

   if (strlen(fsys))                            // filesystem set ?
   {
      TxStrToLower( fsys);                      // for lowercase file names!
      sprintf( chkpart, DFSDISKCHECK "%s.p%2.2hu", fsys, part);
      if ((df = fopen( chkpart, "w")) != NULL)  // create new logfile
      {
         fprintf( df, "DFSee CHKPART analysis for partition: %hu\n\n", part);
         fclose(  df);

         if (TxQueryLogName() != NULL)
         {
            strcpy( logname, TxQueryLogName()); // remember current name
         }
         TxAppendToLogFile( chkpart, TRUE);     // append to new logfile

         sprintf( chkpart, "about -Q -P-#part -B -Q -g ;select partition#"
                           "runscript -B -Q -p:1 -E:i dfs%s.dfs %hu ;may take long ... #"
                           "image -B -Q " DFSDISKCHECK "%s.b%2.2hu 0 1   ;bootsector #"
                           "image -B -Q " DFSDISKCHECK "%s.i%2.2hu 1 1600 -z ;LDR area",
                            fsys, part, fsys, part, fsys, part);

         rc = dfsMultiCommand( chkpart, 0, TRUE, FALSE, TRUE);

         TxAppendToLogFile( logname, TRUE);     // close, reopen old ...

         TxPrint("Finished analysis : CHKPART for partition: %hu\n", part);
      }
      else
      {
         TxPrint( "ChkPart result logfile '%s' cannot be created\n", chkpart);
         rc = ERROR_FILE_NOT_FOUND;
      }
   }
   else
   {
      TxPrint( "Partition-ID %hu is not a valid partition-number.\n", part);
   }
   RETURN(rc);
}                                               // end 'dfsRunChkPartAnalysis'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Fix Partition-bootsector HiddenSectors/geometry fields to match part-tables
/*****************************************************************************/
ULONG dfsFixPbrHsGeoFields
(
   BOOL                fixgeo,                  // IN    Fix Geometry
   BOOL                fixhs,                   // IN    Fix HiddenSectors
   BOOL                confirmed,               // IN    no confirmation
   BOOL                popup                    // IN    use message popup
)
{
   ULONG               rc  = NO_ERROR;
   BOOL                fix = FALSE;
   TXLN                m1  = {0};
   TXLN                m2  = {0};

   ENTER();

   if ((dfsa->FsModeId != DFS_FS_FDISK) && (SINF->p != NULL) && (dfsa->boot != NULL))
   {
      if ( (fixgeo) &&
          ((dfsa->boot->eb.LogGeoHead != SINF->p->geoHeads) ||
           (dfsa->boot->eb.LogGeoSect != SINF->p->geoSecs ) ))
      {
         TxPrint("\n Bootsector Geo-H : %-3lu         => %-3lu",
              dfsa->boot->eb.LogGeoHead, SINF->p->geoHeads);
         TxPrint("\n Bootsector Geo-S : %-3lu         => %-3lu",
              dfsa->boot->eb.LogGeoSect, SINF->p->geoSecs);
         dfsa->boot->eb.LogGeoHead     = SINF->p->geoHeads;
         dfsa->boot->eb.LogGeoSect     = SINF->p->geoSecs;
         sprintf( m1, "Bootsector geometry updated to %u heads and %u sectors to "
                      "match the current geometry in effect for the disk.\n\n",
                       SINF->p->geoHeads, SINF->p->geoSecs);
         fix = TRUE;
      }
      if ((fixhs) &&
          (dfsa->boot->eb.HiddenSectors != SINF->p->partent.BootSectorOffset))
      {
         TxPrint("\n HiddenSectors    : 0x%8.8x  => 0x%8.8x",
             dfsa->boot->eb.HiddenSectors, SINF->p->partent.BootSectorOffset);
         dfsa->boot->eb.HiddenSectors    = SINF->p->partent.BootSectorOffset;
         sprintf( m2, "HiddenSectors field updated to value %8.8x to "
                      "match the related partition-table offset field.\n\n",
                       dfsa->boot->eb.HiddenSectors);
         fix = TRUE;
      }
      if (fix)
      {
         TxPrint( "\n");
         if (dfsa->batch || confirmed ||
            (TxConfirm( 5133, "%s%sReplace sector 0, the partition bootsector "
                     "with the fixed one ? [Y/N] : ", m1, m2)))
         {
            if (DFSTORE_WRITE_ALLOWED)
            {
               rc = dfsWrite( LSN_BOOTR, 1, (BYTE *) dfsa->boot);
               if (rc == NO_ERROR)
               {
                  TxPrint("\nBootrecord successfully updated\n");
               }
            }
            else
            {
               rc = DFS_READ_ONLY;
            }
         }
         else
         {
            TxPrint("\nCommand aborted, no changes made\n");
            rc = DFS_NO_CHANGE;
         }
      }
      else
      {
         sprintf( m1, "HiddenSectors and Geometry fields match "
                      "partition-tables, no fix needed");
         if (popup)
         {
            TxNamedMessage( !(dfsa->batch || confirmed), 0, " INFO: No fix needed ", m1);
         }
         else
         {
            TxPrint( "\n%s\n", m1);
         }
      }
   }
   else                                         // not enough info
   {
      TxPrint("\nNo valid partition and filesystem selected (Mode=xxx), you need to\n"
                "select a partition properly to allow fixing its bootsector values!\n");
      rc = DFS_CMD_FAILED;
   }
   RETURN(rc);
}                                               // end 'dfsFixPbrHsGeoFields'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set volume-serial number in the Partition-bootsector (FAT, FAT32, HPFS ...)
/*****************************************************************************/
ULONG dfsSetVolumeSerialNr
(
   BOOL                popup,                   // IN    use message popup
   char               *newVsn                   // IN    new vsn, "" or NULL
)
{
   ULONG               rc  = NO_ERROR;
   ULN64               vsn = 0;                 // new value
   ULN64               vso = 0;                 // old value
   TXLN                m1  = {0};
   BOOL                updateRequested = FALSE;
   int                 digits = 8;

   ENTER();

   if (((dfsa->FsModeId == DFS_FS_FAT )  ||
        (dfsa->FsModeId == DFS_FS_HPFS)  ||
        (dfsa->FsModeId == DFS_FS_NTFS)  ||
        (dfsa->FsModeId == DFS_FS_E_FAT) ||
        (dfsa->FsModeId == DFS_FS_JFS )) &&
        (dfsa->boot != NULL))
   {
      if (dfsa->FsModeId == DFS_FS_NTFS)
      {
         digits = 16;
         vso = dfsa->boot->nt.SerialNr;
      }
      else if (dfsa->FsModeId == DFS_FS_E_FAT)
      {
         vso = TXmku64( dfsa->boot->ex.SerialNr, 0);
      }
      else if (strncasecmp( dfsa->boot->f2.Type, "FAT32", 5) == 0)
      {
         vso = TXmku64( dfsa->boot->f2.SerialNr, 0);
      }
      else
      {
         vso = TXmku64( dfsa->boot->os.SerialNr, 0);
      }
      if ((newVsn != NULL) && (strlen( newVsn) != 0))
      {
         sscanf( newVsn, "%llx", &vsn);         // get specified hex value
      }
      else                                      // determine default value
      {
         vsn = dfstLSN2Psn( DFSTORE, 0) + 0xee000000; // batch default value
         #if defined (USEWINDOWING)
            if (!dfsa->batch)
            {
               vsn = vso;                       // use old as default value
            }
         #endif
      }

      if (!dfsa->batch)
      {
         #if defined (USEWINDOWING)             // Allow interactive update
         if (popup && txwIsWindow( TXHWND_DESKTOP))
         {
            TXLN      prompt;
            TXTT      value;

            sprintf( prompt, "%s\n\nSerial Number now: %*.*llx\n\n"
                             "Specify %d-digit hexadecimal Serial Number\n",
                              dfstStoreDesc1( DFSTORE) + 10, digits, digits, vso, digits);
            sprintf( value,  "%*.*llx", digits, digits, vsn);

            if (txwPromptBox( TXHWND_DESKTOP, TXHWND_DESKTOP, NULL,
                   prompt, " Set volume serial number in bootrecord ",
                   5161, TXPB_MOVEABLE | TXPB_HCENTER | TXPB_VCENTER,
                   digits + 3, value) != TXDID_CANCEL)
            {
               updateRequested = TRUE;
               sscanf( value, "%llx", &vsn);    // get specified hex value
            }
         }
         else
         {
            updateRequested = TRUE;             // no-popup, confirmed
         }
         #endif
      }
      else
      {
         updateRequested = TRUE;                // batch mode
      }

      if (updateRequested)
      {
         if (dfsa->FsModeId == DFS_FS_NTFS)
         {
            dfsa->boot->nt.SerialNr = vsn;
         }
         else if (dfsa->FsModeId == DFS_FS_E_FAT)
         {
            dfsa->boot->ex.SerialNr = TXu64lo( vsn);
         }
         else if (strncasecmp( dfsa->boot->f2.Type, "FAT32", 5) == 0)
         {
            dfsa->boot->f2.SerialNr = TXu64lo( vsn);
         }
         else
         {
            dfsa->boot->os.SerialNr = TXu64lo( vsn);
         }
         sprintf( m1, "Volume Serial Number updated to value 0x%*.*llx\n\n", digits, digits, vsn);
         TxPrint( "\n");
         if (dfsa->batch || popup || (TxConfirm( 5161,
             "%sUpdate sector 0, the partition bootsector with this change ? [Y/N] : ", m1)))
         {
            if (DFSTORE_WRITE_ALLOWED)
            {
               rc = dfsWrite( LSN_BOOTR, 1, (BYTE *) dfsa->boot);
               if (rc == NO_ERROR)
               {
                  TxPrint("Bootrecord serial : 0x*.*%llx written to the bootsector\n", digits, digits, vsn);
               }
            }
            else
            {
               rc = DFS_READ_ONLY;
            }
         }
         else
         {
            dfsRead( LSN_BOOTR, 1, (BYTE *) dfsa->boot);                     // undo change in in-memory bootrec
            TxPrint("\nCommand aborted, no changes made\n");
            rc = DFS_NO_CHANGE;
         }
      }
   }
   else                                         // not enough info
   {
      TxPrint("\nNo valid partition or filesystem selected (Mode=xxx), you need to\n"
                "select a partition or volume that allows changing the volume serial nr!\n");
      rc = DFS_CMD_FAILED;
   }
   RETURN(rc);
}                                               // end 'dfsSetVolumeSerialNr'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set volume-label string(11) in the Partition-bootsector (FAT, FAT32, HPFS)
/*****************************************************************************/
ULONG dfsSetVolumeLabelBrec
(
   BOOL                popup,                   // IN    use message popup
   char               *newVln                   // INOUT new label, "" or NULL
)
{
   ULONG               rc  = NO_ERROR;
   TXTS                vln = {0};               // new value
   TXTS                vlo = {0};               // old value
   TXLN                m1  = {0};
   BOOL                updateRequested = FALSE;

   ENTER();

   if (((dfsa->FsModeId == DFS_FS_FAT )  ||
        (dfsa->FsModeId == DFS_FS_HPFS)  ||
        (dfsa->FsModeId == DFS_FS_JFS )) &&
        (dfsa->boot != NULL))
   {
      if (strncasecmp(  dfsa->boot->f2.Type, "FAT32", 5) == 0)
      {
         TxCopy( vlo, dfsa->boot->f2.Label, BT_LABL_L + 1);
      }
      else
      {
         TxCopy( vlo, dfsa->boot->os.Label, BT_LABL_L + 1);
      }
      if ((newVln != NULL) && (strlen( newVln) != 0))
      {
         TxCopy( vln, newVln, BT_LABL_L + 1);
      }
      else                                      // determine default value
      {
         sprintf( vln, "Dsk%huPart%hu", SINF->disknr, SINF->partid);  // batch default value
         #if defined (USEWINDOWING)
            if (!dfsa->batch)
            {
               strcpy( vln, vlo);               // use old as default value
            }
         #endif
      }

      if (!dfsa->batch)
      {
         #if defined (USEWINDOWING)             // Allow interactive update
         if (popup && txwIsWindow( TXHWND_DESKTOP))
         {
            TXLN      prompt;

            sprintf( prompt, "%s\n\nVolume label, current value: '%s'\n\n"
                             "Specify new %d-character volume label\n",
                              dfstStoreDesc1( DFSTORE) + 10, vlo, BT_LABL_L);

            if (txwPromptBox( TXHWND_DESKTOP, TXHWND_DESKTOP, NULL,
                   prompt, " Set volume label in bootrecord ",
                   5162, TXPB_MOVEABLE | TXPB_HCENTER | TXPB_VCENTER,
                   BT_LABL_L, vln) != TXDID_CANCEL)
            {
               updateRequested = TRUE;
            }
         }
         else
         {
            updateRequested = TRUE;             // no-popup, confirmed
         }
         #endif
      }
      else
      {
         updateRequested = TRUE;                // batch mode
      }

      if (updateRequested)
      {
         TxStrip( vln, vln, ' ', ' ');          // strip leading/trailing spaces
         if (strncasecmp( dfsa->boot->f2.Type, "FAT32", 5) == 0)
         {
            memcpy( dfsa->boot->f2.Label, vln, BT_LABL_L);
         }
         else
         {
            memcpy( dfsa->boot->os.Label, vln, BT_LABL_L);
         }
         sprintf( m1, "Volume Label updated to '%s'\n\n", vln);
         TxPrint( "\n");
         if (dfsa->batch || popup || (TxConfirm( 5162,
             "%sUpdate sector 0, the partition bootsector with this change ? [Y/N] : ", m1)))
         {
            if (DFSTORE_WRITE_ALLOWED)
            {
               rc = dfsWrite( LSN_BOOTR, 1, (BYTE *) dfsa->boot);
               if (rc == NO_ERROR)
               {
                  TxPrint("New  Volume label : '%s' written to the bootsector\n", vln);

                  if (newVln != NULL)           // output parameter ?
                  {
                     strcpy( newVln, vln);      // return new label
                  }
               }
            }
            else
            {
               rc = DFS_READ_ONLY;
            }
         }
         else
         {
            dfsRead( LSN_BOOTR, 1, (BYTE *) dfsa->boot);                     // undo change in in-memory bootrec
            TxPrint("\nCommand aborted, no changes made\n");
            rc = DFS_NO_CHANGE;
         }
      }
      else
      {
         rc = DFS_USER_ABORT;
      }
   }
   else                                         // not enough info
   {
      TxPrint("\nNo valid partition or filesystem selected (Mode=xxx), you need to\n"
                "select a partition or volume that allows changing the volume label!\n");
      rc = DFS_CMD_FAILED;
   }
   RETURN(rc);
}                                               // end 'dfsSetVolumeLabelBrec'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Test if supplied buffer is RLE compressable for a single byte value
/*****************************************************************************/
BOOL dfsRleCompressable                         // RET   single byte value
(
   BYTE               *buf,                     // IN    sector buffer
   ULONG               bytes                    // IN    size in bytes
)
{
   BOOL                rc = TRUE;               // function return
   BYTE                pbyte;                   // pattern byte
   ULONG               bfour;                   // 4-byte pattern
   ULONG               size = bytes / sizeof(ULONG);
   ULONG              *data = (ULONG *) buf;
   ULONG               i;

   ENTER();

   pbyte = *buf;
   bfour = pbyte | (pbyte << 8) | (pbyte << 16) | (pbyte << 24);

   TRACES(("size: %u ULONGS, pattern: %8.8x\n", size, bfour));

   for (i = 0; i < size; i++, data++)           // check whole buffer
   {
      if (*data != bfour)
      {
         rc = FALSE;
         break;
      }
   }
   BRETURN (rc);
}                                               // end 'dfsRleCompressable'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Test file-existence, prompt user to change a removable medium for read
/*****************************************************************************/
BOOL dfsChangeReadMedium                        // RET   File (now) present
(
   char               *lead,                    // IN    leading info text
   char               *filename                 // IN    full path and filename
)
{
   BOOL                rc = TRUE;               // function return
   TX1K                s1;                      // large message text

   ENTER();

   while (TxFileExists( filename) == FALSE)     // Needed file not there ?
   {
      sprintf( s1, "%s%s\n\nThis required file was not found. "
                   "When the drive is a removable, change the medium "
                   "to the one containing the requested file.\n\n"
                   "Retry on changed removable medium now ? [Y/N]: ",
                    lead, filename);
      if (!TxConfirm( 5217, s1))
      {
         rc = FALSE;                            // No/Cancel, file not there
         break;
      }
   }
   BRETURN (rc);
}                                               // end 'dfsChangeReadMedium'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Prompt user to change a removable medium for write, create PATH when needed
/*****************************************************************************/
BOOL dfsChangeWriteMedium                       // RET   New medium and path OK
(
   char               *filename,                // IN    full path and filename
   LLONG               maxsize                  // IN    max size to be written
)
{
   BOOL                rc = FALSE;              // function return
   TXLN                s1;                      // message text
   TXTM                size;                    // size value
   TXLN                path;                    // path component
   char               *s;
   BOOL                makepath = FALSE;

   ENTER();

   strcpy( path, filename);
   TxRepl( path, FS_PALT_SEP, FS_PATH_SEP);     // fixup ALT separators
   if ((s = strrchr( path, FS_PATH_SEP)) != NULL)
   {
      *s = 0;                                   // terminate path
      if ((path[1] != ':') || (strlen(path) > 2))
      {
         makepath = TRUE;
      }
   }

   strcpy( size, "");
   dfstrSXiB( size, "Maximum space required is : ", (maxsize / dfsGetSectorSize()), "");

   sprintf( s1, "%s\n\nThis file is about to be created/written. "
                "When the drive is a removable, change the medium "
                "to a new (empty) one if needed.\n\n%s\n\n"
                "Continue with create/write now ? [Y/N]: ", filename, size);

   while ((rc == FALSE) && (TxConfirm( 5215, s1)))
   {
      if (!makepath || (TxMakePath( path) == NO_ERROR))
      {
         rc = TRUE;                             // Yes, and path OK
      }
      else                                      // adjust msg to MakePath fail
      {
         sprintf( s1, "%s\n\nCreating the path for this file failed!\n\n"
                      "Make sure no write-protection is active and the medium "
                      "is not full, change to a new (empty) one if needed.\n\n%s\n\n"
                      "Continue with create/write now ? [Y/N]: ", filename, size);
      }
   }
   BRETURN (rc);
}                                               // end 'dfsChangeWriteMedium'
/*---------------------------------------------------------------------------*/

