//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// DFSee utility & display services, get input strings section
//
// Author: J. van Wijk
//
// JvW  18-08-2005 Initial version, split off from DFSUTIL.C
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfshpfs.h>                            // HPFS structure defs
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS  navigation and defs
#include <dfsver.h>                             // DFS  version and naming
#include <dfsulzw.h>                            // DFSee compression interface
#include <dfsutil.h>                            // DFS  utility functions
#include <dfsupart.h>                           // PART utility functions
#include <dfsufdsk.h>                           // FDSK utility functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfstable.h>                           // SLT  utility functions
#include <dfsdgen.h>                            // generic dialogs interface


/*****************************************************************************/
// Select partition or volume from specified options or parameter
// if no parameter, and part/vol is already selected, don't do anything!
/*****************************************************************************/
BOOL dfsSelectPartVol                           // RET   part/volume selected
(
   char               *param,                   // IN    selection parameter
   char               *optstr                   // IN    select-cmd option string
)
{
   BOOL                rc = FALSE;              // function return

   TXTM                s1;
   TXA_OPTION         *opt;                     // option pointer

   ENTER();

   if (TxaOptMutEx((( param != NULL) && (strlen(param))),
                     "pvi", "if 'part' or 'vol' is specified!\n", 0))
   {
      s1[0] = 0;
      if ((param != NULL) && (strlen(param)))
      {
         if ((strncasecmp( param, "a:", 2) == 0) ||
             (strncasecmp( param, "b:", 2) == 0)  )
         {
            sprintf( s1, "vol  %s", param);
         }
         else
         {
            sprintf( s1, "part %s", param);
         }
      }
      else if ((opt = TxaOptValue('p')) != NULL) // from a Partition
      {
         DFSDISKINFO  *d  = dfsGetDiskInfo( dfsGetDiskNumber( 'd', ".", "", FALSE));

         switch (opt->type)
         {
            case TXA_STRING:
               sprintf( s1, "part %s",  opt->value.string);
               break;

            default:
               sprintf( s1, "part %llu", opt->value.number);
               break;
         }
         if ((opt->unit == 'r') && (d) && (d->firstpart))
         {
            strcat( s1, ",r");                  // make it a relative PID
         }
      }
      else if (TxaOption('v'))                  // from a Volume
      {
         sprintf( s1, "vol %s", TxaOptStr('v', "Vol. letter", "C:"));
      }
      else if (TxaOption('i'))                  // from an Image
      {
         sprintf( s1, "im %s",  TxaOptStr('i', "ImgFilename", ""));
      }
      else                                      // no param or switch
      {
         strcpy( s1, "");                       // do nothing at all
      }
      if (strlen(s1))
      {
         if (optstr != NULL)
         {
            strcat( s1, " ");
            strcat( s1, optstr);
         }
         dfsMultiCommand( s1, 0, TRUE, FALSE, TRUE);
      }
      if ( (SINF->partid   !=  0 ) ||
           ((strlen(SINF->drive))          &&
                   (SINF->drive[0] != '-') &&
                   (SINF->drive[0] != ' ')))
      {
         rc = TRUE;                             // partition or volume selected
      }
      TRACES(("partid: %hu, drive: '%s'\n", SINF->partid, SINF->drive));
   }
   BRETURN (rc);
}                                               // end 'dfsSelectPartVol'
/*---------------------------------------------------------------------------*/


/*************************************************************************************************/
// Get desired object type based on specified options, or current open object
/*************************************************************************************************/
DFSOBJECT dfsGetObjectType                      // RET   DISK/PART/VOLD/IMGF
(
   DFST_HANDLE         store,                   // IN    related store
   char               *opts                     // IN    option letter string
)
{
   DFSOBJECT           rc = DFSO_NONE;          // default current open object
   DFSISTORE          *is = NULL;
   char               *s;

   ENTER();

   dfstQueryStoreType( store, &is, NULL);
   if (is != NULL)
   {
      rc = is->object;                          // open object in store
   }
   if (opts && strlen(opts))
   {
      for (s = opts; *s; s++)                   // walk option letters
      {
         if (TxaOption( *s))                    // option set ?
         {
            switch (*s)
            {
               case 'd': case 'D': rc = DFSO_DISK; break;
               case 'p': case 'P': rc = DFSO_PART; break;
               case 'v': case 'V': rc = DFSO_VOLD; break;
               case 'i': case 'I': rc = DFSO_IMGF; break;
            }
            break;                              // out of loop
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsGetObjectType'
/*-----------------------------------------------------------------------------------------------*/


/*************************************************************************************************/
// Get desired object type, and test if object value (to be opened) is specified
/*************************************************************************************************/
BOOL dfsGetObjectValue                          // RET   value specified
(
   DFST_HANDLE         store,                   // IN    related store
   char               *opts,                    // IN    option letter string
   DFSOBJECT          *object                   // OUT   DISK/PART/VOLD/IMGF etc
)
{
   BOOL                rc  = FALSE;
   TXA_OPTION         *opt = NULL;
   DFSISTORE          *is  = NULL;
   char               *s;

   ENTER();

   dfstQueryStoreType( store, &is, NULL);
   if (is != NULL)
   {
      if (is->object == DFSO_NONE)              // nothing opened
      {
         *object = SINF->object;                // default to same object as
      }                                         // the current store
      else
      {
         *object = is->object;                  // open object in store
      }
   }
   if (opts && strlen(opts))                    // explicit object overrule ?
   {
      TRACES(( "*object: %u  opts: '%s'\n", *object, opts));
      for (s = opts; *s; s++)                   // walk option letters
      {
         if ((opt = TxaOptValue( *s)) != NULL)  // option set ?
         {
            switch (*s)
            {
               case 'd': case 'D': *object = DFSO_DISK; break;
               case 'p': case 'P': *object = DFSO_PART; break;
               case 'v': case 'V': *object = DFSO_VOLD; break;
               case 'i': case 'I': *object = DFSO_IMGF; break;
               default:            *object = DFSO_NONE; break;
            }
            if (*object != DFSO_NONE)
            {
               switch (opt->type)
               {
                  case TXA_STRING: rc = TRUE;                     break;
                  case TXA_NUMBER: rc = (opt->value.number != 0); break;
                  default:                                        break;
               }
            }
            break;                              // out of loop
         }
      }
      TRACES(( "*object: %u\n", *object));
   }
   BRETURN (rc);
}                                               // end 'dfsGetObjectValue'
/*-----------------------------------------------------------------------------------------------*/


/*************************************************************************************************/
// Get desired object 'open command', based on specified options
// 20181119 JvW: Removed 'ALLOC' calculation/display not really needed and slow!
/*************************************************************************************************/
ULONG dfsGetObjectOpenCmd                       // RET   result
(
   char               *opts,                    // IN    option letter string
   char               *cmd                      // OUT   command to execute
)                                               //       or empty when none
{
   ULONG               rc = NO_ERROR;
   char               *s;
   TXA_OPTION         *opt;                     // option pointer

   ENTER();
   TRACES(( "option letters: '%s'\n", opts));

   strcpy( cmd, "");
   if (opts && strlen(opts))
   {
      if (TxaOptMutEx( FALSE, opts, "", NULL))  // check if just one option
      {
         for (s = opts; *s; s++)                // walk option letters
         {
            if ((opt = TxaOptValue(*s)) != NULL) // option set ?
            {
               TRACES(( "Option '%c' is set, type is: %d\n", *s, opt->type));
               switch (opt->type)
               {
                  case TXA_NO_VAL:
                     TRACES(( "TXA_NO_VAL\n"));
                     rc = DFS_NO_CHANGE;
                     break;

                  case TXA_STRING:
                     TRACES(( "TXA_STRING: '%s'", opt->value.string));
                     if (strlen( opt->value.string) > 0)
                     {
                        switch (*s)
                        {
                           case 'p': case 'P': sprintf( cmd, "part -Q -q -a- %s", opt->value.string); break;
                           case 'v': case 'V': sprintf( cmd, "vol  -Q    -a- %s", opt->value.string); break;
                           case 'i': case 'I': sprintf( cmd, "im   -Q -B -R- %s", opt->value.string); break;
                           default:  rc = DFS_VALUE_ERROR;                                            break;
                        }
                     }
                     else
                     {
                        rc = DFS_VALUE_ERROR;   // unsupported combination
                     }
                     break;

                  default:
                     TRACES(( "default:     %u'", opt->value.number));
                     if (opt->value.number != 0)
                     {
                        switch (*s)
                        {
                           case 'd': case 'D': sprintf( cmd, "disk -Q -q %llu",     opt->value.number); break;
                           case 'p': case 'P': sprintf( cmd, "part -Q -q -a- %llu", opt->value.number); break;
                           default:  rc = DFS_VALUE_ERROR;                                              break;
                        }
                     }
                     else
                     {
                        rc = DFS_VALUE_ERROR;   // unsupported combination
                     }
                     break;
               }
               TRACES(( "Option '%c' cmd:'%s' rc:%u\n", *s, cmd, rc));
               break;                           // out of loop
            }
         }
      }
      else
      {
         rc = DFS_VALUE_ERROR;                  // unsupported combination
      }
   }
   TRACES(( "Open command: '%s'\n", cmd));
   RETURN (rc);
}                                               // end 'dfsGetObjectOpenCmd'
/*-----------------------------------------------------------------------------------------------*/


/*************************************************************************************************/
// Set desired object CURRENT by opening it, based on specified options
/*************************************************************************************************/
ULONG dfsSetObjectCurrent                       // RET   result
(
   char               *opts,                    // IN    option letter string
   BOOL                verbose                  // IN    show command executing
)
{
   ULONG               rc;                      // result
   TXLN                command;

   ENTER();

   switch (rc = dfsGetObjectOpenCmd( opts, command))
   {
      case NO_ERROR:
         if (strlen( command) != 0)             // got valid open command
         {
            rc = dfsMultiCommand( command, 0, verbose, FALSE, verbose);
            if (verbose && (rc != NO_ERROR))
            {
               switch (rc)
               {
                  case DFS_BAD_STRUCTURE:
                     TxPrint( "\nOpened object '%s' has no valid contents (yet)\n", command);
                     break;

                  default:
                     TxPrint( "\nOpening the object using '%s' ended with a warning/error!\n", command);
                     break;
               }
            }
         }
         break;

      case DFS_NO_CHANGE:                       // no value specified
         rc = NO_ERROR;                         // no cmd to execute, but OK
         break;
   }
   RETURN (rc);
}                                               // end 'dfsSetObjectCurrent'
/*-----------------------------------------------------------------------------------------------*/


/*****************************************************************************/
// Get a clean string value from -a, -u or -h option, or from mixed string
/*****************************************************************************/
USHORT dfsGetMixedString
(
   char               *mixed,                   // IN    mixed string or empty
   char               *descr,                   // IN    usage description
   BOOL                prompt,                  // IN    prompt when no input
   BOOL                upper,                   // IN    options AHUM not ahum
   char               *string                   // OUT   clean string value
)
{
   USHORT              rc = 0;                  // function return
   BOOL                raw_value = (mixed) && (strlen(mixed));
   TXLN                s0, s1;
   TXTS                optchar;
   TXA_OPTION         *opt;

   ENTER();

   strcpy( optchar, (upper) ? "AHUM" : "ahum");
   TRACES(("opts:%s mixed: >>%s<<   descr:'%s'\n", optchar, mixed, descr));

   if (TxaOptMutEx( raw_value, optchar, "if a value is specified!", 0))
   {
      if (raw_value)                            // raw value specified
      {
         rc = TxFormatMixedStr( mixed, string);
      }
      else if ((opt = TxaOptValue(optchar[0])) != NULL) // ASCII format
      {
         switch (opt->type)
         {
            case TXA_NO_VAL: strcpy(  s0, "");                          break;
            case TXA_STRING: strcpy(  s0,         opt->value.string);   break;
            default:         sprintf( s0, "%lld", opt->value.number);   break;
         }
         if ((strlen(s0) == 0) || TxaOption('P'))
         {
            TxPrompt( 5503, TXMAXLN, s0, "ASCII %s string", descr);
         }
         strcpy( string, s0);
         rc = (USHORT) strlen( s0);
      }
      else if ((opt = TxaOptValue(optchar[1])) != NULL) // hex format
      {
         switch (opt->type)
         {
            case TXA_NO_VAL: strcpy(  s1, "");                          break;
            case TXA_STRING: strcpy(  s1, opt->value.string);           break;
            default:
               if      (opt->value.number <= (ULN64) 0xff)
               {
                  sprintf( s1, "%2.2llx", opt->value.number);
               }
               else if (opt->value.number <= (ULN64) 0xffff)
               {
                  sprintf( s1, "%4.4llx", opt->value.number);
               }
               else if (opt->value.number <= (ULN64) 0xffffffff)
               {
                  sprintf( s1, "%8.8llx", opt->value.number);
               }
               else
               {
                  sprintf( s1, "%16.16llx", opt->value.number);
               }
               break;
         }
         if ((strlen(s1) == 0) || TxaOption('P'))
         {
            TxPrompt( 5504, TXMAXLN, s1, "HEX %s string", descr);
         }
         sprintf( s0, "'%s'",   s1);
         rc = TxFormatMixedStr( s0, string);
      }
      else if ((opt = TxaOptValue(optchar[2])) != NULL) // UNICODE format
      {
         switch (opt->type)
         {
            case TXA_NO_VAL: strcpy(  s1, "");                          break;
            case TXA_STRING: strcpy(  s1,         opt->value.string);   break;
            default:         sprintf( s1, "%lld", opt->value.number);   break;
         }
         if ((strlen(s1) == 0) || TxaOption('P'))
         {
            TxPrompt( 5505, TXMAXLN, s1, "UNICODE %s string", descr);
         }
         sprintf( s0, "\"%s\"", s1);
         rc = TxFormatMixedStr( s0, string);
      }
      else if (TxaOptSet(optchar[3]) || prompt) // no data specified, prompt
      {
         if ((opt = TxaOptValue(optchar[3])) != NULL)  // MIXED format
         {
            switch (opt->type)
            {
               case TXA_NO_VAL: strcpy(  s0, "");                          break;
               case TXA_STRING: strcpy(  s0,         opt->value.string);   break;
               default:         sprintf( s0, "%lld", opt->value.number);   break;
            }
         }
         else
         {
            strcpy(  s0, "");
         }
         if ((strlen(s0) == 0) || TxaOption('P'))
         {
            TxPrompt( 5507, TXMAXLN, s0, "ASCII/mixed %s string", descr);
         }
         rc = TxFormatMixedStr( s0, string);
      }
   }
   TRACES(("string: '%s'\n", s0));
   RETURN (rc);
}                                               // end 'dfsGetMixedString'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Determine specified or default-optimal buffersize in sectors, upto a limit
/*****************************************************************************/
ULONG dfsGetBufferSize                          // RET   number of sectors
(
   ULONG               defSize,                 // IN    default size, sectors
   ULONG               limit                    // IN    limit size, sectors
)
{
   ULONG               rc = defSize;             // when NOT specified at all
   TXA_OPTION         *opt;

   ENTER();
   TRACES(("defSize: %u  limit: %u\n", defSize, limit));

   if ((opt = TxaOptValue('b')) != NULL)        // when -b option specified
   {
      TRACES(("-b option used, type: %hhu\n", opt->type));
      switch (opt->type)
      {
         case TXA_NUMBER: rc = (ULONG) opt->value.number; break; // as specified
         case TXA_STRING:
         default:         rc = dfstGeoSectors( DFSTORE);  break; // tracksize
      }
   }
   if (rc > limit)                              // would overflow buffer ...
   {
      rc = limit;                               // limit to maximum
      TxPrint("\nBuffer size for '-b' option limited to %u sectors\n", limit);
   }
   RETURN (rc);
}                                               // end 'dfsGetBufferSize'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get a symbolic or mcs-number start sector-number like 3f . this or .3
/*****************************************************************************/
ULN64 dfsGetSymbolicSN                          // RET   sector-number
(
   char               *spec,                    // IN    symbolic or hex SN
   ULN64               defsn                    // IN    default SN for no spec
)
{
   ULN64               rc = defsn;              // function return
   ULN64               ei;

   ENTER();
   TRACES(( "spec:'%s' defsn:0x%llx\n", spec, defsn));

   if      (strcmp( spec, ".") == 0)
   {
      rc = nav.this;
   }
   else if (strcmp( spec, "$") == 0)
   {
      rc = 0;                                   // maximum size, start at 0
   }
   else if (strlen( spec) > 0)
   {
      if (strchr( "+-.", spec[0]) != NULL)
      {
         if (sscanf( spec +1, "%llu", &ei) == 1)
         {
            switch (spec[0])
            {
               case '+': rc = nav.this + ei; break;
               case '-':
                  if (ei < nav.this)
                  {
                     rc = nav.this - ei; break;
                  }
                  else
                  {
                     rc = 0;
                  }
                  break;


               default:
                  if (ei < dfsa->snlist[0])
                  {
                     rc = dfsa->snlist[ei+1];
                  }
                  break;
            }
         }
      }
      else                                      // symbolic values
      {
         if      (strcasecmp( spec, "this" ) == 0)
         {
            rc = nav.this;
         }
         else if (strcasecmp( spec, "down" ) == 0)
         {
            rc = nav.down;
         }
         else if (strcasecmp( spec, "next" ) == 0)
         {
            rc = nav.down;
         }
         else if (strcasecmp( spec, "up"   ) == 0)
         {
            rc = nav.up;
         }
         else if (strcasecmp( spec, "xtra" ) == 0)
         {
            rc = nav.xtra;
         }
         else if (strcasecmp( spec, "extra") == 0)
         {
            rc = nav.xtra;
         }
         else                                   // must be a number
         {
            BYTE                unit;

            //- Interpret sector (and CL/BL) values as HEX by default
            rc = TxaParseNumber( spec, TX_RADIX_UN_S_CLASS, &unit);
            rc = dfsApplyNumberUnit( rc, unit, DFSTORE);
         }
      }
   }
   RETN64 (rc);
}                                               // end 'dfsGetSymbolicSN'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get a symbolic or mcs-number size value like 3f . this or .3
/*****************************************************************************/
ULN64 dfsGetSymbolicSize
(
   char               *spec,                    // IN    symbolic or hex SN
   ULN64               defsz,                   // IN    default size
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               base                     // IN    base value for relative
)                                               //       values like $ and .
{
   ULN64               rc = defsz;              // function return
   ULN64               ei;
   BYTE                unit;

   ENTER();

   if      (strcmp( spec, ".") == 0)
   {
      rc = nav.this - base +1;                  // upto and including 'this'
   }
   else if (strcmp( spec, "$") == 0)
   {
      rc = dfstGetLogicalSize( store) - base;   // full size
   }
   else if (strlen( spec) > 0)
   {
      if (spec[0] == '.')
      {
         if ((sscanf( spec +1, "%llu", &ei) == 1) && (ei < dfsa->snlist[0]))
         {
            rc = dfsa->snlist[ei+1];
         }
      }
      else                                      // symbolic values
      {
         if (spec[0] == '.')
         {
            if ((sscanf( spec +1, "%llu", &ei) == 1) && (ei < dfsa->snlist[0]))
            {
               if (base <= dfsa->snlist[ei+1])
               {
                  rc = dfsa->snlist[ei+1] - base;
               }
            }
         }
         else                                   // symbolic values
         {
            if      (strcasecmp( spec, "this" ) == 0)
            {
               rc = nav.this - base +1;         // upto and including 'this'
            }
            else if ((strcasecmp( spec, "all" ) == 0) ||
                     (strcasecmp( spec, "end" ) == 0)  )
            {
               rc = dfstGetLogicalSize( store) - base; // full size
            }
            else
            {
               rc = TxaParseNumber( spec, DFSRADIX_SIZE, &unit);
               rc = dfsApplyNumberUnit( rc, unit, store);
            }
         }
      }
   }
   RETN64 (rc);
}                                               // end 'dfsGetSymbolicSize'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get an mcs-number in HEX, OCTAL or DECIMAL
// JvW 28-03-2010: honor supplied default value!
/*****************************************************************************/
ULN64 dfsGetMcsNumber
(
   char               *spec,                    // IN    Hex/Oct/Dec mcs-number
   ULN64               defsz                    // IN    default size
)
{
   ULN64               rc = defsz;              // function return
   BYTE                unit;

   ENTER();
   TRACES(("spec: '%s' defsz: %llx\n", spec, defsz));

   if (strlen( spec) != 0)
   {
      rc = TxaParseNumber( spec, DFSRADIX_SIZE, &unit);
      rc = dfsApplyNumberUnit( rc, unit, DFSTORE);
   }
   RETN64 (rc);
}                                               // end 'dfsGetMcsNumber'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Apply specified number-unit, returning corrected number value; default 's'
// JvW 18-04-2017: Added Tera, Exa and Peta-byte unit specifiers
/*****************************************************************************/
ULN64 dfsApplyNumberUnit                        // RET   unit corrected value
(
   ULN64               value,                   // IN    uncorrected value
   char                unit,                    // IN    unit, k m g c h s
   DFST_HANDLE         store                    // IN    DFS store for GEO
)
{
   ULN64               rc     = value;          // function return
   USHORT              bpsect = dfstGetSectorSize( store);

   ENTER();
   TRACES(("Input : 0x%llx  unit: %c\n", value, unit));

   switch (tolower(unit))
   {
      case 'c': rc *= dfstCylinderSize( store);       break; // cylinders
      case 'h': rc *= dfstGeoSectors(   store);       break; // heads/tracks
      case 'k': rc  = DFSKB2SECTS( rc, bpsect);       break; // Kilobytes
      case 'p': rc *= 1024;                     // no break!    Peta factor
      case 'e': rc *= 1024;                     // no break!    Exa  factor
      case 't': rc *= 1024;                     // no break!    Tera factor
      case 'g': rc *= 1024;                     // no break!    Giga factor
      case 'm': rc  = DFSMB2SECTS( rc, bpsect);       break; // Mega/Giga/Tera/Exea/Peta
      default:                                        break; // sectors, OK
   }
   RETN64 (rc);
}                                               // end 'dfsApplyNumberUnit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get a dotted-decimal number, up to 20 positions (as created by UllDot20)
/*****************************************************************************/
ULN64 dfsGetUllDot20                            // RET   64-bit unsigned number
(
   char               *spec                     // IN    Dotted decimal, 20 pos
)
{
   ULN64               rc = 0;                  // function return
   char               *s = spec;

   ENTER();
   TRACES(("spec: 0x%p = '%s'\n", spec, spec));

   if (strlen( spec) != 0)
   {
      while (*s == ' ')                         // skip leading spaces
      {
         s++;
      }
      while (*s && ((*s == '.') || (isdigit(*s))))
      {
         if (*s != '.')
         {
            rc = rc * 10 + (*s - '0');
         }
         TRACES(("s: 0x%p = '%c'   rc = %llu\n", s, *s, rc));
         s++;
      }
   }
   RETN64 (rc);
}                                               // end 'dfsGetUllDot20'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get chsStyle from specified option or from the dfsa global value
/*****************************************************************************/
ULONG dfsGetChsStyle                            // RET   chsStyle value
(
   char                ch,                      // IN    character
   TXHANDLE            txtype,                  // IN    option or switch
   BOOL                batch                    // IN    no interactive popups
)
{
   ULONG               rc = dfsa->chsStyle;     // default function return
   TXA_OPTION         *opt;                     // option pointer

   ENTER();

   if ((opt = TxaGetOption(txtype, ch)) != NULL) // chs option/switch ?
   {
      switch (opt->type)
      {
         case TXA_STRING:
            if ((opt->value.string[0] != '!') && (opt->value.string[0] != '?'))
            {
               switch (opt->value.string[0])
               {
                  case 'P': case 'p': rc = DFS_CHSTYLE_PQ;  break; // PowerQuest
                  case 'M': case 'm': rc = DFS_CHSTYLE_MS;  break; // Microsoft
                  default:            rc = DFS_CHSTYLE_IBM; break; // IBM / DFSee
               }
               break;
            }
         #if defined (USEWINDOWING)
         case TXA_NO_VAL:                       // option set, but no value yet
            if (!batch)                         // batch will use global default
            {
               if (dfsChsStyleDialog((opt->type == TXA_STRING), &rc)
                   != DFS_NO_CHANGE)
               {
                  TXTS newchs;                  // set -c opt at current level

                  sprintf( newchs, "-c:%u", rc);
                  TxaOptSetItem( newchs);
               }
            }
            break;
         #endif

         default:
            rc = opt->value.number;
            break;
      }
   }
   RETURN (rc);
}                                               // end 'dfsGetChsStyle'
/*---------------------------------------------------------------------------*/

#if defined (USEWINDOWING)
/*****************************************************************************/
// Prompt for new LSN value based on user selections (windowed only)
// 20170818 JvW Removed update for the clustersize, confusing in HEXED
/*****************************************************************************/
ULONG dfsGetGotoLsnValue
(
   char               *title,                   // IN    Dialog title
   ULN64              *sn                       // OUT   new sector number
)
{
   ULONG               rc = NO_ERROR;           // function return
   ULN64               numvalue  = *sn;         // numeric result
   TXLN                message;
   TXTM                chslimit;
   TXTM                mcsvalue;
   TXTM                strvalue  = {0};         // raw string value
   static DFSMODE      lastMode  = DFS_FS_AUX;  // to set cluster default
   static BOOL         isHexadec = TRUE;
   static BOOL         isCluster = FALSE;
   BOOL                doCluster = ((dfsa->FsModeId == DFS_FS_NTFS ) ||
                                    (dfsa->FsModeId == DFS_FS_FAT  ) ||
                                    (dfsa->FsModeId == DFS_FS_E_FAT) ||
                                    (dfsa->FsModeId == DFS_FS_JFS  ) ||
                                    (dfsa->FsModeId == DFS_FS_HFS  ) ||
                                    (dfsa->FsModeId == DFS_FS_XFS  ) ||
                                    (dfsa->FsModeId == DFS_FS_EXT  ) );
   ENTER();

   if (dfsa->FsModeId != lastMode)              // set new default for FS mode
   {
      lastMode  = dfsa->FsModeId;
      isCluster = doCluster;                    // make it the default too
   }
   sprintf(    message, "Specify a single sector (or %s) number to go to",
                                       (doCluster) ? "block/cluster" : "MCS-style");
   if (dfsa->FsModeId == DFS_FS_FDISK)
   {
      sprintf( chslimit, "'C H S' values: 0..%5u 0..%3u 1..%2u",
               dfstGeoCylinders(    DFSTORE) -1,
               dfstGeoHeads(        DFSTORE) -1,
               dfstGeoSectors(      DFSTORE));
      strcat(  message, ",\nor three decimal ");
      strcat(  message, chslimit);
   }
   while ((dfsPromptOptDialog( title, message, 5518, 49, strvalue,
          &isHexadec, "Default single number format is HEXADECIMAL (MCS-number style)",
         (doCluster) ? &isCluster : NULL, "Convert value from CLUSTER to sector number",
          NULL, NULL, NULL, NULL, NULL, NULL)) == NO_ERROR)
   {
      if (strchr( strvalue, ' '))               // spaces found ==> assume CHS
      {
         ULONG         c = 0;
         ULONG         h = 0;
         ULONG         s = 1;

         sscanf( strvalue, "%u %u %u", &c, &h, &s);

         if ((c <  dfstGeoCylinders( DFSTORE)) &&
             (h <  dfstGeoHeads(     DFSTORE)) &&
             (s <= dfstGeoSectors(   DFSTORE)) && (s != 0))
         {
            numvalue = dfstPsn2LSN( DFSTORE, dfstChs2Psn( DFSTORE, c, h, s));
         }
         else
         {
            TxNamedMessage( !dfsa->batch, 0, " ERROR: Invalid CHS values ",
                            "CHS out of range, valid range is \n\n%s", chslimit);
            continue;
         }
      }
      else                                      // some form of sector/cluster/block
      {
         if (isxdigit( strvalue[0])    &&       // valid (hexa) decimal
               strcmp( strvalue, "0x") &&
               strcmp( strvalue, "0t")  )       // no explicit hex/dec MCS
         {
            strcpy( mcsvalue, (isHexadec) ? "0x" : "0t");
         }
         else
         {
            strcpy( mcsvalue, "");
         }
         strcat( mcsvalue, strvalue);
         numvalue = dfsGetSymbolicSN( mcsvalue, numvalue);
      }

      if (isCluster && (strchr( strvalue, ',') == NULL))
      {
         *sn = dfsCl2Sn( numvalue);
      }
      else                                      // MCS type number or explicit
      {
         *sn = numvalue;                        // note: do NOT touch cluster
      }
      if (*sn < dfsGetLogicalSize())
      {
         break;                                 // OK, leave prompt loop
      }
      else
      {
         TxNamedMessage( !dfsa->batch, 0, " ERROR: Invalid sectornumber ", "LSN out of range: 0x%llx", *sn);
      }
   }
   RETURN (rc);
}                                               // end 'dfsGetGotoLsnValue'
/*---------------------------------------------------------------------------*/
#endif
