//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// DFSee utility & display services
//
// Author: J. van Wijk
//
// JvW  17-06-95   Initial version
//
#ifndef    DFSUTIL
   #define DFSUTIL

#define ST__INFO   0x80                         // Info only, no searching
                                                // S   used on searching
                                                //  I  identification only
#define ST_WILDC   '*'                          // SI  any type
#define ST_KNOWN   '+'                          // S   any known type
#define ST_FINAL  ('.' + ST__INFO)              //  I  End-of-Partition marker
#define ST_EMPTY   '0'                          //     ZERO  cleared sector 00
#define ST_BADFE   '5'                          //     DFSee marked  sector FE (bad)
#define ST_FDISK   '6'                          //     FDISK cleared sector F6
#define ST_BOOT2  ('7' + ST__INFO)              //     FAT32 2nd bootsector (2=FAT2)
#define ST_BOOT3  ('9' + ST__INFO)              //     FAT32 3rd bootsector (3=FAT12)
#define ST_FREES   '@'                          //  I  Free space
#define ST_MCDDM   'A'                          //     MAC driver description
#define ST_LUKSH   'C'                          //     LUKS Crypto Header sector
#define ST_GPTHD   'G'                          //     GPT part table header
#define ST_UDATA   'H'                          //     unidentified data sector
#define ST_FDATA  ('I' + ST__INFO)              //     file data extent
#define ST_LVSIG   'L'                          //     LVM BBR  signature sector
#define ST_LVDLT   'P'                          //     LVM drive-linking table
#define ST_LVBBT   'Q'                          //     LVM bad-block rel table
#define ST_RESRV  ('R' + ST__INFO)              //  I  reserved area
#define ST_ULZ16  ('T' + ST__INFO)              //     DFSee compressed, small buffer
#define ST_ULZ32  ('U' + ST__INFO)              //     DFSee compressed, large buffer
#define ST_LVREM   'V'                          //     LVM BBR deleted volume
#define ST_BADSC  ('X' + ST__INFO)              //  I  Bad sector
#define ST_MCDPM   'a'                          //     MAC disk partition map
#define ST_BOOTR   'b'                          //     Filesystem Boot sector
#define ST_EXTBR   'e'                          //     Extended Boot record
#define ST_GPTAS   'g'                          //     GPT part table array sector
#define ST_HDATA  ('h' + ST__INFO)              //     Hex   format half sector
#define ST_LVINF   'l'                          //     LVM DLAT info sector
#define ST_BOOTX  ('n' + ST__INFO)              //     non standard boot rec
#define ST_LVDLF   'p'                          //     LVM drive-linking feature
#define ST_LVBBF   'q'                          //     LVM bad-block rel feature
#define ST_MASTR   'r'                          //     Master Boot record
#define ST_TDATA  ('t' + ST__INFO)              //     Text  format data sector
#define ST_CORES   'v'                          //     Apple Core Storage header
#define ST_MINIM  ('~' + ST__INFO)              //  I  info only, no contents

//- Special sectortype for filesystem-superblocks, independant of current FSMODE
#define ST_SEXT    DFS_FS_EXT
#define ST_SHPFS   DFS_FS_HPFS
#define ST_SHFS    DFS_FS_HFS
#define ST_SAPFS   DFS_FS_APFS
#define ST_SJFS    DFS_FS_JFS
#define ST_SNTFS   DFS_FS_NTFS
#define ST_SRSR    DFS_FS_RSR
#define ST_SXFS    DFS_FS_XFS

//- seperator strings used to make a symbolic-link or hard-link stand out from filename itself
#define DFS_SH_LINK_STR " -> "


// partial base-name (start) definitions for DFSDISK, DFSFAST and DFSCHECK result files
#if   defined (WIN32)
   #define DFSDISK_BASE  "DFSDISKw"
   #define DFSDISKSHORT  "DFSDw"
   #define DFSFAST_BASE  "DFSFASTw"
   #define DFSFASTSHORT  "DFSFw"
   #define DFSDISKCHECK  "DFSw"
#elif defined (DOS32)
   #define DFSDISK_BASE  "DFSDISKd"
   #define DFSDISKSHORT  "DFSDd"
   #define DFSFAST_BASE  "DFSFASTd"
   #define DFSFASTSHORT  "DFSFd"
   #define DFSDISKCHECK  "DFSd"
#elif defined (LINUX)
   #define DFSDISK_BASE  "DFSDISKl"
   #define DFSDISKSHORT  "DFSDl"
   #define DFSFAST_BASE  "DFSFASTl"
   #define DFSFASTSHORT  "DFSFl"
   #define DFSDISKCHECK  "DFSl"
#elif defined (DARWIN)
   #define DFSDISK_BASE  "DFSDISKm"
   #define DFSDISKSHORT  "DFSDm"
   #define DFSFAST_BASE  "DFSFASTm"
   #define DFSFASTSHORT  "DFSFm"
   #define DFSDISKCHECK  "DFSm"
#else
   #define DFSDISK_BASE  "DFSDISKo"
   #define DFSDISKSHORT  "DFSDo"
   #define DFSFAST_BASE  "DFSFASTo"
   #define DFSFASTSHORT  "DFSFo"
   #define DFSDISKCHECK  "DFSo"
#endif

// SN-list file pragmas for Meta-info lines
#define    DFSNIFSMODE   ";;Mode="
#define    DFSNIFSMCMD   ";;Mcmd="
#define    DFSNLISTCOM   ";;Lcompact="
#define    DFSNLISTVER   ";;Ldetail="
#define    DFSNLISTDRV   ";;Ldrive="
#define    DFSNLISTDSK   ";;Ldisk="
#define    DFSNLISTPAR   ";;Lpart="

extern char  mapchar[];                         // Map usage indicators

//- various signature string values, null terminated
extern char sg_lvinf[];                         // LVM info signature
extern char sg_lvsig[];                         // LVM BBR, signature
extern char sg_lvrem[];                         // LVM BBR, removed
extern char sg_lvdlf[];                         // LVM drive-link sector
extern char sg_lvdlt[];                         // LVM drive-link table
extern char sg_lvbbf[];                         // LVM bad-block  sector
extern char sg_lvbbt[];                         // LVM bad-block  table

extern char sg_luksh[];                         // LUKS crypto header

extern char sg_mcddm[];                         // MAC driver description
extern char sg_mcdpm[];                         // MAC disk partition map


// Calculate size in GiB for OS/2 65535 cylinder limit and given GEO values
ULONG dfs16bitCylLimitGiB                       // RET   nr of GiBs for limit
(
   ULONG               heads,                   // IN    Number of heads in GEO
   ULONG               sects                    // IN    Sectors per track in GEO
);

// Return the number of sectors to align partitions on when CReating them
ULONG dfsAlignmentSectors                       // RET   #sectors for CR alignment
(
   char                fspKind,                 // IN    G(pt), P(rim), L(ogical)
   DFSPARTINFO        *f                        // IN    freespace info (geo)
);


// Return the number of sectors to to use as 'GAP' value when CReating them
// Note: overruled by -G:n option, sets default based on specified alignment
ULONG dfsDefaultGapSectors                      // RET   #sectors default GAP value
(
   char                fspKind,                 // IN    G(pt), P(rim), L(ogical)
   DFSPARTINFO        *f                        // IN    freespace info (geo)
);

// Display and return specified information item
ULONG dfsQueryValue                             // RET   queried value
(
   char               *query,                   // IN    item specification
   char               *param                    // IN    argument
);

// Show pseudo graphical map of disk(s) with partition and freespace info
void dfsShowMap
(
   USHORT              dnr,                     // IN    disknr, 0 == all disks
   char                level                    // IN    detail level '-', '+'
);

// Print several DFSee status-line items to the current TxPrintf destination
void dfsTxPrintStatusLine
(
   ULONG               dr                       // IN    dfs return-code
);

// Read sector and Identify type of the sector
BYTE dfsReadIdentifySector                      // RET   type of sector
(
   ULN64               lsn                      // IN    Sector LSN
);

// Identify type of specified sector
BYTE dfsIdentifySector                          // RET   type of sector
(
   ULN64              lsn,                      // IN    Sector LSN
   USHORT              sninfo,                  // IN    info (fat entry, ext index)
   BYTE              *sec                       // IN    sector data
);


// Test if the given MBR/EBR sector has 4 valid entries, with at least one used
BOOL dfsValidXbrTableEntries                    // RET   sector likely is xBR
(
   BYTE              *sec                       // IN    sector data
);

// Locate copy of specified sector, search forward (like find 2nd FAT from 1st)
ULONG dfsLocateDuplicateSector                  // RET   LSN found or 0
(
   ULONG               start,                   // IN    start sector lsn
   ULONG               limit,                   // IN    max sectors to search
   ULONG               length                   // IN    length to match
);

// Locate a sector with specified string/signature at specified offset
ULONG dfsLocateSignatureSector                  // RET   LSN found or 0
(
   ULONG               start,                   // IN    start sector lsn
   ULONG               limit,                   // IN    lsn range to search
   char               *sign,                    // IN    signature data
   ULONG               slen,                    // IN    signature length
   ULONG               soff                     // IN    signature offset
);

// Read sector(s) with BOOT.INI contents, display default line, optional update
ULONG dfsCheckFixNtBootIni
(
   ULN64               sn,                      // IN    sector number
   char               *param,                   // IN    modification parameter
   BOOL                update2nd                // IN    update 2nd occurence too
);                                              //       and force update

// Display specified sectortype in ASCII
void dfsShowSectorType
(
   BYTE               st                        // IN    sector type
);

// Display all known sectortypes with type-char and description in ASCII
void dfsDisplaySectorTypes
(
   void
);

// Create text-array with known sectortypes with type-char and description
char **dfsSectorTypes2Text                      // RET   TXlib style text array
(                                               //       or NULL for alloc fail
   void
);

// Display single sectortype with type-char and colored description in ASCII
void dfsSectorTypeInColor
(
   char              *lead,                     // IN    leading string
   BYTE               st                        // IN    sector type
);

// Return specified sectortype in ASCII
void dfsSectorTypeAsAscii
(
   BYTE               st,                       // IN    sector type
   char              *buf                       // OUT   text buffer
);

// Read, analyse and display sector(s)
ULONG dfsReadAnDisplay
(
   ULN64               lsn,                     // IN    Start LSN
   ULN64               info,                    // IN    FAT entry/EXT Inode/NTFS parent
   BYTE              *stype                     // OUT   sector type found
);

// Read. analyse and display sector(s) based on Physical addressing
ULONG dfsReadDisplayPhysical
(
   ULN64               psn,                     // IN    Start PSN
   BYTE               *stype                    // INOUT sector type
);

// Display sector(s) based on Physical addressing and known (forced) type
ULONG dfsDisplaySector
(
   ULN64              psn,                      // IN    Start PSN
   BYTE               stype,                    // IN    sector/display type
   BYTE              *sect                      // IN    sector data
);

// Show current sector contents as ascii text
ULONG dfsSectorAsAscii
(
   ULN64               sn,                      // IN    lsn of sector
   ULONG               size                     // IN    Size to display
);

// Show current sector contents hexadecimal
ULONG dfsSectorAsHex
(
   ULN64               sn,                      // IN    lsn of sector
   ULONG               size,                    // IN    Size to display
   BOOL                asUlong                  // IN    format as ULONG
);

// Format for Display attributes (FAT); append to string
void dfstrFatAttrib
(
   char               *str,                     // INOUT text string
   BYTE                data                     // IN    data
);

// Check if 128-bit UUID or GUID value in 38 character string is valid
BOOL dfsUidStringIsValid                       // RET   UUID/GUID string valid
(
   char               *uidStr                   // IN    UID String (38 chars)
);

// Return (Linux) 128-bit UID value in 38 char (static) string, endian-swap
char *dfsFsUuidValueString                      // RET   Formatted UUID (static)
(
   BYTE               *uuid                     // IN    UUID data, 16 bytes
);

// Return 128-bit UUID/GUID value in 38 char (static) string, no endian-swap
char *dfsFsUuidValueNoSwap                      // RET   Formatted UUID (static)
(
   BYTE               *uuid                     // IN    UUID data, 16 bytes
);

// Format (Linux) 128-bit UID value in 38 character string; append to string
void dfstrFsUuidValue
(
   char               *str,                     // INOUT UID text string, 38
   BYTE               *uuid,                    // IN    UID binary, 16 bytes
   BOOL                swap                     // IN    perform endian swaps
);

// Convert UUID/GUID string (38) into 16 byte GUID, including byte-swaps
// JvW: 06-02-2018, selectable order first 3 groups, little-endian swap/no-swap
ULONG dfsUidStringToBinary                      // RET   conversion result
(
   char               *str,                     // IN    UID text string, 38
   BYTE               *uuid,                    // OUT   UID binary, 16 bytes
   BOOL                swap                     // IN    perform endian swaps
);

// Display ULONG size in 4-hex digits and Dec KiB/MiB/GiB; to TxPrint output
void dfsUlSiz4
(
   char               *text,                    // IN    leading string
   ULONG               data,                    // IN    data
   char               *trail                    // IN    trailing text
);

// Format ULONG size in 4-hex digits and Dec KiB/MiB/GiB; append to string
void dfstrUlSiz4
(
   char                *str,                    // OUT   resulting string
   char                *text,                   // IN    leading string
   ULONG                data,                   // IN    data
   char                *trail                   // IN    trailing text
);

// Display (sector) size in KiB/MiB/GiB/TiB; to TxPrint output (8 positions)
void dfsSize
(
   char               *text,                    // IN    leading string
   ULONG               data,                    // IN    data
   char               *trail                    // IN    trailing text
);

// Format 64-bit size in Byte/KiB/MiB/GiB/TiB; append to string (8 positions)
char *dfstr64XiB                                // RET   resulting string
(
   char               *str,                     // OUT   resulting string
   char               *text,                    // IN    leading string
   ULN64               data,                    // IN    data (nr of bytes)
   char               *trail                    // IN    trailing text
);

// Display ULONG in decimal DOTTED format up to 4GB, to Tprintf (13 positions)
void dfsUlDot13
(
   char               *text,                    // IN    leading string
   ULONG               data,                    // IN    data (nr of bytes)
   char               *trail                    // IN    trailing text
);

// Format ULONG in DOTTED format up to 4GB; append to string (13 positions)
char *dfstrUlDot13                              // RET   resulting string
(
   char               *str,                     // OUT   resulting string
   char               *text,                    // IN    leading string
   ULONG               data,                    // IN    data (nr of bytes)
   char               *trail                    // IN    trailing text
);

// Display ULL in decimal DOTTED format up to 999Tb, else raw (20 positions)
void dfsUllDot20
(
   char               *text,                    // IN    leading string
   ULN64               data,                    // IN    data (nr of bytes)
   char               *trail                    // IN    trailing text
);

// Format ULL in DOTTED format if < 999Tb else raw; app to str (20 positions)
char *dfstrUllDot20                             // RET   resulting string
(
   char               *str,                     // OUT   resulting string
   char               *text,                    // IN    leading string
   ULN64               data,                    // IN    data (nr of bytes)
   char               *trail                    // IN    trailing text
);

// Format (byte/sect) size Dec KiB/MiB/GiB/TiB; append to string (8 positions)
#define dfstrBytes(s,l,d,t) dfstrS4XiB((s),(l),(d),1,(t))
#define dfstrSize(s,l,d,t)  dfstrS4XiB((s),(l),(d),dfsGetSectorSize(),(t))
char *dfstrS4XiB                                // RET   resulting string
(
   char               *str,                     // OUT   resulting string
   char               *text,                    // IN    leading string
   ULONG               data,                    // IN    data (sectors or bytes)
   USHORT              bps,                     // IN    bytes per sector, or 1)
   char               *trail                    // IN    trailing text
);

// Display (sector) size in 8-hex digits and KiB/MiB/GiB; to TxPrintf output
#define dfsSiz8(l,d,t) dfsSizeBps((l),(d),dfsGetSectorSize(),(t))
void dfsSizeBps
(
   char               *text,                    // IN    leading string
   ULONG               data,                    // IN    data
   USHORT              bps,                     // IN    bytes per sector
   char               *trail                    // IN    trailing text
);

// Format (sector) size in 8-hex digits and KiB/MiB/GiB; append to string
#define dfstrSiz8(s,l,d,t) dfstrSizeBps((s),(l),(d),dfsGetSectorSize(),(t))
char *dfstrSizeBps                              // RET   resulting string
(
   char               *str,                     // OUT   resulting string
   char               *text,                    // IN    leading string
   ULONG               data,                    // IN    data
   USHORT              bps,                     // IN    bytes per sector
   char               *trail                    // IN    trailing text
);

// Format byte size in 13-hex digits + KiB/MiB/GiB/TiB;  to TxPrint (27)
void dfsSz64Byte
(
   char               *text,                    // IN    leading string
   ULN64               data,                    // IN    data
   char               *trail                    // IN    trailing text
);

// Format byte size in 13-hex digits + KiB/MiB/GiB/TiB; append to string (27)
char *dfstrSz64Byte                             // RET   resulting string
(
   char               *str,                     // OUT   resulting string
   char               *text,                    // IN    leading string
   ULN64               data,                    // IN    data
   char               *trail                    // IN    trailing text
);

// Format sector size in 10-hex digits + KiB/MiB/GiB/TiB;  to TxPrint (24)
#define dfsSz64(l,d,t) dfsSz64Bps((l),(d),dfsGetSectorSize(),(t))
void dfsSz64Bps
(
   char               *text,                    // IN    leading string
   ULN64               data,                    // IN    data
   USHORT              bps,                     // IN    bytes per sector
   char               *trail                    // IN    trailing text
);

// Format sector size in 10-hex digits + KiB/MiB/GiB/TiB; append to string (24)
#define dfstrSz64(s,l,d,t) dfstrSz64Bps((s),(l),(d),dfsGetSectorSize(),(t))
char *dfstrSz64Bps                              // RET   resulting string
(
   char               *str,                     // OUT   resulting string
   char               *text,                    // IN    leading string
   ULN64               data,                    // IN    data
   USHORT              bps,                     // IN    bytes per sector
   char               *trail                    // IN    trailing text
);

// Format (sector) size in 7.1 value + KiB/MiB/GiB total 11; to TxPrintf output
void dfsSizeXiB
(
   char               *text,                    // IN    leading string
   ULONG               data,                    // IN    data
   USHORT              bps,                     // IN    bytes per sector
   char               *trail                    // IN    trailing text
);

// Format (sector) size in 7.1 value + KiB/MiB/GiB total 11; append to string
#define dfstrSXiB(s,l,d,t) dfstrSizeXiB(s,l,d,dfsGetSectorSize(),t)
char *dfstrSizeXiB                              // RET   resulting string
(
   char               *str,                     // OUT   resulting string
   char               *text,                    // IN    leading string
   ULONG               data,                    // IN    data
   USHORT              bps,                     // IN    bytes per sector
   char               *trail                    // IN    trailing text
);

// Format (sect64) size in 7.1 value + KiB/MiB/GiB/TiB total 11 TxPrint output
void dfsSz64XiB
(
   char               *text,                    // IN    leading string
   ULN64               data,                    // IN    data
   USHORT              bps,                     // IN    bytes per sector
   char               *trail                    // IN    trailing text
);

// Format (sect64) size in 7.1 value + KiB/MiB/GiB/TiB total 11 append to str
#define dfstrS64XiB(s,l,d,t) dfstrSz64XiB(s,l,d,dfsGetSectorSize(),t)
char *dfstrSz64XiB                              // RET   resulting string
(
   char               *str,                     // OUT   resulting string
   char               *text,                    // IN    leading string
   ULN64               data,                    // IN    data
   USHORT              bps,                     // IN    bytes per sector
   char               *trail                    // IN    trailing text
);

// Display value in 8-hex digits and decimal; to TxPrintf output
void dfsDec8
(
   char               *text,                    // IN    leading string
   ULONG               data,                    // IN    data
   char               *trail                    // IN    trailing text
);

// Format value in 8-hex digits and 10 decimals; 23 positions; append to string
void dfstrDecX
(
   char               *str,                     // OUT   resulting string
   char               *text,                    // IN    leading string
   ULONG               data,                    // IN    data
   char               *trail                    // IN    trailing text
);

// Format value in 8-hex digits and decimal;  append to string
void dfstrDec8
(
   char               *str,                     // OUT   resulting string
   char               *text,                    // IN    leading string
   ULONG               data,                    // IN    data
   char               *trail                    // IN    trailing text
);

// Display value in 4-hex digits and decimal; to TxPrintf output
void dfsDec4
(
   char               *text,                    // IN    leading string
   ULONG               data,                    // IN    data
   char               *trail                    // IN    trailing text
);

// Format value in 4-hex digits and decimal; append to string
void dfstrDec4
(
   char               *str,                     // OUT   resulting string
   char               *text,                    // IN    leading string
   ULONG               data,                    // IN    data
   char               *trail                    // IN    trailing text
);

// Format (sector) size in a compact floating-point Mb value when possible
void dfsSizeMB
(
   USHORT              width,                   // IN    width of field
   ULONG               sectors,                 // IN    number of sectors
   USHORT              bps,                     // IN    bytes per sector
   char               *sb                       // OUT   string buffer
);                                              //       minimum 16 chars

// Display ULONG value in 8-hex digits in color; to TxPrintf output
void dfsLX8
(
   char               *text,                    // IN    leading string
   ULONG               data,                    // IN    data
   char               *color,                   // IN    color
   char               *trail                    // IN    trailing text
);

// Format ULONG value in 8-hex digits in color; append to string buffer
void dfstrLX8
(
   char               *str,                     // OUT   resulting string
   char               *text,                    // IN    leading string
   ULONG               data,                    // IN    data
   char               *color,                   // IN    color
   char               *trail                    // IN    trailing text
);


// Display 64bit value with 0x+8-hex or 10 hex digits in color; to TxPrint output
void dfsX10
(
   char               *text,                    // IN    leading string
   ULN64               data,                    // IN    data
   char               *color,                   // IN    color
   char               *trail                    // IN    trailing text
);

// Format 64bit value with 0x+8-hex or 10 hex digits in color; append to string
char *dfstrX10                                  // RET   resulting string
(
   char               *str,                     // OUT   resulting string
   char               *text,                    // IN    leading string
   ULN64               data,                    // IN    data
   char               *color,                   // IN    color
   char               *trail                    // IN    trailing text
);

// Display 64bit value '-' / ' ' + 0x+8-hex / 10 hex digits; to TxPrint output
void dfsSX11
(
   char               *text,                    // IN    leading string
   ULN64               data,                    // IN    data (as signed!)
   char               *color,                   // IN    (color) string
   char               *trail                    // IN    trailing text
);

// Format 64bit value '-' / ' ' + 0x+8-hex / 10 hex digits; append to string
char *dfstrSX11                                  // RET   resulting string
(
   char               *str,                     // INOUT resulting string
   char               *text,                    // IN    leading string
   ULN64               data,                    // IN    data (as signed!)
   char               *color,                   // IN    (color) string
   char               *trail                    // IN    trailing text
);

// Progress reporting functions, enhanced with 'smart' fast area progress
#define DFSP_NONE     0x0000                    // no progress at all
#define DFSP_PERC     0x0001                    // use percentage in status
#define DFSP_VDEC     0x0002                    // decimal XX of YY values
#define DFSP_ADEC     0x0004                    // decimal aditional value
#define DFSP_BAR      0x0010                    // use single line progress  bar
#define DFSP_RES      0x0020                    // add 'resume' indicator to bar
#define DFSP_NO_ETA   0x0800                    // do not display ETA value

// standard, often used progress combinations
#define DFSP_STAT    (DFSP_PERC | DFSP_ADEC)
#define DFSP_BARS    (DFSP_STAT | DFSP_BAR | DFSP_RES)

// dfsProgress addValue Magic for additional value being string only
#define DFSP_STRONLY         0xFFFFFACE
#define DFSP_STATUS_INTERVAL (TMRMSEC(500))     // about twice per second
#define DFSP_ETA_INTERVAL    (TMR_SEC(  2))     // seconds between ETA guesses
#define DFSP_ETA_REMAINING   2                  // seconds left, still displayed
#define DFSP_ETA_1STGUESS    5                  // seconds before 1st ETA

// Init sector progress indicator for BAR and percentage/ETA display in status-line
// 20170526: Added 'fast' area-progress, for more accurate ETA calculations
void dfsProgressInit
(
   ULN64               base,                    // IN    base number (like LSN)
   ULN64               size,                    // IN    size todo   (like sectors)
   ULN64               fastSize,                // IN    fast 'smart' size or 0
   char               *lead,                    // IN    lead text like Sector:
   char               *operation,               // IN    operation indicator
   ULONG               flags,                   // IN    progress style/options
   ULONG               fastFactor               // IN    Speed factor, fast area
);                                              //       being faster than other

// Resume progress session for BAR
void dfsProgressResume
(
   char               *text                     // IN    optional resume text
);

// Suspend progress session (end BAR painting) but keep status for resume
void dfsProgressSuspend
(
   void
);

// Terminate progress indicator
void dfsProgressTerm
(
   void
);

// Display elapsed time and leading/trailing text for progress-session
double dfsProgressElapsed                       // RET   elapsed time, seconds
(
   char               *lead,                    // IN    lead text
   char               *trail                    // IN    trailing text
);

// Display elapsed time and throughput for initialized progress-session
void dfsProgressThroughput
(
   USHORT              bps                      // IN    bytes per sector
);

// Display elapsed time and sector throughput based on starttime and #sectors
double dfsDisplayThroughput                     // RET   elapsed seconds
(
   TXTIMER             stime,                   // IN    starttime
   ULN64               sectors,                 // IN    sectors handled
   USHORT              bps                      // IN    bytes per sector
);

// Display elapsed seconds and byte throughput based on #seconds and #bytes
void dfsDisByteThroughput
(
   double              seconds,                 // IN    elapsed seconds
   ULN64               bytes,                   // IN    bytes handled (filesize)
   char               *spec                     // IN    Throughput spec, 12 ch
);

// Init progress indicator for multiple items/passes
void dfsProgressItemsTodo
(
   ULONG               todo,                    // IN    total items to
   char               *itemname                 // IN    item-kind indicator
);

// Increment items done for progress indicator (when NOT using ProgressInit!)
void dfsProgressItemsNext
(
   void
);

// DFS show progress in SB-status line (percentage) and/or progress BAR in SB
// 20170526: Added 'fast' area-progress, for more accurate ETA calculations
void dfsProgressShow
(
   ULN64               progress,                // IN    actual progress value
   ULN64               fastProgress,            // IN    progress 'fast' area
   ULN64               addValue,                // IN    additional displ value
   char               *addText                  // IN    additional displ text
);                                              //       addText hex/dec value

// DFS show progress message to standard screen or to STDOUT if -G switch set
void dfsGuiStdMessage
(
   char               *msg                      // IN    progress message
);

// DFS test if important output should be sent to STDOUT if -G switch set
BOOL dfsGuiStdOut
(
   void
);

// Display SLT info for specified LSN, unless related LSN is same as last
BOOL dfsDisplaySltIdInfo                        // OUT   type/contents found
(
   ULN64               sn,                      // IN    lsn to display for
   ULN64              *last                     // INOUT last (don't display)
);

// Display of sectornumbers in the SN-list in format specified by options
void dfsDisplaySectorList
(
   char               *select                   // IN    path+filename wildcard
);

// Sort SN-table ascending or descending, optionaly make each entry unique
void dfsSortSectorList
(
   ULN64              *list,                    // IN    count + list of sn's
   char               *options                  // IN    sort-order, unique
);

// Export list of sectornumbers in the SN-table to a file
ULONG dfsExportSectorList                       // RET   function result
(
   char               *fname,                   // IN    Filename for LSN list
   char               *dname                    // IN    Filename for datafile
);

// Import list of sectornumbers to the SN-table from a file
ULONG dfsImportSectorList                       // RET   function result
(
   char               *fname,                   // IN    Filename for LSN list
   char               *dname                    // IN    Filename for datafile
);


// Translate LSN in 4 possible formats, using current GEO, 1 line display
void dfsDisplayTransLsn
(
   char               *lead,                    // IN    leading string (20)
   ULN64               lsn                      // IN    LSN value
);

// Translate LSN in 4 possible formats, using given GEO, 1 line display
void dfsGeoDispTransLsn
(
   char               *lead,                    // IN    leading string (20)
   ULONG               geoHeads,                // IN    geo heads
   ULONG               geoSecs,                 // IN    geo sectors
   ULN64               lsn                      // IN    LSN value
);

// Translate PSN in 4 possible formats, using given GEO, 1 line display
// 20170602: Uses first character after 'lead' for minus sign, plus max 10 HEX
void dfsGeoDispTransPsn
(
   char               *lead,                    // IN    leading string (20)
   ULONG               geoHeads,                // IN    geo heads
   ULONG               geoSecs,                 // IN    geo sectors
   ULN64               psn                      // IN    PSN value
);

// Display single line with  LSN, determined sector-type, CHS and PSN
void dfsDisplayLsnTypeChsPsn
(
   char               *lead,                    // IN    leading text
   ULN64               lsn,                     // IN    Start PSN
   USHORT              info                     // IN    additonal info (entry)
);

// Display single line with  LSN, determined sector-type, CHS and PSN, use GEO
void dfsGeoDispLsnTypeChsPsn
(
   char               *lead,                    // IN    leading text
   ULONG               geoHeads,                // IN    geo heads
   ULONG               geoSecs,                 // IN    geo sectors
   ULN64               lsn,                     // IN    Start PSN
   USHORT              info                     // IN    additonal info (entry)
);

// Display single line with  sector(s) based on Physical addressing
void dfsDisplaySnType
(
   char               *lead,                    // IN    leading text
   ULN64               lsn,                     // IN    Start PSN
   USHORT              info,                    // IN    additonal info (entry)
   BYTE                st                       // IN    type of sector
);

// Display single line with  sector(s) based on Physical addressing, using GEO
void dfsGeoDispSnType
(
   char               *lead,                    // IN    leading text
   ULONG               geoHeads,                // IN    geo heads
   ULONG               geoSecs,                 // IN    geo sectors
   ULN64               lsn,                     // IN    Start PSN
   USHORT              info,                    // IN    additonal info (entry)
   BYTE                st                       // IN    type of sector
);

// Get DFS return-code description in an (ANSI decorated) ASCII string
char *dfsGetRcString                            // RET   RC description string
(
   ULONG               rc,                      // IN    DFS returncode
   TXLN                message                  // INOUT RC message string
);

// Display DFS return-code
void dfsExplainRC
(
   ULONG               dr                       // IN    DFS returncode
);

// Calculate HPFS-sector checksum
ULONG dfsCheckSum
(
   BYTE               *data                     // IN    data sector
);

// Calculate allocation reliability percentage, as in integer value 0..100
ULONG dfsAllocationReliability                  // RET   Percentage
(
   ULN64               size,                    // IN    total number of sectors
   ULN64               bads                     // IN    number of BAD allocations
);

// Add a value to the standard dfs sector-number list
#define dfsAddSI2List(sn,inf) dfsAdd2List((sn),(inf) | DFSSNINFO,NULL,NULL)
#define dfsAdd2SectorList(sn) dfsAdd2List((sn),0,NULL,NULL)
ULONG dfsAdd2List
(
   ULN64               sn,                      // IN    sector-nr to add
   ULN64               d2,                      // IN    dummy
   char               *df,                      // IN    dummy, optional info
   void               *data                     // INOUT dummy
);

// (Re) Initializee the standard dfs sector-number list to a specific size
ULONG dfsInitList
(
   ULN64               size,                    // IN    requested size
   char               *compact,                 // IN    compact display mode
   char               *verbose                  // IN    verbose display mode
);

// Initialize a Multi-item recovery session, by clearing the status counters
void dfsInitMultiRecover
(
   void
);

// Note that (Handled - Recovered - Failed) is the number of items skipped,
// usually because they do NOT match the recovery selection criteria, but
// also the remainder of the sectorlist when aborting

// Get total number of handled items in a Multi-item recovery session
ULONG dfsGetMrHandled
(
   void
);

// Get total number of recovered items in a Multi-item recovery session
ULONG dfsGetMrRecovered
(
   void
);

// Get total number of failed items in a Multi-item recovery session
ULONG dfsGetMrFailed
(
   void
);

// Recover multiple files using list of sectornumbers in the DFSee SN-list
ULONG dfsRecoverList
(
   char               *path,                    // IN    destination path
   S_RECOVER_PARAM    *param                    // INOUT recovery parameters
);


// DFS recover a single file/directory
ULONG dfsRecoverSingle
(
   ULN64               fn,                      // IN    LSN for file to recover
   ULN64               info,                    // IN    meta info for the LSN
   char               *path,                    // IN    base destination path
   S_RECOVER_PARAM    *param                    // INOUT recovery parameters
);


#define DFS_FS_ITEM_NORMAL     ((char) 'N')     // filter Files and Directories
#define DFS_FS_ITEM_FILES      ((char) 'F')     // filter Files, no Directories
#define DFS_FS_ITEM_DIRS       ((char) 'D')     // filter Directories, no Files
#define DFS_FS_ITEM_BROWSE     ((char) 'B')     // filter Files, but keep all Directories

// Parse file-selection string: wildcard, percentage, min/max, size, timestamp
BOOL dfsParseFileSelection                      // RET   more than a wildcard
(
   char               *spec,                    // IN    selection string
   TXTM                wildcard,                // OUT   wildcard part
   BOOL               *minperc,                 // OUT   percentage is minimum
   ULONG              *percent,                 // OUT   percentage value, or 0
   ULONG              *minsize,                 // OUT   min size sectors, or 0
   ULONG              *maxsize,                 // OUT   max size sectors, or 0
   time_t             *timestamp,               // OUT   Modify timestamp, or 0
   char               *type                     // OUT   D(ir) F(ile) B(rowse)
);                                              //       (unchanged if not set)

// Set allocation bits for filesystem for a range of sectors
ULONG dfsSetAllocForRange
(
   ULN64               start,                   // IN    start sector number
   ULONG               size,                    // IN    size of range to set
   ULONG               clsize,                  // IN    granularity, sect/cl
   BOOL                set                      // IN    SET alloc (or reset)
);

// Check allocation bits for filesystem for a range of sectors
ULONG dfsCheckAllocForRange
(
   ULN64               start,                   // IN    start sector number
   ULN64               size,                    // IN    size of range to check
   ULONG               clsize,                  // IN    granularity, sect/cl
   BOOL                set,                     // IN    check if SET (or free)
   ULN64              *bads                     // INOUT Nr of failing sectors
);

// Assemble string of repeated characters with leading and trailing strings
void dfsRstr
(
   char               *string,                  // OUT   resulting string
   char               *lead1,                   // IN    leading string 1
   char               *lead2,                   // IN    leading string 2
   char               *single,                  // IN    string to repeat
   USHORT              count,                   // IN    repeat count
   char               *trail1,                  // IN    trailing text 1
   char               *trail2                   // IN    trailing text 2
);


// Determine disknumber from spec and option value; support -r, -g and -i too
USHORT dfsGetDiskNumber                         // RET   disknr or FDSK_ANY
(
   char                option,                  // IN    option char to use
   char               *defspec,                 // IN    default spec (* or .)
   char               *spec,                    // IN    disk spec or NULL
   BOOL                reread                   // IN    force reread diskinfo
);

// Return valid disk-id for disk-specification;  <empty> | . | * | nr
// or the range FIRST disk plus LAST disk todo, when 'lastDisk' is not NULL
USHORT dfsParseDiskSpec                         // RET   disk/FDSK_ANY or FIRST
(
   char               *spec,                    // IN    disk select spec
   USHORT             *lastDisk                 // INOUT last in range, 0=invalid
);                                              //       or NULL for single disknr

// Select partition or volume from options or parameter
BOOL dfsSelectPartVol                           // RET   part/volume selected
(
   char               *param,                   // IN    selection parameter
   char               *optstr                   // IN    select-cmd option string
);

// Get desired object type based on specified options, or current open object
DFSOBJECT dfsGetObjectType                      // RET   DISK/PART/VOLD/IMGF
(
   DFST_HANDLE         store,                   // IN    related store
   char               *opts                     // IN    option letter string
);

// Get desired object type, and test if object value (to be opened) is specified
BOOL dfsGetObjectValue                          // RET   value specified
(
   DFST_HANDLE         store,                   // IN    related store
   char               *opts,                    // IN    option letter string
   DFSOBJECT          *object                   // OUT   DISK/PART/VOLD/IMGF etc
);

// Get desired object 'open command', based on specified options
ULONG dfsGetObjectOpenCmd                       // RET   result
(
   char               *opts,                    // IN    option letter string
   char               *cmd                      // OUT   command to execute
);                                              //       or empty when none

// Set desired object CURRENT by opening it, based on specified options
ULONG dfsSetObjectCurrent                       // RET   result
(
   char               *opts,                    // IN    option letter string
   BOOL                verbose                  // IN    show command executing
);

// Get a clean string value from -a, -u or -h option, or from mixed string
USHORT dfsGetMixedString
(
   char               *mixed,                   // IN    mixed string or empty
   char               *descr,                   // IN    usage description
   BOOL                prompt,                  // IN    prompt when no input
   BOOL                upper,                   // IN    options AHUM not ahum
   char               *string                   // OUT   clean string value
);

// Determine specified or default-optimal buffersize in sectors, upto a limit
ULONG dfsGetBufferSize                          // RET   number of sectors
(
   ULONG               defSize,                 // IN    default size, sectors
   ULONG               limit                    // IN    limit size, sectors
);


// Get a symbolic or HEX start sector-number like 3f . this or .3
ULN64 dfsGetSymbolicSN
(
   char               *spec,                    // IN    symbolic or hex SN
   ULN64               defsn                    // IN    default SN for no spec
);


// Get a symbolic or HEX size value like 3f . this or .3
ULN64 dfsGetSymbolicSize
(
   char               *spec,                    // IN    symbolic or hex SN
   ULN64               defsz,                   // IN    default size
   DFST_HANDLE         store,                   // IN    DFS store to be used
   ULN64               base                     // IN    base value for relative
);                                              //       values like $ and .

// Get an mcs-number in HEX, OCTAL or DECIMAL
ULN64 dfsGetMcsNumber
(
   char               *spec,                    // IN    Hex/Oct/Dec mcs-number
   ULN64               defsz                    // IN    default size
);

// Apply specified number-unit, returning corrected number value; default 's'
ULN64 dfsApplyNumberUnit                        // RET   unit corrected value
(
   ULN64               value,                   // IN    uncorrected value
   char                unit,                    // IN    unit, k m g c h s
   DFST_HANDLE         store                    // IN    DFS store for GEO
);

// Get a dotted-decimal number, up to 20 positions (as created by UllDot20)
ULN64 dfsGetUllDot20                            // RET   64-bit unsigned number
(
   char               *spec                     // IN    Dotted decimal, 20 pos
);

// Get chsStyle from specified option or from the dfsa global value
ULONG dfsGetChsStyle                            // RET   chsStyle value
(
   char                ch,                      // IN    character
   TXHANDLE            txtype,                  // IN    option or switch
   BOOL                batch                    // IN    no interactive popups
);

#if defined (USEWINDOWING)
// Prompt for new LSN value based on user selections (windowed only)
ULONG dfsGetGotoLsnValue
(
   char               *title,                   // IN    Dialog title
   ULN64              *sn                       // OUT   new sector number
);
#endif

// Run the DFSDISK script on specified disk, with any options
ULONG dfsRunDfsDiskAnalysis                     // RET   function result
(
   USHORT              disk,                    // IN    disk number
   char               *precision,               // IN    ALL, CYL or ""
   char               *cyls,                    // IN    #cyl or "?"
   char               *head,                    // IN    #heads
   char               *sect,                    // IN    #sectors/track
   char               *superFS,                 // IN    Superblock filesystem
   char               *basefn                   // IN    Base filename
);

// Run the DFSDFAST script on specified disk, with any options
ULONG dfsRunDfsFastAnalysis                     // RET   function result
(
   USHORT              disk,                    // IN    disk number
   char               *basefn                   // IN    Base filename
);

// Run the DOxxxx script on specified partition and optional filesystem
ULONG dfsRunChkPartAnalysis                     // RET   function result
(
   USHORT              part,                    // IN    partition-ID
   char               *fsname                   // IN    optional fsys name
);

// Fix Partition-bootsector HiddenSectors/geometry fields to match part-tables
ULONG dfsFixPbrHsGeoFields
(
   BOOL                fixgeo,                  // IN    Fix Geometry
   BOOL                fixhs,                   // IN    Fix HiddenSectors
   BOOL                confirmed,               // IN    no confirmation
   BOOL                popup                    // IN    use message popup
);

// Set volume-serial number in the Partition-bootsector (FAT, FAT32, HPFS ...)
ULONG dfsSetVolumeSerialNr
(
   BOOL                popup,                   // IN    use message popup
   char               *newvsn                   // IN    new vsn, "" or NULL
);

// Set volume-label string(11) in the Partition-bootsector (FAT, FAT32, HPFS)
ULONG dfsSetVolumeLabelBrec
(
   BOOL                popup,                   // IN    use message popup
   char               *newVln                   // INOUT new label, "" or NULL
);

// Test if supplied buffer is RLE compressable for a single byte value
BOOL dfsRleCompressable                         // RET   single byte value
(
   BYTE               *buf,                     // IN    sector buffer
   ULONG               bytes                    // IN    size in bytes
);

// Test file-existence, prompt user to change a removable medium for read
BOOL dfsChangeReadMedium                        // RET   File (now) present
(
   char               *lead,                    // IN    leading info text
   char               *filename                 // IN    full path and filename
);

// Prompt user to change a removable medium for write, create PATH when needed
BOOL dfsChangeWriteMedium                       // RET   New medium and path OK
(
   char               *filename,                // IN    full path and filename
   LLONG               maxsize                  // IN    max size to be written
);

// Mangle 32-bit LVM-ID value using 3-digit HEX mangle string-value
ULONG dfsLvmIdMangle                            // RET   Mangled Id value
(
   ULONG               id,                      // IN    base Id value
   char               *ms                       // IN    Hex mangle string[3]
);

// Get description for current (DFSTORE/SINF) partition/volume; max 50 chars
// Replace filesystem shown in store by actual/real one for known part/volume
char *dfsDriveDescription                       // RET   text buffer
(
   TXTM                txt                      // OUT   text buffer
);

// Count nr of '1' bits in a bitmap (array of ULONGS)
ULONG dfsUlMapBitCount                          // RET   nr of bits SET
(
   ULONG              *map,                     // IN    Bitmap array
   ULONG               size                     // IN    array size
);

// Parse ALLOC characters per line, based on display screen-width
ULONG dfsAllocCharsPerLine                      // RET   characters per line
(
   char                option                   // IN    option char to use
);

// Get max characters per line, based on display screen-width and SET margin
ULONG dfsGetDisplayMargin                       // RET   characters per line
(
   void
);

// Parse ALLOC detail value from an option plus string-param and supplied size
ULONG dfsAllocItemsPerChar                      // RET   items per character
(
   char                option,                  // IN    option char to use
   char               *param,                   // IN    detail specification
   ULONG               size,                    // IN    total number of items
   ULONG               cols                     // IN    chars per line (64..256)
);

// Copy SIZE data sectors from SRC to DEST, allowing overlapping SRC and DEST
ULONG dfsCopySectorData
(
   ULN64               fLsn,                    // IN    FROM LSN
   ULN64               dLsn,                    // IN    DEST LSN
   ULONG               size                     // IN    SIZE in sectors
);

#endif
