//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// DFSee utility & display services
//
// Author: J. van Wijk
//
// JvW  22-07-95   Initial version, split off from DHPFS.C
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfshpfs.h>                            // HPFS structure defs
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS  navigation and defs
#include <dfsver.h>                             // DFS  version and naming
#include <dfsutil.h>                            // DFS  utility functions

#if !defined (OEMSB)
#include <dfsulzw.h>                            // DFS  compress functions

#include <dfsupart.h>                           // FDISK utility functions
#include <dfsufgpt.h>                           // FDISK GPT defs and functions

#include <dfsspace.h>                           // DFS file-space interface
#include <dfsaext.h>                            // EXT  superblock display
#include <dfsahpfs.h>                           // HPFS superblock display
#include <dfsahfs.h>                            // HFS  superblock display
#include <dfsajfs.h>                            // JFS  superblock display
#include <dfsantfs.h>                           // NTFS superblock display
#include <dfsarsr.h>                            // RSR  superblock display
#include <dfsaxfs.h>                            // XFS  superblock display

static char signature[SG_SPARE] = {SV_SPARE};   // signature spare-block
#endif

//- LVM signature string values, null terminated
char sg_lvinf[ SG_LVINF +1] = {SV_LVINF,0};     // LVM info signature
char sg_lvsig[ SG_LVSIG +1] = {SV_LVSIG,0};     // LVM BBR, signature
char sg_lvrem[ SG_LVSIG +1] = {SV_LVREM,0};     // LVM BBR, removed
char sg_lvdlf[ SG_LVDLF +1] = {SV_LVDLF,0};     // LVM drive-link sector
char sg_lvdlt[ SG_LVDLT +1] = {SV_LVDLT,0};     // LVM drive-link table
char sg_lvbbf[ SG_LVBBF +1] = {SV_LVBBF,0};     // LVM bad-block  sector
char sg_lvbbt[ SG_LVBBT +1] = {SV_LVBBT,0};     // LVM bad-block  table

//- Linux Unified Key Setup, header at start partition or disk
char sg_luksh[ SG_LUKSH +1] = {SV_LUKSH,0};     // LUKS crypto header

//- MAC DDM signature strings
char sg_mcddm[SG_MCDDM] = {SV_MCDDM};           // MAC driver description
char sg_mcdpm[SG_MCDPM] = {SV_MCDPM};           // MAC disk partition map


/*****************************************************************************/
// Identify type of specified sector
/*****************************************************************************/
BYTE dfsIdentifySector                          // RET   type of sector
(
   ULN64               lsn,                     // IN    Sector LSN
   USHORT              sninfo,                  // IN    info (fat entry, ext index)
   BYTE               *sec                      // IN    sector data
)
{
   ULONG               dr;
   BYTE                rc = ST_UDATA;           // rc, sector type
   TXTM                zero;                    // buffer with binary zeros
   char               *bytes = (char *) sec;
   ULONG               psn   = dfstLSN2Psn( DFSTORE, lsn);

   ENTER();
   TRACES(( "lsn:0x0%llx sninfo:%4.4hx\n", lsn, sninfo));
   TRHEXS( 120,  sec,  0x80, "sector data");

   dr = DFSFNCALL(dfsa->FsIdentifySector, lsn, sninfo, (char *) &rc, sec);

   if (dr == DFS_PENDING)                    // not handled
   {
      if (((S_BOOTR*)sec)->Signature == SV_BOOTR)
      {
         FAT32B2      *fat32 = (FAT32B2 *) sec;

         TRACES(("Bootsector signature aa55 present ...\n"));
         memset( zero, 0, sizeof(zero));
         if ((fat32->Signatur1 == SV_BOOT21) &&
             (fat32->Signatur2 == SV_BOOT22)  )
         {
            rc = ST_BOOT2;                      // FAT32 2nd bootsec
         }
         else if (memcmp( ((char *) sec) +1,  "fF", 4) == 0)
         {
            rc = ST_BOOT3;                      // FAT32 3rd bootsec
         }
         else if ((memcmp(&(((S_BOOTR*)sec)->PartitionTable[0]), zero, 64) != 0) &&
             ((((S_BOOTR*)sec)->PartitionTable[0].Status & 0x7a) == 0) &&
             ((((S_BOOTR*)sec)->PartitionTable[1].Status & 0x7a) == 0) &&
             ((((S_BOOTR*)sec)->PartitionTable[2].Status & 0x7a) == 0) &&
             ((((S_BOOTR*)sec)->PartitionTable[3].Status & 0x7a) == 0) )
         {
            rc = (psn == 0) ? ST_MASTR : ST_EXTBR;
         }
         else if ((((S_BOOTR*)sec)->Instruction[0] == DFS_M_OPCO_1) ||
                  (((S_BOOTR*)sec)->Instruction[0] == DFS_M_OPCO_2)  )
         {
            rc = ST_MASTR;
         }
         else if ((TxMemStr(sec, "GRUB", SECTORSIZE) != NULL) ||
                  (TxMemStr(sec, "LILO", SECTORSIZE) != NULL)  )
         {
            rc = (psn == 0) ? ST_MASTR : ST_BOOTR;
         }
         else
         {
            TXTS       label;

            TRACES(("Not a FAT32 2/3 or MBR/EBR, check filesystems ...\n"));

            memcpy( label, ((S_BOOTR*)sec)->os.Type, BT_TYPE_L);

            switch (((S_BOOTR*)sec)->Instruction[0])
            {
               case DFS_B_OPCO_1:
               case DFS_B_OPCO_2:
                  rc = ST_BOOTR;                // regular boot record
                  break;

               default:
                  if ((memcmp( label, "JFS     ", BT_TYPE_L) == 0) ||
                      (memcmp( label, "FAT     ", BT_TYPE_L) == 0) ||
                      (memcmp( label, "HPFS    ", BT_TYPE_L) == 0)  )
                  {
                     rc = ST_BOOTR;             // regular (FS) boot record
                  }
                  else
                  {
                     rc = (psn == 0) ? ST_MASTR : ST_BOOTX; // non-standard boot record
                  }
                  break;
            }
         }
      }
      else if (!memcmp(((S_LVINF*)sec)->Signature,sg_lvinf,SG_LVINF)) rc = ST_LVINF;
      else if (!memcmp(((S_LVSIG*)sec)->Signature,sg_lvsig,SG_LVSIG)) rc = ST_LVSIG;
      else if (!memcmp(((S_LVSIG*)sec)->Signature,sg_lvrem,SG_LVSIG)) rc = ST_LVREM;
      else if (!memcmp(((S_LVDLF*)sec)->Signature,sg_lvdlf,SG_LVDLF)) rc = ST_LVDLF;
      else if (!memcmp(((S_LVDLT*)sec)->Signature,sg_lvdlt,SG_LVDLT)) rc = ST_LVDLT;
      else if (!memcmp(((S_LVBBF*)sec)->Signature,sg_lvbbf,SG_LVBBF)) rc = ST_LVBBF;
      else if (!memcmp(((S_LVBBT*)sec)->Signature,sg_lvbbt,SG_LVBBT)) rc = ST_LVBBT;
#if !defined (OEMSB)
      else if ((     (((S_GPTHDR*)sec)->Signature1 == SV_GPTHDR1) &&
                     (((S_GPTHDR*)sec)->Signature2 == SV_GPTHDR2) ))  rc = ST_GPTHD;
      else if (!memcmp(((S_MCDDM*)sec)->Signature,sg_mcddm,SG_MCDDM)) rc = ST_MCDDM;
      else if (!memcmp(((S_MCDPM*)sec)->Signature,sg_mcdpm,SG_MCDPM)) rc = ST_MCDPM;
      else if (!memcmp(sec, DFSUL_FILESIG, DFSUL_SIG_LEN))            rc = ST_ULZ16;
      else if (!memcmp(sec, DFSUL_FILEBIG, DFSUL_SIG_LEN))            rc = ST_ULZ32;
      else if (!memcmp(((S_LUKSH*)sec)->Signature,sg_luksh,SG_LUKSH)) rc = ST_LUKSH;
      else if ((((S_CORE_ST*)sec)->Signature  == CORE_ST_SIG)       &&
               (((S_CORE_ST*)sec)->crcPad     == CORE_ST_PAD)       &&
               (((S_CORE_ST*)sec)->sectorSize >= SECTORSIZE)        &&
              ((((S_CORE_ST*)sec)->sectorSize %  SECTORSIZE) == 0)  &&
              ((((S_CORE_ST*)sec)->blockSize  %  SECTORSIZE) == 0)   )  rc = ST_CORES;
#endif
      else if ((bytes[0] == (char ) 0xf6) && (dfsCheckSum((BYTE *) bytes) == 0xfffffff0))
      {
         rc = ST_FDISK;                         // FDISK F6 cleared sector
      }
      else if ((bytes[0] == (char ) 0x00) && (dfsCheckSum((BYTE *) bytes) == 0x00000000))
      {
         rc = ST_EMPTY;                         // Completely empty, 0x00 only
      }
      else if ((bytes[0] == (char ) 0xfe) && (dfsCheckSum((BYTE *) bytes) == 0xffffffc0))
      {
         rc = ST_BADFE;                         // Sector marked as BAD, 0xFE only
      }
   }
   TRINIT(70);
      dfsSectorTypeInColor( "Sector identification:", rc); TxPrint("\n");
   TREXIT();
   RETURN (rc);
}                                               // end 'dfsIdentifySector'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Test if the given MBR/EBR sector has 4 valid entries, with at least one used
// NO check for a valid 0x55AA signature to allow interpreting damaged disks
/*****************************************************************************/
BOOL dfsValidXbrTableEntries                    // RET   sector likely is xBR
(
   BYTE               *data                     // IN    data sector
)
{
   TXTM                zero;                    // buffer with binary zeros
   int                 i;
   S_BOOTR            *xbr = (S_BOOTR *) data;
   DFSPARTENTRY       *entry;

   memset( zero, 0, sizeof(zero));
   if (memcmp(&(xbr->PartitionTable[0]), zero, 64) == 0)
   {
      return FALSE;                             // no partition table entries at all
   }

   // First check for invalid entries
   for (i = 0; i < 4; i++)
   {
      entry = &(xbr->PartitionTable[ i]);
      if (memcmp( entry, zero, 16) != 0)        // something in the entry
      {
         if ((entry->BootSectorOffset == 0) ||
             (entry->NumberOfSectors  == 0) ||
             (entry->PartitionType    == 0) ||
             ((entry->Status & 0x7a)  != 0))
         {
            return FALSE;                       // but some invalid values
         }
      }
   }

   //- Then check if there is at least one valid partition defined
   for (i = 0; i < 4; i++)
   {
      entry = &(xbr->PartitionTable[ i]);

      if ((entry->NumberOfSectors     != 0) ||  // non-zero size
          (entry->LastSector.SecCyl   != 0))    // last cylinder not zero
      {
         return TRUE;
      }
   }
   return FALSE;
}                                               // end 'dfsValidXbrTableEntries'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Calculate sector checksum, use special handling for HPFS spare dirty flag
// and for FAT16/FAT32 dirty flags so CRC does not differ between clean/dirty
/*****************************************************************************/
ULONG dfsCheckSum
(
   BYTE               *data                     // IN    data sector
)
{
   ULONG               cs = 0;                  // checksum value

   #if defined (OEMSB)
      cs = TxHpfsCheckSum( data);               // actual CRC calculation
   #else
      S_SPARE         *spr = (S_SPARE *) data;  // data as HPFS spare-block
      ULONG            cspare = 0;
      BYTE             dflag  = 0;              // dirty flag byte
      S_BOOTR         *brt = (S_BOOTR *) data;  // sector as partition BOOTREC
      BYTE            *FsysValue = NULL;        // ptr to FAT dirty flag

      ENTER();

      TRHEXS(70, data, (TxTrLevel > 100) ? dfsGetSectorSize() : 30, "checksum data");

      if      ((strncasecmp( brt->f2.Type, "FAT32", 5) == 0) && // FAT32
               (strncasecmp( brt->os.Type, "FAT",   3) != 0)  )
      {
         FsysValue = &brt->f2.FsysValue;
      }
      else if ((strncasecmp( brt->f2.Type, "FAT32", 5) != 0) && // FAT16
               (strncasecmp( brt->os.Type, "FAT",   3) == 0)  )
      {
         FsysValue = &brt->os.FsysValue;
      }
      if (FsysValue != NULL)                    // we have a FAT dirty flag
      {
         dflag = *FsysValue;                    // save dirty flag value
         *FsysValue = 0;                        // ZERO during CRC calculation
      }

      if (memcmp(spr->Signature, signature, SG_SPARE))
      {
         spr = NULL;                            // not a spare sector
      }
      else                                      // prepare spare-block for
      {                                         // checksumming
         cspare = spr->SpChecksum;
         dflag  = spr->StatusFlag;

         spr->StatusFlag &= ~0x01;              // clear Dirty Flag
         spr->SpChecksum = 0;                   // clear own checksum field
      }

      cs = TxHpfsCheckSum((char *) data);       // actual CRC calculation

      if (spr != NULL)                          // restore cleared fields
      {
         spr->SpChecksum = cspare;
         spr->StatusFlag = dflag;
      }

      if (FsysValue != NULL)                    // we have a FAT dirty flag
      {
         *FsysValue = dflag;                    // restore dirty flag value
      }

   #endif
   RETURN (cs);
}                                               // end 'dfsCheckSum'
/*---------------------------------------------------------------------------                    */


/*****************************************************************************/
// Calculate allocation reliability percentage, as in integer value 0..100
/*****************************************************************************/
ULONG dfsAllocationReliability                  // RET   Percentage
(
   ULN64               size,                    // IN    total number of sectors
   ULN64               bads                     // IN    number of BAD allocations
)
{
   ULONG               rc = 100;                // percentage

   ENTER();
   if ((size != 0) && (bads != 0))
   {
      //- NO ROUNDING UP, one sector BAD must lead to less than 100%
      ULN64         OKx100 = ((ULN64) 100 * (size - bads));
      rc = (ULONG) (OKx100 / size);

      if ((rc == 0) && (bads < size))           // 0% ONLY for ALL bad
      {
         rc = 1;                                // show that at least SOME are OK
      }
   }
   RETURN (rc);
}                                               // end 'dfsAllocationReliability'
/*---------------------------------------------------------------------------*/

#ifndef OEMSB
/*****************************************************************************/
// Parse ALLOC characters per line, based on display screen-width
/*****************************************************************************/
ULONG dfsAllocCharsPerLine                      // RET   characters per line
(
   char                option                   // IN    option char to use
)
{
   ULONG               rc;
   ULONG               value;
   short               screen = dfsGetDisplayMargin();

   ENTER();

   TRACES(("Option: '%c'\n", option));

   if      (screen > 724) value = 704;
   else if (screen > 660) value = 640;
   else if (screen > 596) value = 576;
   else if (screen > 532) value = 512;
   else if (screen > 468) value = 448;
   else if (screen > 404) value = 384;
   else if (screen > 340) value = 320;
   else if (screen > 276) value = 256;
   else if (screen > 212) value = 192;
   else if (screen > 180) value = 160;
   else if (screen > 156) value = 128;          // 6 extra for possible BG#
   else if (screen > 132) value = 112;
   else if (screen > 116) value =  96;
   else if (screen > 100) value =  80;
   else                   value =  64;

   rc = TxaOptNum( option, NULL, value);
   if ((rc == 0) || (rc > 750))
   {
      rc = 64;                                  // sane default
   }
   RETURN (rc);
}                                               // end 'dfsAllocCharsPerLine'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get max characters per line, based on display screen-width and SET margin
/*****************************************************************************/
ULONG dfsGetDisplayMargin                       // RET   characters per line
(
   void
)
{
   ULONG               rc = (dfsa->margin) ? dfsa->margin : dfsa->sbWidth;

   ENTER();
   RETURN (rc);
}                                               // end 'dfsGetDisplayMargin'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Parse ALLOC detail value from an option plus string-param and supplied size
/*****************************************************************************/
ULONG dfsAllocItemsPerChar                      // RET   items per character
(
   char                option,                  // IN    option char to use
   char               *param,                   // IN    detail specification
   ULONG               size,                    // IN    total number of items
   ULONG               cols                     // IN    chars per line (64..256)
)
{
   ULONG               rc = 1;
   ULONG               value;

   ENTER();
   TRACES(("Option: '%c' size:0x%8.8lx param:'%s'\n", option, size, param));

   if ((TxaOptSet( option) ||                   // explict lines, or items/char
        strlen(param) == 0) )                   // default, one screen full
   {
      value = TxaOptNum( option, NULL,
      #if defined (OEM_BRANDED)
                         16);
      #else
                         TxScreenRows() -16);
      #endif
      rc = (size / (cols * value)) + 1;         // round up, avoid more lines
   }
   else if (param[0] == '@')                    // exactly one line (-l:1)
   {
      rc = (size / cols) + 1;                   // round up, avoid more lines
   }
   else if ((strlen(  param)) &&                // hex numeric param,
            isxdigit( param[0]))                // direct sectors/char
   {
      rc = dfsGetSymbolicSize( param, 1, DFSTORE, 0);
   }
   else                                         // use string parameter
   {
      for (value = size, rc = 1; value != 0; value /= 2)
      {
         rc *= 2;                               // round size up to a
      }                                         // power of two, for clean
      switch (param[0])                         // sizes per character
      {                                         // (based on 64 chars/line)
         case '$': rc /= 65536;  break;         // detailed  512    1024
         case '*': rc /= 16384;  break;         // normal    128 .. 256
         case '+': rc /=  4096;  break;         // small      32 .. 64
         case '~': rc /=   256;  break;         // very small, 2 .. 4  lines
         case '-':
         default:  rc /=  1024;  break;         // yields      8 .. 16 lines
      }
   }
   if (rc == 0)
   {
      rc = 1;                                   // sane minimum, max detail
   }
   RETURN (rc);
}                                               // end 'dfsAllocItemsPerChar'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get description for current (DFSTORE/SINF) partition/volume; max 50 chars
// Replace filesystem shown in store by actual/real one for known part/volume
/*****************************************************************************/
char *dfsDriveDescription
(
   TXTM                txt                      // OUT   text buffer
)
{
   TXTT                fs;                      // real filesystem description

   ENTER();

   strcpy( txt, dfstStoreDesc1( DFSTORE) + 10);

   strcpy( fs, "????");
   if      (txt[0] == 'V')
   {
      if (strcmp( SINF->drive, "--") != 0)
      {
         TxFsType( SINF->drive, fs, NULL);
      }
   }
   else if (txt[0] == 'P')
   {
      if (SINF->p != 0)
      {
         strcpy( fs, SINF->p->fsform);
      }
   }
   strcat( fs, "      ");                       // pad with spaces
   memcpy( txt + 24, fs, 6);                    // and replace DFSTORE fs-name

   RETURN (txt);
}                                               // end 'dfsDriveDescription'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Count nr of '1' bits in a bitmap (array of ULONGS)
/*****************************************************************************/
ULONG dfsUlMapBitCount                          // RET   nr of bits SET
(
   ULONG              *map,                     // IN    Bitmap array
   ULONG               size                     // IN    array size
)
{
   ULONG               rc = 0;
   ULONG               word;
   int                 i;

   for (i = 0; i < size; i++)
   {
      word = map[i];
      do
      {
         rc += (word & 1);
      } while (word >>= 1);
   }
   return (rc);
}                                               // end 'dfsUlMapBitCount'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Calculate size in GiB for OS/2 65535 cylinder limit and given GEO values
/*****************************************************************************/
ULONG dfs16bitCylLimitGiB                       // RET   nr of GiBs for limit
(
   ULONG               heads,                   // IN    Number of heads in GEO
   ULONG               sects                    // IN    Sectors per track in GEO
)
{
   ULONG               rc = (heads * sects * 65535) / (2 * 1024 * 1024);

   return (rc);
}                                               // end 'dfs16bitCylLimitGiB'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Return the number of sectors to align partitions on when CReating them
/*****************************************************************************/
ULONG dfsAlignmentSectors                       // RET   #sectors for CR alignment
(
   char                fspKind,                 // IN    G(pt), P(rim), L(ogical)
   DFSPARTINFO        *f                        // IN    freespace or part info (geo)
)
{
   ULONG               rc;
   ULONG               cSC = (f) ? f->cSC      : dfsCylinderSize();
   ULONG               bps = (f) ? f->bpsector : dfsGetSectorSize();

   ENTER();

   switch (dfsa->alignment)
   {
      case DFS_CR_ALIGN_AUTO:
         rc = (fspKind == 'G') ? (0x100000 / bps) : cSC;
         break;

      case DFS_CR_ALIGN_CYL:
         rc = cSC;
         break;

      case DFS_CR_ALIGN_MIB:
      default:
         rc = (0x100000 / bps);  // 1 MiB, depending on BPS
         break;
   }
   RETURN (rc);
}                                               // end 'dfsAlignmentSectors'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// Return the number of sectors to to use as 'GAP' value when CReating them
// Note: overruled by -G:n option, sets default based on specified alignment
/*****************************************************************************/
ULONG dfsDefaultGapSectors                      // RET   #sectors default GAP value
(
   char                fspKind,                 // IN    G(pt), P(rim), L(ogical)
   DFSPARTINFO        *f                        // IN    freespace info (geo)
)
{
   ULONG               rc;
   ULONG               track = (f) ? f->geoSecs : dfstGeoSectors( DFSTORE);

   ENTER();
   TRACES(("fspKind: '%c', alignment: %lu\n", fspKind, dfsa->alignment));

   switch (dfsa->alignment)
   {
      case DFS_CR_ALIGN_AUTO:
         rc = (fspKind == 'G') ? 1 : track;     // should not happen (no GAPS on GPT)
         break;

      case DFS_CR_ALIGN_CYL:
         rc = track;
         break;

      case DFS_CR_ALIGN_MIB:
      default:                                  // numeric value, including 1 MiB
         if (dfsa->alignment <= 64)
         {
            rc = dfsa->alignment;
         }
         else
         {
            rc = (4096 / dfsGetSectorSize());   // default to 4Kb gap for larger (like 1 MiB)
         }
         break;
   }
   RETURN (rc);
}                                               // end 'dfsDefaultGapSectors'
/*---------------------------------------------------------------------------*/


#endif
