//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// REISER utility functions
//
// Author: J. van Wijk
//
// JvW  31-03-2004 Initial version
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsver.h>                             // DFS version info
#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfstable.h>                           // SLT utility functions

#include <dfsrsr.h>                             // RSR disk structures
#include <dfsarsr.h>                            // RSR analysis functions
#include <dfsursr.h>                            // RSR utility functions
#include <dfslrsr.h>                            // RSR slt & error definitions


/*****************************************************************************/
// Identify if FS structures for REISER are present, set fsform and label
/*****************************************************************************/
ULONG dfsRsrFsIdentify
(
   S_BOOTR            *boot,                    // IN    Bootsector ref or NULL
   DFSPARTINFO        *p                        // INOUT partition info
)
{
   ULONG               rc = DFS_FS_UNKNOWN;     // function return
   S_RSR_SUPER        *sup;                     // superblock sector
   char                st;

   ENTER();

   if ((sup = TxAlloc(1, dfsGetSectorSize())) != NULL)
   {
      if (dfsRead( p->basePsn + RSR_LSNSUP2, 1, (BYTE *) sup) == NO_ERROR)
      {
         if ((dfsRsrIdent(RSR_LSNSUP2, 0, &st, sup) != DFS_PENDING) && (st == ST_RSRSUP))
         {
            rc = NO_ERROR;
         }
      }
      if (rc == DFS_FS_UNKNOWN)                 // try alternate (old) location
      {
         if (dfsRead( p->basePsn + RSR_LSNSUP1, 1, (BYTE *) sup) == NO_ERROR)
         {
            if ((dfsRsrIdent(RSR_LSNSUP1, 0, &st, sup) != DFS_PENDING) && (st == ST_RSRSUP))
            {
               rc = NO_ERROR;
            }
         }
      }
      if (rc == NO_ERROR)
      {
         switch (sup->Signature[SG_RSRSUP])
         {
            case '2': strcpy( p->fsform, "Reiser2"); break;
            case '3': strcpy( p->fsform, "Reiser3"); break;
            default:  strcpy( p->fsform, "Reiser");  break;
         }
         TxCopy( p->plabel, (char *) sup->VolumeName, RSR_LEN_LBL +1);
         memcpy( p->fsUuid, sup->Uuid, FS_UUID_LENGTH);
         p->uuidPresent = TRUE;
      }
      TxFreeMem( sup);
   }
   RETURN (rc);
}                                               // end 'dfsRsrFsIdentify'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate Block-nr to LSN, generic interface
/*****************************************************************************/
ULN64 dfsRsrCl2Lsn                              // RET   LSN for this block
(
   ULN64               block,                   // IN    block number
   ULN64               d2,                      // IN    dummy
   char               *p1,                      // IN    dummy
   void               *p2                       // IN    dummy
)
{
   ULN64               lsn = block;

   lsn = block * rsr->SectorsPerBlock;
   return (lsn);
}                                               // end 'dfsRsrCl2Lsn'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate LSN to Block-nr, generic interface
/*****************************************************************************/
ULN64 dfsRsrLsn2Cl                              // RET   block for this LSN
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *p1,                      // IN    dummy
   void               *p2                       // IN    dummy
)
{
   ULN64               block = 0;

   block = lsn / rsr->SectorsPerBlock;
   return (block);
}                                               // end 'dfsRsrLsn2Cl'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Determine allocation-bit for specified LSN, ALLOCATED beyond last block!
/*****************************************************************************/
ULONG dfsRsrAllocated                           // RET   LSN is allocated
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               asn = dfstAreaD2Part( DFSTORE, lsn); // Area aware sn

   if (rsr->Bm.Cache != NULL)                   // cache present
   {
      if (asn < (rsr->Sect))                    // within valid block area
      {
         return((ULONG) dfsRsrBitmapCache( dfsRsrLsn2Block( asn), NULL));
      }
      else                                      // return ALLOCATED for any
      {                                         // lsn beyond last block
         return( 1);                            // to avoid CHECK errors
      }                                         // on non-standard bitmaps
   }
   else
   {
      return(DFS_PSN_LIMIT);
   }
}                                               // end 'dfsRsrAllocated'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Align R/W cache to position for block, and get current allocation-bit
// Note: Area aware to allow usage from FDISK mode too
/*****************************************************************************/
BOOL dfsRsrBitmapCache                          // RET   block is allocated
(
   ULONG               bl,                      // IN    Block number
   ULONG              *cp                       // OUT   position in cache
)                                               //       or L64_NULL if invalid
{
   ULONG               rc = NO_ERROR;
   BOOL                al = TRUE;               // default allocated
   ULONG               group;                   // Group number for block
   ULONG               BmapSect;                // sectornr in Block Bitmap
   ULONG               bytepos = L32_NULL;      // byte position in map
   ULONG               alloc_byte;

   if (rsr->BlocksPerGroup != 0)                // avoid divide by zero
   {
      group = bl / rsr->BlocksPerGroup;         // Group number for block
      if (group != rsr->Bm.Group)               // other block group
      {
         if (rsr->Bm.Dirty)                     // need to flush changes ?
         {
            rc = dfsRsrBitmapFlush( FALSE);     // flush, but keep cache
         }
         if (rc == NO_ERROR)
         {
            if ((rsr->Bm.Group = group) == 0)   // group-0 bitmap after super
            {
               BmapSect = RSR_LSNSUP2 + rsr->SectorsPerBlock;
            }
            else                                // others first block in group
            {
               BmapSect = group * rsr->BlocksPerGroup * rsr->SectorsPerBlock;
            }
            TRACES(("RsrAllocated: Cache bitmap for group %lu\n", group));
            rc = dfsRead( dfstAreaP2Disk( DFSTORE, BmapSect), rsr->SectorsPerBlock, rsr->Bm.Cache);
         }
      }
      if (rc == NO_ERROR)
      {
         bytepos    = (bl % rsr->BlocksPerGroup) >> 3; // byte index in map
         alloc_byte = rsr->Bm.Cache[ bytepos];
         if (((alloc_byte >> (bl & 0x07)) & 0x01) == 0) // LSB is lowest!
         {
            al = FALSE;
         }
      }
      if (cp != NULL)                           // return byte position too ?
      {
         *cp = bytepos;
      }
   }
   return (al);
}                                               // end 'dfsRsrBitmapCache'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create Reiser Bitmap cache structures and initialize the BitMap cache
// Note: Area aware to allow usage from FDISK mode too
/*****************************************************************************/
ULONG dfsRsrBitMapInit                          // RET   rc = 0 if type match
(
   void
)
{
   ULONG               rc = 0;                  // rc, sector match

   ENTER();

   if ((rsr->Bm.Cache = TxAlloc( rsr->SectorsPerBlock, dfsGetSectorSize())) != NULL)
   {
      rsr->Bm.Group = DFSRSR_NOGROUP;           // nothing cached yet
      rsr->Bm.Dirty = FALSE;
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsRsrBitMapInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Flush BitMap cache when Dirty, and optional free its resources
/*****************************************************************************/
ULONG dfsRsrBitmapFlush                         // RET   rc
(
   BOOL                terminate                // IN    terminate cache too
)
{
   ULONG               rc = NO_ERROR;           // rc

   ENTER();

   if (rsr->Bm.Dirty)                           // need to flush changes ?
   {
      TRACES(( "dirty, flush it ...\n"));

      //- Flushing to be refined
      rsr->Bm.Dirty = FALSE;                    // mark it clean again
   }
   else
   {
      TRACES(( "not dirty ...\n"));
   }
   if (terminate)
   {
      TxFreeMem( rsr->Bm.Cache);

      rsr->Bm.Group = DFSRSR_NOGROUP;
   }
   RETURN (rc);
}                                               // end 'dfsRsrBitmapFlush'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Write cached superblock to superblock location on disk
/*****************************************************************************/
ULONG dfsRsrWriteSuperBlocks
(
   void
)
{
   ULONG               rc  = NO_ERROR;

   ENTER();

   if (DFSTORE_WRITE_ALLOWED)
   {
      TxPrint("Superblock update : 0x%2.2lx ", RSR_LSNSUP2);
      rc = dfsWrite(   RSR_LSNSUP2, 1, (BYTE   *) rsr->sup);
      TxPrint( "%s\n", (rc == NO_ERROR) ? "written" : "failed");
   }
   else
   {
      rc = DFS_READ_ONLY;
   }
   RETURN(rc);
}                                               // end 'dfsRsrWriteSuperBlocks'
/*---------------------------------------------------------------------------*/

