//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// Partition utility and iterator functions
//
// Author: J. van Wijk
//
// JvW  18-04-99   Initial version, split from DFSUFDSK
//
#ifndef    DFSUPART
   #define DFSUPART

#define FDSK_ANY      ((USHORT) 0xffff)         // 'any' value for disk/table
#define FDSK_QUIET    ((USHORT) 0x0000)         // all disks, quiet operation

#define DFSPS_HEADERSIZE 512
#define DFSPS_ASCII_SIZE 384
#define DFSPS_DINFO_SIZE (DFSPS_HEADERSIZE - DFSPS_ASCII_SIZE)

#define DFSPS_ILEAD_SIZE  32                    // 2 lines in hexdump
#define DFSPS_ITAIL_SIZE 168                    // PARTINFO partial copy
#define DFSPS_INF64_SIZE  32                    // 2 lines in hexdump :)
#define DFSPS_SINFO_SIZE (DFSPS_ILEAD_SIZE + DFSPS_ITAIL_SIZE)

//- magic values overwriting first 8-bytes in DISKINFO struct stored in PSAVE
#define DFSPS_MAGIC1   ((DFSPARTINFO *) 0xFACE5AFE)
#define DFSPS_MAGIC2   ((DFSPARTINFO *) 0xDEADBEEF)

//- Structure version definitions, 00 used before DFSee 14.5, 01 after that
#define DFSPS_V_00_32BIT   0
#define DFSPS_V_01_64BIT   1

typedef struct dfsps_sec
{
   ULONG               psn32;                   // sector number  (32bit!)
   ULONG               sects;                   // nr of sectors
   BYTE                type;                    // type of sector
   BYTE                pdxVer;                  // version of this structure
   USHORT              reserved;                // filler
   ULONG               infosize;                // size of info   (unused)
   char                info[ DFSPS_ILEAD_SIZE]; // text info area, leading
   char                ibin[ DFSPS_ITAIL_SIZE]; // binary info area
   ULN64               psn;                     // sector number  (64bit!)
   char                in64[ DFSPS_INF64_SIZE]; // text info area, 64bit
} DFSPS_SEC;                                    // end of struct "dfsps_sec"


typedef enum fdsk_cb_phase                      // FDSK callback phase
{
   FDSK_PREPARE,                                // automation and confirmation
   FDSK_PERFORM,                                // perform initial action
   FDSK_PENDING                                 // perform pending action
} FDSK_CB_PHASE;                                // end of enum "fdsk_cb_phase"

typedef enum fdsk_cr_mbr
{
   FDSK_CR_MBR_OFF,                             // do NOT create an MBR
   FDSK_CR_MBR_STD,                             // create standard MBR if needed
   FDSK_CR_MBR_EMPTY                            // create an EMPTY MBR if needed
} FDSK_CR_MBR;                                  // end of enum "fdsk_cr_mbr"


typedef struct fdsk_numval                      // numeric value with options
{
   ULN64               value;                   // OUT   numeric value
   TXTS                prefix;                  // INOUT allowed/used prefix
   TXTS                format;                  // INOUT allowed/used format
   TXTS                type;                    // INOUT allowed/used type
} FDSK_NUMVAL;                                  // end of struct "fdsk_numval"

#define FDSK_NUMZERO  {0,"","dxoDXO", "mkgtchs"}

//- 20170408 JvW: Updated for 64bit SN/size
typedef struct fdsk_cb_info                     // FDSK callback info, most
{                                               // semantics defined by func
   USHORT              disknr;                  // Physical disknr or FDSK_ANY
   USHORT              partnr;                  // Ptable entry number (PTEdit)
   ULONG               creMbr;                  // create MBR if needed
   BOOL                creEbr;                  // create EBR, do not read
   BOOL                related;                 // include related sector(s) too
   BOOL                usebase;                 // use basePsn, not partPsn
   BOOL                nowrite;                 // no confirm, no write-back
   BOOL                verbose;                 // give progress information
   BOOL                confirmed;               // no confirmations wanted
   BOOL                abortNc;                 // abort on 1st NO_CHANGE
   BYTE                stype;                   // selection type
   BYTE                ntype;                   // new or second type
   BOOL                option;                  // select option
   BOOL                flag;                    // informational flag
   BOOL                doAllBBR;                // incl BBR on non 0x35 types
   ULN64               thisSn;                  // this sector number value
   ULN64               sn;                      // function sector number value
   ULN64               number;                  // informational number value (64 bit now! 16.6)
   FDSK_NUMVAL        *pos;                     // position information
   FDSK_NUMVAL        *size;                    // size     information
   void               *more;                    // link to more info
   BOOL                result;                  // result indicator
   TXLN                string;                  // informational line of text
   BOOL                cbOption1;               // callback option 1
   BOOL                cbOption2;               // callback option 2
   BOOL                cbOption3;               // callback option 3
   BOOL                cbOption4;               // callback option 4
   ULONG               cbOptNum1;               // callback number value 32bit
   ULN64               cbOptNum2;               // callback number value 64bit
   TXTT                cbOptStr1;               // callback small string
   #if defined (DEV32)
      BOOL             lvmSync;                 // Sync LVM engine too
      char             syncName[BMSYS_LONG +2]; // Sync volume name
      BYTE             syncLetter;              // Sync driveletter
   #endif
   TXTS                bname;                   // short name (bootmanager)
   char                pname[BMSYS_LONG +2];    // partition name (LVM 20, GPT 36)
   char                vname[BMSYS_LONG +2];    // volume name    (LVM 20)
   DFS_GUID            typeGuid;                // GPT type-GUID as binary (16)
   DFS_GUID            partGuid;                // GPT part-GUID as binary (16)
} FDSK_CB_INFO;                                 // end of struct "fdsk_cb_info"

// Generic FDISK callback function operating on bootrecord partition-tables
// and the corresponding partition-information structures
// Note: The function can be called multiple times, see 'fdsk_cb_phase'
// PENDING phase only called when PERFORM call returns PENDING
// Sequence is: 1) Caller reads info and MBR/EBR sector
//              2) Function performs user-confirmation, return NO_ERROR
//              3) Function performs operation on bootrecord, return PENDING
//              4) Caller writes back bootrecord, calls again with 'pending'
//              5) Function updates the partition information, return NO_ERROR
//
typedef ULONG (* FDSK_FUNCTION)
(
   DFSPARTINFO        *p,                       // INOUT partition info
   S_BOOTR            *br,                      // INOUT sector with Ptable
   FDSK_CB_PHASE       phase,                   // IN    callback phase
   FDSK_CB_INFO       *info                     // INOUT function info+data
);                                              //       like FDSK_CB_INFO


// Generic FDISK callback function operating on multiple sectors of any type
// Note: The function can be called multiple times, see 'fdsk_cb_phase'
// PENDING phase only called when PERFORM call returns PENDING
// Sequence is: 1) Caller reads sectors
//              2) Function performs user-confirmation, return NO_ERROR
//              3) Function performs operation on bootrecord, return PENDING
//              4) Caller writes back bootrecord, calls again with 'pending'
//              5) Function can do required post-processing, return NO_ERROR
//
typedef ULONG (* FDSK_MS_FUNCTION)
(
   BYTE               *sec,                     // INOUT sectors
   FDSK_CB_PHASE       phase,                   // IN    callback phase
   FDSK_CB_INFO       *info                     // INOUT function info+data
);                                              //       like FDSK_CB_INFO


// Generic PRESTORE callback function operating on PsaveD sectors + info
//
typedef ULONG (* FDSK_PS_FUNCTION)
(
   BYTE               *sec,                     // IN    sector contents
   DFSPS_SEC          *psi,                     // IN    PsaveD sector info
   FDSK_CB_INFO       *info                     // INOUT function info+data
);                                              //       like FDSK_CB_INFO


// Write boot-code-part for an MBR sector, optional clean partition-table
ULONG dfsFdskNewMbr
(
   USHORT              disknr,                  // IN    disknr to work on
   BOOL                clean                    // IN    clean part-table
);

// Clear LVM DLAT sectors in MBR track clean (any geo), optional skip current
ULONG dfsFdskClearLvmSectors
(
   USHORT              diskspec,                // IN    disk(s) to work on
   BOOL                skipcurrent              // IN    skip DLAT for current geo
);

// Write NT-signature in MBR sector, with specified or (DFSee) default value
ULONG dfsFdskNtSignature
(
   USHORT              disknr,                  // IN    disknr to work on
   char               *newSig                   // IN    signature value or ""
);

// Test if a given partition-type is hidden
BOOL dfsPartTypeHidable
(
   BYTE                type                     // IN    partition type
);


// Translate PartitionType to ASCII description
BOOL dfsPartTypeDescription                     // RET   hidden type
(
   BYTE                type,                    // IN    partition type
   char               *descr                    // OUT   description string
);


// Compare DFSPARTINFO elements for qsort, sort on: disknr, start-PSN
int dfsComparePartInfo
(
   const void         *one,
   const void         *two
);


// Determine and set type for each identified area of freespace + statistics
// GPT: adjust first/last freespace areas for disk for reserved HDR+PTA area
void dfsSetFreeSpaceType
(
   void
);

// Test if specified freespace type is compatible with partition type PRI/LOG
// or if it is ANY GPT freespace (move/copy to GPT freespace is always OK)
BOOL dfsFreeSpaceTypeMatch                      // RET   fspType is compatible
(
   BYTE                fspType,                 // IN    freespace type
   BOOL                delPriSameDisk,          // IN    Pri samedisk deleted
   BOOL                primary                  // IN    partition is primary
);

// Iterate over all or selected partitions, using ExecOn and callback
ULONG dfsIteratePart                            // all parts, optional ebr
(
   FDSK_FUNCTION       callback,                // IN    operation callback
   FDSK_CB_INFO       *cbi                      // INOUT call data
);


// Iterate over all or selected MBR-partitions, using ExecOn and callback
ULONG dfsIteratePmbr                            // primaries + ext-container
(
   FDSK_FUNCTION       callback,                // IN    operation callback
   FDSK_CB_INFO       *cbi                      // INOUT call data
);


// Iterate over all or selected EBR-partitions, using ExecOn and callback
ULONG dfsIterateExtContainers                   // all ebr's
(
   FDSK_FUNCTION       callback,                // IN    operation callback
   FDSK_CB_INFO       *cbi                      // INOUT call data
);


// Patch the in-memory ebr-chain, re-route a specified link to new info-block
BOOL dfsInsertInMemEbrChain
(
   USHORT              disknr,                  // IN    physical disk nr
   DFSPARTINFO        *oldLink,                 // IN    value to replace
   DFSPARTINFO        *newLink,                 // IN    replacement value
   DFSPARTINFO       **prevEbr                  // OUT   prev ebr or NULL
);


// Iterate over all or selected MBR/EBR sectors and/or related LVM sectors
// cb flag       indicates that MBR/EBR are the primary records to process
// cb related    indicates that both LVM and MBR/EBR need to be processed
// cb option     for LVM, process signature sectors too (BBR, feature info)
// doAllBBR      include signature sectors (BBR) for non 0x35 type too!
ULONG dfsIteratePlvm                            // all partition and/or LVM's
(
   FDSK_MS_FUNCTION    callback,                // IN    operation callback
   FDSK_CB_INFO       *cbi                      // INOUT call data
);


// Process for a single partition, MBR/EBR sectors and/or related LVM sectors
// cb flag       indicates that MBR/EBR are the primary records to process
// cb related    indicates that both LVM and MBR/EBR need to be processed
// cb option     for LVM, the signature sectors need to be processed as well
ULONG dfsProcessPlvm                            // single partition and/or LVM
(
   FDSK_MS_FUNCTION    callback,                // IN    operation callback
   FDSK_CB_INFO       *cbi                      // INOUT call data (.more = p)
);


// Iterate over all or selected Free-space areas, callback (skip dummy disks)
ULONG dfsIterateFree                            // all freespace areas
(
   FDSK_FUNCTION       callback,                // IN    operation callback
   FDSK_CB_INFO       *cbi                      // INOUT call data
);


// Iterate over all sections in a PsaveD file, calling back on each sector
// cb nowrite    indicates read-only, no write operations to be performed
// cb verbose    indicates extra (confirmation) info to be displayed
// cb string     options for the Prestore operation
// cb more       filepointer PsaveD file, opened for read at 1st info section
ULONG dfsIteratePsaved                          // all sections in PsaveD file
(
   FDSK_PS_FUNCTION    callback,                // IN    operation callback
   FDSK_CB_INFO       *cbi                      // INOUT call data
);


// Execute on a partition table in a specified MBR/EBR record
// 1) Read (or create a new) MBR/EBR or PBR sector
// 2) Execute (callback) confirmation for operation to perform
// 3) Execute (callback) operation on specified partition-table (entry)
// 4) If allowed/requested, write the sector back to the disk
// 5) Execute (callback) pending operation after successful write
ULONG dfsExecOnBootRec
(
   DFSPARTINFO        *p,                       // IN    partition to work on
   USHORT              disknr,                  // IN    verify disknr, or 0
   FDSK_FUNCTION       callback,                // IN    operation callback
   FDSK_CB_INFO       *cbi                      // INOUT call data
);


// Execute on one or more sectors of any type
// 1) Read the sector(s)
// 2) Execute (callback) confirmation for operation to perform
// 3) Execute (callback) operation on the sector(s)
// 4) If allowed/requested, write the sector(s) back to the disk
// 5) Execute (callback) pending operation after successful write
ULONG dfsExecOnSectors
(
   FDSK_MS_FUNCTION    callback,                // IN    operation callback
   FDSK_CB_INFO       *cbi                      // INOUT call data
);


// Show contents of one partition-table entry (as in 'walk')
void dfsShowPartEntry
(
   char               *lead,                    // IN    leading text 1st line
   BYTE                nav_st,                  // IN    update nav, using st
   BOOL                showpid,                 // IN    PID on current disk
   ULONG               geoHeads,                // IN    geometry, heads
   ULONG               geoSecs,                 // IN    geometry, sectors
   ULONG               thisPsn,                 // IN    PSN of table sector
   DFSPARTENTRY       *pe                       // IN    part entry ptr
);

// Macro to display a size in megabytes for a partition or freespace in table
// Sizes less than 0.95 MiB in three digits, above that with one (so 1.0 MiB)
#define dfsShowMiBsize(sc,bps) TxPrint((((sc) * (bps)) < 996000) ? "%11.3lf\n" : "%11.1lf\n", TXSMIB((sc), (bps)))


// Show freespace area for specified freespace info structure
void dfsShowFreeSpaceArea
(
   DFSPARTINFO        *f,                       // IN    freespace area info
   int                 cw                       // IN    name column width
);

// Show non-MBR disk (large-floppy/luks) and display its size in the table
void dfsShowNonMbrDiskArea
(
   DFSDISKINFO        *d,                       // IN    disk information
   int                 cw                       // IN    name column width
);

// Show freespace area for specified freespace info structure, comma separated
void dfsShowFreeSpaceSep
(
   DFSPARTINFO        *f,                       // IN    freespace area info
   char               *x,                       // IN    separator string
   char               *eol,                     // IN    end-of-line string
   char               *line                     // IN    line buffer, 1K
);

// Show freespace area for specified freespace, comma separated query format
void dfsShowFreeSpaceQuery
(
   DFSPARTINFO        *f,                       // IN    freespace area info
   char               *x,                       // IN    separator string
   char               *eol,                     // IN    end-of-line string
   char               *line                     // IN    line buffer, 1K
);

// Show non-MBR type of disk info (CRP, FLP etc), comma separated query format
void dfsShowNonMbrQuery
(
   DFSDISKINFO        *d,                       // IN    disk information
   char               *x,                       // IN    separator string
   char               *eol,                     // IN    end-of-line string
   char               *line                     // IN    line buffer, 1K
);

// Show verbose partition-info table for specified partition
void dfsVerboseShowPart
(
   DFSPARTINFO        *p                        // IN    partition info
);


// Show compact partition-info table for specified partition
void dfsCompactShowPart
(
   DFSPARTINFO        *p                        // IN    partition info
);

// Get one-line compact partition description in a string, 50 chars
void dfsGetPartDesc
(
   DFSPARTINFO        *p,                       // IN    partition info
   char               *descr                    // OUT   description, 50 chars
);

// Show errors/warnings for specified partition
void dfsShowPartWarnings
(
   DFSPARTINFO        *p,                       // IN    partition/EBR info
   ULONG               mask,                    // IN    warning filter mask
   ULONG              *minors                   // INOUT minor count, or NULL
);

// Show errors/warnings for specified disk
void dfsShowDiskWarnings
(
   DFSDISKINFO        *d,                       // IN    disk info
   ULONG               mask                     // IN    warning filter mask
);

// Set driveletter for specified partition in PARTINFO structure
ULONG dfsSetDriveLetter
(
   DFSPARTINFO        *p,                       // IN    partition info
   char               *letter                   // IN    new drive letter
);

// Find FIRST GPT partition with given partition-GUID (to check uniqueness)
USHORT dfsGuid2Pid                              // RET   found PID or 0
(
   DFS_GUID            guid                     // IN    partition GUID value
);

// Identify MBR (or an EBR) using CRC and display description
ULONG dfsMbrEbrIdentifyDisplay
(
   BYTE               *mbr,                     // IN    MBR to identify
   char               *lead                     // IN    8 char lead or NULL
);

// Identify MBR (or an EBR) using CRC, return maximum 8 character description
ULONG dfsMbrEbrBootcodeDescription
(
   BYTE               *mbr,                     // IN    MBR/EBR to identify
   char               *descr                    // OUT   Max 8 char description
);

// Read EBR sector from specified sectornumber and get Bootcode description
ULONG dfsGetBootcodeDescription
(
   ULN64               psn,                     // IN    MBR/EBR sectornumber
   char               *description              // OUT   Max 8 char description
);

// Build selection-list with partition info for all partitions present
TXSELIST *dfsPartSelist                         // RET   selection list or NULL
(
   BOOL                incall,                  // IN    include 'all' choice
   USHORT              disknr,                  // IN    disknr or FDSK_ANY
   char                select                   // IN    P=prim, L=log else ALL
);

// Build selection-list with disk info for all physical and virtual disks
TXSELIST *dfsDiskSelist                         // RET   selection list or NULL
(
   BOOL                incall,                  // IN    include 'all' choice
   BOOL                excDummy                 // IN    exclude 'dummy disks'
);

// Build selection-list with freespace info for all physical and virtual disks
TXSELIST *dfsFreeSelist                         // RET   selection list or NULL
(
   void
);

// Enable only matching freespace-areas in the Selist, return enabled count
ULONG dfsFreeSlEnable                           // RET   nr of matching freespaces
(
   TXSELIST           *slFree,                  // IN    List of freespaces
   ULN64               minSize,                 // IN    minimum required size
   BOOL                create,                  // IN    suitable for CREATE
   USHORT              disk,                    // IN    WASTED on this disk OK
   char                fspKind                  // IN    Kind wanted: G, P, L or *
);

// Build selection-list with info for all BMGR-bootable partitions
TXSELIST *dfsBmgrSelist                         // RET   selection list or NULL
(
   void
);

// Get DISK identification string, Make/Model/serial etc, in specified buffer
BOOL dfsGetDiskIdentification                   // RET   Disk ID info present
(
   USHORT              disknr,                  // IN    DFSee disk number
   char               *unixDev,                 // IN    Unix device name, or ""
   int                 length,                  // IN    Fixed output length or 0 (all)
   TXTM                idString                 // OUT   Disk ID string, or error text
);

#endif
