//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// Partition utility, display and iterator functions; main section
//
// Author: J. van Wijk
//
// JvW  18-04-99   Initial version, split from DFSUFDSK
// JvW  03-03-2000 Added default NT-admin signature 'DF5EE318' for newmbr
// JvW  13-04-2000 Fixed typo in message (unlimited)
// JvW  18-04-2000 Changed default NTsignature, now different for each disk
// JvW  29-06-2001 Fixed calculated value (end sector) for non extended
// JvW  21-09-2002 I13X capable MBR code and MBR checksum and determination
// JvW  23-09-2002 Added PartSelist() to create partition selection list
// JvW  18-08-2005 Split off huge parts to dfsuparN.c files, for modularity
//

#include <txlib.h>                              // ANSI controls and logging

#if !defined (OEMSB)
#include <dfsver.h>                             // DFS version info
#endif

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsupart.h>                           // partition utility functions
#include <dfsufdsk.h>                           // Fdisk & translation services
#include <dfsufgpt.h>                           // FDISK GPT defs and functions
#include <dfsutil.h>                            // DFS utility functions
#include <dfsafdsk.h>                           // FDISK generic definitions


#if defined (USEWINDOWING)

static char dfsNoPrimaries[] = "Primary partitions are not supported for this function.";
static char dfsNoLogicals[]  = "Logical partitions are not supported for this function.";

static char dfsNoAccesMsg[] =
 "Check for missing drivers or hardware problems"
                      #if   defined (WIN32)
                       "; Logon as an ADMINISTATOR!";
                      #elif defined (UNIX)
                       "; Logon as 'root' or use 'sudo'!";
                      #elif defined (DOS32)
                       " like bad/disconnected cabling";
                      #else
                       "; On HPFS386: log on as ADMIN!";
                      #endif

// Define sort information for partition lists
static TXSELSORT       dfs_part_sort =
{
  {                                             // actual, initial sort order
     0,                                         // none
     TXS_SORT_TEXT | TXS_SORT_ON_CASE | TXS_SORT_ASCEND  | 47, // (size)
     TXS_SORT_TEXT | TXS_SORT_ON_CASE | TXS_SORT_ASCEND  |  1, // (PID)
     TXS_SORT_TEXT | TXS_SORT_ON_CASE | TXS_SORT_ASCEND  | 36, // (type)
     TXS_SORT_TEXT | TXS_SORT_ON_CASE | TXS_SORT_ASCEND  | 29, // (letter)
     TXS_SORT_TEXT | TXS_SORT_IGNCASE | TXS_SORT_ASCEND  | 41, // (fsys)
     TXS_SORT_TEXT | TXS_SORT_IGNCASE | TXS_SORT_ASCEND  | 17, // (label)
     TXS_SORT_TEXT | TXS_SORT_IGNCASE | TXS_SORT_ASCEND  | 32  // (p/l)
  },
  {                                             // reverse   (c-F8 / c-R)
     "unsorted   -",                            // unsorted  (c-F7 / c-U)
     "total size -",                            // text   1  (c-F2 / c-X)
     "number PID -",                            // text   2  (c-F1 / c-D)
     "syst. type -",                            // text   3  (c-F9 / c-A)
     "dr. letter -",                            // desc   4  (c-F5 / c-T)
     "filesystem -",                            // desc   5  (c-F6 / c-B / c-S / c-F)
     "vol. label -",                            // desc   6  (c-F3 / c-N)
     "prim / log -",                            // desc   7  (c-F4 / c-E)
  },
  ""                                            // current description
};

/*
01 /dev/hda1    ECS22-DEV   D: Pri 0x07 JFS        2047.5 MiB
|  |            |           |  |   |    |     |
1  4            17          29 32  36   41    47
*/

#endif

/*****************************************************************************/
// Test if a given partition-type is hidden
/*****************************************************************************/
BOOL dfsPartTypeHidable
(
   BYTE                type                     // IN    partition type
)
{
   BOOL                rc = FALSE;              // function return

   if (type <= DFS_P_MAXHIDE)
   {
      switch (type & ~DFS_P_PHIDDEN)
      {
         case DFS_P_FAT12     :
         case DFS_P_FAT16     :
         case DFS_P_FAT16X    :
         case DFS_P_BIGDOS    :
         case DFS_P_INST_FS   :
         case DFS_P_FAT32     :
         case DFS_P_FAT32X    :
            rc = TRUE;
            break;

         default:
            break;
      }
   }
   return (rc);
}                                               // end 'dfsPartTypeHidable'
/*---------------------------------------------------------------------------*/



/*****************************************************************************/
// Translate PartitionType to ASCII description
/*****************************************************************************/
BOOL dfsPartTypeDescription                     // RET   hidden type
(
   BYTE                type,                    // IN    partition type
   char               *descr                    // OUT   description string
)
{
   BOOL                rc = FALSE;
   BYTE                ht;

   if (dfsPartTypeHidable(type) && (type > DFS_P_PHIDDEN))
   {
      ht = (BYTE) ((int) type & ~DFS_P_PHIDDEN);
      rc = TRUE;
   }
   else
   {
      ht = type;
   }
   switch (ht)
   {
      case DFS_P_EMPTY     : strcpy( descr, "Deleted! " ); break;
      case DFS_P_FAT12     : strcpy( descr, "FAT12    " ); break;
      case DFS_P_XENIX_S   : strcpy( descr, "XenixRoot" ); break;
      case DFS_P_XENIX_U   : strcpy( descr, "XenixUser" ); break;
      case DFS_P_FAT16     : strcpy( descr, "FAT16-32M" ); break;
      case DFS_P_EXTENDED  : strcpy( descr, "Extended " ); break;
      case DFS_P_BIGDOS    : strcpy( descr, "FAT16    " ); break;
      case DFS_P_INST_FS   : strcpy( descr, "Inst-FSys" ); break;
      case DFS_P_IFSHIDDEN : strcpy( descr, "IFShidden" ); break;
      case DFS_P_AIX_BOOT  : strcpy( descr, "AIX boot " ); break;
      case DFS_P_AIX_DATA  : strcpy( descr, "AIX data " ); break;
      case DFS_P_BOOTMGR   : strcpy( descr, "IBM-BMGR " ); break;
      case DFS_P_FAT32     : strcpy( descr, "FAT32    " ); break;
      case DFS_P_FAT32X    : strcpy( descr, "FAT32-Ext" ); break;
      case DFS_P_FAT16X    : strcpy( descr, "FAT16-Ext" ); break;
      case DFS_P_BIGEXTEND : strcpy( descr, "ExtendBig" ); break;
      case DFS_P_HIDEXTEND : strcpy( descr, "ExtendHid" ); break;
      case DFS_P_RECOVERY  : strcpy( descr, "Recovery " ); break;
      case DFS_P_ASTWINSWP : strcpy( descr, "AstWinSwp" ); break;
      case DFS_P_OXYGENEXT : strcpy( descr, "OxygenExt" ); break;
      case DFS_P_NECDOS_3X : strcpy( descr, "Nec-Dos3x" ); break;
      case DFS_P_WARP_LVM  : strcpy( descr, "Warp-LVM " ); break;
      case DFS_P_THEOS_V32 : strcpy( descr, "TheosV3.2" ); break;
      case DFS_P_THEOS_V4S : strcpy( descr, "TheosV4sp" ); break;
      case DFS_P_THEOS_V4P : strcpy( descr, "TheosV4  " ); break;
      case DFS_P_THEOS_V4X : strcpy( descr, "TheosV4Ex" ); break;
      case DFS_P_PQM_RECOV : strcpy( descr, "PQMrecovr" ); break;
      case DFS_P_VENIX_286 : strcpy( descr, "Venix-286" ); break;
      case DFS_P_WIND_PREP : strcpy( descr, "Wind-PREP" ); break;
      case DFS_P_WIN2XPLDM : strcpy( descr, "W2K/xpLDM" ); break;
      case DFS_P_LINUXDRSW : strcpy( descr, "LinuxDrSw" ); break;
      case DFS_P_GOBACK_DR : strcpy( descr, "GoBack-DR" ); break;
      case DFS_P_QNXVER4P1 : strcpy( descr, "QNXv4-1st" ); break;
      case DFS_P_QNXVER4P2 : strcpy( descr, "QNXv4-2nd" ); break;
      case DFS_P_QNXVER4P3 : strcpy( descr, "QNXv4-3rd" ); break;
      case DFS_P_ONTRACK_R : strcpy( descr, "OnTrack-R" ); break;
      case DFS_P_ONTRACKRW : strcpy( descr, "OnTrackRW" ); break;
      case DFS_P_CPM_RES_1 : strcpy( descr, "CP/M SysV" ); break;
      case DFS_P_ONTRACK_6 : strcpy( descr, "OnTrack-6" ); break;
      case DFS_P_ONTRACKDO : strcpy( descr, "OnTrackDO" ); break;
      case DFS_P_EZ_DRIVE  : strcpy( descr, "EZ-drive " ); break;
      case DFS_P_VFEATURE  : strcpy( descr, "V-feature" ); break;
      case DFS_P_PRI_EDISK : strcpy( descr, "Edisk-mgr" ); break;
      case DFS_P_SPEEDSTR  : strcpy( descr, "Speedstor" ); break;
      case DFS_P_SCO_SYSV  : strcpy( descr, "SCO/SystV" ); break;
      case DFS_P_NOVELL_R2 : strcpy( descr, "Novell-R2" ); break;
      case DFS_P_NOVELL_34 : strcpy( descr, "Novell3/4" ); break;
      case DFS_P_NOVELL386 : strcpy( descr, "Novell386" ); break;
      case DFS_P_NOVELL_X1 : strcpy( descr, "Novell-x1" ); break;
      case DFS_P_NOVELL_X2 : strcpy( descr, "Novell-x2" ); break;
      case DFS_P_NOVELL_X3 : strcpy( descr, "Novell-SS" ); break;
      case DFS_P_APFS_CONT : strcpy( descr, "APFS Cont" ); break;
      case DFS_P_PCIX_UNIX : strcpy( descr, "PC/IX    " ); break;
      case DFS_P_XOSL_BMGR : strcpy( descr, "XOSL-bmgr" ); break;
      case DFS_P_VRAIDPART : strcpy( descr, "VRAIDpart" ); break;
      case DFS_P_MINIX_13  : strcpy( descr, "Minix1.3-" ); break;
      case DFS_P_MINIX_15  : strcpy( descr, "Minix1.5+" ); break;
      case DFS_P_SWAPSOLAR : strcpy( descr, "SunS/SWAP" ); break;
      case DFS_P_LINUXNATV : strcpy( descr, "LinuxNatv" ); break;
      case DFS_P_HIBERNATE : strcpy( descr, "Hibernate" ); break;
      case DFS_P_LINUXEXTX : strcpy( descr, "ExtLinux " ); break;
      case DFS_P_BIGDOS_VS : strcpy( descr, "FAT16Vset" ); break;
      case DFS_P_IFS_MIRR  : strcpy( descr, "MirrorIFS" ); break;
      case DFS_P_AB_LINKRN : strcpy( descr, "LinuxKrnl" ); break;
      case DFS_P_LINUX_LVM : strcpy( descr, "Linux-LVM" ); break;
      case DFS_P_LINUXLUKS : strcpy( descr, "LinuxLUKS" ); break;
      case DFS_P_BIGLINEXT : strcpy( descr, "ExtLinBig" ); break;
      case DFS_P_AMOEBASYS : strcpy( descr, "AmoebaSys" ); break;
      case DFS_P_AMOEBA_BB : strcpy( descr, "Amoeba-BB" ); break;
      case DFS_P_LOGIC1024 : strcpy( descr, "MylexBigX" ); break;
      case DFS_P_SAVE2DISK : strcpy( descr, "Save2Disk" ); break;
      case DFS_P_SAVEDISK2 : strcpy( descr, "Save2-NEC" ); break;
      case DFS_P_FREE_BSD  : strcpy( descr, "FreeBSD  " ); break;
      case DFS_P_OPEN_BSD  : strcpy( descr, "OpenBSD  " ); break;
      case DFS_P_NEXTSTEP  : strcpy( descr, "NextStep " ); break;
      case DFS_P_MACXDATA  : strcpy( descr, "macOSdata" ); break;
      case DFS_P_NETBSD98  : strcpy( descr, "NetBSD-98" ); break;
      case DFS_P_OLISERVE  : strcpy( descr, "OlService" ); break;
      case DFS_P_MACXBOOT  : strcpy( descr, "macOSboot" ); break;
      case DFS_P_AOS2GPVD  : strcpy( descr, "AOS2-GPVD" ); break;
      case DFS_P_MACXHFSP  : strcpy( descr, "macOS HFS" ); break;
      case DFS_P_BSDI_FSYS : strcpy( descr, "BSDI-fsys" ); break;
      case DFS_P_BSDI_SWAP : strcpy( descr, "BSDI-swap" ); break;
      case DFS_P_ACRONIS_H : strcpy( descr, "AcrHidden" ); break;
      case DFS_P_ACRONISSZ : strcpy( descr, "AcronisSZ" ); break;
      case DFS_P_SOLARBOOT : strcpy( descr, "SunS-Boot" ); break;
      case DFS_P_SOLAR_ZFS : strcpy( descr, "Solaris-Z" ); break;
      case DFS_P_CTOS_FSYS : strcpy( descr, "CTOS-fsys" ); break;
      case DFS_P_DRDOSSF12 : strcpy( descr, "DrDosSF12" ); break;
      case DFS_P_DRDOSSF16 : strcpy( descr, "DrDosSF16" ); break;
      case DFS_P_BIGDOSCVS : strcpy( descr, "FAT-badVS" ); break;
      case DFS_P_CTOSMDUMP : strcpy( descr, "CTOSmDump" ); break;
      case DFS_P_IFS_DISAB : strcpy( descr, "DisablIFS" ); break;
      case DFS_P_CPM_86    : strcpy( descr, "CPM/86   " ); break;
      case DFS_P_CPM_RES_2 : strcpy( descr, "CPMreserv" ); break;
      case DFS_P_DELL_UTIL : strcpy( descr, "DELL-Util" ); break;
      case DFS_P_DFSNOWARN : strcpy( descr, "DfsNoWarn" ); break;
      case DFS_P_DFSEETYPE : strcpy( descr, "DFSeeType" ); break;
      case DFS_P_SPEEDS_12 : strcpy( descr, "Speedst12" ); break;
      case DFS_P_DOSACCESS : strcpy( descr, "DosAccess" ); break;
      case DFS_P_SPEEDS_16 : strcpy( descr, "Speedst16" ); break;
      case DFS_P_BEOS_FS   : strcpy( descr, "BeOS-Fsys" ); break;
      case DFS_P_EFI_GPT   : strcpy( descr, "GPT-guard" ); break;
      case DFS_P_EFI_ANY   : strcpy( descr, "Efi-Any-P" ); break;
      case DFS_P_EFI_ESP   : strcpy( descr, "Efi-Sys-P" ); break;
      case DFS_P_SDMAXTOR1 : strcpy( descr, "SdMaxtor1" ); break;
      case DFS_P_DOSSECOND : strcpy( descr, "DosSecond" ); break;
      case DFS_P_SDMAXTOR2 : strcpy( descr, "SdMaxtor2" ); break;
      case DFS_P_BOCHS_VM  : strcpy( descr, "BochsVirt" ); break;
      case DFS_P_VMWARSYST : strcpy( descr, "VmWareSys" ); break;
      case DFS_P_VMWARSWAP : strcpy( descr, "VmWarSwap" ); break;
      case DFS_P_LIN_NONFS : strcpy( descr, "LinxNonFS" ); break;
      case DFS_P_LINUXRAID : strcpy( descr, "LinuxRaid" ); break;
      case DFS_P_PS2SYST   : strcpy( descr, "PS/2-syst" ); break;
      case DFS_P_BADBLOCKS : strcpy( descr, "Xenix-Bbl" ); break;

      default:               strcpy( descr, "Unknown  " ); break;
   }
   return(rc);
}                                               // end 'dfsPartTypeDescription'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Compare DFSPARTINFO elements for qsort, sort on: disknr, start-PSN
/*****************************************************************************/
int dfsComparePartInfo
(
   const void         *one,
   const void         *two
)
{
   DFSPARTINFO        *p1  = (DFSPARTINFO *) one;
   DFSPARTINFO        *p2  = (DFSPARTINFO *) two;
   int                 res;

   if ((res = (int) (p1->disknr - p2->disknr)) == 0) // on same disk
   {
      if      (p1->basePsn > p2->basePsn)
      {
         res = 1;
      }
      else if (p1->basePsn < p2->basePsn)
      {
         res = -1;
      }
   }
   return(res);
}                                               // end 'dfsComparePartInfo'
/*---------------------------------------------------------------------------*/

#if !defined (OEMSB)
/*****************************************************************************/
// Determine and set type for each identified area of freespace + statistics
// GPT: adjust first/last freespace areas for disk for reserved HDR+PTA area
// 20170813 JvW Set special 'unpartionable-space' for non MBR/GPT style disks
/*****************************************************************************/
void dfsSetFreeSpaceType
(
   void
)
{
   USHORT              disk;                    // physical disk number
   DFSPARTINFO        *e;                       // ptr to related ebr info
   DFSPARTINFO        *f;                       // ptr to freespace info
   DFSDISKINFO        *d;                       // ptr to phys disk info
   BOOL                mbrFull;                 // all 4 mbr entries used
   BYTE                fspType;                 // freespace type
   ULN64               fspSize;                 // size of this freespace
   ULN64               gptPtaSectors;           // sectors in GPT PT Array

   ENTER();

   for (disk = 1; disk <= dfsPartitionableDisks(); disk++)
   {
      d = dfsGetDiskInfo(disk);
      if (d != NULL)
      {
         d->freeSpace = 0;
         d->freeSects = 0;

         mbrFull = (((d->primaries == 3) && (d->ebrHead)) || (d->primaries == 4));

         if (d->gptHeader != NULL)              // could be empty (0 partitions)
         {
            S_GPTHDR      *gptHeader = (S_GPTHDR *) d->gptHeader;

            //- calculate minimum nr of sectors MBR + GPT-HDR + GPT-PTA
            gptPtaSectors = ((gptHeader->ptaEntries * gptHeader->ptaEntrySize) / d->bpsector) + 2;
         }
         else
         {
            gptPtaSectors = 0;                  // MBR style!
         }

         for (f = d->fspHead; f != NULL; f = f->fspChain)
         {
            fspSize = f->lastPsn - f->basePsn +1;

            d->freeSpace++;
            d->freeSects += fspSize;

            TRACES(("disk %hu size:%llx PTA:%llx sectors, fspSize:%llx  fspFirst:%llx fspLast:%llx style:%s\n",
                                     disk, d->sectors, gptPtaSectors, fspSize, f->basePsn, f->lastPsn, d->pStyle));

            if ((d->pStyle[0] != DFSP_MBR) && (d->pStyle[0] != DFSP_GPT))
            {
               fspType = DFS_FSP_UNPS;          // Unpartionable space
            }
            else if (gptPtaSectors != 0)        // GPT style freespace
            {
               if      (f->lastPsn <  gptPtaSectors)
               {
                  fspType = DFS_FSP_GPTM;       // GPT PTA + MBR only
               }
               else if (f->basePsn == 0)        // start of disk, PTA + free
               {
                  fspType = DFS_FSP_GPTS;       // MBR + GPT PTA + freespace

                  f->basePsn = gptPtaSectors;                  //- adjust start sector
                  if (f->lastPsn >= (d->sectors -1))
                  {
                     f->lastPsn = d->sectors - gptPtaSectors;  //- adjust end sector
                  }
                  f->sectors = f->lastPsn - f->basePsn + 1;    //- adjust size
               }
               else if (f->lastPsn >= (d->sectors -1))
               {
                  if ((f->basePsn + gptPtaSectors) >= (d->sectors -1))
                  {
                     fspType = DFS_FSP_GPTE;    // GPT PTA at end
                  }
                  else
                  {
                     fspType = DFS_FSP_GPTC;    // GPT PTA at end + freespace

                     f->lastPsn = d->sectors - gptPtaSectors;   // adjust end sector and size
                     f->sectors = f->lastPsn - f->basePsn + 1;
                  }
               }
               else
               {
                  fspType = DFS_FSP_GPTF;       // GPT Freespace
               }
            }
            else                                // MBR style freespaces
            {
               if      (f->lastPsn < f->geoSecs) // Track-Zero space
               {
                  fspType = DFS_FSP_ZERO;
               }
               else if ((fspSize < f->cSC) && (f->lastPsn >= (d->sectors -1)))
               {
                  fspType = DFS_FSP_ENDC;       // partial end-cylinder
               }
               else if (f->lastPsn < f->cSC)    // Cylinder-Zero space
               {
                  fspType = (mbrFull) ? DFS_FSP_NONE : DFS_FSP_PRIM;
               }
               else if ((e = d->ebrHead) == NULL) // no extended partition yet
               {
                  fspType = (mbrFull) ? DFS_FSP_NONE : DFS_FSP_NEWX;
               }
               else                             // extended partition present
               {
                  TRACES(("Disk: %hu freespace-area %hu\n", disk, f->id));
                  TRACES(("f->base:%llx last:%llx\n", f->basePsn, f->lastPsn));
                  TRACES(("e->base:%llx last:%llx\n", e->basePsn, e->lastPsn));
                  if       (f->lastPsn <   e->basePsn -1)
                  {
                     fspType = (mbrFull) ? DFS_FSP_NONE : DFS_FSP_PRIM;
                  }
                  else if (f->flag2 & DFS_F_EMPTYCONTN)
                  {
                     fspType = DFS_FSP_ELOG;    // empty EBR container in FSP!
                  }
                  else if ((f->basePsn <= e->basePsn   ) &&
                           (f->lastPsn >= e->basePsn -1) )
                  {
                     fspType = (mbrFull) ? DFS_FSP_HLOG : DFS_FSP_HEAD;
                  }
                  else if ((f->basePsn >  e->basePsn   ) &&
                           (f->lastPsn <  e->lastPsn   ) )
                  {
                     if ((f->ebrChain != NULL) &&
                         (f->ebrChain->primary == FALSE))
                     {
                        fspType = DFS_FSP_LOG;
                     }
                     else                       // last, or followed by primary
                     {
                        fspType = (mbrFull) ? DFS_FSP_LOG : DFS_FSP_TAIL;
                     }
                  }
                  else if ((f->basePsn <= e->lastPsn +1) &&
                           (f->lastPsn >= e->lastPsn   ) )
                  {
                     fspType = (mbrFull) ? DFS_FSP_TLOG : DFS_FSP_TAIL;
                  }
                  else
                  {
                     fspType = (mbrFull) ? DFS_FSP_NONE : DFS_FSP_PRIM;
                  }
               }
            }
            switch (fspType)
            {
               case DFS_FSP_UNPS : strcpy( f->descr, "Unpartitionable! "); break;
               case DFS_FSP_GPTM : strcpy( f->descr, "MBR + GPT hdr/pta"); break;
               case DFS_FSP_GPTS : strcpy( f->descr, "Fsp + GPT hdr/pta"); break;
               case DFS_FSP_GPTE : strcpy( f->descr, "atEnd GPT pta/hdr"); break;
               case DFS_FSP_GPTC : strcpy( f->descr, "Fsp + GPT pta/hdr"); break;
               case DFS_FSP_GPTF : strcpy( f->descr, "FreeSpace GPT    "); break;
               case DFS_FSP_NONE : strcpy( f->descr, "FreeSpace Wasted "); break;
               case DFS_FSP_PRIM : strcpy( f->descr, "FreeSpace Primary"); break;
               case DFS_FSP_HLOG :
               case DFS_FSP_TLOG :
               case DFS_FSP_LOG  : strcpy( f->descr, "FreeSpace Logical"); break;
               case DFS_FSP_ELOG : strcpy( f->descr, "FreeSpace EXT/Log"); break;
               case DFS_FSP_NEWX :
               case DFS_FSP_HEAD :
               case DFS_FSP_TAIL : strcpy( f->descr, "FreeSpace Pri/Log"); break;
               case DFS_FSP_ENDC : strcpy( f->descr, "Partial Cylinder "); break;
               case DFS_FSP_ZERO : strcpy( f->descr, "MBR +Track-0 Area"); break;
               default           : strcpy( f->descr, "FreeSpace Unknown"); break;
            }
            TRACES(("Freespace-type: %hu  descr: '%s'\n", fspType, f->descr));
            f->partent.PartitionType = fspType;
         }
      }
   }
   VRETURN();
}                                               // end 'dfsSetFreeSpaceType'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Test if specified freespace type is compatible with partition type PRI/LOG
// or if it is ANY GPT freespace (move/copy to GPT freespace is always OK)
/*****************************************************************************/
BOOL dfsFreeSpaceTypeMatch                      // RET   fspType is compatible
(
   BYTE                fspType,                 // IN    freespace type
   BOOL                delPriSameDisk,          // IN    Pri samedisk deleted
   BOOL                primary                  // IN    partition is primary
)
{
   BOOL                rc = TRUE;               // function return

   ENTER();
   TRACES(( "Freespace type: %2.2hx, partition %s\n",
             fspType, (primary) ? "PRI" : "LOG"));

   if (fspType > DFS_FSP_GPT_STYLE)             // GPT is OK, check for MBR
   {
      if (primary)                              // Part to be created is Pri
      {
         if (delPriSameDisk)                    // Primary on same disk will be
         {                                      // deleted before create ...
            rc = (fspType != DFS_FSP_LOG);      // Just real Logical freespace
         }                                      // within EXT container not OK
         else
         {
            //- Pri, Pri/Log OK, but not WASTED
            rc = ((fspType & DFS_FSP_PRIMASK) == DFS_FSP_PRIMASK);
         }
      }
      else
      {
         rc    = ((fspType & DFS_FSP_LOGMASK) == DFS_FSP_LOGMASK);
      }
   }
   BRETURN (rc);
}                                               // end 'dfsFreeSpaceTypeMatch'
/*---------------------------------------------------------------------------*/
#endif


/*****************************************************************************/
// Iterate over all or selected MBR/EBR sectors and/or related LVM sectors
// cb flag       indicates that MBR/EBR are the primary records to process
// cb related    indicates that both LVM and MBR/EBR need to be processed
// cb option     for LVM, process signature sectors too (BBR, feature info)
// doAllBBR      include signature sectors (BBR) for non 0x35 type too!
// 20170815 JvW  Full-function on MBR style disks only, nothing on rest
/*****************************************************************************/
ULONG dfsIteratePlvm                            // all partition and/or LVMs
(
   FDSK_MS_FUNCTION    callback,                // IN    operation callback
   FDSK_CB_INFO       *cbi                      // INOUT call data
)
{
   ULONG               rc = NO_ERROR;           // function return
   DFSDISKINFO        *d;                       // ptr to disk info
   DFSPARTINFO        *e;                       // ptr to EBR  partition info
   DFSPARTINFO        *p;                       // ptr to real partition info
   USHORT              disk;                    // current disk index
   USHORT              part;                    // current partition index
   USHORT              parts;                   // number of partitions
   USHORT              diskselect;              // disk selection

   ENTER();
   if ((callback != NULL) && (cbi != NULL))
   {
      diskselect = cbi->disknr;
      for ( disk = 1;
           (disk <= dfsPartitionableDisks()) && ((rc == NO_ERROR) || (rc == DFS_NO_CHANGE));
            disk++)
      {
         if ((disk == diskselect) || (diskselect == FDSK_ANY))
         {
            TRACES(("IteratePlvm for disk nr: %hu\n", disk));
            cbi->disknr   = disk;               // disk to read from
            cbi->number   = 1;                  // allways one sector
            d = dfsGetDiskInfo( disk);

            if ((d != NULL) && ((d->pStyle[0] == DFSP_MBR) || (d->pStyle[0] == DFSP_GPT)))
            {
               if      (((cbi->option) || (cbi->cbOptNum1 == 1)) && (cbi->nowrite == FALSE) &&
                        ((d->gptHeader != NULL) && (d->gptArray != NULL)))
               {
                  //- LVM write operation on a GPT style disk
                  TxPrint( "\nDisk %hu, is using the GPT partitioning style, incompatible with OS/2 LVM!\n\n", disk);
               }
               else if (((cbi->option) || (cbi->cbOptNum1 == 1)) && (cbi->nowrite == FALSE) && (d->geoSecs <= 1))
               {
                  //- LVM write operation on disk with 1 sector/track
                  TxPrint( "\nDisk %hu, with %hu sectors/track is incompatible with OS/2 LVM!\n"
                             "There is no room for the LVM information sector(s)\n\n", disk, d->geoSecs);
               }
               else if (d->OpenError == FALSE)  // skip unaccessible disks too
               {
                  if (cbi->flag || cbi->related)
                  {
                     cbi->sn       = 0;         // MBR sector
                     rc = dfsExecOnSectors( callback, cbi);
                     if ((rc == DFS_NO_CHANGE) && (cbi->abortNc == FALSE))
                     {
                        rc = NO_ERROR;          // one skipped, continue
                     }
                  }
                  if (((cbi->flag == FALSE) || (cbi->related)) && (d->pStyle[0] == DFSP_MBR) && (rc == NO_ERROR))
                  {
                     cbi->sn       = d->geoSecs -1; // MBR-LVM sector
                     rc = dfsExecOnSectors( callback, cbi);
                     if ((rc == DFS_NO_CHANGE) && (cbi->abortNc == FALSE))
                     {
                        rc = NO_ERROR;          // one skipped, continue
                     }
                     if ((rc == NO_ERROR) && cbi->option)
                     {
                        parts = dfsPartitions(); // handle signature sectors
                        for ( part = 1;
                             (part <= parts) && (rc == NO_ERROR) && !TxAbort();
                              part++)
                        {
                           p = dfsGetPartInfo( part);
                           if (p && (p->disknr == disk) && (p->primary))
                           {
                              if ((cbi->doAllBBR) || // check BBR on non 0x35 too
                                  (p->partent.PartitionType == DFS_P_WARP_LVM))
                              {
                                 TRACES(("Check BBR for PID: %hu at 0x%llx\n", p->id, p->lastPsn));

                                 cbi->sn   = p->lastPsn; // possible LVM-signature
                                 rc = dfsExecOnSectors( callback, cbi);
                                 if ((rc == DFS_NO_CHANGE) && (cbi->abortNc == FALSE))
                                 {
                                    rc = NO_ERROR; // one skipped, continue
                                 }
                                 else if (rc != NO_ERROR)     //- could be 'beyond end-disk'
                                 {                            //- ignore read errors, continue!
                                    TxPrint( "\nDisk %hu, last sector 0x%llu (LVM BBR area) for partition %2.2hu "
                                               "can not be read!\n", disk, p->lastPsn, p->id);
                                    rc = NO_ERROR;
                                 }
                              }
                           }
                        }
                     }
                  }
                  for ( e = d->ebrHead;
                       (e) && (rc == NO_ERROR) && !TxAbort();
                        e = e->ebrChain)
                  {
                     if (cbi->flag || cbi->related)
                     {
                        cbi->sn       = e->basePsn; // EBR sector
                        rc = dfsExecOnSectors( callback, cbi);
                        if ((rc == DFS_NO_CHANGE) && (cbi->abortNc == FALSE))
                        {
                           rc = NO_ERROR;       // one skipped, continue
                        }
                     }
                     if (((cbi->flag == FALSE) || (cbi->related)) && (rc == NO_ERROR))
                     {
                        cbi->sn = e->basePsn + d->geoSecs -1; // EBR-LVM-info
                        rc = dfsExecOnSectors( callback, cbi);
                        if ((rc == DFS_NO_CHANGE) && (cbi->abortNc == FALSE))
                        {
                           rc = NO_ERROR;       // one skipped, continue
                        }
                        if (rc == NO_ERROR)
                        {
                           if (cbi->option)     // handle signature sector
                           {
                              p = dfsGetPartInfo( e->id);
                              if (p != NULL)
                              {
                                 if ((cbi->doAllBBR) || // check BBR on non 0x35 too
                                     (p->partent.PartitionType == DFS_P_WARP_LVM))
                                 {
                                    TRACES(("Check BBR for PID: %hu at 0x%llx\n", e->id, p->lastPsn));

                                    cbi->sn   = p->lastPsn; // possible LVM-signature
                                    rc = dfsExecOnSectors( callback, cbi);
                                    if ((rc == DFS_NO_CHANGE) && (cbi->abortNc == FALSE))
                                    {
                                       rc = NO_ERROR; // one skipped, continue
                                    }
                                    else if (rc != NO_ERROR)     //- could be 'beyond end-disk'
                                    {                            //- ignore read errors, continue!
                                       TxPrint( "\nDisk %hu, last sector 0x%llu (LVM BBR area) for partition %2.2hu "
                                                  "can not be read!\n", disk, p->lastPsn, p->id);
                                       rc = NO_ERROR;
                                    }
                                 }
                              }
                           }
                        }
                     }
                  }
               }
            }
            else if (d != NULL)
            {
               //- TxPrint( "\nDisk %hu, is using the %s partitioning style, incompatible with OS/2 LVM!\n", disk, d->pStyle);
               TRACES(("Skipped disk %hu using partitioning style '%s'\n", disk, d->pStyle));
            }
         }
         cbi->confirmed = FALSE;                // reconfirm for next disk
      }
      cbi->disknr = diskselect;                 // restore original value
   }
   RETURN (rc);
}                                               // end 'dfsIteratePlvm'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Execute on one or more sectors of any type
// 1) Read the sector(s)
// 2) Execute (callback) confirmation for operation to perform
// 3) Execute (callback) operation on the sector(s)
// 4) If allowed/requested, write the sector(s) back to the disk
// 5) Execute (callback) pending operation after successful write
/*****************************************************************************/
ULONG dfsExecOnSectors
(
   FDSK_MS_FUNCTION    callback,                // IN    operation callback
   FDSK_CB_INFO       *cbi                      // INOUT call data
)
{
   ULONG               rc  = NO_ERROR;          // function return
   BYTE               *sec = NULL;              // sector buffer
   TXTM                text;                    // progress text

   ENTER();
   TRACES(("disknr: %hu  sn: %llX  nr: %u\n", cbi->disknr, cbi->sn, cbi->number));
   if (cbi->disknr != 0)                        // specific disk ?
   {
      rc = dfsSelectDisk( cbi->disknr, FALSE, FALSE); // Select disk quietly
   }                                            // otherwise, use STORE as is
   if (rc == NO_ERROR)
   {
      if ((sec = (TxAlloc( cbi->number, dfsGetSectorSize()))) != NULL)
      {
         rc = dfstReadPsn( DFSTORE, cbi->sn, cbi->number, sec);
         if (rc == NO_ERROR)
         {
            cbi->thisSn = cbi->sn;              // current sn for callback
            if (cbi->verbose)
            {
               TxPrint( "Preparing work on %u sectors starting at 0x0%llx "
                        "on disk %u\n", cbi->number, cbi->sn, cbi->disknr);
            }
            if (((callback)( sec, FDSK_PREPARE, cbi)) == NO_ERROR)
            {
               sprintf( text, "Process sectors");
               rc = (callback)( sec, FDSK_PERFORM, cbi);
               if (rc == DFS_PENDING)          // conditional write, call again
               {
                  if (cbi->nowrite)
                  {
                     rc = NO_ERROR;
                  }
                  else                          // writing back desired
                  {
                     if ((cbi->confirmed) || DFSTORE_WRITE_ALLOWED)
                     {
                        rc = dfstWritePsn( DFSTORE, cbi->sn, cbi->number, sec);
                     }
                     else
                     {
                        rc = DFS_NO_CHANGE;
                     }
                  }
                  if (rc == NO_ERROR)           // do the post-processing
                  {
                     rc = (callback)( sec, FDSK_PENDING, cbi);
                  }
               }
            }
            else
            {
               sprintf( text, "Skipped sectors");
               rc = DFS_NO_CHANGE;
            }
            switch (rc)
            {
               case NO_ERROR:
                  strcat( text," finished");
                  break;

               case DFS_NO_CHANGE:
                  strcat( text,", no changes made");
                  break;

               default:
                  strcat( text," failed");
                  break;
            }
            if (cbi->verbose)
            {
               TxPrint("%s\n\n", text);
            }
         }
         else if (rc == DFS_PSN_LIMIT)          // past disk ? (DFSDOS)
         {                                      // or read-error
            rc = NO_ERROR;                      // allow iterator continue
            if (cbi->verbose)
            {
               TxPrint("Process sector skipped on read-error at 0x0%llx\n\n", cbi->sn);
            }
         }
         TxFreeMem( sec);                       // free the memory
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   RETURN (rc);
}                                               // end 'dfsExecOnSectors'
/*---------------------------------------------------------------------------*/


#if defined (USEWINDOWING)
/*****************************************************************************/
// Build selection-list with partition info for all partitions present
/*****************************************************************************/
TXSELIST *dfsPartSelist                         // RET   selection list or NULL
(
   BOOL                incall,                  // IN    include 'all' choice
   USHORT              disknr,                  // IN    disknr or FDSK_ANY
   char                select                   // IN    P=prim, L=log else ALL
)
{
   TXSELIST           *list  = NULL;            // total list
   TXS_ITEM           *item;                    // single item
   DFSPARTINFO        *p;                       // partition info
   ULONG               lsize;                   // list-size
   USHORT              disk  = 0;               // current disk
   USHORT              pid   = 0;               // partition ID
   int                 i;
   USHORT              parts = dfsPartitions();

   ENTER();
   TRACES(("ALL:%s disk:%hu select:'%c'  #part:%hu\n", (incall) ? "YES" : "NO", disknr, select, parts));

   lsize = parts;
   if (disknr == FDSK_ANY)                      // count #disks with any parts
   {
      if (incall)
      {
         lsize += 2;                            // make room for 'ALL' items
      }
      if (parts != 0)                           // only when partitioins there!
      {
         for (disk = 0, i = parts; i; i--)      // all partitions, backwards
         {
            if ((p = dfsGetPartInfo( i)) != NULL)
            {
               if (p->disknr != disk)           // add separator for each
               {                                // disk that has partitions
                  lsize++;
                  disk = p->disknr;             // disk will end at disknr
               }                                // for the first partition
            }
         }
         lsize--;                               // minus one :-)
         pid = 1;                               // start numbering PART items with 1
      }
   }
   else                                         // specific disk, count partitions
   {
      for (lsize = 0, i = 1; i <= parts; i++)
      {
         if ((p = dfsGetPartInfo( i)) != NULL)
         {
            if (p->disknr == disknr)            // part is ON desired disk
            {
               lsize++;
               if (pid == 0)                    // not set yet
               {
                  pid = i;                      // pid of first number on selected disk
               }
            }
         }
      }
   }

   if (lsize != 0)                              // any partitions to list ?
   {
      if (TxSelCreate( lsize, lsize, lsize, TXS_AS_NOSTATIC, FALSE, NULL, &list) == NO_ERROR)
      {
         char           *listdescr;             // list level description

         list->astatus = TXS_AS_NOSTATIC      | // all dynamic allocated
                         TXS_LST_DESC_PRESENT | // with list description
                         TXS_LST_DYN_CONTENTS;

         if ((listdescr  = TxAlloc( 1, TXMAXTM)) != NULL)
         {
            sprintf( listdescr, "partition to be used %s", (incall) ?
                                ", including 'ALL partitions'" : "");

            list->miscInfo = listdescr;
         }
         for (i = 0; i < lsize; i++)            // all list entries
         {
            TRACES(("pid %hu, list entry: %d of %u\n", pid, i, lsize));
            if ((item  = TxAlloc( 1, sizeof(TXS_ITEM))) != NULL)
            {
               list->count    = i +1;           // actual items in list
               list->items[i] = item;           // attach item to list
               item->userdata = (void *) list->count; // assign item number for unsort

               item->helpid = TXWH_USE_OWNER_HELP; // from owner-menu-item

               if (incall && (i < 2))           // first two are ALL-disks
               {
                  if (i == 0)                   // first is entry
                  {
                     if (((item->text = TxAlloc( 1, TXMAXTM)) != NULL) &&
                         ((item->desc = TxAlloc( 1, TXMAXTM)) != NULL)  )
                     {
                        item->value   = TXDID_MAX; // part-id '0'

                        sprintf( item->text, "             |- - - - All partitions - - - -|");
                        sprintf( item->desc, "Perform operation on all partitions listed");
                        item->index  = 17;      // All ... quick-select

                        TRACES(("text: %d '%s'\n", strlen(item->text), item->text));
                        TRACES(("desc: %d '%s'\n", strlen(item->desc), item->desc));
                     }
                  }
                  else                          // second is a separator
                  {
                     item->flags = TXSF_DISABLED | TXSF_SEPARATOR;
                  }
                  continue;                     // with next list item
               }

               if ((p = dfsGetPartInfo( pid)) != NULL) // get info for this pid
               {
                  if ((disknr == FDSK_ANY) && (p->disknr != disk)) // disk separater needed
                  {
                     item->flags = TXSF_DISABLED | TXSF_SEPARATOR;
                     disk = p->disknr;
                     continue;                  // with next list item
                  }
                  if (((item->text = TxAlloc( 1, TXMAXTM)) != NULL) &&
                      ((item->desc = TxAlloc( 1, TXMAXTM)) != NULL)  )
                  {
                     TXTM label;                // plabel, or part of lvm/gpt name
                     TXTM nameinfo;
                     TXTS typeinfo;

                     item->value   = TXDID_MAX + pid;

                     strcpy(     label, p->plabel);
                     TxStrip(    label,     label, ' ', ' '); // remove spaces, if any
                     if (strlen( label) == 0)
                     {
                        TxCopy(  label, p->lvm.VoluName, BT_LABL_L + 1);
                     }
                     if (p->tablenr == GPT_STYLE)
                     {
                        sprintf( nameinfo, "%s",    p->lvm.VoluName);
                        sprintf( typeinfo, "Gpt %2.2hhx00", p->partent.PartitionType);
                     }
                     else
                     {
                        if (p->lvmPresent)      // use volume, partition name
                        {
                           sprintf( nameinfo, "%s on %s",   p->lvm.VoluName, p->lvm.PartName);
                        }
                        else
                        {
                           sprintf( nameinfo, " %-11.11s --No-LVM-info--", label);
                        }
                        sprintf( typeinfo, "%s 0x%2.2hhx", (p->primary) ? "Pri" : "Log",
                                                            p->partent.PartitionType);
                     }
                     sprintf( item->text, "%2.2hu %-12.12s on disk%2hu, %2.2s %-11.11s %-6.6s %9.1lf MiB",
                              pid, p->UnixDevicePart, disk, p->drive, label, p->fsform,
                              TXSMIB( p->sectors, p->bpsector));

                     sprintf( item->desc,
                             "%-12.12s on %s disk%2hu %s %-17.17s %-8.8s %-26.26s",
                              p->UnixDevicePart, dfsMediaTypeDescr( dfsDid2DiskType( disk)),
                              disk, typeinfo, p->descr, p->creatr, nameinfo);

                     //- get unique SelectChar starting at text[1] ...
                     item->index = TxSelGetCharSelect( list, i, 1);

                     //- test JvW, logical only
                     switch (select)
                     {
                        case 'l':
                        case 'L':
                           if (p->primary == TRUE)
                           {
                              item->flags = TXSF_DISABLED;
                              TxSelSetItemInfo( item, dfsNoPrimaries);
                           }
                           break;

                        case 'p':
                        case 'P':
                           if (p->primary == FALSE)
                           {
                              item->flags = TXSF_DISABLED;
                              TxSelSetItemInfo( item, dfsNoLogicals);
                           }
                           break;

                        default:
                           break;
                     }
                     TRACES(("text: %d '%s'\n", strlen(item->text), item->text));
                     TRACES(("desc: %d '%s'\n", strlen(item->desc), item->desc));
                  }
               }
               pid++;                           // to next partition-id
            }
         }
      }
   }
   else
   {
      list = TxSelEmptyList( "- No partitions present on any disk -",
            "No partitions found, all disks empty; MBR tables cleared or disks inaccessible", FALSE);

   }
   if (list != NULL)                            // attach sort information
   {
      list->sortinfo = &dfs_part_sort;
   }
   RETURN( list);
}                                               // end 'dfsPartSelist'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Build selection-list with disk info for all physical and virtual disks
/*****************************************************************************/
TXSELIST *dfsDiskSelist                         // RET   selection list or NULL
(
   BOOL                incall,                  // IN    include 'all' choice
   BOOL                excDummy                 // IN    exclude 'dummy disks'
)
{
   TXSELIST           *list  = NULL;            // total list
   TXS_ITEM           *item;                    // single item
   DFSDISKINFO        *d;                       // disk info
   ULONG               lsize;                   // list-size
   USHORT              disks;                   // #disks incl virtuals
   USHORT              total;                   // disks, incl non-accessible
   USHORT              disk = 0;                // current disk number
   int                 i;
   char               *s;

   ENTER();

   total = dfsGetDiskCount( FALSE);             // any disks present ?

   for (disks = total, i = 0; (i < total) && excDummy; i++)
   {
      if (!dfsDiskAccessible( i+1))
      {
         disks--;                               // exclude non-accessible disks
      }
   }
   TRACES(( "Accessible disks: %hu, from total: %hu\n", disks, total));
   if (disks)                                   // accessible disks present ?
   {
      lsize   = disks;
      if (incall)
      {
         lsize += 2;                            // make room for ALL items
      }
      if (TxSelCreate( lsize, lsize, lsize, TXS_AS_NOSTATIC, FALSE, NULL, &list) == NO_ERROR)
      {
         char           *listdescr;             // list level description

         list->astatus = TXS_AS_NOSTATIC      | // all dynamic allocated
                         TXS_LST_DESC_PRESENT | // with list description
                         TXS_LST_DYN_CONTENTS;

         if ((listdescr  = TxAlloc( 1, TXMAXTM)) != NULL)
         {
            sprintf( listdescr, "disk to be used %s", (incall) ? ", including 'ALL disks'" : "");
            list->miscInfo = listdescr;
         }
         for (i = 0; i < lsize; i++)            // all list entries
         {
            if ((item  = TxAlloc( 1, sizeof(TXS_ITEM))) != NULL)
            {
               list->count    = i +1;           // actual items in list
               list->items[i] = item;           // attach item to list

               item->helpid = TXWH_USE_OWNER_HELP; // from owner-menu-item

               if (incall && (i < 2))           // first two are ALL-disks
               {
                  if (i == 0)                   // first is entry
                  {
                     if (((item->text = TxAlloc( 1, TXMAXTT)) != NULL) &&
                         ((item->desc = TxAlloc( 1, TXMAXTM)) != NULL)  )
                     {
                        item->value   = TXDID_MAX; // disk nr '0'

                        sprintf( item->text, "   |- - - - - All disks - - - - -|");
                        sprintf( item->desc, "Perform operation on all accessible disks, as listed");
                        item->index  = 15;      // All ... quick-select

                        TRACES(("text: %d '%s'\n", strlen(item->text), item->text));
                        TRACES(("desc: %d '%s'\n", strlen(item->desc), item->desc));
                     }
                  }
                  else                          // second is a separator
                  {
                     item->flags = TXSF_DISABLED | TXSF_SEPARATOR;
                  }
                  continue;                     // with next list item
               }

               if ((d = dfsGetDiskInfo( ++disk)) != NULL)
               {
                  if (excDummy)
                  {
                     while (d && d->OpenError)  // skip inaccessible disks
                     {
                        d = dfsGetDiskInfo( ++disk);
                     }
                  }
                  if (d != NULL)                // better safe then sorry :-)
                  {
                     if (((item->text = TxAlloc( 1, TXMAXTM)) != NULL) &&
                         ((item->desc = TxAlloc( 1, TXMAXTM)) != NULL)  )
                     {
                        item->value   = TXDID_MAX + disk;

                        sprintf( item->text, "%2hu %-20.20s %9.1lf MiB",
                                              disk, d->DiskName, TXSMIB( d->sectors, d->bpsector));

                        #if defined (GET_FROM_DISK_EACH_TIME_IS_TOO_SLOW_ON_OS2)
                           //- Length identification matches the standard length from Windows DiskDeviceString (82)
                           sprintf( item->desc, "%-13s           ", d->UnixDeviceDisk); // leading device-name
                           if (dfsGetDiskIdentification( d->disknr, d->UnixDeviceDisk, 82, item->desc + 14) == FALSE)
                        #else
                           //- get from identification string generated by AddMediaMapping
                           if (((s = dfsDid2DiskDescription( disk)) != NULL) && strlen( s))
                           {
                              sprintf( item->desc, "%-13s %-82.82s", d->UnixDeviceDisk, s);
                           }
                           else
                        #endif

                        //- Else generate a generic description from other disk info
                        {
                           sprintf( item->desc, "%-13s %s-disk %hu  %2hu partitions, %hu primary; CHS:%7u %3u %-3u",
                                    d->UnixDeviceDisk, dfsMediaTypeDescr( dfsDid2DiskType(disk)),
                                    disk, d->totalpart, d->primaries, d->geoCyls, d->geoHeads, d->geoSecs);
                        }

                        //- get unique SelectChar starting at text[1] ...
                        item->index = TxSelGetCharSelect( list, i, 1);

                        TRACES(("text: %d '%s'\n", strlen(item->text), item->text));
                        TRACES(("desc: %d '%s'\n", strlen(item->desc), item->desc));
                     }
                  }
               }
            }
         }
      }
   }
   else
   {
      list = TxSelEmptyList( "- No disks present/accessible to DFSee -", dfsNoAccesMsg, FALSE);
   }
   RETURN( list);
}                                               // end 'dfsDiskSelist'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Build selection-list with freespace info for all physical and virtual disks
/*****************************************************************************/
TXSELIST *dfsFreeSelist                         // RET   selection list or NULL
(
   void
)
{
   TXSELIST           *list  = NULL;            // total list
   TXS_ITEM           *item;                    // single item
   DFSDISKINFO        *d;                       // disk info
   DFSPARTINFO        *f = NULL;                // freespace info
   ULONG               lsize;                   // list-size
   USHORT              disks;                   // #disks incl virtuals
   USHORT              frees = 0;               // #freespace areas
   USHORT              disk  = 1;               // current disk number
   ULN64               sectors;
   TXTS                sizeStr;                 // size indicator string
   int                 i;

   ENTER();

   if ((disks = dfsGetDiskCount( FALSE)) != 0)  // any disks present ?
   {
      for (lsize = 0, disk = 1; disk <= disks; disk++)
      {
         if (((d = dfsGetDiskInfo( disk)) != NULL) && (!d->OpenError))
         {
            frees += d->freeSpace;              // actual freespaces
            lsize++;                            // separator line
         }
      }
      lsize --;                                 // no separator on last
      lsize += frees;                           // total list size

      TRACES(( "frees:%u disks:%hu ==> lsize:%u\n", frees, disks, lsize));

      if (TxSelCreate( lsize, lsize, lsize, TXS_AS_NOSTATIC, FALSE, NULL, &list) == NO_ERROR)
      {
         char           *listdescr;             // list level description

         list->astatus = TXS_AS_NOSTATIC      | // all dynamic allocated
                         TXS_LST_DESC_PRESENT | // with list description
                         TXS_LST_DYN_CONTENTS;

         if ((listdescr  = TxAlloc( 1, TXMAXTM)) != NULL)
         {
            sprintf( listdescr, "freespace area to be used");
            list->miscInfo = listdescr;
         }
         i = 0;                                 // start with 1st Fsp-area
         for (disk = 1; disk <= disks; disk++)
         {
            if (((d = dfsGetDiskInfo( disk)) != NULL) && (!d->OpenError))
            {
               for (f = d->fspHead; f != NULL; f = f->fspChain)
               {
                  if ((item = TxAlloc( 1, sizeof(TXS_ITEM))) != NULL)
                  {
                     list->items[i] = item;                //- attach item to list
                     list->count    = ++i;                 //- actual items now in list
                     item->helpid   = TXWH_USE_OWNER_HELP; //- owner-menu-item

                     if (((item->text = TxAlloc( 1, TXMAXTM)) != NULL) &&
                         ((item->desc = TxAlloc( 1, TXMAXTM)) != NULL)  )
                     {
                        item->value   = TXDID_MAX  + f->id;
                        sectors       = f->lastPsn - f->basePsn +1; // fsp size

                        if (sectors > 1000)     // at least 0.5 MiB
                        {
                           sprintf( sizeStr, "%9.1lf MiB", TXSMIB( sectors, f->bpsector));
                        }
                        else
                        {
                           sprintf( sizeStr, "0x%3.3llx sectors", sectors);
                        }
                        sprintf( item->text, "Disk%2hu, Area:%2.2hu %s %s", f->disknr, f->id, f->descr, sizeStr);

                        sprintf( item->desc, "Disk %2hu  Cyl:%9u -%9u ",  f->disknr,
                                                                   (ULONG) (f->basePsn / f->cSC), (ULONG) (f->lastPsn / f->cSC));
                        switch (f->partent.PartitionType)
                        {
                           case DFS_FSP_UNPS : strcat( item->desc, "Unpartionable space! Disk not to be touched!   "); break;
                           case DFS_FSP_NONE : strcat( item->desc, "Outside Log-container and #primaries is maximal"); break;
                           case DFS_FSP_PRIM : strcat( item->desc, "Can be used for a new PRIMARY partition only   "); break;
                           case DFS_FSP_HLOG :
                           case DFS_FSP_TLOG :
                           case DFS_FSP_LOG  : strcat( item->desc, "Can be used for a new LOGICAL partition only   "); break;
                           case DFS_FSP_ELOG : strcat( item->desc, "For a LOGICAL only, for PRIMARY need a CLEANUP!"); break;
                           case DFS_FSP_NEWX :
                           case DFS_FSP_HEAD :
                           case DFS_FSP_TAIL : strcat( item->desc, "Can be used for a primary or logical partition "); break;
                           case DFS_FSP_ENDC : strcat( item->desc, "Too small too align, unaligned small part only "); break;
                           case DFS_FSP_ZERO : strcat( item->desc, "MBR, first-track - too small for real partition"); break;

                           case DFS_FSP_GPTM : strcat( item->desc, "GPT MBR, HDR and Partition Table Array at start"); break;
                           case DFS_FSP_GPTS : strcat( item->desc, "GPT, Freespace plus MBR, HDR and Array at start"); break;
                           case DFS_FSP_GPTE : strcat( item->desc, "GPT, HDR and Partition Table Array at end disk "); break;
                           case DFS_FSP_GPTC : strcat( item->desc, "GPT, Freespace plus HDR and PTA at end of disk "); break;
                           case DFS_FSP_GPTF : strcat( item->desc, "GPT, Freespace area available for new partition"); break;

                           default           : strcat( item->desc, "Unknown freespace type, this is a program bug! "); break;
                        }

                        //- get unique SelectChar starting at 1st Fid digit
                        item->index = TxSelGetCharSelect( list, i-1, 13);

                        TRACES(("text: %d '%s'\n", strlen(item->text), item->text));
                        TRACES(("desc: %d '%s'\n", strlen(item->desc), item->desc));
                     }
                  }
               }
               if (list->count < lsize)         // separator except on last ...
               {
                  if ((item  = TxAlloc( 1, sizeof(TXS_ITEM))) != NULL)
                  {
                     list->items[i] = item;     // attach item to list
                     list->count    = ++i;      // actual items now in list
                     item->flags    = TXSF_DISABLED | TXSF_SEPARATOR;
                  }
               }
            }
         }
         for (i = 0; i < lsize; i++)
         {
            if ((list->items[i]->flags & TXSF_DISABLED) == 0)
            {
               list->selected = i;              // first selectable
               break;
            }
         }
      }
   }
   else
   {
      list = TxSelEmptyList( "- No disks or freespace areas present -",
            "No disks accessible to DFSee or no freespace areas on any of the disks", FALSE);

   }
   RETURN( list);
}                                               // end 'dfsFreeSelist'
/*---------------------------------------------------------------------------*/


/*************************************************************************************************/
// Enable only matching freespace-areas in the Selist, return enabled count
/*************************************************************************************************/
ULONG dfsFreeSlEnable                           // RET   nr of matching freespaces
(
   TXSELIST           *slFree,                  // INOUT List of freespaces
   ULN64               minSize,                 // IN    required size or L64_NULL
   BOOL                create,                  // IN    suitable for CREATE
   USHORT              disk,                    // IN    WASTED on this disk OK
   char                fspKind                  // IN    Kind wanted: G, P, L or *
)
{
   ULONG               rc = 0;                  // function return
   TXS_ITEM           *item;                    // single item
   DFSPARTINFO        *f = NULL;                // freespace info
   int                 i;

   ENTER();

   TRACES(( "list:%8.8x  minsize:%llx  CR:%s  disk:%hu  fspKind: '%c'\n",
             slFree, minSize, (create) ? "YES" : "NO", disk, fspKind));
   if (slFree != NULL)
   {
      for (i = 0; i < slFree->count; i++)
      {
         item = slFree->items[i];
         if ((item->flags & TXSF_SEPARATOR) == 0)
         {
            item->flags |= TXSF_DISABLED;       // init to disabled state
            if ((f = dfsGetFreeInfo( item->value - TXDID_MAX)) != NULL)
            {
               ULN64   mSize = (minSize == L64_NULL) ? (f->geoSecs) : minSize;

               if ((f->lastPsn - f->basePsn +1) >= mSize)
               {
                  if (create)                   // disabled ZERO/NONE types
                  {
                     switch (f->partent.PartitionType) // supply the disable reason for user feedback
                     {
                        case DFS_FSP_ZERO:      // track-zero fsp, never OK
                           TxSelSetItemInfo( item, "This is a (small) area in TRACK-ZERO reserved for the MBR sector "
                                                   "and possibly other housekeeping stuff. Not usable for a partition!");
                           break;

                        case DFS_FSP_GPTM:      // GPT structures at start
                        case DFS_FSP_GPTE:      // GPT structures at end
                           TxSelSetItemInfo( item, "This is a (small) area in at the start/end of the disk reserved "
                                                   "for the GPT header and tables. Not usable for a partition!");
                           break;

                        case DFS_FSP_NONE:      // wasted fsp, not OK on CR
                           TxSelSetItemInfo( item, "This is a freespace area suitable for PRIMARY partitions only "
                                                   "but the maximum number of primaries has been reached already!");
                           break;

                        case DFS_FSP_UNPS:      // Unpartionable,  never OK
                           TxSelSetItemInfo( item, "This disk is NOT an MBR-style or GPT-style partitioned disk (yet) "
                                                   "It may be an encrypted disk or filesystem with your valuable data "
                                                   "or simply have garbage or a factory test-pattern is still there.\n\n"
                                                   "For safety, it can NOT be partitioned by DFSee in this condition!\n\n"
                                                   "If you are sure the disk contents can be destroyed, you can use NEWMBR "
                                                   "or wipe the start of the disk to make it partitionable as MBR or GPT.");
                           break;

                        default:
                           item->flags &= ~TXSF_DISABLED; // reset to enabled
                           if (++rc == 1)
                           {
                              slFree->selected = i; // make it the default
                           }
                           break;
                     }
                  }
                  else                          // must match primary property
                  {                             // or WASTED on same disk & PRI
                     if (dfsFreeSpaceTypeMatch( f->partent.PartitionType,
                                              ((f->disknr == disk) || (disk == FDSK_ANY)),
                                              ((fspKind   == 'P' ) || (fspKind == '*'))))
                     {
                        item->flags &= ~TXSF_DISABLED; // reset to enabled
                        if (++rc == 1)
                        {
                           slFree->selected = i; // make it the default
                        }
                     }
                  }
               }
            }
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsFreeSlEnable'
/*-----------------------------------------------------------------------------------------------*/


/*****************************************************************************/
// Build selection-list with info for BMGR-bootable partitions on PHYS disks
/*****************************************************************************/
TXSELIST *dfsBmgrSelist                         // RET   selection list or NULL
(
   void
)
{
   TXSELIST           *list  = NULL;            // total list
   TXS_ITEM           *item;                    // single item
   DFSPARTINFO        *p;                       // partition info
   DFSDISKINFO        *d;                       // disk info
   ULONG               lsize;                   // list-size
   USHORT              parts;                   // #partitions
   USHORT              boots = 0;               // #bootable on BMGR
   USHORT              pid   = 1;               // partition ID
   int                 i;
   USHORT              bbp[DFS_MAX_PART];       // bootable pids

   ENTER();

   if ((parts = dfsPartitions()) != 0)          // any partitions present ?
   {
      for (pid = 1; pid <= parts; pid++)
      {
         p = dfsGetPartInfo( pid);              // next partition valid ?
         d = dfsGetDiskInfo( p->disknr);

         if ((strlen(p->blabel) > 0) &&         // bootable by bootmgr
             (d->Removable == FALSE)            // not a removable one
              #if !defined( DUMP)
                && (dfsDid2DiskType(p->disknr) == DFSD_PHYS) // and real disk
              #endif
            )
         {
            bbp[boots++] = p->id;               // remember id and count
         }
      }
   }

   TRACES(( "partitions: %hu  BMGR-bootables: %hu\n", parts, boots));

   if (boots != 0)                              // some bootables present
   {
      lsize = boots +2;                         // 2 lines for last/default
      if (TxSelCreate( lsize, lsize, lsize, TXS_AS_NOSTATIC, FALSE, NULL, &list) == NO_ERROR)
      {
         char           *listdescr;             // list level description

         list->astatus = TXS_AS_NOSTATIC      | // all dynamic allocated
                         TXS_LST_DESC_PRESENT | // with list description
                         TXS_LST_DYN_CONTENTS;

         if ((listdescr  = TxAlloc( 1, TXMAXTM)) != NULL)
         {
            sprintf( listdescr, "bootable partition from the IBM BMGR menu");

            list->miscInfo = listdescr;
         }
         list->selected = 2;                    // First real item is default

         for (i = 0; i < lsize; i++)            // all list entries
         {
            TRACES(("format list entry: %d for total of %u\n", i, lsize));
            if ((item  = TxAlloc( 1, sizeof(TXS_ITEM))) != NULL)
            {
               list->count    = i +1;           // actual items in list
               list->items[i] = item;           // attach item to list

               item->helpid = TXWH_USE_OWNER_HELP; // from owner-menu-item

               if (i < 2)                       // first two are special
               {
                  if (i == 0)                   // first is last/default entry
                  {
                     if (((item->text = TxAlloc( 1, TXMAXTT)) != NULL) &&
                         ((item->desc = TxAlloc( 1, TXMAXTM)) != NULL)  )
                     {
                        item->value   = TXDID_MAX; // partition-id 0

                        sprintf( item->text, "      %s", "Last-booted or BM-default");
                        sprintf( item->desc, "%-80.80s",
                                "Use the last booted partition, or the one set as BootManager default");
                        item->index  = 4;       // Last ... quick-select

                        TRACES(("text: %d '%s'\n", strlen(item->text), item->text));
                        TRACES(("desc: %d '%s'\n", strlen(item->desc), item->desc));
                     }
                  }
                  else                          // second is a separator
                  {
                     item->flags = TXSF_DISABLED | TXSF_SEPARATOR;
                  }
                  continue;                     // with next list item
               }

               pid = bbp[i-2];
               p = dfsGetPartInfo( pid);        // next partition valid ?
               if (p != NULL)
               {
                  if (((item->text = TxAlloc( 1, TXMAXTM)) != NULL) &&
                      ((item->desc = TxAlloc( 1, TXMAXTM)) != NULL)  )
                  {
                     TXTM nameinfo;
                     TXTS typeinfo;

                     item->value   = TXDID_MAX + pid;

                     if (p->tablenr == GPT_STYLE)
                     {
                        sprintf( nameinfo, "Named '%s'",    p->lvm.VoluName);
                        sprintf( typeinfo, "Gpt %2.2hhx00", p->partent.PartitionType);
                     }
                     else
                     {
                        if (p->lvmPresent)      // use volume, partition name
                        {
                           sprintf( nameinfo, "%s on %s",   p->lvm.VoluName, p->lvm.PartName);
                        }
                        else
                        {
                           sprintf( nameinfo, " %-11.11s --No-LVM-info--", p->plabel);
                        }
                        sprintf( typeinfo, "%s 0x%2.2hhx", (p->primary) ? "Pri" : "Log",
                                                            p->partent.PartitionType);
                     }
                     sprintf( item->text, "%2.2hu %-12.12s on disk%2hu, %2.2s %-11.11s %-6.6s%9.1lf MiB",
                              pid, p->UnixDevicePart, p->disknr, p->drive, p->plabel, p->fsform,
                              TXSMIB( p->sectors, p->bpsector));

                     sprintf( item->desc,
                             "%-12.12s on %s disk%2hu %s %-17.17s %-8.8s %-26.26s",
                              p->UnixDevicePart, dfsMediaTypeDescr( dfsDid2DiskType( p->disknr)),
                              p->disknr, typeinfo, p->descr, p->creatr, nameinfo);

                     //- get unique SelectChar starting at text[1] ...
                     item->index = TxSelGetCharSelect( list, i, 1);

                     TRACES(("text: %d '%s'\n", strlen(item->text), item->text));
                     TRACES(("desc: %d '%s'\n", strlen(item->desc), item->desc));
                  }
               }
            }
         }
      }
   }
   RETURN( list);
}                                               // end 'dfsBmgrSelist'
/*---------------------------------------------------------------------------*/
#endif                                          // USEWINDOWING

