//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// Partition utility, display and iterator functions; misc section
//
// Author: J. van Wijk
//
// JvW  18-08-2005 Initial version, split from DFSUPART
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsver.h>                             // DFS version info
#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsupart.h>                           // partition utility functions
#include <dfsufdsk.h>                           // Fdisk & translation services
#include <dfsutil.h>                            // DFS utility functions
#include <dfsafdsk.h>                           // FDISK generic definitions
#include <dfsufdsk.h>                           // FDISK utility functions

#if defined (WIN32)
   #include <dfsntreg.h>                        // DFSee NT registry functions
#endif

#if defined (DEV32)

//- Below definitions from Lard Erdman, through email 20181209

#define DSK_WRITEPASSTHRU 0x4F
#define DSK_READPASSTHRU  0x6F

#pragma pack(1)
#define SCSI_INQUIRY            0x12    /* Inquiry command                  */

typedef struct _PassthruATA
{
  UCHAR        byPhysicalUnit;             /* physical unit number 0-n */
                                          /* 0 = Pri/Mas, 1=Pri/Sla, 2=Sec/Mas, etc. */
  struct
  {
    UCHAR        Features;                 /* features register */
    UCHAR        SectorCount;              /* sector count register */
    UCHAR        SectorNumber;             /* sector number register */
    UCHAR        CylinderLow;              /* cylinder low register */
    UCHAR        CylinderHigh;             /* cylinder high register */
    UCHAR        DriveHead;                /* drive/head register */
    UCHAR        Command;                  /* command register */
  } TaskFileIn;
  struct
  {
    UCHAR        Error;                    /* error register */
    UCHAR        SectorCount;              /* sector count register */
    UCHAR        SectorNumber;             /* sector number register */
    UCHAR        CylinderLow;              /* cylinder low register */
    UCHAR        CylinderHigh;             /* cylinder high register */
    UCHAR        Reserved;                 /* unused */
    UCHAR        Status;                   /* status register */
  } TaskFileOut;
  USHORT        RegisterTransferMap;      /* bits to indicate what */
                                          /* registers are to be */
                                          /* transferred to/from H/W */
} PassThruATA, NEAR* NPPassThruATA, FAR* PPassThruATA;

typedef struct DDI_CtrlString
{
            UCHAR       Command;
            UCHAR       LogicalDrive;
            ULONG       datasize;
            ULONG       DeviceTimeout_Sec;              /* device timeout in seconds */
                                                        /* 0 = default */
                                                        /* -1 - no timeout; */
            ULONG       scsi_command_length;
            BYTE        scsi_command[sizeof(PassThruATA)];
} CTRLSTRING, NEAR * NPCTRLSTRING, FAR * PCTRLSTRING;

typedef struct _SCSICDB6
{                                               /* CDB6 - 6 byte Command Descriptor Block        */

        UCHAR   Opcode;             /* CDB Operation Code           */
        UCHAR   Lun_MsbLBA;         /* SCSI LUN & 5 MSB bits of LBA */
        UCHAR   MidLBA;             /* SCSI MID byte of LBA         */
        UCHAR   LsbLBA;             /* SCSI LBA byte of LBA         */
        UCHAR   XferLen;            /* SCSI Xfer length, Alloc length */
        UCHAR   Control;            /* Control byte                 */

} SCSICDB6,FAR *PSCSICDB6, *NPSCSICDB6;


#define SCSI_INQLENGTH                  28
typedef struct  _SCSIInqData
{                                               /* INQ                                           */

        UCHAR    DevType;               /* Periph Qualifier & Periph Dev Type*/
        UCHAR    RMB_TypeMod;           /* rem media bit & Dev Type Modifier */
        UCHAR    Vers;                  /* ISO, ECMA, & ANSI versions        */
        UCHAR    RDF;                   /* AEN, TRMIOP, & response data format*/
        UCHAR    AddLen;                /* length of additional data         */
        UCHAR    Res1;                  /* reserved                          */
        UCHAR    Res2;                  /* reserved                          */
        UCHAR    Flags;                 /* RelADr,Wbus32,Wbus16,Sync,etc.    */
        UCHAR    VendorID[8];           /* Vendor Identification             */
        UCHAR    ProductID[16];         /* Product Identification            */
        UCHAR    ProductRev[4];         /* Product Revision                  */


} SCSI_INQDATA, FAR *PSCSI_INQDATA, *NPSCSI_INQDATA;
#pragma pack()


// Get SCSI style Vendor/Product/Revision didk ID info for driveletter
static BOOL dfsOs2Drive2DiskId                  // RET   Got valid info
(
   char               *drive,                   // IN    driveletter
   TXTM                info                     // OUT   Disk ID info string
);

#endif


#if defined (LINUX)
// Try to retrieve a descriptive name for specified disk/device
static BOOL dfsLinuxDiskModel                   // RET   some name was resolved
(
   USHORT              disknr,                  // IN    Disk number (non-UNIX)
   char               *drive,                   // IN    Disk devicename
   int                 length,                  // IN    size of name buffer
   char               *name                     // OUT   Disk Model Name
);
#endif

/*****************************************************************************/
// Get DISK identification string, Make/Model/serial etc, in specified buffer
/*****************************************************************************/
BOOL dfsGetDiskIdentification                   // RET   Disk ID info present
(
   USHORT              disknr,                  // IN    DFSee disk number
   char               *unixDev,                 // IN    Unix device name, or ""
   int                 length,                  // IN    Fixed output length or 0 (all)
   TXTM                idString                 // OUT   Disk ID string, or error text
)
{
   BOOL                rc = FALSE;
   char               *idInfo = NULL;           // retrieved information string

   ENTER();
   TRACES(("disknr: %hu  unixDev: '%s'\n", disknr, unixDev));

   #if   defined (WIN32)
      idInfo = dfsNtGetDiskDeviceId( disknr);   // from Registry ...

   #elif defined (DOS32)

   #elif defined (LINUX)
   {
      TXTM     cmd;
      sprintf( cmd, "hdparm -i %s 2>&1 | grep 'erial\\|nvalid\\|enied'", unixDev);
      txcExecRedirectIO( cmd, NULL, &idInfo);   // needs SUDO (DFSee std), works for IDE/SATA

      //- check hdparm result, and try an alternative when it failed
      if ((idInfo == NULL) || (strlen( idInfo) < 5) || (strstr( idInfo, "nvalid")) || (strstr( idInfo, "enied")))
      {
         TxFreeMem( idInfo);                    // free possible error output string
         if ((idInfo = TxAlloc( 1, TXMAXTM)) != NULL) // try to get just Model Info instead
         {
            if (dfsLinuxDiskModel( disknr, unixDev, TXMAXTM, idInfo) == FALSE)
            {
               TxFreeMem( idInfo);
            }
         }
      }
      else                                      // we got hdparm lines, terminate after 1st one
      {
         TxRepl(  idInfo, '\n', 0);             // end string at first end of line
         TxStrip( idInfo, idInfo, ' ', ' ');    // strip leading/trailing spaces
      }
   }
   #elif defined (DARWIN)
      //- to be refined, no interface or utility found yet

   #else
   {
      DFSDISKINFO     *d;
      DFSPARTINFO     *p;
      USHORT           pid;
      TXTS             drive;

      //- OS/2, first try to get Disk-ID from SCSI PASSTHRU ioctl (works on USB, probably on SCSI)
      //- Fill in others from the (cached) disk-model strings from OS2AHCI$ / IBMS506$ devices

      if (((d = dfsGetDiskInfo( disknr)) != NULL) && (d->totalpart != 0)) // valid disk, with partitions
      {
         if ((idInfo = TxAlloc( 1, TXMAXTM)) != NULL)
         {
            for (pid = d->firstpart; (pid <= d->lastpart) && (rc == FALSE); pid++)
            {
               if ((p = dfsGetPartInfo( pid)) != NULL)
               {
                  drive[ 1] = ':';              // Fixed  (drive may contain '*' here)
                  drive[ 0] = toupper( p->drive[ 0] );
                  if ((drive[ 0] >= 'C') && (drive[ 0] <= 'Z')) // valid letter
                  {
                     rc = dfsOs2Drive2DiskId( drive, idInfo); // try to get SCSI Vendor/Product/Rev
                     break;                     // only probe ONE valid letter (performance)
                  }
               }
            }
            if (rc == FALSE)                    // got no identification yet
            {
               if (dfsMediaOs2IsPaeRamDisk( disknr, idInfo) == FALSE) // not the QSINIT RAMDISK either
               {
                  if (dfsMediaOs2IdeSataId( disknr, idInfo) == FALSE) // try cached IDE/SATA identification
                  {
                     TxFreeMem( idInfo);
                  }
               }
            }
         }
      }
   }
   #endif

   if (length == 0)                             // no specific length, set to whole text
   {
      length = (idInfo) ? max( strlen( idInfo), TXMAXTM - 1) : 0;
   }
   sprintf( idString, "%-*.*s", length, length, (idInfo) ? idInfo : "");

   if (idInfo != NULL)                          // we have valid info
   {
      TxFreeMem( idInfo);                       // Free allocated ID info
      rc = TRUE;
   }
   TRACES(("idString: '%s'\n", idString));
   BRETURN (rc);
}                                               // end 'dfsGetDiskIdentification'
/*---------------------------------------------------------------------------*/

#if defined (DEV32)
/*****************************************************************************/
// Get SCSI style Vendor/Product/Revision didk ID info for driveletter
/*****************************************************************************/
static BOOL dfsOs2Drive2DiskId                  // RET   Got valid info
(
   char               *drive,                   // IN    driveletter
   TXTM                info                     // OUT   Disk ID info string
)
{
   BOOL                rc = FALSE;              // function return
   HFILE               hDrive = NULLHANDLE;
   ULONG               ulAction = 0UL;
   ULONG               dr;

   ENTER();
   TRACES(("drive: '%s'\n", drive));

   if ((dr = DosOpen( drive, &hDrive, &ulAction, 0, FILE_NORMAL, OPEN_ACTION_OPEN_IF_EXISTS,
                      OPEN_FLAGS_DASD|OPEN_FLAGS_NOINHERIT|OPEN_SHARE_DENYNONE|OPEN_ACCESS_READONLY,
                      NULL)) == NO_ERROR)
   {
      CTRLSTRING       parm;
      CHAR             data[sizeof(SCSI_INQDATA)];
      ULONG            parmlen = sizeof(parm),datalen = sizeof(data);
      NPSCSICDB6       pCmd = (NPSCSICDB6)parm.scsi_command;

      memset(&parm,0,sizeof(parm));
      parm.datasize            = sizeof(data);
      parm.scsi_command_length = sizeof(*pCmd);
      pCmd->Opcode             = SCSI_INQUIRY;
      pCmd->XferLen            = sizeof(data);

      memset(data,0,sizeof(data));

      if ((dr = DosDevIOCtl( hDrive, IOCTL_DISK, DSK_READPASSTHRU,
                             &parm, parmlen, &parmlen,
                             data,  datalen, &datalen)) == NO_ERROR)
      {
         int           i;
         SCSI_INQDATA *id = (SCSI_INQDATA *) data;

         TRDUMP(70, "PASSTHRU data structure 8-8-16-4 bytes:\n", data, 36, 0);

         for ( i = 0; i < SCSI_INQLENGTH; i++)
         {
            if (id->VendorID[ i] == 0)
            {
               id->VendorID[i] = 0x20;          // replace by spaces
            }
         }
         //- Now print the 3 fields in a nice format
         sprintf( info, "%-8.8s  %-16.16s  %-4.4s", id->VendorID, id->ProductID, id->ProductRev);

         TRACES(("info: '%s'\n", info));
         rc = TRUE;
      }
      else
      {
         TRACES(("DosDevIOCtl error: %u\n", dr));
      }
      DosClose(hDrive);
   }
   else
   {
      TRACES(("DosOpen error: %u\n", dr));
   }
   BRETURN( rc)
}                                               // end 'dfsOs2Drive2DiskId'
/*---------------------------------------------------------------------------*/
#endif

#if defined (LINUX)
/*****************************************************************************/
// Try to retrieve a descriptive name for specified disk/device
/*****************************************************************************/
static BOOL dfsLinuxDiskModel                   // RET   some name was resolved
(
   USHORT              disknr,                  // IN    Disk number (non-UNIX)
   char               *drive,                   // IN    Disk devicename
   int                 length,                  // IN    size of name buffer
   char               *name                     // OUT   Disk Model Name
)
{
   BOOL                rc = FALSE;
   FILE               *fh;
   char               *baredev;                 // bare device name
   TXTM                modelfile;               // full path to model file
   TXLN                line;

   ENTER();
   TxFsAutoFailCriticalErrors( TRUE);           // avoid Not-ready pop-ups

   TRACES(("Drive: '%s'\n", drive));

   TxLinuxRootDevice( drive, line);             // Copy and translate root device
   TxStrip( line, line, 0, ' ');                // strip spaces from devicename
   baredev = strrchr( line, '/');

   TRACES(("baredev: '%s'\n", baredev));

   if (baredev != NULL)
   {
      sprintf(  modelfile, "/sys/block%s/device/model", baredev);
      TRACES(( "modelfile file: '%s'\n", modelfile));

      if ((fh = fopen( modelfile, "rb")) != NULL)
      {
         if (fgets( line, TXMAXLN, fh) != NULL)
         {
            TxStrip( line, line, 0, '\n');      // strip trailing end-of-line
            TxStrip( line, line, 0, ' ');       // strip trailing spaces
            TxCopy(  name, line, length);
            if (strlen( name) != 0)
            {
               rc = TRUE;
            }
         }
         fclose( fh);
      }
   }
   TxFsAutoFailCriticalErrors( FALSE);          // enable criterror handler
   BRETURN (rc);
}                                               // end 'dfsLinuxDiskModel'
/*---------------------------------------------------------------------------*/
#endif


/*****************************************************************************/
// Iterate over all or selected Free-space areas, callback (skip dummy disks)
/*****************************************************************************/
ULONG dfsIterateFree                            // all freespace areas
(
   FDSK_FUNCTION       callback,                // IN    operation callback
   FDSK_CB_INFO       *cbi                      // INOUT call data
)
{
   ULONG               rc = NO_ERROR;           // function return
   DFSDISKINFO        *d;                       // ptr to disk info
   DFSPARTINFO        *p;                       // ptr to partition info
   USHORT              disk;                    // current disk index

   ENTER();
   if ((callback != NULL) && (cbi != NULL))
   {
      for ( disk = 1;
           (disk <= dfsPartitionableDisks()) && (rc == NO_ERROR);
            disk++)
      {
         if ((disk == cbi->disknr) || (cbi->disknr  == FDSK_ANY))
         {
            if (((d = dfsGetDiskInfo(disk)) != NULL) && (d->OpenError == FALSE))
            {
               for (p = d->fspHead; (p) && (rc == NO_ERROR); p = p->fspChain)
               {
                  if ((cbi->stype == 0) ||
                      (cbi->stype == p->partent.PartitionType))
                  {
                     rc = (callback)( p, NULL, FDSK_PERFORM, cbi);
                     if ((p->ebrChain  != NULL) && // is there a related ebr
                         (cbi->related)         && // and do we need to call
                         (rc == NO_ERROR))
                     {
                        rc = (callback)( p->ebrChain, NULL, FDSK_PERFORM, cbi);
                     }
                  }
               }
            }
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsIterateFree'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Iterate over all sections in a PsaveD file, calling back on each sector
// cb nowrite    indicates read-only, no write operations to be performed
// cb verbose    indicates extra (confirmation) info to be displayed
// cb string     options for the Prestore operation
// cb more       filepointer PsaveD file, opened for read at 1st info section
// 20180921 JvW  Updated to read sector in ALLOCATED buffer, not generic rbuf
//               since that might be overwritten in verbose restore (display)
/*****************************************************************************/
ULONG dfsIteratePsaved                          // all sections in PsaveD file
(
   FDSK_PS_FUNCTION    callback,                // IN    operation callback
   FDSK_CB_INFO       *cbi                      // INOUT call data
)
{
   ULONG               rc = NO_ERROR;           // function return
   FILE               *df;                      // diskfile ptr
   DFSPS_SEC           psinfo;
   ULONG               nr;
   USHORT              bps = dfsGetSectorSize();
   ULONG               mxs = RBUFBYTES / bps;
   BYTE               *sector = NULL;

   ENTER();
   if ((callback != NULL) && (cbi != NULL))
   {
      if ((sector = TxAlloc( 1, dfsGetSectorSize())) != NULL)
      {
         df = (FILE *) cbi->more;

         do
         {
            if (fread( &psinfo, sizeof(DFSPS_SEC), 1, df) == 1)
            {
               TRACES(("ps.psn: 0x%llx, sects:%lu, type '%c'\n", psinfo.psn, psinfo.sects, psinfo.type));

               if ((nr = psinfo.sects) > mxs)   // fits in rbuf
               {
                  TxPrint( "Section too large, reading %u sectors only\n", mxs);
                  nr = mxs;
               }
               if (fread(  sector, (size_t) bps, (size_t) nr, df) == (size_t) nr)
               {
                  rc = (callback)( sector, &psinfo, cbi);
               }
            }
         } while (!feof(df) && !ferror(df) && !TxAbort());

         TxFreeMem( sector);
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   RETURN (rc);
}                                               // end 'dfsIteratePsaved'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Process for a single partition, MBR/EBR sectors and/or related LVM sectors
// cb flag       indicates that MBR/EBR are the primary records to process
// cb related    indicates that both LVM and MBR/EBR need to be processed
// cb option     for LVM, the signature sectors need to be processed as well
//                        for any type 0x35, or others if cbOption3 set (-C)
/*****************************************************************************/
ULONG dfsProcessPlvm                            // single partition and/or LVM
(
   FDSK_MS_FUNCTION    callback,                // IN    operation callback
   FDSK_CB_INFO       *cbi                      // INOUT call data (.more = p)
)
{
   ULONG               rc = NO_ERROR;           // function return
   DFSPARTINFO        *p;                       // ptr to real partition info

   ENTER();

   if ((callback != NULL) && (cbi        != NULL) &&
       ((p = (DFSPARTINFO *)  cbi->more) != NULL)  )
   {
      TRACES(("ProcessPlvm for part %hu on disk nr: %hu\n", p->id, p->disknr));
      TRACES(("MBR/EBR:%s  related:%s  SigSect:%s  allBBR:%s\n",
             (cbi->flag   )  ? "YES" : "NO",
             (cbi->related)  ? "YES" : "NO",
             (cbi->option )  ? "YES" : "NO",
             (cbi->doAllBBR) ? "YES" : "NO"));

      if (p->geoSecs > 1)
      {
         cbi->disknr = p->disknr;               // disk to read from
         cbi->number = 1;                       // allways one sector

         if (cbi->flag || cbi->related)
         {
            cbi->sn = p->partPsn;               // MBR/EBR sector
            rc = dfsExecOnSectors( callback, cbi);
            if ((rc == DFS_NO_CHANGE) && (cbi->abortNc == FALSE))
            {
               rc = NO_ERROR;                   // one skipped, continue
            }
         }
         if (((cbi->flag == FALSE) || (cbi->related)) && (rc == NO_ERROR))
         {
            cbi->sn = p->partPsn + p->geoSecs -1; // LVM-info sector
            rc = dfsExecOnSectors( callback, cbi);
            if ((rc == DFS_NO_CHANGE) && (cbi->abortNc == FALSE))
            {
               rc = NO_ERROR;                   // one skipped, continue
            }
            if ((rc == NO_ERROR) && cbi->option)
            {
               if ((p->partent.PartitionType == DFS_P_WARP_LVM) ||
                   (cbi->doAllBBR))             // LVM 0x35 type, or doAll
               {
                  cbi->sn = p->lastPsn;         // possible LVM-signature
                  rc = dfsExecOnSectors( callback, cbi);
                  if ((rc == DFS_NO_CHANGE) && (cbi->abortNc == FALSE))
                  {
                     rc = NO_ERROR;             // one skipped, continue
                  }
               }
            }
         }
      }
      else                                      // No room for info
      {
         TxPrint( "\nDisk %hu with %hu sectors/track is NOT compatible "
                  "with LVM!\nThere is no room for the LVM information "
                  "sector(s)\n\n", p->disknr, p->geoSecs);
      }
   }
   RETURN (rc);
}                                               // end 'dfsProcessPlvm'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set driveletter for specified partition in PARTINFO structure
/*****************************************************************************/
ULONG dfsSetDriveLetter
(
   DFSPARTINFO        *p,                       // IN    partition info
   char               *letter                   // IN    new drive letter
)
{
   ULONG               rc = NO_ERROR;           // function return
   BOOL                set = TRUE;              // set permission
   TXTM                volumes;                 // present volumes
   char                dl = (char) toupper(*letter); // uppercase driveletter

   ENTER();

   if (p != NULL)
   {
      if ((dfsa->batch == FALSE) && (dl != '-')) // check for unique letter
      {
         TxFsVolumes( TXFSV_ALL, volumes);      // get present volumes
         if (strchr( volumes, dl) != NULL)
         {
            if (TxConfirm( 5146,
                "Drive letter '%c:' seems to be in use. "
                "(to be checked with the 'vol' command)\n\n"
                "Are you sure you want to set it ? [Y/N] : ", dl))
            {
               TxPrint( "\nMake sure driveletters will be "
                          "unique after you are done!\n");
            }
            else
            {
               set = FALSE;
            }
         }
      }
      if (set)
      {
         if ((p->drive[0] = dl) == '-')
         {
            p->drive[1] = '-';
         }
         else
         {
            p->drive[1] = ':';
         }
         p->drive[2] = '\0';
         TxPrint( "\nDriveletter for partition %2.2u "
                   "on disk %u (%s) set to '%s%s%s'\n",
                     p->id, p->disknr, p->descr, CBG, p->drive, CNN);
      }
      else
      {
         rc = DFS_CMD_FAILED;                   // aborted
      }
   }
   RETURN( rc);
}                                               // end 'dfsSetDriveLetter'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Find FIRST GPT partition with given partition-GUID (to check uniqueness)
/*****************************************************************************/
USHORT dfsGuid2Pid                              // RET   found PID or 0
(
   DFS_GUID            guid                     // IN    partition GUID value
)
{
   USHORT              rc = 0;                  // function return
   USHORT              pid;
   DFSPARTINFO        *p;

   ENTER();

   for (pid = 1; pid <= dfsPartitions(); pid++)
   {
      if ((p = dfsGetPartInfo( pid)) != NULL)
      {
         if (memcmp( guid, p->partGuid, sizeof( DFS_GUID)) == 0)
         {
            rc = pid;
            break;
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsGuid2Pid'
/*---------------------------------------------------------------------------*/


