//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// Partition utility, display and iterator functions; MBR/EBR section
//
// Author: J. van Wijk
//
// JvW  18-08-2005 Initial version, split from DFSUPART
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsver.h>                             // DFS version info
#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsupart.h>                           // partition utility functions
#include <dfsufdsk.h>                           // Fdisk & translation services
#include <dfsutil.h>                            // DFS utility functions
#include <dfsafdsk.h>                           // FDISK generic definitions
#include <dfsufdsk.h>                           // FDISK utility functions

#define FDSK_CRCNLS_SIZE   0x0c0                // small CRC area MBR (NLS)
#define FDSK_CRCMBR_SIZE   0x180                // large CRC area MBR
#define FDSK_NEWMBR_SIZE   0x1bd                // copy size boot-code area
#define FDSK_WNTMBR_SIZE   0x1b8                // size, excl WinNT signature

#define FDSK_NTSIG         DFS_V_EYE            // DFSee NT-like signature

static ULONG dfsFdskMbrTemplate[] =             // From eCS 1.0 (LVM nov 2000)
{
   0x0030b8fa, 0x00bcd08e, 0x33fcfb01, 0x8ed88ec0,  //- 0м3؎
   0x7c00bec0, 0xb97e00bf, 0xa5f30200, 0xc37e2068,  //- |~h ~
   0xbb7efabe, 0x7f807fbe, 0x41740a04, 0x8110c383,  //- ~tA
   0x7c7ffefb, 0xcdc033f1, 0xb208b413, 0x7213cd81,  //- |3r
   0x7fb4b92c, 0xeee881b2, 0x75e40a00, 0xfe3e8120,  //- ,u >
   0x75aa557d, 0x7dbebb18, 0x0a047f80, 0x81b20675,  //- }Uu}u
   0x31ebcb8b, 0x8110c383, 0x7c7dfefb, 0x7fbebbeb,  //- 1}|뻾
   0x3f80c933, 0x0b087580, 0x8b6d75c9, 0x8005ebcb,  //- 3ɀ?uum
   0x6475003f, 0x8110c383, 0x7c7ffefb, 0x75c90be5,  //- ?ud|u
   0xb218cd02, 0x21e86080, 0x51526100, 0x0a0098e8,  //- `!aRQ
   0xbe0574e4, 0x40eb7f0f, 0x817f24be, 0x557ffe3e,  //- t@$>U
   0x5e3575aa, 0x7c00ea5a, 0x41b40000, 0xcd55aabb,  //- u5^Z|AU
   0x81187213, 0x75aa55fb, 0x21fc8012, 0xc1f60d72,  //- rUu!r
   0x66087401, 0x333149b8, 0x3302eb58, 0x300068c0,  //- tfI13X3h0
   0x6664a10f, 0xc30000a3, 0x02ebdb33, 0x0eb410cd,  //- df3
   0x75c00aac, 0xfeebfbf7, 0x61420012, 0x61702064,  //- uBad pa
   0x742d7472, 0x656c6261, 0x0a0d2173, 0x45001200,  //- rt-tables!E
   0x726f7272, 0x616f6c20, 0x676e6964, 0x0d534f20,  //- rror loading OS
   0x0012000a, 0x7953704f, 0x6f6e2073, 0x6f662074,  //- OpSys not fo
   0x21646e75, 0x8b000a0d, 0x0005bfd9, 0x0f300068,  //- und!ٿh0
   0x3e8064a1, 0x74490000, 0x024f8b18, 0xbb01778a,  //- d>ItOw
   0xc0337c00, 0x01b813cd, 0x7313cd02, 0xf27f4f03,  //- |3sO
   0x8b661ec3, 0xa00f0847, 0x0008be1f, 0x0004a366,  //- fGf
   0x08448966, 0xc7c03366, 0xc7001004, 0x00010244,  //- fDf3D
   0x000444c7, 0x0644897c, 0x0c448966, 0x13cdc02b,  //- D|DfD+
   0x13cd42b4, 0x774f0373, 0x00c31ff3, 0x00000000,  //- BsOw
   0x00000000, 0x00000000, 0x00000000, 0x00000000,  //- 
   0x00000000, 0x00010000, FDSK_NTSIG, 0x0000cc33,  //- DFS73..
   0x00000000, 0x00000000, 0x00000000, 0x00000000,  //- 
   0x00000000, 0x00000000, 0x00000000, 0x00000000,  //- 
   0x00000000, 0x00000000, 0x00000000, 0x00000000,  //- 
   0x00000000, 0x00000000, 0x00000000, 0xaa550000   //- U
};


/*****************************************************************************/
// Write boot-code-part for an MBR sector, optional clean partition-table
/*****************************************************************************/
ULONG dfsFdskNewMbr
(
   USHORT              disknr,                  // IN    disknr to work on
   BOOL                clean                    // IN    clean part-table
)
{
   ULONG               rc  = NO_ERROR;          // function return
   S_BOOTR            *mbr = NULL;
   S_BOOTR            *dsk = NULL;              // MBR from other disk

   ENTER();

   rc = dfsSelectDisk( disknr, FALSE, FALSE);   // Select disk (needed for BPS!)
   if (rc == NO_ERROR)
   {
      if (((mbr = TxAlloc( 1, dfsGetSectorSize())) != NULL) &&
          ((dsk = TxAlloc( 1, dfsGetSectorSize())) != NULL)  )
      {
         if (TxaOption('f'))                    // get from specified disk
         {
            USHORT           from = (USHORT) TxaOptNum('f', "Disk", 0);

            if ((rc = dfsSelectDisk( from, FALSE, FALSE)) == NO_ERROR)
            {
               rc = dfstReadPsn( DFSTORE, 0, 1, (BYTE *) dsk);
            }
            if (rc == NO_ERROR)
            {
               TxPrint( "\nSuccessfully read MBR bootcode from disk %hu\n", from);
               rc = dfsSelectDisk( disknr, FALSE, FALSE); // re-Select target disk!
            }
            else
            {
               TxPrint("Error accessing MBR partition-info for disk %hu\n", from);
            }
         }
         if (rc == NO_ERROR)
         {
            DFSDISKINFO     *d;
            FILE            *fp = NULL;
            TXLN             fname;
            TXA_OPTION      *opt;

            memset( mbr, 0, dfsGetSectorSize()); // clean out new mbr

            dfsReadDiskInfo( FDSK_QUIET);       // up-to-date partition count!
            d = dfsGetDiskInfo( disknr);

            if (d && ((d->flags & DFS_F_FSYSTEMONLY) || // large floppy now
                      (d->flags & DFS_F_CRYPTO_DISK) )) // crypto header now
            {
               #if !defined (OEM_BRANDED)       // never allow newmbr FLP/CRP in OEM
                  if (dfsa->batch)              // in DFSee proper, only in batch
               #endif
               {
                  TxPrint("NEWMBR on %s style disk %hu not allowed, you need to WIPE it first!\n", d->pStyle, disknr);
                  rc = DFS_CMD_WARNING;         // need to wipe first ...
               }
               clean = TRUE;                    // force EMPTY tables, ignore garbage!
            }

            //- prepare the new MBR code from file or template
            if (rc == NO_ERROR)
            {
               if (clean == FALSE)              // read 'old' part-table
               {
                  rc = dfstReadPsn( DFSTORE, 0, 1, (BYTE *) mbr);
                  if (d && (rc == NO_ERROR))
                  {
                     TxPrint("Keep existing partition-table with %hu partitions "
                             "being used\n", d->totalpart);
                  }
                  else
                  {
                     TxPrint("Error accessing MBR partition-info for disk %hu\n", disknr);
                  }
               }
               else
               {
                  TxPrint("Partition-table initialized and empty, 0 partitions left\n"
                          "ALL PARTITIONS ARE TO BE DELETED !!!\n");
               }
               mbr->Signature = SV_BOOTR;       // set bootcode and signature

               strcpy( fname, "");
               if ((opt = TxaOptValue('I')) != NULL) // explicit Image
               {
                  switch (opt->type)
                  {
                     case TXA_NO_VAL: strcpy(  fname, "");                          break;
                     case TXA_STRING: strcpy(  fname,         opt->value.string);   break;
                     default:         sprintf( fname, "%lld", opt->value.number);   break;
                  }
                  if (strlen( fname) == 0)
                  {
                     if (dfsa->batch)           // batch-mode, use default name
                     {
                        strcpy( fname, "newmbr.img");
                     }
                     else
                     {
                        TxPrompt( 5531, TXMAXLN, fname, "NEWMBR Image filename");
                     }
                  }
                  if ((fp = fopen( fname, "rb")) == NULL) // not present here
                  {
                     if (TxFindByPath( fname, "PATH", fname) != NULL)
                     {
                        fp = fopen( fname, "rb");
                     }
                  }
                  if (fp != NULL)               // present
                  {
                     if (fread( mbr, 1, FDSK_WNTMBR_SIZE, fp) == FDSK_WNTMBR_SIZE)
                     {
                        TxPrint( "\nSuccessfully read MBR bootcode from: '%s'\n", fname);
                        fclose( fp);
                     }
                     else
                     {
                        TxPrint( "\nError reading from Imagefile '%s', MBR code will be "
                                 "taken\nfrom the DFSee default MBR template!\n", fname);
                        fclose( fp);
                        fp = NULL;              // signal not loaded
                     }
                  }
                  else
                  {
                     TxPrint( "\nMBR Imagefile '%s' not found, MBR code will be "
                              "taken\nfrom the DFSee default MBR template!\n", fname);
                  }
               }

               if (fp == NULL)                  // no imagefile loaded
               {
                  if (TxaOption('f'))           // get from specified disk
                  {
                     TRACES(("memcpy data from other disk mbr 0x%8.8x, size: %d\n", dsk, FDSK_WNTMBR_SIZE));
                     memcpy( mbr, dsk, FDSK_WNTMBR_SIZE);
                  }
                  else                          // use built-in MBR code
                  {
                     if ((TxaOption('n')) ||    // NO MBR code wanted
                         (TxaOption('N'))  )    // NO NT signature too
                     {
                        memset( mbr, 0, FDSK_WNTMBR_SIZE + ((TxaOption('n')) ? 0 : 6));
                        TxPrint( "\nMBR bootcode cleared to zeroes %s NT-signature value.\n",
                                    TxaOption('n') ? "upto" : "including");
                     }
                     else                       // use DFSee default code
                     {
                        TRACES(("memcpy data from DFSee template area, size: %d\n", FDSK_WNTMBR_SIZE));
                        memcpy( mbr, (void *) &dfsFdskMbrTemplate[0], FDSK_WNTMBR_SIZE);
                     }
                  }
               }
            }

            //- confirm and warn when interactive; execute when confirmed or batch
            if (rc == NO_ERROR)
            {
               if ((dfsa->batch) || TxConfirm( 5151,
                   "%sWrite new bootcode in MBR on disk %hu, %s ? [Y/N] : ",
                   (d && (d->flags & DFS_F_FSYSTEMONLY)) ?
                                     "The disk seems to be 'FileSystem Only'! "
                                     "NEWMBR will DESTROY that current (FAT) filesystem and "
                                     "create an MBR with EMPTY partition tables!\n\n" :
                   (d && (d->flags & DFS_F_CRYPTO_DISK)) ?
                                     "The disk seems to be (LUKS) encrypted! "
                                     "NEWMBR will DESTROY that encrypted volume and "
                                     "create an MBR with EMPTY partition tables!\n\n" : "",
                   disknr, (clean) ? "clearing the partition tables,\n\n"
                                     "DELETING CURRENT PARTITIONS, if any" :
                                     "leaving all partition tables intact"))
               {
                  if (DFSTORE_WRITE_ALLOWED)
                  {
                     rc = dfstWritePsn( DFSTORE, 0, 1, (BYTE *) mbr);
                     if (rc == NO_ERROR)
                     {
                        TxPrint("Bootcode in MBR on disk %hu refreshed\n", disknr);
                     }
                  }
                  else
                  {
                     rc = DFS_READ_ONLY;
                  }
               }
               else
               {
                  TxPrint( "Skipped MBR refresh on disk %hu, no change\n", disknr);
               }
            }
         }
         TxFreeMem( mbr);
         TxFreeMem( dsk);
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   RETURN (rc);
}                                               // end 'dfsFdskNewMbr'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Clear LVM DLAT sectors in MBR track clean (any geo), optional skip current
/*****************************************************************************/
ULONG dfsFdskClearLvmSectors
(
   USHORT              diskspec,                // IN    disk(s) to work on
   BOOL                skipcurrent              // IN    skip DLAT for current geo
)
{
   ULONG               rc = NO_ERROR;           // function return
   USHORT              disknr;
   ULN64               sn;

   ENTER();
   TRACES(("diskspec: %hu\n", diskspec));

   for (  disknr = 1;
        ((disknr <= dfsPartitionableDisks()) && (rc == NO_ERROR));
          disknr++)
   {
      if ((disknr == diskspec) || (diskspec == FDSK_ANY))
      {
         rc = dfsSelectDisk( disknr, FALSE, FALSE); // quietly select disk
         if (rc == NO_ERROR)
         {
            for (sn = 1; sn < DFS_2TB_SECT; sn++)
            {
               rc = dfstReadPsn( DFSTORE, sn, 1, rbuf);
               if (rc == NO_ERROR)
               {
                  if (dfsIdentifySector( sn, 0, rbuf) == ST_LVINF)
                  {
                     if (!skipcurrent || (sn != (dfstGeoSectors( DFSTORE) - 1)))
                     {
                        TxPrint("Clear LVM info in DLAT sector at: 0x%llX on disk %hu\n", sn, disknr);
                        memset( rbuf, 0, dfsGetSectorSize());
                        rc = dfstWritePsn( DFSTORE, sn, 1, rbuf);
                     }
                  }
               }
            }
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsFdskClearLvmSectors'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Write NT-signature in MBR sector, with specified or (DFSee) default value
/*****************************************************************************/
ULONG dfsFdskNtSignature
(
   USHORT              disknr,                  // IN    disknr to work on
   char               *newSig                   // IN    signature value or ""
)
{
   ULONG               rc;                      // function return
   S_BOOTR            *mbr;
   ULONG               ntSig;
   DFSDISKINFO        *d;                       // corresponding diskinfo
   BOOL                updateRequested = dfsa->batch;

   ENTER();

   if ((mbr = TxAlloc( 1, dfsGetSectorSize())) != NULL)
   {
      rc = dfsSelectDisk( disknr, FALSE, FALSE); // Select disk
      if (rc == NO_ERROR)
      {
         rc = dfstReadPsn( DFSTORE, 0, 1, (BYTE *) mbr);
         if (rc == NO_ERROR)
         {
            if ((newSig != NULL) && (strlen(newSig) != 0))
            {
               ntSig = dfsGetMcsNumber( newSig, 0);
            }
            else if (TxaOption(TXA_O_QUERY))
            {
               ntSig = mbr->NTsignature;
               updateRequested = FALSE;
            }
            else                                // determine default value
            {
               #if defined (USEWINDOWING)
                  if (dfsa->batch)
                  {
                     ntSig = FDSK_NTSIG + disknr;
                  }
                  else
                  {
                     ntSig = mbr->NTsignature;
                  }
               #else
                  ntSig = FDSK_NTSIG + disknr;
               #endif
            }

            if ((!TxaOption(TXA_O_QUERY)) && (!dfsa->batch))
            {
               #if defined (USEWINDOWING)       // Allow interactive update
               if (txwIsWindow( TXHWND_DESKTOP))
               {
                  TXLN      prompt;
                  TXTT      value;

                  sprintf( prompt, "Disk number %hu, signature now: %8.8x\n\n"
                                   "Specify 8-digit hexadecimal signature\n",
                                    disknr, mbr->NTsignature);
                  sprintf( value,  "%8.8x", ntSig);

                  if (txwPromptBox( TXHWND_DESKTOP, TXHWND_DESKTOP, NULL,
                         prompt, " Set Windows disk signature in MBR ",
                         5141, TXPB_MOVEABLE | TXPB_HCENTER | TXPB_VCENTER,
                         11, value) != TXDID_CANCEL)
                  {
                     updateRequested = TRUE;
                     sscanf( value, "%x", &ntSig); // get specified hex value
                  }
               }
               else
               #endif
               {
                  updateRequested = TxConfirm( 5142,
                     "Write NT-signature value 0x%8.8X in MBR on disk %hu ? [Y/N] : ",
                      ntSig, disknr);
               }
            }

            if (updateRequested)
            {
               if (DFSTORE_WRITE_ALLOWED)
               {
                  mbr->NTsignature = ntSig;
                  rc = dfstWritePsn( DFSTORE, 0, 1, (BYTE *) mbr);
                  if (rc == NO_ERROR)
                  {
                     d = dfsGetDiskInfo( disknr);
                     if (d != NULL)
                     {
                        d->NTsignature = ntSig; // update in-memory info too

                        TxPrint("NT-signature in MBR on disk %hu set to 0x%8.8X\n",
                                 disknr, ntSig);
                     }
                  }
               }
               else
               {
                  rc = DFS_READ_ONLY;
               }
            }
            else
            {
               if (TxaOption(TXA_O_QUERY))
               {
                  if (TxaOptUnSet('v'))         // non-verbose -v- ?
                  {
                     TxPrint( "0x%8.8X\n", ntSig);
                  }
                  else
                  {
                     TxPrint( "Signature  Disk %hu : 0x%8.8X\n", disknr, ntSig);
                  }
               }
               else
               {
                  TxPrint( "Skipped NT-signature for disk %hu, no change\n", disknr);
                  rc = DFS_NO_CHANGE;
               }
            }
         }
         else
         {
            TxPrint("Error accessing MBR-sector (0) for disk %hu\n", disknr);
         }
      }
      TxFreeMem( mbr);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsFdskNtSignature'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Iterate over all or selected MBR-partitions, using ExecOn and callback
/*****************************************************************************/
ULONG dfsIteratePmbr                            // primaries + ext-container
(
   FDSK_FUNCTION       callback,                // IN    operation callback
   FDSK_CB_INFO       *cbi                      // INOUT call data
)
{
   ULONG               rc = NO_ERROR;           // function return
   DFSPARTINFO        *p;                       // ptr to partition info
   DFSDISKINFO        *d;                       // ptr to disk info
   USHORT              parts;                   // number of partitions
   USHORT              part;                    // current partition index
   USHORT              disk   = 0;              // current disk number
   BOOL                ebrtoo = FALSE;          // current ebrtoo flag

   ENTER();
   if ((callback != NULL) && (cbi != NULL))
   {
      parts = dfsPartitions();
      for ( part = 1;
           (part <= parts) && ((rc == NO_ERROR) || (rc == DFS_NO_CHANGE));
            part++)
      {
         if (cbi->cbOption2)                    // start from highest PID
         {                                      // descending order
            p = dfsGetPartInfo( parts - part +1);
         }
         else                                   // normal ascending order
         {
            p = dfsGetPartInfo( part);
         }
         if (p != NULL)                         // may be NULL when deleting!
         {
            if (disk != p->disknr)              // new disk
            {
               ebrtoo = cbi->related;           // do related ebr too
               disk   = p->disknr;
            }
            if (((p->disknr  == cbi->disknr ) || (cbi->disknr  == FDSK_ANY)) &&
                ((cbi->stype == 0) || (cbi->stype == p->partent.PartitionType)))
            {
               if (p->primary)
               {
                  rc = dfsExecOnBootRec( p, 0, callback, cbi);
               }
               else                             // there must be an ebrChain
               {
                  if (ebrtoo)                   // want to handle extended ?
                  {
                     if ((d = dfsGetDiskInfo( disk)) != NULL)
                     {
                        rc = dfsExecOnBootRec( d->ebrHead, 0, callback, cbi);
                        if (rc == DFS_NO_CHANGE)
                        {
                           rc = NO_ERROR;       // one skipped, continue
                        }
                     }
                     ebrtoo = FALSE;            // only one call per disk
                  }
               }
            }
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsIteratePmbr'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Iterate over all or selected EBR-partitions, using ExecOn and callback
/*****************************************************************************/
ULONG dfsIterateExtContainers                   // all ebrs
(
   FDSK_FUNCTION       callback,                // IN    operation callback
   FDSK_CB_INFO       *cbi                      // INOUT call data
)
{
   ULONG               rc = NO_ERROR;           // function return
   DFSDISKINFO        *d;                       // ptr to disk info
   DFSPARTINFO        *e;                       // ptr to EBR partition info
   USHORT              disk;                    // current disk index

   ENTER();
   if ((callback != NULL) && (cbi != NULL))
   {
      for ( disk = 1;
           (disk <= dfsPartitionableDisks()) &&
                    ((rc == NO_ERROR) || (rc == DFS_NO_CHANGE));
            disk++)
      {
         if ((disk == cbi->disknr) || (cbi->disknr  == FDSK_ANY))
         {
            d = dfsGetDiskInfo( disk);
            if (d != NULL)
            {
               for ( e = d->ebrHead;
                    (e) && (rc == NO_ERROR) && !TxAbort();
                     e = e->ebrChain)
               {
                  if ((cbi->stype == 0) ||
                      (cbi->stype == e->partent.PartitionType))
                  {
                     TRACES(( "ebrID:%2hu tablenr:%2hu partnr:%hu partPsn:%llx basePsn:%llx size:%u\n",
                               e->id, e->tablenr, e->partnr, e->partPsn,      e->basePsn,
                                                  DFSECT2MIB(e->sectors,      e->bpsector)));
                     rc = dfsExecOnBootRec( e, 0, callback, cbi);
                     if (rc == DFS_NO_CHANGE)
                     {
                        rc = NO_ERROR;          // one skipped, continue
                     }
                  }
               }
            }
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsIterateExtContainers'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Patch the in-memory ebr-chain, re-route a specified link to new info-block
/*****************************************************************************/
BOOL dfsInsertInMemEbrChain
(
   USHORT              disknr,                  // IN    physical disk nr
   DFSPARTINFO        *oldLink,                 // IN    value to replace
   DFSPARTINFO        *newLink,                 // IN    replacement value
   DFSPARTINFO       **prevEbr                  // OUT   prev ebr or NULL
)
{
   BOOL                rc = FALSE;              // function return
   DFSPARTINFO       **e;                       // address of ptr to ebr info
   DFSDISKINFO        *d;

   ENTER();

   if ((d = dfsGetDiskInfo( disknr)) != NULL)
   {
      TRARGS(("Disk:%hu, ebrHead:%p; replace Link:%p by %p\n", disknr, d->ebrHead, oldLink, newLink));

      e = &d->ebrHead;                          // address of head ptr
      do
      {
         if (e != NULL)
         {
            *prevEbr = *e;                      // previous or NULL
            TRACES(("e:%p  *e:%p  id:%hu = %s\n", e, *e, (*e) ? (*e)->id : 0, (*e) ? (*e)->descr : "-NULL-"));
            if (*e == oldLink)                  // *e can be NULL (empty list)
            {
               *e = newLink;                    // route link to new value
               TRACES(("repl ptr to ID:%hu, descr %s\n", (*e)->id, (*e)->descr));
               rc = TRUE;
            }
            else if (*e == NULL)                // check for end-of-list
            {
               break;
            }
            e = &((*e)->ebrChain);              // address of next ptr
         }
      } while ((e != NULL) && (rc == FALSE));   // or until end-of-list ...
   }
   BRETURN (rc);
}                                               // end 'dfsInsertInMemEbrChain'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Iterate over all or selected partitions, using ExecOn and callback
/*****************************************************************************/
ULONG dfsIteratePart                            // all parts, optional ebr
(
   FDSK_FUNCTION       callback,                // IN    operation callback
   FDSK_CB_INFO       *cbi                      // INOUT call data
)
{
   ULONG               rc = NO_ERROR;           // function return
   DFSPARTINFO        *p;                       // ptr to partition info
   USHORT              parts;                   // number of partitions
   USHORT              part;                    // current partition index
   USHORT              diskselect;              // disk selection

   ENTER();
   if ((callback != NULL) && (cbi != NULL))
   {
      diskselect = cbi->disknr;
      parts = dfsPartitions();
      for ( part = 1;
           (part <= parts) && ((rc == NO_ERROR) || (rc == DFS_NO_CHANGE));
            part++)
      {
         p = dfsGetPartInfo( part);
         if (p && (((p->disknr  == diskselect ) || (diskselect  == FDSK_ANY)) &&
                  ((cbi->stype == 0) || (cbi->stype == p->partent.PartitionType))))
         {
            cbi->disknr   = p->disknr;          // disk to read from
            if ((p->ebrChain != NULL) &&        // is there a related ebr
                (cbi->related))                 // and do we need to call
            {
               rc = dfsExecOnBootRec( p->ebrChain, 0, callback, cbi);
               if (rc == DFS_NO_CHANGE)
               {
                  rc = NO_ERROR;                // one skipped, continue
               }
            }
            if (rc == NO_ERROR)                 // callback for entry itself
            {
               rc = dfsExecOnBootRec( p, 0, callback, cbi);
               switch (rc)
               {
                  case DFS_NO_CHANGE:           // not updated
                  case DFS_ST_MISMATCH:         // was not formatted
                     rc = NO_ERROR;             // one skipped, continue
                     break;

                  default:
                     break;
               }
            }
         }
      }
      cbi->disknr = diskselect;                 // restore original value
   }
   RETURN (rc);
}                                               // end 'dfsIteratePart'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Execute on a partition table in a specified MBR/EBR record
// 1) Read (or create a new) MBR/EBR or PBR sector
// 2) Execute (callback) confirmation for operation to perform
// 3) Execute (callback) operation on specified partition-table (entry)
// 4) If allowed/requested, write the sector back to the disk
// 5) Execute (callback) pending operation after successful write
/*****************************************************************************/
ULONG dfsExecOnBootRec
(
   DFSPARTINFO        *p,                       // IN    partition to work on
   USHORT              disknr,                  // IN    verify disknr, or 0
   FDSK_FUNCTION       callback,                // IN    operation callback
   FDSK_CB_INFO       *cbi                      // INOUT call data
)
{
   ULONG               rc = NO_ERROR;           // function return
   BYTE                brType;                  // boot record type
   ULN64               brPsn;                   // boot record Psn
   S_BOOTR            *br;                      // master/extended boot record
   TXTM                text;                    // progress text

   ENTER();
   TRARGS(("p-info *:%p  ID:%hu  disknr:%hu cbi: 0x%p callback:0x%p\n",
            p, (p) ? p->id : 0, disknr, cbi, callback));

   if ((br = TxAlloc( 1, dfsGetSectorSize())) != NULL)
   {
      if (p != NULL)
      {
         if ((disknr == 0) || (disknr == FDSK_ANY) || (disknr == p->disknr))
         {
            rc = dfsSelectDisk( p->disknr, FALSE, FALSE); // Select disk quietly
            if (rc == NO_ERROR)                 // sets base to 0 (lsn == psn)
            {
               brPsn       = (cbi->usebase) ? p->basePsn : p->partPsn;
               cbi->thisSn = brPsn;

               //- ->sn is in use for dfsIteratePlvm etc (create, bug in 3.17)
               //- new field thisSn introduced for PSAVE callback

               TRACES(("Work on Psn: 0x%llx\n", brPsn));
               if ((cbi->creEbr) &&             // create EBR requested
                   (brPsn != 0))                // and possible EBR sector
               {
                  TxPrint( "Create new EBR at : 0x%llx  on disk %hu\n", brPsn, p->disknr);
                  memset( br, 0, dfsGetSectorSize());
                  br->Signature = SV_BOOTR;     // so it will be recognized ...
                  //- br.PartitionTable[0].BootSectorOffset = L64_NULL;
               }
               else                             // read existing BR for update
               {
                  rc = dfstReadPsn( DFSTORE, brPsn, 1, (BYTE *) br);
               }
               sprintf( text, "Skipped BR");    // default start text
               if (rc == NO_ERROR)
               {
                  brType = dfsIdentifySector( brPsn, 0, (BYTE *) br);
                  if ((cbi->creMbr != FDSK_CR_MBR_OFF) &&   //- create MBR allowed
                      (brPsn == 0))                         //- and possible MBR sector
                  {
                     if (brType != ST_MASTR)    // no MBR recognized yet,
                     {                          // create one on the fly ...
                        if ((brType        != ST_MCDDM) || // except on MCDDM that
                            (br->Signature != SV_BOOTR)  ) // has a PC signature
                        {
                           TxPrint( "Create new MBR on : disk %u\n", p->disknr);
                           TxPrint( "MBR bootcode used : ");
                           if (brType == ST_MCDDM) // MAC driver description
                           {
                              TxPrint( "None, keeping MAC driver-descriptions intact\n");
                           }
                           else if (cbi->creMbr == FDSK_CR_MBR_STD)
                           {
                              TxPrint( "DFSee I13X capable, English messages\n");
                              memset( br, 0, dfsGetSectorSize());
                              memcpy( br, (void *) &dfsFdskMbrTemplate[0],
                                      FDSK_NEWMBR_SIZE); // include NT-signature!
                           }
                           br->Signature = SV_BOOTR; // set PC-style signature
                        }
                        brType  = ST_MASTR;
                     }
                  }
                  if (  cbi->cbOption4      ||  // ignore sector-type (psave)
                      ((brType == ST_BOOTR) && (cbi->usebase)) ||
                       (brType == ST_BOOTX) ||
                       (brType == ST_MASTR) ||
                       (brType == ST_EXTBR))    // supported combination ?
                  {
                     if (cbi->verbose)
                     {
                        dfsCompactShowPart( p);
                     }
                     TRACES(("About to execute callback at: 0x%p\n", callback));
                     if (((callback)( p, br, FDSK_PREPARE, cbi)) == NO_ERROR)
                     {
                        sprintf( text, "Process BR");
                        rc = (callback)( p, br, FDSK_PERFORM, cbi);
                        if (rc == DFS_PENDING)  // conditional write, call again
                        {
                           if (cbi->nowrite)
                           {
                              rc = NO_ERROR;
                           }
                           else                 // writing back desired
                           {
                              if ((cbi->confirmed) || DFSTORE_WRITE_ALLOWED)
                              {
                                 rc = dfstWritePsn( DFSTORE, brPsn, 1, (BYTE *) br);
                              }
                              else
                              {
                                 rc = DFS_NO_CHANGE;
                              }
                           }
                           if (rc == NO_ERROR)  // do the post-processing
                           {
                              rc = (callback)( p, br, FDSK_PENDING, cbi);
                           }
                        }
                     }
                     else
                     {
                        rc = DFS_NO_CHANGE;
                     }
                  }
                  else
                  {
                     if (cbi->verbose)
                     {
                        TxPrint("Sector at : 0x0%llx is not a %s Boot Record\n",
                                 brPsn, (cbi->usebase) ? "Partition" :
                                               "Master or Extended");
                     }
                     rc = DFS_ST_MISMATCH;
                  }
                  nav.xtra = brPsn;             // make it easy to refer to
               }
               else                             // some read error
               {
                  strcat( text, " after read-error, function");
               }
               switch (rc)
               {
                  case NO_ERROR:
                     strcat( text," finished");
                     break;

                  case DFS_NO_CHANGE:
                  case DFS_ST_MISMATCH:
                     strcat( text,", no changes made");
                     break;

                  default:
                     strcat( text," failed");
                     break;
               }
               if (cbi->verbose)
               {
                  TxPrint("%s\n\n", text);
               }
            }
         }
         else
         {
            TxPrint("Partition %hu is not located on disk %hu\n", p->id, disknr);
            rc = DFS_NOT_FOUND;
         }
      }
      TxFreeMem( br);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsExecOnBootRec'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Identify MBR (or an EBR) using CRC and display description
/*****************************************************************************/
ULONG dfsMbrEbrIdentifyDisplay
(
   BYTE               *mbr,                     // IN    MBR/EBR to identify
   char               *lead                     // IN    8 char lead or NULL
)
{
   ULONG               rc = 0;                  // function return
   ULONG               crc;
   char               *area = (char *)    mbr;  // as ascii
   S_MCDDM            *sd   = (S_MCDDM *) mbr;  // as APM header
   TXTM                descr;                   // description

   ENTER();

   rc  = TxCrc32( area, FDSK_CRCMBR_SIZE);
   crc = TxCrc32( area, FDSK_CRCNLS_SIZE);
   if (!memcmp(sd->Signature,sg_mcddm,SG_MCDDM) && // APM without any bootcode
               sd->bpsector && sd->sectors       ) // but possible MBR-signature
   {
      strcpy( descr, "Apple Partition Map. Disk is NOT bootable on a PC!");
   }
   else if (!TxAreaEmpty( area, FDSK_CRCMBR_SIZE, area[0]))
   {
      switch (rc)                               // right value, more precise
      {
         case 0x9e591529:       strcpy( descr, "Lenovo Thinkpad, factory preload MBR");  break;
         case 0xb16c291f:       strcpy( descr, "Windows 7, generic English");            break;
         case 0xc585aea1:       strcpy( descr, "Windows Vista/8/8.1, generic English");  break;
         case 0x50047175:       strcpy( descr, "Windows (W2K/XP) German NLS version");   break;
         case 0x7d590a21:       strcpy( descr, "Classic, Warp-3 .. Win-XP");             break;
         case 0x0b6012f2:       strcpy( descr, "Windows 98 (MS-DOS 7)");                 break;
         case 0xf1680109:       strcpy( descr, "Windows NT version 4");                  break;
         case 0x22c6fd33:       strcpy( descr, "IBM OS/2 Warp 4 English");               break;
         case 0x132f2c65:
         case 0xedc95dc5:
         case 0x2fb01f85:       strcpy( descr, "IBM OS/2 version 2.xx, US/English");     break;
         case 0x2f601f85:       strcpy( descr, "IBM OS/2 version 2.xx, German");         break;
         case 0x2fc0bf03:       strcpy( descr, "IBM OS/2 Warp 4 German");                break;
         case 0x0c8ca699:       strcpy( descr, "DFSee generic bootcode, US-English");    break;
         case 0x01f61820:       strcpy( descr, "VirtualPC or classic DOS");              break;
         case 0xc7a1b67b:       strcpy( descr, "Caldera DrDOS");                         break;
         case 0x00be54bb:       strcpy( descr, "MS-DOS 6, Nederlands");                  break;
         case 0xcce185ed:       strcpy( descr, "IBM OS/2 LVM or W4 from FP9");           break;
         case 0x7ea9d168:       strcpy( descr, "IBM OS/2 LVM, 14.105 May 2006");         break;
         case 0xac7036e0:       strcpy( descr, "IBM OS/2 Warp 3 Connect US");            break;
         case 0x54af7652:       strcpy( descr, "IBM OS/2 Warp 3 English");               break;
         case 0x57534a57:
         case 0xdd3c5a78:       strcpy( descr, "IBM Thinkpad, factory preload MBR");     break;
         case 0x884764d2:
         case 0x99ab7a76:       strcpy( descr, "Iomega Corporation boot code (ZIP)");    break;
         case 0x9d5ab559:       strcpy( descr, "Generic memory-stick bootcode sector");  break;
         case 0xaa3df0e4:       strcpy( descr, "Systemsoft Boot Partition Sector");      break;
         case 0x6f3c3048:       strcpy( descr, "Symantec/Norton Partition Magic");       break;
         case 0xca075bcd:       strcpy( descr, "Veit Kannegieser VPART bmgr");           break;
         case 0xfedd998e:       strcpy( descr, "FreeBSD master boot code");              break;
         case 0x050e59c7:       strcpy( descr, "Linux generic non-GRUB bootcode");       break;
         default:
            switch (crc)                        // left value, less precise (NLS)
            {
               case 0x2e1e78bc: strcpy( descr, "Classic MS orIBM generic bootcode");     break;
               case 0x3669dc25: strcpy( descr, "Windows 7, possibly NLS version");       break;
               case 0x78313bf1: strcpy( descr, "Windows Vista, possibly NLS version");   break;
               case 0xb5be8c03: strcpy( descr, "Windows (W2K/XP) NLS version");          break;
               case 0xfc947e97: strcpy( descr, "IBM OS/2 Warp 3 NLS version");           break;
               case 0x32c34502: strcpy( descr, "Veit Kannegieser OS2LDR.SEK patch");     break;
               case 0x3860051e: strcpy( descr, "PendriveLinux Multi ISO boot from USB"); break;
               case 0x8c619cc9: strcpy( descr, "Puppy SysLinux MBR, DFSee Slacko USB");  break;
               case 0x6f875380: strcpy( descr, "Syslinux generic MBR code (ISOHybrid)"); break;
               case 0x50c0d3f1: strcpy( descr, "Linux GRUB loader/bootmanager");         break;
               case 0xb557f028: strcpy( descr, "Linux GRUB(2) generic bootcode");        break;
               default:
                  if (memcmp( &area[2], "AiRBOOT", 7) == 0)
                  {
                     sprintf( descr, "Airboot BM, %2.2hhx-%2.2hhx-%2.2hhx%2.2hhx version %hhx.%2.2hhx Lang:%c",
                                      area[9], area[10], area[11], area[12], area[13], area[14], area[15]);
                  }
                  else if (TxMemStr( area, "PLoP Boot Manager", SECTORSIZE) != NULL)
                  {
                     strcpy( descr, "Plop Boot Manager");
                  }
                  else if (TxMemStr( area, "SYSCMNDRSYS", SECTORSIZE) != NULL)
                  {
                     strcpy( descr, "V Communications System Commander");
                  }
                  else if (TxMemStr( area, "LILO", SECTORSIZE) != NULL)
                  {
                     strcpy( descr, "Linux LILO loader and bootmanager");
                  }
                  else if (TxMemStr( area, "GRUB", SECTORSIZE) != NULL)
                  {
                     strcpy( descr, "Linux GRUB loader and bootmanager");
                  }
                  else if (TxMemStr( area, "BootMagic", SECTORSIZE) != NULL)
                  {
                     strcpy( descr, "PowerQuest BootMagic bootmanager");
                  }
                  else if (TxMemStr( area, "Apple iPod", SECTORSIZE) != NULL)
                  {
                     strcpy( descr, "Apple iPod partitioned device");
                  }
                  else if (TxMemStr( area, "OS2LDR     ", SECTORSIZE) != NULL)
                  {
                     strcpy( descr, "OS2LDR.SEK based FAT bootsector");
                  }
                  else
                  {
                     strcpy( descr, "MBR bootcode unknown to DFSee");
                  }
                  break;
            }
            break;
      }
      if (TxMemStr( area, "I13X", 500) != NULL)
      {
         strcat( descr, ", I13X");
      }
   }
   else                                         // area is empty
   {
      strcpy( descr, "Bootcode not present. Start of sector is empty!");
   }
   if (lead != NULL)                            // report
   {
      TxPrint( "%s %8.8x : 0x%8.8x = %s\n", lead, crc, rc, descr);
   }
   RETURN (rc);
}                                               // end 'dfsMbrEbrIdentifyDisplay'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Identify MBR (or an EBR) using CRC, return maximum 8 character description
/*****************************************************************************/
ULONG dfsMbrEbrBootcodeDescription
(
   BYTE               *mbr,                     // IN    MBR/EBR to identify
   char               *descr                    // OUT   Max 8 char description
)
{
   ULONG               rc = 0;                  // function return
   ULONG               crc;
   char               *area = (char *) mbr;

   ENTER();

   strcpy( descr, "");

   rc  = TxCrc32( area, FDSK_CRCMBR_SIZE);
   crc = TxCrc32( area, FDSK_CRCNLS_SIZE);
   if (dfsIdentifySector( 0, 0, mbr) == ST_MCDDM)
   {
      strcpy( descr, "MACdrive");
   }
   else if (!TxAreaEmpty( area, FDSK_CRCMBR_SIZE, area[0]))
   {
      switch (rc)                               // right value, more precise
      {
         case 0x9e591529:       strcpy( descr, "Lenovo");   break;
         case 0xb16c291f:       strcpy( descr, "Windows7"); break;
         case 0xc585aea1:       strcpy( descr, "WindowsV"); break;
         case 0x50047175:       strcpy( descr, "WinXP-DE"); break;
         case 0x7d590a21:       strcpy( descr, "XP-Warp3"); break;
         case 0x0b6012f2:       strcpy( descr, "W98/DOS7"); break;
         case 0xf1680109:       strcpy( descr, "Wind-NT4"); break;
         case 0x2fc0bf03:
         case 0xac7036e0:
         case 0x54af7652:
         case 0x22c6fd33:       strcpy( descr, "OS/2Warp"); break;
         case 0x132f2c65:
         case 0xedc95dc5:
         case 0x2f601f85:
         case 0x2fb01f85:       strcpy( descr, "OS/2-2.x"); break;
         case 0x0c8ca699:       strcpy( descr, "DFSee");    break;
         case 0x01f61820:       strcpy( descr, "DOS/VPC");  break;
         case 0xc7a1b67b:       strcpy( descr, "DrDOS");    break;
         case 0x00be54bb:       strcpy( descr, "DOS6-NL");  break;
         case 0xcce185ed:
         case 0x7ea9d168:       strcpy( descr, "OS/2-LVM"); break;
         case 0xdd3c5a78:       strcpy( descr, "Thinkpad"); break;
         case 0x884764d2:
         case 0x99ab7a76:       strcpy( descr, "Iomega");   break;
         case 0x9d5ab559:       strcpy( descr, "USBstick"); break;
         case 0xaa3df0e4:       strcpy( descr, "Sys-soft"); break;
         case 0x6f3c3048:       strcpy( descr, "PQ-Magic"); break;
         case 0xca075bcd:       strcpy( descr, "VPART-BM"); break;
         case 0xfedd998e:       strcpy( descr, "FreeBSD");  break;
         case 0x050e59c7:       strcpy( descr, "Linux");    break;
         default:
            switch (crc)                        // left value, less precise (NLS)
            {
               case 0x2e1e78bc: strcpy( descr, "MS / IBM"); break;
               case 0x3669dc25: strcpy( descr, "Windows7"); break;
               case 0x78313bf1: strcpy( descr, "WindowsV"); break;
               case 0xb5be8c03: strcpy( descr, "Win2K/XP"); break;
               case 0xfc947e97: strcpy( descr, "OS/2Warp"); break;
               case 0x32c34502: strcpy( descr, "OS2LDR");   break;
               case 0x3860051e: strcpy( descr, "Pendrive"); break;
               case 0x50c0d3f1:
               case 0xb557f028: strcpy( descr, "GRUB");     break;
               default:
                  if      (TxMemStr( area, "AiRBOOT", SECTORSIZE) != NULL)
                  {
                     strcpy( descr, "AirBoot");
                  }
                  else if (TxMemStr( area, "SYSCMNDRSYS", SECTORSIZE) != NULL)
                  {
                     strcpy( descr, "SysCmder");
                  }
                  else if (TxMemStr( area, "LILO", SECTORSIZE) != NULL)
                  {
                     strcpy( descr, "LILO");
                  }
                  else if (TxMemStr( area, "GRUB", SECTORSIZE) != NULL)
                  {
                     strcpy( descr, "GRUB");
                  }
                  else if (TxMemStr( area, "BootMagic", SECTORSIZE) != NULL)
                  {
                     strcpy( descr, "PQ-Magic");
                  }
                  else if (TxMemStr( area, "OS2LDR     ", SECTORSIZE) != NULL)
                  {
                     strcpy( descr, "OS2LDR");
                  }
                  break;
            }
            break;
      }
   }
   TRACES(("Short Bootcode desription: '%s'\n", descr));
   RETURN (rc);
}                                               // end 'dfsMbrEbrBootcodeDescription'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read EBR sector from specified sectornumber and get Bootcode description
/*****************************************************************************/
ULONG dfsGetBootcodeDescription
(
   ULN64               psn,                     // IN    MBR/EBR sectornumber
   char               *description              // OUT   Max 8 char description
)
{
   ULONG               rc = NO_ERROR;           // DOS rc
   BYTE               *boot = NULL;             // boot sector buffer

   ENTER();

   if ((boot = TxAlloc( 1, dfsGetSectorSize())) != NULL) // 2 sectors, any size
   {
      rc = dfstReadPsn( DFSTORE, psn, 1, boot);
      if (rc == NO_ERROR)
      {
         dfsMbrEbrBootcodeDescription( boot, description);
      }
      TxFreeMem( boot);
   }
   RETURN (rc);
}                                               // end 'dfsGetBootcodeDescription'
/*---------------------------------------------------------------------------*/

