//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// Partition utility, display and iterator functions; display section
//
// Author: J. van Wijk
//
// JvW  18-08-2005 Initial version, split from DFSUPART
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsver.h>                             // DFS version info
#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsupart.h>                           // partition utility functions
#include <dfsufdsk.h>                           // Fdisk & translation services
#include <dfsutil.h>                            // DFS utility functions
#include <dfsafdsk.h>                           // FDISK generic definitions
#include <dfsufdsk.h>                           // FDISK utility functions


static char dfsLvm20Warning[] =
 "                    This can cause some problems with 'LVM /query',\n"
 "                    eComStation installation and some other tools.\n";
static char dfsLvm20FixTip[] =
 "                    Use 'lvm -P- *' to automatically shorten them.\n";
static char dfsLvmCrcFixTip[] =
 "                    Use 'lvm -c' to fix the LVM CRC and sequence-nr values.\n";
static char dfsFixPbrTip[] =
 "                    It may be harmless, but when it seems to cause problems,\n"
 "                    you can probably fix it by opening the partition and use\n"
 "                    the FIXPBR command or the corresponding menu item being:\n"
 "                    Mode=xxx -> Boot area fixes -> Fix HiddenSectors/Geo value\n";
static char dfsAlignExtTip[] =
 "                    Use the ALIGNEXT command to align its extended container, or\n"
 "                    Mode=FDISK->Part-table cleanup/repair->Align EXT container, Part\n";

/*****************************************************************************/
// Show contents of one partition-table entry (as in 'walk')
/*****************************************************************************/
void dfsShowPartEntry
(
   char               *lead,                    // IN    leading text 1st line
   BYTE                nav_st,                  // IN    update nav, using st
   BOOL                relocated,               // IN    Show PID, but no 'BAD'
   ULONG               geoHeads,                // IN    geometry, heads
   ULONG               geoSecs,                 // IN    geometry, sectors
   ULONG               thisPsn,                 // IN    PSN of table sector
   DFSPARTENTRY       *pe                       // IN    part entry ptr
)
{
   ULONG               c,h,s;                   // cyl, head and sector
   TXTS                text;                    // small string
   ULONG               calcs;                   // Calculated start psn
   ULONG               calce;                   // Calculated end   psn
   ULONG               cc,ch,cs;                // Calculated cyl, head, sect

   ENTER();

   if (dfsIsExtendedType( pe->PartitionType))
   {
      calcs = pe->BootSectorOffset;             // calculate start
      if (nav_st != ST_MASTR)
      {
         calcs += dfsEbrBase();                 // add EBR base offset
      }
      calce = calcs + pe->NumberOfSectors -1;   // calculate end
      if (nav_st != 0)
      {
         nav.down = calcs;                      // Enter goes to start
      }
   }
   else
   {
      calcs    = thisPsn  + pe->BootSectorOffset;
      calce    = calcs    + pe->NumberOfSectors -1;
      if (nav_st != 0)
      {
         nav.up   = calcs;                      // u goes to bootsector
         nav.xtra = calcs;                      // x goes to bootsector too
         if (dfsa->lvmPresent)
         {                                      // or to LVM-info when present
            if (thisPsn == 0)
            {
               nav.xtra = geoSecs -1;           // LVM for MBR is fixed
            }
            else
            {
               nav.xtra--;                      // sector before bootsector
            }                                   // is the LVM info sector
         }
      }
   }

   dfsPartTypeDescription( pe->PartitionType, text);
   TxPrint("%sType 0x%02.2x  = %s%s%s %s %s%s",
           lead, pe->PartitionType, CBC, text,
           CNC, (pe->Status) ? "<---------"       : "          ",
           CNN, (pe->Status) ? "Active/Startable" : "                ");
   if ((relocated) && (SINF->disknr != 0))
   {
      TxPrint(" PID : %02.2hu\n", dfsDiskPsn2PartId( SINF->disknr, calcs));
   }
   else
   {
      TxPrint( "\n");
   }

   dfsX10( " LBA offset value : ", pe->BootSectorOffset, (nav_st != ST_MASTR) ? CBG : CBY, "");
   if (pe->NumberOfSectors == DFS32MAX)
   {
      TxPrint(       "   Sectors: Maximum (whole disk)\n");
   }
   else
   {
      dfsSiz8(       "   Sectors: ", pe->NumberOfSectors,       "\n");
   }

   if (thisPsn != 0)                            // display base for EBR's
   {
      if (dfsIsExtendedType( pe->PartitionType))
      {
         dfsX10( " Offs relative to : ", dfsEbrBase(), CNN, " = EBRbase\n");
      }
      else
      {
         dfsX10( " Offs relative to : ", thisPsn, CNN, " = thisEBR\n");
      }
   }

   strcpy( text, "");                           // any non-MS / not-BAD value

   c = DFSC2CYLIND(pe->FirstSector.SecCyl);
   h =             pe->FirstSector.Head;
   s = DFSC2SECTOR(pe->FirstSector.SecCyl);
   dfsGeoPsn2Chs( calcs, geoHeads, geoSecs, &cc, &ch, &cs);
   dfsX10( " Begin sector PSN : ", calcs,
                   (nav_st != ST_MASTR) ? CBY : CNG, " = ");
   TxPrint("LBAchs:%s%8u %3u %3u%s",  CNY, cc, ch, cs,  CNN);
   if ((cc != c) || (ch != h) || (cs != s))     // calculated & CHS don't match
   {
      TxPrint(" CHS:%s%7u %3u %3u %s", CNC, c, h, s, CNG);
      if      ((h == 255)           && (s == 63     )) TxPrint( "MS");
      else if ((h == (geoHeads -1)) && (s == geoSecs)) TxPrint( "IBM");
      else if ((h == ch)            && (s == cs     )) TxPrint( "PQ");
      else if (!relocated)                             TxPrint( "%sBAD", CBR);
      strcpy( text, ((h == ch) && (s == cs)) ? "PQ" : "IBM");
   }
   TxPrint( "%s\n", CNN);

   c = DFSC2CYLIND( pe->LastSector.SecCyl);
   h =              pe->LastSector.Head;
   s = DFSC2SECTOR( pe->LastSector.SecCyl);
   dfsGeoPsn2Chs( calce, geoHeads, geoSecs, &cc, &ch, &cs);
   dfsX10( " End sector   PSN : ", calce, CNY, " = ");
   TxPrint("LBAchs:%s%8u %3u %3u%s", CNY, cc, ch, cs,  CNN);
   if ((cc != c) || (ch != h) || (cs != s))     // calculated & CHS don't match
   {
      TxPrint(" CHS:%s%7u %3u %3u %s", CNC, c, h, s, CNG);
      if      ((h == 255)           && (s == 63     )) TxPrint( "MS");
      else if ((h == (geoHeads -1)) && (s == geoSecs)) TxPrint( text);
      else if ((h == ch)            && (s == cs     )) TxPrint( "PQ");
      else if (!relocated)                             TxPrint( "%sBAD", CBR);
   }
   TxPrint( "%s\n", CNN);

   VRETURN ();
}                                               // end 'dfsShowPartEntry'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Show freespace area for specified freespace info structure
/*****************************************************************************/
void dfsShowFreeSpaceArea
(
   DFSPARTINFO        *f,                       // IN    freespace area info
   int                 cw                       // IN    name column width
)
{
   ULN64               sectors   = f->sectors;
   ULONG               cyl       = f->basePsn / f->cSC;
   BOOL                numeric   = TxaOption('n');
   BOOL                blocks    = TxaOption('b');
   ULONG               threshold = (numeric) ? 1 : f->geoSecs;
   TXTS                text;                      // small text buffer

   if (sectors > threshold)
   {
      TxPrint((f->id < 100) ? "%s%02.2d%s" : "%s%03.3d%s", CNC, f->id, CNN);
      TxPrint("     %s%s%s", CNC, f->descr, CNN);
      if (numeric)
      {
         sprintf( text, (cyl > 99999999) ? "%9u" : "%8u ", cyl);
         TxPrint("% 10llx% 10llx%s-%9u% *llx",
                        f->basePsn,  f->lastPsn, text,
               (ULONG) (f->lastPsn / f->cSC),
                        cw, sectors);
      }
      else if (blocks)                          // format/blocks
      {
         ULONG    blocksize = TxaOptNum('b', NULL, 1);

         sprintf( text, (cyl > 99999999) ? "%9u" : "%8u ", cyl);
         TxPrint("            %s-%9u%12llu%12llu%*llu",
                        text,
               (ULONG) (f->lastPsn / f->cSC),
                        f->basePsn, f->lastPsn,
                        cw, (f->sectors / blocksize));
      }
      else                                      // labels
      {
         if (dfsa->bmgrDisk || dfsa->lvmPresent)
         {
            strcpy( text,"");
         }
         else // No extra info to display, show size in HEX, sectors
         {
            sprintf( text, "Size 0x%llx sectors", f->sectors);
         }
         TxPrint("-- -- ---- -- --- - - - - -%- *.*s", cw, cw, text);
      }
      dfsShowMiBsize( sectors, f->bpsector);
   }
}                                               // end 'dfsShowFreeSpaceArea'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Show non-MBR disk (large-floppy/luks) and display its size in the table
/*****************************************************************************/
void dfsShowNonMbrDiskArea
(
   DFSDISKINFO        *d,                       // IN    disk information
   int                 cw                       // IN    name column width
)
{
   ULN64               sectors   = d->sectors;
   BOOL                numeric   = TxaOption('n');
   BOOL                blocks    = TxaOption('b');
   char                indicator = ' ';
   TXTS                descr;

   ENTER();
   TRACES(("sectors:0x%llx  pStyle:%s cSC:%u\n", sectors, d->pStyle, d->cSC));

   if (d->Removable)
   {
      indicator = 'r';
   }
   switch (d->pStyle[0])
   {
      case 'C': strcpy( descr, "FullEncryptedDisk"); break;
      case 'F': strcpy( descr, "FileSystemOnlyDsk"); break;
      case 'A': strcpy( descr, "ApplePartitionMap"); break;
      default:  strcpy( descr, "UnknownDiskFormat"); break;
   }
   TxPrint("--%c%s 0%s%2.2s%s%s%s", indicator, CNY, CNN, d->ddrive, CNM, descr, CNN);
   if (numeric)
   {
      TxPrint("         0% 10llx       0 -% 9u% *llx",
              sectors -1, (ULONG) ((sectors -1) / d->cSC), cw, sectors);
   }
   else if (blocks)                             // format/blocks
   {
      ULONG    blocksize = TxaOptNum('b', NULL, 1);

      TxPrint("%-5.5s%-7.7s       0 -% 9u         0% 10llx% *llu",
         d->fsform, d->creatr, (ULONG) ((sectors -1) / d->cSC), sectors -1, cw, (sectors -1) / blocksize);
   }
   else                                         // labels
   {
      TxPrint("%-8.8s%-8.8s%-11.11s%- *.*s", d->fsform, d->creatr, d->dlabel, cw, cw, d->dxInfo);
   }
   TxPrint( "%11.1lf\n", TXSMIB(sectors, d->bpsector));

   VRETURN();
}                                               // end 'dfsShowNonMbrDiskArea'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Show freespace area for specified freespace info structure, comma separated
/*****************************************************************************/
void dfsShowFreeSpaceSep
(
   DFSPARTINFO        *f,                       // IN    freespace area info
   char               *x,                       // IN    separator string
   char               *eol,                     // IN    end-of-line string
   char               *line                     // IN    line buffer, 1K
)
{
   ULN64               sectors;
   ULONG               threshold;

   sectors   = f->sectors;                      // net nr of sectors
   threshold = (TxaOption('n')) ? 1 : f->geoSecs;

   if ((sectors > threshold) && (!TxaOptUnSet('f')))
   {
      sprintf( line, "%2.2d%s%2d%s%c %-13.13s%s%-2.2s%s%s%s"
              "%2.2s%s%-*s%s%s%s%s%s%-8.8s%s%-8.8s%s%-11.11s%s%-20.20s%s"
              "%-20.20s%s%-8.8s%s%12llu%s%10llx%s%12llu%s%10llx%s"
              "%9u%s%9u%s%12llu%s%10llx%s%11.1lf%s",
                        f->id,      x, f->disknr,       x,
                   ' ', "",         x, "",              x,
                        "Free",                         x,
                        "",                             x,
                        #if defined (OEM_BRANDED)
                                  17,
                        #else
                                  19,
                        #endif
                        f->descr,                       x,
                        "  ",       x, "  ",            x,
                        "",         x, "",              x,
                        "",         x, "",              x,
                        "",         x, "",              x,
                        f->basePsn, x, f->basePsn,      x,
                        f->lastPsn, x, f->lastPsn,      x,
               (ULONG) (f->basePsn / f->cSC),           x,
               (ULONG) (f->lastPsn / f->cSC),           x,
                        sectors,    x, sectors,         x,
                TXSMIB( sectors, f->bpsector), eol);
       TxPrint( "%s", line);
       if (TxaExeSwitch('g'))
       {
          printf( "%s", line);
          fflush( stdout);
       }
   }
}                                               // end 'dfsShowFreeSpaceSep'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Show freespace area for specified freespace, comma separated query format
/*****************************************************************************/
void dfsShowFreeSpaceQuery
(
   DFSPARTINFO        *f,                       // IN    freespace area info
   char               *x,                       // IN    separator string
   char               *eol,                     // IN    end-of-line string
   char               *line                     // IN    line buffer, 128
)
{
   ULN64               sectors;
   ULONG               threshold;

   sectors   = f->sectors;                      // net nr of sectors
   threshold = (TxaOption('n')) ? 1 : f->geoSecs;

   if ((sectors > threshold) && (!TxaOptUnSet('f')))
   {
      sprintf( line, " %2.2d%s%d%s%c%s%-2.2s%s%s%s"
              "%-4.4s%s%-11.11s%s%-13.13s%s"
              "%9u%s%9u%s%11.1lf%s",
                        f->id,      x, f->disknr,       x,
                   ' ',             x, "",              x,
                        "Free",     x,
                        "",         x,
                        "",         x,
                        "",         x,
               (ULONG) (f->basePsn / f->cSC),           x,
               (ULONG) (f->lastPsn / f->cSC),           x,
                TXSMIB( sectors, f->bpsector), eol);
       TxPrint( "%s", line);
       if (TxaExeSwitch('g'))
       {
          printf( "%s", line);
          fflush( stdout);
       }
   }
}                                               // end 'dfsShowFreeSpaceQuery'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Show non-MBR type of disk info (CRP, FLP etc), comma separated query format
/*****************************************************************************/
void dfsShowNonMbrQuery
(
   DFSDISKINFO        *d,                       // IN    disk information
   char               *x,                       // IN    separator string
   char               *eol,                     // IN    end-of-line string
   char               *line                     // IN    line buffer, 1K
)
{
   ULN64               sectors   = d->sectors;
   char                indicator = ' ';
   TXTS                descr;

   if (d->Removable)
   {
      indicator = 'r';
   }
   switch (d->pStyle[0])
   {
      case 'C': strcpy( descr, "Encrypted"); break;
      case 'F': strcpy( descr, "LargeFlop"); break;
      default:  strcpy( descr, "Unknown!!"); break;
   }
   sprintf( line, " %c %s%d%s%9.9s%s"
           "%-4.4s%s%-11.11s%s%-13.13s%s"
           "%9u%s%9u%s%11.1lf%s",
                     indicator,  x, d->disknr,    x,
                     descr,      x,
                     d->fsform,  x,
                     d->creatr,  x,
                     d->dxInfo,  x,
                     0,          x,
           (ULONG) ( sectors  /  d->cSC),          x,
             TXSMIB( sectors,    d->bpsector), eol);
   TxPrint( "%s", line);
   if (TxaExeSwitch('g'))
   {
      printf( "%s", line);
      fflush( stdout);
   }
}                                               // end 'dfsShowNonMbrQuery'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Show verbose partition-info table for specified partition, handle -q / -O
/*****************************************************************************/
void dfsVerboseShowPart
(
   DFSPARTINFO        *p                        // IN    partition info
)
{
   USHORT              partype;
   DFSPARTINFO        *e;                       // related ebr partition
   TXTM                tbuf;                    // smallish temporary buffer
   char               *text = NULL;             // accumulating text buffer
   BOOL                loudness = dfsa->verbosity; // local verbosity setting

   ENTER();

   #if defined (OEM_BRANDED)
      if (TxaOption('q'))                       // -q overrules standard -O: setting
      {
         loudness = TXAO_QUIET;
      }
   #endif
   text = TxAlloc( 1, TXMAXLN);
   if ((p != NULL) && (text != NULL))
   {
      ULN64            orgbase = dfstLSN2Psn( DFSTORE, 0);
      ULN64            orgsize = dfsGetLogicalSize();

      dfstLogicalLimits( DFSTORE, p->basePsn, DFS_MAX_PSN);

      partype = p->partent.PartitionType;
      sprintf( text, "\nDFS partition  Id : %2.2d  disk nr : %d  = FixedDisk Diskgeo = ", p->id, p->disknr);
      TxPrint( "%sCHS:%7u %3u %3u\n", text, p->geoCyls, p->geoHeads, p->geoSecs);
      if (loudness > TXAO_QUIET)                // not quiet ?
      {
         strcpy( text, "");
         dfstrX10( text, " P-table for this : ",   p->partPsn, CBM, " = ");
         if      (p->tablenr == GPT_STYLE)
         {
            strcat(  text, "GPT-table,");
         }
         else if (p->tablenr == APM_STYLE)
         {
            strcat(  text, "APM-table,");
         }
         else if (p->tablenr)
         {
            sprintf( tbuf, "EBR:%s%d%s,",     CBM, p->tablenr, CNN);
            strcat(  text, tbuf);
         }
         else
         {
            strcat( text, "MBR,  ");
         }
         TxPrint( "%s entry:%s%d%s", text,  CBM, p->partnr,  CNN);
         if ((e = p->ebrChain) != NULL)         // related ebr info
         {
            strcpy( text, " Previous P-table: ");
            if (e->tablenr)
            {
               sprintf( tbuf, "EBR:%s%d%s,",     CNY, e->tablenr, CNN);
               strcat(  text, tbuf);
            }
            else
            {
               strcat( text, "MBR,  ");
            }
            TxPrint( "%s entry:%s%d%s\n",  text, CNY, e->partnr,  CNN);
         }
         else
         {
            TxPrint( "\n");
         }
      }
      TxStrip( text, p->UnixDevicePart, ' ', ' ');   // strip spaces from EOL
      TxPrint( " System Indicator : 0x%s%2.2x%s       = %s     UNIX devicename: %s%s%s\n",
                                    CBC, partype, CNN, p->descr, CNY, text, CNN );
      TxPrint( " FileSyst. format : %-8s  ", p->fsform);
      if (p->flags & DFS_F_BOOTRECBAD)
      {
         TxPrint( "(Not usable, FORMAT or recovery required)\n");
      }
      else                                      // show BOOT.INI part-nr
      {
         TxPrint( "                Windows BOOT.INI: Partition(%s%hu%s)\n", CBG, p->diskPart, CNN);
      }
      TxPrint( " Partition   type : %s   ", (p->primary) ? "Primary" : "Logical");
      TxPrint( " %s       ", ((dfsPartTypeHidable((BYTE)partype)) && (partype & DFS_P_PHIDDEN)) ? "Hidden " : "Visible");
      TxPrint( " Bootrec-OEM-name: %s\n", p->creatr);

      if (loudness > TXAO_QUIET)                // not quiet ?
      {
         if (partype == DFS_P_BOOTMGR)
         {
            TxPrint( " IBM BMGR version : %s\n", p->plabel);
         }
         else
         {
            TxPrint( " Partition Label  : %s%-11.11s%s", CBG, p->plabel, CNN);
            if (p->uuidPresent)
            {
               strcpy( text, "   Uuid : ");
               dfstrFsUuidValue( text, p->fsUuid, TRUE);
               TxPrint( "%s", text);
            }
            TxPrint( "\n");
         }
         sprintf( text,    " Boot capability  : ");
         if (strlen(p->blabel) > 0)
         {
            sprintf( tbuf, "Bootable by OS/2 BMGR as: %s%s%s BMGR name: %s%s%s",
                            CBC, p->blabel, CNN, CNC, p->bmname, CNN);
         }
         else
         {
            if  (p->partent.Status && p->primary && ((p->flags & DFS_F_BOOTRECBAD) == 0))
            {
               if (p->disknr == 1)
               {
                  sprintf( tbuf, "Bootable directly by any BIOS and MBR-code");
               }
               else
               {
                  sprintf( tbuf, "Only by modern BIOS and MBR-code, supporting disk %d", p->disknr);
               }
            }
            else
            {
               sprintf( tbuf, "Not by BIOS, MBR or OS/2 BMGR, maybe by other bootmanagers");
            }
         }
         strcat(  text,   tbuf);
         TxPrint( "%s\n", text);

         TxPrint(       " Bootsec checksum : 0x%8.8X\n", p->checksum);

         dfsGeoDispTransPsn( " PartTable offset :", p->geoHeads, p->geoSecs, p->partPsn);
         dfsGeoDispTransPsn( " 1st  part sector :", p->geoHeads, p->geoSecs, p->basePsn);
         dfsGeoDispTransPsn( " Last part sector :", p->geoHeads, p->geoSecs, p->lastPsn);
         strcpy( text, "");
         dfstrDec8( text, " LBA offset value : ", p->partent.BootSectorOffset, "      ");
         sprintf(   tbuf, "Cylinder size : %4.4x = %4.2lf MiB", p->cSC, TXSMIB( p->cSC, p->bpsector));
         TxPrint( "%s%s\n", text, tbuf);
      }

      strcpy( text, "");
      dfstrSz64Bps( text, " Partition   size : ", p->sectors, p->bpsector, "  =");
      TxPrint( "%s %10.3lf cylinders\n", text, (double)(((double) p->sectors) / ((double) p->cSC)));

      if (loudness > TXAO_QUIET)                // not quiet ?
      {
         USHORT        cs = (p->scluster) ? p->scluster : (p->bpsector) ? (4096 / p->bpsector) : SECTORSIZE;
         strcpy( text, "");
         dfstrSizeBps( text, (p->scluster) ? " Brec ClusterSize : " : " Default Blk Size : ", cs, p->bpsector, "");
         sprintf( tbuf, " in %hu sector(s) ", cs);
         strcat(  text, tbuf);
         TxPrint( "%s of %hu bytes\n",  text, p->bpsector);

         if (p->lvmPresent)
         {
            DFSDISKINFO *d = dfsGetDiskInfo( p->disknr);

            TxPrint( " LVM Ph disk name : %s%-*.*s%s    Drive letter: %s%c%s",
                  CNC, LVM_NAME_L, LVM_NAME_L, (d != NULL) ? d->DiskName : "",
                  CNN, CBG, (p->lvm.letter) ?  p->lvm.letter : '-', CNN);
            if (p->lvmDriveLinks >= 1)          // LVM-partition (non compat)
            {
               TxPrint( "   LVM-part %s%u%s of %s%u%s", CBY, p->lvmDriveIndex, CNN, CBM, p->lvmDriveLinks, CNN);
            }
            else                                // non-LVM 35, show BPB driveletter
            {
               if (p->pbrSect != NULL)          // we have a bootsector copy
               {
                  if ((strncmp( p->fsform, "HPFS", 4) == 0) ||
                      (strncmp( p->fsform, "JFS",  3) == 0)  )
                  {
                     TxPrint("  in BPB : 0x%2.2hx = ", p->pbrSect->os.FsysValue);
                     if (p->pbrSect->os.FsysValue < 0x80)
                     {
                        TxPrint( "data");
                     }
                     else
                     {
                        TxPrint( "%s%c:%s", CBG, (char) (p->pbrSect->os.FsysValue - 0x80 + 'C'), CNN);
                     }
                  }
               }
            }
            TxPrint( "\n LVM partition    : %s%-*.*s%s    Volume name : %s%-*.*s%s\n",
                  CNG, LVM_NAME_L, LVM_NAME_L, p->lvm.PartName, CNN,
                  CBC, LVM_NAME_L, LVM_NAME_L, p->lvm.VoluName, CNN);
         }
      }
      if ((p->pbrSect) && (TxMemStr(((BYTE *) p->pbrSect) +100, "GRUB", SECTORSIZE -100) != NULL))
      {
         dfsFdskGrubStage1((BYTE *) p->pbrSect, (dfsa->verbosity >= TXAO_VERBOSE));
      }
      if (loudness > TXAO_QUIET)                // not quiet ?
      {
         #if defined (UNIX)
            TxFsShow(  " ",   p->UnixDevicePart);
         #else
            TxFsShow(  "OS ", p->drive);
         #endif
         dfsShowPartWarnings( NULL, 0, NULL);   // signal start of sequence
         dfsShowPartWarnings( p, DFS_F_M_SHOWALL, NULL);
      }
      dfstLogicalLimits( DFSTORE, orgbase, orgbase + orgsize -1);
   }
   TxFreeMem( text);
   VRETURN();
}                                               // end 'dfsVerboseShowPart'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Show compact partition-info table for specified partition
/*****************************************************************************/
void dfsCompactShowPart
(
   DFSPARTINFO        *p                        // IN    partition info
)
{
   TXTM                tbuf;                    // temporary buffer
   char               *text = NULL;             // accumulating text buffer

   ENTER();

   text = TxAlloc( 1, TXMAXLN);
   if ((p != NULL) && (text != NULL))
   {
      TRACES(("p*:%p\n"));
      TxPrint("DFS partition  Id : %s%02.2d%s    on   Phdisk number:%s%d%s\n", CNC,  p->id, CNN, CBC, p->disknr,  CNN);
      strcpy(   text, "");
      dfstrX10( text, " Part-table   PSN : ",        p->partPsn, CBM, " = ");
      sprintf(  tbuf, "%sBR(%d),",                  (p->tablenr) ? "E" : "M", p->tablenr);
      strcat(   text, tbuf);
      TxPrint(       "%s entry:%s%d%s\n", text, CBY, p->partnr,  CNN);
      sprintf( text, " System Indicator : 0x%2.2x       = ", p->partent.PartitionType);
      strcat(  text, p->descr);
      sprintf( tbuf, " %s%s%s:%s%s%s ",  CNC, (p->primary) ? "Pri" : "Log", CNN, CBM,  p->fsform, CNN);
      strcat(   text, tbuf);
      TxPrint("%s%s%s  %s%s%s\n",  text, CBG,  p->plabel, CBC,  p->blabel, CNN);
      dfsSz64( " Partition   size : ",   p->sectors, "\n");
      dfsGeoDispTransPsn( "First part sector :", p->geoHeads, p->geoSecs, p->basePsn);
      dfsGeoDispTransPsn( "Last  part sector :", p->geoHeads, p->geoSecs, p->lastPsn);
      TxPrint("\n");                         // make prev info stand out
   }
   TxFreeMem( text);
   VRETURN();
}                                               // end 'dfsCompactShowPart'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get one-line compact partition description in a string, 50 chars
/*****************************************************************************/
void dfsGetPartDesc
(
   DFSPARTINFO        *p,                       // IN    partition info
   char               *descr                    // OUT   description, 50 chars
)
{
   ENTER();
   if (descr != NULL)
   {
      if (p != NULL)
      {
         sprintf(   descr, "Part %2.2hu: %-7.7s %-8.8s %-11.11s ", p->id, p->fsform, p->creatr, p->plabel);
         dfstrSXiB( descr, "", p->sectors, "");
      }
      else
      {
         strcpy( descr, "Not an existing partition (yet/anymore)");
      }
   }
   VRETURN();
}                                               // end 'dfsGetPartDesc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Show errors/warnings for specified partition
/*****************************************************************************/
void dfsShowPartWarnings
(
   DFSPARTINFO        *p,                       // IN    partition/EBR info
   ULONG               mask,                    // IN    warning filter mask
   ULONG              *minors                   // INOUT minor count, or NULL
)
{
   TXTS                idpd;                    // identification id/pd
   TXTT                part;                    // partition id string
   TXTT                text;                    // boundary text
   ULONG               warnings;
   ULONG               warning2;                // 2nd set
   ULONG               mask2 = 0;               // derived mask for 2nd set
   static BOOL         lvmcrcseq;               // LVM CRC/SEQ fixtip given
   static BOOL         lvmquery;                // LVM query explanation given
   static BOOL         causes;                  // explained possibe causes
   static BOOL         active;                  // explained ACTIVE problems
   static BOOL         multiples;               // explained multiple driveletters
   static BOOL         lindevice;
   static BOOL         extInt13WarningGiven;
   static BOOL         fixpbrWarningGiven;
   static BOOL         gptWarningGiven;
   static BOOL         ldmWarningGiven;
   static BOOL         alignextWarningGiven;

   if (p == NULL)                               // signal start of sequence
   {
      lvmcrcseq            = FALSE;
      lvmquery             = FALSE;
      causes               = FALSE;
      multiples            = FALSE;
      active               = FALSE;
      lindevice            = FALSE;
      extInt13WarningGiven = FALSE;
      fixpbrWarningGiven   = FALSE;
      gptWarningGiven      = FALSE;
      ldmWarningGiven      = FALSE;
      alignextWarningGiven = FALSE;
      if (minors)
      {
         *minors = 0;
      }
   }
   else                                         // show warnings for this one
   {
      warnings = p->flags;                      // 1st set of warnings
      warning2 = p->flag2;                      // 2nd set of warnings
      TRACES(("Pre  mask PID:%u  mask1:%8.8x  warnings:%8.8x  warning2:%8.8x\n", p->id, mask, warnings, warning2));
      if (dfsa->warnLevel <= 1)                 // if not pedantic, mask
      {
         if ((minors) && (mask == DFS_F_M_MAJOR)) // count suppressed minor warnings
         {
            if (dfsa->warnLevel == 0)           // first mask off TRIVIAL warnings
            {
               warnings &= ~DFS_F_M_TRIVIAL;    // leaving only non-trivial minor warnings
               warning2 &= ~DFS_F2M_TRIVIAL;
            }
            *minors += TxUlBitCount( warnings & DFS_F_M_MINOR);
            *minors += TxUlBitCount((warning2 & DFS_F2M_MINOR) & ~DFS_F_I13XNEEDED);
         }
         warnings &= mask;
         switch (mask)
         {
            case DFS_F_M_MAJOR: mask2 = DFS_F2M_MAJOR;   break;
            case DFS_F_M_MINOR: mask2 = DFS_F2M_MINOR;   break;
            default:            mask2 = DFS_F_M_SHOWALL; break;
         }
         warning2 &= mask2;
      }
      TRACES(("Post mask PID:%u  mask2:%8.8x  warnings:%8.8x  warning2:%8.8x\n", p->id, mask2, warnings, warning2));
      if (p->id != 0)
      {
         sprintf( idpd, (p->id < 100) ? "%s%2.2d=%2.2s" : "%s%3.3d=%1.1s",
                  (p->pitype == DFS_PI_EBR_CHAIN) ? "EBR" : "Pid", p->id, p->drive);
      }
      else                                      // no id, list PD + entry nr
      {
         sprintf( idpd, "PD %d @:%d", p->disknr, p->partnr);
      }

      //==================================================== ERRORS
      sprintf( part, " %s %sERROR%s   : ", idpd, CBR, CNN);
      if ((warnings & DFS_F_MULTIP_ACT) ||
          (warning2 & DFS_F_NONSTDFLAG)  )
      {
         if (warnings & DFS_F_MULTIP_ACT)
         {
            TxPrint( "%sMultiple partitions are marked ACTIVE on this disk\n", part);
         }
         if (warning2 & DFS_F_NONSTDFLAG)
         {
            TxPrint( "%sPartition table has FLAG byte other than 0x00/0x80\n", part);
         }
         if (active == FALSE)                   // explain only once
         {
            if (p->partPsn == 0)                // is it the first, in MBR ?
            {
               TxPrint("%20.20sThis might cause MBR bootcode to fail booting the system!\n", "");
            }
            TxPrint("%20.20sTo fix, use the 'cleanup' command or corresponding menu:\n"
                    "%20.20sMode=FDISK -> Cleanup partition tables -> ... disk(s)\n", "", "");
            active = TRUE;
         }
      }
      if (warning2 & DFS_F_EXTC_BLINK)
      {
         TxPrint( "%sNext extented container at lower PSN, BACKWARD link!\n", part);
         TxPrint( "%20.20sData loss possible when updating partition tables.\n", "");
      }
      if (warning2 & DFS_F_EXTC_NLINK)
      {
         TxPrint( "%sEx\1tended is linked from a higher PSN, BACKWARD link!\n", part);
      }
      if (warning2 & DFS_F_EXTC_LLOOP)
      {
         TxPrint( "%sNext extented container already linked, illegal LOOP!\n", part);
         TxPrint( "%20.20sData loss possible when updating partition tables.\n", "");
      }
      if (warning2 & DFS_F_EXTC_NLOOP)
      {
         TxPrint( "%sExtented is linked in a circular way, illegal LOOP!\n", part);
      }
      if (warnings & DFS_F_OVERLAPREV)
      {
         TxPrint( "%sStart of partition OVERLAPS previous one (data corruption)!\n", part);
      }

      if ((p->partent.PartitionType != DFS_P_DFSEETYPE) && // ignore warnings on temporary type
          (p->partent.PartitionType != DFS_P_DFSNOWARN)  ) // like DFSEE and  TMP/BACKUP ones
      {
         //-==================================================== MINOR warnings
         sprintf( part, " %s warning : ", idpd);

         if (warnings & DFS_F_CHSLBA_ERR)       // JvW 20140627: not an ERROR anymore
         {                                      // JvW 20180905: made a MINOR warning
            if      (warnings & DFS_F_CHSLBA_S13)
            {
               TxPrint("%sMismatch Geometry + CHS-Begin with LBA offset\n", part);
            }
            if (warnings & DFS_F_CHSLBA_E13)
            {
               TxPrint("%sMismatch Geometry + CHS-End with LBA offset+size\n", part);
            }
            sprintf( part, " %s warning : ", idpd);
            if (warnings & DFS_F_CHSDUM_S13)
            {
               TxPrint("%sMismatch Geometry + CHS-Begin int13-dummy value\n", part);
            }
            if (warnings & DFS_F_CHSDUM_E13)
            {
               TxPrint("%sMismatch Geometry + CHS-End   int13-dummy value\n", part);
            }
            if (causes == FALSE)                // explain only once
            {
               TxPrint(" possible causes: - change in BIOS level or BIOS settings for LBA\n");
               TxPrint("                  - changed disk adapter, firmware or device driver)\n");
               TxPrint("                  - bad CHS (dummy) values by buggy partitioning tool\n");
               causes = TRUE;
            }
         }
         if (warnings & DFS_F_LOGICALACT)
         {
            TxPrint( "%sLogical partition is marked as startable! (active)\n", part);
         }
         if (warnings & DFS_F_ACT_NOT1ST)
         {
            TxPrint( "%sActive (startable), but not on 1st disk\n", part);
            TxPrint("%20.20sRequires modern BIOS or smart MBR code to boot\n", "");
         }
         if (warnings & DFS_F_BIGEXTENDF)
         {
            TxPrint( "%sA windows-compatible extended-partition type 0x0f is used,\n"
                     "%20.20sOld operating systems might not see all logical partitions.\n",
                        part, "");
         }
         if (warning2 & DFS_F_GUIDDUPLIC)
         {
            TxPrint( "%sThe GPT partition GUID for this partition is NOT unique!\n"
                     "%20.20sYou can use the Partition Table Editor to fix this.\n",
                      part, "");
         }
         if (warning2 & DFS_F_LINUXHDA63)
         {
            TxPrint( "%sPartition device-nr too large, might be invissible to Linux!\n", part);
            if (lindevice == FALSE)
            {
               TxPrint("%20.20sIt may be vissible using device-mapper or similar ...\n", "");
               lindevice = TRUE;
            }
         }
         if (warning2 & DFS_F_LVM_NO_SIG)
         {
            TxPrint(      "%sPartition is of LVM type 0x35, has LVM-information  (DLAT),\n"
                     "%20.20sbut lacks the LVM-signature sector with LVM-features (BBR).\n"
                     "%20.20sEither delete/recreate it with DFSee (CR in recovery mode),\n"
                     "%20.20sor when it is NOT a multi-partition spanned volume, change\n"
                     "%20.20sthe type to 0x07 (bootable JFS) not needing any BBR info.\n"
                     "%20.20s(Requires non-IBM JFS.IFS from ArcaOS or eComStation 2.x)\n",
                      part, "", "", "", "", "");
         }
         if (warning2 & DFS_F_SIG_NO_LVM)
         {
            TxPrint(      "%sPartition is NOT of LVM type 0x35, it is a COMPATIBILTY type,\n"
                     "%20.20sbut DOES have an LVM-signature sector with LVM-features (BBR).\n"
                     "%20.20sThis may lead to 'Disk corrupt' messages from the LVM program!\n"
                     "%20.20sFix it using the command 'lvm -C %hu' to clear the BBR sector.\n",
                      part, "", "", "", p->id);
         }
         if (warning2 & DFS_F_BMNOTW2KBM)
         {
            TxPrint( "%sBMGR NOT (fully) protected against Windows-2000 chkdsk\n"
                     "%20.20sUse the 'BMFIX %hu' command to protect it\n", part, "", p->disknr);
         }
         if (warning2 & DFS_F_BMBADCHS)
         {
            TxPrint( "%sBMGR CHS values for boot or data sector location incorrect\n"
                     "%20.20sUse the 'BMFIX %hu' command to update them\n", part, "", p->disknr);
         }
         if (warning2 & DFS_F_LVMBADSIZE)
         {
            TxPrint( "%sLVM DLAT size or position mismatch with partition tables\n"
                     "%20.20sUse the 'LVM %hu' command to synchronize them\n", part, "", p->id);
         }
         if (warning2 & DFS_F_LVMBOOTBAD)
         {
            TxPrint( "%sLVM boot-disk-id differs from first partition\n", part);
         }
         if (warning2 & DFS_F_HIDDENSBAD)
         {
            TxPrint( "%sBootsector HiddenSectors field does NOT match part-table!\n", part);
            if (fixpbrWarningGiven == FALSE)
            {
               TxPrint( "%s", dfsFixPbrTip);
               fixpbrWarningGiven = TRUE;
            }
         }
         if (warning2 & DFS_F_BOOTGEOBAD)
         {
            TxPrint( "%sBootsector GEO Heads/Sectors fields do NOT match geometry!\n", part);
            if (fixpbrWarningGiven == FALSE)
            {
               TxPrint( "%s", dfsFixPbrTip);
               fixpbrWarningGiven = TRUE;
            }
         }
         strcpy( text,  " (cyl boundary)\n");
         if (warnings & DFS_F_PRIM_NOT_0)
         {
            TxPrint( "%sPrimary does not start on   Head-0/Sector-1%s",    part, text);
         }
         if (warnings & DFS_F_PLOG_NOT_1)
         {
            TxPrint( "%sPartition does not start on Head-1/Sector-1%s",    part, text);
         }
         if (warnings & DFS_F_PART_NOT_C)
         {
            TxPrint( "%sPartition doesn't end on last Head & Sector%s",    part, text);
         }
         if (warnings & DFS_F_EXTC_NOT_C)
         {
            TxPrint( "%sExtended does not end on last Head & Sector%s",    part, text);
         }

         //-==================================================== MAJOR warnings
         sprintf( part, " %s %sWARNING%s : ", idpd, CBY, CNN);
         if (warning2 & DFS_F_OVERLAPMBR)
         {
            TxPrint( "%sStart partition overlaps MBR/GPT tables! Possible ISOHybrid\n", part);
         }
         if (warnings & DFS_F_BOOTRECBAD)
         {
            if (p->partent.PartitionType == DFS_P_WARP_LVM)
            {
               TxPrint( "%sNot formatted or LVM/JFS multiple partition\n", part);
            }
            else
            {
               TxPrint( "%sNot formatted or invalid boot record\n", part);
            }
         }
         if (warnings & DFS_F_EXTEND_ACT)
         {
            if (p->tablenr == 0)                // MBR container (extended)
            {
               TxPrint( "%sExtended container is marked as startable! (active)\n"
                        "%20.20sCould be needed for some bootloaders (GRUB2), but\n"
                        "%20.20smay cause boot-problems with some older BIOS'es.\n", part, "", "");
            }
            else                                // EBR container (logical)
            {
               TxPrint( "%sLogical container is marked as startable! (active)\n", part);
            }
         }
         if (warning2 & DFS_F_DLATINPART)
         {
            TxPrint( "%sStart partition overlaps DLAT, incompatible with OS/2 LVM!\n", part);
            if ((p->primary == FALSE) && (alignextWarningGiven == FALSE))
            {
               TxPrint( "%s", dfsAlignExtTip);  // valid for logicals only
               alignextWarningGiven = TRUE;
            }
         }
         if (warning2 & DFS_F_NOBOOTLETT)
         {
            TxPrint( "%sBootable JFS or HPFS but no driveletter assigned in LVM!\n", part);
         }
         if (warning2 & DFS_F_BADBPBLETT)
         {
            TxPrint( "%sBootable JFS or HPFS driveletter mismatch BPB versus LVM!\n", part);
         }
         if (warning2 & DFS_F_PRIM_INEXT)
         {
            TxPrint( "%sExtended partition contains a PRIMARY partition!\n", part);
         }
         if (warning2 & DFS_F_F32INF16TP)
         {
            TxPrint( "%sFAT32 FS found in FAT16 systemtype (06/0e) expected 0b/0c!\n", part);
         }
         if (warning2 & DFS_F_F16INF32TP)
         {
            TxPrint( "%sFAT16 FS found in FAT32 systemtype (0b/0c) expected 06/0e!\n", part);
         }
         if (warning2 & DFS_F_BRDUPLICAT)
         {
            TxPrint( "%sPartition bootsector is a match to MULTIPLE driveletters!", part);
            if (multiples == FALSE)             // explain only once
            {
               TxPrint( " This\n"
               #if defined (WIN32)
                  "%20.20smay happen on changing a driveletter on an in-use partition, or it\n"
               #endif
                  "%20.20scould be caused by imaging or cloning. You can make it unique\n"
                  "%20.20sby opening the partition and updating the volume serial-number:\n"
                  "%20.20sMode=xxx ->Boot area fixes/updates ->Set Volume Serial Number\n"
                  "%20.20swhere 'xxx' is the filesystem on the partition (FAT, HPFS etc).",
               #if defined (WIN32)
                   "",
               #endif
                   "", "", "", "");
               multiples = TRUE;
            }
            TxPrint( "\n");
         }
         if (warning2 & DFS_F_WASTECONTN)
         {
            TxPrint( "%sWasted freespace inside extended container, not shown by DFSee!\n", part);
            if (alignextWarningGiven == FALSE)
            {
               TxPrint( "%s", dfsAlignExtTip);
               alignextWarningGiven = TRUE;
            }
         }
         if (warning2 & DFS_F_EMPTYCONTN)
         {
            TxPrint( "%sLeading freespace contains an EMPTY extended container!\n"
                     "%20.20sUse the CLEANUP command to remove it from the EBR chain.\n", part, "");
         }
         if (warning2 & DFS_F_LVMBADCRCI)
         {
            TxPrint( "%sCRC value for the LVM-info sector (DLAT) is BAD!\n", part);
            if (lvmcrcseq == FALSE)
            {
               TxPrint( "%s", dfsLvmCrcFixTip);
               lvmcrcseq = TRUE;
            }
         }
         if (warning2 & DFS_F_LVMBADCRCS)
         {
            TxPrint( "%sCRC value for the LVM-signature sector is BAD!\n", part);
            if (lvmcrcseq == FALSE)
            {
               TxPrint( "%s", dfsLvmCrcFixTip);
               lvmcrcseq = TRUE;
            }
         }
         if (warning2 & DFS_F_LVMBADSEQN)
         {
            TxPrint( "%sDrivelink sequence-nr in LVM BBR area is ZERO!\n", part);
            if (lvmcrcseq == FALSE)
            {
               TxPrint( "%s", dfsLvmCrcFixTip);
               lvmcrcseq = TRUE;
            }
         }
         if (warning2 & DFS_F_LVMNAME_20)
         {
            TxPrint( "%sPartition or Volume-name for LVM is too long!\n", part);
            if (lvmquery == FALSE)
            {
               TxPrint( "%s%s", dfsLvm20Warning, dfsLvm20FixTip);
               lvmquery = TRUE;
            }
         }
         if (warning2 & DFS_F_LVMNAMEBAD)
         {
            TxPrint( "%sLVM disk-name used differs from first partition\n", part);
         }
         if (warning2 & DFS_F_LVMTHISBAD)
         {
            TxPrint( "%sLVM disk-id used differs from first partition\n", part);
         }
         if (warnings & DFS_F_EFIGPTDISK)
         {
            TxPrint( "%sThe disk appears to be partitioned using the GPT Partition style.\n", part);
            if (gptWarningGiven == FALSE)       // explain only once
            {
               TxPrint( "%20.20sThis is part of the Intel Itanium EFI specification as used\n"
                        "%20.20s(optional) with some Windows versions and (Intel) MAC systems.\n"
                        "%20.20sDo NOT use MBR-style only partitioning tools like OS/2 LVM or\n"
                        "%20.20sFDISK. In DFSee use 'set gpt on' for automatic GPT handling.\n",
                        "", "", "", "");
               gptWarningGiven  =  TRUE;
            }
         }
         if (warnings & DFS_F_WINLDMDISK)
         {
            TxPrint( "%sThe disk appears to be partitioned as a Windows 'Dynamic Disk'.\n", part);
            if (ldmWarningGiven == FALSE)       // explain only once
            {
               TxPrint( "%20.20sThis relies on a proprietary LDM database in unpartitioned\n"
                        "%20.20sspace which is NOT accounted for in the MBR partition table.\n"
                        "%20.20sDo NOT use MBR style partition-tools like DFSee to modify,\n"
                        "%20.20sunless you want to clear it completely (newmbr -c, wipe).\n",
                         "", "", "", "");
               ldmWarningGiven  =  TRUE;
            }
         }
         if (warnings & DFS_F_ACT_HIDDEN)
         {
            TxPrint( "%sActive (startable), but also hidden\n",              part);
         }
         if (warnings & DFS_F_EXTC_NOT_0)
         {
            TxPrint( "%sExtended does not start on  Head-0/Sector-1%s",      part, text);
         }
         strcpy( text,  " ext-container\n");
         if (warnings & DFS_F_LOG_BEFORE)
         {
            TxPrint( "%sLogical partition starts before EBR%s",              part, text);
         }
         if (warnings & DFS_F_LOG_BEYOND)
         {
            TxPrint( "%sLogical partition extends beyond EBR%s",             part, text);
         }
         if (warnings & DFS_F_EXT_BEFORE)
         {
            TxPrint( "%sExtended partition starts before MBR%s",             part, text);
         }
         if (warnings & DFS_F_EXT_BEYOND)
         {
            TxPrint( "%sExtended partition extends beyond MBR%s",            part, text);
         }
         strcpy( text,  " end of disk");
         if (warnings & DFS_F_PART_EDISK)
         {
            TxPrint( "%sPartition extends beyond%s (for S-Geo)\n",           part, text);
         }
         if (warnings & DFS_F_EXTC_EDISK)
         {
            TxPrint( "%sExtended partition extends beyond%s (for S-Geo)\n",  part, text);
         }
         #if defined (DOS32)
         {
            if ((warnings   & DFS_F_EXTC_EDISK) ||
                (warnings   & DFS_F_PART_EDISK)  )
            {
               if ((p->geoCyls < 1024) && (extInt13WarningGiven == FALSE ))
               {
                  TxPrint("%20.20sIt is likely that DFSee can not see "
                          "the whole disk\n%20.20sbecause no extended "
                          "Int13 support is available\n", "", "");
               }
               TxPrint("%20.20sIt is possible that the BIOS reports "
                       "1 or 2 cylinders\n%20.20stoo few to allow for "
                       "a test/calibration cylinder.\n", "", "");
               TxPrint("%20.20sThis could be a (SCSI) BIOS or device "
                       "driver problem,\n%20.20sor you are running "
                       "the program from a virtual DosBox\n", "", "");
               extInt13WarningGiven = TRUE;     // no more of same WARNING ...
            }
         }
         #endif
         strcpy( text,  "IBM BootManager");
         if (warnings & DFS_F_BMGR_NOT12)
         {
            TxPrint( "%sThe %s found is NOT on either disk 1 or 2,\n", part, text);
            TxPrint( "%20.20sto use it you either need it on disk 1 or 2, or your\n", "");
            TxPrint( "%20.20sBIOS must support booting from that other disk\n", "");
         }
         if (warnings & DFS_F_BMGR_OTHER)
         {
            TxPrint( "%s%s found on disk other than 1 or 2\n", part, text);
         }
         if (warnings & DFS_F_BMGR_INACT)
         {
            TxPrint( "%s%s on disk 1 is not startable (active)\n", part, text);
         }
         if (warnings & DFS_F_NO_ACTIVE1)
         {
            TxPrint( "%sNO partition is marked active on the first disk\n", part);
            TxPrint( "%20.20sBooting requires modern BIOS or smart MBR code!\n", "");
         }
         if (warnings & DFS_F_NO_ACTIVES)
         {
            TxPrint( "%sNO partition is marked active on any disk\n", part);
            TxPrint( "%20.20sBooting with a standard BIOS and MBR will fail!\n", "");
         }

         //- check I13X dependancy for bootable partitions (NOT using flags!)

         if ((p->flag2 & DFS_F_I13XNEEDED) &&   // requires 'I13X' to boot
             ((strlen(p->blabel) > 0) ||        // bootable by BMGR
              (p->primary && (p->partent.Status)))) // or active primary
         {
            ULONG         c,h,s;

            dfsGeoPsn2Chs( p->basePsn, p->geoHeads, p->geoSecs, &c, &h, &s);

            if (c > 1023)                       // requires I13X ?
            {
               if ((dfsa->bmgrDisk != 0) &&     // if Bootmanager is present
                   (strlen(p->blabel) > 0))     // and this one on BMGR menu
               {
                  if (dfsa->bmgrI13X == DFSI13X_NOTTHERE) // not supported by BMGR
                  {
                     TxPrint( "%sPartition might not show up on the BMGR menu!\n", part);
                     TxPrint( "%20.20sCylinder > 1023 but no known I13X capable BMGR present!\n"
                              "%20.20sReinstall BMGR on disk %hu using LVM, DFSee or recent FDISK\n",
                              "", "", dfsa->bmgrDisk);
                  }
                  if (dfsa->mbr1I13X == FALSE)  // not supported by the MBR
                  {
                     TxPrint( "%sPartition may not boot correctly without 'I13X' support!\n", part);
                     TxPrint( "%20.20sCylinder > 1023 but no known I13X capable MBR-code!\n"
                              "%20.20sRefresh MBR-code on disk %hu using LVM or DFSee 'newmbr'\n"
                              "%20.20sor update the bootcode in the partition using 'fixboot'\n",
                              "", "", dfsa->bmgrDisk, "");
                  }
               }
               if ((dfsa->mbr1I13X == FALSE) && // not supported by MBR code
                    p->primary               && // and a primary partition,
                    p->partent.Status        && // ACTIVE, possible OS/2 boot
                   (p->partent.PartitionType == DFS_P_INST_FS))
               {
                  TxPrint( "%sCylinder > 1023 but MBR does not have I13X support!\n", part);
                  TxPrint( "%20.20sRefresh MBR-code on disk %hu using LVM or DFSee 'newmbr'\n"
                           "%20.20sor update the bootcode in the partition using 'fixboot'\n",
                           "", dfsa->bmgrDisk, "");
               }
            }
         }
      }
      else
      {
         TRACES(("Warnings ignored for selected partition-types\n"));
      }
   }
}                                               // end 'dfsShowPartWarnings'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Show errors/warnings for specified disk
/*****************************************************************************/
void dfsShowDiskWarnings
(
   DFSDISKINFO        *d,                       // IN    disk info
   ULONG               mask                     // IN    warning filter mask
)
{
   TXTS                idpd;                    // identification id/pd
   TXTT                disk;                    // disk id string
   ULONG               warnings;


   warnings = d->flags;                         // 1st set of warnings
   if (dfsa->warnLevel <= 1)                    // if not pedantic, mask
   {
      warnings &= (mask | 0x03);                // preserve PTE entry nr
   }

   sprintf( idpd, "Disk %2hu ", d->disknr);

   //- sprintf( disk, " %s ERROR   : ", idpd);
   //- sprintf( disk, " %s warning : ", idpd);

   sprintf( disk, " %s WARNING : ", idpd);

   if ((d->disknr == 1) && ((dfsa->bmgrI13X == DFSI13X_REQUIRED) && !dfsa->mbr1I13X))
   {
      TxPrint( "%sIBM BMGR requires 'I13X' but MBR does not seem to set that!\n", disk);
      TxPrint( "%20.20sHPFS partitions beyond cyl 1023 may not boot/be seen!\n", "");
      TxPrint( "%20.20sUse the 'NEWMBR 1' command to refresh the MBR code\n", "");
   }
   if (warnings & DFS_F_LVMDISK_20)
   {
      TxPrint( "%sDisk-name used with LVM is too long!\n", disk);
      TxPrint( "%s", dfsLvm20Warning);
      TxPrint( "%20.20sUse 'lvm -P- -n' to automatically shorten them\n",
               "", d->disknr);
   }
   if (warnings & DFS_F_LVMCYL_64K)
   {
      TxPrint( "%sLVM geometry cylinder count exceeds OS/2 kernel limit of 65535\n", disk);
      TxPrint( "%20.20sThe end of the disk will NOT be accessible from OS/2 or eCS!\n", "");
      TxPrint( "%20.20sWhen possible, adjust Head/Sectors values for larger cylinders.\n", "");
   }
   if (warnings & DFS_F_PTYPE_0x00)
   {
      TxPrint( "%sMBR partition table has systemtype DELETED (0x00) for entry: %hu!\n", disk, (warnings & 0x03));
      #if defined (DEV32)
      TxPrint( "%20.20sMay cause problems with the OS/2 LVM.EXE program, and deratives.\n", "");
      #endif
   }
   if (warnings & DFS_F_INVALIDPTE)
   {
      TxPrint( "%sMBR sector has invalid partition table entry nr: %hu\n",
                disk, (warnings >> 2) & 0x03);
      TxPrint( "%20.20sThe size in sectors is ZERO\n", "");
   }
   if (warnings & DFS_F_MULTI_EXT)
   {
      TxPrint( "%sMBR contains multiple EXTENDED containers, entry:%hu\n",
                disk, (warnings >> 4) & 0x30);
   }
   if (warnings & DFS_F_MULTI_GPT)
   {
      TxPrint( "%sMBR contains multiple GPT guards (0xEE), entry:%hu\n",
                disk, (warnings >> 4) & 0x30);
   }
   if (warnings & DFS_F_GPT_U_512)
   {
      TxPrint( "%sUnexpected GPT header at byte offset 512, for sectorsize 4K.\n", disk);
      TxPrint( "%20.20sYou may need to configure the disk/controller for 512\n"
               "%20.20sbytes per sector (often called 'compatibility mode').\n", "", "");
   }
   if (warnings & DFS_F_GPT_U_4096)
   {
      TxPrint( "%sUnexpected GPT header at byte offset 4K, for sectorsize 512.\n", disk);
      TxPrint( "%20.20sYou may need to configure the disk/controller for 4096\n"
               "%20.20sbytes per sector (disabling the 'compatibility mode').\n", "", "");
   }
   if (warnings & DFS_F_GPTINVALID)
   {
      TxPrint( "%sGPT guard partition present, but GPT Header/Array is invalid.\n", disk);
      TxPrint( "%20.20sYou may delete the guard partition and recreate to recover.\n", "");
   }
   if (warnings & DFS_F_GPT_BADCRC)
   {
      TxPrint( "%sGPT header holds incorrect CRC values for header or Array.\n", disk);
      TxPrint( "%20.20sYou can rewrite the GPT from known info to fix this ('gpt -fix')\n", "");
   }
   if (warnings & DFS_F_GPT_BADPRI)
   {
      TxPrint( "%sGPT primary header info at disk start is missing or invalid.\n", disk);
      TxPrint( "%20.20sYou can rewrite the GPT from known info to fix this ('gpt -fix')\n", "");
   }
   if (warnings & DFS_F_GPT_BADALT)
   {
      TxPrint( "%sGPT alternate header info at disk end is missing or invalid.\n", disk);
      TxPrint( "%20.20sYou can rewrite the GPT from known info to fix this ('gpt -fix')\n", "");
   }
   if (warnings & DFS_F_GPTSMGUARD)
   {
      TxPrint( "%sGPT guard partition (0xEE) is smaller than the whole disk.\n", disk);
      TxPrint( "%20.20sThis is non-standard, when not using an MBR/GPT 'hybrid'.\n"
               "%20.20sYou may delete it, and recreate it properly to recover.\n", "", "");
   }
   if (warnings & DFS_F_GPTMBRPART)
   {
      TxPrint( "%sGPT partition table is present together with the shown MBR partitions!\n", disk);
      TxPrint( "%20.20sUse the 'gpt' command to show the GPT table contents in detail", "");
      if (dfsa->gptAuto == FALSE)
      {
         TxPrint( ", or\n%20.20senable Automatic GPT partition handling with 'set gpt on'", "");
      }
      TxPrint( ".\n");
   }
   if (warnings & DFS_F_GPT_NOGUARD)
   {
      TxPrint( "%sGPT tables found and shown, but no GPT guard partition in the MBR!\n", disk);
      TxPrint( "%20.20sEither remove the GPT tables ('gpt2mbr'), to keep the MBR style,\n"
               "%20.20sor remove any MBR partitions ('mbr2gpt'), to make it valid GPT.\n"
               "%20.20s(Expert: Mode=Fdisk-> Partition table operations-> Convert ...)\n", "", "", "");
   }
   if (warnings & DFS_F_APMMBRPART)
   {
      TxPrint( "%sApple Partition Map present together with the shown MBR partitions (hybrid)!\n", disk);
      TxPrint( "%20.20sUse the command 'disk %hu' to show the APM table contents in detail.\n", "", d->disknr);
      TxPrint( "%20.20sDelete MBR partitions and remove '55 AA' sig to make it a APM only.\n",  "");
   }
   if (warnings & DFS_F_FSYSTEMONLY)
   {
      TxPrint( "%sNo partitions detected, FileSystem Only (large floppy format)\n", disk);
      #if defined (DEV32)
         if (strcasecmp(d->fsform, "FAT32") == 0)
         {
            TxPrint( "%20.20sFAT32 large floppy will NOT work on OS/2 or eCS!\n", "");
         }
      #endif
   }
   if (warnings & DFS_F_CRYPTO_DISK)
   {
      TxPrint( "%sStart sectors suggest this is a fully encrypted disk (LUKS)\n", disk);
   }
   if ((warnings & DFS_F_OS2_64K_CYL) && (d->gptHeader == NULL)) // no warning on GPT
   {
      #if !defined (DEV32)
         if (dfsa->OS2FsPresent)                // always show on OS/2 itself
      #endif
      {
         ULONG     os2Limit = dfs16bitCylLimitGiB( d->geoHeads, d->geoSecs);

         TxPrint( "%sLogical geometry (see L-Geo) for disk exceeds 65535 cylinders,\n", disk);
         TxPrint( "%20.20sthe OS/2 or eCS kernel can only use up to %u GiB reliably!\n"
                  "%20.20sUse 255 heads and 127 or 255 sectors/track and repartition to\n"
                  "%20.20suse maximum disk size, or limit OS/2 usage to first %u GiB.\n",
                  "", os2Limit, "", "", os2Limit);
      }
   }
   if (warnings & DFS_F_MBR_NO_SIG)
   {
      TxPrint( "%sMBR contains valid table entries but no 0x55AA signature!\n", disk);
   }
   if (warnings & DFS_F_EBR_NO_SIG)
   {
      TxPrint( "%sAt least one EBR contains valid table but no 0x55AA signature!\n", disk);
   }
}                                               // end 'dfsShowDiskWarnings'
/*---------------------------------------------------------------------------*/

