//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// NTFS utility functions
//
// Author: J. van Wijk
//
// JvW  22-11-1997 Initial version, split off from DFSANTFS
// JvW  04-04-2017 Updated LSN to 64bit, MFT-numbers to 64bit
//
#ifndef    DFSUNTFS
   #define DFSUNTFS


#define MFT_NULL      0xffffffffffffffffULL

// Find MFT-record for specified path, starting at known root-directory MFT
ULONG dfsNtfsFindPath
(
   ULN64               loud,                    // IN    Show progress
   ULN64               d2,                      // IN    dummy
   char               *path,                    // IN    path specification
   void               *vp                       // OUT   found dir/file MFT
);

// Read initial DIR-entries (attribute 90) from directory MFT LSN
BOOL dfsNtfsMft2DirEntries                      // OUT   DIR entries resolved
(
   ULONG               mft,                     // IN    directory MFT number
   A_IDXROOT         **idx                      // OUT   INDX Root structure
);

// Find DIR-entry for specified name, start at DIRBLOCK LSN, walk the Btree
BOOL dfsNtfsDirIndx2Entry                       // OUT   entry found
(
   ULN64               dirMft,                  // IN    MFTnr for directory
   ULONG               loud,                    // IN    Show progress
   char               *entry,                   // IN    entry specification
   S_DIRENTRY         *dindex,                  // IN    start Index ROOT area
   ULN64              *mft,                     // OUT   found dir/file MFT nr
   ULN64              *size                     // OUT   file/dir size in bytes
);                                              //       from the directory-entry

// Replace sn-list by contents of a single Directory, storing MFT-rec Lsn's
ULONG dfsNtfsMakeBrowseList
(
   ULN64               mftLsn,                  // IN    MFT-rec sectornumber
   ULN64               info,                    // IN    unused
   char               *options,                 // IN    option string (~ = 8.3)
   void               *param                    // INOUT unused
);

// Get allocation information for file-DATA into integrated-SPACE structure
ULONG dfsNtfsGetAllocSpace
(
   ULN64               mftLsn,                  // IN    MFT-record Lsn
   ULN64               info,                    // IN    unused
   char               *str,                     // IN    unused
   void               *param                    // INOUT Integrated SPACE
);

// Write supplied data area into an MFT-record (resident data attribute)
ULONG dfsNtfsWriteMetaSpace
(
   ULN64               num1,                    // IN    unused
   ULN64               num2,                    // IN    unused
   char               *str,                     // IN    unused
   void               *param                    // IN    Integrated SPACE
);

// Add NTFS attributes description to a string
void dfstrNtfsAttrib
(
   char               *str,                     // INOUT string
   ULONG               FileAttributes           // IN    NTFS attributes
);

// Check and fixup an NTFS multi-sector structure using the fixup-table
ULONG dfsNtfsFixupStructure                     // RET   nr of invalid sector
(
   BYTE               *fs,                      // IN    ptr to first sector
   ULONG               sectors                  // IN    #sectors in structure
);

// Prepare a fixed-up multi-sector NTFS structure for next write (Un-Fixup :-)
ULONG dfsNtfsUnfixStructure                     // RET   nr of invalid sector
(
   BYTE               *fs                       // IN    ptr to first sector
);

// Display sequence of NTFS Directory entries, and add to LSN list
ULONG dfsNtfsDirEntries                         // RET   nr of entries listed
(
   ULN64               dirMft,                  // IN    MFT number directory
   S_DIRENTRY         *entries,                 // IN    Sequence of Dir entries
   BOOL                header                   // IN    present a header too
);

// Convert NTFS NTIME (64 bit) to DFS standard date/time string
char *dfsNtfsTime2str                           // RET   string value
(
   NTIME              *nt,                      // IN    ptr NT time value
   char               *dtime                    // INOUT ptr to string buffer
);

// Show NTFS Attribute type as value and descriptive string
ULONG dfsNtfsShowAttrType                       // RET   unrecognized type
(
   char               *lead,                    // IN    leading text
   ULONG               atype,                   // IN    attribute type
   char               *dcol,                    // IN    descr color (ANSI)
   char               *tail                     // IN    tail text
);

// Find MFT for Root-directory, starting at some LSN in the volume
// 20171114 JvW Updated to allow input of 'desired' parentMFT (for hard links)
BOOL dfsNtfsFindRootMft                         // OUT   root found
(
   ULN64               start,                   // IN    starting lsn
   BOOL                verbose,                 // IN    display mode
   BOOL                dos83,                   // IN    favor DOS8.3 shortname
   ULN64               desiredParent,           // IN    desired parent MFT, or 0
   char               *path,                    // OUT   combined path string
   ULN64              *lsn                      // OUT   root MFT-LSN or NULL
);

// Find parent MFT for the given (DIR or FILE) MFT, including path string
// 20171114 JvW Updated to allow input of 'desired' parentMFT (for hard links)
BOOL dfsNtfsMft2Parent                          // OUT   real parent found
(
   ULN64               fn,                      // IN    starting lsn
   BOOL                verbose,                 // IN    display mode
   BOOL                dos83,                   // IN    favor DOS8.3 shortname
   ULN64               parent,                  // IN    desired parent MFT, or 0
   char               *path,                    // OUT   combined path, TXMAXLN
   ULN64              *lsn                      // OUT   found parent MFT LSN
);

// DFS NTFS read and fixup an INDX record based on its sector number
ULONG dfsNtfsReadFixIndx                        // RET   BAD_STRUCTURE
(
   ULN64               lsn,                     // IN    INDX LSN
   S_DIRINDX         **dirrec                   // OUT   INDX fixed-up record
);

// DFS NTFS read and fixup an MFT record based on its sector number
ULONG dfsNtfsReadFixMft                         // RET   BAD_STRUCTURE
(
   ULN64               lsn,                     // IN    MFT LSN
   BYTE               *stype,                   // OUT   MFT sector-type
   S_MFTFILE         **mftrec                   // OUT   MFT fixed-up record
);

// DFS NTFS unfix and write an MFT record based on its sector number
ULONG dfsNtfsFixWriteMft                        // RET   BAD_STRUCTURE
(
   ULN64               lsn,                     // IN    MFT LSN
   S_MFTFILE          *mft                      // IN    MFT fixed-up record
);

// DFS NTFS verify allocation integrity for an MFT record (normal / deleted)
ULONG dfsNtfsMftLsnCheckAlloc                   // RET   check result
(
   ULN64               lsn,                     // IN    MFT LSN
   char               *options,                 // IN    result options
   BYTE               *stype,                   // OUT   sector-type
   ULN64              *totals,                  // INOUT total sectors
   ULN64              *invalids,                // INOUT invalid sectors
   ULN64              *datalsn                  // OUT   first data lsn
);

// DFS NTFS verify allocation integrity for given MFT record
ULONG dfsNtfsMftRecCheckAlloc                   // RET   check result
(
   S_MFTFILE          *mft,                     // IN    MFT record
   char               *options,                 // IN    result options
   ULN64              *totals,                  // INOUT total sectors
   ULN64              *invalids,                // INOUT invalid sectors
   ULN64              *datalsn                  // OUT   first data lsn
);

// Display file allocation and path info for LSN
ULONG dfsNtfsFileInfo                           // RET   LSN is valid Fnode
(
   ULN64               lsn,                     // IN    Fnode LSN
   ULN64               parent,                  // IN    desired parent MFT LSN or 0
   char               *select,                  // IN    Fnode select wildcard
   void               *lead                     // IN    leading text
);

// Get (long) filename, related parent-MFT and RealSize from given MFT-record
// 20171114 JvW Updated to allow input of 'desired' parentMFT (for hard links)
ULONG dfsNtfsMft2NameInfo                       // RET   MFT is valid MFT rec
(
   S_MFTFILE          *sd,                      // IN    base MFT rec, fixed-up
   BOOL                dos83,                   // IN    favor DOS8.3 shortname
   ULN64              *parent,                  // INOUT parent MFT, or NULL
   ULN64              *fileSize,                // OUT   RealSize bytes, or NULL
   char               *fname                    // OUT   ASCII filename string
);                                              //       maximum size: TXMAXLN

// DFS NTFS get S_SPACE allocation structure for attribute(s) in MFT (by lsn)
// Note: Just gets the allocation data, not data itself  (use MftLsnAttrData)
ULONG dfsNtfsMftLsn2Alloc                       // RET   BAD_STRUCTURE
(
   ULN64               lsn,                     // IN    MFT LSN
   ULONG               type,                    // IN    type of attribute MA_
   USHORT              atid,                    // IN    attribute-id or IA_ANY
   BYTE               *mftype,                  // OUT   MFT sectortype or NULL
   DFSISPACE          *isp                      // OUT   SPACE, resident or not
);

// Get attribute data for MFT attribute specified by type/id (resident/non-res)
ULONG dfsNtfsMftLsnAttrData                     // RET   error code
(
   ULN64               lsn,                     // IN    MFT LSN
   ULONG               type,                    // IN    type of attribute MA_
   USHORT              atid,                    // IN    attribute-id or IA_ANY
   char               *atname,                  // IN    attribute name or NULL
   ULONG               smax,                    // IN    maximum size wanted or 0
   ULONG              *size,                    // OUT   length of data returned
   void              **data                     // OUT   attribute data or NULL
);                                              //       use TxFreeMem to free

// Get attribute data for MFT attribute specified by type/id (resident/non-res)
ULONG dfsNtfsGetMftAttrData                     // RET   error code
(
   S_MFTFILE          *sd,                      // IN    base MFT rec, fixed-up
   ULONG               type,                    // IN    type of attribute MA_
   USHORT              atid,                    // IN    attribute-id or IA_ANY
   char               *atname,                  // IN    attribute name or NULL
   ULONG               smax,                    // IN    maximum size wanted or 0
   ULONG              *size,                    // OUT   length of data returned
   void              **data                     // OUT   attribute data or NULL
);                                              //       use TxFreeMem to free

// Get reference to attribute in MFT specified by type/id/name   (res/non-res)
// Does NOT support attributes located in secondary MFT records! (DFS_PENDING)
ULONG dfsNtfsGetMftAttrRef                      // RET   NO_ERROR, DFS_PENDING
(                                               //       or DFS_NOT_FOUND
   S_MFTFILE          *sd,                      // IN    base MFT rec, fixed-up
   ULONG               type,                    // IN    type of attribute MA_
   USHORT              atid,                    // IN    attribute id or IA_ANY
   char               *atname,                  // IN    attribute name or NULL
   S_MFTATTR         **ref                      // OUT   attribute ref or NULL
);

// Update an attribute in MFT                             (resident/non-res)
// Modify existing or create new, MFT-part only (non-resident is in runlist)
ULONG dfsNtfsUpdateMftAttr                      // RET   NO_ERROR, DFS_PENDING
(                                               //       or DFS_NOT_FOUND
   S_MFTFILE          *sd,                      // IN    base MFT rec, fixed-up
   S_MFTATTR          *attr                     // IN    attribute reference
);

// Get/Set NTFS Volume information in the $Volume special file
ULONG dfsNtfsGetSetVolumeInfo
(
   USHORT             *flags,                   // INOUT Volume flags (dirty)
   USHORT              mask,                    // IN    flags update mask
   char               *volname,                 // INOUT Volume name
   BOOL                setname,                 // IN    Set volume name (NI)
   double             *version,                 // OUT   NTFS volume version
   BOOL                setvers                  // IN    Set volume version
);                                              //       not implemented (NI)

// Get/Set NTFS LOG status information from the $LogFile special file
ULONG dfsNtfsGetSetLogStatus                    // RET   function result
(
   USHORT             *flags,                   // INOUT LOG flags (unmounted)
   USHORT              mask                     // IN    flags update mask
);                                              //       not implemented (NI)

// Find specific attribute in an MFT, return S_SPACE allocation data for a
// non-resident attribute or allocated copy of the data for a resident one
// Handles ATTRIBUTE lists including referenses to OTHER MFT records too
ULONG dfsNtfsSpAlloc                            // RET   error code
(
   S_MFTFILE          *sd,                      // IN    base MFT rec, fixed-up
   ULONG               type,                    // IN    type of attribute MA_
   USHORT              atid,                    // IN    attribute-id or IA_ANY
   char               *atname,                  // IN    attribute name or NULL
   DFSISPACE          *isp                      // OUT   SPACE, resident or not
);

// Find attribute in OTHER (external) MFT, return S_SPACE allocation data for
// a non-resident attribute or allocated copy of the data for a resident one
ULONG dfsNtfsSpExtern                           // RET   error code
(
   ULONG               mftnr,                   // IN    MFT record number
   ULONG               type,                    // IN    type of attribute MA_
   USHORT              atid,                    // IN    attribute-id or IA_ANY
   DFSISPACE          *isp                      // OUT   SPACE, resident or not
);

// Find specific attribute in an MFT, return S_SPACE allocation data for a
// non-resident attribute or allocated copy of the data for a resident one
// Handles simple resident or non-resident attributes in THIS MFT record
ULONG dfsNtfsSpNonRes                           // RET   error code
(
   S_MFTFILE          *sd,                      // IN    base MFT rec, fixed-up
   ULONG               type,                    // IN    type of attribute MA_
   USHORT              atid,                    // IN    attribute-id or IA_ANY
   char               *atname,                  // IN    attribute name or NULL
   DFSISPACE          *isp                      // OUT   SPACE, resident or not
);

// Compare specified name with name for an MFT attribute, if any
BOOL dfsNtfsAttrNameMatch                       // RET   name matches attribute
(
   S_MFTATTR          *at,                      // IN    reference to attribute
   char               *name                     // IN    ASCII attribute name
);

// Compare specified name with name for an attribute-element, if any
BOOL dfsNtfsElemNameMatch                       // RET   name matches attribute
(
   A_ATTRELEM         *el,                      // IN    ref to attribute element
   char               *name                     // IN    ASCII attribute name
);

// Create S_SPACE structure for an NTFS Runlist
ULONG dfsNtfsAllocChain                         // RET   error code
(
   BYTE               *runlist,                 // IN    runlist reference
   ULONG               virtual,                 // IN    1st virtual cluster
   ULONG               clusters,                // IN    clusters in fragment
   USHORT              length,                  // IN    length of run-list
   ULONG               info,                    // IN    additional SPACE info
   ULONG              *chunks,                  // OUT   nr of space entries
   S_SPACE           **space                    // OUT   space allocation
);

// Create a standard NTFS RunList for an S_SPACE array
ULONG dfsNtfsMkRunList                          // RET   rc
(
   ULONG               chunks,                  // IN    nr of space entries
   S_SPACE            *space,                   // IN    space allocation
   USHORT             *length,                  // OUT   bytes in new runlist
   BYTE              **runlist                  // OUT   allocated list or NULL
);

// Display S_SPACE structure for an NTFS non-resident attribute
ULONG dfsNtfsSpDisplay                          // RET   error code
(
   S_MFTFILE          *sd,                      // IN    base MFT rec, fixed-up
   ULONG               type,                    // IN    type of attribute MA_
   USHORT              atid                     // IN    attribute-id or IA_ANY
);

// Translate Directory Btree ordinal to DirBlock (INDX) logical sector nr
ULN64 dfsNtfsBtree2Lsn                          // RET   Logical sector nr
(
   ULN64               mft,                     // IN    DIR MFT record number
   ULONG               ordinal                  // IN    Btree ordinal, relative
);                                              //       block INDX Allocation

// Translate MFT record number to logical sector nr
ULN64 dfsNtfsMft2Lsn                            // RET   Logical sector nr
(
   ULN64               mft                      // IN    MFT record number
);

// Translate MFT-mirror record number to logical sector nr
ULN64 dfsNtfsMir2Lsn                            // RET   Logical sector nr
(
   ULN64               mft                      // IN    MFT record number
);

// Translate (mft record) LSN to MFT record number
ULN64 dfsNtfsLsn2Mft                           // RET   MFT record number
(
   ULN64               lsn                      // IN    Logical sector nr
);

#define dfsNtfsClust2Lsn(cl) (ULN64) dfsNtfsCl2Lsn((cl),0,NULL,NULL)
// Translate Cluster-nr to LSN
ULN64 dfsNtfsCl2Lsn                             // RET   LSN for this cluster
(
   ULN64               cluster,                 // IN    cluster number
   ULN64               d2,                      // IN    dummy
   char               *p1,                      // IN    dummy
   void               *p2                       // IN    dummy
);

// Iterate over MFT records (using a callback function)
ULONG dfsNtfsMftIterator                        // RET   next result
(
   DFS_UL_FUNCT        callback,                // IN    callback for each MFT
   ULN64               minmft,                  // IN    first mft to process
   ULN64               maxmft                   // IN    last  mft to process
);

// NTFS generic Iterate: "mft" = MFT records (returning LSN for each)
ULN64 dfsNtfsLsnIterator                        // RET   next LSN to work on
(
   ULN64               this,                    // IN    this LSN (or special)
   ULN64               d2,                      // IN    dummy
   char               *p1,                      // IN    options: "mft"
   void               *p2                       // INOUT ptr to ULN64 index
);                                              //       or NULL

#define dfsNtfsLsn2Clust(sn) dfsNtfsLsn2Cl((ULN64)(sn),0,NULL, NULL)
// Translate LSN to Cluster-nr
ULN64 dfsNtfsLsn2Cl                             // RET   cluster for this LSN
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *p1,                      // IN    dummy
   void               *p2                       // IN    dummy
);

// Get MFT-record usage status from $MFT Bitmap cache
BOOL dfsNtfsMftMapped                           // RET   MFT record is in-use
(
   ULN64               mft                      // IN    MFT number
);

// Determine allocation-bit for specified LSN
ULONG dfsNtfsAllocated                          // RET   LSN is allocated
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);

// Align R/W cache to position for cluster, and get current allocation-bit
BOOL dfsNtfsBitmapCache                         // RET   cluster is allocated
(
   ULN64               cl,                      // IN    cluster
   ULONG              *bp                       // OUT   byte position or NULL
);                                              //       = L32_NULL if invalid

// Flush BitMap cache when Dirty, and optional free its resources
ULONG dfsNtfsBitmapFlush                        // RET   rc
(
   BOOL                terminate                // IN    terminate cache too
);

// Set allocate status for LSN to specified value (beyond FS, to end of bitmap)
ULONG dfsNtfsSetAlloc                           // RET   LSN allocation set
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *value,                   // IN    NULL = not allocated
   void               *ref                      // IN    dummy (for NTFS)
);

// Set allocate status for CLUSTER to specified value
ULONG dfsNtfsSetCluster                         // RET   CL  allocation set
(
   ULN64               cl,                      // IN    Cluster
   BOOL                value                    // IN    SET allocation bit
);

// Set allocate status for DFSISPACE to specified value
ULONG dfsNtfsSetSpaceAlloc                     // RET   allocation set
(
   DFSISPACE          *si,                      // IN    space info
   BOOL                value                    // IN    SET allocation bit
);

// Copy a valid NTFS bootrecord based on (hopefully) available spare-copy
ULONG dfsNtfsCopySpareBR
(
   void
);

// Get label-string from the "Volume" MFT record
BOOL dfsNtfsVolumeLabel                         // RET   Label found
(
   S_BOOTR            *boot,                    // IN    boot record for vol
   ULN64               offset,                  // IN    Offset for boot-rec
   char               *label                    // OUT   label string
);

// Create valid NTFS NTLDR sector (second sector of $Boot), and write to disk
ULONG dfsNtfsMkNtldrSec
(
   BOOL                vw7810                   // IN    Vista/W7/8/10 BOOTMGR
);

// Create valid NTFS boot record based on p-table info (or GEO) and template
ULONG dfsNtfsMkBootRec
(
   ULONG               mftLsn,                  // IN    sectornr for $MFT or 0
   BOOL                vw7810                   // IN    Vista/W7/8/10 BOOTMGR
);

#endif
