//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// NTFS utility functions
//
// Author: J. van Wijk
//
// JvW  04-12-97   Initial version
// JvW  06-01-2000 Fixed trap on resident MFT allocations
// JvW  12-05-2000 Removed obsolete dfsNtfsUndeleteSNtable function
// JvW  22-06-2000 Less stringent validity test on fixboot sectors
// JvW  07-07-2001 Fixed trap on invalid date/time (gmtime returns NULL!)
// JvW  24-01-2016 Added Windows-7 variant of the bootsector template code
// JvW  04-04-2017 Updated LSN to 64bit, MFT-numbers to 64bit
// JvW  16-11-2017 Introduce 64bit desired-parent MFT-number in various functions
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsver.h>                             // DFS version info
#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfstable.h>                           // SLT utility functions

#include <dfsantfs.h>                           // NTFS analysis functions
#include <dfsuntfs.h>                           // NTFS utility functions

// REXX value = d2x(((369*365+89)*86400)*10000000)
#define  NTM_LO    ((ULONG) 0xd53e8000)           // delta between 1-1-1601 base
#define  NTM_HI    ((ULONG) 0x019db1de)           // and ANSI 1-1-1970 (100 ns)

#define  NTM_HI_M  ((double) 65536 * (double) (65536))
#define  NTM_NS_D  ((double) 10000000)          // 100ns slots per second


typedef struct ntfs_fixup                       // first sector of 'fixed-up'
{                                               // structure
   BYTE         signature[4];                   // sector signature bytes
   USHORT       offset;                         // byte offset to fixup table
   USHORT       size;                           // total size of fixup table
   BYTE         filler[FIXUP_AREA_SIZE - 10];  // size of fixup area - 10
   USHORT       generation;                     // generation value for sector
} NTFS_FIXUP;                                   // end of struct "ntfs_fixup"


static ULONG dfsNtfsBootTemplateXP[] =          // From Windows-XP, US-EN
{
   0x4e9052eb, 0x20534654, 0x00202020, 0x00000802, // [.R.NTFS    .....]
   0x00000000, 0x0000f800, 0x00f0003f, 0x0000003f, // [........?...?...]
   0x00000000, 0x00800080, 0x0257da40, 0x00000000, // [........@.W.....]
   0x000c0000, 0x00000000, 0x00257da4, 0x00000000, // [.........}%.....]
   0x000000f6, 0x00000001, 0x0093daf9, 0x5a0093fa, // [...............Z]
   0x00000000, 0x8ec033fa, 0x7c00bcd0, 0x07c0b8fb, // [.....3.....|....]
   0x16e8d88e, 0x0d00b800, 0xdb33c08e, 0x000e06c6, // [..........3.....]
   0x0053e810, 0x680d0068, 0x8acb026a, 0xb4002416, // [..S.h..hj....$..]
   0x7313cd08, 0xffffb905, 0x0f66f18a, 0x6640c6b6, // [...s......f...@f]
   0x80d1b60f, 0xe2f73fe2, 0xedc0cd86, 0x0f664106, // [.....?.......Af.]
   0xf766c9b7, 0x20a366e1, 0x41b4c300, 0x8a55aabb, // [..f..f. ...A..U.]
   0xcd002416, 0x810f7213, 0x75aa55fb, 0x01c1f609, // [.$...r...U.u....]
   0x06fe0474, 0x66c30014, 0x66061e60, 0x660010a1, // [t......f`..f...f]
   0x001c0603, 0x20063b66, 0x3a820f00, 0x6a661e00, // [....f;. ...:..fj]
   0x06506600, 0x10686653, 0x80000100, 0x0000143e, // [.fP.Sfh.....>...]
   0x000c850f, 0x80ffb3e8, 0x0000143e, 0x0061840f, // [........>.....a.]
   0x168a42b4, 0x1f160024, 0x13cdf48b, 0x075b5866, // [.B..$.......fX[.]
   0x58665866, 0x662deb1f, 0x0f66d233, 0x00180eb7, // [fXfX..-f3.f.....]
   0xfef1f766, 0x66ca8ac2, 0xc166d08b, 0x36f710ea, // [f......f..f....6]
   0xd686001a, 0x0024168a, 0xe4c0e88a, 0xb8cc0a06, // [......$.........]
   0x13cd0201, 0x0019820f, 0x2005c08c, 0x66c08e00, // [........... ...f]
   0x001006ff, 0x000e0eff, 0xff6f850f, 0x61661f07, // [..........o...fa]
   0x01f8a0c3, 0xa00009e8, 0x03e801fb, 0xfeebfb00, // [................]
   0xf08b01b4, 0x74003cac, 0xbb0eb409, 0x10cd0007, // [.....<.t........]
   0x0dc3f2eb, 0x6420410a, 0x206b7369, 0x64616572, // [.....A disk read]
   0x72726520, 0x6f20726f, 0x72756363, 0x00646572, // [ error occurred.]
   0x544e0a0d, 0x2052444c, 0x6d207369, 0x69737369, // [..NTLDR is missi]
   0x0d00676e, 0x4c544e0a, 0x69205244, 0x6f632073, // [ng...NTLDR is co]
   0x6572706d, 0x64657373, 0x500a0d00, 0x73736572, // [mpressed...Press]
   0x72744320, 0x6c412b6c, 0x65442b74, 0x6f74206c, // [ Ctrl+Alt+Del to]
   0x73657220, 0x74726174, 0x00000a0d, 0x00000000, // [ restart........]
   0x00000000, 0x00000000, 0xc9b3a083, 0xaa550000  // [..............U.]
};


static ULONG dfsNtfsBootTemplateW7[] =          // From Windows-7, US-EN
{
   0x4e9052eb, 0x20534654, 0x00202020, 0x00000802, // .R.NTFS    .....
   0x00000000, 0x0000f800, 0x00f0003f, 0x00003b10, // ........?....;..
   0x00000000, 0x00800080, 0x15094cdf, 0x00000000, // .........L......
   0x000c0000, 0x00000000, 0x00000002, 0x00000000, // ................
   0x000000f6, 0x00000001, 0x6a142afc, 0x706a1466, // .........*.jf.jp
   0x00000000, 0x8ec033fa, 0x7c00bcd0, 0x07c068fb, // .....3.....|.h..
   0x66681e1f, 0x1688cb00, 0x8166000e, 0x4e00033e, // ..hf......f.>..N
   0x75534654, 0xbb41b415, 0x13cd55aa, 0xfb810c72, // TFSu..A..U..r...
   0x0675aa55, 0x0001c1f7, 0xdde90375, 0xec831e00, // U.u.....u.......
   0x001a6818, 0x168a48b4, 0xf48b000e, 0x13cd1f16, // .h...H..........
   0x18c4839f, 0x721f589e, 0x0b063be1, 0xa3db7500, // .....X.r.;...u..
   0x2ec1000f, 0x1e04000f, 0xb9db335a, 0xc82b2000, // ........Z3... +.
   0x1106ff66, 0x0f160300, 0xffc28e00, 0xe8001606, // f...............
   0xc82b004b, 0x00b8ef77, 0x661acdbb, 0x2d75c023, // K.+.w......f#.u-
   0x54fb8166, 0x75415043, 0x02f98124, 0x161e7201, // f..TCPAu$....r..
   0x16bb0768, 0x160e7068, 0x66000968, 0x66536653, // h...hp..h..fSfSf
   0x16161655, 0x6601b868, 0xcd070e61, 0xbfc0331a, // U...h..fa....3..
   0xd8b91028, 0xaaf3fc0f, 0x90015fe9, 0x1e606690, // (........_...f`.
   0x11a16606, 0x06036600, 0x661e001c, 0x00000068, // .f...f.....fh...
   0x06506600, 0x00016853, 0xb4001068, 0x0e168a42, // .fP.Sh..h...B...
   0x8b1f1600, 0x6613cdf4, 0x665a5b59, 0x1f596659, // .......fY[ZfYfY.
   0x0016820f, 0x1106ff66, 0x0f160300, 0xffc28e00, // ....f...........
   0x7500160e, 0x661f07bc, 0xf8a0c361, 0x0009e801, // ...u...fa.......
   0xe801fba0, 0xebf40003, 0x8b01b4fd, 0x003cacf0, // ..............<.
   0x0eb40974, 0xcd0007bb, 0xc3f2eb10, 0x20410a0d, // t.............A
   0x6b736964, 0x61657220, 0x72652064, 0x20726f72, // disk read error
   0x7563636f, 0x64657272, 0x420a0d00, 0x4d544f4f, // occurred...BOOTM
   0x69205247, 0x696d2073, 0x6e697373, 0x0a0d0067, // GR is missing...
   0x544f4f42, 0x2052474d, 0x63207369, 0x72706d6f, // BOOTMGR is compr
   0x65737365, 0x0a0d0064, 0x73657250, 0x74432073, // essed...Press Ct
   0x412b6c72, 0x442b746c, 0x74206c65, 0x6572206f, // rl+Alt+Del to re
   0x72617473, 0x000a0d74, 0xd6bea98c, 0xaa550000  // start.........U.
};

static S_BOOTR *brt = (S_BOOTR *) &dfsNtfsBootTemplateW7;


static ULONG dfsNtfsNtldrFirstSec[] =           // From Windows-XP, US-EN
{
   0x004e0005, 0x004c0054, 0x00520044, 0x00240004, // [..N.T.L.D.R...$.]
   0x00330049, 0xe0000030, 0x30000000, 0x00000000, // [I.3.0......0....]
   0x00000000, 0x00000000, 0x00000000, 0x00000000, // [................]
   0x00000000, 0x00000000, 0x00000000, 0x00000000, // [................]
   0x00000000, 0x00000000, 0x00000000, 0x00000000, // [................]
   0x00000000, 0x12eb0000, 0x00009090, 0x00000000, // [................]
   0x00000000, 0x00000000, 0xc88c0000, 0xe0c1d88e, // [................]
   0xe08bfa04, 0xfe03e8fb, 0x06b70f66, 0x0f66000b, // [........f.....f.]
   0x000d1eb6, 0x66e3f766, 0x66024ea3, 0x00400e8b, // [....f..f.N.f..@.]
   0x0f00f980, 0xf6000e8f, 0x01b866d9, 0x66000000, // [.........f.....f]
   0x08ebe0d3, 0x4ea16690, 0xe1f76602, 0x0252a366, // [.....f.N.f..f.R.]
   0x1eb70f66, 0x3366000b, 0xf3f766d2, 0x0256a366, // [f.....f3.f..f.V.]
   0x66040de8, 0x024a0e8b, 0x220e8966, 0x0e036602, // [...f..J.f..".f..]
   0x89660252, 0x6602260e, 0x02520e03, 0x2a0e8966, // [R.f..&.f..R.f..*]
   0x0e036602, 0x89660252, 0x66023a0e, 0x02520e03, // [.f..R.f..:.f..R.]
   0x420e8966, 0x90b86602, 0x66000000, 0x02220e8b, // [f..B.f.....f..".]
   0x6608ece8, 0x840fc00b, 0xa366fe57, 0xb866022e, // [...f....W.f...f.]
   0x000000a0, 0x260e8b66, 0x08d3e802, 0x0232a366, // [....f..&....f.2.]
   0x00b0b866, 0x8b660000, 0xe8022a0e, 0xa36608c1, // [f.....f..*....f.]
   0xa1660236, 0x0b66022e, 0x24840fc0, 0x788067fe, // [6.f...f....$.g.x]
   0x850f0008, 0x6667fe1b, 0x6710508d, 0x67044203, // [......gf.P.g.B.g]
   0x48b60f66, 0x0e89660c, 0x66670262, 0x6608488b, // [f..H.f..b.gf.H.f]
   0x025e0e89, 0x025ea166, 0x0eb70f66, 0x3366000b, // [..^.f.^.f.....f3]
   0xf1f766d2, 0x0266a366, 0x0242a166, 0x5e060366, // [.f..f.f.f.B.f..^]
   0x46a36602, 0x3e836602, 0x0f000232, 0x66001984, // [.f.F.f.>2......f]
   0x02363e83, 0xc8840f00, 0x1e8b66fd, 0x071e0236, // [.>6......f..6...]
   0x463e8b66, 0x0192e802, 0x0eb70f66, 0xb8660200, // [f.>F....f.....f.]
   0x00000202, 0x660796e8, 0x840fc00b, 0x6667090a, // [.......f......gf]
   0x071e008b, 0x3a3e8b66, 0x05cee802, 0x023aa166, // [....f.>:....f.:.]
   0x0080bb66, 0xb9660000, 0x00000000, 0x0000ba66, // [f.....f.....f...]
   0xace80000, 0xc00b6600, 0x003e850f, 0x0080b966, // [.....f....>.f...]
   0xa1660000, 0x59e8023a, 0xc00b6608, 0x08c8840f  // [..f.:..Y.f......]
};


static ULONG dfsNtfsNtldrVistaSec[] =           // From Windows-Vista
{
   0x00420007, 0x004f004f, 0x004d0054, 0x00520047, // ..B.O.O.T.M.G.R.
   0x00240004, 0x00330049, 0xe0000030, 0x30000000, // ..$.I.3.0......0
   0x00000000, 0x00000000, 0x00000000, 0x00000000, // ................
   0x00000000, 0x00000000, 0x00000000, 0x00000000, // ................
   0x00000000, 0x00000000, 0x00000000, 0x00000000, // ................
   0x00000000, 0x22eb0000, 0x00059090, 0x0054004e, // ......."....N.T.
   0x0044004c, 0x00000052, 0x00000000, 0x00000000, // L.D.R...........
   0x00000000, 0x00000000, 0x0f660000, 0x000b06b7, // ..........f.....
   0x1eb60f66, 0xf766000d, 0x52a366e3, 0x0e8b6602, // f.....f..f.R.f..
   0xf9800040, 0x0e8f0f00, 0x66d9f600, 0x000001b8, // @..........f....
   0xe0d36600, 0x669008eb, 0x660252a1, 0xa366e1f7, // .f.....f.R.f..f.
   0x0f660266, 0x000b1eb7, 0x66d23366, 0xa366f3f7, // f.f.....f3.f..f.
   0x95e80256, 0x0e8b6604, 0x8966024e, 0x6602260e, // V....f..N.f..&.f
   0x02660e03, 0x2a0e8966, 0x0e036602, 0x89660266, // ..f.f..*.f..f.f.
   0x66022e0e, 0x02660e03, 0x3e0e8966, 0x0e036602, // ...f..f.f..>.f..
   0x89660266, 0x6602460e, 0x000090b8, 0x0e8b6600, // f.f..F.f.....f..
   0x83e80226, 0xc00b6609, 0xfe53840f, 0x0232a366, // &....f....S.f.2.
   0x00a0b866, 0x8b660000, 0xe8022a0e, 0xa366096a, // f.....f..*..j.f.
   0xb8660236, 0x000000b0, 0x2e0e8b66, 0x0958e802, // 6.f.....f.....X.
   0x023aa366, 0x0232a166, 0x0fc00b66, 0x67fe2084, // f.:.f.2.f.... .g
   0x00087880, 0xfe17850f, 0x508d6667, 0x42036710, // .x......gf.P.g.B
   0x0f666704, 0x660c48b6, 0x02720e89, 0x488b6667, // .gf..H.f..r.gf.H
   0x0e896608, 0xa166026e, 0x0f66026e, 0x000b0eb7, // .f..n.f.n.f.....
   0x66d23366, 0xa366f1f7, 0xa1660276, 0x03660246, // f3.f..f.v.f.F.f.
   0x66026e06, 0x66024aa3, 0x02363e83, 0x1d840f00, // .n.f.J.f.>6.....
   0x3e836600, 0x0f00023a, 0x66fdc484, 0x023a1e8b, // .f.>:......f..:.
   0x8b66071e, 0x66024a3e, 0xe8022ea1, 0x0f6601e0, // ..f.>J.f......f.
   0x02000eb7, 0x0202b866, 0x22e80000, 0xc00b6608, // ....f......".f..
   0x0016850f, 0x0eb70f66, 0xb866025a, 0x0000025c, // ....f...Z.f.\...
   0x66080ce8, 0x840fc00b, 0x66670c42, 0x071e008b, // ...f....B.gf....
   0x3e3e8b66, 0x063fe802, 0x023ea166, 0x0020bb66, // f.>>..?.f.>.f. .
   0xb9660000, 0x00000000, 0x0000ba66, 0xe4e80000  // ..f.....f.......
};

//- $MFT secondary signature value
static char sg_mft_0[] = {(char)0x04, (char)0x03,  // length bytes
                          (char)0x24, (char)0x00}; // $ it IS a system file

static char ntfsDirHeaderText[] =
 " Nr    MftRecLSN  Creation/LastAccess Attr. Modified / Filename             Filesize    Mft-nr";

static char ntfsDirHeaderLine[] =
 " ===== ========== =================== ===== =================== ==================== =========";


// Add ROOT/INDX file/dir MFT-rec LSN's to the sectorlist (recursive in Btree)
static ULONG dfsNtfsDirEntries2List
(
   ULONG               guard,                   // IN    recursion guard counter
   ULN64               dirMft,                  // IN    MFTnr for directory
   BOOL                dos83,                   // IN    favor DOS8.3 shortname
   S_DIRENTRY         *first                    // IN    first Entry in sequence
);

// Add DirBlock sequence  file/dir MFT-rec LSN's to the sectorlist (recursive)
static ULONG dfsNtfsDirIndx2List
(
   ULONG               guard,                   // IN    recursion guard counter
   ULONG               dirMft,                  // IN    MFTnr for directory
   BOOL                dos83,                   // IN    favor DOS8.3 shortname
   ULONG               bDown                    // IN    Btree-down DirBlock LSN
);

// Read sn into rbuf and check if it is a valid NTFS bootrec for this partition
static BOOL dfsNtfsValidBootRec
(
   ULN64               sn                       // IN    logical sector-number
);


/*****************************************************************************/
// Find MFT-record for specified path, starting at known root-directory MFT
/*****************************************************************************/
ULONG dfsNtfsFindPath
(
   ULN64               loud,                    // IN    Show progress
   ULN64               d2,                      // IN    dummy
   char               *path,                    // IN    path specification
   void               *vp                       // OUT   found dir/file MFT Lsn
)                                               //       and exact byteSize
{
   ULONG               rc  = NO_ERROR;
   TXLN                part;
   char               *p   = path;
   int                 l;
   A_IDXROOT          *idxRoot;
   S_DIRENTRY         *dirArea;                 // DIRENTRY area
   ULN64               curMft = MFT_ROOT;
   DFS_PARAMS         *parm = (DFS_PARAMS *) vp;
   ULN64               bytesize;                // size of file or DIR (from DIR-entry)

   ENTER();
   TRACES(("path: '%s'", path));

   if (loud)
   {
      TxPrint("RootDir   MFT  nr : %12.12llX   find path: '%s'\n", curMft, path);
   }
   parm->Lsn      = dfsNtfsMft2Lsn( curMft);    // mft Lsn for ROOT
   parm->Number   = 0;
   parm->byteSize = 0;                          // size for ROOT
   while ((rc == NO_ERROR) && strlen(p) && !TxAbort())
   {
      if ((l = strcspn(p, FS_PATH_STR)) != 0)
      {
         strncpy(part, p, l);                   // isolate part
         part[l] = '\0';
         p += l;                                // skip part
         if (*p == FS_PATH_SEP)
         {
            p++;                                // skip '\'
         }
      }
      if (strlen(part))
      {
         if (loud)
         {
            TxPrint("Directory MFT LSN : 0x%12.12llX", dfsNtfsMft2Lsn( curMft));
         }
         TRACES(( "Search MFT: %llx for '%s'\n", curMft, part));
         if (dfsNtfsMft2DirEntries( curMft, &idxRoot))
         {
            dirArea = idxRoot->EntryList;
            if (dfsNtfsDirIndx2Entry( curMft, loud, part, dirArea, &curMft, &bytesize))
            {
               if (loud)
               {
                  TxPrint(" - MFT %12.12llX  for '%s'\n", curMft, part);
               }
               if (*p == '\0')                  // end of string, found!
               {
                  parm->Lsn      = dfsNtfsMft2Lsn( curMft);
                  parm->Number   = 0;           // no additional info
                  parm->byteSize = bytesize;
                  parm->Flag     = TRUE;        // Size is from DIR-entry
                  TRACES(("MFT-lsn:0x%llx  size:0x%llx\n", parm->Lsn, parm->byteSize));
               }
            }
            else
            {
               if (loud)
               {
                  TxPrint(" - Search failed for '%s'\n", part);
               }
               rc = ERROR_PATH_NOT_FOUND;
            }
            TxFreeMem( idxRoot);                // free initial INDX root
         }
         else
         {
            rc = ERROR_PATH_NOT_FOUND;
         }
      }
      else
      {
         rc = ERROR_PATH_NOT_FOUND;
      }
   }
   RETURN (rc);
}                                               // end 'dfsNtfsFindPath'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read initial DIR-entries (attribute 90) from directory MFT LSN
/*****************************************************************************/
BOOL dfsNtfsMft2DirEntries                      // OUT   DIR entries resolved
(
   ULONG               mft,                     // IN    directory MFT number
   A_IDXROOT         **idx                      // OUT   INDX Root structure
)
{
   BOOL                rc  = FALSE;
   ULONG               size;                    // size of dir area read

   ENTER();
   TRACES(( "Read INDX Root for MFT: %8.8x\n", mft));

   if (dfsNtfsMftLsnAttrData( dfsNtfsMft2Lsn( mft),
       MA_IDXROOT, IA_ANY, NULL, SECTORSIZE,
       &size, (void **) idx) == NO_ERROR)
   {
      rc = TRUE;
   }
   TRACES(( "IdxRoot allocated at %8.8x\n", *idx));
   BRETURN (rc);
}                                               // end 'dfsNtfsMft2DirEntries'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Find DIR-entry for specified name, start at DIRBLOCK LSN, walk the Btree
/*****************************************************************************/
BOOL dfsNtfsDirIndx2Entry                       // OUT   entry found
(
   ULN64               dirMft,                  // IN    MFTnr for directory
   ULONG               loud,                    // IN    Show progress
   char               *entry,                   // IN    entry specification
   S_DIRENTRY         *dindex,                  // IN    start Index ROOT area
   ULN64              *mft,                     // OUT   found dir/file MFT nr
   ULN64              *size                     // OUT   file/dir size in bytes
)                                               //       from the directory-entry
{
   BOOL                rc   = FALSE;
   int                 cp   = 1;                // compare result; 1 == greater
   S_DIRENTRY         *de;                      // Single directory entry
   S_DIRINDX          *Indx = NULL;             // INDX   directory structure
   BYTE               *pos  = (BYTE *) dindex;  // positional ptr in list
   TXLN                fname;                   // upercase candidate name ASCII
   TXLN                uname;                   // upercase search-name    ASCII

   ENTER();
   TRACES(( "Search dirMFT:%llx for '%s'\n", dirMft, entry));

   TxStrToUpper( strcpy( uname, entry));
   while ((rc == FALSE) && (pos != NULL) && (!TxAbort()))
   {
      do
      {
         de = (S_DIRENTRY *) pos;               // current position as ENTRY

         if      (de->Length > 0x20)            // regular entry, filedata
         {
            TxUnic2Ascii( de->Filename, fname, de->NameLen);
            TxStrToUpper( fname);               // make all uppercase

            cp = strcmp( uname, fname);
            TRACES(( "Compare result %u for '%s' - '%s'\n", cp, fname, entry));
         }
         else if (de->Length >= 0x10)           // Down-Right Btree pointer,
         {                                      // after last real name ...
            TRACES(( "End of index, down-right link if there\n"));
            cp = -1;                            // Force following the link
         }

         if      (cp < 0)                       // Need to follow the link
         {
            if (de->DirFlags & IDX_BTREE_PTR)   // B-tree down pointer present
            {
               ULONG        *bto;               // B-tree ordinal 0..n
               ULN64         bdown;             // B-tree INDX block LSN

               bto   = (ULONG *) (pos + de->Length - 8);
               bdown = dfsNtfsBtree2Lsn( dirMft,  *bto);

               TRACES(( "Btree DOWN ordinal: %u = %llx\n", *bto, bdown));

               if (loud)
               {
                  TxPrint("\nBtree %s% 3.3u%s DIR LSN : 0x%llX", CBM, *bto, CNN, bdown);
               }
               //- Recursive implementation is possibe too, getting
               //- new INDX block and then call this function again
               //- Will require more memory and is not needed here.

               TxFreeMem( Indx);                // free any previous
               if (dfsNtfsReadFixIndx( bdown, &Indx) == NO_ERROR)
               {
                  pos = (((BYTE *) &Indx->HeaderSize) + Indx->HeaderSize);
                  TRACES(( "First DIRENTRY at: %8.8x\n", pos));
               }
               else
               {
                  pos = NULL;                   // INDX error, end search
               }
               break;                           // continue with next INDX
            }
            else
            {
               pos = NULL;                      // no link, end search
            }
         }
         else if (cp == 0)                      // equal, entry found
         {
            if (strlen(entry) == strlen(fname))
            {
               *mft  = dfsMftPure(de->Mftnr);   // return MFT-nr
               *size = de->Size;                // and bytesize
               rc = TRUE;
               break;
            }
         }
         else
         {
            pos  += de->Length;                 // keep searching until END
         }
      } while ((pos != NULL) &&
               ((de->DirFlags & IDX_BTREE_END) == 0) &&
                (de->Length > 20) && !TxAbort());
   }
   TxFreeMem( Indx);
   BRETURN (rc);
}                                               // end 'dfsNtfsDirIndx2Entry'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Replace sn-list by contents of a single Directory, storing MFT-rec Lsn's
// Filter out the NTFS system files (\$xxx in ROOT)
/*****************************************************************************/
ULONG dfsNtfsMakeBrowseList
(
   ULN64               mftLsn,                  // IN    MFT-rec sectornumber
   ULN64               info,                    // IN    unused
   char               *options,                 // IN    option string (~ 8.3)
   void               *param                    // INOUT unused
)
{
   ULONG               rc  = NO_ERROR;
   A_IDXROOT          *idxRoot;
   S_DIRENTRY         *dirArea;                 // DIRENTRY area
   ULN64               curMft;
   ULN64               parentLsn;
   BOOL                dos83 = (strchr(options, '~') != NULL); // prefer 8.3 names

   ENTER();
   TRACES(("mftLsn:0x%llx  options:'%s'  dos83: %s\n", mftLsn, options, (dos83) ? "TRUE" : "FALSE"));

   if ((curMft = dfsNtfsLsn2Mft( mftLsn)) != MFT_NULL)     // validate sectornumber
   {
      if (dfsNtfsMft2DirEntries( curMft, &idxRoot))
      {
         dfsInitList(0, "-f -P", "-d");         // optimal for menu file-recovery

         if (dfsNtfsMft2Parent( mftLsn, FALSE, FALSE, 0, NULL, &parentLsn))
         {
            TRACES(("List marked as 1ST_PARENT\n"));
            DFSBR_SnlistFlag |= DFSBR_1ST_PARENT;
            dfsAdd2SectorList( parentLsn);      // real parent found
         }
         dirArea = idxRoot->EntryList;
         rc = dfsNtfsDirEntries2List( 0, curMft, dos83, dirArea);
         TxFreeMem( idxRoot);                   // free initial INDX root
      }
      else                                      // no index-ROOT attribute
      {
         rc = DFS_BAD_STRUCTURE;
      }
   }
   else
   {
      rc = DFS_BAD_STRUCTURE;
   }
   RETURN (rc);
}                                               // end 'dfsNtfsMakeBrowseList'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get allocation information for file-DATA into integrated-SPACE structure
/*****************************************************************************/
ULONG dfsNtfsGetAllocSpace
(
   ULN64               mftLsn,                  // IN    MFT-record Lsn
   ULN64               info,                    // IN    unused
   char               *str,                     // IN    unused
   void               *param                    // INOUT Integrated SPACE
)
{
   ULONG               rc = NO_ERROR;
   DFSISPACE          *ispace = (DFSISPACE *) param;

   ENTER();

   rc = dfsNtfsMftLsn2Alloc( mftLsn, MA_DATA, IA_ANY, NULL, ispace);
   RETURN (rc);
}                                               // end 'dfsNtfsGetAllocSpace'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Write supplied data area into an MFT-record (resident data attribute)
/*****************************************************************************/
ULONG dfsNtfsWriteMetaSpace
(
   ULN64               num1,                    // IN    unused
   ULN64               num2,                    // IN    unused
   char               *str,                     // IN    unused
   void               *param                    // IN    Integrated SPACE
)
{
   ULONG               rc = NO_ERROR;
   DFSISPACE          *is = (DFSISPACE *) param;
   BYTE                st = ST_UDATA;
   S_MFTFILE          *mft;

   ENTER();
   TRACES(( "MFtlsn:0x%llx offset:%u size:%llu\n", is->meta, is->offset, is->byteSize));

   if (((is->chunks == 0) && (is->space != NULL)) && //- resident data
       ((is->byteSize < ((ntfs->MftRecSize * dfsGetSectorSize()) - is->offset - 8))))
   {
      if ((rc = dfsNtfsReadFixMft( is->meta, &st, &mft)) == NO_ERROR)
      {
         BYTE         *resident = ((BYTE *) mft) +  is->offset;
         size_t        size     = (size_t)          is->byteSize;

         TRACES(("memcpy: %u bytes from: %8.8x to %8.8x\n", size, is->space, resident));
         memcpy(  resident, is->space, size);

         TRHEXS(70, is->space, size, "Modified data, is->space")

         rc = dfsNtfsFixWriteMft(  is->meta, mft); // write modified MFT record back
         TxFreeMem( mft);
      }
   }
   else
   {
      rc = DFS_ST_MISMATCH;
   }
   RETURN (rc);
}                                               // end 'dfsNtfsWriteMetaSpace'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add ROOT/INDX file/dir MFT-rec LSN's to the sectorlist (recursive in Btree)
/*****************************************************************************/
static ULONG dfsNtfsDirEntries2List
(
   ULONG               guard,                   // IN    recursion guard counter
   ULN64               dirMft,                  // IN    MFTnr for directory
   BOOL                dos83,                   // IN    favor DOS8.3 shortname
   S_DIRENTRY         *first                    // IN    first Entry in sequence
)
{
   ULONG               rc  = NO_ERROR;
   S_DIRENTRY         *de;                      // Single directory entry
   S_DIRINDX          *Indx = NULL;             // INDX   directory structure
   BYTE               *pos  = (BYTE *) first;   // positional ptr in sequence

   ENTER();

   do
   {
      de = (S_DIRENTRY *) pos;                  // current position as ENTRY

      TRACES(("de->Length:%2hx DirFlags:%2.2hhx  NameType:%2hhx  Mft:%llx\n",
               de->Length, de->DirFlags, de->FileNameType,   de->Mftnr));

      if (de->DirFlags & IDX_BTREE_PTR)         // B-tree down pointer present
      {
         ULONG        *bto;                     // B-tree ordinal 0..n
         ULN64         bdown;                   // B-tree INDX block LSN

         bto   = (ULONG *) (pos + de->Length - 8);
         bdown = dfsNtfsBtree2Lsn( dirMft,  *bto);

         TRACES(( "Btree DOWN ordinal: %u = 0x%llx\n", *bto, bdown));

         rc = dfsNtfsDirIndx2List( guard, dirMft, dos83, bdown); // recursive
      }
      if (de->Length > 0x20)                    // regular entry, file/dir
      {
         if ((de->FileNameType == FN_DOS83) == dos83) // 8.3 type OK
         {
            A_STANDARD  *sa = NULL;             // standard attribute
            ULONG        atSize;                // attribute size
            ULN64        mftLsn = dfsNtfsMft2Lsn( dfsMftPure(de->Mftnr));
            BOOL         filtered = FALSE;

            if (dfsa->browseShowHidden == FALSE) // need to check hidden/system attribute
            {
               if ((dirMft == MFT_ROOT) && (dfsMftPure(de->Mftnr) < MFT_1STUSER)) // $xxx system file
               {
                  filtered = TRUE;
               }
               else if (de->Filename[0] == (USHORT) '.') // .filenames are hidden
               {
                  filtered = TRUE;
               }
               else                             // need to check attribute
               {
                  if (dfsNtfsMftLsnAttrData( mftLsn, MA_STANDARD, IA_ANY, NULL, SECTORSIZE, &atSize, (void **) &sa) == NO_ERROR)
                  {
                     if (sa && (atSize >= sizeof(A_STANDARD)))
                     {
                        if ((sa->FileAttributes & FATTR_SYSTEM) ||
                            (sa->FileAttributes & FATTR_HIDDEN)  )
                        {
                           filtered = TRUE;
                        }
                     }
                     TxFreeMem( sa);            // free attribute data
                  }
               }
            }
            if (!filtered)
            {
               dfsAdd2SectorList( mftLsn);      // add LSN
            }
         }
      }
      pos  += de->Length;                       // keep searching until END
   } while (((de->DirFlags & IDX_BTREE_END) == 0) &&
             (de->Length > 20) && (rc == NO_ERROR));

   TxFreeMem( Indx);
   RETURN (rc);
}                                               // end 'dfsNtfsDirEntries2List'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add DirBlock sequence  file/dir MFT-rec LSN's to the sectorlist (recursive)
/*****************************************************************************/
static ULONG dfsNtfsDirIndx2List
(
   ULONG               guard,                   // IN    recursion guard counter
   ULONG               dirMft,                  // IN    MFTnr for directory
   BOOL                dos83,                   // IN    favor DOS8.3 shortname
   ULONG               bDown                    // IN    Btree-down DirBlock LSN
)
{
   ULONG               rc  = NO_ERROR;
   S_DIRINDX          *Indx = NULL;             // INDX   directory structure
   BYTE               *pos;                     // positional ptr in sequence

   ENTER();

   TRACES(("bDown:%8.8x for mft:%8.8x\n", bDown, dirMft));

   if (guard > 100)                             // sanity recursion check
   {
      rc = DFS_BAD_STRUCTURE;
   }
   else
   {
      if (dfsNtfsReadFixIndx( bDown, &Indx) == NO_ERROR)
      {
         pos = (((BYTE *) &Indx->HeaderSize) + Indx->HeaderSize);

         rc = dfsNtfsDirEntries2List( guard +1, dirMft, dos83, (S_DIRENTRY *) pos);
      }
      TxFreeMem( Indx);
   }
   RETURN (rc);
}                                               // end 'dfsNtfsDirIndx2List'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add NTFS attributes description to a string
/*****************************************************************************/
void dfstrNtfsAttrib
(
   char               *str,                     // INOUT string
   ULONG               FileAttributes           // IN    NTFS attributes
)
{
   dfstrFatAttrib( str, (BYTE)  FileAttributes);
   if (FileAttributes & FA_TEMPORARY    ) strcat( str, " Temporary"     );
   if (FileAttributes & FA_SPARSE_FILE  ) strcat( str, " Sparse"        );
   if (FileAttributes & FA_REPARSE_POINT) strcat( str, " ReparsePoint"  );
   if (FileAttributes & FA_COMPRESSED   ) strcat( str, " Compressed"    );
   if (FileAttributes & FA_OFFLINE      ) strcat( str, " Offline"       );
   if (FileAttributes & FA_NOT_CNT_INDXD) strcat( str, " NotCntIndexed" );
   if (FileAttributes & FA_ENCRYPTED    ) strcat( str, " Encrypted"     );
   if (FileAttributes & FA_NAME_IDX_PRES) strcat( str, " NameIdxPresent");
   if (FileAttributes & FA_VIEW_IDX_PRES) strcat( str, " ViewIdxPresent");
   return;
}                                               // end 'dfstrNtfsAttrib'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Check and fixup an NTFS multi-sector structure using the fixup-table
// Note: may be called on ANY sector type so check boundaries etc!
/*****************************************************************************/
ULONG dfsNtfsFixupStructure                     // RET   nr of invalid sector
(
   BYTE               *fs,                      // IN    ptr to first sector
   ULONG               sectors                  // IN    #sectors in structure
)
{
   ULONG               rc = NO_ERROR;
   NTFS_FIXUP         *ns = (NTFS_FIXUP *) fs;  // ptr to first fixup sector

   ENTER();

   TRACES(("Fixup table offset   : %4.4x  size: %4.4x  sectors: %u\n",
                    ns->offset, ns->size, sectors));

   if (sectors &&
        (ns->size     != 0) &&
       ((ns->size -1) <= (sectors * dfsGetSectorSize() / FIXUP_AREA_SIZE)) &&
        (ns->offset   >  7) &&
        (ns->offset   <  FIXUP_AREA_SIZE))
   {
      USHORT          *ft = (USHORT *) (fs + ns->offset);
      USHORT           i;

      TRACES(("Structure generation : %4.4x\n", ft[0]));

      for (i = 1; (i < ns->size) && (rc == NO_ERROR); i++)
      {
         TRACES(("Sector: %2u, sector-generation: %4.4x, Fixup: %4.4x\n",
                  i-1,    ns[i-1].generation, ft[i]));
         if (ns[i-1].generation == ft[0])       // valid generation value ?
         {
            ns[i-1].generation  =  ft[i];       // fix it up
         }
         else
         {
            rc = i;                             // index of invalid sector
         }
      }
   }
   else
   {
      rc = DFS_BAD_STRUCTURE;
   }
   RETURN( rc);
}                                               // end 'dfsNtfsFixupStructure'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Prepare a fixed-up multi-sector NTFS structure for next write (Un-Fixup :-)
/*****************************************************************************/
ULONG dfsNtfsUnfixStructure                     // RET   nr of invalid sector
(
   BYTE               *fs                       // IN    ptr to first sector
)
{
   ULONG               rc = NO_ERROR;
   NTFS_FIXUP         *ns = (NTFS_FIXUP *) fs;  // ptr to first fixup sector
   USHORT             *ft = (USHORT *) (fs + ns->offset);
   USHORT              i;

   ENTER();

   TRACES(("Fixup table offset   : %4.4x\n", ns->offset));
   TRACES(("Fixup total sectors  : %4.4x\n", ns->size));
   TRACES(("Structure generation : %4.4x\n", ft[0]));
   if ((ns->size   !=  0) &&
       (ns->offset >= 40) &&                    // sanity checks
       (ns->offset <  FIXUP_AREA_SIZE))
   {
      ft[0]++;                                  // update generation count

      for (i = 1; (i < ns->size) && (rc == NO_ERROR); i++)
      {
         ft[i] = ns[i-1].generation;            // put fixup value in table
         ns[i-1].generation  =  ft[0];          // and replace by generation

         TRACES(("Sector: %2u, generation: %4.4x, Fixup: %4.4x\n", i-1, ft[0], ft[i]));
      }
      //- to be refined, should timestamp in standard attribute be updated too ?
   }
   else
   {
      rc = DFS_BAD_STRUCTURE;
   }
   RETURN( rc);
}                                               // end 'dfsNtfsUnfixStructure'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display sequence of NTFS Directory entries, and add to LSN list
/*****************************************************************************/
ULONG dfsNtfsDirEntries                         // RET   nr of entries listed
(
   ULN64               dirMft,                  // IN    MFT number directory
   S_DIRENTRY         *entries,                 // IN    Sequence of Dir entries
   BOOL                header                   // IN    present a header too
)
{
   S_DIRENTRY         *de;                      // Single directory entry
   BYTE               *pos = (BYTE *) entries;  // positional ptr in list
   ULONG               ei = 0;                  // entry index
   int                 cl = TXaNWnZ;            // LSN color
   int                 ac;                      // alternate color fg+bg
   int                 gc;                      // grey FG, alternate BG
   ULN64               mftlsn;                  // MFT starting sector
   TX1K                text;                    // text buffer, MAX_PATH + line
   TXTM                tbuf;                    // temporary text buffer
   A_STANDARD         *sa = NULL;               // standard attribute
   ULONG               atSize;                  // attribute size

   ENTER();

   if (header)
   {
      TxPrint( "\n%s\n%s", ntfsDirHeaderText, ntfsDirHeaderLine);
   }
   TxPrint( "\n");
   if (!TxaOptUnSet('l'))                       // create a new list for DIR
   {
      dfsInitList(0, "-f -P", "-d");            // optimal for menu file-recovery
   }
   do
   {
      de = (S_DIRENTRY *) pos;                  // current position as ENTRY

      if (de->DirFlags & IDX_BTREE_PTR)         // B-tree down pointer present
      {
         ULONG        *bto;                     // B-tree ordinal 0..n
         ULN64         bdown;                   // B-tree INDX block LSN

         bto   = (ULONG *) (pos + de->Length - 8);
         bdown = dfsNtfsBtree2Lsn( dfsMftPure( dirMft),  *bto);

         sprintf(  text, ".%5.5u", ei);
         dfstrX10( text, " ", bdown, (char *) ((ei == 0) ? CBG : CNN), " --");
         sprintf(  tbuf, " Btree - ordinal: %s%u%s\n", CBM, *bto, CNN);
         strcat(   text, tbuf);
         if (ei == 0)
         {
            nav.down = bdown;
         }
         TxPrint( "%s", text);
         ei++;
         if (!TxaOptUnSet('l'))                 // create a new list for DIR
         {
            dfsAdd2SectorList( bdown);          // add to list
         }
      }

      if (de->Length > 0x20)                    // regular entry, filedata
      {
         mftlsn = dfsNtfsMft2Lsn( dfsMftPure( de->Mftnr));
         if (ei % 2)
         {
            ac = TXaBWnC;
            gc = TXaBZnC;
         }
         else
         {
            ac = TXaNWnZ;
            gc = TXaBZnZ;
         }
         if (ei == 0)                           // first entry in the list
         {
            cl = TXaBGnZ;
            nav.down = mftlsn;
         }
         else
         {
            cl = ac;
         }
         sprintf(  text, "%s.%5.5u", ansi[ac], ei);
         dfstrX10( text, "", mftlsn, ansi[cl], ansi[ac]);
         strcat(   text, " ");

         strcat(   text, dfsNtfsTime2str(&de->Create, tbuf));

         if (dfsNtfsMftLsnAttrData( mftlsn, MA_STANDARD, IA_ANY, NULL, SECTORSIZE, &atSize, (void **) &sa) == NO_ERROR)
         {
            if (sa && (atSize >= sizeof(A_STANDARD)))
            {
               dfstrFatAttrib( text, (BYTE)  sa->FileAttributes);
            }
            else
            {
               strcat(   text, " noatt");       // fake FAT-type attributes
            }
            TxFreeMem( sa);                     // free attribute data
         }
         else
         {
            strcat(   text, " -bad-");          // fake FAT-type attributes
         }

         if ((de->Modify.hi != de->Create.hi) ||
             (de->Modify.lo != de->Create.lo)  )
         {
            strcat( text, " ");
            strcat( text, dfsNtfsTime2str(&de->Modify, tbuf));
         }
         else
         {
            strcat( text, "                    ");
         }
         if ((de->Size == 0) && (de->Alloc == 0))
         {
            strcat( text, "                      ");
         }
         else
         {
            dfstrUllDot20( text, " ", de->Size, " "); // real file size
         }
         sprintf( tbuf, "%9llx%s\n%s", dfsMftPure( de->Mftnr), CGE, ansi[ac]);
         strcat(  text, tbuf);

         if ((de->Access.hi != de->Create.hi) ||
             (de->Access.lo != de->Create.lo)  )
         {
            sprintf( tbuf, " %saccessed%s%s ", ansi[gc], CNN, ansi[ac]);
            strcat(  text, tbuf);
            strcat(  text, dfsNtfsTime2str(&de->Access, tbuf));
         }
         else
         {
            sprintf( tbuf, " %28.28s", "");
            strcat(  text, tbuf);
         }
         strcat(  text, "       ");             // attributes / Flags column

         strcat(  text, ansi[Ccol((CcY | CcI), Ccbg(ac))]); // Yellow
         TxUnicAppend( de->Filename, text, de->NameLen);
         sprintf( tbuf, "%s%s%s\n", ansi[Ccol( CcW, Ccbg(ac))], CGE, CNN);
         strcat(  text, tbuf);

         TxPrint( "%s", text);

         ei++;
         if (!TxaOptUnSet('l'))                 // create a new list for DIR
         {
            dfsAdd2SectorList( mftlsn);         // add to list
         }
      }
      pos  += de->Length;

   } while (((de->DirFlags & IDX_BTREE_END) == 0) &&
             (de->Length > 20) && !TxAbort());

   if (header && (ei > 12))                     // reversed header at end
   {
      TxPrint( "%s\n%s\n", ntfsDirHeaderLine, ntfsDirHeaderText);
   }
   RETURN (ei);
}                                               // end 'dfsNtfsDirEntries'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Convert NTFS NTIME (64 bit) to DFS standard date/time string
/*****************************************************************************/
char *dfsNtfsTime2str                           // RET   string value
(
   NTIME              *nt,                      // IN    ptr NT time value
   char               *dtime                    // INOUT ptr to string buffer
)
{
   time_t              tm;                      // time in compiler format
   struct tm          *gm;

   ENTER();

   tm = txWinFileTime2t( nt, ntfs->TimeZone);
   if (((nt->hi != 0) || (nt->lo != 0)) && ((gm = gmtime( &tm)) != NULL))
   {
      strftime(dtime, TXMAXTM, "%Y-%m-%d %H:%M:%S", gmtime( &tm));
   }
   else                                         // invalid, before 1-1-1970
   {
      sprintf( dtime, "-%8.8x-%8.8x-", nt->hi, nt->lo);
   }
   TRACES(("Formatted date/time: %s\n", dtime));

   RETURN( dtime);
}                                               // end 'dfsNtfsTime2str'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Show NTFS Attribute type as value and descriptive string
/*****************************************************************************/
ULONG dfsNtfsShowAttrType                       // RET   un-recognized type
(
   char               *lead,                    // IN    leading text
   ULONG               atype,                   // IN    attribute type
   char               *dcol,                    // IN    descr color (ANSI)
   char               *tail                     // IN    tail text
)
{
   ULONG               rc  = NO_ERROR;
   TXTM                text;

   switch (atype)
   {
      case MA_STANDARD  : sprintf( text,  "Standard Info ");  break;
      case MA_ATTRLIST  : sprintf( text,  "Attribute list");  break;
      case MA_FILENAME  : sprintf( text,  "File Name     ");  break;
      case MA_VERSION   : sprintf( text,  "VolVers/Obj-Id");  break;
      case MA_SECURITY  : sprintf( text,  "Security Descr");  break;
      case MA_VOLNAME   : sprintf( text,  "Volume name   ");  break;
      case MA_VOLINFO   : sprintf( text,  "Volume Info   ");  break;
      case MA_DATA      : sprintf( text,  "Data Stream   ");  break;
      case MA_IDXROOT   : sprintf( text,  "Index Root    ");  break;
      case MA_DIRINDX   : sprintf( text,  "Index Alloc   ");  break;
      case MA_BITMAP    : sprintf( text,  "Alloc Bitmap  ");  break;
      case MA_SYMLINK   : sprintf( text,  "SymLnk/Reparse");  break;
      case MA_EAINFO    : sprintf( text,  "ExtAttr Info  ");  break;
      case MA_EADATA    : sprintf( text,  "ExtAttr Data  ");  break;
      case MA_PROPSET   : sprintf( text,  "Property Set  ");  break;
      case MA_LOGUSTR   : sprintf( text,  "LoggedUtStream");  break;
      case MA_ALLATTRS  : sprintf( text,  "ALL attributes");  break;
      default:            sprintf( text,  "%s-Unknown Attr ",  CBR);
         rc = DFS_ST_MISMATCH;
         break;
   }
   TxPrint( "%s%4.4X%s : %s%s%s%s", lead, atype, CNN, dcol, text, CNN, tail);
   return( rc);
}                                               // end 'dfsNtfsShowAttrType'
/*---------------------------------------------------------------------------*/



/*****************************************************************************/
// Find MFT for Root-directory, starting at some LSN in the volume
// 20171114 JvW Updated to allow input of 'desired' parentMFT (for hard links)
/*****************************************************************************/
BOOL dfsNtfsFindRootMft                         // OUT   root found
(
   ULN64               start,                   // IN    starting lsn
   BOOL                verbose,                 // IN    display mode
   BOOL                dos83,                   // IN    favor DOS8.3 shortname
   ULN64               desiredParent,           // IN    desired parent MFT, or 0
   char               *path,                    // OUT   combined path, TXMAXLN
   ULN64              *lsn                      // OUT   root MFT-LSN or NULL
)
{
   BOOL                rc  = FALSE;
   ULN64               frm = start;             // Current LSN startpoint
   ULN64               par = 0;                 // Parent  LSN found
   BOOL                parent;

   ENTER();
   TRACES(("start:0x%8.8llx  dos83:%s desiredParent:%llx\n", start, (dos83) ? "YES" : "NO ", desiredParent));

   if (verbose)
   {
      dfsX10("Searching at  LSN : ", frm, CBW, TREOLN);
      dfsProgressInit( start, ntfs->SpareLsn - start, 0, "Sector:", "searched", DFSP_STAT, 0);
   }
   if (path)
   {
      *path = '\0';
   }
   do
   {
      parent = dfsNtfsMft2Parent( frm, verbose, dos83, desiredParent, path, &par);
      if (parent)                               // follow MFT recs towards root
      {
         frm = par;
         desiredParent = 0;                     // desired ONLY at original start MFT record!
      }
      else
      {
         TRACES(("par: %llx\n", par));
         if (par < 2)
         {
            if (verbose)
            {
               if ((par == 1) && (frm > 2))     // parent LSN was no Mft!
               {
                  TxPrint("%s(invalid/deleted)%s\n", CNC, CNN);
                  dfsX10("Searching at  LSN : ", frm, CBW, TREOLN);
               }
               if ((frm % 0x800) == 0)          // signal each MiB searched
               {
                  if (TxQueryLogFile( NULL, NULL)) // if no logging, move cursor
                  {
                     TxPrint(".%s", TREOLN);
                  }
                  else
                  {
                     TxPrint("%s%s%8.8X%s%s", CL8, CBW, frm, CNN, TREOLN);
                  }
                  dfsProgressShow( frm, 0, 0, NULL);
               }
            }
            else
            {
               par = 3;                         // force loop-exit
            }
            frm++;
         }
      }
   } while ((parent || (par < 2)) && !TxAbort());

   if (verbose)
   {
      dfsProgressTerm();
      TXRNOLN();                                // EOLN when not tracing
   }
   if (par == frm)                              // Root is its own parent
   {
      if (lsn)
      {
         *lsn = par;
      }
      rc   = TRUE;
   }
   TRACES(("Found root 0x%llx for Mft %llx in '%s'\n", (lsn) ? *lsn : 0, start, path));
   BRETURN (rc);
}                                               // end 'dfsNtfsFindRootMft'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Find parent MFT for the given (DIR or FILE) MFT, including path string
// 20171114 JvW Updated to allow input of 'desired' parentMFT (for hard links)
/*****************************************************************************/
BOOL dfsNtfsMft2Parent                          // OUT   real parent found
(
   ULN64               fn,                      // IN    starting lsn
   BOOL                verbose,                 // IN    display mode
   BOOL                dos83,                   // IN    favor DOS8.3 shortname
   ULN64               parent,                  // IN    desired parent MFT, or 0
   char               *path,                    // OUT   combined path, TXMAXLN
   ULN64              *lsn                      // OUT   found parent MFT LSN
)
{
   BOOL                rc  = FALSE;             // not a valid parent
   ULONG               dr  = NO_ERROR;
   S_MFTFILE          *mft;
   ULN64               parentMft = parent;
   BYTE                st;
   TXLN                fname;                   // filename component
   BOOL                directory;
   BOOL                orphaned = FALSE;        // parent not a directory

   ENTER();
   TRACES(("fn:0x%8.8llx  dos83:%s parent:%llx\n", fn, (dos83) ? "YES" : "NO ", parent));

   *lsn = L64_NULL;                             // invalid, simulated end

   if ((dr = dfsNtfsReadFixMft( fn, &st, &mft)) == NO_ERROR)
   {
      if (path)
      {
         TRACES(( "path at start: '%s'\n", path));
      }
      directory = ((st == ST_MFTDIRB) || (st == ST_MFTDELD));
      if (dfsNtfsMft2NameInfo( mft, dos83, &parentMft, NULL, fname) == NO_ERROR)
      {
         *lsn = dfsNtfsMft2Lsn( parentMft);
         if (directory)
         {
            if (strcmp(fname, ".") == 0)        // root directory
            {
               strcpy( fname, "");              // don't show '.' in path
            }
            strcat( fname, FS_PATH_STR);        // add directory separator
         }
         else if (path && strlen( path))        // pre-pends MUST be a DIR!
         {
            sprintf(fname, "%c%12.12llX.DIR%c", FS_PATH_SEP, fn, FS_PATH_SEP);
            orphaned = TRUE;
         }
      }
      else                                      // make unique name
      {
         sprintf( fname, "-mft-%12.12llx-at-%12.12llx-", dfsNtfsLsn2Mft( fn), fn);
      }
      if (verbose)
      {
         TxPrint("%s  found MFT   LSN : 0x%12.12llX   %s short-name : %s%s",
                  TRNOLN, fn, (directory) ? "Dir." : "File", fname, TREOLN);
      }
      if (path != NULL)                         // path wanted ?
      {
         if (strlen(fname) + strlen(path) + 30 < dfsa->maxPath)
         {
            strcat( fname, path);               // append existing path
            strcpy( path, fname);               // and copy back
         }
         else                                   // try shortened version
         {
            sprintf(fname, "%12.12llX.DIR%c", fn, FS_PATH_SEP);
            if (strlen(fname) + strlen(path) + 10 < dfsa->maxPath)
            {
               strcat( fname, path);            // append existing path
               strcpy( path, fname);            // and copy back
            }
            else                                // truncate this component
            {
               TRACES(( "Path component '%s' discarded due to length limit\n", fname));
               sprintf(fname, "x%c", FS_PATH_SEP); // last resort
               if (strlen(fname) + strlen(path) < dfsa->maxPath)
               {
                  strcat( fname, path);         // append existing path
                  strcpy( path, fname);         // and copy back
               }
            }
         }
      }
      if (!orphaned && (*lsn != fn))            // different LSN, real parent
      {
         rc = TRUE;
      }
      TxFreeMem( mft);
   }
   else if ((dr == DFS_ST_MISMATCH) || (dr == DFS_BAD_STRUCTURE))
   {
      if (path && strlen(path))                 // unknown part of path
      {
         sprintf(fname, "%c%12.12llX.DIR%c", FS_PATH_SEP, fn, FS_PATH_SEP);
         if (strlen(fname) + strlen(path) < dfsa->maxPath)
         {
            strcat( fname, path);               // append existing path
            strcpy( path, fname);               // and copy back
         }
         else                                   // discard this component
         {
            TRACES(( "Path component '%s' discarded due to length limit\n", fname));
         }
      }
      if (fn == *lsn)                           // should be MFT LSN
      {
         *lsn = 1;                              // signal invalid Mft-LSN
      }
      else
      {
         *lsn = 0;                              // signal continue-search
      }
   }
   TRACES(("fn:0x%llx parentMft:0x%llx lsn:0x%llx  path: '%s'\n", fn, parentMft, *lsn, path));
   BRETURN (rc);
}                                               // end 'dfsNtfsMft2Parent'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DFS NTFS read and fixup an INDX record based on its sector number
/*****************************************************************************/
ULONG dfsNtfsReadFixIndx                        // RET   BAD_STRUCTURE
(
   ULN64               lsn,                     // IN    INDX LSN
   S_DIRINDX         **dirrec                   // OUT   INDX fixed-up record
)
{
   ULONG               rc = NO_ERROR;
   S_DIRINDX          *dir;                     // INDX record
   BYTE                st = ST_UDATA;

   ENTER();

   //- Note: this assumes INDX-records are always contiguous, they could be
   //-       at DIFFERENT clusters with a < 4096 byte clustersize!
   //-       In that case you need to read the rest by interpreting
   //-       the allocation chain, not simply the next LSN!
   //-       dfsNtfsBtree2Lsn() could return array of 8 sector addresses ...
   //- Perhaps the filesystem takes care NOT to fragment individual records ???

   if ((dir = TxAlloc( ntfs->DirRecSize, dfsGetSectorSize())) != NULL)
   {
      rc = dfsRead( lsn, ntfs->DirRecSize, (BYTE   *) dir);
      if (rc == NO_ERROR)
      {
         st = dfsIdentifySector(lsn, 0, (BYTE   *) dir);
         switch ((rc = dfsNtfsFixupStructure( (BYTE   *) dir, ntfs->DirRecSize)))
         {
            case NO_ERROR:
               switch (st)
               {
                  case ST_DIRINDX:
                     break;

                  default:
                     rc = DFS_ST_MISMATCH;      // signal type mismatch
                     break;
               }
               break;

            case DFS_BAD_STRUCTURE:
               TxPrint( "Index record at LSN : 0x%llx has bad fixup structures.\n", lsn);
               break;

            default:
               TxPrint( "Index record at LSN : 0x%llx has a fixup error for sector: %u\n", lsn, rc -1);
               rc = DFS_BAD_STRUCTURE;          // signal invalid structure
               break;
         }
      }
      if (rc == NO_ERROR)
      {
         *dirrec = dir;                         // return the DIR record
      }
      else                                      // free the memory here!
      {
         TxFreeMem( dir);
      }
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN(rc);
}                                               // end 'dfsNtfsReadFixIndx'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DFS NTFS read and fixup an MFT record based on its sector number
/*****************************************************************************/
ULONG dfsNtfsReadFixMft                         // RET   BAD_STRUCTURE
(
   ULN64               lsn,                     // IN    MFT LSN
   BYTE               *stype,                   // OUT   MFT sector-type
   S_MFTFILE         **mftrec                   // OUT   MFT fixed-up record
)
{
   ULONG               rc = NO_ERROR;
   S_MFTFILE          *mft;                     // MFT record
   BYTE                st = ST_UDATA;

   ENTER();

   //- Note: this assumes MFT-records are always contiguous, they could be
   //-       at DIFFERENT clusters with a 512 byte clustersize!
   //-       In that case you need to read the 2nd half by interpreting
   //-       the allocation chain, not simply the next LSN!
   //- This could be implemented fairly easy by using SspaceReadFilePart()
   //- using the MFT-nr and MFTrec-size to calculate offset & #sectors
   //- ==> Make an exception for MFT-0 (since it needs the MFT S_SPACE :-)


   //- Could be made conditional, if clustersize < MFTrec-size
   //- Perhaps the filesystem takes care NOT to fragment individual records ???

   //- Might need to split NtfsReadFixMft into a Read and a Fixup part
   //- and use that everywhere (read for display should NOT fixup yet :-)

   if ((mft = TxAlloc( ntfs->MftRecSize, dfsGetSectorSize())) != NULL)
   {
      rc = dfsRead( lsn, ntfs->MftRecSize, (BYTE   *) mft);
      if (rc == NO_ERROR)
      {
         st = dfsIdentifySector(lsn, 0, (BYTE   *) mft);
         switch ((rc = dfsNtfsFixupStructure( (BYTE   *) mft, ntfs->MftRecSize)))
         {
            case NO_ERROR:
               switch (st)
               {
                  case ST_MFTFILE:
                  case ST_MFTDELF:
                  case ST_MFTSECF:
                  case ST_MFTSECD:
                  case ST_MFTDIRB:
                  case ST_MFTDELD:
                  case ST_MFGHOST:
                  case ST_MFTEMPT:
                     break;

                  default:
                     if (dfstAreaP2Disk( DFSTORE, LSN_BOOTR) != LSN_BOOTR)
                     {
                        st = ST_MFTFILE;        // Fake good MFT in Area mode
                     }
                     else                       // regular FS mode, fail ...
                     {
                        rc = DFS_ST_MISMATCH;   // signal type mismatch
                     }
                     break;
               }
               break;

            case DFS_BAD_STRUCTURE:             // message only when fixups are present
               if ((mft->FixupOffset >= 4) && (mft->FixupOffset < FIXUP_AREA_SIZE))
               {
                  TxPrint( "MFT record at LSN : 0x%llx has bad fixup structures.\n", lsn);
               }
               break;

            default:
               TxPrint( "MFT record at LSN : 0x%llx has a fixup error for sector: %u\n", lsn, rc -1);
               rc = DFS_BAD_STRUCTURE;          // signal invalid structure
               break;
         }
      }
      if (rc == NO_ERROR)
      {
         *mftrec = mft;                         // return the MFT record
      }
      else                                      // free the memory here!
      {
         TxFreeMem( mft);
      }
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   *stype = st;
   TRACES(("sectortype: 0x%2.2hhx = '%c'\n", st, st));
   RETURN(rc);
}                                               // end 'dfsNtfsReadFixMft'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DFS NTFS unfix and write an MFT record based on its sector number
/*****************************************************************************/
ULONG dfsNtfsFixWriteMft                        // RET   BAD_STRUCTURE
(
   ULN64               lsn,                     // IN    MFT LSN
   S_MFTFILE          *mft                      // IN    MFT fixed-up record
)
{
   ULONG               rc = NO_ERROR;
   ULONG               nr = dfsNtfsLsn2Mft(lsn);

   ENTER();

   //- Note: this assumes MFT-records are always contiguous, see ReadFixMft()

   switch ((rc = dfsNtfsUnfixStructure( (BYTE   *) mft)))
   {
      case NO_ERROR:
         rc = dfsWrite( lsn, ntfs->MftRecSize, (BYTE   *) mft);
         if ((rc == NO_ERROR) && (nr < 4))      // write to mirror as well!
         {                                      // write failure not critical
            dfsWrite( dfsNtfsMir2Lsn(nr), ntfs->MftRecSize, (BYTE   *) mft);
         }
         break;

      default:
         rc = DFS_BAD_STRUCTURE;                // signal invalid structure
         break;
   }
   RETURN(rc);
}                                               // end 'dfsNtfsFixWriteMft'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DFS NTFS verify allocation integrity for an MFT record (normal / deleted)
/*****************************************************************************/
ULONG dfsNtfsMftLsnCheckAlloc                   // RET   check result
(
   ULN64               lsn,                     // IN    MFT LSN
   char               *options,                 // IN    result options
   BYTE               *stype,                   // OUT   sector-type
   ULN64              *totals,                  // INOUT total sectors
   ULN64              *invalids,                // INOUT invalid sectors
   ULN64              *datalsn                  // OUT   first data lsn
)
{
   ULONG               rc  = NO_ERROR;
   ULN64               mftnr;                   // corresponding MFT nr
   S_MFTFILE          *mft;
   BYTE                st = ST_UDATA;

   ENTER();

   if ((mftnr = dfsNtfsLsn2Mft(lsn)) != MFT_NULL) // validate MFT LSN
   {
      if ((rc = dfsNtfsReadFixMft( lsn, &st, &mft)) == NO_ERROR)
      {
         rc = dfsNtfsMftRecCheckAlloc( mft, options, totals, invalids, datalsn);
         TxFreeMem( mft);
      }
   }
   else
   {
      rc = DFS_PENDING;
   }
   *stype = st;
   RETURN(rc);
}                                               // end 'dfsNtfsMftLsnCheckAlloc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DFS NTFS verify allocation integrity for given MFT record
/*****************************************************************************/
ULONG dfsNtfsMftRecCheckAlloc                   // RET   check result
(
   S_MFTFILE          *mft,                     // IN    MFT record
   char               *options,                 // IN    result options
   ULN64              *totals,                  // INOUT total sectors
   ULN64              *invalids,                // INOUT invalid sectors
   ULN64              *datalsn                  // OUT   first data lsn
)
{
   ULONG               rc  = NO_ERROR;
   DFSISPACE           isp;                     // allocation space/size
   ULONG               i;                       // number alloc sections
   ULN64               j;                       // number of sectors
   ULN64               sn;                      // Data LSN
   BOOL                verbose = (strchr(options, 'v') != NULL);
   BOOL                allattr = (strchr(options, 'a') != NULL);
   BOOL                skipchk = (strchr(options, 's') != NULL);
   ULONG               clsize  = (ULONG) ntfs->ClustSize;
   USHORT              col = 0;
   BOOL                alok = FALSE;            // allocation OK
   BOOL                ldot = FALSE;            // last dot status
   ULONG               dots = 0;                // dots of same color

   ENTER();

   TRACES(( "Bm.Size: %u\n", ntfs->Bm.Size));

   if (ntfs->Bm.Size != 0)                      // allocation BitMap present ?
   {
      //- 'a' option forces allocation check on ALL attributes, otherwise
      //- just the data attribute (useful for file undelete/recovery)

      rc = dfsNtfsSpAlloc( mft, (allattr) ? MA_ALLATTRS : MA_DATA, IA_ANY, NULL, &isp);
      if ((rc == NO_ERROR) && (isp.space != NULL))
      {
         if (isp.chunks == 0)                   // resident data
         {
            *totals = (isp.byteSize >> 9) +1;   // resident size in sectors
            *invalids = 0;                      // no invalids possible :-)
         }
         else
         {
            ULN64      spaceSectors = dfsSspaceSectors( FALSE, isp.chunks, isp.space);

            *datalsn = isp.space[0].start;      // first data sector, location

            if (skipchk)
            {
               *totals = spaceSectors;          // just report size, ignore allocation
            }                                   // (will be reported as 100% OK!)
            else                                // SPACE allocation, full alloc check
            {
               #if defined (USEWINDOWING)
                  if (spaceSectors > DFSECT_BIGF)
                  {
                     txwSetSbviewStatus( "Checking allocation for a huge file ...", cSchemeColor);
                  }
               #endif
               for (i = 0; (i < isp.chunks) && !TxAbort(); i++)
               {
                  if (verbose)
                  {
                     if (++col >= dfsGetDisplayMargin())
                     {
                        TxPrint("\n");
                        col = 1;
                     }
                     TxPrint("");              // start of allocation extent
                  }
                  if (isp.space[ i].start != LSN_SPARSE) // normal extent ?
                  {
                     for ( j  = 0;              // all clusters in extent
                          (j <  isp.space[ i].size) && !TxAbort();
                           j += clsize)
                     {
                        sn = isp.space[ i].start + j;

                        switch (dfsNtfsAllocated(sn, 0, NULL, NULL))
                        {
                           case FALSE:          // must be a deleted file
                              alok = ((mft->MftFlags & AF_REC_INUSE) == 0);
                              break;

                           case TRUE:           // must be a regular file
                              alok = ((mft->MftFlags & AF_REC_INUSE) != 0);
                              break;

                           default:
                              alok = FALSE; rc = DFS_PSN_LIMIT;
                              break;
                        }
                        if (verbose)
                        {
                           //- start with 1 dot per block (4KiB) the one per 1024 (4MiB)
                           if ((alok != ldot) || ((++dots) < 33) || ((dots % 1024) == 0))
                           {
                              if (alok != ldot)
                              {
                                 ldot = alok;
                                 dots = 0;
                              }
                              if (++col >= dfsGetDisplayMargin())
                              {
                                 TxPrint("\n");
                                 col = 1;
                              }
                              TxPrint("%s%s%s", alok ? CBG : CBR,
                                                alok ? "" : "",
                                                CNN);
                           }
                        }
                        if (!alok)
                        {
                           (*invalids) += clsize;
                        }
                        (*totals) += clsize;
                     }
                  }
                  else                          // sparse extent
                  {
                     (*totals) += isp.space[ i].size * clsize;
                     if (verbose)
                     {
                        if (++col >= dfsGetDisplayMargin())
                        {
                           TxPrint("\n");
                           col = 1;
                        }
                        TxPrint( "%s%s%s", CNC, "", CNN);
                     }
                  }
               }
               if (rc == NO_ERROR)
               {
                  if (*invalids)
                  {
                     rc = DFS_BAD_STRUCTURE;
                  }
               }
               else                             // structure / PSN-limit error
               {
                  *invalids = clsize;           // to be refined
                  *totals   = *invalids;
               }
               if (verbose)
               {
                  TxPrint("\n");
                  switch (rc)
                  {
                     case NO_ERROR:
                        TxPrint("No allocation errors detected\n");
                        break;

                     case DFS_PSN_LIMIT:
                        TxPrint("Sector numbers are too large for this NTFS volume!\n");
                        break;

                     default:
                        TxPrint("Allocation error detected on %u out of %u sectors, "
                                "save/undelete unreliable!\n", *invalids, *totals);
                        break;
                  }
               }
            }
         }
      }
      TxFreeMem( isp.space);                    // non-res SPACE or res data
   }
   else
   {
      rc = DFS_BAD_STRUCTURE;
   }
   RETURN(rc);
}                                               // end 'dfsNtfsMftRecCheckAlloc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display file allocation and path info for LSN
/*****************************************************************************/
ULONG dfsNtfsFileInfo                           // RET   LSN is valid MFT rec
(
   ULN64               lsn,                     // IN    MFT LSN
   ULN64               parent,                  // IN    desired parent MFT LSN or 0
   char               *select,                  // IN    MFT select wildcard
   void               *param                    // INOUT leading text/shortpath
)
{
   ULONG               rc = NO_ERROR;
   TXLN                fpath;
   ULN64               root;
   ULN64               size      = 0;           // allocated size, sectors
   ULN64               bads      = 0;           // bad allocated sectors
   BYTE                st        = 0;
   TXLN                text;
   ULONG               percent   = 0;
   ULONG               threshold = 0;           // threshold percentage
   BOOL                isMinimum = TRUE;        // threshold is minimum value
   ULN64               location  = 0;
   TXLN                temp;                    // location text
   ULONG               minS      = 0;           // minimal size
   ULONG               maxS      = 0;           // maximal size
   char                itemType  = 0;           // type Dir, File, Browse, or ANY
   time_t              modTstamp = 0;           // Modify timestamp value, or 0
   time_t              lastModified = 0;        // Std-attribute, MOD timestamp
   char               *lead    = param;
   BOOL                verbose = (strcmp( lead, ":") != 0);
   BOOL                output;                  // no Printf, just return fpath
   BOOL                alcheck;                 // Execute a full allocation check
   ULN64               mftnr;                   // corresponding MFT nr
   S_MFTFILE          *mft;
   ULN64               fsize     = 0;
   ULN64               parentMft = 0;

   ENTER();
   TRARGS(("LSN:0x%llX, parent:0x%llx, select: '%s', lead: '%s'\n", lsn, parent, select, lead));

   if (strcmp( lead, "!") == 0)                 // Skip allocation check wanted
   {
      output  = FALSE;                          // no output
      alcheck = FALSE;                          // and no allocation check
      *lead   = 0;                              // discard 'garbage' in lead string
   }
   else                                         // Possible output/alloc-check wanted
   {
      output  = (*lead != 0);                   // output if any lead given (except '!')
      alcheck = TRUE;                           // but always an alloc check if no  '!'
   }
   TRACES(("verbose:%u output:%u alcheck:%u\n", verbose, output, alcheck));

   if ((mftnr = dfsNtfsLsn2Mft(lsn)) != L64_NULL)
   {
      if ((rc = dfsNtfsReadFixMft( lsn, &st, &mft)) == NO_ERROR)
      {
         dfsParseFileSelection( select, text, &isMinimum, &threshold, &minS, &maxS, &modTstamp, &itemType);
         if (verbose)
         {
            //- Allow skipping the allocation-check when not needed (Speedup BROWSE display)
            rc = dfsNtfsMftRecCheckAlloc( mft, (alcheck) ? "" : "s", &size, &bads, &location);
         }
         else                                   // no size/percentage info
         {
            isMinimum = TRUE; threshold = 0;    // no percentage filter
            minS      = 0;    maxS      = 0;    // no size filter
         }
         switch (st)
         {
            case ST_MFTSECF:                    // to be refined ?
            case ST_MFTSECD:                    // trace to baseMFT ?
               rc = DFS_PENDING;
               break;                           // Yes, see MFT display

            case ST_MFTDIRB:                    // normal or deleted DIR
            case ST_MFTDELD:
               if      (itemType == DFS_FS_ITEM_BROWSE) // no filtering on Directories, take all.
               {
                  strcpy( text, "");            // no wildcard
                  isMinimum = TRUE; threshold = 0;      //- no percentage filter
                  modTstamp = 0;                        //- no timestamp filter
                  minS      = 0;    maxS      = 0;      //- no size filter
               }
               else if (itemType == DFS_FS_ITEM_FILES) // filter files, discard Directories
               {
                  TRACES(("mismatch on wanted item-type being FILE\n"));
                  rc = DFS_ST_MISMATCH;         // item-type mismatch
                  break;
               }
            case ST_MFTFILE:                    // normal or deleted files
            case ST_MFTDELF:
            case ST_MFGHOST:
               if      (itemType == DFS_FS_ITEM_DIRS) // filter dirs, discard files
               {
                  TRACES(("mismatch on wanted item-type being DIR\n"));
                  rc = DFS_ST_MISMATCH;         // item-type mismatch
                  break;
               }
               if (parent != 0)                 // a desired parent specified
               {
                  parentMft = dfsNtfsLsn2Mft( parent);
                  TRACES(("desired parent MFT number: %llx\n", parentMft));
               }
               dfsNtfsFindRootMft( lsn, FALSE, (*((char *) param) == '~'),  parentMft, fpath, &root);
               if ((strlen(text) == 0) || (TxStrWicmp(fpath, text) >= 0))
               {
                  if (verbose)
                  {
                     if      (location == 0)
                     {
                        strcpy(  temp, "  resident ");
                        rc = NO_ERROR;          // allocation always OK
                     }
                     else if (location == LSN_SPARSE)
                     {
                        strcpy(  temp, "    sparse ");
                        rc = NO_ERROR;          // allocation always OK
                     }
                     else
                     {
                        sprintf( temp, "%10llX ", location);
                     }
                     percent = dfsAllocationReliability( size, bads);

                     TRACES(( "percent:%u threshold:%u size:%llu minS:%u maxS:%u\n",
                               percent, threshold, size, minS, maxS));
                  }
                  if ((BOOL)((percent >= threshold)) == isMinimum)
                  {
                     S_MFTATTR  *at;            // attribute data, within same MFT

                     if (dfsNtfsGetMftAttrRef( mft, MA_STANDARD, IA_ANY, NULL, &at) == NO_ERROR)
                     {
                        A_STANDARD *sad = (A_STANDARD *) (((BYTE *) at) + at->R.Offset);
                        lastModified = txWinFileTime2t( &sad->Modify, 0);
                     }
                     if ((lastModified > modTstamp) &&                         //- Timestamp match
                         ((size >= minS) && ((size <= maxS) || (maxS == 0))))  //- Size march
                     {
                        if (verbose)            // size and optional allocation reliability
                        {
                           if (alcheck)
                           {
                              sprintf( text, "%s%c %s%3u%%%s ", CBM, st, (rc == NO_ERROR) ?  CBG  :  CBR, percent, CNN);
                           }
                           else                 // no reliability percentage
                           {
                              sprintf( text, "%s%c     %s ", CBM, st, CNN);
                           }
                           strcat(    text, temp);
                           dfstrSize( text, CBC, size, " ");
                           dfstrBytes(text, CBZ, 0,    " "); // XATTR/EA-size
                        }
                        else
                        {
                           strcpy( text, "");   // remove any stray text, non-verbose
                        }
                        if (output)             // not totally silent?
                        {
                           strcat( lead, text);
                           TxPrint( "%s%s%s%s%s\n", lead, CBY, fpath, CNN, CGE);
                        }
                        else                    // return info in 'select'
                        {
                           DFS_PARAMS  parm;

                           if (lastModified != 0)
                           {
                              strftime( temp, TXMAXTM, "%Y-%m-%d %H:%M:%S", gmtime( &lastModified));
                           }
                           else
                           {
                              strcpy( temp, "-no-date- -no-time-");
                           }
                           TxStripAnsiCodes( text);
                           sprintf( select, "%s %s", text, temp);

                           //- Get exact bytesize from the directory-entry, via path
                           if (dfsNtfsFindPath( FALSE, 0, fpath + 1, &parm) == NO_ERROR)
                           {
                              fsize = parm.byteSize;
                           }
                           dfstrUllDot20( select, "", fsize, "");
                        }
                        rc = NO_ERROR;
                     }
                     else
                     {
                        TRACES(("mismatch on SIZE\n"));
                        rc = DFS_ST_MISMATCH;   // file-size mismatch
                     }
                  }
                  else
                  {
                     TRACES(("mismatch on PERCENTAGE\n"));
                     rc = DFS_ST_MISMATCH;      // percentage mismatch
                  }
               }
               else
               {
                  TRACES(("mismatch on WILDCARD-select\n"));
                  rc = DFS_ST_MISMATCH;         // wildcard mismatch
               }
               strcpy( param, fpath);           // return the found name
               break;

            default:
               rc = DFS_PENDING;
               break;
         }
         TxFreeMem( mft);
      }
   }
   else
   {
      rc = DFS_PENDING;                         // not an MFT
   }
   RETURN(rc);
}                                               // end 'dfsNtfsFileInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get (long) filename, related parent-MFT and RealSize from given MFT-record
// 20171114 JvW Updated to allow input of 'desired' parentMFT (for hard links)
/*****************************************************************************/
ULONG dfsNtfsMft2NameInfo                       // RET   MFT is valid MFT rec
(
   S_MFTFILE          *sd,                      // IN    base MFT rec, fixed-up
   BOOL                dos83,                   // IN    favor DOS8.3 shortname
   ULN64              *parent,                  // INOUT parent MFT number, or 0
   ULN64              *fileSize,                // OUT   RealSize bytes, or NULL
   char               *fname                    // OUT   ASCII filename string
)                                               //       maximum size: TXMAXLN
{
   ULONG               rc = DFS_NOT_FOUND;
   S_MFTATTR          *at;                      // ptr to attribute
   ULONG               sc;                      // sanity counter
   ULN64               desiredParent = *parent;

   ENTER();
   TRACES(("dos83:%s desiredParent:%llx\n", (dos83) ? "YES" : "NO", desiredParent));

   strcpy( fname, "");
   for ( sc = 0,      at = (S_MFTATTR *) ((BYTE *) sd + sd->AttribOffset);
        (sc < 30) && (at->Type != MA_END_LIST);
         sc++,        at = ((S_MFTATTR *)((BYTE *) at + at->Length)))
   {
      if (at->Type == MA_FILENAME)              // filename attribute found
      {
         U_MFTATTR    *av;                      // Attribute value data

         av = (U_MFTATTR *) ((BYTE *) at + at->R.Offset);
         TxUnic2Ascii( av->Fname.Filename, fname, av->Fname.FileNameLength);
         *parent = dfsMftPure( av->Fname.MftParentMft);
         TRACES(("Parent candidate: %llx for '%s'\n", *parent, fname));

         if (fileSize != NULL)
         {
            if (av->Fname.RealSize != 0)
            {
               *fileSize = av->Fname.RealSize;  // actual size of the file data
            }
            else                                // real size 0, use allocated
            {
               *fileSize = av->Fname.Allocated;
            }
         }
         rc = NO_ERROR;

         if ((desiredParent == 0) || (desiredParent == *parent)) // don't care, or MATCH
         {
            TRACES(("FileNameType: 0x%2.2hx\n", av->Fname.FileNameType));

            //- quit when the nametype 8.3 versus others matches, continue otherwise.
            if ((av->Fname.FileNameType == FN_DOS83) == dos83)
            {
               TRACES(("Match!\n"));
               break;
            }
         }
      }
   }
   TRACES(("Selected Parent:0x%llx fname:'%s'\n", desiredParent, fname));
   RETURN(rc);
}                                               // end 'dfsNtfsMft2NameInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DFS NTFS get S_SPACE allocation structure for attribute(s) in MFT (by lsn)
// Note: Just gets the allocation data, not data itself  (use MftLsnAttrData)
/*****************************************************************************/
ULONG dfsNtfsMftLsn2Alloc                       // RET   BAD_STRUCTURE
(
   ULN64               lsn,                     // IN    MFT LSN
   ULONG               type,                    // IN    type of attr for files
   USHORT              atid,                    // IN    attribute-id or IA_ANY
   BYTE               *mftype,                  // OUT   MFT sectortype or NULL
   DFSISPACE          *isp                      // OUT   SPACE, resident or not
)
{
   ULONG               rc  = NO_ERROR;
   ULN64               mftnr;                   // corresponding MFT nr
   S_MFTFILE          *mft;
   BYTE                st = ST_UDATA;

   ENTER();
   TRARGS(("Get S_SPACE for mft LSN: 0x%llx, attribute: %2.2x\n", lsn, type));

   isp->chunks   = 0;                           // initialize to empty
   isp->space    = NULL;
   isp->byteSize = 0;
   isp->meta     = lsn;                         // may be updated for extern-resident

   if ((mftnr = dfsNtfsLsn2Mft(lsn)) != MFT_NULL) // validate mft LSN
   {
      if ((rc = dfsNtfsReadFixMft( lsn, &st, &mft)) == NO_ERROR)
      {
         //- to be refined, may add a flag in interface to include/exclude
         //- MFT-records for deleted files (if flag -> break)

         switch (st)
         {
            case ST_MFTDELF:
            case ST_MFTDELD:
            // if (exclude_deleted)
            // {
            //    break;
            // }
            case ST_MFTFILE:
            case ST_MFTSECF:
            case ST_MFGHOST:
            case ST_MFTDIRB:
            case ST_MFTSECD:
               rc = dfsNtfsSpAlloc( mft, type, IA_ANY, NULL, isp);
               break;

            case ST_MFTEMPT:                    // no data, but RC is Ok
               break;

            default:
               rc = DFS_ST_MISMATCH;
               break;
         }
         TxFreeMem( mft);
      }
   }
   else
   {
      rc = DFS_PENDING;
   }
   if (mftype != NULL)
   {
      *mftype = st;                             // return type too
   }
   TRACES(("type: %2.2hx = '%c' chunks: %u  space at: %8.8x\n",
                   st, st, isp->chunks, isp->space));
   RETURN(rc);
}                                               // end 'dfsNtfsMftLsn2Alloc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get attribute data for MFT attribute specified by type/id (resident/non-res)
/*****************************************************************************/
ULONG dfsNtfsMftLsnAttrData                     // RET   error code
(
   ULN64               lsn,                     // IN    MFT LSN
   ULONG               type,                    // IN    type of attribute MA_
   USHORT              atid,                    // IN    attribute-id or IA_ANY
   char               *atname,                  // IN    attribute name or NULL
   ULONG               smax,                    // IN    maximum size wanted or 0
   ULONG              *size,                    // OUT   length of data returned
   void              **data                     // OUT   attribute data or NULL
)                                               //       use TxFreeMem to free
{
   ULONG               rc = NO_ERROR;
   S_MFTFILE          *mft;
   BYTE                st = ST_UDATA;

   ENTER();
   TRACES(( "MFT lsn:%8.8hx attr-type:%4.4x attr-id:%hu\n", lsn, type, atid));

   if ((rc = dfsNtfsReadFixMft( lsn, &st, &mft)) == NO_ERROR)
   {
      rc = dfsNtfsGetMftAttrData( mft, type, atid, atname, smax, size, data);
      if ((rc == NO_ERROR) && (type == MA_STANDARD) && ((mft->MftFlags & AF_DIRECTORY) != 0))
      {
         A_STANDARD   *sa = (A_STANDARD *) *data;  //- Get access to attribute byte
         sa->FileAttributes |= FATTR_DIRECTORY;    //- inject FAT-style directory flag
      }                                            //- to be used for directory display
      TxFreeMem( mft);
   }
   RETURN (rc);
}                                               // end 'dfsNtfsMftLsnAttrData'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get attribute data for MFT attribute specified by type/id (resident/non-res)
/*****************************************************************************/
ULONG dfsNtfsGetMftAttrData                     // RET   error code
(
   S_MFTFILE          *sd,                      // IN    base MFT rec, fixed-up
   ULONG               type,                    // IN    type of attribute MA_
   USHORT              atid,                    // IN    attribute-id or IA_ANY
   char               *atname,                  // IN    attribute name or NULL
   ULONG               smax,                    // IN    maximum size wanted or 0
   ULONG              *size,                    // OUT   length of data returned
   void              **data                     // OUT   attribute data or NULL
)                                               //       use TxFreeMem to free
{
   ULONG               rc = NO_ERROR;
   DFSISPACE           isp;                     // allocation space/size
   ULONG               sectors = 0;
   BYTE               *buffer  = NULL;          // allocated data buffer

   ENTER();
   TRACES(( "MFT seq:%4.4hx  attr-type:%4.4x  attr-id:%hu\n", sd->Sequence, type, atid));

   *size = 0;                                   // initialize to no data

   rc = dfsNtfsSpAlloc( sd, type, atid, atname, &isp);
   if ((rc == NO_ERROR) && (isp.space != NULL)) // some data returned
   {
      if (isp.chunks == 0)                      // resident data, already
      {                                         // allocated exact size
         *size = (ULONG) isp.byteSize;          // always return length
         if (data != NULL)
         {
            *data = isp.space;
            TRHEXS( 70, isp.space, min(0x80, isp.byteSize), "Resident Attribute RAW data");
         }
         else                                   // free space, if any
         {
            TxFreeMem( isp.space);
         }
      }
      else                                      // non-resident, replace sSpace
      {                                         // by the real attribute data
         if ((smax == 0) ||                     // no limit specified or
             (smax >= isp.byteSize))            // not larger than wanted
         {
            sectors = dfsSspaceSectors( FALSE, isp.chunks, isp.space);
            *size   = (ULONG) isp.byteSize;     // saved attribute size
         }
         else
         {
            sectors = smax    / dfsGetSectorSize(); // rounded DOWN to sectors!
            *size   = sectors * dfsGetSectorSize(); // net returned size
         }
         if (sectors != 0)                      // any data present ?
         {
            if ((buffer = TxAlloc( sectors, dfsGetSectorSize())) != NULL)
            {
               rc = dfsSspaceReadFilePart( isp.chunks, isp.space, 0, sectors, buffer);
               TRHEXS( 70, buffer, 0x80, "Non-Resident Attribute RAW data");
            }
         }
         if (data != NULL)
         {
            *data = buffer;
         }
         else                                   // free buffer, if any
         {
            TxFreeMem( buffer);                 // free the buffer, if any
         }
         TxFreeMem( isp.space);                 // free the S_SPACE, if any
      }
   }
   TRACES(("size: %u  data *: %8.8x\n", *size, *data));
   RETURN (rc);
}                                               // end 'dfsNtfsGetMftAttrData'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get reference to attribute in MFT specified by type/id/name   (res/non-res)
// Does NOT support attributes located in secondary MFT records! (DFS_PENDING)
/*****************************************************************************/
ULONG dfsNtfsGetMftAttrRef                      // RET   NO_ERROR, DFS_PENDING
(                                               //       or DFS_NOT_FOUND
   S_MFTFILE          *sd,                      // IN    base MFT rec, fixed-up
   ULONG               type,                    // IN    type of attribute MA_
   USHORT              atid,                    // IN    attribute id or IA_ANY
   char               *atname,                  // IN    attribute name or NULL
   S_MFTATTR         **ref                      // OUT   attribute ref or NULL
)
{
   ULONG               rc = DFS_NOT_FOUND;
   S_MFTATTR          *at;                      // ptr to attribute
   ULONG               sc;                      // sanity counter

   ENTER();
   TRACES(( "MFT seq:%4.4hx  attr-type:%4.4x  attr-id:%hu attr-name:'%s'\n",
             sd->Sequence, type, atid, (atname) ? atname : "-none-"));

   *ref = NULL;
   for ( sc = 0,      at = (S_MFTATTR *) ((BYTE *) sd + sd->AttribOffset);
        (sc < 30) && (at->Type != MA_END_LIST) && (rc == DFS_NOT_FOUND);
         sc++,        at = ((S_MFTATTR *)((BYTE *) at + at->Length)))
   {
      TRACES(("type: 0x%8.8x  at->Type: %8.8x  atid:%4.4hx at->Ident:%4.4hx\n",
               type,           at->Type,         atid,       at->Ident));

      if (((type   == MA_ALLATTRS) || (type == at->Type ))   &&
          ((atid   == IA_ANY)      || (atid == at->Ident))   &&
          ((atname == NULL)        || (dfsNtfsAttrNameMatch( at, atname))))
      {
         TRACES(( "Attribute found, offset: %3.3x  type %4.4x  id:%hu\n",
                   (ULONG)((BYTE *) at - (BYTE *) sd), at->Type, at->Ident));
         *ref = at;
         rc = NO_ERROR;
      }
      else if (at->Type == MA_ATTRLIST)         // attribute list found
      {
         rc = DFS_PENDING;                      // could be in other MFT
      }
   }
   RETURN (rc);
}                                               // end 'dfsNtfsGetMftAttrRef'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Update an attribute in MFT                             (resident/non-res)
// Modify existing or create new, MFT-part only (non-resident is in runlist)
/*****************************************************************************/
ULONG dfsNtfsUpdateMftAttr                      // RET   NO_ERROR, DFS_PENDING
(                                               //       or DFS_NOT_FOUND
   S_MFTFILE          *sd,                      // IN    base MFT rec, fixed-up
   S_MFTATTR          *attr                     // IN    attribute reference
)
{
   ULONG               rc = DFS_NOT_FOUND;
   ULONG               type = attr->Type;       // type of attribute MA_
   USHORT              atid = attr->Ident;      // attribute-id
   S_MFTATTR          *at;                      // ptr to attribute

   ENTER();
   TRACES(( "MFT seq:%4.4hx  attr-type:%4.4x  attr-id:%hu\n",
             sd->Sequence, type, atid));

   if ((rc = dfsNtfsGetMftAttrRef( sd, type, atid, NULL, &at)) == NO_ERROR)
   {
      //- to be refined, updated found attribute in MFT with new
      //- data from supplied attribute. Note: size may have changed!
   }
   RETURN (rc);
}                                               // end 'dfsNtfsUpdateMftAttr'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get/Set NTFS Volume information in the $Volume special file
/*****************************************************************************/
ULONG dfsNtfsGetSetVolumeInfo                   // RET   function result
(
   USHORT             *flags,                   // INOUT Volume flags (dirty)
   USHORT              mask,                    // IN    flags update mask
   char               *volname,                 // INOUT Volume name
   BOOL                setname,                 // IN    Set volume name (NI)
   double             *version,                 // OUT   NTFS volume version
   BOOL                setvers                  // IN    Set volume version
)                                               //       not implemented (NI)
{
   ULONG               rc = NO_ERROR;           // function return
   BYTE                st = ST_UDATA;
   S_MFTFILE          *mft;                     // Volume mft record
   S_MFTATTR          *at;                      // Attribute
   BOOL                updated = FALSE;         // MFT data updated

   ENTER();
   TRACES(( "flags:%4.4hx mask:%4.4x volname*:%8.8x setname:%s version*:%8.8x\n",
            *flags,       mask,       volname,       (setname) ? "YES" : "NO", version));

   if ((rc = dfsNtfsReadFixMft( dfsNtfsMft2Lsn( MFT_VOLUME),
                                &st, &mft)) == NO_ERROR)
   {
      if ((rc = dfsNtfsGetMftAttrRef( mft, MA_VOLINFO, IA_ANY, NULL, &at)) == NO_ERROR)
      {
         if ((at->DataExtern == FALSE) && (at->Type == MA_VOLINFO))
         {
            A_VOLINFO          *info;           // Volume Attribute data
            USHORT              volflags;       // Volume flags

            info = (A_VOLINFO *) (((BYTE *) at) + at->R.Offset);

            if (mask != 0)                      // Set/Reset flags using mask
            {
               volflags = info->VolFlags;
               info->VolFlags = ((*flags & mask) | (volflags & ~mask));
               updated  = (volflags != info->VolFlags);

               TRACES(( "volflags: %4.4hx => %4.4hx\n", volflags, info->VolFlags));
            }
            *flags = info->VolFlags;            // final value in record

            if (version != NULL)
            {
               *version = (double) (info->MajorVer) +
                          (double) (info->MinorVer) / 10.0;
            }
         }
         else
         {
            rc = DFS_BAD_STRUCTURE;
         }
      }
      else
      {
         TxPrint("The $Volume MFT record does NOT contain a 0x70 Volume-info attribute!\n");
      }

      if ((rc == NO_ERROR) && (volname != NULL))
      {
         if (dfsNtfsGetMftAttrRef( mft, MA_VOLNAME, IA_ANY, NULL, &at) == NO_ERROR)
         {
            if ((at->DataExtern == FALSE) && (at->Type == MA_VOLNAME))
            {
               USHORT *name;

               name = (USHORT *) (((BYTE *) at) + at->R.Offset);

               if (setname)                     // update of name requested
               {
                  int   movesize  = (ntfs->MftRecSize * dfsGetSectorSize()) -
                                    ((BYTE *) at - (BYTE *) mft) - 0x58;  //- size of rest, for max volname size
                  ULONG newlength = ((2 * strlen( volname)) + 7) + 0x18;  //- including attr-header, prepare modulo
                  newlength -= (newlength % 8);                           //- rounded up to modulo 8 attr-size

                  TRACES(("movesize: %d  newlength: 0x%2.2x\n", movesize, newlength));

                  memmove(((BYTE *) at) + newlength, ((BYTE *) at) + at->Length, movesize); // overlap!

                  //- then write Unicode-converted name into the VolumeName attribute
                  at->Length = newlength;
                  at->R.Size = strlen( volname) * 2;   //- nr of bytes, unicode
                  memset( name, 0, at->R.Size);        //- clean out with ZEROs first
                  TxAscii2Unic( volname, name, strlen( volname));

                  updated = TRUE;
               }
               else                             // Get the Volume label
               {
                  TxUnic2Ascii( name, volname, (USHORT)  (at->R.Size / 2));
               }
            }
            else
            {
               rc = DFS_BAD_STRUCTURE;
            }
         }
         else
         {
            TxPrint("The $Volume MFT record does NOT contain a 0x60 Volume-name attribute!\n");
         }
      }

      if ((rc == NO_ERROR) && (updated))        // write $Volume MFT back
      {
         rc = dfsNtfsFixWriteMft( dfsNtfsMft2Lsn( MFT_VOLUME), mft);

         if (rc == NO_ERROR)
         {
            if (setname)
            {
               TxPrint( "New  Volume label : '%s' written to $Volume MFT-record\n", volname);
            }
         }

         //- to be refined, should last-update timestamp be updated too ?
         //- if so, should be handled by FixWriteMft
      }
      TxFreeMem( mft);
   }
   RETURN (rc);
}                                               // end 'dfsNtfsGetSetVolumeInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get/Set NTFS LOG status information from the $LogFile special file
/*****************************************************************************/
ULONG dfsNtfsGetSetLogStatus                    // RET   function result
(
   USHORT             *flags,                   // INOUT LOG flags (unmounted)
   USHORT              mask                     // IN    flags update mask
)                                               //       not implemented (NI)
{
   ULONG               rc = NO_ERROR;           // function return
   BYTE                st = ST_UDATA;
   S_MFTFILE          *mft;                     // Volume mft record
   BOOL                updated = FALSE;         // MFT data updated

   ENTER();
   TRACES(( "flags:%4.4hx mask:%4.4x\n", *flags, mask));

   if ((rc = dfsNtfsReadFixMft( dfsNtfsMft2Lsn( MFT_LOGFILE),
                                &st, &mft)) == NO_ERROR)
   {
      S_LOGRSTR       *rstr;                    // Logfile restart record
      ULONG            atSize;

      //- to be refined, only read first sector for now (no update anyway)

      if ((rc = dfsNtfsGetMftAttrData( mft, MA_DATA, IA_ANY, NULL, dfsGetSectorSize(),
                                       &atSize, (void **) &rstr)) == NO_ERROR)
      {
         if ((atSize != 0) && (rstr != NULL) &&
             (rstr->LraOffset < atSize))         // avoid traps on bad contents
         {
            S_LRAREA     *area = (S_LRAREA *) (((BYTE *) rstr) + rstr->LraOffset);
            USHORT        lraflags;             // Log RestartArea flags

            lraflags = area->LraFlags;
            if (mask != 0)                      // Set/Reset flags using mask
            {
               area->LraFlags = ((*flags & mask) | (lraflags | ~mask));
               updated = (*flags != area->LraFlags);
            }
            *flags = area->LraFlags;            // final value in record
         }
         else
         {
            rc = DFS_BAD_STRUCTURE;
         }
         TxFreeMem( rstr);
      }

      if ((rc == NO_ERROR) && (updated))        // write RestartArea's back
      {
         //- rc = dfsNtfsFixWriteMft( dfsNtfsMft2Lsn( MFT_LOGFILE), mft);
         //- to be refined, write RestartArea's (first 2 clusters, 8K)
         //- Using the dfsSspaceWriteFilePart with the DFSISPACE data
         //- and perhaps update timestamps in the MFT itself ?
      }
      TxFreeMem( mft);
   }
   TRACES(( "flags:%4.4hx\n", *flags));
   RETURN (rc);
}                                               // end 'dfsNtfsGetSetLogStatus'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Find specific attribute in an MFT, return S_SPACE allocation data for a
// non-resident attribute or allocated copy of the data for a resident one
// Handles ATTRIBUTE lists including referenses to OTHER MFT records too
/*****************************************************************************/
ULONG dfsNtfsSpAlloc                            // RET   error code
(
   S_MFTFILE          *sd,                      // IN    base MFT rec, fixed-up
   ULONG               type,                    // IN    type of attribute MA_
   USHORT              atid,                    // IN    attribute-id or IA_ANY
   char               *atname,                  // IN    attribute name or NULL
   DFSISPACE          *isp                      // OUT   SPACE, resident or not
)                                               //       or resident data area
{
   ULONG               rc = NO_ERROR;
   S_MFTATTR          *at;                      // ptr to attribute
   ULONG               sc;                      // sanity counter

   ENTER();
   TRACES(( "MFT seq:%4.4hx  attr-type:%4.4x  attr-id:%hu  attr-name: '%s'\n",
             sd->Sequence, type, atid, (atname) ? atname : "-none-"));
   TRHEXS( 70, sd, 0xc0, "S_MFTFILE  MFT record");

   isp->chunks   = 0;                           // initialize to empty
   isp->space    = NULL;
   isp->byteSize = 0;

   for ( sc = 0,      at = (S_MFTATTR *) ((BYTE *) sd + sd->AttribOffset);
        (sc < 30) && (at->Type != MA_END_LIST) && (rc == NO_ERROR);
         sc++,        at = ((S_MFTATTR *)((BYTE *) at + at->Length)))
   {
      if (at->Type == MA_ATTRLIST)              // attribute list found
      {
         break;
      }
   }
   if ((type     != MA_ATTRLIST) &&             // list itself NOT listed :-)
       (at->Type == MA_ATTRLIST))               // attribute list found
   {
      ULONG            this;                    // size of this element
      ULONG            done;
      ULN64            base = MFT_NULL;         // base mft record nr
      ULN64            emft;                    // element MFT number
      A_ATTRELEM      *el;
      DFSISPACE        nsp;                     // new space
      U_MFTATTR       *av;                      // Attribute value data
      ULONG            atSize;                  // attribute data size

      if ((rc = dfsNtfsGetMftAttrData( sd, MA_ATTRLIST, at->Ident, NULL, 0,
                                       &atSize, (void **) &av)) == NO_ERROR)
      {
         if ((at->DataExtern) &&                // attrlist itself non-resident
             (type == MA_ALLATTRS))             // want all; add as first one
         {
            rc = dfsNtfsSpNonRes( sd, MA_ATTRLIST, at->Ident, NULL, isp);
         }
         TRACES(( "attr-list has been read, now read each listed attribute ...\n"));
         TRHEXS( 70, at, atSize, "Total list: MA_ATTRLIST");
         for ( done = 0,          sc = 0;
              (done < atSize) && (rc == NO_ERROR);
               done += this,      sc++)
         {
            el   = (A_ATTRELEM *) (((char *) av) + done);
            emft = dfsMftPure( el->MftRecordNr);
            this = el->RecLength;

            TRHEXS( 70, el, this, "List element: MFT attribute");
            TRACES(("Type:%4hx  done:%4x  this:%4x  atSize:%4x\n", el->Type, done, this, atSize));

            if ((this != 0)               &&
                (this <= (atSize - done)) &&
                 sc   <  NTFS_SANE_ATTRLIST)
            {
               if (base == MFT_NULL)            // not registred yet
               {
                  base = emft;                  // remember base
               }

               TRACES(("type: 0x%8.8x  el->Type: %8.8x  atid:%4.4hx el->Ident:%4.4hx\n",
                        type,           el->Type,         atid,       el->Ident));

               if (((type   == MA_ALLATTRS) || (type == el->Type ))   &&
                   ((atid   == IA_ANY)      || (atid == el->Ident))   &&
                   ((atname == NULL)        || (dfsNtfsElemNameMatch( el, atname))))
               {
                  if (emft != base)             // located in other MFT!
                  {
                     rc = dfsNtfsSpExtern( emft, el->Type, el->Ident, &nsp);
                  }
                  else                          // same MFT, already loaded
                  {
                     rc = dfsNtfsSpNonRes( sd, el->Type, el->Ident, atname, &nsp);
                  }
                  if (rc == NO_ERROR)
                  {
                     if ((nsp.chunks != 0) && (nsp.space != NULL)) //- join new with rest
                     {
                        rc = dfsSspaceJoin( isp->chunks,  isp->space, nsp.chunks, nsp.space,
                                           &isp->chunks, &isp->space);
                     }
                     else if ((nsp.chunks == 0) &&  (nsp.space != NULL))    //- new is resident
                     {
                        if ((isp->chunks == 0) && (isp->space == NULL) &&   //- nothing yet
                            (type != MA_ALLATTRS)) //- don't want any resident for ALLATTRS
                        {
                           isp->space    = nsp.space;     //- attach allocated data
                           isp->byteSize = nsp.byteSize;
                           isp->meta     = nsp.meta;
                           break;                         //- ignore rest of attributes!
                        }
                        else                              //- resident after something else
                        {
                           TRACES(( "discarding resident data!\n"));
                           TxFreeMem( nsp.space);
                        }
                     }
                  }
               }
            }
            else
            {
               rc = DFS_BAD_STRUCTURE;
            }
         }
         TxFreeMem( av);                        // free attribute data
      }
   }
   else                                         // simple MFT, read non-res
   {
      rc = dfsNtfsSpNonRes( sd, type, atid, atname, isp);
   }
   TRACES(("chunks: %u  bytesize:0x%llx  space *:%8.8x  meta:%8.8x offset:%hu\n",
            isp->chunks, isp->byteSize, isp->space, isp->meta, isp->offset));
   RETURN (rc);
}                                               // end 'dfsNtfsSpAlloc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Find attribute in OTHER (external) MFT, return S_SPACE allocation data for
// a non-resident attribute or allocated copy of the data for a resident one
/*****************************************************************************/
ULONG dfsNtfsSpExtern                           // RET   error code
(
   ULONG               mftnr,                   // IN    MFT record number
   ULONG               type,                    // IN    type of attribute MA_
   USHORT              atid,                    // IN    attribute-id or IA_ANY
   DFSISPACE          *isp                      // OUT   SPACE, resident or not
)
{
   ULONG               rc  = NO_ERROR;
   ULN64               lsn = dfsNtfsMft2Lsn( mftnr);
   S_MFTFILE          *mft;

   ENTER();
   TRACES(( "MFT:%8.8x  attr-type:%4.4x  attr-id:%hu\n", mftnr, type, atid));

   isp->chunks   = 0;                           // initialize to empty
   isp->space    = NULL;
   isp->byteSize = 0;

   if ((mft = TxAlloc( ntfs->MftRecSize, dfsGetSectorSize())) != NULL)
   {
      rc = dfsRead( lsn, ntfs->MftRecSize, (BYTE   *) mft);
      if (rc == NO_ERROR)
      {
         switch ((rc = dfsNtfsFixupStructure( (BYTE   *) mft, ntfs->MftRecSize)))
         {
            case NO_ERROR:
               //- Note: attribute in this other MFT always identified by type/atid
               if ((rc = dfsNtfsSpNonRes( mft, type, atid, NULL, isp)) == NO_ERROR)
               {
                  isp->meta = dfsNtfsMft2Lsn( mftnr);
               }
               break;

            case DFS_BAD_STRUCTURE:
               TxPrint( "Related MFT 0x%x at LSN 0x%llx has bad fixup structures.\n", mftnr, lsn);
               break;

            default:
               rc = DFS_ST_MISMATCH;            // signal invalid structure
               TxPrint("related MFT 0x%x at LSN 0x%llx has a fixup error for sector: %u\n", mftnr, lsn, rc -1);
               break;
         }
      }
      TxFreeMem( mft);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   TRACES(("chunks: %u  bytesize: 0x%llx  space *: %8.8x\n", isp->chunks, isp->byteSize, isp->space));
   RETURN (rc);
}                                               // end 'dfsNtfsSpExtern'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Find specific attribute in an MFT, return S_SPACE allocation data for a
// non-resident attribute or allocated copy of the data for a resident one
// Handles simple resident or non-resident attributes in THIS MFT record
/*****************************************************************************/
ULONG dfsNtfsSpNonRes                           // RET   error code
(
   S_MFTFILE          *sd,                      // IN    base MFT rec, fixed-up
   ULONG               type,                    // IN    type of attribute MA_
   USHORT              atid,                    // IN    attribute-id or IA_ANY
   char               *atname,                  // IN    attribute name or NULL
   DFSISPACE          *isp                      // OUT   SPACE, resident or not
)
{
   ULONG               rc = NO_ERROR;
   S_MFTATTR          *at;                      // ptr to attribute
   USHORT              size;
   ULONG               info;                    // combined ST_ and MA_ info
   BYTE                st;                      // MFT sector type (File/Dir)
   ULONG               extents = 0;             // total extents
   S_SPACE            *sp      = NULL;          // total S_SPACE / res data
   ULONG               sc;                      // sanity counter

   ENTER();
   TRACES(( "MFT seq:%4.4hx  attr-type:%4.4x  attr-id:%hu  attr-name:'%s'\n",
             sd->Sequence, type, atid, (atname) ? atname : "-none-"));

   isp->chunks   = 0;                           // initialize to empty
   isp->space    = NULL;
   isp->byteSize = 0;

   for ( sc = 0,      at = (S_MFTATTR *) ((BYTE *) sd + sd->AttribOffset);
        (sc < 30) && (at->Type != MA_END_LIST) && (rc == NO_ERROR);
         sc++,        at = ((S_MFTATTR *)((BYTE *) at + at->Length)))
   {
      TRACES(("type: 0x%8.8x  at->Type: %8.8x  atid:%4.4hx at->Ident:%4.4hx\n",
               type,           at->Type,         atid,       at->Ident));

      if (((type   == MA_ALLATTRS) || (type == at->Type ))   &&
          ((atid   == IA_ANY)      || (atid == at->Ident))   &&
          ((atname == NULL)        || (dfsNtfsAttrNameMatch( at, atname))))
      {
         TRACES(( "Attribute found, offset: %3.3x  type %4.4x  id:%hu\n",
                   (ULONG)((BYTE *) at - (BYTE *) sd), at->Type, at->Ident));

         if (at->DataExtern)                    // non-resident, allocate and
         {                                      // copy allocation structure
            ULONG      ext;                     // new extents
            S_SPACE   *new;                     // new S_SPACE

            st   = dfsIdentifySector( dfsNtfsClust2Lsn( dfsa->boot->nt.MftClus), 0, (BYTE   *) sd);
            switch (st)
            {
               case ST_MFTFILE:
               case ST_MFTSECF: info = STIB_MFTFILE + (at->Type >> 4); break;
               case ST_MFTDIRB:
               case ST_MFTSECD: info = STIB_MFTDIRB + (at->Type >> 4); break;
               default:         info = st;                             break;
            }

            TRACES(("STMA st: 0x%2.2hx = '%c' at->Type: 0x%2.2hx  info: %8.8x\n",
                     st, st, at->Type, info));

            size = at->Length - at->N.Offset;

            if (at->N.SegFirst <= at->N.SegLast)
            {
               rc = dfsNtfsAllocChain( ((BYTE *) at) + at->N.Offset,
                                                       at->N.SegFirst,
                                                       at->N.SegLast - at->N.SegFirst +1,
                                                       size, info, &ext, &new);
            }
            else
            {
               TxPrint("\n   Runlist  error : Last cluster LCN < first, on MFT seq: 0x%4.4hx\n", sd->Sequence);
               rc = DFS_BAD_STRUCTURE;
            }

            if ((rc == NO_ERROR) && (ext != 0) && (new != NULL))
            {                                   // EXACT size for SaveAs (with MA_DATA only!)
               isp->byteSize = at->N.Size;      // external attr size (filesize, if _DATA)
               rc = dfsSspaceJoin( extents, sp, ext, new, &extents, &sp);
            }
            if (type != MA_ALLATTRS)            // no join for all types ?
            {
               break;                           // specific type found (speed)
            }
         }
         else if (type != MA_ALLATTRS)          // allocate and copy resident (incl empty)
         {
            BYTE *atdata = ((BYTE *) at) + at->R.Offset;

            if ( (atdata - ((BYTE *) sd) + at->R.Size) <= (ntfs->MftRecSize * dfsGetSectorSize()))
            {
               if ((sp = TxAlloc( at->R.Size + 1, 1)) != NULL) // extra byte, for empty files
               {
                  if (at->R.Size > 0)
                  {
                     memcpy( sp, atdata, at->R.Size);
                  }
                  isp->byteSize = at->R.Size;
                  isp->offset   = (USHORT) (atdata - ((BYTE *) sd));
               }
               else
               {
                  rc = DFS_ALLOC_ERROR;
               }
            }
            else                                // extends beyond MFT!
            {
               TxPrint("\n   MFT attr error : Resident extends beyond MFT, on MFT seq: 0x%4.4hx\n", sd->Sequence);
               rc = DFS_BAD_STRUCTURE;
            }
            break;                              // specific type found (speed)
         }
      }
   }
   isp->chunks = extents;
   isp->space  = sp;

   TRACES(("chunks: %u  offset: %hu   bytesize: 0x%llx  space *: %8.8x\n",
            isp->chunks, isp->offset,  isp->byteSize,    isp->space));
   RETURN (rc);
}                                               // end 'dfsNtfsSpNonRes'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Compare specified name with name for an MFT attribute, if any
/*****************************************************************************/
BOOL dfsNtfsAttrNameMatch                       // RET   name matches attribute
(
   S_MFTATTR          *at,                      // IN    reference to attribute
   char               *name                     // IN    ASCII attribute name
)
{
   BOOL                rc = FALSE;

   if (name && strlen(name))                    // valid name specified
   {
      if (at->NameLen != 0)                     // and attribute name present
      {
         char         *ucn = (char *) at + at->Offset;

         rc = (TxUnicStrncmp(ucn, name, strlen(name)) == 0);
      }
   }
   return (rc);
}                                               // end 'dfsNtfsAttrNameMatch'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Compare specified name with name for an attribute-element, if any
/*****************************************************************************/
BOOL dfsNtfsElemNameMatch                       // RET   name matches attribute
(
   A_ATTRELEM         *el,                      // IN    ref to attribute element
   char               *name                     // IN    ASCII attribute name
)
{
   BOOL                rc = FALSE;

   if (name && strlen(name))                    // valid name specified
   {
      if (el->NameLen != 0)                     // and attribute name present
      {
         char         *ucn = (char *) &(el->Name);

         rc = (TxUnicStrncmp(ucn, name, strlen(name)) == 0);
      }
   }
   return (rc);
}                                               // end 'dfsNtfsElemNameMatch'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create S_SPACE structure for an NTFS Runlist including sparse/compr support
/*****************************************************************************/
ULONG dfsNtfsAllocChain                         // RET   error code
(
   BYTE               *runlist,                 // IN    runlist reference
   ULONG               virtual,                 // IN    1st virtual cluster
   ULONG               clusters,                // IN    clusters in fragment
   USHORT              length,                  // IN    length of run-list
   ULONG               info,                    // IN    additional SPACE info
   ULONG              *chunks,                  // OUT   nr of space entries
   S_SPACE           **space                    // OUT   space allocation
)
{
   ULONG               rc = NO_ERROR;
   ULONG               extents = 0;             // nr of extents
   S_SPACE            *sp      = NULL;
   BYTE               *r   = runlist;           // ptr to runlist byte
   ULONG               lcn = 0;                 // Logical CN on disk
   ULONG               cl  = 0;                 // nr of clusters total
   ULONG               cn;                      // nr of clusters in fragment
   LONG                on;                      // offset from last LCN
   ULONG               sm;                      // sign-extend mask
   USHORT              cb;                      // byte count clusters
   USHORT              ob;                      // byte count offset
   USHORT              i;                       // byte counter
   ULONG               ext;                     // extent counter

   ENTER();
   TRARGS(("Runlist bytes: %hu, first cl: %8.8x  clusters: %8.8x\n",
            length, virtual, clusters));

   while ((((USHORT)(r - runlist)) < length) && // handle runlist until length
             (*r != 0x00))                      // or termination byte found
   {
      cb  = ((*r) & 0x0f);                      // bytes in fragment-clusters
      ob  = ((*r) & 0xf0) >> 4;                 // bytes in fragment-offset

      for ( i = 0, cn = 0;                      // read all cluster-bytes
           (i < cb) && (((USHORT)(r - runlist)) < length);
            i++)
      {
         cn += ((ULONG) (*(++r))) << (8 * i);   // read cl-byte and advance
      }

      for ( i = 0;                              // read all offset-bytes
           (i < ob) && (((USHORT)(r - runlist)) < length);
            i++)
      {
         ++r;                                   // just skip them
      }
      extents++;                                // count the extent
      cl += cn;                                 // total nr of clusters sofar
      r++;                                      // advance to next fragment
      TRACES(( "handled: %hu bytes, extents:% 3u  clusters:% 10u = %8.8x\n",
                ((USHORT)(r - runlist)), extents, cl, cl));
   }
   if (cl != clusters)
   {
      TxPrint("   Runlist  error : %8.8x clusters found, %8.8x expected\n",
                  cl, clusters);
   }
   //-   else                              // ignore errors, use partial info
   if (extents != 0)                       // allocate enough SPACE
   {
      sp = TxAlloc(extents, sizeof(S_SPACE));
      if (sp != NULL)
      {
         for (r = runlist, cl = 0, lcn = 0, ext = 0; ext < extents; ext++)
         {
            cb  = ((*r) & 0x0f);                // bytes in fragment-clusters
            ob  = ((*r) & 0xf0) >> 4;           // bytes in fragment-offset

            for ( i = 0, cn = 0;                // read all cluster-bytes
                 (i < cb) && (((USHORT)(r - runlist)) < length);
                  i++)
            {
               cn += ((ULONG) (*(++r))) << (8 * i); // read cl-byte and advance
            }

            if (ob != 0)                        // not a sparse run element ?
            {
               for ( i = 0, on = 0, sm = 0;     // read all offset-bytes
                    (i < ob) && (((USHORT)(r - runlist)) < length);
                     i++)
               {
                  on += ((ULONG) (*(++r))) << (8 * i); // read offset-byte and advance
                  sm += ((ULONG) ( 0xff )) << (8 * i); // update corresponding mask
               }
               if (*r > 0x7f)                   // indicates negative offset
               {
                  on |= ~sm;                    // sign extend with mask
               }

               lcn += on;                       // start-LCN for this fragment

               sp[ext].start = dfsNtfsClust2Lsn( lcn);
            }
            else                                // sparse/compressed element
            {
               sp[ext].start = LSN_SPARSE;      // indicate no data on disk
            }
            sp[ext].size  = cn             * ntfs->ClustSize;
            sp[ext].index = (virtual + cl) * ntfs->ClustSize;
            sp[ext].info  = info;

            cl += cn;                           // total nr of clusters sofar
            r++;                                // advance to next fragment
         }
      }
   }
   if (sp != NULL)
   {
      TRINIT(70);
         dfsSspaceDisplay( SD_DEFAULT | SD_ALLINFO, extents, sp);
      TREXIT();
   }
   else if (extents > 0)
   {
      rc = DFS_ALLOC_ERROR;
   }
   *space  = sp;
   *chunks = extents;
   TRARGS(("Clusters found: %8.8x, chunks: %u\n", cl, extents));
   RETURN (rc);
}                                               // end 'dfsNtfsAllocChain'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create a standard NTFS RunList for an S_SPACE array, incl sparse/compressed
/*****************************************************************************/
ULONG dfsNtfsMkRunList                          // RET   rc
(
   ULONG               chunks,                  // IN    nr of space entries
   S_SPACE            *space,                   // IN    space allocation
   USHORT             *length,                  // OUT   bytes in new runlist
   BYTE              **runlist                  // OUT   allocated list or NULL
)
{
   ULONG               rc = NO_ERROR;
   ULONG               asize = chunks * 13 +8;  // worst case maximum size
   ULONG               chunk;
   BYTE               *rl;

   ENTER();
   TRACES(("Chunks: %u, space: %8.8X\n", chunks, space));

   if (space != NULL)
   {
      if ((rl = TxAlloc( asize, 1)) != NULL)
      {
         USHORT        len = 0;                 // #bytes in list
         LLONG         lcn = 0;                 // logical lcn on disk
         LLONG         ncn;                     // new cluster number
         LLONG         on;                      // offset from last LCN
         ULONG         cl;                      // #clusters in run
         USHORT        cb;                      // byte count clusters
         USHORT        ob;                      // byte count offset
         BYTE          bt = 0;                  // byte value to add

         TRINIT(70);
            dfsSspaceDisplay( SD_DEFAULT, chunks, space);
         TREXIT();

         //- to be refined, when cl and .size becomes 64-bit (alloc 13 => 17)
         for (chunk = 0; (chunk < chunks) && space; chunk++)
         {
            cl  = space[chunk].size  / ntfs->ClustSize;
            for ( cb = 0;
                 (cb < sizeof(ULONG)) && ((cl != 0) || (bt > 0x7f));
                  cb++)
            {
               bt = cl & 0xff;
               rl[ len + cb +1 ] = bt;          // copy least-significant byte
               cl >>= 8;                        // and divide to get remainder
            }

            if (space[chunk].start != LSN_SPARSE) // not a sparse element ?
            {
               ncn    = space[chunk].start / ntfs->ClustSize;
               on     = ncn - lcn;
               lcn    = ncn;                    // next reference

               if (on > 0)                      // positive offset
               {
                  for ( ob = 0, bt = 0xff;
                       (ob < sizeof(LLONG)) &&  ((on != 0) || (bt > 0x7f));
                        ob++)
                  {
                     bt = on & 0xff;
                     rl[ len + cb + ob +1 ] = bt; // copy least-significant byte
                     on >>= 8;                  // and divide to get remainder
                  }
               }
               else                             // negative offset
               {
                  for ( ob = 0, bt = 0;
                       (ob < sizeof(LLONG)) && ((on != -1) || (bt < 0x80));
                        ob++)
                  {
                     bt = on & 0xff;
                     rl[ len + cb + ob +1 ] = bt; // copy least-significant byte
                     on >>= 8;                  // and divide to get remainder
                  }
               }
            }
            else                                // sparse extent, no position
            {
               ob = 0;                          // position length is ZERO!
            }
            rl[ len] = ((ob << 4) | cb) & 0xff; // add combined length byte
            len     +=  (cb + ob +1);           // update total list length
         }
         rl[ len++] = 0;                        // terminate the runlist
                                                // (already 0 from Txalloc :-)
         TRHEXS( 70, rl, (ULONG) len, "RunList");

         *length  = (len + 7) & 0xfff8;         // make modulo 8, rounded up
         *runlist = rl;
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   else
   {
      rc = DFS_BAD_STRUCTURE;
   }
   RETURN (rc);
}                                               // end 'dfsNtfsMkRunList'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display S_SPACE structure for an NTFS non-resident attribute, with check
/*****************************************************************************/
ULONG dfsNtfsSpDisplay                          // RET   error code
(
   S_MFTFILE          *sd,                      // IN    base MFT rec, fixed-up
   ULONG               type,                    // IN    type of attribute MA_
   USHORT              atid                     // IN    attribute-id or IA_ANY
)
{
   ULONG               rc = NO_ERROR;
   ULONG               sdo;
   DFSISPACE           isp;                     // integrated SPACE allocation
   ULN64               data;

   ENTER();

   TRACES(( "MFT seq:%4.4hx  attr-type:%4.4x  attr-id:%hu\n", sd->Sequence, type, atid));

   rc = dfsNtfsSpAlloc( sd, type, atid, NULL, &isp);
   if ((rc == NO_ERROR) && (isp.space != NULL) && (isp.chunks != 0)) // not resident
   {
      ULN64            size = 0;
      ULN64            nr   = 0;

      rc = dfsNtfsMftRecCheckAlloc( sd, "", &size, &nr, &data);
      TxPrint( " Allocation Check : ");
      switch (rc)
      {
         case DFS_BAD_STRUCTURE:
            TxPrint("%sFAIL%s     ", CBR, CNN);
            if (nr == 0)
            {
               TxPrint("$Bitmap info not available!\n");
            }
            else
            {
               dfsSz64("failing sectors : ",  nr, "\n");
            }
            break;

         case DFS_PSN_LIMIT:
            TxPrint("%sERROR%s, sector nrs are too large!\n", CBR, CNN);
            break;

         default:
            TxPrint("%sOK%s%s\n", CBG, CNN, (type == MA_DATA) ?
                    "         Saveto/Recover/Undelete of filedata is possible" : "");
            break;
      }

      sdo = SD_CLUSTER | SD_TOPLINE;
      if (type == MA_DATA)                      // for data only, alloc for DIR
      {                                         // is handled by the Btree ...
         if (!TxaOptUnSet('X'))                 // unless -X- specified
         {
            sdo |= SD_NAVDOWN;                  // next is data sector
         }
      }
      if (isp.chunks > 1)
      {
         sdo |= SD_SUMMARY;
      }
      if (isp.chunks > 32)                      // multi-page is likely
      {
         if (dfsa->verbosity >= TXAO_VERBOSE)   // explicit verbose
         {
            sdo |= SD_BOTLINE;
         }
         else
         {
            sdo |= SD_LIMIT32;                  // just first 32 + summary
         }
      }
      if (TxaOption('X'))
      {
         sdo |= SD_ALLINFO;                     // incl Index and Info
      }
      dfsSspaceDisplay( sdo, isp.chunks, isp.space);
   }
   TxFreeMem( isp.space);
   RETURN (rc);
}                                               // end 'dfsNtfsSpDisplay'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate Directory Btree ordinal to DirBlock (INDX) logical sector nr
/*****************************************************************************/
ULN64 dfsNtfsBtree2Lsn                          // RET   Logical sector nr
(
   ULN64               mft,                     // IN    DIR MFT record number
   ULONG               ordinal                  // IN    Btree ordinal, relative
)                                               //       block INDX Allocation
{
   ULN64               lsn = 0x0badbadbadULL;   // 10 digit BAD value
   ULN64               rsn = ordinal * ntfs->ClustSize; // rel sector in alloc

   ENTER();
   TRARGS(("mft: %llX, ordinal: %u\n", mft, ordinal));

   if ((ntfs->Da.All.space == NULL) ||          // no cache there yet
       (ntfs->Da.Mft   != mft))                 // or for other mft #
   {
      if (ntfs->Da.All.space != NULL)           // remove existing one
      {
         TxFreeMem( ntfs->Da.All.space);        // free Dir Alloc cache
      }
      dfsNtfsMftLsn2Alloc( dfsNtfsMft2Lsn( mft), MA_DIRINDX, IA_ANY, NULL, &ntfs->Da.All);
      ntfs->Da.Mft = mft;
   }
   if (ntfs->Da.All.space != NULL)              // have Dir Alloc cache now
   {
      lsn = dfsSspaceRsn2Lsn( ntfs->Da.All.chunks, ntfs->Da.All.space, rsn);
   }
   RETN64 (lsn);
}                                               // end 'dfsNtfsBtree2Lsn'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate MFT record number to logical sector nr
/*****************************************************************************/
ULN64 dfsNtfsMft2Lsn                            // RET   Logical sector nr
(
   ULN64               mft                      // IN    MFT record number
)
{
   ULN64               lsn;
   ULN64               rsn = mft * ntfs->MftRecSize;

   ENTER();
   TRARGS(("mft:0x%llX\n", mft));

   if (mft == MFT_MFT)
   {
      lsn = dfsNtfsClust2Lsn(dfsa->boot->nt.MftClus);
   }
   else
   {
      if (ntfs->MftAll.space != NULL)           // MFT allocation known ?
      {
         lsn = dfsSspaceRsn2Lsn( ntfs->MftAll.chunks, ntfs->MftAll.space, rsn);
      }
      else                                      // calculate for 1st extent
      {
         lsn = dfsNtfsClust2Lsn( dfsa->boot->nt.MftClus) + rsn;
      }
   }
   RETN64 (lsn);
}                                               // end 'dfsNtfsMft2Lsn'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate MFT-mirror record number to logical sector nr
/*****************************************************************************/
ULN64 dfsNtfsMir2Lsn                            // RET   Logical sector nr
(
   ULN64               mft                      // IN    MFT record number
)
{
   ULN64               lsn;
   ULN64               rsn = mft * ntfs->MftRecSize;

   ENTER();
   TRARGS(("mft:0x%llX\n", mft));

   if (mft == MFT_MFT)
   {
      lsn = dfsNtfsClust2Lsn(dfsa->boot->nt.MftCopy);
   }
   else
   {
      //- to be refined, check allocation for Mirror (or rsn limit)
      lsn = dfsNtfsClust2Lsn(dfsa->boot->nt.MftCopy) + rsn;
   }
   RETN64 (lsn);
}                                               // end 'dfsNtfsMir2Lsn'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate (mft record) LSN to MFT record number
/*****************************************************************************/
ULN64  dfsNtfsLsn2Mft                           // RET   MFT record number
(
   ULN64               lsn                      // IN    Logical sector nr
)
{
   ULN64               mft = L64_NULL;
   ULN64               rsn = L64_NULL;

   ENTER();
   TRARGS(("lsn: %llX\n", lsn));

   if (ntfs->MftAll.space != NULL)                  // MFT allocation known ?
   {
      rsn = dfsSspaceLsn2Rsn( ntfs->MftAll.chunks, ntfs->MftAll.space, lsn);
   }
   else
   {
      rsn = lsn - dfsNtfsClust2Lsn( dfsa->boot->nt.MftClus);
   }
   if (rsn != L64_NULL)                         // valid ?
   {
      mft = rsn / ntfs->MftRecSize;
   }
   RETN64 (mft);
}                                               // end 'dfsNtfsLsn2Mft'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Iterate over MFT records (using a callback function)
/*****************************************************************************/
ULONG dfsNtfsMftIterator                        // RET   next result
(
   DFS_UL_FUNCT        callback,                // IN    callback for each MFT
   ULN64               minmft,                  // IN    first mft to process
   ULN64               maxmft                   // IN    last  mft to process
)
{
   ULONG               rc = NO_ERROR;
   ULN64               last = minmft;
   ULN64               lsn;

   ENTER();
   TRARGS(("first mft: 0x%llX   last mft: 0x%llX\n", minmft, maxmft));

   dfsNtfsLsnIterator( DFS_ITER_INIT, 0, "mft", &last);

   while (( rc == NO_ERROR) && (last <= maxmft) &&
          ((lsn = dfsNtfsLsnIterator( DFS_ITER_NEXT, 0, "mft", &last)) != L64_NULL))
   {
      rc = DFSFNCAUL( callback, lsn, 0, "mft", &last);
   }
   dfsNtfsLsnIterator( DFS_ITER_TERM, 0, "mft", NULL);
   RETURN( rc);
}                                               // end 'dfsNtfsMftIterator'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// NTFS generic Iterate: "mft" = MFT records (returning LSN for each)
/*****************************************************************************/
ULN64 dfsNtfsLsnIterator                        // RET   next LSN to work on
(
   ULN64               this,                    // IN    this LSN (or special)
   ULN64               d2,                      // IN    dummy
   char               *p1,                      // IN    options: "mft"
   void               *p2                       // INOUT ptr to ULN64 index
)                                               //       or NULL
{
   ULN64               rc  = L64_NULL;
   static ULN64        mft = MFT_NULL;          // last used mft number
   ULN64              *index = (ULN64 *) p2;

   switch (*p1)
   {
      case 'm':                                 // mft iterate
      default:
         if     ((this == DFS_ITER_INIT) ||
                 (this == DFS_ITER_TERM))
         {
            if ((index != NULL) && (*index != 0))
            {
               mft = *index -1;                 // init to start index
            }
            else
            {
               mft = MFT_NULL;
            }
         }
         else if (this == DFS_ITER_FIRST)
         {
            mft = 0;
         }
         else                                   // DFS_ITER_NEXT
         {
            if (mft == MFT_NULL)
            {
               mft = 0;
            }
            else if (++mft >= ntfs->MftRecCount)
            {
               mft = MFT_NULL;
            }
         }
         if (mft != MFT_NULL)
         {
            rc = dfsNtfsMft2Lsn( mft);
         }
         if (index != NULL)                     // when requested, supply the
         {                                      // underlying index value too
            *index = mft;
         }
         break;
   }
   return( rc);
}                                               // end 'dfsNtfsLsnIterator'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate Cluster-nr to LSN, Area aware
/*****************************************************************************/
ULN64 dfsNtfsCl2Lsn                             // RET   LSN for this cluster
(
   ULN64               cluster,                 // IN    cluster number
   ULN64               d2,                      // IN    dummy
   char               *p1,                      // IN    dummy
   void               *p2                       // IN    dummy
)
{
   ULN64               lsn;

   ENTER();
   TRACES(( "cluster: %llx  clsize:%u\n", cluster, ntfs->ClustSize));
   lsn = dfstAreaP2Disk( DFSTORE, (cluster * ntfs->ClustSize));
   RETN64(lsn);
}                                               // end 'dfsNtfsCl2Lsn'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate LSN to Cluster-nr, Area aware
/*****************************************************************************/
ULN64 dfsNtfsLsn2Cl                             // RET   cluster for this LSN
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *p1,                      // IN    dummy
   void               *p2                       // IN    dummy
)
{
   ULN64               cluster = 0;

   if (lsn == DFS_MAX_PSN)                      // query max Cluster-nr
   {
      cluster = ntfs->Clust;
   }
   else
   {
      cluster = dfstAreaD2Part( DFSTORE, lsn) / ntfs->ClustSize;
   }
   return (cluster);
}                                               // end 'dfsNtfsLsn2Cl'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get MFT-record usage status from $MFT Bitmap cache
/*****************************************************************************/
BOOL dfsNtfsMftMapped                           // RET   MFT record is in-use
(
   ULN64               mft                      // IN    MFT number
)
{
   BOOL                used  = TRUE;            // default in-use
   ULN64               index = mft >> 3;        // byte index in map
   BYTE                byte;

   if (index < ntfs->MftMap.Size)               // inside our cached map ?
   {
      byte = ntfs->MftMap.Data[ index];
      if (((byte >> (mft & 0x07)) & 0x01) == 0) // LSB is lowest!
      {
         used = FALSE;
      }
   }
   else
   {
      used = FALSE;
   }
   return (used);
}                                               // end 'dfsNtfsMftMapped'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Determine allocation-bit for specified LSN, ALLOCATED beyond last cluster!
/*****************************************************************************/
ULONG dfsNtfsAllocated                          // RET   LSN is allocated
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               asn = dfstAreaD2Part( DFSTORE, lsn); // Area aware sn

   if ((ntfs->Bm.Size != 0) && (asn < ntfs->Bm.LimitLsn)) // inside the bitmap
   {
      if (asn < (ntfs->Clust * ntfs->ClustSize)) // within valid cluster area
      {
         return((ULONG) dfsNtfsBitmapCache(dfsNtfsLsn2Clust( asn), NULL));
      }
      else                                      // return ALLOCATED for any
      {                                         // lsn beyond last cluster
         return( 1);                            // to avoid CHECK errors
      }                                         // on non-standard bitmaps
   }
   else
   {
      return(DFS_PSN_LIMIT);
   }
}                                               // end 'dfsNtfsAllocated'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Align R/W cache to position for cluster, and get current allocation-bit
/*****************************************************************************/
BOOL dfsNtfsBitmapCache                         // RET   cluster is allocated
(
   ULN64               cl,                      // IN    LCN
   ULONG              *bp                       // OUT   byte position in cache
)                                               //       or L32_NULL if invalid
{
   ULONG               rc = NO_ERROR;
   BOOL                al = TRUE;               // default allocated
   ULONG               cs = ntfs->ClustSize;    // cluster size in sectors
   ULONG               byte;                    // byte pos in sector
   ULN64               rsn;                     // relative sector in Bitmap
   ULONG               bytepos = L32_NULL;      // byte position in cluster
   BYTE                alloc_byte;

   if (cs != 0)                                 // avoid divide by zero
   {
      rsn = cl / (8 * dfsGetSectorSize());      // find relative BM sector for cl

      if (((rsn <  ntfs->Bm.First) ||           // if rsn is not in current
           (rsn >= ntfs->Bm.First +             // cached range of rsn's
                   ntfs->Bm.Size)))             // read new range ...
      {
         if (ntfs->Bm.Dirty)                    // need to flush changes ?
         {
            rc = dfsNtfsBitmapFlush( FALSE);    // flush, but keep cache
         }
         if (rc == NO_ERROR)
         {
            ntfs->Bm.First = (rsn / cs) * cs;
            TRACES(("Read  new cache Bm.First: %llx\n", ntfs->Bm.First));
            dfsSspaceReadFilePart( ntfs->Bm.All.chunks,
                                   ntfs->Bm.All.space,
                                   ntfs->Bm.First,
                                   ntfs->Bm.Size,
                          (BYTE *) ntfs->Bm.Buffer);
         }
      }
      if (rc == NO_ERROR)
      {
         byte       = (cl % (8 * dfsGetSectorSize())) / 8; // byte index in sector
         bytepos    = ((rsn - ntfs->Bm.First) * dfsGetSectorSize()) + byte;

         if ((bytepos < (ntfs->Bm.Size * dfsGetSectorSize())) && ntfs->Bm.Buffer != NULL)
         {
            alloc_byte = ntfs->Bm.Buffer[ bytepos];
            if (((alloc_byte >> (cl & 0x07)) & 0x01) == 0) // LSB is lowest!
            {
               al = FALSE;
            }
         }
         else                                   // signal error condition
         {
            TRACES(("bytepos %u out of range\n", bytepos));
            bytepos = L32_NULL;
         }
      }
      if (bp != NULL)                           // return byte position too ?
      {
         *bp = bytepos;
      }
   }
   return (al);
}                                               // end 'dfsNtfsBitmapCache'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Flush BitMap cache when Dirty, and optional free its resources
/*****************************************************************************/
ULONG dfsNtfsBitmapFlush                        // RET   rc
(
   BOOL                terminate                // IN    terminate cache too
)
{
   ULONG               rc = NO_ERROR;           // rc

   ENTER();

   TRACES(("BitMap cache Bm.First: %8.8x, ", ntfs->Bm.First));
   if (ntfs->Bm.Dirty)                          // need to flush changes ?
   {
      TRACES(( "dirty, flush it ...\n"));
      rc = dfsSspaceWriteFilePart( ntfs->Bm.All.chunks,
                                   ntfs->Bm.All.space,
                                   ntfs->Bm.First,
                                   ntfs->Bm.Size,
                          (BYTE *) ntfs->Bm.Buffer);

      ntfs->Bm.Dirty = FALSE;                   // mark it clean again
   }
   else
   {
      TRACES(( "not dirty ...\n"));
   }
   if (terminate)
   {
      TxFreeMem( ntfs->Bm.All.space);
      TxFreeMem( ntfs->Bm.Buffer);

      ntfs->Bm.Size     = 0;
      ntfs->Bm.LimitLsn = 0;
   }
   RETURN (rc);
}                                               // end 'dfsNtfsBitmapFlush'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set allocate status for LSN to specified value (beyond FS, to end of bitmap)
/*****************************************************************************/
ULONG dfsNtfsSetAlloc                           // RET   LSN allocation set
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *value,                   // IN    NULL = not allocated
   void               *ref                      // IN    dummy (for NTFS)
)
{
   if ((ntfs->Bm.Size != 0) && (lsn < ntfs->Bm.LimitLsn)) // inside the bitmap
   {
      return( dfsNtfsSetCluster(dfsNtfsLsn2Clust( lsn), (value != NULL)));
   }
   else
   {
      return( DFS_PSN_LIMIT);
   }
}                                               // end 'dfsNtfsSetAlloc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set allocate status for CLUSTER to specified value
/*****************************************************************************/
ULONG dfsNtfsSetCluster                         // RET   CL  allocation set
(
   ULN64               cl,                      // IN    Cluster
   BOOL                value                    // IN    SET allocation bit
)
{
   ULONG               rc = NO_ERROR;
   ULONG               bytepos;
   BYTE                alloc_byte;
   BYTE                bit_mask = (1 << (cl & 0x07));

   dfsNtfsBitmapCache( cl, &bytepos);
   if (bytepos != L32_NULL)                     // cluster position valid ?
   {
      alloc_byte = ntfs->Bm.Buffer[ bytepos];

      if (value)                                // set the allocation bit
      {
         ntfs->Bm.Buffer[ bytepos] = alloc_byte |  bit_mask;
      }
      else                                      // reset allocation bit
      {
         ntfs->Bm.Buffer[ bytepos] = alloc_byte & ~bit_mask;
      }
      TRACES(( "%s cache-byte at 0x%4.4x = -%6.6x- for cl:%10.10llx "
                  "from 0x%2.2hx to 0x%2.2hx\n",
                (value) ? "Set" : "Clr", bytepos, cl / 8, cl,
                alloc_byte, ntfs->Bm.Buffer[ bytepos]));

      ntfs->Bm.Dirty = TRUE;                    // mark cache dirty
   }
   else
   {
      rc = DFS_PSN_LIMIT;
   }
   return (rc);
}                                               // end 'dfsNtfsSetCluster'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set allocate status for DFSISPACE to specified value
/*****************************************************************************/
ULONG dfsNtfsSetSpaceAlloc                     // RET   allocation set
(
   DFSISPACE          *si,                      // IN    space info
   BOOL                value                    // IN    SET allocation bit
)
{
   ULONG               rc = NO_ERROR;
   ULONG               ac;                      // allocation chunk
   ULN64               cl;                      // cluster
   ULN64               rsn;                     // relative sector
   S_SPACE            *s;                       // one space info chunk

   ENTER();

   for (ac = 0; (ac < si->chunks) && si->space; ac++)
   {
      s = &(si->space[ac]);
      if (s->start != LSN_SPARSE)               // non-sparse chunk ?
      {
         for (rsn = 0; rsn < s->size; rsn += ntfs->ClustSize)
         {
            cl = (s->start + rsn) / ntfs->ClustSize;
            TRACES(( "Set cl %10.10x %s\n", cl, (value) ? "ALLOC" : "FREE"));

            dfsNtfsSetCluster( cl, value);
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsNtfsSetSpaceAlloc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Copy a valid NTFS boot record based on (hopefully) available spare-copy
/*****************************************************************************/
ULONG dfsNtfsCopySpareBR
(
   void
)
{
   ULONG               rc  = DFS_ST_MISMATCH;
   ULN64               lsn;
   ULONG               i;

   ENTER();

   for ( i = 0, lsn  = ntfs->SpareLsn;
        (i < 16) && (rc == DFS_ST_MISMATCH);
         i++,   lsn /= 2)
   {
      if (dfsNtfsValidBootRec( lsn))
      {
         rc = NO_ERROR;
      }
   }
   if (rc == NO_ERROR)                          // valid bootrec found
   {
      TxPrint( "\nThe following matching bootsector has been found:\n\n");
      dfsDisplaySector( dfstLSN2Psn( DFSTORE, LSN_BOOTR), ST_BOOTR, rbuf);
      if ((dfsa->batch) || (TxConfirm( 5137, "Do you want to replace the "
                "bootsector (sector 0) with this spare one ? [Y/N] : ")))
      {
         if (DFSTORE_WRITE_ALLOWED)
         {
            rc = dfsWrite( LSN_BOOTR, 1, rbuf);
            if (rc == NO_ERROR)
            {
               TxPrint("\nNTFS boot record successfully updated\n");
            }
         }
         else
         {
            rc = DFS_READ_ONLY;
         }
      }
      else
      {
         TxPrint("\nCommand aborted, no changes made\n");
         rc = DFS_NO_CHANGE;
      }
   }
   else
   {
      TxPrint("\nNo valid and matching NTFS boot record found\n");
   }
   RETURN(rc);
}                                               // end 'dfsNtfsCopySpareBR'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read sn into rbuf and check if it is a valid NTFS bootrec for this partition
/*****************************************************************************/
static BOOL dfsNtfsValidBootRec
(
   ULN64               sn                       // IN    logical sector-number
)
{
   BOOL                rc = FALSE;              // function return
   ULONG               dr;
   S_BOOTR            *sd = (S_BOOTR *) rbuf;
   TXTS                creatr;                  // creatr string in bootsect

   TxPrint("Check possible NTFS boot record at LSN 0x%llx: ", sn);
   dr = dfsRead( sn, 1, rbuf);
   if (dr == NO_ERROR)
   {
      if (dfsIdentifySector(LSN_BOOTR, 0, rbuf) == ST_BOOTR)
      {
         memcpy( creatr, sd->OpSys,    BT_SYST_L);
         creatr[BT_SYST_L] = '\0';
         if (strncasecmp( creatr, "NTFS", 4) == 0) // NTFS bootrec
         {
            if (SINF->p || (SINF->p->sectors >= sd->nt.VolSize))
            {
               if ((sd->nt.MftClus > 0) && (sd->nt.MftClus < sd->nt.VolSize) &&
                   (sd->nt.MftCopy > 0) && (sd->nt.MftCopy < sd->nt.VolSize)  )
               {
                  rc = TRUE;                    // good enough :-)
               }
               else
               {
                  TxPrint("MFT positions invalid\n");
               }
            }
            else
            {
               TxPrint("#sectors %llx is too large\n", sd->nt.VolSize);
            }
         }
         else
         {
            TxPrint("incorrect NTFS signature\n");
         }
      }
      else
      {
         TxPrint("not a boot record\n");
      }
   }
   else
   {
      TxPrint("read failure\n");
   }
   return (rc);
}                                               // end 'dfsNtfsValidBootRec'
/*---------------------------------------------------------------------------*/

#define NTFS_MFTSIGN   0x00
#define NTFS_MFTNAME   0x454c4946
#define NTFS_VOLATTR   0x35
#define NTFS_VOLNAME   0x006f0056
#define NTFS_LABATTR   0x188

/*****************************************************************************/
// Get label-string from the "Volume" MFT record
/*****************************************************************************/
BOOL dfsNtfsVolumeLabel                         // RET   Label found
(
   S_BOOTR            *boot,                    // IN    boot record for vol
   ULN64               offset,                  // IN    Offset for boot-rec
   char               *label                    // OUT   label string
)
{
   ULONG               rc = NO_ERROR;
   BOOL                found = FALSE;
   ULN64               volmft;                  // LSN of MFT[3] = volume
   S_MFTFILE          *mft;                     // ptr to MFT record
   ULONG              *mfl;                     // MFT as array of longs
   S_MFTATTR          *at;                      // ptr to attribute
   ULONG               sc;                      // sanity counter

   ENTER();

   //- Note: dfsa->boot is NOT valid when calling this (might be NULL!)
   //- Note: NTFS not initialized so ntfs-> fields not reliable either

   volmft = offset + (boot->nt.MftClus * boot->eb.ClustSize) +
          dfsNtfsV2S( boot->nt.MftSize,  boot->eb.ClustSize) * MFT_VOLUME;

   TRACES(("Volume-MFT at LSN 0x%llX\n", volmft));

   if ((rc = dfsRead( volmft, 2, rbuf)) == NO_ERROR) // read the MFT record
   {
      mfl  = (ULONG    *) rbuf;
      mft  = (S_MFTFILE*) rbuf;

      TRACES(("MFT at %8.8X, sign: %8.8x  volattr: %8.8x\n", rbuf, mfl[ NTFS_MFTSIGN], mfl[ NTFS_VOLATTR]));

      //- to be refined, could do a Fixup and real Identify here ...

      if (mfl[ NTFS_MFTSIGN] == NTFS_MFTNAME)
      {
         for ( sc = 0,      at = (S_MFTATTR *) ((BYTE *) mft + mft->AttribOffset);
              (sc < 30) && (at->Type != MA_END_LIST) && (!found);
               sc++,        at = ((S_MFTATTR *)((BYTE *) at  + at->Length)))
         {
            TRACES(("ATTR  %8.8x at: %8.8x\n", at->Type, at));
            if ((at->Type == MA_VOLNAME) && (at->Length != 0))
            {
               U_MFTATTR    *av;                // Attribute value data
               TXTM          text;

               av = (U_MFTATTR *) ((BYTE *) at + at->R.Offset);

               TxUnic2Ascii( av->Vname.VolumeName, text, (USHORT) (at->R.Size /2));
               text[ BT_LABL_L] = 0;
               TRACES(("Label found in MFT[3] : '%s'\n", text));
               strcpy( label, text);
               found = TRUE;
            }
         }
      }
   }
   RETURN (found);
}                                               // end 'dfsNtfsVolumeLabel'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create valid NTFS NTLDR sector (second sector of $Boot), and write to disk
/*****************************************************************************/
ULONG dfsNtfsMkNtldrSec
(
   BOOL                vw7810                   // IN    Vista/W7/8/10 BOOTMGR
)
{
   ULONG               rc = DFS_NO_CHANGE;
   BYTE               *ntldr = (vw7810) ? (BYTE *) &dfsNtfsNtldrVistaSec
                                        : (BYTE *) &dfsNtfsNtldrFirstSec;
   ENTER();
   TRACES(("Type: %s\n", (vw7810) ? "Vista/W7/8/10 BOOTMGR" : "NT/XP NTLDR"));

   TxPrint( "\nThe following NTLDR sector has been prepared:\n\n");
   dfsDisplaySector( dfstLSN2Psn( DFSTORE, NTLDR_LSN), ST_HDATA, ntldr);
   if ((dfsa->batch) || (TxConfirm( 5136, "Do you want to replace the "
                  "first NTLDR sector with the created one ? [Y/N] : ")))
   {
      if (DFSTORE_WRITE_ALLOWED)
      {
         rc = dfsWrite( NTLDR_LSN,  1, ntldr);
         if (rc == NO_ERROR)
         {
            TxPrint("\nFirst sector of NTLDR successfully updated\n");
         }
      }
      else
      {
         rc = DFS_READ_ONLY;
      }
   }
   else
   {
      TxPrint("\nCommand aborted, no changes made\n");
   }
   RETURN(rc);
}                                               // end 'dfsNtfsMkNtldrSec'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create valid NTFS boot record based on p-table info (or GEO) and template
// 20180614 JvW: Supply actual DEFAULT clustersize for partitions < 2 GiB
/*****************************************************************************/
ULONG dfsNtfsMkBootRec
(
   ULONG               mftLsn,                  // IN    sectornr for $MFT or 0
   BOOL                vw7810                   // IN    Vista/W7/8/10 BOOTMGR
)
{
   ULONG               rc = NO_ERROR;
   ULN64               partSectors;
   ULN64               lsn = mftLsn;
   S_MFTFILE          *mft;
   BYTE                st = ST_UDATA;
   USHORT              disknr = 1;
   time_t              tt = time( &tt);         // current date/time

   ENTER();
   TRACES(("Type: %s\n", (vw7810) ? "Vista/W7/8/10 BOOTMGR" : "NT/XP NTLDR"));

   brt = (vw7810) ? (S_BOOTR *) &dfsNtfsBootTemplateW7
                  : (S_BOOTR *) &dfsNtfsBootTemplateXP;

   if (SINF->p != NULL)                         // partition info available
   {
      disknr                = SINF->p->disknr;
      partSectors           = SINF->p->partent.NumberOfSectors;
      brt->eb.HiddenSectors = SINF->p->partent.BootSectorOffset;
      brt->eb.LogGeoSect    = SINF->p->geoSecs;
      brt->eb.LogGeoHead    = SINF->p->geoHeads;
   }
   else                                         // use global geo info
   {
      TxPrint( "\n%sNo partition-table info%s is available, this will "
               "make the 'HiddenSectors'\nfield incorrect when this "
               "is a primary partition.\n", CBR, CNN);
      TxPrint( "The logical geometry used now is %s%hu%s Heads "
               "with %s%hu%s Sectors\n",
                CBY, dfstGeoHeads(     DFSTORE), CNN,
                CBY, dfstGeoSectors(   DFSTORE), CNN);
      TxPrint("Use the 'diskgeo' command to change this when incorrect\n");

      partSectors           = dfstGetLogicalSize( DFSTORE);
      brt->eb.HiddenSectors = dfstGeoSectors( DFSTORE);
      brt->eb.LogGeoSect    = dfstGeoSectors( DFSTORE);
      brt->eb.LogGeoHead    = dfstGeoHeads( DFSTORE);
   }

   if (TxaOptSet('c'))                          // clustersize overrule
   {
      brt->eb.ClustSize = (BYTE) TxaOptNum( 'c', NULL, dfsGetClusterSize());
   }
   else                                         // use standard MS defaults
   {
      if (dfsGetSectorSize() == SECTORSIZE)     // 512 byte sectors
      {
         if      (partSectors <= (DFSECT_2GIB / 4)) // less than 512 MiB
         {
            brt->eb.ClustSize = 1;
         }
         else if (partSectors <= (DFSECT_2GIB / 2)) // less than 1024 MiB
         {
            brt->eb.ClustSize = 2;
         }
         else if (partSectors <= (DFSECT_2GIB / 1)) // less than 2048
         {
            brt->eb.ClustSize = 4;
         }
         else                                   // most common, use 4 KiB clusters
         {
            brt->eb.ClustSize = 8;
         }
      }
      else                                      // Most likely 4 KiB, 1 sect/Cl
      {
         brt->eb.ClustSize = 1;
      }
   }
   ntfs->ClustSize = brt->eb.ClustSize;         // update our anchor block too
   ntfs->Clust     =  partSectors / ntfs->ClustSize;
   dfstSetClusterSize( DFSTORE, (USHORT) ntfs->ClustSize);

   if (lsn == 0)                                // not overruled by param
   {
      lsn = ntfs->MftBaseLsn;                   // from Init, or FindMft
   }
   brt->nt.MftClus = lsn / brt->eb.ClustSize;

   //- Best guess: MFTcopy halfway the partition :-)
   brt->nt.MftCopy = (partSectors / 2) / brt->eb.ClustSize;

   if ((dfsNtfsReadFixMft( lsn, &st, &mft)   == NO_ERROR) && // is an MFT record
       (TxMemStr( mft, sg_mft_0, SECTORSIZE) != NULL)      ) // looks like $MFT
   {
      TxFreeMem( mft);
      if (dfsNtfsReadFixMft( lsn +2, &st, &mft) == NO_ERROR) // read $MFTMirr
      {
         DFSISPACE           isp;               // integrated SPACE allocation

         if (dfsNtfsSpAlloc( mft, MA_DATA, IA_ANY, NULL, &isp) == NO_ERROR)
         {
            if (isp.chunks)                     // $MFTMirr data extent
            {
               brt->nt.MftCopy = isp.space->start / brt->eb.ClustSize;
            }
            TxFreeMem( isp.space);              // free the S_SPACE, if any
         }
         TxFreeMem( mft);
      }
   }
   else                                         // lsn is not an MFT record
   {
      TXLN             warning;

      sprintf( warning, "WARNING: default sector 0x0%llx does NOT seem to contain the MFT-record\n"
                        "for the $MFT special file. The resulting bootsector will be unreliable!", lsn);

      TxPrint( "%s\n", warning);
      if ((!dfsa->batch) && (!TxConfirm( 5134, "%s\n\n"
                     "It is better to CANCEL the fixboot now, and run the FINDMFT command "
                     "or menu-item first, to try and find the correct $MFT location\n\n"
                     "Continue with the FIXBOOT operation anyway ? [Y/N] : ", warning)))
      {
         rc = DFS_NO_CHANGE;
      }
   }

   if (rc == NO_ERROR)
   {
      brt->nt.VolSize = partSectors;
      brt->nt.SerialNr = TXmku64( DFS_V_EYE, (ULONG) tt); // DFSee eyecatcher + timestamp

      TxPrint( "\nThe following new NTFS bootsector has been prepared:\n\n");
      dfsDisplaySector( dfstLSN2Psn( DFSTORE, LSN_BOOTR), ST_BOOTR, (BYTE   *) brt);
      if ((dfsa->batch) || (TxConfirm( 5135, "Do you want to replace the "
                     "bootsector (sector 0) with the created one ? [Y/N] : ")))
      {
         if (DFSTORE_WRITE_ALLOWED)
         {
            rc = dfsWrite( LSN_BOOTR, 1, (BYTE   *) brt);
            if (rc == NO_ERROR)
            {
               TxPrint("\nNTFS boot record successfully updated\n");
            }
         }
         else
         {
            rc = DFS_READ_ONLY;
         }
      }
      else
      {
         rc = DFS_NO_CHANGE;
      }
   }
   RETURN(rc);
}                                               // end 'dfsNtfsMkBootRec'
/*---------------------------------------------------------------------------*/

