//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// DFSee data compression functions, based on hashed-LZW algorithm
//
//
// 04-06-1998 JvW First released version                    (DFSee 2.67)
// 05-06-1998 JvW Fixed memory-leak                         (DFSee 2.68)
// 12-06-1998 JvW Increased decode-stack from 4000 to 9999  (LPT   1.90)
// 20-06-1998 JvW Analysed stack-runaway bug
// 27-10-2002 JvW Removed assert() usage (Watcom PMWIN)     (DFSee 5.04)
// 29-12-2002 JvW Fix status/progress flicker + performance (DFSee 5.08)
// 31-12-2004 JvW Set decode-stack to 16000 now             (DFSee 7.00)
// 24-07-2005 JvW Moved to DFSee source tree, out of TXLIB  (DFSee 7.10)
// 02-12-2016 JvW Set decode-stack to 64000 now             (DFSee 14.0)
// 17-01-2017 JvW Cleanup interface for buffer-only         (DFSee 14.2)

#include <txlib.h>                              // TxLib interface

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsulzw.h>                            // DFSee compression interface

#define DFSUL_SIGNATURE   0xFACEFEED            // valid magic signature value
#define DFSUL_OBSOLETE    0xDEADBEEF            // invalid signature value

typedef struct dfsul_status
{
   ULONG               signature;               // DFSUL signature value
   ULONG               method;                  // compression method
   ULONG               reserved[32];            // max 128 bytes method specific
} DFSUL_STATUS;                                 // end of struct "dfsul_status"


typedef struct dfsul_buffer
{
   BYTE               *buffer;                  // start of buffer
   ULONG               pos;                     // current position
   ULONG               length;                  // length available data
} DFSUL_BUFFER;                                 // end of struct "dfsul_buffer"


typedef struct dfsul_io_buf
{
   DFSUL_BUFFER        get;                     // input  or 'get' buffer data
   DFSUL_BUFFER        put;                     // output or 'put' buffer data
} DFSUL_IO_BUF;                                 // end of struct "dfsul_io_buf"


// Buffer get-byte routine, read from a memory buffer
static int dfsUlBufGetByte                      // RET   byte from buffer
(
   DFSUL_IO_BUF       *io                       // IN    buffer descriptors
);

// Buffer put-byte routine, write to a memory buffer only
static void dfsUlBufPutByte
(
   int                 byte,                    // IN    data byte to put
   DFSUL_IO_BUF       *io                       // IN    buffer descriptors
);


// Compression routine, multi method dispatcher
static ULONG dfsUlCompress
(
   DFSUL_HANDLE        ulh,                     // IN    dfsUl handle
   ULONG               method,                  // IN    compression method
   DFSUL_IO_BUF       *bufio                    // INOUT data buffer info
);

// expansion routine, multi method dispatcher
static ULONG dfsUlUncompress
(
   DFSUL_HANDLE        ulh,                     // IN    dfsUl handle
   ULONG               method,                  // IN    compression method
   DFSUL_IO_BUF       *bufio                    // INOUT data buffer info
);


//============ START of LZW-specific definitions and functions ===============

#define DFSLZW_INIBITS   9                      // nr of bits to start with
#define DFSLZW_BITMASK   ((ULONG) 0x0f)         // nr-of-bits in method

#define DFSLZW_MAXBIT(n) (( 1 <<( n )) -1)      // max_value formula macro


#define DFSLZW_TSIZE10    1193
#define DFSLZW_TSIZE11    2399
#define DFSLZW_TSIZE12    4799
#define DFSLZW_TSIZE13    9587                  // prime values,
#define DFSLZW_TSIZE14   19183                  // little > 2^ maxbits

#define DFSLZW_STACK     64000                  // decode stack-size

#define DFSLZW_CLRTAB    256                    // Code to flush the table
#define DFSLZW_TERMIN    257                    // To mark EOF Condition
#define DFSLZW_CODE_1    258                    // First available code
#define DFSLZW_CPOINT    100                    // Nr of chars before check

#define DFSLZW_UNUSED  ((USHORT) 0xffff)        // unused code-value

// LZW variant for the status structure
typedef struct dfslzw_status
{
   ULONG               signature;               // DFSUL signature value
   ULONG               method;                  // compression method
   USHORT              bits;                    // nr of bits in code-values
   USHORT              maxcode;                 // maximum value in 'bits'
   USHORT              maxbits;                 // maximum nr of bits
   ULONG               bgot;                    // bytes got from input
   ULONG               bput;                    // bytes put out
   ULONG               cpoint;                  // statistic checkpoint
   USHORT              icount;                  // input  bit count
   USHORT              ocount;                  // output bit count
   ULONG               ibits;                   // input  bit buffer
   ULONG               obits;                   // output bit buffer
   USHORT              tabsize;                 // size of LZW hash-buffers
   USHORT             *cvalue;                  // array of code-values
   USHORT             *prefix;                  // array of prefix codes
   char               *append;                  // array of append characters
   char               *decode;                  // decode-stack
} DFSLZW_STATUS;                                // end of struct "dfslzw_status"


// Initialize all values for an LZW instance of the dfsUl abstract-data-type
static ULONG dfsLzwInitialize
(
   DFSLZW_STATUS      *lzws,                    // IN    LZW status
   BOOL                compress                 // IN    compress
);

// Terminate use of allocated resources for an LZW instance
static void dfsLzwFree
(
   DFSLZW_STATUS      *lzws                     // IN    LZW status
);

// Expansion routine, LZW class
static ULONG dfsLzwUncompress
(
   DFSLZW_STATUS      *lzws,                    // IN    LZW status
   DFSUL_IO_BUF       *bufio                    // INOUT data buffer info
);

// Compression routine, LZW class
static ULONG dfsLzwCompress
(
   DFSLZW_STATUS      *lzws,                    // IN    LZW status
   DFSUL_IO_BUF       *bufio                    // INOUT data buffer info
);

// Get byte from input, LZW class
static USHORT dfsLzwInputCode
(
   DFSLZW_STATUS      *lzws,                    // IN    LZW status
   DFSUL_IO_BUF       *bufio                    // INOUT data buffer info
);

// Hashing routine, LZW class
static USHORT dfsLzwFindCodeMatch
(
   DFSLZW_STATUS      *lzws,                    // IN    LZW status
   USHORT              hash_prefix,
   USHORT              hash_character
);

// decode string, LZW class
static char *dfsLzwDecodeString
(
   DFSLZW_STATUS      *lzws,                    // IN    LZW status
   char               *buffer,
   USHORT              code
);


// output code, LZW class
static void dfsLzwOutputCode
(
   DFSLZW_STATUS      *lzws,                    // IN    LZW status
   DFSUL_IO_BUF       *bufio,                   // INOUT data buffer info
   USHORT              code                     // IN    new output value
);


/*****************************************************************************/
// Register a new instance of the dfsUl abstract-data-type
/*****************************************************************************/
ULONG dfsUlRegister                             // RET   result
(
   DFSUL_HANDLE       *ulh                      // OUT   dfsUl handle
)
{
   ULONG               rc = NO_ERROR;
   DFSUL_STATUS       *uls;                     // pack status ptr

   ENTER();

   if ((uls = TxAlloc(1, sizeof(DFSUL_STATUS))) != NULL)
   {
      uls->signature = DFSUL_SIGNATURE;         // Signature, rest is ZEROED!
   }
   else
   {
      rc = TX_ALLOC_ERROR;
   }
   *ulh = (DFSUL_HANDLE) uls;
   TRACES(("Alloc %6u to uls: %8.8lX\n", sizeof(DFSUL_STATUS), uls));
   RETURN( rc);
}                                               // end 'dfsUlRegister'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Terminate use of an instance of the dfsUl abstract-data-type
/*****************************************************************************/
void dfsUlTerminate
(
   DFSUL_HANDLE        ulh                      // IN    dfsUl handle
)
{
   DFSUL_STATUS       *uls = (DFSUL_STATUS *) ulh;

   ENTER();

   if ((uls != NULL) && (uls->signature == DFSUL_SIGNATURE))
   {
      if (DFSUL_METHOD(uls->method) <= DFSUL_MLZWMAX)
      {
         dfsLzwFree( (DFSLZW_STATUS *) uls);
      }
      uls->signature = DFSUL_OBSOLETE;
      TRACES(("Free uls at: %8.8lX\n", uls));
      TxFreeMem( uls);
   }
   VRETURN();
}                                               // end 'dfsUlTerminate'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Buffer Compression routine
/*****************************************************************************/
ULONG dfsUlBufCompress
(
   DFSUL_HANDLE        ulh,                     // IN    dfsUl handle
   ULONG               method,                  // IN    compression method
   BYTE               *ibuf,                    // IN    input buffer
   ULONG               ilen,                    // IN    input buffer length
   BYTE               *obuf,                    // OUT   output buffer
   ULONG               olen,                    // IN    output buffer length
   ULONG              *size                     // OUT   compressed size
)
{
   ULONG               rc = NO_ERROR;
   DFSUL_IO_BUF        io;                     // input/output descriptors

   ENTER();

   TRARGS(("ulh: %p, method: %8.8lX\n", ulh, method));
   TRARGS(("ibuf-Crc32: %8.8lX\n", TxCrc32( ibuf, ilen)));

   io.get.buffer = ibuf;
   io.get.length = ilen;
   io.get.pos    = 0;

   io.put.buffer = obuf;
   io.put.length = olen;
   io.put.pos    = 0;

   rc = dfsUlCompress( ulh, method, &io);
   if (rc == NO_ERROR)
   {
      if ((io.put.pos >= io.put.length) ||      // doesn't fit in ouput
          (io.put.pos >= io.get.length) )       // or larger than input
      {
         TRACES(("No compression achieved!\n"));
         rc = TX_NO_COMPRESS;                   // no compression achieved
      }
   }
   *size = io.put.pos;
   TRACES(("obuf-Crc32: %8.8lX\n", TxCrc32( obuf, min( *size, olen))));
   TRACES(("Compressed: %9lu bytes into %9lu, ratio: % 2.2lf\n", ilen, *size, ((double) ilen / (double) *size)));
   RETURN( rc);
}                                               // end 'dfsUlBufCompress'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Buffer Un-compress routine
/*****************************************************************************/
ULONG dfsUlBufUncompress
(
   DFSUL_HANDLE        ulh,                     // IN    dfsUl handle
   ULONG               method,                  // IN    compression method
   BYTE               *ibuf,                    // IN    input buffer
   ULONG               ilen,                    // IN    input buffer length
   BYTE               *obuf,                    // OUT   output buffer, or NULL
   ULONG               olen,                    // IN    output buffer length
   ULONG              *size                     // OUT   un-compressed size
)
{
   ULONG               rc = NO_ERROR;
   DFSUL_IO_BUF        io;                     // input/output descriptors

   ENTER();

   TRARGS(("ulh: %p, method: %8.8lX\n", ulh, method));
   TRARGS(("ibuf-Crc32: %8.8lX\n", TxCrc32( ibuf, ilen)));

   io.get.buffer = ibuf;
   io.get.length = ilen;
   io.get.pos    = 0;

   io.put.buffer = obuf;
   io.put.length = olen;
   io.put.pos    = 0;

   //- Allow no-output by specifying no put-function on null obuf (but VALID olen!)
   rc = dfsUlUncompress( ulh, method, &io);
   if (rc == NO_ERROR)
   {
      if (io.put.pos > io.put.length)           // write position too large!
      {
         rc = TX_FAILED;
      }
   }
   *size = io.put.pos;
   if (obuf)
   {
      TRACES(("obuf-Crc32: %8.8lX\n", TxCrc32( obuf, *size)));
   }
   TRACES(("Uncompressed: %9lu bytes into %9lu, ratio: % 2.2lf\n", ilen, *size, ((double) *size / (double) ilen)));
   RETURN( rc);
}                                               // end 'dfsUlBufUncompress'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Buffer get-byte routine, read from a memory buffer
/*****************************************************************************/
static int dfsUlBufGetByte                      // RET   byte from buffer
(
   DFSUL_IO_BUF       *io                       // IN    buffer descriptors
)
{
   int                 byte;

   if (io->get.pos < io->get.length)            // more data available ?
   {
      byte = (io->get.buffer[io->get.pos++]) & 0xff;
   }
   else
   {
      byte = EOF;
   }
   return( byte);
}                                               // end 'dfsUlBufGetByte'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Buffer put-byte routine, write to a memory buffer only
/*****************************************************************************/
static void dfsUlBufPutByte
(
   int                 byte,                    // IN    data byte to put
   DFSUL_IO_BUF       *io                       // IN    buffer descriptors
)
{
   if ((io->put.buffer != NULL) && (io->put.pos < io->put.length)) // actual write allowed?
   {
      io->put.buffer[io->put.pos] = (BYTE) (byte & 0xff);
   }
   io->put.pos++;                               // always increment, can be used by caller
}                                               // end 'dfsUlBufPutByte'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Compression routine, multi method dispatcher
/*****************************************************************************/
static ULONG dfsUlCompress
(
   DFSUL_HANDLE        ulh,                     // IN    dfsUl handle
   ULONG               method,                  // IN    compression method
   DFSUL_IO_BUF       *bufio                    // INOUT data buffer info
)
{
   ULONG               rc  = NO_ERROR;
   DFSUL_STATUS       *uls = (DFSUL_STATUS *) ulh;

   ENTER();

   if ((uls != NULL) && (uls->signature == DFSUL_SIGNATURE))
   {
      uls->method = method;
      if (DFSUL_METHOD(uls->method) <= DFSUL_MLZWMAX)
      {
         rc = dfsLzwCompress( (DFSLZW_STATUS *) uls, bufio);
      }
      else                                      // other algorithms
      {
         rc = DFS_VALUE_ERROR;
      }
   }
   else
   {
      rc = TX_NO_INITIALIZE;
   }
   RETURN(rc);
}                                               // end 'dfsUlCompress'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// expansion routine, multi method dispatcher
/*****************************************************************************/
static ULONG dfsUlUncompress
(
   DFSUL_HANDLE        ulh,                     // IN    dfsUl handle
   ULONG               method,                  // IN    compression method
   DFSUL_IO_BUF       *bufio                    // INOUT data buffer info
)
{
   ULONG               rc  = NO_ERROR;
   DFSUL_STATUS       *uls = (DFSUL_STATUS *) ulh;

   ENTER();

   if ((uls != NULL) && (uls->signature == DFSUL_SIGNATURE))
   {
      uls->method = method;
      if (DFSUL_METHOD(uls->method) <= DFSUL_MLZWMAX)
      {
         rc = dfsLzwUncompress( (DFSLZW_STATUS *) uls, bufio);
      }
      else                                      // other algorithms
      {
         rc = DFS_VALUE_ERROR;
      }
   }
   else
   {
      rc = TX_NO_INITIALIZE;
   }
   RETURN(rc);
}                                               // end 'dfsUlUncompress'
/*---------------------------------------------------------------------------*/


//============ START of LZW-specific implementation functions ================


/*****************************************************************************/
// Initialize all values for a given instance of the dfsUl abstract-data-type
/*****************************************************************************/
static ULONG dfsLzwInitialize
(
   DFSLZW_STATUS      *lzws,                    // IN    LZW status
   BOOL                compress                 // IN    compress
)
{
   ULONG               rc = NO_ERROR;
   USHORT              tabsize = 0;

   ENTER();

   if (DFSUL_METHOD(lzws->method) <= DFSUL_MLZWMAX)
   {
      lzws->bits    = DFSLZW_INIBITS;           // nr of bits in code-values
      lzws->maxcode = DFSLZW_MAXBIT(DFSLZW_INIBITS);
      lzws->maxbits = (USHORT) (lzws->method & DFSLZW_BITMASK); // max nr of bits allowed
      lzws->bgot    = 0;                        // bytes got from input
      lzws->bput    = 0;                        // bytes put out
      lzws->cpoint  = DFSLZW_CPOINT;            // statistic checkpoint
      lzws->icount  = 0;                        // input  bit count
      lzws->ocount  = 0;                        // output bit count
      lzws->ibits   = 0;                        // input  bit buffer
      lzws->obits   = 0;                        // output bit buffer

      switch (lzws->maxbits)
      {
         case DFSUL_MLZW_10: tabsize = DFSLZW_TSIZE10; break;
         case DFSUL_MLZW_11: tabsize = DFSLZW_TSIZE11; break;
         case DFSUL_MLZW_12: tabsize = DFSLZW_TSIZE12; break;
         case DFSUL_MLZW_13: tabsize = DFSLZW_TSIZE13; break;
         default:            tabsize = DFSLZW_TSIZE14; break;
      }
      if (tabsize > lzws->tabsize)              // (bigger) LZW tables needed ?
      {
         dfsLzwFree( lzws);                     // free old ones
      }
      if (lzws->prefix == NULL)                 // not allocated yet ?
      {
         lzws->prefix = TxAlloc(tabsize, sizeof(USHORT));
         if (lzws->prefix == NULL)
         {
            rc = TX_ALLOC_ERROR;
         }
         TRACES(("Alloc %6u *2 to prefix: %8.8lX\n", tabsize, lzws->prefix));
      }
      if (lzws->append == NULL)                 // not allocated yet ?
      {
         lzws->append = TxAlloc(tabsize, sizeof(char));
         if (lzws->append == NULL)
         {
            rc = TX_ALLOC_ERROR;
         }
         TRACES(("Alloc %6u *1 to append: %8.8lX\n", tabsize, lzws->append));
      }
      if (compress)                             // alloc compress buffers
      {
         if (lzws->cvalue == NULL)              // not allocated yet ?
         {
            lzws->cvalue = TxAlloc(tabsize, sizeof(USHORT));
            if (lzws->cvalue == NULL)
            {
               rc = TX_ALLOC_ERROR;
            }
            TRACES(("Alloc %6u *2 to cvalue: %8.8lX\n", tabsize, lzws->cvalue));
         }
      }
      else                                      // alloc uncompress buffers
      {
         if (lzws->decode == NULL)              // not allocated yet ?
         {
            lzws->decode = TxAlloc(DFSLZW_STACK, sizeof(char));
            if (lzws->decode == NULL)
            {
               rc = TX_ALLOC_ERROR;
            }
            TRACES(("Alloc %6u *1 to decode: %8.8lX\n", DFSLZW_STACK, lzws->decode));
         }
      }
      if (rc == NO_ERROR)
      {
         lzws->tabsize = tabsize;               // allocated tabsize
      }
      else                                      // free aquired resources
      {
         dfsLzwFree( lzws);
      }
   }
   RETURN(rc);
}                                               // end 'dfsLzwInitialize'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Terminate use of allocated resources for an LZW instance
/*****************************************************************************/
static void dfsLzwFree
(
   DFSLZW_STATUS      *lzws                     // IN    LZW status
)
{
   TxFreeMem( lzws->cvalue);
   TxFreeMem( lzws->prefix);
   TxFreeMem( lzws->append);
   TxFreeMem( lzws->decode);
}                                               // end 'dfsLzwFree'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Compression routine, LZW class
/*****************************************************************************/
static ULONG dfsLzwCompress
(
   DFSLZW_STATUS      *lzws,                    // IN    LZW status
   DFSUL_IO_BUF       *bufio                    // INOUT data buffer info
)
{
   ULONG               rc  = NO_ERROR;
   USHORT              next_code = DFSLZW_CODE_1;
   USHORT              character;
   USHORT              string_code;
   USHORT              index;
   USHORT              i;                       // All purpose integer
   ULONG               ratio_new;               // New ratio as a percentage
   ULONG               ratio_old = 100;         // Original ratio at 100%

   ENTER();

   dfsLzwInitialize( lzws, TRUE);
   for (i = 0; i < lzws->tabsize; i++)          // Init string table first
   {
      lzws->cvalue[i] = DFSLZW_UNUSED;
   }
   string_code = dfsUlBufGetByte( bufio);       // Get the first code

   TRACES(("Kb : "));
   while((character = dfsUlBufGetByte( bufio)) != (USHORT) EOF)
   {
      #if defined (DUMP)                        // possible trace
         if (!(++lzws->bgot % 1024))
         {                                      // Count bytes and pacifier
            TRINTF(("."));
         }
      #else
         ++lzws->bgot;                          // just count
      #endif
      index = dfsLzwFindCodeMatch( lzws, string_code, character);
      if (lzws->cvalue[index] != DFSLZW_UNUSED)
      {
         string_code = lzws->cvalue[index];
      }
      else
      {
         if (next_code <= lzws->maxcode )
         {
            lzws->cvalue[index] = next_code++;
            lzws->prefix[index] = string_code;
            lzws->append[index] = (char) character;
         }
         dfsLzwOutputCode( lzws, bufio, string_code);
         string_code = character;
         if (next_code > lzws->maxcode)
         {                                      // Is table Full?
            if ( lzws->bits < lzws->maxbits)
            {                                   // Any more bits?
               lzws->maxcode = DFSLZW_MAXBIT(++lzws->bits); // Incr code size
               TRINTF(("+"));
            }
            else if (lzws->bgot > lzws->cpoint)
            {                                   // At checkpoint?
               if (lzws->bits == lzws->maxbits)
               {
                  ratio_new = lzws->bput*100/lzws->bgot; // New compression ratio
                  if (ratio_new > ratio_old)
                  {                             // Has ratio degraded?
                     dfsLzwOutputCode( lzws, bufio, DFSLZW_CLRTAB);
                     TRINTF(("C"));
                     lzws->bits    = DFSLZW_INIBITS;
                     next_code     = DFSLZW_CODE_1; // Reset to DFSLZW_CODE_1
                     lzws->maxcode = DFSLZW_MAXBIT(lzws->bits); // Re-Initialize
                     lzws->bgot    = lzws->bput = 0;
                     ratio_old     = 100;        // Reset compression ratio
                     for (i = 0; i < lzws->tabsize; i++) // Reset code value array
                     {
                        lzws->cvalue[i] = DFSLZW_UNUSED;
                     }
                  }
                  else
                  {                             // NO, then save new
                     ratio_old = ratio_new;     // compression ratio
                  }
               }
               lzws->cpoint = lzws->bgot + DFSLZW_CPOINT; // Set new checkpoint
            }
         }
      }
   }
   dfsLzwOutputCode( lzws, bufio, string_code);
   if (next_code == lzws->maxcode)
   {                                            // Handles special case for
      ++lzws->bits;                             // bit-increment on EOF
      TRINTF(("C"));
   }
   dfsLzwOutputCode( lzws, bufio, DFSLZW_TERMIN);
   dfsLzwOutputCode( lzws, bufio, 0);           // Flush the output buffer
   dfsLzwOutputCode( lzws, bufio, 0);
   dfsLzwOutputCode( lzws, bufio, 0);
   TRINTF(("\n"));                              // put next at start of line
   RETURN(rc);
}                                               // end 'dfsLzwCompress'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// expansion routine, LZW class
/*****************************************************************************/
static ULONG dfsLzwUncompress
(
   DFSLZW_STATUS      *lzws,                    // IN    LZW status
   DFSUL_IO_BUF       *bufio                    // INOUT data buffer info
)
{
   ULONG               rc = NO_ERROR;
   USHORT              next_code  = DFSLZW_CODE_1;
   USHORT              new_code   = 0;
   USHORT              old_code   = 0;
   USHORT              character  = 0;
   USHORT              counter    = 0;
   BOOL                clear_flag = TRUE;       // clear the code value array
   char               *string;

   ENTER();

   dfsLzwInitialize( lzws, FALSE);
   TRACES(("Progress in Kb : %s", TREOLN));
   while (( rc == NO_ERROR) && ((new_code = dfsLzwInputCode( lzws, bufio)) != DFSLZW_TERMIN))
   {
      if (clear_flag)
      {                                         // Initialize or Re-Init
         clear_flag = FALSE;
         old_code   = new_code;
         character  = old_code;
         dfsUlBufPutByte( old_code, bufio);     // output one character
         continue;
      }
      if (new_code == DFSLZW_CLRTAB)
      {                                         // Clear string table
         clear_flag  = TRUE;
         lzws->bits  = DFSLZW_INIBITS;
         next_code   = DFSLZW_CODE_1;
         TRINTF(("C"));
         lzws->maxcode = DFSLZW_MAXBIT(lzws->bits);
         TRACES(("Cleared string table, maxcode: %4.4x\n", lzws->maxcode))
         continue;
      }
      if (++counter == 1024)
      {                                         // Pacifier
         counter=0;
         TRINTF(("."));
      }
      if (new_code >= next_code)
      {                                         // Check for string+char+string
         lzws->decode[0] = (char) character;
         string = dfsLzwDecodeString( lzws, lzws->decode+1, old_code);
      }
      else
      {
         string = dfsLzwDecodeString( lzws, lzws->decode, new_code);
      }

      if (string != NULL)
      {
         character = *string;                   // Output string in reverse
         while (string >= lzws->decode)
         {
            dfsUlBufPutByte( *string--, bufio); // output one character
         }
         if (next_code <= lzws->maxcode)
         {                                      // Add to table if not full
            lzws->prefix[next_code]   = old_code;
            lzws->append[next_code++] = (char) character;
            if ((next_code  == lzws->maxcode) &&
                (lzws->bits  < lzws->maxbits))
            {
               TRINTF(("+"));
               lzws->maxcode = DFSLZW_MAXBIT(++lzws->bits);
               TRACES(("Incremented bits, maxcode: %4.4x\n", lzws->maxcode))
            }
         }
         old_code = new_code;
      }
      else                                      // decode stack error!
      {
         if (bufio->put.buffer != NULL)         // if output wanted
         {
            TxPrint( "Internal decode-stack overflow error uncompressing the data!\n");
         }
         rc = TX_ABORTED;
      }
   }
   TRINTF(("\n"));                              // put next at start of line
   RETURN(rc);
}                                               // end 'dfsLzwUncompress'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Hashing routine, LZW class
/*****************************************************************************/
static USHORT dfsLzwFindCodeMatch
(
   DFSLZW_STATUS      *lzws,                    // IN    LZW status
   USHORT              hash_prefix,
   USHORT              hash_character
)
{
   int                 index;
   USHORT              offset;

   index = (hash_character << (lzws->maxbits -8)) ^ hash_prefix;
   if (index == 0 )
   {
      offset = 1;
   }
   else
   {
      offset = lzws->tabsize - index;
   }
   while(lzws->cvalue[index] != DFSLZW_UNUSED)
   {
      if (lzws->prefix[index] == hash_prefix &&
          lzws->append[index] == (char) hash_character)
      {
         break;
      }
      index -= offset;
      if (index < 0)
      {
         index += lzws->tabsize;
      }
   }
   return(index);
}                                               // end 'dfsLzwFindCodeMatch'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// decode string, into a reverse-order stack, LZW class
/*****************************************************************************/
static char *dfsLzwDecodeString                  // RET   buffer or NULL (error)
(
   DFSLZW_STATUS      *lzws,                    // IN    LZW status
   char               *buffer,
   USHORT              code
)
{
   int                 i = 0;

   while (code > 255)
   {
      if (code >= lzws->tabsize)
      {
         TxPrint( "\nDecompress code %hu exceeds current table limit %hu!\n",
                                code, lzws->tabsize);
         return NULL;
      }
      *buffer++ = lzws->append[code];
      code      = lzws->prefix[code];
      i++;
      //- TRACES(("DecodeString, i:%5d code:%hu\n", i, code));
      if (i >= DFSLZW_STACK)
      {
         return NULL;
      }
   }
   *buffer = (char) code;
   return(buffer);
}                                               // end 'dfsLzwDecodeString'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Input a variable length code, LZW class
/*****************************************************************************/
static USHORT dfsLzwInputCode
(
   DFSLZW_STATUS      *lzws,                    // IN    LZW status
   DFSUL_IO_BUF       *bufio                    // INOUT data buffer info
)
{
   USHORT              return_value;
   int                 input_byte = 0;

   while ((lzws->icount <= 24) && (input_byte != EOF))
   {
      input_byte    = dfsUlBufGetByte( bufio);
      lzws->ibits  |= (ULONG) input_byte << (24 - lzws->icount);
      lzws->icount += 8;
      //- TRACES(("InputCode, ibits: 0x%8.8lx  icount:%3hu  input_byte: 0x%16.16x = %d\n",
      //-          lzws->ibits, lzws->icount, input_byte, input_byte));
   }
   if (input_byte != EOF)
   {
      return_value   = (USHORT) (lzws->ibits >> (32-lzws->bits));
      lzws->ibits  <<= lzws->bits;
      lzws->icount  -= lzws->bits;
   }
   else                                         // error, reached EOF
   {
      return_value = DFSLZW_TERMIN;             // force end of unpack
   }
   return(return_value);
}                                               // end 'dfsLzwInputCode'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Output a variable length code, LZW class
/*****************************************************************************/
static void dfsLzwOutputCode
(
   DFSLZW_STATUS      *lzws,                    // IN    LZW status
   DFSUL_IO_BUF       *bufio,                   // INOUT data buffer info
   USHORT              code                     // IN    new code byte
)
{
   lzws->obits  |= (ULONG) code << (32 - lzws->bits - lzws->ocount);
   lzws->ocount += lzws->bits;
   while (lzws->ocount >= 8)
   {
      if (bufio->put.buffer != NULL)            // if output wanted
      {
         dfsUlBufPutByte((int) (lzws->obits >> 24), bufio);
      }
      lzws->obits  <<= 8;
      lzws->ocount  -= 8;
      lzws->bput++;                             // compression monitoring
   }
}                                               // end 'dfsLzwOutputCode'
/*---------------------------------------------------------------------------*/

