//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// JFS utility functions
//
// Author: J. van Wijk
//
// JvW  19-04-2001 Initial version
//
#ifndef    DFSUJFS
   #define DFSUJFS

#include <dfsjfs.h>                             // JFS structure definitions


//- Translate PXD and XAD  style .hi + .lo fields into a single 64bit value
#define dfsJfsPxdValue(pxd) ((ULN64) (pxd).lo  + ((ULN64) (pxd).hi << 32))
#define dfsJfsXadValue(xad) ((ULN64) (xad).rlo + ((ULN64) (xad).rhi << 32))

                                                // superblock id's
#define JFSB_1ST           1
#define JFSB_2ND           2
#define JFSB_BOTH          3

// Identify if FS structures for JFS are present, set fsform and label
ULONG dfsJfsFsIdentify
(
   S_BOOTR            *boot,                    // IN    Bootsector ref or NULL
   DFSPARTINFO        *p                        // INOUT partition info
);


// Find INODE for specified path, starting at known root-directory INODE
ULONG dfsJfsFindPath
(
   ULN64               loud,                    // IN    Show progress
   ULN64               d2,                      // IN    dummy
   char               *path,                    // IN    path specification
   void               *vp                       // OUT   found dir/file INODE
);

// Set the alternate label location, in the superblock, using specified label
ULONG dfsJfsSetAltBrecLabel
(
   ULN64               v1,                      // IN    unused
   ULN64               v2,                      // IN    unused
   char               *str,                     // IN    new label value (11)
   void               *param                    // IN    unused
);

// Update filename in-place, with a new name of the exact same length
ULONG dfsJfsUpdateFileName
(
   ULN64               inoLsn,                  // IN    Inode sectornumber
   ULN64               info,                    // IN    unused
   char               *str,                     // IN    current filename
   void               *param                    // IN    new same length filename
);

// Replace sn-list by contents of a single Directory (InoLsn, no info)
ULONG dfsJfsMakeBrowseList                      // RET   LSN is valid INODE
(
   ULN64               inoLsn,                  // IN    Inode sectornumber
   ULN64               info,                    // IN    unused
   char               *str,                     // IN    unused
   void               *param                    // INOUT unused
);

// Get allocation information for file-DATA into integrated-SPACE structure
// 20161215: Include EA-data in separate allocation (in the ispace)
ULONG dfsJfsGetAllocSpace
(
   ULN64               inoLsn,                  // IN    Inode sectornumber
   ULN64               info,                    // IN    unused
   char               *str,                     // IN    unused
   void               *param                    // INOUT Integrated SPACE
);

#define dfsJfsBlock2Lsn(bl) (ULN64) dfsJfsCl2Lsn((bl),0,NULL,NULL)
// Translate Block-nr to LSN, generic interface
ULN64 dfsJfsCl2Lsn                              // RET   LSN for this block
(
   ULN64               block,                   // IN    block number
   ULN64               d2,                      // IN    dummy
   char               *p1,                      // IN    dummy
   void               *p2                       // IN    dummy
);

#define dfsJfsLsn2Block(sn) dfsJfsLsn2Cl((ULN64)(sn),0,NULL, NULL)
// Translate LSN to Block-nr, generic interface
ULN64 dfsJfsLsn2Cl                              // RET   block for this LSN
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *p1,                      // IN    dummy
   void               *p2                       // IN    dummy
);

// Determine allocation-bit for specified LSN, ALLOCATED beyond last block!
ULONG dfsJfsAllocated                           // RET   LSN is allocated
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);

// Align R/W cache to position for block, and get current allocation-bit
BOOL dfsJfsBitmapCache                          // RET   block is allocated
(
   ULN64               bl,                      // IN    Block number
   ULONG              *cp                       // OUT   position in cache
);                                              //       or L32_NULL if invalid

// Extract BitMap allocation S_SPACE structure and initialize the BitMap cache
ULONG dfsJfsBitMapInit                          // RET   rc = 0 if type match
(
   void
);

// Flush BitMap cache when Dirty, and optional free its resources
ULONG dfsJfsBitmapFlush                         // RET   rc
(
   BOOL                terminate                // IN    terminate cache too
);

// Set allocate status for LSN to specified value (can go beyond FS, to end bitmap)
ULONG dfsJfsSetAlloc                            // RET   LSN allocation set
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *value,                   // IN    NULL = not allocated
   void               *ref                      // IN    dummy (for HPFS)
);

// Set allocate status for BLOCK to specified value
ULONG dfsJfsSetBlock                            // RET   BL  allocation set
(
   ULN64               bl,                      // IN    Block
   BOOL                value                    // IN    SET allocation bit
);

// Write cached superblock to 1st and 2nd superblock location on disk
ULONG dfsJfsWriteSuperBlocks
(
   ULONG               sbid                     // IN    superblock id
);

// Quickly find normal/deleted Inodes using the Inode MAP
ULONG dfsJfsQuickFindInodes
(
   BOOL                verbose,                 // IN    List files while found
   ULONG               first,                   // IN    First inode todo
   char               *types,                   // IN    Inode types to search
   char               *select                   // IN    Selection string for
);                                              //       resolved name (if any)

// DFS JFS read and check type for a JFS Inode based on its sector number
ULONG dfsJfsReadChkInode
(
   ULN64               lsn,                     // IN    Inode LSN
   BYTE               *stype,                   // OUT   Inode type, or NULL
   S_JINODE          **inode                    // OUT   Inode structure
);

#define JFS_TREE_INDENT     2

// Convert Inode external-data description (Xtree) to an S_SPACE structure
// There will be a separate S_SPACE for the actual data, and tree meta-data
ULONG dfsJfsInodeData2Space
(
   S_JINODE           *inode,                   // IN    Inode with alloc info
   ULONG               showTree,                // IN    display indent or 0
   DFSISPACE          *is,                      // OUT   Integrated S_SPACE
   ULONG              *metaChunks,              // OUT   nr of meta extents
   S_SPACE           **metaSpace                // OUT   meta extent space
);


// Convert Xtree page external-data to an S_SPACE structure (could recurse)
// There will be a separate S_SPACE for the actual data, and tree meta-data
ULONG dfsJfsXtpData2Space
(
   ULN64               block,                   // IN    Tree page block nr
   ULONG               showTree,                // IN    display indent or 0
   DFSISPACE          *is,                      // OUT   Integrated S_SPACE
   ULONG              *metaChunks,              // OUT   nr of meta extents
   S_SPACE           **metaSpace                // OUT   meta extent space
);


// Convert array of extent-allocation-descriptors to an S_SPACE structure
ULONG dfsJfsXads2Space
(
   S_XAD              *xad,                     // IN    XAD Array
   ULONG               extents,                 // IN    Nr of XAD structures
   DFSISPACE          *is                       // OUT   Integrated S_SPACE
);


// Display file allocation and path info for LSN
ULONG dfsJfsFileInfo                            // RET   LSN is valid Inode
(
   ULN64               lsn,                     // IN    Inode LSN
   ULN64               d2,                      // IN    dummy
   char               *select,                  // IN    Inode select wildcard
   void               *param                    // INOUT leading text/shortpath
);


// Convert JFS JTIME (64 bit) to DFS standard date/time string
char *dfsJfsTime2str                            // RET   string value
(
   JTIME              *jt,                      // IN    ptr to JFS time value
   char               *dtime                    // INOUT ptr to string buffer
);

// Find Inode and PATH upto Root-directory, starting at some Inode LSN
BOOL dfsJfsFindRootInode                        // OUT   root found
(
   ULN64               start,                   // IN    starting lsn
   char               *path,                    // OUT   combined path, TXMAXLN
   ULN64              *lsn                      // OUT   found dir/file Inode LSN
);

// Find parent Inode for the given (DIR or FILE) Inode, add name to PATH
BOOL dfsJfsInode2Parent                         // OUT   real parent found
(
   ULN64               fn,                      // IN    starting lsn
   char               *path,                    // OUT   combined path, TXMAXLN
   ULN64              *lsn                      // OUT   found dir/file Inode LSN
);

// Resolve related Inode name from cache, .LONGNAME or magic-area
BOOL dfsJfsResolveInodeName                     // RET   real filename found
(
   ULN64               lsn,                     // IN    Inode lsn
   S_JINODE           *ino,                     // IN    Inode sector
   BYTE                st,                      // IN    Inode type
   ULONG              *parent,                  // OUT   Parent inode# or NULL
   char               *fname                    // OUT   related name
);                                              //       min TXMAXLN

// Append JFS Directory multiple-slot Unicode filename to ASCII string
char *dfsJfsFileNameAppend                      // RET   string value
(
   S_JDSLOT           *slot,                    // IN    inode dir slot array
   char               *asc,                     // INOUT ptr to string buffer
   BYTE                index,                   // IN    slot with name start
   BYTE                maxslot                  // IN    maximum slot number
);

// Append JFS Btree Splitpoint multiple-slot Unicode filename to ASCII string
char *dfsJfsBtspNameAppend                      // RET   string value
(
   S_JDSLOT           *slot,                    // IN    inode dir slot array
   char               *asc,                     // INOUT ptr to string buffer
   BYTE                index,                   // IN    slot with name start
   BYTE                maxslot                  // IN    maximum slot number
);

// Display(S_PXD) address and size in 8-hex digits and Mb/Kb or Bytes
void dfsSpxd                                    // in the resulting display
(                                               // ** indicates addr>32 bits!!
   char               *text,                    // IN    leading string
   ULONG               bsize,                   // IN    block size in sectors
   S_PXD              *pxd,                     // IN    physical data extent
   char               *trail                    // IN    trailing text
);

// Format (S_PXD) address and size in 8-hex digits and Mb/Kb or Bytes; append
void dfstrSpxd                                  // in the resulting string,
(                                               // ** indicates addr>32 bits!!
   char               *str,                     // OUT   resulting string
   char               *text,                    // IN    leading string
   ULONG               bsize,                   // IN    block size in sectors
   S_PXD              *pxd,                     // IN    physical data extent
   char               *trail                    // IN    trailing text
);

// DFS JFS verify allocation integrity for a given INODE (normal / deleted)
ULONG dfsJfsInodeCheckAlloc                     // RET   check result
(
   S_JINODE           *ino,                     // IN    INODE structure
   BYTE                iType,                   // IN    INODE type
   char               *options,                 // IN    result options
   ULN64              *totals,                  // INOUT total sectors
   ULN64              *invalids,                // INOUT invalid sectors
   ULN64              *datalsn                  // OUT   first data lsn
);                                              //       or NULL if not needed

// DFS JFS verify allocation integrity for given INODE SPACE allocation
ULONG dfsJfsSpaceCheckAlloc                     // RET   check result
(
   DFSISPACE          *is,                      // IN    data allocation
   BOOL                deleted,                 // IN    delete status
   char               *options,                 // IN    result options
   ULN64               maxSect,                 // IN    max nr of sectors
   ULN64              *totals,                  // INOUT total sectors
   ULN64              *invalids                 // INOUT invalid sectors
);

// Translate Inode number to logical sector nr
ULN64 dfsJfsInode2Lsn                           // RET   Logical sector 64bit
(
   ULONG               ino,                     // IN    Inode number
   S_IMAP             *map                      // IN    Inode mapping
);

// Translate (Inode sector) LSN to INODE number
ULONG  dfsJfsLsn2Inode                           // RET   INODE number
(
   ULN64               lsn,                     // IN    Logical sector nr
   S_IMAP             *map                      // IN    Inode mapping
);

// Read allocation info for Aggr/Fileset Inodes from the Fileset Inode Table
ULONG dfsJfsFsInodeAlloc
(
   S_JINODE           *ino,                     // IN    Super Inode sector
   S_IMAP             *map                      // INOUT Inode map info
);

// Add any EA data info for an Inode to std FEA2-list structure
ULONG dfsJfsReadEaInode
(
   S_JINODE           *ino,                     // IN    Inode sector
   ULONG              *size,                    // OUT   size in bytes
   S_FEA2LIST        **pfeal                    // OUT   list of FEA2's
);

// Retrieve value for specified EA using Inode info
ULONG dfsJfsEaGetValue
(
   S_JINODE           *ino,                     // IN    Inode sector
   char               *name,                    // IN    EA name
   ULONG              *len,                     // OUT   EA data size
   void              **data                     // OUT   EA data (allocated)
);


// Analyse IAG page, optional add Inode-extents to supplied inode extent map
ULONG dfsJfsIagAddExtents
(
   S_IAG              *iag,                     // IN    IAG structure buffer
   ULONG               start,                   // IN    start index in map
   ULONG               limit,                   // IN    size of inode map
   ULN64              *extMap,                  // OUT   Inode extent map or NULL
   ULONG              *errors                   // INOUT Invalid ext lsns or NULL
);

// Analyse IAG page, calculate summary maps for extent-alloc and inode-status
ULONG dfsJfsIagMkSummary
(
   S_IAG              *iag,                     // IN    IAG structure buffer
   ULONG              *extAllocation,           // OUT   Inode extent alloc map
   ULONG              *inoFreeStatus,           // OUT   Inode free status map
   ULONG              *extUsed,                 // OUT   used extents 0..128
   ULONG              *inoUsed                  // OUT   used inodes  0..4096
);


// Create a valid JFS  bootrecord based on partition-table, template and SB
ULONG dfsJfsMkBootRec
(
   S_JFSUP            *sup,                     // IN    optional superblock
   BOOL                standard,                // IN    use standard JFS code
   char               *drive                    // IN    optional bootdrive
);


// Copy JFS bad-blocks list (special Inode 0x5c) to the DFSee (bad) sectorlist
ULONG dfsJfsExportBadList                      // RET
(
   ULN64             *list,                     // IN    LSN list structure
   ULONG              size                      // IN    max nr of entries
);


// Copy DFSee (bad) sectorlist to the JFS bad-blocks list (special Inode 0x5c)
// sectorlist may be empty (to clear the bad sectors)
ULONG dfsJfsImportBadList                      // RET
(
   ULN64             *list                      // IN    LSN list structure
);

#endif
