//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// JFS utility functions
//
// Author: J. van Wijk
//
// JvW  04-12-1997 Initial version
// JvW  12-05-2000 Removed obsolete dfsJfsUndeleteSNtable function
// JvW  03-04-2017 Updated for 64bit sectornumbers


#include <txlib.h>                              // ANSI controls and logging

#include <dfsver.h>                             // DFS version info
#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfstable.h>                           // SLT utility functions

#include <dfsjfs.h>                             // JFS disk structures
#include <dfsajfs.h>                            // JFS analysis functions
#include <dfsujfs.h>                            // JFS utility functions
#include <dfsljfs.h>                            // JFS slt & error definitions
#include <dfsos2ea.h>                           // HPFS/JFS/FAT EA handling


static ULONG dfsJfsStdTemplate[] =              // From eCS 256 MiB partition
{
   0x49001fac, 0x34204d42, 0x0030352e, 0x00010802,  //-  IBM 4.50
   0x00020002, 0x0086f800, 0x00ff003f, 0x0000003f,  //-  ??
   0x0003ebd1, 0xbd290180, 0x44559365, 0x65655346,  //-  )eUDFSee
   0x4a647453, 0x464a5346, 0x20202053, 0x79202020,  //-  StdJFSJFS      y
   0x40000017, 0x00373900, 0x0017790c, 0x3920000d,  //-  @97y 9
   0x0064004f, 0x00080008, 0x00027870, 0x00000053,  //-  OdpxS
   0x1c0277b5, 0x00000000, 0x1c0277b5, 0x00027880,  //-  wwx
   0xffffffff, 0x1c027dbd, 0x00000000, 0x0000035e,  //-  }^
   0x000085e6, 0x000278b0, 0xffffffff, 0x00000008,  //-  x
   0x00000002, 0x27000000, 0x0002787c, 0x00027928,  //-  '|x(y
   0x1ff51eb8, 0x00000008, 0x00000008, 0x00000064,  //-  d
   0x00093920, 0x0000000d, 0x0002790c, 0x00063900,  //-   9y9
   0x00004000, 0x00027920, 0x00027928, 0x00000000,  //-  @ y(y
   0x00000000, 0x00027980, 0x00004000, 0x00000020,  //-  y@ 
   0x002479b0, 0x0004001b, 0x00000000, 0x00063900,  //-  y$9
   0x000279b4, 0x788879dc, 0x79dc1c02, 0x16000017,  //-  yyxy
   0x004f4440, 0x001779c8, 0x3920000d, 0x0044004f,  //-  @DOy 9OD
   0x00080008, 0x0002792c, 0x00000053, 0x1c0277b5,  //-  ,ySw
   0x00000000, 0x1c0277b5, 0x0002793c, 0xffffffff,  //-  w<y
   0x1c027dbd, 0x00000000, 0x00007c46, 0x00001000,  //-  }F|
   0x0002796c, 0xffffffff, 0x00000008, 0x00000002,  //-  ly
   0x27000000, 0x00027938, 0x000279e4, 0x1ff51eea,  //-  '8yy
   0x00000008, 0x00000008, 0x00000044, 0x00093920,  //-  D 9
   0x0000000d, 0x000279c8, 0x00094440, 0x00001600,  //-  y@D
   0x000279dc, 0x000279e4, 0x00000000, 0x00023920,  //-  yy 9
   0x78887a54, 0x7a541c02, 0x00240017, 0x00177a30,  //-  TzxTz$0z
   0x00177a58, 0x7a5f0001, 0x00630017, 0x00080008,  //-  Xz_zc
   0x000279c8, 0x00000053, 0x1c0277b5, 0x00000000,  //-  ySw
   0x1c0277b5, 0x000279d8, 0xffffffff, 0x1c027dbd,  //-  wy}
   0x00000000, 0x1cef8e48, 0x00001000, 0x00027a08,  //-  Hz
   0xffffffff, 0x00000008, 0x00000002, 0x27000000,  //-  '
   0x000279d4, 0x00027a64, 0x1ff5290b, 0xaa550008   //-  ydz)U
};

/*
static ULONG dfsJfsEcs2Template[] =             // From eCS 2.0 GA
{
   0x499054eb, 0x34204d42, 0x0030352e, 0x00000002,  //- .T.IBM 4.50.....
   0x00000000, 0x0000f800, 0x00f0003f, 0x0000003f,  //- ........?...?...
   0x020ca3e1, 0x0c298580, 0x427c9a99, 0x61746f6f,  //- ......)...|Boota
   0x4a656c62, 0x464a5346, 0x20202053, 0x00002020,  //- bleJFSJFS     ..
   0x00000000, 0x00100000, 0x00000020, 0x00000000,  //- ........ .......
   0x00000000, 0x33fa0000, 0xbcd08ec0, 0xbbfb7c00,  //- .......3.....|..
   0xc38e07c0, 0x542d12cd, 0xfff02500, 0x8e06e0c1,  //- ......-T.%......
   0xff3366d8, 0x66db3366, 0x001ca126, 0x06012666,  //- .f3.f3.f&...f&..
   0x2666003e, 0x00421e11, 0x0020b81e, 0x5be8f633,  //- >.f&..B... .3..[
   0x26661f00, 0x66003ea1, 0x66003ea3, 0x0042a126,  //- ..f&.>.f.>.f&.B.
   0x0042a366, 0x0024a026, 0x810024a2, 0x6102003e,  //- f.B.&.$..$..>..a
   0xb80e7419, 0xd88e07c0, 0xbe0200a1, 0xa1e800c7,  //- .t..............
   0x9cb81e00, 0x49cb5019, 0x6c61766e, 0x63206469,  //- .....P.Invalid c
   0x2065646f, 0x20726f66, 0x2053464a, 0x59532d00,  //- ode for JFS .-SY
   0x32303253, 0x90002d37, 0x90909090, 0xda8c521e,  //- S2027-.......R..
   0x48a31f06, 0x4c168900, 0x4a368900, 0x0046be00,  //- ...H...L..6J..F.
   0x003ea166, 0x004ea366, 0x0042a166, 0x0052a366,  //- f.>.f.N.f.B.f.R.
   0x4e3e0166, 0x1e116600, 0x42b40052, 0x0024168a,  //- f.>N.f..R..B..$.
   0x13cd00b0, 0xe40a0473, 0x1f5a0375, 0x54a150c3,  //- ....s...u.Z..P.T
   0x0062e800, 0xe80052a1, 0x50a1005c, 0x0056e800,  //- ..b..R..\..P..V.
   0xe8004ea1, 0xa1660050, 0x8a580046, 0x00ddbec2,  //- .N..P.f.F.X.....
   0x90000fe8, 0x90909090, 0x90909090, 0x90909090,  //- ................
   0x66fc9090, 0xffa8ac50, 0x0eb40974, 0xcd0007bb,  //- ...fP...t.......
   0xfbf2eb10, 0x50665866, 0x00002566, 0xc166ffff,  //- ....fXfPf%....f.
   0x11e810e8, 0xb43ab000, 0x0007bb0e, 0x586610cd,  //- ......:.......fX
   0xeb0003e8, 0x8a50c3fe, 0xb1f024c4, 0xe8e8d204,  //- ......P..$......
   0x5058001d, 0x0f24c48a, 0x580014e8, 0xb1f02450,  //- ..XP..$....XP$..
   0xe8e8d204, 0x50580009, 0x02e80f24, 0x04c35800,  //- ......XP$....X..
   0x7e393c30, 0xb4070402, 0x0007bb0e, 0x4fc310cd,  //- 0<9~...........O
   0x444c3253, 0x534f0052, 0x4f4f4232, 0x63280054,  //- S2LDR.OS2BOOT.(c
   0x30322029, 0x4d203031, 0x79736e65, 0x56422073,  //- ) 2010 Mensys BV
   0x00000000, 0x00000000, 0x00000000, 0xaa558500   //- ..............U.
};
*/

static ULONG dfsJfsBootTemplate[] =             // From JFS 1.9.7 ArcaOS
{
   0x499054eb, 0x34204d42, 0x0030352e, 0x00000002, // .T.IBM 4.50.....
   0x00000000, 0x0000f800, 0x00ff003f, 0x0000003f, // ........?...?...
   0x0026badf, 0xbe298a80, 0x4a234455, 0x312d5346, // ..&...).UD#JFS-1
   0x372d392d, 0x464a0000, 0x20202053, 0x00002020, // -9-7..JFS     ..
   0x00000000, 0x00100000, 0x0000001f, 0x00000000, // ................
   0x00000000, 0x33fa0000, 0xbcd08ec0, 0xbbfb7c00, // .......3.....|..
   0xc38e07c0, 0x542d12cd, 0xfff02500, 0x8e06e0c1, // ......-T.%......
   0xff3366d8, 0x66db3366, 0x001ca126, 0x06012666, // .f3.f3.f&...f&..
   0x2666003e, 0x00421e11, 0x001fb81e, 0x5be8f633, // >.f&..B.....3..[
   0x26661f00, 0x66003ea1, 0x66003ea3, 0x0042a126, // ..f&.>.f.>.f&.B.
   0x0042a366, 0x0024a026, 0x810024a2, 0x6102003e, // f.B.&.$..$..>..a
   0xb80e7419, 0xd88e07c0, 0xbe0200a1, 0xa1e800c7, // .t..............
   0x9cb81e00, 0x49cb5019, 0x6c61766e, 0x63206469, // .....P.Invalid c
   0x2065646f, 0x20726f66, 0x2053464a, 0x59532d00, // ode for JFS .-SY
   0x32303253, 0x90002d37, 0x90909090, 0xda8c521e, // S2027-.......R..
   0x48a31f06, 0x4c168900, 0x4a368900, 0x0046be00, // ...H...L..6J..F.
   0x003ea166, 0x004ea366, 0x0042a166, 0x0052a366, // f.>.f.N.f.B.f.R.
   0x4e3e0166, 0x1e116600, 0x42b40052, 0x0024168a, // f.>N.f..R..B..$.
   0x13cd00b0, 0xe40a0473, 0x1f5a0375, 0x54a150c3, // ....s...u.Z..P.T
   0x0062e800, 0xe80052a1, 0x50a1005c, 0x0056e800, // ..b..R..\..P..V.
   0xe8004ea1, 0xa1660050, 0x8a580046, 0x00ddbec2, // .N..P.f.F.X.....
   0x90000fe8, 0x90909090, 0x90909090, 0x90909090, // ................
   0x66fc9090, 0xffa8ac50, 0x0eb40974, 0xcd0007bb, // ...fP...t.......
   0xfbf2eb10, 0x50665866, 0x00002566, 0xc166ffff, // ....fXfPf%....f.
   0x11e810e8, 0xb43ab000, 0x0007bb0e, 0x586610cd, // ......:.......fX
   0xeb0003e8, 0x8a50c3fe, 0xb1f024c4, 0xe8e8d204, // ......P..$......
   0x5058001d, 0x0f24c48a, 0x580014e8, 0xb1f02450, // ..XP..$....XP$..
   0xe8e8d204, 0x50580009, 0x02e80f24, 0x04c35800, // ......XP$....X..
   0x7e393c30, 0xb4070402, 0x0007bb0e, 0x4fc310cd, // 0<9~...........O
   0x444c3253, 0x534f0052, 0x4f4f4232, 0x00000054, // S2LDR.OS2BOOT...
   0x00000000, 0x00000000, 0x00000000, 0x00000000, // ................
   0x00000000, 0x00000000, 0x00000000, 0xaa558a00  // ..............U.
};

static S_BOOTR *brt = (S_BOOTR *) &dfsJfsBootTemplate;



// Search filename in INODE DIR-entries and BTree pages from dir INODE LSN
// Optional rename to SAME LENGTH new name
static BOOL dfsJfsSearchDinode                  // OUT   DIR slots resolved
(
   ULONG               inode,                   // IN    directory INODE nr
   ULONG               loud,                    // IN    Show progress
   char               *rename,                  // IN    same length new name
   char               *entry,                   // IN    entry specification
   ULONG              *nextIno                  // OUT   found Inode number
);


// Search filename in Directory Btree+ page, recurse into tree
// Optional rename to SAME LENGTH new name
static BOOL dfsJfsSearchDtreePage               // RET   dir entry found
(
   ULONG               loud,                    // IN    Show progress
   char               *entry,                   // IN    entry specification
   char               *rename,                  // IN    same length new name
   ULONG               pageLsn,                 // IN    start LSN of page
   ULONG              *ino                      // OUT   found entry inode
);


// Find DIR-entry for specified name, start in slot table, follow the tree
// Optional rename to SAME LENGTH new name, caller must WRITE BACK slot-array!
static BOOL dfsJfsDirSlot2Entry                 // OUT   entry found
(
   ULONG               loud,                    // IN    Show progress
   char               *entry,                   // IN    entry specification
   char               *rename,                  // IN    same length new name
   S_JDSLOT           *slot,                    // IN    Directory slot array
   BYTE                maxslot,                 // IN    maximum nr of slots
   BYTE               *sTable,                  // IN    sort-table array
   BYTE                sCount,                  // IN    used entries in sTable
   BOOL                treeNodes,               // IN    tree internal nodes
   ULONG              *ino                      // OUT   found dir/file INODE nr
);


// Update JFS Directory multiple-slot Unicode filename from ASCII str (rename)
static void dfsJfsFileNameUpdate
(
   S_JDSLOT           *slot,                    // INOUT inode dir slot array
   char               *asc,                     // IN    Ascii (new) filename
   BYTE                index,                   // IN    slot with name start
   BYTE                maxslot                  // IN    maximum slot number
);


// Add inoLsn for all entries in Directory Btree+ page to sectorlist, recurse
static ULONG dfsJfsDtreePage2List               // RET   dir entry found
(
   ULONG               pageLsn                  // IN    start LSN of page
);

// Add DIR-entry inoLSNs to list, start at slot table, iterate over whole tree
static ULONG dfsJfsDirSlots2List
(
   S_JDSLOT           *slot,                    // IN    Directory slot array
   BYTE                maxslot,                 // IN    maximum nr of slots
   BYTE               *sTable,                  // IN    sort-table array
   BYTE                sCount,                  // IN    used entries in sTable
   BOOL                treeNodes                // IN    tree internal nodes
);


/*****************************************************************************/
// Find INODE for specified path, starting at known root-directory INODE
/*****************************************************************************/
ULONG dfsJfsFindPath
(
   ULN64               loud,                    // IN    Show progress
   ULN64               d2,                      // IN    dummy
   char               *path,                    // IN    path specification
   void               *vp                       // OUT   found dir/file INODE
)
{
   ULONG               rc  = NO_ERROR;
   TXLN                part;
   char               *p   = path;
   int                 l;
   ULONG               curIno = JFS_ROOT_INODE;
   DFS_PARAMS         *parm = (DFS_PARAMS *) vp;

   ENTER();

   if (loud)
   {
      TxPrint("RootDir  Inode nr : 0x%08.8X   find path: '%s'\n", curIno, path);
   }
   parm->Lsn    = dfsJfsInode2Lsn( curIno, &jfs->Fs1);
   parm->Number = 0;
   while ((rc == NO_ERROR) && strlen(p) && !TxAbort())
   {
      if ((l = strcspn(p, FS_PATH_STR)) != 0)
      {
         strncpy(part, p, l);                   // isolate part
         part[l] = '\0';
         p += l;                                // skip part
         if (*p == FS_PATH_SEP)
         {
            p++;                                // skip '\'
         }
      }
      if (strlen(part))
      {
         if (loud)
         {
            TxPrint("Directory Inode @ : 0x%08.8X",
                     dfsJfsInode2Lsn( curIno, &jfs->Fs1));
         }
         if (dfsJfsSearchDinode( curIno, loud, part, NULL, &curIno))
         {
            if (loud)
            {
               TxPrint(" - Inode %08.8X for '%s'\n", curIno, part);
            }
            if (*p == '\0')                     // end of string, found!
            {
               BYTE                st;
               S_JINODE           *ino;

               parm->Lsn    = dfsJfsInode2Lsn(curIno, &jfs->Fs1);
               parm->Number = 0;                // no additional info
               parm->Flag   = FALSE;            // Size NOT from DIR-entry

               if (dfsJfsReadChkInode( parm->Lsn, &st, &ino) == NO_ERROR)
               {
                  parm->byteSize = ino->FileSize;
                  TxFreeMem( ino);
               }
            }
         }
         else
         {
            if (loud)
            {
               TxPrint(" - Search failed  for '%s'\n", part);
            }
            rc = ERROR_PATH_NOT_FOUND;
         }
      }
      else
      {
         rc = ERROR_PATH_NOT_FOUND;
      }
   }
   RETURN (rc);
}                                               // end 'dfsJfsFindPath'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Search filename in INODE DIR-entries and BTree pages from dir INODE LSN
// Optional rename to SAME LENGTH new name
/*****************************************************************************/
static BOOL dfsJfsSearchDinode                  // OUT   DIR slots resolved
(
   ULONG               inode,                   // IN    directory INODE nr
   ULONG               loud,                    // IN    Show progress
   char               *entry,                   // IN    entry specification
   char               *rename,                  // IN    same length new name
   ULONG              *nextIno                  // OUT   found Inode number
)
{
   BOOL                rc = FALSE;
   BYTE                st;
   S_JINODE           *ino;
   ULONG               inodeLsn = dfsJfsInode2Lsn( inode, &jfs->Fs1);

   ENTER();
   TRACES(( "Search Dinode: %8.8x for '%s'\n", inode, entry));

   if (dfsJfsReadChkInode( inodeLsn, &st, &ino) == NO_ERROR)
   {
      if (st == ST_JIDIR)                       // Fileset directory
      {
         rc = dfsJfsDirSlot2Entry( loud, entry, rename,
                                   ino->Slot, JFS_IN_SLOTS,
                                   ino->Dir.sTable, ino->Dir.sCount,
                                  (ino->Dir.rflag & JFS_BT_INTN),
                                   nextIno);
         if ((rc == TRUE) && (rename))          // need to write back
         {
            if (DFSTORE_WRITE_ALLOWED)
            {
               if (dfsWrite( inodeLsn, 1, (BYTE *) ino) != NO_ERROR)
               {
                  rc = FALSE;
               }
            }
         }
      }
      TxFreeMem( ino);
   }
   BRETURN (rc);
}                                               // end 'dfsJfsSearchDinode'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Search filename in Directory Btree+ page, recurse into tree
// Optional rename to SAME LENGTH new name
/*****************************************************************************/
static BOOL dfsJfsSearchDtreePage               // RET   dir entry found
(
   ULONG               loud,                    // IN    Show progress
   char               *entry,                   // IN    entry specification
   char               *rename,                  // IN    same length new name
   ULONG               pageLsn,                 // IN    start LSN of page
   ULONG              *ino                      // OUT   found entry inode
)
{
   ULONG               rc = NO_ERROR;           // rc, sector match
   BYTE                id = ST_UDATA;           // type of sector
   BYTE               *sb;                      // sector buffer
   BYTE               *st;                      // sort table location

   ENTER();
   TRACES(( "Search Dpage at %8.8x for '%s'\n", pageLsn, entry));

   if (loud)
   {
      TxPrint("\nDtree - Page  LSN : 0x%08.8X", pageLsn);
   }
   if ((sb = TxAlloc(1, JFS_PSIZE)) != NULL)
   {
      if ((rc = dfsRead( pageLsn, JFS_PSECT, sb)) == NO_ERROR)
      {
         id = dfsIdentifySector( pageLsn, 0, sb); // double-check sector type
         if (id == ST_JDTPG)
         {
            S_DTHEAD  *sd = (S_DTHEAD *) sb;

            st = ((BYTE *) sd) + (sd->stIndex * JFS_SLOTSIZE); // sort-table

            rc = dfsJfsDirSlot2Entry( loud, entry, rename,
                                      sd->slot,
                                     (sd->self.len * jfs->SectorsPerBlock * JFS_DB_SLOTS) -1,
                                      st, sd->sCount,
                                     (sd->rflag & JFS_BT_INTN),
                                      ino);
            if ((rc == TRUE) && (rename))   // need to write back
            {
               if (DFSTORE_WRITE_ALLOWED)
               {
                  if (dfsWrite( pageLsn, JFS_PSECT, sb) != NO_ERROR)
                  {
                     rc = FALSE;
                  }
               }
               else
               {
                  rc = DFS_READ_ONLY;
               }
            }
         }
      }
      TxFreeMem( sb);
   }
   BRETURN (rc);
}                                               // end 'dfsJfsSearchDtreePage'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Find DIR-entry for specified name, start in slot table, follow the tree
// Optional rename to SAME LENGTH new name, caller must WRITE BACK slot-array!
/*****************************************************************************/
static BOOL dfsJfsDirSlot2Entry                 // OUT   entry found
(
   ULONG               loud,                    // IN    Show progress
   char               *entry,                   // IN    entry specification
   char               *rename,                  // IN    same length new name
   S_JDSLOT           *slot,                    // IN    Directory slot array
   BYTE                maxslot,                 // IN    maximum nr of slots
   BYTE               *sTable,                  // IN    sort-table array
   BYTE                sCount,                  // IN    used entries in sTable
   BOOL                treeNodes,               // IN    tree internal nodes
   ULONG              *ino                      // OUT   found dir/file INODE nr
)
{
   BOOL                rc = FALSE;
   TXLN                fname;                   // upercase candidate name ASCII
   TXLN                uname;                   // upercase search-name    ASCII
   S_JDSLOT           *de;                      // single directory slot
   BYTE                si;                      // sort index
   BYTE                slotnr;
   ULONG               bdown = 0;

   ENTER();

   TxStrToUpper( strcpy( uname, entry));

   TRACES(( "Slot:%8.8x maxslot:%u  sTable:%8.8x sCount:%u  entry: '%s'\n",
             slot, maxslot, sTable, sCount, entry));

   for ( si = 0;
        (si < sCount) && (rc == FALSE);         // used entries in sort-table
         si++)
   {
      slotnr = sTable[(int) si] -1;             // ZERO based slotnumber
      de     = slot + slotnr;                   // Slot with Inode + name

      TRACES(( "si: %u  index: %hu = slotnr: %hu\n", si, sTable[(int) si], slotnr));

      if (treeNodes)                            // Btree split format
      {
         strcpy( fname, "");
         dfsJfsBtspNameAppend( slot, fname, slotnr, maxslot);
         if (strlen( fname))                    // splitpoint name present
         {
            TxStrToUpper( fname);               // make all uppercase
            if (strncmp( uname, fname, strlen(uname)) < 0) // split is larger
            {
               TRACES(( "Bdown: %8.8x just before '%s'\n", bdown, fname));
               break;                           // found our splitpoint
            }
         }
         bdown = dfsJfsBlock2Lsn( dfsJfsPxdValue( de->t.Page));
      }
      else                                      // regular directory format
      {
         if (de->i.Inode < jfs->Fs1.Max)        // sanity check
         {
            strcpy( fname, "");
            dfsJfsFileNameAppend( slot, fname, slotnr, maxslot);
            if (strlen( fname))
            {
               TxStrToUpper( fname);            // make all uppercase
               if (strcmp( uname, fname) == 0)  // exact match
               {
                  if (rename != NULL)           // rename the file in-place
                  {                             // keeping exact same length
                     dfsJfsFileNameUpdate( slot, rename, slotnr, maxslot);
                  }
                  if (ino != NULL)
                  {
                     *ino = de->i.Inode;        // Inode number
                  }
                  rc   = TRUE;
               }
            }
         }
      }
   }
   if (treeNodes && (bdown != 0))               // search the found Btree link
   {
      rc = dfsJfsSearchDtreePage( loud, entry, rename, bdown, ino);
   }
   BRETURN (rc);
}                                               // end 'dfsJfsDirSlot2Entry'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set the alternate label location, in the superblock, using specified label
/*****************************************************************************/
ULONG dfsJfsSetAltBrecLabel
(
   ULN64               v1,                      // IN    unused
   ULN64               v2,                      // IN    unused
   char               *str,                     // IN    new label value (11)
   void               *param                    // IN    unused
)
{
   ULONG               rc = NO_ERROR;

   ENTER();
   TRACES(( "label: '%s'\n", str));

   memcpy( jfs->sup->FsVolume, str, BT_LABL_L);
   rc = dfsJfsWriteSuperBlocks( TxaOptNum('s', NULL, JFSB_BOTH));
   if (rc == NO_ERROR)
   {
      TxPrint( "New  Volume label : '%s' synchronized to both superblocks\n", str);
   }
   RETURN (rc);
}                                               // end 'dfsJfsSetAltBrecLabel'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Update filename in-place, with a new name of the exact same length
/*****************************************************************************/
ULONG dfsJfsUpdateFileName
(
   ULN64               inoLsn,                  // IN    Inode sectornumber
   ULN64               info,                    // IN    unused
   char               *str,                     // IN    current filename
   void               *param                    // IN    new same length filename
)
{
   ULONG               rc = NO_ERROR;
   ULN64               dirInoLsn;
   char               *newName = (char *) param;

   ENTER();
   TRACES(( "inoLsn: %llx  name:'%s' newName:'%s'\n", inoLsn, str, newName));

   if (dfsJfsInode2Parent( inoLsn, NULL, &dirInoLsn))
   {
      if (dfsJfsSearchDinode( dfsJfsLsn2Inode( dirInoLsn, &jfs->Fs1),
                              FALSE, str, newName, NULL))
      {
         ULONG        inode = dfsJfsLsn2Inode( inoLsn, &jfs->Fs1);

         //- Need to update the Name/Parent cache, or name will stick for a while!
         if ((jfs->Ic)                 &&       // name cache there
             (inode < jfs->Fs1.Max)    &&       // in normal inode range
             (jfs->Ic[ inode].Name)     )       // and name present
         {
            strcpy( jfs->Ic[ inode].Name, newName); // direct copy to cache
         }
      }
      else
      {
         rc = DFS_CMD_FAILED;                   // either not found (unlikely)
      }                                         // or write-back failed ...
   }
   else
   {
      rc = DFS_ST_MISMATCH;
   }
   RETURN (rc);
}                                               // end 'dfsJfsUpdateFileName'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Replace sn-list by contents of a single Directory (InoLsn, no info)
/*****************************************************************************/
ULONG dfsJfsMakeBrowseList                      // RET   LSN is valid INODE
(
   ULN64               inoLsn,                  // IN    Inode sectornumber
   ULN64               info,                    // IN    unused
   char               *str,                     // IN    unused
   void               *param                    // INOUT unused
)
{
   ULONG               rc = NO_ERROR;
   BYTE                st;
   S_JINODE           *ino;

   ENTER();
   TRACES(( "inoLsn: %llx\n", inoLsn));

   if ((rc = dfsJfsReadChkInode( inoLsn, &st, &ino)) == NO_ERROR)
   {
      if (st == ST_JIDIR)                       // Fileset directory
      {
         dfsInitList(0, "-f -P", "-d");         // optimal for menu file-recovery

         if (ino->Self != JFS_ROOT_INODE)       // no .. for ROOT (consistency)
         {
            TRACES(("List marked as 1ST_PARENT\n"));
            DFSBR_SnlistFlag |= DFSBR_1ST_PARENT;
            if (jfs->Ic)                        // get parent from cache
            {
               dfsAdd2SectorList( dfsJfsInode2Lsn( jfs->Ic[ ino->Self].Parent, &jfs->Fs1));
            }
            else                                // return ROOT as parent when not known
            {
               dfsAdd2SectorList( dfsJfsInode2Lsn( JFS_ROOT_INODE, &jfs->Fs1));
            }
         }
         rc = dfsJfsDirSlots2List( ino->Slot,
                                   JFS_IN_SLOTS,
                                   ino->Dir.sTable, // sort-table
                                   ino->Dir.sCount,
                                  (ino->Dir.rflag & JFS_BT_INTN));
      }
      else                                      // not a directory
      {
         rc = DFS_BAD_STRUCTURE;
      }
      TxFreeMem( ino);
   }
   RETURN (rc);
}                                               // end 'dfsJfsMakeBrowseList'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get allocation information for file-DATA into integrated-SPACE structure
// 20161215: Include EA-data in separate allocation (in the ispace)
/*****************************************************************************/
ULONG dfsJfsGetAllocSpace
(
   ULN64               inoLsn,                  // IN    Inode sectornumber
   ULN64               info,                    // IN    unused
   char               *str,                     // IN    unused
   void               *param                    // INOUT Integrated SPACE
)
{
   ULONG               rc = NO_ERROR;
   S_JINODE           *ino;
   DFSISPACE          *ispace = (DFSISPACE *) param;

   ENTER();

   ispace->space  = NULL;
   ispace->chunks = 0;

   if ((rc = dfsJfsReadChkInode( inoLsn, NULL, &ino)) == NO_ERROR)
   {
      if (ino->FileSize != 0)
      {
         rc = dfsJfsInodeData2Space( ino, 0, ispace, NULL, NULL);
      }
      if (rc == NO_ERROR)
      {
         rc = dfsJfsReadEaInode(  ino, &ispace->xasize, (S_FEA2LIST **) &ispace->xattrs);
         if ((rc == NO_ERROR) && (ispace->xattrs != NULL))
         {
            ispace->xatype = DFSXA_FEA2;
         }
      }
      TxFreeMem( ino);
   }
   RETURN (rc);
}                                               // end 'dfsJfsGetAllocSpace'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add inoLsn for all entries in Directory Btree+ page to sectorlist, recurse
/*****************************************************************************/
static ULONG dfsJfsDtreePage2List               // RET   dir entry found
(
   ULONG               pageLsn                  // IN    start LSN of page
)
{
   ULONG               rc = NO_ERROR;           // rc, sector match
   BYTE                id = ST_UDATA;           // type of sector
   BYTE               *sb;                      // sector buffer
   BYTE               *st;                      // sort table location

   ENTER();
   TRACES(( "DtreePage at %8.8x\n", pageLsn));

   if ((sb = TxAlloc(1, JFS_PSIZE)) != NULL)
   {
      if ((rc = dfsRead( pageLsn, JFS_PSECT, sb)) == NO_ERROR)
      {
         id = dfsIdentifySector( pageLsn, 0, sb); // double-check sector type
         if (id == ST_JDTPG)
         {
            S_DTHEAD  *sd = (S_DTHEAD *) sb;

            st = ((BYTE *) sd) + (sd->stIndex * JFS_SLOTSIZE); // sort-table

            rc = dfsJfsDirSlots2List( sd->slot,
                                     (sd->self.len * jfs->SectorsPerBlock * JFS_DB_SLOTS) -1,
                                      st, sd->sCount,
                                     (sd->rflag & JFS_BT_INTN));
         }
      }
      TxFreeMem( sb);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsJfsDtreePage2List'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add DIR-entry inoLSNs to list, start at slot table, iterate over whole tree
/*****************************************************************************/
static ULONG dfsJfsDirSlots2List
(
   S_JDSLOT           *slot,                    // IN    Directory slot array
   BYTE                maxslot,                 // IN    maximum nr of slots
   BYTE               *sTable,                  // IN    sort-table array
   BYTE                sCount,                  // IN    used entries in sTable
   BOOL                treeNodes                // IN    tree internal nodes
)
{
   ULONG               rc = NO_ERROR;           // rc, sector match
   S_JDSLOT           *de;                      // single directory slot
   BYTE                si;                      // sort index
   BYTE                slotnr;

   ENTER();

   TRACES(( "Slot:%8.8x maxslot:%u  sTable:%8.8x sCount:%u\n",
             slot, maxslot, sTable, sCount));

   for ( si = 0;
        (si < sCount) && (rc == NO_ERROR);      // used entries in sort-table
         si++)
   {
      slotnr = sTable[(int) si] -1;             // ZERO based slotnumber
      de     = slot + slotnr;                   // Slot with Inode + name

      TRACES(( "si: %u  index: %hu = slotnr: %hu\n", si, sTable[(int) si], slotnr));

      if (treeNodes)                            // Btree split format
      {
         rc = dfsJfsDtreePage2List( dfsJfsBlock2Lsn( dfsJfsPxdValue( de->t.Page)));
      }
      else                                      // regular dir entry, add inode LSN
      {
         BOOL          filtered = FALSE;
         ULN64         inodeLsn = dfsJfsInode2Lsn( de->i.Inode, &jfs->Fs1);

         if (dfsa->browseShowHidden == FALSE)   // need to check hidden/system attribute
         {
            if (de->i.FileName[0] == '.')       // filename starts with DOT, hidden (UNIX like)
            {
               filtered = TRUE;
            }
            else                                // check regular FS attributes (FAT like)
            {
               BYTE                attr;
               S_JINODE           *ino;

               if (dfsJfsReadChkInode( inodeLsn, NULL, &ino) == NO_ERROR)
               {
                  attr = (BYTE) (ino->Mode >> JFS_FATTRSHIFT);
                  if ((attr & FATTR_SYSTEM) || (attr & FATTR_HIDDEN))
                  {
                     filtered = TRUE;
                  }
                  TxFreeMem( ino);
               }
            }
         }
         if (!filtered)
         {
            rc = dfsAdd2SectorList( inodeLsn);
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsJfsDirSlots2List'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate Block-nr to LSN, generic interface
/*****************************************************************************/
ULN64 dfsJfsCl2Lsn                              // RET   LSN for this block
(
   ULN64               block,                   // IN    block number
   ULN64               d2,                      // IN    dummy
   char               *p1,                      // IN    dummy
   void               *p2                       // IN    dummy
)
{
   ULN64               lsn = block;

   lsn = block * jfs->SectorsPerBlock;
   return (lsn);
}                                               // end 'dfsJfsCl2Lsn'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate LSN to Block-nr, generic interface
/*****************************************************************************/
ULN64 dfsJfsLsn2Cl                              // RET   block for this LSN
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *p1,                      // IN    dummy
   void               *p2                       // IN    dummy
)
{
   ULN64               block = 0;

   block = lsn / jfs->SectorsPerBlock;
   return (block);
}                                               // end 'dfsJfsLsn2Cl'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Determine allocation-bit for specified LSN, ALLOCATED beyond last block!
/*****************************************************************************/
ULONG dfsJfsAllocated                           // RET   LSN is allocated
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               asn = dfstAreaD2Part( DFSTORE, lsn); // Area aware sn

   if ((jfs->Bm.dmap != NULL) && (asn < jfs->Bm.LimitLsn)) // inside the bitmap
   {
      if (asn < (jfs->Sect))                    // within valid block area
      {
         return((ULONG) dfsJfsBitmapCache( dfsJfsLsn2Block( asn), NULL));
      }
      else                                      // return ALLOCATED for any
      {                                         // lsn beyond last block
         return( 1);                            // to avoid CHECK errors
      }                                         // on non-standard bitmaps
   }
   else
   {
      return(DFS_PSN_LIMIT);
   }
}                                               // end 'dfsJfsAllocated'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Align R/W cache to position for block, and get current allocation-bit
/*****************************************************************************/
BOOL dfsJfsBitmapCache                          // RET   block is allocated
(
   ULN64               bl,                      // IN    Block number
   ULONG              *cp                       // OUT   position in cache
)                                               //       or L32_NULL if invalid
{
   ULONG               rc = NO_ERROR;
   BOOL                al = TRUE;               // default allocated
   ULONG               rdm;                     // relative dmap page in map
   ULONG               dmapPageIndex;           // pagenr in Block Alloc Map
   ULONG               u32pos = L32_NULL;       // u32 position in map
   ULONG               alloc_u32;

   //- ULONG               bit_mask = (1 << (0x1f - (bl & 0x1f))); // for trace verify!

   if (bl < jfs->VolumeBlocks)                  // within valid block range?
   {
      rdm = (ULONG) (bl >> 13);                 // find relative page for bl

      if (rdm != jfs->Bm.Rdmap)                 // other relative dmap page
      {
         if (jfs->Bm.Dirty)                     // need to flush changes ?
         {
            rc = dfsJfsBitmapFlush( FALSE);     // flush, but keep cache
         }
         if (rc == NO_ERROR)
         {
            jfs->Bm.Rdmap = rdm;
            dmapPageIndex = rdm + (rdm >> 10) + (rdm >> 20) + 4;
            TRACES(("Read bitmap file page: %u for dmap %u\n", dmapPageIndex, rdm));
            rc = dfsSspaceReadFilePart( jfs->Bm.File.chunks,
                                        jfs->Bm.File.space,
                                        dmapPageIndex * JFS_PSECT, JFS_PSECT,
                               (BYTE *) jfs->Bm.dmap);
            if (rc == NO_ERROR)
            {
               TRACES(("Dmap rdm: %u, start Block:%12.12llx  Free blocks:%u from:%u\n",
                        rdm, jfs->Bm.dmap->blStart, jfs->Bm.dmap->bFree, jfs->Bm.dmap->blocks));
            }
         }
      }
      if (rc == NO_ERROR)
      {
         u32pos    = (ULONG) (bl &  0x1fff) >> 5; // u32 index in map
         alloc_u32 = jfs->Bm.dmap->blBitMap[ u32pos];
         if (((alloc_u32 << (bl & 0x1f)) & 0x80000000) == 0) // MSB is lowest!
         {
            al = FALSE;
         }
         //- TRACES(("Cache:%12.12llx u32pos:%8.8x u32:%8.8x mask:%8.8x (Free: %u)  = %s\n",
         //-          bl, u32pos, alloc_u32, bit_mask, jfs->Bm.dmap->bFree, (al) ? "USED" : "FREE"));

      }
   }
   if (cp != NULL)                              // return dword position too ?
   {
      *cp = u32pos;                             // L64_NULL when invalid range
   }
   return (al);
}                                               // end 'dfsJfsBitmapCache'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Extract BitMap allocation S_SPACE structure and initialize the BitMap cache
// Note: Area aware to allow usage from FDISK mode too
/*****************************************************************************/
ULONG dfsJfsBitMapInit                          // RET   rc = 0 if type match
(
   void
)
{
   ULONG               rc = 0;                  // rc, sector match

   ENTER();

   jfs->Bm.LimitLsn = 0;                        // signal no BitMaps present
   if ((jfs->Bm.dmap = TxAlloc( 1, sizeof(S_DMAP))) != NULL)
   {
      rc = dfsRead( dfstAreaP2Disk( DFSTORE, LSN_AITAB + JFS_BLOCKAMAP_INODE), 1, rbuf);
      if (rc == NO_ERROR)
      {
         //- create Area-corrected Sspace, so ReadFilePart needs no changes
         rc = dfsJfsInodeData2Space( (S_JINODE *) rbuf, 0, &jfs->Bm.File, NULL, NULL);
         if ((rc == NO_ERROR) && (jfs->Bm.File.chunks != 0))
         {
            rc = dfsSspaceReadFilePart( jfs->Bm.File.chunks,
                                        jfs->Bm.File.space,
                                        0, JFS_PSECT, rbuf); // control page (hdr)
            if (rc == NO_ERROR)
            {
               S_BMAP *cp = (S_BMAP *) rbuf;    // BAM control page
               jfs->Bm.LimitLsn  = cp->blocks * jfs->SectorsPerBlock;
               jfs->Bm.FreeSect  = cp->bFree  * jfs->SectorsPerBlock;

               jfs->Bm.Dirty = FALSE;
               jfs->Bm.Rdmap = 0;
               rc = dfsSspaceReadFilePart( jfs->Bm.File.chunks,
                                           jfs->Bm.File.space,
                                           4 * JFS_PSECT, JFS_PSECT, // first dmap page
                                  (BYTE *) jfs->Bm.dmap);
            }
         }
      }
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsJfsBitMapInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Flush BitMap cache when Dirty, and optional free its resources
/*****************************************************************************/
ULONG dfsJfsBitmapFlush                         // RET   rc
(
   BOOL                terminate                // IN    terminate cache too
)
{
   ULONG               rc = NO_ERROR;           // rc
   ULONG               rdm;                     // relative dmap page in map
   ULONG               dmapPageIndex;           // pagenr in Block Alloc Map

   ENTER();

   if (jfs->Bm.Dirty)                          // need to flush changes ?
   {
      TRACES(( "dirty, flush it ...\n"));

      //- update tree (binary buddy) + summary pages L2, L1, L0 to make it perfect (or require CHKDSK)
      //- Summary pages are located just after the header (block 0), in block 1, 2 and 3.

      //- Sync the persistent bitmap to the working one
      memcpy( jfs->Bm.dmap->blPrsMap, jfs->Bm.dmap->blBitMap, JFS_DMAPSZ * sizeof(ULONG));

      rdm = jfs->Bm.Rdmap;
      dmapPageIndex = rdm + (rdm >> 10) + (rdm >> 20) + 4;
      rc = dfsSspaceWriteFilePart( jfs->Bm.File.chunks,
                                   jfs->Bm.File.space,
                                   dmapPageIndex * JFS_PSECT, JFS_PSECT,
                          (BYTE *) jfs->Bm.dmap);

      jfs->Bm.Dirty = FALSE;                    // mark it clean again
   }
   else
   {
      TRACES(( "not dirty ...\n"));
   }
   if (terminate)
   {
      TxFreeMem( jfs->Bm.File.space);
      TxFreeMem( jfs->Bm.dmap);

      jfs->Bm.Rdmap    = 0;
      jfs->Bm.LimitLsn = 0;
   }
   RETURN (rc);
}                                               // end 'dfsJfsBitmapFlush'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set allocate status for LSN to specified value (can go beyond FS, to end bitmap)
/*****************************************************************************/
ULONG dfsJfsSetAlloc                            // RET   LSN allocation set
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *value,                   // IN    NULL = not allocated
   void               *ref                      // IN    dummy
)
{
   if ((jfs->Bm.dmap != NULL) && (lsn < jfs->Bm.LimitLsn)) // inside the bitmap
   {
      return( dfsJfsSetBlock( dfsJfsLsn2Block( lsn), (value != NULL)));
   }
   else
   {
      return( DFS_PSN_LIMIT);
   }
}                                               // end 'dfsJfsSetAlloc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set allocate status for BLOCK to specified value
/*****************************************************************************/
ULONG dfsJfsSetBlock                            // RET   BL  allocation set
(
   ULN64               bl,                      // IN    Block
   BOOL                value                    // IN    SET allocation bit
)
{
   ULONG               rc = NO_ERROR;
   ULONG               u32pos;
   ULONG               alloc_u32;
   ULONG               bit_mask = (1 << (0x1f - (bl & 0x1f))); // MSB is lowest!

   ENTER();

   TRACES(("Set bl:%llx to %s using bit_mask:%8.8x\n", bl, (value) ? "ALLOCATED" : "FREE", bit_mask));

   dfsJfsBitmapCache( bl, &u32pos);             // position cache at page/block
   if (u32pos != L32_NULL)                      // block position valid ?
   {
      alloc_u32 = jfs->Bm.dmap->blBitMap[ u32pos];

      if (value)                                // set the allocation bit
      {
         if ((alloc_u32 & bit_mask) == 0)       // bit is not set now
         {
            if (jfs->Bm.dmap->bFree != 0)
            {
               jfs->Bm.dmap->bFree--;
            }
            jfs->Bm.dmap->blBitMap[ u32pos] = alloc_u32 |  bit_mask;
         }
      }
      else                                      // reset allocation bit
      {
         if ((alloc_u32 & bit_mask) != 0)       // bit is set now
         {
            if (jfs->Bm.dmap->bFree < jfs->Bm.dmap->blocks)
            {
               jfs->Bm.dmap->bFree++;
            }
            jfs->Bm.dmap->blBitMap[ u32pos] = alloc_u32 & ~bit_mask;
         }
      }
      TRACES(( "%s cache-u32 at 0x%4.4x for bl:%8.8x "
                  "from 0x%8.8x to 0x%8.8x  (Free now: %u)\n",
                (value) ? "Set" : "Clr", u32pos, bl, alloc_u32,
                 jfs->Bm.dmap->blBitMap[ u32pos], jfs->Bm.dmap->bFree));

      jfs->Bm.Dirty = TRUE;                     // mark cache dirty
   }
   else
   {
      rc = DFS_PSN_LIMIT;
   }
   RETURN (rc);
}                                               // end 'dfsJfsSetBlock'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Write cached superblock to 1st and 2nd superblock location on disk
// Make sure the jfs-sup cache always is a copy of the 1st superblock
/*****************************************************************************/
ULONG dfsJfsWriteSuperBlocks
(
   ULONG               sbid                     // IN    superblock id
)
{
   ULONG               rc  = NO_ERROR;
   ULN64               d2  = NO_ERROR;

   ENTER();

   if (DFSTORE_WRITE_ALLOWED)
   {
      TxPrint("Superblock update : ");
      if (sbid & JFSB_1ST)
      {
         TxPrint( "1st location 0x%2.2x ", LSN_JFSUP);
         rc = dfsWrite(   LSN_JFSUP, 1, (BYTE   *) jfs->sup);
         TxPrint( "%s; ", (rc == NO_ERROR) ? "written" : "failed");
      }
      if (sbid & JFSB_2ND)
      {
         TxPrint( "2nd location 0x%2.2x ", LSN_JFSU2);
         d2 = dfsWrite(LSN_JFSU2, 1, (BYTE   *) jfs->sup);
         TxPrint( "%s", (d2 == NO_ERROR) ? "written" : "failed");
         if (rc == NO_ERROR)
         {
            rc = d2;
         }
      }
      if (sbid == JFSB_2ND)                     // re-sync super cache
      {
         dfsRead( LSN_JFSUP, 1, (BYTE   *) jfs->sup);
      }
      TxPrint( "\n");
   }
   else
   {
      rc = DFS_READ_ONLY;
   }
   RETURN(rc);
}                                               // end 'dfsJfsWriteSuperBlocks'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Quickly find normal/deleted Inodes using the Inode MAP
/*****************************************************************************/
ULONG dfsJfsQuickFindInodes
(
   BOOL                verbose,                 // IN    List files while found
   ULONG               first,                   // IN    First inode todo
   char               *types,                   // IN    Inode types to search
   char               *select                   // IN    Selection string for
)                                               //       resolved name (if any)
{
   ULONG               rc = NO_ERROR;           // function return
   BYTE                st;
   S_JINODE           *ino;
   ULONG               inode;
   ULN64               lsn;
   BOOL                match;
   TXLN                text;                    // name / result text buffer
   ULONG               ei = 0;                  // entries listed

   ENTER();

   if (jfs->Fs1.Inode)                          // mapping available ?
   {
      dfsInitList(0, "-f -P", "-d");            // optimal for menu file-recovery
      dfsProgressInit( 0, jfs->Fs1.Max, 0, " Inode:", "searched",
                      (verbose) ? DFSP_STAT | DFSP_VDEC : DFSP_BARS | DFSP_VDEC, 0);

      for ( inode = first; inode < jfs->Fs1.Max; inode++)
      {
         lsn = dfsJfsInode2Lsn( inode, &jfs->Fs1);
         if ((lsn != L64_NULL) && (lsn != 0))
         {
            if ((rc = dfsJfsReadChkInode( lsn, &st, &ino)) == NO_ERROR)
            {
               if ((st == ST_JINOD) || (st == ST_JIDEL) ||
                   (st == ST_JIDIR) || (st == ST_JIDDI))
               {
                  if (strchr( types, st) != NULL)
                  {
                     if (select && strlen(select))
                     {
                        dfsJfsResolveInodeName( lsn, ino, st, NULL, text);
                        match = (strstr( text, select) != NULL);
                     }
                     else                       // no name selection
                     {
                        match = TRUE;
                     }
                     if (match)
                     {
                        dfsAdd2SectorList( lsn);

                        if (verbose)            // list all 100% alloc OK
                        {
                           sprintf( text, ".%5.5u = %12.12llX : ", ei++, lsn);
                           dfsJfsFileInfo( lsn, 0, "*%100", text);
                        }
                     }
                  }
               }
               TxFreeMem( ino);
            }
            else if (rc == DFS_ST_MISMATCH)     // ignore sector type mismatch
            {
               rc = NO_ERROR;
            }
            else if (dfsa->eStrategy != TXAE_IGNORE) // unless -E:i
            {
               break;
            }
         }
         if ((inode & 0xff) == 0)               // every 256, unallocated too
         {
            if (TxAbort())
            {
               break;
            }
            else
            {
               dfsProgressShow( inode, 0, dfsa->snlist[0], "Found:");
            }
         }
      }
      dfsProgressTerm();
      sprintf( text, ", found %llu Inodes\n", dfsa->snlist[0]);
      dfsProgressElapsed( "Elapsed time      : ", text);
   }
   RETURN (rc);
}                                               // end 'dfsJfsQuickFindInodes'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DFS JFS read and check type for a JFS Inode based on its sector number
/*****************************************************************************/
ULONG dfsJfsReadChkInode
(
   ULN64               lsn,                     // IN    Inode LSN
   BYTE               *stype,                   // OUT   Inode type, or NULL
   S_JINODE          **inode                    // OUT   Inode structure
)
{
   ULONG               rc = NO_ERROR;
   BYTE                st = ST_UDATA;
   S_JINODE           *ino;

   ENTER();
   TRARGS(("Inode LSN: %llX\n", lsn));

   if ((ino = TxAlloc( 1, sizeof( S_JINODE))) != NULL)
   {
      if ((rc = dfsRead( lsn, 1, (BYTE   *) ino)) == NO_ERROR)
      {
         switch (st = dfsIdentifySector( lsn, 0, (BYTE   * ) ino))
         {
            case ST_JINAG:
            case ST_JINOD:
            case ST_JIDEL:
            case ST_JIDIR:
            case ST_JIDDI:                      // any Inode type is OK
               break;

            default:
               rc = DFS_ST_MISMATCH;
               break;
         }
      }
      if (rc != NO_ERROR)
      {
         TxFreeMem( ino);
      }
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   *inode = ino;
   if (stype)
   {
      *stype = st;
   }
   RETURN(rc);
}                                               // end 'dfsJfsReadChkInode'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Convert Inode external-data description (Xtree) to an S_SPACE structure
// There will be a separate S_SPACE for the actual data, and tree meta-data
/*****************************************************************************/
ULONG dfsJfsInodeData2Space
(
   S_JINODE           *inode,                   // IN    Inode with alloc info
   ULONG               showTree,                // IN    display indent or 0
   DFSISPACE          *is,                      // OUT   Integrated S_SPACE
   ULONG              *metaChunks,              // OUT   nr of meta extents
   S_SPACE           **metaSpace                // OUT   meta extent space
)
{
   ULONG               rc = NO_ERROR;           // function return
   int                 i;
   ULONG               xads;
   DFSISPACE           isNew;                   // data space
   ULONG               mex    = 0;              // meta extents
   S_SPACE            *mSpace = NULL;           // meta space

   ENTER();
   TRACES(( "Inode:%8.8x  space:%8.8x\n", inode->Self, is));

   memset( is,     0, sizeof(DFSISPACE));       // initialize to all zero
   memset( &isNew, 0, sizeof(DFSISPACE));       // initialize to all zero

   is->clsize = jfs->SectorsPerBlock;

   if ((!S_ISDIR( inode->Mode  )) &&            // Dir has NO data allocation
       (inode->File.xCount >   2) &&            // external data available
       (inode->File.xCount <= 18) &&            // upto 16 extents descriptors
       (inode->FileSize    >   0)  )            // and not an empty file
   {
      xads = inode->File.xCount-2;              // nr of XAD links in Inode
      TRACES(( "XADs in inode: %u  is->chunks: %u\n", xads, is->chunks));

      if      (inode->File.rflag & JFS_BT_LEAF) // XAD links to data extents
      {
         rc = dfsJfsXads2Space( inode->Xad, xads, is);
      }
      else if (inode->File.rflag & JFS_BT_INTN) // XAD links to internal nodes
      {
         if (metaChunks && metaSpace)           // want meta stuff too
         {
            if ((mSpace = TxAlloc( xads, sizeof(S_SPACE))) != NULL)
            {
               for (i = 0; i < xads; i++)
               {
                  mSpace[i].start = inode->Xad[i].pe.lo  * is->clsize; // location
                  mSpace[i].size  = inode->Xad[i].pe.len * is->clsize; // #sectors
               }
               *metaChunks = xads;
               *metaSpace  = mSpace;
            }
            else
            {
               rc = DFS_ALLOC_ERROR;
            }
         }

         mSpace = NULL;
         for (i = 0; (i < xads) && (rc == NO_ERROR); i++)
         {
            rc = dfsJfsXtpData2Space( inode->Xad[i].pe.lo,
                                      showTree, &isNew, &mex, &mSpace);
            if (rc == NO_ERROR)
            {                                   // join main data stream
               rc = dfsSspaceJoin( is->chunks,     is->space,
                                   isNew.chunks,   isNew.space,
                                  &is->chunks,    &is->space);

               if ((rc == NO_ERROR)          &&
                   (metaChunks && metaSpace) && // join lower-level meta-data
                   (mex != 0)  &&   (mSpace)  ) // if any was created ...
               {
                  rc = dfsSspaceJoin( *metaChunks, *metaSpace, mex, mSpace,
                                       metaChunks,  metaSpace);
               }
            }
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsJfsInodeData2Space'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Convert Xtree page external-data to an S_SPACE structure (could recurse)
// There will be a separate S_SPACE for the actual data, and tree meta-data
// *metaChunks MUST be initialized to ZERO by the top level caller!
/*****************************************************************************/
ULONG dfsJfsXtpData2Space
(
   ULN64               block,                   // IN    Tree page block nr
   ULONG               showTree,                // IN    display indent or 0
   DFSISPACE          *is,                      // INOUT Integrated S_SPACE
   ULONG              *metaChunks,              // INOUT nr of meta extents
   S_SPACE           **metaSpace                // INOUT meta extent space
)
{
   ULONG               rc = NO_ERROR;           // function return
   int                 i;
   ULONG               xads;
   DFSISPACE           isNew;                   // data space
   ULONG               mex    = 0;              // meta extents
   S_SPACE            *mSpace = NULL;           // meta space
   S_XTPAGE           *page;
   ULN64               pageLsn = dfsJfsBlock2Lsn( block);

   ENTER();
   TRACES(( "block:%llx is->chunks:%u is->space: %8.8x\n", block, is->chunks, is->space));

   if (showTree)
   {
      TxPrint("  Btree+ Page @Bl : %*.*s0x%12.12llX size 4.0 KiB  LSN: 0x%12.12llX\n",
                 (showTree - JFS_TREE_INDENT), (showTree - JFS_TREE_INDENT), "", block, pageLsn);
      showTree += JFS_TREE_INDENT;              // for next level ...
   }
   memset( &isNew, 0, sizeof(DFSISPACE));       // initialize to all zero

   if ((page = TxAlloc( 1, JFS_PSIZE)) != NULL) // allocate one page
   {
      if ((rc = dfsRead( pageLsn, JFS_PSECT, (BYTE   *) page)) == NO_ERROR)
      {
         if (dfsIdentifySector( pageLsn, 0, (BYTE   * ) page) == ST_JXTPG)
         {
            if (page->header.entries > 2)       // external data available
            {
               xads = (page->header.entries -2) & 0xff; // XAD links in page
               TRACES(( "XADs in xPage: %u  is->chunks: %u\n", xads, is->chunks));
               if      (page->header.tflag & JFS_BT_LEAF)
               {
                  rc = dfsJfsXads2Space( page->header.extent, xads, is);
               }
               else if (page->header.tflag & JFS_BT_INTN) // more internal nodes
               {
                  if (metaChunks && metaSpace)  // want meta stuff too
                  {
                     if ((mSpace = TxAlloc( xads, sizeof(S_SPACE))) != NULL)
                     {
                        for (i = 0; i < xads; i++)
                        {
                           mSpace[i].start = page->header.extent[i].pe.lo * jfs->SectorsPerBlock;
                           mSpace[i].size  = JFS_PSECT; // fixed size, 1 page
                        }
                        *metaChunks = xads;
                        *metaSpace  = mSpace;
                     }
                     else
                     {
                        rc = DFS_ALLOC_ERROR;
                     }
                  }

                  mSpace = NULL;
                  for (i = 0; (i < xads) && (rc == NO_ERROR); i++)
                  {                             // recursive tree walk!
                     rc = dfsJfsXtpData2Space( page->header.extent[i].pe.lo,
                                               showTree, &isNew, &mex, &mSpace);
                     if (rc == NO_ERROR)
                     {                          // join main data stream
                        rc = dfsSspaceJoin( is->chunks,     is->space,
                                            isNew.chunks,   isNew.space,
                                           &is->chunks,     &is->space);

                        if ((rc == NO_ERROR)          &&
                            (metaChunks && metaSpace) && // join lower-level
                            (mex != 0)  &&   (mSpace)  ) // if any was created
                        {
                           rc = dfsSspaceJoin( *metaChunks, *metaSpace, mex, mSpace,
                                                metaChunks,  metaSpace);
                        }
                     }
                  }
               }
            }
         }
         else                                   // incorrect page type
         {
            rc = DFS_BAD_STRUCTURE;
         }
      }
      TxFreeMem( page);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsJfsXtpData2Space'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Convert array of extent-allocation-descriptors to an S_SPACE structure
// Note: Area aware to allow usage from FDISK mode too (S_SPACE adjustment)
/*****************************************************************************/
ULONG dfsJfsXads2Space
(
   S_XAD              *xad,                     // IN    XAD Array
   ULONG               extents,                 // IN    Nr of XAD structures
   DFSISPACE          *is                       // OUT   Integrated S_SPACE
)
{
   ULONG               rc = NO_ERROR;           // function return
   int                 i;

   ENTER();

   memset( is, 0, sizeof(DFSISPACE));           // initialize to all zero

   is->clsize = jfs->SectorsPerBlock;

   //- to be refined, need to prepare for SPARSE files by allowing the relative
   //- start block (xad->rlo) for an extent to be larger than last-sector +1
   //- In such cases, add a SPARSE S_SPACE extent using start-sector LSN_SPARSE

   //- To allow sparse elements at higher level too, largest handled relative
   //- sector must be supplied/returned (INOUT ?)
   //- so the next iteration may start with a sparse extent

   //- Must allocate extra S_SPACE entries, worst case is (extents * 2)
   //- is->chunks will become larger than 'extents' in that case (++ on sparse)

   if ((is->chunks = extents) != 0)
   {
      if ((is->space = TxAlloc( extents, sizeof(S_SPACE))) != NULL)
      {
         for (i = 0; i < extents; i++)
         {
            is->space[i].start = dfstAreaP2Disk( DFSTORE, (xad[i].pe.lo  * is->clsize));
            is->space[i].size  =                           xad[i].pe.len * is->clsize;
         }
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   RETURN (rc);
}                                               // end 'dfsJfsXads2Space'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display file allocation and path info for LSN
/*****************************************************************************/
ULONG dfsJfsFileInfo                            // RET   LSN is valid INODE
(
   ULN64               lsn,                     // IN    Inode LSN
   ULN64               d2,                      // IN    dummy
   char               *select,                  // IN    Inode select wildcard
   void               *param                    // INOUT leading text/shortpath
)
{
   ULONG               rc = NO_ERROR;
   TXLN                fpath;
   ULN64               root;
   ULN64               size = 0;                // allocated size, sectors
   ULN64               bads = 0;                // bad allocated sectors
   BYTE                st = ST_JINOD;
   TXLN                text;
   ULONG               percent   = 100;
   ULONG               threshold = 0;           // threshold percentage
   BOOL                isMinimum = TRUE;        // threshold is minimum value
   ULN64               location  = 0;
   TXTM                temp;                    // location/date text
   ULONG               minS;                    // minimal size
   ULONG               maxS;                    // maximal size
   char                itemType  = 0;           // type Dir, File, Browse, or ANY
   time_t              modTstamp = 0;           // Modify timestamp value, or 0
   char               *lead    = param;
   BOOL                verbose = (strcmp( lead, ":") != 0);
   BOOL                output;                  // no Printf, just return fpath
   BOOL                alcheck;                 // Execute a full allocation check
   S_JINODE           *ino;

   ENTER();
   TRARGS(("LSN: %llX, select: '%s', lead: '%s'\n", lsn, select, lead));

   if (strcmp( lead, "!") == 0)                 // Skip allocation check wanted
   {
      output  = FALSE;                          // no output
      alcheck = FALSE;                          // and no allocation check
      *lead   = 0;                              // discard 'garbage' in lead string
   }
   else                                         // Possible output/alloc-check wanted
   {
      output  = (*lead != 0);                   // output if any lead given (except '!')
      alcheck = TRUE;                           // but always an alloc check if no  '!'
   }
   TRACES(("verbose:%u output:%u alcheck:%u\n", verbose, output, alcheck));

   if ((rc = dfsJfsReadChkInode( lsn, &st, &ino)) == NO_ERROR)
   {
      if (verbose)
      {
         rc = dfsJfsInodeCheckAlloc( ino, st, (alcheck) ? "" : "s", &size, &bads, &location);
      }
      dfsParseFileSelection( select, text, &isMinimum, &threshold, &minS, &maxS, &modTstamp, &itemType);
      switch (st)
      {
         case ST_JIDIR:                         // normal or deleted directory
         case ST_JIDDI:
            if      (itemType == DFS_FS_ITEM_BROWSE)  // no filtering on Directories, take all.
            {
               strcpy( text, "");               // no wildcard
               isMinimum = TRUE; threshold = 0; // no percentage filter
               modTstamp = 0;                   // no timestamp filter
               minS      = 0;    maxS      = 0; // no size filter
            }
            else if (itemType == DFS_FS_ITEM_FILES) // filter files, discard Directories
            {
               rc = DFS_ST_MISMATCH;            // item-type mismatch
               break;
            }
         case ST_JINOD:                         // normal or deleted file
         case ST_JIDEL:
            if      (itemType == DFS_FS_ITEM_DIRS) // filter dirs, discard files
            {
               rc = DFS_ST_MISMATCH;            // item-type mismatch
               break;
            }
            dfsJfsFindRootInode( lsn, fpath, &root);
            if ((strlen(text) == 0) || (TxStrWicmp( fpath, text) >= 0))
            {
               if (verbose)                     // include alloc and size
               {
                  if (location == 0)
                  {
                     if (size == 0)
                     {
                        strcpy(  temp, "     empty ");
                        rc = NO_ERROR;          // allocation always OK
                     }
                     else
                     {
                        strcpy(  temp, "  resident ");
                        rc = NO_ERROR;          // allocation always OK
                     }
                  }
                  else if (location == L64_NULL)
                  {
                     strcpy(     temp, "    sparse ");
                  }
                  else
                  {
                     sprintf( temp, "%10llX ", location);
                  }
                  percent = dfsAllocationReliability( size, bads);

                  TRACES(( "percent:%u threshold:%u size:%u minS:%u maxS:%u\n",
                            percent, threshold, size, minS, maxS));
               }
               if ((BOOL)((percent >= threshold)) == isMinimum)            //- percentage match
               {
                  if ((ino->LastModify.hi > modTstamp) &&                  //- Timestamp match
                      ((size >= minS) && ((size <= maxS) || (maxS == 0)))) //- Size match
                  {
                     if (verbose)               // include alloc and size
                     {
                        if (alcheck)
                        {
                           sprintf( text, "%s%c %s%3u%%%s ", CBM, st,
                                    (rc == NO_ERROR) ?  CBG  :  CBR, percent, CNN);
                        }
                        else                    // no reliability percentage
                        {
                           sprintf( text, "%s%c     %s ", CBM, st, CNN);
                        }
                        strcat(    text, temp);
                        dfstrSize( text, CBC, size,          " "); // file-size, sect
                        dfstrBytes(text, CBZ, ino->Eax.size, " "); // EA-size, bytes
                     }
                     else
                     {
                        strcpy( text, "");      // remove any stray text, non-verbose
                     }
                     if (output)                // not totally silent?
                     {
                        strcat( lead, text);
                        TxPrint( "%s%s%s%s%s\n", lead, CBY, fpath, CNN, CGE);
                     }
                     else                       // return info in 'select'
                     {
                        dfsJfsTime2str( &ino->LastModify, temp);
                        TxStripAnsiCodes( text);
                        sprintf( select, "%s %s",   text, temp);
                        dfstrUllDot20( select, "", ino->FileSize, "");
                     }
                     rc = NO_ERROR;
                  }
                  else
                  {
                     rc = DFS_ST_MISMATCH;      // file-size mismatch
                  }
               }
               else
               {
                  rc = DFS_ST_MISMATCH;         // percentage mismatch
               }
            }
            else
            {
               rc = DFS_ST_MISMATCH;            // wildcard mismatch
            }
            strcpy( param, fpath);              // return the found name
            break;

         default:
            rc = DFS_PENDING;
            break;
      }
      TxFreeMem( ino);                          // not needed anymore
   }
   RETURN(rc);
}                                               // end 'dfsJfsFileInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Convert JFS JTIME (64 bit) to DFS standard date/time string
/*****************************************************************************/
char *dfsJfsTime2str                            // RET   string value
(
   JTIME              *jt,                      // IN    ptr to JFS time value
   char               *dtime                    // INOUT ptr to string buffer
)
{
   time_t              tm = (time_t) jt->hi;    // time in compiler format
   struct tm          *gm;

   ENTER();

   if ((jt->hi != 0) && ((gm = gmtime( &tm)) != NULL))
   {
      //- Note: time_t officially is a SIGNED value (1901 .. 2038, 0 = 1970)
      strftime(dtime, TXMAXTM, "%Y-%m-%d %H:%M:%S", gmtime( &tm));
   }
   else                                         // invalid, out of range TM
   {
      sprintf( dtime, "-%8.8x-%8.8x-", jt->hi, jt->lo);
   }
   TRACES(("Formatted date/time: %s\n", dtime));

   RETURN( dtime);
}                                               // end 'dfsJfsTime2str'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Find Inode and PATH upto Root-directory, starting at an Inode LSN
/*****************************************************************************/
BOOL dfsJfsFindRootInode                        // OUT   root found
(
   ULN64               start,                   // IN    starting lsn
   char               *path,                    // OUT   combined path, TXMAXLN
   ULN64              *lsn                      // OUT   found dir/file Inode LSN
)
{
   BOOL                rc  = FALSE;
   ULN64               cInodeLsn = start;       // Current LSN startpoint
   ULN64               parentLsn = 0;           // Parent  LSN found
   BOOL                isParent;
   ULONG               guard     = 100;         // sanity count

   ENTER();
   if (path)
   {
      *path = '\0';                             // start with empty path
   }
   do
   {
      isParent = dfsJfsInode2Parent( cInodeLsn, path, &parentLsn);
      if (isParent)
      {
         cInodeLsn = parentLsn;                 // follow Inodes towards root
      }
   } while (isParent && guard-- && !TxAbort());

   if (parentLsn == cInodeLsn)                  // Root is its own parent
   {
      *lsn = parentLsn;
      rc   = TRUE;
   }
   TRACES(("Found root 0x%llX for Inode 0x%llX in '%s'\n", *lsn, start, path));
   BRETURN (rc);
}                                               // end 'dfsJfsFindRootInode'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Find parent Inode for the given (DIR or FILE) Inode, add name to PATH
/*****************************************************************************/
BOOL dfsJfsInode2Parent                          // OUT   real parent found
(
   ULN64               fn,                      // IN    starting lsn
   char               *path,                    // OUT   combined path, TXMAXLN
   ULN64              *lsn                      // OUT   found dir/file Inode LSN
)
{
   BOOL                rc  = FALSE;             // not a valid parent
   ULONG               parentInode;
   TXLN                fname;                   // filename component
   BYTE                st;
   BOOL                orphaned = FALSE;        // parent not a directory
   S_JINODE           *ino;

   ENTER();

   *lsn  = L64_NULL;                            // invalid, simulated end

   TRACES(( "fn:%llx  path:'%s'\n", fn, path));

   if (dfsJfsReadChkInode( fn, &st, &ino) == NO_ERROR)
   {
      dfsJfsResolveInodeName( fn, ino, st, &parentInode, fname);
      *lsn = dfsJfsInode2Lsn( parentInode, &jfs->Fs1);

      TRACES(( "Parent inode:%8.8x  lsn:%8.8x\n", parentInode, *lsn));

      if (S_ISDIR( ino->Mode))
      {
         strcat( fname, FS_PATH_STR);           // add directory separator
      }
      else if (path && strlen( path))           // pre-pends MUST be a DIR!
      {
         sprintf( fname, "%c%12.12llX.DIR%c", FS_PATH_SEP, fn, FS_PATH_SEP);
         orphaned = TRUE;
      }
      if (parentInode == JFS_SYS_PARENT)        // system Inode
      {
          *lsn = fn;                            // fake ROOT reached ...
      }
      else if ((parentInode == 0) || (parentInode == L32_NULL)) // unknown
      {
         orphaned = TRUE;
      }
      if (!orphaned && (*lsn != fn))            // different LSN, real parent
      {
         rc = TRUE;
      }

      if (path != NULL)                         // path wanted
      {
         if (strlen(fname) + strlen(path) + 30 < dfsa->maxPath)
         {
            strcat( fname, path);               // append existing path
            strcpy( path, fname);               // and copy back
         }
         else                                   // try shortened version
         {
            sprintf(fname, "%12.12llX.DIR%c", fn, FS_PATH_SEP);
            if (strlen(fname) + strlen(path) + 10 < dfsa->maxPath)
            {
               strcat( fname, path);            // append existing path
               strcpy( path, fname);            // and copy back
            }
            else                                // truncate this component
            {
               TRACES(( "Path component '%s' discarded due to length limit\n", fname));
               sprintf(fname, "x%c", FS_PATH_SEP); // last resort
               if (strlen(fname) + strlen(path) < dfsa->maxPath)
               {
                  strcat( fname, path);         // append existing path
                  strcpy( path, fname);         // and copy back
               }
            }
         }
      }
      TxFreeMem( ino);

      TRACES(( "lsn:%llx  path:'%s'\n", fn, path));
   }
   BRETURN (rc);
}                                               // end 'dfsJfsInode2Parent'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Resolve related Inode name from cache, .LONGNAME or magic-area
/*****************************************************************************/
BOOL dfsJfsResolveInodeName                     // RET   real filename found
(
   ULN64               lsn,                     // IN    Inode lsn
   S_JINODE           *ino,                     // IN    Inode sector
   BYTE                st,                      // IN    Inode type
   ULONG              *parent,                  // OUT   Parent inode# or NULL
   char               *fname                    // OUT   related name
)                                               //       min TXMAXLN
{
   BOOL                rc = TRUE;               // function return
   ULONG               parIno = L32_NULL;

   ENTER();

   if (st != ST_JINAG)
   {
      if (ino->Self == JFS_ROOT_INODE)          // ROOT itself
      {
         strcpy( fname, "");                    // empty name
         parIno = JFS_ROOT_INODE;               // fixed Inode number
      }
      else if ((jfs->Ic)                     && // name cache there
               (ino->Self < jfs->Fs1.Max)    && // in normal inode range
               (jfs->Ic[ino->Self].Name)      ) // and name present
      {
         strcpy( fname, jfs->Ic[ ino->Self].Name); // direct copy from cache
         parIno       = jfs->Ic[ ino->Self].Parent;
      }
      else if ((ino->fMagic.magic  == DFS_JFS_MAGIC) && // try magic parent/name
               (ino->fMagic.parent <  jfs->Fs1.Max)  &&
               (ino->fMagic.parent >= JFS_ROOT_INODE) )
      {
         TxCopy( fname, ino->fMagic.fname, DFS_JFS_ML);
         parIno       = ino->fMagic.parent;
      }
      else                                      // try .LONGNAME EA
      {
         ULONG  eal;                            // EA length
         char  *eav;                            // EA value

         if (dfsJfsEaGetValue( ino, DFS_JFS_LFN_EA, &eal, (void **) &eav) == NO_ERROR)
         {
            strcpy( fname, eav);                // LFN is null-terminated str
            TxFreeMem( eav);                    // free the allocated memory
         }
         else                                   // no .LONGNAME either
         {
            if ((st == ST_JIDIR) || (st == ST_JIDDI))
            {
               sprintf( fname, "%c%12.12llX.DIR", FS_PATH_SEP, lsn);
            }
            else                                // must be a file ...
            {
               sprintf( fname, "-ino-%8.8x-at-%12.12llx-", ino->Self, lsn);
            }
            rc = FALSE;                         // name is faked
         }
      }
   }
   if (parent)
   {
      *parent = parIno;
   }
   TRACES(( "Parent: %x Name: '%s'\n", parIno, fname));
   BRETURN (rc);
}                                               // end 'dfsJfsResolveInodeName'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Append JFS Directory multiple-slot Unicode filename to ASCII string
/*****************************************************************************/
char *dfsJfsFileNameAppend                      // RET   string value
(
   S_JDSLOT           *slot,                    // IN    inode dir slot array
   char               *asc,                     // INOUT ptr to string buffer
   BYTE                index,                   // IN    slot with name start
   BYTE                maxslot                  // IN    maximum slot number
)
{
   int                 todo;                    // length left todo
   int                 len;                     // length in this fragment
   BYTE                next;

   ENTER();

   if ((index <= maxslot) && ((todo = slot[(int)index].i.NameLen) != 0))
   {
      if ((next = slot[(int)index].i.Dnext) != JFS_LASTSLOT)
      {
         next--;                                // make it ZERO based
         if (jfs->sup->Flag & JFS_DIR_INDEX)
         {
            len = 11;                           // Linux style
         }
         else
         {
            len = 13;                           // Classic (OS2) style
         }
      }
      else
      {
         len = todo;
      }
      TxUnicAppend( slot[(int)index].i.FileName, asc, len);
      todo -= len;

      while ((next <= maxslot) && (todo > 0))   // more continuation slots
      {
         if ((len = todo) > 15)
         {
            len = 15;
         }
         TxUnicAppend( slot[(int)next].n.FileName, asc, len);
         next  = slot[(int)next].n.Dnext -1;    // ZERO based next slot
         todo -= len;
      }
   }
   TRACES(( "Index:%u asc:'%s'\n", index, asc));
   RETURN( asc);
}                                               // end 'dfsJfsFileNameAppend'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Update JFS Directory multiple-slot Unicode filename from ASCII str (rename)
/*****************************************************************************/
static void dfsJfsFileNameUpdate
(
   S_JDSLOT           *slot,                    // INOUT inode dir slot array
   char               *asc,                     // IN    Ascii (new) filename
   BYTE                index,                   // IN    slot with name start
   BYTE                maxslot                  // IN    maximum slot number
)
{
   int                 todo;                    // length left todo
   int                 len;                     // length in this fragment
   BYTE                next;
   char               *fragment = asc;          // start of name fragment

   ENTER();
   TRACES(("slot:%llu of %llu, new name: '%s'\n", index, maxslot, asc));

   TRHEXS( 70, &slot[(int)index], 0x80, "JFS-dir-slot");
   if ((index <= maxslot) && ((todo = slot[(int)index].i.NameLen) != 0))
   {
      if ((next = slot[(int)index].i.Dnext) != JFS_LASTSLOT)
      {
         next--;                                // make it ZERO based
         if (jfs->sup->Flag & JFS_DIR_INDEX)
         {
            len = 11;                           // Linux style
         }
         else
         {
            len = 13;                           // Classic (OS2) style
         }
      }
      else
      {
         len = todo;
      }
      TRACES(("Update first %d chars using '%s'\n", len, fragment));
      TxAscii2Unic( fragment, slot[(int)index].i.FileName, len);
      todo     -= len;
      fragment += len;

      while ((next <= maxslot) && (todo > 0))   // more continuation slots
      {
         if ((len = todo) > 15)
         {
            len = 15;
         }
         TRACES(("Update first %d chars using '%s'\n", len, fragment));
         TxAscii2Unic( fragment, slot[(int)next].n.FileName, len);
         next      = slot[(int)next].n.Dnext -1; // ZERO based next slot
         todo     -= len;
         fragment += len;
      }
   }
   TRHEXS( 70, &slot[(int)index], 0x80, "JFS-dir-slot");
   VRETURN();
}                                               // end 'dfsJfsFileNameUpdate'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Append JFS Btree Splitpoint multiple-slot Unicode filename to ASCII string
/*****************************************************************************/
char *dfsJfsBtspNameAppend                      // RET   string value
(
   S_JDSLOT           *slot,                    // IN    inode dir slot array
   char               *asc,                     // INOUT ptr to string buffer
   BYTE                index,                   // IN    slot with name start
   BYTE                maxslot                  // IN    maximum slot number
)
{
   int                 todo;                    // length left todo
   int                 len;                     // length in this fragment
   BYTE                next;

   ENTER();

   if ((index <= maxslot) && ((todo = slot[(int)index].t.NameLen) != 0))
   {
      if ((next = slot[(int)index].t.Dnext) != JFS_LASTSLOT)
      {
         next--;                                // make it ZERO based
         len = 11;                              // Tree slot name length
      }
      else
      {
         len = todo;
      }
      TxUnicAppend( slot[(int)index].t.FileName, asc, len);
      todo -= len;

      while ((next <= maxslot) && (todo > 0))   // more continuation slots
      {
         if ((len = todo) > 15)
         {
            len = 15;
         }
         TxUnicAppend( slot[(int)next].n.FileName, asc, len);
         next  = slot[(int)next].n.Dnext -1;    // ZERO based next slot
         todo -= len;
      }
   }
   TRACES(( "Index:%u asc:'%s'\n", index, asc));
   RETURN( asc);
}                                               // end 'dfsJfsBtspNameAppend'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display(S_PXD) address and size in 8-hex digits and Mb/Kb or Bytes
/*****************************************************************************/
void dfsSpxd                                    // in the resulting display
(                                               // ** indicates addr>32 bits!!
   char               *text,                    // IN    leading string
   ULONG               bsize,                   // IN    block size in sectors
   S_PXD              *pxd,                     // IN    physical data extent
   char               *trail                    // IN    trailing text
)
{
   TXLN                str;

   strcpy(    str, "");
   dfstrSpxd( str, text, bsize, pxd, trail);
   TxPrint(   str);
}                                               // end 'dfsSpxd'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Format (S_PXD) address and size in 8-hex digits and Mb/Kb or Bytes; append
/*****************************************************************************/
void dfstrSpxd                                  // in the resulting string,
(                                               // ** indicates addr>32 bits!!
   char               *str,                     // OUT   resulting string
   char               *text,                    // IN    leading string
   ULONG               bsize,                   // IN    block size in sectors
   S_PXD              *pxd,                     // IN    physical data extent
   char               *trail                    // IN    trailing text
)
{
   TXLN                form;

   dfstrX10(  str, text, dfsJfsPxdValue( *pxd), "", "   ");
   sprintf(        form, "Size: 0x%4.4x blocks = ",  pxd->len);
   dfstrSiz8( str, form,                    (bsize * pxd->len), trail);
}                                               // end 'dfstrSpxd'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DFS JFS verify allocation integrity for a given INODE (normal / deleted)
/*****************************************************************************/
ULONG dfsJfsInodeCheckAlloc                     // RET   check result
(
   S_JINODE           *ino,                     // IN    INODE structure
   BYTE                iType,                   // IN    INODE type
   char               *options,                 // IN    result options
   ULN64              *totals,                  // INOUT total sectors
   ULN64              *invalids,                // INOUT invalid sectors
   ULN64              *datalsn                  // OUT   first data lsn
)                                               //       or NULL if not needed
{
   ULONG               rc  = NO_ERROR;
   DFSISPACE           is;
   BOOL                skipchk = (strchr(options, 's') != NULL);

   ENTER();

   if (jfs->Bm.dmap != NULL)                    // allocation BitMap present ?
   {
      switch (iType)
      {
         case ST_JINAG:                         // only data-type inodes
         case ST_JINOD:                         // have regular allocation
         case ST_JIDEL:
            if (ino->FileSize != 0)
            {
               rc = dfsJfsInodeData2Space( ino, 0, &is, NULL, NULL);
               if ((rc == NO_ERROR) && (is.space != NULL))
               {
                  ULN64      spaceSectors = dfsSspaceSectors( FALSE, is.chunks, is.space);

                  if (datalsn)
                  {
                     *datalsn = is.space[0].start; // first data sector
                  }
                  if (skipchk)
                  {
                     *totals = spaceSectors;          // just report size, ignore allocation
                  }                                   // (will be reported as 100% OK!)
                  else                                // SPACE allocation, full alloc check
                  {
                     ULONG maxSects = (((ino->FileSize -1) / jfs->sup->BlockSize) +1) * jfs->SectorsPerBlock;

                     #if defined (USEWINDOWING)
                        if (spaceSectors > DFSECT_BIGF)
                        {
                           txwSetSbviewStatus( "Checking allocation for a huge file ...", cSchemeColor);
                        }
                     #endif
                     rc = dfsJfsSpaceCheckAlloc( &is, ((iType != ST_JINAG) &&
                                                       (iType != ST_JINOD) &&
                                                       (iType != ST_JIDIR)),
                                                        options, maxSects, totals, invalids);
                  }
               }
               TxFreeMem( is.space);            // free SPACE
            }
            else
            {
               *totals   = 0;
               *invalids = 0;
            }
            break;

         default:                               // rest assumed OK
            break;
      }
   }
   else
   {
      rc = DFS_BAD_STRUCTURE;
   }
   RETURN(rc);
}                                               // end 'dfsJfsInodeCheckAlloc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DFS JFS verify allocation integrity for given INODE SPACE allocation
// 20161203: Add initial check on extentsize versus MAX, fail if wrong
/*****************************************************************************/
ULONG dfsJfsSpaceCheckAlloc                     // RET   check result
(
   DFSISPACE          *is,                      // IN    data allocation
   BOOL                deleted,                 // IN    delete status
   char               *options,                 // IN    result options
   ULN64               maxSect,                 // IN    max nr of sectors
   ULN64              *totals,                  // INOUT total sectors
   ULN64              *invalids                 // INOUT invalid sectors
)
{
   ULONG               rc  = NO_ERROR;
   ULONG               i;                       // number alloc sections
   ULONG               j;                       // number of sectors
   ULN64               sn;                      // Data LSN
   BOOL                verbose = (strchr(options, 'v') != NULL);
   ULONG               clsize  = (ULONG) jfs->SectorsPerBlock;
   USHORT              col = 0;
   BOOL                alok = FALSE;            // allocation OK
   BOOL                ldot = (BOOL) 2;         // last dot status
   ULONG               dots = 0;                // dots of same color

   ENTER();
   TRACES(("maxSect: %llx\n", maxSect));

   if (is->space != NULL)
   {
      for (i = 0; (i < is->chunks) && (rc == NO_ERROR); i++)
      {
         if ((is->space[ i].size <= maxSect) && (is->space[ i].start != 0))
         {
            if (verbose)
            {
               if (++col >= dfsGetDisplayMargin())
               {
                  TxPrint("\n");
                  col = 1;
               }
               TxPrint("");                    // start of allocation extent
               dots = 0;
            }
            if (is->space[ i].start != LSN_SPARSE) // normal extent ?
            {
               for ( j = 0; (j < is->space[ i].size) && (rc == NO_ERROR); j += clsize)
               {
                  sn = is->space[ i].start + j;

                  switch (dfsJfsAllocated( sn, 0, NULL, NULL))
                  {
                     case FALSE: alok =  deleted;                   break;
                     case TRUE:  alok = !deleted;                   break;
                     default:    alok =  FALSE; rc = DFS_PSN_LIMIT; break;
                  }
                  if (verbose)
                  {
                     //- start with 1 dot per block (4KiB) then one per 1024 (4MiB)
                     if ((alok != ldot) || ((++dots) < 33) || ((dots % 1024) == 0))
                     {
                        if (alok != ldot)
                        {
                           ldot = alok;
                           dots = 0;
                        }
                        if (++col >= dfsGetDisplayMargin())
                        {
                           TxPrint("\n");
                           col = 1;
                        }
                        TxPrint("%s%s%s", alok ? CBG : CBR, alok ? "" : "", CNN);
                     }
                  }
                  if (!alok)
                  {
                     (*invalids) += clsize;
                  }
                  (*totals) += clsize;
               }
            }
            else                                // sparse extent
            {
               (*totals) += is->space[ i].size * clsize;
               if (verbose)
               {
                  if (++col >= dfsGetDisplayMargin())
                  {
                     TxPrint("\n");
                     col = 1;
                  }
                  TxPrint( "%s%s%s", CNC, "", CNN);
               }
            }
         }
         else                                   // invalid extent, fail whole file!
         {
            rc = DFS_BAD_STRUCTURE;
         }
      }
      if (rc == NO_ERROR)
      {
         if (*invalids)                         // some sectors failed ...
         {
            rc = DFS_BAD_STRUCTURE;
         }
      }
      else                                      // structure / PSN-limit error
      {
         *invalids = maxSect;                   // fail on WHOLE file size
         *totals   = maxSect;
      }
      if (verbose)
      {
         TxPrint("\n");
         switch (rc)
         {
            case NO_ERROR:
               TxPrint("No allocation errors detected\n");
               break;

            case DFS_PSN_LIMIT:
               TxPrint("Sector numbers are too large for this JFS volume!\n");
               break;

            default:
               TxPrint("Allocation error detected on %llu out of %llu sectors, "
                       "save/undelete unreliable!\n", *invalids, *totals);
               break;
         }
      }
   }
   RETURN(rc);
}                                               // end 'dfsJfsSpaceCheckAlloc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate Inode number to logical sector nr
/*****************************************************************************/
ULN64 dfsJfsInode2Lsn                           // RET   Logical sector 64bit
(
   ULONG               ino,                     // IN    Inode number
   S_IMAP             *map                      // IN    Inode mapping
)
{
   ULN64               lsn = L64_NULL;

   //- note: trace removed for optimum speed (QuickFind)

   if ((ino < map->Max) && (map->Inode != NULL))
   {
      if ((lsn = map->Inode[(ino / JFS_INOEXT)]) != L64_NULL) // allocated
      {
         lsn += (ino % JFS_INOEXT);
      }
      else
      {
         lsn = 0;                               // not allocated yet ...
      }
   }
   return (lsn);
}                                               // end 'dfsJfsInode2Lsn'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate (Inode sector) LSN to INODE number
/*****************************************************************************/
ULONG  dfsJfsLsn2Inode                          // RET   INODE number
(
   ULN64               lsn,                     // IN    Logical sector nr
   S_IMAP             *map                      // IN    Inode mapping
)
{
   ULN64               ino = L64_NULL;
   ULONG               ext;
   ULN64               base;
   ULONG               mapsize = map->Max / JFS_INOEXT;

   ENTER();
   TRARGS(("lsn: %llX\n", lsn));

   if (map->Inode != NULL)                      // INODE allocation known ?
   {
      for (ext = 0; ext < mapsize; ext++)
      {
         base = map->Inode[ext];
         if ((lsn >= base) && (lsn < (base + JFS_INOEXT)))
         {
            ino = (ULN64) (ext * JFS_INOEXT) + (lsn - base);
            break;
         }
      }
   }
   RETURN ((ULONG) ino);
}                                               // end 'dfsJfsLsn2Inode'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read allocation info for Aggr/Fileset Inodes from the Fileset Inode Table
/*****************************************************************************/
ULONG dfsJfsFsInodeAlloc
(
   S_JINODE           *ino,                     // IN    Super Inode sector
   S_IMAP             *map                      // INOUT Inode map info
)
{
   ULONG               rc  = NO_ERROR;
   DFSISPACE           iam;                     // inode allocation map SPACE
   ULN64               size;                    // sectors in IAM
   ULN64               rsn;                     // RSN in IAM file
   ULONG               isi;                     // startindex in Inode map
   ULONG               iagNr;                   // iag sequence number
   ULONG               extents;                 // nr of Inode extents
   ULONG               inIag;                   // allocated Inodes in IAG
   S_IAG              *iag;                     // IAG structure buffer

   ENTER();

   TxFreeMem( map->Inode);                      // free existing, if any
   TxFreeMem( map->Space);
   map->Chunks = 0;                             // nr of allocated/used
   map->Max    = 0;                             // nr of allocated/used
   map->Used   = 0;
   map->Alloc  = 0;
   if ((iag = TxAlloc( 1, sizeof( S_IAG))) != NULL)
   {
      if ((rc = dfsJfsInodeData2Space( ino, 0, &iam, NULL, NULL)) == NO_ERROR)
      {
         size        = dfsSspaceSectors( FALSE, iam.chunks, iam.space);
         extents     = ( (size - JFS_IAGCPG) / JFS_IAGSEC) * JFS_IAGEXT;
         map->Max    = extents * JFS_INOEXT;
         map->Chunks = iam.chunks;
         map->Space  = iam.space;

         if ((map->Inode = TxAlloc( extents, sizeof( ULN64))) != NULL)
         {
            if (extents > 200)
            {
               dfsProgressInit( 0, (extents / JFS_IAGEXT), 0, "InodeAG",
                                "usage checked", DFSP_STAT | DFSP_VDEC, 0);
            }
            for (  rsn  = JFS_IAGCPG, isi  = 0,          iagNr = 0;
                 ((rsn +  JFS_IAGSEC)     <= size)  &&   (rc == NO_ERROR);
                   rsn += JFS_IAGSEC, isi += JFS_IAGEXT, iagNr++)
            {
               rc = dfsSspaceReadFilePart( iam.chunks, iam.space,
                                           rsn, JFS_IAGSEC, (BYTE *) iag);
               if (rc == NO_ERROR)
               {
                  inIag = (JFS_IAGEXT - iag->extFreeCount) * JFS_INOEXT;

                  if ((iag->blAgStart < jfs->TotalBlocks) && // IAG sanity check
                      (iag->iagNumber     <   JFS_IAGMAX) &&
                      (iag->extFreeCount  <=  JFS_IAGEXT) &&
                      (iag->inoFreeCount  <=  inIag     )  )
                  {
                     map->Alloc +=  inIag;
                     map->Used  += (inIag - iag->inoFreeCount);

                     rc = dfsJfsIagAddExtents( iag, isi, extents, map->Inode, NULL);
                  }
               }
               if (extents > 200)
               {
                  dfsProgressShow( (rsn / JFS_IAGSEC), 0, 0, NULL);
               }
            }
            if (extents > 200)
            {
               dfsProgressTerm();
            }
         }
         else
         {
            rc = DFS_ALLOC_ERROR;
         }
      }
      TxFreeMem( iag);                          // IAG buffer
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN(rc);
}                                               // end 'dfsJfsFsInodeAlloc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add any EA data info for an Inode to std FEA2-list structure
/*****************************************************************************/
ULONG dfsJfsReadEaInode
(
   S_JINODE           *ino,                     // IN    Inode sector
   ULONG              *size,                    // OUT   size in bytes
   S_FEA2LIST        **pfeal                    // OUT   list of FEA2's
)
{
   ULONG               rc    = NO_ERROR;
   S_FEA2LIST         *feal  = NULL;            // list of FEA2's
   ULONG               bsz   = ino->Eax.size;   // byte size

   ENTER();

   if (bsz != 0)                                // is there EA info ?
   {
      TRACES(("Eax.size: %u\n", bsz));
      if ((feal = TxAlloc(1, EA_BIGBUF_SIZE)) != NULL)
      {
         if (ino->Eax.dflag & JFS_EA_INTERN)    // is it an inline EA ?
         {
            rc = dfsOs2EaReadArea( ino->Eai.value, bsz, feal);
         }
         else                                   // in external extent
         {
            rc = dfsOs2EaReadExtern( jfs->SectorsPerBlock * ino->Eax.pe.lo,
                                     4, bsz, feal);
         }
         TRHEXS(70, feal, feal->cbList +10, "EA list");
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   *size  = bsz;
   *pfeal = feal;
   RETURN (rc);
}                                               // end 'dfsJfsReadEaInode'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Retrieve value for specified EA using Inode info
/*****************************************************************************/
ULONG dfsJfsEaGetValue
(
   S_JINODE           *ino,                     // IN    Inode sector
   char               *name,                    // IN    EA name
   ULONG              *len,                     // OUT   EA data size
   void              **data                     // OUT   EA data (allocated)
)
{
   ULONG               rc = DFS_NOT_FOUND;      // function return

   ENTER();
   TRACES(("EA name: %s\n", name));

   if (ino->Eax.size != 0)                      // is there EA info ?
   {
      if (ino->Eax.dflag & JFS_EA_INTERN)       // is it an inline EA ?
      {
         rc = dfsEaBlockGetValue( name, ino->Eai.value,
                                        ino->Eai.size, len, data);
      }
      else                                      // in external extent
      {
         ULONG         sects = (ino->Eax.size / dfsGetSectorSize()) +1;
         S_JFSEA      *ea;

         TRACES(( "Allocate %u sectors of EA data\n", sects));

         if ((ea = TxAlloc( sects, dfsGetSectorSize())) != NULL) // alloc EA
         {
            if ((rc = dfsRead( jfs->SectorsPerBlock * ino->Eax.pe.lo,
                               sects, (BYTE   *) ea)) == NO_ERROR)
            {
               if (ea->size <= ino->Eax.size)   // sanity check, avoid traps!
               {
                  rc = dfsEaBlockGetValue( name, ea->value, ea->size, len, data);
               }
               else
               {
                  rc = DFS_NOT_FOUND;
               }
            }
            else
            {
               TxFreeMem( ea);
            }
         }
         else
         {
            rc = DFS_ALLOC_ERROR;
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsJfsEaGetValue'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Analyse IAG page, optional add Inode-extents to supplied inode extent map
/*****************************************************************************/
ULONG dfsJfsIagAddExtents
(
   S_IAG              *iag,                     // IN    IAG structure buffer
   ULONG               start,                   // IN    start index in map
   ULONG               limit,                   // IN    size of inode map
   ULN64              *extMap,                  // OUT   Inode extent map or NULL
   ULONG              *errors                   // INOUT Invalid ext lsns or NULL
)
{
   ULONG               rc = NO_ERROR;
   int                 ei;                      // extent index
   int                 mw;                      // mapword index
   int                 mb;                      // bit in a mapword
   ULONG               index  = start;          // Inode reference index
   ULONG               mword;
   ULN64               eBlock;                  // extent blocknumber 64bit
   ULN64               lsn;
   ULONG               bpe = (JFS_INOEXT / jfs->SectorsPerBlock); // blocks per extent

   ENTER();

   TRACES(( "IAG number: %u\n", iag->iagNumber));
   for (ei = 0, mw = 0; mw < JFS_SMAPSZ; mw++)  // iterate over all possible
   {                                            // Inode extents in the IAG
      mword = iag->extBitMap[ mw];
      for ( mb = 0;
           (mb < 32) && (index < limit);
            mb++,  ei++, index++)
      {
         if (mword & 0x80000000)                // Allocated if MSB set
         {
            eBlock  = dfsJfsPxdValue( iag->inoExtent[ei]);
            lsn = dfsJfsBlock2Lsn( eBlock);
            if (extMap)
            {
               extMap[index] = lsn;
            }
            if (errors && ((lsn < LSN_AITAB) || (lsn > jfs->Sect) || (iag->inoExtent[ei].len != bpe)))
            {
               (*errors)++;
            }
            // Always dump, even the good ones, to compare the bad ones with ...
            if ((dfsa->warnLevel > 1) && (dfsa->verbosity >= TXAO_MAXIMUM)) // extreme verbose :-)
            {
                TxPrint("Inode ext len = %u   bpe = %u         "
                        "lsn 0x%10.10llx   = Bl 0x%10.10llx\n",
                         iag->inoExtent[ei].len, bpe, lsn, eBlock);
            }
         }
         else if (extMap)                       // insert sparse extent to
         {                                      // keep the Inode numbering
            extMap[index] = LSN_SPARSE;         // consistent (fixed)
         }
         mword <<= 1;                           // shift left by 1 bit
      }
   }
   RETURN(rc);
}                                               // end 'dfsJfsIagAddExtents'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Analyse IAG page, calculate summary maps for extent-alloc and inode-status
/*****************************************************************************/
ULONG dfsJfsIagMkSummary
(
   S_IAG              *iag,                     // IN    IAG structure buffer
   ULONG              *extAllocation,           // OUT   Inode extent alloc map
   ULONG              *inoFreeStatus,           // OUT   Inode free status map
   ULONG              *extUsed,                 // OUT   used extents 0..128
   ULONG              *inoUsed                  // OUT   used inodes  0..4096
)
{
   ULONG               rc = NO_ERROR;
   int                 ei;                      // extent index
   int                 mw;                      // mapword index
   int                 mb;                      // bit in a mapword
   ULONG               aword = 0;
   ULONG               fword = 0;
   ULONG               bpe = (JFS_INOEXT / jfs->SectorsPerBlock); // blocks per extent

   ENTER();

   TRACES(( "IAG number: %u\n", iag->iagNumber));
   for (mw = 0, ei = 0, *extUsed = 0;
        mw < JFS_SMAPSZ;
        mw++)                                   // iterate over all possible
   {                                            // Inode extents in the IAG
      for (mb = 0; mb < 32; mb++,  ei++)
      {
         aword <<= 1;                           // shift left by 1 bit
         fword <<= 1;                           // result upto now
         fword  |= 1;                           // add next bit
         if ((dfsJfsPxdValue( iag->inoExtent[ei]) != 0) &&
             (iag->inoExtent[ei].len == bpe))   // extent seems allocated
         {
            (*extUsed)++;                       // count total allocated
            aword |= 1;                         // and register in map

            if (iag->inoBitMap[ei] != 0xffffffff)
            {
               fword &= ~1;                     // at least one free inode
            }
         }
      }
      extAllocation[mw] = aword;
      inoFreeStatus[mw] = fword;
   }
   *inoUsed = dfsUlMapBitCount( iag->inoBitMap, JFS_IAGEXT);
   RETURN(rc);
}                                               // end 'dfsJfsIagMkSummary'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create a valid JFS  boot record based on partition-table, template and SB
/*****************************************************************************/
ULONG dfsJfsMkBootRec
(
   S_JFSUP            *sup,                     // IN    optional superblock
   BOOL                standard,                // IN    use standard JFS code
   char               *drive                    // IN    optional bootdrive
)
{
   ULONG               rc  = NO_ERROR;
   ULN64               partSectors;

   ENTER();

   brt = (S_BOOTR *) ((standard) ? &dfsJfsStdTemplate : &dfsJfsBootTemplate);

   if ((SINF->p != NULL) || ((sup != NULL) && (sup->SectSize != 0)))
   {                                            // check partinfo & valid SB
      char             letter = 0;

      memcpy( brt->OpSys, DFS_VLABEL, BT_SYST_L);
      if (SINF->p != NULL)                      // partition info available
      {
         partSectors           =  SINF->p->sectors
                               - (SINF->p->geoSecs * SINF->p->geoHeads);

         brt->eb.HiddenSectors = SINF->p->partent.BootSectorOffset;
         brt->eb.LogGeoSect    = SINF->p->geoSecs;
         brt->eb.LogGeoHead    = SINF->p->geoHeads;
      }
      else                                      // use global geo info
      {
         TxPrint( "\n%sNo partition-table info%s is available, this will "
                  "make the 'HiddenSectors'\nfield incorrect when this "
                  "is a primary partition.\n", CBR, CNN);
         TxPrint( "The logical geometry used now is %s%hu%s Heads "
                  "with %s%hu%s Sectors\n",
                   CBY, dfstGeoHeads(   DFSTORE), CNN,
                   CBY, dfstGeoSectors( DFSTORE), CNN);
         TxPrint("Use the 'diskgeo' command to change this when incorrect\n");

         partSectors           = 0;
         brt->eb.HiddenSectors = dfstGeoSectors( DFSTORE);
         brt->eb.LogGeoSect    = dfstGeoSectors( DFSTORE);
         brt->eb.LogGeoHead    = dfstGeoHeads(   DFSTORE);
      }
      if ((sup != NULL) && (sup->SectSize != 0))
      {
         ULONG spblock    = sup->BlockSize / sup->SectSize;
         ULN64 supSectors = sup->TotalSec     + // total (normal) sectors  +
               (sup->InlineLog.len * spblock) + // size of inline-log area +
               (sup->FsckSpace.len * spblock);  // size of FSCK space area

         if ((partSectors - supSectors) < spblock)
         {
            brt->eb.BigSectors =  partSectors;  // closer match (not rounded)
         }
         else                                   // use SUPER calculated value
         {
            brt->eb.BigSectors =  supSectors;   // minimum required size
         }
         brt->eb.ClustSize  = (BYTE) spblock;   // calculated sectors/block

         memcpy( brt->os.Label, sup->FsVolume, BT_LABL_L);
      }
      else          // use size from partition table minus 1 cylinder (for LVM)
      {
         TxPrint( "\n%sNo JFS superblock%s is available, this will make the "
                 "resulting bootsector\nless reliable when there are multiple "
                 "partitions in this JFS volume\n\n", CBR, CNN);

         brt->eb.BigSectors = partSectors;
      }
      if (brt->eb.BigSectors <= 0xffff)         // use the right 'size' field
      {
         brt->eb.Sectors      = (USHORT) brt->eb.BigSectors;
         brt->eb.BigSectors   = 0;
      }

      if (drive && *drive)                      // letter/reset specified ?
      {
         if (*drive != '-')
         {
            letter = *drive;
         }
      }
      else if (        (SINF->p)              && // partition info there
               (strlen( SINF->p->blabel) > 0) && // bootable by bootmgr
               (isalpha(SINF->p->drive[ 0]  ) )) // and letter available
      {
         letter =       SINF->p->drive[ 0];
      }
      if (letter)
      {
         brt->os.FsysValue = toupper( letter) - 'C' + 0x80;
         if (!standard)
         {
            BYTE      *sectordata = (BYTE *) brt;

            sectordata[ JFSBOOTALTERNATELETTER] = brt->os.FsysValue;
         }
         TxPrint("Boot driveletter '%c:' has been set in the bootsector\n", toupper( letter));
      }
      else                                      // default
      {
         brt->os.FsysValue = 0;
         TxPrint("No boot driveletter has been set, data partition only\n");
      }
      TxPrint( "\nThe following %s JFS bootsector has been prepared:\n\n",
                      (standard) ? "standard" : "bootable");
      dfsDisplaySector( dfstLSN2Psn( DFSTORE, LSN_BOOTR), ST_BOOTR, (BYTE   *) brt);
      if ((dfsa->batch) || (TxConfirm( 5132, "Do you want to replace the "
          "bootsector (sector 0) with the created %s one ? [Y/N] : ",
                      (standard) ? "standard" : "bootable" )))
      {
         if (DFSTORE_WRITE_ALLOWED)
         {
            rc = dfsWrite( LSN_BOOTR, 1, (BYTE   *) brt);
            if (rc == NO_ERROR)
            {
               TxPrint("\nJFS boot record successfully updated\n");
               if (!standard)
               {
                  TxNamedMessage( !dfsa->batch, 5132, " INFO: Add JFS LDR recommended ",
                     "To make this partition really bootable, you may need to add/fix the JFS LDR "
                     "code in the boot-track as well. Use the menu:\n\n "
                     "Mode=JFS -> Boot area fixes -> Restore JFS LDR imagefile");
               }
            }
         }
         else
         {
            rc = DFS_READ_ONLY;
         }
      }
      else
      {
         TxPrint("\nCommand aborted, no changes made\n");
         rc = DFS_NO_CHANGE;
      }
   }
   else                                         // not enough info
   {
      TxPrint("No partition info and no superblock available, "
              "create not possible\n");
      rc = DFS_CMD_FAILED;
   }
   RETURN(rc);
}                                               // end 'dfsJfsMkBootRec'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Identify if FS structures for JFS are present, set fsform and label
/*****************************************************************************/
ULONG dfsJfsFsIdentify
(
   S_BOOTR            *boot,                    // IN    Bootsector ref or NULL
   DFSPARTINFO        *p                        // INOUT partition info
)
{
   ULONG               rc = DFS_FS_UNKNOWN;     // function return
   S_JFSUP            *sup;                     // superblock sector
   char                st;

   ENTER();

   if ((sup = TxAlloc(1, dfsGetSectorSize())) != NULL)
   {
      if (dfsRead( p->basePsn + LSN_JFSUP, 1, (BYTE *) sup) == NO_ERROR)
      {
         if ((dfsJfsIdent(LSN_JFSUP, 0, &st, sup) != DFS_PENDING) &&
             ((st == ST_JFSUP) || (st == ST_JSBAD)))
         {
            rc = NO_ERROR;                      // OK or BAD 1st superblock present
         }
      }
      if (rc == DFS_FS_UNKNOWN)                 // try alternate location
      {
         if (dfsRead( p->basePsn + LSN_JFSU2, 1, (BYTE *) sup) == NO_ERROR)
         {
            if ((dfsJfsIdent(LSN_JFSU2, 0, &st, sup) != DFS_PENDING) &&
                ((st == ST_JFSUP) || (st == ST_JSBAD)))
            {
               rc = NO_ERROR;                   // OK or BAD 2nd superblock present
            }
         }
      }
      if (rc == NO_ERROR)
      {
         strcpy( p->fsform, "JFS");
         TxCopy( p->plabel, sup->FsVolume, BT_LABL_L +1);

         // TxCopy( p->plabel, sup->linux.lxVolName, JFS_LEN_LBL + 1);
         memcpy( p->fsUuid, sup->linux.lxVolUuid, FS_UUID_LENGTH);
         p->uuidPresent = TRUE;
      }
      TxFreeMem( sup);
   }
   RETURN (rc);
}                                               // end 'dfsJfsFsIdentify'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Copy JFS bad-blocks list (special Inode 0x5c) to the DFSee (bad) sectorlist
/*****************************************************************************/
ULONG dfsJfsExportBadList                      // RET
(
   ULN64             *list,                     // IN    LSN list structure
   ULONG              size                      // IN    max nr of entries
)
{
   ULONG               rc = NO_ERROR;
   S_JINODE           *ino;
   DFSISPACE           bbl;                     // bad-blocks Inode SPACE
   ULN64               sectors;                 // #sectors in badblock list
   ULN64               todo = 0;                // sectors to be added
   ULN64               done;                    // sectors added to list

   ENTER();

   dfsInitList( 0, "-w", "-f -P");              // empty DFSee snlist
   if ((rc = dfsJfsReadChkInode( JFS_BADBLOCK_LSN, NULL, &ino)) == NO_ERROR)
   {
      if (ino->FileSize != 0)
      {
         if ((rc = dfsJfsInodeData2Space( ino, 0, &bbl, NULL, NULL)) == NO_ERROR)
         {
            sectors = dfsSspaceSectors( FALSE, bbl.chunks, bbl.space);

            if (sectors > ((ULN64) size))       // bad-sector space in FS
            {
               TxPrint("\nWarning, only %u bad-sectors can be registred in DFSee!\n", size);
            }
            todo = min( sectors, size);

            TRACES(("Exporting %u BAD sectors to the sectorlist\n"));

            //- iterate over SPACE, add each RSN -> LSN to the snlist
            for (done = 0; done < todo; done++)
            {
               list[ done + 1] = dfsSspaceRsn2Lsn( bbl.chunks, bbl.space, done);
            }
         }
         *list = todo;
         TxFreeMem( bbl.space);
      }
      TxFreeMem( ino);
   }
   RETURN (rc);
}                                               // end 'dfsJfsExportBadList'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Copy DFSee (bad) sectorlist to the JFS bad-blocks list (special Inode 0x5c)
// sectorlist may be empty (to clear the bad sectors)
/*****************************************************************************/
ULONG dfsJfsImportBadList                      // RET
(
   ULN64             *list                      // IN    LSN list structure
)
{
   ULONG               rc = NO_ERROR;
   S_JINODE           *ino;
   DFSISPACE           is;                      // bad blocks SPACE
   ULN64               sectors;                 // #sectors in badblock list
   ULONG               chunk;
   ULN64               aBlock;                  // Absolute block number
   ULN64               rBlock;                  // relative blocknumber
   ULONG               cBlocks;                 // Chunk size in blocks
   time_t              tm;                      // c-runtime time_t format
   int                 i;

   ENTER();

   rc = dfsSectorList2Space( JFS_IN_XADS, &is); // get bads into a SPACE
   if ((rc == NO_ERROR) || (rc == DFS_CMD_WARNING))
   {
      sectors = dfsSspaceSectors( FALSE, is.chunks, is.space);
      if (rc == DFS_CMD_WARNING)
      {
         TxPrint( "\nWarning: Some listed bad sectors ignored, badblocks limited to %u ranges!\n", JFS_IN_XADS);
      }

      //- to be refined, may do a block-free check here, based on SLT and the SPACE info
      //- and extra warnings if some not FREE, possibly with a confirmation before continuing
      //- Update: OR combine SectorList2Space with that logic, making it reusable

      if ((rc = dfsJfsReadChkInode( JFS_BADBLOCK_LSN, NULL, &ino)) == NO_ERROR)
      {
         if (ino->File.xCount > 2)              // existing bad blocks listed?
         {
            for (chunk = 0; chunk < (ino->File.xCount -2); chunk++)
            {
//             rBlock  = ino->Xad[ chunk].rlo    + ((ULN64) ino->Xad[ chunk].rhi   << 32);
//             aBlock  = ino->Xad[ chunk].pe.lo  + ((ULN64) ino->Xad[ chunk].pe.hi << 32);

               rBlock  = dfsJfsXadValue( ino->Xad[ chunk]);
               aBlock  = dfsJfsPxdValue( ino->Xad[ chunk].pe);
               cBlocks = ino->Xad[ chunk].pe.len;

               for (i = 0; i < cBlocks; i++)    // set each block to FREE
               {
                  dfsJfsSetBlock( aBlock + i, FALSE);
               }
            }
         }
         memset( &(ino->Xad[0]), 0, (JFS_IN_XADS * sizeof( S_XAD)));     // clean up XAD area

         for (rBlock = 0, chunk = 0; (chunk < is.chunks) && (rc == NO_ERROR); chunk++)
         {
            aBlock  = dfsSn2Cl( is.space[ chunk].start);
            cBlocks = is.space[ chunk].size / dfsGetClusterSize();

            ino->Xad[ chunk].rhi    = rBlock >> 32;
            ino->Xad[ chunk].rlo    = rBlock  & 0xffffffff;

            ino->Xad[ chunk].pe.hi  = aBlock >> 32;
            ino->Xad[ chunk].pe.lo  = aBlock  & 0xffffffff;
            ino->Xad[ chunk].pe.len = cBlocks;

            for (i = 0; i < cBlocks; i++)       // set each block to ALLOCATED
            {
               dfsJfsSetBlock( aBlock + i, TRUE);
            }

            rBlock += cBlocks;

            TRACES(("Set %4u blocks as BAD, starting at Bl:%8.8x = sect:0x%llx\n",
                     cBlocks, ino->Xad[ chunk].pe.lo, is.space[ chunk].start));
         }
         ino->FileSize      = sectors * dfsGetSectorSize();
         ino->Allocated     = sectors / dfsGetClusterSize();
         ino->File.xCount   = ((BYTE)  is.chunks) + 2;

         time( &tm);                            // get current time
         ino->LastStatus.hi = (ULONG) tm;       // timestamp last inode update
         rc = dfsWrite( JFS_BADBLOCK_LSN, 1, (BYTE   *) ino);

         TxFreeMem( ino);
         dfsJfsBitmapFlush( FALSE);             // flush, but keep cache
      }
      TxFreeMem( is.space);
   }
   RETURN (rc);
}                                               // end 'dfsJfsImportBadList'
/*---------------------------------------------------------------------------*/


