//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// ISO utility functions
//
// Author: J. van Wijk
//
// JvW  10-01-2019 Initial version, derived from HFS
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsver.h>                             // DFS version info
#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfstable.h>                           // SLT utility functions

#include <dfsiso.h>                             // ISO disk structures
#include <dfsaiso.h>                            // ISO analysis functions
#include <dfsuiso.h>                            // ISO utility functions
#include <dfsliso.h>                            // ISO slt & error definitions


// Add (sub) directory to ISO path cache, recursing over its subdirectories
static ULONG dfsIsoDirAddPathCache
(
   ULONG               recurseLevel,            // IN    recursion level (limit)
   char               *thisPath,                // IN    path from root to THIS dir
   DFSISPACE          *dirIsp                   // IN    DIR allocation in ISPACE
);

// Put contents of a Directory SSPACE (DirLsn, entry pairs) into sn-list
static ULONG dfsIsoDirSpace2List
(
   DFSISPACE          *dirIsp                   // IN    DIR allocation in ISPACE
);

// Find specific name in a (sub) directory SPACE structure
static BOOL dfsIsoDirSpace2Entry                // RET   name found
(
   char               *name,                    // IN    subdir or filename
   ULONG               chunks,                  // IN    nr of space entries
   S_SPACE            *space,                   // IN    space allocation
   ULN64              *lsn,                     // OUT   Dir-sector LSN found
   ULONG              *info,                    // OUT   Dir entry-nr found
   S_ISO_DIRENTRY     *dirent                   // OUT   Directory entry copy
);


/*****************************************************************************/
// Find Leaf-Node LSN+Index for specified path, starting at the root-directory
/*****************************************************************************/
ULONG dfsIsoFindPath
(
   ULN64               loud,                    // IN    Show progress
   ULN64               d2,                      // IN    dummy
   char               *path,                    // IN    path specification
   void               *vp                       // OUT   found dir/file NODE+index
)
{
   ULONG               rc  = NO_ERROR;
   ULN64               fs;                      // Current FNODE LSN
   ULN64               ds  = 0;                 // Current DIR LSN
   TXLN                part;
   char               *p   = path;
   int                 l;
   S_ISO_DIRENTRY      de;                      // directory entry (next)
   DFSISPACE           isp;                     // Space structure
   DFS_PARAMS         *parm  = (DFS_PARAMS *) vp;
   ULONG               entry = 0;

   ENTER();
   fs = dfsIsoLE32(iso->vd->p.RootDirEntry.Location); // start LSN
   if (loud)
   {
      dfsX10( "RootDirectory LSN : ", fs, CNN, "   ");
      TxPrint("find path: '%s'\n", path);
   }
   parm->Lsn    = fs;
   parm->Number = strlen(p) ? DFSSNINFO : 0;    // no info for ROOT!
   while ((rc == NO_ERROR) && strlen(p) && !TxAbort())
   {
      if ((l = strcspn(p, FS_PATH_STR)) != 0)
      {
         strncpy(part, p, l);                   // isolate part
         part[l] = '\0';
         p += l;                                // skip part
         if (*p == FS_PATH_SEP)
         {
            p++;                                // skip '\'
         }
      }
      if (strlen(part))
      {
         if (dfsIsoGetAllocSpace( fs, entry | DFSSNINFO, NULL, &isp) == NO_ERROR)
         {
            if (loud)
            {
               dfsX10( "Base DirBlock LSN : ", fs, CNN, TREOLN);
            }
            if (dfsIsoDirSpace2Entry( part, isp.chunks, isp.space, &ds, &entry, &de))
            {
               if (loud)
               {
                  dfsX10( " - Lsn ", ds, CNN, ", ");
                  TxPrint("entry %2u for '%s'\n", entry, part);
               }
               if (*p == '\0')                          //- end of string, found!
               {
                  parm->Lsn      = ds;                  //- DIR sector number
                  parm->Number   = entry | DFSSNINFO;   //- entry 0..50 in that sector
                  parm->Flag     = TRUE;                //- Size is from DIR-entry
                  parm->byteSize = dfsIsoLE32( de.FileSize);
               }
               else
               {
                  if ((de.FileFlags & ISO_FLAG_SUBDIR) == 0)
                  {
                     if (loud)
                     {
                        TxPrint("Object at sector  : %8.8X is not a directory\n",
                                                           dfsIsoLE32( de.Location));
                     }
                     rc = ERROR_PATH_NOT_FOUND;
                  }
               }
               fs = ds;                         // next level LSN to check is in ds+entry
            }
            else
            {
               if (loud)
               {
                  TxPrint(" - Search failed   for '%s'\n", part);
               }
               rc = ERROR_PATH_NOT_FOUND;
            }
            free( isp.space);                   // free the memory !
         }
      }
      else
      {
         rc = ERROR_PATH_NOT_FOUND;
      }
   }
   RETURN (rc);
}                                               // end 'dfsIsoFindPath'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Find specific name in a (sub) directory SPACE structure
/*****************************************************************************/
static BOOL dfsIsoDirSpace2Entry                // RET   name found
(
   char               *name,                    // IN    subdir or filename
   ULONG               chunks,                  // IN    nr of space entries
   S_SPACE            *space,                   // IN    space allocation
   ULN64              *lsn,                     // OUT   Dir-sector LSN found
   ULONG              *info,                    // OUT   Dir entry-nr found
   S_ISO_DIRENTRY     *dirent                   // OUT   Directory entry copy
)
{
   BOOL                found = FALSE;
   ULONG               chunk;                   // index in space-area
   ULONG               sect;                    // sectors to handle
   ULN64               dirsec;                  // current dir sector LSN
   TXLN                fileName;                // ISO long filename
   BYTE               *dirbuf;
   S_ISO_DIRENTRY     *isodir;                  // Fat directory sector

   ENTER();
   TRARGS(("Name to find : '%s'\n", name));

   if ((dirbuf = TxAlloc( 1, iso->BlockSize)) != NULL) // allocate one sector
   {
      isodir = (S_ISO_DIRENTRY *) dirbuf;

      for ( chunk = 0;
           (chunk < chunks) && !TxAbort() && !found && (space);
            chunk++)
      {
         for ( sect = 0;
              (sect < space[chunk].size)  && !TxAbort() && !found;
               sect++)
         {
            dirsec  = space[chunk].start + sect;
            if (dfsRead(dirsec, 1, dirbuf) == NO_ERROR)
            {
               BYTE               *at;          // current position
               S_ISO_DIRENTRY     *curDir;      // current as directory struct
               ULONG               blockRemain; // remaining size in this block
               ULONG               entry;       // index in ISO directory

               blockRemain = iso->BlockSize;    // bytes remaining in block
               at          = dirbuf;            // set to start of this block
               entry       = 0;                 // first entry in this block

               do
               {
                  curDir = (S_ISO_DIRENTRY *) at;

                  if (curDir->RecordLength > 0) // handle this DIR entry record
                  {
                     if      ((curDir->NameLength == 1) && (curDir->FileName[0] == 0x00))
                     {
                        strcpy( fileName, ".");
                     }
                     else if ((curDir->NameLength == 1) && (curDir->FileName[0] == 0x01))
                     {
                        strcpy( fileName, "..");
                     }
                     else
                     {
                        if (iso->vdUnicode)     // This VD uses Unicode (Joliet)
                        {
                           TxUnic2Ascii( (USHORT *) curDir->FileName, fileName, curDir->NameLength / 2);
                        }
                        else
                        {
                           TxCopy( fileName, (char *) curDir->FileName,  curDir->NameLength + 1);
                        }
                        if (fileName[ strlen( fileName) - 2] == ';') // fileversion postfix
                        {
                           fileName[ strlen( fileName) - 2] = 0; // remove the postfix
                        }
                     }
                     //- to be refined, may get RockRidge altName as well (replacing fileName)

                     TRACES(("dirsec: 0x%llx entry:%u name: '%s'\n", dirsec, entry, fileName));

                     if (strcasecmp( fileName, name) == 0)       // long-filename match
                     {
                        if (lsn)
                        {
                           *lsn = dirsec;
                        }
                        if (info)
                        {
                           *info = entry;
                        }
                        if (dirent)
                        {
                           *dirent = *curDir;   // copies first 34 bytes of the entry only!
                        }
                        TRACES(("FOUND name at sector: 0x%llx entry %u\n", dirsec, entry));
                        found = TRUE;
                     }
                     entry++;                   // to next entry, until found

                     at          += curDir->RecordLength; // advance to next
                     blockRemain -= curDir->RecordLength; // and keep count ...
                  }
               } while ((blockRemain > 0) && (curDir->RecordLength > 0) && (!found));
            }
         }
      }
      TxFreeMem( dirbuf);
   }
   BRETURN(found);
}                                               // end 'dfsIsoDirSpace2Entry'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Allocate ISO volumedesc, and read it from byte-offset calculated from vdi
// Note: Area aware to allow usage from FDISK mode too
/*****************************************************************************/
S_ISO_VOLUMEDESC *dfsIsoGetVolumeDesc                // RET    TxAlloc'ed volumedesc
(
   ULONG               vdi                      // IN    Volume Descriptor Index
)
{
   S_ISO_VOLUMEDESC   *vd = NULL;               // function return

   ENTER();
   TRACES(("vdi: %u\n", vdi));

   if ((vd = TxAlloc( 1, dfsGetSectorSize())) != NULL)
   {
      if (dfsRead( dfstAreaP2Disk( DFSTORE, ISO_LSNVD( iso->vdi)), 1, (BYTE *) vd) == NO_ERROR)
      {
         TRHEXS( 70,  vd,  0x100, "Volume Descriptor");
      }
      else                                      // no volumedesc (yet)
      {
         //- to be refined, could try to read from backup location, if there are any
      }
   }
   RETURN (vd);
}                                               // end 'dfsIsoGetVolumeDesc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Identify if FS structures for ISO are present
/*****************************************************************************/
ULONG dfsIsoFsIdentify
(
   S_BOOTR            *boot,                    // IN    Bootsector ref or NULL
   DFSPARTINFO        *p                        // INOUT partition info
)
{
   ULONG               rc = DFS_FS_UNKNOWN;     // function return
   S_ISO_VOLUMEDESC   *vd;                      // volumedesc sector
   char                st;

   ENTER();

   if ((vd = dfsIsoGetVolumeDesc( iso->vdi)) != NULL) // vdi to be refined
   {
      if ((dfsIsoIdent( ISO_LSNVD( iso->vdi), 0, &st, vd) != DFS_PENDING) && (st == ST_ISOSVD))
      {
         //- to be refined
      }
      TxFreeMem( vd);
   }
   RETURN (rc);
}                                               // end 'dfsIsoFsIdentify'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display file allocation and path info for LSN + index
/*****************************************************************************/
ULONG dfsIsoFileInfo                            // RET   LSN is valid NODE
(
   ULN64               dirLsn,                  // IN    DirBlock sectornumber
   ULN64               fileIdx,                 // IN    File dir-record index
   char               *select,                  // IN    file selection string
   void               *param                    // INOUT leading text / path
)
{
   ULONG               rc = NO_ERROR;
   BYTE               *sb = NULL;               // sector buffer
   S_ISO_DIRENTRY     *de;                      // ISO directory entry
   BYTE                st = 0;                  // dir-entry type, default none
   ULN64               dirsect = dirLsn;
   USHORT              entry   = fileIdx & 0x7f; // allows up to 128 per dirblock
   TXLN                shortpath;
   TXLN                text;
   TXTT                temp;                    // location text
   ULONG               threshold = 0;           // threshold percentage
   BOOL                isMinimum = TRUE;        // threshold is minimum value
   ULONG               size      = 0;           // nr of allocated sectors
   ULONG               location  = 0;           // location value; LSN/Cluster
   ULONG               minS      = 0;           // minimal size
   ULONG               maxS      = 0;           // maximal size
   char                itemType  = 0;           // type Dir, File, Browse, or ANY
   time_t              modTstamp = 0;           // Modify timestamp value, or 0
   char               *lead    = param;
   BOOL                verbose = (strcmp( lead, ":") != 0);
   BOOL                output;                  // no Printf, just return fpath

   ENTER();
   TRARGS(("LSN:0x%llX, info:0x%2.2x select:'%s', lead:'%s'\n", dirLsn, fileIdx, select, lead));

   if (strcmp( lead, "!") == 0)                 // Skip allocation check wanted
   {
      output  = FALSE;                          // no output
      *lead   = 0;                              // discard 'garbage' in lead string
   }
   else                                         // Possible output/alloc-check wanted
   {
      output  = (*lead != 0);                   // output if any lead given (except '!')
   }
   TRACES(("verbose:%u output:%u\n", verbose, output));

   if (fileIdx & DFSSNINFO)
   {
      if ((sb = TxAlloc(1, iso->BlockSize)) != NULL)
      {
         if ((rc = dfsRead(dirsect, 1, sb)) == NO_ERROR)
         {
            if ((de = dfsIsoDirIndex2Entry( sb, entry)) != NULL)
            {
               //- to be refined, may want to read RR modified time when available
               time_t  modifiedTime = dfsIsoFileTime2t( &de->FileDateTime);

               st = (de->FileFlags & ISO_FLAG_SUBDIR) ? 'D' : 'f';

               dfsParseFileSelection( select, text, &isMinimum, &threshold, &minS, &maxS, &modTstamp, &itemType);
               switch (st)
               {
                  case 'D':
                     if      (itemType == DFS_FS_ITEM_BROWSE) // no filtering on Directories, take all.
                     {
                        strcpy( text, "");      // no wildcard
                        modTstamp = 0;          // no timestamp filter
                        minS      = 0;
                        maxS      = 0;          // no size filter
                     }
                     else if (itemType == DFS_FS_ITEM_FILES) // filter files, discard Directories
                     {
                        rc = DFS_ST_MISMATCH;   // item-type mismatch
                        break;
                     }
                  case 'f':
                  default:
                     if      (itemType == DFS_FS_ITEM_DIRS) // filter dirs, discard files
                     {
                        rc = DFS_ST_MISMATCH;   // item-type mismatch
                        break;
                     }
                     dfsIsoLsnInfo2Path( dirLsn, fileIdx, shortpath);

                     if ((strlen(text) == 0) || (TxStrWicmp(shortpath, text) >= 0))
                     {
                        if (dfsIsoLE32( de->FileSize) == 0)  // empty
                        {
                           size = 0;
                        }
                        else
                        {
                           size = ((dfsIsoLE32( de->FileSize) -1) / iso->BlockSize) + 1;
                        }
                        location = dfsIsoLE32( de->Location);

                        if (de->FileFlags & ISO_FLAG_SUBDIR)
                        {
                           TRACES(("Filter shortpath length: %d = '%s'\n", strlen( shortpath), shortpath));
                           if ((location == dfsIsoLE32(iso->vd->p.RootDirEntry.Location)) && (location == dirLsn))
                           {
                              strcpy( shortpath, FS_PATH_STR);
                           }
                           else                 // append PATH_SEP to subdirectories
                           {
                              if (shortpath[strlen(shortpath) -1] != '.') // not on . and ..
                              {
                                 strcat( shortpath, FS_PATH_STR);
                              }
                           }
                           TRACES(("Result shortpath: '%s'\n", shortpath));
                        }

                        if ((modifiedTime > modTstamp) &&                        //- Timestamp match
                            ((size >= minS) && ((size <= maxS) || (maxS == 0)))) //- Size match
                        {
                           if (verbose)         // include alloc and size
                           {
                              sprintf(   text, "%s%c      %s  %8X ", CBM, st, CNN, location);
                              dfstrSize( text, CBC, size, " ");
                              strcat(    text, "         "); // empty EA/Xattr column
                           }
                           else
                           {
                              strcpy( text, ""); // remove any stray text, non-verbose
                           }
                           if (output)          // not totally silent?
                           {
                              strcat(    lead, text);
                              TxPrint( "%s%s%s%s%s\n", lead, CBY, shortpath, CNN, CGE);
                           }
                           else                 // return info in 'select'
                           {
                              time_t tt = dfsIsoFileTime2t( &de->FileDateTime);
                              strftime( temp, TXMAXTT, "%Y-%m-%d %H:%M:%S", localtime( &tt));

                              TxStripAnsiCodes( text);
                              sprintf( select, "%s %s", text, temp);
                              dfstrUllDot20( select, "", dfsIsoLE32( de->FileSize), "");
                           }
                           rc = NO_ERROR;
                        }
                        else
                        {
                           TRACES(("file-size mismatch\n"));
                           rc = DFS_ST_MISMATCH; // file-size mismatch
                        }
                     }
                     else
                     {
                        rc = DFS_ST_MISMATCH;   // wildcard mismatch
                     }
                     strcpy(  param, shortpath); // return the found name
                     break;
               }
            }
            else
            {
               rc = DFS_ST_MISMATCH;            // info is NOT a valid entrynr
            }
         }
         else if (dirsect < iso->BlockCount)    // should be there
         {
            dfsX10("\nError reading sector : ", dirsect, CBR, "\n");
         }
         TxFreeMem(sb);
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   else
   {
      rc = DFS_ST_MISMATCH;                     // info is NOT a valid entrynr
   }
   RETURN(rc);
}                                               // end 'dfsIsoFileInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Replace sn-list by contents of a single Directory (InoLsn, InoIdx pairs)
/*****************************************************************************/
ULONG dfsIsoMakeBrowseList                      // RET   LSN is valid INODE
(
   ULN64               lsn,                     // IN    Directory LSN
   ULN64               info,                    // IN    dir entry, 0 for ROOT
   char               *str,                     // IN    unused
   void               *param                    // INOUT unused
)
{
   ULONG               rc = NO_ERROR;
   DFSISPACE           is;                      // Directory in ISPACE

   ENTER();
   TRARGS(("LSN:0x%llX, info:0x%2.2x\n", lsn, info));

   if ((info & DFSSNINFO) ||                    // valid entry, get the DIR sector
       (lsn == dfsIsoLE32(iso->vd->p.RootDirEntry.Location))) // or the ROOT
   {
      if ((rc = dfsIsoGetAllocSpace( lsn, info, NULL, &is)) == NO_ERROR)
      {
         rc = dfsIsoDirSpace2List( &is);
         TxFreeMem( is.space);                  // free the SPACE (calloc/realloc)
      }
   }
   RETURN(rc);
}                                               // end 'dfsIsoMakeBrowseList'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Put contents of a Directory SSPACE (DirLsn, entry pairs) into sn-list
/*****************************************************************************/
static ULONG dfsIsoDirSpace2List
(
   DFSISPACE          *dirIsp                   // IN    DIR allocation in ISPACE
)
{
   ULONG               rc = NO_ERROR;
   ULONG               dirSize;                 // total DIR size in SPACE
   ULN64               dirsec = 0;              // current relative dirblock LSN
   BYTE               *block;
   BYTE               *at;                      // current position
   S_ISO_DIRENTRY     *curDir;                  // current as directory struct
   ULONG               dirSizeDone;             // DIR block size handled
   ULONG               blockRemain;             // remaining size in this block
   ULONG               entry;                   // index in iso directory

   ENTER();

   if ((block = TxAlloc( 1, iso->BlockSize)) != NULL)
   {
      if (!TxaOptUnSet('l'))                    // create a new list for DIR
      {
         dfsInitList(0, "-f -P", "-d");         // optimal for menu file-recovery
      }

      dirSize     = dfsSspaceSectors( TRUE, dirIsp->chunks, dirIsp->space) * iso->BlockSize;
      dirSizeDone = 0;
      do
      {
         if ((rc = dfsSspaceReadFilePart( dirIsp->chunks, dirIsp->space, dirsec, 1, block)) == NO_ERROR)
         {
            blockRemain = iso->BlockSize;       // bytes remaining in block
            at          = block;                // set to start of this block
            entry       = 0;                    // first entry in this block

            do
            {
               curDir = (S_ISO_DIRENTRY *) at;

               if (curDir->RecordLength > 0)    // display this DIR entry record
               {
                  if ((curDir->NameLength == 1) && (curDir->FileName[0] <= 0x01)) // . or .. entry
                  {
                     if ((curDir->FileName[0] == 0x01) && // a .. entry, that is NOT in the ROOT
                         (dirIsp->space[0].start != dfsIsoLE32(iso->vd->p.RootDirEntry.Location)))
                     {
                        if (dfsa->snlist[0] == 0) // it is the first entry to be added
                        {
                           TRACES(("List marked as 1ST_PARENT\n"));
                           DFSBR_SnlistFlag |= DFSBR_1ST_PARENT;
                           dfsAddSI2List( dfsSspaceRsn2Lsn( dirIsp->chunks, dirIsp->space, dirsec), entry);
                        }
                     }
                  }
                  else                          // regular entries
                  {
                     BOOL          filtered = FALSE;

                     if (dfsa->browseShowHidden == FALSE) // check hidden attribute and DOT f-name
                     {
                        if ( (curDir->FileName[0] == '.') ||
                            ((curDir->FileFlags & ISO_FLAG_HIDDEN) != 0))
                        {
                           filtered = TRUE;
                        }
                     }
                     if (!filtered)
                     {
                        dfsAddSI2List( dfsSspaceRsn2Lsn( dirIsp->chunks, dirIsp->space, dirsec), entry);
                     }
                  }
                  entry++;

                  at          += curDir->RecordLength; // advance to next
                  blockRemain -= curDir->RecordLength; // and keep count ...
               }
            } while ((blockRemain > 0) && (curDir->RecordLength > 0) && (rc == NO_ERROR) && (!TxAbort()));

            dirSizeDone += iso->BlockSize;      // one more DIR block handled
            dirsec++;                           // to next relative sector in ISPACE
         }
      } while ((dirSizeDone < dirSize) && (rc == NO_ERROR) && (!TxAbort()));

      TxFreeMem( block);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsIsoDirSpace2List'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get allocation information for DIR or FILE into integrated-SPACE structure
// starting from a given directory entry, or for ROOT (when info value is 0)
/*****************************************************************************/
ULONG dfsIsoGetAllocSpace
(
   ULN64               lsn,                     // IN    Directory LSN   or 0
   ULN64               info,                    // IN    directory entry or 0
   char               *str,                     // IN    unused
   void               *param                    // INOUT Integrated SPACE
)
{
   ULONG               rc = NO_ERROR;
   DFSISPACE          *ispace = (DFSISPACE *) param;
   BYTE               *sb = NULL;               // sector buffer
   S_ISO_DIRENTRY     *de;                      // ISO directory entry
   USHORT              entry   = info & 0x7f;   // remove info flag (entry < 60)

   ENTER();
   TRACES(("lsn: 0x%llx  info: 0x%x\n", lsn, info));

   ispace->space = NULL;

   if (info & DFSSNINFO)                        // valid entry, get the DIR sector
   {
      if ((sb = TxAlloc(1, dfsGetSectorSize())) != NULL)
      {
         if ((rc = dfsRead( lsn, 1, sb)) == NO_ERROR)
         {
            switch (dfsIdentifySector( lsn, 0, sb))
            {
               case ST_ISODIR:
                  if (((de = dfsIsoDirIndex2Entry( sb, entry)) != NULL) &&
                       (dfsIsoLE32( de->Location) != 0))
                  {
                     rc = dfsIsoAllocChain( dfsIsoLE32( de->Location),
                                            dfsIsoLE32( de->FileSize),
                                                        de->FileUnitSize,
                                                        de->InterleaveGap,
                                       &ispace->chunks, &ispace->space);
                  }
                  else
                  {
                     rc = DFS_BAD_STRUCTURE;
                  }
                  break;

               default:
                  rc = DFS_ST_MISMATCH;         // not a directory sector
                  break;
            }
         }
         TxFreeMem(sb);
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   else if (info == 0)                          // ROOT directory
   {
      rc = dfsIsoAllocChain( dfsIsoLE32(iso->vd->p.RootDirEntry.Location),
                             dfsIsoLE32(iso->vd->p.RootDirEntry.FileSize),
                                        iso->vd->p.RootDirEntry.FileUnitSize,
                                        iso->vd->p.RootDirEntry.InterleaveGap,
                       &ispace->chunks, &ispace->space);
   }
   else                                         // info/entry is mandatory!
   {
      rc = DFS_BAD_STRUCTURE;
   }
   RETURN (rc);
}                                               // end 'dfsIsoGetAllocSpace'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create S_SPACE structure for an ISO allocation chain, supporting interleave
// Note: fileUnitSize and interleaveGap should be ZERO for directories!
/*****************************************************************************/
ULONG dfsIsoAllocChain
(
   ULONG               block,                   // IN    starting block nr
   ULONG               maxsize,                 // IN    file-size
   BYTE                fileUnitSize,            // IN    #contiguous blocks, or 0
   BYTE                interleaveGap,           // IN    #blocks between FU, or 0
   ULONG              *chunks,                  // OUT   nr of space entries
   S_SPACE           **space                    // OUT   space allocation
)
{
   ULONG               rc = NO_ERROR;
   ULONG               blocks  = 0;             // nr of blocks
   ULONG               this    = block;         // current block
   ULONG               extSize = 0;             // sectors in this chunk
   ULONG               extents = 0;             // nr of extents
   ULONG               curExt  = 0;             // current extent
   S_SPACE            *sp      = NULL;

   ENTER();
   TRARGS(("Input block: %8.8X, maxsize: %u\n", block, maxsize));

   if (maxsize > 0)
   {
      blocks = ((maxsize - 1) / dfsGetSectorSize()) + 1;

      if ((fileUnitSize == 0) || (interleaveGap == 0)) // contiguous
      {
         extents = 1;
         extSize = blocks;
      }
      else
      {
         extents = (blocks / fileUnitSize) + 1;
         extSize = fileUnitSize;
      }

      if ((sp = (S_SPACE *) TxAlloc( extents, sizeof(S_SPACE))) != NULL)
      {
         for (curExt = 0; curExt < extents; curExt++)
         {
            sp[ curExt].size  = extSize;
            sp[ curExt].start = this;

            this += (extSize + interleaveGap);  // next start sector, after interleave
         }
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }

   if (sp != NULL)
   {
      TRACES(( "sp[extents -1].start: 0x%llx        .size: 0x%llx\n",
                sp[extents -1].start, sp[extents -1].size));

      *space  = sp;
      *chunks = extents;
   }
   else
   {
      *space  = NULL;                           // dont return random value!
      *chunks = 0;
   }
   RETURN (rc);
}                                               // end 'dfsIsoAllocChain'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Find full PATH to Root-directory, starting at some Node LSN and index info
/*****************************************************************************/
ULONG dfsIsoLsnInfo2Path                        // RET   result
(
   ULN64               lsn,                     // IN    Directory LSN
   ULONG               info,                    // IN    dir entry, 0 for ROOT
   char               *path                     // OUT   combined path string
)
{
   ULONG               rc = NO_ERROR;
   BYTE               *sb  = NULL;              // sector buffer
   S_ISO_DIRENTRY     *de;                      // ISO directory entry
   TXLN                fileName;
   ULONG               pcIndex;                 // index in path-cache
   char               *pcPath;                  // found path in cache

   ENTER();
   TRACES(("lsn: 0x%llx, entry: 0x%2.2x\n", lsn, info));

   if (path)
   {
      *path = '\0';                             // start with empty path

      if ((sb = TxAlloc( 1, iso->BlockSize)) != NULL)
      {
         if ((rc = dfsRead( lsn, 1, sb)) == NO_ERROR)
         {
            switch (dfsIdentifySector( lsn, 0, sb))
            {
               case ST_ISODIR:
                  if ((de = dfsIsoDirIndex2Entry( sb, DFSSNIGET( info))) != NULL)
                  {
                     //- find enclosing directory path in the path-cache, linear search
                     for (pcPath = NULL, pcIndex = 0; pcIndex < iso->pathCache.sdcUsed; pcIndex++)
                     {
                        if (iso->pathCache.sdCache[ pcIndex].DirBlockLsn == lsn) // found our dir block
                        {
                           pcPath = iso->pathCache.sdCache[ pcIndex].Path;
                           break;
                        }
                     }
                     if (pcPath != NULL)        // found dirblock in cache
                     {
                        TRACES(("found dir-path: '%s'\n", pcPath));
                        strcpy( path, pcPath);  // start with cached path to this directory block
                     }
                     if (strlen( path) > 0)
                     {
                        if ((de->NameLength == 1) && (de->FileName[0] <= 0x01)) // . or ..
                        {
                           path[ strlen( path) - 1] = 0; // remove trailing path separator

                           if (de->FileName[0] == 0x01) // .. directory
                           {
                              if ((pcPath = strrchr( path, FS_PATH_SEP)) != NULL)
                              {
                                 *pcPath = 0;   // remove last path component, leaving parent path
                              }
                           }
                           if (strlen( path) == 0) // empty now,
                           {
                           }
                           else
                           {
                           }
                        }
                        else                    // regular filenames
                        {
                           if (iso->vdUnicode)  // This VD uses Unicode (Joliet)
                           {
                              TxUnic2Ascii( (USHORT *) de->FileName, fileName, de->NameLength / 2);
                           }
                           else
                           {
                              TxCopy( fileName, (char *) de->FileName,  de->NameLength + 1);
                           }
                           if (fileName[ strlen( fileName) - 2] == ';') // fileversion postfix
                           {
                              fileName[ strlen( fileName) - 2] = 0; // remove the postfix
                           }
                           strcat( path, fileName); // add this filename to retrieved directory path
                        }
                     }
                  }
                  else
                  {
                     rc = DFS_ST_MISMATCH;
                  }
                  break;

               default:
                  rc = DFS_PENDING;
                  break;
            }
         }
         TxFreeMem(sb);
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   TRACES(("Found path to root for 0x%llX+%4.4x:'%s'\n", lsn, info, path));
   RETURN(rc);
}                                               // end 'dfsIsoLsnInfo2Path'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Build cache with (sub) directoryblock to full-path from ROOT
/*****************************************************************************/
ULONG dfsIsoBuildDirPathCache
(
   void
)
{
   ULONG               rc = NO_ERROR;           // function return
   DFSISPACE           rootIsp;                 // ROOT DIR sectors in ISPACE

   ENTER();

   dfsIsoFreeDirPathCache();                    // free existing cache, and re-init

   if ((iso->vd->vdType == ISO_VDT_PRIMARY) || (iso->vd->vdType == ISO_VDT_SUPPLEMENTAL))
   {
      iso->pathCache.sdcSize = max( ISO_SDC_MIN_SIZE, iso->BlockCount / 16);
      iso->pathCache.sdcDirs = 1;               // start count with ROOT directory

      if ((iso->pathCache.sdCache = TxAlloc( iso->pathCache.sdcSize, sizeof( ISOSDCELEM))) != NULL)
      {
         if ((rc = dfsIsoGetAllocSpace( dfsIsoLE32(iso->vd->p.RootDirEntry.Location), 0, NULL, &rootIsp)) == NO_ERROR)
         {
            txwAllowUserStatusMessages( TRUE);  // only our own messages now
            rc = dfsIsoDirAddPathCache( 0, FS_PATH_STR, &rootIsp);
            txwAllowUserStatusMessages( FALSE); // resume default status text

            TxFreeMem( rootIsp.space);
         }
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   RETURN (rc);
}                                               // end 'dfsIsoBuildDirPathCache'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add (sub) directory to ISO path cache, recursing over its subdirectories
/*****************************************************************************/
static ULONG dfsIsoDirAddPathCache
(
   ULONG               recurseLevel,            // IN    recursion level (limit)
   char               *thisPath,                // IN    path from root to THIS dir
   DFSISPACE          *dirIsp                   // IN    DIR allocation in ISPACE
)
{
   ULONG               rc = NO_ERROR;
   ULONG               dirSize;                 // total DIR size in SPACE
   ULN64               dirsec = 0;              // current relative dirblock LSN
   BYTE               *block;
   BYTE               *at;                      // current position
   S_ISO_DIRENTRY     *de;                      // current as directory struct
   ULONG               dirSizeDone;             // DIR block size handled
   ULONG               blockRemain;             // remaining size in this block
   ULONG               entry;                   // index in fat directory
   TXLN                fileName;                // filename for an entry
   TXLN                subDirPath;              // path from root to a  SUBDIR
   DFSISPACE           subDirIsp;               // SUBDIR allocation in ISPACE

   ENTER();
   TRACES(("recurseLevel: %u first sector: 0x%llx  path: '%s'\n", recurseLevel, dirIsp->space[0].start, thisPath));

   if (recurseLevel < ISO_MAX_DIR_DEPTH)
   {
      //- first add all sectors for this directory to the cache, with path string
      dirSize = dfsSspaceSectors( TRUE, dirIsp->chunks, dirIsp->space); // #sectors in directory
      for (dirsec = 0; dirsec < dirSize; dirsec++)
      {
         if ((entry = iso->pathCache.sdcUsed) < iso->pathCache.sdcSize)
         {
            if ((iso->pathCache.sdCache[ entry].Path = TxAlloc( 1, strlen( thisPath) + 1)) != NULL)
            {
               strcpy( iso->pathCache.sdCache[ entry].Path, thisPath);
               iso->pathCache.sdCache[ entry].DirBlockLsn = dfsSspaceRsn2Lsn( dirIsp->chunks, dirIsp->space, dirsec);
            }
            else
            {
               break;
            }
            iso->pathCache.sdcUsed++;           // advance to next cache entry
         }
      }
      sprintf( fileName, "Path cache %6u at: %s", iso->pathCache.sdcUsed, thisPath);
      txwSetSbviewStatus( fileName, cSchemeColor); // acts as progress indicator on statusline

      //- then traverse this directory to process any subdirectories recursively
      if ((block = TxAlloc( 1, iso->BlockSize)) != NULL)
      {
         dirSize     = dfsSspaceSectors( TRUE, dirIsp->chunks, dirIsp->space) * iso->BlockSize;
         dirSizeDone = 0;
         dirsec      = 0;                       // start at first block of directory
         do
         {
            if ((rc = dfsSspaceReadFilePart( dirIsp->chunks, dirIsp->space, dirsec, 1, block)) == NO_ERROR)
            {
               blockRemain = iso->BlockSize;    // bytes remaining in block
               at          = block;             // set to start of this block
               entry       = 0;                 // first entry in this block

               do
               {
                  de = (S_ISO_DIRENTRY *) at;

                  if (de->RecordLength > 0)     // display this DIR entry record
                  {
                     if (de->FileFlags & ISO_FLAG_SUBDIR) // if it is a directory
                     {
                        if ((de->NameLength > 1) || (de->FileName[0] > 0x01)) // not the . or .. entry
                        {
                           if (iso->vdUnicode)  // This VD uses Unicode (Joliet)
                           {
                              TxUnic2Ascii( (USHORT *) de->FileName, fileName, de->NameLength / 2);
                           }
                           else
                           {
                              TxCopy( fileName, (char *) de->FileName,  de->NameLength + 1);
                           }
                           if (fileName[ strlen( fileName) - 2] == ';') // fileversion postfix
                           {
                              fileName[ strlen( fileName) - 2] = 0; // remove the postfix
                           }

                           if ((rc = dfsIsoGetAllocSpace( dfsSspaceRsn2Lsn( dirIsp->chunks, dirIsp->space, dirsec),
                                                          entry | DFSSNINFO, NULL, &subDirIsp)) == NO_ERROR)
                           {
                              TRACES(("entry: %u subdir name: '%s'\n", entry, fileName));

                              sprintf( subDirPath, "%s%s%s", thisPath, fileName, FS_PATH_STR); // path for this subdir

                              rc = dfsIsoDirAddPathCache( recurseLevel + 1, subDirPath, &subDirIsp);

                              TxFreeMem( subDirIsp.space);
                           }
                           iso->pathCache.sdcDirs++; // count for statistic reporting
                        }
                     }
                     else
                     {
                        iso->pathCache.sdcFiles++; // count for statistic reporting
                     }
                     entry++;
                     at          += de->RecordLength; // advance to next
                     blockRemain -= de->RecordLength; // and keep count ...
                  }
               } while ((blockRemain > 0) && (de->RecordLength > 0) && (rc == NO_ERROR));

               dirSizeDone += iso->BlockSize;   // one more DIR block handled
               dirsec++;                        // to next relative sector in ISPACE
            }
         } while ((dirSizeDone < dirSize) && (rc == NO_ERROR) && (!TxAbort()));
         TxPrint( "%s", CNN);

         TxFreeMem( block);
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   else
   {
      rc = DFS_BAD_STRUCTURE;
   }
   RETURN( rc);
}                                               // end 'dfsIsoDirAddPathCache'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Free cache with (sub) directoryblock to full-path information
/*****************************************************************************/
void dfsIsoFreeDirPathCache
(
   void
)
{
   ULONG               i;

   ENTER();

   for (i = 0; i < iso->pathCache.sdcUsed; i++)
   {
      TxFreeMem(iso->pathCache.sdCache[ i].Path); // free the path strings
   }
   TxFreeMem( iso->pathCache.sdCache);          // free existing cache array

   iso->pathCache.sdcSize  = 0;
   iso->pathCache.sdcUsed  = 0;
   iso->pathCache.sdcDirs  = 0;
   iso->pathCache.sdcFiles = 0;

   VRETURN();
}                                               // end 'dfsIsoFreeDirPathCache'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Determine allocation-bit for specified LSN, ALLOCATED beyond last block!
/*****************************************************************************/
ULONG dfsIsoAllocated                           // RET   LSN is allocated
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   return( 1);                                  // always ALLOCATED!
}                                               // end 'dfsIsoAllocated'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Convert ISO Vol-Descriptor DATE/TIME to DFS standard date/time string (19)
/*****************************************************************************/
char *dfsIsoVtime2str                           // RET   string value
(
   S_ISO_VTIME        *vtm,                     // IN    ISO time value
   char               *dtime                    // INOUT ptr to string buffer
)
{
   ENTER();

   sprintf( dtime, "%4.4s-%2.2s-%2.2s %2.2s:%2.2s:%2.2s,%2.2s", vtm->Year, vtm->Month, vtm->Day,
                                        vtm->Hour, vtm->Minute, vtm->Second, vtm->Hundredths);

   RETURN( dtime);
}                                               // end 'dfsIsoTime2str'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Convert ISO DATE/TIME (binary struct) to standard C time_t value
/*****************************************************************************/
time_t dfsIsoFileTime2t                         // RET   time_t representation
(
   S_ISO_FTIME        *ftm                      // IN    ISO file time value
)
{
   time_t              tm = 0;
   struct tm           stm;

   ENTER();

   stm.tm_year = ftm->Year;                    // derive struct tm value from ftm
   stm.tm_mon  = ftm->Month -1;
   stm.tm_mday = ftm->Day;
   stm.tm_hour = ftm->Hour;
   stm.tm_min  = ftm->Minute;
   stm.tm_sec  = ftm->Second;

   tm = mktime( &stm);                          // create a time_t value for this

   RETURN( tm);
}                                               // end 'dfsIsoFileTime2t'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate ISO directory Index to a directory entry pointer
/*****************************************************************************/
S_ISO_DIRENTRY *dfsIsoDirIndex2Entry
(
   BYTE               *block,                   // IN    One directory block
   ULONG               index                    // IN    Index for wanted entry
)
{
   S_ISO_DIRENTRY     *rc = NULL;
   BYTE               *at;                      // current position
   S_ISO_DIRENTRY     *curDir;                  // current as directory struct
   ULONG               blockRemain;             // remaining size in this block
   ULONG               entry;                   // index in fat directory

   ENTER();
   TRACES(("index: %u\n", index));

   blockRemain = iso->BlockSize;                // bytes remaining in block
   at          = block;                         // set to start of this block
   entry       = 0;                             // first entry in this block

   while (blockRemain > 0)
   {
      curDir = (S_ISO_DIRENTRY *) at;

      if (entry == index)                       // we got to the wanted entry
      {
         rc = curDir;
         break;
      }

      if (curDir->RecordLength > 0)
      {
         entry++;
         at          += curDir->RecordLength;   // advance to next
         blockRemain -= curDir->RecordLength;   // and keep count ...
      }
      else
      {
         break;
      }
   }
   RETPTR (rc);
}                                               // end 'dfsIsoDirIndex2Entry'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display all available info about an ISO directory entry
/*****************************************************************************/
ULONG dfsIsoShowDirectoryEntry
(
   ULN64               dirBlock,                // IN    lsn of ISO dir block
   ULONG               entry                    // IN    index of dir entry
)
{
   ULONG               rc;
   BYTE               *sb  = NULL;              // sector buffer
   S_ISO_DIRENTRY     *de;                      // ISO directory entry

   ENTER();
   TRACES(("dirBlock: 0x%llx, entry: 0x%2.2x\n", dirBlock, entry));

   if ((sb = TxAlloc( 1, iso->BlockSize)) != NULL)
   {
      if ((rc = dfsRead( dirBlock, 1, sb)) == NO_ERROR)
      {
         switch (dfsIdentifySector( dirBlock, 0, sb))
         {
            case ST_ISODIR:
               if ((de = dfsIsoDirIndex2Entry( sb, entry)) != NULL)
               {
                  if (de->FileFlags & ISO_FLAG_SUBDIR) // if it is a directory
                  {
                     TRACES(("Cache ParentLsn: 0x%llx : %u for LSN 0x%llx\n", dirBlock, entry, dfsIsoLE32( de->Location)));
                     iso->dirCache.SubdirLsn   = dfsIsoLE32( de->Location);
                     iso->dirCache.ParentLsn   = dirBlock;
                     iso->dirCache.ParentEntry = entry | DFSSNINFO;
                  }
                  if (!TxaOptUnSet('X'))        // unless -X- specified
                  {
                     nav.down = dfsIsoLE32( de->Location);
                  }
                  TxPrint("\n");
                  dfsIsoDirHeader( "DirBlock   entry", 0); // display header
                  dfsX10("   ", dirBlock, CBC, " ");
                  TxPrint("%s%2x%s ", CNC, entry, CNN);
                  dfsIsoShowDirEntry( de, CcZ);
                  TxPrint("\n");
               }
               else
               {
                  TxPrint( "Directory entry index: %d not found!\n", entry);
                  rc = DFS_ST_MISMATCH;
               }
               break;

            default:
               rc = DFS_PENDING;
               break;
         }
      }
      else if (dirBlock < iso->BlockCount)       // should be there
      {
         dfsX10("\nError reading  block : ", dirBlock, CBR, "\n");
      }
      TxFreeMem(sb);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN( rc);
}                                               // end 'dfsIsoShowDirectoryEntry'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// ISO filesystem, display DIR header/footer
/*****************************************************************************/
void dfsIsoDirHeader
(
   char               *lead,                    // IN    lead text-string
   ULONG               items                    // IN    DIR-lines shown (0 = TOP)
)
{
   TXLN                line;

   sprintf(line, " ================ ========== ======== ====== ============= ======================================\n");
   if (items > 0)
   {
      TxPrint( line);
   }
   TxPrint(                    " %s Creation Date/Time  Attrib      Filesize Filename (optional posix name, yellow)\n", lead);
   if (items == 0)
   {
      TxPrint( line);
   }
}                                               // end 'dfsIsoDirHeader'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display ISO directory entry on a single line (optional 2nd for Posix LFN)
/*****************************************************************************/
void dfsIsoShowDirEntry
(
   S_ISO_DIRENTRY     *entry,                   // IN    ISO directory entry
   BYTE                bg                       // IN    current background
)
{
   TXLN                fileName;                // large filename buffer
   TXTT                text;                    // smaller text buffer
   time_t              tt;                      // time value

   tt = dfsIsoFileTime2t( &entry->FileDateTime);
   strftime( text, TXMAXTT, "%Y-%m-%d %H:%M:%S ", localtime( &tt));
   dfstrIsoAttrib( text, entry->FileFlags);     // add the attribute characters  (6)
   dfstrUlDot13(   text, " ", dfsIsoLE32( entry->FileSize), ""); // add filesize (13)

   if      ((entry->NameLength == 1) && (entry->FileName[0] == 0x00))
   {
      strcpy( fileName, ".");
   }
   else if ((entry->NameLength == 1) && (entry->FileName[0] == 0x01))
   {
      strcpy( fileName, "..");
   }
   else
   {
      if (iso->vdUnicode)                       // This VD uses Unicode (Joliet)
      {
         TxUnic2Ascii( (USHORT *) entry->FileName, fileName, entry->NameLength / 2);
      }
      else
      {
         TxCopy( fileName, (char *) entry->FileName,  entry->NameLength + 1);
      }
      if (fileName[ strlen( fileName) - 2] == ';') // fileversion postfix
      {
         fileName[ strlen( fileName) - 2] = 0;  // remove the postfix
      }
   }
   TxPrint( " %s %s", text, fileName);

   //- handle possible Rock Ridge extensions, display Posix info etc on separate line
   if ((entry->NameLength + 21) < entry->RecordLength) // extra info could be present
   {
      SSR_SP_RECORD   *sp;
      BYTE            *at = entry->FileName;
      int              todo = entry->RecordLength - entry->NameLength - 21;
      TXLN             altName;                 // Alternative filename buffer
      TXTS             posixMode;               // Posix attribute string, filemode
      TXTS             posixUser;               // Posix attribute string, user-id
      TXTS             posixGroup;              // Posix attribute string, group-id
      ULONG            rrRecords = 0;           // Number of supported RockRidge records seen

      at += entry->NameLength;
      if ((entry->NameLength & 0x01) == 0)      // even length, skip padding byte
      {
         at++;
      }
      sp = (SSR_SP_RECORD *) at;
      if ((sp->Signature == RRP_SUSPI_DATA_SIG) && (sp->Length == 7) && (sp->SuspSig == RRP_SUSPI_CHECKSIG))
      {
         TRACES(("Set SUSP skip bytes to: %hhu\n", sp->LenSkip));
         iso->SuspSkipBytes = sp->LenSkip;
         at += sp->Length;                      // skip to first real SUSP record (RR)
      }
      else                                      // skip specified number of bytes
      {
         at += iso->SuspSkipBytes;
      }
      TRACES(("SUSP present, entry start: %p, first SUSP record at: %p", entry, at));

      //- iterate over extension records, and display the Rock-Ridge extra file info

      strcpy( text,  "                    ");   // 20 char empty modify time  (default)
      strcpy( posixMode,  "       ");           //  7 char empty posix mode   (default)
      strcpy( posixUser,  "       ");           //  7 char empty posix user   (default)
      strcpy( posixGroup, "       ");           //  6 char empty posix group  (default)
      strcpy( altName, "");                     // empty alternate name       (default)

      sp = (SSR_SP_RECORD *) at;                // next area as SUSP record
      do
      {
         switch (sp->Signature)                 // handle the wanted records only
         {
            case RRP_TIMES_DATA_SIG:
               {
                  SSR_TF_RECORD *px = (SSR_TF_RECORD *) at;

                  if (px->TfFlags & RRP_TFFLG_MODIFY) // last modify is present
                  {
                     if (px->TfFlags & RRP_TFFLG_LONG_FORM) // 17 character date format
                     {
                        S_ISO_VTIME *vtime = (S_ISO_VTIME *) px->Data; // point to first date/time

                        if (px->TfFlags & RRP_TFFLG_CREATION) // there is a creation date/time
                        {
                           vtime++;             // increment to next date/time slot
                        }
                        dfsIsoVtime2str( vtime, text);
                        text[ 19] = ' ';
                        text[ 20] = 0;          // replace hundredths-sec by a space
                     }
                     else                       // standard 7 byte date format (like filetime)
                     {
                        S_ISO_FTIME *ftime = (S_ISO_FTIME *) px->Data; // point to first date/time

                        if (px->TfFlags & RRP_TFFLG_CREATION) // there is a creation date/time
                        {
                           ftime++;             // increment to next date/time slot
                        }
                        tt = dfsIsoFileTime2t( ftime);
                        strftime( text, TXMAXTT, "%Y-%m-%d %H:%M:%S ", localtime( &tt));
                     }
                     rrRecords++;               // signal output is desired
                  }
               }
               break;

            case RRP_POSIX_DATA_SIG:
               {
                  SSR_PX_RECORD *px = (SSR_PX_RECORD *) at;

                  if (dfsIsoLE32( px->FileMode) != 0)
                  {
                     sprintf( posixMode,  "%6.6o ", dfsIsoLE32( px->FileMode));
                  }
                  if (dfsIsoLE32( px->UserId) != 0)
                  {
                     sprintf( posixUser,  "%6.6u ", dfsIsoLE32( px->UserId));
                  }
                  if (dfsIsoLE32( px->GroupId) != 0)
                  {
                     sprintf( posixGroup, "%6.6u ", dfsIsoLE32( px->GroupId));
                  }
                  rrRecords++;                  // signal output is desired
               }
               break;

            case RRP_ANAME_DATA_SIG:
               {
                  SSR_NM_RECORD *nm = (SSR_NM_RECORD *) at;

                  if (nm->NmFlags == 0)         // not the . or .. entries etc
                  {
                     if (iso->vdUnicode)        // This VD uses Unicode (Joliet)
                     {
                        TxUnic2Ascii( (USHORT *) nm->AltName, altName, (nm->Length - 5) / 2);
                     }
                     else
                     {
                        TxCopy( altName, (char *) nm->AltName,  nm->Length - 5 + 1);
                     }
                     if (strcmp( fileName, altName) == 0) // names are identical, dont display
                     {
                        strcpy( altName, "");
                     }
                     else
                     {
                        rrRecords++;            // signal output is desired
                     }
                  }
               }
               break;

            default:                            // ignore other records for now
               break;
         }

         if (sp->Length >= 4)                   // advance to next SUSP/RR record
         {
            at   += sp->Length;
            todo -= sp->Length;
            sp = (SSR_SP_RECORD *) at;          // next area as SUSP record
         }
         else
         {
            break;                              // exit on error condition
         }
      } while ((todo > 4) && (sp->Version == 1));

      if (rrRecords > 0)                        // there is RockRidge info
      {
         TxPrint( "%s\n mTm+fMode+UsGrp: %s%s%s%s", CGE, text, posixMode, posixUser, posixGroup);
         if (strlen( altName))                  // add altName in bright yellow
         {
            TxPrint( "%s%s%s", ansi[Ccol((CcY | CcI), bg)], altName,
                               ansi[Ccol( CcW       , bg)]);
         }
      }
   }
}                                               // end 'dfsIsoShowDirEntry'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Format ISO file/directory attribute bits into a 6 character string
/*****************************************************************************/
void dfstrIsoAttrib
(
   char               *str,                     // INOUT text string
   BYTE                data                     // IN    data
)
{
   TXTS                attrib;

   strcpy( attrib, "      ");
   if (data & ISO_FLAG_NEXT_R) attrib[0] = '+';
   if (data & ISO_FLAG_ASSO_F) attrib[1] = 'A';
   if (data & ISO_FLAG_EXATTR) attrib[2] = 'S';
   if (data & ISO_FLAG_HIDDEN) attrib[3] = 'H';
   if (data & ISO_FLAG_OWNGRP) attrib[4] = 'O';
   if (data & ISO_FLAG_SUBDIR) attrib[5] = 'D';

   strcat( str, attrib);
}                                               // end 'dfstrIsoAttrib'
/*---------------------------------------------------------------------------*/


