//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// HPFS utility functions
//
// Author: J. van Wijk
//
// JvW  22-11-97   Initial version, split off from DFSAHPFS
//
#ifndef    DFSUHPFS
   #define DFSUHPFS

// Find FNODE for specified path, starting at root-directory FNODE
ULONG dfsHpfsFindPath
(
   ULN64               loud,                    // IN    Show progress
   ULN64               d2,                      // IN    dummy
   char              *path,                     // IN    path specification
   void              *vp                        // OUT   found dir/file FNODE
);

// Resolve DIRBL LSN from directory FNODE LSN
BOOL dfsHpfsFnode2Dirblock                      // OUT   DIRBL resolved
(
   ULONG              fn,                       // IN    starting FNODE lsn
   ULONG             *lsn                       // OUT   found DirBlock lsn
);

// Find DIR-entry for Fnode, start at DIRBLOCK LSN, walk Btree; return DIR-entry
// Input is either a complete filename or a partial (15) plus the Fnode LSN
// Optional rename to SAME LENGTH new name
S_DIRENT *dfsHpfsDirblockFindEntry              // RET   DIR-entry found or NULL
(                                               //       (allocated, TxFreeMem!)
   ULONG               loud,                    // IN    Show progress
   char               *rename,                  // IN    same length new name
   char               *entry,                   // IN    entry (partial) filename
   ULONG               fnode,                   // IN    Fnode LSN or 0
   ULONG               dirbl                    // IN    starting DIRBLOCK lsn
);

// Find FNODE for Root-directory, starting at some LSN in the volume
BOOL dfsHpfsFindRootFnode                       // OUT   root found
(
   ULONG               start,                   // IN    starting lsn
   BOOL                verbose,                 // IN    display mode
   BOOL                search,                  // IN    search while no FNODE
   char               *path,                    // OUT   combined path string
   ULONG              *lsn                      // OUT   found dir/file FNODE
);

// Find parent FNODE for the given (DIR or FILE) FNODE
BOOL dfsHpfsFnode2Parent                        // OUT   real parent found
(
   ULONG               fn,                      // IN    starting lsn
   BOOL                verbose,                 // IN    display mode
   char               *path,                    // OUT   combined path string
   ULONG              *lsn                      // OUT   found dir/file FNODE
);

// DFS HPFS read and check type for a HPFS Fnode based on its sector number
ULONG dfsHpfsReadChkFnode                       // RET   BAD_STRUCTURE
(
   ULONG               lsn,                     // IN    fnode LSN
   BYTE               *stype,                   // OUT   Fnode type
   S_FNODE           **fnode                    // OUT   Fnode structure
);

// Update filename in-place(FNODE + DIR), with a new name of exact same length
ULONG dfsHpfsUpdateFileName
(
   ULN64               fnoLsn,                  // IN    Fnode sectornumber
   ULN64               info,                    // IN    unused
   char               *str,                     // IN    current filename
   void               *param                    // IN    new same length filename
);

// Replace sn-list by contents of a single Directory, storing Inode Lsn's
ULONG dfsHpfsMakeBrowseList
(
   ULN64               fnoLsn,                  // IN    Fnode sectornumber
   ULN64               info,                    // IN    unused
   char               *str,                     // IN    unused
   void               *param                    // INOUT unused
);

// Get allocation information for file-DATA into integrated-SPACE structure
// 20161215: Include EA-data in separate allocation (in the ispace)
ULONG dfsHpfsGetAllocSpace
(
   ULN64               fnoLsn,                  // IN    Fnode sectornumber
   ULN64               info,                    // IN    unused
   char               *str,                     // IN    unused
   void               *param                    // INOUT Integrated SPACE
);

// DFS HPFS write-fnode to disk (SaveAs fnode to file/directory)
ULONG dfsHpfsFnodeSaveAs
(
   ULN64               fn64,                    // IN    fnode LSN
   ULN64               d2,                      // IN    dummy
   char               *path,                    // IN    full destination path
   void               *ren                      // IN    new name (like 8.3)
);

// DFS HPFS verify allocation integrity for a file (normal / deleted)
ULONG dfsHpfsFnodeCheckAlloc                    // RET   BAD_STRUCTURE
(
   S_FNODE            *sd,                      // IN    fnode structure
   char               *options,                 // IN    result options
   BYTE                stype,                   // IN    sector-type
   ULONG              *totals,                  // INOUT total sectors
   ULONG              *invalids,                // INOUT invalid sectors
   ULONG              *datalsn                  // OUT   first data sector
);

// DFS HPFS verify allocation integrity for an alloc sector (normal / deleted)
ULONG dfsHpfsAllocCheckAlloc
(
   ULONG               sn,                      // IN    Alloc-sector LSN
   BYTE                st,                      // IN    type of fnode
   ULONG               RelSecFirst,             // IN    1st RelSec if a leaf
   char               *options,                 // IN    result options
   ULONG              *totals,                  // INOUT total sectors
   ULONG              *invalids,                // INOUT invalid sectors
   ULONG              *size,                    // INOUT total size handled
   ULONG              *datalsn                  // OUT   first data sector
);

// Count nr of '1' bits in a BYTE
int dfsHpfsBitCount                             // RET   nr of bits SET
(
   BYTE               ch                        // IN    one byte
);

// Display file allocation and path info for LSN
ULONG dfsHpfsFileInfo                           // RET   LSN is valid Fnode
(
   ULN64               fn64,                    // IN    Fnode LSN
   ULN64               d2,                      // IN    dummy
   char               *select,                  // IN    Fnode select wildcard
   void               *lead                     // IN    leading text
);

// Convert HPFS time_t (32 bit) to DFS standard date/time string
char *dfsHpfsTime2str                           // RET   string value
(
   ULONG               htime,                   // IN    HPFS time value
   char               *dtime                    // INOUT ptr to string buffer
);

// Determine allocation-bit for specified LSN
ULONG dfsHpfsAllocated                          // RET   LSN is allocated
(
   ULN64               sn64,                    // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);

// Test if sector is in a bitmap area mentioned in BMT (is a bitmap sector)
BOOL dfsInBitMapArea
(
   ULONG               lsn                      // IN    candidate LSN
);

// Set allocate status for LSN to specified value
ULONG dfsHpfsSetAlloc                           // RET   LSN allocation set
(
   ULN64               sn64,                    // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *value,                   // IN    NULL = not allocated
   void               *ref                      // IN    dummy (for HPFS)
);

// Copy HPFS bad-sector list to the DFSee (bad) sectorlist
ULONG dfsHpfsExportBadList                      // RET
(
   ULN64             *list,                     // IN    LSN list structure
   ULONG              size                      // IN    max nr of entries
);

// Copy DFSee (bad) sectorlist to the HPFS bad-sector list
ULONG dfsHpfsImportBadList                      // RET
(
   ULN64             *list                      // IN    LSN list structure
);

// Determine Area size of EA's for an Fnode (including deleted)
ULONG dfsHpfsSizeEaFnode
(
   ULONG               sn,                      // IN    LSN for the Fnode
   S_FNODE            *sd                       // IN    Fnode structure
);

// Add any EA data info for an Fnode to std FEA2-list structure
ULONG dfsHpfsReadEaFnode
(
   ULONG               sn,                      // IN    LSN for the Fnode
   S_FNODE            *sd,                      // IN    Fnode structure
   ULONG              *size,                    // OUT   size in bytes
   S_FEA2LIST        **pfeal                    // OUT   list of FEA2's
);

// Calculate checksums and write super- and spare-block from hpfs structure
ULONG dfsHpfsWriteSuperSpare
(
   void
);

// Iterate over directory-structures finding files, and execute callback
ULONG dfsHpfsIterator
(
   ULN64               recurse,                 // IN    recurse subtrees
   ULN64               type,                    // IN    do 'D', 'f' or both
   char               *path,                    // IN    path specification
   void               *p                        // IN    iterate parameters
);

// Create valid HPFS bootrecord based on p-table info, superblock and template
ULONG dfsHpfsMkBootRec
(
   S_SUPER            *sup,                     // IN    optional superblock
   char               *drive                    // IN    optional bootdrive
);

// Get allocation for deleted or ALL FNODE sectors into single SPACE structure
// Traverses whole volume as fast as possible, will be relatively slow ...
ULONG dfsHpfsDelFnodes2Space
(
   BOOL                delOnly,                 // IN    deleted FNODEs only
   ULONG              *nr,                      // OUT   nr of space entries
   S_SPACE           **sp                       // OUT   space allocation
);

// Get allocation for ALL referenced FNODE sectors into single SPACE structure
ULONG dfsHpfsRefFnodes2Space
(
   BOOL                progressSectors,         // IN    progress in sectors, not DIRs
   ULONG              *nr,                      // OUT   nr of space entries
   S_SPACE           **sp                       // OUT   space allocation
);

#endif
