//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// HPFS utility functions
//
// Author: J. van Wijk
//
// JvW  08-04-2021 Use rc CMD_WARNING on FileSaveAs alloc errors, considered OK
// JvW  22-11-1997 Initial version, split off from DFSAHPFS
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsver.h>                             // DFS version info
#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfstable.h>                           // SLT utility functions

#include <dfsahpfs.h>                           // HPFS analysis functions
#include <dfsuhpfs.h>                           // HPFS utility functions
#include <dfsos2ea.h>                           // HPFS/JFS/FAT EA handling


#if defined (WIN32)
   #include <dfsntreg.h>                        // NT error detail reporting
#endif

static ULONG        bm_asn = L32_NULL;          // bitmap sector cache ASN

static ULONG dfsHpfsBootSecClassic[] =          // From Warp V4 500 MiB
{
   0x4f904deb, 0x32203253, 0x00302e30, 0x00010102,  //-  MOS2 20.0
   0x00000000, 0x0000f800, 0x00000000, 0x00000000,  //-  ??
   0x00000000, 0x00280080, 0x44df5ee1, 0x31655346,  //-  ....(tuDFSe13HPFS
   0x46504833, 0x50482053, 0x20205346, 0x00002020,  //-  3HPFSHPFS    
   0x00000000, 0x00000000, 0x00000000, 0xfa000000,  //-  
   0xd08ec033, 0xfb7c00bc, 0x8e07c0b8, 0x0024a1d8,  //-  3м|ء$
   0xcd004da3, 0x004c2d12, 0xc1fff025, 0x4ba306e0,  //-  M-L%K
   0x3e06c700, 0xc7000000, 0x00004006, 0x4506c700,  //-  >@E
   0xa1001400, 0xc08e004b, 0x21e8db2b, 0x4b36ff00,  //-  K+!6K
   0x02726800, 0x26004da1, 0x26004da3, 0xa10024a2,  //-  hrM&M&$
   0xa326001c, 0x1ea1001c, 0x1ea32600, 0x5350cb00,  //-  &&PS
   0xa1065251, 0x168b003e, 0x06030040, 0x1613001c,  //-  QR>@
   0xa150001e, 0x26f6001a, 0xc88b0018, 0xa3f1f758,  //-  P&X
   0xc28b0042, 0x001836f6, 0x2688c4fe, 0x25a20044,  //-  B6Ĉ&D%
   0x0018a100, 0x0044062a, 0x45063b40, 0xa1037600,  //-  *D@;Ev
   0xb4500045, 0x42168b02, 0xd206b100, 0x44360ae6,  //-  EPB6D
   0x86ca8b00, 0x24168be9, 0x5813cd00, 0x06012a72,  //-  ʆ$Xr*
   0x1683003e, 0x29000040, 0x76004506, 0x05e0c10b,  //-  >@)Ev
   0xd003c28c, 0x8bebc28e, 0x5b595a07, 0x6ebec358,  //-  ЎZY[Xþn
   0xbe08eb08, 0x03eb0202, 0xe80158be, 0x3ebe0009,  //-  X>
   0x0003e802, 0xacfeebfb, 0x0974003c, 0x07bb0eb4,  //-  <t
   0xeb10cd00, 0x001dc3f2, 0x69642041, 0x72206b73,  //-  A disk r
   0x20646165, 0x6f727265, 0x636f2072, 0x72727563,  //-  ead error occurr
   0x0d2e6465, 0x4f07000a, 0x524b3253, 0x4f064c4e,  //-  ed.OS2KRNLO
   0x444c3253, 0x534f0752, 0x4f4f4232, 0x00000054,  //-  S2LDROS2BOOT
   0x00000000, 0x00000000, 0x00000000, 0x00000000,  //-  
   0x00000000, 0x00000000, 0x00000000, 0x00000000,  //-  
   0x00000000, 0x00000000, 0x00000000, 0x00000000,  //-  
   0x00000000, 0x00000000, 0x00000000, 0x00000000,  //-  
   0x00000000, 0x00000000, 0x00000000, 0x00000000,  //-  
   0x00000000, 0x00000000, 0x00000000, 0x00000000,  //-  
   0x00000000, 0x00000000, 0x00000000, 0xaa550000   //-  U
};

#define BT_SYST_IBM45 "IBM 4.50"                // original IBM system name

static ULONG dfsHpfsBootTemplate[] =            // From eCS 1.1 1040.9 MiB
{
   0x49904eeb, 0x34204d42, 0x0030352e, 0x00000002,  //-  lNEIBM 4.50....
   0x00020000, 0x0000f800, 0x00f0003f, 0x0000003f,  //-  ....o..?.q.?...
   0x00000000, 0x00280080, 0x44df5ee1, 0x31655346,  //-  ....(tuDFSe13HPFS
   0x46504833, 0x50482053, 0x20205346, 0x00002020,  //-  3HPFSHPFS    
   0x00000000, 0x00000000, 0x00000000, 0x00000000,  //-  ................
   0x8ec033fa, 0x7c00bcd0, 0x07c0b8fb, 0x24a1d88e,  //-  .3+A++.|v++.A+i$
   0x004da300, 0x4c2d12cd, 0xfff02500, 0xa306e0c1,  //-  .uM.=-L.%q +au
   0x0068004b, 0x64a10f30, 0x003e8166, 0x33314900,  //-  K.h.0idfu>..I13
   0xc6057558, 0x01004f06, 0x003e06c7, 0x06c70000,  //-  Xu+O.+>...+
   0x00000040, 0x004506c7, 0x4ba10014, 0x2bc08e00,  //-  @...+E..iK.A++
   0x0023e8db, 0x004b36ff, 0xa1028268, 0xa326004d,  //-  *i#. 6K.heiM.&u
   0xa226004d, 0xa1660024, 0x6626001c, 0xa0001ca3,  //-  M.&o$.fi.&fu.a
   0xa226004f, 0x50cb004f, 0x06525153, 0x8b003ea1,  //-  O.&oO.+PSQRi>.i
   0x03004016, 0x13001c06, 0x80001e16, 0x01004f3e,  //-  @...C>O.
   0xa1506c74, 0x26f6001a, 0xc88b0018, 0xa3f1f758,  //-  tlPi..:&.i+X~ru
   0xc28b0042, 0x001836f6, 0x2688c4fe, 0x25a20044,  //-  B.i+:6..-e&D.o%
   0x0018a100, 0x0044062a, 0x45063b40, 0xa1037600,  //-  .i.*D.@;E.vi
   0xb4500045, 0x42168b02, 0xd206b100, 0x44360ae6,  //-  E.P+iB.=+g.6D
   0x86ca8b00, 0x24168be9, 0x5813cd00, 0x06015d72,  //-  .i+aji$.=Xr]
   0x1683003e, 0x29000040, 0x76004506, 0x05e0c10b,  //-  >.a@..)E.v+a
   0xd003c28c, 0x84ebc28e, 0x5b595a07, 0x8b57c358,  //-  i++A+la.ZY[X+Wi
   0x8b00243e, 0x8d00450e, 0xc7027236, 0x89001004,  //-  >$.iE.i6r+.e
   0x448c045c, 0x024c8906, 0x89084489, 0xc7660a54,  //-  \iDeLeD.eT.f+
   0x00000c44, 0x42b40000, 0x13cdd78b, 0xebad725f,  //-  D.....+Bi+=_ril
   0x087ebec7, 0x02be08eb, 0xbe03eb02, 0x09e801ab,  //-  ++~.l.+l+2i.
   0x023ebe00, 0xfb0003e8, 0x3cacfeeb, 0xb4097400,  //-  .+>i.vl.4<.t.+
   0x0007bb0e, 0xf2eb10cd, 0x41001dc3, 0x73696420,  //-  +..=l>+.A dis
   0x6572206b, 0x65206461, 0x726f7272, 0x63636f20,  //-  k read error occ
   0x65727275, 0x0a0d2e64, 0x534f0700, 0x4e524b32,  //-  urred.....OS2KRN
   0x534f064c, 0x52444c32, 0x32534f07, 0x544f4f42,  //-  LOS2LDR.OS2BOOT
   0x00000000, 0x00000000, 0x00000000, 0x00000000,  //-  ................
   0x00000000, 0x00000000, 0x00000000, 0xaa550000,  //-  ..............U]
};


static S_BOOTR *brt = (S_BOOTR *) &dfsHpfsBootTemplate;

// Get all sectors administred in an allocation sector into an S_SPACE
static ULONG dfsHpfsAllocSectSpace
(
   ULONG               sn,                      // IN    Alloc-sector LSN
   ULONG               RelSecFirst,             // IN    1st RelSec if a leaf
   ULONG              *size,                    // INOUT total size handled
   DFSISPACE          *is                       // OUT   allocation SPACE
);

// Add Dirblock-sector file/dir fnodes to the sectorlist (recursive in Btree)
static ULONG dfsHpfsDirblock2List
(
   ULONG               guard,                   // IN    recursion guard counter
   ULONG               sn                       // IN    Dirblock LSN
);

// Recursively iterate over FNODES, directory and files, execute callback
static ULONG dfsHpfsIterateFnode
(
   ULONG               rs,                      // IN    recurse subtrees
   ULONG               type,                    // IN    do 'D', 'f' or both
   ULONG               fn,                      // IN    Fnode LSN
   DFS_PARAMS         *cp                       // IN    callback params
);

// Recursively iterate over DIRBLOCK, directory and files, execute callback
static ULONG dfsHpfsIterateDirBlock
(
   ULONG               rs,                      // IN    recurse subtrees
   ULONG               type,                    // IN    do 'D', 'f' or both
   ULONG               sn,                      // IN    Dirblock LSN
   DFS_PARAMS         *cp                       // IN    callback params
);

// Add SPACE for specified FNODE LSN to the accumulating total FNODE SPACE
static ULONG dfsHpfsAddFnode2Space
(
   ULN64               sn64,                    // IN    sector-nr for FNODE
   ULN64               info,                    // IN    dummy (0)
   char               *df,                      // IN    dummy, optional info
   void               *data                     // INOUT callback data, SPACE
);

/*****************************************************************************/
// Find FNODE for specified path, starting at root-directory FNODE
/*****************************************************************************/
ULONG dfsHpfsFindPath
(
   ULN64               loud,                    // IN    Show progress
   ULN64               d2,                      // IN    dummy
   char               *path,                    // IN    path specification
   void               *vp                       // OUT   found dir/file
)
{
   ULONG               rc  = NO_ERROR;
   ULONG               fs;                      // Current FNODE LSN
   ULONG               ds  = 0;                 // Current DIR LSN
   S_DIRENT           *entry;                   // directory entry
   TXLN                part;
   char               *p   = path;
   int                 l;
   DFS_PARAMS         *parm = (DFS_PARAMS *) vp;

   ENTER();

   fs = hpfs->root;                             // start LSN
   if (loud)
   {
      TxPrint("RootDir Fnode LSN : 0x%08.8X   find path: '%s'\n", fs, path);
   }
   parm->Lsn    = fs;
   parm->Number = 0;
   while ((rc == NO_ERROR) && strlen(p) && !TxAbort())
   {
      if ((l = strcspn(p, FS_PATH_STR)) != 0)
      {
         strncpy(part, p, l);                   // isolate part
         part[l] = '\0';
         p += l;                                // skip part
         if (*p == FS_PATH_SEP)
         {
            p++;                                // skip '\'
         }
      }
      if (strlen(part))
      {
         if (dfsHpfsFnode2Dirblock(fs, &ds))
         {
            if (loud)
            {
               TxPrint("Base DirBlock LSN : 0x%08.8X", ds);
            }
            if ((entry = dfsHpfsDirblockFindEntry( loud, NULL, part, 0, ds)) != NULL)
            {
               fs = entry->Fnode;
               if (loud)
               {
                  TxPrint(" - Fnode %08.8X for '%s'\n", fs, part);
               }
               if (*p == '\0')                  // end of string, found!
               {
                  parm->Lsn      = fs;
                  parm->Number   = 0;           // no additional info
                  parm->Flag     = TRUE;        // Size is from DIR-entry
                  parm->byteSize = entry->FileSize;
               }
               TxFreeMem( entry);               // free directory entry
            }
            else
            {
               if (loud)
               {
                  TxPrint(" - Search failed  for '%s'\n", part);
               }
               nav.down = ds;
               rc = ERROR_PATH_NOT_FOUND;
            }
         }
         else
         {
            rc = ERROR_PATH_NOT_FOUND;
         }
      }
      else
      {
         rc = ERROR_PATH_NOT_FOUND;
      }
   }
   RETURN (rc);
}                                               // end 'dfsHpfsFindPath'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Resolve DIRBL LSN from directory FNODE LSN
/*****************************************************************************/
BOOL dfsHpfsFnode2Dirblock                      // OUT   DIRBL resolved
(
   ULONG               fn,                      // IN    starting FNODE lsn
   ULONG              *lsn                      // OUT   found DirBlock lsn
)
{
   BOOL                rc  = FALSE;
   ULONG               fs  = fn;                 // Current FNODE LSN
   ULONG               dr  = NO_ERROR;
   BYTE               *sb;                      // sector buffer
   S_FNODE            *sd;
   BYTE                st;

   ENTER();

   if ((sb = TxAlloc(1, dfsGetSectorSize())) != NULL)
   {
      dr = dfsRead(fs, 1, sb);
      st = dfsIdentifySector(fs, 0, sb);
      if ((dr == NO_ERROR) && ((st == ST_FNDIR) || (st == ST_FNDDI)))
      {
         sd = (S_FNODE *) sb;
         if ((sd->DirFlag) && (sd->Al.Directory.Zero == 0)) // Valid Directory
         {
            *lsn = sd->Al.Directory.DirBlock;
            rc = TRUE;
         }
      }
      TxFreeMem( sb);
   }
   BRETURN (rc);
}                                               // end 'dfsHpfsFnode2Dirblock'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Find DIR-entry for Fnode, start at DIRBLOCK LSN, walk Btree; return DIR-entry
// Input is either a complete filename or a partial (15) plus the Fnode LSN
// Optional rename to SAME LENGTH new name
/*****************************************************************************/
S_DIRENT *dfsHpfsDirblockFindEntry              // RET   DIR-entry found or NULL
(                                               //       (allocated, TxFreeMem!)
   ULONG               loud,                    // IN    Show progress
   char               *rename,                  // IN    same length new name
   char               *entry,                   // IN    entry (partial) filename
   ULONG               fnode,                   // IN    Fnode LSN or 0
   ULONG               dirbl                    // IN    starting DIRBLOCK lsn
)
{
   S_DIRENT           *rc  = NULL;
   ULONG               ds  = dirbl;             // Current DIR LSN
   ULONG               dr  = NO_ERROR;
   int                 cp  = 1;                 // compare result; 1 == greater
   BYTE               *sb;                      // sector buffer
   S_DIRBL            *sd;
   S_DIRENT           *de;
   ULONG              *bdown;
   TXLN                fname;                   // upercase candidate name ASCII
   TXLN                uname;                   // upercase search-name    ASCII
   ULONG               dsWriteLsn;              // write-back dir-LSN for rename

   ENTER();

   TRACES(("fname: '%s' Fnode:%8.8x DirBlock:%8.8x\n", entry, fnode, dirbl));

   if ((sb = TxAlloc( HPFS_DIRBL_SECTS, dfsGetSectorSize())) != NULL)
   {
      TxStrToUpper( strcpy( uname, entry));
      while ((rc == NULL) && (ds != 0) && (dr == NO_ERROR))
      {
         dr = dfsRead( ds, HPFS_DIRBL_SECTS, sb);
         if (dr == NO_ERROR)
         {
            if (dfsIdentifySector( ds, 0, sb) == ST_DIRBL)
            {
               dsWriteLsn = ds;
               sd = (S_DIRBL *) sb;
               ds = 0;                          // default no next DIRBL
               for (de = &(sd->FirstEntry);
                    de->Length != 0;
                    de = (S_DIRENT *) ((char *) (de) + de->Length))
               {
                  memcpy( fname, de->FileName, de->NameLength);
                  fname[(int) de->NameLength] = '\0';
                  TxStrToUpper( fname);

                  if ((fnode != 0) && (strlen( uname) > HPFS_FNAME_LEN)) // truncated (fnode) name
                  {
                     cp = strncmp( uname, fname, HPFS_FNAME_LEN);
                  }
                  else
                  {
                     cp = strcmp( uname, fname);
                  }
                  TRACES(( "Compare result %u for '%s' - '%s'\n", cp, fname, entry));
                  if      (cp < 0)
                  {
                     if (de->Flag & 0x04)       // B-tree DOWN pointer
                     {
                        bdown = (ULONG *) ((char *) (de) + de->Length - sizeof(ULONG));
                        ds    = *bdown;
                        if (loud)
                        {
                           TxPrint("\nBtree - DirBl LSN : 0x%08.8X", ds);
                        }
                        break;
                     }
                  }
                  else if (cp == 0)             // equal, entry found
                  {
                     if (strlen(entry) == de->NameLength) // full length, incl '---'
                     {
                        if ((fnode == 0) ||     // not specified fnode, or
                            (fnode == de->Fnode)) // matching specified fnode
                        {
                           size_t  size = sizeof( S_DIRENT) + strlen( de->FileName);

                           if ((rc = TxAlloc(1, size)) != NULL)
                           {
                              memcpy( rc, de, size); // directory entry structure
                           }
                           if (rename != NULL)  // rename and write back entry
                           {
                              if (DFSTORE_WRITE_ALLOWED)
                              {
                                 strcpy( de->FileName, rename);
                                 if (dfsWrite( dsWriteLsn, HPFS_DIRBL_SECTS, sb) != NO_ERROR)
                                 {
                                    TxFreeMem( rc); // free, and set to NULL
                                 }
                              }
                           }
                           break;
                        }
                     }
                  }
                  if (de->Flag & 0x08)          // Special "END" entry
                  {
                     break;                     // out of entry-loop
                  }
               }
            }
            else
            {
               dr = DFS_ST_MISMATCH;            // break out of loop!
            }
         }
      }
      TxFreeMem( sb);
   }
   BRETURN (rc);
}                                               // end 'dfsHpfsDirblockFindEntry'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Find FNODE for Root-directory, starting at some LSN in the volume
/*****************************************************************************/
BOOL dfsHpfsFindRootFnode                       // OUT   root found
(
   ULONG               start,                   // IN    starting lsn
   BOOL                verbose,                 // IN    display mode
   BOOL                search,                  // IN    search while no FNODE
   char               *path,                    // OUT   combined path string
   ULONG              *lsn                      // OUT   found dir/file FNODE
)
{
   BOOL                rc  = FALSE;
   ULONG               frm = start;             // Current LSN startpoint
   ULONG               par = 0;                 // Parent  LSN found
   BOOL                parent;
   ULONG               sanity = 32;             // maximum iterations

   ENTER();
   if (verbose)
   {
      dfsX10("Start search  LSN : ", start, CBW, "");
      dfsProgressInit( start, hpfs->sup->TotalSec - start, 0, "Sector:", "searched", DFSP_STAT, 0);
   }
   if (path)
   {
      *path = '\0';
   }
   do
   {
      parent = dfsHpfsFnode2Parent( frm, verbose, path, &par);
      if (parent)                               // follow fnode towards root
      {
         frm = par;
         sanity--;                              // count up-references
      }
      else                                      // ROOT found, or no Fnode
      {
         if (par == 0)                          // no Fnode found
         {
            if (path && strlen( path))          // had found a file Fnode
            {
               if (search)                      // if keep searching to find
               {                                // the root, reset the path
                  *path = '\0';
               }
               if (verbose)
               {
                  TxPrint("     %s(damaged DIR Fnode)%s\n", CNC, CNN);
                  dfsX10("  restart at  LSN : ", frm, CBW, "");
               }
            }
            if (verbose)
            {
               dfsProgressShow( frm + 1, 0, 0, NULL);
            }
            frm++;
         }
      }
   } while (((parent) || ((par == 0) && (search))) && (sanity) && !TxAbort());
   if (verbose)
   {
      TXRNOLN();                                // EOLN when not tracing
   }
   if (par == frm)                              // Root is its own parent
   {
      *lsn = par;
      rc   = TRUE;
   }
   else if (sanity == 0)
   {
      TxPrint( "\nSearching from SN : %8.8x aborted after 32 iterations "
                 "Fnode links\nseem to form an endless loop.\nTry searching "
                 "again, using a different startsector.\n\n", start);
   }
   TRACES(("Found root %8.8X for Fnode %8.8X in '%s'\n", *lsn, start, path));
   if (verbose)
   {
      dfsProgressTerm();
   }
   BRETURN (rc);
}                                               // end 'dfsHpfsFindRootFnode'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Find parent FNODE for the given (DIR or FILE) FNODE
/*****************************************************************************/
BOOL dfsHpfsFnode2Parent                        // OUT   real parent found
(
   ULONG               fn,                      // IN    starting lsn
   BOOL                verbose,                 // IN    display mode
   char               *path,                    // OUT   combined path string
   ULONG              *lsn                      // OUT   found dir/file FNODE
)
{
   BOOL                rc  = FALSE;             // not a valid parent
   ULONG               dr  = NO_ERROR;
   BYTE               *sb  = NULL;              // sector buffer
   S_FNODE            *sd;
   BYTE                st;
   TX1K                fname;                   // shortname (+ path)
   ULONG               ds;                      // Dirblock LSN
   S_DIRENT           *entry = NULL;            // directory entry

   ENTER();
   *lsn = L32_NULL;                             // invalid, simulated end
   if ((sb = TxAlloc(1, dfsGetSectorSize())) != NULL)
   {
      dr = dfsRead(fn, 1, sb);
      if (dr == NO_ERROR)
      {
         st = dfsIdentifySector(fn, 0, sb);
         TRACES(( "path at start: '%s'\n", path));
         if (  (st == ST_FNDIR)  ||              // directory fnode always OK
               (st == ST_FNDDI)  ||
             (((st == ST_FNODE)  ||              // fnode only valid when last
               (st == ST_FNDEL)) &&              // in chain (leaf node), so
              (( path != NULL)   &&              // the path must be empty
               ( strlen(path) == 0))) )
         {
            sd = (S_FNODE *) sb;
            *lsn = sd->ParentDir;
            memset(fname, '-', TXMAX1K);        // create (truncated) filename
            memcpy(fname, sd->Name, min(HPFS_FNAME_LEN,(size_t) sd->NameLength));
            fname[(int) sd->NameLength] = '\0';

            if (sd->NameLength > HPFS_FNAME_LEN) // will be truncated now ...
            {
               //- try to resolve complete long-filename from directory entry, if there
               if (dfsHpfsFnode2Dirblock( sd->ParentDir, &ds))
               {
                  if ((entry = dfsHpfsDirblockFindEntry( 0, NULL, fname, fn, ds)) != NULL)
                  {
                     //- Copy complete filename from the resolved directory entry
                     memcpy( fname, entry->FileName, entry->NameLength);
                     fname[(int) entry->NameLength] = '\0';

                     TxFreeMem( entry);         // free directory entry
                  }
               }
            }

            if (sd->DirFlag)
            {
               strcat( fname, FS_PATH_STR);
            }
            if (verbose)
            {
               TxPrint("\n  found Fnode LSN : 0x%08.8X   %s : %-15s",
                        fn, (sd->DirFlag) ? "Dir." : "File", fname);
            }
            if (path)
            {
               strcat( fname, path);            // append existing path
               strcpy( path, fname);            // and copy back
               if ((TxaOption('u')) &&          // force unique naming
                   (st != ST_FNDIR)  )          // but NOT on regular DIR
               {
                  sprintf(fname, "%8.8X_", fn);
                  strcat( fname, path);         // append existing path
                  strcpy( path, fname);         // and copy back
               }
            }
            if (sd->ParentDir != fn)            // different LSN, real parent
            {
               rc = TRUE;
            }
         }
         else                                   // not a valid Fnode
         {
            if ((path) && (strlen(path)))       // unknown part of path
            {                                   // but NOT first!
               sprintf(fname, "%c%8.8X.DIR%c", FS_PATH_SEP, fn, FS_PATH_SEP);
               strcat( fname, path);            // append existing path
               strcpy( path, fname);            // and copy back
            }
            *lsn = 0;                           // signal continue-search
         }
         TRACES(( "path at end  : '%s'\n", path));
      }
      TxFreeMem(sb);
      TRACES(("started at: %8.8x, result lsn: %8.8x\n", fn, *lsn));
   }
   BRETURN (rc);
}                                               // end 'dfsHpfsFnode2Parent'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DFS HPFS read and check type for a HPFS Fnode based on its sector number
/*****************************************************************************/
ULONG dfsHpfsReadChkFnode                       // RET   BAD_STRUCTURE
(
   ULONG               lsn,                     // IN    fnode LSN
   BYTE               *stype,                   // OUT   Fnode type
   S_FNODE           **fnode                    // OUT   Fnode structure
)
{
   ULONG               rc = NO_ERROR;
   BYTE               *sb = NULL;               // sector buffer
   BYTE                st = ST_UDATA;

   ENTER();
   if ((sb = TxAlloc( 1, dfsGetSectorSize())) != NULL)
   {
      if ((rc = dfsRead( lsn, 1, sb)) == NO_ERROR)
      {
         switch (st = dfsIdentifySector( lsn, 0, sb))
         {
            case ST_FNODE:
            case ST_FNDEL:
            case ST_FNDIR:
            case ST_FNDDI:
               break;

            default:
               rc = DFS_ST_MISMATCH;
               break;
         }
      }
      if (rc != NO_ERROR)
      {
         TxFreeMem( sb);
      }
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   *fnode = (S_FNODE *) sb;
   *stype = st;

   TRACES(("lsn:%8.8x  sb:%8.8x  type:%c\n", lsn, sb, st));
   RETURN(rc);
}                                               // end 'dfsHpfsReadChkFnode'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Update filename in-place(FNODE + DIR), with a new name of exact same length
/*****************************************************************************/
ULONG dfsHpfsUpdateFileName
(
   ULN64               fnoLsn,                  // IN    Fnode sectornumber
   ULN64               info,                    // IN    unused
   char               *str,                     // IN    current filename
   void               *param                    // IN    new same length filename
)
{
   ULONG               rc = NO_ERROR;
   ULONG               dirFnoLsn;
   char               *newName = (char *) param;
   S_DIRENT           *entry;                   // directory entry
   ULONG               ds   = 0;                // DIR sector LSN
   BYTE                id   = ST_UDATA;         // type of sector
   S_FNODE            *sd   = NULL;             // Fnode structure, allocated
   TXLN                path = {0};              // unused, but needed in API

   ENTER();
   TRACES(( "fnoLsn: 0x%llx  name:'%s' newName:'%s'\n", fnoLsn, str, newName));

   if (dfsHpfsFnode2Parent( fnoLsn, FALSE, path, &dirFnoLsn))
   {
      if (dfsHpfsFnode2Dirblock( dirFnoLsn, &ds))
      {
         //- find, and in the same action also RENAME the directory-entry for this file
         if ((entry = dfsHpfsDirblockFindEntry( 0, newName, str, 0, ds)) != NULL)
         {
            //- Update the partial filename in the Fnode too
            if ((rc = dfsHpfsReadChkFnode( (ULONG) fnoLsn, &id, &sd)) == NO_ERROR)
            {
               memcpy( sd->Name, newName, min( HPFS_FNAME_LEN, strlen( newName)));
               rc = dfsWrite( fnoLsn, 1, (BYTE *) sd); // write back
               TxFreeMem( sd);                  // free Fnode
            }
            TxFreeMem( entry);                  // free directory entry
         }
         else
         {
            rc = DFS_CMD_FAILED;                // either not found (unlikely)
         }                                      // or write-back failed ...
      }
      else
      {
         rc = DFS_BAD_STRUCTURE;
      }
   }
   else
   {
      rc = DFS_ST_MISMATCH;
   }
   RETURN (rc);
}                                               // end 'dfsHpfsUpdateFileName'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Replace sn-list by contents of a single Directory, storing Inode Lsn's
/*****************************************************************************/
ULONG dfsHpfsMakeBrowseList
(
   ULN64               fnoLsn,                  // IN    Fnode sectornumber
   ULN64               info,                    // IN    unused
   char               *str,                     // IN    unused
   void               *param                    // INOUT unused
)
{
   ULONG               rc  = NO_ERROR;
   BYTE                id  = ST_UDATA;          // type of sector
   S_FNODE            *sd;

   ENTER();

   if ((rc = dfsHpfsReadChkFnode( fnoLsn, &id, &sd)) == NO_ERROR)
   {
      if (id == ST_FNDIR)                       // non-deleted DIR only
      {
         dfsInitList(0, "-f -P", "-d");         // optimal for menu file-recovery

         if (sd->ParentDir != fnoLsn)           // parent not same, not ROOT
         {
            TRACES(("List marked as 1ST_PARENT\n"));
            DFSBR_SnlistFlag |= DFSBR_1ST_PARENT;
            dfsAdd2SectorList( sd->ParentDir);
         }
         rc = dfsHpfsDirblock2List( 0, sd->Al.Directory.DirBlock);
      }
      TxFreeMem(sd);
   }
   RETURN(rc);
}                                               // end 'dfsHpfsMakeBrowseList'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add Dirblock-sector file/dir fnodes to the sectorlist (recursive in Btree)
/*****************************************************************************/
static ULONG dfsHpfsDirblock2List
(
   ULONG               guard,                   // IN    recursion guard counter
   ULONG               sn                       // IN    Dirblock LSN
)
{
   ULONG               rc  = NO_ERROR;
   BYTE                id  = ST_UDATA;          // type of sector
   BYTE               *sb  = NULL;              // sector buffer

   ENTER();
   if (guard > 100)                             // sanity recursion check
   {
      rc = DFS_BAD_STRUCTURE;
   }
   else
   {
      if ((sb = TxAlloc( HPFS_DIRBL_SECTS, dfsGetSectorSize())) != NULL)
      {
         TRACES(("LSN: %8.8X, guard:%u\n", sn, guard));
         rc = dfsRead(sn, HPFS_DIRBL_SECTS, sb);
         if (rc == NO_ERROR)
         {
            id = dfsIdentifySector(sn, 0, sb);  // double-check sector type
            if (id == ST_DIRBL)
            {
               S_DIRBL      *sd = (S_DIRBL *) sb;
               S_DIRENT     *de;
               BYTE          df;
               ULONG        *bdown;

               for ( de = &(sd->FirstEntry);
                    (de->Length != 0) && (rc == NO_ERROR);
                     de = (S_DIRENT *) ((char *) (de) + de->Length))
               {
                  df = de->Flag;
                  if (df & HPFS_DF_BDOWN)       // B-tree DOWN pointer
                  {
                     bdown = (ULONG *) ((char *) (de) + de->Length - sizeof(ULONG));
                     dfsHpfsDirblock2List( guard +1, *bdown);
                  }
                  if ((df & HPFS_DF_BGEND) == 0) // no special entry
                  {
                     BOOL          filtered = FALSE;

                     if (dfsa->browseShowHidden == FALSE) // need to check hidden/system attribute
                     {
                        if ((de->FatAttrib & FATTR_SYSTEM) ||
                            (de->FatAttrib & FATTR_HIDDEN) ||
                            (de->FileName[0] == '.'))
                        {
                           filtered = TRUE;
                        }
                     }
                     if (!filtered)
                     {
                        dfsAdd2SectorList( de->Fnode);
                     }
                  }
                  if (df & HPFS_DF_EODIR)       // Special "END" entry
                  {
                     break;                     // out of entry-loop
                  }
               }
            }
            else                                // invalid Dirblock LSN
            {
               rc = DFS_BAD_STRUCTURE;
            }
         }
         TxFreeMem(sb);
      }
   }
   RETURN(rc);
}                                               // end 'dfsHpfsDirblock2List'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get allocation information for file-DATA into integrated-SPACE structure
// 20161215: Include EA-data in separate allocation (in the ispace)
/*****************************************************************************/
ULONG dfsHpfsGetAllocSpace
(
   ULN64               fnoLsn,                  // IN    Fnode sectornumber
   ULN64               info,                    // IN    unused
   char               *str,                     // IN    unused
   void               *param                    // INOUT Integrated SPACE
)
{
   ULONG               rc  = NO_ERROR;
   DFSISPACE          *ispace = (DFSISPACE *) param;
   S_FNODE            *sd;
   ULONG               i;                       // number alloc sections
   BYTE                st;

   ENTER();
   TRARGS(("Fnode LSN: 0x%llX\n", fnoLsn));

   ispace->chunks = 0;
   ispace->space  = NULL;

   if ((rc = dfsHpfsReadChkFnode( (ULONG) fnoLsn, &st, &sd)) == NO_ERROR)
   {
      ULONG            al;
      BOOL             node;                    // fnode contains nodes
      BOOL             done = FALSE;            // stop-condition in display
      ULONG            filesize = 0;            // size to handle
      ULONG            RelSec = 0;              // Fnode starts with RelSec 0
      DFSISPACE        isNew;                   // data space to be added

      ispace->byteSize = sd->FileLength;        // make available for SaveAs

      if ((st == ST_FNODE) || (st == ST_FNDEL)) // it is a regular File
      {                                         // so get data allocation
         isNew.chunks = 0;                      // initialize to empty
         isNew.space  = NULL;

         //- Note: AlInfo Flag is not reliable, it is zapped when deleting
         node = (sd->Al.Leaf[0].RelSecData != 0);
         al   = sd->AlInfo.UsedEntries + sd->AlInfo.FreeEntries;

         for ( i=0; (i < al) && (rc == NO_ERROR) && (!done); i++)
         {
            if (node)                           // NODE allocation block
            {
               filesize = sd->FileLength;
               rc = dfsHpfsAllocSectSpace( sd->Al.Node[i].ChildBlock, RelSec, &filesize, &isNew);
               RelSec = sd->Al.Node[i].RelSecNext;
            }
            else                                // LEAF, create single extent SPACE
            {
               isNew.chunks = 1;
               isNew.space  = (S_SPACE *) TxAlloc( 1, sizeof(S_SPACE));

               if (isNew.space != NULL)
               {
                  isNew.space->size  = sd->Al.Leaf[i].LengthData;
                  isNew.space->start = sd->Al.Leaf[i].DataBlock;
               }
               else
               {
                  rc = DFS_ALLOC_ERROR;
               }
               filesize = ((sd->Al.Leaf[i].RelSecData  +
                            sd->Al.Leaf[i].LengthData) *
                            dfsGetSectorSize());
            }
            if (rc == NO_ERROR)
            {                                   // join file data stream
               rc = dfsSspaceJoin( ispace->chunks,  ispace->space,
                                   isNew.chunks,    isNew.space,
                                  &ispace->chunks, &ispace->space);
            }
            done = (filesize >= sd->FileLength);
         }
      }
      if (rc == NO_ERROR)
      {
         rc = dfsHpfsReadEaFnode( fnoLsn, sd, &ispace->xasize, (S_FEA2LIST **) &ispace->xattrs);
         if ((rc == NO_ERROR) && (ispace->xattrs != NULL))
         {
            ispace->xatype = DFSXA_FEA2;
         }
      }
      TxFreeMem( sd);
   }
   else
   {
      rc = DFS_BAD_STRUCTURE;
   }
   RETURN(rc);
}                                               // end 'dfsHpfsGetAllocSpace'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get all sectors administred in an allocation sector into an S_SPACE
/*****************************************************************************/
static ULONG dfsHpfsAllocSectSpace
(
   ULONG               sn,                      // IN    Alloc-sector LSN
   ULONG               RelSecFirst,             // IN    1st RelSec if a leaf
   ULONG              *size,                    // INOUT total size handled
   DFSISPACE          *is                       // OUT   allocation SPACE
)
{
   ULONG               rc  = NO_ERROR;
   BYTE               *sb  = NULL;              // sector buffer
   BYTE                stype;                   // type of this sector

   ENTER();
   TRARGS(("RelSecFirst: %8.8X\n", RelSecFirst));
   if ((sb = TxAlloc(1, dfsGetSectorSize())) != NULL)
   {
      TRACES(("SPACE alloc for LSN: %8.8X\n", sn));
      if ((rc = dfsRead(sn, 1, sb)) == NO_ERROR)
      {
         stype = dfsIdentifySector(sn, 0, sb);
         if ((stype == ST_ALLOC) || (stype == ST_ALDEL))
         {
            S_ALLOC      *sd = (S_ALLOC *) sb;
            ULONG         al;
            BYTE          af = sd->AlInfo.Flag;
            BOOL          node;                 // fnode contains nodes
            BOOL          done = FALSE;         // stop-condition in display
            ULONG         i;
            ULONG         filesize = *size;     // total size to handle
            ULONG         RelSec   = RelSecFirst;
            DFSISPACE     isNew;                // data space to be added

            isNew.chunks = 0;                   // initialize to empty
            isNew.space  = NULL;

            //- Note: AlInfo Flag is not reliable, it is zapped when deleting
            node = (sd->Al.Leaf[0].RelSecData != RelSecFirst) || (af & HPFS_AL_NODE);
            al   = sd->AlInfo.UsedEntries + sd->AlInfo.FreeEntries;

            for ( i=0; (i < al) && (rc == NO_ERROR) && (!done); i++)
            {
               if (node)                        // NODE allocation block
               {
                  rc = dfsHpfsAllocSectSpace( sd->Al.Node[i].ChildBlock, RelSec, size, &isNew);
                  RelSec = sd->Al.Node[i].RelSecNext;
               }
               else                             // LEAF allocation block
               {
                  isNew.chunks = 1;
                  isNew.space  = (S_SPACE *) TxAlloc( 1, sizeof(S_SPACE));

                  if (isNew.space != NULL)
                  {
                     isNew.space->size  = sd->Al.Leaf[i].LengthData;
                     isNew.space->start = sd->Al.Leaf[i].DataBlock;
                  }
                  else
                  {
                     rc = DFS_ALLOC_ERROR;
                  }
                  *size = ((sd->Al.Leaf[i].RelSecData  +
                            sd->Al.Leaf[i].LengthData) *
                            dfsGetSectorSize());
               }
               if (rc == NO_ERROR)
               {                                // join file data stream
                  rc = dfsSspaceJoin( is->chunks,   is->space,
                                      isNew.chunks, isNew.space,
                                     &is->chunks,  &is->space);
               }
               done = (*size >= filesize);
            }
         }
         else
         {
            rc = DFS_BAD_STRUCTURE;
         }
      }
      TxFreeMem(sb);
   }
   RETURN(rc);
}                                               // end 'dfsHpfsAllocSectSpace'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DFS HPFS write-fnode to disk (SaveAs fnode to file/directory)
// 20150318: Recover original timestamps/name from DIR for non-deleted files
// 20161201: Add progress indicator in status line for files over 1 Mb
// 20190617: Use HpfsGetAllocSpace and SspaceFileSaveAs, no dedicated code
/*****************************************************************************/
ULONG dfsHpfsFnodeSaveAs
(
   ULN64               fn64,                    // IN    fnode LSN, 64bit
   ULN64               d2,                      // IN    dummy
   char               *fpath,                   // IN    full destination path
   void               *recp                     // INOUT recovery parameters
)
{
   ULONG               rc  = NO_ERROR;
   S_RECOVER_PARAM    *param = (S_RECOVER_PARAM *) recp;
   S_FNODE            *sd = NULL;               // Fnode structure
   DFSISPACE           isp;                     // allocation SPACE info
   TXLN                name;                    // file to create
   TXLN                safname;                 // resulting SaveAs name
   ULONG               fn = (ULONG) fn64;       // Fnode 32bit Lsn
   BYTE                st;                      // sector type
   TXLN                fname;                   // resolved filename
   ULONG               ds;                      // Dirblock LSN
   S_DIRENT           *entry = NULL;            // directory entry

   ENTER();
   TRARGS(("Fnode LSN: %8.8X, to path: '%s'\n", fn, fpath));

   if ((rc = dfsHpfsReadChkFnode( fn, &st, &sd)) == NO_ERROR)
   {
      memset( &isp, 0, sizeof( DFSISPACE));

      if ((rc = dfsHpfsGetAllocSpace( fn64, 0, NULL, &isp)) == NO_ERROR)
      {
         if ((param->newname == NULL) || (*param->newname == 0)) // no newname present
         {
            //- First, create full-length 'stuffed' filename from Fnode
            memset(fname, '-', TXMAXLN);
            memcpy(fname, sd->Name, min(HPFS_FNAME_LEN,(size_t) sd->NameLength));
            fname[(int) sd->NameLength] = '\0';

            if ((st == ST_FNDIR) || (st == ST_FNODE)) // regular file/dir, not deleted
            {
               if (dfsHpfsFnode2Dirblock( sd->ParentDir, &ds))
               {
                  if ((entry = dfsHpfsDirblockFindEntry( 0, NULL, fname, fn, ds)) != NULL)
                  {
                     //- Copy complete filename from the resolved directory entry
                     memcpy( fname, entry->FileName, entry->NameLength);
                     fname[(int) entry->NameLength] = '\0';
                  }
               }
            }
         }
         else                                   // rename specified
         {
            strcpy( fname, param->newname);
         }
         strcpy( name, fname);

         if (param->unique)                     // force unique naming on FILE name
         {
            sprintf(fname, "%8.8X_%s", fn, name);
            strcpy( name, fname);               // copy back
         }

         if ((strlen(fpath) + strlen(name) +1) < dfsa->maxPath) // length OK now
         {
            rc = dfsSspaceFileSaveAs( &isp, ((st == ST_FNDIR) || (st == ST_FNDDI)), // isDirectory
                                            ((st == ST_FNDEL) || (st == ST_FNDDI)), // isDeleted
                                             param->noAllocCheck, param->name83, fpath, name, safname);
            if ((rc == NO_ERROR)       ||       // set original timestamps
                (rc == DFS_CMD_WARNING) )       // even when allocation errors present
            {
               if (entry != NULL)               // original DIR entry present
               {
                  time_t      cre = entry->CreationTime;
                  time_t      acc = entry->LastAccessTime;
                  time_t      mod = entry->LastModDate;

                  TRACES(("HPFS recover timestamps for '%s'\n", safname));
                  TxSetFileTime( safname, &cre, &acc, &mod);
               }
               if (param->recFname)
               {
                  strcpy( param->recFname, safname);
               }
            }
         }
         else
         {
            rc = ERROR_FILENAME_EXCED_RANGE;
         }
      }
      TxFreeMem( entry);                        // free directory entry
      TxFreeMem( sd);
   }
   else
   {
      TxPrint("SaveTo only implemented on Fnode sectors\n");
   }
   RETURN(rc);
}                                               // end 'dfsHpfsFnodeSaveAs'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DFS HPFS verify allocation integrity for a file (normal / deleted)
/*****************************************************************************/
ULONG dfsHpfsFnodeCheckAlloc                    // RET   BAD_STRUCTURE
(
   S_FNODE            *sd,                      // IN    fnode structure
   char               *options,                 // IN    result options
   BYTE                stype,                   // IN    sector-type
   ULONG              *totals,                  // INOUT total sectors
   ULONG              *invalids,                // INOUT invalid sectors
   ULONG              *datalsn                  // OUT   first data sector
)
{
   ULONG               rc  = NO_ERROR;
   ULONG               i;                       // number alloc sections
   ULONG               j;                       // number of sectors
   ULONG               sn;                      // Data LSN
   BOOL                verbose = (strchr(options, 'v') != NULL);
   BOOL                skipchk = (strchr(options, 's') != NULL);
   USHORT              col = 0;

   ENTER();
   TRACES(("sd:%8.8x stype:%c\n", sd, stype));

   if ((stype == ST_FNODE) || (stype == ST_FNDEL))
   {
      ULONG      al;
      BOOL       node;                          // fnode contains nodes
      BOOL       done = FALSE;                  // stop-condition in display
      BOOL       alok = FALSE;                  // allocation OK
      ULONG      filesize;
      ULONG      RelSec = 0;                    // Fnode starts with RelSec 0

      //- Note: AlInfo Flag is not reliable, it is zapped when deleting
      node = (sd->Al.Leaf[0].RelSecData != 0);
      al   = sd->AlInfo.UsedEntries + sd->AlInfo.FreeEntries;

      for ( i=0;
           (i < al) && (rc == NO_ERROR) && (!done);
            i++)
      {
         if (node)                              // NODE allocation block
         {
            filesize = sd->FileLength;
            rc = dfsHpfsAllocCheckAlloc( sd->Al.Node[i].ChildBlock,
                                         stype, RelSec, options,
                                         totals, invalids,
                                         &filesize, datalsn);
            RelSec = sd->Al.Node[i].RelSecNext;
         }
         else                                   // LEAF allocation block
         {
            if (i == 0)
            {
               *datalsn = sd->Al.Leaf[0].DataBlock; // first data LSN
            }
            if (verbose)
            {
               if (++col >= dfsGetDisplayMargin())
               {
                  TxPrint("\n");
                  col = 1;
               }
               TxPrint("");                    // start of allocation extent
            }
            if (skipchk)
            {
               *totals += sd->Al.Leaf[i].LengthData;  //- just report size, ignore allocation
            }                                         //- (will be reported as 100% OK!)
            else                                      //- full alloc check
            {
               for ( j=0, sn = sd->Al.Leaf[i].DataBlock;
                    (j < sd->Al.Leaf[i].LengthData) && (rc == NO_ERROR);
                     j++, sn++)
               {
                  switch (dfsHpfsAllocated( sn, 0, NULL, NULL))
                  {
                     case FALSE: alok = (stype == ST_FNDEL);          break;
                     case TRUE:  alok = (stype == ST_FNODE);          break;
                     default:    alok = FALSE; rc = DFS_PSN_LIMIT; break;
                  }
                  if (verbose)
                  {
                     if (++col >= dfsGetDisplayMargin())
                     {
                        TxPrint("\n");
                        col = 1;
                     }
                     TxPrint("%s%s%s", alok ? CBG : CBR,
                                       alok ? "" : "",
                                       CNN);
                  }
                  if (!alok)
                  {
                     (*invalids)++;
                  }
                  (*totals)++;
               }
            }
            filesize = ((sd->Al.Leaf[i].RelSecData  +
                         sd->Al.Leaf[i].LengthData) *
                         dfsGetSectorSize());
         }
         done = (filesize >= sd->FileLength);
      }
      if (rc == NO_ERROR)
      {
         if (*invalids)
         {
            rc = DFS_BAD_STRUCTURE;
         }
      }
      else                                      // structure / PSN-limit error
      {
         *invalids = (ULONG) ((sd->FileLength +dfsGetSectorSize()-1) /
                                               dfsGetSectorSize());
         *totals   = *invalids;
      }
      if (verbose)
      {
         TxPrint("\n");
         switch (rc)
         {
            case NO_ERROR:
               TxPrint("No allocation errors detected\n");
               break;

            case DFS_PSN_LIMIT:
               TxPrint("Sector numbers are too large for this HPFS volume!\n");
               break;

            default:
               TxPrint("Allocation error detected on %u out of %u sectors, "
                       "save/undelete unreliable!\n", *invalids, *totals);
               break;
         }
      }
   }
   else                                         // must be a directory Fnode
   {
      *datalsn  = sd->Al.Directory.DirBlock;    // first DIRBLOCK LSN
      *totals   = HPFS_DIRBL_SECTS;
      *invalids = 0;                            // to be refined (check alloc)
   }
   RETURN(rc);
}                                               // end 'dfsHpfsFnodeCheckAlloc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DFS HPFS verify allocation integrity for an alloc sector (normal / deleted)
/*****************************************************************************/
ULONG dfsHpfsAllocCheckAlloc
(
   ULONG               sn,                      // IN    Alloc-sector LSN
   BYTE                st,                      // IN    type of fnode
   ULONG               RelSecFirst,             // IN    1st RelSec if a leaf
   char               *options,                 // IN    result options
   ULONG              *totals,                  // INOUT total sectors
   ULONG              *invalids,                // INOUT invalid sectors
   ULONG              *size,                    // INOUT total size handled
   ULONG              *datalsn                  // OUT   first data sector
)
{
   ULONG               rc  = NO_ERROR;
   BYTE               *sb  = NULL;              // sector buffer
   BYTE                stype;                   // type of this sector
   BOOL                verbose = (strchr(options, 'v') != NULL);
   BOOL                skipchk = (strchr(options, 's') != NULL);

   ENTER();
   TRARGS(("RelSecFirst: %8.8X  totals: %X  invs: %X\n",
            RelSecFirst, *totals, *invalids));
   if ((sb = TxAlloc(1, dfsGetSectorSize())) != NULL)
   {
      rc = dfsRead(sn, 1, sb);
      if (rc == NO_ERROR)
      {
         if (verbose)
         {
            TxPrint("%s%s%s", CBC, CNN, TREOLN); // start of Alloc sector
         }
         stype = dfsIdentifySector(sn, 0, sb);
         if ((stype == ST_ALLOC) || (stype == ST_ALDEL))
         {
            S_ALLOC      *sd = (S_ALLOC *) sb;
            ULONG         al;
            BYTE          af = sd->AlInfo.Flag;
            BOOL          node;                 // fnode contains nodes
            BOOL          done = FALSE;         // stop-condition in display
            BOOL          alok = FALSE;         // allocation OK
            ULONG         i;
            ULONG         j;                    // number of sectors
            ULONG         sect;                 // Data LSN
            ULONG         filesize = *size;     // total size to write
            ULONG         RelSec = RelSecFirst;

            //- Note: AlInfo Flag is not reliable, it is zapped when deleting
            node = (sd->Al.Leaf[0].RelSecData != RelSecFirst) || (af & HPFS_AL_NODE);
            al   =  sd->AlInfo.UsedEntries + sd->AlInfo.FreeEntries;
            for (i=0; (i < al) && (!done); i++)
            {
               if (node)                        // NODE allocation block
               {
                  rc = dfsHpfsAllocCheckAlloc(sd->Al.Node[i].ChildBlock,
                                              st, RelSec, options,
                                              totals, invalids,
                                              size, datalsn);
                  RelSec = sd->Al.Node[i].RelSecNext;
               }
               else                             // LEAF allocation block
               {
                  if (verbose)
                  {
                     TxPrint("");              // start of allocation extent
                  }
                  if (RelSec == 0)              // very first data block
                  {
                     *datalsn = sd->Al.Leaf[0].DataBlock; // first data LSN
                  }
                  if (skipchk)
                  {
                     *totals += sd->Al.Leaf[i].LengthData;  //- just report size, ignore allocation
                  }                                         //- (will be reported as 100% OK!)
                  else                                      //- full alloc check
                  {
                     for ( j=0, sect = sd->Al.Leaf[i].DataBlock;
                          (j <  sd->Al.Leaf[i].LengthData) && (rc == NO_ERROR);
                           j++, sect++)
                     {
                        switch (dfsHpfsAllocated( sect, 0, NULL, NULL))
                        {
                           case FALSE: alok = (st == ST_FNDEL);          break;
                           case TRUE:  alok = (st == ST_FNODE);          break;
                           default:    alok = FALSE; rc = DFS_PSN_LIMIT; break;
                        }
                        if (verbose)
                        {
                           TxPrint("%s%s%s", alok ? CBG : CBR,
                                             alok ? "" : "",
                                             CNN);
                        }
                        if (!alok)
                        {
                           (*invalids)++;
                        }
                        (*totals)++;
                     }
                  }
                  *size = ((sd->Al.Leaf[i].RelSecData  +
                            sd->Al.Leaf[i].LengthData) *
                            dfsGetSectorSize());
               }
               done = (*size >= filesize);
            }
            TXREOLN();
         }
         else
         {
            (*invalids)++;                      // report back alloc sector
            (*totals)++;                        // with simulated size of 1
            rc = DFS_BAD_STRUCTURE;
         }
      }
      else
      {
         rc = DFS_PSN_LIMIT;                    // un-readable LSN
      }
      TxFreeMem(sb);
   }
   if (verbose && (rc != NO_ERROR))
   {
      TxPrint("\nCheck on allocation-sector %8.8X not possible: ", sn);
      switch (rc)
      {
         case DFS_PSN_LIMIT:       TxPrint("read error\n");           break;
         case DFS_BAD_STRUCTURE:   TxPrint("contents damaged\n");     break;
      }
   }
   RETURN(rc);
}                                               // end 'dfsHpfsAllocCheckAlloc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Count nr of '1' bits in a BYTE
/*****************************************************************************/
int dfsHpfsBitCount                             // RET   nr of bits SET
(
   BYTE               ch                        // IN    one byte
)
{
   int                 rc   = 0;
   BYTE                byte = ch;

   do
   {
      rc += (byte & 1);
   } while (byte >>= 1);
   return (rc);
}                                               // end 'dfsHpfsBitCount'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display file allocation and path info for LSN
/*****************************************************************************/
ULONG dfsHpfsFileInfo                           // RET   LSN is valid Fnode
(
   ULN64               fn64,                    // IN    Fnode LSN
   ULN64               d2,                      // IN    dummy
   char               *select,                  // IN    Fnode select wildcard
   void               *param                    // INOUT leading text/shortpath
)
{
   ULONG               rc = NO_ERROR;
   ULONG               lsn = (ULONG) fn64;
   TXLN                shortpath;
   ULONG               root;
   ULONG               size;
   ULONG               bads;
   BYTE                st = 0;
   TXLN                text;
   TXLN                temp;
   ULONG               percent   = 0;
   ULONG               threshold = 0;           // threshold percentage
   BOOL                isMinimum = TRUE;        // threshold is minimum value
   ULONG               location  = 0;
   ULONG               minS      = 0;           // minimal size
   ULONG               maxS      = 0;           // maximal size
   char                itemType  = 0;           // type Dir, File, Browse, or ANY
   time_t              modTstamp = 0;           // Modify timestamp value, or 0
   char               *lead    = param;
   BOOL                verbose = (strcmp( lead, ":") != 0);
   BOOL                output;                  // no Printf, just return fpath
   BOOL                alcheck;                 // Execute a full allocation check
   S_FNODE            *sd;                      // Fnode for this file
   ULONG               ds;                      // directoryblock LSN
   S_DIRENT           *entry = NULL;            // directory entry, for Mod-time
   TXLN                fname;

   ENTER();
   TRARGS(("LSN: %8.8X, select: '%s', lead: '%s'\n", lsn, select, lead));

   if (strcmp( lead, "!") == 0)                 // Skip allocation check wanted
   {
      output  = FALSE;                          // no output
      alcheck = FALSE;                          // and no allocation check
      *lead   = 0;                              // discard 'garbage' in lead string
   }
   else                                         // Possible output/alloc-check wanted
   {
      output  = (*lead != 0);                   // output if any lead given (except '!')
      alcheck = TRUE;                           // but always an alloc check if no  '!'
   }

   if ((rc = dfsHpfsReadChkFnode( lsn, &st, &sd)) == NO_ERROR)
   {
      bads = size = 0;
      if (verbose)
      {
         //- Allow skipping the allocation-check when not needed (Speedup BROWSE display)
         rc = dfsHpfsFnodeCheckAlloc( sd, (alcheck) ? "" : "s", st, &size, &bads, &location);
      }
      dfsParseFileSelection( select, text, &isMinimum, &threshold, &minS, &maxS, &modTstamp, &itemType);
      switch (st)
      {
         case ST_FNDIR:                         // normal or deleted dir
         case ST_FNDDI:
            if      (itemType == DFS_FS_ITEM_BROWSE)  // no filtering on Directories, take all.
            {
               strcpy( text, "");               // no wildcard
               isMinimum = TRUE; threshold = 0; // no percentage filter
               modTstamp = 0;                   // no timestamp filter
               minS      = 0;    maxS      = 0; // no size filter
            }
            else if (itemType == DFS_FS_ITEM_FILES) // filter files, discard Directories
            {
               rc = DFS_ST_MISMATCH;            // item-type mismatch
               break;
            }
         case ST_FNODE:                         // normal or deleted file
         case ST_FNDEL:
            if      (itemType == DFS_FS_ITEM_DIRS) // filter dirs, discard files
            {
               rc = DFS_ST_MISMATCH;            // item-type mismatch
               break;
            }
            dfsHpfsFindRootFnode( lsn, FALSE, FALSE, shortpath, &root);
            if ((strlen(text) == 0) || (TxStrWicmp(shortpath, text) >= 0))
            {
               if (verbose)
               {
                  percent = dfsAllocationReliability( size, bads);

                  TRACES(( "percent:%u threshold:%u size:%u minS:%u maxS:%u\n",
                            percent, threshold, size, minS, maxS));
               }
               if ((BOOL)((percent >= threshold)) == isMinimum)
               {
                  //- first resolve DIR-entry in parent, to get to the Modify-timestamp
                  if (dfsHpfsFnode2Dirblock( sd->ParentDir, &ds))
                  {
                     if (sd->ParentDir != lsn)  // it is not the root
                     {
                        memset(fname, '-', TXMAXLN); // create (truncated) filename
                        memcpy(fname, sd->Name, min(HPFS_FNAME_LEN,(size_t) sd->NameLength));
                        fname[(int) sd->NameLength] = '\0';
                     }
                     else                       // ROOT, get 'begin' DIR entry for timestamp
                     {
                        fname[0] = 0x01;        // name for the special BEGIN entry
                        fname[1] = 0x01;
                        fname[2] = '\0';
                     }
                     TRACES(("Length: %u  fname: '%s'\n", sd->NameLength, fname));

                     entry = dfsHpfsDirblockFindEntry( 0, NULL, fname, lsn, ds);
                  }
                  if (((entry == NULL) || (entry->LastModDate > modTstamp)) &&  //- Timestamp match
                      ((size >= minS) && ((size <= maxS) || (maxS == 0))))      //- Size match
                  {
                     if (verbose)               // include alloc and size
                     {
                        if (alcheck)
                        {
                           sprintf( text, "%s%c %s%3u%%%s   %8X ", CBM, st, (rc == NO_ERROR) ?  CBG  :  CBR, percent, CNN, location);
                        }
                        else                    // no reliability percentage
                        {
                           sprintf( text, "%s%c     %s   %8X ", CBM, st, CNN, location);
                        }
                        dfstrSize( text, CBC, size, " ");
                        dfstrBytes(text, CBZ, dfsHpfsSizeEaFnode( lsn, sd), " "); // EA-size, bytes
                     }
                     else
                     {
                        strcpy( text, "");      // remove any stray text, non-verbose
                     }
                     if (output)                // not totally silent?
                     {
                        strcat( lead, text);
                        TxPrint( "%s%s%s%s%s\n", lead, CBY, shortpath, CNN, CGE);
                     }
                     else                       // return info in 'select'
                     {
                        strcpy( temp, "-no-date- -no-time-");
                        if (entry != NULL)
                        {
                           dfsHpfsTime2str( entry->LastModDate, temp);
                        }
                        TxStripAnsiCodes( text);
                        sprintf( select, "%s %s",   text, temp);
                        dfstrUllDot20( select, "", sd->FileLength, "");
                     }
                     rc = NO_ERROR;
                  }
                  else
                  {
                     rc = DFS_ST_MISMATCH;      // file-size mismatch
                  }
                  TxFreeMem( entry);            // free directory entry
               }
               else
               {
                  rc = DFS_ST_MISMATCH;         // percentage mismatch
               }
            }
            else
            {
               rc = DFS_ST_MISMATCH;            // wildcard mismatch
            }
            strcpy(  param, shortpath);         // return the found name
            break;

         default:
            rc = DFS_PENDING;
            break;
      }
      TxFreeMem( sd);
   }
   RETURN(rc);
}                                               // end 'dfsHpfsFileInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Convert HPFS time_t (32 bit) to DFS standard date/time string
/*****************************************************************************/
char *dfsHpfsTime2str                           // RET   string value
(
   ULONG               htime,                   // IN    HPFS time value
   char               *dtime                    // INOUT ptr to string buffer
)
{
   time_t              tm = (time_t) htime;     // time in compiler format
   struct tm          *gm;

   ENTER();

   if ((htime != 0) && ((gm = gmtime( &tm)) != NULL))
   {
      //- Note: time_t officially is a SIGNED value (1901 .. 2038, 0 = 1970)
      strftime(dtime, TXMAXTM, "%Y-%m-%d %H:%M:%S", gmtime( &tm));
   }
   else                                         // invalid, out of range TM
   {
      sprintf( dtime, "-%8.8x-%8.8x-", htime, 0); // no high precision there
   }
   TRACES(("Formatted date/time: %s\n", dtime));

   RETURN( dtime);
}                                               // end 'dfsHpfsTime2str'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Determine allocation-bit for specified LSN
// Note: Area aware to allow usage from FDISK mode too
/*****************************************************************************/
ULONG dfsHpfsAllocated                          // RET   LSN is allocated
(
   ULN64               sn64,                    // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               al = NO_ERROR;           // not allocated
   ULONG               lsn = (ULONG) sn64;
   ULONG               bm;
   BYTE                alloc_byte;
   ULONG               asn = dfstAreaD2Part( DFSTORE, lsn);

   if (asn < hpfs->sup->TotalSec)               // within volume boundary ?
   {
      if (hpfs->bmt != NULL)
      {
         bm = hpfs->bmt[asn >> 14];
         if (bm != bm_asn)                      // Not cached bitmap ?
         {
            TRACES(("HpfsAllocated: cache bitmap at %8.8X for LSN %8.8X\n", bm, lsn));
            if (dfsRead( dfstAreaP2Disk( DFSTORE, bm), HPFS_BTMAP_SECTS, (BYTE   *) hpfs->bmc) != NO_ERROR)
            {
               TRACES(( "BITMAP read failure for ASN %8.8X, assume all ALLOCATED!\n", bm));
               memset( hpfs->bmc, 0, (HPFS_BTMAP_SECTS * SECTORSIZE)); // fake as ALLOCATED
            }
            bm_asn = bm;                        // ALWAYS cache!
         }
         alloc_byte = hpfs->bmc[(asn & 0x3fff) >> 3];
         if (((alloc_byte >> (asn & 0x07)) & 0x01) == 0)
         {
            al = TRUE;
         }
      }
   }
   else
   {
      al = DFS_PSN_LIMIT;
   }
   return (al);
}                                               // end 'dfsHpfsAllocated'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Test if sector is in a bitmap area mentioned in BMT (is a bitmap sector)
/*****************************************************************************/
BOOL dfsInBitMapArea
(
   ULONG               lsn                      // IN    candidate LSN
)
{
   BOOL                rc = FALSE;              // function return
   ULONG               bm;                      // bitmap LSN

   if (lsn < hpfs->sup->TotalSec)               // within volume boundary ?
   {
      if (hpfs->bmt != NULL)
      {
         bm = hpfs->bmt[lsn >> 14];             // start of 4 sector bitmap

         if ((lsn >= bm) && (lsn < bm + HPFS_BTMAP_SECTS)) // in the bitmap itself ?
         {
            rc = TRUE;
         }
      }
   }
   return (rc);
}                                               // end 'dfsInBitMapArea'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set allocate status for LSN to specified value
/*****************************************************************************/
ULONG dfsHpfsSetAlloc                           // RET   LSN allocation set
(
   ULN64               sn64,                    // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *value,                   // IN    NULL = not allocated
   void               *ref                      // IN    dummy (for HPFS)
)
{
   ULONG               rc = NO_ERROR;           // not allocated
   ULONG               lsn = (ULONG) sn64;
   ULONG               bm;
   BYTE                byte_mask;

   ENTER();

   //- to be refined, use the cache for writing too, by adding a Dirty
   //- flag to the cache structure and a Flush function (like NTFS)

   if (lsn < hpfs->sup->TotalSec)               // within volume boundary ?
   {
      if (hpfs->bmt != NULL)
      {
         if (DFSTORE_WRITE_ALLOWED)
         {
            bm = hpfs->bmt[lsn >> 14];
            if (bm != bm_asn)                   // Not cached bitmap ?
            {
               TRACES(("Cache-read bitmaps for LSN %8.8X\n", lsn));
               if (dfsRead(bm, HPFS_BTMAP_SECTS, (BYTE   *) hpfs->bmc) == NO_ERROR)
               {
                  bm_asn = bm;
               }
            }
            byte_mask = (BYTE) (0x01 << (lsn & 0x07));
            if (value)                          // set allocated => bit 0
            {
               hpfs->bmc[(lsn & 0x3fff) >> 3] &= ~byte_mask;
            }
            else                                // set to free => bit 1
            {
               hpfs->bmc[(lsn & 0x3fff) >> 3] |= byte_mask;
            }
            TRACES(("Write for set alloc bit for LSN %8.8X to %s\n",
                    lsn, (value) ? "allocated" : "free"));
            rc = dfsWrite(bm, HPFS_BTMAP_SECTS, (BYTE   *) hpfs->bmc);
         }
         else
         {
            rc = DFS_READ_ONLY;
         }
      }
   }
   else
   {
      rc = DFS_PSN_LIMIT;
   }
   RETURN(rc);
}                                               // end 'dfsHpfsSetAlloc'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// Format HPFS directory entry flag attributes, append to string
/*****************************************************************************/
void dfstrHpfsAttrib
(
   char               *str,                     // INOUT resulting string
   BYTE                data                     // IN    data
)
{
   TXTS                flagbit;

   strcpy( flagbit, "     ");
   if (data & 0x10) flagbit[0] = 'E';
   if (data & 0x80) flagbit[1] = 'N';
   if (data & 0x02) flagbit[2] = 'A';
   if (data & 0x20) flagbit[3] = 'X';
   if (data & 0x40) flagbit[4] = 'S';
   strcat( str, flagbit);
}                                               // end 'dfstrHpfsAttrib'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Copy HPFS bad-sector list to the DFSee (bad) sectorlist
/*****************************************************************************/
ULONG dfsHpfsExportBadList                      // RET
(
   ULN64             *list,                     // IN    LSN list structure
   ULONG              size                      // IN    max nr of entries
)
{
   ULONG               rc = 0;                  // rc, sector match
   BYTE               *sb = NULL;               // sector buffer
   ULONG              *bs;
   ULONG               sn;
   ULONG               done;
   ULONG               todo;
   ULONG               bads;
   ULONG               NextSect = hpfs->sup->BadList;

   ENTER();

   if (hpfs->sup->BadSec > size)                // bad-sector space in FS
   {
      TxPrint("\nWarning, only %u bad-sectors can be registred in DFSee!\n", size);
   }
   if ((sb = TxAlloc(4, dfsGetSectorSize())) != NULL)
   {
      bs    = (ULONG *) sb;
      bads  = min( size, hpfs->sup->BadSec);
      done  = 0;
      do
      {
         rc = dfsRead(NextSect, 4, sb);         // linked-list of BS-sects
         if (rc == NO_ERROR)
         {
            todo = min( HPFS_BADSECS, bads - done);
            for (sn = 1; (sn <= todo) && (!TxAbort()); sn++)
            {
               list[++done] = bs[sn];
            }
            NextSect = bs[0];                   // LSN ptr to next sector
         }
      } while ((rc == NO_ERROR) &&
               (NextSect != 0)  &&
               (done < bads ));
      TxPrint("\nFinished copying %u LSN's to DFS list\n", bads);
      *list = bads;
      TxFreeMem(sb);
   }
   RETURN (rc);
}                                               // end 'dfsHpfsExportBadList'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Copy DFSee (bad) sectorlist to the HPFS bad-sector list
/*****************************************************************************/
ULONG dfsHpfsImportBadList                      // RET
(
   ULN64             *list                      // IN    LSN list structure
)
{
   ULONG               rc = 0;                  // rc, sector match
   BYTE               *sb = NULL;               // sector buffer
   ULONG              *bs;
   ULONG               sn;
   ULONG               nr;
   ULONG               todo;                    // nr of badsecs to to
   ULONG               bads;                    // index of badsec in list
   ULONG               extents;                 // nr of allocated extents
   S_SPACE             chunk;                   // allocation for chunk
   ULONG               NextSect;                // start LSN for next chunk

   ENTER();

   if ((sb = TxAlloc(4, dfsGetSectorSize())) != NULL)
   {
      bs   = (ULONG *) sb;                      // sectors as LSN array
      NextSect = hpfs->sup->BadList;            // first badsect chunk
      do                                        // free current bad-list
      {                                         // except the first one
         rc = dfsRead(NextSect, 4, sb);         // linked-list of BS-sects
         if (rc == NO_ERROR)
         {
            //- Mark the sectors currently listed as 'FREE' again
            for (sn = 1; sn <= HPFS_BADSECS; sn++)
            {
               if (bs[sn] != 0)                 // entry is in use
               {
                  DFSFNCALL(dfsa->FsLsnSetAlloc, bs[sn], 0, NULL, NULL);
               }
            }

            //- Free extra sectors (not 1st four) for large bad-sector lists
            if ((NextSect = bs[0]) != 0)        // LSN ptr to next sector
            {
               for (nr = 0; nr < 4; nr++)
               {
                  DFSFNCALL(dfsa->FsLsnSetAlloc,  NextSect + nr, 0, NULL, NULL);
               }
            }
         }
      } while ((rc == NO_ERROR) && (NextSect != 0));

      //- set allocation for the bad sectors in the new supplied list
      for (nr = 1; nr <= *list; nr++)           // set each bad-sec allocated
      {
         DFSFNCALL(dfsa->FsLsnSetAlloc, list[nr], 0, "set", NULL);
      }

      bads = 1;                                 // first index to read
      todo = *list;                             // nr of bad-sectors in list
      NextSect = hpfs->sup->BadList;            // first badsect chunk
      do
      {
         nr = min( HPFS_BADSECS, todo);         // bad-sects in this chunk
         for (sn = 1; sn <= HPFS_BADSECS; sn++)
         {
            if (sn <= nr)                       // still part of the list
            {
               bs[sn] = (ULONG) list[bads++];
            }
            else                                // fill up with zero LSN's
            {
               bs[sn] = 0;
            }
         }
         todo -= nr;                            // rest of bad-sects todo
         if (todo > 0)                          // allocate new disk-space
         {
            if (dfsSlTableSearchFree(4, 4, 1, &extents, &chunk))
            {
               dfsSlTableAllocate(   4, 4, extents, ST_BADSL, LSN_SUPER, 0, &chunk);
               bs[0] = chunk.start;
            }
            else                                // no room to extend the list
            {
               TxPrint("Insufficient free diskspace for bad-sector list.\n");
               bs[0] = 0;                       // No next chunk available
            }
         }
         else                                   // nothing left todo
         {
            bs[0] = 0;                          // No next chunk needed
         }
         if ((rc = dfsWrite(NextSect, 4, sb)) == NO_ERROR) // write this chunk
         {
            dfsX10( "Write table at LSN: ", NextSect,  CBC, "\n");
         }
         else
         {
            TxPrint("Write bad-sector list failed, disk not locked ?\n");
         }
         NextSect = bs[0];
      } while ((rc == NO_ERROR) && (NextSect != 0));
      if (rc == NO_ERROR)
      {
         hpfs->sup->BadSec = *list - todo;
         if (dfsWrite(LSN_SUPER, 1, (BYTE   *) hpfs->sup) == NO_ERROR)
         {
            if (hpfs->sup->BadSec != 0)
            {
               TxPrint("\nFinished putting %u LSN's in "
                       "HPFS bad-sector list\n", *list - todo);
            }
            else
            {
               TxPrint( "HPFS bad-sector list reset to ZERO bad sectors.\n");
            }
         }
         else
         {
            TxPrint("Write Superblock failed, disk not locked ?\n");
         }
      }
      TxFreeMem(sb);
   }
   RETURN (rc);
}                                               // end 'dfsHpfsImportBadList'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Determine Area size of EA's for an Fnode (including deleted)
/*****************************************************************************/
ULONG dfsHpfsSizeEaFnode
(
   ULONG               sn,                      // IN    LSN for the Fnode
   S_FNODE            *sd                       // IN    Fnode structure
)
{
   ULONG               asize = 0;               // ea area size (limit)
   BYTE                stype;                   // type of this sector

   ENTER();
   if (sd->EAsExtern.size != 0)
   {
      asize = sd->EAsExtern.size;
   }
   if ((sd->EAsFnLength != 0) && (sd->EAsFnLength <= HPFS_MAX_ACL_EA))
   {
      asize = (ULONG) sd->EAsFnLength;
   }
   stype = dfsIdentifySector(sn, 0, (BYTE   *) sd);
   if ((asize == 0) && ((stype  == ST_FNDEL) || (stype  == ST_FNDDI)))
   {
      S_EABLK         *ea = (S_EABLK *) sd->EasOrAcl;

      //- To be refined, walk complete EA chain and determine real size
      //-
      if (((ea->Flag & 0x7c)               == 0   ) &&  // only allow 'sane' EA
          ( ea->NameLength                 != 0   ) &&
          ( ea->Name[0]                    != '\0') &&
          ( ea->Name[(int) ea->NameLength] == '\0') &&
          ( ea->DataLength                 != 0   ) )
      {
         asize = HPFS_MAX_ACL_EA;               // assume maximum size
      }
   }
   RETURN(asize);
}                                               // end 'dfsHpfsSizeEaFnode'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add any EA data info for an Fnode to std FEA2-list structure
/*****************************************************************************/
ULONG dfsHpfsReadEaFnode
(
   ULONG               sn,                      // IN    LSN for the Fnode
   S_FNODE            *sd,                      // IN    Fnode structure
   ULONG              *size,                    // OUT   size in bytes
   S_FEA2LIST        **pfeal                    // OUT   list of FEA2's
)
{
   ULONG               rc    = NO_ERROR;
   ULONG               asize = 0;               // ea area size (limit)
   S_FEA2LIST         *feal  = NULL;            // list of FEA2's

   ENTER();
   if ((asize = dfsHpfsSizeEaFnode( sn, sd)) != 0) // EA's present ?
   {
      if ((feal = TxAlloc(1, EA_BIGBUF_SIZE)) != NULL)
      {
         if (sd->EAsExtern.size != 0)
         {
            rc = dfsOs2EaReadExtern( sd->EAsExtern.ptr,  0, asize, feal);
         }
         else
         {
            rc = dfsOs2EaReadArea((S_EABLK *) sd->EasOrAcl, asize, feal);
         }
         TRHEXS(70, feal, feal->cbList +10, "EA list");
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   *size  = asize;
   *pfeal = feal;
   RETURN (rc);
}                                               // end 'dfsHpfsReadEaFnode'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Calculate checksums and write super- and spare-block from hpfs structure
/*****************************************************************************/
ULONG dfsHpfsWriteSuperSpare
(
   void
)
{
   ULONG               rc = NO_ERROR;
   time_t              tm;                      // c-runtime time_t format

   ENTER();

   if (DFSTORE_WRITE_ALLOWED)
   {
      time( &tm);                               // get current time
      hpfs->sup->OptimDate = (ULONG) tm;        // timestamp last update
      sprintf(         hpfs->sup->VolumeName,   // Volumename / DFS string
               "%s ver %s", DFS_N, DFS_V);      // ... was here!
      if ((hpfs->sup->VerMajor > 2) || (hpfs->sup->VerMinor > 3))
      {
         hpfs->spr->StatusFlag |= 0x80;         // Version may not be supported
      }
      hpfs->spr->SuChecksum = dfsCheckSum((BYTE *) hpfs->sup);
      hpfs->spr->SpChecksum = dfsCheckSum((BYTE *) hpfs->spr);
      if ((rc = dfsWrite(LSN_SUPER, 1,  (BYTE   *) hpfs->sup))   == NO_ERROR)
      {
         if ((rc = dfsWrite(LSN_SPARE, 1, (BYTE   *) hpfs->spr)) == NO_ERROR)
         {
            TxPrint("\nSuper- and spare-block updated and written to disk\n");
         }
         else
         {
            TxPrint("Write Spareblock failed\n");
         }
      }
      else
      {
         TxPrint("Write Superblock failed\n");
      }
   }
   else
   {
      rc = DFS_READ_ONLY;
   }
   RETURN(rc);
}                                               // end 'dfsHpfsWriteSuperSpare'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create valid HPFS boot record based on p-table info, superblock and template
/*****************************************************************************/
ULONG dfsHpfsMkBootRec
(
   S_SUPER            *sup,                     // IN    optional superblock
   char               *drive                    // IN    optional bootdrive
)
{
   ULONG               rc  = DFS_NO_CHANGE;
   ULONG               partSectors;

   ENTER();

   if ((SINF->p != NULL) || ((sup != NULL) && (sup->TotalSec != 0)))
   {
      char             letter = 0;

      if (TxaOption('c'))                       // classic OS2 20.0 bootsector
      {
         brt = (S_BOOTR *) &dfsHpfsBootSecClassic;
         TxPrint("Classic, non I13X-capable bootcode selected, may NOT boot"
                 "on partitions located beyond cylinder 1024.\n");
      }
      else
      {
         brt = (S_BOOTR *) &dfsHpfsBootTemplate;
         if (TxaOption( DFS_O_I13X))            // keep explicit I13X check
         {
            TxPrint("I13X-capable bootcode selected, requires I13X capable MBR too!\n");
            brt->os.RestCode[ HPFS_I13X_CHECK   ] = DFS_B_OPCJNE;
            brt->os.RestCode[ HPFS_I13X_CHECK +1] = 0x05;
         }
         else
         {
            TxPrint("I13X-capable bootcode selected, no I13X check (allow GRUB etc).\n");
            brt->os.RestCode[ HPFS_I13X_CHECK   ] = DFS_B_OPCNOP;
            brt->os.RestCode[ HPFS_I13X_CHECK +1] = DFS_B_OPCNOP;
         }
      }
      if (drive && *drive)                      // letter/reset specified ?
      {
         if (*drive != '-')
         {
            letter = *drive;
         }
      }
      else if (        (SINF->p)              && // partition info there
               (strlen( SINF->p->blabel) > 0) && // bootable by bootmgr
               (isalpha(SINF->p->drive[ 0]  ) )) // and letter available
      {
         letter =       SINF->p->drive[ 0];
      }
      if (letter)
      {
         brt->os.FsysValue = toupper( letter) - 'C' + 0x80;
         TxPrint("Boot driveletter '%c:' has been set in the bootsector\n",
                             toupper( letter));
      }
      else                                      // default
      {
         brt->os.FsysValue = 0;
         TxPrint("No boot driveletter has been set, data partition only\n");
      }
      if (SINF->p != NULL)                      // partition info available
      {
         partSectors           = SINF->p->partent.NumberOfSectors;
         brt->eb.HiddenSectors = SINF->p->partent.BootSectorOffset;
         brt->eb.LogGeoSect    = SINF->p->geoSecs;
         brt->eb.LogGeoHead    = SINF->p->geoHeads;
      }
      else                                      // use global geo info
      {
         TxPrint( "\n%sNo partition-table info%s is available, this will "
                  "make the 'HiddenSectors'\nfield incorrect when this "
                  "is a primary partition.\n", CBR, CNN);
         TxPrint( "The logical geometry used now is %s%hu%s Heads "
                  "with %s%hu%s Sectors\n",
                   CBY, dfstGeoHeads(     DFSTORE), CNN,
                   CBY, dfstGeoSectors(   DFSTORE), CNN);
         TxPrint("Use the 'diskgeo' command to change this when incorrect\n");

         partSectors           = 0;
         brt->eb.HiddenSectors = dfstGeoSectors( DFSTORE);
         brt->eb.LogGeoSect    = dfstGeoSectors( DFSTORE);
         brt->eb.LogGeoHead    = dfstGeoHeads(   DFSTORE);
      }
      if ((sup != NULL) && (sup->TotalSec != 0))
      {
         ULONG supSectors = sup->TotalSec;      // total (normal) sectors

         if ((partSectors - supSectors) < 4)    // HPFS size is modulo 4
         {
            brt->eb.BigSectors =  partSectors;  // closer match (not rounded)
         }
         else                                   // use SUPER calculated value
         {
            brt->eb.BigSectors =  supSectors;   // minimum required size
         }
      }
      else                                      // use size from p-table
      {
         brt->eb.BigSectors = partSectors;
      }
      if (brt->eb.BigSectors <= 0xffff)         // use the right size field
      {
         brt->eb.Sectors      = (USHORT) brt->eb.BigSectors;
         brt->eb.BigSectors   = 0;
      }
      brt->os.SerialNr = (ULONG) dfstLSN2Psn( DFSTORE, 0) + 0xee000000;

      TxPrint( "\nThe following new HPFS bootsector has been prepared:\n\n");
      dfsDisplaySector( dfstLSN2Psn( DFSTORE, LSN_BOOTR), ST_BOOTR, (BYTE   *) brt);
      if ((dfsa->batch) || (TxConfirm( 5126, "Do you want to replace the "
                     "bootsector (sector 0) with the created one ? [Y/N] : ")))
      {
         if (DFSTORE_WRITE_ALLOWED)
         {
            rc = dfsWrite( LSN_BOOTR, 1, (BYTE   *) brt);
            if (rc == NO_ERROR)
            {
               TxPrint("\nHPFS boot record successfully updated\n");
            }
         }
         else
         {
            rc = DFS_READ_ONLY;
         }
      }
      else
      {
         TxPrint("\nCommand aborted, no changes made\n");
      }
   }
   else
   {
      TxPrint("No partition info and no superblock available, "
              "create not possible\n");
   }
   RETURN(rc);
}                                               // end 'dfsHpfsMkBootRec'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Iterate over directory-structures finding files, and execute callback
/*****************************************************************************/
ULONG dfsHpfsIterator
(
   ULN64               recurse,                 // IN    recurse subtrees
   ULN64               type,                    // IN    do 'D', 'f' or 'A'
   char               *path,                    // IN    path spec, info only
   void               *p                        // IN    iterate parameters
)
{
   ULONG               dr;
   DFS_PARAMS         *ip  = (DFS_PARAMS *) p;

   ENTER();

   //- Specified path translated to an LSN and optional entry-nr in params
   dr = dfsHpfsIterateFnode((recurse) ? recurse +1 : 0, type, ip->Lsn, ip);
   RETURN(dr);
}                                               // end 'dfsHpfsIterator'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Recursively iterate over FNODES, directory and files, execute callback
/*****************************************************************************/
static ULONG dfsHpfsIterateFnode
(
   ULONG               rs,                      // IN    recurse subtrees
   ULONG               type,                    // IN    do 'D', 'f' or both
   ULONG               fn,                      // IN    Fnode LSN
   DFS_PARAMS         *cp                       // IN    callback params
)                                               //       .Flag is 'incl deleted'
{
   ULONG               rc = NO_ERROR;
   BYTE                id = ST_UDATA;           // type of sector
   S_FNODE            *sd;                      // FNODE

   ENTER();

   if ((rc = dfsHpfsReadChkFnode( fn, &id, &sd)) == NO_ERROR)
   {
      rc = DFS_ST_MISMATCH;
      switch (id)
      {
         case ST_FNODE:
         case ST_FNDIR:
            rc = NO_ERROR;
            //- type for this fnode not excluded?
            if (((id == ST_FNODE) && (type != 'D') && (type != 'd')) ||
                ((id == ST_FNDIR) && (type != 'F') && (type != 'f'))  )
            {
               #if defined (USEWINDOWING)
                  if (txwIsWindow( TXHWND_DESKTOP))
                  {
                     if (TxTmrTimerExpired( dfsa->statusTimer))
                     {
                        TXTM    status;
                        sprintf( status, "Fnode : 0x%8.8x '%-15.15s', listed now: %llu",
                                 fn, sd->Name, dfsa->snlist[0]);
                        txwSetSbviewStatus( status, cSchemeColor);
                        dfsa->statusTimer = TxTmrSetTimer( DFSP_STATUS_INTERVAL);
                     }
                  }
               #endif
               rc = (cp->Func)(fn, 0, (char *) &id, cp); //- execute the callback
            }
            if ((sd->DirFlag) ||                // Directory
                (sd->ParentDir == fn))          // Root Directory
            {
               if ((rs != 1) && !TxAbort())     // recursion ?
               {
                  rc = dfsHpfsIterateDirBlock(rs -1, type, sd->Al.Directory.DirBlock, cp);
               }
            }
            break;

         default:
            break;
      }
      TxFreeMem( sd);
   }
   RETURN(rc);
}                                               // end 'dfsHpfsIterateFnode'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Recursively iterate over DIRBLOCK, directory and files, execute callback
/*****************************************************************************/
static ULONG dfsHpfsIterateDirBlock
(
   ULONG               rs,                      // IN    recurse subtrees
   ULONG               type,                    // IN    do 'D', 'f' or both
   ULONG               sn,                      // IN    Dirblock LSN
   DFS_PARAMS         *cp                       // IN    callback params
)
{
   ULONG               rc  = NO_ERROR;
   BYTE                id  = ST_UDATA;          // type of sector
   BYTE               *sb  = NULL;              // sector buffer

   ENTER();
   if ((sb = TxAlloc( HPFS_DIRBL_SECTS, dfsGetSectorSize())) != NULL)
   {
      rc = dfsRead(sn, HPFS_DIRBL_SECTS, sb);
      if (rc == NO_ERROR)
      {
         id = dfsIdentifySector(sn, 0, sb);     // double-check sector type
         if (id == ST_DIRBL)
         {
            S_DIRBL      *sd = (S_DIRBL *) sb;
            S_DIRENT     *de;
            BYTE          df;
            ULONG        *bdown;

            for ( de = &(sd->FirstEntry);
                 (de->Length != 0) && (rc == NO_ERROR) && !TxAbort();
                  de = (S_DIRENT *) ((char *) (de) + de->Length))
            {
               df = de->Flag;
               if (df & HPFS_DF_BDOWN)          // B-tree DOWN pointer
               {
                  bdown = (ULONG *) ((char *) (de) + de->Length - sizeof(ULONG));
                  dfsHpfsIterateDirBlock( rs, type, *bdown, cp);
               }
               if ((df & HPFS_DF_BGEND) == 0)   // no special entry
               {
                  rc = dfsHpfsIterateFnode( rs, type, de->Fnode, cp);
               }
               if (df & HPFS_DF_EODIR)          // Special "END" entry
               {
                  break;                        // out of entry-loop
               }
            }
         }
         else                                   // invalid Dirblock LSN
         {
            rc = DFS_ST_MISMATCH;
         }
      }
      TxFreeMem(sb);
   }
   RETURN(rc);
}                                               // end 'dfsHpfsIterateDirBlock'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get allocation for deleted or ALL FNODE sectors into single SPACE structure
// Traverses whole volume as fast as possible, will be relatively slow ...
/*****************************************************************************/
ULONG dfsHpfsDelFnodes2Space
(
   BOOL                delOnly,                 // IN    deleted FNODEs only
   ULONG              *nr,                      // OUT   nr of space entries
   S_SPACE           **sp                       // OUT   space allocation
)
{
   ULONG               rc = NO_ERROR;
   ULONG               newextents = 0;          // new SPACE extents
   S_SPACE            *newsp      = NULL;
   ULONG               chunks = 0;              // chunks in new space structure
   S_SPACE            *space  = NULL;           // new space structure
   USHORT              bps = dfsGetSectorSize();
   ULONG               bs;                      // sectors per buffer
   ULONG               sn;                      // current buffer sn
   ULONG               snlimit = hpfs->sup->TotalSec;
   ULONG               rsec;                    // relative sectornr in buffer
   BYTE                st;                      // sectortype

   ENTER();

   bs = dfsGetBufferSize( DFSOPTIMALBUF, DFSMAXBUFSIZE);

   dfsProgressInit( 0, snlimit, 0, "Volume:", "FNODES", DFSP_STAT, 0);

   for ( sn  = 0;
        (sn < snlimit) && (rc == NO_ERROR) && (!TxAbort());
         sn += bs)
   {
      if ((snlimit - sn) < bs)                  // if almost at end
      {
         bs = snlimit - sn;                     // remaining sectors
      }

      rc = dfsRead( sn, bs, rbuf);
      if (rc == NO_ERROR)
      {
         for (rsec = 0; rsec < bs; rsec++)
         {
            if (DFSFNCALL(dfsa->FsIdentifySector, sn + rsec, 0,
                            (char *) &st, rbuf + (rsec * bps)) == NO_ERROR)
            {
               switch (st)
               {
                  case ST_FNODE:
                  case ST_FNDIR:
                     if (delOnly)
                     {
                        break;                  // skip FNODEs for non-deleted
                     }                          // fall-through if doing ALL
                  case ST_FNDEL:
                  case ST_FNDDI:
                     //- Create single-sector space allocation for this new FNODE
                     newsp = TxAlloc( 1, sizeof(S_SPACE));
                     if (newsp != NULL)
                     {
                        //- populate the single SPACE structure
                        newextents   = 1;
                        newsp->start = sn + rsec;
                        newsp->size  = newextents;

                        //- join with accumulator space, and free the old ones
                        rc = dfsSspaceJoin( chunks, space, newextents, newsp, &chunks, &space);
                     }
                     else
                     {
                        rc = DFS_ALLOC_ERROR;
                     }
                     break;

                  default:                      // ignore other sectortypes
                     break;
               }
            }
         }
         //- Show progress once for each buffer
         dfsProgressShow( sn + bs, 0, chunks, "Found:");
      }
   }
   dfsProgressTerm();

   *sp = space;
   *nr = chunks;
   RETURN (rc);
}                                               // end 'dfsHpfsDelFnodes2Space'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get allocation for ALL referenced FNODE sectors into single SPACE structure
// Traverses directory tree structure only, will be relatively fast ...
/*****************************************************************************/
ULONG dfsHpfsRefFnodes2Space
(
   BOOL                progressSectors,         // IN    progress in sectors, not DIRs
   ULONG              *nr,                      // OUT   nr of space entries
   S_SPACE           **sp                       // OUT   space allocation
)
{
   ULONG               rc = NO_ERROR;
   DFS_PARAMS          params;                  // generic function params
   ULONG               chunks = 0;              // chunks in space structure
   S_SPACE            *space  = NULL;           // space structure

   ENTER();

   //- Start with empty SPACE, first FNODE to add is the ROOT
   params.Lsn    = hpfs->root;                  // start sector for iterator
   params.Number = 0;                           // not a specific entry
   params.Func   = dfsHpfsAddFnode2Space;       // Iterator callback function
   params.Misc   = space;
   params.Count  = chunks;                      // Callback IN/OUT structure
   params.Flag   = progressSectors;

   //- Iterate over all FNODES and DIR blocks recursively and add them to SPACE
   //- Note: type unequal to 'D' or 'f' will do both, 'A' includes deleted ones
   rc = dfsHpfsIterator( -1, 'A', NULL, &params);
   if (rc == NO_ERROR)
   {
      chunks =             params.Count;
      space  = (S_SPACE *) params.Misc;
   }
   *sp = space;
   *nr = chunks;
   RETURN (rc);
}                                               // end 'dfsHpfsRefFnodes2Space'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add SPACE for specified FNODE LSN to the accumulating total FNODE SPACE
/*****************************************************************************/
static ULONG dfsHpfsAddFnode2Space
(
   ULN64               sn64,                    // IN    sector-nr for FNODE
   ULN64               info,                    // IN    dummy (0)
   char               *df,                      // IN    dummy, optional info
   void               *data                     // INOUT callback data, SPACE
)
{
   ULONG               rc      = NO_ERROR;
   ULONG               sn      = (ULONG) sn64;
   DFS_PARAMS         *ip      = (DFS_PARAMS *) data;
   ULONG               extents =                ip->Count;
   S_SPACE            *sp      = (S_SPACE *)    ip->Misc;
   ULONG               chunks  = 1;
   S_SPACE            *space   = NULL;            // SPACE for this DIR

   ENTER();

   //- Create single-sector space allocation for this new FNODE
   space = TxAlloc( 1, sizeof(S_SPACE));
   if (space != NULL)
   {
      //- populate the single SPACE structure
      space->start = sn;
      space->size  = chunks;

      //- join with accumulator space, and free the old ones
      rc = dfsSspaceJoin( extents, sp, chunks, space, &extents, &sp);
      if (rc == NO_ERROR)
      {
         ip->Count = extents;
         ip->Misc  = sp;                        // pass back joined SPACE
      }
   }
   RETURN (rc);
}                                               // end 'dfsHpfsAddFnode2Space'
/*---------------------------------------------------------------------------*/
