//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// HFS utility functions
//
// Author: J. van Wijk
//
// JvW  16-07-2007 Initial version, derived from RSR
//
#ifndef    DFSUHFS
   #define DFSUHFS

#include <dfshfs.h>                             // HFS structure definitions

#define DFS_FS_HFS1        ((ULONG)   1)        // FsIdentify HFS classic


// Find Leaf-Node LSN+Index for specified path, starting at the root-directory
ULONG dfsHfsFindPath
(
   ULN64               loud,                    // IN    Show progress
   ULN64               d2,                      // IN    dummy
   char               *path,                    // IN    path specification
   void               *vp                       // OUT   found dir/file NODE+index
);

// Allocate HFS superblock, and read it from byte-offset 1024 into filesystem
// Note: Area aware to allow usage from FDISK mode too
S_HFS_SUPER *dfsHfsGetSuperBlock                // RET    TxAlloc'ed superblock
(
   DFSPARTINFO        *p                        // IN    NULL if partition opened
);                                              //       or PARTINFO * for a disk


// Identify if FS structures for HFS are present
ULONG dfsHfsFsIdentify
(
   S_BOOTR            *boot,                    // IN    Bootsector ref or NULL
   DFSPARTINFO        *p                        // INOUT partition info
);

// Update the LABEL string in a (tree of) HFS Index/Leaf nodes (recursive)
ULONG dfsHfsNodeSetLabel
(
   ULONG               nodeNr,                  // IN    Catalog (root) node NR
   ULONG               sanity,                  // IN    Sanity recursion check
   char               *oldLabel,                // IN    Old label (verification)
   char               *newLabel                 // IN    New label string
);

// Display file allocation and path info for LSN + index
ULONG dfsHfsFileInfo                            // RET   LSN is valid NODE
(
   ULN64               leafLsn,                 // IN    Leaf-node sectornumber
   ULN64               leafIdx,                 // IN    Leaf-node record index
   char               *select,                  // IN    file selection string
   void               *param                    // INOUT leading text / path
);

// Replace sn-list by contents of a single Directory (InoLsn, InoIdx pairs)
ULONG dfsHfsMakeBrowseList                      // RET   LSN is valid INODE
(
   ULN64               leafLsn,                 // IN    Leaf-node sectornumber
   ULN64               leafIdx,                 // IN    Leaf-node record index
   char               *str,                     // IN    unused
   void               *param                    // INOUT unused
);

// Find FDALIAS actual-DIR CnID for a given (Node) LSN and index info value
ULONG dfsHfsFdLsnInfo2CnId                      // RET   CnID, or 0 when failed
(
   ULN64               nodeLsn,                 // IN    node lsn
   ULONG               index                    // IN    node index
);

// Get allocation information for file-DATA into integrated-SPACE structure
ULONG dfsHfsGetAllocSpace
(
   ULN64               leafLsn,                 // IN    Leaf-node sectornumber
   ULN64               leafIdx,                 // IN    Leaf-node record index
   char               *str,                     // IN    unused
   void               *param                    // INOUT Integrated SPACE
);

// Find full PATH to Root-directory, starting at some Node LSN and index info
ULONG dfsHfsLsnInfo2Path                        // RET   result
(
   ULN64               start,                   // IN    starting node lsn
   ULONG               index,                   // IN    catalog record index
   char               *path                     // OUT   combined path string
);

// Find related CnID for given (Node) LSN and index info value; optional dType
ULONG dfsHfsLsnInfo2CnId                        // RET   CnID, or 0 when failed
(
   ULN64               nodeLsn,                 // IN    node lsn
   ULONG               index,                   // IN    node index
   char               *dType                    // OUT   detailed type, or NULL
);                                              //       D,H,f,h,s

// Find catalog-record for a given Node-structure and index info value
USHORT dfsHfsNodeIdx2Rec                        // RET   recordtype, or 0 when failed
(
   BYTE               *node,                    // IN    node structure
   ULONG               index,                   // IN    node index
   U_HFS_CAT_RECORD  **rec                      // OUT   file/folder catalog record
);

#define dfsHfsBlock2Lsn(bl) (ULN64) dfsHfsCl2Lsn((bl),0,NULL,NULL)
// Translate Block-nr to LSN, generic interface
ULN64 dfsHfsCl2Lsn                              // RET   LSN for this block
(
   ULN64               block,                   // IN    block number
   ULN64               d2,                      // IN    dummy
   char               *p1,                      // IN    dummy
   void               *p2                       // IN    dummy
);

#define dfsHfsLsn2Block(sn) dfsHfsLsn2Cl((ULN64)(sn),0,NULL, NULL)
// Translate LSN to Block-nr, generic interface
ULN64 dfsHfsLsn2Cl                              // RET   block for this LSN
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *p1,                      // IN    dummy
   void               *p2                       // IN    dummy
);

// Determine allocation-bit for specified LSN, ALLOCATED beyond last block!
ULONG dfsHfsAllocated                           // RET   LSN is allocated
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);

// Align R/W cache to position for block, and get current allocation-bit
BOOL dfsHfsBitmapCache                          // RET   block is allocated
(
   ULONG               bl,                      // IN    Block number
   ULONG              *cp                       // OUT   position in cache
);                                              //       or L64_NULL if invalid

// Extract BitMap allocation S_SPACE structure and initialize the BitMap cache
// Note: Area aware to allow usage from FDISK mode too
ULONG dfsHfsBitMapInit                          // RET   rc = 0 if type match
(
   void
);

// Flush BitMap cache when Dirty, and optional free its resources
ULONG dfsHfsBitmapFlush                         // RET   rc
(
   BOOL                terminate                // IN    terminate cache too
);

// Set allocate status for LSN to specified value (beyond FS, to end of bitmap)
ULONG dfsHfsSetAlloc                            // RET   LSN allocation set
(
   ULN64               lsn,                     // IN    LSN
   char               *value,                   // IN    NULL = not allocated
   void               *ref                      // IN    dummy (for HFS)
);

// Set allocate status for BLOCK to specified value
ULONG dfsHfsSetBlock                            // RET   BL  allocation set
(
   ULONG               bl,                      // IN    Block
   BOOL                value                    // IN    SET allocation bit
);

// Search thread-record for CnID, returning parent-CnID / item-name / isDir
ULONG dfsHfsCnId2Parent                         // RET   result
(
   ULONG               cnId,                    // IN    CNID of item to search
   ULONG              *parent,                  // OUT   parent-CnID       or NULL
   char               *name,                    // OUT   NAME of item, ""  or NULL
   BOOL               *isDir                    // OUT   item is DIR flag, or NULL
);

// Extract Catalog allocation S_SPACE structure and cache the RootNode
ULONG dfsHfsCatalogInit
(
   void
);

// Search catalog B-tree for specified key, get Leafnode + optional rec-index
ULONG dfsHfsSearchCatalog                       // RET   Cat NODE number (leaf)
(
   ULONG               parent,                  // IN    CNID of parent to search
   char               *name,                    // IN    NAME of item, "" or NULL
   USHORT             *record,                  // OUT   record-index,    or NULL
   BYTE              **leaf                     // OUT   ptr to leafnode, or NULL
);

// Compare catalog-record-key to supplied parent and optional name
int dfsHfsCatalogKeyCompare                     // RET   0 = EQ; 1 = rKey larger
(                                               //              -1 = rKey smaller
   S_HFS_CATALOG_KEY  *recordKey,               // IN    record key
   ULONG               parent,                  // IN    parent ID to compere
   char               *name                     // IN    name to compare, or NULL
);

// HFS read and check type for an HFS+ Catalog Node from its sector number
ULONG dfsHfsReadChkCatNode
(
   ULN64               lsn,                     // IN    Inode LSN
   BYTE               *stype,                   // OUT   Node type
   BYTE              **catNode                  // OUT   Catalog Node structure
);

// HFS filesystem, display THREAD-style header/footer
void dfsHfsThreadHeader
(
   ULONG               items                    // IN    DIR-lines shown (0 = TOP)
);

// HFS filesystem, display DIR header/footer
void dfsHfsDirHeader
(
   char               *lead,                    // IN    lead text-string     (20)
   ULONG               items                    // IN    DIR-lines shown (0 = TOP)
);

// Translate Catalog-Node-ID to logical sector nr
ULN64 dfsHfsNode2Lsn                           // RET   Logical sector 64bit
(
   ULONG               nodeNr                   // IN    node number
);

// Translate (Catalog Node sector) LSN to Catalog NODE-Number
ULONG dfsHfsLsn2Node                            // RET   Catalog NODE-Number
(
   ULN64               lsn                      // IN    Logical sector nr
);

// Display HFS Catalog-record entry on two lines, in 'dfsee directory' format
void dfsHfsShowCatRecord
(
   BYTE               *keyRec,                  // IN    key followed by record
   BYTE                bg                       // IN    current background
);

// Read ASCII SYMLINK string from specified symlink sectornumber
ULONG dfsHfsResolveSymLink
(
   ULN64               slnkLsn,                 // IN    symlink sectornumer
   char               *linkname                 // OUT   resolved symlink name
);

// Create ASCII FOLDERALIAS string from specified 'alis' sectornumber
ULONG dfsHfsResolveFdAlias
(
   ULN64               fdalLsn,                 // IN    folder alias sectornumer
   char               *linkname                 // OUT   resolved folder-alias name
);

// Convert HFS DATE/TIME (32 bit) to standard C time_t value
time_t dfsHfsFileTime2t                         // RET   time_t representation
(
   ULONG               hfsdate                  // IN    HFS time value
);

// Convert HFS DATE/TIME (32 bit) to DFS standard date/time string
char *dfsHfsTime2str                            // RET   string value
(
   ULONG               hfsdate,                 // IN    HFS time value
   char               *dtime                    // INOUT ptr to string buffer
);

// Display HFS style data (1-jan-1904, big-endian) to TxPrint output
void dfsHfsPrintDateTime
(
   char               *lead,                    // IN    leading string
   ULONG               hfsdate                  // IN    HFS style date
);

// Create SPACE allocation structure from an HFS ForkData structure
ULONG dfsHfsFork2Space
(
   S_HFS_FORKDATA     *fork,                    // IN    fork data
   ULONG              *nr,                      // OUT   nr of space entries
   S_SPACE           **sp                       // OUT   space allocation
);

// Display HFS Forkdata to TxPrint output
void dfsHfsPrintForkdata
(
   char               *lead,                    // IN    leading string
   ULONG               options,                 // IN    SD_ display options
   S_HFS_FORKDATA     *fork                     // IN    HFS fork data
);

// Read stored offset for a record in a node, based on its index
USHORT dfsHfsRecordOffset                       // RET   offset to record
(
   USHORT              index,                   // IN    record index in node
   void               *node                     // IN    any HFS node
);

// Return fixed-length string (6) with node-type description
char *dfsHfsNodeType                            // RET   description string
(
   BYTE                kind                     // IN    kind of node
);

// Return fixed-length string (4) with catalog-record type description
char *dfsHfsCatRecType                          // RET   description string
(
   USHORT              crType                   // IN    catalog record type
);

#endif
