//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// HFS utility functions
//
// Author: J. van Wijk
//
// JvW  16-07-2007 Initial version, derived from RSR
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsver.h>                             // DFS version info
#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfstable.h>                           // SLT utility functions

#include <dfshfs.h>                             // HFS disk structures
#include <dfsahfs.h>                            // HFS analysis functions
#include <dfsuhfs.h>                            // HFS utility functions
#include <dfslhfs.h>                            // HFS slt & error definitions


// Update Mac-Unicode name at specified offset in node, fixup table of offsets
static ULONG dfsNodeChangeName
(
   BYTE               *node,                    // IN    Catalog tree node
   USHORT              offset,                  // IN    Byte offset of name
   char               *newLabel                 // IN    New label string ASCII
);


/*****************************************************************************/
// Find Leaf-Node LSN+Index for specified path, starting at the root-directory
/*****************************************************************************/
ULONG dfsHfsFindPath
(
   ULN64               loud,                    // IN    Show progress
   ULN64               d2,                      // IN    dummy
   char               *path,                    // IN    path specification
   void               *vp                       // OUT   found dir/file NODE+index
)
{
   ULONG               rc  = NO_ERROR;
   TXLN                part;
   char               *p   = path;
   int                 l;
   ULONG               nodeNr;
   ULONG               curCnId = HFS_ID_RootFolder;
   USHORT              nodeIdx = 0;             // ROOT is 1st record (folder 1, label)
   ULONG               nodeLsn = dfsHfsNode2Lsn( TxBE32( hfs->Catalog.Btree->hdr.bthFirstLeaf));
   BYTE               *node = NULL;
   DFS_PARAMS         *parm = (DFS_PARAMS *) vp;

   ENTER();

   if (loud)
   {
      TxPrint("Root folder  CnID : 0x%08.8X   find path: '%s'\n", curCnId, path);
   }
   parm->Lsn    = nodeLsn;                      // initial values, for ROOT (empty path)
   parm->Number = (ULONG) nodeIdx | DFSSNINFO;
   while ((rc == NO_ERROR) && strlen(p) && !TxAbort())
   {
      if ((l = strcspn(p, FS_PATH_STR)) != 0)
      {
         strncpy(part, p, l);                   // isolate part
         part[l] = '\0';
         p += l;                                // skip part
         if (*p == FS_PATH_SEP)
         {
            p++;                                // skip '\'
         }
      }
      if (strlen(part))
      {
         if (loud)
         {
            TxPrint("Search  folder Id : 0x%08.8X", curCnId);
         }
         nodeNr = dfsHfsSearchCatalog( curCnId, part, &nodeIdx, &node);
         if ((nodeNr != 0) && ((nodeNr != L32_NULL)))
         {
            U_HFS_CAT_RECORD   *catData;
            USHORT              recType;

            recType = dfsHfsNodeIdx2Rec( node, nodeIdx, &catData);
            if ((recType == HFS_CR_File) || (recType == HFS_CR_Folder))
            {
               curCnId = TxBE32( catData->d.crCnID);
               if (loud)
               {
                  TxPrint(" - CnID  %08.8X for '%s'\n", curCnId, part);
               }
               if (*p == '\0')                  // end of string, found!
               {
                  parm->Lsn    = dfsHfsNode2Lsn( nodeNr);
                  parm->Number = (ULONG) nodeIdx | DFSSNINFO;
                  parm->Flag   = FALSE;         // Size NOT from DIR-entry
                  if (recType == HFS_CR_File)   // get the filesize
                  {
                     parm->byteSize = TxBE64( catData->file.crDataFork.logicalSize) +    //- return sum of DATA
                                      TxBE64( catData->file.crRsrcFork.logicalSize);     //- plus RESOURCE fork
                  }
               }
            }
            else
            {
               rc = DFS_BAD_STRUCTURE;
            }
            TxFreeMem( node);
         }
         else
         {
            rc = ERROR_PATH_NOT_FOUND;
         }
         if (rc != NO_ERROR)
         {
            if (loud)
            {
               TxPrint(" - Search failed  for '%s'\n", part);
            }
         }
      }
      else
      {
         rc = ERROR_PATH_NOT_FOUND;
      }
   }
   RETURN (rc);
}                                               // end 'dfsHfsFindPath'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Allocate HFS superblock, and read it from byte-offset 1024 into filesystem
// Note: Area aware to allow usage from FDISK mode too
/*****************************************************************************/
S_HFS_SUPER *dfsHfsGetSuperBlock                // RET    TxAlloc'ed superblock
(
   DFSPARTINFO        *p                        // IN    NULL if partition opened
)                                               //       or PARTINFO * for a disk
{
   S_HFS_SUPER        *super = NULL;            // function return
   BYTE               *buffer;
   int                 sectors;                 // sectors to read SB

   ENTER();

   sectors = (HFS_SUPSIZE / dfsGetSectorSize()); // #sectors to read super 'block'
   if ((buffer = TxAlloc( sectors, dfsGetSectorSize())) != NULL)
   {
      if (dfsRead( (p) ? p->basePsn : dfstAreaP2Disk( DFSTORE, LSN_BOOTR), sectors, buffer) == NO_ERROR)
      {
         //- allocate at least same size as largest sector, so the Identify functions
         //- HfsIdent() and IdentifySector() can safely access upto GetSectorSize()

         if ((super = TxAlloc(1, HFS_SUPSIZE)) != NULL)
         {
            memcpy( super, buffer + 1024, sizeof( S_HFS_SUPER));
         }
      }
      else                                      // no superblock (yet)
      {
         //- to be refined, could try to read from backup location, if there are any
      }
      TxFreeMem( buffer);
   }
   RETURN (super);
}                                               // end 'dfsHfsGetSuperBlock'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Identify if FS structures for HFS are present
/*****************************************************************************/
ULONG dfsHfsFsIdentify
(
   S_BOOTR            *boot,                    // IN    Bootsector ref or NULL
   DFSPARTINFO        *p                        // INOUT partition info
)
{
   ULONG               rc = DFS_FS_UNKNOWN;     // function return
   S_HFS_SUPER        *sup;                     // superblock sector
   char                st;
   ULN64               basePsn = (p) ? p->basePsn : dfstAreaP2Disk( DFSTORE, LSN_BOOTR);

   ENTER();

   if ((sup = dfsHfsGetSuperBlock( p)) != NULL)
   {
      if ((dfsHfsIdent(HFS_LSNSUP1, 0, &st, sup) != DFS_PENDING) && (st == ST_HFSSUP))
      {
         BYTE            *block = (BYTE *) sup;           //- re-use superblock as disk buffer
         S_BT_HDR_NODE   *node  = (S_BT_HDR_NODE *) sup;  //- Header-Node   alias to buffer
         S_BT_ROOTNODE   *root  = (S_BT_ROOTNODE *) sup;  //- Root-Node     alias to buffer
         ULONG            spblk = TxBE32( sup->BlockSize) / dfsGetSectorSize(); // real blocksize (could be > 4096)
         ULONG            bsect = HFS_SUPSIZE             / dfsGetSectorSize(); // allocated by GetSuper.. == 4096!
         ULN64            catSn = TxBE32( sup->CatalogFile.extents[0].startBlock) * spblk;

         TRACES(("basePsn: 0x0%llx  bsect: %u  spblk: %u  catSn: 0x0%llx\n", basePsn, bsect, spblk, catSn));

         //- read (first 4096 bytes of) the Catalog-B-tree HEADER NODE block using its Fork data in superblock
         if (dfsRead( basePsn + catSn, bsect, block) == NO_ERROR)
         {
            ULONG         sectsPerNode = TxBE16( node->hdr.bthNodeSize) / dfsGetSectorSize();
            ULN64         rootDirDelta = TxBE32( node->hdr.bthRootNode) * sectsPerNode;

            TRACES(("sectsPerNode: %u  rootDirDelta: 0x0%llx\n", sectsPerNode, rootDirDelta));

            //- Read (first 4096 bytes of) the tree-root node, and get the name (LABEL) from that
            if (dfsRead( basePsn + catSn + rootDirDelta, bsect, block) == NO_ERROR)
            {
               TXLN    label;

               TRHEXS(70, block, 0x80, "root-folder node");
               TxMacUniStr2Ascii( &(root->Label.name), label);
               TxCopy( p->plabel, label, TXMAXTS); // limit to available size in partinfo

               TRACES(("Ascii label value: '%s'\n", label));
            }
         }
         strcpy( p->fsform, "HFS");
         rc = NO_ERROR;
      }
      TxFreeMem( sup);
   }
   RETURN (rc);
}                                               // end 'dfsHfsFsIdentify'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Update the LABEL string in a (tree of) HFS Index/Leaf nodes (recursive)
/*****************************************************************************/
ULONG dfsHfsNodeSetLabel
(
   ULONG               nodeNr,                  // IN    Catalog (root) node NR
   ULONG               sanity,                  // IN    Sanity recursion check
   char               *oldLabel,                // IN    Old label (verification)
   char               *newLabel                 // IN    New label string
)
{
   ULONG               rc = NO_ERROR;           // function return
   ULN64               nodeLsn;
   BOOL                isDirty = FALSE;         // Node contents changed
   BYTE               *node;                    // Node structure index/leaf
   BYTE                st;                      // sector type (node type)

   ENTER();
   TRACES(("nodeNr: %8.8x sanity:%u old:'%s' new:'%s'\n", nodeNr, sanity, oldLabel, newLabel));

   if (sanity > 0)                              // more recursion levels allowed
   {
      nodeLsn = dfsHfsNode2Lsn( nodeNr);        // needed again for write
      if ((rc = dfsHfsReadChkCatNode( nodeLsn, &st, &node)) == NO_ERROR)
      {
         int                deltaSize = 2 * (strlen( newLabel) - strlen( oldLabel)); // Unicode!
         USHORT             keyOffset = dfsHfsRecordOffset( 0, node); // first record holds label in key
         S_HFS_CATALOG_KEY *key       = (S_HFS_CATALOG_KEY *) (node + keyOffset);
         USHORT             keyLength = TxBE16( key->keyLength);
         S_BT_NODE_DESC    *desc = (S_BT_NODE_DESC *) node;
         USHORT             nRecs = TxBE16( desc->nRecords); // number of records
         USHORT             freeB = hfs->Catalog.bpNode - ((nRecs + 1) * sizeof( USHORT))
                                      - dfsHfsRecordOffset( nRecs, node);

         TRACES(("nRecs:%hu  freeB:%hu  deltaSize: %d\n", nRecs, freeB, deltaSize));
         TRHEXS(70, node, hfs->Catalog.bpNode, "Node before update");

         switch (desc->nKind)
         {
            case BT_IndexNode:
               if (freeB >= deltaSize)
               {
                  ULONG nextLevelNodeId = TxBE32( *((ULONG *) (node + keyOffset + keyLength + 2)));

                  //- recurse into next node-level BEFORE updating this one!
                  rc = dfsHfsNodeSetLabel( nextLevelNodeId, sanity - 1, oldLabel, newLabel);

                  if (rc == NO_ERROR)           // update when lower levels OK
                  {
                     keyOffset      = dfsHfsRecordOffset( 0, node); // offset key 1st record
                     key            = (S_HFS_CATALOG_KEY *) (node + keyOffset);
                     key->keyLength = TxBE16( 2 * strlen( newLabel) + 6); // Uni string + size/parent

                     rc = dfsNodeChangeName( node, keyOffset + 6, newLabel);
                     if (rc == NO_ERROR)
                     {
                        isDirty = TRUE;         // request write-back
                     }
                  }
               }
               else
               {
                  rc = DFS_CMD_FAILED;
               }
               break;

            case BT_LeafNode:
               if (freeB >= (2 * deltaSize))
               {
                  keyOffset      = dfsHfsRecordOffset( 0, node); // offset key 1st record
                  key            = (S_HFS_CATALOG_KEY *) (node + keyOffset);
                  key->keyLength = TxBE16( 2 * strlen( newLabel) + 6); // Uni string + size/parent

                  rc = dfsNodeChangeName( node, keyOffset + 6, newLabel);
                  if (rc == NO_ERROR)
                  {
                     keyOffset   = dfsHfsRecordOffset( 1, node); // offset key 2nd record (Thread)
                     key         = (S_HFS_CATALOG_KEY *) (node + keyOffset);
                     keyLength   = TxBE16( key->keyLength);

                     //- update name in THREAD record data area
                     rc = dfsNodeChangeName( node, keyOffset + keyLength + 10, newLabel);
                     if (rc == NO_ERROR)
                     {
                        isDirty = TRUE;         // request write-back
                     }
                  }
               }
               else
               {
                  rc = DFS_CMD_FAILED;
               }
               break;

            default:                            // should not happen, error
               rc = DFS_ST_MISMATCH;
               break;
         }
         if (isDirty)                           // node contents changed
         {
            TRHEXS(70, node, hfs->Catalog.bpNode, "Node with LABEL update");

            rc = dfsWrite( nodeLsn, hfs->Catalog.spNode, node);
            if (rc == NO_ERROR)
            {
               TxPrint( "\nCatalog %s node 0x%8.8x updated to use label '%s'\n",
                           (desc->nKind == BT_LeafNode) ? "Leaf " : "Index", nodeNr, newLabel);
            }
         }
         TxFreeMem( node);
      }
   }
   RETURN (rc);
}                                               // end 'dfsHfsNodeSetLabel'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Update Mac-Unicode name at specified offset in node, fixup table of offsets
/*****************************************************************************/
static ULONG dfsNodeChangeName
(
   BYTE               *node,                    // IN    Catalog tree node
   USHORT              offset,                  // IN    Byte offset of name
   char               *newLabel                 // IN    New label string ASCII
)
{
   ULONG               rc = NO_ERROR;           // function return
   TXLN                oldLabel;
   int                 deltaSize;
   S_MACUNI_STRING    *macUni = (S_MACUNI_STRING *) (node + offset);
   S_BT_NODE_DESC     *desc   = (S_BT_NODE_DESC *) node;
   USHORT              nRecs  = TxBE16( desc->nRecords); // number of records

   ENTER();

   //- read existing (old label) Mac-Unicode string
   TxMacUniStr2Ascii( macUni, oldLabel);
   deltaSize = 2 * (strlen( newLabel) - strlen( oldLabel)); // can be negative!

   if (deltaSize != 0)
   {
      USHORT           rec;
      USHORT           tableValue;              // offset value from table
      USHORT          *tableEntry;              // ptr to table value
      USHORT           moveSize = dfsHfsRecordOffset( nRecs, node) - offset;

      TRACES(("deltaSize: %d  Size to move: 0x%4.4hx = %hu\n", deltaSize, moveSize, moveSize));

      //- move remainder of data/records up or down, depending on new stringsize
      if      (deltaSize > 0)                   // new string longer, move up
      {
         TRACES(("move TO: 0x%4.4hx FROM 0x%4.4hx size: 0x%4.4hx\n", offset + deltaSize, offset, moveSize));
         memmove( node + offset + deltaSize, node + offset, moveSize);    //- move, allow overlap!
      }
      else if (deltaSize < 0)                   // new string shorter, move down
      {
         TRACES(("move TO: 0x%4.4hx FROM 0x%4.4hx size: 0x%4.4hx\n", offset, offset - deltaSize, moveSize));
         memmove( node + offset, node + offset - deltaSize, moveSize);    //- move, allow overlap!
         memset(  node + offset + moveSize + deltaSize, 0, - deltaSize);  //- wipe new freespace clean
         TRACES(("Wipe offset: 0x%4.4hx  size: %d\n", offset + moveSize + deltaSize, - deltaSize));
      }

      //- Update the offset table at the end of the node, for #rec + 1 (freespace)
      tableEntry = (USHORT *) (node + hfs->Catalog.bpNode - sizeof( USHORT));
      for (rec = 0; rec <= nRecs; rec++)
      {
         tableValue = TxBE16( *tableEntry);

         if (tableValue > offset)               // entry is affected by move
         {
            TRACES(("Fixup rec %3hu from 0x%4.4hx to 0x%4.4hx\n", rec, tableValue, tableValue + deltaSize));
            tableValue += deltaSize;
            *tableEntry = TxBE16( tableValue);  // write updated value to table
         }
         tableEntry--;                          // point to next value in table
      }
   }

   //- Now write the new Mac-Unicode string to original location
   TxAscii2MacUniStr( newLabel, macUni);

   RETURN (rc);
}                                               // end 'dfsNodeChangeName'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display file allocation and path info for LSN + index
// Supports automatic resolve of hardlinks, when enabled in the (menu) option
/*****************************************************************************/
ULONG dfsHfsFileInfo                            // RET   LSN is valid NODE
(
   ULN64               leafLsn,                 // IN    Leaf-node sectornumber
   ULN64               leafIdx,                 // IN    Leaf-node record index
   char               *select,                  // IN    file selection string
   void               *param                    // INOUT leading text / path
)
{
   ULONG               rc = NO_ERROR;
   TXLN                fpath;
   ULONG               xsize     = 0;           // EA/XATTR/ResourceFork size
   ULN64               fsize     = 0;           // filesize in bytes
   ULN64               size      = 0;           // allocated size, sectors
   ULN64               bads      = 0;           // bad allocated sectors
   TXLN                text;
   ULONG               percent   = 100;
   ULONG               threshold = 0;           // threshold percentage
   BOOL                isMinimum = TRUE;        // threshold is minimum value
   ULN64               location  = 0;           // LSN for all file data
   ULN64               aliasLsn  = 0;           // LSN for alias resource fork
   TXLN                temp;                    // location/size/date/symlink text
   ULONG               minS      = 0;           // minimal size
   ULONG               maxS      = 0;           // maximal size
   char                itemType  = 0;           // type Dir, File, Browse, or ANY
   time_t              modTstamp = 0;           // Modify timestamp value, or 0
   char               *lead    = param;
   BOOL                verbose = (strcmp( lead, ":") != 0);
   BOOL                output;                  // no Printf, just return fpath
   BOOL                alcheck;                 // Execute a full allocation check
   BYTE                st = 0;                  // node-type or Dir/file designator
   BYTE               *node = NULL;             // leaf node for file
   USHORT              offset;                  // record offset in leaf node
   BOOL                isSymLink = FALSE;
   BOOL                isHrdLink = FALSE;
   BOOL                isFdAlias = FALSE;

   ENTER();
   TRARGS(("LSN: 0x%llX, index: %llX  select: '%s', lead: '%s'\n", leafLsn, leafIdx, select, lead));

   if (strcmp( lead, "!") == 0)                 // Skip allocation check wanted
   {
      output  = FALSE;                          // no output
      alcheck = FALSE;                          // and no allocation check
      *lead   = 0;                              // discard 'garbage' in lead string
   }
   else                                         // Possible output/alloc-check wanted
   {
      output  = (*lead != 0);                   // output if any lead given (except '!')
      alcheck = TRUE;                           // but always an alloc check if no  '!'
   }
   TRACES(("verbose:%u output:%u alcheck:%u\n", verbose, output, alcheck));

   if ((leafIdx & DFSSNINFO) && (dfsHfsReadChkCatNode( leafLsn, &st, &node) == NO_ERROR) && (st == ST_HFSCLF))
   {
      if ((offset = dfsHfsRecordOffset( DFSSNIGET( leafIdx), node)) != 0)
      {
         S_HFS_CATALOG_KEY *key       = (S_HFS_CATALOG_KEY *) (node + offset);
         USHORT             keyLength = TxBE16( key->keyLength);
         U_HFS_CAT_RECORD  *catData   = (U_HFS_CAT_RECORD *) (((BYTE *) key) + keyLength + 2);
         USHORT             recType   = TxBE16( catData->t.crRecType);

         dfsParseFileSelection( select, text, &isMinimum, &threshold, &minS, &maxS, &modTstamp, &itemType);
         /* to be refined, check allocation (for files only)
         if (verbose)
         {
            //- Allow skipping the allocation-check when not needed (Speedup BROWSE display)
            rc = dfsHfsCatCheckAlloc( catData, (alcheck) ? "" : "s", &fsize, &size, &bads, &location);
         }
         else                                   // no size/percentage info
         */
         {
            isMinimum = TRUE; threshold = 0;    // no percentage filter
            minS      = 0;    maxS      = 0;    // no size filter
         }

         isSymLink = ((catData->d.f.type == HFS_FT_SYMLINK) && (catData->d.f.creator == HFS_FC_SYMLINK));
         isHrdLink = ((catData->d.f.type == HFS_FT_HRDLINK) && (catData->d.f.creator == HFS_FC_HRDLINK));
         isFdAlias = ((catData->d.f.type == HFS_FT_FDALIAS) && (catData->d.f.creator == HFS_FC_FDALIAS));

         TRACES(("recType: %hu\n", recType));
         switch (recType)
         {
            case HFS_CR_Folder:                 // normal dir
               if      (itemType == DFS_FS_ITEM_BROWSE) // no filtering on Directories, take all.
               {
                  strcpy( text, "");            // no wildcard
                  isMinimum = TRUE; threshold = 0; // no percentage filter
                  modTstamp = 0;                // no timestamp filter
                  minS      = 0;    maxS      = 0; // no size filter
               }
               else if ((itemType == DFS_FS_ITEM_FILES) || // filter files, discard Directories
                        (isFdAlias == TRUE))    // or it is a DIR disguised as file!
               {
                  rc = DFS_ST_MISMATCH;         // item-type mismatch
                  break;
               }
            case HFS_CR_File:                   // normal file

               if ((itemType  == DFS_FS_ITEM_DIRS) && // filter dirs, discard files
                   (isFdAlias == FALSE))        // and it is NOT a DIR disguised as file!
               {
                  rc = DFS_ST_MISMATCH;         // item-type mismatch
                  break;
               }
               dfsHfsLsnInfo2Path( leafLsn, DFSSNIGET( leafIdx), fpath);
               if ((strlen(text) == 0) || (TxStrWicmp( fpath, text) >= 0))
               {
                  if (recType == HFS_CR_File)
                  {
                     //- resolve hardlink, if any, to get proper size and location for display and recovery
                     if ((isHrdLink) && (hfs->hlMetaFolderId != 0) && (HfsHardlinksAutoResolve == TRUE))
                     {
                        ULONG              nodeNr;
                        USHORT             index; // index in new node
                        TXTT               tbuf; // hardlink iNode name buffer

                        TxFreeMem( node);   // free original node, to be replaced with resolved one

                        sprintf( tbuf, "iNode%d", TxBE32( catData->d.crBsdInfo.iNodeNum));
                        nodeNr = dfsHfsSearchCatalog( hfs->hlMetaFolderId, tbuf, &index, &node);
                        if ((nodeNr != 0) && ((nodeNr != L32_NULL)))
                        {
                           if (dfsHfsNodeIdx2Rec( node, index, &catData) == HFS_CR_File)
                           {
                              TRACES(("Resolved hardlink, morphed catData to that record for location/sizes\n"));
                           }
                        }
                     }

                     if ((isFdAlias) && (hfs->fdAliasFolderId != 0))
                     {
                        aliasLsn = dfsHfsBlock2Lsn( TxBE32( catData->file.crRsrcFork.extents[0].startBlock));
                        if (HfsFldrAliasAutoResolve == FALSE)
                        {
                           //- detailed-type 'H' and non-zero location -> handle as a file in the browser
                           location = aliasLsn; // signal to BROWSER to NOT enter this as a directory now!
                           fsize    = TxBE64( catData->file.crRsrcFork.logicalSize);
                           size = (fsize + dfsGetSectorSize() -1) / dfsGetSectorSize();
                        }
                     }
                     else
                     {
                        location = dfsHfsBlock2Lsn( TxBE32( catData->file.crDataFork.extents[0].startBlock));
                        fsize    = TxBE64( catData->file.crDataFork.logicalSize) +    //- return sum of DATA
                                   TxBE64( catData->file.crRsrcFork.logicalSize);     //- plus RESOURCE fork
                        size = (fsize + dfsGetSectorSize() -1) / dfsGetSectorSize();
                     }
                  }
                  if (verbose)                  // include alloc and size
                  {
                     if (location == 0)
                     {
                        if (fsize == 0)
                        {
                           if (recType == HFS_CR_File)
                           {
                              if (isFdAlias)
                              {
                                 sprintf( temp, "%s  fldAlias%s ", CBZ, CNN);
                              }
                              else
                              {
                                 sprintf( temp, "%s     empty%s ", CBZ, CNN);
                              }
                           }
                           else
                           {
                              sprintf(    temp, "%s       dir%s ", CBZ, CNN);
                           }
                           rc = NO_ERROR;       // allocation always OK
                        }
                        else                    // there IS something, perhaps resource fork
                        {
                           location = dfsHfsBlock2Lsn( TxBE32( catData->file.crRsrcFork.extents[0].startBlock));
                           if (location == 0)
                           {
                              strcpy(  temp, "           ");
                           }
                           else
                           {
                              sprintf( temp, "%sr:%s%8llX ", CBZ, CNN, location); // signal reource location
                           }
                           rc = NO_ERROR;       // allocation always OK
                        }
                     }
                     else
                     {
                        sprintf( temp, "%10llX ", location);
                     }
                     percent = dfsAllocationReliability( size, bads);

                     TRACES(( "percent:%u threshold:%u size:%u minS:%u maxS:%u\n",
                               percent, threshold, size, minS, maxS));
                  }
                  if ((BOOL)((percent >= threshold)) == isMinimum)
                  {
                     ULONG modTime = TxBE32( catData->d.crModContTime); // contents modify time, native endianess

                     if ((HfsDate2Tm( modTime) > modTstamp) &&                //- Timestamp match time_t
                         ((size >= minS) && ((size <= maxS) || (maxS == 0)))) //- Size match
                     {
                        if (verbose)            // include alloc and size
                        {
                           st = (recType == HFS_CR_Folder) ? 'D' :
                                               (isSymLink) ? 's' :
                                               (isFdAlias) ? 'H' :
                                               (isHrdLink) ? 'h' : 'f';
                           if (alcheck)
                           {
                              sprintf( text, "%s%c %s%3u%%%s ", CBM, st, (rc == NO_ERROR) ? CBG : CBR, percent, CNN);
                           }
                           else                 // no reliability percentage
                           {
                              sprintf( text, "%s%c%s      ", CBM, st, CNN);
                           }
                           strcat(     text, temp);

                           if ((recType == HFS_CR_Folder) || (isFdAlias && (HfsFldrAliasAutoResolve == TRUE)))
                           {
                              strcpy( temp, "         "); // just leave empty for now
                           }
                           else                 // files, report size in sectors, KiB..TiB, set RF size
                           {
                              strcpy( temp, "");
                              dfstrSize( temp, CBC, size, " ");
                              xsize = (ULONG) TxBE64( catData->file.crRsrcFork.logicalSize);
                           }
                           dfstrBytes(temp, CBZ, xsize,   " "); // EA/XATTR/RF-size (should be sum of ...)
                           strcat( text, temp);
                        }
                        else
                        {
                           strcpy( text, "");   // remove any stray text, non-verbose
                        }
                        if (output)             // not totally silent?
                        {
                           strcat( lead, text);
                           TxPrint( "%s%s%s", lead, CBY, fpath);
                           if ((isSymLink) || ((isHrdLink) && (HfsHardlinksAutoResolve == FALSE))
                                           || ((isFdAlias) && (HfsFldrAliasAutoResolve == FALSE)))
                           {
                              if (strlen( fpath) < 19)
                              {
                                 TxPrint( "%*.*s", (19 - strlen( fpath)), (19 - strlen( fpath)), "");
                              }
                              if      (isSymLink)
                              {
                                 dfsHfsResolveSymLink( location, temp);
                              }
                              else if (isHrdLink)
                              {
                                 sprintf( temp, "/%s/iNode%d", HFS_SD_HARDLINK, TxBE32( catData->d.crBsdInfo.iNodeNum));
                              }
                              else
                              {
                                 dfsHfsResolveFdAlias( aliasLsn, temp);
                              }
                              TxPrint( "%s%s%s%s", CNN, DFS_SH_LINK_STR, CBC, temp);
                           }
                           TxPrint( "%s%s\n", CNN, CGE);
                        }
                        else                    // return info in 'select' and 'param' (Browse)
                        {
                           if ((isSymLink) || ((isHrdLink) && (HfsHardlinksAutoResolve == FALSE))
                                           || ((isFdAlias) && (HfsFldrAliasAutoResolve == FALSE)))
                           {
                              if      (isSymLink)
                              {
                                 dfsHfsResolveSymLink( location, temp);
                              }
                              else if (isHrdLink)
                              {
                                 sprintf( temp, "/%s/iNode%d", HFS_SD_HARDLINK, TxBE32( catData->d.crBsdInfo.iNodeNum));
                              }
                              else
                              {
                                 dfsHfsResolveFdAlias( aliasLsn, temp);
                              }
                              strcat( fpath, DFS_SH_LINK_STR); // indicator for symbolic link in BROWSE
                              strcat( fpath, temp);
                           }
                           dfsHfsTime2str( catData->d.crModContTime, temp);
                           TxStripAnsiCodes( text);
                           sprintf( select, "%s %s",        text, temp);
                           dfstrUllDot20( select, "", fsize, "");
                        }
                        rc = NO_ERROR;
                     }
                     else
                     {
                        rc = DFS_ST_MISMATCH;   // file-size mismatch
                     }
                  }
                  else
                  {
                     rc = DFS_ST_MISMATCH;      // percentage mismatch
                  }
               }
               else
               {
                  rc = DFS_ST_MISMATCH;         // wildcard mismatch
               }
               strcpy( param, fpath);           // return the found name
               break;

            default:
               rc = DFS_PENDING;
               break;
         }
      }
      TxFreeMem( node);
   }
   else
   {
      rc = DFS_PENDING;                         // not a catalog node with index
   }
   RETURN(rc);
}                                               // end 'dfsHfsFileInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Replace sn-list by contents of a single Directory (InoLsn, InoIdx pairs)
/*****************************************************************************/
ULONG dfsHfsMakeBrowseList                      // RET   LSN is valid INODE
(
   ULN64               leafLsn,                 // IN    Leaf-node sectornumber
   ULN64               leafIdx,                 // IN    Leaf-node record index
   char               *str,                     // IN    unused
   void               *param                    // INOUT unused
)
{
   ULONG               rc = NO_ERROR;
   ULONG               cnId;                    // cnid of folder to browse
   ULONG               parent;                  // cnid of its parent, if any
   BOOL                isDirectory;
   ULN64               thisNodeNr;              // node nr  to be handled next (auto)
   ULN64               thisNodeLsn;             // node LSN to be handled next (auto)
   ULN64               entryLsn;
   USHORT              entryIdx;
   TXLN                fileName;
   ULONG               parentParent;            // parent of parent (to get lsn/index)
   char                detailedType;            // file/Dir/symlink/hardlink/folder-alias

   ENTER();
   TRARGS(("LSN: 0x%llX, index: %llX\n", leafLsn, leafIdx));

   dfsInitList(0, "-f -P", "-d");               // optimal for menu file-recovery

   cnId = dfsHfsLsnInfo2CnId( leafLsn, DFSSNIGET( leafIdx), &detailedType);
   if (cnId != 0)
   {
      rc = dfsHfsCnId2Parent( cnId, &parent, NULL, &isDirectory);
      if ((rc == NO_ERROR) && ((detailedType == 'D') || (detailedType == 'H'))) //DIR of Folder-alias only
      {
         if (cnId > HFS_ID_RootFolder)          // add .. for parent folder
         {
            //- get leaf-Lsn and index for the parent Folder entry
            rc = dfsHfsCnId2Parent( parent, &parentParent, fileName, &isDirectory);
            if (rc == NO_ERROR)
            {
               TRACES(("parentParent CNID %x name '%s'\n", parentParent, fileName));
               if ((thisNodeNr = dfsHfsSearchCatalog( parentParent, fileName, &entryIdx, NULL)) != 0)
               {
                  TRACES(("List marked as 1ST_PARENT\n"));
                  DFSBR_SnlistFlag |= DFSBR_1ST_PARENT;

                  entryLsn = dfsHfsNode2Lsn( thisNodeNr);

                  rc = dfsAddSI2List( entryLsn, entryIdx);
               }
            }
         }
         if (rc == NO_ERROR)
         {
            ULONG            parentId = 0;      // last parent-ID evaluated
            BYTE            *node;

            if ((node = TxAlloc( hfs->Catalog.spNode, dfsGetSectorSize())) != NULL)
            {
               S_BT_NODE_DESC  *desc = (S_BT_NODE_DESC *) node;

               if (detailedType == 'H')         // hardlinked directory, folder-alias
               {
                  cnId = dfsHfsFdLsnInfo2CnId( leafLsn, DFSSNIGET( leafIdx)); // Cnid of actual directory
               }

               //- determine first leaf-node for the folder to browse
               thisNodeNr  = dfsHfsSearchCatalog( cnId, NULL, NULL, NULL);
               if (thisNodeNr != 0)
               {
                  thisNodeLsn = dfsHfsNode2Lsn( thisNodeNr);

                  TRACES(("Traverse leaf nodes from node %u, for entries with parent CNID %x\n", thisNodeNr, cnId));
                  do                            // iterate over leaf-nodes, collect entries for cnid
                  {
                     if (dfsRead( thisNodeLsn, hfs->Catalog.spNode, node) == NO_ERROR)
                     {
                        S_HFS_CATALOG_KEY *key;
                        U_HFS_CAT_RECORD  *catData; // catalog data record folder/file/thread
                        USHORT             rec;
                        USHORT             nRecs = TxBE16( desc->nRecords); // number of records

                        for (rec = 0; rec < nRecs; rec++)
                        {
                           USHORT     this = dfsHfsRecordOffset( rec,     node);
                           USHORT     keyLength;
                           USHORT     recType;  // record-type folder/file (or -thread)
                           BOOL       filtered;

                           key       = (S_HFS_CATALOG_KEY *) (node + this);
                           keyLength = TxBE16( key->keyLength);
                           parentId  = TxBE32( key->parentId);

                           catData = (U_HFS_CAT_RECORD *) (((BYTE *) key) + keyLength + 2);
                           recType = TxBE16( catData->t.crRecType);

                           TRACES(("rec:%2u recType:%hu Parent:%x\n", rec, recType, parentId));

                           if ((parentId == cnId) && ((recType == HFS_CR_File) || (recType == HFS_CR_Folder)))
                           {
                              filtered = FALSE; // entry is NOT filtered out
                              if ((str) && (strlen( str) != 0) && // option string present (possible exclude)
                                  ((HfsHardlinksAutoResolve == FALSE) || (HfsFldrAliasAutoResolve == FALSE)))
                              {
                                 if (dfsHfsLsnInfo2CnId( thisNodeLsn, rec, &detailedType) != 0)
                                 {
                                    if (strchr( str, 'L') != NULL) // exclude links, when not auto resolved
                                    {
                                       if ((detailedType == 'h') && (HfsHardlinksAutoResolve == FALSE))
                                       {
                                          TRACES(("filter EXCLUDED hardlinked file\n"));
                                          filtered = TRUE;
                                       }
                                       if ((detailedType == 'H') && (HfsFldrAliasAutoResolve == FALSE))
                                       {
                                          TRACES(("filter EXCLUDED Folder alias\n"));
                                          filtered = TRUE;
                                       }
                                    }
                                 }
                              }

                              if (!filtered)    // filter on filename
                              {
                                 TxMacUniStr2Ascii( &(key->name), fileName); // name for this entry
                                 TRACES(("Filter check: '%s'\n", fileName));

                                 if      ((parentId == HFS_ID_RootFolder) && (hfs->hlMetaFolderId  != 0) &&
                                    HfsHardlinksAutoResolve && (strcmp( fileName, HFS_SD_HARDLINK) == 0)  )
                                 {
                                    TRACES(("filter hardlinks meta DIR\n"));
                                    filtered = TRUE; // don't show the meta directory
                                 }
                                 else if ((parentId == HFS_ID_RootFolder) && (hfs->fdAliasFolderId != 0) &&
                                    HfsFldrAliasAutoResolve && (strcmp( fileName, HFS_SD_FLDALIAS) == 0)  )
                                 {
                                    TRACES(("filter folderalias meta DIR\n"));
                                    filtered = TRUE; // don't show the meta directory
                                 }
                                 else if (dfsa->browseShowHidden == FALSE)
                                 {
                                    if (fileName[0] == '.')
                                    {
                                       TRACES(("filter a DOT hidden file\n"));
                                       filtered = TRUE; // don't show HIDDEN directories/files
                                    }
                                 }
                              }
                              if (!filtered)
                              {
                                 rc = dfsAddSI2List( thisNodeLsn, rec); // add to list
                              }
                           }
                           else if (parentId > cnId) // rest is larger, quit linear search
                           {
                              TRACES(("Rest is larger, quit search ...\n"));
                              break;
                           }
                        }
                     }
                     if (parentId == cnId)      // still matching parent-ID, next leaf node
                     {
                        thisNodeNr  = TxBE32( desc->lNext);
                        thisNodeLsn = dfsHfsNode2Lsn( thisNodeNr);
                     }
                     else                       // last Node (for this parent) handled
                     {
                        thisNodeNr  = 0;        // terminate ...
                     }
                  } while ((rc == NO_ERROR) && (thisNodeNr != 0) && !TxAbort());
               }
               TxFreeMem( node);
            }
            else
            {
               rc = DFS_ALLOC_ERROR;
            }
         }
      }
      else
      {
         rc = DFS_ST_MISMATCH;                  // not a directory
      }
   }
   RETURN(rc);
}                                               // end 'dfsHfsMakeBrowseList'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Find FDALIAS actual-DIR CnID for a given (Node) LSN and index info value
/*****************************************************************************/
ULONG dfsHfsFdLsnInfo2CnId                      // RET   CnID, or 0 when failed
(
   ULN64               nodeLsn,                 // IN    node lsn
   ULONG               index                    // IN    node index
)
{
   ULONG               rc = 0;
   BYTE                st;
   USHORT              recType;
   BYTE               *node;
   U_HFS_CAT_RECORD   *catData  = NULL;
   S_HFS_ALIS_FORK    *alisData = NULL;
   ULN64               lsn;

   ENTER();
   TRACES(("Node-Lsn: 0x0%llx index: 0x%x\n", nodeLsn, index));

   if (dfsHfsReadChkCatNode( nodeLsn, &st, &node) == NO_ERROR)
   {
      if (st == ST_HFSCLF)                      // must be a leaf record
      {
         recType = dfsHfsNodeIdx2Rec( node, index, &catData);
         if (recType == HFS_CR_File)            // alias record is a FILE!
         {
            if ((catData->d.f.type == HFS_FT_FDALIAS) && (catData->d.f.creator == HFS_FC_FDALIAS))
            {
               if ((alisData = TxAlloc( 1, dfsGetSectorSize())) != NULL)
               {
                  lsn = dfsHfsBlock2Lsn( TxBE32( catData->file.crRsrcFork.extents[0].startBlock));
                  if (dfsRead( lsn, 1, (BYTE *) alisData) == NO_ERROR)
                  {
                     if ((alisData->al_signature        == HFS_SIGN_ALIAS)   && // validate the ALIS
                         (TxBE16( alisData->al_hfsplus) == HFS_PLUS_SIGNATURE)) // resourcefork data
                     {
                        TXTM    tbuf;
                        ULONG   nodeNr;
                        USHORT  index;

                        TxFreeMem( node);       // free initial fdalias node
                        sprintf( tbuf, "dir_%d", TxBE32( alisData->al_dirRef));
                        nodeNr = dfsHfsSearchCatalog( hfs->fdAliasFolderId, tbuf, &index, &node);
                        if ((nodeNr != 0) && ((nodeNr != L32_NULL)))
                        {
                           recType = dfsHfsNodeIdx2Rec( node, index, &catData);
                           if (recType == HFS_CR_Folder) // resolved record is a DIRECTORY
                           {
                              rc = TxBE32( catData->d.crCnID);
                           }
                        }
                     }
                  }
                  TxFreeMem( alisData);
               }
               else
               {
                  rc = DFS_ALLOC_ERROR;
               }
            }
         }
      }
      TxFreeMem( node);
   }
   RETURN(rc);
}                                               // end 'dfsHfsFdLsnInfo2CnId'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get allocation information for file-DATA into integrated-SPACE structure
// Supports automatic resolve of hardlinks, when enabled in the (menu) option
// For an empty file with a resource fork, will return the resourcefork alloc
/*****************************************************************************/
ULONG dfsHfsGetAllocSpace
(
   ULN64               leafLsn,                 // IN    Leaf-node sectornumber
   ULN64               leafIdx,                 // IN    Leaf-node record index
   char               *str,                     // IN    unused
   void               *param                    // INOUT Integrated SPACE
)
{
   ULONG               rc = NO_ERROR;
   DFSISPACE          *ispace = (DFSISPACE *) param;
   BYTE               *node = NULL;             // Catalog node buffer
   BYTE                st;                      // sector type (node type)
   USHORT              offset;

   ENTER();
   TRARGS(("LSN: 0x%llX, index: %llX\n", leafLsn, leafIdx));

   ispace->space    = NULL;                     // init to empty object
   ispace->chunks   = 0;
   ispace->byteSize = 0;

   if ((rc = dfsHfsReadChkCatNode( leafLsn, &st, &node)) == NO_ERROR)
   {
      if ((offset = dfsHfsRecordOffset( DFSSNIGET( leafIdx), node)) != 0)
      {
         S_HFS_CATALOG_KEY *key       = (S_HFS_CATALOG_KEY *) (node + offset);
         USHORT             keyLength = TxBE16( key->keyLength);
         U_HFS_CAT_RECORD  *catData   = (U_HFS_CAT_RECORD *) (((BYTE *) key) + keyLength + 2);
         USHORT             recType   = TxBE16( catData->t.crRecType);

         if (recType == HFS_CR_File)
         {
            BOOL isHrdLink = ((catData->d.f.type == HFS_FT_HRDLINK) && (catData->d.f.creator == HFS_FC_HRDLINK));
            BOOL isFdAlias = ((catData->d.f.type == HFS_FT_FDALIAS) && (catData->d.f.creator == HFS_FC_FDALIAS));

            if (isHrdLink)
            {
               TXTM               tbuf;         // hardlink iNode name buffer

               if ((hfs->hlMetaFolderId != 0) && (HfsHardlinksAutoResolve == TRUE))
               {
                  ULONG              nodeNr;
                  USHORT             index;     // index in new node

                  TxFreeMem( node);         // free original node, to be replaced with resolved one

                  sprintf( tbuf, "iNode%d", TxBE32( catData->d.crBsdInfo.iNodeNum));
                  nodeNr = dfsHfsSearchCatalog( hfs->hlMetaFolderId, tbuf, &index, &node);
                  if ((nodeNr != 0) && ((nodeNr != L32_NULL)))
                  {
                     if (dfsHfsNodeIdx2Rec( node, index, &catData) == HFS_CR_File)
                     {
                        TRACES(("Resolved hardlink, morphed catData to that record for sizes\n"));
                     }
                  }
               }
               else                             // no meta-dir/auto-resolve; return the link filename
               {
                  sprintf( tbuf, "/%s/iNode%d", HFS_SD_HARDLINK, TxBE32( catData->d.crBsdInfo.iNodeNum));
                  ispace->byteSize = strlen( tbuf) + 1;
                  ispace->chunks   = 0;         // indicate it is a single memory-area, no chunks
                  if ((ispace->space = TxAlloc( 1, ispace->byteSize)) != NULL)
                  {
                     strcpy( (char *) ispace->space, tbuf); // results in recovery like a symlink
                  }
                  else
                  {
                     rc = DFS_ALLOC_ERROR;
                  }
                  catData = NULL;               // signal no regular SPACE allocation
               }
            }
            else if (isFdAlias)
            {
               //- return the folder-alias linkname as a single allocation ? (or keep whle resource-fork)
            }

            if (catData != NULL)                // get SPACE for normal file or resolved hardlink
            {
               ispace->byteSize = TxBE64( catData->file.crDataFork.logicalSize); // datafork size
               if (ispace->byteSize != 0)
               {
                  rc = dfsHfsFork2Space( &(catData->file.crDataFork), &(ispace->chunks), &(ispace->space));
               }
               else
               {
                  ispace->byteSize = TxBE64( catData->file.crRsrcFork.logicalSize); // resourcefork size
                  if (ispace->byteSize != 0)
                  {
                     rc = dfsHfsFork2Space( &(catData->file.crRsrcFork), &(ispace->chunks), &(ispace->space));
                  }
               }
            }
         }
      }
      TxFreeMem( node);
   }
   RETURN (rc);
}                                               // end 'dfsHfsGetAllocSpace'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Find full PATH to Root-directory, starting at some Node LSN and index info
/*****************************************************************************/
ULONG dfsHfsLsnInfo2Path                        // RET   result
(
   ULN64               start,                   // IN    node lsn
   ULONG               index,                   // IN    node index
   char               *path                     // OUT   combined path string
)
{
   ULONG               rc = NO_ERROR;
   ULONG               cnId;
   ULONG               parent;
   TXLN                thisName;                // name for current component
   BOOL                isDirectory;
   ULONG               sanity = 100;

   ENTER();
   if (path)
   {
      *path = '\0';
   }

   cnId = dfsHfsLsnInfo2CnId( start, index, NULL);
   while ((cnId != 0) && (rc == NO_ERROR) && sanity--)
   {
      if (cnId <= 2)                            // root, or worse :)
      {
         strcpy( thisName, "");                 // don't want the volumename!
         isDirectory = TRUE;
         parent = 0;                            // terminate after this ...
      }
      else
      {
         rc = dfsHfsCnId2Parent( cnId, &parent, thisName, &isDirectory);
      }
      if (rc == NO_ERROR)
      {
         if (isDirectory)
         {
            strcat( thisName, FS_PATH_STR);
         }
         strcat( thisName, path);               // append previous parts
         strcpy( path, thisName);               // and copy back ...

         cnId = parent;                         // to next path component
      }
   }
   TRACES(("Found path to root for 0x%llX+%4.4x:'%s'\n", start, index, path));
   RETURN(rc);
}                                               // end 'dfsHfsLsnInfo2Path'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Find related CnID for given (Node) LSN and index info value; optional dType
/*****************************************************************************/
ULONG dfsHfsLsnInfo2CnId                        // RET   CnID, or 0 when failed
(
   ULN64               nodeLsn,                 // IN    node lsn
   ULONG               index,                   // IN    node index
   char               *dType                    // OUT   detailed type, or NULL
)                                               //       D,H,f,h,s
{
   ULONG               rc = 0;
   BYTE                st;
   BYTE               *node;
   U_HFS_CAT_RECORD   *catData;
   USHORT              recType;

   ENTER();
   TRACES(("Node-Lsn: 0x0%llx index: 0x%x\n", nodeLsn, index));

   if (dfsHfsReadChkCatNode( nodeLsn, &st, &node) == NO_ERROR)
   {
      if (st == ST_HFSCLF)
      {
         recType = dfsHfsNodeIdx2Rec( node, index, &catData);
         if ((recType == HFS_CR_File) || (recType == HFS_CR_Folder))
         {
            rc = TxBE32( catData->d.crCnID);

            if (dType != NULL)                  // need detailed record type ?
            {
               BOOL isSymLink = ((catData->d.f.type == HFS_FT_SYMLINK) && (catData->d.f.creator == HFS_FC_SYMLINK));
               BOOL isHrdLink = ((catData->d.f.type == HFS_FT_HRDLINK) && (catData->d.f.creator == HFS_FC_HRDLINK));
               BOOL isFdAlias = ((catData->d.f.type == HFS_FT_FDALIAS) && (catData->d.f.creator == HFS_FC_FDALIAS));

               *dType = (recType == HFS_CR_Folder) ? 'D' : // regular Directory
                                       (isSymLink) ? 's' : // symlink
                                       (isFdAlias) ? 'H' : // Hardlinked directory (folder alias)
                                       (isHrdLink) ? 'h' : // hardlinked file
                                                     'f' ; // regular file
            }
         }
      }
      TxFreeMem( node);
   }
   RETURN(rc);
}                                               // end 'dfsHfsLsnInfo2CnId'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Find catalog-record for a given Node-structure and index info value
/*****************************************************************************/
USHORT dfsHfsNodeIdx2Rec                        // RET   recordtype, or 0 when failed
(
   BYTE               *node,                    // IN    node structure
   ULONG               index,                   // IN    node index
   U_HFS_CAT_RECORD  **rec                      // OUT   file/folder catalog record
)
{
   USHORT              rc = 0;
   USHORT              offset;

   ENTER();

   if ((offset = dfsHfsRecordOffset( index, node)) != 0)
   {
      S_HFS_CATALOG_KEY *key       = (S_HFS_CATALOG_KEY *) (node + offset);
      USHORT             keyLength = TxBE16( key->keyLength);
      U_HFS_CAT_RECORD  *catData   = (U_HFS_CAT_RECORD *) (((BYTE *) key) + keyLength + 2);

      *rec = catData;                           // return the catalog record

      rc = TxBE16( catData->t.crRecType);
   }
   RETURN(rc);
}                                               // end 'dfsHfsNodeIdx2Rec'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate Block-nr to LSN, generic interface
/*****************************************************************************/
ULN64 dfsHfsCl2Lsn                              // RET   LSN for this block
(
   ULN64               block,                   // IN    block number
   ULN64               d2,                      // IN    dummy
   char               *p1,                      // IN    dummy
   void               *p2                       // IN    dummy
)
{
   ULN64               lsn = block;

   lsn = block * hfs->SectorsPerBlock;
   return (lsn);
}                                               // end 'dfsHfsCl2Lsn'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate LSN to Block-nr, generic interface
/*****************************************************************************/
ULN64 dfsHfsLsn2Cl                              // RET   block for this LSN
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *p1,                      // IN    dummy
   void               *p2                       // IN    dummy
)
{
   ULN64               block = 0;

   block = lsn / hfs->SectorsPerBlock;
   return (block);
}                                               // end 'dfsHfsLsn2Cl'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Determine allocation-bit for specified LSN, ALLOCATED beyond last block!
/*****************************************************************************/
ULONG dfsHfsAllocated                           // RET   LSN is allocated
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULN64               asn = dfstAreaD2Part( DFSTORE, lsn);  //- Area aware sn

   if ((hfs->Bm.Map != NULL) && (asn < hfs->Bm.LimitLsn))    //- inside the bitmap
   {
      if (asn < (hfs->SectorCount))             // within valid block area
      {
         return((ULONG) dfsHfsBitmapCache( dfsHfsLsn2Block( asn), NULL));
      }
      else                                      // return ALLOCATED for any
      {                                         // lsn beyond last block
         return( 1);                            // to avoid CHECK errors
      }                                         // on non-standard bitmaps
   }
   else
   {
      return(DFS_PSN_LIMIT);
   }
}                                               // end 'dfsHfsAllocated'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Align R/W cache to position for block, and get current allocation-bit
/*****************************************************************************/
BOOL dfsHfsBitmapCache                          // RET   block is allocated
(
   ULONG               bl,                      // IN    Block number
   ULONG              *cp                       // OUT   position in cache
)                                               //       or L64_NULL if invalid
{
   ULONG               rc = NO_ERROR;
   BOOL                al = TRUE;               // default allocated
   ULONG               rmp;                     // relative dmap page in map
   ULONG               u8pos = L32_NULL;        // u8 position in map
   ULONG               alloc_u8;

   if (hfs->BlockBits != 0)                     // avoid divide by zero
   {
      rmp = bl / hfs->BlockBits;                // find relative page for bl

      if (rmp != hfs->Bm.Rbmap)                 // other relative map page
      {
         if (hfs->Bm.Dirty)                     // need to flush changes ?
         {
            rc = dfsHfsBitmapFlush( FALSE);     // flush, but keep cache
         }
         if (rc == NO_ERROR)
         {
            hfs->Bm.Rbmap = rmp;
            TRACES(("Read bitmap page: %u for block %u\n", rmp, bl));
            dfsSspaceReadFilePart( hfs->Bm.File.chunks, hfs->Bm.File.space,
                                   hfs->SectorsPerBlock * rmp,
                                   hfs->SectorsPerBlock, (BYTE *) hfs->Bm.Map);
         }
      }
      if (rc == NO_ERROR)
      {
         u8pos    = (bl %  hfs->BlockBits) / 8; // byte index in map
         alloc_u8 = hfs->Bm.Map[ u8pos];
         if (((alloc_u8 << (bl & 0x07)) & 0x80) == 0) // MSB is lowest!
         {
            al = FALSE;
         }
      }
      if (cp != NULL)                           // return byte position too ?
      {
         *cp = u8pos;
      }
   }
   return (al);
}                                               // end 'dfsHfsBitmapCache'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Extract BitMap allocation S_SPACE structure and initialize the BitMap cache
// Note: Area aware to allow usage from FDISK mode too
/*****************************************************************************/
ULONG dfsHfsBitMapInit                          // RET   rc = 0 if type match
(
   void
)
{
   ULONG               rc = 0;                  // rc, sector match

   ENTER();

   hfs->Bm.LimitLsn = 0;                        // signal no BitMaps present

   if ((hfs->Bm.Map = TxAlloc( hfs->SectorsPerBlock, dfsGetSectorSize())) != NULL)
   {
      if (hfs->sup && hfs->sup->AllocationFile.logicalSize != 0)
      {
         //- create Area-corrected Sspace, so ReadFilePart needs no changes
         rc = dfsHfsFork2Space( &(hfs->sup->AllocationFile), &hfs->Bm.File.chunks, &hfs->Bm.File.space);
         if ((rc == NO_ERROR) && (hfs->Bm.File.chunks != 0))
         {
            rc = dfsSspaceReadFilePart( hfs->Bm.File.chunks,
                                        hfs->Bm.File.space,
                                     0, hfs->SectorsPerBlock,
                               (BYTE *) hfs->Bm.Map);
            if (rc == NO_ERROR)
            {
               hfs->Bm.LimitLsn = dfsSspaceSectors( FALSE, hfs->Bm.File.chunks, hfs->Bm.File.space) * hfs->BlockBits;
               hfs->Bm.Dirty    = FALSE;
               hfs->Bm.Rbmap    = 0;
            }
         }
      }
      else
      {
         TxPrint( "Bitmap allocation file has ZERO length!\n");
         rc = DFS_BAD_STRUCTURE;
      }
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsHfsBitMapInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Flush BitMap cache when Dirty, and optional free its resources
/*****************************************************************************/
ULONG dfsHfsBitmapFlush                         // RET   rc
(
   BOOL                terminate                // IN    terminate cache too
)
{
   ULONG               rc = NO_ERROR;           // rc

   ENTER();

   if (hfs->Bm.Dirty)                          // need to flush changes ?
   {
      TRACES(( "dirty, flush it ...\n"));

      //- to be implemented, write back the bitmap page

      hfs->Bm.Dirty = FALSE;                    // mark it clean again
   }
   else
   {
      TRACES(( "not dirty ...\n"));
   }
   if (terminate)
   {
      TxFreeMem( hfs->Bm.File.space);
      TxFreeMem( hfs->Bm.Map);

      hfs->Bm.Rbmap    = 0;
      hfs->Bm.LimitLsn = 0;
   }
   RETURN (rc);
}                                               // end 'dfsHfsBitmapFlush'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set allocate status for LSN to specified value (beyond FS, to end of bitmap)
/*****************************************************************************/
ULONG dfsHfsSetAlloc                            // RET   LSN allocation set
(
   ULN64               lsn,                     // IN    LSN
   char               *value,                   // IN    NULL = not allocated
   void               *ref                      // IN    dummy
)
{
   if ((hfs->Bm.Map != NULL) && (lsn < hfs->Bm.LimitLsn)) // inside the bitmap
   {
      return( dfsHfsSetBlock( dfsHfsLsn2Block( lsn), (value != NULL)));
   }
   else
   {
      return( DFS_PSN_LIMIT);
   }
}                                               // end 'dfsHfsSetAlloc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set allocate status for BLOCK to specified value
/*****************************************************************************/
ULONG dfsHfsSetBlock                            // RET   BL  allocation set
(
   ULONG               bl,                      // IN    Block
   BOOL                value                    // IN    SET allocation bit
)
{
   ULONG               rc = NO_ERROR;
   ULONG               bytePos;
   ULONG               allocByte;
   ULONG               bit_mask = (1 << (bl & 0x07));

   dfsHfsBitmapCache( bl, &bytePos);
   if (bytePos != L32_NULL)                     // block position valid ?
   {
      allocByte = hfs->Bm.Map[ bytePos];

      if (value)                                // set the allocation bit
      {
         hfs->Bm.Map[ bytePos] = allocByte |  bit_mask;
      }
      else                                      // reset allocation bit
      {
         hfs->Bm.Map[ bytePos] = allocByte & ~bit_mask;
      }
      TRACES(( "%s cache-u8 at 0x%4.4x = -%6.6x- for bl:%8.8x "
                  "from 0x%2.2hx to 0x%2.2hx\n",
                (value) ? "Set" : "Clr", bytePos, bl / 32, bl,
                allocByte, hfs->Bm.Map[ bytePos]));

      hfs->Bm.Dirty = TRUE;                     // mark cache dirty
   }
   else
   {
      rc = DFS_PSN_LIMIT;
   }
   return (rc);
}                                               // end 'dfsHfsSetBlock'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Extract Catalog allocation S_SPACE structure and cache the RootNode
/*****************************************************************************/
ULONG dfsHfsCatalogInit
(
   void
)
{
   ULONG               rc = 0;                  // rc, sector match
   S_BT_HDR_NODE      *rnSector = NULL;         // first sector, to get spNode value

   ENTER();

   if ((rnSector = TxAlloc( 1, dfsGetSectorSize())) != NULL)
   {
      if (hfs->sup && hfs->sup->CatalogFile.logicalSize != 0)
      {
         //- create Area-corrected Sspace, so ReadFilePart needs no changes
         rc = dfsHfsFork2Space( &(hfs->sup->CatalogFile), &hfs->Catalog.File.chunks, &hfs->Catalog.File.space);
         if ((rc == NO_ERROR) && (hfs->Bm.File.chunks != 0))
         {
            rc = dfsSspaceReadFilePart( hfs->Catalog.File.chunks,
                                        hfs->Catalog.File.space,
                                     0, 1,  (BYTE *) rnSector);
            if (rc == NO_ERROR)
            {
               hfs->Catalog.bpNode = TxBE16( rnSector->hdr.bthNodeSize);
               hfs->Catalog.spNode = hfs->Catalog.bpNode / dfsGetSectorSize();

               if ((hfs->Catalog.Btree = TxAlloc( hfs->Catalog.spNode, dfsGetSectorSize())) != NULL)
               {
                  rc = dfsSspaceReadFilePart( hfs->Catalog.File.chunks,
                                              hfs->Catalog.File.space,
                                           0, hfs->Catalog.spNode, (BYTE *) hfs->Catalog.Btree);
               }
               else
               {
                  rc = DFS_ALLOC_ERROR;
               }
            }
         }
      }
      else
      {
         TxPrint( "Catalog file has ZERO length!\n");
         rc = DFS_BAD_STRUCTURE;
      }
      TxFreeMem( rnSector);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsHfsCatalogInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Search thread-record for CnID, returning parent-CnID / item-name / isDir
/*****************************************************************************/
ULONG dfsHfsCnId2Parent                         // RET   result
(
   ULONG               cnId,                    // IN    CNID of item to search
   ULONG              *parent,                  // OUT   parent-CnID       or NULL
   char               *name,                    // OUT   NAME of item, ""  or NULL
   BOOL               *isDir                    // OUT   item is DIR flag, or NULL
)
{
   ULONG               rc = DFS_NOT_FOUND;      // function return, default not found
   USHORT              index;
   USHORT              offset;
   BYTE               *node;

   ENTER();

   if (dfsHfsSearchCatalog( cnId, NULL, &index, &node) != 0)
   {
      if ((offset = dfsHfsRecordOffset( index, node)) != 0)
      {
         S_HFS_CATALOG_KEY *key       = (S_HFS_CATALOG_KEY *) (node + offset);
         USHORT             keyLength = TxBE16( key->keyLength);
         U_HFS_CAT_RECORD  *catData   = (U_HFS_CAT_RECORD *) (((BYTE *) key) + keyLength + 2);
         USHORT             recType   = TxBE16( catData->t.crRecType);

         if ((recType == HFS_CR_Dthread) || (recType == HFS_CR_Fthread))
         {
            if (name != NULL)
            {
               TxMacUniStr2Ascii( &(catData->t.name), name);
            }
            if (isDir != NULL)
            {
               *isDir = (recType == HFS_CR_Dthread);
            }
            if (parent != NULL)
            {
               *parent = TxBE32( catData->t.parentId);
            }
            rc = NO_ERROR;
         }
      }
   }
   TRACES(("parent: %x for cnId: %x = '%s'\n", (parent) ? *parent : 0, cnId, (name) ? name : ""));
   RETURN (rc);
}                                               // end 'dfsHfsCnId2Parent'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Search catalog B-tree for specified key, get Leafnode + optional rec-index
/*****************************************************************************/
ULONG dfsHfsSearchCatalog                       // RET   Cat NODE number (leaf)
(
   ULONG               parent,                  // IN    CNID of parent to search
   char               *name,                    // IN    NAME of item, "" or NULL
   USHORT             *record,                  // OUT   record-index,    or NULL
   BYTE              **leaf                     // OUT   ptr to leafnode, or NULL
)
{
   ULONG               rc = 0;                  // function return, default not found
   BYTE               *node;
   S_BT_NODE_DESC     *desc;                    // node description header
   ULN64               lsn;
   ULONG               sanity = BT_SANE_LEVELS; // limit levels of Indexes to search

   ENTER();
   TRACES(("parent: 0x%8.8x  name: '%s'\n", parent, (name) ? name : ""));

   if (hfs->Catalog.Btree != NULL)
   {
      if ((node = TxAlloc( hfs->Catalog.spNode, dfsGetSectorSize())) != NULL)
      {
         ULONG         nodeNr = TxBE32( hfs->Catalog.Btree->hdr.bthRootNode);

         desc = (S_BT_NODE_DESC *) node;
         do
         {
            TRACES(("node:0x%8.8x\n", nodeNr));
            if ((lsn = dfsHfsNode2Lsn( nodeNr)) != L64_NULL)
            {
               if (dfsRead( lsn, hfs->Catalog.spNode, node) == NO_ERROR)
               {
                  if (desc->nKind == BT_IndexNode)
                  {
                     //- iterate over index-records; find LARGEST key less or equal to reference

                     S_HFS_CATALOG_KEY *rKey;
                     USHORT             rec;
                     USHORT             nRecs = TxBE16( desc->nRecords); // number of records

                     for (rec = 0; rec < nRecs; rec++)
                     {
                        USHORT     this = dfsHfsRecordOffset( rec,     node);
                        USHORT     keyLength;
                        int        keyComp;     // key compare result

                        TRACES(("Index record: %3u  offset:0x%4.4hx\n", rec, this));

                        rKey    = (S_HFS_CATALOG_KEY *) (node + this);
                        keyComp = dfsHfsCatalogKeyCompare( rKey, parent, name);

                        //- to be refined, logic does not seem right here
                        //- check with simple values (integer?)
                        if (keyComp <= 0)       // record-key is less or equal, node is candidate
                        {
                           keyLength = TxBE16( rKey->keyLength);
                           nodeNr    = TxBE32( *((ULONG *) (node + this + keyLength + 2)));
                        }
                        else                    // record key was larger
                        {
                           TRACES(("next level node: 0x%8.8x\n", nodeNr));
                           break;               // nodeNr is the branch to follow
                        }
                     }
                  }
                  else
                  {
                     TRACES(("Exit index-searching on (leaf) node: 0x%8.8x\n", nodeNr));
                  }
               }
            }
         } while (sanity-- && (desc->nKind == BT_IndexNode));

         if (desc->nKind == BT_LeafNode)
         {
            //- iterate over leaf-records (file, folder) until key(mis)match

            S_HFS_CATALOG_KEY *rKey;
            USHORT             rec;
            USHORT             nRecs = TxBE16( desc->nRecords); // number of records

            for (rec = 0; rec < nRecs; rec++)
            {
               USHORT     this = dfsHfsRecordOffset( rec,     node);
               int        keyComp;              // key compare result

               TRACES(("Leaf  record: %3u  offset:0x%4.4hx\n", rec, this));

               rKey    = (S_HFS_CATALOG_KEY *) (node + this);
               keyComp = dfsHfsCatalogKeyCompare( rKey, parent, name);

               if (keyComp == 0)                // key match found
               {
                  TRACES(("Leaf record found: %u\n", rec));

                  if (record != NULL)
                  {
                     *record = rec;
                  }
                  if (leaf != NULL)
                  {
                     *leaf = node;
                  }
                  rc = nodeNr;                  // return leaf node number
                  break;
               }
               else if (keyComp > 0)            // record key was larger, abort
               {
                  TRACES(("Leaf node abort search on record: %u\n", rec));

                  rc = L32_NULL;                // signal search failed ...
                  break;                        // wanted key is not present
               }
            }
         }
         if ((leaf == NULL) || (*leaf != node)) // node not returned to caller
         {
            TxFreeMem( node);
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsHfsSearchCatalog'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Compare catalog-record-key to supplied parent and optional name
/*****************************************************************************/
int dfsHfsCatalogKeyCompare                     // RET   0 = EQ; 1 = rKey larger
(                                               //              -1 = rKey smaller
   S_HFS_CATALOG_KEY  *recordKey,               // IN    record key
   ULONG               parent,                  // IN    parent ID to compere
   char               *name                     // IN    name to compare, or NULL
)
{
   int                 rc = 0;                  // function return, default to 'match'
   USHORT              keyLength;
   ULONG               parentId;
   TXLN                keyName;

   ENTER();

   keyLength = TxBE16( recordKey->keyLength);
   parentId  = TxBE32( recordKey->parentId);

   TRACES(("parent:0x%8.8x Record-parent:0x%8.8x\n", parent, parentId));
   if (parentId == parent)                      // match for primary key part ?
   {
      if ((name != NULL) && (*name != 0))       // secondary key non-empty ?
      {
         TxMacUniStr2Ascii( &(recordKey->name), keyName);

         TRACES(("name: '%s'  Record-string: '%s'\n", name, keyName));

         if (strcmp( name, HFS_SD_HARDLINK) == 0)
         {
            if (strcmp( keyName, HFS_SD_HARDLINK) == 0)
            {
               rc = 0;                          // matched on special HARDLINK dir name
            }
            else
            {
               rc = -1;
            }
         }
         else if (recordKey->name.unicode[0] != 0) // record-key not another 'special' ?
         {
            if (hfs->Catalog.Btree->hdr.bthCompareType == HFS_CaseFolding)
            {
               rc = strcasecmp( keyName, name); // case insensitive
            }
            else
            {
               rc = strcmp(     keyName, name); // case sensitive
            }
            TRACES(("name compare result: %d\n", rc));
         }
         else if (recordKey->name.length != 0)  // record name is not empty, but starts with ZERO
         {
            rc =  1;                            // recordKey larger than reference, by definition
            TRACES(("recordKey has leading ZERO, larger by definition\n"));
         }
         else                                   // keyname not empty, but record-key IS
         {
            rc = -1;                            // recordKey smaller than reference, by definition
            TRACES(("recordKey is EMPTY, smaller by definition\n"));
         }
      }
      else if (recordKey->name.length != 0)     // key-name is empty, record name is not
      {
         rc =  1;                               // recordKey larger  than reference
      }
   }
   else if (parentId > parent)
   {
      rc =  1;                                  // recordKey larger  than reference
   }
   else
   {
      rc = -1;                                  // recordKey smaller than reference
   }
   RETURN (rc);
}                                               // end 'dfsHfsCatalogKeyCompare'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate Catalog-Node-Number to logical sector nr
/*****************************************************************************/
ULN64 dfsHfsNode2Lsn                           // RET   Logical sector 64bit
(
   ULONG               nodeNr                   // IN    node number
)
{
   ULN64               lsn = L64_NULL;

   ENTER();
   TRARGS(("nodeNr: %X\n", nodeNr));

   if ((hfs->Catalog.File.space) && (hfs->Catalog.Btree))
   {
      lsn = dfsSspaceRsn2Lsn( hfs->Catalog.File.chunks,
                              hfs->Catalog.File.space,
                              hfs->Catalog.spNode * nodeNr);
   }
   RETN64 (lsn);
}                                               // end 'dfsHfsNode2Lsn'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate (Catalog Node sector) LSN to Catalog NODE-Number
/*****************************************************************************/
ULONG  dfsHfsLsn2Node                           // RET   Catalog NODE-Number
(
   ULN64               lsn                      // IN    Logical sector nr
)
{
   ULONG               cnid = L32_NULL;
   ULN64               rsn;

   ENTER();
   TRARGS(("lsn: %llX\n", lsn));

   if ((hfs->Catalog.File.space) && (hfs->Catalog.Btree))
   {
      rsn = dfsSspaceLsn2Rsn( hfs->Catalog.File.chunks,
                              hfs->Catalog.File.space, lsn);
      if (rsn != L64_NULL)
      {
         cnid = rsn / hfs->Catalog.spNode;
      }
   }
   RETURN (cnid);
}                                               // end 'dfsHfsLsn2Node'
/*---------------------------------------------------------------------------*/

// to be refined, could implement caching for (a few?) nodes read
/*****************************************************************************/
// HFS read and check type for an HFS+ Catalog Node from its sector number
/*****************************************************************************/
ULONG dfsHfsReadChkCatNode
(
   ULN64               lsn,                     // IN    Node LSN
   BYTE               *stype,                   // OUT   Node type
   BYTE              **catNode                  // OUT   Catalog Node structure
)
{
   ULONG               rc = NO_ERROR;
   BYTE                st = ST_UDATA;
   BYTE               *node;

   ENTER();
   TRARGS(("Inode LSN: %llX\n", lsn));

   if ((node = TxAlloc( hfs->Catalog.spNode, dfsGetSectorSize())) != NULL)
   {
      if ((rc = dfsRead( lsn, hfs->Catalog.spNode, node)) == NO_ERROR)
      {
         switch ((st = dfsIdentifySector( lsn, 0, node)))
         {
            case ST_HFSCAT:
            case ST_HFSCIN:
            case ST_HFSCLF:                     // any node type is OK
            case ST_HFSCMP:
            case ST_HFSCHD:
               break;

            default:
               rc = DFS_ST_MISMATCH;
               break;
         }
      }
      if (rc != NO_ERROR)
      {
         TxFreeMem( node);
      }
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   *catNode = node;
   *stype = st;
   RETURN(rc);
}                                               // end 'dfsHfsReadChkCatNode'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// HFS filesystem, display THREAD-style header/footer
/*****************************************************************************/
void dfsHfsThreadHeader
(
   ULONG               items                    // IN    DIR-lines shown (0 = TOP)
)
{
   TXLN                line;

   sprintf( line, " ======= ==== ====== ============= ============ ============\n");
   if (items > 0)
   {
      TxPrint( line);
   }
   TxPrint(       " Nr      rec# R-Type this CnID     Parent-CnID  Name\n");
   if (items == 0)
   {
      TxPrint( line);
   }
}                                               // end 'dfsHfsThreadHeader'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// HFS filesystem, display DIR-style header/footer
/*****************************************************************************/
void dfsHfsDirHeader
(
   char               *lead,                    // IN    lead text-string     (20)
   ULONG               items                    // IN    DIR-lines shown (0 = TOP)
)
{
   TXLN                line;

   sprintf( line, " ==================== =================== ==== =================== ==================== ====================\n");
   if (items > 0)
   {
      TxPrint( line);
   }
   TxPrint(                         " %s Creation/LastAccess Attr Modified / Filename -> Link     Filesize   Resource-fork size\n", lead);
   if (items == 0)
   {
      TxPrint( line);
   }
}                                               // end 'dfsHfsDirHeader'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display HFS Catalog-record entry on two lines, in 'dfsee directory' format
// Supports automatic resolve of hardlinks, when enabled in the (menu) option
/*****************************************************************************/
void dfsHfsShowCatRecord
(
   BYTE               *keyRec,                  // IN    key followed by record
   BYTE                bg                       // IN    current background
)
{
   S_HFS_CATALOG_KEY  *key = (S_HFS_CATALOG_KEY *) keyRec;
   S_HFS_FILE         *file;                    // record, as ORIGINAL file-record
   S_HFS_FILE         *fnode;                   // same or hardlinked, as file-record  (size info)
   S_HFS_DIRINFO      *dir;                     // same or hardlinked, as directory record (dates)
   USHORT              keyLength;
   USHORT              recType;                 // record-type folder/file (or -thread)
   USHORT              fileMode;
   TXTM                tbuf;
   TXLN                text;
   ULN64               forkSize;
   BOOL                isSymLink;               // symlink, linkname is in the data-fork
   BOOL                isHrdLink;               // hardlink, link-ref crBsdInfo.iNodeNum -> File meta directory
   BOOL                isFdAlias;               // Folder alias, linked to numbered folder in DIR meta-directory
   BYTE               *node = NULL;             // optional node, used to resolve hardlinks

   ENTER();

   keyLength = TxBE16( key->keyLength);
   file      = (S_HFS_FILE *) (keyRec + keyLength + 2);
   fnode     = file;                            // fileinfo, starting as the original record
   dir       = (S_HFS_DIRINFO *) file;          // dirinfo,  starting as the original record
   recType   = TxBE16( dir->crRecType);
   isSymLink = ((file->dir.f.type == HFS_FT_SYMLINK) && (file->dir.f.creator == HFS_FC_SYMLINK));
   isHrdLink = ((file->dir.f.type == HFS_FT_HRDLINK) && (file->dir.f.creator == HFS_FC_HRDLINK));
   isFdAlias = ((file->dir.f.type == HFS_FT_FDALIAS) && (file->dir.f.creator == HFS_FC_FDALIAS));

   TRACES(("keyRec:%p  keyLength:%4.4hx  dir/file:%p\n", keyRec, keyLength, dir));

   if ((isHrdLink) && (hfs->hlMetaFolderId != 0) && (HfsHardlinksAutoResolve == TRUE))
   {
      ULONG              nodeNr;
      USHORT             index;                 // index in new node
      U_HFS_CAT_RECORD  *catData;               // new record (union of dir and file structures)

      sprintf( tbuf, "iNode%d", TxBE32( file->dir.crBsdInfo.iNodeNum));
      nodeNr = dfsHfsSearchCatalog( hfs->hlMetaFolderId, tbuf, &index, &node);
      if ((nodeNr != 0) && ((nodeNr != L32_NULL)))
      {
         if (dfsHfsNodeIdx2Rec( node, index, &catData) == HFS_CR_File)
         {
            TRACES(("Resolved hardlink, morph to that record for dates and sizes\n"));
            fnode = (S_HFS_FILE *) catData; // fileinfo, now to the resolved hardlink record
            dir   = (S_HFS_DIRINFO *) file; // dirinfo,  now to the resolved hardlink record
         }
      }
   }

   //- first line
   strcpy( text, " ");
   strcat( text, dfsHfsTime2str( dir->crCreTime, tbuf));

   if (recType == HFS_CR_File)                  // File attributes, show sym/hard link
   {
      TRACES(("f.type: 0x%8.8x  f.creator: 0x%8.8x", dir->f.type, dir->f.creator));
      if      (isSymLink) strcat( text, " Slnk");
      else if (isHrdLink) strcat( text, " Hlnk");
      else if (isFdAlias) strcat( text, " Alis");
      else                strcat( text, "     "); // to be refined, other attributes ?
   }
   else                                         // Folder attributes, to be refined
   {
      strcat( text, "    D");                   // to be refined (FAT-like) attributes
   }

   if ((dir->crModContTime != 0) && (dir->crModContTime != dir->crCreTime))
   {
      strcat( text, " ");
      strcat( text, dfsHfsTime2str( dir->crModContTime, tbuf));
   }
   else
   {
      strcat( text, "                    ");
   }

   if (recType == HFS_CR_File)                  // use possibly morphed record (hardlink)
   {
      forkSize = TxBE64( fnode->crDataFork.logicalSize);
      dfstrUllDot20( text, " ", forkSize, "");
      if ((forkSize = TxBE64( fnode->crRsrcFork.logicalSize)) != 0)
      {
         dfstrUllDot20( text, " ", forkSize, "");
      }
   }
   sprintf( tbuf, "%s\n", CGE);
   strcat(  text, tbuf);

   //- second line
   sprintf( tbuf, "          id %8x ", TxBE32( dir->crCnID));
   strcat(  text, tbuf);

   if ((dir->crAccTime != 0) && (dir->crAccTime != dir->crCreTime))
   {
      strcat( text, dfsHfsTime2str( dir->crAccTime, tbuf));
   }
   else
   {
      strcat( text, "                   ");
   }

   if ((fileMode = TxBE16( dir->crBsdInfo.fileMode) & 0777) != 0)
   {
      sprintf( tbuf, "  %3o ", fileMode);
      strcat(  text, tbuf);
   }
   else
   {
      strcat(  text, "      ");
   }

   strcat(  text, ansi[Ccol((CcY | CcI), bg)]); // Yellow
   TxPrint( "%s", text);

   TxMacUniStr2Ascii( &(key->name), text);

   if ((isSymLink) || (isHrdLink) || isFdAlias) // it is some kind of link elsewhere ...
   {
      if (strlen( text) < 19)
      {
         sprintf( tbuf, "%*.*s", (int)(19 - strlen( text)), (int)(19 - strlen( text)), "");
         strcat(  text, tbuf);
      }
      TxPrint( "%s%s%s%s",  text, ansi[Ccol( CcW, bg)], DFS_SH_LINK_STR, ansi[Ccol((CcC | CcI), bg)]);

      if (isSymLink)                            // resolve and show the symlink path too
      {
         dfsHfsResolveSymLink( dfsHfsBlock2Lsn( TxBE32( file->crDataFork.extents[0].startBlock)), text);
      }
      else if (isHrdLink)                       // resolve and show the hardlink path
      {
         sprintf( text, "/%s/iNode%d", HFS_SD_HARDLINK, TxBE32( file->dir.crBsdInfo.iNodeNum));
      }
      else                                      // resolve and show the FolderAlias path
      {
         dfsHfsResolveFdAlias( dfsHfsBlock2Lsn( TxBE32( file->crRsrcFork.extents[0].startBlock)), text);
      }
   }
   sprintf( tbuf, "%s%s%s\n", ansi[Ccol( CcW, bg)], CGE, CNN);
   strcat(  text, tbuf);
   TxPrint( "%s", text);

   TxFreeMem( node);                            // free optional node, if any

   VRETURN();
}                                               // end 'dfsHfsShowCatRecord'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read ASCII SYMLINK string from specified symlink sectornumber
/*****************************************************************************/
ULONG dfsHfsResolveSymLink
(
   ULN64               slnkLsn,                 // IN    symlink sectornumer
   char               *linkname                 // OUT   resolved symlink name
)
{
   ULONG               rc = NO_ERROR;           // function return
   BYTE               *buffer;

   ENTER();

   strcpy( linkname, "-");                      // in case of failure ...
   if ((buffer = TxAlloc( 1, dfsGetSectorSize())) != NULL)
   {
      if ((rc = dfsRead( slnkLsn, 1, buffer)) == NO_ERROR)
      {
         strcpy( linkname, (char *) buffer);    // name is zero-terminated in sector
      }
      TxFreeMem( buffer);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsHfsResolveSymLink'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create ASCII FOLDERALIAS string from specified 'alis' sectornumber
/*****************************************************************************/
ULONG dfsHfsResolveFdAlias
(
   ULN64               fdalLsn,                 // IN    folder alias sectornumer
   char               *linkname                 // OUT   resolved folder-alias name
)
{
   ULONG               rc = NO_ERROR;           // function return
   S_HFS_ALIS_FORK    *alisData = NULL;

   ENTER();

   strcpy( linkname, "-");                      // in case of failure ...
   if ((alisData = TxAlloc( 1, dfsGetSectorSize())) != NULL)
   {
      if ((rc = dfsRead( fdalLsn, 1, (BYTE *) alisData)) == NO_ERROR)
      {
         if ((alisData->al_signature        == HFS_SIGN_ALIAS)   && // validate the ALIS
             (TxBE16( alisData->al_hfsplus) == HFS_PLUS_SIGNATURE)) // resourcefork data
         {
            sprintf( linkname, "/%s/dir_%d", HFS_SD_FLDALIAS, TxBE32( alisData->al_dirRef));
         }
      }
      TxFreeMem( alisData);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsHfsResolveFdAlias'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Convert HFS DATE/TIME (32 bit) to DFS standard date/time string (19)
/*****************************************************************************/
char *dfsHfsTime2str                            // RET   string value
(
   ULONG               hfsdate,                 // IN    HFS time value
   char               *dtime                    // INOUT ptr to string buffer
)
{
   ULONG               dt = TxBE32( hfsdate);   // date-time in native endianess
   time_t              tm = HfsDate2Tm( dt);    // and in c-runtime time_t format
   struct tm          *gm;

   ENTER();

   if ((hfsdate != 0) && ((gm = gmtime( &tm)) != NULL))
   {
      //- Note: time_t officially is a SIGNED value (1901 .. 2038, 0 = 1970)
      strftime(dtime, TXMAXTM, "%Y-%m-%d %H:%M:%S", gmtime( &tm));
   }
   else                                         // invalid, out of range TM
   {
      sprintf( dtime, "-%8.8x-%8.8x-", dt, hfsdate); // show both endian forms
   }
   TRACES(("Formatted date/time: %s\n", dtime));

   RETURN( dtime);
}                                               // end 'dfsHfsTime2str'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Convert HFS DATE/TIME (32 bit) to standard C time_t value
/*****************************************************************************/
time_t dfsHfsFileTime2t                         // RET   time_t representation
(
   ULONG               hfsdate                  // IN    HFS time value
)
{
   ULONG               dt = TxBE32( hfsdate);   // date-time in native endianess
   time_t              tm = HfsDate2Tm( dt);    // and in c-runtime time_t format
   struct tm          *gm;

   ENTER();

   if ((hfsdate != 0) && ((gm = gmtime( &tm)) == NULL)) // invalid, out of range TM
   {
      tm = 0;                                   // default to 1970 ...
   }
   RETURN( tm);
}                                               // end 'dfsHfsFileTime2t'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display HFS style data (1-jan-1904, big-endian) to TxPrint output
/*****************************************************************************/
void dfsHfsPrintDateTime
(
   char               *lead,                    // IN    leading string
   ULONG               hfsdate                  // IN    HFS style date
)
{
   ULONG               dt = TxBE32( hfsdate);   // date-time in native endianess
   time_t              tm = HfsDate2Tm( dt);    // and in c-runtime time_t format

   TxPrint( "%s0x%8.8x = %-24.24s\n", lead, dt, (dt == 0) ? "Never" : ctime( &tm));
}                                               // end 'dfsHfsPrintDateTime'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create SPACE allocation structure from an HFS-plus ForkData structure
// Note: Area aware to allow usage from FDISK mode too (S_SPACE adjustment)
// TO BE REFINED, need to incorporate EXTENTS tree if more than 8 extents!
// which may require an additional paramater like file cnid
// (or create different higher level function, ptr to File record)
/*****************************************************************************/
ULONG dfsHfsFork2Space
(
   S_HFS_FORKDATA     *fork,                    // IN    fork data
   ULONG              *nr,                      // OUT   nr of space entries
   S_SPACE           **sp                       // OUT   space allocation
)
{
   ULONG               rc  = NO_ERROR;
   ULONG               chunks = 0;              // nr of space entries
   S_SPACE            *space  = NULL;           // space allocation
   ULONG               i;

   ENTER();

   for (i = 0; i < HFS_EXT_PER_REC; i++)
   {
      if (fork->extents[i].blockCount != 0)
      {
         chunks++;
      }
   }
   if (chunks != 0)
   {
      if ((space = TxAlloc( chunks, sizeof(S_SPACE))) != NULL)
      {
         for (i = 0; i < chunks; i++)
         {
            space[i].start = dfstAreaP2Disk(  DFSTORE, dfsHfsBlock2Lsn( TxBE32(fork->extents[i].startBlock)));
            space[i].size  =                           dfsHfsBlock2Lsn( TxBE32(fork->extents[i].blockCount));
         }
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   *sp = space;
   *nr = chunks;
   RETURN (rc);
}                                               // end 'dfsHfsFork2Space'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display HFS Forkdata to TxPrint output
/*****************************************************************************/
void dfsHfsPrintForkdata
(
   char               *lead,                    // IN    leading string
   ULONG               options,                 // IN    SD_ display options
   S_HFS_FORKDATA     *fork                     // IN    HFS fork data
)
{
   ULONG               blocks = TxBE32( fork->totalBlocks);
   ULONG               extents;
   S_SPACE            *exspace;

   dfsUllDot20( lead, TxBE64( fork->logicalSize), " bytes,   ");
   dfsSz64("Sectors : ", ((ULN64) blocks * hfs->SectorsPerBlock), "");

   if (fork->logicalSize != 0)
   {
      TxPrint( "\n");
      if (dfsHfsFork2Space( fork, &extents, &exspace) == NO_ERROR)
      {
         dfsSspaceDisplay( SD_BLOCKS | options, extents, exspace);
         TxFreeMem( exspace);
      }
   }
   else
   {
      TxPrint( "\n");
   }
}                                               // end 'dfsHfsPrintForkdata'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read stored offset for a record in a node, based on its index
/*****************************************************************************/
USHORT dfsHfsRecordOffset                       // RET   offset to record
(
   USHORT              index,                   // IN    record index 0 .. n
   void               *node                     // IN    any HFS node
)
{
   USHORT              rc = 0;                  // function return, invalid
   S_BT_NODE_DESC     *desc = (S_BT_NODE_DESC *) node;

   ENTER();
   TRACES(("index: %hu\n", index));

   if (index <= desc->nRecords)
   {
      USHORT offset = hfs->Catalog.bpNode - ((index + 1) * sizeof( USHORT));

      rc = TxBE16( *((USHORT *) ((BYTE *) node + offset)));
   }
   RETURN (rc);
}                                               // end 'dfsHfsRecordOffset'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Return fixed-length string (6) with node-type description
/*****************************************************************************/
char *dfsHfsNodeType                            // RET   description string
(
   BYTE                kind                     // IN    kind of node
)
{
   switch (kind)
   {
      case BT_LeafNode   : return( "Leaf  "); break;
      case BT_IndexNode  : return( "Index "); break;
      case BT_HeaderNode : return( "Header"); break;
      case BT_MapNode    : return( "Map   "); break;
      default:             return( "Bad!  "); break;
   }
}                                               // end 'dfsHfsNodeType'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Return fixed-length string (9) with catalog-record type description
/*****************************************************************************/
char *dfsHfsCatRecType                          // RET   description string
(
   USHORT              crType                   // IN    catalog record type
)
{
   switch (crType)
   {
      case HFS_CR_Folder : return( "Folder"); break;
      case HFS_CR_File   : return( "File  "); break;
      case HFS_CR_Dthread: return( "D-thrd"); break;
      case HFS_CR_Fthread: return( "F-thrd"); break;
      default:             return( "T-BAD-"); break;
   }
}                                               // end 'dfsHfsCatRecType'
/*---------------------------------------------------------------------------*/


