//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// FDISK GPT related utility functions
//
// Author: J. van Wijk
//
// JvW  03-09-2015   Initial version, cloned from DFSUFDSK
//
#ifndef    DFSUFGPT
   #define DFSUFGPT


#define GPTHDR_PRIMARY_PSN      1
#define GPTHDR_SIG_LEN          8

#define GPT_DEF_PTA_COUNT     128               // default nr of entries in PTA
#define GPT_DEF_PTA_ESIZE     128               // default size in byte for an entry
#define GPT_DEF_REVISION     0x10000            // default header revision

#define SV_GPTHDR1     0x20494645               // 1st GPT HDR signature: EFI
#define SV_GPTHDR2     0x54524150               // 2nd GPT HDR signature: PART
typedef struct s_gpthdr                         // GPT header sector (PSN 1)
{
   ULONG               Signature1;              // little endian "EFI "
   ULONG               Signature2;              // little endian "PART"
   ULONG               Revision;                // GPT revision (usually 0x10000)
   ULONG               hdrSize;                 // structure size in bytes
   ULONG               hdrCRC;                  // 32-bit CRC over header
   ULONG               reserved1;               // reserved, must be 0
   ULN64               thisPSN;                 // LBA for this hdr sector
   ULN64               althPSN;                 // LBA for alternate hdr sector
   ULN64               firstPSN;                // LBA first usable partition sector
   ULN64               lastPSN;                 // LBA last  usable partition sector
   DFS_GUID            diskGuid;                // GUID for this whole disk
   ULN64               ptaPSN;                  // LBA of first part array sector
   ULONG               ptaEntries;              // Nr of entries in the PTA
   ULONG               ptaEntrySize;            // Size of one entry (usually 128)
   ULONG               ptaCRC;                  // 32-bit CRC over partition array
                                                // rest of sector must be all 0
} S_GPTHDR;                                     // end of struct "s_gpthdr"


// GPT attribute flag values, 64-bit definition (ULN64) top 16 bits are NOT unique!
#define GPT_ATTR_SYSTEM_PARTITION   0x0000000000000001ULL // bit  0; should not be modified ...
#define GPT_ATTR_TO_BE_SW_IGNORED   0x0000000000000002ULL // bit  1; to be ignored by all EFI sw
#define GPT_ATTR_LEGACY_BIOS_BOOT   0x0000000000000004ULL // bit  2; like MBR partition 0x80 flag

// Chromium/Chrome-OS specific attributes
#define GPT_ATTR_CHR_BOOT_PRIOR_0   0x0001000000000000ULL // bit 48; Chrome OS LSB
#define GPT_ATTR_CHR_BOOT_PRIOR_1   0x0002000000000000ULL // bit 49; Ch-boot-prio/Apple Boot
#define GPT_ATTR_CHR_BOOT_PRIOR_2   0x0004000000000000ULL // bit 50; Chrome OS boot priority
#define GPT_ATTR_CHR_BOOT_PRIOR_3   0x0008000000000000ULL // bit 51; Chrome OS MSB
#define GPT_ATTR_CHR_REMN_TRIES_0   0x0010000000000000ULL // bit 52; Chrome OS LSB
#define GPT_ATTR_CHR_REMN_TRIES_1   0x0020000000000000ULL // bit 53; Chrome OS remaining tries
#define GPT_ATTR_CHR_REMN_TRIES_2   0x0040000000000000ULL // bit 54; Chrome OS
#define GPT_ATTR_CHR_REMN_TRIES_3   0x0080000000000000ULL // bit 55; Chrome OS MSB
#define GPT_ATTR_CHR_SUCCESS_BOOT   0x0100000000000000ULL // bit 56; Chrome OS successful boot

// Microsoft specific attributes
#define GPT_ATTR_MS_PART_READONLY   0x1000000000000000ULL // bit 60; partition is read-only
#define GPT_ATTR_MS_PART_ISHIDDEN   0x4000000000000000ULL // bit 62; partition is hidden
#define GPT_ATTR_MS_NO_AUTO_MOUNT   0x8000000000000000ULL // bit 63; no auto-mount (driveletter)

// Apple specific attributes
#define GPT_ATTR_APPLE_BOOT_PART    0x0002000000000000ULL // bit 49; Apple (OS-X) bootable ?

#define GPT_MASK_DEFINED_FLAGS      0xd1ff000000000007ULL // keep value up-to-date!

// OS/2 LVM compatibilty solutions with GPT
// Only REQUIRED info is the driveletter (perhaps 'bootable' flag too ?),
// rest can be derived from name/type/GUID that is part of the GPT itself
// This would allow a GPT style, when OS2DASD/OS2LVM are updated to read that info
// instead of the regular MBR+DLAT sectors. Would be compatible with Windows/MAC
// Upper 16 bits of attrFlags are reserved for specific partition use (bit 48..63)
// Could be used to store a driveletter (DLA) with an additional unique pattern
// Another option is to store the driveletter as part of the (36 char Unicode) name
// For example 'Some name to use   DLA:x'

#define GPT_PNAME_LEN  36                       // 72 bytes, 36 Unicode characters

typedef struct s_gptentry                       // GPT entry structure (in PTA sectors)
{
   DFS_GUID            typeGuid;                // GUID for type of the partition
   DFS_GUID            partGuid;                // GUID for this specific partition
   ULN64               firstPSN;                // LBA first sector of partition
   ULN64               lastPSN;                 // LBA last  sector of partition
   ULN64               attrFlags;               // Attribute flags for GPT
   USHORT              partName[GPT_PNAME_LEN]; // partition name length (Unicode)
} S_GPTENTRY;                                   // end of struct "s_gptentry"

#define GptGuidMatch(g1,g2) (memcmp(g1,g2,DFS_GUID_LENGTH) == 0)


// DFSee variant of the UUID_DEFINE macro, that defines them little-endian, as on-disk
// Allows use of the byte-definitions in the order as usually published
#if defined (DFS_GUID_CREATE)
   #define DFS_MKGUID(name,u0,u1,u2,u3,u4,u5,u6,u7,u8,u9,u10,u11,u12,u13,u14,u15) \
          DFS_GUID name = {u3,u2,u1,u0,u5,u4,u7,u6,u8,u9,u10,u11,u12,u13,u14,u15}
#else
   #define DFS_MKGUID(name,u0,u1,u2,u3,u4,u5,u6,u7,u8,u9,u10,u11,u12,u13,u14,u15) \
      extern DFS_GUID name
#endif


// GPT GUID definitions, values taken from Wikipedia 'PT partition' page
// Stored little-endian, so can be compared directly to on-disk values
// Instantiated in DFSUFGPT only, extern in all other modules
DFS_MKGUID(dfsGuid_APFS_CONTAINER      ,0x7C,0x34,0x57,0xEF,0x00,0x00,0x11,0xAA,0xAA,0x11,0x00,0x30,0x65,0x43,0xEC,0xAC);
DFS_MKGUID(dfsGuid_APPLE_BOOT          ,0x42,0x6F,0x6F,0x74,0x00,0x00,0x11,0xAA,0xAA,0x11,0x00,0x30,0x65,0x43,0xEC,0xAC);
DFS_MKGUID(dfsGuid_APPLE_CORE_STORAGE  ,0x53,0x74,0x6F,0x72,0x61,0x67,0x11,0xAA,0xAA,0x11,0x00,0x30,0x65,0x43,0xEC,0xAC);
DFS_MKGUID(dfsGuid_APPLE_LABEL         ,0x4C,0x61,0x62,0x65,0x6C,0x00,0x11,0xAA,0xAA,0x11,0x00,0x30,0x65,0x43,0xEC,0xAC);
DFS_MKGUID(dfsGuid_APPLE_RAID          ,0x52,0x41,0x49,0x44,0x00,0x00,0x11,0xAA,0xAA,0x11,0x00,0x30,0x65,0x43,0xEC,0xAC);
DFS_MKGUID(dfsGuid_APPLE_RAID_OFFLINE  ,0x52,0x41,0x49,0x44,0x5F,0x4F,0x11,0xAA,0xAA,0x11,0x00,0x30,0x65,0x43,0xEC,0xAC);
DFS_MKGUID(dfsGuid_APPLE_TV_RECOVERY   ,0x52,0x65,0x63,0x6F,0x76,0x65,0x11,0xAA,0xAA,0x11,0x00,0x30,0x65,0x43,0xEC,0xAC);
DFS_MKGUID(dfsGuid_APPLE_UFS           ,0x55,0x46,0x53,0x00,0x00,0x00,0x11,0xAA,0xAA,0x11,0x00,0x30,0x65,0x43,0xEC,0xAC);
DFS_MKGUID(dfsGuid_APPLE_ZFS           ,0x6A,0x89,0x8C,0xC3,0x1D,0xD2,0x11,0xB2,0x99,0xA6,0x08,0x00,0x20,0x73,0x66,0x31);
DFS_MKGUID(dfsGuid_BIOS_BOOT           ,0x21,0x68,0x61,0x48,0x64,0x49,0x6E,0x6F,0x74,0x4E,0x65,0x65,0x64,0x45,0x46,0x49);
DFS_MKGUID(dfsGuid_CHROME_FUTURE       ,0x2E,0x0A,0x75,0x3D,0x9E,0x48,0x43,0xB0,0x83,0x37,0xB1,0x51,0x92,0xCB,0x1B,0x5E);
DFS_MKGUID(dfsGuid_CHROME_KERNEL       ,0xFE,0x3A,0x2A,0x5D,0x4F,0x32,0x41,0xA7,0xB7,0x25,0xAC,0xCC,0x32,0x85,0xA3,0x09);
DFS_MKGUID(dfsGuid_CHROME_ROOTFS       ,0x3C,0xB8,0xE2,0x02,0x3B,0x7E,0x47,0xDD,0x8A,0x3C,0x7F,0xF2,0xA1,0x3C,0xFC,0xEC);
DFS_MKGUID(dfsGuid_DFSEE_UNKNOWN       ,0xdf,0x5e,0xe1,0x30,0xdf,0x00,0x4d,0xf5,0x4d,0xf5,0x20,0x15,0x11,0x18,0x14,0x44);
DFS_MKGUID(dfsGuid_EFI_SYSTEM          ,0xC1,0x2A,0x73,0x28,0xF8,0x1F,0x11,0xD2,0xBA,0x4B,0x00,0xA0,0xC9,0x3E,0xC9,0x3B);
DFS_MKGUID(dfsGuid_FREEBSD_BOOT        ,0x83,0xBD,0x6B,0x9D,0x7F,0x41,0x11,0xDC,0xBE,0x0B,0x00,0x15,0x60,0xB8,0x4F,0x0F);
DFS_MKGUID(dfsGuid_FREEBSD_DATA        ,0x51,0x6E,0x7C,0xB4,0x6E,0xCF,0x11,0xD6,0x8F,0xF8,0x00,0x02,0x2D,0x09,0x71,0x2B);
DFS_MKGUID(dfsGuid_FREEBSD_SWAP        ,0x51,0x6E,0x7C,0xB5,0x6E,0xCF,0x11,0xD6,0x8F,0xF8,0x00,0x02,0x2D,0x09,0x71,0x2B);
DFS_MKGUID(dfsGuid_FREEBSD_UFS         ,0x51,0x6E,0x7C,0xB6,0x6E,0xCF,0x11,0xD6,0x8F,0xF8,0x00,0x02,0x2D,0x09,0x71,0x2B);
DFS_MKGUID(dfsGuid_FREEBSD_VINUM_VM    ,0x51,0x6E,0x7C,0xB8,0x6E,0xCF,0x11,0xD6,0x8F,0xF8,0x00,0x02,0x2D,0x09,0x71,0x2B);
DFS_MKGUID(dfsGuid_FREEBSD_ZFS         ,0x51,0x6E,0x7C,0xBA,0x6E,0xCF,0x11,0xD6,0x8F,0xF8,0x00,0x02,0x2D,0x09,0x71,0x2B);
DFS_MKGUID(dfsGuid_HP_UX_DATA          ,0x75,0x89,0x4C,0x1E,0x3A,0xEB,0x11,0xD3,0xB7,0xC1,0x7B,0x03,0xA0,0x00,0x00,0x00);
DFS_MKGUID(dfsGuid_HP_UX_SERVICE       ,0xE2,0xA1,0xE7,0x28,0x32,0xE3,0x11,0xD6,0xA6,0x82,0x7B,0x03,0xA0,0x00,0x00,0x00);
DFS_MKGUID(dfsGuid_IBM_GENPARALLEL_FS  ,0x37,0xAF,0xFC,0x90,0xEF,0x7D,0x4E,0x96,0x91,0xC3,0x2D,0x7A,0xE0,0x55,0xB1,0x74);
DFS_MKGUID(dfsGuid_LENOVO_BOOT         ,0xBF,0xBF,0xAF,0xE7,0xA3,0x4F,0x44,0x8A,0x9A,0x5B,0x62,0x13,0xEB,0x73,0x6C,0x22);
DFS_MKGUID(dfsGuid_LINUX_DATA          ,0x0F,0xC6,0x3D,0xAF,0x84,0x83,0x47,0x72,0x8E,0x79,0x3D,0x69,0xD8,0x47,0x7D,0xE4);
DFS_MKGUID(dfsGuid_LINUX_EXTEND_BOOT   ,0xbc,0x13,0xc2,0xff,0x59,0xe6,0x42,0x62,0xa3,0x52,0xb2,0x75,0xfd,0x6f,0x71,0x72);
DFS_MKGUID(dfsGuid_LINUX_HOME          ,0x93,0x3A,0xC7,0xE1,0x2E,0xB4,0x4F,0x13,0xB8,0x44,0x0E,0x14,0xE2,0xAE,0xF9,0x15);
DFS_MKGUID(dfsGuid_LINUX_LUKS          ,0xCA,0x7D,0x7C,0xCB,0x63,0xED,0x4C,0x53,0x86,0x1C,0x17,0x42,0x53,0x60,0x59,0xCC);
DFS_MKGUID(dfsGuid_LINUX_LVM           ,0xE6,0xD6,0xD3,0x79,0xF5,0x07,0x44,0xC2,0xA2,0x3C,0x23,0x8F,0x2A,0x3D,0xF9,0x28);
DFS_MKGUID(dfsGuid_LINUX_PLAIN_DMCRYPT ,0x7F,0xFE,0xC5,0xC9,0x2D,0x00,0x49,0xB7,0x89,0x41,0x3E,0xA1,0x0A,0x55,0x86,0xB7);
DFS_MKGUID(dfsGuid_LINUX_RAID          ,0xA1,0x9D,0x88,0x0F,0x05,0xFC,0x4D,0x3B,0xA0,0x06,0x74,0x3F,0x0F,0x84,0x91,0x1E);
DFS_MKGUID(dfsGuid_LINUX_RESERVED      ,0x8D,0xA6,0x33,0x39,0x00,0x07,0x60,0xC0,0xC4,0x36,0x08,0x3A,0xC8,0x23,0x09,0x08);
DFS_MKGUID(dfsGuid_LINUX_ROOT_ARM_32   ,0x69,0xda,0xd7,0x10,0x2c,0xe4,0x4e,0x3c,0xb1,0x6c,0x21,0xa1,0xd4,0x9a,0xbe,0xd3);
DFS_MKGUID(dfsGuid_LINUX_ROOT_ARM_64   ,0xb9,0x21,0xb0,0x45,0x1d,0xf0,0x41,0xc3,0xaf,0x44,0x4c,0x6f,0x28,0x0d,0x3f,0xae);
DFS_MKGUID(dfsGuid_LINUX_ROOT_X86_32   ,0x44,0x47,0x95,0x40,0xf2,0x97,0x41,0xb2,0x9a,0xf7,0xd1,0x31,0xd5,0xf0,0x45,0x8a);
DFS_MKGUID(dfsGuid_LINUX_ROOT_X86_64   ,0x4f,0x68,0xbc,0xe3,0xe8,0xcd,0x4d,0xb1,0x96,0xe7,0xfb,0xca,0xf9,0x84,0xb7,0x09);
DFS_MKGUID(dfsGuid_LINUX_SRV_DATA      ,0x3B,0x8F,0x84,0x25,0x20,0xE0,0x4F,0x3B,0x90,0x7F,0x1A,0x25,0xA7,0x6F,0x98,0xE8);
DFS_MKGUID(dfsGuid_LINUX_SWAP          ,0x06,0x57,0xFD,0x6D,0xA4,0xAB,0x43,0xC4,0x84,0xE5,0x09,0x33,0xC8,0x4B,0x4F,0x4F);
DFS_MKGUID(dfsGuid_MAC_OS_X_HFS_PLUS   ,0x48,0x46,0x53,0x00,0x00,0x00,0x11,0xAA,0xAA,0x11,0x00,0x30,0x65,0x43,0xEC,0xAC);
DFS_MKGUID(dfsGuid_MBR_PARTITIONING    ,0x02,0x4D,0xEE,0x41,0x33,0xE7,0x11,0xD3,0x9D,0x69,0x00,0x08,0xC7,0x81,0xF3,0x9F);
DFS_MKGUID(dfsGuid_MICROSOFT_RESERVED  ,0xE3,0xC9,0xE3,0x16,0x0B,0x5C,0x4D,0xB8,0x81,0x7D,0xF9,0x2D,0xF0,0x02,0x15,0xAE);
DFS_MKGUID(dfsGuid_MIDNIGHTBSD_BOOT    ,0x85,0xD5,0xE4,0x5E,0x23,0x7C,0x11,0xE1,0xB4,0xB3,0xE8,0x9A,0x8F,0x7F,0xC3,0xA7);
DFS_MKGUID(dfsGuid_MIDNIGHTBSD_DATA    ,0x85,0xD5,0xE4,0x5A,0x23,0x7C,0x11,0xE1,0xB4,0xB3,0xE8,0x9A,0x8F,0x7F,0xC3,0xA7);
DFS_MKGUID(dfsGuid_MIDNIGHTBSD_SWAP    ,0x85,0xD5,0xE4,0x5B,0x23,0x7C,0x11,0xE1,0xB4,0xB3,0xE8,0x9A,0x8F,0x7F,0xC3,0xA7);
DFS_MKGUID(dfsGuid_MIDNIGHTBSD_UFS     ,0x03,0x94,0xEf,0x8B,0x23,0x7E,0x11,0xE1,0xB4,0xB3,0xE8,0x9A,0x8F,0x7F,0xC3,0xA7);
DFS_MKGUID(dfsGuid_MIDNIGHTBSD_VINUM   ,0x85,0xD5,0xE4,0x5C,0x23,0x7C,0x11,0xE1,0xB4,0xB3,0xE8,0x9A,0x8F,0x7F,0xC3,0xA7);
DFS_MKGUID(dfsGuid_MIDNIGHTBSD_ZFS     ,0X85,0xD5,0xE4,0x5D,0x23,0x7C,0x11,0xE1,0xB4,0xB3,0xE8,0x9A,0x8F,0x7F,0xC3,0xA7);
DFS_MKGUID(dfsGuid_NETBSD_CONCAT       ,0x2D,0xB5,0x19,0xC4,0xB1,0x0F,0x11,0xDC,0xB9,0x9B,0x00,0x19,0xD1,0x87,0x96,0x48);
DFS_MKGUID(dfsGuid_NETBSD_ENCRYPTED    ,0x2D,0xB5,0x19,0xEC,0xB1,0x0F,0x11,0xDC,0xB9,0x9B,0x00,0x19,0xD1,0x87,0x96,0x48);
DFS_MKGUID(dfsGuid_NETBSD_FFS          ,0x49,0xF4,0x8D,0x5A,0xB1,0x0E,0x11,0xDC,0xB9,0x9B,0x00,0x19,0xD1,0x87,0x96,0x48);
DFS_MKGUID(dfsGuid_NETBSD_LFS          ,0x49,0xF4,0x8D,0x82,0xB1,0x0E,0x11,0xDC,0xB9,0x9B,0x00,0x19,0xD1,0x87,0x96,0x48);
DFS_MKGUID(dfsGuid_NETBSD_RAID         ,0x49,0xF4,0x8D,0xAA,0xB1,0x0E,0x11,0xDC,0xB9,0x9B,0x00,0x19,0xD1,0x87,0x96,0x48);
DFS_MKGUID(dfsGuid_NETBSD_SWAP         ,0x49,0xF4,0x8D,0x32,0xB1,0x0E,0x11,0xDC,0xB9,0x9B,0x00,0x19,0xD1,0x87,0x96,0x48);
DFS_MKGUID(dfsGuid_OPENBSD_DATA        ,0x82,0x4C,0xC7,0xA0,0x36,0xA8,0x11,0xE3,0x89,0x0A,0x95,0x25,0x19,0xAD,0x3F,0x61);
DFS_MKGUID(dfsGuid_OS2_ARCAOS_TYPE_1   ,0x90,0xB6,0xFF,0x38,0xB9,0x8F,0x43,0x58,0xA2,0x1F,0x48,0xF3,0x5B,0x4A,0x8A,0xD3);
DFS_MKGUID(dfsGuid_QNX_POWERSAFE       ,0xCE,0xF5,0xA9,0xAD,0x73,0xBC,0x46,0x01,0x89,0xF3,0xCD,0xEE,0xEE,0xE3,0x21,0xA1);
DFS_MKGUID(dfsGuid_SOLARIS_ALTERNATE   ,0x6A,0x92,0x83,0xA5,0x1D,0xD2,0x11,0xB2,0x99,0xA6,0x08,0x00,0x20,0x73,0x66,0x31);
DFS_MKGUID(dfsGuid_SOLARIS_BACKUP      ,0x6A,0x8B,0x64,0x2B,0x1D,0xD2,0x11,0xB2,0x99,0xA6,0x08,0x00,0x20,0x73,0x66,0x31);
DFS_MKGUID(dfsGuid_SOLARIS_BOOT        ,0x6A,0x82,0xCB,0x45,0x1D,0xD2,0x11,0xB2,0x99,0xA6,0x08,0x00,0x20,0x73,0x66,0x31);
DFS_MKGUID(dfsGuid_SOLARIS_HOME        ,0x6A,0x90,0xBA,0x39,0x1D,0xD2,0x11,0xB2,0x99,0xA6,0x08,0x00,0x20,0x73,0x66,0x31);
DFS_MKGUID(dfsGuid_SOLARIS_RESERVED1   ,0x6A,0x94,0x5A,0x3B,0x1D,0xD2,0x11,0xB2,0x99,0xA6,0x08,0x00,0x20,0x73,0x66,0x31);
DFS_MKGUID(dfsGuid_SOLARIS_RESERVED2   ,0x6A,0x96,0x30,0xD1,0x1D,0xD2,0x11,0xB2,0x99,0xA6,0x08,0x00,0x20,0x73,0x66,0x31);
DFS_MKGUID(dfsGuid_SOLARIS_RESERVED3   ,0x6A,0x98,0x07,0x67,0x1D,0xD2,0x11,0xB2,0x99,0xA6,0x08,0x00,0x20,0x73,0x66,0x31);
DFS_MKGUID(dfsGuid_SOLARIS_RESERVED4   ,0x6A,0x96,0x23,0x7F,0x1D,0xD2,0x11,0xB2,0x99,0xA6,0x08,0x00,0x20,0x73,0x66,0x31);
DFS_MKGUID(dfsGuid_SOLARIS_RESERVED5   ,0x6A,0x8D,0x2A,0xC7,0x1D,0xD2,0x11,0xB2,0x99,0xA6,0x08,0x00,0x20,0x73,0x66,0x31);
DFS_MKGUID(dfsGuid_SOLARIS_ROOT        ,0x6A,0x85,0xCF,0x4D,0x1D,0xD2,0x11,0xB2,0x99,0xA6,0x08,0x00,0x20,0x73,0x66,0x31);
DFS_MKGUID(dfsGuid_SOLARIS_SWAP        ,0x6A,0x87,0xC4,0x6F,0x1D,0xD2,0x11,0xB2,0x99,0xA6,0x08,0x00,0x20,0x73,0x66,0x31);
DFS_MKGUID(dfsGuid_SOLARIS_USR         ,0x6A,0x89,0x8C,0xC3,0x1D,0xD2,0x11,0xB2,0x99,0xA6,0x08,0x00,0x20,0x73,0x66,0x31);
DFS_MKGUID(dfsGuid_SOLARIS_VAR         ,0x6A,0x8E,0xF2,0xE9,0x1D,0xD2,0x11,0xB2,0x99,0xA6,0x08,0x00,0x20,0x73,0x66,0x31);
DFS_MKGUID(dfsGuid_SONY_BOOT           ,0xF4,0x01,0x97,0x32,0x06,0x6E,0x4E,0x12,0x82,0x73,0x34,0x6C,0x56,0x41,0x49,0x4F);
DFS_MKGUID(dfsGuid_VMWARE_COREDUMP     ,0x9D,0x27,0x53,0x80,0x40,0xAD,0x11,0xDB,0xBF,0x97,0x00,0x0C,0x29,0x11,0xD1,0xB8);
DFS_MKGUID(dfsGuid_VMWARE_VMFS         ,0xAA,0x31,0xE0,0x2A,0x40,0x0F,0x11,0xDB,0x95,0x90,0x00,0x0C,0x29,0x11,0xD1,0xB8);
DFS_MKGUID(dfsGuid_WINDOWS_BASIC_DATA  ,0xEB,0xD0,0xA0,0xA2,0xB9,0xE5,0x44,0x33,0x87,0xC0,0x68,0xB6,0xB7,0x26,0x99,0xC7);
DFS_MKGUID(dfsGuid_WINDOWS_LDM_DATA    ,0xAF,0x9B,0x60,0xA0,0x14,0x31,0x4F,0x62,0xBC,0x68,0x33,0x11,0x71,0x4A,0x69,0xAD);
DFS_MKGUID(dfsGuid_WINDOWS_LDM_META    ,0x58,0x08,0xC8,0xAA,0x7E,0x8F,0x42,0xE0,0x85,0xD2,0xE1,0xE9,0x04,0x34,0xCF,0xB3);
DFS_MKGUID(dfsGuid_WINDOWS_RECOVERY    ,0xDE,0x94,0xBB,0xA4,0x06,0xD1,0x4D,0x40,0xA1,0x6A,0xBF,0xD5,0x01,0x79,0xD6,0xAC);


// Return ptr to 17-char description string for the given GUID partition-type
char *dfsGptGuidDescription                     // RET   ptr to description
(
   DFS_GUID            typeGuid                 // IN    DFS_GUID byte array
);

// Return most likely related part-GUID for the given MBR type byte
BYTE *dfsMbrType2GptGuid                        // RET   ptr to binary GUID
(
   BYTE                mbrType                  // IN    MBR style type byte
);

// Create GUID value based on TIME and RANDOM values, 'df5' plus given ID
void dfsGptMkGuid
(
   USHORT              id,                      // IN    object ID (modulo 64)
   BYTE               *guid                     // OUT   16 bytes GUID value
);

// Determine if given LSN is inside the GPT area at start or end of the disk
BOOL dfsGptLsnInTableArea                       // RET   LSN is inside GPT area
(
   DFSDISKINFO        *d,                       // IN    current disk info
   ULN64               lsn,                     // IN    LSN to check area for
   ULN64              *start,                   // OUT   start LSN of area
   ULN64              *final                    // OUT   final LSN of area
);

// Read and validate GPT header sector, and attached partition table array (PTA)
ULONG dfsGptReadValidateLba                     // RET   OK, bad-sig, bad-crc
(
   ULN64               lba,                     // IN    LBA for the HDR sector
   S_GPTHDR          **hdrSect,                 // OUT   GPT header sector  or NULL
   S_GPTENTRY         *ptArray[]                // OUT   allocated PT Array or NULL
);

// Validate GPT header sector (usually PSN 1, after MBR) and return PT array
ULONG dfsGptValidateHdr                         // RET   OK, bad-sig, bad-crc
(
   S_GPTHDR           *sd,                      // IN    GPT header sector
   S_GPTENTRY         *ptArray[]                // OUT   allocated PT array or NULL
);

// Display GPT header sector (usually PSN 1, after MBR)
ULONG dfsGptHdr                                 // RET   OK, bad-sig, bad-crc
(
   BYTE               *sector                   // IN    Sector buffer
);

// Read GPT header and PTA to disk-info, from alternate or primary location
ULONG dfsGptReadAlternate                       // RET   OK, bad-sig, bad-crc
(
   DFSDISKINFO        *d,                       // IN    Disk info
   ULN64               altPsn,                  // IN    LBA for alternate
   BOOL               *priValid,                // OUT   primary   is valid (too)
   BOOL               *altValid                 // OUT   alternate is valid (too)
);                                              //       for MBR guard partition

// Create valid GPT header and partition-arrays, optional recover from existing
ULONG dfsGptCreateFromMbr                       // RET   OK, bad-sig, bad-crc
(
   BOOL                recover,                 // IN    try to recover existing
   DFSPARTINFO        *f                        // IN    freespace/partition info
);                                              //       for MBR guard partition


// Fixup CRC and Write primary plus alternate GPT header sectors and PTA areas
// Note: Supplied header may be either the primary or the alternate header
ULONG dfsGptFixupWriteAll                       // RET   OK, bad-sig, bad-crc
(
   S_GPTHDR           *gptHdr,                  // IN    GPT header sector
   S_GPTENTRY          ptArray[]                // IN    PT array (whole sectors!)
);

// Select disk and get reference to a GPT entry, and optional header/table info
S_GPTENTRY *dfsGptDiskGetEntry                  // RET   GPT entry ref or NULL
(
   FDSK_CB_INFO       *cbp,                     // IN    info, disknr, partnr
   S_GPTHDR          **hdr,                     // OUT   GPT header, or NULL
   S_GPTENTRY        **table                    // OUT   GPT table,  or NULL
);

// Find the NEXT used entry in a GPT array, starting from the current one
BOOL dfsGptNextEntry                            // RET    cbd->partnr changed
(
   FDSK_CB_INFO       *cbp                      // INOUT info, current partnr
);

// Find the PREVIOUS used entry in a GPT array, starting from the current one
BOOL dfsGptPrevEntry                            // RET    cbd->partnr changed
(
   FDSK_CB_INFO       *cbp                      // INOUT info, current partnr
);

// Change partitiontype to specified value, and write back all tables
ULONG dfsGptSetType                             // RET   OK, bad-sig, bad-crc
(
   DFSPARTINFO        *p,                       // IN    current partition info
   FDSK_CB_INFO       *cbp                      // IN    disk/part info + type
);

#if defined (USEWINDOWING)

#define DFSGPT_CUSTOM  0xCC                     // Item value for CUSTOM entry

// Build selection-list with the most important GPT partition types + custom
TXSELIST *dfsGptTypeSelist                      // RET   selection list or NULL
(
   BOOL                addCustom                // IN    Add 'manual edit' as 1st
);
#endif                                          // USEWINDOWING


#endif
