//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// FDISK GPT related utility functions
//
// Author: J. van Wijk
//
// JvW  03-09-2015   Initial version, cloned from DFSUFDSK
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsupart.h>                           // FDISK partition functions
#include <dfsufdsk.h>                           // FDISK utility functions

#define DFS_GUID_CREATE
#include <dfsufgpt.h>                           // FDISK GPT utility functions
#undef  DFS_GUID_CREATE


#ifndef OEMSB
// GPT 32-bit CRC table and calculation

static ULONG dfsGptCrc32Table[ 256] =
{
   0x00000000, 0x77073096, 0xee0e612c, 0x990951ba, 0x076dc419, 0x706af48f, 0xe963a535, 0x9e6495a3,
   0x0edb8832, 0x79dcb8a4, 0xe0d5e91e, 0x97d2d988, 0x09b64c2b, 0x7eb17cbd, 0xe7b82d07, 0x90bf1d91,
   0x1db71064, 0x6ab020f2, 0xf3b97148, 0x84be41de, 0x1adad47d, 0x6ddde4eb, 0xf4d4b551, 0x83d385c7,
   0x136c9856, 0x646ba8c0, 0xfd62f97a, 0x8a65c9ec, 0x14015c4f, 0x63066cd9, 0xfa0f3d63, 0x8d080df5,
   0x3b6e20c8, 0x4c69105e, 0xd56041e4, 0xa2677172, 0x3c03e4d1, 0x4b04d447, 0xd20d85fd, 0xa50ab56b,
   0x35b5a8fa, 0x42b2986c, 0xdbbbc9d6, 0xacbcf940, 0x32d86ce3, 0x45df5c75, 0xdcd60dcf, 0xabd13d59,
   0x26d930ac, 0x51de003a, 0xc8d75180, 0xbfd06116, 0x21b4f4b5, 0x56b3c423, 0xcfba9599, 0xb8bda50f,
   0x2802b89e, 0x5f058808, 0xc60cd9b2, 0xb10be924, 0x2f6f7c87, 0x58684c11, 0xc1611dab, 0xb6662d3d,
   0x76dc4190, 0x01db7106, 0x98d220bc, 0xefd5102a, 0x71b18589, 0x06b6b51f, 0x9fbfe4a5, 0xe8b8d433,
   0x7807c9a2, 0x0f00f934, 0x9609a88e, 0xe10e9818, 0x7f6a0dbb, 0x086d3d2d, 0x91646c97, 0xe6635c01,
   0x6b6b51f4, 0x1c6c6162, 0x856530d8, 0xf262004e, 0x6c0695ed, 0x1b01a57b, 0x8208f4c1, 0xf50fc457,
   0x65b0d9c6, 0x12b7e950, 0x8bbeb8ea, 0xfcb9887c, 0x62dd1ddf, 0x15da2d49, 0x8cd37cf3, 0xfbd44c65,
   0x4db26158, 0x3ab551ce, 0xa3bc0074, 0xd4bb30e2, 0x4adfa541, 0x3dd895d7, 0xa4d1c46d, 0xd3d6f4fb,
   0x4369e96a, 0x346ed9fc, 0xad678846, 0xda60b8d0, 0x44042d73, 0x33031de5, 0xaa0a4c5f, 0xdd0d7cc9,
   0x5005713c, 0x270241aa, 0xbe0b1010, 0xc90c2086, 0x5768b525, 0x206f85b3, 0xb966d409, 0xce61e49f,
   0x5edef90e, 0x29d9c998, 0xb0d09822, 0xc7d7a8b4, 0x59b33d17, 0x2eb40d81, 0xb7bd5c3b, 0xc0ba6cad,
   0xedb88320, 0x9abfb3b6, 0x03b6e20c, 0x74b1d29a, 0xead54739, 0x9dd277af, 0x04db2615, 0x73dc1683,
   0xe3630b12, 0x94643b84, 0x0d6d6a3e, 0x7a6a5aa8, 0xe40ecf0b, 0x9309ff9d, 0x0a00ae27, 0x7d079eb1,
   0xf00f9344, 0x8708a3d2, 0x1e01f268, 0x6906c2fe, 0xf762575d, 0x806567cb, 0x196c3671, 0x6e6b06e7,
   0xfed41b76, 0x89d32be0, 0x10da7a5a, 0x67dd4acc, 0xf9b9df6f, 0x8ebeeff9, 0x17b7be43, 0x60b08ed5,
   0xd6d6a3e8, 0xa1d1937e, 0x38d8c2c4, 0x4fdff252, 0xd1bb67f1, 0xa6bc5767, 0x3fb506dd, 0x48b2364b,
   0xd80d2bda, 0xaf0a1b4c, 0x36034af6, 0x41047a60, 0xdf60efc3, 0xa867df55, 0x316e8eef, 0x4669be79,
   0xcb61b38c, 0xbc66831a, 0x256fd2a0, 0x5268e236, 0xcc0c7795, 0xbb0b4703, 0x220216b9, 0x5505262f,
   0xc5ba3bbe, 0xb2bd0b28, 0x2bb45a92, 0x5cb36a04, 0xc2d7ffa7, 0xb5d0cf31, 0x2cd99e8b, 0x5bdeae1d,
   0x9b64c2b0, 0xec63f226, 0x756aa39c, 0x026d930a, 0x9c0906a9, 0xeb0e363f, 0x72076785, 0x05005713,
   0x95bf4a82, 0xe2b87a14, 0x7bb12bae, 0x0cb61b38, 0x92d28e9b, 0xe5d5be0d, 0x7cdcefb7, 0x0bdbdf21,
   0x86d3d2d4, 0xf1d4e242, 0x68ddb3f8, 0x1fda836e, 0x81be16cd, 0xf6b9265b, 0x6fb077e1, 0x18b74777,
   0x88085ae6, 0xff0f6a70, 0x66063bca, 0x11010b5c, 0x8f659eff, 0xf862ae69, 0x616bffd3, 0x166ccf45,
   0xa00ae278, 0xd70dd2ee, 0x4e048354, 0x3903b3c2, 0xa7672661, 0xd06016f7, 0x4969474d, 0x3e6e77db,
   0xaed16a4a, 0xd9d65adc, 0x40df0b66, 0x37d83bf0, 0xa9bcae53, 0xdebb9ec5, 0x47b2cf7f, 0x30b5ffe9,
   0xbdbdf21c, 0xcabac28a, 0x53b39330, 0x24b4a3a6, 0xbad03605, 0xcdd70693, 0x54de5729, 0x23d967bf,
   0xb3667a2e, 0xc4614ab8, 0x5d681b02, 0x2a6f2b94, 0xb40bbe37, 0xc30c8ea1, 0x5a05df1b, 0x2d02ef8d
};


#define GPT_UNKNOWN_GUID     "Custom  GPT GUID!"


// Count and optional display GPT partition table array (PTA), from memory
static ULONG dfsGptShowPta                      // RET   nr of valid GPT types
(
   S_GPTENTRY          ptArray[],               // IN    allocated PT array
   ULONG               ptaEntries,              // IN    max nr of entries
   ULONG               ptaEntrySize,            // IN    size of entry in bytes
   BOOL                verbose                  // IN    display each partition
);

// Sort PTA entries on starting PSN of each partition
static void dfsGptSortPta                       // RET   nr of valid GPT types
(
   S_GPTENTRY          ptArray[],               // IN    allocated PT array
   ULONG               ptaEntries,              // IN    max nr of entries
   ULONG               ptaEntrySize             // IN    size of entry in bytes
);

// Compare GPT ENTRY elements for qsort, sort on start-sector except 0
static int dfsComparePtaEntry
(
   const void         *one,
   const void         *two
);

// Read GPT partition table array (PTA), starting from supplied first PSN
static ULONG dfsGptReadPta                      // RET   alloc/read errors
(
   ULN64               ptaPsn,                  // IN    PSN of table sector
   ULONG               ptaEntries,              // IN    max nr of entries
   ULONG               ptaEntrySize,            // IN    size of entry in bytes
   S_GPTENTRY         *ptArray[]                // OUT   allocated PT array
);                                              //       (use TxFreeMem!)


// Write GPT partition table array (PTA), starting from supplied first PSN
static ULONG dfsGptWritePta                     // RET   write errors
(
   ULN64               ptaPsn,                  // IN    PSN of table sector
   ULONG               ptaEntries,              // IN    max nr of entries
   ULONG               ptaEntrySize,            // IN    size of entry in bytes
   S_GPTENTRY          ptArray[]                // IN    PT array sectors
);

// Calculate a GPT-stype 32-bit CRC value over specified byte array
static ULONG dfsGptCrc32                        // RET   32-bit GPT CRC
(
   BYTE               *data,                    // IN    data to calculate CRC for
   ULONG               len                      // IN    length of the data
);


/*****************************************************************************/
// Return ptr to 17-char description string for the given GUID partition-type
/*****************************************************************************/
char *dfsGptGuidDescription                     // RET   ptr to description
(
   DFS_GUID            typeGuid                 // IN    DFS_GUID byte array
)
{
   if      (GptGuidMatch( typeGuid, dfsGuid_APFS_CONTAINER     )) return "APFS  Container  ";
   else if (GptGuidMatch( typeGuid, dfsGuid_APPLE_BOOT         )) return "Apple Boot       ";
   else if (GptGuidMatch( typeGuid, dfsGuid_APPLE_CORE_STORAGE )) return "Apple CoreStorage";
   else if (GptGuidMatch( typeGuid, dfsGuid_APPLE_LABEL        )) return "Apple Label      ";
   else if (GptGuidMatch( typeGuid, dfsGuid_APPLE_RAID         )) return "Apple RAID       ";
   else if (GptGuidMatch( typeGuid, dfsGuid_APPLE_RAID_OFFLINE )) return "Apple RAIDoffline";
   else if (GptGuidMatch( typeGuid, dfsGuid_APPLE_TV_RECOVERY  )) return "Apple TV Recovery";
   else if (GptGuidMatch( typeGuid, dfsGuid_APPLE_UFS          )) return "Apple UFS        ";
   else if (GptGuidMatch( typeGuid, dfsGuid_APPLE_ZFS          )) return "Apple ZFS        ";
   else if (GptGuidMatch( typeGuid, dfsGuid_BIOS_BOOT          )) return "BIOS Boot (noEFI)";
   else if (GptGuidMatch( typeGuid, dfsGuid_CHROME_FUTURE      )) return "Chrome Future use";
   else if (GptGuidMatch( typeGuid, dfsGuid_CHROME_KERNEL      )) return "Chrome Kernel    ";
   else if (GptGuidMatch( typeGuid, dfsGuid_CHROME_ROOTFS      )) return "Chrome Root-FS   ";
   else if (GptGuidMatch( typeGuid, dfsGuid_DFSEE_UNKNOWN      )) return "DFSee / Unknown  ";
   else if (GptGuidMatch( typeGuid, dfsGuid_EFI_SYSTEM         )) return "EFI System  (ESP)";
   else if (GptGuidMatch( typeGuid, dfsGuid_FREEBSD_BOOT       )) return "FreeBSD Boot     ";
   else if (GptGuidMatch( typeGuid, dfsGuid_FREEBSD_DATA       )) return "FreeBSD Data     ";
   else if (GptGuidMatch( typeGuid, dfsGuid_FREEBSD_SWAP       )) return "FreeBSD Swap     ";
   else if (GptGuidMatch( typeGuid, dfsGuid_FREEBSD_UFS        )) return "FreeBSD UFS      ";
   else if (GptGuidMatch( typeGuid, dfsGuid_FREEBSD_VINUM_VM   )) return "FreeBSD Vinum VM ";
   else if (GptGuidMatch( typeGuid, dfsGuid_FREEBSD_ZFS        )) return "FreeBSD ZFS      ";
   else if (GptGuidMatch( typeGuid, dfsGuid_HP_UX_DATA         )) return "HP-UX Data       ";
   else if (GptGuidMatch( typeGuid, dfsGuid_HP_UX_SERVICE      )) return "HP-UX Service    ";
   else if (GptGuidMatch( typeGuid, dfsGuid_IBM_GENPARALLEL_FS )) return "IBM GenParallelFS";
   else if (GptGuidMatch( typeGuid, dfsGuid_LENOVO_BOOT        )) return "LENOVO Boot (ESP)";
   else if (GptGuidMatch( typeGuid, dfsGuid_LINUX_DATA         )) return "Linux Data       ";
   else if (GptGuidMatch( typeGuid, dfsGuid_LINUX_EXTEND_BOOT  )) return "Linux Extend-Boot";
   else if (GptGuidMatch( typeGuid, dfsGuid_LINUX_HOME         )) return "Linux /home      ";
   else if (GptGuidMatch( typeGuid, dfsGuid_LINUX_LUKS         )) return "Linux LUKS encr. ";
   else if (GptGuidMatch( typeGuid, dfsGuid_LINUX_LVM          )) return "Linux LVM        ";
   else if (GptGuidMatch( typeGuid, dfsGuid_LINUX_PLAIN_DMCRYPT)) return "Linux Pln-DmCrypt";
   else if (GptGuidMatch( typeGuid, dfsGuid_LINUX_RAID         )) return "Linux RAID       ";
   else if (GptGuidMatch( typeGuid, dfsGuid_LINUX_RESERVED     )) return "Linux Reserved   ";
   else if (GptGuidMatch( typeGuid, dfsGuid_LINUX_ROOT_ARM_32  )) return "Linux Root arm-32";
   else if (GptGuidMatch( typeGuid, dfsGuid_LINUX_ROOT_ARM_64  )) return "Linux Root arm-64";
   else if (GptGuidMatch( typeGuid, dfsGuid_LINUX_ROOT_X86_32  )) return "Linux Root x86-32";
   else if (GptGuidMatch( typeGuid, dfsGuid_LINUX_ROOT_X86_64  )) return "Linux Root x86-64";
   else if (GptGuidMatch( typeGuid, dfsGuid_LINUX_SRV_DATA     )) return "Linux /srv data  ";
   else if (GptGuidMatch( typeGuid, dfsGuid_LINUX_SWAP         )) return "Linux Swap       ";
   else if (GptGuidMatch( typeGuid, dfsGuid_MAC_OS_X_HFS_PLUS  )) return "Mac OS X HFS+    ";
   else if (GptGuidMatch( typeGuid, dfsGuid_MBR_PARTITIONING   )) return "MBR Partitioning ";
   else if (GptGuidMatch( typeGuid, dfsGuid_MICROSOFT_RESERVED )) return "MS Reserved (MSR)";
   else if (GptGuidMatch( typeGuid, dfsGuid_MIDNIGHTBSD_BOOT   )) return "MidnightBSD Boot ";
   else if (GptGuidMatch( typeGuid, dfsGuid_MIDNIGHTBSD_DATA   )) return "MidnightBSD Data ";
   else if (GptGuidMatch( typeGuid, dfsGuid_MIDNIGHTBSD_SWAP   )) return "MidnightBSD Swap ";
   else if (GptGuidMatch( typeGuid, dfsGuid_MIDNIGHTBSD_UFS    )) return "MidnightBSD UFS  ";
   else if (GptGuidMatch( typeGuid, dfsGuid_MIDNIGHTBSD_VINUM  )) return "MidnightBSD Vinum";
   else if (GptGuidMatch( typeGuid, dfsGuid_MIDNIGHTBSD_ZFS    )) return "MidnightBSD ZFS  ";
   else if (GptGuidMatch( typeGuid, dfsGuid_NETBSD_CONCAT      )) return "NetBSD  Concat   ";
   else if (GptGuidMatch( typeGuid, dfsGuid_NETBSD_ENCRYPTED   )) return "NetBSD  Encrypted";
   else if (GptGuidMatch( typeGuid, dfsGuid_NETBSD_FFS         )) return "NetBSD  FFS      ";
   else if (GptGuidMatch( typeGuid, dfsGuid_NETBSD_LFS         )) return "NetBSD  LFS      ";
   else if (GptGuidMatch( typeGuid, dfsGuid_NETBSD_RAID        )) return "NetBSD  RAID     ";
   else if (GptGuidMatch( typeGuid, dfsGuid_NETBSD_SWAP        )) return "NetBSD  Swap     ";
   else if (GptGuidMatch( typeGuid, dfsGuid_OPENBSD_DATA       )) return "OpenBSD Data     ";
   else if (GptGuidMatch( typeGuid, dfsGuid_OS2_ARCAOS_TYPE_1  )) return "OS2 ArcaOS Type 1";
   else if (GptGuidMatch( typeGuid, dfsGuid_QNX_POWERSAFE      )) return "QNX Powersafe    ";
   else if (GptGuidMatch( typeGuid, dfsGuid_SOLARIS_ALTERNATE  )) return "Solaris Alternate";
   else if (GptGuidMatch( typeGuid, dfsGuid_SOLARIS_BACKUP     )) return "Solaris Backup   ";
   else if (GptGuidMatch( typeGuid, dfsGuid_SOLARIS_BOOT       )) return "Solaris Boot     ";
   else if (GptGuidMatch( typeGuid, dfsGuid_SOLARIS_HOME       )) return "Solaris /home    ";
   else if (GptGuidMatch( typeGuid, dfsGuid_SOLARIS_RESERVED1  )) return "Solaris Reserved1";
   else if (GptGuidMatch( typeGuid, dfsGuid_SOLARIS_RESERVED2  )) return "Solaris Reserved2";
   else if (GptGuidMatch( typeGuid, dfsGuid_SOLARIS_RESERVED3  )) return "Solaris Reserved3";
   else if (GptGuidMatch( typeGuid, dfsGuid_SOLARIS_RESERVED4  )) return "Solaris Reserved4";
   else if (GptGuidMatch( typeGuid, dfsGuid_SOLARIS_RESERVED5  )) return "Solaris Reserved5";
   else if (GptGuidMatch( typeGuid, dfsGuid_SOLARIS_ROOT       )) return "Solaris Root     ";
   else if (GptGuidMatch( typeGuid, dfsGuid_SOLARIS_SWAP       )) return "Solaris Swap     ";
   else if (GptGuidMatch( typeGuid, dfsGuid_SOLARIS_USR        )) return "Solaris /usr     ";
   else if (GptGuidMatch( typeGuid, dfsGuid_SOLARIS_VAR        )) return "Solaris /var     ";
   else if (GptGuidMatch( typeGuid, dfsGuid_SONY_BOOT          )) return "SONY Boot   (ESP)";
   else if (GptGuidMatch( typeGuid, dfsGuid_VMWARE_COREDUMP    )) return "VMware CoreDump  ";
   else if (GptGuidMatch( typeGuid, dfsGuid_VMWARE_VMFS        )) return "VMware VMFS data ";
   else if (GptGuidMatch( typeGuid, dfsGuid_WINDOWS_BASIC_DATA )) return "Windows BasicData";
   else if (GptGuidMatch( typeGuid, dfsGuid_WINDOWS_LDM_META   )) return "Windows LDM Meta ";
   else if (GptGuidMatch( typeGuid, dfsGuid_WINDOWS_LDM_DATA   )) return "Windows LDM Data ";
   else if (GptGuidMatch( typeGuid, dfsGuid_WINDOWS_RECOVERY   )) return "Windows Recovery ";
   else                                                           return GPT_UNKNOWN_GUID;
}                                               // end 'dfsGptGuidDescription'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Return most likely related part-GUID for the given MBR type byte
/*****************************************************************************/
BYTE *dfsMbrType2GptGuid                        // RET   ptr to binary GUID
(
   BYTE                mbrType                  // IN    MBR style type byte
)
{
   BYTE               *Uuid;

   switch (mbrType)
   {
      case DFS_P_EFI_ESP   : Uuid = dfsGuid_EFI_SYSTEM;         break;
      case DFS_P_FAT32X    : Uuid = dfsGuid_MICROSOFT_RESERVED; break;
      case DFS_P_INST_FS   : Uuid = dfsGuid_WINDOWS_BASIC_DATA; break;
      case DFS_P_WIN2XPLDM : Uuid = dfsGuid_WINDOWS_LDM_DATA;   break;
      case DFS_P_IFSHIDDEN : Uuid = dfsGuid_WINDOWS_RECOVERY;   break;
      case DFS_P_LINUXNATV : Uuid = dfsGuid_LINUX_DATA;         break;
      case DFS_P_LIN_NONFS : Uuid = dfsGuid_LINUX_RAID;         break;
      case DFS_P_LINUXRAID : Uuid = dfsGuid_LINUX_RAID;         break;
      case DFS_P_SWAPSOLAR : Uuid = dfsGuid_LINUX_SWAP;         break;
      case DFS_P_LINUX_LVM : Uuid = dfsGuid_LINUX_LVM;          break;
      case DFS_P_LINUXLUKS : Uuid = dfsGuid_LINUX_LUKS;         break;
      case DFS_P_MACXHFSP  : Uuid = dfsGuid_MAC_OS_X_HFS_PLUS;  break;
      case DFS_P_MACXBOOT  : Uuid = dfsGuid_APPLE_BOOT;         break;
      case DFS_P_APFS_CONT : Uuid = dfsGuid_APFS_CONTAINER;     break;
      case DFS_P_SOLARBOOT : Uuid = dfsGuid_SOLARIS_BOOT;       break;
      default:               Uuid = dfsGuid_DFSEE_UNKNOWN;      break;
   }
   return( Uuid);
}                                               // end 'dfsMbrType2GptGuid'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create GUID value based on TIME and RANDOM values, 'df5' plus given ID
/*****************************************************************************/
void dfsGptMkGuid
(
   USHORT              id,                      // IN    object ID (modulo 64)
   BYTE               *guid                     // OUT   16 bytes GUID value
)
{
   static BOOL         seeded = FALSE;
   int                 i;

   ENTER();

   if (!seeded)
   {
      srand( time( NULL));
      seeded = TRUE;
   }

   //- set bytes 0..3 to the current time (time_t standard)
   time( (time_t *) guid);

   //- set bytes 4..5 to execution time   (clock_t standard)
   *((clock_t *) (&guid[4])) = (USHORT) clock();

   //- set bytes 6..7 to UUID type-1 and DFSee signature   (little endian!)
   guid[ 7] = 0x4d;                             // 4 signals 'random' UUID
   guid[ 6] = 0xf5;                             // rest is 'df5' signature

   //- set byte  8    to UUID type-2 and clipped object ID (little endian!)
   guid[ 8] = (0x80 | (id & 0x3F));             // range 0x80 .. 0xBF

   //- set bytes 9..15 to a randomized byte sequence using rand()

   for (i = 9; i <= 15; i++)
   {
      guid[ i] = ((rand() >> 7) & 0xFF);        // use upper-8 from 15 bits
   }
   TRACES(("Object ID: %hu, GUID: %s\n", id, dfsFsUuidValueString( guid)));
   VRETURN();
}                                               // end 'dfsGptMkGuid'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Determine if given LSN is inside the GPT area at start or end of the disk
/*****************************************************************************/
BOOL dfsGptLsnInTableArea                       // RET   LSN is inside GPT area
(
   DFSDISKINFO        *d,                       // IN    current disk info
   ULN64               lsn,                     // IN    LSN to check area for
   ULN64              *start,                   // OUT   start LSN of area
   ULN64              *final                    // OUT   final LSN of area
)
{
   BOOL                rc = FALSE;


   if ((d->gptHeader != NULL) && (d->gptArray != NULL))
   {
      S_GPTHDR      *gptHeader = (S_GPTHDR *) d->gptHeader;

      if      (lsn < gptHeader->firstPSN)            // in MBR/GPT area at start
      {
         *start = 0;
         *final = gptHeader->firstPSN -1;
         rc = TRUE;
      }
      else if (lsn > gptHeader->lastPSN)
      {
         *start = gptHeader->lastPSN  +1;
         *final = d->sectors -1;
         rc = TRUE;
      }
   }
   BRETURN (rc);
}                                               // end 'dfsGptLsnInTableArea'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read and validate GPT header sector, and attached partition table array (PTA)
/*****************************************************************************/
ULONG dfsGptReadValidateLba                     // RET   OK, bad-sig, bad-crc
(
   ULN64               lba,                     // IN    LBA for the HDR sector
   S_GPTHDR          **hdrSect,                 // OUT   GPT header sector  or NULL
   S_GPTENTRY         *ptArray[]                // OUT   allocated PT Array or NULL
)
{
   ULONG               rc = NO_ERROR;           // rc, alloc / CRC errors
   S_GPTHDR           *sd = NULL;

   ENTER();
   TRACES(("lba:0x%llx\n", lba));

   if ((sd = (S_GPTHDR *) TxAlloc( 1, dfsGetSectorSize())) != NULL)
   {
      if ((rc = dfstReadPsn( DFSTORE, lba, 1, (BYTE *) sd)) == NO_ERROR)
      {
         rc = dfsGptValidateHdr( sd, ptArray);
         if ((rc == NO_ERROR) || (rc == DFS_BAD_STRUCTURE))
         {
            if (hdrSect != NULL)
            {
               TRACES(("hdrSect struct at: %8.8x\n", sd));
               *hdrSect = sd;                   // pass to caller
            }
            else
            {
               TxFreeMem( sd);                  // free when not desired
            }
         }
         else
         {
            TxFreeMem( sd);                     // free when validate failed
         }
      }
      else
      {
         TxFreeMem( sd);                        // free when read failed
      }
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsGptReadValidateLba'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Validate GPT header sector (usually PSN 1, after MBR) and return PT array
/*****************************************************************************/
ULONG dfsGptValidateHdr                         // RET   OK, bad-sig, bad-crc
(
   S_GPTHDR           *sd,                      // IN    GPT header sector
   S_GPTENTRY         *ptArray[]                // OUT   allocated PT array or NULL
)
{
   ULONG               rc = NO_ERROR;           // rc, alloc / CRC errors
   S_GPTENTRY         *pt = NULL;
   ULONG               headCrc;
   ULONG               gptaCrc;                 // calculated over PT array

   ENTER();

   if ((sd->Signature1 == SV_GPTHDR1)       &&  // Sanity check first ...
       (sd->Signature2 == SV_GPTHDR2)       &&
       (sd->hdrSize    <= dfsGetSectorSize()))
   {
      gptaCrc    = sd->hdrCRC;                            //- temporarily save CRC
      sd->hdrCRC = 0;                                     //- must be 0 during calc
      headCrc    = dfsGptCrc32((BYTE *) sd, sd->hdrSize); //- calculate header CRC
      sd->hdrCRC = gptaCrc;                               //- restore the CRC

      rc = dfsGptReadPta( sd->ptaPSN, sd->ptaEntries, sd->ptaEntrySize, &pt);
      if (rc == NO_ERROR)
      {
         gptaCrc = dfsGptCrc32( (BYTE *) pt, (sd->ptaEntries * sd->ptaEntrySize));
         if ((gptaCrc != sd->ptaCRC) || (headCrc != sd->hdrCRC))
         {
            rc = DFS_BAD_STRUCTURE;
         }
      }
      if (ptArray != NULL)
      {
         TRACES(("ptArray struct at: %8.8x\n", pt));
         *ptArray = pt;
      }
      else
      {
         TxFreeMem( pt);
      }
   }
   else
   {
      rc = DFS_ST_MISMATCH;
   }
   RETURN (rc);
}                                               // end 'dfsGptValidateHdr'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display GPT header sector (usually PSN 1, after MBR)
/*****************************************************************************/
ULONG dfsGptHdr                                 // RET   OK, bad-sig, bad-crc
(
   BYTE               *sector                   // IN    Sector buffer
)
{
   ULONG               rc = NO_ERROR;           // rc, alloc / CRC errors
   S_GPTHDR           *sd = (S_GPTHDR *) sector;
   S_GPTENTRY         *pt = NULL;
   ULONG               headCrc;
   ULONG               gptaCrc;                 // calculated over PT array
   TXTM                text;

   ENTER();

   gptaCrc    = sd->hdrCRC;                            //- temporarily save CRC
   sd->hdrCRC = 0;                                     //- must be 0 during calc
   headCrc    = dfsGptCrc32( sector, sd->hdrSize);     //- calculate header CRC
   sd->hdrCRC = gptaCrc;                               //- restore the CRC

   rc = dfsGptReadPta( sd->ptaPSN, sd->ptaEntries, sd->ptaEntrySize, &pt);
   if (rc == NO_ERROR)
   {
      gptaCrc = dfsGptCrc32( (BYTE *) pt, (sd->ptaEntries * sd->ptaEntrySize));
   }

   TxCopy( text, (char *) sd, GPTHDR_SIG_LEN +1);

   TxPrint( "GPT hdr signature : %s%s%s                 Header revision: 0x%8.8x\n",
                                 CBC, text, CNN,          sd->Revision);
   TxPrint( "GPT  header  size : %-3u bytes                header  CRC-32 : 0x%s%8.8x%s  (%s)\n",
                                 sd->hdrSize,  (headCrc == sd->hdrCRC) ? CBG : CBR, sd->hdrCRC, CNN,
                                               (headCrc == sd->hdrCRC) ? "ok" : "bad");
   TxPrint( "PSN  start of PTA : 0x%16.16llx       PTArray CRC-32 : 0x%s%8.8x%s  (%s)\n",
                                 sd->ptaPSN,   (gptaCrc == sd->ptaCRC) ? CBG : CBR, sd->ptaCRC, CNN,
                                               (gptaCrc == sd->ptaCRC) ? "ok" : "bad");
   TxPrint( "PSN  this  header : 0x%16.16llx       PSN alt. header: 0x%16.16llx\n",
                                 sd->thisPSN,             sd->althPSN);
   TxPrint( "PSN  first usable : 0x%16.16llx       PSN last usable: 0x%16.16llx\n",
                                 sd->firstPSN,            sd->lastPSN);
   TxPrint( "PTA nr of entries : %-8u                 PTA entry size : %u bytes\n",
                                  sd->ptaEntries,          sd->ptaEntrySize);

   dfsSz64( "Calculated   size : ", sd->althPSN, "\n");

   sprintf( text, "GUID for the disk : %s\n", dfsFsUuidValueString( sd->diskGuid));

   if ((rc == NO_ERROR) && ((gptaCrc != sd->ptaCRC) || (headCrc != sd->hdrCRC) ||
                     (sd->Signature1 != SV_GPTHDR1) || (sd->Signature2 != SV_GPTHDR2)))
   {
      TxPrint( "\n");
      if ((sd->Signature1 != SV_GPTHDR1) || (sd->Signature2 != SV_GPTHDR2)  )
      {
         TxPrint( "GPT header sector has an invalid signature, no GPT info present ?\n");
      }
      if (headCrc != sd->hdrCRC)
      {
         TxPrint( "CRC over GPT header does not match CRC stored in the header itself!\n");
      }
      if (gptaCrc != sd->ptaCRC)
      {
         TxPrint( "CRC over Partition Table Array does not match CRC stored in header!\n");
      }
      rc = DFS_BAD_STRUCTURE;
   }
   if (pt != NULL)
   {
      dfsGptShowPta( pt, sd->ptaEntries, sd->ptaEntrySize, TRUE);
      TxFreeMem( pt);
   }
   RETURN (rc);
}                                               // end 'dfsGptHdr'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Count and optional display GPT partition table array (PTA), from memory
/*****************************************************************************/
static ULONG dfsGptShowPta                      // RET   nr of valid GPT types
(
   S_GPTENTRY          ptArray[],               // IN    allocated PT array
   ULONG               ptaEntries,              // IN    max nr of entries
   ULONG               ptaEntrySize,            // IN    size of entry in bytes
   BOOL                verbose                  // IN    display each partition
)
{
   ULONG               rc = 0;                  // number of valid GPT partitions
   ULONG               i;
   ULN64               partSize;
   BYTE               *ep;                      // BYTE aligned entry pointer
   S_GPTENTRY         *sd;
   TXTM                text;
   char               *gptDescription;          // GPT partition description

   ENTER();

   for (i = 0,           ep  = (BYTE *) ptArray;
        i < ptaEntries;
        i++,             ep += ptaEntrySize)
   {
      if (!TxAreaEmpty( ep, sizeof(S_GPTENTRY), 0))
      {
         sd = (S_GPTENTRY *) ep;                // PTA entry structure pointer

         gptDescription = dfsGptGuidDescription( sd->typeGuid);
         if (strcmp( gptDescription, GPT_UNKNOWN_GUID) != 0)
         {
            rc++;                               // count as valid
         }

         if (verbose)
         {
            sprintf( text, "\nGUID part %s%3u%s type: ", CBY, i, CNN);
            dfstrFsUuidValue( text, sd->typeGuid, TRUE);
            TxPrint( "%s = %s%s%s\n", text,  CBC, gptDescription, CNN);

            strcpy( text, "GUID for this part: ");
            dfstrFsUuidValue( text, sd->partGuid, TRUE);
            TxPrint( "%s\n", text);

            TxPrint(  "Partition   start : 0x%s%16.16llx%s    end : 0x%16.16llx\n",
                    (i == 1) ? CBG : CNN,  sd->firstPSN, CNN,       sd->lastPSN);
            if      (i == 1)
            {
               nav.down = (ULONG) sd->firstPSN; // default to start of 2nd partition
            }
            TxPrint(  "P-attribute flags : 0x%16.16llx\n", sd->attrFlags);
            partSize = sd->lastPSN - sd->firstPSN +1;
            dfsSz64( "Calculated   size : ", partSize, "\n");

            strcpy( text, "GPT partition name: ");
            strcat( text,  CBM);
            TxUnicAsciiAppend( sd->partName, text, GPT_PNAME_LEN * 2);
            TxPrint( "%-*.*s%s  PID : %02.2hu\n",  GPT_PNAME_LEN + 43,
                                                   GPT_PNAME_LEN + 43, text, CNN,
                                dfsDiskPsn2PartId( SINF->disknr, sd->firstPSN));
         }
      }
   }
   if (verbose)
   {
      TxPrint( "\nThe GPT partition table array above, has %u used entries out of %u.\n",
                                                     rc,                ptaEntries);
   }
   RETURN( rc);
}                                               // end 'dfsGptShowPta'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Sort PTA entries on starting PSN of each partition
/*****************************************************************************/
static void dfsGptSortPta                       // RET   nr of valid GPT types
(
   S_GPTENTRY          ptArray[],               // IN    allocated PT array
   ULONG               ptaEntries,              // IN    max nr of entries
   ULONG               ptaEntrySize             // IN    size of entry in bytes
)
{
   ENTER();

   qsort( ptArray, ptaEntries, ptaEntrySize, dfsComparePtaEntry);

   VRETURN();
}                                               // end 'dfsGptSortPta'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Compare GPT ENTRY elements for qsort, sort on start-sector except 0
/*****************************************************************************/
static int dfsComparePtaEntry
(
   const void         *one,
   const void         *two
)
{
   S_GPTENTRY         *p1  = (S_GPTENTRY *) one;
   S_GPTENTRY         *p2  = (S_GPTENTRY *) two;
   ULN64               u1  = p1->firstPSN - 1;
   ULN64               u2  = p2->firstPSN - 1;  // makes 0 largest value!
   int                 res = 0;

   TRACES(( "ComparePtaEntry 1: %12.12llx 2: %12.12llx\n", u1, u2));

   if (u1 != u2)
   {
      res = (int) ((u1 < u2) ? -1 : +1);
   }
   return (res);
}                                               // end 'dfsComparePtaEntry'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Select disk and get reference to a GPT entry, and optional header/table info
/*****************************************************************************/
S_GPTENTRY *dfsGptDiskGetEntry                  // RET   GPT entry ref or NULL
(
   FDSK_CB_INFO       *cbp,                     // IN    info, disknr, partnr
   S_GPTHDR          **hdr,                     // OUT   GPT header, or NULL
   S_GPTENTRY        **table                    // OUT   GPT table,  or NULL
)
{
   S_GPTENTRY         *rc = NULL;
   DFSDISKINFO        *d;                       // ptr to disk info
   S_GPTHDR           *gptHdr;                  // GPT header for the disk
   S_GPTENTRY         *gptTable;

   ENTER();
   TRACES(("Disk:%hu  entry:%hu\n", cbp->disknr, cbp->partnr));

   //- Get diskinfo and make sure it is the current object for read/write
   if (((d = dfsGetDiskInfo(cbp->disknr)) != NULL) &&
       ((    dfsSelectDisk( cbp->disknr, FALSE, FALSE)) == NO_ERROR))
   {
      if (((gptHdr   = (S_GPTHDR   *) d->gptHeader) != NULL) &&
          ((gptTable = (S_GPTENTRY *) d->gptArray ) != NULL)  )
      {
         TRACES(("Header:%p  Table:%p\n", gptHdr, gptTable));

         if (cbp->partnr < gptHdr->ptaEntries)
         {
            if (hdr)
            {
               *hdr = gptHdr;
            }
            if (table)
            {
               *table = gptTable;
            }
            rc = &(gptTable[ cbp->partnr]);
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsGptDiskGetEntry'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Find the NEXT used entry in a GPT array, starting from the current one
/*****************************************************************************/
BOOL dfsGptNextEntry                            // RET    cbd->partnr changed
(
   FDSK_CB_INFO       *cbp                      // INOUT info, current partnr
)                                               //       and partition Start
{
   BOOL                rc = FALSE;              // number of valid GPT partitions
   S_GPTHDR           *gptHdr;                  // GPT header for the disk
   DFSDISKINFO        *d;

   ENTER();
   TRACES(("old entry: %u  start-psn:%llx\n", cbp->partnr, cbp->sn));

   if ((( d      = dfsGetDiskInfo(cbp->disknr)) != NULL) &&
        ((gptHdr = (S_GPTHDR *) d->gptHeader)   != NULL) &&
         (                      d->gptArray     != NULL)  )
   {
      int              i;
      BYTE            *ep;                      // BYTE aligned entry pointer

      for (i = cbp->partnr + 1; i < gptHdr->ptaEntries; i++)
      {
         ep = (BYTE *) d->gptArray + (i * gptHdr->ptaEntrySize);
         TRACES(("entry:%u, address:%p\n", i, ep));
         TRHEXS( 500,  ep, sizeof(S_GPTENTRY), "gpt_entry");
         if (!TxAreaEmpty( ep, sizeof(S_GPTENTRY), 0))
         {
            cbp->sn     = ((S_GPTENTRY *) ep)->firstPSN;
            cbp->partnr = i;
            rc = TRUE;                          // found next non-empty entry
            break;
         }
      }
   }
   TRACES(("new entry: %u  start-psn:%llx\n", cbp->partnr, cbp->sn));
   BRETURN( rc);
}                                               // end 'dfsGptNextEntry'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Find the PREVIOUS used entry in a GPT array, starting from the current one
/*****************************************************************************/
BOOL dfsGptPrevEntry                            // RET    cbd->partnr changed
(
   FDSK_CB_INFO       *cbp                      // INOUT info, current partnr
)
{
   BOOL                rc = FALSE;              // number of valid GPT partitions
   S_GPTHDR           *gptHdr;                  // GPT header for the disk
   DFSDISKINFO        *d;

   ENTER();
   TRACES(("old entry: %u  start-psn:%llx\n", cbp->partnr, cbp->sn));

   if ((( d      = dfsGetDiskInfo(cbp->disknr)) != NULL) &&
        ((gptHdr = (S_GPTHDR *) d->gptHeader)   != NULL) &&
         (                      d->gptArray     != NULL)  )
   {
      int              i;
      BYTE            *ep;                      // BYTE aligned entry pointer

      for (i  = cbp->partnr - 1; i >= 0; i--)
      {
         ep = (BYTE *) d->gptArray + (i * gptHdr->ptaEntrySize);
         TRACES(("entry:%u, address:%p\n", i, ep));
         TRHEXS( 500,  ep, sizeof(S_GPTENTRY), "gpt_entry");
         if (!TxAreaEmpty( ep, sizeof(S_GPTENTRY), 0))
         {
            cbp->sn     = ((S_GPTENTRY *) ep)->firstPSN;
            cbp->partnr = i;
            rc = TRUE;                          // found prev non-empty entry
            break;
         }
      }
   }
   TRACES(("new entry: %u  start-psn:%llx\n", cbp->partnr, cbp->sn));
   BRETURN( rc);
}                                               // end 'dfsGptPrevEntry'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read GPT header and PTA to disk-info, from alternate or primary location
/*****************************************************************************/
ULONG dfsGptReadAlternate                       // RET   OK, bad-sig, bad-crc
(
   DFSDISKINFO        *d,                       // IN    Disk info
   ULN64               altPsn,                  // IN    LBA for alternate
   BOOL               *priValid,                // OUT   primary   is valid (too)
   BOOL               *altValid                 // OUT   alternate is valid (too)
)                                               //       for MBR guard partition
{
   ULONG               rc = NO_ERROR;           // rc, alloc / CRC errors
   S_GPTHDR           *sd = NULL;
   S_GPTENTRY         *pt = NULL;

   ENTER();
   TRACES(("altPsn:0x%llx\n", altPsn));

   TxFreeMem( d->gptHeader);                    // free possible existing info
   TxFreeMem( d->gptArray);

   //- Read from alternate location
   if (dfsGptReadValidateLba( altPsn, &sd, &pt) == NO_ERROR)
   {
      *altValid = TRUE;
   }
   else
   {
      *altValid = FALSE;                        // including CRC errors
   }

   //- Read from primary location
   rc = dfsGptReadValidateLba( GPTHDR_PRIMARY_PSN,
                              (S_GPTHDR   **) &d->gptHeader,
                              (S_GPTENTRY **) &d->gptArray);
   if (rc == NO_ERROR)
   {
      *priValid = TRUE;
      TxFreeMem( sd);                           // free alternate info
      TxFreeMem( pt);

      if (*altValid == FALSE)                   // Alt was bad (end of disk)
      {                                         // retry with end-of-guard
         if (dfsGptReadValidateLba( ((S_GPTHDR *) d->gptHeader)->althPSN, NULL, NULL) == NO_ERROR)
         {
            *altValid = TRUE;                   // ALT is at non-standard
         }                                      // location, but is valid
      }
   }
   else
   {
      *priValid = FALSE;

      if (*altValid)                            // use alternate info ?
      {
         d->gptHeader = sd;
         d->gptArray  = pt;
         rc = NO_ERROR;
      }
   }
   RETURN( rc);
}                                               // end 'dfsGptReadAlternate'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create valid GPT header and partition-arrays, optional recover from existing
/*****************************************************************************/
ULONG dfsGptCreateFromMbr                       // RET   OK, bad-sig, bad-crc
(
   BOOL                recover,                 // IN    try to recover existing
   DFSPARTINFO        *f                        // IN    freespace/partition info
)                                               //       for MBR guard partition
{
   ULONG               rc = NO_ERROR;           // rc, alloc / CRC errors

   ENTER();

   rc = dfsSelectDisk( f->disknr, FALSE, FALSE);
   if (rc == NO_ERROR)
   {
      if (DFSTORE_WRITE_ALLOWED)
      {
         DFSDISKINFO  *d;
         BOOL          primaryOK;
         BOOL          alternateOK;

         TxPrint("Create GPT tables : HDR/PTA on disk %hu, recover existing: %s\n",
                                              f->disknr, (recover) ? "YES" : "NO");
         d = dfsGetDiskInfo( f->disknr);
         if (recover)
         {
            rc = dfsGptReadAlternate( d, f->lastPsn, &primaryOK, &alternateOK);
            if (rc == DFS_BAD_STRUCTURE)        // bad CRC, OK but warning ...
            {
               TxPrint( "\nGPT area recovered successfully but CRC values are not consistent!\n");
               rc = NO_ERROR;
            }
            if (rc == NO_ERROR)
            {
               if (alternateOK == FALSE)
               {
                  TxPrint( "\nPrimary GPT area recovered successfully, alternate area at end was bad!\n");
               }
               else if (primaryOK == FALSE)
               {
                  TxPrint( "\nPrimary GPT area was bad, GPT header and PTA recovered from alternate at end!\n");
               }
               else
               {
                  TxPrint( "\nPrimary GPT area recovered successfully, alternate area at end was OK too.\n");
               }
            }
            else
            {
               TxPrint( "\nRecovery of primary or alternate GPT areas not possible, creating EMPTY GPT.\n");
            }
         }
         else
         {
            rc = DFS_PENDING;                   // create empty from scratch
         }

         if (rc != NO_ERROR)
         {
            S_GPTHDR    *sd = NULL;
            S_GPTENTRY  *pt = NULL;
            ULONG        ptaEntries   = GPT_DEF_PTA_COUNT;
            ULONG        ptaEntrySize = GPT_DEF_PTA_ESIZE;
            ULONG        arraySize;      // size of array in bytes
            ULONG        arraySectors;

            //- No recovery wanted, or recovery failed. Create HDR/PTA from scratch
            if ((sd = (S_GPTHDR *) TxAlloc( 1, dfsGetSectorSize())) != NULL)
            {
               arraySize    = ptaEntries * ptaEntrySize;
               arraySectors = (arraySize + dfsGetSectorSize() -1) / dfsGetSectorSize();

               if ((pt = TxAlloc( arraySectors, dfsGetSectorSize())) != NULL)
               {
                  //- Fill in non-zero HDR fields (Rest and PTA zeroed by Alloc)

                  sd->Signature1    = SV_GPTHDR1;
                  sd->Signature2    = SV_GPTHDR2;
                  sd->Revision      = GPT_DEF_REVISION;
                  sd->hdrSize       = sizeof( S_GPTHDR);
                  sd->thisPSN       = GPTHDR_PRIMARY_PSN;
                  sd->althPSN       = f->lastPsn;
                  sd->firstPSN      = sd->thisPSN + 1 + arraySectors;
                  sd->lastPSN       = sd->althPSN - 1 - arraySectors;
                  sd->ptaPSN        = sd->thisPSN + 1;
                  sd->ptaEntries    = ptaEntries;
                  sd->ptaEntrySize  = ptaEntrySize;

                  dfsGptMkGuid( f->disknr, sd->diskGuid);

                  //- make sure CRC value are OK, for in-memory use (write recalculates)
                  sd->ptaCRC = dfsGptCrc32((BYTE *) pt, (ptaEntries * ptaEntrySize));
                  sd->hdrCRC = dfsGptCrc32((BYTE *) sd, sd->hdrSize);

                  TxFreeMem( d->gptHeader);     // free possible existing info
                  TxFreeMem( d->gptArray);
                  d->gptHeader = sd;            // attach created HDR/PTA
                  d->gptArray  = pt;

                  rc = NO_ERROR;
               }
               else
               {
                  TxFreeMem( sd);               // free when validate failed
                  rc = DFS_ALLOC_ERROR;
               }
            }
            else
            {
               rc = DFS_ALLOC_ERROR;
            }
         }
         if (rc == NO_ERROR)
         {
            rc = dfsGptFixupWriteAll( d->gptHeader, d->gptArray);

            if (rc == NO_ERROR)
            {
               TxPrint("\n");
               dfsGptHdr((BYTE *) d->gptHeader);
            }
         }
      }
      else
      {
         rc = DFS_READ_ONLY;
      }
   }
   RETURN (rc);
}                                               // end 'dfsGptCreateFromMbr'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read GPT partition table array (PTA), starting from supplied first PSN
/*****************************************************************************/
static ULONG dfsGptReadPta                      // RET   alloc/read errors
(
   ULN64               ptaPsn,                  // IN    PSN of table sector
   ULONG               ptaEntries,              // IN    max nr of entries
   ULONG               ptaEntrySize,            // IN    size of entry in bytes
   S_GPTENTRY         *ptArray[]                // OUT   allocated PT array
)                                               //       (use TxFreeMem!)
{
   ULONG               rc = DFS_ALLOC_ERROR;    // rc, sector match
   ULONG               arraySize;               // size of array in bytes
   ULONG               arraySectors;
   BYTE               *ep;                      // BYTE aligned entry pointer

   ENTER();

   *ptArray = NULL;
   if ((ptaEntries   >=   4) && (ptaEntries   <= 4096) &&
       (ptaEntrySize >= 128) && (ptaEntrySize <= 4096)) // sanity check
   {
      arraySize    = ptaEntries * ptaEntrySize;
      arraySectors = (arraySize + dfsGetSectorSize() -1) / dfsGetSectorSize();

      if ((ep = TxAlloc( arraySectors, dfsGetSectorSize())) != NULL)
      {
         if ((rc = dfstReadPsn( DFSTORE, ptaPsn, arraySectors, ep)) == NO_ERROR)
         {
            TRACES(("ptArray at: 0x%llx\n", ep));
            *ptArray = (S_GPTENTRY *) ep;       // pass address of the array
         }
         else
         {
            TxFreeMem( ep);                     // free when read failed
         }
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   else
   {
      rc = DFS_BAD_STRUCTURE;
   }
   RETURN (rc);
}                                               // end 'dfsGptReadPta'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Fixup CRC and Write primary plus alternate GPT header sectors and PTA areas
// Note: Supplied header may be either the primary or the alternate header
/*****************************************************************************/
ULONG dfsGptFixupWriteAll                       // RET   OK, bad-sig, bad-crc
(
   S_GPTHDR           *gptHdr,                  // IN    GPT header sector
   S_GPTENTRY          ptArray[]                // IN    PT array (whole sectors!)
)
{
   ULONG               rc = NO_ERROR;           // rc, alloc / CRC errors
   S_GPTHDR           *sd;                      // IN    GPT header sector

   ENTER();

   if ((gptHdr->Signature1 == SV_GPTHDR1) && (gptHdr->Signature2 == SV_GPTHDR2))
   {
      if ((sd = (S_GPTHDR *) TxAlloc( 1, dfsGetSectorSize())) != NULL)
      {
         //- Copy GPT header strcuture into start of sector (512..4096)
         memcpy((BYTE *) sd, (BYTE *) gptHdr, sizeof( S_GPTHDR));

         //- Sort the PTA on startsector
         dfsGptSortPta( ptArray, sd->ptaEntries, sd->ptaEntrySize);

         //- Fixup CRC values. for THIS instance of the header
         sd->ptaCRC = dfsGptCrc32((BYTE *) ptArray, (sd->ptaEntries * sd->ptaEntrySize));
         sd->hdrCRC = 0;                                     //- must be 0 during calc
         sd->hdrCRC = dfsGptCrc32((BYTE *) sd, sd->hdrSize); //- calculate header CRC

         rc = dfstWritePsn( DFSTORE, sd->thisPSN, 1, (BYTE *) sd);
         if (rc == NO_ERROR)
         {
            rc = dfsGptWritePta( sd->ptaPSN, sd->ptaEntries, sd->ptaEntrySize, ptArray);
            if (rc == NO_ERROR)
            {
               ULONG   arraySize;               // size of array in bytes
               ULONG   arraySectors;

               //- Morph HEADER to alternate header (or vice-versa)
               arraySize    = sd->ptaEntries * sd->ptaEntrySize;
               arraySectors = (arraySize + dfsGetSectorSize() -1) / dfsGetSectorSize();

               if (sd->thisPSN == GPTHDR_PRIMARY_PSN) // we are primary now
               {
                  sd->ptaPSN  = sd->althPSN - arraySectors;
                  sd->thisPSN = sd->althPSN;
                  sd->althPSN = GPTHDR_PRIMARY_PSN;
               }
               else                             // we are alternate now
               {
                  sd->ptaPSN = GPTHDR_PRIMARY_PSN + 1;
                  sd->althPSN = sd->thisPSN;
                  sd->thisPSN = GPTHDR_PRIMARY_PSN;
               }

               //- Fixup CRC values. for second (morphed) instance of the header
               sd->hdrCRC = 0;                                     //- must be 0 during calc
               sd->hdrCRC = dfsGptCrc32((BYTE *) sd, sd->hdrSize); //- calculate header CRC

               rc = dfstWritePsn( DFSTORE, sd->thisPSN, 1, (BYTE *) sd);
               if (rc == NO_ERROR)
               {
                  rc = dfsGptWritePta( sd->ptaPSN, sd->ptaEntries, sd->ptaEntrySize, ptArray);
               }
            }
         }
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   else
   {
      rc = DFS_ST_MISMATCH;
   }
   RETURN (rc);
}                                               // end 'dfsGptFixupWriteAll'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Write GPT partition table array (PTA), starting from supplied first PSN
/*****************************************************************************/
static ULONG dfsGptWritePta                     // RET   write errors
(
   ULN64               ptaPsn,                  // IN    PSN of table sector
   ULONG               ptaEntries,              // IN    max nr of entries
   ULONG               ptaEntrySize,            // IN    size of entry in bytes
   S_GPTENTRY          ptArray[]                // IN    PT array sectors
)
{
   ULONG               rc;
   ULONG               arraySize;               // size of array in bytes
   ULONG               arraySectors;

   ENTER();
   TRACES(("ptaPsn: 0x%llx\n", ptaPsn));

   if ((ptaEntries   >=   4) && (ptaEntries   <= 4096) &&
       (ptaEntrySize >= 128) && (ptaEntrySize <= 4096)) // sanity check
   {
      arraySize    = ptaEntries * ptaEntrySize;
      arraySectors = (arraySize + dfsGetSectorSize() -1) / dfsGetSectorSize();

      rc = dfstWritePsn( DFSTORE, ptaPsn, arraySectors, (BYTE *) ptArray);
   }
   else
   {
      rc = DFS_BAD_STRUCTURE;
   }
   RETURN (rc);
}                                               // end 'dfsGptWritePta'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Calculate a GPT-stype 32-bit CRC value over specified byte array
/*****************************************************************************/
static ULONG dfsGptCrc32                        // RET   32-bit GPT CRC
(
   BYTE               *data,                    // IN    data to calculate CRC for
   ULONG               len                      // IN    length of the data
)
{
   ULONG               rc = 0xFFFFFFFF;         // CRC seed value
   BYTE               *pb = data;

   ENTER();
   TRACES(("GPT 32-bit CRC for data at 0x%8.8x, length: %u bytes\n", data, len));

   while (len--)
   {
      rc = dfsGptCrc32Table[(rc ^ *pb++) & 0xFF] ^ (rc >> 8);
   }
   rc ^= 0xFFFFFFFF;

   RETURN (rc);
}                                               // end 'dfsGptCrc32'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Change partitiontype to specified value, and write back all tables
/*****************************************************************************/
ULONG dfsGptSetType                             // RET   OK, bad-sig, bad-crc
(
   DFSPARTINFO        *p,                       // IN    current partition info
   FDSK_CB_INFO       *cbp                      // IN    disk/part info + type
)
{
   ULONG               rc = NO_ERROR;
   TXLN                s1;
   S_GPTHDR           *gptHdr;                  // GPT header for the disk
   S_GPTENTRY         *gptTable;
   S_GPTENTRY         *pe;
   TXTM                descr;

   ENTER();

   if ((pe = dfsGptDiskGetEntry( cbp, &gptHdr, &gptTable)) != NULL)
   {
      strcpy( descr, dfsGptGuidDescription( cbp->typeGuid));
      TxStrip( descr, descr, ' ', ' ');

      sprintf(   s1, "Partition %hu disk %hu, %s %s\nLabel '%s' by %s\nPartition name: ",
                    p->id, p->disknr,  "GPT-style", p->fsform, p->plabel, p->creatr);
      TxUnicAsciiAppend( pe->partName, s1, GPT_PNAME_LEN * 2);
      dfstrSizeBps( s1, "\nPartition size: ", p->sectors, p->bpsector, "\n\n");
      if (TxConfirm( 5083, "%sChange the type for this "
                           "partition to '%s' ? [Y/N] : ", s1, descr))
      {
         memcpy( pe->typeGuid, cbp->typeGuid, sizeof(DFS_GUID));

         rc = dfsGptFixupWriteAll( gptHdr, gptTable);
         if (rc == NO_ERROR)
         {
            TxPrint( "GPT info written successfully to primary and alternate location on disk %hu.\n", cbp->disknr);
         }
      }
      else
      {
         rc = DFS_CMD_FAILED;                   // aborted
      }
   }
   else
   {
      rc = DFS_BAD_STRUCTURE;
   }
   RETURN (rc);
}                                               // end 'dfsGptSetType'
/*---------------------------------------------------------------------------*/


#if defined (USEWINDOWING)
/*****************************************************************************/
// Build selection-list with the most important GPT partition types + custom
/*****************************************************************************/
TXSELIST *dfsGptTypeSelist                      // RET   selection list or NULL
(
   BOOL                addCustom                // IN    Add 'manual edit' as 1st
)
{
   TXSELIST           *list  = NULL;            // total list
   TXS_ITEM           *item;                    // single item
   ULONG               lsize;                   // list-size
   int                 i;

   ENTER();

   lsize = (addCustom) ? 35 : 34;               // Keep in sync with switch CODE!
   if (TxSelCreate( lsize, lsize, lsize, TXS_AS_NOSTATIC, FALSE, NULL, &list) == NO_ERROR)
   {
      list->astatus = TXS_AS_NOSTATIC      |    // all dynamic allocated
                      TXS_LST_DESC_PRESENT |    // with list description
                      TXS_LST_DYN_CONTENTS;

      #if   defined (LINUX)
         list->selected = 16;                   // Default Linux Data
      #elif defined (DARWIN)
         list->selected = 8;                    // Default HFS+
      #else
         list->selected = 3;                    // Default Windows-Data
      #endif

      for (i = 0; i < lsize; i++)               // all list entries
      {
         TRACES(("GPT type list entry: %d for total of %u\n", i, lsize));
         if ((item  = TxAlloc( 1, sizeof(TXS_ITEM))) != NULL)
         {
            list->count    = i +1;              // actual items in list
            list->items[i] = item;              // attach item to list

            item->helpid = TXWH_USE_WIN_HELP;   // from list-window itself

            if (((item->text = TxAlloc( 1, TXMAXTM)) != NULL) &&
                ((item->desc = TxAlloc( 1, TXMAXTM)) != NULL)  )
            {
               item->value = TXDID_MAX + i;     // set user value

               switch (i)
               {
                  case 0:
                     strcpy( item->text, "fe00|UEFI/GPT |");
                     strcat( item->text, dfsGptGuidDescription( dfsGuid_EFI_SYSTEM ));
                     strcpy( item->desc, dfsFsUuidValueString(  dfsGuid_EFI_SYSTEM ));
                     break;

                  case 1:
                     strcpy( item->text, "bb00|BootMgrs |");
                     strcat( item->text, dfsGptGuidDescription( dfsGuid_BIOS_BOOT ));
                     strcpy( item->desc, dfsFsUuidValueString(  dfsGuid_BIOS_BOOT ));
                     break;

                  case 2:
                     strcpy( item->text, "0c01|Microsoft|");
                     strcat( item->text, dfsGptGuidDescription( dfsGuid_MICROSOFT_RESERVED ));
                     strcpy( item->desc, dfsFsUuidValueString(  dfsGuid_MICROSOFT_RESERVED ));
                     break;

                  case 3:
                     strcpy( item->text, "0700|Microsoft|");
                     strcat( item->text, dfsGptGuidDescription( dfsGuid_WINDOWS_BASIC_DATA ));
                     strcpy( item->desc, dfsFsUuidValueString(  dfsGuid_WINDOWS_BASIC_DATA ));
                     break;

                  case 4:
                     strcpy( item->text, "4201|Microsoft|");
                     strcat( item->text, dfsGptGuidDescription( dfsGuid_WINDOWS_LDM_META ));
                     strcpy( item->desc, dfsFsUuidValueString(  dfsGuid_WINDOWS_LDM_META ));
                     break;

                  case 5:
                     strcpy( item->text, "4200|Microsoft|");
                     strcat( item->text, dfsGptGuidDescription( dfsGuid_WINDOWS_LDM_DATA ));
                     strcpy( item->desc, dfsFsUuidValueString(  dfsGuid_WINDOWS_LDM_DATA ));
                     break;

                  case 6:
                     strcpy( item->text, "2700|Microsoft|");
                     strcat( item->text, dfsGptGuidDescription( dfsGuid_WINDOWS_RECOVERY ));
                     strcpy( item->desc, dfsFsUuidValueString(  dfsGuid_WINDOWS_RECOVERY ));
                     break;

                  case 7:
                     strcpy( item->text, "ab00|Apple OSX|");
                     strcat( item->text, dfsGptGuidDescription( dfsGuid_APPLE_BOOT ));
                     strcpy( item->desc, dfsFsUuidValueString(  dfsGuid_APPLE_BOOT ));
                     break;

                  case 8:
                     strcpy( item->text, "af00|Apple OSX|");
                     strcat( item->text, dfsGptGuidDescription( dfsGuid_MAC_OS_X_HFS_PLUS ));
                     strcpy( item->desc, dfsFsUuidValueString(  dfsGuid_MAC_OS_X_HFS_PLUS ));
                     break;

                  case 9:
                     strcpy( item->text, "af03|Apple OSX|");
                     strcat( item->text, dfsGptGuidDescription( dfsGuid_APPLE_LABEL ));
                     strcpy( item->desc, dfsFsUuidValueString(  dfsGuid_APPLE_LABEL ));
                     break;

                  case 10:
                     strcpy( item->text, "af01|Apple OSX|");
                     strcat( item->text, dfsGptGuidDescription( dfsGuid_APPLE_RAID ));
                    strcpy( item->desc, dfsFsUuidValueString(  dfsGuid_APPLE_RAID ));
                     break;

                  case 11:
                     strcpy( item->text, "af02|Apple OSX|");
                     strcat( item->text, dfsGptGuidDescription( dfsGuid_APPLE_RAID_OFFLINE ));
                     strcpy( item->desc, dfsFsUuidValueString(  dfsGuid_APPLE_RAID_OFFLINE ));
                     break;

                  case 12:
                     strcpy( item->text, "af6a|Apple OSX|");
                     strcat( item->text, dfsGptGuidDescription( dfsGuid_APPLE_ZFS ));
                     strcpy( item->desc, dfsFsUuidValueString(  dfsGuid_APPLE_ZFS ));
                     break;

                  case 13:
                     strcpy( item->text, "a800|Apple OSX|");
                     strcat( item->text, dfsGptGuidDescription( dfsGuid_APPLE_UFS ));
                     strcpy( item->desc, dfsFsUuidValueString(  dfsGuid_APPLE_UFS ));
                     break;

                  case 14:
                     strcpy( item->text, "af04|Apple OSX|");
                     strcat( item->text, dfsGptGuidDescription( dfsGuid_APPLE_TV_RECOVERY ));
                     strcpy( item->desc, dfsFsUuidValueString(  dfsGuid_APPLE_TV_RECOVERY ));
                     break;

                  case 15:
                     strcpy( item->text, "af05|Apple OSX|");
                     strcat( item->text, dfsGptGuidDescription( dfsGuid_APPLE_CORE_STORAGE ));
                     strcpy( item->desc, dfsFsUuidValueString(  dfsGuid_APPLE_CORE_STORAGE ));
                     strcat( item->desc, " (Possible FileVault)");
                     break;

                  case 16:
                     strcpy( item->text, "af73|APFS Cont|");
                     strcat( item->text, dfsGptGuidDescription( dfsGuid_APFS_CONTAINER ));
                     strcpy( item->desc, dfsFsUuidValueString(  dfsGuid_APFS_CONTAINER ));
                     break;

                  case 17:
                     strcpy( item->text, "8200|Linux    |");
                     strcat( item->text, dfsGptGuidDescription( dfsGuid_LINUX_SWAP ));
                     strcpy( item->desc, dfsFsUuidValueString(  dfsGuid_LINUX_SWAP ));
                     break;

                  case 18:
                     strcpy( item->text, "8300|Linux    |");
                     strcat( item->text, dfsGptGuidDescription( dfsGuid_LINUX_DATA ));
                     strcpy( item->desc, dfsFsUuidValueString(  dfsGuid_LINUX_DATA ));
                     break;

                  case 19:
                     strcpy( item->text, "8301|Linux    |");
                     strcat( item->text, dfsGptGuidDescription( dfsGuid_LINUX_RESERVED ));
                     strcpy( item->desc, dfsFsUuidValueString(  dfsGuid_LINUX_RESERVED ));
                     break;

                  case 20:
                     strcpy( item->text, "8302|Linux    |");
                     strcat( item->text, dfsGptGuidDescription( dfsGuid_LINUX_HOME ));
                     strcpy( item->desc, dfsFsUuidValueString(  dfsGuid_LINUX_HOME ));
                     break;

                  case 21:
                     strcpy( item->text, "8303|Linux    |");
                     strcat( item->text, dfsGptGuidDescription( dfsGuid_LINUX_ROOT_X86_32 ));
                     strcpy( item->desc, dfsFsUuidValueString(  dfsGuid_LINUX_ROOT_X86_32 ));
                     break;

                  case 22:
                     strcpy( item->text, "8304|Linux    |");
                     strcat( item->text, dfsGptGuidDescription( dfsGuid_LINUX_ROOT_X86_64 ));
                     strcpy( item->desc, dfsFsUuidValueString(  dfsGuid_LINUX_ROOT_X86_64 ));
                     break;

                  case 23:
                     strcpy( item->text, "8306|Linux    |");
                     strcat( item->text, dfsGptGuidDescription( dfsGuid_LINUX_SRV_DATA ));
                     strcpy( item->desc, dfsFsUuidValueString(  dfsGuid_LINUX_SRV_DATA ));
                     break;

                  case 24:
                     strcpy( item->text, "8308|Linux    |");
                     strcat( item->text, dfsGptGuidDescription( dfsGuid_LINUX_PLAIN_DMCRYPT ));
                     strcpy( item->desc, dfsFsUuidValueString(  dfsGuid_LINUX_PLAIN_DMCRYPT ));
                     break;

                  case 25:
                     strcpy( item->text, "8309|Linux    |");
                     strcat( item->text, dfsGptGuidDescription( dfsGuid_LINUX_LUKS ));
                     strcpy( item->desc, dfsFsUuidValueString(  dfsGuid_LINUX_LUKS ));
                     break;

                  case 26:
                     strcpy( item->text, "8e00|Linux    |");
                     strcat( item->text, dfsGptGuidDescription( dfsGuid_LINUX_LVM ));
                     strcat( item->desc, dfsFsUuidValueString(  dfsGuid_LINUX_LVM ));
                     break;

                  case 27:
                     strcpy( item->text, "fd00|Linux    |");
                     strcat( item->text, dfsGptGuidDescription( dfsGuid_LINUX_RAID ));
                     strcpy( item->desc, dfsFsUuidValueString(  dfsGuid_LINUX_RAID ));
                     break;

                  case 28:
                     strcpy( item->text, "0521|OS2 TYPE1|");
                     strcat( item->text, dfsGptGuidDescription( dfsGuid_OS2_ARCAOS_TYPE_1 ));
                     strcpy( item->desc, dfsFsUuidValueString(  dfsGuid_OS2_ARCAOS_TYPE_1 ));
                     break;

                  case 29:
                     strcpy( item->text, "c001|H-Packard|");
                     strcat( item->text, dfsGptGuidDescription( dfsGuid_HP_UX_DATA ));
                     strcpy( item->desc, dfsFsUuidValueString(  dfsGuid_HP_UX_DATA ));
                     break;

                  case 30:
                     strcpy( item->text, "c002|H-Packard|");
                     strcat( item->text, dfsGptGuidDescription( dfsGuid_HP_UX_SERVICE ));
                     strcpy( item->desc, dfsFsUuidValueString(  dfsGuid_HP_UX_SERVICE ));
                     break;

                  case 31:
                     strcpy( item->text, "b100|QNX      |");
                     strcat( item->text, dfsGptGuidDescription( dfsGuid_QNX_POWERSAFE ));
                     strcpy( item->desc, dfsFsUuidValueString(  dfsGuid_QNX_POWERSAFE ));
                     break;

                  case 32:
                     strcpy( item->text, "fc00|VMware   |");
                     strcat( item->text, dfsGptGuidDescription( dfsGuid_VMWARE_COREDUMP ));
                     strcpy( item->desc, dfsFsUuidValueString(  dfsGuid_VMWARE_COREDUMP ));
                     break;

                  case 33:
                     strcpy( item->text, "fb00|VMware   |");
                     strcat( item->text, dfsGptGuidDescription( dfsGuid_VMWARE_VMFS ));
                     strcpy( item->desc, dfsFsUuidValueString(  dfsGuid_VMWARE_VMFS ));
                     break;

                  case 34:
                     item->value = TXDID_MAX + DFSGPT_CUSTOM;
                     strcpy( item->text, "Manually set custom GUID value");
                     strcpy( item->desc, "Specify a custom 128-bit GUID value, manually");
                     break;

                  default:
                     strcpy( item->text, "Missing definition!");
                     strcpy( item->desc, "Program bug, list size error");
                     break;
               }

               //- get unique SelectChar starting at text[1] ...
               //- item->index = TxSelGetCharSelect( list, i, 1);

               TRACES(("text: %d '%s'\n", strlen(item->text), item->text));
               TRACES(("desc: %d '%s'\n", strlen(item->desc), item->desc));
            }
         }
      }
   }
   RETURN( list);
}                                               // end 'dfsGptTypeSelist'
/*---------------------------------------------------------------------------*/
#endif                                          // USEWINDOWING

#endif
