//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// FDISK utility functions
//
// Author: J. van Wijk
//
// JvW  21-02-1999   Initial version, cloned from DFSUNTFS
// JvW  02-07-2000   Moved CountDisks from dfsector.c
// JvW  06-07-2000   Changed CountDisks to GetDiskCount, phys + virt
// JvW  03-09-2015   Split off the GPT specific definitions and code
//
#ifndef    DFSUFDSK
   #define DFSUFDSK

#if defined (DEV32)
// Synchronize CURRENT LIVE in-memory LVM engine to changed on-disk situation
ULONG dfsLvmSyncEngine
(
   BOOL                verbose,                 // IN    display LVM action
   BOOL                reread,                  // IN    reread diskinfo too
   char               *volume,                  // IN    LVM volume name
   char                letter                   // IN    new letter or 0
);                                              //       without the 'lvm /'
#endif

// Set the correct filesystem type on a partition for specified FS-name
ULONG dfsSetPartTypeForFS                       // RET   result
(
   DFSPARTINFO        *p,                       // IN    partition information
   char               *fs                       // IN    filesystem name
);

typedef struct s_lvinit                         // LVM initial partition info
{
   BYTE                OnBmMenu;                // partition is on the BMGR menu
   BYTE                letter;                  // uppercase volume letter
   char                VoluName[LVM_NAME_L +1]; // Logical volume name
} S_LVINIT;                                     // end of struct "s_lvinit"

// Save initial LVM info from current tables to be used by Volume Conversion
S_LVINIT *dfsLvmVcuPreservation                 // RET    table S_LVINIT[#part]
(                                               //        pid is index in array
   void
);

// Get number of partitionable physical or virtual disks
USHORT dfsGetDiskCount                          // RET   number of disks
(
   BOOL                verbose                  // IN    Report to screen too
);

// Return valid part-id for part-specification;  <empty> | . | * | nr | drive
USHORT dfsParsePartSpec                         // RET   0=invalid, nr or ANY
(
   char               *spec,                    // IN    part select spec
   USHORT              disk,                    // IN    related disk (or ANY)
   DFSPARTINFO       **pinfo                    // OUT   partinfo pointer
);


// Return valid part-type for specification;    hex | symbolic
BYTE dfsParsePartType                           // RET   0=invalid or Ptype
(
   char               *spec,                    // IN    part type spec
   FDSK_CB_INFO       *cbi                      // IN    size & position info
);

// Return valid GPT part-type for specification;    hex-UUID-string | symbolic
BOOL dfsParseGptType                            // RET   type is valid UUID now
(
   char               *spec                     // INOUT part type spec / UUID
);

// Parse a hex/dec location value, convert to valid start-cylinder
DFSPARTINFO *dfsParseLocation                   // RET   freespace area to use
(
   char               *spec,                    // IN    specification string
   char               *spos,                    // IN    position specification
   char                fspKind,                 // IN    G(pt), P(rim), L(ogical)
   FDSK_NUMVAL        *size                     // IN    raw size value
);                                              // note: first char is default


// Parse a hex/dec position value, convert Mb/Cyl etc to freespace-id (raw)
void dfsPosition2Fspid
(
   char               *spec,                    // IN    position specification
   char                fspKind,                 // IN    G(pt), P(rim), L(ogical)
   DFSDISKINFO        *d,                       // IN    diskinfo (geo etc)
   USHORT             *fid                      // OUT   freespace-id
);


// Parse a hex/dec size value, convert Mb/Cyl to cylinder-boundary sector-nr
void dfsParseNewSize
(
   char               *spec,                    // IN    specification string
   char                fspKind,                 // IN    G(pt), P(rim), L(ogical)
   DFSPARTINFO        *f,                       // IN    freespace info (geo)
   FDSK_NUMVAL        *pos,                     // IN    RAW position value
   FDSK_NUMVAL        *nv                       // INOUT defaults/result value
);


// Parse a hex/dec position value, convert Mb/Cyl to cyl-boundary sector-nr
ULONG dfsParsePosition
(
   char               *spec,                    // IN    specification string
   char                fspKind,                 // IN    G(pt), P(rim), L(ogical)
   ULONG               ptGap,                   // IN    gap to ptable, sectors
   DFSPARTINFO        *f,                       // INOUT freespace info (geo)
   FDSK_NUMVAL        *size,                    // INOUT resolved size value
   FDSK_NUMVAL        *nv                       // INOUT defaults/result value
);


// Parse a hex/dec size value from an option, convert Mb/Gb/Kb/cyl/track to sec
BOOL dfsOptionSizeSectors                       // RET   option value OK
(
   char                option,                  // IN    option char to use
   DFST_HANDLE         store,                   // IN    store handle for geo
   char                defUnit,                 // IN    default unit s,m etc
   ULN64               defaultSn,               // IN    default value
   ULN64              *sectors                  // OUT   nr of sectors
);


// Parse a hex/dec size value, convert Mb/sec to cyl using specified head/sect
void dfsParseCylinderSize
(
   char               *spec,                    // IN    specification string
   ULONG               hds,                     // IN    Geo, nr of heads
   ULONG               spt,                     // IN    Geo, nr of sectors
   ULONG               bps,                     // IN    Geo, bytes per sector
   FDSK_NUMVAL        *nv                       // INOUT defaults/result value
);


// Parse a hex/dec numeric value with leading options and trailing fmt/type
// Number syntax: [leading-options]number[,[h|d][type-indicator]]
void dfsParseNumValue
(
   char               *spec,                    // IN    specification string
   FDSK_NUMVAL        *nv                       // INOUT defaults/result value
);                                              // note: first char is default

// Make in-memory ebrChain and ebrHead consistent after any updates
// Note: Preparation for on-disk structure update with IteratePebr()
ULONG dfsFdskUpdateExtChain
(
   USHORT              disk                     // IN    disknr to work on
);


// Count number of hidable and currently visible primary partitions in Ptable
USHORT dfsVisPrimaries
(
   S_BOOTR            *br                       // IN    master/extended br
);

// Translate Physical sector number (PSN) to Head-Cylinder-Sector, for GEO
ULONG dfsGeoPsn2Chs                             // RET   error (max psn)
(
   ULN64               psn,                     // IN    physical sector nr
   ULONG               geoHeads,                // IN    geo heads
   ULONG               geoSecs,                 // IN    geo sectors
   ULONG              *cylinder,                // OUT   cylinder number
   ULONG              *head,                    // OUT   head number
   ULONG              *sector                   // OUT   first sector number
);

// Translate Cylinder-Head-Sector to Physical sector number (PSN), use Geo
ULN64 dfsGeoChs2Psn                             // RET   physical sector nr
(
   ULONG               geoHeads,                // IN    geo heads
   ULONG               geoSecs,                 // IN    geo sectors
   ULONG               cylinder,                // IN    cylinder number
   ULONG               head,                    // IN    head number
   ULONG               sector                   // IN    first sector number
);

// Test if a given (starting) sector number is exactly on a track/cyl boundary
ULONG  dfsOnTrackBoundary                       // RET   head-nr on
(
   ULN64               psn,                     // IN    physical sector nr
   ULONG               geoHeads,                // IN    geo heads
   ULONG               geoSecs                  // IN    geo sectors
);


// Find an partition-table entry of specified type in a partition-record
BOOL dfsFindPtableEntry
(
   S_BOOTR            *br,                      // IN    master/extended br
   BYTE                type,                    // IN    type to find
   USHORT             *pi                       // IN    prefered  index
                                                // OUT   partition index
);


// Fill Ptable-entry values for a partition specified by CBI and PARTINFO
ULONG dfsFillPtableEntry
(
   DFSPARTINFO        *p,                       // IN    partition information
   FDSK_CB_INFO       *cbi,                     // IN    create information
   DFSPARTENTRY       *pe                       // OUT   partition-table entry
);


// Select physical disk for Physical reading (aka 'open physical')
ULONG dfsSelectDisk
(
   USHORT              disknr,                  // IN    phys disknr to open
   BOOL                verbose,                 // IN    show disk geo
   BOOL                reset                    // IN    reset disk geo
);


// Determine base-PSN for RBA-addressing of extended boot records
ULONG dfsEbrBase                                // RET   PSN of first EBR
(
   void
);


// Find latest MBR/EBR sector before the specified PSN
ULONG dfsPreceedingTable                        // RET   PSN of last MBR/EBR
(
   ULONG               rPSN                     // IN    reference PSN
);


// Check if supplied type is one of the extended's
BOOL dfsIsExtendedType
(
   BYTE                type                     // IN    partition-type
);


// Calculate relative, logarithmic size for a give number of sectors
USHORT dfsLogarithmicSize                       // RET   Log size 1..45, for
(                                               //
   ULONG               sectors                  // IN    32-bit nr of sectors
);


// Display FS bootsector, with std BPB, NTFS, BMGR, EFAT or other specials
ULONG dfsBootSector                             // RET   rc = 0 if type match
(
   BYTE               *sector                   // IN    Sector data
);

// Display Master- or Extended- Boot Record with Partition table
ULONG dfsPartitionRecord                        // RET   rc = 0 if type match
(
   BYTE               *sector,                  // IN    Sector buffer
   BYTE                t,                       // IN    sector type
   ULONG               tablePsn                 // IN    PSN of table sector
);

// Display GRUB stage1 specific part of an MBR or partition bootrecord
ULONG dfsFdskGrubStage1
(
   BYTE               *sector,                  // IN    Sector data
   BOOL                verbose                  // IN    complete, incl stage2
);

// Read/verify and display GRUB stage2 (or 1.5) code as identified by stage1
ULONG dfsFdskGrubStage2
(
   USHORT              disknr,                  // IN    Disknr stage1/2
   ULONG               s2psn                    // IN    Stage2 1st sector PSN
);

// Display LVM disk/volume information sector (last sector MBR/EBR track)
ULONG dfsFdskLvmInfo
(
   BYTE               *sector                   // IN    Sector data
);

// Display LVM signature/feature sector (end of non-compat partitions)
ULONG dfsFdskLvmSign
(
   BYTE               *sector                   // IN    Sector data
);

// Display LVM drive-linking feature, master including table sectors
ULONG dfsFdskLvmDlf
(
   BYTE               *sector                   // IN    Sector data (4 sects!)
);

// Display LVM drive-linking table. (based on non-zero entries, not count!)
ULONG dfsFdskLvmDlt
(
   BYTE               *sector                   // IN    Sector data
);

// Display LVM bad-block-rel feature, master sector, admin only
ULONG dfsFdskLvmBBf
(
   BYTE               *sector                   // IN    Sector data
);

// Display LVM bad-block-rel table. (based on non-zero entries, not count!)
ULONG dfsFdskLvmBBt
(
   BYTE               *sector                   // IN    Sector data
);

// Check and display LVM type sector-CRC value, correct when BAD
ULONG dfsFdskLvmProcessCRC
(
   BYTE               *sector,                  // IN    Sector data
   ULONG              *actualCRC                // INOUT CRC value in sector
);

// Display LUKS Crypto Header (Linux disk/partition/container key-management)
ULONG dfsFdskLuksHdr
(
   BYTE               *sector                   // IN    Sector data
);

// Display Apple Core Storage header (used for native HFS or FileVault on GPT)
ULONG dfsFdskCoreStg
(
   BYTE               *sector                   // IN    Sector data
);

// Get the BMGR information index for a given partition-table entry on a disk
// Note: when no existing found, the index will be set to a 'free' entry
BOOL dfsFdskGetBmiIndex                         // RET   existing bmi found
(
   USHORT              disk,                    // IN    disk number
   DFSPARTENTRY       *pe,                      // IN    partition-table entry
   BMPRIMARY          *bmi,                     // IN    start of BMI array
   ULONG              *index                    // OUT   index (or DFS32MAX)
);


// Get partition-data if DLAT entry is for a current, existing partition
DFSPARTINFO *dfsFdskDlatEntry2Part              // RET   partition info or NULL
(
   S_LVPART           *entry,                   // IN    DLAT entry
   USHORT              disknr                   // IN    disknumber for entry
);


// Reboot the system, using external and/or internal means
BOOL dfsRebootSystem                            // RET   reboot succesfull
(
   BOOL                flush                    // IN    Flush FS/buffers
);

#endif
