//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// FDISK utility functions
//
// Author: J. van Wijk
//
// JvW  21-02-1999   Initial version, cloned from DFSUNTFS
// JvW  18-04-1999   Moved some stuff to new DFSUPART
// JvW  04-04-2000   Fixed trap using dfsFdskGetBmiIndex (uninitialized)
// JvW  13-04-2000   Fixed typo in message (unlimited)
// JvW  05-06-2000   Added nr of cylinders to BOOTR display
// JvW  02-07-2000   Moved CountDisks from dfsector.c
// JvW  03-04-2001   Added LVM specific utility functions like CRC
// JvW  29-06-2001   Fixed max-head on CHS entries with cyl > 1023
// JvW  03-09-2015   Split off the GPT specific definitions and code
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsver.h>                             // DFS version info
#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#if !defined (OEMSB)
#include <dfsspace.h>                           // DFS  file-space interface
#include <dfsantfs.h>                           // NTFS  interface (bootRec)
#include <dfsajfs.h>                            // JFS interface   (bootrec)
#include <dfsefat.h>                            // EFAT definitions
#include <dfsaefat.h>                           // EFAT interface  (bootRec)
#include <dfsuefat.h>                           // EFAT interface  (bootRec)
#endif
#include <dfsupart.h>                           // FDISK partition functions
#include <dfsafdsk.h>                           // FDISK analysis functions

#include <dfsufdsk.h>                           // FDISK utility functions
#include <dfsufgpt.h>                           // FDISK GPT defs and functions

#if !defined (OEMSB)
#include <dfscfdsk.h>                           // FDISK command functions
#endif
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfsosapi.h>                           // OS specific stuff
#include <dfsfat.h>                             // FAT FS definitions


// Display and check LVM feature class information
static void dfsFdskLvmProcessClass
(
   ULONG               ActualClass,             // IN    feature class
   BOOL                TopOfClass               // IN    LVM top of class
);

// Display LVM feature class information
static void dfsFdskLvmDisplayClass
(
   char               *lead,                    // IN    lead string
   ULONG               Class,                   // IN    feature class
   char               *tail                     // IN    tail string
);


#ifndef OEMSB

static char *dfs_pthex =                        // header for Ptable hex dump
  "        +==+--------+==+--------+------------+-----------+\n";


/*****************************************************************************/
// Set the correct filesystem type on a partition for specified FS-name
/*****************************************************************************/
ULONG dfsSetPartTypeForFS                       // RET   result
(
   DFSPARTINFO        *p,                       // IN    partition information
   char               *fs                       // IN    filesystem name
)
{
   ULONG               rc = NO_ERROR;           // function return
   FDSK_CB_INFO        cbi;                     // callback info
   BYTE                desired;
   BYTE                current;
   TXTM                cmd;

   ENTER();

   memset( &cbi, 0, sizeof(FDSK_CB_INFO));
   cbi.more     = p;                            // attach partition info
   cbi.number   = p->sectors;                   // size to be used
   cbi.sn       = p->basePsn;                   // PSN of bootsector

   current      = p->partent.PartitionType;
   desired      = dfsParsePartType( fs, &cbi);  // matching type for FS-name
   if (desired != current)                      // needs updating ...
   {
      TRACES(( "Change type for part %hu from %2.2hx to %2.2hx\n",
                      p->id, current, desired));

      cbi.disknr    = p->disknr;                // disknr to use
      cbi.ntype     = desired;
      cbi.confirmed = TRUE;
      strcpy( cbi.string, fs);                  // original 'new' type spec
      rc = dfsExecOnBootRec( p, cbi.disknr, dfsFdskSetType, &cbi);

      if (rc == NO_ERROR)
      {
         sprintf( cmd, "part %hu -q", p->id);   // select partition, silent
         dfsMultiCommand( cmd, 0, FALSE, FALSE, FALSE);

         TxPrint( "Type for part  ID : %2.2hu changed from %2.2hx to %2.2hx\n",
                                     p->id, current, desired);
      }
   }
   RETURN (rc);
}                                               // end 'dfsSetPartTypeForFS'
/*---------------------------------------------------------------------------*/


#if defined (DEV32)                             // useful on OS/2 only
/*****************************************************************************/
// Synchronize CURRENT LIVE in-memory LVM engine to changed on-disk situation
// Empty or NULL volume will refresh the removable media assignments (PRMs)
/*****************************************************************************/
ULONG dfsLvmSyncEngine
(
   BOOL                verbose,                 // IN    display LVM action
   BOOL                reread,                  // IN    reread diskinfo too
   char               *volume,                  // IN    LVM volume or ""
   char                letter                   // IN    new letter or 0
)
{
   ULONG               rc = NO_ERROR;           // function return

   ENTER();
   TRACES(( "Volume: '%s'  driveletter: %hu\n", volume, letter));

   if (dfsLvmEnginePresent())
   {
      USHORT           disk = SINF->disknr;
      USHORT           part = SINF->partid;
      TXTM             cmd;

      //- ALL disks must be closed to be able to use the LVM.DLL engine!
      DFSFNCALL(dfsa->FsClose,0,0,NULL,NULL);   // FS specific handler
      dfstClose( DFST_SYSTEM);                  // Close system  store
      dfstClose( DFST_ST_ONE);                  // Close first   store
      dfstClose( DFST_ST_TWO);                  // Close second  store
      SINF->disknr  = 0;
      SINF->partid  = 0;                        // mark status as closed
      strcpy(SINF->drive, "--");
      strcpy(SINF->afsys, "--none--");

      rc = dfsLvmEngineExecute( verbose, volume, letter);

      if (reread)
      {
         dfsReadDiskInfo( FDSK_QUIET);          // re-read diskinfo
      }
      if (disk != 0)
      {
         if (part == 0)
         {
            sprintf( cmd, "disk %hu -q", disk);
         }
         else
         {
            sprintf( cmd, "part %hu -q", part);
         }
         dfsMultiCommand( cmd, 0, FALSE, FALSE, FALSE);
      }
   }
   RETURN (rc);
}                                               // end 'dfsLvmSyncEngine'
/*---------------------------------------------------------------------------*/
#endif

#endif

/*****************************************************************************/
// Translate Physical sector number (PSN) to Head-Cylinder-Sector, for GEO
/*****************************************************************************/
ULONG dfsGeoPsn2Chs                             // RET   error (max psn)
(
   ULN64               psn,                     // IN    physical sector nr
   ULONG               geoHeads,                // IN    geo heads
   ULONG               geoSecs,                 // IN    geo sectors
   ULONG              *cylinder,                // OUT   cylinder number
   ULONG              *head,                    // OUT   head number
   ULONG              *sector                   // OUT   first sector number
)
{
   ULONG               dr = 0;                  // DOS rc
   ULONG               ulSpc;                   // sectors per cylinder
   ULONG               ulHeadSec;               // heads per sector

   if (psn != DFS_MAX_PSN)
   {
      if (geoSecs  == 0)
      {
         geoSecs    = 63;
      }
      if (geoHeads == 0)
      {
         geoHeads   = 255;
      }
      ulSpc = geoHeads * geoSecs;

      if (ulSpc && geoSecs)
      {
         ulHeadSec = (ULONG) (psn % ulSpc);
         *cylinder = (ULONG) (psn / ulSpc);
         *head     = (ULONG) (ulHeadSec / geoSecs);
         *sector   = (ULONG) (ulHeadSec % geoSecs) +1;
      }
   }
   else
   {
      dr = DFS_PSN_LIMIT;
      *cylinder = 0;
      *head     = 0;
      *sector   = 1;
   }
   return (dr);
}                                               // end 'dfsGeoPsn2Chs'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate Cylinder-Head-Sector to Physical sector number (PSN), use Geo
/*****************************************************************************/
ULN64 dfsGeoChs2Psn                             // RET   physical sector nr
(
   ULONG               geoHeads,                // IN    geo heads
   ULONG               geoSecs,                 // IN    geo sectors
   ULONG               cylinder,                // IN    cylinder number
   ULONG               head,                    // IN    head number
   ULONG               sector                   // IN    first sector number
)
{
   ULN64               psn = DFS_MAX_PSN;       // default invalid value

   psn = ((cylinder * geoHeads * geoSecs) +
          (head                * geoSecs) +
        (((sector) ? sector -1 : 0)));
   return (psn);
}                                               // end 'dfsGeoChs2Psn'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Test if a given (starting) sector number is exactly on a track/cyl boundary
/*****************************************************************************/
ULONG  dfsOnTrackBoundary                       // RET   head-nr on
(
   ULN64               psn,                     // IN    physical sector nr
   ULONG               geoHeads,                // IN    geo heads
   ULONG               geoSecs                  // IN    geo sectors
)
{
   ULONG               rc = L32_NULL;           // function return
   ULONG               c,h,s;                   // chs numbering

   dfsGeoPsn2Chs( psn, geoHeads, geoSecs,       // calculate CHS using geo
                  &c, &h, &s);
   if (s == 1)                                  // on a track boundary
   {
      rc = h;                                   // return head-number
   }
   return (rc);
}                                               // end 'dfsOnTrackBoundary'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Select physical disk for Physical reading (aka 'open physical')
/*****************************************************************************/
ULONG dfsSelectDisk
(
   USHORT              disknr,                  // IN    phys disknr to open
   BOOL                verbose,                 // IN    show disk geo
   BOOL                reset                    // IN    reset disk geo
)
{
   ULONG               rc = DFS_NO_DEVICE;      // DOS rc
   ULN64               lastsec;

   ENTER();

   TRACES(( "disknr: %hu  Reset:%s  Verbose:%s \n",
             disknr,     (reset) ? "YES" : "NO ", (verbose) ? "YES" : "NO "));
   rc = dfstOpenDisk( DFSTORE, disknr, verbose, reset);

   lastsec = dfstRawDiskSize( DFSTORE) -1;      // use non rounded down size
   if (lastsec == 0)
   {
      lastsec = DFS_MAX_PSN;
   }
   dfstLogicalLimits(  DFSTORE, 0, lastsec);    // back to physical
   dfstSetClusterSize( DFSTORE, 1);
   SINF->disknr = disknr;
   RETURN (rc);
}                                               // end 'dfsSelectDisk'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Determine base-PSN for RBA-addressing of extended boot records
/*****************************************************************************/
ULONG dfsEbrBase                                // RET   PSN of first EBR
(
   void
)
{
   ULONG               rc = 0;                  // DOS rc
   S_BOOTR            *mbr;                     // master boot record
   int                 pi;                      // partition index
   ULONG               ePSN = 0;                // first EBR, base for others

   ENTER();

   if ((mbr = TxAlloc(1, dfsGetSectorSize())) != NULL)
   {
      rc = dfstReadPsn( DFSTORE, 0, 1, (BYTE   *) mbr);
      if (rc == NO_ERROR)
      {
         if (mbr->Signature == SV_BOOTR)
         {
            for (pi = 0; pi < 4; pi++)
            {
               if (dfsIsExtendedType( mbr->PartitionTable[pi].PartitionType))
               {
                  ePSN = mbr->PartitionTable[pi].BootSectorOffset;
                  break;
               }
            }
         }
      }
      TxFreeMem( mbr);
   }
   RETURN (ePSN);
}                                               // end 'dfsEbrBase'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Check if supplied type is one of the extended's
/*****************************************************************************/
BOOL dfsIsExtendedType
(
   BYTE                type                     // IN    partition-type
)
{
   BOOL                rc = FALSE;              // function return

   switch (type)
   {
      case DFS_P_EXTENDED:
      case DFS_P_BIGEXTEND:
      case DFS_P_HIDEXTEND:
      case DFS_P_LINUXEXTX:
      case DFS_P_BIGLINEXT:
         rc = TRUE;
         break;

      default:
         break;
   }
   return (rc);
}                                               // end 'dfsIsExtendedType'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Calculate relative, logarithmic size for a give number of sectors
/*****************************************************************************/
USHORT dfsLogarithmicSize                       // RET   Log size 1..45, for
(                                               //
   ULONG               sectors                  // IN    32-bit nr of sectors
)
{
   USHORT              rc = 0;                  // function return
   ULONG               value = sectors >> 11;   // get rid of lowest 11 bits
   USHORT              lastbit  = 0;            // remembered last bit

   while (value > 1)                            // calculate 1.414 based
   {                                            // logarithmic size
      lastbit = (USHORT) (value & 0x0001);
      rc     += 2;
      value  /= 2;

      //- TRACES(("Lastbit: %x  value: %8.8x  rc: %hu\n", lastbit, value, rc));
   }
   rc += lastbit;
   rc |= 0x001;
   return( rc);
}                                               // end 'dfsLogarithmicSize'
/*---------------------------------------------------------------------------*/


#ifndef OEMSB
/*****************************************************************************/
// Display FS bootsector, with std BPB, NTFS, BMGR, EFAT or other specials
/*****************************************************************************/
ULONG dfsBootSector                             // RET   rc = 0 if type match
(
   BYTE               *sector                   // IN    Sector buffer
)
{
   ULONG               rc = 0;                  // rc, sector match
   S_BOOTR            *sd = (S_BOOTR *) sector;
   TXTS                plabel;                  // partition/bootsect label
   TXTS                creatr;                  // creatr string in bootsect
   TXTS                fsform;                  // Filesystem info bootsect
   DFS_OS_BOOT         bootos;                  // bootsector type
   TXLN                text;
   ULN64               sectors;                 // number of sectors
   BYTE                FsysValue;

   ENTER();

   TxDisplHexDump( sector, LEN_BOOTR);
   TxPrint("\n");

   if (memcmp(&(sd->Instruction[2]), "LILO", 4) == 0) // Linux LILO bootrec
   {
      TxPrint("Bootsector format : %sLILO%s, the Linux Loader\n", CBC, CNN);
      TxPrint("PC style BPB info : not present\n");
      //- to be refined, add some LILO detail display
   }
   else if (TxMemStr( sector + 100, "GRUB", SECTORSIZE -100) != NULL)
   {
      TxPrint("Bootsector format : %sGRUB%s, GRand Unified Bootloader, stage1 code\n", CBC, CNN);
      TxPrint("PC style BPB info : not present\n");
      dfsFdskGrubStage1( sector, TRUE);         // show GRUB details
   }
   else
   {
      memcpy( creatr, sd->OpSys,    BT_SYST_L);
      creatr[BT_SYST_L] = '\0';
      TRACES(("OpSys string: '%s'\n", creatr));
      if      (strncasecmp( creatr, "NTFS", 4) == 0) // NTFS bootrec
      {
         bootos = DFS_NT;
         strcpy( fsform,  creatr);
         strcpy( creatr, "Win NT");
         strcpy( plabel, "");
      }
      else if (memcmp( sd->f2.Type, "FAT32", 5) == 0) // FAT32 bootrec
      {
         bootos = DFS_F2;
         strcpy( fsform, "FAT32");
         memcpy( plabel, sd->f2.Label, BT_LABL_L);
         plabel[BT_LABL_L] = '\0';
      }
      else
      {
         bootos = DFS_OS;
         memcpy( fsform, sd->os.Type,  BT_TYPE_L);
         fsform[BT_TYPE_L] = '\0';
         memcpy( plabel, sd->os.Label, BT_LABL_L);
         plabel[BT_LABL_L] = '\0';
      }

      if (strncmp( creatr, "APJ&WN", 6) == 0)   // IBM BootManager
      {
         double        bmv = (double) creatr[6];

         TxPrint("Bootsector format : %s%s%s", CBM, fsform, CNN);
         if ((sd->eb.FatOffset   >= DFSFDSK_BMGRSEC) &&
             (sd->eb.RootEntries == 0))         // W2KBM protected ?
         {
            TxPrint( " (IBM BMGR protected against W2K chkdsk)");
         }
         else if (sd->eb.FatOffset == DFSFDSK_BMGROLD)
         {
            TxPrint( " (BM partly protected by older DFSee, use BMFIX)");
         }
         else
         {
            TxPrint( " (BM NOT protected against W2K chkdsk, use BMFIX)");
         }
         TxPrint("\nPart. creator     : %*.*s ", 6, 6, creatr);
         TxPrint( " = IBM BMGR version: %3.1lf\n", (bmv / 10));

         //- consistency check and warning not possible, PSN is unknown here
         TxPrint("BMGR datasector @ : CHS:%7lu %3u %-3u  ",
                                      DFSC2CYLIND(sd->os.BmDataSecCyl),
                                                  sd->os.BmDataHead,
                                      DFSC2SECTOR(sd->os.BmDataSecCyl));
         TxPrint(           "boot @ : %sCHS:%7lu %3u %-3u%s\n",    CBY,
                                      DFSC2CYLIND(sd->os.BmBootSecCyl),
                                                  sd->os.BmBootHead,
                                      DFSC2SECTOR(sd->os.BmBootSecCyl), CNN);
         nav.up = 1;                            // point to data sector
      }
      else if (strncmp( creatr, "EXFAT", 5) == 0) // EFAT filesystem
      {
         bootos = DFS_EX;
         strcpy( fsform,  creatr);
         strcpy( creatr, "Win NT");
         if (sd->ex.CodeEnd == DFSEX_MACOSX_CODE_END)
         {
            strcpy( creatr, "Mac OSX");
         }
         else
         {
            strcpy( creatr, "Windows");
         }
         dfsEfatRootLabel( sd, 0, plabel);

         TxPrint(    "Bootsector format : %s%s%s\n", CBM, fsform, CNN);
         TxPrint(    "Partition label   : %s%s%s\n", CNC, plabel, CNN);

         TxPrint(    "Bytes per sector  :     0x%4.4X = %4u (2^%u)",
                     (1 << sd->ex.BpsShift), (1 << sd->ex.BpsShift), sd->ex.BpsShift);
         TxPrint( "  0x%4.4X = %u sectors/cluster (2^%u)\n",
                     (1 << sd->ex.SpcShift), (1 << sd->ex.SpcShift), sd->ex.SpcShift);

         sprintf(   text, "  Offset 1st  FAT : %s0x%8.8X%s", CBG, sd->ex.FatOffset, CNN);
         dfstrSize( text, " = ",        (ULONG) sd->ex.FatOffset, "");
         TxPrint( "%s\n", text);

         sprintf(   text, "  Number of FAT's : %10u\n",  (USHORT) sd->ex.NrOfFats);
         dfstrSiz8( text, "  Sectors per FAT : ",        (ULONG)  sd->ex.FatLength, "");
         TxPrint( "%s\n", text);

         TxPrint(         "  RootDir Cluster : 0x%8.8X     LSN = 0x%8.8X\n",
                             sd->ex.RootCluster,             sd->ex.ClHeapOffset +
                           ((sd->ex.RootCluster - 2) * (1 << sd->ex.SpcShift)));

         sprintf(    text, "Vol nr of sectors : 0x%12.12llX = ", sd->ex.VolumeLength);
         dfstr64XiB( text, " ", (sd->ex.VolumeLength * (1 << sd->ex.BpsShift)), "");
         TxPrint( "%s  Cyls:%10.3lf (Dsk L-Geo)\n", text,
                       (((double) sd->ex.VolumeLength) / ((double) dfsCylinderSize())));
         sprintf(    text, "Partition Offset  : 0x%12.12llX = ", sd->ex.PartitionOffset);
         dfstr64XiB( text, " ", (sd->ex.PartitionOffset * (1 << sd->ex.BpsShift)), "\n");
         TxPrint( "%s", text);

         TxPrint("Data area   usage :        %2u%%   Filesystem layout version: %u.%02.2u\n",
                                sd->ex.PercentUsed, sd->ex.FsRevMajor, sd->ex.FsRevMinor);


         TxPrint("Physical disk nr  :       0x%02.2X = %7u    "
            "(Std = 0x80)\n",               (USHORT)  sd->ex.BiosDrive,
                                            (USHORT)  sd->ex.BiosDrive);

         TxPrint("Volume flags      :     0x%4.4X = %s-FAT  status = %s   %s\n",
              sd->ex.VolumeFlags,
             (sd->ex.VolumeFlags & EFAT_FAT2ACTIVE) ? "2nd"   : "1st",
             (sd->ex.VolumeFlags & EFAT_VOL_DIRTY)  ? "DIRTY" : "CLEAN",
             (sd->ex.VolumeFlags & EFAT_PENDINGBAD) ? "Bad sectors Pending" : "");

         strcpy( text, "");
         dfstrSiz8(    text, "First   FAT   LSN : ", sd->ex.FatOffset,    "\n");
         dfstrSiz8(    text, "Cluster heap  LSN : ", sd->ex.ClHeapOffset, "\n");
         TxPrint("%s", text);
         sprintf(      text, "Cluster heap size : 0x%8.8X =", sd->ex.ClusterCount);
         dfstr64XiB(   text, "    ",                  ((LLONG) sd->ex.ClusterCount *
                                (1 << sd->ex.SpcShift) * (1 << sd->ex.BpsShift)), "\n");
         TxPrint("%s", text);

         TxPrint("Volume-serial nr  : 0x%08.8X\n",  sd->ex.SerialNr);
      }
      else                                      // regular (BPB style) bootsectors
      {
         TxPrint(     "Bootsector format : %s%s%s\n", CBM, fsform, CNN);
         TxPrint(     "Part. creator     : %s\n",          creatr);
         if (bootos != DFS_NT)
         {
            TxPrint(  "Partition label   : %s%s%s\n", CNC, plabel, CNN);
         }
         TxPrint(     "Bytes per sector  :     0x%4.4X = %4u", sd->eb.SectSize, sd->eb.SectSize);
         if (sd->eb.ClustSize != 0)
         {
            TxPrint( "          %u sectors/cluster", (USHORT) sd->eb.ClustSize);
         }
         TxPrint("\n");
         switch (bootos)
         {
            case DFS_OS:
               if ((sd->eb.FatOffset           != 0) &&
                   (sd->eb.NrOfFats            != 0) &&
                   (sd->eb.RootEntries         != 0) &&
                   (sd->eb.FatSectors          != 0) &&
                   (strncmp( fsform, "JFS", 3) != 0) )
               {
                  sprintf(   text, "  Offset 1st  FAT :     %s0x%4.4X%s", CBG, sd->eb.FatOffset, CNN);
                  dfstrSize( text, " = ",        (ULONG) sd->eb.FatOffset, "");
                  TxPrint( "%s\n", text);

                  sprintf(   text, "  Number of FAT's : %10u\n",  (USHORT) sd->eb.NrOfFats);
                  dfstrSiz8( text, "  Sectors per FAT : ",        (ULONG)  sd->eb.FatSectors, "");
                  TxPrint( "%s\n", text);

                  strcpy(    text, "");
                  dfstrSiz8( text, "  Sectors in ROOT : ",       (ULONG) ((sd->eb.RootEntries * sizeof(S_FATDIR)) / dfsGetSectorSize()), "");
                  TxPrint( "%s for %s%u%s directory entries\n", text, CBY, sd->eb.RootEntries, CNN);
               }
               else
               {
                  //- No FAT comptible information seen
               }
               break;

            case DFS_F2:
               if ((sd->eb.FatOffset   != 0) &&
                   (sd->eb.NrOfFats    != 0) &&
                   (sd->f2.FatSectors  != 0) )
               {
                  sprintf(   text, "  Offset 1st  FAT :     %s0x%4.4X%s",
                                                    CBG, sd->eb.FatOffset, CNN);
                  dfstrSize( text, " = ",        (ULONG) sd->eb.FatOffset, "");
                  TxPrint( "%s\n", text);

                  sprintf(   text, "  Number of FAT's : %10u\n",
                                                (USHORT) sd->eb.NrOfFats);
                  dfstrSiz8( text, "  Sectors per FAT : ",
                                                (ULONG)  sd->f2.FatSectors, "");
                  TxPrint( "%s\n  FAT32 extFlags  :     0x%4.4hx = ", text, sd->f2.extFlags);
                  if (sd->f2.extFlags & DFSF2_NO_FAT_MIRROR)
                  {
                     TxPrint( "Mirroring disabled, active FAT : %hu\n",
                                (sd->f2.extFlags & DFSF2_FAT_MIRROR_MASK));
                  }
                  else
                  {
                     TxPrint( "Standard, all FATs are mirrored\n");
                  }
                  TxPrint(         "  Entries in ROOT : %s%s%s\n",
                                                  CBY,   "unlimited", CNN);
                  TxPrint(         "  RootDir Cluster : 0x%8.8x\n",
                                                         sd->f2.RootCluster);
               }
               break;

            default:
               break;
         }
         if (bootos == DFS_NT)
         {
            strcpy( text, "Vol");
            sectors = sd->nt.VolSize;
         }
         else if (sd->eb.BigSectors)
         {
            strcpy( text, "Big");
            sectors = sd->eb.BigSectors;
         }
         else
         {
            strcpy( text, "Std");
            sectors = sd->eb.Sectors;
         }
         TxPrint("Media type        :       0x%02.2X = %s\n", sd->eb.MediaType,
                  (sd->eb.MediaType == 0xF8) ? "harddisk" : "diskette");
         TxPrint( "BPB formatted Geo : 0x%2.2X, 0x%2.2X = H:%4u S:%3u  ",
                                         sd->eb.LogGeoHead,    sd->eb.LogGeoSect,
                                         sd->eb.LogGeoHead,    sd->eb.LogGeoSect);
         if ((sd->eb.LogGeoHead  *  sd->eb.LogGeoSect) != 0)
         {
            TxPrint( "Cylinders:%10.3lf (BrBPB-Geo)\n",
                (((double) sectors) / ((double) sd->eb.LogGeoHead  *  sd->eb.LogGeoSect)));
         }
         else
         {
            TxPrint( "Cylinders:     Unknown!\n");
         }
         dfstrSz64( text, " nr of sectors : ", sectors, " =");
         TxPrint( "%s Cylinders:%10.3lf (Dsk L-Geo)\n", text,
                            (((double) sectors) / ((double) dfsCylinderSize())));
         if (dfsCylinderSize() != 0)
         {
            TxPrint( "%s Cylinders:%10.3lf (Dsk L-Geo)\n", text,
                               (((double) sectors) / ((double) dfsCylinderSize())));
         }
         else
         {
            TxPrint( "%s Cylinders:     Unknown!\n", text);
         }
         if (bootos == DFS_F2)
         {
            TxPrint("Physical disk nr  :       0x%02.2X = %7u    "
               "(Std = 0x80)\n",               (USHORT)  sd->f2.PhysDriveNr,
                                               (USHORT)  sd->f2.PhysDriveNr);
            TxPrint("F32 BPB signature :       0x%02.2X = %7u    "
               "(Std = 0x28, 0x29 or 0x80)\n", (USHORT)  sd->f2.Signature,
                                               (USHORT)  sd->f2.Signature);
            FsysValue = sd->f2.FsysValue;
         }
         else
         {
            TxPrint("Physical disk nr  :       0x%02.2X = %7u      "
               "(Std = 0x80)\n",               (USHORT)  sd->os.PhysDriveNr,
                                               (USHORT)  sd->os.PhysDriveNr);
            TxPrint("Ext-BPB signature :       0x%02.2X = %7u      "
               "(Std = 0x28, 0x29 or 0x80)\n", (USHORT)  sd->os.Signature,
                                               (USHORT)  sd->os.Signature);
            FsysValue = sd->os.FsysValue;
         }
         if ((strncmp( fsform, "HPFS", 4) == 0) ||
             (strncmp( fsform, "JFS",  3) == 0)  )
         {
            TxPrint("Dr. letter in BPB :       0x%2.2hx = ", (USHORT) (FsysValue));
            if (FsysValue < 0x80)
            {
               TxPrint( "Not bootable, data partition\n");
            }
            else
            {
               TxPrint( "%s%c:%s\n", CBG, (char) (FsysValue - 0x80 + 'C'), CNN);
            }
            if (strncmp( fsform, "JFS", 3) == 0)
            {
               dfsJfsBootSectorDetails( sd);
            }
         }
         else if (strncmp( fsform, "FAT", 3) == 0)
         {
            TxPrint("FAT DIRTY status  :       0x%2.2hx = ", FsysValue);
            if (FsysValue == 0)
            {
               TxPrint( "CLEAN\n");
            }
            else if (FsysValue <= 0x03)
            {
               TxPrint( "DIRTY, surface-SCAN/CHKDSK required (MS)\n");
            }
            else
            {
               TxPrint( "DIRTY ? non-standard value!\n");
            }
         }

         switch (bootos)
         {
            case DFS_NT:
               dfsX10(  "Master File Table : ", sd->nt.MftClus, CBG, "\n");
               dfsX10(  "MFT duplicate LCN : ", sd->nt.MftCopy, CNC, "\n");

               strcpy( text, "");
               dfstrSiz8( text, "FILE  record size : ",
                   dfsNtfsV2S(  sd->nt.MftSize, sd->eb.ClustSize),        "");
               dfstrX10(  text, "    (size value : ", sd->nt.MftSize, CNG,")");
               TxPrint( "%s\n", text);

               strcpy( text, "");
               dfstrSiz8( text, "INDX  record size : ",
                   dfsNtfsV2S(  sd->nt.DirSize, sd->eb.ClustSize),        "");
               dfstrX10(  text, "    (size value : ", sd->nt.DirSize, CNY,")");
               TxPrint( "%s\n", text);
               TxPrint("Volume-serial nr  : 0x%16.16llX          (used as UUID by Linuxes)\n", sd->nt.SerialNr);
               break;

            case DFS_F2:
               TxPrint("Volume-serial nr  : 0x%08.8X\n",  sd->f2.SerialNr);
               break;

            default:
               TxPrint("Volume-serial nr  : 0x%08.8X\n",  sd->os.SerialNr);
               break;
         }
         nav.up = sectors -1;                   // last sector in filesystem
         strcpy( text, "HiddenSectors     : ");
         dfstrSiz8( text, "", sd->eb.HiddenSectors, "  ");
         sectors = (ULONG) sd->eb.LogGeoHead  * sd->eb.LogGeoSect;
         TxPrint( "%sCylinder size : 0x%4.4llX = %5.2lf MiB\n", text, sectors, TXSMIB( sectors, dfsGetSectorSize()));
      }
   }
   nav.down = dfsa->FsEntry;                    // next sector to visit
   RETURN (rc);
}                                               // end 'dfsBootSector'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display Master- or Extended- Boot Record with Partition table
/*****************************************************************************/
ULONG dfsPartitionRecord                        // RET   rc = 0 if type match
(
   BYTE               *sector,                  // IN    Sector buffer
   BYTE                t,                       // IN    sector type
   ULONG               tablePsn                 // IN    PSN of table sector
)
{
   ULONG               rc = 0;                  // rc, sector match
   S_BOOTR            *sd = (S_BOOTR *) sector;
   DFSPARTENTRY       *pe;                      // part entry ptr
   USHORT              i;
   TXTM                text;
   BOOL                gptPresent = FALSE;      // GPT type partitions
   BYTE                secType    = 0;

   ENTER();

   nav.down = 0;                                // default back to MBR/bootrec
   if (t == ST_MASTR)
   {
      if (sd->NTsignature != 0)                 // touched by NT Disk Admin
      {
         TxPrint("MBR informational : 0x%8.8X = Possible NT Disk "
                 "Administrator signature\n", sd->NTsignature);
      }
      dfsMbrEbrIdentifyDisplay( sector, "MBR crc "); // check bootcode & display
      if (TxMemStr( sector + 100, "GRUB", SECTORSIZE -100) != NULL)
      {
         dfsFdskGrubStage1( sector, (dfsa->verbosity >= TXAO_VERBOSE)); // show GRUB summary
      }
      //- to be refined, add some LILO summary display

   }
   else                                         // extended boot record
   {
      dfsMbrEbrIdentifyDisplay( sector, "EBR crc "); // check bootcode & display
      if ((sd->BootMgrFlag & BT_BM_BOOT) == BT_BM_BOOT)
      {
         TxCopy( text, sd->BootMgrName, 9);
         TxPrint("EBR informational : %-10s   = IBM Bootmanager menu name, "
                 "or LVM placeholder\n", text);
      }
   }
   TxPrint("Offset   Fl|%sBeginCHS%s|%sTp%s|%s EndCHS %s|%s "
           "BR-offset  %s|sectors     %s[%sPartition Table %s]%s\n",
           (t != ST_MASTR) ? CBY : CNG, CNN, CBC, CNN, CNY, CNN,
           (t != ST_MASTR) ? CBG : CBY, CNN, CNC, CNN, CNC, CNN);
   TxDisplayHex(dfs_pthex, (BYTE *) &(sd->PartitionTable[0]),   0x40,
                          ((BYTE *) &(sd->PartitionTable[0])) - sector);
   TxPrint( dfs_pthex);

   for ( i=0; i < 4; i++)                             //- walk partition table
   {
      pe = &sd->PartitionTable[i];
      if (!TxAreaEmpty( pe, sizeof(DFSPARTENTRY), 0))
      {
         if ((t == ST_MASTR) && (tablePsn == 0) && (pe->PartitionType == DFS_P_EFI_GPT))
         {
            gptPresent = TRUE;
         }
         sprintf( text, "Partition index %u : ", i);
         dfsShowPartEntry( text, t, TRUE,             //- show PID, but no 'BAD'
                           dfstGeoHeads(   DFSTORE),
                           dfstGeoSectors( DFSTORE), tablePsn, pe);
      }
   }
   if ((t == ST_MASTR) && (tablePsn == 0))      // it is the MBR
   {
      if (gptPresent && dfsa->gptAuto)
      {
         //- display GPT info but keep MBR current!
         TxPrint("\n");
         dfsReadDisplayPhysical( GPTHDR_PRIMARY_PSN, &secType);
         nav.this = 0;
         nav.this_sninfo = 0;
         nav.down = 0;                          // stop any (WALK) iteration
      }
      else if (!memcmp(((S_MCDDM*)sector)->Signature,sg_mcddm,SG_MCDDM))
      {
         TxPrint( "\nAPM partitions (use 't A -v' to display APM header):\n");
         rc = dfsFdskMacPartMap( TRUE);         // verbose, partitions only
      }
   }
   RETURN (rc);
}                                               // end 'dfsPartitionRecord'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display GRUB stage1 specific part of an MBR or partition bootrecord
/*****************************************************************************/
ULONG dfsFdskGrubStage1
(
   BYTE               *sector,                  // IN    Sector data
   BOOL                verbose                  // IN    complete, incl stage2
)
{
   ULONG               rc = 0;                  // rc, sector match
   S_BOOTR            *s1 = (S_BOOTR *) sector;
   USHORT              disknr = s1->g1.grBootDisk;
   USHORT              pid    = 0;
   DFSPARTINFO        *p;

   ENTER();

   if (disknr && (disknr != 0xff))              // BIOS style disk 0x80 = disk 1
   {
      disknr = (disknr & 7) +1;
   }
   else
   {
      disknr = SINF->disknr;
   }
   pid    = dfsDiskPsn2PartId( disknr, s1->g1.grStage2Psn);

   TxPrint(   "GRUB stageX  @PSN : 0x%8.8x", s1->g1.grStage2Psn);
   if ((p = dfsGetPartInfo( pid)) != NULL)
   {
      TxPrint("   Located on  : %s%s%s  /boot/grub   PID: %02hu\n",
                             CBY, p->UnixDevicePart, CNN, pid);
      if (verbose)
      {
         nav.xtra = s1->g1.grStage2Psn - p->basePsn; // LSN of GRUB stage 2
      }
   }
   else
   {
      if (s1->g1.grStage2Psn < dfstGeoSectors( DFSTORE)) // in MBR track
      {
         TxPrint("   GRUB intermediate stage 1.5 code.\n");
      }
      else
      {
         TxPrint("   Disk and partition can not be resolved!\n");
      }
   }
   if (verbose)
   {
      TxPrint("StageX  segm:addr : %4.4hx:%4.4hx", s1->g1.grStage2Segm, s1->g1.grStage2Addr);
      TxPrint(  "    Forcing LBA : %s\n",         (s1->g1.grForceLBA) ? "Yes" : "No");
      TxPrint("Stage1 install by : %hu.%hu      ", s1->g1.grMajor,      s1->g1.grMinor);
      TxPrint(  "    Boot Disk # : 0x%2.2hx",      s1->g1.grBootDisk);
      TxPrint(      "   = disknr : %hu\n",         disknr);

      rc = dfsFdskGrubStage2( disknr, s1->g1.grStage2Psn);
   }
   RETURN (rc);
}                                               // end 'dfsFdskGrubStage1'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read/verify and display GRUB stage2 (or 1.5) code as identified by stage1
/*****************************************************************************/
ULONG dfsFdskGrubStage2
(
   USHORT              disknr,                  // IN    Disknr stage1/2
   ULONG               s2psn                    // IN    Stage2 1st sector PSN
)
{
   ULONG               rc = 0;                  // rc, sector match
   DFSG2INFO          *s2;
   ULONG               s2sects = (sizeof( DFSG2INFO) + dfsGetSectorSize() -1) /
                                                       dfsGetSectorSize();
   ENTER();
   TRACES(( "s2psn: %8.8x, sects: %u\n", s2psn, s2sects));

   if ((s2 = TxAlloc( s2sects, dfsGetSectorSize())) != NULL)
   {
      if ((rc = dfstReadPsn( DFSTORE, s2psn, s2sects, (BYTE *) s2)) == NO_ERROR)
      {
         char         *str = s2->grVersion + strlen( s2->grVersion) +1;

         TRACES(( "Self:%8.8x Ver:'%s' Str:'%s'\n", s2->grStage2Self, s2->grVersion, str));

         if ((TxMemStr((BYTE *) s2, GR_MSG_STAGE, sizeof( DFSG2INFO)) != NULL) &&
             (s2->grStage2Self  ==  s2psn +1))
         {
            TxPrint("StageX install by : %hu.%hu      ", s2->grMajor, s2->grMinor);
            TxPrint(  "    LoadSegment : 0x%4.4hx = ",   s2->grStageXSegm);
            switch (s2->grStageXSegm)
            {
               case GR_SEG_STAGE15: TxPrint( "stage 1.5 code\n");         break;
               case GR_SEG_STAGE2:  TxPrint( "stage 2 code\n");           break;
               default:             TxPrint( "UNKNOWN stage segment!\n"); break;
            }
            TxPrint("GRUB code version : %s    ", s2->grVersion);
            dfsSiz8("     Code size   : ",        s2->grStage2Sect, "\n");
            if (s2->grStageXSegm == GR_SEG_STAGE15)
            {
               USHORT        pid;
               USHORT        dev;
               DFSPARTINFO  *p;

               str += 2;                        // move to device-number
               dev = (USHORT) (*str) +1;
               pid = dfsDevNr2PartId( disknr, dev);

               str += 2;                        // move to stage string
               if ((p = dfsGetPartInfo( pid)) != NULL)
               {
                  TxPrint( "GRUB config is in : %s%s%s   Grub name   : "
                           "(hd%hu,%hu) %15.15s PID : %02hu\n",
                             CBY, p->UnixDevicePart, CNN, disknr -1, dev -1, "", pid);
               }
               else
               {
                  TxPrint("GRUB config files :    Disk and partition can not be resolved!\n");
               }
               TxPrint( "GRUB next stage   : %s\n", str);
            }
            else
            {
               TxPrint( "GRUB Config file  : %s\n", str);
            }
         }
         else
         {
            TxPrint( "GRUB Stage-X area : Loading text is invalid!   "
                     "%sUnlikely to boot!%s\n", CBR, CNN);
            rc = DFS_BAD_STRUCTURE;
         }
      }
      else
      {
         TxPrint( "GRUB Stage-X area : 0x%8.8x can not be read!\n", s2psn);
      }
      TxFreeMem( s2);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsFdskGrubStage2'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display LVM disk/volume information sector (last sector MBR/EBR track)
/*****************************************************************************/
ULONG dfsFdskLvmInfo
(
   BYTE               *sector                   // IN    Sector data
)
{
   ULONG               rc = 0;                  // rc, sector match
   S_LVINF            *sd = (S_LVINF *) sector;
   S_LVPART           *lp;                      // LVM part info
   TXLN                geo;
   int                 pi;                      // partition index

   ENTER();

   sprintf( geo, "Disk geometry Cyl :%6u H:%3u S:%2u = %4.2lf MiB ",
                  sd->geoCyls, sd->geoHeads, sd->geoSecs,
                  TXSMIB( (sd->geoHeads * sd->geoSecs), dfsGetSectorSize()));
   dfstrSiz8( geo, "Size : ", (sd->geoCyls * sd->geoHeads * sd->geoSecs), "");
   TxPrint( "%s\n", geo);
   TxPrint("    Drive Serial# : 0x%8.8X = %-10u Name : %-*.*s\n",
                sd->ThisDiskId, sd->ThisDiskId, LVM_NAME_L,LVM_NAME_L, sd->DiskName);
   TxPrint("    BootDrSerial# : 0x%8.8X = %-10u\n",  sd->BootDiskId, sd->BootDiskId);
   TxPrint("    Install Flags : 0x%8.8X    %s\n", sd->InstallFlags,
                (sd->RebootFlag) ? "Boot : flag set" : "");

   dfsFdskLvmProcessCRC( sector, &(sd->LvmCRC));

   for (pi = 0; pi < 4; pi++)
   {
      lp = &(sd->part[pi]);
      if (lp->sectors != 0)
      {
         ULONG         c,h,s;                   // cyl, head, sect
         ULONG         last = lp->partPsn +lp->sectors -1;
         USHORT        pid  = dfsLvmSnP2PartId( SINF->disknr, lp->PartitId);

         dfsGeoPsn2Chs( lp->partPsn,  sd->geoHeads, sd->geoSecs, &c, &h, &s);
         TxPrint("Partition 1st PSN : 0x%8.8X = %-10u ", lp->partPsn, lp->partPsn);
         TxPrint("    %sCHS:%7lu %3u %-3u%s = %s%8.8x%s\n", CNY, c, h, s, CNN, CNG, lp->partPsn, CNN);
         dfsGeoPsn2Chs( last,         sd->geoHeads, sd->geoSecs, &c, &h, &s);
         TxPrint("    Part last PSN : 0x%8.8X = %-10u ", last, last);
         TxPrint("    %sCHS:%7lu %3u %-3u%s = %s%8.8x%s\n", CNY, c, h, s, CNN, CNG, last, CNN);
         TxPrint("    PartitionSize : 0x%8.8X = %-10u", lp->sectors, lp->sectors);
         dfsSizeXiB(        " = ", lp->sectors, dfsGetSectorSize(), "\n");
         TxPrint("    Volume letter : %s",     CBG);
         if (lp->letter == '*')
         {
            TxPrint("* (auto)%s",  CNN);
         }
         else if (lp->letter)
         {
            TxPrint("%c:      %s", lp->letter, CNN);
         }
         else
         {
            TxPrint("none    %s",  CNN);
         }
         TxPrint("%16.16sbmgr : %s%s%s  %s", "",
            (lp->OnBmMenu)    ?  CBC   : "",
            (lp->OnBmMenu)    ? "YES, on menu  "   : "not selectable", CNN,
            (lp->Installable) ? "(Installable)  "  : "       ");

         if (dfsFdskDlatEntry2Part( lp, SINF->disknr) == NULL)
         {
            TxPrint( "(%snot CURRENT!%s)", CBR, CNN);
         }

         TxPrint("\n    VolumeSerial# : 0x%8.8X = %-10u Name : %-*.*s\n",
               lp->VolumeId, lp->VolumeId, LVM_NAME_L,LVM_NAME_L, lp->VoluName);
         TxPrint(  "    Part. Serial# : 0x%8.8X = %-10u Name : %-*.*s PID: %02.2hu\n",
               lp->PartitId, lp->PartitId, LVM_NAME_L,LVM_NAME_L, lp->PartName, pid);

         if (pi == 0)
         {
            nav.xtra = lp->partPsn - dfstGeoSectors( DFSTORE);
         }
         nav.down = lp->partPsn;                // first sector of partition
         nav.up   = last;                       // LVM-sign or NTFS spare
      }
   }
   RETURN (rc);
}                                               // end 'dfsFdskLvmInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display LVM signature/feature sector (end of non-compat partitions)
/*****************************************************************************/
ULONG dfsFdskLvmSign
(
   BYTE               *sector                   // IN    Sector data
)
{
   ULONG               rc = 0;                  // rc, sector match
   S_LVSIG            *sd = (S_LVSIG *) sector;
   S_LVFEAT           *lf;                      // LVM feature info
   int                 fi;                      // feature index
   ULONG               first = sd->firstPsn;    // remember start PSN ...
   ULONG               autoDisplayDlPsn = 0;    // auto display drive-link
   ULONG               autoDisplayBbPsn = 0;    // auto display BadBlockRs
   ULONG               autoDisplayFkEbr = sd->fakeEbrPsn;
   USHORT              pid;

   ENTER();

   pid = dfsLvmSnP2PartId( SINF->disknr, sd->PartitId);
   TxPrint("LVM VolumeSerial# : 0x%8.8X = %-10u Name : %-*.*s\n",
             sd->VolumeId, sd->VolumeId, LVM_NAME_L, LVM_NAME_L, sd->VoluName);
   TxPrint("    Part. Serial# : 0x%8.8X = %-10u Name : %-*.*s PID: %02.2hu\n",
             sd->PartitId, sd->PartitId, LVM_NAME_L, LVM_NAME_L, sd->PartName, pid);
   TxPrint("    BootDrSerial# : 0x%8.8X = %-10u Disk : %-*.*s\n",
             sd->BootDiskId, sd->BootDiskId, LVM_NAME_L, LVM_NAME_L, sd->DiskName);
   TxPrint("    Volume letter : %s", CBG);
   if (sd->letter == '*')
   {
      TxPrint("* (auto)%s",  CNN);
   }
   else if (sd->letter)
   {
      TxPrint("%c:      %s\n",                sd->letter, CNN);
   }
   else
   {
      TxPrint("none    %s\n",  CNN);
   }
   dfsDisplayTransLsn("    Part  1st PSN : ", sd->firstPsn);
   dfsDisplayTransLsn("    Part last PSN : ", sd->lastPsn);
   dfsSiz8("    Part     size : ",            sd->sectors, "\n");
   dfsSiz8("    Reserved area : ",            sd->reserved, "\n");
   dfsSiz8("    Reported size : ",            sd->reported, "\n");
   TxPrint("    LVM   version : %hu.%hu\n",   sd->majVers, sd->minVers);
   TxPrint("    fake EBR  PSN : 0x%8.8X\n",  sd->fakeEbrPsn);
   TxPrint("    Sequence   nr : 0x%08.8X",   sd->SequenceNr);
   TxPrint(          "   Next : 0x%08.8X\n", sd->NextAggregate);
   TxPrint("    LVM   comment : %-*.*s\n",    LVM_COMM_L, LVM_COMM_L, sd->pComment);

   dfsFdskLvmProcessCRC( sector, &(sd->LvmCRC));

   for (fi = 0; fi < LVM_FEATURES; fi++)
   {
      lf = &(sd->feature[fi]);
      if (lf->sectors != 0)
      {
         ULONG         fbase;                   // base for feature SN

         switch (lf->FeatureId)                 // set absolute/relative SN
         {
            case LVMF_BAD_BLOCKR: fbase = sd->firstPsn;                 break;
            default:              fbase = 0;                            break;
         }
         TxPrint(  "Feature prim. PSN : 0x%08.8X",   lf->priPsn + fbase);
         TxPrint(      "   Feature-id : % -8u   = ", lf->FeatureId);
         switch (lf->FeatureId)
         {
            case LVMF_DRIVE_LINK: TxPrint( "Drive Linking\n"         ); break;
            case LVMF_BAD_BLOCKR: TxPrint( "Bad Block Relocation\n"  ); break;
            default:              TxPrint( "Unknown (plugin)\n"      ); break;
         }
         TxPrint(  "    Secondary PSN : 0x%08.8X",   lf->secPsn  + fbase);
         dfsSiz8(      "   alloc size : ",            lf->sectors, "\n");
         TxPrint(  "    Version    nr : %hu.%hu  ",   lf->majVers, lf->minVers);
         TxPrint( "        status-ind : %sactive\n", (lf->active) ? "" : "in");
         switch (lf->FeatureId)                 // set next LSN to view
         {
            case LVMF_DRIVE_LINK: autoDisplayDlPsn = lf->priPsn + fbase;  break;
            case LVMF_BAD_BLOCKR: autoDisplayBbPsn = lf->priPsn + fbase;  break;
            default:                                                      break;
         }
      }
   }
   if (autoDisplayBbPsn != 0)
   {
      BYTE                st = 0;               // use autodetect

      TxPrint("\nAutomatic display : Related Bad Block Relocation sectors ...\n");
      rc = dfsReadAnDisplay( autoDisplayBbPsn - dfstLSN2Psn(DFSTORE, 0), 0, &st);
   }
   if (autoDisplayDlPsn != 0)
   {
      BYTE                st = 0;               // use autodetect

      nav.xtra = autoDisplayDlPsn;              // to keep it around ...
      TxPrint("\nAutomatic display : Related Drive Linking sectors ...\n");
      rc = dfsReadAnDisplay( autoDisplayDlPsn - dfstLSN2Psn(DFSTORE, 0), 0, &st);
   }
   if (autoDisplayFkEbr != 0)
   {
      BYTE               *sec = NULL;           // sector buffer

      if ((sec = (TxAlloc( 1, dfsGetSectorSize()))) != NULL)
      {
         rc = dfstReadPsn( DFSTORE, autoDisplayFkEbr, 1, sec);
         if (rc == NO_ERROR)
         {
            //- dfstSetLastPsn( DFSTORE, first - dfstGeoSectors( DFSTORE));
            TxPrint("\nAutomatic display : Related fake-EBR sector "
                    "at 0x%s%8.8X%s relocated to 0x%s%8.8X%s\n",
              CBG, autoDisplayFkEbr, CNN, CBM, first - dfstGeoSectors( DFSTORE), CNN);

            dfsPartitionRecord( sec, ST_EXTBR, first - dfstGeoSectors( DFSTORE));
         }
         TxFreeMem( sec);                       // free the memory
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   nav.down = first -1;                         // most likely the LVM info
   RETURN (rc);                                 // remembered next SN
}                                               // end 'dfsFdskLvmSign'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display LVM drive-linking feature, master including table sectors
/*****************************************************************************/
ULONG dfsFdskLvmDlf
(
   BYTE               *sector                   // IN    Sector data
)
{
   ULONG               rc = 0;                  // rc, sector match
   S_LVDLF            *sd = (S_LVDLF *) sector;
   ULONG               li;                      // link index

   ENTER();

   dfsFdskLvmProcessClass( sd->ActualClass, sd->TopOfClass);
   dfsFdskLvmProcessCRC( sector, &(sd->LvmCRC));

   TxPrint( "  sequence number : % -8u    ", sd->SequenceNr);
   TxPrint(     " Links in use : % -8u\n",   sd->LinksInUse);

   TxPrint( "  feature  seq-nr : % -8u    ", sd->FeatureSequenceNr);
   TxPrint(     " Aggregate-S# : % -8u\n",   sd->AggregateSerNr);

   for (li = 0; (li < sd->LinksInUse) && (li < LVDLT_SIZE1); li++)
   {
      ULONG            lvmsnp = sd->LinkTable[ li].PartitionSerialNr;

      TxPrint(" % 2u: DriveSerial# : 0x%8.8X, "
                    " Part Serial# : 0x%8.8X = %-10u PID: %02.2hu\n",
                      li, sd->LinkTable[ li].DriveSerialNr, lvmsnp, lvmsnp,
                      dfsLvmSnP2PartId( SINF->disknr, lvmsnp));
   }
   RETURN (rc);
}                                               // end 'dfsFdskLvmDlf'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display LVM drive-linking table. (based on non-zero entries, not count!)
/*****************************************************************************/
ULONG dfsFdskLvmDlt
(
   BYTE               *sector                   // IN    Sector data
)
{
   ULONG              rc = 0;                   // rc, sector match
   S_LVDLT            *sd = (S_LVDLT *) sector;
   ULONG               li;                      // link index

   ENTER();

   dfsFdskLvmProcessCRC( sector, &(sd->LvmCRC));

   TxPrint( "  sequence number : % -8u\n", sd->SequenceNr);

   for ( li = 0;
        (li < LVDLT_SIZEN) && (sd->LinkTable[li].DriveSerialNr != 0);
         li++)
   {
      ULONG            lvmsnp = sd->LinkTable[ li].PartitionSerialNr;

      TxPrint(" % 2u: DriveSerial# : 0x%8.8X,  "
                     "Part Serial# : 0x%8.8X = %-10u PID: %02.2hu\n",
                      li, sd->LinkTable[ li].DriveSerialNr, lvmsnp, lvmsnp,
                      dfsLvmSnP2PartId( SINF->disknr, lvmsnp));
   }
   TxPrint( " Nr of links used : % -8u\n", li);
   RETURN (rc);
}                                               // end 'dfsFdskLvmDlt'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display LVM bad-block-rel feature, master sector, admin only
/*****************************************************************************/
ULONG dfsFdskLvmBBf
(
   BYTE               *sector                   // IN    Sector data
)
{
   ULONG               rc = 0;                  // rc, sector match
   S_LVBBF            *sd = (S_LVBBF *) sector;

   ENTER();

   dfsFdskLvmProcessClass( sd->ActualClass, sd->TopOfClass);
   dfsFdskLvmProcessCRC( sector, &(sd->LvmCRC));

   TxPrint( "  sequence number : % -8u    ", sd->SequenceNr);
   TxPrint( " Nr of links used : % -8u\n",   sd->TableEntriesInUse);
   TxPrint( " size repl. table : % -8u    ", sd->TableSize);
   TxPrint( " Sectors in table : % -8u\n",   sd->SectorsPerTable);
   TxPrint( " First repl.  LSN : 0x%8.8X  ", sd->FirstReplSector);
   TxPrint( " Last  repl.  LSN : 0x%8.8X\n", sd->LastReplSector );
   TxPrint( " repl. sector cnt : % -8u    ", sd->ReplSectorCount);
   TxPrint( " BBR global flags : 0x%8.8X\n", sd->Flags);
   TxPrint( "  feature  seq-nr : % -8u\n",   sd->FeatureSequenceNr);

   RETURN (rc);
}                                               // end 'dfsFdskLvmBBf'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display LVM bad-block-rel table. (based on non-zero entries, not count!)
/*****************************************************************************/
ULONG dfsFdskLvmBBt
(
   BYTE               *sector                   // IN    Sector data
)
{
   ULONG               rc = 0;                  // rc, sector match
   S_LVBBT            *sd = (S_LVBBT *) sector;
   ULONG               li;                      // link index

   ENTER();

   dfsFdskLvmProcessCRC( sector, &(sd->LvmCRC));

   TxPrint( "  sequence number : % -8u\n", sd->SequenceNr);

   for ( li = 0;
        (li < LVDLT_SIZEN) && (sd->BbrTable[li].Badsector != 0xffffffff);
         li++)
   {
      TxPrint(" % 2u: Bad sector # : 0x%8.8X,  Replacement sect : 0x%8.8X\n",
                  li, sd->BbrTable[ li].Badsector,
                      sd->BbrTable[ li].Replacement);
   }
   TxPrint( " Nr of links used : % -8u\n", li);
   RETURN (rc);
}                                               // end 'dfsFdskLvmBBt'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display and check LVM feature class information
/*****************************************************************************/
static void dfsFdskLvmProcessClass
(
   ULONG               ActualClass,             // IN    feature class
   BOOL                TopOfClass               // IN    LVM top of class
)
{
   ENTER();

   dfsFdskLvmDisplayClass( "LVM Feature class : ", ActualClass, "        ");
   TxPrint( "Top of Class: %s\n", (TopOfClass) ? "Yes" : "No");
   VRETURN( );
}                                               // end 'dfsFdskLvmProcessClass'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display LVM feature class information
/*****************************************************************************/
static void dfsFdskLvmDisplayClass
(
   char               *lead,                    // IN    lead string
   ULONG               Class,                   // IN    feature class
   char               *tail                     // IN    tail string
)
{
   ENTER();

   TxPrint( "%s%10u = %s",  lead, Class, CBC);
   switch (Class)
   {
      case LVM_CLASS_PARTITION:  TxPrint( "Partition ");  break;
      case LVM_CLASS_AGGREGATE:  TxPrint( "Aggregate ");  break;
      case LVM_CLASS_VOLUME:     TxPrint( "Volume    ");  break;
      default:                   TxPrint( "Unknown!! ");  break;
   }
   TxPrint( "%s%s", CNN, tail);
   VRETURN( );
}                                               // end 'dfsFdskLvmDisplayClass'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Check and display LVM type sector-CRC value, correct when BAD
/*****************************************************************************/
ULONG dfsFdskLvmProcessCRC
(
   BYTE               *sector,                  // IN    Sector data
   ULONG              *actualCRC                // INOUT CRC value in sector
)
{
   ULONG               rc = 0;                  // rc, CRC OK
   ULONG               sectorCRC;               // original CRC

   ENTER();

   sectorCRC  = *actualCRC;
   *actualCRC = 0;                              // prepare for calculation
   *actualCRC = TxCalculateLvmCrc( sector, SECTORSIZE);
   TxPrint( "    LVM sectorCRC : 0x%s%8.8X%s   ",
      (*actualCRC == sectorCRC) ? CNG : CBR,  sectorCRC, CNN);
   if (*actualCRC == sectorCRC)
   {
      TxPrint( "checked, OK\n\n");
   }
   else
   {
      TxPrint( "BAD! : 0x%8.8X is the correct CRC value\n\n", *actualCRC);
      rc = DFS_BAD_STRUCTURE;
   }
   RETURN (rc);
}                                               // end 'dfsFdskLvmProcessCRC'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display LUKS Crypto Header (Linux disk/partition/container key-setup)
// Note: Will only display up to 6 of the 8 keys (first header sector only)
/*****************************************************************************/
ULONG dfsFdskLuksHdr
(
   BYTE               *sector                   // IN    Sector data
)
{
   ULONG               rc = 0;                  // rc, sector match
   S_LUKSH            *sd = (S_LUKSH *) sector;

   ENTER();

   TxPrint("LUKS signature ID : %*.*s\n",  SG_LUKSH,     SG_LUKSH, sd->Signature      );
   TxPrint("Header    version : %hu\n",                    TxBE16( sd->version)       );
   TxPrint("Cipher       Name : %-*.*s\n", LUKS_STR_L, LUKS_STR_L, sd->cipherName     );
   TxPrint("Cipher       Mode : %-*.*s\n", LUKS_STR_L, LUKS_STR_L, sd->cipherMode     );
   TxPrint("HashSpecification : %-*.*s\n", LUKS_STR_L, LUKS_STR_L, sd->hashSpec       );
   TxPrint("Crypto key length : %u bits\n",                TxBE32( sd->keyBytes) * 8  );
   TxPrint("Crypto area  UUID : %-*.*s\n", LUKS_UUIDL, LUKS_UUIDL, sd->uuidString     );

   TxPrint("Crypto key slot 0 : %s\n", (TxBE32( sd->keyblock[0].active) == LUKS_KEY_ENABLED) ? "ENABLED" : "-");
   TxPrint("Crypto key slot 1 : %s\n", (TxBE32( sd->keyblock[1].active) == LUKS_KEY_ENABLED) ? "ENABLED" : "-");
   TxPrint("Crypto key slot 2 : %s\n", (TxBE32( sd->keyblock[2].active) == LUKS_KEY_ENABLED) ? "ENABLED" : "-");
   TxPrint("Crypto key slot 3 : %s\n", (TxBE32( sd->keyblock[3].active) == LUKS_KEY_ENABLED) ? "ENABLED" : "-");
   TxPrint("Crypto key slot 4 : %s\n", (TxBE32( sd->keyblock[4].active) == LUKS_KEY_ENABLED) ? "ENABLED" : "-");
   TxPrint("Crypto key slot 5 : %s\n", (TxBE32( sd->keyblock[5].active) == LUKS_KEY_ENABLED) ? "ENABLED" : "-");
   TxPrint("Crypto key slot 6 : %s\n", "?");
   TxPrint("Crypto key slot 7 : %s\n", "?");    // 6 and 7 are beyond the 1st sector ...

   dfsX10( "Payload    Offset : ",      TxBE32( sd->payloadOffset), CBG, "");
   dfsSize( " = ",                      TxBE32( sd->payloadOffset), " from start\n");
   nav.down   =                         TxBE32( sd->payloadOffset);

   RETURN (rc);
}                                               // end 'dfsFdskLuksHdr'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display Apple Core Storage header (used for native HFS or FileVault on GPT)
/*****************************************************************************/
ULONG dfsFdskCoreStg
(
   BYTE               *sector                   // IN    Sector data
)
{
   ULONG               rc = 0;                  // rc, sector match
   S_CORE_ST          *sd = (S_CORE_ST *) sector;

   ENTER();

   TxPrint(     "Header    version : %hu\n",      sd->version);
   TxPrint(     "Sectorsize, bytes : %u\n",       sd->sectorSize);
   TxPrint(     "Blocksize,  bytes : %u\n",       sd->blockSize);
   dfsX10(      "Meta label Offset : ",           sd->metaOffset, CBG, "\n");

   dfsSz64Byte( "MetadataBlocksize : ",  (ULN64)  sd->metaSize, " = ");
   dfsUlDot13(   "",                              sd->metaSize,  " bytes\n");

   TxPrint(     "Crypto key length : %u bits\n",  sd->keyLength * 8);
   TxPrint(     "CryptoAlg version : %u\n",       sd->cryptoVersion);

   //- Note: these UUID values seem to be byte-swapped ON-disk already (unlike GPT etc)
   TxPrint(     "Aes-Xts-Key1 UUID : %s\n", dfsFsUuidValueNoSwap( sd->aesXtsKey));
   TxPrint(     "PhysicalVolume ID : %s\n", dfsFsUuidValueNoSwap( sd->physVolumeUuid));
   TxPrint(     "Log Vol Group  ID : %s\n", dfsFsUuidValueNoSwap( sd->logVolGroupUuid));

   RETURN (rc);
}                                               // end 'dfsFdskCoreStg'
/*---------------------------------------------------------------------------*/


#endif


/*****************************************************************************/
// Get the BMGR information index for a given partition-table entry on a disk
// Note: when no existing found, the index will be set to a 'free' entry
// Note: BOOT flag (1st byte) not tested, this avoids unwanted mismatches
/*****************************************************************************/
BOOL dfsFdskGetBmiIndex                         // RET   existing bmi found
(
   USHORT              disk,                    // IN    disk number
   DFSPARTENTRY       *pe,                      // IN    partition-table entry
   BMPRIMARY          *bmi,                     // IN    start of BMI array
   ULONG              *index                    // OUT   index (or DFS32MAX)
)
{
   ULONG               bi = DFS32MAX;           // bmi index value
   DFSPARTINFO        *p;                       // partinfo
   BMPRIMARY          *b;                       // BMI entry
   USHORT              pi;                      // partition index
   USHORT              i;                       // bm-info index

   ENTER();
   TRARGS(("disk:%hu ptype:%hu\n", disk, pe->PartitionType));
   TRHEXS(70, pe,   0x10, "partition entry ");
   TRHEXS(70, bmi,  0x80, "bmi data");

   *index = DFS32MAX;                           // init return value
   if ((disk != 0) && (disk < DFS_MAX_DISK))
   {
      for (i = 0; (i < 4) && (*index == DFS32MAX); i++)
      {
         bi = 4 * (disk -1) + i;                // start of table for this disk
         b  = &(bmi[bi]);

         if ((pe->FirstSector.Head   == b->FirstSector.Head   ) &&
             (pe->FirstSector.SecCyl == b->FirstSector.SecCyl ) )
         {
            *index = bi;
            TRACES(("Found exact match at bm index: %u\n", bi));
         }
      }

      //- when no exact match, try to find an obsolete or free entry
      for (i = 0; (i < 4) && (*index == DFS32MAX); i++)
      {
         bi = 4 * (disk -1) + i;
         b  = &(bmi[bi]);

         for ( pi  = 1,                   *index  = bi;
              (pi <= dfsPartitions()) && (*index != DFS32MAX);
               pi++)
         {
            p = dfsGetPartInfo( pi);
            if ((p) && (p->disknr == disk) && (p->primary))
            {
               if ((p->partent.FirstSector.Head   == b->FirstSector.Head  ) &&
                   (p->partent.FirstSector.SecCyl == b->FirstSector.SecCyl) )
               {                                // No, not free/obsolete
                  *index = DFS32MAX;            // bm-primary still valid
                  TRACES(("Valid primary     at bm index: %u\n", bi));
               }
            }
         }
      }
      TRACES(("resulting index: %u\n", *index));
   }
   BRETURN (*index != DFS32MAX);
}                                               // end 'dfsFdskGetBmiIndex'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get partition-data if DLAT entry is for a current, existing partition
/*****************************************************************************/
DFSPARTINFO *dfsFdskDlatEntry2Part              // RET   partition info or NULL
(
   S_LVPART           *entry,                   // IN    DLAT entry
   USHORT              disknr                   // IN    disknumber for entry
)
{
   DFSPARTINFO        *rc = NULL;               // function return
   DFSPARTINFO        *p;
   USHORT              pid;

   ENTER();

   TRACES(( "entry->partPsn: %llx  disknr: %u, partitions:%hu\n",
             entry->partPsn,       disknr,   dfsPartitions()));

   for (pid = dfsPartitions(); pid; pid--)
   {
      p = dfsGetPartInfo( pid);
      if ((p->basePsn == entry->partPsn) && (p->disknr == disknr))
      {
         rc = p;                                // there is a partition on same
         break;                                 // disk with the same start PSN
      }
   }
   RETURN (rc);
}                                              // end 'dfsFdskDlatEntry2Part'
/*---------------------------------------------------------------------------*/

