//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// FDISK utility functions, LVM and CREATE section
//
// Author: J. van Wijk
//
// JvW  18-08-2005   Initial version, splitt off from DFSUFDSK
// JvW  18-04-1999   Moved some stuff to new DFSUPART
// JvW  04-04-2000   Fixed trap using dfsFdskGetBmiIndex (uninitialized)
// JvW  13-04-2000   Fixed typo in message (unlimited)
// JvW  05-06-2000   Added nr of cylinders to BOOTR display
// JvW  02-07-2000   Moved CountDisks from dfsector.c
// JvW  03-04-2001   Added LVM specific utility functions like CRC
// JvW  29-06-2001   Fixed max-head on CHS entries with cyl > 1023
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsver.h>                             // DFS version info
#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsmdisk.h>                           // Disk store memory disks
#include <dfsutil.h>                            // DFS utility functions
#include <dfsspace.h>                           // DFS  file-space interface
#include <dfstable.h>                           // SLT utility functions
#include <dfsantfs.h>                           // NTFS  interface (bootRec)
#include <dfsupart.h>                           // FDISK partition functions

#include <dfsafdsk.h>                           // FDISK analysis functions
#include <dfsufdsk.h>                           // FDISK utility functions
#include <dfscfdsk.h>                           // FDISK command functions
#include <dfsmedia.h>                           // Partitionable Media manager


/*****************************************************************************/
// Save initial LVM info from current tables to be used by Volume Conversion
/*****************************************************************************/
S_LVINIT *dfsLvmVcuPreservation                 // RET    table S_LVINIT[#part]
(                                               //        pid is index in array
   void
)
{
   S_LVINIT           *rc = NULL;               // function return
   USHORT              pc;                      // partition count
   USHORT              pi;                      // partition index

   ENTER();

   if ((pc = dfsPartitions()) != 0)            // any partition info there ?
   {
      if ((rc = TxAlloc( pc +1, sizeof( S_LVINIT))) != NULL)
      {
         DFSPARTINFO  *p;

         for (pi = 1; pi <= pc; pi++)
         {
            if ((p = dfsGetPartInfo(pi)) != NULL)
            {
               if ((p->drive[0] != '-') && (p->drive[0] != 0))
               {
                  rc[ pi].letter = p->drive[0]; // has a driveletter
               }
               if      (strlen(p->blabel) > 0)  // bootable by bootmgr
               {
                  rc[ pi].OnBmMenu = 1;

                  strcpy( rc[pi].VoluName, p->blabel);
               }
               else if (strlen(p->plabel) > 0)  // has a volume-label
               {
                  if ((p->plabel[0] != '') &&  // not the Bootmanager
                      (p->plabel[0] != ' ') &&  // not a space-filler
                      (strstr( p->blabel, "NONAME") == NULL))
                  {
                     strcpy( rc[pi].VoluName, p->plabel);
                  }
               }
            }
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsLvmVcuPreservation'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Make in-memory ebrChain and ebrHead consistent after any updates
// also reset FLAG byte in the entries for EBR to ZERO (except MBR container)
// Note: Preparation for on-disk structure update with IteratePebr()
/*****************************************************************************/
ULONG dfsFdskUpdateExtChain
(
   USHORT              disk                     // IN    disknr to work on
)
{
   ULONG               rc = NO_ERROR;           // function return
   DFSPARTINFO        *e;                       // ptr to ebr  info
   DFSPARTINFO        *p;                       // ptr to related part info
   DFSDISKINFO        *d;                       // ptr to related disk info
   ULONG               partPsn = 0;             // last valid Ptable PSN
   ULONG               ebrBase = 0;             // start of extended container
   ULONG               ebrLast = 0;             // end of container
   FDSK_CB_INFO        cbi;                     // info struct for FillPtable

   ENTER();

   //- The logic in this function is broken with regard to the 'e->id' being 0
   //- It is assumed that means the container is empty, but it really means
   //- this is the MBR-container! However, trying to fix this has not resulted
   //- in a properly functioning cleanup yet (delete leaves empty containers!)
   //- The logic-flaw corrupts the EBR-chain info as described in P#1345
   //- Since that problem is solved otherwise, leave it as is for now ...

   d = dfsGetDiskInfo( disk);
   if (d != NULL)
   {
      //- find correct ebrBase and ebrLast values
      for (e = d->ebrHead; e; e = e->ebrChain)
      {
         if (e->id != 0)                        // EBR in-use / non-empty ?
         {
            if (ebrBase == 0)
            {
               ebrBase = e->basePsn;
            }
            if ((p = dfsGetPartInfo( e->id)) != NULL)
            {
               if (p->partPsn == e->basePsn)    // is this our partition ?
               {                                // (may not be on a 'create'!)
                  e->lastPsn = p->lastPsn;                 //- sync container size
                  e->sectors = e->lastPsn - e->basePsn +1; //- with the partition
               }
            }
            ebrLast    = e->lastPsn;            // current end position
            TRACES(( "ebrBase: %8.8lx ebrLast %8.8lx after id: %hu\n", ebrBase, ebrLast, e->id));

            e->partPsn = partPsn;               // make it reference the last
            partPsn    = e->basePsn;            // and remember for next one
         }
      }

      //- update first non-empty with new ebrBase and ebrLast values
      for (e = d->ebrHead; e; e = e->ebrChain)
      {
         if (e->id != 0)                        // EBR in-use / non-empty ?
         {
            e->basePsn = ebrBase;               // Base for first in chain
            e->lastPsn = ebrLast;               // Last sector Ext container
            e->sectors = ebrLast - ebrBase +1;  // adjust container size
            break;
         }
      }

      //- update all corresponding partitiontable-entries with new values
      //- and reset any non-zero FLAG bytes for them
      for (e = d->ebrHead; e; e = e->ebrChain)
      {
         if (e->id != 0)                        // All except MBR container
         {
            memset( &cbi, 0, sizeof(cbi));      // initial callback info
            cbi.sn      = e->basePsn;
            cbi.number  = e->sectors;
            cbi.ntype   = e->partent.PartitionType;
            dfsFillPtableEntry( e, &cbi, &(e->partent));

            //- reset ACTIVE flag in containers, except the EXTENDED one
            //- in the MBR to allow active extended partition for GRUB2
            if ((e->partent.Status != 0) && (e->partPsn != 0))
            {
               dfsX10("Fix Part-Table at : ", e->partPsn, CBG,
                      ", reset ACTIVE logical container!\n");
               e->partent.Status = 0;           // make sure it is not ACTIVE
            }                                   // to avoid active containers
         }
         else
         {
            memset( &e->partent, 0, sizeof(DFSPARTENTRY)); // all ZEROES
         }
      }

      //- update the BootSectorOffset for all extended-containers
      for (e = d->ebrHead; e; e = e->ebrChain)
      {
         if (e->id != 0)                        // EBR in-use / non-empty ?
         {
            if (e->partPsn == 0)                // first, in MBR
            {
               e->partent.BootSectorOffset = e->basePsn;
            }
            else                                // chained, in EBR
            {
               e->partent.BootSectorOffset = e->basePsn - ebrBase;
            }
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsFdskUpdateExtChain'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Find an partition-table entry of specified type in a partition-record
/*****************************************************************************/
BOOL dfsFindPtableEntry
(
   S_BOOTR            *br,                      // IN    master/extended br
   BYTE                type,                    // IN    type to find
   USHORT             *pi                       // IN    prefered  index
                                                // OUT   partition index
)
{
   BOOL                rc = FALSE;              // function return
   USHORT              i = (*pi < 4) ? *pi : 0; // limit to range 0..3

   ENTER();
   TRARGS(("Find type: %2.2x, prefered entry: %hu\n", type, i));

   if (br->PartitionTable[i].PartitionType == type) // prefered one OK ?
   {
      *pi = i;
      rc = TRUE;
      TRACES(("Prefered OK: %hu\n", i));
   }
   else                                         // search through all entries
   {
      for (i = 0; (i < 4) && (rc == FALSE); i++)
      {
         if (br->PartitionTable[i].PartitionType == type)
         {
            *pi = i;
            rc  = TRUE;
            TRACES(("found index: %hu\n", i));
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsFindPtableEntry'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Count number of hidable and currently visible primary partitions in Ptable
/*****************************************************************************/
USHORT dfsVisPrimaries
(
   S_BOOTR            *br                       // IN    master/extended br
)
{
   USHORT              rc = 0;                  // function return
   USHORT              i;
   BYTE                type;

   ENTER();

   for (i = 0; i < 4; i++)
   {
      type = br->PartitionTable[i].PartitionType;
      if (dfsPartTypeHidable( type))
      {
         rc++;
      }
   }
   RETURN (rc);
}                                               // end 'dfsVisPrimaries'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Fill Ptable-entry values for a partition specified by CBI and PARTINFO
/*****************************************************************************/
ULONG dfsFillPtableEntry
(
   DFSPARTINFO        *p,                       // IN    partition information
   FDSK_CB_INFO       *cbi,                     // IN    create information
   DFSPARTENTRY       *pe                       // OUT   partition-table entry
)
{
   ULONG               rc = NO_ERROR;           // function return
   ULONG               last;                    // last sn
   ULONG               c,h,s;                   // chs numbering
   ULONG               chsStyle;

   ENTER();
   TRACES(( "start PSN:%llx size:%8.8lx using geo H:% 4hu S:% 3hu\n",
             cbi->sn, cbi->number, p->geoHeads, p->geoSecs));

   chsStyle = dfsGetChsStyle( 'c', TXA_CUR, dfsa->batch); // style from 'c' option

   last = (ULONG) cbi->sn + cbi->number -1;     // last sector number

   dfsGeoPsn2Chs( cbi->sn, p->geoHeads, p->geoSecs, &c, &h, &s);
   if (c < 1024)                                // fits in combined SecCyl ?
   {
      pe->FirstSector.SecCyl = DFSCOMBINE(s,c); // make combined sec-cyl value
      pe->FirstSector.Head   = (BYTE) h;
   }
   else                                         // CHS dummy placeholder value
   {
      switch (chsStyle)
      {
         case DFS_CHSTYLE_PQ:
            pe->FirstSector.SecCyl = DFSCOMBINE(s, 1023);
            pe->FirstSector.Head   = (BYTE)    (h);
            break;

         case DFS_CHSTYLE_MS:
            pe->FirstSector.SecCyl = DFSCOMBINE(63, 1023);
            pe->FirstSector.Head   = (BYTE)    (255);
            break;

         default:
            pe->FirstSector.SecCyl = DFSCOMBINE(p->geoSecs, 1023);
            pe->FirstSector.Head   = (BYTE)    (p->geoHeads -1);
            break;
      }
   }

   dfsGeoPsn2Chs( last, p->geoHeads, p->geoSecs, &c, &h, &s);
   if (c < 1024)                                // fits in combined SecCyl ?
   {
      pe->LastSector.SecCyl  = DFSCOMBINE(s,c); // make combined sec-cyl value
      pe->LastSector.Head    = (BYTE) h;
   }
   else                                         // CHS dummy placeholder value
   {
      switch (chsStyle)
      {
         case DFS_CHSTYLE_PQ:
            pe->LastSector.SecCyl = DFSCOMBINE(s, 1023);
            pe->LastSector.Head   = (BYTE)    (h);
            break;

         case DFS_CHSTYLE_MS:
            pe->LastSector.SecCyl = DFSCOMBINE(63, 1023);
            pe->LastSector.Head   = (BYTE)    (255);
            break;

         default:
            pe->LastSector.SecCyl = DFSCOMBINE(p->geoSecs, 1023);
            pe->LastSector.Head   = (BYTE)    (p->geoHeads -1);
            break;
      }
   }

   pe->PartitionType   = cbi->ntype;            // partition type
   pe->NumberOfSectors = cbi->number;           // partition size

   //- Note: BootSectorOffset for extended-container is NOT filled here,
   //-       this must be done separately, using ebrBase (see UpdateExtChain)

   if (dfsIsExtendedType(cbi->ntype) == FALSE)
   {
      if (cbi->flag)                            // primary partition
      {
         pe->BootSectorOffset = (ULONG) cbi->sn; // first sector number
      }
      else                                      // logical partition
      {
         if (cbi->cbOptNum1)                    // specific ptGap specified ?
         {
            pe->BootSectorOffset = cbi->cbOptNum1;
         }
         else
         {
            pe->BootSectorOffset = p->geoSecs;  // one-track offset to EBR
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsFillPtableEntry'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Find latest MBR/EBR sector before the specified PSN
/*****************************************************************************/
ULONG dfsPreceedingTable                        // RET   PSN of last MBR/EBR
(
   ULONG               rPSN                     // IN    reference PSN
)
{
   ULONG               rc = 0;                  // DOS rc
   S_BOOTR            *tbr;                     // master boot record
   int                 pi;                      // partition index
   ULONG               tPSN = 0;                // table PSN last read
   ULONG               lPSN = 0;                // first EBR, base for others
   ULONG               ePSN = 0;                // EBR offset

   ENTER();
   if ((tbr = TxAlloc( 1, dfsGetSectorSize())) != NULL)
   {
      while ((rc == NO_ERROR) && (lPSN < rPSN))
      {
         TRACES(("read Ptable sector at: %8.8lx for ref: %8.8lx\n", lPSN, rPSN));
         tPSN = lPSN;
         rc = dfstReadPsn( DFSTORE, tPSN, 1, (BYTE   *) tbr);
         if (rc == NO_ERROR)
         {
            if (tbr->Signature == SV_BOOTR)
            {
               for (pi = 0; pi < 4; pi++)
               {
                  if (dfsIsExtendedType( tbr->PartitionTable[pi].PartitionType))
                  {
                     lPSN = tbr->PartitionTable[pi].BootSectorOffset;
                     if (tPSN == 0)             // first, in MBR ?
                     {
                        ePSN  = lPSN;           // remember offset
                     }
                     else
                     {
                        lPSN += ePSN;           // add EBR offset
                     }
                     TRACES(("Next Ptable sector at: %8.8lx\n", lPSN));
                     break;
                  }
               }
               if (pi == 4)                     // no EBR link found ?
               {
                  rc = DFS_NOT_FOUND;           // this was last in chain ...
               }
            }
            else
            {
               rc = DFS_ST_MISMATCH;
            }
         }
      }
      TxFreeMem( tbr);
   }
   RETURN (tPSN);                               // last table read, or 0
}                                               // end 'dfsPreceedingTable'
/*---------------------------------------------------------------------------*/

