//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// FDISK utility functions, Get values and parse section
//
// Author: J. van Wijk
//
// JvW  18-08-2005   Initial version, split off from DFSUFDSK
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsver.h>                             // DFS version info
#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsmdisk.h>                           // Disk store memory disks
#include <dfsutil.h>                            // DFS utility functions
#include <dfsspace.h>                           // DFS  file-space interface
#include <dfstable.h>                           // SLT utility functions
#include <dfsantfs.h>                           // NTFS  interface (bootRec)
#include <dfsupart.h>                           // FDISK partition functions

#include <dfsafdsk.h>                           // FDISK analysis functions
#include <dfsufdsk.h>                           // FDISK utility functions
#include <dfsufgpt.h>                           // FDISK GPT utility functions
#include <dfscfdsk.h>                           // FDISK command functions
#include <dfsmedia.h>                           // Partitionable Media manager


/*****************************************************************************/
// Return valid part-id for part-specification;  <empty> | . | * | nr | drive
/*****************************************************************************/
USHORT dfsParsePartSpec                         // RET   0=invalid, nr or ANY
(
   char               *spec,                    // IN    part select spec
   USHORT              disk,                    // IN    related disk (or ANY)
   DFSPARTINFO       **pinfo                    // OUT   partinfo pointer
)
{
   USHORT              rc = 0;                  // function return
   DFSDISKINFO        *d  = dfsGetDiskInfo( disk);
   TXA_OPTION         *opt;

   ENTER();
   TRARGS(("spec='%s' for disk: %4.4hx\n", spec, disk));
   if ((spec != NULL) && (strlen(spec)))        // partition as parameter
   {
      switch (spec[0])
      {
         case '\0':                             // empty
         case '.':                              // .
            rc = 0;
            break;

         case '*':                              // wildcard
         case '0':                              // or ZERO (from ListBox)
            if (strlen(spec) == 1)              // unless multi-digit
            {
               rc = FDSK_ANY;
               break;
            }
         default:
            if (isdigit(spec[0]))               // partition nr specified
            {
               rc = (USHORT) atoi(spec);
               if ((d) && (d->firstpart) && strstr( spec, ",r"))
               {
                  rc += d->firstpart -1;        // calculate absolute PID
               }
            }
            else                                // possible drive letter
            {
               rc = dfsDrive2PartId( spec);     // find drive-letter in admin
            }
            break;
      }
   }
   else if ((opt = TxaOptValue('p')) != NULL)   // from a Partition
   {
      switch (opt->type)
      {
         case TXA_STRING:
            if (strcasecmp( opt->value.string, "AUTO") == 0)
            {
               if (SINF->partid != 0)
               {
                  rc = SINF->partid;            // default current partition
               }
               else                             // use autoPid, usually 'last'
               {                                // partition used, or a new one
                  rc = dfsa->autoPid;           // just created ...
               }
            }
            else                                // assume numeric value
            {
               rc = (USHORT) atoi(opt->value.string);
            }
            break;

         default:
            rc = (USHORT) opt->value.number;
            break;
      }
      if ((opt->unit == 'r') && (d) && (d->firstpart))
      {
         rc += d->firstpart -1;                 // calculate absolute PID
      }
   }
   if (rc != FDSK_ANY)
   {
      if ((*pinfo = dfsGetPartInfo(rc)) == NULL)
      {
         rc = 0;                                // signal invalid part-spec
      }
   }
   else
   {
      *pinfo = dfsGetPartInfo( 1);              // return 1st part as start
   }                                            // (is a bit dangerous!)
   RETURN (rc);
}                                               // end 'dfsParsePartSpec'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Return valid part-type for specification;    hex | symbolic
/*****************************************************************************/
BYTE dfsParsePartType                           // RET   0=invalid or Ptype
(
   char               *spec,                    // IN    part type spec
   FDSK_CB_INFO       *cbi                      // IN    size & position info
)
{
   BYTE                rc = 0;                  // function return, invalid

   ENTER();
   TRARGS(("type='%s'\n", spec));
   if (spec[0] != '\0')                         // non empty
   {
      if ( ( isxdigit(spec[0])) &&              // first is HEX
          (( txHasMcsPrefix( spec)) ||          // 0x/0n/0t prefix
           ( strlen(spec) == 1)     ||          // single digit or 2 HEX
           ((strlen(spec) == 2) && (isxdigit(spec[1]))) ))
      {
         rc = (BYTE) TxaParseNumber( spec, DFSRADIX_TYPE, NULL);
      }
      else                                      // symbolic or invalid
      {
         if      (strncasecmp(spec, "FAT", 3 ) == 0)
         {
            double     size = 1000;             // default size in MB
            ULONG      lcyl = 1000;             // default last cylinder

            if (cbi != NULL)                    // size / position known
            {
               DFSPARTINFO *p = (DFSPARTINFO *) cbi->more;

               size = TXSMIB(  cbi->number, p->bpsector);
               lcyl = (cbi->sn + cbi->number -1) / p->cSC;
            }
            TRACES(( "FAT - size: %8.1lf MiB - last cyl: %lu\n", size, lcyl));

            if      (strcasecmp( spec, "FAT12"  ) == 0)  rc = DFS_P_FAT12;
            else if (strcasecmp( spec, "FAT16"  ) == 0)
            {
               rc = (size <  32)   ? DFS_P_FAT16 :
                    (lcyl >= 1024) ? DFS_P_FAT16X
                                   : DFS_P_BIGDOS;
            }
            else if (strcasecmp( spec, "FAT32"  ) == 0)
            {
               rc = (lcyl >= 1024) ? DFS_P_FAT32X : DFS_P_FAT32;
            }
            else                                // determine exact type ...
            {
               if (size >  2048)                // too large for 16-bit
               {
                  rc = (lcyl >= 1024) ? DFS_P_FAT32X : DFS_P_FAT32;
               }
               else if (lcyl >= 1024)           // must be 'extended'
               {
                  rc = DFS_P_FAT16X;
               }
               else                             // classic FAT16
               {
                  rc = (size < 16) ? DFS_P_FAT12 :
                       (size < 32) ? DFS_P_FAT16
                                   : DFS_P_BIGDOS;
               }
            }
         }
         else if (strcasecmp( spec, "BEOS"   ) == 0)  rc = DFS_P_BEOS_FS;
         else if (strcasecmp( spec, "BIGX"   ) == 0)  rc = DFS_P_BIGEXTEND;
         else if (strcasecmp( spec, "BJFS"   ) == 0)  rc = DFS_P_INST_FS;
         else if (strcasecmp( spec, "BMGR"   ) == 0)  rc = DFS_P_BOOTMGR;
         else if (strncasecmp(spec, "EXT", 3 ) == 0)  rc = DFS_P_LINUXNATV;
         else if (strcasecmp( spec, "CONT"   ) == 0)  rc = DFS_P_EXTENDED;
         else if (strcasecmp( spec, "GPT"    ) == 0)  rc = DFS_P_EFI_GPT;
         else if (strcasecmp( spec, "GUARD"  ) == 0)  rc = DFS_P_EFI_GPT;
         else if (strcasecmp( spec, "HPFS"   ) == 0)  rc = DFS_P_INST_FS;
         else if (strcasecmp( spec, "IFS"    ) == 0)  rc = DFS_P_INST_FS;
         else if (strcasecmp( spec, "JFS"    ) == 0)  rc = DFS_P_WARP_LVM;
         else if (strcasecmp( spec, "LINUX"  ) == 0)  rc = DFS_P_LINUXNATV;
         else if (strcasecmp( spec, "LVM"    ) == 0)  rc = DFS_P_WARP_LVM;
         else if (strcasecmp( spec, "NTFS"   ) == 0)  rc = DFS_P_INST_FS;
         else if (strncasecmp(spec, "REIS",4 ) == 0)  rc = DFS_P_LINUXNATV;
         else if (strcasecmp( spec, "SWAP"   ) == 0)  rc = DFS_P_SWAPSOLAR;
         else
         {
            rc = DFS_P_DFSEETYPE;
            if (cbi == NULL)                    // not 2nd parse on CREATE
            {
               TxNamedMessage( !dfsa->batch, 5025, " WARNING: Invalid symbolic type ",
                          "Symbolic partition-type '%s' is unknown, probably a mistake ...\n\n"
                          "Type 0xdf = DFSee-TMP will be used.", spec);
            }
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsParsePartType'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Return valid GPT part-type for specification: hex-UUID | symbolic | hex 2/4
/*****************************************************************************/
BOOL dfsParseGptType                            // RET   type is valid UUID now
(
   char               *spec                     // INOUT part type spec / UUID
)
{
   BYTE                rc = TRUE;               // function return, default valid
   char               *hdig = spec;             // start of actual hex-digits

   ENTER();
   TRARGS(("type IN='%s'\n", spec));
   if (spec[0] != '\0')                         // non empty
   {
      if ((hdig[0] == '0') && ((hdig[1] == 'x') || (hdig[1] == 'X')))
      {
         hdig += 2;                             // skip leading 0x or 0X prefix
      }
      if (dfsUidStringIsValid( spec) == FALSE)  // symbolic or invalid
      {
         if      ((strncasecmp( spec, "EFI",  3) == 0) ||
                  (strncasecmp( hdig, "ef",   2) == 0)  )
         {
            strcpy( spec, dfsFsUuidValueString( dfsGuid_EFI_SYSTEM ));
         }
         else if ((strncasecmp( spec, "BIOS", 3) == 0) ||
                  (strncasecmp( hdig, "bb",   2) == 0)  )
         {
            strcpy( spec, dfsFsUuidValueString( dfsGuid_BIOS_BOOT ));
         }
         else if ((strncasecmp( spec, "WIN",  3) == 0) ||
                  (strncasecmp( spec, "FAT",  3) == 0) ||
                  (strncasecmp( spec, "NTFS", 4) == 0) ||
                  (strncasecmp( spec, "EFAT", 4) == 0) ||
                  (strncasecmp( hdig, "07",   2) == 0)  )
         {
            strcpy( spec, dfsFsUuidValueString( dfsGuid_WINDOWS_BASIC_DATA ));
         }
         else if ((strncasecmp( spec, "REC",  3) == 0) ||
                  (strncasecmp( hdig, "27",   2) == 0)  )
         {
            strcpy( spec, dfsFsUuidValueString( dfsGuid_WINDOWS_RECOVERY ));
         }
         else if ((strncasecmp( spec, "AOT1", 4) == 0) ||
                  (strncasecmp( hdig, "0521", 4) == 0)  )
         {
            strcpy( spec, dfsFsUuidValueString( dfsGuid_OS2_ARCAOS_TYPE_1 ));
         }
         else if ((strncasecmp( spec, "SWAP", 4) == 0) ||
                  (strncasecmp( hdig, "82",   2) == 0)  )
         {
            strcpy( spec, dfsFsUuidValueString( dfsGuid_LINUX_SWAP ));
         }
         else if ((strncasecmp( spec, "LIN",  3) == 0) ||
                  (strncasecmp( spec, "EXT",  3) == 0) ||
                  (strncasecmp( hdig, "83",   2) == 0)  )
         {
            strcpy( spec, dfsFsUuidValueString( dfsGuid_LINUX_DATA ));
         }
         else if ((strncasecmp( spec, "HOME", 4) == 0) ||
                  (strncasecmp( hdig, "8302", 4) == 0)  )
         {
            strcpy( spec, dfsFsUuidValueString( dfsGuid_LINUX_HOME ));
         }
         else if ((strncasecmp( spec, "ROOT32", 6) == 0) ||
                  (strncasecmp( hdig, "8303",   4) == 0)  )
         {
            strcpy( spec, dfsFsUuidValueString( dfsGuid_LINUX_ROOT_X86_32 ));
         }
         else if ((strncasecmp( spec, "ROOT64", 6) == 0) ||
                  (strncasecmp( hdig, "8304",   4) == 0)  )
         {
            strcpy( spec, dfsFsUuidValueString( dfsGuid_LINUX_ROOT_X86_64 ));
         }
         else if ((strncasecmp( spec, "SRV",  3) == 0) ||
                  (strncasecmp( hdig, "8306", 4) == 0)  )
         {
            strcpy( spec, dfsFsUuidValueString( dfsGuid_LINUX_SRV_DATA ));
         }
         else if ((strncasecmp( spec, "DMCR", 4) == 0) ||
                  (strncasecmp( hdig, "8308", 4) == 0)  )
         {
            strcpy( spec, dfsFsUuidValueString( dfsGuid_LINUX_PLAIN_DMCRYPT ));
         }
         else if ((strncasecmp( spec, "LUKS", 4) == 0) ||
                  (strncasecmp( hdig, "8309", 4) == 0)  )
         {
            strcpy( spec, dfsFsUuidValueString( dfsGuid_LINUX_LUKS ));
         }
         else if ((strncasecmp( spec, "RAID", 4) == 0) ||
                  (strncasecmp( hdig, "fd",   2) == 0)  )
         {
            strcpy( spec, dfsFsUuidValueString( dfsGuid_LINUX_RAID ));
         }
         else if ((strncasecmp( spec, "APPL", 4) == 0) ||
                  (strncasecmp( hdig, "ab",   2) == 0)  )
         {
            strcpy( spec, dfsFsUuidValueString( dfsGuid_APPLE_BOOT ));
         }
         else if ((strncasecmp( spec, "APFS", 4) == 0) ||
                  (strncasecmp( hdig, "af73", 4) == 0) ||
                  (strncasecmp( hdig, "73",   2) == 0)  )
         {
            strcpy( spec, dfsFsUuidValueString( dfsGuid_APFS_CONTAINER ));
         }
         else if ((strncasecmp( spec, "HFS",  3) == 0) ||
                  (strncasecmp( hdig, "af",   2) == 0)  )
         {
            strcpy( spec, dfsFsUuidValueString( dfsGuid_MAC_OS_X_HFS_PLUS ));
         }
         else
         {
            rc = FALSE;
         }
      }
   }
   BRETURN (rc);
}                                               // end 'dfsParseGptType'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Parse a hex/dec location value, select (and align) matching freespace area
/*****************************************************************************/
DFSPARTINFO *dfsParseLocation                   // RET   freespace area to use
(
   char               *spec,                    // IN    location specification
   char               *spos,                    // IN    position specification
   char                fspKind,                 // IN    G(pt), P(rim), L(ogical)
   FDSK_NUMVAL        *size                     // IN    raw size value
)                                               // note: first char is default
{
   DFSPARTINFO        *rc = NULL;               // function return
   DFSPARTINFO        *f  = NULL;               // fsp  information
   DFSDISKINFO        *d  = NULL;               // disk information
   USHORT              fspace = 0;              // freespace id given
   USHORT              fspid  = 0;              // freespace id used
   USHORT              first  = 1;
   USHORT              disk;
   USHORT              disks  = 0;
   USHORT              disknr = 0;
   BOOL                align  = TRUE;           // align to cylinder/MiB boundaries

   ENTER();

   if (TxaOptUnSet( DFS_O_ALIGN) ||             // Explicit 'unaligned' request
       (size->type[0] == 's') ||                // Size specified in sectors
       (strstr( spos,   ",s") ))                // Start position in sectors
   {
      align = FALSE;
   }
   TRARGS(("loc:'%s'  pos:'%s'  size:%llx,%s  fspKind: %c  align: %s\n",
            spec, spos, size->value, size->type, fspKind, (align) ? "yes" : "no"));

   disks = dfsPartitionableDisks();             // nr of disks
   if (disks != 0)
   {
      switch (*spec)
      {
         case 0:                                // no spec ==> all disks
            break;

         case '@':                              // freespace id
            sscanf( spec +1, "%hu", &fspace);   // read nr if specified
            fspid = fspace;                     // use explicit id given
            break;

         default:                               // disk number
            sscanf( spec, "%hu", &disknr);      // read nr if specified
            TRACES(("disks:%hu  disknr:%hu\n", disks, disknr));
            switch (disknr)
            {
               case FDSK_ANY:                   // specific 'all disks'
                  break;

               case 0:                          // current disk only
                  first = SINF->disknr;
                  disks = SINF->disknr;
                  break;

               default:
                  if (disknr <= disks)          // valid specified disk
                  {
                     first = disknr;
                     disks = disknr;
                  }
                  else
                  {
                     TxPrint("Disk number %hu is not valid\n", disknr);
                     disks = 0;
                  }
                  break;
            }
            break;
      }
   }

   if (disks != 0)                              // disks to examine ?
   {
      TRACES(("first: %hu  disks: %hu  disknr: %hu  fspid: %hu\n",
               first,      disks,      disknr,      fspid));
      for (disk = first; (disk <= disks) && (rc == NULL); disk++)
      {
         TRACES(("disk: %hu  disks: %hu  disknr: %hu  fspid: %hu\n",
                  disk,      disks,      disknr,      fspid));
         if ((disknr == 0) || (disknr == disk))
         {
            d = dfsGetDiskInfo( disk);          // info for whole disk
            if (d != NULL)
            {
               ULN64   alBase = 0;              // aligned base PSN
               ULN64   alLast = 0;              // aligned last PSN
               ULN64   fsize  = 0;              // size of this freespace area
               ULN64   rsize  = 0;              // requested size in sectors

               if (fspace == 0)                 // no explict id given
               {
                  dfsPosition2Fspid( spos, fspKind, d, &fspid);
               }

               for (f = d->fspHead; f && (rc == NULL); f = f->fspChain)
               {
                  TRACES(("disk: %hu  f->id: %hu  type: %hx\n", disk, f->id, (USHORT) f->partent.PartitionType));
                  if ((fspid == 0) || (fspid == f->id))
                  {
                     ULONG   kmgAlign = dfsAlignmentSectors(fspKind, f);

                     fsize = f->sectors;
                     rsize = size->value;

                     if (size->type[0] == 'c')  // cylinder specification
                     {
                        rsize *= f->cSC;        // make it sectors
                     }

                     TRACES(("rsize: 0x%llx  fsize:0x%llx\n", rsize, fsize));

                     if (align)   // Calculate ALIGNED size (and start/end) for freespace
                     {
                        if ( (f->basePsn > (f->geoSecs + 1)) || // align if not 1st track (MBR)
                            ((f->basePsn) && (fspKind == 'G'))) // or non-zero fsp start  (GPT)
                        {
                           alBase = (((f->basePsn - 1) / kmgAlign) + 1) * kmgAlign;
                           if (alBase >= (f->basePsn + fsize))
                           {
                              alBase = f->basePsn; // aligned was beyond END!
                           }
                        }
                        else
                        {
                           alBase = f->basePsn;
                        }
                        if (f->lastPsn > kmgAlign)
                        {
                           alLast = (((f->lastPsn + 1) / kmgAlign) * kmgAlign) - 1;
                        }
                        else
                        {
                           alLast = 0;          // force 'can not be aligned'
                        }

                        if (alLast > alBase)    // avoid negative size on very small FSP
                        {
                           fsize = alLast - alBase + 1;
                        }
                        else
                        {
                           if ((fsize >= rsize) || (rsize == 0) || (fspid != 0))
                           {
                              align = FALSE;    // will qualify, but is NOT aligned
                              dfsSz64("Matched freespace : ",  fsize, " can not be aligned (too small)\n");
                           }
                        }
                        TRACES(("Aligned fsize: 0x%llx  Base: 0x%llx  Last: 0x%llx\n", fsize, alBase, alLast));
                     }

                     if ((fsize >= rsize) ||    // area large enough ?
                         (rsize == 0)     ||    // or whole area wanted
                         (fspid != 0)      )    // or specific area
                     {
                        switch (fspKind)
                        {
                           case 'G':            // GPT
                              switch (f->partent.PartitionType)
                              {
                                 case DFS_FSP_GPTS :
                                 case DFS_FSP_GPTC :
                                 case DFS_FSP_GPTF : rc = f; break;
                                 default           :         break;
                              }
                              break;

                           case 'P':            // we need a primary
                              switch (f->partent.PartitionType)
                              {
                                 case DFS_FSP_PRIM :
                                 case DFS_FSP_NEWX :
                                 case DFS_FSP_ENDC :
                                 case DFS_FSP_HEAD :
                                 case DFS_FSP_TAIL : rc = f; break;
                                 default           :         break;
                              }
                              break;

                           default:             // we need a logical
                              switch (f->partent.PartitionType)
                              {
                                 case DFS_FSP_LOG  :
                                 case DFS_FSP_ELOG :
                                 case DFS_FSP_HLOG :
                                 case DFS_FSP_TLOG :
                                 case DFS_FSP_NEWX :
                                 case DFS_FSP_ENDC :
                                 case DFS_FSP_HEAD :
                                 case DFS_FSP_TAIL : rc = f; break;
                                 default           :         break;
                              }
                              break;
                        }
                     }
                     else
                     {
                        TRACES(("fsp area %2u, %8.1lf MB is too small\n", f->id, TXSMIB(fsize, f->bpsector)));
                     }
                     if ((rc == NULL) && (fspid == f->id)) // specific area given
                     {
                        TxPrint("%s partition cannot be created in freespace area %u type %s\n",
                                (fspKind == 'G') ? "GPT" : (fspKind == 'P') ? "Primary" : "Logical", f->id, f->descr);
                     }
                  }
               }
               if ((rc != NULL) && (align))
               {
                  if (rc->basePsn < alBase)     // start was adjusted for alignment
                  {
                     ULN64 adjust = alBase - rc->basePsn;

                     dfsSz64Bps("Aligned start Fsp : ", adjust, rc->bpsector, "     from begin to start next");
                     TxPrint(" %s.\n", (fspKind == 'G') ? "MiB" : "Cyl");
                     rc->basePsn = alBase;
                  }
                  if (rc->lastPsn > alLast)     // end was adjusted for alignment
                  {
                     ULN64 adjust = rc->lastPsn - alLast;

                     dfsSz64Bps("Aligned  end  Fsp : ", adjust, rc->bpsector, "     from end-Fsp to end last");
                     TxPrint(" %s.\n", (fspKind == 'G') ? "MiB" : "Cyl");
                     rc->lastPsn = alLast;
                  }
                  rc->sectors = fsize;
               }
            }
         }
      }
      if ((rc == NULL) && (fspid == 0))         // no specific area given
      {
         TxPrint("%s partition cannot be created on specified disk(s), there\n"
                 "is insufficient, or not the right type of freespace available.\n",
                 (fspKind == 'G') ? "GPT" : (fspKind == 'P') ? "Primary" : "Logical");
         if (fspKind == 'G')
         {
            TxPrint( "Before creating a GPT partition, you first need to create a GPT guard partition\n"
                     "in the MBR on an (empty) disk, by using the 'cr gpt' command, or when using the\n"
                     "menu, select 'GPT-guard' from the create-partition dialog type selection list.\n");
         }
      }
   }
   if (rc)
   {
      TRARGS(("Disknr:%hu  id:%02.2u  Descr: %s Geo Cyl:%5u H:%3u S:%3u\n",
               rc->disknr, rc->id, rc->descr, rc->geoCyls, rc->geoHeads, rc->geoSecs));
   }
   RETURN (rc);
}                                               // end 'dfsParseLocation'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Parse a hex/dec position value, convert MB/Cyl etc to freespace-id (raw)
/*****************************************************************************/
void dfsPosition2Fspid
(
   char               *spec,                    // IN    specification string
   char                fspKind,                 // IN    G(pt), P(rim), L(ogical)
   DFSDISKINFO        *d,                       // IN    diskinfo (geo etc)
   USHORT             *fid                      // OUT   freespace-id
)
{
   ULN64               rv = 0;                  // resulting value
   ULN64               ov;                      // offset    value
   ULN64               limit;                   // limit for {} operations
   FDSK_NUMVAL         pos = FDSK_NUMZERO;
   FDSK_NUMVAL        *nv  = &pos;              // result value
   DFSPARTINFO        *f   = NULL;              // fsp  information
   BOOL                align  = TRUE;           // align to cylinder/MiB boundaries
   ULONG               kmgAlign;

   ENTER();

   if (TxaOptUnSet( DFS_O_ALIGN) ||             // Explicit 'unaligned' request
       (strstr( spec,   ",s") ))                // Size spec is in sectors
   {
      align = FALSE;
   }
   kmgAlign = dfsAlignmentSectors(fspKind, NULL);

   TRARGS(("Geo C:%4lu  H:%3lu  S:%3lu   cyl size:%8.8lx alignment:0x%lx  align: %s\n",
            d->geoCyls, d->geoHeads, d->geoSecs, d->cSC, kmgAlign, (align) ? "yes" : "no"));

   strcpy( pos.prefix,  "+-*@{}");
   strcpy( pos.format, "dxoDXO");
   dfsParseNumValue( spec, nv);                 // parse the spec
   if (d && (d->cSC != 0))
   {
      ov = nv->value;                           // raw value
      switch (nv->type[0])                      // resulting number type
      {
         case 's':                              // NEVER correct position
            break;                              // specified in exact sectors!

         case 'h': case 't':                    // Heads (or tracks :-)
            ov *= d->geoSecs;                   // multiply with sectors/track
            break;

         case 'c':                              // Cylinders
            ov *= d->cSC;                       // multiply with cyl-size
            break;

         case 'k':                              // Kilobytes
            ov  = DFSKB2SECTS( ov, d->bpsector);
            if ((ov != 0) && align)             // perform cylinder roundup
            {
               ov--;                            // subtract 1 before rounding
               ov /= kmgAlign;                  // aligned, rounded down
               ov += 1;                         // round up nr of units
               ov *= kmgAlign;                  // multiply with alignment
            }
            break;

         default:                               // Megabytes or Gigabytes
            ov  = DFSMB2SECTS( ov, d->bpsector);
            if (nv->type[0] == 'g')
            {
               ov *= 1024;                      // Giga correction
            }
            if ((ov != 0) && align)             // perform cylinder roundup
            {
               ov--;                            // subtract 1 before rounding
               ov /= kmgAlign;                  // aligned, rounded down
               ov += 1;                         // round up nr of units
               ov *= kmgAlign;                  // multiply with alignment
            }
            break;
      }
      limit = 1024 * d->cSC;                    // INT13 limit
      if (limit > d->sectors)                   // disk is smaller
      {
         limit  = d->sectors;                   // raw size, round down
         if (align)
         {
            limit /= kmgAlign;                  // Cyl/MiB, rounded down
            limit *= kmgAlign;                  // multiply with Cyl/MiB-size
         }
      }
      limit--;                                  // limit is last valid sn

      switch (nv->prefix[0])                    // given prefix character
      {
         case '@':                              // absolute position on disk
            rv = ov;
            break;

         case '{':                              // end just before 1024 limit
            rv = limit - ov;
            break;

         case '}':                              // start just after 1024 limit
            rv = limit + ov;
            break;

         default:                               // relative to freespace, will
            rv = DFS_MAX_PSN;                   // be handled by ParsePosition
            break;
      }
      for ( f  = d->fspHead, *fid  = 0;         // find freespace area (id)
           (f != NULL)  &&  (*fid == 0);        // with this raw position
            f  = f->fspChain)
      {
         if ((rv >= f->basePsn) && (rv <= f->lastPsn))
         {
            *fid = f->id;                       // found matching freespace
         }
      }
   }
   TRACES(("Calculated raw position: %llx, freespace-id: %hu\n", rv, *fid));
   VRETURN();
}                                               // end 'dfsPosition2Fspid'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Parse a hex/dec size value, convert MB/Cyl to cylinder-boundary sector-nr
/*****************************************************************************/
void dfsParseNewSize
(
   char               *spec,                    // IN    specification string
   char                fspKind,                 // IN    G(pt), P(rim), L(ogical)
   DFSPARTINFO        *f,                       // IN    freespace info (geo)
   FDSK_NUMVAL        *pos,                     // IN    RAW position value
   FDSK_NUMVAL        *nv                       // INOUT defaults/result value
)
{
   ULN64               rv;                      // resulting value
   ULONG               track1 = 0;              // wasted sectors on 1st cyl
   BOOL                align  = TRUE;           // align to cylinder/MiB boundaries
   ULONG               kmgAlign;

   ENTER();

   if (TxaOptUnSet( DFS_O_ALIGN) ||             // Explicit 'unaligned' request
       (nv->type[0] == 's') ||                  // Default size is in sectors
       (strstr( spec,   ",s") ))                // Size spec is in sectors
   {
      align = FALSE;
   }
   kmgAlign = dfsAlignmentSectors(fspKind, f);
   TRARGS(("size:'%s'  fspKind: %c  kmgAlign: 0x%lx  align: %s\n", spec, fspKind, kmgAlign, (align) ? "yes" : "no"));
   TRARGS(("pre:'%s'  fmt:'%s'  type: '%s'  value: %llu\n", nv->prefix, nv->format, nv->type, nv->value));

   dfsParseNumValue( spec, nv);                 // parse the spec
   rv = nv->value;                              // raw value
   switch (nv->type[0])                         // resulting number type
   {
      case 's':                                 // NEVER try to correct exact
         break;                                 // size specified in sectors!

      case 'h': case 't':                       // Heads (or tracks :-)
         if (f != NULL)
         {
            rv *= f->geoSecs;                   // multiply with sectors/track
         }
         rv += track1;                          // include needed waste
         break;

      case 'c':                                 // Cylinders
         if (f != NULL)
         {
            rv *= f->cSC;                       // multiply with cyl-size
         }
         break;

      case 'k':                                 // Kilobytes
         rv  = DFSKB2SECTS( rv, (f) ? f->bpsector : dfsGetSectorSize());
         rv += track1;                          // include needed waste
         if (align)                             // align desired, roundup
         {
            if (rv != 0)                        // perform cylinder roundup
            {
               rv--;                            // subtract 1 before rounding
               rv /= kmgAlign;                  // aligned, rounded down
               rv += 1;                         // round up nr of units
               rv *= kmgAlign;                  // multiply with alignment
            }
         }
         break;

      default:                                  // Megabytes or Gigabytes
         rv  = DFSMB2SECTS( rv, (f) ? f->bpsector : dfsGetSectorSize());
         if (nv->type[0] == 'g')
         {
            rv *= 1024;                         // Giga correction
         }
         rv += track1;                          // include needed waste
         if (align)                             // align desired, roundup
         {
            if (rv != 0)                        // perform cylinder roundup
            {
               rv--;                            // subtract 1 before rounding
               rv /= kmgAlign;                  // aligned, rounded down
               rv += 1;                         // round up nr of units
               rv *= kmgAlign;                  // multiply with alignment
            }
         }
         break;
   }
   if (f == NULL)                               // no geo info
   {
      TxPrint("Specified    size : ");
      if (nv->type[0] == 'c')                   // cylinders
      {
         TxPrint("%llu Cyl\n", rv);
      }
      else
      {
         if (rv == 0)                           // no size given
         {
            TxPrint( "%29snone, use maximum\n", "");
         }
         else
         {
            dfsSz64Bps( "", rv, dfsGetSectorSize(), "\n");
         }
      }
   }
   else
   {
      TxPrint("Aligned PART size : ");
      if (rv == 0)                              // no size given
      {
         TxPrint( "%29snone, use maximum\n", "");
      }
      else
      {
         dfsSz64Bps( "", rv, f->bpsector, "\n");
      }
   }
   nv->value = rv;                              // update value structure
   VRETURN();
}                                               // end 'dfsParseNewSize'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Parse a hex/dec position value, convert MB/Cyl to cyl-boundary sector-nr
// For a logical MBR-style, the EBR location is returned in f->partPsn
/*****************************************************************************/
ULONG dfsParsePosition
(
   char               *spec,                    // IN    specification string
   char                fspKind,                 // IN    G(pt), P(rim), L(ogical)
   ULONG               ptGap,                   // IN    gap to ptable, sectors
   DFSPARTINFO        *f,                       // INOUT freespace info (geo)
   FDSK_NUMVAL        *size,                    // INOUT resolved size value
   FDSK_NUMVAL        *nv                       // INOUT defaults/result value
)
{
   ULONG               rc = NO_ERROR;           // function return
   ULN64               rv = 0;                  // resulting value
   ULN64               ov;                      // offset    value
   ULN64               limit;                   // limit for {} operations
   DFSDISKINFO        *d;                       // disk info
   TXTS                object;                  // small variable text object
   ULONG               track1 = 0;              // wasted sectors on 1st cyl
   ULN64               fsize;                   // size of freespace area
   ULN64               adjust;                  // adjust  value
   BOOL                align  = TRUE;           // align to cylinder/MiB boundaries
   ULONG               kmgAlign;

   ENTER();

   if (TxaOptUnSet( DFS_O_ALIGN) ||             // Explicit 'unaligned' request
       (nv->type[0] == 's') ||                  // Default pos is in sectors
       (strstr( spec,   ",s") ))                // Pos spec is in sectors
   {
      align = FALSE;
   }
   kmgAlign = dfsAlignmentSectors(fspKind, f);
   TRARGS(("pos:'%s'  fspKind: %c  ptGap: 0x%8.8lx  kmgAlign: 0x%lx  align: %s\n",
            spec,     fspKind,     ptGap,           kmgAlign, (align) ? "yes" : "no"));
   TRARGS(("pre:'%s'  fmt:'%s'  type: '%s'  value: %llx\n", nv->prefix, nv->format, nv->type, nv->value));
   TRARGS(("Geo C:%4lu  H:%3lu  S:%3lu   cyl size:%8.8lx\n", f->geoCyls, f->geoHeads, f->geoSecs, f->cSC));

   dfsParseNumValue( spec, nv);                 // parse the spec
   if (kmgAlign != 0)                           // avoid divide by zero
   {
      ov = nv->value;                           // raw value to start with
      switch (nv->type[0])                      // resulting number type
      {
         case 's':                              // NET sectors
            break;

         case 'h': case 't':                    // Heads (or tracks :-)
            ov *= f->geoSecs;                   // multiply with sectors/track
            break;

         case 'c':                              // Cylinders
            ov *= f->cSC;                       // multiply with cyl-size
            break;

         case 'k':                              // Kilobytes
            ov  = DFSKB2SECTS( ov, f->bpsector);
            if ((ov != 0) && align)             // perform cylinder roundup
            {
               ov--;                            // subtract 1 before rounding
               ov /= kmgAlign;                  // aligned, rounded down
               ov += 1;                         // round up nr of units
               ov *= kmgAlign;                  // multiply with alignment
            }
            break;

         default:                               // Megabytes or Gigabytes
            ov  = DFSMB2SECTS( ov, f->bpsector);
            if (nv->type[0] == 'g')
            {
               ov *= 1024;                      // Giga correction
            }
            if ((ov != 0) && align)             // perform cylinder roundup
            {
               ov--;                            // subtract 1 before rounding
               ov /= kmgAlign;                  // aligned, rounded down
               ov += 1;                         // round up nr of units
               ov *= kmgAlign;                  // multiply with alignment
            }
            break;
      }
      d = dfsGetDiskInfo( f->disknr);           // info for whole disk
      limit = 1024 * f->cSC;
      if (limit > d->sectors)                   // disk is smaller
      {
         limit  = d->sectors;                   // raw size, round to cyls
         if (align)
         {
            limit /= kmgAlign;                  // Cyl/MiB, rounded down
            limit *= kmgAlign;                  // multiply with Cyl/MiB-size
         }
         strcpy( object, "End-of-disk");
      }
      else
      {
         strcpy( object, "Cylinder-1024");
      }
      limit--;                                  // limit is last valid sn
      fsize = f->sectors;                       // size of freespace area

      TRACES(("base:%llx last:%llx fsize:%llx ov:%llx size:%llx\n",
               f->basePsn,  f->lastPsn, fsize, ov, size->value));
      TxPrint("New part position : ");
      switch (nv->prefix[0])                    // given prefix character
      {
         case '-':                              // relative to end freespace
            if ((f->lastPsn+1 - size->value - ov) >= f->basePsn)
            {
               if (size->value == 0)            // use maximum area
               {
                  size->value = fsize - ov;
               }
               rv = f->lastPsn+1 - size->value - ov - f->basePsn;
               dfsSz64Bps( "", ov, f->bpsector, " before END freespace\n");
            }
            else
            {
               dfsSz64Bps( "", ov, f->bpsector, "     ERROR:\n");
               TxPrint("Negative position too large for selected size and freespace area\n");
               rc = DFS_VALUE_ERROR;
            }
            break;

         case '@':                              // absolute position on disk
            if ((ov >= f->basePsn) && (ov <= f->lastPsn))
            {
               rv = ov - f->basePsn;
               if (size->value == 0)            // use maximum area
               {
                  size->value = fsize - rv;
               }
               dfsSz64Bps(                     "", rv, f->bpsector, "     from beginning of freespace\n");
               dfsSz64Bps( "                    ", ov, f->bpsector, "     from start of disk\n");
            }
            else
            {
               dfsSz64Bps( "", ov, f->bpsector, "     ERROR:\n");
               TxPrint("Partition on absolute position does not fit in selected freespace area\n");
               rc = DFS_VALUE_ERROR;
            }
            break;

         case '{':                              // end just before 1024 limit
            if (((limit - ov - size->value) >= f->basePsn)  &&
                ((limit - ov              ) <= f->lastPsn +1))
            {
               rv = limit - ov - size->value - f->basePsn +1;
               if (size->value == 0)            // use maximum area
               {
                  size->value = rv;
                  rv = 0;
               }
               dfsSz64Bps( "", ov, f->bpsector, "     before %s\n");
            }
            else
            {
               dfsSz64Bps( "", ov, f->bpsector, "     ERROR:\n");
               TxPrint("%s - position is not inside selected freespace area\n", object);
               rc = DFS_VALUE_ERROR;
            }
            break;

         case '}':                              // start just after 1024 limit
            if (((limit + ov              ) >= f->basePsn) &&
                ((limit + ov + size->value) <= f->lastPsn)  )
            {
               rv = limit + ov - f->basePsn +1;
               if (size->value == 0)            // use maximum area
               {
                  size->value = fsize - ov;
               }
               dfsSz64Bps( "", ov, f->bpsector, "     after %s\n");
            }
            else
            {
               dfsSz64Bps( "", ov, f->bpsector, "     ERROR:\n");
               TxPrint("%s + position is not inside selected freespace area\n", object);
               rc = DFS_VALUE_ERROR;
            }
            break;

         default:                               // relative to begin freespace
            if ((f->basePsn + ov + size->value) <= f->lastPsn +1)
            {
               if (size->value == 0)            // use maximum area
               {
                  size->value = fsize - ov;
               }
               rv = ov;
               dfsSz64Bps( "", ov, f->bpsector, "     after BEGIN freespace\n");
            }
            else
            {
               dfsSz64Bps( "", ov, f->bpsector, "     ERROR:\n");
               TxPrint("Specified partition does not fit in selected freespace area\n");
               rc = DFS_VALUE_ERROR;
            }
            break;
      }

      if (fspKind != 'G')                       // no track1 issue with GPT
      {
         //- calculate track1 AFTER knowing exactly where the partition will be!
         TRACES(("pre:  rv:%llx  size:%llx  base:%llx  track1:%8.8lx\n",
                  rv, size->value, f->basePsn, track1));
         if (fspKind == 'P')                    // set part-table sector-nr
         {
            f->partPsn = 0;                     // partition table in MBR
            if ((f->basePsn + rv) < f->geoSecs) // start inside 1st track
            {                                   // round-up to start 2nd track
               if (align)                       // when alignment wanted
               {
                  if ((f->basePsn == 0) &&
                      (ptGap < f->geoSecs))     // -G specified for 1st primary
                  {
                     track1 = ptGap;            // first track holds MBR
                  }
                  else
                  {
                     track1 = f->geoSecs - f->basePsn - rv;
                  }
               }
               else if (rv == 0)                // just make enough room for MBR
               {
                  track1 = 1;
               }
            }
         }
         else                                   // logical, don't use cyl-0
         {
            TRACES(("CYL1 fix LOG, base:%llx  rv:%llx  cSC:%8.8lx\n", f->basePsn, rv, f->cSC));

            if ((f->basePsn + rv) < f->cSC)     // within 1st cylinder ?
            {
               if (align)
               {
                  adjust = f->cSC - f->basePsn -rv; // adjust to start of cyl-1
                  rv          += adjust;        // adjust relative position
                  size->value -= adjust;        // adjust NET size too
                  TRACES(("adjust:%llx  rv:%llx  value:%llx  fsize:%llx\n", adjust, rv, size->value, fsize));

                  if ((rv + size->value + f->cSC) <= fsize) // if enough freespace
                  {                             // add exactly one cylinder
                     size->value += f->cSC;     // to maintain the minimum
                  }                             // specified size
               }
               else if (rv == 0)                // just make enough room for EBR
               {
                  rv          += 1;             // EBR directly after MBR, no gap
                  size->value -= 1;
               }
            }
            f->partPsn = f->basePsn + rv;       // partition table EBR sector
            track1     = ptGap;                 // first track holds EBR
         }
         rv           += track1;                // adjust relative position
         size->value  -= track1;                // adjust NET size
         dfsSz64Bps( "Calculated   size : ", size->value, f->bpsector, "\n");
      }
      TRACES(("post: rv:%llx  size:%llx  base:%llx  track1:%8.8lx\n", rv, size->value, f->basePsn, track1));
   }
   nv->value = rv;                              // update value structure
   RETURN (rc);
}                                               // end 'dfsParsePosition'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Parse a hex/dec size value from an option, convert MB/GB/KB/cyl/track to sec
/*****************************************************************************/
BOOL dfsOptionSizeSectors                       // RET   option value OK
(
   char                option,                  // IN    option char to use
   DFST_HANDLE         store,                   // IN    store handle for geo
   char                defUnit,                 // IN    default unit s,m etc
   ULN64               defSn,                   // IN    default value
   ULN64              *sectors                  // OUT   nr of sectors
)
{
   BOOL                rc = TRUE;               // function return
   ULN64               sn = defSn;              // resulting value
   TXA_OPTION         *opt;                     // option pointer

   ENTER();
   TRACES(("Option: '%c' store: %lu default: %llx\n", option, store, defSn));

   if ((opt = TxaOptValue( option)) != NULL)    // get the option details
   {
      switch (opt->type)
      {
         case TXA_STRING:                       // error unless this is "find"
            if (option != 'f' || strcmp(opt->value.string, "Current"))
            {
               TxPrint( "\n* Need a NUMBER value for the '-%c' option, using default!", option);
            }
         case TXA_NO_VAL:
            rc = FALSE;
            break;

         default:                               // convert, default is MiB!
            sn = dfsApplyNumberUnit( opt->value.number,
                                    (opt->unit == TXA_DFUNIT) ? defUnit : opt->unit,
                                     store);
            break;
      }
   }
   TRACES(("Resulting size in sectors: 0x%llx\n", sn));
   *sectors = sn;
   RETURN (rc);
}                                               // end 'dfsOptionSizeSectors'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Parse a hex/dec size value, convert MB/sec to cyl using specified head/sect
/*****************************************************************************/
void dfsParseCylinderSize
(
   char               *spec,                    // IN    specification string
   ULONG               hds,                     // IN    Geo, nr of heads
   ULONG               spt,                     // IN    Geo, nr of sectors
   ULONG               bps,                     // IN    Geo, bytes per sector
   FDSK_NUMVAL        *nv                       // INOUT defaults/result value
)
{
   ULN64               rv;                      // resulting value
   ULONG               cSC = hds * spt;         // sectors per cylinder

   ENTER();
   TRARGS(("size:'%s'  using %hu heads  %hu sectors and %hu BpSec\n", spec, hds, spt, bps));
   TRARGS(("prefix:'%s' fmt:'%s'  type: '%s'  value: %llu\n", nv->prefix, nv->format, nv->type, nv->value));

   dfsParseNumValue( spec, nv);                 // parse the spec
   rv = nv->value;                              // raw value
   switch (nv->type[0])                         // resulting number type
   {
      case 'h': case 't':                       // Heads (or tracks :-)
         rv *= spt;                             // multiply with sectors/track
      case 's':                                 // sectors, round to nearest cyl
         rv = ((rv -1) / cSC) +1;               // round to nearest cyl
         break;

      case 'c':                                 // Cylinders, value is OK
         break;

      case 'k':                                 // Kilobytes
         rv = DFSKB2SECTS( rv, bps);            // convert to sectors first
         rv = ((rv -1) / cSC) +1;               // round to nearest cyl
         break;

      default:
         rv = DFSMB2SECTS( rv, bps);            // convert to sectors first
         if (nv->type[0] == 'g')
         {
            rv *= 1024;                         // Giga correction
         }
         rv = ((rv -1) / cSC) +1;               // round to nearest cyl
         break;
   }
   nv->value = rv;                              // resulting cylinder value
   VRETURN();
}                                               // end 'dfsParseCylinderSize'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Parse a hex/dec numeric value with leading options and trailing fmt/type
// Number syntax: [leading-options]number[,[h|d][type-indicator]]
/****************************************************************************/
void dfsParseNumValue
(
   char               *spin,                    // IN    specification string
   FDSK_NUMVAL        *nv                       // INOUT defaults/result value
)                                               // note: first char is default
{
   TXTM                spec;                    // local copy of spec
   char               *s  = spec;
   char               *ft;                      // optional format/type
   char                numbase = nv->format[0]; // format from postfix, or prefix!

   ENTER();

   TRARGS(("spin:'%s'  pre:'%s'  fmt:'%s'  type: '%s'  value: %lu\n",
            spin, nv->prefix, nv->format, nv->type, nv->value));

   strcpy( spec, spin);                         // make modifiable copy

   while ((*s) && (!isxdigit(*s)))              // prefix given ?
   {
      s++;                                      // advance to start of number
   }
   if (s != spec)                               // prefix given
   {
      memset(  nv->prefix, 0,    TXMAXTS);
      strncpy( nv->prefix, spec, max((size_t)(TXMAXTS-1), (size_t) (s - spec)));
   }
   else                                         // no prefix
   {
      strcpy( nv->prefix, "");
   }
   if ((ft = strchr( spec, ',')) != NULL)       // specified format/type
   {
      *ft = '\0';                               // terminate the numeric part
      ft++;                                     // advance to format
      if (strchr(nv->format, *ft))              // valid format spec
      {
         numbase = *ft;                         // base INDICATOR from postfix
         ft++;                                  // advance to type string
      }
      if (strlen(ft))                           // specified ?
      {
         memset(  nv->type, 0,  TXMAXTS);
         strncpy( nv->type, ft, TXMAXTS-1);
      }
   }
   nv->format[1] = '\0';

   //- Allow a specific HEX or OCTAL prefix on these numbers as well (overrides FORMAT TYPE!)
   if ((*s == '0') && (((*(s+1)) == 'x') || ((*(s+1)) == 'X') || // base 16 (HEX)
                       ((*(s+1)) == 'o') || ((*(s+1)) == 'O') || // base  8 (OCT)
                       ((*(s+1)) == 't') || ((*(s+1)) == 'T') )) // base 10 (DEC)
   {
      numbase = *(s+1);                         // BASE indicator from prefix
      if ((numbase == 't') || (numbase == 'T')) // decimal prefix 0t or 0T
      {
         numbase = 'd';                         // use standard decimal format specifier
      }
      s += 2;                                   // skip prefix
   }
   nv->format[0] = numbase;                     // might be used by caller!

   TRACES(("prefix: '%s'  type: '%s'  digits: '%s'  numbase: '%c'\n", nv->prefix, nv->type, s, numbase));

   if (strlen(s))                               // number part present ?
   {
      switch (numbase)
      {
         case 'x':
         case 'X':
            sscanf( s, "%llx", &nv->value);     // hexadecimal
            break;

         case 'o':
         case 'O':
            sscanf( s, "%llo", &nv->value);     // octal
            break;

         default:                               // decimal, unsigned
            sscanf( s, "%llu", &nv->value);
            break;
      }
   }
   else
   {
      nv->value = 0;
   }
   TRARGS(("OUT type: '%s'  value: %llu = 0x%llx\n", nv->type, nv->value, nv->value));
   VRETURN();
}                                               // end 'dfsParseNumValue'
/*---------------------------------------------------------------------------*/

