//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// FAT utility functions
//
// Author: J. van Wijk
//
// JvW  22-11-97   Initial version, split off from DFSAFAT
//
#ifndef    DFSUFAT
   #define DFSUFAT

#include <dfsfat.h>                             // FAT structure definitions
#include <dfsafat.h>                            // FAT cache definitions

#define DFS_FAT_INVALIDS ":;\"|\\=?*<>,+[]/"

// Display file allocation and path info for LSN
ULONG dfsFatFileInfo                            // RET   result
(
   ULN64               lsn,                     // IN    Directory LSN
   ULN64               entry,                   // IN    directory entry
   char               *select,                  // IN    File select wildcard
   void               *lead                     // IN    leading text
);


// Replace sn-list by contents of a single Directory (DirLsn, entry pairs)
ULONG dfsFatMakeBrowseList                      // RET   result
(
   ULN64               lsn,                     // IN    Directory LSN
   ULN64               info,                    // IN    directory entry
   char               *str,                     // IN    unused
   void               *param                    // INOUT unused
);

// Get allocation information for file-DATA into integrated-SPACE structure
// 20161215: Include EA-data in separate allocation (in the ispace)
ULONG dfsFatGetAllocSpace
(
   ULN64               lsn,                     // IN    Directory LSN
   ULN64               info,                    // IN    directory entry
   char               *str,                     // IN    unused
   void               *param                    // INOUT Integrated SPACE
);

// Get long or short (8.3) name for given LSN and entry number for DIR sector
ULONG dfsFatLsnInfo2Name                        // RET   result
(
   ULN64               lsn,                     // IN    Directory LSN
   ULONG               info,                    // IN    directory entry
   char               *name                     // OUT   directory entry name
);


// Find full PATH to Root-directory, starting at some dir LSN and entry info
ULONG dfsFatLsnInfo2Path                        // RET   result und
(
   ULN64               start,                   // IN    starting lsn
   ULONG               entry,                   // IN    directory entry info
   char               *path                     // OUT   combined path string
);

// Determine type of whole directory cluster and cache it for later reuse
BYTE dfsFatDirClusterType
(
   ULN64               lsn                      // IN    LSN of a (DIR) sector
);

// Determine allocation-bit for specified LSN
ULONG dfsFatAllocated                           // RET   LSN is allocated
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);

// Set allocate status for LSN to specified value
ULONG dfsFatSetAlloc                            // RET   LSN allocation set
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *value,                   // IN    NULL = not allocated
   void               *ref                      // IN    dummy
);

// Translate EA-index to relative sector-number
ULN64 dfsFatEaIndex2Rsn                         // RET   RSN for this EA index
(
   USHORT              index                    // IN    EA index number
);

// Determine exact size in bytes, for an OS/2 style EA (FAT16 only!)
ULONG dfsFatEAByteSize                          // RET EA size in bytes
(
   ULN64               lsn                      // IN    1st lsn of EA cluster
);

#define dfsFatClust2LSN(cl) (ULN64) dfsFatCl2Lsn((cl),0,NULL,NULL)
// Translate Cluster-nr to LSN
ULN64 dfsFatCl2Lsn                              // RET   LSN for this cluster
(
   ULN64               cluster,                 // IN    cluster number
   ULN64               d2,                      // IN    dummy
   char               *p1,                      // IN    dummy
   void               *p2                       // IN    dummy
);

#define dfsFatLSN2Clust(sn) dfsFatLsn2Cl((ULN64)(sn),0,NULL, NULL)
// Translate LSN to Cluster-nr
ULN64 dfsFatLsn2Cl                              // RET   cluster for this LSN
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *p1,                      // IN    dummy
   void               *p2                       // IN    dummy
);

// Get cluster-value from a Directory entry structure
// Note: May return non-zero on a FAT32 dir-entry with an LFN!
ULONG dfsFatDir2Clust                           // RET   cluster value
(
   S_FATDIR           *entry                    // IN    FAT directory entry
);

// Set cluster-value in a Directory entry structure
void dfsClust2FatDir
(
   ULONG               cluster,                 // IN    cluster value to set
   S_FATDIR           *entry                    // INOUT FAT directory entry
);

// Get cluster-value from the Fat Cache structure (32 bit entries)
ULONG dfsFatValue                               // RET   cluster value
(
   ULONG               clust                    // IN    cluster to get value
);

// Set cluster-value in the Fat Cache structure (32 bit entries)
ULONG dfsFatSetCluster
(
   ULONG               clust,                   // IN    cluster to get value
   ULONG               value                    // IN    FAT-value to be set
);

// Reset bad-sector admin, change each BAD cluster in the FAT to FREE
ULONG dfsFatResetBadClus                        // RET   ref cluster or 0
(
   void
);

// Search reference to specified cluster in FAT (cluster pointing to this one)
// Note: Dont use regular cache to avoid trashing that just by searching here
ULONG dfsFatFindReference                       // RET   ref cluster or 0
(
   ULONG               cluster                  // IN    target cluster
);

// Check if cluster could be the 1st root-DIR sector (DIR but no '..' entry)
BOOL dfsFatRootCandidate                        // RET   Could be a root DIR
(
   ULONG               cluster                  // IN    target cluster
);

// Calulate 32-bit CRC over FAT-contents (to compare FAT1 and FAT2 perhaps :-)
ULONG dfsFatCalculateFatCRC                     // RET   32-bit CRC
(
   ULONG               fatlsn                   // IN    first sector of FAT
);

// Set the active FAT LSN to be used for fat references using the cache
void dfsFatSetFatInUse
(
   ULONG               fat2Use                  // IN    first sector of FAT
);

// Copy FAT-part from disk to the DFS FAT cache structure (32 bit entries)
ULONG dfsFatGetFAT                              // RET
(
   ULONG               clust,                   // IN    cluster value to get
   DFSFATCACHE        *cache                    // INOUT cache structure used
);

// Flush specified FAT cache (32 bit entries) back to disk FatInUse (+ other)
ULONG dfsFatFlushFAT                            // RET   result
(
   DFSFATCACHE        *cache                    // INOUT cache structure to write
);

// Show logical FAT
void dfsFatShowFAT
(
   ULONG               size,                    // IN    nr of entries
   ULONG               start,                   // IN    first cl# to show
   char               *options                  // IN    display options
);

// Count number of used Dir-entries for S_SPACE structure
ULONG dfsFatDirUsed                             // RET   matching entries
(
   ULONG               chunks,                  // IN    nr of space entries
   S_SPACE            *space                    // IN    space allocation
);

// Get label-string from the FAT Root-directory
BOOL dfsFatRootLabel                            // RET   Label found
(
   S_BOOTR            *boot,                    // IN    boot record for vol
   ULN64               offset,                  // IN    Offset for boot-rec
   char               *label                    // OUT   label string
);

// Set the alternate label location, in ROOT-directory, using specified label
ULONG dfsFatSetAltBrecLabel
(
   ULN64               v1,                      // IN    unused
   ULN64               v2,                      // IN    unused
   char               *str,                     // IN    new label value (11)
   void               *param                    // IN    unused
);

// Create S_SPACE structure for a FAT allocation chain
ULONG dfsFatAllocChain                          // RET   nr of clusters
(
   ULONG               cluster,                 // IN    starting cluster nr
   ULONG               maxsize,                 // IN    max file-size, or 0
   ULONG              *errors,                  // INOUT error flags (or NULL)
   ULONG              *chunks,                  // OUT   nr of space entries
   S_SPACE           **space                    // OUT   space allocation
);

// Check if name is a valid FAT filename
BOOL dfsValidFatName
(
   const void         *fn,                      // IN    ptr to FAT name
   USHORT              len                      // IN    length to check
);

// Display all available info about a directory entry
ULONG dfsFatShowDirectoryEntry
(
   ULN64               dirsect,                 // IN    lsn of FAT dir sector
   int                 entry                    // IN    index of dir entry
);

// FAT filesystem, display DIR header/footer
void dfsFatDirHeader
(
   char               *lead,                    // IN    lead text-string
   ULONG               items                    // IN    DIR-lines shown (0 = TOP)
);

// Add any EA data info for a FAT16 EA-index to std FEA2-list structure
ULONG dfsFatReadEaIndex
(
   USHORT              eaIndex,                 // IN    FAT EA-index
   ULONG              *size,                    // OUT   size in bytes
   S_FEA2LIST        **pfeal                    // OUT   list of FEA2's
);

// Determine complete EA-data allocation from an LSN and display it
ULONG dfsFatEAdataDisplay
(
   ULN64               lsn                      // IN    1st lsn of EA cluster
);

// Read one cluster EA-data from an LSN and display it
ULONG dfsFatDisplayEA
(
   S_EACLUST          *ea,                      // IN    EA data cluster(s)
   USHORT              size                     // IN    buffer size
);

// Read one cluster EA-data from an LSN and display it
ULONG dfsFatDispEAindex
(
   S_EAINDEX          *eai,                     // IN    EA index cluster(s)
   USHORT              size                     // IN    buffer size
);

// Display FAT directory entry on a single line
void dfsFatShowDirEntry
(
   S_FATDIR           *entry,                   // IN    FAT directory entry
   BYTE                bg                       // IN    current background
);

// Display VFAT part of directory entry on a single line
void dfsFatShowVfdEntry
(
   S_FATDIR           *fe,                      // IN    FAT directory entry
   char               *lfn,                     // IN    lfn string
   BYTE                bg                       // IN    current background
);

// Get de long-filename (LFN) belonging to the specified (short) FAT dir entry
ULONG dfsFatExtractLfn                          // RET   LFN length found, or 0
(
   BYTE               *buf,                     // IN    start of buffer (check)
   S_FATDIR           *de,                      // IN    directory entry (short)
   char               *name                     // OUT   long filename or ""
);

// Insert another UNICODEd lfn-slot in front of the ASCII lfn-string
void dfsFatAddLfnSlot
(
   S_VFSLOT           *entry,                   // IN    VFAT lfn slot
   char               *name                     // INOUT LFN string
);

// Display FAT allocation chain from cluster
void dfsFatShowAlloc
(
   ULONG               cluster                  // IN    Cluster number
);

// Convert null-terminated name to a valid directory-entry name (8.3)
void dfsFatName2Entry
(
   char               *fn,                      // IN    ptr to FAT name
   S_FATDIR           *de                       // OUT   directory entry
);

// Convert a valid directory-entry name (8.3) to a null-terminated name
void dfsFatEntry2Name
(
   S_FATDIR           *de,                      // IN    directory entry
   char               *fn                       // OUT   ptr to FAT name
);

// Retrieve first-cluster number for specified Dir-sector-LSN + index (LsnInf)
ULONG dfsLsnInf2Cluster                         // RET   First cluster or 0
(
   ULN64               lsn,                     // IN    Directory LSN
   ULONG               info                     // IN    directory entry
);

// Retrieve filesize for specified Dir-sector-LSN + index (LsnInf)
ULONG dfsLsnInf2FileSize                        // RET   First cluster or 0
(
   ULN64               lsn,                     // IN    Directory LSN
   ULONG               info                     // IN    directory entry
);

// retrieve directory-entry for specified Dir-sector-LSN + index (LsnInf)
ULONG dfsLsnInf2Entry
(
   ULN64               lsn,                     // IN    Directory LSN
   ULONG               info,                    // IN    directory entry
   S_FATDIR           *direntry                 // OUT   directory entry
);

// Create SPACE allocation structure from a Directory LSN (root or other)
ULONG dfsDirLsn2Space
(
   ULN64               lsn,                     // IN    lsn for DIR sector
   ULONG              *nr,                      // OUT   nr of space entries
   S_SPACE           **sp                       // OUT   space allocation
);

// Iterate over directory-structures finding files, and execute callback
ULONG dfsFatIterator
(
   ULN64               recurse,                 // IN    recurse subtrees
   ULN64               type,                    // IN    do 'D', 'f' or both
   char               *path,                    // IN    path specification
   void               *p                        // IN    iterate parameters
);

// Create valid FAT(32) bootrecord based on fat-> info from FatInit
ULONG dfsFatMkBootRec
(
   char                osFat16,                 // IN    OS preference FAT16
   BOOL                verbose,                 // IN    verbose and confirm
   USHORT              rootEntries,             // IN    root-entries or 0
   ULONG               serialNr,                // IN    Vol-serialNr or 0
   char               *label                    // IN    Label or ""  or NULL
);

// Get/Set DIRTY bits in FAT16/32 bootsector to specified value
BYTE dfsFatDirtyStatusBootrec                   // RET   final dirty value
(                                               //       or RC when > 2
   BOOL                update,                  // IN    update value
   BYTE                dirty                    // IN    new value: 0, 1, 2
);

// Get/Set DIRTY bits in FAT16/32 FAT-area entry 1 to specified value
BYTE dfsFatDirtyStatusFatarea                   // RET   final dirty value
(                                               //       or RC when > 2
   BOOL                update,                  // IN    update value
   BYTE                dirty                    // IN    new value: 0, 1, 2
);

// Read allocation for ALL directory sectors into a single SPACE structure
ULONG dfsFatAllDirs2Space
(
   BOOL                progressSectors,         // IN    progress in sectors, not DIRs
   ULONG              *nr,                      // OUT   nr of space entries
   S_SPACE           **sp                       // OUT   space allocation
);

// Update EVERY cluster reference in each directory entry in SPACE for resize
ULONG dfsFatFixDirs4Resize
(
   ULONG               eFsCl,                   // IN    extra clusters each FAT
   ULONG               ext,                     // IN    nr of space entries
   S_SPACE            *spc                      // IN    ALL-DIR space allocation
);

// Update FAT area for resize, COPY moved-part, fixup references, clear rest
ULONG dfsFatFixRfat4Resize
(
   ULONG               eFsCl                    // IN    extra clusters each FAT
);

// Get updated cluster-value for intended resize operation based on shifting
ULONG dfsFatResizedValue                        // RET   updated cluster value
(
   ULONG               clust,                   // IN    current cluster value
   ULONG               efsc                     // IN    extra FAT clusters/FAT
);

#endif
