//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// FAT utility functions
//
// Author: J. van Wijk
//
// JvW  15-08-2001 Added first-free-LSN display to alloc (truncate)
// JvW  12-05-2000 Removed obsolete dfsFatUndeleteSNtable function
// JvW  04-12-1997 Initial version
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsver.h>                             // DFS version info
#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfstable.h>                           // SLT utility functions

#include <dfsupart.h>                           // FDISK partition functions
#include <dfsafdsk.h>                           // FDISK display & analysis
#include <dfscfdsk.h>                           // FDISK util, FAT sectors

#include <dfsafat.h>                            // FAT analysis functions
#include <dfsufat.h>                            // FAT utility functions
#include <dfslfat.h>                            // FAT slt & error definitions
#include <dfsbfat.h>                            // FAT bootsector functions
#include <dfsos2ea.h>                           // HPFS/JFS/FAT EA handling


// Put contents of a Directory SSPACE (DirLsn, entry pairs) into sn-list
static ULONG dfsFatDirSpace2List
(
   DFSISPACE          *dirData                  // IN    Directory clusters
);

// Recursively iterate over FAT directory and files, execute callback
static ULONG dfsFatIterFiles
(
   ULN64               rs,                      // IN    recurse subtrees
   ULONG               type,                    // IN    do 'D', 'f' or both
   ULONG               chunks,                  // IN    nr of S_SPACE chunks
   S_SPACE            *space,                   // IN    space structure array
   DFS_PARAMS         *cp                       // IN    callback params
);

// Add SPACE for specified directory to the accumulating total directory SPACE
static ULONG dfsFatAddDir2Space
(
   ULN64               sn,                      // IN    sector-nr DIR sector
   ULN64               info,                    // IN    coded directory entry#
   char               *df,                      // IN    dummy, optional info
   void               *data                     // INOUT callback data, SPACE
);


//- relative age of last read/write action to a cache structure
static ULONG        lastCacheStamp = 0;      // last age value issued

//- progress counter for FatIterator (is not always the sector-list!)
static ULONG        iterationsDone = 0;

/*****************************************************************************/
// Display file allocation and path info for LSN
/*****************************************************************************/
ULONG dfsFatFileInfo                            // RET   result
(
   ULN64               lsn,                     // IN    Directory LSN
   ULN64               info,                    // IN    directory entry
   char               *select,                  // IN    File select wildcard
   void               *param                    // INOUT leading text/shortpath
)
{
   ULONG               rc = NO_ERROR;
   BYTE               *sb = NULL;               // sector buffer
   S_FATDIR           *fe;                      // Fat directory entry
   BYTE                st = 0;                  // dir-entry type, default none
   ULN64               dirsect = lsn;
   USHORT              entry   = info & 0x0f;
   TXLN                shortpath;
   TXLN                text;
   TXTT                temp;                    // location text
   ULONG               percent   = 0;
   ULONG               threshold = 0;           // threshold percentage
   BOOL                isMinimum = TRUE;        // threshold is minimum value
   ULONG               size      = 0;           // nr of allocated sectors
   ULONG               location  = 0;           // location value; LSN/Cluster
   ULONG               minS      = 0;           // minimal size
   ULONG               maxS      = 0;           // maximal size
   char                itemType  = 0;           // type Dir, File, Browse, or ANY
   time_t              modTstamp = 0;           // Modify timestamp value, or 0
   char               *lead    = param;
   BOOL                verbose = (strcmp( lead, ":") != 0);
   BOOL                output;                  // no Printf, just return fpath
   BOOL                alcheck;                 // Execute a full allocation check
   ULONG               eaSize  = 0;             // size of OS/2 EA (FAT16)

   ENTER();
   TRARGS(("LSN:0x%llX, info:0x%2.2x select:'%s', lead:'%s'\n", lsn, info, select, lead));

   if (strcmp( lead, "!") == 0)                 // Skip allocation check wanted
   {
      output  = FALSE;                          // no output
      alcheck = FALSE;                          // and no allocation check
      *lead   = 0;                              // discard 'garbage' in lead string
   }
   else                                         // Possible output/alloc-check wanted
   {
      output  = (*lead != 0);                   // output if any lead given (except '!')
      alcheck = TRUE;                           // but always an alloc check if no  '!'
   }
   TRACES(("verbose:%u output:%u alcheck:%u\n", verbose, output, alcheck));

   if (info & DFSSNINFO)
   {
      if ((sb = TxAlloc(1, dfsGetSectorSize())) != NULL)
      {
         if ((rc = dfsRead(dirsect, 1, sb)) == NO_ERROR)
         {
            switch (dfsIdentifySector(dirsect, 0, sb))
            {
               case ST_ROOTD:
               case ST_SUBDR:
               case ST_SUBRT:
               case ST_DIREC:
                  fe = &(((S_FATDIR *) sb)[entry]);
                  st = (fe->FatAttrib & FATTR_DIRECTORY) ? 'D' : 'f';

                  dfsParseFileSelection( select, text, &isMinimum, &threshold, &minS, &maxS, &modTstamp, &itemType);
                  switch (st)
                  {
                     case 'D':
                        if      (itemType == DFS_FS_ITEM_BROWSE) // no filtering on Directories, take all.
                        {
                           strcpy( text, "");   // no wildcard
                           isMinimum = TRUE; threshold = 0; // no percentage filter
                           modTstamp = 0;       // no timestamp filter
                           minS      = 0;    maxS      = 0; // no size filter
                        }
                        else if (itemType == DFS_FS_ITEM_FILES) // filter files, discard Directories
                        {
                           rc = DFS_ST_MISMATCH; // item-type mismatch
                           break;
                        }
                     case 'f':
                     default:
                        if      (itemType == DFS_FS_ITEM_DIRS) // filter dirs, discard files
                        {
                           rc = DFS_ST_MISMATCH; // item-type mismatch
                           break;
                        }
                        if ( (fe->FatAttrib != VFAT_ATTRIB) && (fe->FatAttrib & FATTR_LABEL))
                        {
                           st   = 'L';
                           if (output)
                           {
                              sprintf( shortpath, "%sLabel:%s ", CNN, CBG);
                           }
                           else
                           {
                              strcpy( shortpath, "");
                           }
                           dfsFatEntry2Name( fe, shortpath + strlen(shortpath));
                        }
                        else                    // non-label, (long) filename
                        {
                           //- Find path upto ROOT when SLT is available, or bare name otherwise
                           dfsFatLsnInfo2Path( lsn, info, shortpath);
                        }
                        if ((strlen(text) == 0) || (TxStrWicmp(shortpath, text) >= 0))
                        {
                           if      (fe->FatAttrib & FATTR_DIRECTORY)
                           {
                              size = fat->ClustSize;
                              location = dfsFatDir2Clust( fe);
                              if (location == 0) // ROOT!
                              {
                                 strcpy( shortpath, FS_PATH_STR);
                                 percent = 100; // fake recoverability
                              }
                              else              // append PATH_SEP to subdirectories
                              {
                                 if (shortpath[strlen(shortpath) -1] != '.') // not on . and ..
                                 {
                                    strcat( shortpath, FS_PATH_STR);
                                 }
                              }
                           }
                           else                 // regular file
                           {
                              if (fe->fsize == 0) // empty
                              {
                                 size = 0;
                              }
                              else
                              {
                                 size = ((fe->fsize -1) / dfsGetSectorSize()) +1;
                              }
                              location = dfsFatDir2Clust( fe);
                           }
                           if ((verbose) && (location >= 2))
                           {
                              //- Note: Modelled after EFAT, but DOES not work on FAT(32) because
                              //-       alocation and allocated sectors can NOT be determined
                              //-       as soon as the FAT-chain is damaged or cleared in delete
                              //-       As a minimum, size should be taken from directory and NOT
                              //-       from the damaged FAT-chain (S_SPACE)

                              DFSISPACE       is;
                              ULONG           ef   = 0; // error flags (not used here)
                              ULN64           size = 0; // allocated size in sectors
                              ULN64           bads = 0; // sectors with bad allocation

                              if (dfsFatAllocChain( location, 0, &ef, &is.chunks, &is.space) != 0)
                              {
                                 is.clsize = fat->ClustSize;
                                 size      = dfsSspaceSectors( TRUE, is.chunks, is.space);

                                 if (alcheck)
                                 {
                                    #if defined (USEWINDOWING)
                                       if (size > DFSECT_BIGF)
                                       {
                                          txwSetSbviewStatus( "Checking allocation for a huge file ...", cSchemeColor);
                                       }
                                    #endif
                                    dfsCheckAllocForSpace( &is, 0, (fe->Name[0] != FAT_DIRDEL), &bads);
                                    percent = dfsAllocationReliability( size, bads);
                                 }
                                 else
                                 {
                                    isMinimum = TRUE; threshold = 0; // no percentage filter
                                 }
                                 TxFreeMem( is.space); // free the SPACE
                              }
                              TRACES(( "percent:%u threshold:%u size:%u minS:%u maxS:%u\n",
                                        percent, threshold, size, minS, maxS));
                           }
                           if ((BOOL)((percent >= threshold)) == isMinimum)
                           {
                              time_t  modifiedTime = txOS2FileTime2t( &fe->date.u, &fe->time.u);

                              if ((modifiedTime > modTstamp) &&                        //- Timestamp match
                                  ((size >= minS) && ((size <= maxS) || (maxS == 0)))) //- Size match
                              {
                                 if (!TxaOption('X') || (fe->OS2EA != 0))
                                 {
                                    if (verbose) // include alloc and size
                                    {
                                       strcpy( temp, "");
                                       if (TxaOption('I')) // use EA-index if present
                                       {
                                          if (fe->OS2EA != 0)
                                          {
                                             if (TxaOption('C')) // display as relative-cluster
                                             {
                                                location = dfsFatEaIndex2Rsn(fe->OS2EA) / fat->ClustSize;
                                                sprintf( temp, "%8X", location);
                                             }
                                             else
                                             {
                                                location = (ULONG) fe->OS2EA;
                                                sprintf( temp, "EA:%7u", location);
                                             }
                                          }
                                       }
                                       else     // LSN/Cluster displays
                                       {
                                          if (TxaOption('e')) // EA LSN/Cluster if present
                                          {
                                             if (fe->OS2EA != 0)
                                             {
                                                location = dfsSspaceRsn2Lsn( fat->EaData.Chunks, fat->EaData.Space,
                                                              dfsFatEaIndex2Rsn(fe->OS2EA));
                                                if (TxaOption('C'))
                                                {
                                                   location = dfsFatLSN2Clust( location);
                                                }
                                             }
                                          }
                                          else
                                          {
                                             location = dfsFatDir2Clust( fe);
                                             if (!TxaOption('C') && (location != 0))
                                             {
                                                location = dfsFatClust2LSN( location);
                                             }
                                          }
                                          if (location != 0)
                                          {
                                             sprintf( temp, "%8X", location);
                                          }
                                       }
                                       if (alcheck)
                                       {
                                          sprintf(   text, "%s%c %s%3u%% %s%10.10s ", CBM, st,
                                                     (rc == NO_ERROR) ?  CBG : CBR, percent, CNN, temp);
                                       }
                                       else     // no reliability percentage
                                       {
                                          sprintf(   text, "%s%c      %s%10.10s ", CBM, st, CNN, temp);
                                       }
                                       dfstrSize( text, CBC, size, " ");

                                       if ((fe->OS2EA) && (fat->FatBits != 32))
                                       {
                                          eaSize = dfsFatEAByteSize(
                                                      dfsSspaceRsn2Lsn( fat->EaData.Chunks, fat->EaData.Space,
                                                         dfsFatEaIndex2Rsn( fe->OS2EA)));
                                       }
                                       dfstrBytes(text, CBZ, eaSize, " "); // XATTR/EA-size
                                    }
                                    else
                                    {
                                       strcpy( text, ""); // remove any stray text, non-verbose
                                    }
                                    if (output) // not totally silent?
                                    {
                                       strcat(    lead, text);
                                       TxPrint( "%s%s%s%s%s\n", lead, CBY, shortpath, CNN, CGE);
                                    }
                                    else        // return info in 'select'
                                    {
                                       sprintf( temp, "%4.4u-%2.2u-%2.2u %2.2u:%2.2u:%2.2u",
                                                fe->date.d.year + 1980, fe->date.d.month, fe->date.d.day,
                                                fe->time.t.hours, fe->time.t.minutes, fe->time.t.twosecs * 2);

                                       TxStripAnsiCodes( text);
                                       sprintf( select, "%s %s", text, temp);
                                       dfstrUllDot20( select, "", fe->fsize, "");
                                    }
                                    rc = NO_ERROR;
                                 }
                                 else           // skip files without EAs
                                 {
                                    TRACES(("OS2EA mismatch\n"));
                                    rc = DFS_ST_MISMATCH; // OS2EA mismatch
                                 }
                              }
                              else
                              {
                                 TRACES(("file-size mismatch\n"));
                                 rc = DFS_ST_MISMATCH; // file-size mismatch
                              }
                           }
                           else
                           {
                              TRACES(("percentage mismatch\n"));
                              rc = DFS_ST_MISMATCH; // percentage mismatch
                           }
                        }
                        else
                        {
                           rc = DFS_ST_MISMATCH; // wildcard mismatch
                        }
                        strcpy(  param, shortpath); // return the found name
                        break;
                  }
                  break;

               default:
                  rc = DFS_PENDING;
                  break;
            }
         }
         else if (dirsect < fat->Sect)          // should be there
         {
            dfsX10("\nError reading sector : ", dirsect, CBR, "\n");
         }
         TxFreeMem(sb);
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   else
   {
      rc = DFS_ST_MISMATCH;                     // info is NOT a valid entrynr
   }
   RETURN(rc);
}                                               // end 'dfsFatFileInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Replace sn-list by contents of a single Directory (DirLsn, entry pairs)
/*****************************************************************************/
ULONG dfsFatMakeBrowseList                      // RET   result
(
   ULN64               lsn,                     // IN    Directory LSN
   ULN64               info,                    // IN    dir entry, 0 for ROOT
   char               *str,                     // IN    unused
   void               *param                    // INOUT unused
)
{
   ULONG               rc = NO_ERROR;
   BYTE               *sb = NULL;               // sector buffer
   S_FATDIR           *fe;                      // Fat directory entry
   ULN64               dirsect = lsn;
   USHORT              entry   = info & 0x7f;
   ULONG               location  = 0;           // location value; LSN/Cluster
   DFSISPACE           is;
   ULONG               ef;                      // error flags (not used here)

   ENTER();
   TRARGS(("LSN:0x%llX, info:0x%2.2x\n", lsn, info));

   if (info & DFSSNINFO)                        // valid entry, get the DIR sector
   {
      if ((sb = TxAlloc(1, dfsGetSectorSize())) != NULL)
      {
         if ((rc = dfsRead(dirsect, 1, sb)) == NO_ERROR)
         {
            switch (dfsIdentifySector(dirsect, 0, sb))
            {
               case ST_ROOTD:
               case ST_SUBDR:
               case ST_SUBRT:
               case ST_DIREC:
                  rc = DFS_BAD_STRUCTURE;       // error, unless proven otherwise
                  fe = &(((S_FATDIR *) sb)[entry]);
                  if      (fe->FatAttrib & FATTR_DIRECTORY)
                  {
                     if ((location = dfsFatDir2Clust( fe)) >= 2)
                     {
                        if (dfsFatAllocChain( location, 0, &ef, &is.chunks, &is.space) != 0)
                        {
                           rc = NO_ERROR;
                        }
                        //- to be refined, assume single cluster allocation
                     }
                     else                       // .. pointing to ROOT!
                     {
                        rc = dfsDirLsn2Space( fat->Root, &is.chunks, &is.space);
                     }
                  }
                  break;

               default:
                  rc = DFS_ST_MISMATCH;         // not a directory sector
                  break;
            }
         }
         TxFreeMem(sb);
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   else                                         // it is the ROOT itself
   {
      rc = dfsDirLsn2Space( fat->Root, &is.chunks, &is.space);
   }
   if ((rc == NO_ERROR) && (is.space))
   {
      rc = dfsFatDirSpace2List( &is);
      free( is.space);                          // free the SPACE (calloc/realloc)
   }
   RETURN(rc);
}                                               // end 'dfsFatMakeBrowseList'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get allocation information for file-DATA into integrated-SPACE structure
// 20161215: Include EA-data in separate allocation (in the ispace)
/*****************************************************************************/
ULONG dfsFatGetAllocSpace
(
   ULN64               lsn,                     // IN    Directory LSN
   ULN64               info,                    // IN    directory entry
   char               *str,                     // IN    unused
   void               *param                    // INOUT Integrated SPACE
)
{
   ULONG               rc = NO_ERROR;
   DFSISPACE          *ispace = (DFSISPACE *) param;
   BYTE               *sb = NULL;               // sector buffer
   S_FATDIR           *fe;                      // Fat directory entry
   USHORT              entry   = info & 0x7f;
   ULONG               location  = 0;           // location value; LSN/Cluster
   ULONG               ef;                      // error flags (not used here)

   ENTER();

   ispace->space = NULL;

   if (info & DFSSNINFO)                        // valid entry, get the DIR sector
   {
      if ((sb = TxAlloc(1, dfsGetSectorSize())) != NULL)
      {
         if ((rc = dfsRead( lsn, 1, sb)) == NO_ERROR)
         {
            switch (dfsIdentifySector( lsn, 0, sb))
            {
               case ST_ROOTD:
               case ST_SUBDR:
               case ST_SUBRT:
               case ST_DIREC:
                  rc = DFS_BAD_STRUCTURE;       // error, unless proven otherwise
                  fe = &(((S_FATDIR *) sb)[entry]);
                  if ((location = dfsFatDir2Clust( fe)) >= 2)
                  {
                     if (dfsFatAllocChain( location, 0, &ef, &ispace->chunks, &ispace->space) != 0)
                     {
                        rc = NO_ERROR;
                     }
                  }
                  else                          // .. pointing to ROOT!
                  {
                     rc = dfsDirLsn2Space( fat->Root, &ispace->chunks, &ispace->space);
                  }
                  if ((rc == NO_ERROR) && (fe->OS2EA) && (fat->FatBits != 32))
                  {
                     rc = dfsFatReadEaIndex(  fe->OS2EA, &ispace->xasize, (S_FEA2LIST **) &ispace->xattrs);
                     if ((rc == NO_ERROR) && (ispace->xattrs != NULL))
                     {
                        ispace->xatype = DFSXA_FEA2;
                     }
                  }
                  break;

               default:
                  rc = DFS_ST_MISMATCH;         // not a directory sector
                  break;
            }
         }
         TxFreeMem(sb);
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   else                                         // it is the ROOT itself
   {
      rc = dfsDirLsn2Space( fat->Root, &ispace->chunks, &ispace->space);
   }
   RETURN (rc);
}                                               // end 'dfsFatGetAllocSpace'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Put contents of a Directory SSPACE (DirLsn, entry pairs) into sn-list
/*****************************************************************************/
static ULONG dfsFatDirSpace2List
(
   DFSISPACE          *dirData                  // IN    Directory clusters
)
{
   ULONG               rc = NO_ERROR;
   USHORT              bps = dfsGetSectorSize();
   USHORT              entries = (bps / sizeof(S_FATDIR));
   BYTE               *dirbuf;
   S_FATDIR           *fatdir;                  // Fat directory sector
   S_VFSLOT           *fl;                      // ptr to LFN entry struct
   USHORT              entry;                   // entry index in sector
   ULONG               chunk;                   // dir space chunk
   ULONG               sect;                    // relative sector number
   ULONG               dlsn;                    // sector number

   ENTER();

   dirbuf = TxAlloc( 1, bps);                   // allocate one sector
   if ((dirbuf != NULL) && (dirData->space != NULL))
   {
      dfsInitList(0, "-f -P", "-d");            // optimal for menu file-recovery
      fatdir = (S_FATDIR *) dirbuf;
      for ( chunk = 0;
           (chunk < dirData->chunks) && (rc == NO_ERROR);
            chunk++)                            // walk all alloc chunks
      {
         for ( sect = 0;
              (sect < dirData->space[chunk].size) && (rc == NO_ERROR);
               sect++)                          // each sector in chunk
         {
            dlsn = dirData->space[chunk].start + sect;
            rc = dfsRead( dlsn, 1, dirbuf);
            switch (dfsIdentifySector( dlsn, 0, dirbuf))
            {
               case ST_ROOTD:
               case ST_SUBDR:
               case ST_SUBRT:
               case ST_DIREC:
                  for ( entry = 0;
                       (entry < entries) && (rc == NO_ERROR);
                        entry++)                // each dir-entry in sector
                  {
                     if ((fatdir[entry].FatAttrib & FATTR_LABEL) == 0) // skip label
                     {
                        switch (fatdir[entry].Name[0])
                        {
                           case FAT_DIRFREE:    // free entry
                           case FAT_DIRDEL:     // deleted entry
                              break;

                           case FAT_DIRDOT:     // . or .. entry
                              if (fatdir[entry].Name[1] != FAT_DIRDOT) // it is . not ..
                              {
                                 break;         // falltrough for ..
                              }
                              else              // .. directory, set PARENT flag
                              {
                                 if (dfsa->snlist[0] == 0)
                                 {
                                    TRACES(("List marked as 1ST_PARENT\n"));
                                    DFSBR_SnlistFlag |= DFSBR_1ST_PARENT;
                                 }
                              }
                           default:             // used, but could be lfn
                              fl = (S_VFSLOT *) &(fatdir[entry]);
                              if ((fl->FatAttrib == VFAT_ATTRIB) &&
                                  (fl->clust     == VFAT_CLUSTR) &&
                                  (fl->Vzero     == 0))
                              {
                                 TRACES(("Entry :%5u Lfn slotId: 0x%2.2hx\n", entry, fl->SlotId));
                              }
                              else              // regular DIR entry for file or directory
                              {
                                 BOOL          filtered = FALSE;

                                 if (dfsa->browseShowHidden == FALSE) // need to check hidden/system attribute
                                 {
                                    if ((fatdir[entry].FatAttrib & FATTR_SYSTEM) ||
                                        (fatdir[entry].FatAttrib & FATTR_HIDDEN) ||
                                        (fatdir[entry].Name[0] == '.'))
                                    {
                                       filtered = TRUE;
                                    }
                                 }
                                 if (!filtered)
                                 {
                                    rc = dfsAddSI2List( dlsn, entry);
                                 }
                              }
                              break;
                        }
                     }
                  }
                  break;

               default:
                  break;
            }
         }
      }
      TxFreeMem( dirbuf);
   }
   RETURN (rc);
}                                               // end 'dfsFatDirSpace2List'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get long or short (8.3) name for given LSN and entry number for DIR sector
/*****************************************************************************/
ULONG dfsFatLsnInfo2Name                        // RET   result
(
   ULN64               lsn,                     // IN    Directory LSN
   ULONG               info,                    // IN    directory entry info
   char               *name                     // OUT   directory entry name
)
{
   ULONG               rc = NO_ERROR;

   ENTER();
   TRARGS(("LSN:0x%llX, info:0x%2.2x\n", lsn, info));

   if (info & DFSSNINFO)
   {
      BYTE            *sb = NULL;            // sector buffer
      S_FATDIR        *fe;                   // Fat directory entry
      S_VFSLOT        *fl;                   // Fat LFN slot  entry (alias)
      ULN64            dirsect = lsn;
      USHORT           entry   = info & 0x0f;
      USHORT           bps = dfsGetSectorSize();
      USHORT           entries = (bps / sizeof(S_FATDIR)); // entries per sector
      USHORT           i;

      //- Need to allocate/read previous and next sector to always allow reading complete LFN
      if ((sb = TxAlloc( 4, bps)) != NULL)
      {
         if ((rc = dfsRead(dirsect -2, 4, sb)) == NO_ERROR)
         {
            switch (dfsIdentifySector(dirsect, 0, sb + (2 * bps)))
            {
               case ST_ROOTD:
               case ST_SUBDR:
               case ST_SUBRT:
               case ST_DIREC:
                  fl = &(((S_VFSLOT *) sb)[32 + entry]); // could be lfn or SHORT entry, need SHORT!
                  for (i = 0; i < entries; i++) // sync to (next) SHORT entry, at most over 1 sector
                  {
                     if ((fl->FatAttrib == VFAT_ATTRIB) && (fl->clust == VFAT_CLUSTR) && (fl->Vzero == 0))
                     {
                        TRACES(("Entry is an LFN slot, try next one: %hu\n", ++entry));
                        fl++;                   // go to the NEXT entry, as long as it is an LFN slot
                     }
                     else
                     {
                        fe = (S_FATDIR *) fl;   // we found the corresponding SHORT name dir slot
                     }
                  }
                  if ((TxaOptUnSet( DFS_O_LONGFN)) ||         //- if -lfn- option used
                      (dfsFatExtractLfn( sb, fe, name) == 0)) //- or no lfn present
                  {
                     dfsFatEntry2Name( fe, name); // use the short-name directly
                  }
                  break;

               default:
                  strcpy( name, "");
                  rc = DFS_ST_MISMATCH;
                  break;
            }
         }
         TxFreeMem(sb);
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   else
   {
      rc = DFS_ST_MISMATCH;                     // info is NOT a valid entrynr
   }
   TRACES(("Name: '%s'\n", name));
   RETURN(rc);
}                                               // end 'dfsFatLsnInfo2Name'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Find full PATH to Root-directory, starting at some dir LSN and entry info
/*****************************************************************************/
ULONG dfsFatLsnInfo2Path                        // RET   result und
(
   ULN64               start,                   // IN    starting lsn
   ULONG               entry,                   // IN    directory entry info
   char               *path                     // OUT   combined path string
)
{
   ULONG               rc = NO_ERROR;
   ULN64               ref = start;             // Reference/Parent LSN
   USHORT              info = entry;            // entry info from SLT
   BYTE                st;
   ULONG               sanity = 20;             // maximum iterations
   ULONG               sltIndex;                // dummy SLT location
   TX1K                fname;                   // shortname (+ path)

   ENTER();
   if (path)
   {
      *path = '\0';
   }
   rc = dfsFatLsnInfo2Name( start, info, path); // get name for THIS dir entry

   if (rc == NO_ERROR)
   {
      if (dfsSlTableStatus(NULL) == SLT_READY) // there is an SLT
      {
         do
         {
            if (dfsSlTableFind( ref, &ref, &info, &st, &sltIndex))   //- Parent, next REF
            {
               if (ref == LSN_BOOTR)            // Root-DIR refrences to boot
               {
                  strcpy( fname, FS_PATH_STR);  // ROOT
                  strcat( fname, path);         // append existing path
                  strcpy( path, fname);         // and copy back
                  break;
               }
               else
               {
                  if (strcmp( path, "..") == 0) // entry is the parent, SKIP and remove ".."
                  {
                     strcpy(  path, "");
                  }
                  else
                  {
                     if ((rc = dfsFatLsnInfo2Name( ref, info | DFSSNINFO, fname)) == NO_ERROR)
                     {
                        TRACES(("'%s' + '%s'\n", fname, path));
                        if (path[0] != 0)       // when existing path already
                        {
                           strcat( fname, FS_PATH_STR); // add separator
                        }
                        strcat( fname, path);   // append existing path
                        strcpy( path, fname);   // and copy back
                     }
                  }
               }
            }
            else
            {
               rc = DFS_NOT_FOUND;
            }
            sanity--;
         } while ((rc == NO_ERROR) && (sanity));
      }
      else
      {
         rc = DFS_CMD_WARNING;
      }
   }
   TRACES(("Found path to root for 0x%llX+%4.4x:'%s'\n", start, entry, path));

   RETURN(rc);
}                                               // end 'dfsFatLsnInfo2Path'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Determine type of whole directory cluster and cache it for later reuse
/*****************************************************************************/
BYTE dfsFatDirClusterType                       // RET   DIR type or UDATA
(
   ULN64               lsn                      // IN    LSN of a (DIR) sector
)
{
   BYTE                rc = ST_UDATA;           // function return
   BYTE               *sb = NULL;               // sector buffer
   static ULN64        csFirst = 0;             // first in cached range
   static ULN64        csLast  = 0;             // last
   static BYTE         csType  = ST_UDATA;      // the associated type info
   ULN64               dirsect = csFirst;       // first sector of DIR cluster

   ENTER();
   TRACES(("lsn:0x%llx csF:0x%llx csL:0x%llx csT:%c\n", lsn, csFirst, csLast, csType));

   if ((lsn >= fat->ClTwo) && (lsn < fat->Sect)) // possible (sub) directory
   {
      if ((lsn >= csFirst) && (lsn <= csLast))  // in cached range ?
      {
         rc = csType;                           // use cached value
      }
      else                                      // determine and cache new type
      {
         if ((sb = TxAlloc( fat->ClustSize, dfsGetSectorSize())) != NULL)
         {
            dirsect = dfsFatClust2LSN( dfsFatLSN2Clust( lsn));
            if (dfsRead( dirsect, fat->ClustSize, sb) == NO_ERROR)
            {
               S_VFSLOT  *vf;                   // entry for LFN fragment
               S_FATDIR  *sf;                   // entry, with short name
               S_FATDIR  *fatdir = (S_FATDIR *) sb;
               BOOL       fdir   = TRUE;        // possible FAT directory
               ULONG      entry;                // index in fat directory
               S_FATDIR   refdent;              // reference DIR-entry
               ULONG      labels = 0;           // number of labels (max 1!)
               USHORT     fatEntries;           // entries per sector

               fatEntries = (dfsGetSectorSize() * fat->ClustSize) / sizeof(S_FATDIR);
               for (entry = 0; (entry < fatEntries) && fdir; entry++)
               {
                  sf = &(fatdir[entry]);        // ptr to current entry

                  TRACEX(("de:%3u cl:%8.8X fsize:%u name[0]: %2.2X, name: '%*.*s'\n",
                     entry, dfsFatDir2Clust(sf), sf->fsize,
                      sf->Name[0], FAT_N, FAT_N, sf->Name));

                  switch (sf->Name[0])
                  {
                     case FAT_DIRFREE:
                        memset(     &refdent,  FAT_DIRFREE, sizeof(S_FATDIR));
                        if (memcmp( sf, &refdent, sizeof(S_FATDIR)) != 0)
                        {
                           fdir = FALSE;
                           TRACEX(("dir-entry failed on non-zero fields in entry: %u\n", entry));
                        }
                        break;

                     case FAT_DIRDEL:           // deleted always OK (may be zeroed!)
                        break;

                     default:                   // now check for LFN first
                        vf = (S_VFSLOT *) sf;

                        if ( (vf->FatAttrib   == VFAT_ATTRIB) &&
                             (vf->clust       == VFAT_CLUSTR) &&
                             (vf->Vzero       == 0          ) &&
                           (((vf->SlotId >  0x00) && (vf->SlotId <= 0x14)) ||
                            ((vf->SlotId >  0x40) && (vf->SlotId <= 0x54)) ||
                             (vf->SlotId == 0xe5)                        )  )
                        {
                           TRACEX(("Valid VFAT LFName slot at entry: %u\n", entry));
                        }
                        else
                        {
                           memset( &refdent, sf->Name[0], sizeof(S_FATDIR));
                           if (memcmp( sf, &refdent, sizeof(S_FATDIR)) != 0)
                           {
                              if ((dfsFatDir2Clust( sf) <= fat->MapClust) &&
                                  (dfsValidFatName( sf->Name, FAT_NSIZE + FAT_ESIZE)))
                              {
                                 if ((sf->date.u == 0)         || //- no zero date (1-1-1980)
                                     (sf->date.u > MAX_S_DATE) || //- allow valid date and
                                     (sf->time.u > MAX_S_TIME)  ) //- valid time only
                                 {
                                    TRACEX(("dir-entry failed on invalid date/time, entry: %u\n", entry));
                                    fdir = FALSE;
                                 }
                                 else           // looks valid ...
                                 {
                                    if ((sf->FatAttrib & FATTR_LABEL) && (++labels))
                                    {
                                       TRACEX(("dir-entry failed on multiple-labels, entry: %u\n", entry));
                                       fdir = FALSE;
                                    }
                                    else
                                    {
                                       TRACEX(("Valid FAT standard slot at entry: %u\n", entry));
                                    }
                                 }
                              }
                              else              // Invalid clusternr or name
                              {
                                 TRACEX(("dir-entry failed on invalid cluster or name in entry: %u\n", entry));
                                 fdir = FALSE;
                              }
                           }
                           else                 // whole entry same byte
                           {                    // must be an invalid one!
                              fdir = FALSE;
                              TRACEX(("dir-entry failed on all-same-bytes in entry: %u\n", entry));
                           }
                        }
                        break;
                  }
               }
               if (fdir)                        // valid DIR cluster
               {
                  if ((memcmp( fatdir[0].Name, ".          ", FAT_NSIZE + FAT_ESIZE) == 0) &&
                      (memcmp( fatdir[1].Name, "..         ", FAT_NSIZE + FAT_ESIZE) == 0)  )
                  {
                     if (dfsFatDir2Clust(&fatdir[1]) == 0)
                     {
                        rc = ST_SUBRT;          // first sector of cluster
                     }                          // is subdirectory from ROOT
                     else
                     {
                        rc = ST_SUBDR;          // first sector of cluster
                     }                          // is 1st of a subdirectory
                  }
                  else
                  {
                     rc = ST_DIREC;             // any kind of dir-sector
                  }
               }
               csType  = rc;                    // update type cache
               csFirst = dirsect;
               csLast  = dirsect + fat->ClustSize -1;
            }
            TxFreeMem( sb);
         }
      }
      if (((rc == ST_SUBDR) || (rc == ST_SUBRT)) && (lsn != dirsect))
      {
         rc = ST_DIREC;                         // non 1st kind of dir-sector
      }
   }
   RETURN (rc);
}                                               // end 'dfsFatDirClusterType'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Determine allocation-bit for specified LSN
/*****************************************************************************/
ULONG dfsFatAllocated                           // RET   LSN is allocated
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               al = NO_ERROR;           // not allocated
   ULONG               value;                   // cluster allocation value
   ULONG               cluster;
   ULONG               asn = dfstAreaD2Part( DFSTORE, lsn); // Area aware sn

   if (asn < fat->Sect)                         // within volume boundary ?
   {
      if (asn < fat->ClTwo)                     // Boot/Fat/Root area
      {
         al = TRUE;
      }
      else
      {
         if ((cluster = dfsFatLSN2Clust(asn)) == fat->CachedClNumber)
         {
            al = fat->CachedClAlloc;            // use cached value
         }
         else
         {
            value = dfsFatValue( cluster);
            if (value != 0)                     // not free
            {
               if (value != FAT_BADCLUST)       // and not BAD
               {
                  al = TRUE;
               }
            }
            fat->CachedClAlloc  = al;
            fat->CachedClNumber = cluster;
         }
      }
   }
   else
   {
      al = DFS_PSN_LIMIT;
   }
   return (al);
}                                               // end 'dfsFatAllocated'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set allocate status for LSN to specified value
/*****************************************************************************/
ULONG dfsFatSetAlloc                            // RET   LSN allocation set
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *value,                   // IN    NULL = not allocated
   void               *ref                      // IN    dummy
)
{
   ULONG               rc = NO_ERROR;           // not allocated

   ENTER();
   if (lsn < fat->Sect)                         // within volume boundary ?
   {
      //- to be refined (set to cluster value ?)
      //- or to <eof> value, awaiting real FAT-chain update ...
   }
   else
   {
      rc = DFS_PSN_LIMIT;
   }
   RETURN(rc);
}                                               // end 'dfsFatSetAlloc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate EA-index to relative sector-number
/*****************************************************************************/
ULN64 dfsFatEaIndex2Rsn                         // RET   RSN for this EA index
(
   USHORT              index                    // IN    EA index number
)
{
   ULN64               rsn = 0;

   ENTER();
   TRARGS(("index: %4.4X\n", index));

   if (index < fat->EaData.MapSize)             // avoid trap on invalid index
   {
      rsn = (fat->EaData.Index->base[0] + fat->EaData.Map[index]) * fat->ClustSize;
   }
   RETN64 (rsn);
}                                               // end 'dfsFatEaIndex2Rsn'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate Cluster-nr to LSN
/*****************************************************************************/
ULN64 dfsFatCl2Lsn                              // RET   LSN for this cluster
(
   ULN64               cluster,                 // IN    cluster number
   ULN64               d2,                      // IN    dummy
   char               *p1,                      // IN    dummy
   void               *p2                       // IN    dummy
)
{
   ULN64               lsn = cluster;

   if ((cluster < 2) && (fat->Root != 0))       // CL 0/1 and initialized ?
   {
      lsn = fat->Root;                          // allow cluster 0/1 in DIR
   }                                            // entries to point to root
   else
   {
      lsn = fat->ClTwo + ((cluster -2) * fat->ClustSize);
   }
   return (lsn);
}                                               // end 'dfsFatCl2Lsn'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate LSN to Cluster-nr
/*****************************************************************************/
ULN64 dfsFatLsn2Cl                              // RET   cluster for this LSN
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *p1,                      // IN    dummy
   void               *p2                       // IN    dummy
)
{
   ULN64               cluster = 2;             // minimal valid cluster

   if (lsn == DFS_MAX_PSN)                      // query max Cluster-nr
   {
      cluster = fat->RawClust - 1;              // highest is #clusters - 1
   }
   else if (lsn >= fat->ClTwo)
   {
      cluster = ((lsn - fat->ClTwo) / fat->ClustSize) + 2;
   }
   return (cluster);
}                                               // end 'dfsFatLsn2Cl'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get cluster-value from a Directory entry structure
// Note: May return non-zero on a FAT32 dir-entry with an LFN!
/*****************************************************************************/
ULONG dfsFatDir2Clust                           // RET   cluster value
(
   S_FATDIR           *entry                    // IN    FAT directory entry
)
{
   ULONG               cluster = (ULONG) entry->clust;

   if (fat->FatBits == 32)                      // combine Hi-part of 32-bit
   {
      cluster += (((ULONG) entry->clustHi) << 16);
   }
   return (cluster);
}                                               // end 'dfsFatDir2Clust'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set cluster-value in a Directory entry structure
/*****************************************************************************/
void dfsClust2FatDir
(
   ULONG               cluster,                 // IN    cluster value to set
   S_FATDIR           *entry                    // INOUT FAT directory entry
)
{
   entry->clust = (USHORT) (cluster & 0xFFFF);

   if (fat->FatBits == 32)                      // split-off Hi-part of 32-bit
   {
      entry->clustHi = (USHORT) (cluster >> 16);
   }
}                                               // end 'dfsClust2FatDir'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Check if cluster could be the 1st root-DIR sector (DIR but no '..' entry)
/*****************************************************************************/
BOOL dfsFatRootCandidate                        // RET   Could be a root DIR
(
   ULONG               cluster                  // IN    target cluster
)
{
   BOOL                rc = FALSE;              // function return
   BYTE               *sb = NULL;               // sector buffer
   ULONG               dirsect;

   ENTER();
   TRARGS(("cluster: %8.8x\n", cluster));

   if ((sb = TxAlloc(1, dfsGetSectorSize())) != NULL)
   {
      dirsect = dfsFatClust2LSN( cluster);
      if (dfsRead( dirsect, 1, sb) == NO_ERROR)
      {
         switch (dfsIdentifySector( dirsect, 0, sb))
         {
            case ST_ROOTD:
            case ST_DIREC:
               rc = TRUE;
               break;

            default:
               break;
         }
      }
      TxFreeMem(sb);
   }
   BRETURN (rc);
}                                               // end 'dfsFatRootCandidate'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get cluster-value from the Fat Cache structure (32 bit entries)
/*****************************************************************************/
ULONG dfsFatValue                               // RET   cluster value
(
   ULONG               clust                    // IN    cluster to get value
)
{
   ULONG               value = 0;               // rc, cluster value
   DFSFATCACHE        *this = &fat->CacheA;
   DFSFATCACHE        *oldest;

   if ((clust < this->First) || (clust >= (this->First + DFSFAT_CACHE)))
   {
      this = &fat->CacheB;
      if ((clust < this->First) || (clust >= (this->First + DFSFAT_CACHE)))
      {
         //- Not in either cache, select a cache to be reloaded with needed info
         oldest = (this->Stamp < fat->CacheA.Stamp) ? this : &fat->CacheA;
         if (((this->Dirty == TRUE ) && (fat->CacheA.Dirty == TRUE )) ||
             ((this->Dirty == FALSE) && (fat->CacheA.Dirty == FALSE))  )
         {
            this = oldest;                      // same, use oldest, causing toggle
         }
         else                                   // reload the non-Dirty one, to allow
         {                                      // successive Rd/Wr without thrashing
            if (this->Dirty == TRUE)            // B is dirty
            {
               this = &fat->CacheA;             // so reload and usa A
            }
         }
         TRACES(("FatValue for 0x%8.8x, not cached, reload: cache-%c\n", clust, this->Id));
         if (this->Dirty == TRUE)               // selected cache is Dirty
         {
            //- Flush the dirty cache before reloading it
            dfsFatFlushFAT( this);
         }
         dfsFatGetFAT( clust, this);            // reload needed part of FAT
      }
   }
   value = this->Value[clust % DFSFAT_CACHE];   // get cluster value from cache

   return (value);
}                                               // end 'dfsFatValue'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set cluster-value in the Fat Cache structure (32 bit entries, internally)
/*****************************************************************************/
ULONG dfsFatSetCluster                          // RET   NO_CHANGE / NO_ERROR
(
   ULONG               clust,                   // IN    cluster to set value for
   ULONG               value                    // IN    FAT-value to be set
)
{
   ULONG               rc = NO_ERROR;
   DFSFATCACHE        *this = &fat->CacheA;
   DFSFATCACHE        *oldest;

   if ((clust < this->First) || (clust >= (this->First + DFSFAT_CACHE)))
   {
      this = &fat->CacheB;
      if ((clust < this->First) || (clust >= (this->First + DFSFAT_CACHE)))
      {
         //- Not in either cache, select a cache to be reloaded with needed info
         oldest = (this->Stamp < fat->CacheA.Stamp) ? this : &fat->CacheA;
         if (((this->Dirty == TRUE ) && (fat->CacheA.Dirty == TRUE )) ||
             ((this->Dirty == FALSE) && (fat->CacheA.Dirty == FALSE))  )
         {
            this = oldest;                      // same, use oldest, causing toggle
         }
         else                                   // reload the Dirty one, to allow
         {                                      // successive Rd/Wr without thrashing
            if (this->Dirty == FALSE)           // B is clean
            {
               this = &fat->CacheA;             // so reload and use A
            }
         }
         TRACES(("SetCluster for 0x%8.8x, not cached, reload: cache-%c\n", clust, this->Id));
         if (this->Dirty == TRUE)               // selected cache is Dirty
         {
            //- Flush the dirty cache before reloading it
            dfsFatFlushFAT( this);
         }
         rc = dfsFatGetFAT( clust, this);       // reload needed part of FAT
      }
   }
   if (this->Value[clust % DFSFAT_CACHE] != value)
   {
      if (this->Dirty == FALSE)
      {
         TRACES(("Cache %c now DIRTY on writing: 0x%8.8x(%7x)\n", this->Id, clust, value));
      }
      this->Value[clust % DFSFAT_CACHE] = value; // set cluster value in cache
      this->Dirty = TRUE;
   }
   else
   {
      rc = DFS_NO_CHANGE;                       // value written same as existing
   }
   return (rc);
}                                               // end 'dfsFatSetCluster'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Reset bad-sector admin, change each BAD cluster in the FAT to FREE
/*****************************************************************************/
ULONG dfsFatResetBadClus                        // RET   ref cluster or 0
(
   void
)
{
   ULONG               rc = NO_ERROR;           // function return
   ULONG               cBase;                   // lowest cl in cache
   ULONG               cOffs;                   // offset cl in cache
   ULONG               cFirst;                  // first offset to check
   ULONG               cLast;                   // last offset to check
   ULONG               bads = 0;                // total bad clusters
   DFSFATCACHE        *cache = &fat->CacheA;

   ENTER();
   TRACES(( "Fat1:0x%llx Fat2:0x%llx InUse:0x%llx\n", fat->Fat1, fat->Fat2, fat->FatInUse));

   TxPrint( "\nScanning FAT area for clusters marked 'BAD' ...\n");

   //- Since FAT areas might be changed, it is fine to use and invalidate the caches
   fat->CacheB.First  = FAT_NO_CLUST;           // invalidate B cache

   for ( cBase  = 0;                            // Big steps, whole cache ...
        (cBase <= fat->MapClust) && (rc == NO_ERROR) && !TxAbort();
         cBase += DFSFAT_CACHE)
   {
      dfsFatGetFAT( cBase, cache);              // cache needed part of FAT

      cFirst = (cBase == 0) ? 2 : 0;            // first valid cl is 2
      if ((cBase + DFSFAT_CACHE) > fat->MapClust)
      {
         cLast = fat->MapClust % DFSFAT_CACHE;     // last valid offset
      }
      else
      {
         cLast = DFSFAT_CACHE -1;               // last offset within cache
      }
      for (cOffs = cFirst; (cOffs <= cLast) && (rc == 0); cOffs++)
      {
         if (cache->Value[cOffs] == FAT_BADCLUST)
         {
            TRACES(( "Reset BAD cluster nr: %8.8x\n", cBase + cOffs));
            cache->Value[cOffs] = FAT_FREECLUS;
            bads++;
            cache->Dirty = TRUE;
         }
      }
      if (cache->Dirty)
      {
         rc = dfsFatFlushFAT( cache);
      }
   }
   if (bads > 0)
   {
      dfsSz64( "\nTotal bad sectors : ", (ULN64) bads * fat->ClustSize, ", have been reset\n");
   }
   else
   {
      TxPrint( "\nNo bad sectors present, no action required ...\n");
   }
   RETURN (rc);
}                                               // end 'dfsFatResetBadClus'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Search reference to specified cluster in FAT (cluster pointing to this one)
// Note: Dont use regular cache to avoid trashing that just by searching here
/*****************************************************************************/
ULONG dfsFatFindReference                       // RET   ref cluster or 0
(
   ULONG               cluster                  // IN    target cluster
)
{
   ULONG               rc = 0;                  // function return
   ULONG               cBase;                   // lowest cl in cache
   ULONG               cOffs;                   // offset cl in cache
   ULONG               cFirst;                  // first offset to check
   ULONG               cLast;                   // last offset to check
   DFSFATCACHE         cache;                   // allocated local cache

   ENTER();
   TRARGS(("Find cluster ref for: %8.8x, in FAT at 0x%llx\n", cluster, fat->FatInUse));

   //- avoid use of FatValue() and use local cache to gain speed and avoid cache trashing
   if ((cache.Value = TxAlloc(DFSFAT_CACHE, sizeof(ULONG))) != NULL)
   {
      for ( cBase  = 0;                         // Big steps, whole cache ...
           (cBase <= fat->MapClust) && (rc == 0) && !TxAbort();
            cBase += DFSFAT_CACHE)
      {
         dfsFatGetFAT( cBase, &cache);          // cache needed part of FAT

         cFirst = (cBase == 0) ? 2 : 0;         // first valid cl is 2
         if ((cBase + DFSFAT_CACHE) > fat->MapClust)
         {
            cLast = fat->MapClust % DFSFAT_CACHE; // last valid offset
         }
         else
         {
            cLast = DFSFAT_CACHE -1;            // last offset within cache
         }
         for (cOffs = cFirst; (cOffs <= cLast) && (rc == 0); cOffs++)
         {
            if (cache.Value[cOffs] == cluster)
            {
               rc = cBase + cOffs;
            }
         }
      }
      TxFreeMem( cache.Value);
   }
   RETURN (rc);
}                                               // end 'dfsFatFindReference'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Calulate 32-bit CRC over FAT-contents (to compare FAT1 and FAT2 perhaps :-)
/*****************************************************************************/
ULONG dfsFatCalculateFatCRC                     // RET   32-bit CRC
(
   ULONG               fatlsn                   // IN    first sector of FAT
)
{
   ULONG               rc = 0;                  // function return
   ULONG               cBase;                   // lowest cl in cache

   ENTER();
   TRARGS(("fatlsn: 0x%llx\n", fatlsn));

   #if defined (USEWINDOWING)
      if (txwIsWindow( TXHWND_DESKTOP))
      {
         dfsa->statusTimer = TxTmrSetTimer( TMR_SEC( 1)); // only after 1 second ...
      }
   #endif
   dfsFatSetFatInUse( fatlsn);                  // flush cache and switch FAT

   for ( cBase  = 0;                            // Big steps, whole cache ...
        (cBase <= fat->MapClust) && !TxAbort();
         cBase += DFSFAT_CACHE)
   {
      #if defined (USEWINDOWING)
         if (txwIsWindow( TXHWND_DESKTOP))
         {
            if (TxTmrTimerExpired( dfsa->statusTimer))
            {
               TXTM    status;
               sprintf( status, "Calculate CRC FAT%c, done %6u of %u",
                       (fatlsn == fat->Fat1) ? '1' :
                       (fatlsn == fat->Fat2) ? '2' : 'x',
                        cBase, fat->MapClust);
               txwSetSbviewStatus( status, cSchemeColor);
               dfsa->statusTimer = TxTmrSetTimer( DFSP_STATUS_INTERVAL);
            }
         }
      #endif

      dfsFatGetFAT(  cBase, &fat->CacheA);      // cache needed part of FAT
      rc ^= TxCrc32( fat->CacheA.Value, DFSFAT_CACHE * sizeof(ULONG));
      TRACES(( "CRC value from cluster %8.8x = %8.8x\n", cBase, rc));
   }
   RETURN (rc);
}                                               // end 'dfsFatCalculateFatCRC'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set the active FAT LSN to be used for fat references using the cache
/*****************************************************************************/
void dfsFatSetFatInUse
(
   ULONG               fat2Use                  // IN    first sector of FAT
)
{
   ENTER();
   TRARGS(("fat2Use: 0x%llx\n", fat2Use));

   if (fat->FatInUse != fat2Use)
   {
      //- Flush changes to disk, BEFORE switching to other FAT disk area
      if (fat->CacheA.Dirty)
      {
         dfsFatFlushFAT( &fat->CacheA);
      }
      if (fat->CacheB.Dirty)
      {
         dfsFatFlushFAT( &fat->CacheB);
      }
      fat->FatInUse      = fat2Use;             // switch
      fat->CacheA.First  = FAT_NO_CLUST;
      fat->CacheB.First  = FAT_NO_CLUST;        // invalidate FAT caches
   }
   VRETURN();
}                                               // end 'dfsFatSetFatInUse'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Copy FAT-part from disk to specified FAT cache structure (32 bit entries)
/*****************************************************************************/
ULONG dfsFatGetFAT                              // RET
(
   ULONG               clust,                   // IN    cluster value to get
   DFSFATCACHE        *cache                    // INOUT cache structure used
)
{
   ULONG               rc  = NO_ERROR;
   ULN64               lsn;                     // LSN of 1st sector to read
   ULONG               sectors;                 // nr of FAT sectors to read
   BYTE               *dfat;                    // FAT sector buffer
   BYTE               *byte;                    // value pointer
   ULONG               i;
   ULONG               value;
   ULONG               clusters;                // cluster values in chunk

   ENTER();

   cache->First  = clust -  (clust % DFSFAT_CACHE);

   if ((clusters = (fat->MapClust - cache->First)) > DFSFAT_CACHE)
   {
      clusters = DFSFAT_CACHE;
   }

   TRARGS(("FAT LSN:0x%llx for cl:0x%8.8x, first:0x%8.8x, clusters:0x%4.4x  cache:%c\n",
            fat->FatInUse, clust, cache->First, clusters, cache->Id));

   if ((fat->FatInUse != 0) && (clusters != 0))
   {
      switch (fat->FatBits)
      {
         case 32:                               // no separate sector-buffer
            dfat    = (BYTE   *) cache->Value;
            lsn     = cache->First   / (dfsGetSectorSize() / sizeof(ULONG));
            sectors = ((clusters -1) / (dfsGetSectorSize() / sizeof(ULONG))) +1;
            break;

         case 16:
            dfat    = TxAlloc(DFSFAT_CACHE, sizeof(USHORT));
            lsn     = cache->First   / (dfsGetSectorSize() / sizeof(USHORT));
            sectors = ((clusters -1) / (dfsGetSectorSize() / sizeof(USHORT))) +1;
            break;

         default:                               // allways start at 1st
            dfat    = TxAlloc(DFSFAT_CACHE, sizeof(USHORT));
            lsn     = 0;

            //- calculate maximum nr of sectors first, avoiding buffer overwrite
            sectors = DFSFAT_CACHE  / (dfsGetSectorSize() / sizeof(USHORT));
            if (fat->FatSectors < sectors)
            {
               sectors = fat->FatSectors;       // limit to real nr of sects
            }
            break;
      }
      TRACES(( "Reading %u FAT sectors, relative lsn:0x%llx\n", sectors, lsn));

      lsn += fat->FatInUse;                     // add to start of FAT lsn
      if ((dfat != NULL) && (cache->Value != NULL))
      {
         //- make sure whole cache-buffer is ZERO to get reliable CRC
         //- values for partially filled ones (the last one)

         memset( cache->Value, 0, (DFSFAT_CACHE * sizeof(ULONG)));

         if ((rc = dfsRead( dfstAreaP2Disk( DFSTORE, lsn), sectors, dfat)) == NO_ERROR)
         {
            switch (fat->FatBits)
            {
               case 12:
                  byte = (BYTE *) dfat;
                  for ( i  = 0;
                       (i  < DFSFAT_CACHE) && (i <= fat->MapClust);
                        i += 2, byte += 3)
                  {
                     value = (ULONG) (*byte | ((*(byte+1) & 0x0f) << 8));
                     if ((i == 0) || (value > 0xff5)) // special value ?
                     {
                        value |= 0x0ffff000;    // extend to 32-bit
                     }
                     cache->Value[i] = value;

                     value = (ULONG) (((*(byte+1) & 0xf0) >> 4) | (*(byte+2)) << 4);
                     if ((i == 0) || (value > 0xff5)) // special value ?
                     {
                        value |= 0x0ffff000;    // extend to 32-bit
                     }
                     cache->Value[i+1] = value;
                  }
                  break;

               case 16:
                  for (i = 0; (i < DFSFAT_CACHE) && (i <= fat->MapClust); i++)
                  {
                     value = (ULONG) (((USHORT *) dfat)[i]);
                     if (value > 0xfff5)        // special FAT16 value ?
                     {
                        value |= 0x0fff0000;    // extend to 32-bit
                     }
                     cache->Value[i] = value;
                  }
                  break;

               default:                         // 32 bit, no convert needed
                  break;
            }
            if (++lastCacheStamp == 0)          // when wrapped, reset BOTH caches ages
            {
               fat->CacheA.Stamp = 0;
               fat->CacheB.Stamp = 0;
               lastCacheStamp = 1;              // will be assigned to current one
            }
            cache->Stamp = lastCacheStamp;
            cache->Dirty = FALSE;               // clean now

            TRHEXS(70,dfat, fat->FatBits * 4, "start of physical FAT");
            TRACES(("Converted to FAT cache, age is: 0x%8.8x, contents:\n", lastCacheStamp));
            TRHEXS(70,cache->Value, 0x80, "start of FAT cache");
         }
         else
         {
            dfsX10("\nFailed to read FAT sectors at LSN : ", lsn, CBR, "\n");
         }
         if ((ULONG *) dfat != cache->Value)    // temp buffer allocated ?
         {
            TxFreeMem(dfat);
         }
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   else
   {
      rc = DFS_BAD_STRUCTURE;
   }
   RETURN (rc);
}                                               // end 'dfsFatGetFAT'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Flush specified FAT cache (32 bit entries) back to disk FatInUse (+ other)
/*****************************************************************************/
ULONG dfsFatFlushFAT                            // RET   result
(
   DFSFATCACHE        *cache                    // INOUT cache structure to write
)
{
   ULONG               rc  = NO_ERROR;
   ULN64               lsn;                     // LSN of 1st sector to read
   ULONG               sectors;                 // nr of FAT sectors to read
   ULONG              *lfat;                    // logical FAT area, 32 bit
   BYTE               *dfat;                    // FAT sector buffer
   BYTE               *byte;                    // value pointer
   ULONG               i;
   ULONG               clusters;                // cluster values in chunk

   ENTER();

   if ((clusters = (fat->MapClust - cache->First)) > DFSFAT_CACHE)
   {
      clusters = DFSFAT_CACHE;
   }
   TRARGS(("FAT LSN:0x%llx first:0x%8.8x, clusters:0x%8.8x  cache:%c\n",
                  fat->Fat1, cache->First, clusters, cache->Id));

   if (clusters != 0)
   {
      switch (fat->FatBits)
      {
         case 32:                               // no separate sector-buffer
            dfat    = (BYTE   *) cache->Value;
            lsn     = cache->First   / (dfsGetSectorSize() / sizeof(ULONG));
            sectors = ((clusters -1) / (dfsGetSectorSize() / sizeof(ULONG))) +1;
            break;

         case 16:
            dfat    = TxAlloc(DFSFAT_CACHE, sizeof(USHORT));
            lsn     = cache->First   / (dfsGetSectorSize() / sizeof(USHORT));
            sectors = ((clusters -1) / (dfsGetSectorSize() / sizeof(USHORT))) +1;
            break;

         default:                               // always start at 1st
            dfat    = TxAlloc(DFSFAT_CACHE, sizeof(USHORT));
            lsn     = 0;

            //- calculate maximum nr of sectors first, avoiding buffer overwrite
            sectors = DFSFAT_CACHE  / (dfsGetSectorSize() / sizeof(USHORT));
            if (fat->FatSectors < sectors)
            {
               sectors = fat->FatSectors;       // limit to real nr of sects
            }
            break;
      }

      lfat = cache->Value;
      if ((dfat != NULL) && (lfat != NULL))
      {
         //- convert from cache to dfat area (fat12/16) first
         switch (fat->FatBits)
         {
            case 12:
               byte = dfat;
               for (i = 0; i < clusters; i+=2)
               {
                  *byte++ = (BYTE) (  lfat[i]   & 0x0ff);
                  *byte++ = (BYTE) (((lfat[i]   & 0xf00) >> 8) |
                                    ((lfat[i+1] & 0x00f) << 4));
                  *byte++ = (BYTE) (((lfat[i+1] & 0xff0) >> 4));
               }
               break;

            case 16:
               for (i = 0; i < clusters; i++)
               {
                  ((USHORT *) dfat)[i] = (USHORT) lfat[i] & 0xffff;
               }
               break;

            default:                            // 32 bit, no convert needed
               break;
         }

         TRACES(( "Writing %u FAT sectors, relative lsn:0x%llx\n", sectors, lsn));
         //- write to first-fat area
         if ((rc = dfsWrite( dfstAreaP2Disk( DFSTORE, lsn + fat->FatInUse), sectors, dfat)) == NO_ERROR)
         {
            TRHEXS(70,cache->Value, 0x80, "start of FAT cache");
            TRACES(("Converted to physical FAT, contents:\n"));
            TRHEXS(70,dfat, fat->FatBits * 4, "start of physical FAT");

            //- write to second-fat area, IFF active FAT is Fat1 or Fat2 (not a temp one)
            if      ((fat->FatInUse == fat->Fat1) && (fat->Fat2 != 0)) // 1 used, 2 present
            {
               dfsWrite( dfstAreaP2Disk( DFSTORE, lsn + fat->Fat2), sectors, dfat);
            }
            else if ((fat->FatInUse == fat->Fat2) && (fat->Fat1 != 0)) // 2 used, 1 present
            {
               dfsWrite( dfstAreaP2Disk( DFSTORE, lsn + fat->Fat1), sectors, dfat);
            }
            if (++lastCacheStamp == 0)          // when wrapped, reset BOTH caches ages
            {
               fat->CacheA.Stamp = 0;
               fat->CacheB.Stamp = 0;
               lastCacheStamp = 1;              // will be assigned to current one
            }
            cache->Stamp = lastCacheStamp;
            cache->Dirty = FALSE;               // clean again now
         }
         else
         {
            dfsX10("\nFailed to write FAT sectors at LSN : ", lsn + fat->FatInUse, CBR, "\n");
         }
         if ((ULONG *) dfat != cache->Value)   // temp buffer allocated ?
         {
            TxFreeMem(dfat);
         }
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   else
   {
      rc = DFS_BAD_STRUCTURE;
   }
   RETURN (rc);
}                                               // end 'dfsFatFlushFAT'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Show logical FAT
/*****************************************************************************/
void dfsFatShowFAT
(
   ULONG               size,                    // IN    nr of entries
   ULONG               start,                   // IN    first cl# to show
   char               *options                  // IN    display options
)
{
   ULONG               i;
   ULONG               value;
   TXTM                tbuf;
   TXLN                text;

   ENTER();

   strcpy( text, "");
   for (i = 0; (i < size) && !TxAbort() && fat->CacheA.Value; i++)
   {
      if ((i % 8) == 0)
      {
         TxPrint( "%s\n%s%07.7X = %s", text, CBZ, start + i, CNN);
         strcpy( text, "");
      }
      value = dfsFatValue(start + i);
      if ((start + i) < 2)                      // Invalid cluster numbers
      {
         if ((start + i) == 0)
         {
            sprintf( tbuf, "<%sMedia-type%s %2.2x%s> ",
                     CNC, CBC, (USHORT) (value & 0xff), CNN);
            strcat(  text, tbuf);
         }
      }
      else
      {
         if (value > FAT_MAXCLUST)
         {
            if (value >= FAT_MINEOFCL)
            {
               sprintf( tbuf, " <%seof%s>  ", CNC, CNN);
            }
            else
            {
               sprintf( tbuf, " <%sbad%s>  ", CBR, CNN);
            }
         }
         else
         {
            if (value == 0)
            {
               sprintf( tbuf, " <free> ");
            }
            else
            {
               sprintf( tbuf, "%7X ", value);
            }
         }
         strcat(  text, tbuf);
      }
   }
   TxPrint("%s\n", text);
   VRETURN();
}                                               // end 'dfsFatShowFAT'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Count number of used Dir-entries for S_SPACE structure
/*****************************************************************************/
ULONG dfsFatDirUsed                             // RET   matching entries
(
   ULONG               chunks,                  // IN    nr of space entries
   S_SPACE            *space                    // IN    space allocation
)
{
   ULONG               rc;
   ULONG               used = 0;                // nr of used entries
   ULONG               chunk;                   // index in space-area
   ULONG               sect;                    // sectors to handle
   ULONG               entry;                   // index in fat directory
   USHORT              bps = dfsGetSectorSize();
   USHORT              entries = (bps / sizeof(S_FATDIR));
   BYTE               *dirbuf;
   S_FATDIR           *fatdir;                  // Fat directory sector

   ENTER();

   TRACES(("Read %u chunks into %u entries of %u bytes\n",
            chunks, entries, sizeof(S_FATDIR)));

   if ((dirbuf = TxAlloc( 1, bps)) != NULL)     // allocate one sector
   {
      fatdir = (S_FATDIR *) dirbuf;

      for (chunk = 0; (chunk < chunks)  && (!TxAbort()); chunk++)
      {
         for (sect = 0; (sect < space[chunk].size)  && (!TxAbort()); sect++)
         {
            if ((rc = dfsRead(space[chunk].start + sect, 1, dirbuf)) == NO_ERROR)
            {
               for (entry = 0; entry < entries; entry++)
               {
                  switch (fatdir[entry].Name[0])
                  {
                     case FAT_DIRFREE:          // free entry
                     case FAT_DIRDEL:           // deleted entry
                        break;

                     default:                   // used
                        used++;
                        break;
                  }
               }
            }
         }
      }
      TxFreeMem( dirbuf);
   }
   RETURN (used);
}                                               // end 'dfsFatDirUsed'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get label-string from the FAT Root-directory
/*****************************************************************************/
BOOL dfsFatRootLabel                            // RET   Label found
(
   S_BOOTR            *boot,                    // IN    boot record for vol
   ULN64               offset,                  // IN    Offset for boot-rec
   char               *label                    // OUT   label string
)
{
   ULONG               rc = NO_ERROR;
   BOOL                found = FALSE;
   ULONG               size;                    // sectors in Root dir
   ULN64               root;                    // LSN of first root sector
   ULONG               sect;                    // sectors to handle
   ULONG               entry;                   // index in fat directory
   USHORT              bps = dfsGetSectorSize();
   USHORT              entries = (bps / sizeof(S_FATDIR));
   BYTE               *dirbuf;
   S_FATDIR           *fatdir;                  // Fat directory sector

   ENTER();

   //- Note: dfsa->boot is NOT valid when calling this (might be NULL!)

   if ((dirbuf = TxAlloc( 1, bps)) != NULL)     // allocate one sector
   {
      fatdir = (S_FATDIR *) dirbuf;

      if (strncasecmp( boot->f2.Type, "FAT32", 5) == 0) // FAT32 bootrec
      {
         size  =  boot->eb.ClustSize;           // minimum size of Root-Dir
         root  = (ULN64) (boot->f2.RootCluster -2) * size +
                  boot->eb.FatOffset + (boot->f2.FatSectors * boot->eb.NrOfFats);
         TRACES(("Root LSN: 0x%llX  cluster-size: %u\n", root, size));
      }
      else
      {
         size  = (boot->eb.RootEntries * sizeof(S_FATDIR)) / bps;
         root  = (ULN64) boot->eb.FatOffset + (boot->eb.FatSectors * boot->eb.NrOfFats);
      }

      TRACES(("Root-Dir with %u sectors at LSN 0x%llX\n", size, root));
      TRACES(("%u entries of %u bytes, S_DATIM size: %u\n",
               boot->eb.RootEntries,
               (ULONG) sizeof(S_FATDIR),
               (ULONG) sizeof(S_DATIM)));
      for (sect = 0; (sect < size) && (rc == NO_ERROR) && !found; sect++)
      {
         rc = dfsRead(offset + root + sect, 1, dirbuf);
         if (rc == NO_ERROR)
         {
            for (entry = 0; (entry < entries) && !found; entry++)
            {
               switch (fatdir[entry].Name[0])
               {
                  case FAT_DIRFREE:             // free entry
                  case FAT_DIRDEL:              // deleted entry
                     break;

                  default:                      // used, and not a VFAT entry
                     if ((fatdir[entry].FatAttrib != VFAT_ATTRIB) ||
                         (fatdir[entry].clust     != VFAT_CLUSTR) )
                     {
                        if (fatdir[entry].FatAttrib & FATTR_LABEL)
                        {
                           TRACES(("Label found in Dir-entry %u\n", entry));
                           memcpy( label,             fatdir[entry].Name, FAT_NSIZE);
                           memcpy( label + FAT_NSIZE, fatdir[entry].Ext,  FAT_ESIZE);
                           label[  FAT_NSIZE + FAT_ESIZE] = '\0';
                           found = TRUE;
                        }
                     }
                     break;
               }
            }
         }
      }
      TxFreeMem( dirbuf);
   }
   RETURN (found);
}                                               // end 'dfsFatRootLabel'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set the alternate label location, in ROOT-directory, using specified label
/*****************************************************************************/
ULONG dfsFatSetAltBrecLabel
(
   ULN64               v1,                      // IN    unused
   ULN64               v2,                      // IN    unused
   char               *str,                     // IN    new label value (11)
   void               *param                    // IN    unused
)
{
   ULONG               rc = NO_ERROR;
   ULONG               found = 0;               // relative sector to write
   ULONG               size;                    // sectors in Root dir
   ULN64               root;                    // LSN of first root sector
   ULONG               sect;                    // relative sector to read
   ULONG               entry;                   // index in fat directory
   USHORT              bps = dfsGetSectorSize();
   USHORT              entries = (bps / sizeof(S_FATDIR));
   BYTE               *dirbuf;
   S_FATDIR           *fatdir;                  // Fat directory sector
   TXTS                label;                   // space-padded label value

   ENTER();

   //- Note: dfsa->boot IS valid when calling this (FS initialized)

   if ((dirbuf = TxAlloc( 1, bps)) != NULL)     // allocate one sector
   {
      sprintf( label, "%-11.11s", str);         // pad with spaces

      fatdir = (S_FATDIR *) dirbuf;

      if (strncasecmp( dfsa->boot->f2.Type, "FAT32", 5) == 0) // FAT32 bootrec
      {
         size  =  dfsa->boot->eb.ClustSize;           // minimum size of Root-Dir
         root  = (ULN64) (dfsa->boot->f2.RootCluster -2) * size +
                  dfsa->boot->eb.FatOffset + (dfsa->boot->f2.FatSectors * dfsa->boot->eb.NrOfFats);
         TRACES(("Root LSN: 0x%llX  cluster-size: %u\n", root, size));
      }
      else
      {
         size  = (dfsa->boot->eb.RootEntries * sizeof(S_FATDIR)) / bps;
         root  = (ULN64) dfsa->boot->eb.FatOffset + (dfsa->boot->eb.FatSectors * dfsa->boot->eb.NrOfFats);
      }

      TRACES(("Root-Dir with %u sectors at LSN 0x%llX\n", size, root));
      TRACES(("%u entries of %u bytes, S_DATIM size: %u\n", dfsa->boot->eb.RootEntries,
               (ULONG) sizeof(S_FATDIR), (ULONG) sizeof(S_DATIM)));

      for (sect = 0; (sect < size) && (rc == NO_ERROR) && (found == 0); sect++)
      {
         rc = dfsRead( LSN_BOOTR + root + sect, 1, dirbuf);
         if (rc == NO_ERROR)
         {
            for (entry = 0; (entry < entries) && (found == 0); entry++)
            {
               switch (fatdir[entry].Name[0])
               {
                  case FAT_DIRFREE:             // free entry
                  case FAT_DIRDEL:              // deleted entry
                     break;

                  default:                      // used, and not a VFAT entry
                     if ((fatdir[entry].FatAttrib != VFAT_ATTRIB) ||
                         (fatdir[entry].clust     != VFAT_CLUSTR) )
                     {
                        if (fatdir[entry].FatAttrib & FATTR_LABEL)
                        {
                           TRACES(("Set label in existing Dir-entry %u\n", entry));
                           memcpy( fatdir[entry].Name, label,             FAT_NSIZE);
                           memcpy( fatdir[entry].Ext,  label + FAT_NSIZE, FAT_ESIZE);
                           found = LSN_BOOTR + root + sect;
                        }
                     }
                     break;
               }
            }
         }
      }
      if (!found)                               // not found/updated yet, find free
      {
         for (sect = 0; (sect < size) && (rc == NO_ERROR) && (found == 0); sect++)
         {
            rc = dfsRead( LSN_BOOTR + root + sect, 1, dirbuf);
            if (rc == NO_ERROR)
            {
               for (entry = 0; (entry < entries) && (found == 0); entry++)
               {
                  switch (fatdir[entry].Name[0])
                  {
                     case FAT_DIRFREE:          // free entry
                     case FAT_DIRDEL:           // deleted entry
                        TRACES(("Set label in Free/Deleted Dir-entry %u\n", entry));
                        memcpy( fatdir[entry].Name, label,             FAT_NSIZE);
                        memcpy( fatdir[entry].Ext,  label + FAT_NSIZE, FAT_ESIZE);
                        fatdir[entry].FatAttrib = FATTR_LABEL;
                        found = LSN_BOOTR + root + sect;
                        break;

                     default:                   // used
                        break;
                  }
               }
            }
         }
      }
      if (found)
      {
         rc = dfsWrite( found, 1, dirbuf);
         if (rc == NO_ERROR)
         {
            TxPrint( "New  Volume label : '%s' synchronized to ROOT directory\n", str);
         }
      }
      else                                      // not found/updated at all
      {
         rc = DFS_CMD_FAILED;
      }
      TxFreeMem( dirbuf);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsFatSetAltBrecLabel'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create S_SPACE structure for a FAT allocation chain
/*****************************************************************************/
ULONG dfsFatAllocChain                          // RET   nr of clusters
(
   ULONG               cluster,                 // IN    starting cluster nr
   ULONG               maxsize,                 // IN    max file-size, or 0
   ULONG              *errors,                  // INOUT error flags
   ULONG              *chunks,                  // OUT   nr of space entries
   S_SPACE           **space                    // OUT   space allocation
)
{
   ULONG               cl;                      // nr of clusters
   ULONG               this    = cluster;       // current cluster
   ULONG               last    = 0;             // last handled cluster
   ULONG               size    = 0;             // sectors in this chunk
   ULONG               extents = 0;             // nr of extents
   ULONG               areas   = 0;             // nr of space-areas
   ULONG               climit  = fat->MapClust; // limit on nr of clusters
   S_SPACE            *sp      = NULL;

   ENTER();

   if (maxsize != 0)                            // explicit area-size given
   {                                            // limit == clusters +1 !!!
      climit = (maxsize / (fat->ClustSize * dfsGetSectorSize())) +2;
   }
   TRARGS(("Input cluster: %8.8X, maxsize: %8.8X  Cluster limit: %8.8x\n",
                  cluster,        maxsize,               climit));

   for (cl = 1; (cl <= climit) && (cl != 0); cl++)
   {
      TRACES(( "IN-FOR cl: %uthis:%8.8x last:%8.8x extents: %u size:%u\n",
                       cl,   this,       last,      extents,    size));
      if (this == (last +1))                    // cluster in sequence ?
      {
         size += fat->ClustSize;
      }
      else                                      // create new chunk
      {
         if (areas <= extents)                  // areas used up ?
         {
            areas = areas * 8 + 16;
            TRACES(("(re)Allocate %u space chunks, rel cl %u\n", areas, cl))
            sp = (S_SPACE *) realloc( sp, (size_t) (areas+1) * sizeof(S_SPACE));
         }
         if (sp != NULL)
         {
            if (last != 0)                      // not first extent ?
            {
               sp[extents].size = size;         // set size of last extent
               extents++;                       // and move to next one
            }
            size = fat->ClustSize;              // start with 1 cluster
            sp[extents].size  = size;
            sp[extents].start = dfsFatClust2LSN(this);
         }
         else
         {
            cl = 0;
         }
      }
      if (this >= FAT_MINEOFCL)                 // EOF value
      {
         break;                                 // leave
      }
      else                                      // follow the chain
      {
         last = this;
         if ((last >= 2) && (last <= fat->MapClust)) // OK to follow ?
         {
            this = dfsFatValue(last);
            if ((this <= 1) || (last <= 1))     // free cluster, error
            {
               if (errors)
               {
                  *errors |= EF_ALLOC_CL_FREE;
               }
               sp[extents].size = size;         // set size of last extent
               extents++;                       // and set #extents
               break;
            }
            else if (this == FAT_BADCLUST)
            {
               if (errors)
               {
                  *errors |= EF_ALLOC_CL_BADS;
               }
               sp[extents].size = size;         // set size of last extent
               extents++;                       // and set #extents
               break;
            }
         }
         else
         {
            if (errors)
            {
               if (last == FAT_BADCLUST)
               {
                  *errors |= EF_ALLOC_CL_BADS;
               }
               else
               {
                  *errors |= EF_CLUSTER_RANGE;
               }
            }
            sp[extents].size = size;            // set size of last extent
            extents++;                          // and set #extents
            break;
         }
      }
      if (cl == climit)                         // maximum clusters reached
      {
         if (errors)
         {
            *errors |= EF_RUNAWAY_CHAIN;
         }
      }
   }
   TRACES(( "ENDFOR cl: %u this:%8.8x last:%8.8x extents: %u size:%lu\n",
                    cl,    this,      last,      extents,    size));
   if (sp != NULL)
   {
      TRACES(( "sp[extents -1].start: 0x%llx        .size: 0x%llx\n",
                sp[extents -1].start, sp[extents -1].size));

      *space  = sp;
      *chunks = extents;
      TRINIT(70);
         dfsSspaceDisplay( SD_DEFAULT, extents, sp);
      TREXIT();
   }
   else
   {
      *space  = NULL;                           // dont return random value!
      *chunks = 0;
      cl = 1;                                   // signal (alloc) failure
   }
   if (errors)
   {
      TRACES(("error flags: 0x%8.8x\n", *errors));
   }
   RETURN (cl -1);
}                                               // end 'dfsFatAllocChain'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Check if name is a valid FAT filename
/*****************************************************************************/
BOOL dfsValidFatName
(
   const void         *fn,                      // IN    ptr to FAT name
   USHORT              len                      // IN    length to check
)
{
   TXLN                name;
   USHORT              lok;

   memset(  name, 0,  TXMAXTM);
   strncpy( name, fn, len);

   lok = strcspn( name, DFS_FAT_INVALIDS);

   TRACES(("FatName: '%*.*s', len: %u, strlen: %u,  length-ok: %u\n",
            len, len, name, len, strlen(name), lok));
   return (lok == len);
}                                               // end 'dfsValidFatName'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// Convert null-terminated name to a valid directory-entry name (8.3)
/*****************************************************************************/
void dfsFatName2Entry
(
   char               *fn,                      // IN    ptr to FAT name
   S_FATDIR           *de                       // OUT   directory entry
)
{
   TXLN                local;
   char               *p;

   ENTER();
   strcpy( local, fn);
   TxStrToUpper( local);
   memset( de->Name, ' ', FAT_NSIZE + FAT_ESIZE);
   if ((p = strrchr(local, '.')) != NULL)       // extention present ?
   {
      *p++ = '\0';                              // terminate base-name
      memcpy(de->Ext, p, strlen(p));            // copy ext
   }
   memcpy( de->Name, local, strlen(local));     // copy base
   TRACES(("Converted '%s' to :\n", fn));
   TRHEXS(70, de, 16, "S_FATDIR directory entry");
   VRETURN();
}                                               // end 'dfsFatName2Entry'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// Convert a valid directory-entry name (8.3) to a valid null-terminated name
/*****************************************************************************/
void dfsFatEntry2Name
(
   S_FATDIR           *de,                      // IN    directory entry
   char               *fn                       // OUT   ptr to FAT name
)
{
   char               *p;

   ENTER();
   memset( fn, '\0', FAT_NSIZE + FAT_ESIZE +2);
   memcpy( fn, de->Name, FAT_NSIZE);            // copy base
   for (p = fn + strlen(fn) -1; *p == ' '; p--)
   {
      *p = '\0';
   }
   if ((de->Ext[0] != ' ') || (de->Ext[1] != ' ') || (de->Ext[2] != ' '))
   {
      if ((de->FatAttrib & FATTR_LABEL) == 0)   // unless this is a label
      {                                         // add the DOT seperator
         *(++p) = '.';
      }
      memcpy( ++p, de->Ext, FAT_ESIZE);         // copy ext
      for (p = fn + strlen(fn) -1; *p == ' '; p--)
      {
         *p = '\0';
      }
   }
   if (fn[0] == FAT_DIRDEL)                     // deleted file(name) ?
   {
      fn[0] = '!';                              // make it a valid name
   }
   TRHEXS(70, de, 16, "S_FATDIR directory entry");
   TRACES(("Converted to : '%s'\n", fn));
   VRETURN();
}                                               // end 'dfsFatEntry2Name'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display all available info about a directory entry, including VFAT & LFN
/*****************************************************************************/
ULONG dfsFatShowDirectoryEntry
(
   ULN64               dirsect,                 // IN    lsn of FAT dir sector
   int                 entry                    // IN    index of dir entry
)
{
   ULONG               rc;
   BYTE               *sb  = NULL;              // sector buffer
   S_FATDIR           *fe;                      // Fat directory entry
   TXLN                lfn;                     // VFAT long filename
   USHORT              bps = dfsGetSectorSize();
   USHORT              entries = (bps / sizeof(S_FATDIR)); // entries per sector

   ENTER();
   TRACES(("dirsect: 0x%llx, entry: 0x%2.2x\n", dirsect, entry));

   //- Need to allocate/read previous sectors to allow reading complete LFN
   if ((sb = TxAlloc( 3, bps)) != NULL)
   {
      strcpy( lfn, "");                         // init first to not-present
      if ((rc = dfsRead(dirsect -2, 3, sb)) == NO_ERROR)
      {
         switch (dfsIdentifySector(dirsect, 0, sb + (2 * bps)))
         {
            case ST_ROOTD:
            case ST_SUBDR:
            case ST_SUBRT:
            case ST_DIREC:
               fe = &(((S_FATDIR *) sb)[(2 * entries) + entry]);

               dfsFatExtractLfn( sb, fe, lfn);  // extract LFN, if present

               TxPrint("\n");
               dfsFatDirHeader( "DirSector  entry", 0); // display header
               dfsX10("   ", dirsect, CBC, " ");
               TxPrint("%s%2x%s ", CNC, entry, CNN);
               dfsFatShowDirEntry( fe, CcZ);
               TxPrint("\n");
               dfsFatShowVfdEntry( fe, lfn, CcZ);
               TxPrint("\n");

               if ((fe->fsize != 0) || (fe->FatAttrib & FATTR_DIRECTORY))
               {
                  dfsFatShowAlloc( dfsFatDir2Clust(fe));
               }

               if ((fe->OS2EA) && (fat->FatBits != 32))
               {
                  nav.xtra = dfsSspaceRsn2Lsn( fat->EaData.Chunks, fat->EaData.Space,
                                dfsFatEaIndex2Rsn( fe->OS2EA));
                  TxPrint("\n");
                  rc = dfsFatEAdataDisplay( nav.xtra);
               }
               break;

            default:
               rc = DFS_PENDING;
               break;
         }
      }
      else if (dirsect < fat->Sect)             // should be there
      {
         dfsX10("\nError reading sector : ", dirsect, CBR, "\n");
      }
      TxFreeMem(sb);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN( rc);
}                                               // end 'dfsFatShowDirectoryEntry'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// FAT filesystem, display DIR header/footer
/*****************************************************************************/
void dfsFatDirHeader
(
   char               *lead,                    // IN    lead text-string
   ULONG               items                    // IN    DIR-lines shown (0 = TOP)
)
{
   TXLN                line;

   sprintf(line, " ================ ========== ======== ===== ============ ============= %s =======\n",
                                                 (fat->FatBits == 32) ? "       " : "=======");
   if (items > 0)
   {
      TxPrint( line);
   }
   TxPrint(                    " %s Date (mod) Time     Attr. Filename          Filesize %s Cluster\n", lead,
                                                 (fat->FatBits == 32) ? "       " : "EA-Clus");
   if (items == 0)
   {
      TxPrint( line);
   }
}                                               // end 'dfsFatDirHeader'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add any EA data info for a FAT16 EA-index to std FEA2-list structure
/*****************************************************************************/
ULONG dfsFatReadEaIndex
(
   USHORT              eaIndex,                 // IN    FAT EA-index
   ULONG              *size,                    // OUT   size in bytes
   S_FEA2LIST        **pfeal                    // OUT   list of FEA2's
)
{
   ULONG               rc    = NO_ERROR;
   S_FEA2LIST         *feal  = NULL;            // list of FEA2's
   S_EACLUST          *sb    = NULL;            // EA buffer
   USHORT              sc;                      // allocated sectors for EA
   ULONG               rsn;                     // relative sector in EA DATA
   ULONG               lsn;
   ULONG               bsz = 0;                 // byte size

   ENTER();

   rsn = dfsFatEaIndex2Rsn( eaIndex);
   lsn = dfsSspaceRsn2Lsn( fat->EaData.Chunks, fat->EaData.Space, rsn);
   bsz = dfsFatEAByteSize( lsn);
   sc  = ((bsz + FAT_EAOFFS) / dfsGetSectorSize()) +1;

   TRACES(("Allocate %u sectors for EA-size %u\n", sc, bsz));
   if ((sb = TxAlloc(sc, dfsGetSectorSize())) != NULL)
   {
      rc = dfsSspaceReadFilePart( fat->EaData.Chunks,
                                  fat->EaData.Space,
                                  rsn, sc, (BYTE *) sb);
      if (rc == NO_ERROR)
      {
         if ((feal = TxAlloc(1, EA_BIGBUF_SIZE)) != NULL)
         {
            rc = dfsOs2EaReadArea( sb->eadata, sb->ealength, feal);
            TRHEXS(70, feal, feal->cbList +10, "EA list");
         }
         else
         {
            rc = DFS_ALLOC_ERROR;
         }
      }
      TxFreeMem(sb);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   *size  = bsz;
   *pfeal = feal;
   RETURN (rc);
}                                               // end 'dfsFatReadEaIndex'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Determine complete EA-data allocation from an LSN and display it
/*****************************************************************************/
ULONG dfsFatEAdataDisplay
(
   ULN64               lsn                      // IN    1st lsn of EA cluster
)
{
   ULONG               rc = NO_ERROR;
   S_EACLUST          *sb = NULL;               // EA buffer
   USHORT              sc;                      // allocated sectors for EA

   ENTER();

   sc = ((dfsFatEAByteSize( lsn) + FAT_EAOFFS) / dfsGetSectorSize()) +1;
   TRACES(("Allocate %u sectors for EA-size %u\n", sc, sb->ealength));
   if ((sb = TxAlloc(sc, dfsGetSectorSize())) != NULL)
   {
      rc = dfsSspaceReadFilePart( fat->EaData.Chunks,
                                  fat->EaData.Space,
                dfsSspaceLsn2Rsn( fat->EaData.Chunks,
                                  fat->EaData.Space,
                                  lsn),
                                  sc, (BYTE *) sb);
      if (rc == NO_ERROR)
      {
         rc = dfsFatDisplayEA( sb, (USHORT) (sc * dfsGetSectorSize()));
      }
      TxFreeMem(sb);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN( rc);
}                                               // end 'dfsFatEAdataDisplay'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Determine exact size in bytes, for an OS/2 style EA (FAT16 only!)
/*****************************************************************************/
ULONG dfsFatEAByteSize                          // RET EA size in bytes
(
   ULN64               lsn                      // IN    1st lsn of EA cluster
)
{
   ULONG               rc = 0;
   S_EACLUST          *sb = NULL;               // EA buffer

   ENTER();

   if ((sb = TxAlloc(1, dfsGetSectorSize())) != NULL)
   {
      if ((rc = dfsRead(lsn, 1, (BYTE   *) sb)) == NO_ERROR)
      {
         rc = sb->ealength;
      }
      TxFreeMem(sb);
   }
   RETURN( rc);
}                                               // end 'dfsFatEAByteSize'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read one cluster EA-data from an LSN and display it
/*****************************************************************************/
ULONG dfsFatDisplayEA
(
   S_EACLUST          *ea,                      // IN    EA data cluster(s)
   USHORT              size                     // IN    buffer size
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   TxPrint("Size of EA data   : %-5u ", ea->ealength);
   TxPrint("for filename: %s%s%s", CBY, ea->fname, CNN);
   TxPrint(", EA data index : %u\n",    ea->index);

   dfsEaBlockDisplay( "EA data. sf area  : ", ea->eadata, size -FAT_EAOFFS);

   RETURN( rc);
}                                               // end 'dfsFatDisplayEA'
/*---------------------------------------------------------------------------*/



/*****************************************************************************/
// Read one cluster EA-data from an LSN and display it
/*****************************************************************************/
ULONG dfsFatDispEAindex
(
   S_EAINDEX          *eai,                     // IN    EA index cluster(s)
   USHORT              size                     // IN    buffer size
)
{
   ULONG               rc = NO_ERROR;

   ENTER();

   TxPrint("Nr index clusters : %u\n", eai->base[0]);
   TxPrint("Max used EA index : %u\n", eai->base[EAI_SIZE -1]);

   RETURN( rc);
}                                               // end 'dfsFatDispEAindex'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display FAT directory entry on a single line
/*****************************************************************************/
void dfsFatShowDirEntry
(
   S_FATDIR           *entry,                   // IN    FAT directory entry
   BYTE                bg                       // IN    current background
)
{
   ULONG               cluster = 0;
   TXTM                tbuf;
   TXLN                text;

   sprintf( text, " %4.4u-%2.2u-%2.2u %2.2u:%2.2u:%2.2u",
              entry->date.d.year + 1980,
              entry->date.d.month,
              entry->date.d.day,
              entry->time.t.hours,
              entry->time.t.minutes,
              entry->time.t.twosecs * 2);

   dfstrFatAttrib( text, entry->FatAttrib);


   sprintf( tbuf, " %*.*s %*.*s",
      FAT_NSIZE, FAT_NSIZE, entry->Name,
      FAT_ESIZE, FAT_ESIZE, entry->Ext);
   strcat( text, tbuf);

   if (entry->FatAttrib & (FATTR_DIRECTORY | FATTR_LABEL))
   {
      strcat( text, "              ");
   }
   else
   {
      dfstrUlDot13( text, " ", entry->fsize, "");
   }
   cluster = dfsFatDir2Clust( entry);
   if ((fat->FatBits != 32) && (entry->OS2EA))  // OS/2 style EA index
   {
      BYTE             fg = (bg == CcZ) ? CcW : CcW | CcI;

      sprintf( tbuf, "%s%8llX%s",
              ansi[Ccol((CcG | CcI), bg)],      // bright green on background
              dfsFatLSN2Clust(
                 dfsSspaceRsn2Lsn( fat->EaData.Chunks, fat->EaData.Space,
                    dfsFatEaIndex2Rsn( entry->OS2EA))),
              ansi[Ccol(fg, bg)]);              // white on background
      strcat( text, tbuf);
   }
   else
   {
      strcat( text, "        ");
   }
   if (cluster != 0)
   {
      sprintf( tbuf, "%8X", cluster);          // can be 32-bit
      strcat( text, tbuf);
   }
   else if (entry->FatAttrib & FATTR_LABEL)     // a LABEL
   {
      strcat( text, " Label 0");
   }
   else if (entry->FatAttrib & FATTR_DIRECTORY) // probably the root ..
   {
      strcat( text, " Root  0");
   }
   else                                         // must be an empty file/dir
   {
      strcat( text, " Empty 0");
   }
   TxPrint( "%s", text);
}                                               // end 'dfsFatShowDirEntry'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display VFAT part of directory entry on a single line, IFF VFAT is present
/*****************************************************************************/
void dfsFatShowVfdEntry
(
   S_FATDIR           *fe,                      // IN    FAT directory entry
   char               *lfn,                     // IN    lfn string
   BYTE                bg                       // IN    current background
)
{
   if ((fe->Vadate.u  != 0) ||
       (fe->Vcdate.u  != 0) ||
       (fe->Vctime.u  != 0) ||
       (strlen( lfn)) != 0   )
   {
      if ((fe->Vadate.u != 0) && (fe->Vadate.u != 0x21))
      {
         TxPrint( "%sAcc:%4.4u-%2.2u-%2.2u ", ansi[Ccol((CcZ | CcI), bg)], // grey on background
                        fe->Vadate.d.year + 1980, fe->Vadate.d.month, fe->Vadate.d.day);
      }
      else
      {
         TxPrint( "             ");
      }
      if ((fe->Vcdate.u != 0) && (fe->Vcdate.u != 0x21))
      {
         TxPrint( " C:%s%4.4u-%2.2u-%2.2u", ansi[Ccol( CcW, bg)], // white on background
                        fe->Vcdate.d.year + 1980, fe->Vcdate.d.month, fe->Vcdate.d.day);
      }
      else
      {
         TxPrint( "   %s          ", ansi[Ccol( CcW, bg)]);   // white on background
      }

      if (fe->Vctime.u != 0)
      {
         int      seconds = fe->Vctime.t.twosecs * 2 + (fe->Vctime10ms / 100);

         TxPrint( " %2.2u:%2.2u:%2.2u:%2.2u ",
                  fe->Vctime.t.hours,
                  fe->Vctime.t.minutes,
                  seconds,
                  fe->Vctime10ms % 100);
      }
      else
      {
         TxPrint( "             ");
      }
      if (strlen( lfn))
      {
         TxPrint( "   %s%s%s", ansi[Ccol((CcY | CcI), bg)], lfn,
                               ansi[Ccol( CcW       , bg)]);
      }
      TxPrint("%s\n", CGE);
   }
}                                               // end 'dfsFatShowVfdEntry'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Insert another UNICODEd lfn-slot in front of the ASCII lfn-string
/*****************************************************************************/
void dfsFatAddLfnSlot
(
   S_VFSLOT           *entry,                   // IN    VFAT lfn slot
   char               *name                     // INOUT LFN string
)
{
   TXLN                lfn;

   ENTER();

   TxUnic2Ascii( entry->lfn5, lfn, 5);
   TxUnicAppend( entry->lfn6, lfn, 6);
   TxUnicAppend( entry->lfn2, lfn, 2);

   TRACES(("Append '%s' to this '%s'\n", name, lfn));
   strcat( lfn, name);
   strcpy( name, lfn);
   VRETURN();
}                                               // end 'dfsFatAddLfnSlot'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display FAT allocation chain from cluster
/*****************************************************************************/
void dfsFatShowAlloc
(
   ULONG               cluster                  // IN    Cluster number
)
{
   S_SPACE            *sp = NULL;
   ULONG               nr;
   ULONG               ef = 0;                  // Initialize error flag!

   dfsX10("Space for cluster : ", cluster, CBM, "\n");
   if ((cluster == 0) && (fat->Root != 0))      // fake for Root directory
   {
      dfsDirLsn2Space( fat->Root, &nr, &sp);
   }
   else
   {
      dfsFatAllocChain( cluster, 0, &ef, &nr, &sp);
   }
   if (sp != NULL)
   {
      ULONG            sdo = SD_TOPLINE | SD_SUMMARY;

      if (!TxaOptUnSet('X'))                    // unless -X- specified
      {
         sdo |= SD_NAVDOWN;                     // next is data sector
      }
      nr = dfsSspaceDisplay( sdo, nr, sp);
      free( sp);
   }
   else if (!TxaOptUnSet('X'))                  // unless -X- specified
   {
      nav.down = dfsFatClust2LSN( cluster);     // allow digging down anyway
   }                                            // FAT contents may be wrong
   if (ef != 0)
   {
      dfsFatDispError( ef, 0, "\nWarning: ", NULL);
   }
}                                               // end 'dfsFatShowAlloc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Retrieve first-cluster number for specified Dir-sector-LSN + index (LsnInf)
/*****************************************************************************/
ULONG dfsLsnInf2Cluster                         // RET   First cluster or 0
(
   ULN64               lsn,                     // IN    Directory LSN
   ULONG               info                     // IN    directory entry
)
{
   ULONG               rc  = 0;
   S_FATDIR            fatdir;                  // Fat directory sector

   ENTER();

   if (dfsLsnInf2Entry( lsn, info, &fatdir) == NO_ERROR)
   {
      rc = dfsFatDir2Clust( &fatdir);
   }
   RETURN (rc);
}                                               // end 'dfsLsnInf2Cluster'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Retrieve filesize for specified Dir-sector-LSN + index (LsnInf)
/*****************************************************************************/
ULONG dfsLsnInf2FileSize                        // RET   First cluster or 0
(
   ULN64               lsn,                     // IN    Directory LSN
   ULONG               info                     // IN    directory entry
)
{
   ULONG               rc  = 0;
   S_FATDIR            fatdir;                  // Fat directory sector

   ENTER();

   if (dfsLsnInf2Entry( lsn, info, &fatdir) == NO_ERROR)
   {
      rc = fatdir.fsize;
   }
   RETURN (rc);
}                                               // end 'dfsLsnInf2FileSize'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Retrieve directory-entry for specified Dir-sector-LSN + index (LsnInf)
/*****************************************************************************/
ULONG dfsLsnInf2Entry
(
   ULN64               lsn,                     // IN    Directory LSN
   ULONG               info,                    // IN    directory entry
   S_FATDIR           *direntry                 // OUT   directory entry
)
{
   ULONG               rc  = NO_ERROR;
   USHORT              bps = dfsGetSectorSize();
   BYTE               *dirbuf;
   S_FATDIR           *fatdir;                  // Fat directory sector

   ENTER();
   TRARGS(("Input LSN: 0x%llX, entry:%u\n", lsn, DFSSNIGET( info)));

   if ((dirbuf = TxAlloc( 1, bps)) != NULL)     // allocate one sector
   {
      fatdir = (S_FATDIR *) dirbuf;

      rc = dfsRead( lsn, 1, dirbuf);
      if (rc == NO_ERROR)
      {
         *direntry = fatdir[ DFSSNIGET( info)];
      }
      TxFreeMem( dirbuf);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsLsnInf2Entry'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create SPACE allocation structure from a Directory LSN (root or other)
/*****************************************************************************/
ULONG dfsDirLsn2Space
(
   ULN64               lsn,                     // IN    lsn for DIR sector
   ULONG              *nr,                      // OUT   nr of space entries
   S_SPACE           **sp                       // OUT   space allocation
)
{
   ULONG               rc  = NO_ERROR;
   ULONG               chunks;                  // nr of space entries
   S_SPACE            *space = NULL;            // space allocation
   ULONG               ef = 0;                  // Initialize error flag!

   ENTER();
   TRARGS(("Input LSN: 0x%llX\n", lsn));

   if ((lsn >= (fat->Root + fat->Roots)) ||     // not the root in FAT12 / 16
       (fat->FatBits == 32))                    // Root has sub-dir structure
   {
      if (dfsFatAllocChain( dfsFatLSN2Clust(lsn), 0, &ef, &chunks, &space) == 0)
      {
         if (ef != 0)                           // FAT area probably damaged
         {
            dfsFatDispError( ef, 0, "\nWarning: ", NULL);

            chunks = 1;                         // just return first cluster
            if ((space = (S_SPACE *) calloc((size_t) chunks, sizeof(S_SPACE))) != NULL)
            {
               space->start = lsn;              // start of cluster
               space->size  = fat->ClustSize;   // size of chunk
            }
            else
            {
               rc = DFS_ALLOC_ERROR;
            }
         }
         else
         {
            rc = DFS_ALLOC_ERROR;
         }
      }
   }
   else                                         // one chunk, root
   {
      chunks = 1;
      if ((space = (S_SPACE *) calloc((size_t) chunks, sizeof(S_SPACE))) != NULL)
      {
         space->start = fat->Root;              // complete Root directory
         space->size  = fat->Roots;
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   *sp = space;
   *nr = chunks;
   RETURN (rc);
}                                               // end 'dfsDirLsn2Space'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Iterate over directory-structures finding files, and execute callback
/*****************************************************************************/
ULONG dfsFatIterator
(
   ULN64               recurse,                 // IN    recurse subtrees
   ULN64               type,                    // IN    do 'D', 'f' or both
   char               *path,                    // IN    path spec, info only
   void               *p                        // IN    iterate parameters
)
{
   ULONG               rc  = NO_ERROR;
   DFS_PARAMS         *ip  = (DFS_PARAMS *) p;  // Iterator parameters
   S_FATDIR            de;                      // directory entry
   ULONG               nr;                      // Alloc chunks
   S_SPACE            *sp  = NULL;              // Alloc chunk array

   ENTER();

   //- Specified path translated to an LSN and optional entry-nr in params
   TRACES(("Lsn: 0x%llx  Entry: %u\n", ip->Lsn, ip->Number));

   memset( &de, 0, sizeof( S_FATDIR));          // default CL 0 ==> ROOT

   if (ip->Number & DFSSNINFO)                  // reference to a DIR entry
   {
      dfsLsnInf2Entry( ip->Lsn, ip->Number, &de);
   }
   if (dfsDirLsn2Space(dfsFatClust2LSN(dfsFatDir2Clust(&de)), &nr, &sp) == NO_ERROR)
   {
      iterationsDone = 0;                      // initialize static progress
      rc = dfsFatIterFiles( recurse, type, nr, sp, ip);
      free( sp);
   }
   RETURN (rc);
}                                               // end 'dfsFatIterator'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create valid FAT(32) boot record based on fat-> info from FatInit
/*****************************************************************************/
ULONG dfsFatMkBootRec
(
   char                osFat16,                 // IN    OS preference FAT16
   BOOL                verbose,                 // IN    verbose and confirm
   USHORT              rootEntries,             // IN    root-entries or 0
   ULONG               serialNr,                // IN    Vol-serialNr or 0
   char               *label                    // IN    Label or ""  or NULL
)
{
   ULONG               rc  = DFS_NO_CHANGE;
   ULONG               partSectors;
   S_BOOTR            *brt;
   char                os  = (char) toupper( osFat16);
   TXTS                string;
   TXTS                osname;
   ULONG               serial = serialNr;

   ENTER();

   TRACES(( "Requested OS: '%c'\n", os));
   if ((SINF->p != NULL) || ((fat->Fat1 != 0) && (fat->Fat2 > fat->Fat1)))
   {
      if (serial == 0)                          // create default
      {
         serial = dfstLSN2Psn( DFSTORE, 0) + 0xee000000;
      }
      if (label && strlen(label))
      {
         if (strlen(label))
         {
            strcpy( string, label);
         }
         else
         {
            strcpy( string, "           ");     // empty label string
         }
      }
      else
      {
         sprintf( string, "DFS%8.8x", serial); // default label
      }
      if ((brt = TxAlloc(1, dfsGetSectorSize())) != NULL)
      {
         if (fat->FatBits == 32)                // larger than 2GB
         {
            os = '9';                           // use Win-98 bootcode
         }
         else if ((osFat16 == 0) || (strchr("IPMNOVW", os) == NULL))
         {
            os = 'M';                           // default to MS-DOS
         }

         dfsMakeFatOsString(   os, osname);
         dfsInsertFatBootCode( os, brt);

         switch (dfstStoreType( DFSTORE))
         {
            case DFST_PHDISK:                   // possibly partitioned
            case DFST_MEMORY:
               if (SINF->p != NULL)             // partition info available
               {
                  partSectors           = SINF->p->sectors;
                  brt->eb.HiddenSectors = SINF->p->partent.BootSectorOffset;
                  brt->eb.LogGeoSect    = SINF->p->geoSecs;
                  brt->eb.LogGeoHead    = SINF->p->geoHeads;
               }
               else                             // large floppy
               {
                  partSectors           = fat->Sect;
                  brt->eb.HiddenSectors = 0;
                  brt->eb.LogGeoSect    = dfstGeoSectors( DFSTORE);
                  brt->eb.LogGeoHead    = dfstGeoHeads(   DFSTORE);
                  if (fat->FatBits == 32)
                  {
                     TxPrint( "\nWARNING: Large floppy format FAT32 is "
                              "NOT supported on OS/2 or eComstation!"
                     #if defined (DEV32)
                              "\nTo use FAT32, you need to create a primary partition first!"
                     #endif
                              "\n\n");
                  }
               }
               break;

            default:
               if (verbose)
               {
                  TxPrint( "\n%sNo partition-table info%s is available, this will "
                           "make the 'HiddenSectors'\nfield incorrect when this "
                           "is a primary partition.\n", CBR, CNN);
                  TxPrint( "The logical geometry used now is %s%hu%s Heads "
                           "with %s%hu%s Sectors\n",
                            CBY, dfstGeoHeads(     DFSTORE), CNN,
                            CBY, dfstGeoSectors(   DFSTORE), CNN);
                  TxPrint("Use the 'diskgeo' command to change this when incorrect\n\n");
               }
               partSectors           = fat->Sect;
               brt->eb.HiddenSectors = dfstGeoSectors( DFSTORE);
               brt->eb.LogGeoSect    = dfstGeoSectors( DFSTORE);
               brt->eb.LogGeoHead    = dfstGeoHeads(   DFSTORE);
               break;
         }
         //- Fill in the extended boot parameter block (BPB)

         brt->eb.SectSize         = dfsGetSectorSize(); // 512 .. 4096
         brt->eb.ClustSize        = (BYTE)   fat->ClustSize;
         brt->eb.FatOffset        = (USHORT) fat->Fat1;
         brt->eb.NrOfFats         = 2;
         if ((partSectors > 0xfff) && (brt->eb.HiddenSectors != 0))
         {
            brt->eb.MediaType     = 0xf8;
         }
         else                                   // small FAT-FS or large floppy
         {
            brt->eb.MediaType     = 0xf0;
         }
         if (partSectors > 0xffff)
         {
            brt->eb.Sectors       = 0;
            brt->eb.BigSectors    = partSectors;
         }
         else
         {
            brt->eb.Sectors       = (USHORT) partSectors;
            brt->eb.BigSectors    = 0;
         }

         //- Fill in the specific fat32 or generic os structures ...
         if (fat->FatBits == 32)
         {
            brt->eb.RootEntries   = 0;
            brt->eb.FatSectors    = 0;
            brt->f2.FatSectors    = fat->FatSectors;
            brt->f2.RootCluster   = dfsFatLSN2Clust( fat->Root);
            brt->f2.FsInfoLsn     = 1;          // 2nd bootsector, FSP info
            brt->f2.SpBootLsn     = 6;          // spare bootsector LSN
            brt->f2.PhysDriveNr   = 0x80;
            brt->f2.Signature     = 0x29;
            brt->f2.SerialNr      = serial;
            memcpy( brt->f2.Label, string, 11);
            memcpy( brt->f2.Type,  "FAT32   ", 8);
         }
         else
         {
            brt->eb.RootEntries   = (rootEntries != 0)  ?  rootEntries :
                                            (fat->FatBits == 12) ? 224 : 512;
            brt->eb.FatSectors    = (USHORT) fat->FatSectors;
            brt->os.PhysDriveNr   = 0x80;
            brt->os.Signature     = 0x29;
            brt->os.SerialNr      = serial;
            memcpy( brt->os.Label, string, 11);
            memcpy( brt->os.Type,  "FAT     ", 8);
         }

         if (verbose)
         {
            TxPrint( "\nThe following new FAT bootsector has been prepared:\n\n");
            dfsDisplaySector( dfstLSN2Psn( DFSTORE, LSN_BOOTR), ST_BOOTR, (BYTE   *) brt);
         }
         if ((dfsa->batch) || (verbose == FALSE) ||
             (TxConfirm( 5129, "A new FAT bootsector has been prepared with "
                               "bootcode for OS:\n\n      %s\n\n"
                               "Do you want to replace the current "
                               "filesystem bootsector (sector 0)\n"
                               "by the newly created one ? [Y/N] : ", osname)))
         {
            if (DFSTORE_WRITE_ALLOWED)
            {
               rc = dfsWrite( LSN_BOOTR, 1, (BYTE   *) brt);
               if ((rc == NO_ERROR) && verbose)
               {
                  TxPrint("\nFAT boot record successfully updated\n");
               }
            }
            else
            {
               rc = DFS_READ_ONLY;
            }
         }
         TxFreeMem( brt);
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   else
   {
      TxPrint("No partition info and no FAT positions available, "
              "create not possible\n");
   }
   RETURN(rc);
}                                               // end 'dfsFatMkBootRec'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get/Set DIRTY bits in FAT16/32 bootsector to specified value
/*****************************************************************************/
BYTE dfsFatDirtyStatusBootrec                   // RET   final dirty value
(                                               //       or RC when > 2
   BOOL                update,                  // IN    update value
   BYTE                dirty                    // IN    new value: 0, 1, 2
)
{
   ULONG               rc = 0;
   S_BOOTR            *brt;
   BYTE               *FsysValue;

   ENTER();

   if ((brt = TxAlloc(1, dfsGetSectorSize())) != NULL)
   {
      if ((rc = dfsRead( LSN_BOOTR, 1, (BYTE   *) brt)) == NO_ERROR)
      {
         if (fat->FatBits == 32)
         {
            FsysValue = &brt->f2.FsysValue;
         }
         else
         {
            FsysValue = &brt->os.FsysValue;
         }
         if (update)
         {
            if (DFSTORE_WRITE_ALLOWED)
            {
               *FsysValue = dirty;              // update value
               if ((dfsWrite( LSN_BOOTR, 1, (BYTE   *) brt)) == NO_ERROR)
               {
                  TxPrint("\nFAT boot record successfully updated with new DIRTY status\n");
               }
               else
               {
                  TxPrint("\nFAT boot record update for dirty-flag has failed!\n");
               }
            }
            else
            {
               rc = DFS_READ_ONLY;
            }
         }
         rc = *FsysValue;
      }
      else
      {
         TxPrint("\nFAT bootrecord read error!\n");
      }
      TxFreeMem( brt);
   }
   RETURN(rc);
}                                               // end 'dfsFatDirtyStatusBootrec'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// Get/Set DIRTY bits in FAT16/32 FAT-area entry 1 to specified value
/*****************************************************************************/
BYTE dfsFatDirtyStatusFatarea                   // RET   final dirty value
(                                               //       or RC when > 2
   BOOL                update,                  // IN    update value
   BYTE                dirty                    // IN    new value: 0, 1, 2
)
{
   ULONG               rc = 0;
   BYTE               *sec;
   BYTE               *FsysValue;

   ENTER();

   if ((fat->FatBits > 12) && ((sec = TxAlloc(1, dfsGetSectorSize())) != NULL))
   {
      //- Use entry 1 from FAT, but NOT for a FAT12 system (MS spec FAT16/32 only)

      if ((rc = dfsRead( fat->Fat1, 1, sec)) == NO_ERROR)
      {
         if (fat->FatBits == 32)
         {
            FsysValue = &sec[7];
         }
         else
         {
            FsysValue = &sec[3];
         }
         if (update && (fat->Fat1 != 0))
         {
            if (DFSTORE_WRITE_ALLOWED)
            {
               if (fat->FatBits == 32)
               {
                  *FsysValue |= 0x0c;              //- set upper 2 bits of 28-bit FAT entry
                  if (dirty & 0x01)
                  {
                     *FsysValue &= 0xf7;           //- reset bit-27 if dirty
                  }
                  if (dirty & 0x02)
                  {
                     *FsysValue &= 0xfb;           //- reset bit-26 if scan required
                  }
               }
               else
               {
                  *FsysValue |= 0xc0;              //- set upper 2 bits of 16-bit FAT entry
                  if (dirty & 0x01)
                  {
                     *FsysValue &= 0x7f;           //- reset bit-15 if dirty
                  }
                  if (dirty & 0x02)
                  {
                     *FsysValue &= 0xbf;           //- reset bit-14 if scan required
                  }
               }
               if (((rc = dfsWrite( fat->Fat1, 1, sec)) == NO_ERROR) &&
                   ((rc = dfsWrite( fat->Fat2, 1, sec)) == NO_ERROR)  )
               {
                  TxPrint("FAT areas successfully updated with new DIRTY status\n");
               }
               else
               {
                  TxPrint("\nFAT areas update for dirty-flag has failed!\n");
               }
            }
            else
            {
               rc = DFS_READ_ONLY;
            }
         }

         // Read back and interpret the dirty flags now present in the sector
         if (fat->FatBits == 32)
         {
            if ((*FsysValue & 0x08) == 0)       // bit-27 flagged
            {
               rc |= 0x01;
            }
            if ((*FsysValue & 0x04) == 0)       // bit-26 flagged
            {
               rc |= 0x02;
            }
         }
         else
         {
            if ((*FsysValue & 0x80) == 0)       // bit-15 flagged
            {
               rc |= 0x01;
            }
            if ((*FsysValue & 0x40) == 0)       // bit-14 flagged
            {
               rc |= 0x02;
            }
         }
      }
      else
      {
         TxPrint("\nFAT area-1 read error!\n");
      }
      TxFreeMem( sec);
   }
   RETURN(rc);
}                                               // end 'dfsFatDirtyStatusFatarea'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get de long-filename (LFN) belonging to the specified (short) FAT dir entry
/*****************************************************************************/
ULONG dfsFatExtractLfn                          // RET   LFN length found, or 0
(
   BYTE               *buf,                     // IN    start of buffer (check)
   S_FATDIR           *de,                      // IN    directory entry (short)
   char               *name                     // OUT   long filename or ""
)
{
   S_VFSLOT           *fl;                      // Fat LFN slot  entry
   ULONG               i;

   ENTER();

   strcpy( name, "");                           // init first to not-present
   fl = (S_VFSLOT *) de -1;                     // start at FAT-entry minus 1
   for (i = 1; i < 20; i++)                     // limit nr of slots to search
   {
      TRACES(("Test Lfn slot Id: %2.2x pos: %u\n", fl->SlotId, i));
      if ((fl->FatAttrib == VFAT_ATTRIB) && (fl->clust == VFAT_CLUSTR) && (fl->Vzero == 0))
      {
         if ((ULONG) (fl->SlotId & ~VFAT_SLOT_1) != i)
         {
            TxPrint("\nWarning: LFN slot %u, bad sequence\n", i);
         }
         if ((fl->SlotId & VFAT_SLOT_1) == VFAT_SLOT_1)
         {
            break;                              // correct 1st of sequence
         }
         else                                   // skip to next slot
         {
            if ((BYTE *) fl <= buf)             // at start of buffer
            {
               break;                           // abort searching furher
            }
            else
            {
               fl--;
            }
         }
      }
      else                                      // hit a non VFAT LFN slot
      {
         if (i > 1)
         {
            TxPrint("\nWarning: LFN slot %u, bad terminator\n", i);
         }
         break;
      }
   }
   while ((fl->FatAttrib == VFAT_ATTRIB) &&
          (fl->clust     == VFAT_CLUSTR) &&
          (fl->Vzero     == 0          ) &&
          (i > 0)                         )
   {
      TRACES(("Add  Lfn slot Id: %2.2x\n", fl->SlotId));
      dfsFatAddLfnSlot( fl, name);
      fl++;                                     // follow lfn-slots up to short-entry
      i--;                                      // safe-guard, counting back
   }
   RETURN( (ULONG) strlen( name));
}                                               // end 'dfsFatExtractLfn'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Recursively iterate over FAT directory and files, execute callback
/*****************************************************************************/
static ULONG dfsFatIterFiles
(
   ULN64               rs,                      // IN    recurse subtrees
   ULONG               type,                    // IN    do 'D', 'f' or both
   ULONG               chunks,                  // IN    nr of S_SPACE chunks
   S_SPACE            *space,                   // IN    space structure array
   DFS_PARAMS         *cp                       // IN    callback params
)
{
   ULONG               rc = NO_ERROR;
   ULONG               chunk;                   // index in space-area
   ULONG               sect;                    // sectors to handle
   USHORT              entry;                   // index in fat directory
   ULN64               dlsn;                    // Directory sector LSN
   ULONG               ef = 0;                  // Initialize error flag!
   ULONG               nr;                      // Alloc chunks
   S_SPACE            *sp = NULL;               // Alloc chunk array
   BYTE                id  = ST_UDATA;          // type of sector
   USHORT              bps = dfsGetSectorSize();
   USHORT              entries = (bps / sizeof(S_FATDIR));
   BYTE               *dirbuf;
   S_FATDIR           *fatdir;                  // Fat directory sector
   S_VFSLOT           *fl;                      // ptr to LFN entry struct

   ENTER();
   TRARGS(("Recursion: %d, chunks: %u\n", rs, chunks));

   dirbuf = TxAlloc( 1, bps);                   // allocate one sector
   if ((dirbuf != NULL) && (space != NULL))
   {
      fatdir = (S_FATDIR *) dirbuf;
      for ( chunk = 0;
           (chunk < chunks) && (rc == NO_ERROR) && !TxAbort();
            chunk++)                            // walk all alloc chunks
      {
         TRACES(( "chunk size: 0x%llx  start lsn: 0x%llx\n", space[chunk].size, space[chunk].start));
         for ( sect = 0;
              (sect < space[chunk].size) && (rc == NO_ERROR) && !TxAbort();
               sect++)                          // each sector in chunk
         {
            if (cp->Flag == TRUE)               // progress in sectors
            {
               iterationsDone++;
            }
            dlsn = space[chunk].start + sect;
            TRACES(( "Read directory sector at: 0x%llx\n", dlsn));

            rc = dfsRead(dlsn, 1, dirbuf);

            switch (dfsIdentifySector( dlsn, 0, dirbuf))
            {
               case ST_ROOTD:
               case ST_SUBDR:
               case ST_SUBRT:
               case ST_DIREC:
                  for ( entry = 0;
                       (entry < entries) && (rc == NO_ERROR) && !TxAbort();
                        entry++)                // each dir-entry in sector
                  {
                     if (dfsFatDir2Clust(&fatdir[entry]) != 0) // space allocated ?
                     {
                        switch (fatdir[entry].Name[0])
                        {
                           case FAT_DIRDEL:     // deleted entry
                              TRACES(("Erased:%5u Name: '%*.*s'\n", entry, FAT_N, FAT_N, fatdir[entry].Name));
                              break;

                           case FAT_DIRDOT:     // . or .. entry
                              TRACES(("DirDot:%5u Name: '%*.*s'\n", entry, FAT_N, FAT_N, fatdir[entry].Name));
                              break;

                           case FAT_DIRFREE:    // free entry
                              TRACES(("Free  :%5u Name: '%*.*s'\n", entry, FAT_N, FAT_N, fatdir[entry].Name));
                              break;

                           default:             // used, but could be lfn
                              fl = (S_VFSLOT *) &(fatdir[entry]);
                              if ((fl->FatAttrib == VFAT_ATTRIB) &&
                                  (fl->clust     == VFAT_CLUSTR) &&
                                  (fl->Vzero     == 0))
                              {
                                 TRACES(("Entry :%5u Lfn slotId: 0x%2.2hx\n", entry, fl->SlotId));
                              }
                              else
                              {
                                 BOOL isDir = ((fatdir[entry].FatAttrib & FATTR_DIRECTORY) != 0);
                                 TRACES(("Entry :%5u Name: '%*.*s' type:%u isDir:%s\n", entry,
                                          FAT_N, FAT_N, fatdir[entry].Name, type, (isDir) ? "YES" : "NO"));

                                 if (((isDir == FALSE) && (type != 'D') && (type != 'd')) || //- type for this entry
                                     ((isDir == TRUE ) && (type != 'F') && (type != 'f'))  ) //- not excluded ?
                                 {
                                    if (cp->Flag == FALSE) // progress in entries
                                    {
                                       iterationsDone++;
                                    }
                                    #if defined (USEWINDOWING)
                                       if (txwIsWindow( TXHWND_DESKTOP))
                                       {
                                          if (TxTmrTimerExpired( dfsa->statusTimer))
                                          {
                                             TXTM    status;

                                             if (cp->Flag == FALSE) // progress in entries
                                             {
                                                sprintf( status, "DIRsec: 0x%8.8llx '%-11.11s', done:%6u DIR entries",
                                                                  dlsn, fatdir[entry].Name, iterationsDone);
                                             }
                                             else
                                             {
                                                sprintf( status, "DIRsec: 0x%8.8llx '%-11.11s', done:%6u sectors",
                                                                  dlsn, fatdir[entry].Name, iterationsDone);
                                             }
                                             txwSetSbviewStatus( status, cSchemeColor);
                                             dfsa->statusTimer = TxTmrSetTimer( DFSP_STATUS_INTERVAL);
                                          }
                                       }
                                    #endif
                                    rc = (cp->Func)( dlsn, entry | DFSSNINFO, (char *) &id, cp); // execute callback
                                 }
                                 else
                                 {
                                    TRACES(("Skipping entry: %u\n", entry));
                                 }
                                 if (fatdir[entry].FatAttrib & FATTR_DIRECTORY)
                                 {
                                    if (rs != 1) // recursion ?
                                    {
                                       ef = 0;
                                       dfsFatAllocChain( dfsFatDir2Clust(
                                                         &fatdir[entry]),
                                                          fatdir[entry].fsize,
                                                         &ef, &nr, &sp);
                                       if (sp != NULL)
                                       {
                                          rc = dfsFatIterFiles( rs -1, type, nr, sp, cp);
                                          TxFreeMem( sp);
                                       }
                                    }
                                 }
                              }
                              break;
                        }
                     }
                     else
                     {
                        TRACES(("NoData:%5u Name: '%*.*s'\n", entry, FAT_N, FAT_N, fatdir[entry].Name));
                     }
                  }
                  break;

               default:
                  break;
            }
         }
      }
      TxFreeMem( dirbuf);
   }
   RETURN( rc);
}                                               // end 'dfsFatIterFiles'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read allocation for ALL directory sectors into a single SPACE structure
/*****************************************************************************/
ULONG dfsFatAllDirs2Space
(
   BOOL                progressSectors,         // IN    progress in sectors, not DIRs
   ULONG              *nr,                      // OUT   nr of space entries
   S_SPACE           **sp                       // OUT   space allocation
)
{
   ULONG               rc = NO_ERROR;
   DFS_PARAMS          params;                  // generic function params
   ULONG               chunks = 0;              // chunks in space structure
   S_SPACE            *space  = NULL;           // space structure

   ENTER();

   //- Start with ROOT allocation itself (fixed chunk FAT16 or clusters on FAT32)
   if (dfsDirLsn2Space(fat->Root, &chunks, &space) == NO_ERROR)
   {
      params.Lsn    = fat->Root;                // start sector for iterator
      params.Number = 0;                        // not a specific entry
      params.Func   = dfsFatAddDir2Space;       // Iterator callback function
      params.Misc   = space;
      params.Count  = chunks;                   // Callback IN/OUT structure
      params.Flag   = progressSectors;

      //- Iterate over all directories recursively and add them to the SPACE
      rc = dfsFatIterator( -1, 'D', NULL, &params);
      if (rc == NO_ERROR)
      {
         chunks =             params.Count;
         space  = (S_SPACE *) params.Misc;
      }
   }
   *sp = space;
   *nr = chunks;
   RETURN (rc);
}                                               // end 'dfsFatAllDirs2Space'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add SPACE for specified directory to the accumulating total directory SPACE
/*****************************************************************************/
static ULONG dfsFatAddDir2Space
(
   ULN64               sn,                      // IN    sector-nr DIR sector
   ULN64               info,                    // IN    coded directory entry#
   char               *df,                      // IN    dummy, optional info
   void               *data                     // INOUT callback data, SPACE
)
{
   ULONG               rc      = NO_ERROR;
   DFS_PARAMS         *ip      = (DFS_PARAMS *) data;
   ULONG               extents =                ip->Count;
   S_SPACE            *sp      = (S_SPACE *)    ip->Misc;
   ULONG               chunks;
   S_SPACE            *space   = NULL;            // SPACE for this DIR

   ENTER();

   //- Get space allocation for this new directory
   dfsFatAllocChain( dfsLsnInf2Cluster( sn, info), 0, NULL, &chunks, &space);
   if (space != NULL)
   {
      //- join with accumulator space, and free the old ones
      rc = dfsSspaceJoin( extents, sp, chunks, space, &extents, &sp);
      if (rc == NO_ERROR)
      {
         ip->Count = extents;
         ip->Misc  = sp;                        // pass back joined SPACE
      }
   }
   RETURN (rc);
}                                               // end 'dfsFatAddDir2Space'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Update EVERY cluster reference in each directory entry in SPACE for resize
/*****************************************************************************/
ULONG dfsFatFixDirs4Resize
(
   ULONG               eFsCl,                   // IN    extra clusters each FAT
   ULONG               ext,                     // IN    nr of space entries
   S_SPACE            *spc                      // IN    ALL-DIR space allocation
)
{
   ULONG               rc = NO_ERROR;
   ULONG               sTodo = dfsSspaceSectors( TRUE, ext, spc);
   ULONG               rSect;                   // relative dir-sector in SPACE
   USHORT              bps = dfsGetSectorSize();
   USHORT              entries = (bps / sizeof(S_FATDIR));
   USHORT              entry;
   BYTE               *dirbuf;
   S_FATDIR           *fatdir;                  // Fat directory sector
   S_VFSLOT           *fl;                      // ptr to LFN entry struct
   ULONG               clValue;
   ULONG               nwValue;
   BOOL                dirty;                   // DIR sector modified

   ENTER();
   TRACES(("EFSC:%u  ext:%u  DIR sectors:%u\n", eFsCl, ext, sTodo));

   dirbuf = TxAlloc( 1, bps);                   // allocate one sector
   if ((dirbuf != NULL) && (spc != NULL))
   {
      fatdir = (S_FATDIR *) dirbuf;

      //- Iterate over all DIR entries in the SPACE, and fixup the Cl reference
      for (rSect = 0; (rSect < sTodo) && (rc == NO_ERROR); rSect++)
      {
         rc = dfsSspaceReadFilePart( ext, spc, rSect, 1, dirbuf);
         if (rc == NO_ERROR)
         {
            dirty = FALSE;                      // After read, now modify, then write ...
            for (entry = 0; entry < entries; entry++)
            {
               fl = (S_VFSLOT *) &(fatdir[entry]);
               if ((fl->FatAttrib == VFAT_ATTRIB) &&
                   (fl->clust     == VFAT_CLUSTR) &&
                   (fl->Vzero     == 0))
               {
                  TRACES(("Skip  :%5u Lfn slotId: 0x%2.2hx\n", entry, fl->SlotId));
               }
               else
               {
                  clValue = dfsFatDir2Clust( &( fatdir[ entry]));

                  if (clValue != 0)             // when not free (no alloc)
                  {
                     #if defined (USEWINDOWING)
                        if (txwIsWindow( TXHWND_DESKTOP))
                        {
                           if (TxTmrTimerExpired( dfsa->statusTimer) && (fatdir[entry].Name[0] != '.'))
                           {
                              TXTM    status;
                              sprintf( status, "Updating entry '%-11.11s', DIR sector %5u of %u",
                                       fatdir[entry].Name, rSect, sTodo);
                              txwSetSbviewStatus( status, cSchemeColor);
                              dfsa->statusTimer = TxTmrSetTimer( DFSP_STATUS_INTERVAL);
                           }
                        }
                     #endif
                     nwValue = dfsFatResizedValue( clValue, eFsCl);

                     TRACES(("Fixup :%5u Name: '%*.*s'  clValue: %8.8x -> %8.8x\n",
                              entry, FAT_N, FAT_N, fatdir[entry].Name, clValue, nwValue));

                     dfsClust2FatDir( nwValue, &( fatdir[ entry]));
                     dirty = TRUE;
                  }
               }
            }
            if (dirty)                          // sector was changed
            {
               rc = dfsSspaceWriteFilePart( ext, spc, rSect, 1, dirbuf);
            }
         }
      }
      TxFreeMem( fatdir);
   }
   RETURN (rc);
}                                               // end 'dfsFatFixDirs4Resize'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Update FAT area for resize, COPY moved-part, fixup references, clear rest
/*****************************************************************************/
ULONG dfsFatFixRfat4Resize
(
   ULONG               eFsCl                    // IN    extra clusters each FAT
)
{
   ULONG               rc = NO_ERROR;
   ULONG               cl;                      // (src) cluster number, read
   ULONG               nw;                      // (new) cluster number, write
   ULONG               clValue;
   ULONG               nwValue;
   ULONG               nFDC = fat->NrOfFats * eFsCl + 2; // new first data-cluster

   ENTER();
   TRACES(("EFSC:0x%8.8x  nFDC:0x%8.8x  TruncCluster:0x%8.8x\n", eFsCl, nFDC, fat->TruncCluster));

   TRACES(("Copy entries 2 .. (nFDC -1) to current first free at TruncCluster\n"));
   for ( cl = 2,             nw  = fat->TruncCluster;
        (cl < nFDC)   &&    (rc == NO_ERROR);
         cl++,               nw++)
   {
      clValue = dfsFatValue( cl);
      TRACES(("Copy  cl: 0x%8.8x(%7x) -> 0x%8.8x\n", cl, clValue, nw));
      rc = dfsFatSetCluster( nw, clValue);
      if (rc == DFS_NO_CHANGE)
      {
         rc = NO_ERROR;
      }
   }
   TRACES(("Update ALL cluster references, and move them down over (#fats*EFSC)\n"));
   for ( cl = nFDC,          nw  = 2;
        (rc == NO_ERROR) && (nw < fat->TruncCluster);
         cl++,               nw++)
   {
      clValue = dfsFatValue( cl);

      #if defined (USEWINDOWING)
         if (txwIsWindow( TXHWND_DESKTOP))
         {
            if (TxTmrTimerExpired( dfsa->statusTimer))
            {
               TXTM    status;
               sprintf( status, "Updating FAT internal chains, cluster %6u of %u", nw, fat->TruncCluster);
               txwSetSbviewStatus( status, cSchemeColor);
               dfsa->statusTimer = TxTmrSetTimer( DFSP_STATUS_INTERVAL);
            }
         }
      #endif
      nwValue = dfsFatResizedValue( clValue, eFsCl);

      rc = dfsFatSetCluster( nw, nwValue);
      if (rc == DFS_NO_CHANGE)
      {
         rc = NO_ERROR;
      }
      else                                      // value really changed
      {
         TRACES(("Fixup cl: 0x%8.8x(%lx) -> 0x%8.8x(%7x)\n", cl, clValue, nw, nwValue));
      }
   }
   TRACES(("Clear next (2*EFSC) FAT positions to zero (values have been moved down)\n"));
   for ( cl = 2,             nw  = fat->TruncCluster;
        (cl < nFDC)   &&    (rc == NO_ERROR);
         cl++,               nw++)
   {
      TRACES(("Clear cl: 0x%8.8x\n", nw));
      rc = dfsFatSetCluster( nw, 0);
      if (rc == DFS_NO_CHANGE)
      {
         rc = NO_ERROR;
      }
   }
   //- Flush changes to disk, BEFORE this updated FAT is copied to FAT1/2 !!!
   if (fat->CacheA.Dirty)
   {
      rc = dfsFatFlushFAT( &fat->CacheA);
   }
   if (fat->CacheB.Dirty)
   {
      rc = dfsFatFlushFAT( &fat->CacheB);
   }
   RETURN (rc);
}                                               // end 'dfsFatFixRfat4Resize'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get updated cluster-value for intended resize operation based on shifting
/*****************************************************************************/
ULONG dfsFatResizedValue                        // RET   updated cluster value
(
   ULONG               clust,                   // IN    current cluster value
   ULONG               efsc                     // IN    extra FAT clusters/FAT
)
{
   ULONG               value = clust;           // rc, updated cluster value
   ULONG               fatShift;                // data shift in clusters

   if ((clust != 0) && (clust < fat->RawClust)) // no FREE, and no BAD/EOF
   {
      fatShift = (efsc * fat->NrOfFats);        // data shift in clusters
      if (clust < (fatShift + 2))               // in shifted data, Cl 2 and up
      {
         value += (fat->TruncCluster - 2);      // shifted to end of current used,
      }                                         // Cl#2 was moved to TruncCluster
      value -= fatShift;                        // Delta between the old and new
   }                                            // location of the Cl#2 data cluster
   return (value);
}                                               // end 'dfsFatResizedValue'
/*---------------------------------------------------------------------------*/

