//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// EXT2/3/4 utility functions
//
// Author: J. van Wijk
//
// JvW  30-03-2004 Initial version
// JvW  09-06-2016 Updated for specific EXT4 stuff and 4096 BPS compatibility
//
#ifndef    DFSUEXT
   #define DFSUEXT

#include <dfsext.h>                             // EXT structure definitions


// Find INODE for specified path, starting at known root-directory INODE
ULONG dfsExtFindPath
(
   ULN64               loud,                    // IN    Show progress
   ULN64               d2,                      // IN    dummy
   char               *path,                    // IN    path specification
   void               *vp                       // OUT   found dir/file INODE
);

// DFS EXT verify allocation integrity for a given INODE (normal / deleted)
ULONG dfsExtInodeCheckAlloc                     // RET   check result
(
   S_EXT_INODE        *inode,                   // IN    Inode structure
   char               *options,                 // IN    result options
   ULN64              *fsize,                   // OUT   exact filesize in bytes
   ULN64              *totals,                  // INOUT total sectors
   ULN64              *invalids,                // INOUT invalid sectors
   ULN64              *datalsn                  // OUT   first data lsn
);                                              //       or NULL if not needed

// DFS EXT verify allocation integrity for given INODE SPACE allocation
ULONG dfsExtSpaceCheckAlloc                     // RET   check result
(
   DFSISPACE          *is,                      // IN    data allocation
   BOOL                deleted,                 // IN    delete status
   char               *options,                 // IN    result options
   ULN64              *totals,                  // INOUT total sectors
   ULN64              *invalids,                // INOUT invalid sectors
   ULN64              *datalsn                  // OUT   first data lsn
);                                              //       or NULL if not needed

// Translate Inode number to a combination of sectornumber and index in that sector
ULN64 dfsExtInode2LsnIndex                      // RET   Inode Sector or L64_NULL
(
   ULONG               ino,                     // IN    Inode number
   USHORT             *isi                      // OUT   Inode Sector Index
);

// Translate a combination of sectornumber and index to Inode# or 0 (invalid)
ULONG dfsExtLsnIndex2Inode                      // RET   Inode number or 0
(
   ULN64               lsn,                     // IN    Inode Sector number
   USHORT              isi                      // IN    Inode Sector Index
);

// Read and display all available info about an Inode record
ULONG dfsExtReadDisplayInode
(
   ULN64               inosect,                 // IN    lsn of Inode sector
   USHORT              index                    // IN    index of the Inode
);

// Read a single Inode-record into supplied buffer
ULONG dfsExtGetInodeRecord
(
   ULN64               inosect,                 // IN    lsn of Inode sector
   USHORT              index,                   // IN    index of the Inode
   S_EXT_INODE        *inodeData                // OUT   Inode record contents
);

// Allocate EXT superblock, and read it from byte-offset 1024 into filesystem
S_EXT_SUPER *dfsExtGetSuperBlock                // RET    TxAlloc'ed superblock
(
   DFSPARTINFO        *p                        // IN    NULL if partition opened
);                                              //       or PARTINFO * for a disk

// Identify if FS structures for EXT2 or EXT3 are present, set fsform and label
ULONG dfsExtFsIdentify
(
   S_BOOTR            *boot,                    // IN    Bootsector ref or NULL
   DFSPARTINFO        *p                        // INOUT partition info
);

// Convert EXT INODE (64 bit) time to DFS standard date/time string
char *dfsExtTime2str                            // RET   string value
(
   ULONG               ext_t,                   // IN    time_t compatible SEC
   ULONG               extra,                   // IN    2bit SEC + 30bit NSEC
   char               *dtime                    // INOUT ptr to string buffer
);

// Find Inode and PATH upto Root-directory, starting at an Inode number
BOOL dfsExtFindRootInode                        // OUT   root found
(
   ULONG               start,                   // IN    starting inode number
   char               *path,                    // OUT   combined path, TXMAXLN
   ULONG              *ino                      // OUT   found dir/file Inode#
);

// Find parent Inode for the given (DIR/FILE) Inode number, add name to PATH
BOOL dfsExtInode2Parent                          // OUT   real parent found
(
   ULONG               this,                    // IN    this inode number
   char               *path,                    // OUT   combined path, TXMAXLN
   ULONG              *parent                   // OUT   parent dir/file Inode
);

// Resolve related Inode name from cache, or generate one from Inode+index
BOOL dfsExtResolveInodeName                     // RET   real filename found
(
   ULN64               lsn,                     // IN    Inode lsn for sector
   USHORT              idx,                     // IN    Inode index in sector
   S_EXT_INODE        *ino,                     // IN    Inode record
   ULONG              *parent,                  // OUT   Parent Inode# or NULL
   char               *fname                    // OUT   related name
);                                              //       min TXMAXLN

// Display file allocation and path info for LSN
ULONG dfsExtFileInfo                            // RET   LSN is valid INODE
(
   ULN64               inoLsn,                  // IN    Inode sectornumber
   ULN64               inoIdx,                  // IN    Inode index in sector
   char               *select,                  // IN    Inode select wildcard
   void               *param                    // INOUT leading text/shortpath
);

// Replace sn-list by contents of a single Directory (InoLsn, InoIdx pairs)
ULONG dfsExtMakeBrowseList                      // RET   LSN is valid INODE
(
   ULN64               inoLsn,                  // IN    Inode sectornumber
   ULN64               inoIdx,                  // IN    Inode index in sector
   char               *str,                     // IN    unused
   void               *param                    // INOUT unused
);

// Get allocation information for file-DATA into integrated-SPACE structure
ULONG dfsExtGetAllocSpace
(
   ULN64               inoLsn,                  // IN    Inode sectornumber
   ULN64               inoIdx,                  // IN    Inode index in sector
   char               *str,                     // IN    unused
   void               *param                    // INOUT Integrated SPACE
);

#define dfsExtBlock2Lsn(bl) (ULN64) dfsExtCl2Lsn((bl),0,NULL,NULL)
// Translate Block-nr to LSN, generic interface
ULN64 dfsExtCl2Lsn                              // RET   LSN for this block
(
   ULN64               block,                   // IN    block number
   ULN64               d2,                      // IN    dummy
   char               *p1,                      // IN    dummy
   void               *p2                       // IN    dummy
);

#define dfsExtLsn2Block(sn) dfsExtLsn2Cl((ULN64)(sn),0,NULL, NULL)
// Translate LSN to Block-nr, generic interface
ULN64 dfsExtLsn2Cl                              // RET   block for this LSN
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *p1,                      // IN    dummy
   void               *p2                       // IN    dummy
);


// Determine allocation-bit for specified LSN, ALLOCATED beyond last block!
ULONG dfsExtAllocated                           // RET   LSN is allocated
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);


// Align R/O cache to position for Inode, and get current allocation-bit
BOOL dfsExtInoBitmapCache                       // RET   Inode is allocated
(
   ULONG               ino                      // IN    Inode number
);


// Create EXT2/3/4 Bitmap cache structures and initialize the InoBitMap cache
ULONG dfsExtInoBitMapInit                       // RET   rc = 0 if type match
(
   void
);


// Flush Inode BitMap cache when Dirty, and optional free its resources
ULONG dfsExtInoBitmapFlush                      // RET   rc
(
   BOOL                terminate                // IN    terminate cache too
);

// Align R/W cache to position for block, and get current allocation-bit
BOOL dfsExtBlkBitmapCache                       // RET   block is allocated
(
   ULONG               bl,                      // IN    Block number
   ULONG              *cp                       // OUT   position in cache
);                                              //       or L64_NULL if invalid


// Create EXT2/3 Bitmap cache structures and initialize the BitMap cache
// Note: Area aware to allow usage from FDISK mode too
ULONG dfsExtBlkBitMapInit                       // RET   rc = 0 if type match
(
   void
);


// Flush BitMap cache when Dirty, and optional free its resources
ULONG dfsExtBlkBitmapFlush                      // RET   rc
(
   BOOL                terminate                // IN    terminate cache too
);


// Initialize EXT Group Descriptor array cache from first group, area aware
ULONG dfsExtGroupDescriptors
(
   ULONG               verbose                  // IN    Display verbosity
);


// Display information and block-bitmap graphic for a single block-group
ULONG dfsExtShowBgBitmap
(
   ULONG               group,                   // IN    Block Group number
   ULONG               verbosity,               // IN    0=silent 1=summary 2=bitmap
   ULN64              *usedBlocks,              // INOUT used blocks, from bitmap
   ULN64              *usedDescr,               // INOUT used blocks, from descr.
   ULN64              *usedDiff                 // INOUT descriptor/bitmap diff
);


// Write cached superblock to 1st group superblock location on disk
ULONG dfsExtWriteSuperBlocks
(
   void
);


// Quickly find normal (non-deleted) Inodes using the Inode MAP
ULONG dfsExtQuickFindInodes
(
   BOOL                verbose,                 // IN    List files while found
   ULONG               first,                   // IN    First inode todo
   char               *types,                   // IN    Inode types to search (ignored)
   char               *select                   // IN    Selection string for
);                                              //       resolved name (if any)


// Convert Inode allocation info (Iblocks/Extents) to an S_SPACE, incl. meta
ULONG dfsExtInodeData2Space
(
   S_EXT_INODE        *inode,                   // IN    Inode with alloc info
   DFSISPACE          *is,                      // OUT   Integrated S_SPACE, data
   DFSISPACE          *m1,                      // OUT   meta data 1, L1 or Leaf
   DFSISPACE          *m2                       // OUT   meta data 2, L2 or Index
);


// Allocate memory and read one or more blocks from disk, freemem on read fail
ULONG dfsExtAllocReadBlk
(
   ULONG               block,                   // IN    first Block#
   ULONG               blks,                    // IN    number of blocks
   BYTE              **data                     // OUT   Allocated and read data
);

#endif
