//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// EXT2/3/4 utility functions
//
// Author: J. van Wijk
//
// JvW  30-03-2004 Initial version
// JvW  09-06-2016 Updated for specific EXT4 stuff and 4096 BPS compatibility
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsver.h>                             // DFS version info
#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfstable.h>                           // SLT utility functions

#include <dfsext.h>                             // EXT disk structures
#include <dfsaext.h>                            // EXT analysis functions
#include <dfsuext.h>                            // EXT utility functions
#include <dfslext.h>                            // EXT slt & error definitions


// Search filename in DIR-blocks referenced by specified INODE
static BOOL dfsExtSearchDinode                  // OUT   DIR slots resolved
(
   ULONG               thisIno,                 // IN    directory INODE nr
   ULONG               loud,                    // IN    Show progress
   char               *entry,                   // IN    entry specification
   ULONG              *nextIno                  // OUT   found Inode number
);

// Search filename in a Directory, in an S_SPACE defined chain of blocks
static BOOL dfsExtSearchDirBlocks               // RET   dir entry found
(
   ULONG               loud,                    // IN    Show progress
   char               *entry,                   // IN    entry specification
   DFSISPACE          *dirData,                 // IN    Directory blocks
   ULONG              *ino                      // OUT   found entry inode
);

// Put contents of a Directory SSPACE (InoLsn, InoIdx pairs) into sn-list
static ULONG dfsExtDirSpace2List
(
   BOOL                notRoot,                 // IN    Dir is NOT the ROOT
   DFSISPACE          *dirData                  // IN    Directory blocks
);

// Convert allocation LEAF/INDEX NODE structure into an S_SPACE (recursive)
static ULONG dfsExtAnyNode2Space
(
   S_EXT_EXTNODE      *anyNode,                 // IN    Leaf or Index node
   ULN64              *blksTodo,                // INOUT maximum blocks todo
   DFSISPACE          *is,                      // INOUT Integrated S_SPACE, data
   DFSISPACE          *m1,                      // INOUT meta data 1, L1 or Leaf
   DFSISPACE          *m2                       // INOUT meta data 2, L2 or Index
);

// Handle allocation INDEX NODE, read and process each block#-entry as AnyNode
static ULONG dfsExtIndexNode2Space
(
   S_EXT_EXTNODE      *indexNode,               // IN    Index node with extents
   ULN64              *blksTodo,                // INOUT maximum blocks todo
   DFSISPACE          *is,                      // INOUT Integrated S_SPACE, data
   DFSISPACE          *m1,                      // INOUT meta data 1, L1 or Leaf
   DFSISPACE          *m2                       // INOUT meta data 2, L2 or Index
);

// Convert allocation LEAF-NODE (depth==0) structure into an S_SPACE structure
static ULONG dfsExtLeafNode2Space
(
   S_EXT_EXTNODE      *leafNode,                // IN    Leaf node with extents
   ULN64              *blksTodo,                // INOUT maximum blocks todo
   ULONG              *chunks,                  // OUT   nr of space entries
   S_SPACE           **space                    // OUT   space allocation
);

// Convert classic BLOCKS-INDIRECT structure into an S_SPACE structure
static ULONG dfsExtIblocks2Space
(
   S_EXT_BLKINDIRECT  *bi,                      // IN    block-indirect struct
   ULN64              *blksTodo,                // INOUT maximum blocks todo
   DFSISPACE          *is,                      // INOUT Integrated S_SPACE, data
   DFSISPACE          *m1,                      // INOUT meta data 1, L1 or Leaf
   DFSISPACE          *m2                       // INOUT meta data 2, L2 or Index
);


// Add classic INDIRECT level 1 structure to existing base S_SPACE structure
static ULONG dfsExtL1Indirect2Space
(
   ULONG               block,                   // IN    Block# indirect1
   ULN64              *blksTodo,                // INOUT maximum blocks todo
   DFSISPACE          *is,                      // INOUT Integrated S_SPACE, data
   DFSISPACE          *m1,                      // INOUT meta data 1, L1 or Leaf
   DFSISPACE          *m2                       // INOUT meta data 2, L2 or Index
);

// Add classic INDIRECT level 2 structure to existing base S_SPACE structure
static ULONG dfsExtL2Indirect2Space
(
   ULONG               block,                   // IN    Block# indirect1
   ULN64              *blksTodo,                // INOUT maximum blocks todo
   DFSISPACE          *is,                      // INOUT Integrated S_SPACE, data
   DFSISPACE          *m1,                      // INOUT meta data 1, L1 or Leaf
   DFSISPACE          *m2                       // INOUT meta data 2, L2 or Index
);

// Add classic INDIRECT level 3 structure to existing base S_SPACE structure
static ULONG dfsExtL3Indirect2Space
(
   ULONG               block,                   // IN    Block# indirect1
   ULN64              *blksTodo,                // INOUT maximum blocks todo
   DFSISPACE          *is,                      // INOUT Integrated S_SPACE, data
   DFSISPACE          *m1,                      // INOUT meta data 1, L1 or Leaf
   DFSISPACE          *m2                       // INOUT meta data 2, L2 or Index
);

// Create S_SPACE structure for an array of EXT BLOCK numbers
static ULONG dfsExtAllocSequence                // RET   possible alloc error
(
   ULONG              *blkArray,                // IN    array of blocknumbers
   ULONG               entries,                 // IN    nr of block# in array
   ULN64              *blksTodo,                // INOUT maximum blocks todo
   ULONG              *chunks,                  // OUT   nr of space entries
   S_SPACE           **space                    // OUT   space allocation
);


/*****************************************************************************/
// Find INODE for specified path, starting at known root-directory INODE
/*****************************************************************************/
ULONG dfsExtFindPath
(
   ULN64               loud,                    // IN    Show progress
   ULN64               d2,                      // IN    dummy
   char               *path,                    // IN    path specification
   void               *vp                       // OUT   found dir/file INODE
)
{
   ULONG               rc  = NO_ERROR;
   TXLN                part;
   char               *p   = path;
   int                 l;
   ULONG               curIno = EXT_I_ROOT;
   USHORT              inoIdx;
   ULONG               inoLsn;
   DFS_PARAMS         *parm = (DFS_PARAMS *) vp;

   ENTER();

   if (loud)
   {
      TxPrint("RootDir  Inode nr : 0x%08.8X   find path: '%s'\n", curIno, path);
   }
   parm->Lsn    = dfsExtInode2LsnIndex( curIno, &inoIdx);
   parm->Number = (ULONG) inoIdx | DFSSNINFO;
   while ((rc == NO_ERROR) && strlen(p) && !TxAbort())
   {
      if ((l = strcspn(p, FS_PATH_STR)) != 0)
      {
         strncpy(part, p, l);                   // isolate part
         part[l] = '\0';
         p += l;                                // skip part
         if (*p == FS_PATH_SEP)
         {
            p++;                                // skip '\'
         }
      }
      if (strlen(part))
      {
         if (loud)
         {
            inoLsn = dfsExtInode2LsnIndex( curIno, &inoIdx);
            TxPrint("Directory Inode @ : 0x%08.8X^%hX", inoLsn, inoIdx);
         }
         if (dfsExtSearchDinode( curIno, loud, part, &curIno))
         {
            if (loud)
            {
               TxPrint(" - Inode %08.8X for '%s'\n", curIno, part);
            }
            if (*p == '\0')                     // end of string, found!
            {
               S_EXT_INODE        *inode;

               parm->Lsn    = dfsExtInode2LsnIndex( curIno, &inoIdx);
               parm->Number = (ULONG) inoIdx | DFSSNINFO;
               parm->Flag   = FALSE;            // Size NOT from DIR-entry

               if ((inode = TxAlloc( 1, ext->sup->InodeSize)) != NULL)
               {
                  if ((rc = dfsExtGetInodeRecord( parm->Lsn, inoIdx, inode)) == NO_ERROR)
                  {
                     parm->byteSize = TXmku64( inode->SizeLo, inode->SizeHi);
                  }
                  TxFreeMem( inode);
               }
            }
         }
         else
         {
            if (loud)
            {
               TxPrint(" - Search failed  for '%s'\n", part);
            }
            rc = ERROR_PATH_NOT_FOUND;
         }
      }
      else
      {
         rc = ERROR_PATH_NOT_FOUND;
      }
   }
   RETURN (rc);
}                                               // end 'dfsExtFindPath'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Search filename in DIR-blocks referenced by specified INODE
/*****************************************************************************/
static BOOL dfsExtSearchDinode                  // OUT   DIR slots resolved
(
   ULONG               thisIno,                 // IN    directory INODE nr
   ULONG               loud,                    // IN    Show progress
   char               *entry,                   // IN    entry specification
   ULONG              *nextIno                  // OUT   found Inode number
)
{
   BOOL                rc = FALSE;
   USHORT              inoIdx;
   ULONG               inoLsn;
   S_EXT_INODE        *inode;
   DFSISPACE           is;

   ENTER();
   TRACES(( "Search Dinode: %8.8x for '%s'\n", thisIno, entry));

   if ((inode = TxAlloc( 1, ext->sup->InodeSize)) != NULL)
   {
      inoLsn = dfsExtInode2LsnIndex( thisIno, &inoIdx);
      if ((rc = dfsExtGetInodeRecord( inoLsn,  inoIdx, inode)) == NO_ERROR)
      {
         if (dfsExtInodeData2Space( inode, &is, NULL, NULL) == NO_ERROR)
         {
            rc = dfsExtSearchDirBlocks( loud, entry, &is, nextIno);
         }
         TxFreeMem( is.space);                  // free SPACE
      }
      TxFreeMem( inode);
   }
   BRETURN (rc);
}                                               // end 'dfsExtSearchDinode'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Search filename in a Directory, in an S_SPACE defined chain of blocks
/*****************************************************************************/
static BOOL dfsExtSearchDirBlocks               // RET   dir entry found
(
   ULONG               loud,                    // IN    Show progress
   char               *entry,                   // IN    entry specification
   DFSISPACE          *dirData,                 // IN    Directory blocks
   ULONG              *ino                      // OUT   found entry inode
)
{
   BOOL                rc = FALSE;
   ULONG               totalDirSects;           // sectors in SSPACE
   ULONG               done = 0;                // sectors handled
   BYTE               *dirBlock;                // one block of DIR data
   S_EXT_DIRENTRY     *de;                      // single directory slot
   ULONG               offset = 0;              // byte offset into block
   TXLN                fName;                   // filename
   TXLN                uname;                   // upercase search-name    ASCII

   ENTER();

   TxStrToUpper( strcpy( uname, entry));

   if (loud)
   {
      TxPrint("\nDirblock, 1st LSN : 0x%08.8X", dirData->space[0].start);
   }
   if ((dirBlock = TxAlloc( 1, ext->BlockSize)) != NULL)
   {
      totalDirSects = dfsSspaceSectors( TRUE, dirData->chunks, dirData->space);

      for (done = 0; done < totalDirSects; done += ext->SectorsPerBlock)
      {
         rc = dfsSspaceReadFilePart( dirData->chunks, dirData->space,
                                     done, ext->SectorsPerBlock, dirBlock);

         while ((rc == FALSE) && (offset < ext->BlockSize) && !TxAbort())
         {
            de = (S_EXT_DIRENTRY *) ((BYTE *) dirBlock + offset);

            if (de->Inode == 0)                 // unused entry, end of DIR
            {
               if (de->RecLen == 0)
               {
                  //- should not happen if all RecLen values are correct
                  TRACES(("Aborting dirwalk on de->Inode 0 and RecLen 0\n"));
                  done = totalDirSects;         // abort SSPACE-block loop too
                  break;                        // end the loop
               }
            }
            TxCopy( fName, de->Name, de->NameLen + 1); // null-terminated name
            if (strlen( fName))
            {
               TxStrToUpper( fName);            // make all uppercase
               if (strcmp( uname, fName) == 0)  // exact match
               {
                  *ino = de->Inode;             // Inode number
                  rc   = TRUE;
               }
            }
            offset += de->RecLen;               // to next DIR entry, or end
         }
      }
      TxFreeMem( dirBlock);
   }
   TRACES(("ino:%8.8x\n", *ino));
   BRETURN (rc);
}                                               // end 'dfsExtSearchDirBlocks'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DFS EXT verify allocation integrity for a given INODE (normal / deleted)
/*****************************************************************************/
ULONG dfsExtInodeCheckAlloc                     // RET   check result
(
   S_EXT_INODE        *inode,                   // IN    Inode structure
   char               *options,                 // IN    result options
   ULN64              *fsize,                   // OUT   exact filesize in bytes
   ULN64              *totals,                  // INOUT total sectors
   ULN64              *invalids,                // INOUT invalid sectors
   ULN64              *datalsn                  // OUT   first data lsn
)                                               //       or NULL if not needed
{
   ULONG               rc  = NO_ERROR;
   DFSISPACE           is;

   ENTER();

   rc = dfsExtInodeData2Space( inode, &is, NULL, NULL);
   if (rc == NO_ERROR)
   {
      rc = dfsExtSpaceCheckAlloc( &is, FALSE, options, totals, invalids, datalsn);
   }
   *fsize = TXmku64( inode->SizeLo, inode->SizeHi); // return exact size in bytes too
   TxFreeMem( is.space);                        // free SPACE
   RETURN(rc);
}                                               // end 'dfsExtInodeCheckAlloc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DFS EXT verify allocation integrity for given INODE SPACE allocation
/*****************************************************************************/
ULONG dfsExtSpaceCheckAlloc                     // RET   check result
(
   DFSISPACE          *is,                      // IN    data allocation
   BOOL                deleted,                 // IN    delete status
   char               *options,                 // IN    result options
   ULN64              *totals,                  // INOUT total sectors
   ULN64              *invalids,                // INOUT invalid sectors
   ULN64              *datalsn                  // OUT   first data lsn
)                                               //       or NULL if not needed
{
   ULONG               rc  = NO_ERROR;
   ULONG               i;                       // number alloc sections
   ULN64               j;                       // number of sectors
   ULN64               sn;                      // Data LSN
   BOOL                verbose = (strchr(options, 'v') != NULL);
   BOOL                skipchk = (strchr(options, 's') != NULL);
   ULONG               clsize  = ext->SectorsPerBlock;
   USHORT              col = 0;
   BOOL                alok = FALSE;            // allocation OK
   BOOL                ldot = (BOOL) 2;         // last dot status
   ULONG               dots = 0;                // dots of same color

   ENTER();

   if (is->space != NULL)
   {
      ULN64      spaceSectors = dfsSspaceSectors( FALSE, is->chunks, is->space);

      if (datalsn)
      {
         *datalsn = is->space[0].start;         // first data sector
      }

      if (skipchk)
      {
         *totals = spaceSectors;                // just report size, ignore allocation
      }                                         // (will be reported as 100% OK!)
      else                                      // SPACE allocation, full alloc check
      {
         #if defined (USEWINDOWING)
            if (spaceSectors > DFSECT_BIGF)
            {
               txwSetSbviewStatus( "Checking allocation for a huge file ...", cSchemeColor);
            }
         #endif
         for (i = 0; (i < is->chunks) && !TxAbort(); i++)
         {
            if (verbose)
            {
               if (++col >= dfsGetDisplayMargin())
               {
                  TxPrint("\n");
                  col = 1;
               }
               TxPrint("");                    // start of allocation extent
               dots = 0;
            }
            if (is->space[ i].start != L64_NULL) // normal extent ?
            {
               for ( j  = 0;                    // all block in extent
                    (j <  is->space[ i].size) && !TxAbort();
                     j += clsize)
               {
                  sn = is->space[ i].start + j;

                  switch (dfsExtAllocated( sn, 0, NULL, NULL))
                  {
                     case FALSE: alok =  deleted;                   break;
                     case TRUE:  alok = !deleted;                   break;
                     default:    alok =  FALSE; rc = DFS_PSN_LIMIT; break;
                  }
                  if (verbose)
                  {
                     //- start with 1 dot per block (4KiB) then one per 1024 (4MiB)
                     if ((alok != ldot) || ((++dots) < 33) || ((dots % 1024) == 0))
                     {
                        if (alok != ldot)
                        {
                           ldot = alok;
                           dots = 0;
                        }
                        if (++col >= dfsGetDisplayMargin())
                        {
                           TxPrint("\n");
                           col = 1;
                        }
                        TxPrint("%s%s%s", alok ? CBG : CBR,
                                          alok ? "" : "",
                                          CNN);
                     }
                  }
                  if (!alok)
                  {
                     (*invalids) += clsize;
                  }
                  (*totals) += clsize;
               }
            }
            else                                // sparse extent
            {
               (*totals) += is->space[ i].size * clsize;
               if (verbose)
               {
                  if (++col >= dfsGetDisplayMargin())
                  {
                     TxPrint("\n");
                     col = 1;
                  }
                  TxPrint( "%s%s%s", CNC, "", CNN);
               }
            }
         }
         if (rc == NO_ERROR)
         {
            if (*invalids)
            {
               rc = DFS_BAD_STRUCTURE;
            }
         }
         else                                   // structure / PSN-limit error
         {
            *invalids = clsize;                 // to be refined
            *totals   = *invalids;
         }
         if (verbose)
         {
            TxPrint("\n");
            switch (rc)
            {
               case NO_ERROR:
                  TxPrint("No allocation errors detected\n");
                  break;

               case DFS_PSN_LIMIT:
                  TxPrint("Sector numbers are too large for this EXT filesystem!\n");
                  break;

               default:
                  TxPrint("Allocation error detected on %llu out of %llu sectors, "
                          "save/undelete unreliable!\n", *invalids, *totals);
                  break;
            }
         }
      }
   }
   TRACES(("invalids:%llu  totals:%llu\n", *invalids, *totals));
   RETURN(rc);
}                                               // end 'dfsExtSpaceCheckAlloc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate Inode number to a combination of sectornumber and index in that sector
/*****************************************************************************/
ULN64 dfsExtInode2LsnIndex                      // RET   Inode Sector or L64_NULL
(
   ULONG               ino,                     // IN    Inode number
   USHORT             *isi                      // OUT   Inode Sector Index
)
{
   ULN64               rc = L64_NULL;           // function return
   ULONG               r_ino;                   // Relative Inode in group
   ULONG               group;                   // Group number for block
   S_EXT_GROUPDESCR   *gd;                      // group descriptor structure
   ULONG               InodesPerSector;

   ENTER();

   //- avoid traps when not intialized, sup not present, divide by zero ...
   if (ino && ino <= ext->InodeMax && ext->sup && ext->sup->InodesPerGroup && ext->sup->InodeSize)
   {
      r_ino = (ino - 1) % ext->sup->InodesPerGroup; // Relative Inode in group
      group = (ino - 1) / ext->sup->InodesPerGroup; // Group number for Inode
      gd    = &(ext->GroupDescTable[group]);

      InodesPerSector = dfsGetSectorSize() / ext->sup->InodeSize;

      *isi = r_ino % InodesPerSector;
      rc = ((ULN64) gd->InodeTable * ext->SectorsPerBlock) + (r_ino / InodesPerSector);
   }
   else if (ino > ext->InodeMax)
   {
      TRACES(("ERROR! ino out of range!\n"));
   }
   TRACES(("ino:%8.8x -> lsn:%llx  index:%hu\n", ino, rc, *isi));
   RETN64 (rc);
}                                               // end 'dfsExtInode2LsnIndex'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate a combination of sectornumber and index to Inode# or 0 (invalid)
/*****************************************************************************/
ULONG dfsExtLsnIndex2Inode                      // RET   Inode number or 0
(
   ULN64               lsn,                     // IN    Inode Sector number
   USHORT              isi                      // IN    Inode Sector Index
)
{
   ULONG               rc = 0;                  // function return
   ULONG               r_ino;                   // Relative Inode in group
   ULONG               group;                   // Group number for block
   S_EXT_GROUPDESCR   *gd;                      // group descriptor structure
   ULONG               first;                   // first InodeTable sector
   ULONG               size;                    // size Inodetable in sectors
   ULONG               InodesPerSector;

   ENTER();
   TRACES(("lsn:%8.8x index:%hhu\n", lsn, isi));

   if (ext->sup->InodeSize != 0)
   {
      InodesPerSector = dfsGetSectorSize() / ext->sup->InodeSize;

      for (group = 0; group < ext->Groups; group++)
      {
         gd    = &(ext->GroupDescTable[group]);

         first = gd->InodeTable        * ext->SectorsPerBlock;
         size  = ext->InodeTableBlocks * ext->SectorsPerBlock;

         if ((lsn >= first) && (lsn < (first + size))) // in LSN range for this group?
         {
            r_ino  = ((lsn - first) * InodesPerSector) + (isi & ~DFSSNINFO);
            rc     = r_ino + (group * ext->sup->InodesPerGroup) + 1;
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsExtLsnIndex2Inode'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read and display all available info about an Inode record
/*****************************************************************************/
ULONG dfsExtReadDisplayInode
(
   ULN64               inosect,                 // IN    lsn of Inode sector
   USHORT              index                    // IN    index of the Inode
)
{
   ULONG               rc;
   USHORT              bps = dfsGetSectorSize();
   BYTE               *sb  = NULL;              // sector buffer
   ULONG               ino;

   ENTER();
   TRACES(("inosect: 0x%llx, index: 0x%2.2hhx\n", inosect, index));

   if ((sb = TxAlloc( 1, bps)) != NULL)
   {
      ino = dfsExtLsnIndex2Inode( inosect, index);
      if (ino != 0)                             // valid INO# ?
      {
         if ((rc = dfsRead( inosect, 1, sb)) == NO_ERROR)
         {
            TxPrint( "Calculated Inode# : 0x%8.8x = %10u "
                     "for sect : 0x%s%12.12llx%s index: %s%hhu%s\n",
                      ino, ino, CBM, inosect, CNN, CBC, index, CNN);

            rc = dfsExtShowInodeRecord( sb, inosect, index);

            //- set auto display/Status for THIS Inode
            nav.this        = inosect;
            nav.this_sninfo = index | DFSSNINFO;
         }
         else if (inosect < ext->Sect)          // should be there
         {
            dfsX10("\nError reading sector : ", inosect, CBR, "\n");
         }
      }
      else
      {
         rc = DFS_PENDING;
      }
      TxFreeMem(sb);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN( rc);
}                                               // end 'dfsExtReadDisplayInode'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read a single Inode-record into supplied buffer
/*****************************************************************************/
ULONG dfsExtGetInodeRecord
(
   ULN64               inosect,                 // IN    lsn of Inode sector
   USHORT              index,                   // IN    index of the Inode
   S_EXT_INODE        *inodeData                // OUT   Inode record contents
)
{
   ULONG               rc;
   USHORT              bps = dfsGetSectorSize();
   BYTE               *sb  = NULL;              // sector buffer

   ENTER();
   TRACES(("inosect: 0x%x, index: 0x%2.2hhx\n", inosect, index));

   if ((inosect < ext->Sect) && (index < (dfsGetSectorSize() / ext->sup->InodeSize)))
   {
      if ((sb = TxAlloc( 1, bps)) != NULL)
      {
         if ((rc = dfsRead( inosect, 1, sb)) == NO_ERROR)
         {
            BYTE *inodeRecord = sb + (index * ext->sup->InodeSize);

            memcpy( inodeData, inodeRecord, ext->sup->InodeSize);

            TRACES(("sb:%8.8x inodeRecord:%8.8x inodeData:%8.8x\n", sb, inodeRecord, inodeData));
            TRHEXS( 300,  inodeData,  160, "inodeData");
         }
         TxFreeMem(sb);
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   else
   {
      rc = DFS_BAD_STRUCTURE;
   }
   RETURN( rc);
}                                               // end 'dfsExtGetInodeRecord'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Allocate EXT superblock, and read it from byte-offset 1024 into filesystem
// Note: Area aware to allow usage from FDISK mode too
/*****************************************************************************/
S_EXT_SUPER *dfsExtGetSuperBlock                // RET    TxAlloc'ed superblock
(
   DFSPARTINFO        *p                        // IN    NULL if partition opened
)                                               //       or PARTINFO * for a disk
{
   S_EXT_SUPER        *super = NULL;            // function return
   BYTE               *buffer;
   int                 sectors;                 // sectors to read SB

   ENTER();

   sectors = (4096 / dfsGetSectorSize());       // #sectors to read 4096 bytes
   if ((buffer = TxAlloc( sectors, dfsGetSectorSize())) != NULL)
   {
      if (dfsRead( (p) ? p->basePsn : dfstAreaP2Disk( DFSTORE, LSN_BOOTR), sectors, buffer) == NO_ERROR)
      {
         //- allocate at least same size as largest sector, so the Identify functions
         //- ExtIdent() and IdentifySector() can safely access upto GetSectorSize()

         if ((super = TxAlloc(1, 4096)) != NULL)
         {
            memcpy( super, buffer + 1024, sizeof( S_EXT_SUPER));
         }
      }
      else                                      // no superblock (yet)
      {
         //- to be refined, could try to read from backup location
         //- with estimated blocks/group and blocksize (Txprint backup found)
         //- when no backup, init to 'sensible' values
         //- to avoid crashes in other functions
      }
      TxFreeMem( buffer);
   }
   RETURN (super);
}                                               // end 'dfsExtGetSuperBlock'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Identify if FS structures for EXT2/3/4 are present, set fsform and label
/*****************************************************************************/
ULONG dfsExtFsIdentify
(
   S_BOOTR            *boot,                    // IN    Bootsector ref or NULL
   DFSPARTINFO        *p                        // INOUT partition info
)
{
   ULONG               rc = DFS_FS_UNKNOWN;     // function return
   S_EXT_SUPER        *sup;                     // superblock sector
   char                st;

   ENTER();

   if ((sup = dfsExtGetSuperBlock( p)) != NULL)
   {
      //- will identify on either real superblock, or super-part of EXTBSU
      if ((dfsExtIdent( 0, 0, &st, sup) != DFS_PENDING) && (st == ST_EXTSUP))
      {
         TRACES(( "it IS an EXT superblock\n"));
         if (sup->FeatureCompat & EXT_FCO_HAS_JOURNAL)
         {
            if ((sup->FeatureCompat & EXT_FCO_SPARSE_SUPER2 ) ||
                (sup->FeatureRoComp & EXT_FRO_EXTRA_ISIZE   ) ||
                (sup->FeatureRoComp & EXT_FRO_DIR_NLINK     ) ||
                (sup->FeatureInComp & EXT_FIN_META_BG       ) ||
                (sup->FeatureInComp & EXT_FIN_EXTENTS       ) ||
                (sup->FeatureInComp & EXT_FRO_HUGE_FILE     ) ||
                (sup->FeatureInComp & EXT_FRO_BIGALLOC      ) ||
                (sup->FeatureInComp & EXT_FRO_METADATA_CSUM ) ||
                (sup->FeatureInComp & EXT_FRO_REPLICA       ) ||
                (sup->FeatureInComp & EXT_FRO_READONLY      ) ||
                (sup->FeatureInComp & EXT_FRO_PROJECT       ) ||
                (sup->FeatureInComp & EXT_FIN_EA_INODE      ) ||
                (sup->FeatureInComp & EXT_FIN_DIRDATA       ) ||
                (sup->FeatureInComp & EXT_FIN_CSUM_SEED     ) ||
                (sup->FeatureInComp & EXT_FIN_LARGEDIR      ) ||
                (sup->FeatureInComp & EXT_FIN_INLINE_DATA   ) ||
                (sup->FeatureInComp & EXT_FIN_ENCRYPT       ) ||
                (sup->FeatureInComp & EXT_FIN_FLEX_BG       ) ||
                (sup->FeatureInComp & EXT_FIN_64BIT         ) ||
                (sup->FeatureInComp & EXT_FIN_MMP           )  )
            {
               strcpy( p->fsform, "EXT4");
            }
            else                                // must be an EXT3
            {
               strcpy( p->fsform, "EXT3");
            }
         }
         else                                   // must be an EXT2
         {                                      // or EXT3 with JRNL off
            strcpy( p->fsform, "EXT2");
         }
         TxCopy( p->plabel, sup->VolumeName, EXT_LEN_LBL +1);
         memcpy( p->fsUuid, sup->Uuid,       FS_UUID_LENGTH);
         p->uuidPresent = TRUE;
         rc = NO_ERROR;
      }
      TxFreeMem( sup);
   }
   RETURN (rc);
}                                               // end 'dfsExtFsIdentify'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Convert EXT INODE (64 bit) time to DFS standard date/time string
/*****************************************************************************/
char *dfsExtTime2str                            // RET   string value
(
   ULONG               ext_t,                   // IN    time_t compatible SEC
   ULONG               extra,                   // IN    2bit SEC + 30bit NSEC
   char               *dtime                    // INOUT ptr to string buffer
)
{
   time_t              tm = (time_t) ext_t;     // time in compiler format
   struct tm          *gm;

   ENTER();

   if ((ext_t != 0) && ((gm = gmtime( &tm)) != NULL))
   {
      //- Note: time_t officially is a SIGNED value (1901 .. 2038, 0 = 1970)
      strftime(dtime, TXMAXTM, "%Y-%m-%d %H:%M:%S", gmtime( &tm));
   }
   else                                         // invalid, out of range TM
   {
      sprintf( dtime, "-%8.8x-%8.8x-", ext_t, extra);
   }
   TRACES(("Formatted date/time: %s\n", dtime));

   RETURN( dtime);
}                                               // end 'dfsExtTime2str'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Find Inode and PATH upto Root-directory, starting at an Inode number
/*****************************************************************************/
BOOL dfsExtFindRootInode                        // OUT   root found
(
   ULONG               start,                   // IN    starting inode number
   char               *path,                    // OUT   combined path, TXMAXLN
   ULONG              *ino                      // OUT   found dir/file Inode#
)
{
   BOOL                rc  = FALSE;
   ULONG               cInode = start;          // Current Inode startpoint
   ULONG               parent = 0;              // Parent  Inode found
   BOOL                isParent;
   ULONG               guard  = 100;            // sanity count

   ENTER();
   if (path)
   {
      *path = '\0';                             // start with empty path
   }
   do
   {
      isParent = dfsExtInode2Parent( cInode, path, &parent);
      if (isParent)
      {
         cInode = parent;                       // follow Inodes towards root
      }
   } while (isParent && guard-- && !TxAbort());

   if (parent == cInode)                        // Root is its own parent
   {
      *ino = parent;
      rc   = TRUE;
   }
   TRACES(("Found root %8.8X for Inode %8.8X in '%s'\n", *ino, start, path));
   BRETURN (rc);
}                                               // end 'dfsExtFindRootInode'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Find parent Inode for the given (DIR/FILE) Inode number, add name to PATH
/*****************************************************************************/
BOOL dfsExtInode2Parent                         // OUT   real parent found
(
   ULONG               this,                    // IN    this inode number
   char               *path,                    // OUT   combined path, TXMAXLN
   ULONG              *parent                   // OUT   parent dir/file Inode
)
{
   BOOL                rc  = FALSE;             // not a valid parent
   ULONG               parentInode;
   TXLN                fname;                   // filename component
   BOOL                orphaned = FALSE;        // parent not a directory
   USHORT              inoIdx;
   ULN64               inoLsn = dfsExtInode2LsnIndex( this,   &inoIdx);
   S_EXT_INODE        *inode;

   ENTER();

   *parent = L32_NULL;                          // invalid, simulated end

   TRACES(( "this:%8.8x  path:'%s'\n", this, path));

   if ((inode = TxAlloc( 1, ext->sup->InodeSize)) != NULL)
   {
      if ((rc = dfsExtGetInodeRecord( inoLsn, inoIdx, inode)) == NO_ERROR)
      {
         dfsExtResolveInodeName( inoLsn, inoIdx, inode, &parentInode, fname);
         *parent = parentInode;

         TRACES(( "Parent inode:%8.8x\n", parentInode));

         if (S_ISDIR( inode->Mode))
         {
            strcat( fname, FS_PATH_STR);        // add directory separator
         }
         else if (path && strlen( path))        // pre-pends MUST be a DIR!
         {
            sprintf( fname, "%c%12.12llX-%hX.DIR%c", FS_PATH_SEP, inoLsn, inoIdx, FS_PATH_SEP);
            orphaned = TRUE;
         }
         if (parentInode == EXT_SYS_PARENT)     // system Inode (like Journal)
         {
             *parent = this;                    // fake ROOT reached ...
         }
         else if ((parentInode == 0) || (parentInode == L32_NULL)) // unknown
         {
            orphaned = TRUE;
         }
         if (!orphaned && (*parent != this))    // different inode#, real parent
         {
            rc = TRUE;
         }

         if (path != NULL)                      // path wanted
         {
            if (strlen(fname) + strlen(path) + 30 < dfsa->maxPath)
            {
               strcat( fname, path);            // append existing path
               strcpy( path, fname);            // and copy back
            }
            else                                // try shortened version
            {
               sprintf(fname, "%12.12llX-%hX.DIR%c", inoLsn, inoIdx, FS_PATH_SEP);
               if (strlen(fname) + strlen(path) + 10 < dfsa->maxPath)
               {
                  strcat( fname, path);         // append existing path
                  strcpy( path, fname);         // and copy back
               }
               else                             // truncate this component
               {
                  TRACES(( "Path component '%s' discarded due to length limit\n", fname));
                  sprintf(fname, "x%c", FS_PATH_SEP); // last resort
                  if (strlen(fname) + strlen(path) < dfsa->maxPath)
                  {
                     strcat( fname, path);      // append existing path
                     strcpy( path, fname);      // and copy back
                  }
               }
            }
         }
         TxFreeMem( inode);

         TRACES(( "this ino#:%8.8x  path:'%s'\n", this, path));
      }
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   BRETURN (rc);
}                                               // end 'dfsExtInode2Parent'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Resolve related Inode name from cache, or generate one from Inode+index
/*****************************************************************************/
BOOL dfsExtResolveInodeName                     // RET   real filename found
(
   ULN64               lsn,                     // IN    Inode lsn for sector
   USHORT              idx,                     // IN    Inode index in sector
   S_EXT_INODE        *ino,                     // IN    Inode record
   ULONG              *parent,                  // OUT   Parent Inode# NULL
   char               *fname                    // OUT   related name
)                                               //       min TXMAXLN
{
   BOOL                rc = TRUE;               // function return
   ULONG               parIno  = L32_NULL;
   ULONG               selfIno = dfsExtLsnIndex2Inode( lsn, idx);

   ENTER();

   //- to be refined, calculate inode-number, 2 is ROOT (and some other specials!)

   //- Resolve ROOT and JOURNL even when NO namecache exists ?
   if (selfIno == EXT_I_ROOT)                   // ROOT itself
   {
      strcpy( fname, "");                       // empty name
      parIno = EXT_I_ROOT;                      // fixed Inode number
   }
   else if (selfIno == EXT_I_RESIZE)           // RESIZE inode/file
   {
      strcpy( fname, EXT_TXT_RESIZE);           // resize name
      parIno = EXT_SYS_PARENT;                  // fixed Inode number
   }
   else if (selfIno == EXT_I_JOURNAL)           // JOURNAL file
   {
      strcpy( fname, EXT_TXT_JOURNAL);          // journal name
      parIno = EXT_SYS_PARENT;                  // fixed Inode number
   }
   else if ((ext->Ic)                     &&    // name cache there
            (selfIno <= ext->InodeMax)    &&    // in normal inode range
            (ext->Ic[selfIno].Name )       )    // and name present
   {
      strcpy( fname, ext->Ic[ selfIno].Name); // direct copy from cache
      parIno       = ext->Ic[ selfIno].Parent;
   }

   /* to be refined, test/use XATTR lfn/lnk when present
   else if ((ino->fMagic.magic  == DFS_EXT_MAGIC) && // try magic parent/name
            (ino->fMagic.parent <  jfs->Fs1.Max)  &&
            (ino->fMagic.parent >= JFS_ROOT_INODE) )
   {
      TxCopy( fname, ino->fMagic.fname, DFS_EXT_ML);
      parIno       = ino->fMagic.parent;
   }
   */
   else                                         // Construct a unique fake name
   {
      if (S_ISDIR( ino->Mode))
      {
         sprintf( fname, "%c%12.12llX-%hX.DIR", FS_PATH_SEP, lsn, idx);
      }
      else                                      // must be a file ...
      {
         sprintf( fname, "-ino-%8.8x-at-%12.12llx-%hx-", selfIno, lsn, idx);
      }
      rc = FALSE;                               // name is faked, OK for SaveTo
   }
   if (parent)
   {
      *parent = parIno;
   }
   TRACES(( "Parent: %8.8x Name: '%s'\n", parIno, fname));
   BRETURN (rc);
}                                               // end 'dfsExtResolveInodeName'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display file allocation and path info for LSN
/*****************************************************************************/
ULONG dfsExtFileInfo                            // RET   LSN is valid INODE
(
   ULN64               inoLsn,                  // IN    Inode sectornumber
   ULN64               inoIdx,                  // IN    Inode index in sector
   char               *select,                  // IN    Inode select wildcard
   void               *param                    // INOUT leading text/shortpath
)
{
   ULONG               rc = NO_ERROR;
   TXLN                fpath;
   ULONG               root;                    // Root inode#
   ULN64               fsize     = 0;           // filesize in bytes
   ULN64               size      = 0;           // allocated size, sectors
   ULN64               bads      = 0;           // bad allocated sectors
   TXLN                text;
   ULONG               percent   = 100;
   ULONG               threshold = 0;           // threshold percentage
   BOOL                isMinimum = TRUE;        // threshold is minimum value
   ULN64               location  = 0;
   TXTT                temp;                    // location/size/date text
   ULONG               minS      = 0;           // minimal size
   ULONG               maxS      = 0;           // maximal size
   char                itemType  = 0;           // type Dir, File, Browse, or ANY
   time_t              modTstamp = 0;           // Modify timestamp value, or 0
   char               *lead    = param;
   BOOL                verbose = (strcmp( lead, ":") != 0);
   BOOL                output;                  // no Printf, just return fpath
   BOOL                alcheck;                 // Execute a full allocation check
   ULONG               ino = dfsExtLsnIndex2Inode( inoLsn, inoIdx);
   S_EXT_INODE        *inode;
   ULONG               MtExtra = 0;             // Modification time, extra bits
   BYTE                st = 0;

   ENTER();
   TRARGS(("LSN: 0x%llX, index: %hX  select: '%s', lead: '%s'\n", inoLsn, inoIdx, select, lead));

   if (strcmp( lead, "!") == 0)                 // Skip allocation check wanted
   {
      output  = FALSE;                          // no output
      alcheck = FALSE;                          // and no allocation check
      *lead   = 0;                              // discard 'garbage' in lead string
   }
   else                                         // Possible output/alloc-check wanted
   {
      output  = (*lead != 0);                   // output if any lead given (except '!')
      alcheck = TRUE;                           // but always an alloc check if no  '!'
   }
   TRACES(("verbose:%u output:%u alcheck:%u\n", verbose, output, alcheck));

   if ((ino > 1) && (ino <= ext->InodeMax) && (inoIdx & DFSSNINFO)) // Inode with valid index
   {
      if ((inode = TxAlloc( 1, ext->sup->InodeSize)) != NULL)
      {
         if ((rc = dfsExtGetInodeRecord( inoLsn, inoIdx & ~DFSSNINFO, inode)) == NO_ERROR)
         {
            st = (S_ISDIR( inode->Mode)) ? ST_EXTDIN : ST_EXTINO;

            dfsParseFileSelection( select, text, &isMinimum, &threshold, &minS, &maxS, &modTstamp, &itemType);
            if (verbose)
            {
               //- Allow skipping the allocation-check when not needed (Speedup BROWSE display)
               rc = dfsExtInodeCheckAlloc( inode, (alcheck) ? "" : "s", &fsize, &size, &bads, &location);
            }
            else                                // no size/percentage info
            {
               isMinimum = TRUE; threshold = 0; // no percentage filter
               modTstamp = 0;                   // no timestamp filter
               minS      = 0;    maxS      = 0; // no size filter
            }
            switch (st)
            {
               case ST_EXTDIN:                   // normal dir
                  if      (itemType == DFS_FS_ITEM_BROWSE) // no filtering on Directories, take all.
                  {
                     strcpy( text, "");         // no wildcard
                     isMinimum = TRUE; threshold = 0; // no percentage filter
                     modTstamp = 0;             // no timestamp filter
                     minS      = 0;    maxS      = 0; // no size filter
                  }
                  else if (itemType == DFS_FS_ITEM_FILES) // filter files, discard Directories
                  {
                     rc = DFS_ST_MISMATCH;      // item-type mismatch
                     break;
                  }
               case ST_EXTINO:                   // normal file
                  if      (itemType == DFS_FS_ITEM_DIRS) // filter dirs, discard files
                  {
                     rc = DFS_ST_MISMATCH;      // item-type mismatch
                     break;
                  }
                  dfsExtFindRootInode( dfsExtLsnIndex2Inode( inoLsn, inoIdx), fpath, &root);
                  if ((strlen(text) == 0) || (TxStrWicmp( fpath, text) >= 0))
                  {
                     if (verbose)               // include alloc and size
                     {
                        if (location == 0)
                        {
                           if (fsize == 0)
                           {
                              strcpy(  temp, "     empty ");
                              rc = NO_ERROR;    // allocation always OK
                           }
                           else
                           {
                              strcpy(  temp, "    inline ");
                              rc = NO_ERROR;    // allocation always OK
                           }
                        }
                        else
                        {
                           sprintf( temp, "%10llX ", location);
                        }
                        percent = dfsAllocationReliability( size, bads);

                        TRACES(( "percent:%u threshold:%u size:%u minS:%u maxS:%u\n",
                                  percent, threshold, size, minS, maxS));
                     }
                     if ((BOOL)((percent >= threshold)) == isMinimum)
                     {
                        if ((inode->Mtime > modTstamp) &&                        //- Timestamp match
                            ((size >= minS) && ((size <= maxS) || (maxS == 0)))) //- Size match
                        {
                           if (verbose)         // include alloc and size
                           {
                              if (alcheck)
                              {
                                 sprintf( text, "%s%c %s%3u%%%s ", CBM, st, (rc == NO_ERROR) ? CBG : CBR, percent, CNN);
                              }
                              else              // no reliability percentage
                              {
                                 sprintf( text, "%s%c%s      ", CBM, st, CNN);
                              }
                              strcat(     text, temp);

                              if ((size == 0) && (fsize <= 60)) // report inlined size in bytes
                              {
                                 sprintf( temp, "%4llu   B ", fsize);
                              }
                              else              // report size in sectors, KiB..TiB
                              {
                                 strcpy( temp, "");
                                 dfstrSize( temp, CBC, size, " ");
                              }
                              dfstrBytes(temp, CBZ, 0,    " "); // XATTR/EA-size
                              strcat( text, temp);
                           }
                           else
                           {
                              strcpy( text, ""); // remove any stray text, non-verbose
                           }
                           if (output)          // not totally silent?
                           {
                              strcat( lead, text);
                              TxPrint( "%s%s%s%s\n", lead, CBY, fpath, CNN, CGE);
                           }
                           else                 // return info in 'select'
                           {
                              if (ext->sup->InodeSize > 128) // extra fields are available
                              {
                                 MtExtra = inode->MtExtra;
                              }
                              dfsExtTime2str( inode->Mtime, MtExtra, temp);
                              TxStripAnsiCodes( text);
                              sprintf( select, "%s %s",        text, temp);
                              dfstrUllDot20( select, "", TXmku64( inode->SizeLo, inode->SizeHi), "");
                           }
                           rc = NO_ERROR;
                        }
                        else
                        {
                           rc = DFS_ST_MISMATCH; // file-size mismatch
                        }
                     }
                     else
                     {
                        rc = DFS_ST_MISMATCH;   // percentage mismatch
                     }
                  }
                  else
                  {
                     rc = DFS_ST_MISMATCH;      // wildcard mismatch
                  }
                  strcpy( param, fpath);        // return the found name
                  break;

               default:
                  rc = DFS_PENDING;
                  break;
            }
         }
         TxFreeMem( inode);
      }
      else
      {
         rc = DFS_BAD_STRUCTURE;
      }
   }
   else
   {
      rc = DFS_PENDING;                         // not an inode
   }
   RETURN(rc);
}                                               // end 'dfsExtFileInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Replace sn-list by contents of a single Directory (InoLsn, InoIdx pairs)
/*****************************************************************************/
ULONG dfsExtMakeBrowseList                      // RET   LSN is valid INODE
(
   ULN64               inoLsn,                  // IN    Inode sectornumber
   ULN64               inoIdx,                  // IN    Inode index in sector
   char               *str,                     // IN    unused
   void               *param                    // INOUT unused
)
{
   ULONG               rc = NO_ERROR;
   S_EXT_INODE        *inode;
   DFSISPACE           is;
   ULONG               ino = dfsExtLsnIndex2Inode( inoLsn, inoIdx);

   ENTER();
   TRARGS(("LSN: 0x%llX, index: %hX\n", inoLsn, inoIdx));

   if ((inode = TxAlloc( 1, ext->sup->InodeSize)) != NULL)
   {
      if ((rc = dfsExtGetInodeRecord( inoLsn,  inoIdx & ~DFSSNINFO, inode)) == NO_ERROR)
      {
         if (S_ISDIR( inode->Mode))
         {
            rc = dfsExtInodeData2Space( inode, &is, NULL, NULL);
            if (rc == NO_ERROR)
            {
               rc = dfsExtDirSpace2List((ino != EXT_I_ROOT), &is);
            }
            TxFreeMem( is.space);               // free SPACE
         }
         else                                   // not a directory
         {
            rc = DFS_BAD_STRUCTURE;
         }
      }
      TxFreeMem( inode);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN(rc);
}                                               // end 'dfsExtMakeBrowseList'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get allocation information for file-DATA into integrated-SPACE structure
/*****************************************************************************/
ULONG dfsExtGetAllocSpace
(
   ULN64               inoLsn,                  // IN    Inode sectornumber
   ULN64               inoIdx,                  // IN    Inode index in sector
   char               *str,                     // IN    unused
   void               *param                    // INOUT Integrated SPACE
)
{
   ULONG               rc = NO_ERROR;
   S_EXT_INODE        *inode;
   DFSISPACE          *ispace = (DFSISPACE *) param;

   ENTER();

   if ((inode = TxAlloc( 1, ext->sup->InodeSize)) != NULL)
   {
      if ((rc = dfsExtGetInodeRecord( inoLsn,  inoIdx & ~DFSSNINFO, inode)) == NO_ERROR)
      {
         rc = dfsExtInodeData2Space( inode, ispace, NULL, NULL);
      }
      TxFreeMem( inode);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsExtGetAllocSpace'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Put contents of a Directory SSPACE (InoLsn, InoIdx pairs) into sn-list
/*****************************************************************************/
static ULONG dfsExtDirSpace2List
(
   BOOL                notRoot,                 // IN    Dir is NOT the ROOT
   DFSISPACE          *dirData                  // IN    Directory blocks
)
{
   ULONG               rc = NO_ERROR;
   ULONG               totalDirSects;           // sectors in SSPACE
   ULONG               done = 0;                // sectors handled
   BYTE               *dirBlock;                // one block of DIR data
   S_EXT_DIRENTRY     *de;                      // single directory slot
   ULONG               offset = 0;              // byte offset into block
   USHORT              inoIdx;
   ULONG               inoLsn;

   ENTER();

   if ((dirBlock = TxAlloc( 1, ext->BlockSize)) != NULL)
   {
      dfsInitList(0, "-f -P", "-d");            // optimal for menu file-recovery
      totalDirSects = dfsSspaceSectors( TRUE, dirData->chunks, dirData->space);

      for (done = 0; done < totalDirSects; done += ext->SectorsPerBlock)
      {
         rc = dfsSspaceReadFilePart( dirData->chunks, dirData->space,
                                     done, ext->SectorsPerBlock, dirBlock);

         while ((rc == FALSE) && (offset < ext->BlockSize) && !TxAbort())
         {
            de = (S_EXT_DIRENTRY *) ((BYTE *) dirBlock + offset);

            if (de->Inode == 0)                 // unused entry, end of DIR
            {
               if (de->RecLen == 0)
               {
                  //- should not happen if all RecLen values are correct
                  TRACES(("Aborting dirwalk on de->Inode 0 and RecLen 0\n"));
                  done = totalDirSects;         // abort SSPACE-block loop too
                  break;                        // end the loop
               }
            }
            else if ((de->NameLen > 1) || (de->Name[0] != '.'))  //- always skip '.' entry
            {                                                    //- and skip '..' for ROOT
               if ((notRoot) || (de->Name[0] != '.') || (de->Name[1] != '.'))
               {
                  BOOL          filtered = FALSE;

                  if ((dfsa->snlist[0] == 0) && (de->Name[0] == '.') && (de->Name[1] == '.'))
                  {
                     TRACES(("List marked as 1ST_PARENT\n"));
                     DFSBR_SnlistFlag |= DFSBR_1ST_PARENT;
                  }
                  else                          // regular entry, filter on DOT names when needed
                  {
                     if (dfsa->browseShowHidden == FALSE) // need to check hidden/system attribute
                     {
                        if (de->Name[0] == '.') // filename starts with DOT, hidden (UNIX like)
                        {
                           filtered = TRUE;
                        }
                     }
                  }
                  if (!filtered)
                  {
                     inoLsn = dfsExtInode2LsnIndex( de->Inode, &inoIdx);
                     rc = dfsAddSI2List( inoLsn, inoIdx);
                  }
               }
            }
            offset += de->RecLen;               // to next DIR entry, or end
         }
      }
      TxFreeMem( dirBlock);
   }
   RETURN (rc);
}                                               // end 'dfsExtDirSpace2List'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate Block-nr to LSN, generic interface, Area aware
/*****************************************************************************/
ULN64 dfsExtCl2Lsn                              // RET   LSN for this block
(
   ULN64               block,                   // IN    block number
   ULN64               d2,                      // IN    dummy
   char               *p1,                      // IN    dummy
   void               *p2                       // IN    dummy
)
{
   ULN64               lsn = block;

   lsn = dfstAreaP2Disk( DFSTORE, (block * ext->SectorsPerBlock));
   return (lsn);
}                                               // end 'dfsExtCl2Lsn'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate LSN to Block-nr, generic interface, Area aware
/*****************************************************************************/
ULN64 dfsExtLsn2Cl                              // RET   block for this LSN
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *p1,                      // IN    dummy
   void               *p2                       // IN    dummy
)
{
   ULN64               block = 0;

   block = dfstAreaD2Part( DFSTORE, lsn) / ext->SectorsPerBlock;
   return (block);
}                                               // end 'dfsExtLsn2Cl'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Determine allocation-bit for specified LSN, ALLOCATED beyond last block!
/*****************************************************************************/
ULONG dfsExtAllocated                           // RET   LSN is allocated
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               asn = dfstAreaD2Part( DFSTORE, lsn); // Area aware sn

   if (ext->Bm.Cache != NULL)                   // cache present
   {
      if (asn < (ext->Sect))                    // within valid block area
      {
         //- to be refined, reenable when EXT_FIN_META_BG is understood and supported
         if (ext->sup->FeatureInComp & EXT_FIN_META_BG) // unknown/unsupported now!
         {
            return( 1);                         // always allocated!
         }                                      // overruling 'SMART'
         else
         {
            return((ULONG) dfsExtBlkBitmapCache( dfsExtLsn2Block( asn), NULL));
         }
      }
      else                                      // return ALLOCATED for any
      {                                         // lsn beyond last block
         return( 1);                            // to avoid CHECK errors
      }                                         // on non-standard bitmaps
   }
   else
   {
      return(DFS_PSN_LIMIT);
   }
}                                               // end 'dfsExtAllocated'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Align R/O cache to position for Inode, and get current allocation-bit
/*****************************************************************************/
BOOL dfsExtInoBitmapCache                       // RET   Inode is allocated
(
   ULONG               ino                      // IN    Inode number
)
{
   ULONG               rc = NO_ERROR;
   BOOL                al = TRUE;               // default allocated
   ULONG               group;                   // Group number for block
   ULONG               ImapSect;                // sectornr in Inode Bitmap
   ULONG               bytepos = L32_NULL;      // byte position in map
   ULONG               alloc_byte;
   S_EXT_GROUPDESCR   *gd;                      // group descriptor structure

   if (ext->sup->InodesPerGroup)                // avoid divide by zero
   {
      group = ino / ext->sup->InodesPerGroup;    // Group number for block
      gd    = &(ext->GroupDescTable[group]);

      //- when bitmap not there (yet), there are NO Inodes in use at all
      if (gd->bgFlags & EXT_BGF_INODE_UNINIT)   // bitmap not there
      {
         al = FALSE;                            // inodes ALL free
      }
      else                                      // bitmap is there, use that
      {
         if (group != ext->Im.Group)            // other block group
         {
            TRACES(("ExtInodeAlloc: Read bitmap cache for group %u\n",   group));
            ext->Im.Group   = group;
            ImapSect = ext->SectorsPerBlock * gd->InodeBitmap;
            rc = dfsRead( ImapSect, ext->Im.Sects, ext->Im.Cache);
         }
         if (rc == NO_ERROR)
         {
            bytepos    = (ino % ext->sup->InodesPerGroup) >> 3; // byte index in map
            alloc_byte = ext->Im.Cache[ bytepos];
            if (((alloc_byte >> (ino & 0x07)) & 0x01) == 0) // LSB is lowest!
            {
               al = FALSE;
            }
         }
      }
   }
   return (al);
}                                               // end 'dfsExtInoBitmapCache'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create EXT2/3/4 Bitmap cache structures and initialize the InoBitMap cache
/*****************************************************************************/
ULONG dfsExtInoBitMapInit                       // RET   rc = 0 if type match
(
   void
)
{
   ULONG               rc = 0;                  // rc, sector match

   ENTER();

   ext->Im.Sects = ((ext->sup->InodesPerGroup -1) / (8 * dfsGetSectorSize())) +1;

   if ((ext->Im.Cache = TxAlloc( ext->Im.Sects, dfsGetSectorSize())) != NULL)
   {
      ext->Im.Group = DFSEXT_NOGROUP;           // nothing cached yet
      ext->Im.Dirty = FALSE;
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsExtInoBitMapInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Flush Inode BitMap cache when Dirty, and optional free its resources
/*****************************************************************************/
ULONG dfsExtInoBitmapFlush                      // RET   rc
(
   BOOL                terminate                // IN    terminate cache too
)
{
   ULONG               rc = NO_ERROR;           // rc

   ENTER();

   if (ext->Im.Dirty)                           // need to flush changes ?
   {
      TRACES(( "dirty, flush it ...\n"));

      //- Flushing to be refined
      ext->Im.Dirty = FALSE;                    // mark it clean again
   }
   else
   {
      TRACES(( "not dirty ...\n"));
   }
   if (terminate)
   {
      TxFreeMem( ext->Im.Cache);

      ext->Im.Group = DFSEXT_NOGROUP;
   }
   RETURN (rc);
}                                               // end 'dfsExtInoBitmapFlush'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Align R/W cache to position for block, and get current allocation-bit
// Note: Area aware to allow usage from FDISK mode too
/*****************************************************************************/
BOOL dfsExtBlkBitmapCache                       // RET   block is allocated
(
   ULONG               bl,                      // IN    Block number
   ULONG              *cp                       // OUT   position in cache
)                                               //       or L32_NULL if invalid
{
   ULONG               rc = NO_ERROR;
   BOOL                al = TRUE;               // default allocated
   ULONG               group;                   // Group number for block
   ULONG               BmapSect;                // sectornr in Block Bitmap
   ULONG               bytepos = L32_NULL;      // byte position in map
   ULONG               alloc_byte;
   S_EXT_GROUPDESCR   *gd;                      // group descriptor structure

   //- to be refined, make sure ALL fields are available, in AreaInit mode too!
   if ((ext->sup != NULL) && (ext->sup->BlocksPerGroup)) // avoid NULL ptr and divide by zero
   {
      group = bl / ext->sup->BlocksPerGroup;    // Group number for block
      gd    = &(ext->GroupDescTable[group]);

      //- when bitmap not there (yet), only SB, GDT and GdtReserved are allocated
      if (gd->bgFlags & EXT_BGF_BLOCK_UNINIT)   // bitmap not there
      {
         if (dfsExtIsBackupSuperBG( group))     // SB+GDT backup group
         {
            ULONG     firstBlock = group * ext->sup->BlocksPerGroup;

            if (bl > (firstBlock + 1 + ext->GdtBlocks + ext->GdtReservedBlocks))
            {
               al = FALSE;                      // calculated as free
            }
         }
         else                                   // not a SB+GDT backup group
         {
            al = FALSE;                         // non backup are ALL free
         }
      }
      else                                      // bitmap is there, use that
      {
         if (group != ext->Bm.Group)            // other block group
         {
            if (ext->Bm.Dirty)                  // need to flush changes ?
            {
               rc = dfsExtBlkBitmapFlush( FALSE); // flush, but keep cache
            }
            if (rc == NO_ERROR)
            {
               TRACES(("ExtBlkBitmapCache: Read bitmap cache for group %u\n",   group));
               ext->Bm.Group   = group;
               BmapSect = ext->SectorsPerBlock * gd->BlockBitmap;
               rc = dfsRead( dfstAreaP2Disk( DFSTORE, BmapSect), ext->Bm.Sects, ext->Bm.Cache);
            }
         }
         if (rc == NO_ERROR)
         {
            bytepos    = (bl % ext->sup->BlocksPerGroup) >> 3; // byte index in map
            alloc_byte = ext->Bm.Cache[ bytepos];
            if (((alloc_byte >> (bl & 0x07)) & 0x01) == 0) // LSB is lowest!
            {
               al = FALSE;
            }
            TRLEVX(150,("ExtBlkBitmapCache bytepos: %8.8x al:%s\n", bytepos, (al) ? "Yes" : "No"));
         }
         if (cp != NULL)                        // return byte position too ?
         {
            *cp = bytepos;
         }
      }
   }
   return (al);
}                                               // end 'dfsExtBlkBitmapCache'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create EXT2/3/4 Bitmap cache structures and initialize the BlkBitMap cache
// Note: Area aware to allow usage from FDISK mode too
/*****************************************************************************/
ULONG dfsExtBlkBitMapInit                       // RET   rc = 0 if type match
(
   void
)
{
   ULONG               rc = 0;                  // rc, sector match

   ENTER();

   ext->Bm.Sects = ((ext->sup->BlocksPerGroup -1) / (8 * dfsGetSectorSize())) +1;

   if ((ext->Bm.Cache = TxAlloc( ext->Bm.Sects, dfsGetSectorSize())) != NULL)
   {
      ext->Bm.Group = DFSEXT_NOGROUP;           // nothing cached yet
      ext->Bm.Dirty = FALSE;
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsExtBlkBitMapInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Flush Block BitMap cache when Dirty, and optional free its resources
/*****************************************************************************/
ULONG dfsExtBlkBitmapFlush                      // RET   rc
(
   BOOL                terminate                // IN    terminate cache too
)
{
   ULONG               rc = NO_ERROR;           // rc

   ENTER();

   if (ext->Bm.Dirty)                           // need to flush changes ?
   {
      TRACES(( "dirty, flush it ...\n"));

      //- Flushing to be refined
      ext->Bm.Dirty = FALSE;                    // mark it clean again
   }
   else
   {
      TRACES(( "not dirty ...\n"));
   }
   if (terminate)
   {
      TxFreeMem( ext->Bm.Cache);

      ext->Bm.Group = DFSEXT_NOGROUP;
   }
   RETURN (rc);
}                                               // end 'dfsExtBlkBitmapFlush'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Initialize EXT Group Descriptor array cache from first group, area aware
/*****************************************************************************/
ULONG dfsExtGroupDescriptors
(
   ULONG               verbose                  // IN    Display verbosity
)
{
   ULONG               rc = NO_ERROR;
   ULONG               gdSectors;               // size in sectors
   ULONG               gdFirstSect = ext->SectorsPerBlock; // 2nd block, is block 1
   ULONG               Directories = 0;
   ULN64               FreeBlocks  = 0;
   ULONG               InodeFree   = 0;
   ULONG               InodeUsed   = 0;
   ULONG               i;

   ENTER();

   if (ext->BlockSize <= 1024)                  // superblock in own block, GDT moves up!
   {
      gdFirstSect += ext->SectorsPerBlock;      // 3rd block, is block 2
   }
   ext->Groups     = ((ext->Block  -1) / ext->sup->BlocksPerGroup) +1;
   gdSectors       = ((ext->Groups -1) / (dfsGetSectorSize() / ext->GroupDescrSize)) +1;

   ext->GdtBlocks  = ((gdSectors   -1) / ext->SectorsPerBlock) +1;

   TRACES(( "%u-byte descriptors in %u sectors starting at: %u for %u Groups; GdtBlocks: %u\n",
             ext->GroupDescrSize, gdSectors, gdFirstSect, ext->Groups, ext->GdtBlocks));

   if ((ext->GroupDescTable = TxAlloc( gdSectors, dfsGetSectorSize())) != NULL)
   {
      if ((rc = dfsRead( dfstAreaP2Disk( DFSTORE, gdFirstSect), gdSectors, (BYTE *) ext->GroupDescTable)) == NO_ERROR)
      {
         if (verbose & TXAO_VERBOSE)
         {
            TxPrint( " Group FreeInodes Bbm Blk# Ibm Blk# Itb Blk# 1stInode FreeBlocks    #Dirs Flag\n");
            TxPrint( " ===== ========== ======== ======== ======== ======== ========== ======== ====\n");
         }
         for (i = 0; i < ext->Groups; i++)      // iterate over groups
         {
            if ((ext->GroupDescTable[i].BlockBitmap != 0) &&
                (ext->GroupDescTable[i].InodeBitmap != 0) &&
                (ext->GroupDescTable[i].InodeTable  != 0) &&
                (ext->GroupDescTable[i].FreeBlocks  <= ext->sup->BlocksPerGroup) &&
                (ext->GroupDescTable[i].FreeInodes  <= ext->sup->InodesPerGroup)  ) // BG sanity check
            {
               Directories +=                             ext->GroupDescTable[i].Directories;
               FreeBlocks  +=                             ext->GroupDescTable[i].FreeBlocks;
               InodeUsed   += (ext->sup->InodesPerGroup - ext->GroupDescTable[i].FreeInodes);
               InodeFree   +=                             ext->GroupDescTable[i].FreeInodes;
            }
            else
            {
               TRACES(("Invalid descriptor for group: %u\n", i));
            }
            if (verbose & TXAO_VERBOSE)
            {
               //- to be refined for 64-bit blockgroups, combine hi+lo value pairs before display
               TxPrint( "%s%c%s%5u %10hu %8.8x %8.8x %8.8x %8.8x %10hu %8hu 0x%2.2hx\n",
                           CBC, (dfsExtIsBackupSuperBG( i)) ? 'S' : ' ', CNN, i,
                                                  ext->GroupDescTable[i].FreeInodes,
                                                  ext->GroupDescTable[i].BlockBitmap,
                                                  ext->GroupDescTable[i].InodeBitmap,
                                                  ext->GroupDescTable[i].InodeTable,
                   i * ext->sup->InodesPerGroup,  ext->GroupDescTable[i].FreeBlocks,
                                                  ext->GroupDescTable[i].Directories,
                                                  ext->GroupDescTable[i].bgFlags);
            }
         }

         if (verbose > TXAO_QUIET)
         {
            if (verbose & TXAO_VERBOSE)
            {
               TxPrint( " ----- ==========+    used:                           ==========+ =======+\n");
            }
            else
            {
               TxPrint( "Groups FreeInodes UsedInodes   Range of valid Inode#  FreeBlocks    #Dirs\n");
               TxPrint( "====== ========== ==========   =====================  ========== ========\n");

            }
            TxPrint(    "% 6u %10u %10u   Inodes 0 - 0x%8.8x  %10u % 8hu\n\n",
                      ext->Groups, InodeFree, InodeUsed,
                      ext->sup->Inodes -1, FreeBlocks, Directories);

            dfsSz64( "Free #sectors  FS : ",   FreeBlocks * ext->SectorsPerBlock,  "  ");
            dfsSz64(             "total : ",                ext->Sect,       "\n");
            dfsSz64( "Used #sectors  FS : ",                ext->Sect -
                                              (FreeBlocks * ext->SectorsPerBlock), "  ");
            TxPrint(             "equals % 3.1lf%%\n",       (double) 100.0 *
                       ((double) (ext->Block - FreeBlocks) / (double) ext->Block));
         }
         TRHEXS( 500,  ext->GroupDescTable, gdSectors * dfsGetSectorSize(), "EXTn GroupDescTable");

         ext->FreeSect = FreeBlocks * ext->SectorsPerBlock; // Used for SLT build progress

         ext->InodeTableBlocks  = ((ext->sup->InodesPerGroup -1) / (ext->BlockSize / ext->sup->InodeSize)) + 1;

         //- Relies on the BBM following the GDT-reserve directly in group 0    (which seems to be always)
         //- However, using the special Inode-7 allocation info would be more exact        (to be refined)
         ext->GdtReservedBlocks = ext->GroupDescTable[0].BlockBitmap - ext->GdtBlocks -1;
         if (ext->BlockSize <= 1024)            // 1 KiB blocks
         {
            ext->GdtReservedBlocks -= 1;        // adjust for separate BOOT and SUPER blocks in group 0
         }
         dfsa->FsSltRecSize = InodeUsed * 8 + 1000; // average 8 extents per file
         ext->InodeUsed     = InodeUsed;        // keep around for NP cache building
      }
      else
      {
         TxPrint("\nError reading the Group Descriptor Tables from sector 0x%8.8x\n", gdFirstSect);
      }
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);                                 // when needed
}                                               // end 'dfsExtGroupDescriptors'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display information and block-bitmap graphic for a single block-group
/*****************************************************************************/
ULONG dfsExtShowBgBitmap
(
   ULONG               group,                   // IN    Block Group number
   ULONG               verbosity,               // IN    0=silent 1=summary 2=bitmap
   ULN64              *usedBitmap,              // INOUT used blocks, from bitmap
   ULN64              *usedDescr,               // INOUT used blocks, from descr.
   ULN64              *usedDiff                 // INOUT descriptor/bitmap diff
)
{
   ULONG               rc = NO_ERROR;
   ULONG               i  = 0;                  // index in data-area
   ULONG               a  = 0;                  // index in ascii line
   ULONG               l  = 0;                  // line number, 0 based
   ULONG               lbase;                   // base block-number for line
   TX1K                ascii;                   // display-array
   S_EXT_GROUPDESCR   *gd;                      // group descriptor structure
   BYTE               *BlockBitMap;
   ULONG               firstBlock;              // Blocknr for first in this BM
   ULONG               thisBlock;               // temporary blocknumber
   ULONG               mapSize;                 // Blocks mapped in this BM
   ULONG               bits;                    // bits set for this character
   ULONG               used = 0;                // total blocks with bit set
   ULONG               descUsed   = 0;          // Blocks used, from descriptor
   LONG                difference = 0;          // diff between descr and bitmap
   static char         map[20] = " 123456789abcdef#";

   ENTER();

   if (group < ext->Groups)                     // Valid group number
   {
      gd              = &(ext->GroupDescTable[group]);
      firstBlock      = group * ext->sup->BlocksPerGroup;
      if ((firstBlock + ext->sup->BlocksPerGroup) <= ext->Block)
      {
         mapSize    = ext->BlockSize;           // #bytes in bitmap
         descUsed   = ext->sup->BlocksPerGroup - gd->FreeBlocks;
      }
      else                                      // use EXACT number of blocks!
      {
         mapSize    = (ext->Block - firstBlock) / 8;
         descUsed   = (ext->Block - firstBlock) - gd->FreeBlocks;
      }
      TRACES(("group: %u  flag:%hx  First:%x Size:%u  map:'%s'\n",
               group, gd->bgFlags, firstBlock, mapSize, map));

      if (verbosity >= TXAO_VERBOSE)            // include details and bitmap
      {
         TxPrint("\n%s%c%s  Block group %s%6u%s Free Blocks:%10hu   Used:%7u   Directories:%6hu\n",
                     CBC, (dfsExtIsBackupSuperBG( group)) ? 'S' : ' ', CNN,
                     CBY, group, CNN, gd->FreeBlocks, descUsed, gd->Directories);

         TxPrint( "BlockGroupDescriptor  Free Inodes:%10hu   Used:%5u   BGroupFlags:0x%4.4hx\n",
                   gd->FreeInodes,  ext->sup->InodesPerGroup - gd->FreeInodes,  gd->bgFlags);

         thisBlock = firstBlock + 1 + ext->GdtBlocks;
         TxPrint("GdtReserved Size:%4u blocks  @Bl:0x%8.8x = %10u = Sector: 0x%8.8x\n",
             ext->GdtReservedBlocks, thisBlock, thisBlock, ext->SectorsPerBlock * thisBlock);

         //- Note: BlockBitmap and InodeBitmap are 1 block in size, by definition!
         thisBlock = gd->BlockBitmap;
         TxPrint("BlockBitmap Size:   1 block   @Bl:0x%8.8x = %10u = Sector: 0x%8.8x\n",
                                     thisBlock, thisBlock, ext->SectorsPerBlock * thisBlock);

         thisBlock = gd->InodeBitmap;
         TxPrint("InodeBitmap Size:   1 block   @Bl:0x%8.8x = %10u = Sector: 0x%8.8x\n",
                                     thisBlock, thisBlock, ext->SectorsPerBlock * thisBlock);

         thisBlock = gd->InodeTable;
         TxPrint("Inode Table Size:%4u blocks  @Bl:0x%8.8x = %10u = Sector: 0x%8.8x\n",
             ext->InodeTableBlocks,  thisBlock, thisBlock, ext->SectorsPerBlock * thisBlock);

         TxPrint(  "        %s%s%s\n", CBC, BLIN, CNN);
      }
      if (gd->bgFlags & EXT_BGF_BLOCK_UNINIT) // bitmap not there
      {
         if (dfsExtIsBackupSuperBG( group))     // SB+GDT backup group
         {
            used = 1 + ext->GdtBlocks + ext->GdtReservedBlocks; // calculated usage
         }
         if (verbosity >= TXAO_VERBOSE)         // include details and bitmap
         {
            sprintf( ascii, "# Bitmap marked BLOCK_UNINIT. Reserved blocks calculated: %u", used);
            TxPrint("%08.08X%s%s%-64s%s%s\n", firstBlock, CBC, CNC, ascii, CBC, CNN);
            a = 64;                             // need full-line at bottom
         }
      }
      else
      {
         if ((BlockBitMap = TxAlloc( 1, ext->BlockSize)) != NULL)
         {
            rc = dfsRead( ext->SectorsPerBlock * gd->BlockBitmap,
                          ext->SectorsPerBlock,  BlockBitMap);
            if (rc == NO_ERROR)
            {
               for (i = 0, a = 0, l = 0; i < mapSize;)
               {
                  if (verbosity >= TXAO_VERBOSE) // include details and bitmap
                  {
                     if (a == 0)
                     {
                        memset(ascii, 0, TXMAXTM);
                     }
                     bits  = TxUlBitCount( (ULONG) (BlockBitMap[ i++]));
                     bits += TxUlBitCount( (ULONG) (BlockBitMap[ i++]));
                     used += bits;

                     ascii[a++] = map[ bits];   // one char represents 16 blocks

                     TRLEVX(200,("Bytes:%5u bits:%2u used:%5u ascii:'%s'\n", i, bits, used, ascii));

                     if ((a == 64) || (i >= mapSize)) // 64 chars done, or end of map
                     {
                        lbase = firstBlock + (l * 1024);
                        TxPrint("%08.08X%s%s%s", lbase, CBC, CNC, ascii);

                        if (a == 64)            // full line
                        {
                           TxPrint("%s%s\n", CBC, CNN);
                        }
                        else
                        {
                           TxPrint("%s%.*s%s\n", CBC, (int) (64-a-1), BLIN, CNN);
                        }
                        if (i < mapSize)
                        {
                           a = 0;               // keep a value when at end
                        }
                        l++;
                     }
                  }
                  else                          // just count the used bits in every byte
                  {
                     used += TxUlBitCount( (ULONG) (BlockBitMap[ i++]));
                  }
               }
            }
            TxFreeMem( BlockBitMap);
         }
         else
         {
            rc = DFS_ALLOC_ERROR;
         }
      }
      difference = ((LONG) used) - ((LONG) descUsed);
      if (verbosity > TXAO_SILENT)              // not silent
      {
         if (verbosity >= TXAO_VERBOSE)         // include details and bitmap
         {
            TxPrint("        %s%.*s%s\n", CBC, (USHORT) a, BLIN, CNN);
            TxPrint("   Block group %s%6u%s Free Blocks in Bitmap:%6u Used:%5u ",
                         CBY, group,     CNN, (mapSize * 8) - used,     used);
         }
         else
         {
            TxPrint( "BlockGroup %5u Flg: %s%c %s%c %s%c%c  Used Bitmap:%6u Descriptor:%6u ",
                      group,     CBC,  (dfsExtIsBackupSuperBG( group)) ? 'S' : ' ', CBY,
                                  (gd->bgFlags & EXT_BGF_BLOCK_UNINIT) ? 'B' : ' ', CNN,
                                  (gd->bgFlags & EXT_BGF_INODE_UNINIT) ? 'I' : ' ',
                                  (gd->bgFlags & EXT_BGF_INODE_ZEROED) ? 'z' : ' ', used, descUsed);
         }
         if (difference != 0)
         {
            TxPrint( "Difference %s%+6d%s\n", CBR, difference, CNN);
         }
         else
         {
            TxPrint( "%sOK%s\n", CBG, CNN);
         }
      }
   }
   else
   {
      TxPrint("\nBlock group #%5u exceeds FS size of %u groups.\n", group, ext->Groups);
   }
   if (usedBitmap)                              // add blocks used
   {
      *usedBitmap += used;
   }
   if (usedDescr)                               // add blocks used
   {
      *usedDescr  += descUsed;
   }
   if (usedDiff)                                // add ABSOLUTE difference
   {
      if      (difference > 0)
      {
         *usedDiff   += difference;
      }
      else if (difference < 0)
      {
         *usedDiff   -= difference;
      }
   }
   RETURN (rc);
}                                               // end 'dfsExtShowBgBitmap'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Write cached superblock to 1st block-group superblock location on disk
// At LSN 2 for 512 BPS, 1 for 1024, and 0 for 2048/4096 (at offset 1024)
/*****************************************************************************/
ULONG dfsExtWriteSuperBlocks
(
   void
)
{
   ULONG               rc  = NO_ERROR;

   ENTER();

   // To be refined to write to other (sparse) superblock locations too ?
   if (DFSTORE_WRITE_ALLOWED)
   {
      if (dfsGetSectorSize() > 1024)            // SUPER will be at offset 1024
      {
         BYTE               *buffer;
         TxPrint("Superblock update : 1st one at 0x%2.2x, offset 1024 bytes: ", LSN_EXTSUP);

         if ((buffer = TxAlloc( 1, 4096)) != NULL)            // max 4096 BPS
         {
            if (dfsRead( LSN_EXTSUP, 1, buffer) == NO_ERROR)  //- Read existing data
            {                                                 //- Modify (merge super)
               memcpy( buffer + 1024, (BYTE *) ext->sup, sizeof( S_EXT_SUPER));
               rc = dfsWrite(   LSN_EXTSUP, 1, buffer);       //- Write back
            }
            TxFreeMem( buffer);
         }
         else
         {
            rc = DFS_ALLOC_ERROR;
         }
      }
      else
      {
         TxPrint("Superblock update : 1st one at 0x%2.2x, whole sector: ", LSN_EXTSUP);
         rc = dfsWrite(   LSN_EXTSUP, 1, (BYTE *) ext->sup);
      }
      TxPrint( "%s\n", (rc == NO_ERROR) ? "written" : "failed");
   }
   else
   {
      rc = DFS_READ_ONLY;
   }
   RETURN(rc);
}                                               // end 'dfsExtWriteSuperBlocks'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Quickly find normal (non-deleted) Inodes using the Inode MAP
/*****************************************************************************/
ULONG dfsExtQuickFindInodes
(
   BOOL                verbose,                 // IN    List files while found
   ULONG               first,                   // IN    First inode todo
   char               *types,                   // IN    Inode types to search D/f
   char               *select                   // IN    Selection string for
)                                               //       resolved name (if any)
{
   ULONG               rc = NO_ERROR;           // function return
   BYTE                st;                      // File or Directory inode
   S_EXT_INODE        *inode;
   ULONG               ino;
   ULN64               inoLsn;                  // Inode LSN
   USHORT              inoIdx;                  // Inode index
   BOOL                match;
   TXLN                text;                    // name / result text buffer
   ULONG               ei = 0;                  // entries listed

   ENTER();

   if ((inode = TxAlloc( 1, ext->sup->InodeSize)) != NULL)
   {
      dfsInitList(0, "-f -P", "-d");            // optimal for menu file-recovery
      dfsProgressInit( 0, ext->InodeMax, 0, " Inode:", "searched", (verbose) ? DFSP_STAT : DFSP_BARS, 0);

      for (ino = first; ino <= ext->InodeMax; ino++)
      {
         if (dfsExtInoBitmapCache( ino))        // Inode number in-use ?
         {
            inoLsn = dfsExtInode2LsnIndex( ino, &inoIdx);
            if ((inoLsn != L64_NULL) && (inoLsn != 0))
            {
               if ((rc = dfsExtGetInodeRecord( inoLsn, inoIdx, inode)) == NO_ERROR)
               {
                  st = (S_ISDIR( inode->Mode)) ? 'D' : 'f';
                  if (strchr( types, st) != NULL)
                  {
                     if (select && strlen(select))
                     {
                        dfsExtResolveInodeName( inoLsn, inoIdx, inode, NULL, text);
                        match = (strstr( text, select) != NULL);
                     }
                     else                       // no name selection
                     {
                        match = TRUE;
                     }
                     if (match)
                     {
                        dfsAddSI2List( inoLsn, inoIdx);

                        if (verbose)            // list all 100% alloc OK
                        {
                           sprintf( text, ".%5.5u = %12.12llX^%hx : ",  ei++, inoLsn, inoIdx);
                           dfsExtFileInfo( inoLsn, inoIdx, "*%100", text);
                        }
                     }
                  }
               }
               else if (rc == DFS_ST_MISMATCH)  // ignore sector type mismatch
               {
                  rc = NO_ERROR;
               }
               else
               {
                  break;
               }
            }
         }
         if ((ino & 0xff) == 0)                 // every 256, unallocated too
         {
            if (TxAbort())
            {
               break;
            }
            else
            {
               dfsProgressShow( ino, 0, dfsa->snlist[0], "Found:");
            }
         }
      }
      dfsProgressTerm();
      sprintf( text, ", found %llu Inodes\n", dfsa->snlist[0]);
      dfsProgressElapsed( "Elapsed time      : ", text);
      TxFreeMem( inode);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsExtQuickFindInodes'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Convert Inode allocation info (Iblocks/Extents) to an S_SPACE, incl. meta
/*****************************************************************************/
ULONG dfsExtInodeData2Space
(
   S_EXT_INODE        *inode,                   // IN    Inode with alloc info
   DFSISPACE          *is,                      // OUT   Integrated S_SPACE, data
   DFSISPACE          *m1,                      // OUT   meta data 1, L1 or Leaf
   DFSISPACE          *m2                       // OUT   meta data 2, L2 or Index
)
{
   ULONG               rc = NO_ERROR;           // function return
   ULN64               Blocks = 0;              // nr of blocks needed, limit

   ENTER();

   memset( is,     0, sizeof(DFSISPACE));       // initialize to all zero

   if ((inode->BlocksLo != 0) || (inode->BlocksHi != 0)) // not inlined (symlink)
   {
      is->clsize = ext->SectorsPerBlock;

      if ((m1 != NULL) && (m2 != NULL))
      {
         memset( m1, 0, sizeof(DFSISPACE));     // initialize to all zero
         memset( m2, 0, sizeof(DFSISPACE));     // initialize to all zero
      }

      //- Note: inode->BlocksLo+Hi can be too large for really allocated space!
      //-       as observed with a 600Kb file on EXT2, so use filesize instead.
      Blocks = ((TXmku64( inode->SizeLo, inode->SizeHi) -1) / ext->BlockSize) + 1;

      if (Blocks != 0)
      {
         //- first test for the special Inode-7 allocation (RESIZE info), 1 block
         if ((inode->Alloc.gdt.Block[0]  == 0) &&
             (inode->Alloc.gdt.Block[1]  == 0) &&
             (inode->Alloc.gdt.Block[2]  == 0) &&
             (inode->Alloc.gdt.Block[3]  == 0) &&
             (inode->Alloc.gdt.Reserved1 == 0) &&
             (inode->Alloc.gdt.infoBlock != 0) && // single Block of RESIZE-info
             (inode->Alloc.gdt.Reserved2 == 0)  )
         {
            S_SPACE   *sp = TxAlloc( 1, sizeof(S_SPACE));

            if (sp != NULL)
            {
               sp->size   = ext->SectorsPerBlock;
               sp->start  = dfsExtBlock2Lsn( inode->Alloc.gdt.infoBlock);

               is->chunks = 1;
               is->space  = sp;
            }
         }
         else if (inode->Flags & EXT_FL_EXTENTS) // inode uses EXTENTS for allocation
         {
            if ((m1 != NULL) && (m2 != NULL))
            {
               m1->spid = ST_EXTALF;
               m2->spid = ST_EXTAIX;
            }
            rc = dfsExtAnyNode2Space( &inode->Alloc.ex, &Blocks, is, m1, m2);
         }
         else                                   // inode uses classic IBLOCKS
         {
            if ((m1 != NULL) && (m2 != NULL))
            {
               m1->spid = ST_EXTBI1;
               m2->spid = ST_EXTBI2;
            }
            rc = dfsExtIblocks2Space( &inode->Alloc.bi, &Blocks, is, m1, m2);
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsExtInodeData2Space'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Convert allocation LEAF/INDEX NODE structure into an S_SPACE (recursive)
/*****************************************************************************/
static ULONG dfsExtAnyNode2Space
(
   S_EXT_EXTNODE      *anyNode,                 // IN    Leaf or Index node
   ULN64              *blksTodo,                // INOUT maximum blocks todo
   DFSISPACE          *is,                      // INOUT Integrated S_SPACE, data
   DFSISPACE          *m1,                      // INOUT meta data 1, L1 or Leaf
   DFSISPACE          *m2                       // INOUT meta data 2, L2 or Index
)
{
   ULONG               rc = NO_ERROR;           // function return
   DFSISPACE           isNew;                   // data space to be added

   ENTER();

   TRACES(("hdr.Magic: %4.4hx  hdr.Depth: %hhu  entries: %u\n",
            anyNode->hdr.ehMagic, anyNode->hdr.ehDepth, anyNode->hdr.ehEntries));


   if (anyNode->hdr.ehDepth == 0)               // it is a leaf node
   {
      memset( &isNew, 0, sizeof(DFSISPACE));    // initialize to all zero

      rc = dfsExtLeafNode2Space( anyNode, blksTodo, &isNew.chunks, &isNew.space);
      if (rc == NO_ERROR)
      {                                         // join file data stream
         rc = dfsSspaceJoin( is->chunks,     is->space,
                             isNew.chunks,   isNew.space,
                            &is->chunks,     &is->space);
      }
   }
   else
   {
      rc = dfsExtIndexNode2Space( anyNode, blksTodo, is, m1, m2);
   }
   RETURN (rc);
}                                               // end 'dfsExtAnyNode2Space'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Handle allocation INDEX NODE, read and process each block#-entry as AnyNode
/*****************************************************************************/
static ULONG dfsExtIndexNode2Space
(
   S_EXT_EXTNODE      *indexNode,               // IN    Index node with extents
   ULN64              *blksTodo,                // INOUT maximum blocks todo
   DFSISPACE          *is,                      // INOUT Integrated S_SPACE, data
   DFSISPACE          *m1,                      // INOUT meta data 1, L1 or Leaf
   DFSISPACE          *m2                       // INOUT meta data 2, L2 or Index
)
{
   ULONG               rc = NO_ERROR;           // function return
   int                 i;
   S_EXT_EXTNODE      *nlNode;                  // Next-Level (any) NODE block

   ENTER();

   if ((indexNode->hdr.ehMagic   == EXT_EXTENT_MAGIC) && // it is indeed an index node
       (indexNode->hdr.ehDepth   >  0               ) && // with valid depth
       (indexNode->hdr.ehDepth   <  9               ) && // and valid nr of entries
       (indexNode->hdr.ehEntries < (ext->BlockSize / sizeof(S_EXT_EXTINDEX))))
   {
      ULONG           entries = indexNode->hdr.ehEntries;

      for (i = 0; (i < entries) && (rc == NO_ERROR); i++)
      {
         if ((rc = dfsExtAllocReadBlk( indexNode->ent[i].ix.eiBlockLo,
                                       1, (BYTE **) &nlNode)) == NO_ERROR)
         {
            if ((m1 != NULL) && (m2 != NULL))
            {
               S_SPACE   *sp = TxAlloc( 1, sizeof(S_SPACE));

               if (sp != NULL)
               {
                  sp->size  = ext->SectorsPerBlock;
                  sp->start = dfsExtBlock2Lsn( indexNode->ent[i].ix.eiBlockLo);

                  if (nlNode->hdr.ehDepth == 0) // block contains leaf-nodes
                  {
                     TRACES(("Add %llu sectors at 0x%llx to META-1 = LEAF space\n", sp->size, sp->start));
                     rc = dfsSspaceJoin( m1->chunks,  m1->space, 1, sp,
                                        &m1->chunks, &m1->space);
                  }
                  else                          // block contains index-nodes
                  {
                     TRACES(("Add %llu sectors at 0x%llx to META-2 = INDEX space\n", sp->size, sp->start));
                     rc = dfsSspaceJoin( m2->chunks,  m2->space, 1, sp,
                                        &m2->chunks, &m2->space);
                  }
               }
               else
               {
                  rc = DFS_ALLOC_ERROR;
               }
            }
            if (rc == NO_ERROR)
            {
               rc = dfsExtAnyNode2Space( nlNode, blksTodo, is, m1, m2);
            }
            TxFreeMem( nlNode);
         }
         TRACES(("Handled entry %3u of %3u, blksTodo: %lld\n", i+1, entries, *blksTodo));
      }
   }
   else                                         // it is a leaf, error
   {
      rc = DFS_BAD_STRUCTURE;
   }
   RETURN (rc);
}                                               // end 'dfsExtIndexNode2Space'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Convert allocation LEAF-NODE (depth==0) structure into an S_SPACE structure
/*****************************************************************************/
static ULONG dfsExtLeafNode2Space
(
   S_EXT_EXTNODE      *leafNode,                // IN    Leaf node with extents
   ULN64              *blksTodo,                // INOUT maximum blocks todo
   ULONG              *chunks,                  // OUT   nr of space entries
   S_SPACE           **space                    // OUT   space allocation
)
{
   ULONG               rc = NO_ERROR;           // function return
   int                 i;
   ULONG               blocks;                  // net blocks in one extent
   ULONG               extents = 0;             // nr of extents in leaf-node
   S_SPACE            *sp      = NULL;

   ENTER();

   if ((leafNode->hdr.ehMagic   == EXT_EXTENT_MAGIC) && // it is indeed an index node
       (leafNode->hdr.ehDepth   == 0               ) && // with valid depth and entries
       (leafNode->hdr.ehEntries < (ext->BlockSize / sizeof(S_EXT_EXTINDEX))))
   {
      extents = leafNode->hdr.ehEntries;

      TRACES(("Allocate %u space chunks, rel bl %u\n", extents, leafNode->ent[0].lf.elRelBlock))
      sp = (S_SPACE *) TxAlloc( extents, sizeof(S_SPACE));

      if ((sp = (S_SPACE *) TxAlloc( extents, sizeof(S_SPACE))) != NULL)
      {
         for (i = 0; i < extents; i++)          // copy extent info
         {
            blocks = leafNode->ent[i].lf.elLength;
            if (blocks > 32768)                 // blocks are un-initialized, correct count
            {
               blocks -= 32768;
            }
            sp[i].size  = blocks * ext->SectorsPerBlock;
            sp[i].start = dfsExtBlock2Lsn( leafNode->ent[i].lf.elStartLo);

            *blksTodo -= blocks;                // subtract blocks handled now

            TRACES(("Handled entry %3u of %3u, blksTodo: %lld\n", i+1, extents, *blksTodo));
         }
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   else                                         // not a leaf, error
   {
      rc = DFS_BAD_STRUCTURE;
   }
   *chunks = extents;
   *space  = sp;
   RETURN (rc);
}                                               // end 'dfsExtLeafNode2Space'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Convert classic BLOCKS-INDIRECT structure into an S_SPACE structure
/*****************************************************************************/
static ULONG dfsExtIblocks2Space
(
   S_EXT_BLKINDIRECT  *bi,                      // IN    block-indirect struct
   ULN64              *blksTodo,                // INOUT maximum blocks todo
   DFSISPACE          *is,                      // INOUT Integrated S_SPACE, data
   DFSISPACE          *m1,                      // INOUT meta data 1, L1 or Leaf
   DFSISPACE          *m2                       // INOUT meta data 2, L2 or Index
)
{
   ULONG               rc = NO_ERROR;           // function return

   ENTER();

   rc = dfsExtAllocSequence( &bi->Block[0], EXT_BLOCKPTRS,
                              blksTodo, &is->chunks, &is->space);

   if ((rc == NO_ERROR) && (*blksTodo > 0) && (bi->Indirect1 != 0))
   {
      rc = dfsExtL1Indirect2Space( bi->Indirect1, blksTodo, is, m1, m2);

      if ((rc == NO_ERROR) && (*blksTodo > 0) && (bi->Indirect2 != 0))
      {
         rc = dfsExtL2Indirect2Space( bi->Indirect2, blksTodo, is, m1, m2);

         if ((rc == NO_ERROR) && (*blksTodo > 0) && (bi->Indirect3 != 0))
         {
            rc = dfsExtL3Indirect2Space( bi->Indirect3, blksTodo, is, m1, m2);
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsExtIblocks2Space'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add classic INDIRECT level 1 structure to existing base S_SPACE structure
/*****************************************************************************/
static ULONG dfsExtL1Indirect2Space
(
   ULONG               block,                   // IN    Block# indirect1
   ULN64              *blksTodo,                // INOUT maximum blocks todo
   DFSISPACE          *is,                      // INOUT Integrated S_SPACE, data
   DFSISPACE          *m1,                      // INOUT meta data 1, L1 or Leaf
   DFSISPACE          *m2                       // INOUT meta data 2, L2 or Index
)
{
   ULONG               rc = NO_ERROR;           // function return
   DFSISPACE           isNew;                   // data space to be added
   ULONG              *lev1;                    // block of Level-1  Block#
   ULONG               blkNrsPerBlock = ext->BlockSize / sizeof(ULONG);

   ENTER();

   if ((rc = dfsExtAllocReadBlk( block, 1, (BYTE **) &lev1)) == NO_ERROR)
   {
      if (m1 != NULL)                           // add level-1 meta alloc?
      {
         S_SPACE   *sp = (S_SPACE *) TxAlloc( 1, sizeof(S_SPACE));

         if (sp != NULL)
         {
            sp->size  = ext->SectorsPerBlock;
            sp->start = dfsExtBlock2Lsn( block);

            TRACES(("Add %llu sectors at 0x%llx to META-1 = L1 space\n", sp->size, sp->start));
            rc = dfsSspaceJoin( m1->chunks,  m1->space, 1, sp,
                               &m1->chunks, &m1->space);
         }
         else
         {
            rc = DFS_ALLOC_ERROR;
         }
      }
      if (rc == NO_ERROR)
      {
         memset( &isNew, 0, sizeof(DFSISPACE)); // initialize to all zero
         rc = dfsExtAllocSequence( &lev1[0],  blkNrsPerBlock,
                                    blksTodo, &isNew.chunks, &isNew.space);
         if (rc == NO_ERROR)
         {                                      // join file data stream
            rc = dfsSspaceJoin( is->chunks,     is->space,
                                isNew.chunks,   isNew.space,
                               &is->chunks,     &is->space);
         }
      }
      TxFreeMem( lev1);
   }
   RETURN (rc);
}                                               // end 'dfsExtL1Indirect2Space'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add classic INDIRECT level 2 structure to existing base S_SPACE structure
/*****************************************************************************/
static ULONG dfsExtL2Indirect2Space
(
   ULONG               block,                   // IN    Block# indirect1
   ULN64              *blksTodo,                // INOUT maximum blocks todo
   DFSISPACE          *is,                      // INOUT Integrated S_SPACE, data
   DFSISPACE          *m1,                      // INOUT meta data 1, L1 or Leaf
   DFSISPACE          *m2                       // INOUT meta data 2, L2 or Index
)
{
   ULONG               rc = NO_ERROR;           // function return
   int                 i;
   ULONG              *lev2;                    // block of Level-2  Block#
   ULONG               blkNrsPerBlock = ext->BlockSize / sizeof(ULONG);

   ENTER();

   if ((rc = dfsExtAllocReadBlk( block, 1, (BYTE **) &lev2)) == NO_ERROR)
   {
      if (m2 != NULL)                           // add level-2 meta alloc?
      {
         S_SPACE   *sp = (S_SPACE *) TxAlloc( 1, sizeof(S_SPACE));

         if (sp != NULL)
         {
            sp->size  = ext->SectorsPerBlock;
            sp->start = dfsExtBlock2Lsn( block);

            TRACES(("Add %llu sectors at 0x%llx to META-2 = L2 space\n", sp->size, sp->start));
            rc = dfsSspaceJoin( m2->chunks,  m2->space, 1, sp,
                               &m2->chunks, &m2->space);
         }
         else
         {
            rc = DFS_ALLOC_ERROR;
         }
      }
      for (i = 0; (i < blkNrsPerBlock) && (*blksTodo > 0) && (rc == NO_ERROR); i++)
      {
         rc = dfsExtL1Indirect2Space( lev2[i], blksTodo, is, m1, m2);
      }
      TxFreeMem( lev2);
   }
   RETURN (rc);
}                                               // end 'dfsExtL2Indirect2Space'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add classic INDIRECT level 3 structure to existing base S_SPACE structure
/*****************************************************************************/
static ULONG dfsExtL3Indirect2Space
(
   ULONG               block,                   // IN    Block# indirect1
   ULN64              *blksTodo,                // INOUT maximum blocks todo
   DFSISPACE          *is,                      // INOUT Integrated S_SPACE, data
   DFSISPACE          *m1,                      // INOUT meta data 1, L1 or Leaf
   DFSISPACE          *m2                       // INOUT meta data 2, L2 or Index
)
{
   ULONG               rc = NO_ERROR;           // function return
   int                 i;
   ULONG              *lev3;                    // block of Level-3  Block#
   ULONG               blkNrsPerBlock = ext->BlockSize / sizeof(ULONG);

   ENTER();

   if ((rc = dfsExtAllocReadBlk( block, 1, (BYTE **) &lev3)) == NO_ERROR)
   {
      for (i = 0; (i < blkNrsPerBlock) && (*blksTodo > 0) && (rc == NO_ERROR); i++)
      {
         rc = dfsExtL2Indirect2Space( lev3[i], blksTodo, is, m1, m2);
      }
      TxFreeMem( lev3);
   }
   RETURN (rc);
}                                               // end 'dfsExtL3Indirect2Space'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create S_SPACE structure for an array of EXT BLOCK numbers
/*****************************************************************************/
static ULONG dfsExtAllocSequence                // RET   possible alloc error
(
   ULONG              *blkArray,                // IN    array of blocknumbers
   ULONG               entries,                 // IN    nr of block# in array
   ULN64              *blksTodo,                // INOUT maximum blocks todo
   ULONG              *chunks,                  // OUT   nr of space entries
   S_SPACE           **space                    // OUT   space allocation
)
{
   ULONG               rc = NO_ERROR;
   ULONG               bl;                      // nr of blocks handled
   ULONG               todo    = entries;       // max blocks to handle now
   ULONG               this    = 0;             // current block#
   ULONG               last    = 0;             // last handled block
   ULONG               size    = 0;             // sectors in this chunk
   ULONG               extents = 0;             // nr of extents now added
   ULONG               areas   = 0;             // nr of space-areas allocated
   S_SPACE            *sp      = NULL;

   ENTER();

   if (*blksTodo < todo)
   {
      todo = (ULONG) *blksTodo;
   }
   TRARGS(("Array at: %8.8X, entries:%u todo:%u out of:%lld\n", blkArray, entries, todo, *blksTodo));

   for (bl = 0; bl < todo; bl++)
   {
      this = blkArray[bl];

      TRACES(( "IN-FOR bl:%4u this:%8.8x last:%8.8x extents:%4u size:%u\n",
                       bl,    this,      last,      extents,    size));

      if (this == (last +1))                    // block in sequence ?
      {
         size += ext->SectorsPerBlock;          // just add its size
      }
      else                                      // create new chunk
      {
         if (areas <= extents)                  // areas used up ?
         {
            areas = areas * 8 + 16;
            TRACES(("(re)Allocate %u space chunks, rel bl %u\n", areas, bl))
            sp = (S_SPACE *) realloc( sp, (size_t) (areas+1) * sizeof(S_SPACE));
         }
         if (sp != NULL)
         {
            if (last != 0)                      // not first extent ?
            {
               sp[extents].size = size;         // set size of last extent
               extents++;                       // and move to next one
            }
            size = ext->SectorsPerBlock;        // start with 1 block
            sp[extents].size  = size;
            sp[extents].start = dfsExtBlock2Lsn(this);
         }
         else
         {
            extents = 0;                        // all was lost in re-alloc
            rc = DFS_ALLOC_ERROR;
            break;
         }
      }
      last = this;                              // will be zero for non-full sequence
      if (this == 0)                            // end of allocated blocks, before limit
      {
         break;                                 // leave loop
      }
   }
   if (last != 0)                               // pending extent reaching entries ?
   {
      sp[extents].size = size;                  // set size of last extent
      extents++;                                // and move to next one
   }
   TRACES(( "ENDFOR bl:%4u this:%8.8x last:%8.8x extents:%4u size:%u\n",
                    bl,    this,      last,      extents,    size));
   if (sp != NULL)
   {
      *blksTodo -= bl;                          // subtract blocks handled now
      TRACES(( "Last extent start: 0x%llx, size: 0x%llx, blocks added:%u, blksTodo:%lld\n",
             sp[extents -1].start, sp[extents -1].size, bl, *blksTodo));
   }
   *chunks = extents;
   *space  = sp;
   RETURN (rc);
}                                               // end 'dfsExtAllocSequence'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Allocate memory and read one or more blocks from disk, freemem on read fail
/*****************************************************************************/
ULONG dfsExtAllocReadBlk
(
   ULONG               block,                   // IN    first Block#
   ULONG               blks,                    // IN    number of blocks
   BYTE              **data                     // OUT   Allocated and read data
)
{
   ULONG               rc = NO_ERROR;           // function return

   ENTER();

   TRACES(("Read %u blocks, starting at block: 0x%8.8x\n", blks, block));

   if ((*data = TxAlloc( blks, ext->BlockSize)) != NULL) // allocate one page
   {
      rc = dfsRead( dfsExtBlock2Lsn( block), blks * ext->SectorsPerBlock, *data);
      if (rc != NO_ERROR)
      {
         TxFreeMem( *data);
      }
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsExtAllocReadBlk'
/*---------------------------------------------------------------------------*/
