//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// EXFAT utility functions
//
// Author: J. van Wijk
//
// JvW  19-06-2014   Initial version, derived from DFSUFAT
//
#ifndef    DFSUEXFT
   #define DFSUEXFT

#include <dfsexft.h>                            // EXFAT structure definitions
#include <dfsaexft.h>                           // EXFAT cache definitions

#define DFS_EXFT_INVALIDS ":;\"|\\=?*<>,+[]/"

// Display file allocation and path info for LSN
ULONG dfsExftFileInfo                           // RET   result
(
   ULONG               lsn,                     // IN    Directory LSN
   ULONG               entry,                   // IN    directory entry
   char               *select,                  // IN    File select wildcard
   void               *lead                     // IN    leading text
);

// Replace sn-list by contents of a single Directory (DirLsn, entry pairs)
ULONG dfsExftMakeBrowseList                     // RET   result
(
   ULONG               lsn,                     // IN    Directory LSN
   ULONG               info,                    // IN    dir entry, 0 for ROOT
   char               *str,                     // IN    unused
   void               *param                    // INOUT unused
);

// Get allocation information for file-DATA into integrated-SPACE structure
ULONG dfsExftGetAllocSpace
(
   ULONG               lsn,                     // IN    Directory LSN
   ULONG               info,                    // IN    directory entry
   char               *str,                     // IN    unused
   void               *param                    // INOUT Integrated SPACE
);

// Initialize the Directory-Parent cache structure by walking the tree
ULONG dfsExftInitDpCache
(
   void
);

// Search DP-cache for parent Lsn+index for specified directory sector number
ULONG dfsExftSearchParent                  // RET   NOT_FOUND when fail
(
   ULONG               dirLsn,                  // IN    sector number
   LSN                *parentLsn,               // OUT   parent dirsector LSN
   BYTE               *parentIdx                // OUT   parent index in sector
);

// Get long or short (8.3) name for given LSN and entry number for DIR sector
ULONG dfsExftLsnInfo2Name                       // RET   result
(
   ULONG               lsn,                     // IN    Directory LSN
   ULONG               info,                    // IN    directory entry
   char               *name                     // OUT   directory entry name
);


// Find full PATH to Root-directory, starting at some dir LSN and entry info
ULONG dfsExftLsnInfo2Path                       // RET   result und
(
   LSN                 start,                   // IN    starting lsn
   ULONG               entry,                   // IN    directory entry info
   char               *path                     // OUT   combined path string
);

// Determine type of whole directory cluster and cache it for later reuse
BYTE dfsExftDirClusterType
(
   LSN                 lsn                      // IN    LSN of a (DIR) sector
);

#define dfsExftClust2LSN(cl) (LSN) dfsExftCl2Lsn((cl),0,NULL,NULL)
// Translate Cluster-nr to LSN
ULONG dfsExftCl2Lsn                             // RET   LSN for this cluster
(
   ULONG               cluster,                 // IN    cluster number
   ULONG               d2,                      // IN    dummy
   char               *p1,                      // IN    dummy
   void               *p2                       // IN    dummy
);

#define dfsExftLSN2Clust(sn) dfsExftLsn2Cl((ULONG)(sn),0,NULL, NULL)
// Translate LSN to Cluster-nr
ULONG dfsExftLsn2Cl                             // RET   cluster for this LSN
(
   ULONG               lsn,                     // IN    LSN
   ULONG               d2,                      // IN    dummy
   char               *p1,                      // IN    dummy
   void               *p2                       // IN    dummy
);

// Get cluster-value from a Directory entry structure
ULONG dfsExftDir2Clust                          // RET   cluster value
(
   S_EXFTDIR           *entry                    // IN    EXFAT directory entry
);

// Set cluster-value in a Directory entry structure
void dfsExftClust2Dir
(
   ULONG               cluster,                 // IN    cluster value to set
   S_EXFTDIR          *entry                    // INOUT EXFAT directory entry
);

// Get cluster-value from the Fat Cache structure (32 bit entries)
ULONG dfsExftValue                              // RET   cluster value
(
   ULONG               clust                    // IN    cluster to get value
);

// Set cluster-value in the Fat Cache structure (32 bit entries)
ULONG dfsExftSetCluster
(
   ULONG               clust,                   // IN    cluster to get value
   ULONG               value                    // IN    FAT-value to be set
);

// Reset bad-sector admin, change each BAD cluster in the FAT to FREE
ULONG dfsExftResetBadClus                       // RET   ref cluster or 0
(
   void
);

// Search reference to specified cluster in FAT (cluster pointing to this one)
// Note: Dont use regular cache to avoid trashing that just by searching here
ULONG dfsExftFindReference                      // RET   ref cluster or 0
(
   ULONG               cluster                  // IN    target cluster
);

// Check if cluster could be the 1st root-DIR sector (DIR but no '..' entry)
BOOL dfsExftRootCandidate                       // RET   Could be a root DIR
(
   ULONG               cluster                  // IN    target cluster
);

// Calulate 32-bit CRC over FAT-contents (to compare FAT1 and FAT2 perhaps :-)
ULONG dfsExftCalculateFatCRC                    // RET   32-bit CRC
(
   ULONG               fatlsn                   // IN    first sector of FAT
);

// Set the active FAT LSN to be used for fat references using the cache
void dfsExftSetFatInUse
(
   ULONG               fat2Use                  // IN    first sector of FAT
);

// Copy FAT-part from disk to the DFS FAT cache structure (32 bit entries)
ULONG dfsExftGetFAT                             // RET
(
   ULONG               clust,                   // IN    cluster value to get
   DFSEXFTCACHE        *cache                   // INOUT cache structure used
);

// Flush specified FAT cache (32 bit entries) back to disk FatInUse (+ other)
ULONG dfsExftFlushFAT                           // RET   result
(
   DFSEXFTCACHE        *cache                   // INOUT cache structure to write
);

// Show logical FAT
void dfsExftShowFAT
(
   ULONG               size,                    // IN    nr of entries
   ULONG               start,                   // IN    first cl# to show
   char               *options                  // IN    display options
);

// Count number of used Dir-entries for S_SPACE structure
ULONG dfsExftDirUsed                            // RET   matching entries
(
   ULONG               chunks,                  // IN    nr of space entries
   S_SPACE            *space                    // IN    space allocation
);

// Get label-string from the FAT Root-directory
BOOL dfsExftRootLabel                           // RET   Label found
(
   S_BOOTR            *boot,                    // IN    boot record for vol
   LSN                 offset,                  // IN    Offset for boot-rec
   char               *label                    // OUT   label string
);

// Get starting cluster for the specified Bitmap file from root directory
BOOL dfsExftRootGetBmCluster                    // RET   Bitmap cluster or 0
(
   S_BOOTR            *boot,                    // IN    boot record for vol
   LSN                 offset,                  // IN    Offset for boot-rec
   BYTE                mapnr,                   // IN    Bitmap 1st is 0
   ULONG              *cluster,                 // OUT   cluster number
   ULONG              *byteSize                 // OUT   size in bytes
);

// Create S_SPACE structure for an EXFAT stream directory entry
ULONG dfsExftAllocStream                        // RET   nr of clusters
(
   S_EXFTDIR          *stream,                  // Fat dir-entry ptr, STREAM
   ULONG              *errors,                  // INOUT error flags
   ULONG              *chunks,                  // OUT   nr of space entries
   S_SPACE           **space                    // OUT   space allocation
);

// Create S_SPACE structure for a FAT allocation chain
ULONG dfsExftAllocChain                         // RET   nr of clusters
(
   ULONG               cluster,                 // IN    starting cluster nr
   ULONG               maxsize,                 // IN    max file-size, or 0
   ULONG              *errors,                  // INOUT error flags (or NULL)
   ULONG              *chunks,                  // OUT   nr of space entries
   S_SPACE           **space                    // OUT   space allocation
);

// Check if name is a valid EXFAT filename
BOOL dfsValidExftName
(
   const void         *fn,                      // IN    ptr to EXFAT name
   USHORT              len                      // IN    length to check
);

// Display all available info about a directory entry, including C0/C1 entries
ULONG dfsExftShowDirectoryEntry
(
   LSN                 dirsect,                 // IN    lsn of FAT dir sector
   int                 entry                    // IN    File, Stream + Name entries
);

// FAT filesystem, display DIR header/footer
void dfsExftDirHeader
(
   char               *lead,                    // IN    lead text-string
   BOOL                head                     // IN    header (!footer)
);

// Display EXFAT directory entry on two lines, all except filename
void dfsExftShowDirEntry
(
   S_EXFTDIR          *entry,                   // IN    EXFAT FILE entry
   S_EXFTDIR          *stream,                  // IN    EXFAT STREAM entry
   int                 ac                       // IN    active color fg+bg
);

// Display FAT allocation chain from STREAM directory entry
void dfsExftShowAlloc
(
   S_EXFTDIR          *entry                    // IN    STREAM dir entry
);

// Convert null-terminated name to a valid dir-entry name (multiple C1 entries)
void dfsExftName2Entry
(
   char               *fn,                      // IN    ptr to FILE name
   S_EXFTDIR          *de                       // OUT   File, Stream + Name entries
);

// Convert a valid dir-entry name (multiple C1) to valid null-terminated name
// Asumes there are 2 more sectors of DIR entries BEFORE and AFTER specified one
void dfsExftEntry2Name
(
   S_EXFTDIR          *de,                      // IN    File, Stream + Name entries
   char               *fn                       // OUT   ptr to FILE name
);

// Retrieve first-cluster number for specified Dir-sector-LSN + index (LsnInf)
ULONG dfsExftLsnInf2Cluster                         // RET   First cluster or 0
(
   ULONG               lsn,                     // IN    Directory LSN
   ULONG               info                     // IN    directory entry
);

// retrieve directory-entry for specified Dir-sector-LSN + index (LsnInf)
ULONG dfsExftLsnInf2Entry
(
   ULONG               lsn,                     // IN    Directory LSN
   ULONG               info,                    // IN    directory entry
   S_EXFTDIR          *direntry                 // OUT   directory entry
);

// Retrieve STREAM directory-entry for specified Dir-sector-LSN + index (LsnInf)
// Input entry (lsn+inf) should point to a FILE, STREAM, BITMAP or UPCASE entry
ULONG dfsExftLsnInf2Stream
(
   ULONG               lsn,                     // IN    Directory LSN
   ULONG               info,                    // IN    directory entry index
   S_EXFTDIR          *direntry                 // OUT   directory entry (STREAM)
);

// Create SPACE allocation structure from a Directory LSN (root or other)
ULONG dfsExftDirLsn2Space
(
   LSN                 lsn,                     // IN    lsn for DIR sector
   ULONG              *nr,                      // OUT   nr of space entries
   S_SPACE           **sp                       // OUT   space allocation
);

// Iterate over directory-structures finding files, and execute callback
ULONG dfsExftIterator
(
   ULONG               recurse,                 // IN    recurse subtrees
   ULONG               type,                    // IN    do 'D', 'f' or both
   char               *path,                    // IN    path specification
   void               *p                        // IN    iterate parameters
);

// Create valid FAT(32) bootrecord based on exft-> info from ExfatInit
ULONG dfsExftMkBootRec
(
   BOOL                verbose,                 // IN    verbose and confirm
   ULONG               serialNr                 // IN    Vol-serialNr or 0
);

// Get/Set DIRTY bit in EXFAT bootsector to specified value
BOOL dfsExftDirtyStatusBootrec                  // RET   final dirty status
(                                               //
   BOOL                update,                  // IN    update value
   BOOL                dirty                    // IN    new value
);

// Read allocation for ALL directory sectors into a single SPACE structure
ULONG dfsExftAllDirs2Space
(
   BOOL                progressSectors,         // IN    progress in sectors, not DIRs
   ULONG              *nr,                      // OUT   nr of space entries
   S_SPACE           **sp                       // OUT   space allocation
);

// Determine allocation-bit for specified LSN, ALLOCATED beyond last cluster!
ULONG dfsExftAllocated                          // RET   LSN is allocated
(
   ULONG               lsn,                     // IN    LSN
   ULONG               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);

// Align R/W cache to position for cluster, and get current allocation-bit
BOOL dfsExftBitmapCache                         // RET   cluster is allocated
(
   ULONG               cluster,                 // IN    LCN
   ULONG              *bp                       // OUT   byte position in cache
);                                              //       or LSN_NULL if invalid

// Flush BitMap cache when Dirty, and optional free its resources
ULONG dfsExftBitmapFlush                        // RET   rc
(
   BOOL                terminate                // IN    terminate cache too
);

// Set allocate status for LSN to specified value (beyond FS, to end of bitmap)
ULONG dfsExftSetAlloc                           // RET   LSN allocation set
(
   ULONG               lsn,                     // IN    LSN
   ULONG               d2,                      // IN    dummy
   char               *value,                   // IN    NULL = not allocated
   void               *ref                      // IN    dummy (for EXFT)
);

// Set allocate status for CLUSTER to specified value
ULONG dfsExftSetBmCluster                       // RET   CL  allocation set
(
   ULONG               cl,                      // IN    Cluster
   BOOL                value                    // IN    SET allocation bit
);

// Read, calculate checksums and write primary and secondary boot area structures
ULONG dfsExftCrcSyncBootAreas
(
   LSN                 source,                  // IN    LSN of source area
   int                 area                     // IN    destination area 0=both/1/2
);

// Calculate EXFAT boot-area specific checksum, using published algorithm
ULONG dfsExftBootCheckSum                       // RET   EXFAT boot style crc
(
   BYTE               *data,                    // IN    Boot area (>= 11 sect)
   ULONG               size                     // IN    Size of data area
);

// Read EXFAT boot-area checksum reference from sector and check consistency
ULONG dfsExftStoredChecksum                     // RET   EXFAT boot style crc
(
   BYTE               *data,                    // IN    Boot area (>= 12 sect)
   ULONG               size                     // IN    Size of data area
);


#endif
