//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// ExFAT utility functions
//
// Author: J. van Wijk
//
// JvW  19-06-2014   Initial version, derived from DFSUFAT
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsver.h>                             // DFS version info
#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfstable.h>                           // SLT utility functions

#include <dfsupart.h>                           // FDISK partition functions
#include <dfsafdsk.h>                           // FDISK display & analysis
#include <dfscfdsk.h>                           // FDISK util, FAT sectors

#include <dfsaexft.h>                           // EXFT display & analysis
#include <dfsuexft.h>                           // EXFT utility functions
#include <dfslexft.h>                           // EXFT SLT functions


// Put contents of a Directory SSPACE (DirLsn, entry pairs) into sn-list
static ULONG dfsExftDirSpace2List
(
   ULONG               chunks,                  // IN    nr of space entries
   S_SPACE            *space                    // IN    space allocation
);

// Compare Dir-Parent cache elements for qsort, sort on cluster number
static int dfsExftCompareDpItem
(
   const void         *one,
   const void         *two
);

// Add each clusternumber for an EFAT directory to the Directory-Parent cache
static ULONG dfsExftAdd2DpCache
(
   ULONG               chunks,                  // IN    nr of S_SPACE chunks
   S_SPACE            *space,                   // IN    space structure array
   LSN                 parentLsn,               // IN    parent dirsector LSN
   BYTE                parentIdx                // IN    parent index in sector
);

// Recursively iterate over FAT directory and files, execute callback
static ULONG dfsExftIterFiles
(
   ULONG               rs,                      // IN    recurse subtrees
   ULONG               type,                    // IN    do 'D', 'f' or both
   ULONG               chunks,                  // IN    nr of S_SPACE chunks
   S_SPACE            *space,                   // IN    space structure array
   DFS_PARAMS         *cp                       // IN    callback params
);

// Add SPACE for specified directory to the accumulating total directory SPACE
static ULONG dfsExftAddDir2Space
(
   ULONG               sn,                      // IN    sector-nr DIR sector
   ULONG               info,                    // IN    coded directory entry#
   char               *df,                      // IN    dummy, optional info
   void               *data                     // INOUT callback data, SPACE
);


//- relative age of last read/write action to a cache structure
static ULONG        lastCacheStamp = 0;      // last age value issued

//- progress counter for FatIterator (is not always the sector-list!)
static ULONG        iterationsDone = 0;

/*****************************************************************************/
// Display file allocation and path info for LSN (list -f etc)
/*****************************************************************************/
ULONG dfsExftFileInfo                           // RET   result
(
   ULONG               lsn,                     // IN    Directory LSN
   ULONG               info,                    // IN    directory entry
   char               *select,                  // IN    File select wildcard
   void               *param                    // INOUT leading text/shortpath
)
{
   ULONG               rc = NO_ERROR;
   BYTE               *sb = NULL;               // sector buffer
   S_EXFTDIR          *efile;                   // Fat directory entry, FILE
   S_EXFTDIR          *stream;                  // Fat directory entry, STREAM
   BYTE                st = 0;                  // sector type, default none
   LSN                 dirsect = lsn;
   USHORT              entry   = info & 0x0f;
   TXLN                shortpath;
   TXTM                text;
   TXTT                temp;                    // location/date text
   ULONG               percent   = 0;
   ULONG               threshold = 0;           // threshold percentage
   BOOL                isMinimum = TRUE;        // threshold is minimum value
   ULONG               size      = 0;           // nr of allocated sectors
   ULONG               location  = 0;           // location value; LSN/Cluster
   ULONG               minS;                    // minimal size
   ULONG               maxS;                    // maximal size
   char                itemType;                // type Dir, File, Browse ...
   char               *lead    = param;
   BOOL                verbose = (strcmp( lead, ":") != 0);
   BOOL                doprint = (*lead);       // no Printf, just return fpath

   ENTER();
   TRARGS(("LSN:%8.8lX, info:0x%2.2lx select:'%s', lead:'%s'\n", lsn, info, select, lead));

   dfsParseFileSelection( select, text, &isMinimum, &threshold, &minS, &maxS, &itemType);
   if ((info & DFSSNINFO) && (dirsect > 2))
   {
      if ((sb = TxAlloc(5, dfsGetSectorSize())) != NULL) // 2 extra sectors BEFORE and AFTER
      {
         if ((rc = dfsRead(dirsect - 2, 5, sb)) == NO_ERROR)
         {
            BYTE      *dirdata = sb + (2 * dfsGetSectorSize());
            int        sanity;

            switch (dfsIdentifySector(dirsect, 0, dirdata))
            {
               case ST_ROOTD:
               case ST_DIREC:
                  efile = &(((S_EXFTDIR *) dirdata)[entry]);
                  st    = (efile->fl.Attrib & FATTR_DIRECTORY) ? 'D' : 'f';

                  switch (st)
                  {
                     case 'D':
                        if      (itemType == DFS_FS_ITEM_BROWSE) // no filtering on Directories, take all.
                        {
                           strcpy( text, "");   // no wildcard
                           isMinimum = TRUE; threshold = 0; // no percentage filter
                           minS      = 0;    maxS      = 0; // no size filter
                        }
                        else if (itemType == DFS_FS_ITEM_FILES) // filter files, discard Directories
                        {
                           rc = DFS_ST_MISMATCH; // item-type mismatch
                           break;
                        }
                     case 'f':
                     default:
                        if      (itemType == DFS_FS_ITEM_DIRS) // filter dirs, discard files
                        {
                           rc = DFS_ST_MISMATCH; // item-type mismatch
                           break;
                        }
                        if ((efile->EntryType & ~EXFAT_DIRBIT_IN_USE) == EXFAT_DEC_VLABEL)
                        {
                           //- Create specific name string for a LABEL direcory entry
                           sprintf( shortpath, "%sLabel:%s ", CNN, CBG);
                           dfsExftEntry2Name( efile, shortpath + strlen(shortpath));
                        }
                        else                    // non-label, (long) filename
                        {
                           //- Find path upto ROOT when SLT is available, or bare name otherwise
                           //- includes handling systemfiles (bitmap/upcase) with explicit name
                           dfsExftLsnInfo2Path( lsn, info, shortpath);
                        }
                        if ((strlen(text) == 0) || (TxStrWicmp(shortpath, text) >= 0))
                        {
                           stream = efile;      // stream is same entry for bitmap/upcase
                           switch (efile->EntryType & ~EXFAT_DIRBIT_IN_USE)
                           {
                              case EXFAT_DEC_VLABEL:
                                 st = 'L';
                                 break;

                              case EXFAT_ANY_STREAM:
                              case EXFAT_ANY_FILENAME: // need to search backwards to FILE entry
                                 sanity = 2 * dfsGetSectorSize() / sizeof(S_EXFTDIR);
                                 while (((efile->EntryType & ~EXFAT_DIRBIT_IN_USE) != EXFAT_DEC_FILE) && (sanity--))
                                 {
                                    efile--;    // move backwards over 1 entry
                                 }
                                 if ((efile->EntryType & ~EXFAT_DIRBIT_IN_USE) != EXFAT_DEC_FILE)
                                 {
                                    break;      // no FILE entry found, fail
                                 }
                              case EXFAT_DEC_FILE:
                                 stream = efile +1; // advance to actual stream entry
                                 if ((efile->fl.Attrib & FATTR_DIRECTORY) != 0)
                                 {
                                    st   = 'D';
                                    strcat( shortpath, FS_PATH_STR);
                                 }
                                 else
                                 {
                                    st   = 'f';
                                 }
                                 if (stream->st.Length != 0)
                                 {
                                    size = ((stream->st.Length -1) / dfsGetSectorSize()) +1;
                                 }
                                 location = dfsExftDir2Clust( stream);
                                 break;

                              case EXFAT_DEC_BITMAP:
                                 st   = 'S';
                                 if (efile->bm.Length != 0)
                                 {
                                    size = ((efile->bm.Length -1) / dfsGetSectorSize()) +1;
                                 }
                                 location = dfsExftDir2Clust( efile);
                                 break;

                              case EXFAT_DEC_UPCASE:
                                 st   = 'S';
                                 if (efile->up.Length != 0)
                                 {
                                    size = ((efile->up.Length -1) / dfsGetSectorSize()) +1;
                                 }
                                 location = dfsExftDir2Clust( efile);
                                 break;

                              default:
                                 break;
                           }

                           if ((verbose) && (location >= 2)) // check allocation (percentage)
                           {
                              DFSISPACE       is;
                              ULONG           ef; // error flags (not used here)
                              ULONG           size; // allocated size in sectors
                              ULONG           bads; // sectors with bad allocation

                              if (dfsExftAllocStream( stream, &ef, &is.chunks, &is.space) != 0)
                              {
                                 is.clsize = exft->ClustSize;
                                 size      = dfsSspaceSectors( TRUE, is.chunks, is.space);
                                 dfsCheckAllocForSpace( &is, 0,
                                   ((efile->EntryType & EXFAT_DIRBIT_IN_USE) != 0), &bads);

                                 percent = dfsAllocationReliability( size, bads);

                                 free( is.space); // free the SPACE (calloc/realloc)
                              }
                              TRACES(( "percent:%lu threshold:%lu size:%lu minS:%lu maxS:%lu\n",
                                        percent, threshold, size, minS, maxS));
                           }

                           if ((BOOL)((percent >= threshold)) == isMinimum)
                           {
                              if ((size >= minS) && ((size <= maxS) || (maxS == 0)))
                              {
                                 if (verbose)   // include alloc and size
                                 {
                                    strcpy( temp, "");
                                    if (!TxaOption('C') && (location != 0))
                                    {
                                       location = dfsExftClust2LSN( location);
                                    }
                                    if (location != 0)
                                    {
                                       sprintf( temp, "%8lX", location);
                                    }
                                    sprintf(   text, "%s%c %s%3lu%% %s%8.8s ", CBM, st,
                                               (rc == NO_ERROR) ?  CBG : CBR, percent, CNN, temp);
                                    dfstrSize( text, CBC, size, " ");
                                    dfstrBytes(text, CBZ, 0,    " "); // XATTR/EA-size
                                 }
                                 if (doprint)   // not totally silent?
                                 {
                                    strcat(    lead, text);
                                    TxPrint( "%s%s%s%s%s\n", lead, CBY, shortpath, CNN, CGE);
                                 }
                                 else           // return info in 'select'
                                 {
                                    sprintf( temp, "%4.4u-%2.2u-%2.2u %2.2u:%2.2u:%2.2u",
                                       efile->fl.DtModify.d.year + 1980, efile->fl.DtModify.d.month, efile->fl.DtModify.d.day,
                                       efile->fl.TmModify.t.hours, efile->fl.TmModify.t.minutes, efile->fl.TmModify.t.twosecs * 2 +
                                                                   efile->fl.MsModify / 100);
                                    TxStripAnsiCodes( text);
                                    sprintf( select, "%s %s", text, temp);
                                    dfstrUllDot20( select, "", stream->st.Length, "");
                                 }
                                 rc = NO_ERROR;
                              }
                              else
                              {
                                 TRACES(("file-size mismatch\n"));
                                 rc = DFS_ST_MISMATCH; // file-size mismatch
                              }
                           }
                           else
                           {
                              TRACES(("percentage mismatch\n"));
                              rc = DFS_ST_MISMATCH; // percentage mismatch
                           }
                        }
                        else
                        {
                           rc = DFS_ST_MISMATCH; // wildcard mismatch
                        }
                        strcpy(  param, shortpath); // return the found name
                        break;
                  }
                  break;

               default:
                  rc = DFS_PENDING;
                  break;
            }
         }
         else if (dirsect < exft->Sect)          // should be there
         {
            dfsLsx("\nError reading sector : ", dirsect, CBR, "\n");
         }
         TxFreeMem(sb);
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   else if (dirsect == exft->Root)              // ROOT, fake 1 cluster dir
   {
      if (itemType != DFS_FS_ITEM_FILES)        // unless files-only
      {
         sprintf(   text, "%s%c%s      %8lX ",  CBM, 'D', CNN, dirsect);
         dfstrSize( text, CBC, exft->ClustSize, " ");
         if (doprint)                           // not totally silent?
         {
            strcat(    lead, text);
            TxPrint( "%s%s%s%s%s\n", lead, CBY, FS_PATH_STR, CNN, CGE);
         }
         else                                   // return info in 'select'
         {
            sprintf( temp, "01-12-2006 12:00:00"); // no date available for ROOT
            TxStripAnsiCodes( text);
            sprintf( select, "%s %s", text, temp);
         }
         strcpy(  param, FS_PATH_STR);          // return the found name
      }
      else                                      // Root does not match item type
      {
         rc = DFS_ST_MISMATCH;
      }
   }
   else
   {
      rc = DFS_ST_MISMATCH;                     // info is NOT a valid entrynr
   }
   RETURN(rc);
}                                               // end 'dfsExftFileInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Replace sn-list by contents of a single Directory (DirLsn, entry pairs)
/*****************************************************************************/
ULONG dfsExftMakeBrowseList                     // RET   result
(
   ULONG               lsn,                     // IN    Directory LSN
   ULONG               info,                    // IN    dir entry, 0 for ROOT
   char               *str,                     // IN    unused
   void               *param                    // INOUT unused
)
{
   ULONG               rc = NO_ERROR;
   BYTE               *sb = NULL;               // sector buffer
   S_EXFTDIR          *efile;                   // Fat directory entry, FILE
   S_EXFTDIR          *stream;                  // Fat directory entry, STREAM
   LSN                 dirsect = lsn;
   USHORT              entry   = info & 0x7f;
   ULONG               location  = 0;           // location value; LSN/Cluster
   DFSISPACE           is;
   ULONG               ef;                      // error flags (not used here)

   ENTER();
   TRARGS(("LSN:%8.8lX, info:0x%2.2lx\n", lsn, info));

   dfsInitList(0, "-f -P", "-d");               // optimal for menu file-recovery
   if ((info & DFSSNINFO) && (dirsect > 2))     // not the ROOT, and failsafe for lsn-2
   {
      ULONG            dotdotLsn;
      BYTE             dotdotInfo;

      if (dfsExftSearchParent( dirsect, &dotdotLsn, &dotdotInfo) == NO_ERROR)
      {
         DFSBR_SnlistFlag |= DFSBR_1ST_PARENT;
         dfsAdd2List( dotdotLsn, dotdotInfo, NULL, NULL); // start the list with ..
      }
      if ((sb = TxAlloc(5, dfsGetSectorSize())) != NULL) // 2 extra sectors BEFORE and AFTER
      {
         if ((rc = dfsRead(dirsect - 2, 5, sb)) == NO_ERROR)
         {
            BYTE      *dirdata = sb + (2 * dfsGetSectorSize());
            int        sanity;

            switch (dfsIdentifySector(dirsect, 0, dirdata))
            {
               case ST_ROOTD:
               case ST_DIREC:
                  rc = DFS_BAD_STRUCTURE;       // error, unless proven otherwise
                  efile  = &(((S_EXFTDIR *) dirdata)[entry]);
                  switch (efile->EntryType)
                  {
                     case EXFAT_DIR_STREAM:
                     case EXFAT_DIR_FILENAME:   // need to search backwards to FILE entry
                        sanity = 2 * dfsGetSectorSize() / sizeof(S_EXFTDIR);
                        while ((efile->EntryType != EXFAT_DIR_FILE) && (sanity--))
                        {
                           efile--;             // move backwards over 1 entry
                        }
                        if (efile->EntryType != EXFAT_DIR_FILE)
                        {
                           break;               // no FILE entry found, fail
                        }
                     case EXFAT_DIR_FILE:
                        stream = efile +1;      // advance to actual stream entry

                        if ((efile->fl.Attrib & FATTR_DIRECTORY) != 0) // must be a DIR
                        {
                           if ((location = dfsExftDir2Clust( stream)) >= 2)
                           {
                              if (dfsExftAllocStream( stream, &ef, &is.chunks, &is.space) != 0)
                              {
                                 rc = NO_ERROR;
                              }
                           }
                        }
                        break;

                     case EXFAT_DIR_VLABEL:     // only want a directory (aka 'FILE')
                     case EXFAT_DIR_BITMAP:     // so ignore anything else
                     case EXFAT_DIR_UPCASE:
                     default:
                        break;
                  }
                  break;

               default:
                  rc = DFS_ST_MISMATCH;         // not a directory sector
                  break;
            }
         }
         TxFreeMem(sb);
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   else                                         // must be ROOT itself
   {
      rc = dfsExftDirLsn2Space( lsn, &is.chunks, &is.space);
   }
   if ((rc == NO_ERROR) && (is.space))
   {
      rc = dfsExftDirSpace2List( is.chunks, is.space);
      free( is.space);                          // free the SPACE (calloc/realloc)
   }
   RETURN(rc);
}                                               // end 'dfsExftMakeBrowseList'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get allocation information for file-DATA into integrated-SPACE structure
/*****************************************************************************/
ULONG dfsExftGetAllocSpace
(
   ULONG               lsn,                     // IN    Directory LSN
   ULONG               info,                    // IN    directory entry
   char               *str,                     // IN    unused
   void               *param                    // INOUT Integrated SPACE
)
{
   ULONG               rc = NO_ERROR;
   DFSISPACE          *ispace = (DFSISPACE *) param;
   BYTE               *sb = NULL;               // sector buffer
   S_EXFTDIR          *efile;                   // Fat directory entry, FILE
   S_EXFTDIR          *stream;                  // Fat directory entry, STREAM
   USHORT              entry   = info & 0x7f;
   ULONG               location  = 0;           // location value; LSN/Cluster
   ULONG               ef;                      // error flags (not used here)

   ENTER();

   ispace->space = NULL;

   if ((info & DFSSNINFO) && ( lsn > 2))     // not the ROOT, and failsafe for lsn-2
   {
      if ((sb = TxAlloc(5, dfsGetSectorSize())) != NULL) // 2 extra sectors BEFORE and AFTER
      {
         if ((rc = dfsRead( lsn - 2, 5, sb)) == NO_ERROR)
         {
            BYTE      *dirdata = sb + (2 * dfsGetSectorSize());
            int        sanity;

            switch (dfsIdentifySector( lsn, 0, dirdata))
            {
               case ST_ROOTD:
               case ST_DIREC:
                  rc = DFS_BAD_STRUCTURE;       // error, unless proven otherwise
                  efile  = &(((S_EXFTDIR *) dirdata)[entry]);
                  switch (efile->EntryType)
                  {
                     case EXFAT_DIR_STREAM:
                     case EXFAT_DIR_FILENAME:   // need to search backwards to FILE entry
                        sanity = 2 * dfsGetSectorSize() / sizeof(S_EXFTDIR);
                        while ((efile->EntryType != EXFAT_DIR_FILE) && (sanity--))
                        {
                           efile--;             // move backwards over 1 entry
                        }
                        if (efile->EntryType != EXFAT_DIR_FILE)
                        {
                           break;               // no FILE entry found, fail
                        }
                     case EXFAT_DIR_FILE:
                        stream = efile +1;      // advance to actual stream entry
                        if ((location = dfsExftDir2Clust( stream)) >= 2)
                        {
                           if (dfsExftAllocStream( stream, &ef, &ispace->chunks, &ispace->space) != 0)
                           {
                              rc = NO_ERROR;
                           }
                        }
                        break;

                     case EXFAT_DIR_VLABEL:     // only want a directory (aka 'FILE')
                     case EXFAT_DIR_BITMAP:     // so ignore anything else
                     case EXFAT_DIR_UPCASE:
                     default:
                        break;
                  }
                  break;

               default:
                  rc = DFS_ST_MISMATCH;         // not a directory sector
                  break;
            }
         }
         TxFreeMem(sb);
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   else                                         // must be ROOT itself
   {
      rc = dfsExftDirLsn2Space( lsn, &ispace->chunks, &ispace->space);
   }
   RETURN (rc);
}                                               // end 'dfsExftGetAllocSpace'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Put contents of a Directory SSPACE (DirLsn, entry pairs) into sn-list
/*****************************************************************************/
static ULONG dfsExftDirSpace2List
(
   ULONG               chunks,                  // IN    nr of space entries
   S_SPACE            *space                    // IN    space allocation
)
{
   ULONG               rc = NO_ERROR;
   ULONG               chunk;                   // index in space-area
   ULONG               sect;                    // sectors to handle
   ULONG               entry;                   // index in fat directory
   S_EXFTDIR           prevEntryBuffer;         // Previous entry, contents
   S_EXFTDIR          *prev;                    // Previous entry, pointer
   S_EXFTDIR          *this;                    // Fat directory entry
   LSN                 dirsec;                  // current dir sector LSN
   LSN                 prevsec;                 // previous dir sector LSN
   USHORT              bps = dfsGetSectorSize();
   USHORT              entries = (bps / sizeof(S_EXFTDIR));
   BYTE               *dirbuf;
   S_EXFTDIR          *fatdir;                  // Fat directory sector

   ENTER();

   if ((dirbuf = TxAlloc( 1, bps)) != NULL)     // allocate one sector
   {
      fatdir = (S_EXFTDIR *) dirbuf;
      prev   = &prevEntryBuffer;

      prevsec = space[0].start;                 // safe start value for prev

      for (chunk = 0; (chunk < chunks)  && !TxAbort(); chunk++)
      {
         for (sect = 0; (sect < space[chunk].size)  && !TxAbort(); sect++)
         {
            dirsec  = space[chunk].start + sect;
            if ((rc = dfsRead(dirsec, 1, dirbuf)) == NO_ERROR)
            {
               for (entry = 0; (entry < entries) && !TxAbort(); entry++)
               {
                  this = &(fatdir[entry]);
                  switch (this->EntryType)      // normal only, not deleted
                  {
                     case EXFAT_DIR_BITMAP:     // to be refined, ignore or include?
                     case EXFAT_DIR_UPCASE:
                         //- dfsAdd2List( dirsec, entry | DFSSNINFO, NULL, NULL);
                        break;

                     case EXFAT_DIR_FILE:
                        dfsAdd2List( dirsec, entry | DFSSNINFO, NULL, NULL);
                        break;

                     case EXFAT_DIR_STREAM:
                     case EXFAT_DIR_FILENAME:
                     case EXFAT_DIR_VLABEL:
                     case EXFAT_DIR_EODIR:      // free entry
                     default:                   // others (deleted/renamed etc)
                        break;
                  }
                  prevEntryBuffer = *this;      // make a persistent copy!
                  prevsec = dirsec;             // and remember its LSN (STREAM to FILE backtracking)
               }
            }
         }
      }
      TxFreeMem( dirbuf);
   }
   RETURN (rc);
}                                               // end 'dfsExftDirSpace2List'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Initialize the Directory-Parent cache structure by walking the tree
/*****************************************************************************/
ULONG dfsExftInitDpCache
(
   void
)
{
   ULONG               rc  = NO_ERROR;
   S_SPACE            *root;                    // allocation for Root
   ULONG               rootchunks;              // nr of Root fragments

   ENTER();

   if ((exft->ExftFaked == FALSE) &&            // no faked values (no bootrec)
       (exft->CacheA.Value != NULL))            // valid fat structures
   {
      //- having snlist filled on start s not good for Browse default
      //- dfsInitList(0, "-f -P", "-d");            // optimal for menu file-recovery

      rc = dfsExftDirLsn2Space( exft->Root, &rootchunks, &root);
      if (rc == NO_ERROR)
      {
         //- recursively add all directories that have subdirectories
         rc = dfsExftAdd2DpCache( rootchunks, root, exft->Root, 0);
         if (rc == NO_ERROR)
         {
            qsort( exft->Dp, exft->dpCacheSize, // sort on Cluster number
                   sizeof(DFSEXFTDPCACHE), dfsExftCompareDpItem);
         }
      }
      if (root)
      {
         free( root);                           // re-alloc'ed no TxFreeMem !
      }
   }
   else
   {
      rc = DFS_ST_MISMATCH;                     // bad structures
   }
   RETURN (rc);
}                                               // end 'dfsExftInitDpCache'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Compare Dir-Parent cache elements for qsort, sort on cluster number
/*****************************************************************************/
static int dfsExftCompareDpItem
(
   const void         *one,
   const void         *two
)
{
   DFSEXFTDPCACHE     *p1  = (DFSEXFTDPCACHE *) one;
   DFSEXFTDPCACHE     *p2  = (DFSEXFTDPCACHE *) two;
   int                 res = 0;

   ENTER();

   if (p1->Cluster != p2->Cluster)
   {
      res = (int) ((p1->Cluster < p2->Cluster) ? -1 : +1);
   }
   RETURN(res);
}                                               // end 'dfsExftCompareDpItem'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add each clusternumber for an EFAT directory to the Directory-Parent cache
// Add parent Lsn+entry to the sector-list too, to be able to inspect on Open
/*****************************************************************************/
static ULONG dfsExftAdd2DpCache
(
   ULONG               chunks,                  // IN    nr of S_SPACE chunks
   S_SPACE            *space,                   // IN    space structure array
   LSN                 parentLsn,               // IN    parent dirsector LSN
   BYTE                parentIdx                // IN    parent index in sector
)
{
   ULONG               rc = NO_ERROR;
   ULONG               chunk;                   // index in space-area
   ULONG               sect;                    // sectors to handle
   USHORT              entry;                   // index in fat directory
   S_EXFTDIR           prevEntryBuffer;         // Previous entry, contents
   S_EXFTDIR          *prev;                    // Previous entry, pointer
   S_EXFTDIR          *this;                    // Fat directory entry
   S_EXFTDIR          *fatdir;                  // Fat directory sector
   LSN                 dlsn;                    // Directory sector LSN
   LSN                 prevFileLsn;             // Previous  LSN,   FILE entry
   USHORT              prevFileEntry;           // Previous  entry, FILE
   S_SPACE            *sp = NULL;               // Alloc chunk array
   USHORT              fatEntries;              // entries per sector
   ULONG               lastCachedCluster = 0;   // last CL-number added to cache
   ULONG               currentCluster;          // CL-number for this entry
   ULONG               ef;                      // error flags (not used here)
   ULONG               nr;                      // Alloc chunks

   ENTER();

   TRACES(("chunks: %lu, parentLsn: %8.8lX, parentIdx: %hhx\n", chunks, parentLsn, parentIdx));

   if ((fatdir = TxAlloc(1, dfsGetSectorSize())) != NULL)
   {
      prev   = &prevEntryBuffer;

      for ( chunk = 0;
           (chunk < chunks) && (rc == NO_ERROR);
            chunk++)                            // walk all alloc chunks
      {
         for ( sect = 0;
              (sect < space[chunk].size) && (rc == NO_ERROR);
               sect++)                          // each sector in chunk
         {
            fatEntries = dfsGetSectorSize() / sizeof(S_EXFTDIR);
            dlsn = space[chunk].start + sect;
            rc = dfsRead( dlsn, 1, (BYTE   *) fatdir);
            switch (dfsIdentifySector( dlsn, 0, (BYTE   *) fatdir))
            {
               case ST_ROOTD:
               case ST_DIREC:
                  for ( entry = 0;
                       (entry < fatEntries) && (rc == NO_ERROR);
                        entry++)                // each dir-entry in sector
                  {
                     this = &(fatdir[entry]);

                     if (this->EntryType)
                     {
                        TRACES(("Chunk: %lu, Sect: %lu, entry %2u, type:0x%2.2hx\n", chunk, sect, entry, this->EntryType));
                     }
                     switch (this->EntryType)
                     {
                        case EXFAT_DIR_FILE:
                           prevFileLsn   = dlsn;    //- remember LSN and entry
                           prevFileEntry = entry;   //- for the STREAM processing
                           break;

                        case EXFAT_DIR_STREAM:
                           if (prev->EntryType == EXFAT_DIR_FILE)
                           {
                              if ((prev->fl.Attrib & FATTR_DIRECTORY) != 0)
                              {
                                 TRACES(("Process subdir at LSN: %8.8lx, entry: %hx\n", prevFileLsn, prevFileEntry));
                                 dfsExftAllocStream( this, &ef, &nr, &sp);
                                 if (sp != NULL)
                                 {
                                    currentCluster = dfsExftLSN2Clust( prevFileLsn);
                                    if (currentCluster != lastCachedCluster)
                                    {
                                       //- having snlist filled on start s not good for Browse default
                                       //- dfsAdd2List( parentLsn, parentIdx, NULL, NULL);

                                       TRACES(("Add cl %8.8lx with parent %8.8lx-%hhx at entry %lu\n",
                                                currentCluster, parentLsn, parentIdx, exft->dpCacheSize));

                                       if (exft->dpCacheAlloc <= exft->dpCacheSize) // areas used up ?
                                       {
                                          exft->dpCacheAlloc = exft->dpCacheAlloc * 8 + 1024;
                                          TRACES(("(re)Allocate %lu DP-cache entries\n", exft->dpCacheAlloc));
                                          exft->Dp = (DFSEXFTDPCACHE *) realloc( exft->Dp,
                                            (size_t) (exft->dpCacheAlloc) * sizeof(DFSEXFTDPCACHE));
                                       }
                                       exft->Dp[ exft->dpCacheSize].Cluster   = currentCluster;
                                       exft->Dp[ exft->dpCacheSize].ParentLsn = parentLsn;
                                       exft->Dp[ exft->dpCacheSize].ParentIdx = parentIdx;
                                       exft->dpCacheSize++;
                                       lastCachedCluster = currentCluster;
                                    }
                                    rc = dfsExftAdd2DpCache( nr, sp,
                                                             prevFileLsn,
                                                             prevFileEntry | DFSSNINFO);
                                    free( sp);  // re-alloc'ed no TxFreeMem !
                                 }
                              }
                           }
                           else
                           {
                              TRACES(( "EFAT stream, unexpected at this position!\n"));
                           }
                           break;

                        case EXFAT_DIR_BITMAP:
                        case EXFAT_DIR_UPCASE:
                        default:                // other types, deleted, name etc
                           //- no action
                           break;
                     }
                     prevEntryBuffer = *this;   // make persistent copy, each time!
                  }
                  break;

               default:                         // invalid DIR sector, skip
                  break;
            }
         }
      }
      TxFreeMem( fatdir);
   }
   RETURN( rc);
}                                               // end 'dfsExftAdd2DpCache'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Search DP-cache for parent Lsn+index for specified directory sector number
/*****************************************************************************/
ULONG dfsExftSearchParent                  // RET   NOT_FOUND when fail
(
   ULONG               dirLsn,                  // IN    DIR sector number
   LSN                *parentLsn,               // OUT   parent dirsector LSN
   BYTE               *parentIdx                // OUT   parent index in sector
)
{
   ULONG               rc = DFS_NOT_FOUND;
   ULONG               cl = dfsExftLSN2Clust( dirLsn); // DIR cluster, with subdir
   ULONG               i;

   ENTER();

   if (exft->Dp)                                // cache is available
   {
      for (i = 0; i < exft->dpCacheSize; i++)
      {
         TRACES(("Item:%3lu cl:%8.8lx Cluster:%8.8lx\n", i, cl, exft->Dp[ i].Cluster));
         if       (cl == exft->Dp[ i].Cluster)
         {
            *parentLsn = exft->Dp[ i].ParentLsn;
            *parentIdx = exft->Dp[ i].ParentIdx;
            rc = NO_ERROR;
            break;                              // found parent
         }
         else if  (cl <  exft->Dp[ i].Cluster)
         {
            break;                              // past our cl number
         }                                      // in sorted array
      }
   }
   TRACES(("dirLsn:%8.8lX = cl:%8.8lx -> parent Lsn: %8.8lx-%hhx\n",
            dirLsn,         cl,         *parentLsn,   *parentIdx));
   RETURN( rc);
}                                               // end 'dfsExftSearchParent'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get long filename for given LSN and entry number for DIR sector
/*****************************************************************************/
ULONG dfsExftLsnInfo2Name                        // RET   result
(
   ULONG               lsn,                     // IN    Directory LSN
   ULONG               info,                    // IN    directory entry info
   char               *name                     // OUT   directory entry name
)
{
   ULONG               rc = NO_ERROR;

   ENTER();
   TRARGS(("LSN:%8.8lX, info:0x%2.2lx\n", lsn, info));

   if (info & DFSSNINFO)
   {
      BYTE               *sb = NULL;            // sector buffer
      S_EXFTDIR           *efile;               // Fat directory entry
      LSN                 dirsect = lsn;
      USHORT              entry   = info & ~DFSSNINFO;

      //- Need to allocate/read 5 sectors to allow reading LFN up to 255 characters
      if ((sb = TxAlloc(5, dfsGetSectorSize())) != NULL) // 2 extra sectors BEFORE and AFTER
      {
         if ((rc = dfsRead(dirsect - 2, 5, sb)) == NO_ERROR)
         {
            BYTE      *dirdata = sb + (2 * dfsGetSectorSize());

            switch (dfsIdentifySector(dirsect, 0, dirdata))
            {
               case ST_ROOTD:
               case ST_DIREC:
                  efile   = &(((S_EXFTDIR *) dirdata)[entry]);

                  dfsExftEntry2Name( efile, name); // get real or faked file name
                  break;

               default:
                  strcpy( name, "");
                  rc = DFS_ST_MISMATCH;
                  break;
            }
         }
         TxFreeMem(sb);
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   else
   {
      rc = DFS_ST_MISMATCH;                     // info is NOT a valid entrynr
   }
   RETURN(rc);
}                                               // end 'dfsExftLsnInfo2Name'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Find full PATH to Root-directory, starting at some dir LSN and entry info
/*****************************************************************************/
ULONG dfsExftLsnInfo2Path                        // RET   result und
(
   LSN                 start,                   // IN    starting lsn
   ULONG               entry,                   // IN    directory entry info
   char               *path                     // OUT   combined path string
)
{
   ULONG               rc = NO_ERROR;
   LSN                 ref = start;             // Reference/Parent LSN
   BYTE                info = entry;            // entry info from SLT
   BYTE                st;
   ULONG               sanity = 20;             // maximum iterations
   ULONG               sltIndex;                // dummy SLT location
   TX1K                fname;                   // shortname (+ path)

   ENTER();
   if (path)
   {
      *path = '\0';
   }

   rc = dfsExftLsnInfo2Name( start, info, path); // get name for THIS dir entry

   if (rc == NO_ERROR)
   {
      if (dfsSlTableStatus(NULL) == SLT_READY) // there is an SLT
      {
         do
         {
            if (dfsSlTableFind( ref, &ref, &info, &st, &sltIndex))   //- Parent, next REF
            {
               if (ref == LSN_BOOTR)            // Root-DIR refrences to boot
               {
                  strcpy( fname, FS_PATH_STR);  // ROOT
                  strcat( fname, path);         // append existing path
                  strcpy( path, fname);         // and copy back
                  break;
               }
               else
               {
                  if ((rc = dfsExftLsnInfo2Name( ref, info, fname)) == NO_ERROR)
                  {
                     TRACES(("'%s' + '%s'\n", fname, path));
                     strcat( fname, FS_PATH_STR);
                     strcat( fname, path);      // append existing path
                     strcpy( path, fname);      // and copy back
                  }
               }
            }
            else
            {
               rc = DFS_NOT_FOUND;
            }
            sanity--;
         } while ((rc == NO_ERROR) && (sanity) && !TxAbort());
      }
      else
      {
         rc = DFS_CMD_WARNING;
      }
   }
   TRACES(("Found path to root for %8.8lX+%2.2lx:'%s'\n", start, entry, path));

   RETURN(rc);
}                                               // end 'dfsExftLsnInfo2Path'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Determine type of whole directory cluster and cache it for later reuse
/*****************************************************************************/
BYTE dfsExftDirClusterType                      // RET   DIR type or UDATA
(
   LSN                 lsn                      // IN    LSN of a (DIR) sector
)
{
   BYTE                rc = ST_UDATA;           // function return
   BYTE               *sb = NULL;               // sector buffer
   static LSN          csFirst = 0;             // first in cached range
   static LSN          csLast  = 0;             // last
   static BYTE         csType  = ST_UDATA;      // the associated type info
   LSN                 dirsect = csFirst;       // first sector of DIR cluster

   ENTER();
   TRACES(("lsn:%8.8lx csF:%8.8lx csL:%8.8lx csT:%c\n", lsn, csFirst, csLast, csType));

   if ((lsn >= exft->ClHeap) && (lsn < exft->Sect)) // possible (sub) directory
   {
      if ((lsn >= csFirst) && (lsn <= csLast))  // in cached range ?
      {
         rc = csType;                           // use cached value
      }
      else                                      // determine and cache new type
      {
         if ((sb = TxAlloc( exft->ClustSize, dfsGetSectorSize())) != NULL)
         {
            dirsect = dfsExftClust2LSN( dfsExftLSN2Clust( lsn));
            if (dfsRead( dirsect, exft->ClustSize, sb) == NO_ERROR)
            {
               S_EXFTDIR *sf;                   // entry, with short name
               S_EXFTDIR *fatdir = (S_EXFTDIR *) sb;
               BOOL       fdir   = TRUE;        // possible FAT directory
               ULONG      entry;                // index in fat directory
               S_EXFTDIR  refdent;              // reference DIR-entry
               USHORT     fatEntries;           // entries per sector

               fatEntries = (dfsGetSectorSize() * exft->ClustSize) / sizeof(S_EXFTDIR);
               for (entry = 0; (entry < fatEntries) && fdir; entry++)
               {
                  sf = &(fatdir[entry]);        // ptr to current entry

                  switch (sf->EntryType & ~EXFAT_DIRBIT_IN_USE)
                  {
                     case EXFAT_DEC_VLABEL:     // accept valid types, may do more checks
                        break;

                     case EXFAT_DEC_BITMAP:     // like clustervalue < limit and not 0
                     case EXFAT_DEC_UPCASE:
                        break;

                     case EXFAT_DEC_FILE:
                        if      ((sf->fl.DtAccess.u == 0)         || //- no zero date (1-1-1980)
                                 (sf->fl.DtAccess.u > MAX_S_DATE) || //- allow valid date and
                                 (sf->fl.TmAccess.u > MAX_S_TIME)  ) //- valid time only
                        {
                           TRACEX(("dir-entry failed on invalid ACCESS date/time, entry: %lu\n", entry));
                           fdir = FALSE;
                        }
                        else if ((sf->fl.DtModify.u == 0)         || //- no zero date (1-1-1980)
                                 (sf->fl.DtModify.u > MAX_S_DATE) || //- allow valid date and
                                 (sf->fl.TmModify.u > MAX_S_TIME) || //- valid time only
                                 (sf->fl.MsModify   > MAX_S_MS10)  ) //- valid 10-msecs only
                        {
                           TRACEX(("dir-entry failed on invalid MODIFY date/time, entry: %lu\n", entry));
                           fdir = FALSE;
                        }
                        else if ((sf->fl.DtCreate.u == 0)         || //- no zero date (1-1-1980)
                                 (sf->fl.DtCreate.u > MAX_S_DATE) || //- allow valid date and
                                 (sf->fl.TmCreate.u > MAX_S_TIME) || //- valid time only
                                 (sf->fl.MsCreate   > MAX_S_MS10)  ) //- valid 10-msecs only
                        {
                           TRACEX(("dir-entry failed on invalid CREATE date/time, entry: %lu\n", entry));
                           fdir = FALSE;
                        }
                        break;

                     case EXFAT_ANY_STREAM:
                        //- to be refined, do some basic checks like valid cluster
                        break;

                     case EXFAT_ANY_FILENAME:
                        //- no checks possible (any UNICODE string is valid)
                        break;

                     case EXFAT_DIR_VOLGUID:
                     case EXFAT_DIR_TEXFATPAD:
                     case EXFAT_DIR_WINCEACT:
                        //- to be refined
                        break;

                     case EXFAT_DIR_EODIR:
                        memset(     &refdent,  0, sizeof(S_EXFTDIR));
                        if (memcmp( sf, &refdent, sizeof(S_EXFTDIR)) != 0)
                        {
                           fdir = FALSE;
                           TRACEX(("dir-entry failed on non-zero fields in entry: %lu\n", entry));
                        }
                        break;

                     default:
                        if ((sf->EntryType & EXFAT_DIRBITSUNUSED) != 0)
                        {
                           fdir = FALSE;
                           TRACEX(("dir-entry failed on invalid entrytype in entry: %lu\n", entry));
                        }
                        break;
                  }
               }
               if (fdir)                        // valid DIR cluster
               {
                  rc = ST_DIREC;                // any kind of dir-sector
               }
               csType  = rc;                    // update type cache
               csFirst = dirsect;
               csLast  = dirsect + exft->ClustSize -1;
            }
            TxFreeMem( sb);
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsExftDirClusterType'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate Cluster-nr to LSN
/*****************************************************************************/
ULONG dfsExftCl2Lsn                             // RET   LSN for this cluster
(
   ULONG               cluster,                 // IN    cluster number
   ULONG               d2,                      // IN    dummy
   char               *p1,                      // IN    dummy
   void               *p2                       // IN    dummy
)
{
   ULONG               lsn;

   if ((cluster < 2) && (exft->Root != 0))      // CL 0/1 and initialized ?
   {
      lsn = exft->Root;                         // allow cluster 0/1 in DIR
   }                                            // entries to point to root
   else                                         // real clusters in ClHeap
   {
      lsn = exft->ClHeap + ((cluster -2) * exft->ClustSize);
   }
   //- TRACES(("Cl:0x%8.8lx => lsn:0x%8.8lx\n", cluster, lsn));
   return (lsn);
}                                               // end 'dfsExftCl2Lsn'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate LSN to Cluster-nr
/*****************************************************************************/
ULONG dfsExftLsn2Cl                             // RET   cluster for this LSN
(
   ULONG               lsn,                     // IN    LSN
   ULONG               d2,                      // IN    dummy
   char               *p1,                      // IN    dummy
   void               *p2                       // IN    dummy
)
{
   ULONG               cluster = 2;             // minimal valid cluster

   if (lsn == DFS_MAX_PSN)                      // query max Cluster-nr
   {
      cluster = exft->RawClust - 1;              // highest is #clusters - 1
   }
   else if (lsn >= exft->ClHeap)
   {
      cluster = ((lsn - exft->ClHeap) / exft->ClustSize) + 2;
   }
   else if (lsn >= exft->FatOffset)
   {
      cluster = 1;                              // simulate, CL 1 is FAT area
   }
   else
   {
      cluster = 0;                              // simulate, CL 0, upto 1st FAT
   }
   //- TRACES(("lsn:0x%8.8lx => Cl:0x%8.8lx\n", lsn, cluster));
   return (cluster);
}                                               // end 'dfsExftLsn2Cl'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get cluster-value from a Directory entry structure
/*****************************************************************************/
ULONG dfsExftDir2Clust                          // RET   cluster value
(
   S_EXFTDIR           *entry                   // IN    FAT directory entry
)
{
   ULONG               cluster = entry->st.Cluster;

   return (cluster);
}                                               // end 'dfsExftDir2Clust'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set cluster-value in a Directory entry structure
/*****************************************************************************/
void dfsExftClust2Dir
(
   ULONG               cluster,                 // IN    cluster value to set
   S_EXFTDIR           *entry                   // INOUT FAT directory entry
)
{
   entry->st.Cluster = cluster;
}                                               // end 'dfsExftClust2Dir'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Check if cluster could be the 1st root-DIR sector (DIR but no '..' entry)
/*****************************************************************************/
BOOL dfsExftRootCandidate                       // RET   Could be a root DIR
(
   ULONG               cluster                  // IN    target cluster
)
{
   BOOL                rc = FALSE;              // function return
   BYTE               *sb = NULL;               // sector buffer
   ULONG               dirsect;

   ENTER();
   TRARGS(("cluster: %8.8lx\n", cluster));

   if ((sb = TxAlloc(1, dfsGetSectorSize())) != NULL)
   {
      dirsect = dfsExftClust2LSN( cluster);
      if (dfsRead( dirsect, 1, sb) == NO_ERROR)
      {
         switch (dfsIdentifySector( dirsect, 0, sb))
         {
            case ST_ROOTD:
            case ST_DIREC:
               rc = TRUE;
               break;

            default:
               break;
         }
      }
      TxFreeMem(sb);
   }
   BRETURN (rc);
}                                               // end 'dfsExftRootCandidate'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get cluster-value from the Fat Cache structure (32 bit entries)
/*****************************************************************************/
ULONG dfsExftValue                              // RET   cluster value
(
   ULONG               clust                    // IN    cluster to get value
)
{
   ULONG               value = 0;               // rc, cluster value
   DFSEXFTCACHE        *this = &exft->CacheA;
   DFSEXFTCACHE        *oldest;

   if ((clust < this->First) || (clust >= (this->First + DFSEXFT_CACHE)))
   {
      this = &exft->CacheB;
      if ((clust < this->First) || (clust >= (this->First + DFSEXFT_CACHE)))
      {
         //- Not in either cache, select a cache to be reloaded with needed info
         oldest = (this->Stamp < exft->CacheA.Stamp) ? this : &exft->CacheA;
         if (((this->Dirty == TRUE ) && (exft->CacheA.Dirty == TRUE )) ||
             ((this->Dirty == FALSE) && (exft->CacheA.Dirty == FALSE))  )
         {
            this = oldest;                      // same, use oldest, causing toggle
         }
         else                                   // reload the non-Dirty one, to allow
         {                                      // successive Rd/Wr without thrashing
            if (this->Dirty == TRUE)            // B is dirty
            {
               this = &exft->CacheA;            // so reload and usa A
            }
         }
         TRACES(("FatValue for 0x%8.8lx, not cached, reload: cache-%c\n", clust, this->Id));
         if (this->Dirty == TRUE)               // selected cache is Dirty
         {
            //- Flush the dirty cache before reloading it
            dfsExftFlushFAT( this);
         }
         dfsExftGetFAT( clust, this);           // reload needed part of FAT
      }
   }
   value = this->Value[clust % DFSEXFT_CACHE];  // get cluster value from cache

   return (value);
}                                               // end 'dfsExftValue'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set cluster-value in the Fat Cache structure (32 bit entries)
/*****************************************************************************/
ULONG dfsExftSetCluster                         // RET   NO_CHANGE / NO_ERROR
(
   ULONG               clust,                   // IN    cluster to get value
   ULONG               value                    // IN    FAT-value to be set
)
{
   ULONG               rc = NO_ERROR;
   DFSEXFTCACHE        *this = &exft->CacheA;
   DFSEXFTCACHE        *oldest;

   if ((clust < this->First) || (clust >= (this->First + DFSEXFT_CACHE)))
   {
      this = &exft->CacheB;
      if ((clust < this->First) || (clust >= (this->First + DFSEXFT_CACHE)))
      {
         //- Not in either cache, select a cache to be reloaded with needed info
         oldest = (this->Stamp < exft->CacheA.Stamp) ? this : &exft->CacheA;
         if (((this->Dirty == TRUE ) && (exft->CacheA.Dirty == TRUE )) ||
             ((this->Dirty == FALSE) && (exft->CacheA.Dirty == FALSE))  )
         {
            this = oldest;                      // same, use oldest, causing toggle
         }
         else                                   // reload the Dirty one, to allow
         {                                      // successive Rd/Wr without thrashing
            if (this->Dirty == FALSE)           // B is clean
            {
               this = &exft->CacheA;            // so reload and use A
            }
         }
         TRACES(("SetCluster for 0x%8.8lx, not cached, reload: cache-%c\n", clust, this->Id));
         if (this->Dirty == TRUE)               // selected cache is Dirty
         {
            //- Flush the dirty cache before reloading it
            dfsExftFlushFAT( this);
         }
         rc = dfsExftGetFAT( clust, this);      // reload needed part of FAT
      }
   }
   if (this->Value[clust % DFSEXFT_CACHE] != value)
   {
      if (this->Dirty == FALSE)
      {
         TRACES(("Cache %c now DIRTY on writing: 0x%8.8lx(%7lx)\n", this->Id, clust, value));
      }
      this->Value[clust % DFSEXFT_CACHE] = value; // set cluster value in cache
      this->Dirty = TRUE;
   }
   else
   {
      rc = DFS_NO_CHANGE;                       // value written same as existing
   }
   return (rc);
}                                               // end 'dfsExftSetCluster'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Reset bad-sector admin, change each BAD cluster in the FAT to FREE
/*****************************************************************************/
ULONG dfsExftResetBadClus                       // RET   ref cluster or 0
(
   void
)
{
   ULONG               rc = NO_ERROR;           // function return
   ULONG               cBase;                   // lowest cl in cache
   ULONG               cOffs;                   // offset cl in cache
   ULONG               cFirst;                  // first offset to check
   ULONG               cLast;                   // last offset to check
   ULONG               bads = 0;                // total bad clusters
   DFSEXFTCACHE        *cache = &exft->CacheA;

   ENTER();
   TRACES(( "Fat1:%8.8lx Fat2:%8.8lx InUse:%8.8lx\n", exft->Fat1, exft->Fat2, exft->FatInUse));

   TxPrint( "\nScanning FAT area for clusters marked 'BAD' ...\n");

   //- Since FAT areas might be changed, it is fine to use and invalidate the caches
   exft->CacheB.First  = EXFT_NO_CLUST;         // invalidate B cache

   for ( cBase  = 0;                            // Big steps, whole cache ...
        (cBase <= exft->MapClust) && (rc == NO_ERROR) && !TxAbort();
         cBase += DFSEXFT_CACHE)
   {
      dfsExftGetFAT( cBase, cache);              // cache needed part of FAT

      cFirst = (cBase == 0) ? 2 : 0;            // first valid cl is 2
      if ((cBase + DFSEXFT_CACHE) > exft->MapClust)
      {
         cLast = exft->MapClust % DFSEXFT_CACHE;     // last valid offset
      }
      else
      {
         cLast = DFSEXFT_CACHE -1;               // last offset within cache
      }
      for (cOffs = cFirst; (cOffs <= cLast) && (rc == 0); cOffs++)
      {
         if (cache->Value[cOffs] == EXFAT_BADCLUST)
         {
            TRACES(( "Reset BAD cluster nr: %8.8lx\n", cBase + cOffs));
            cache->Value[cOffs] = EXFAT_FREECLUS;
            bads++;
            cache->Dirty = TRUE;
         }
      }
      if (cache->Dirty)
      {
         rc = dfsExftFlushFAT( cache);
      }
   }
   if (bads > 0)
   {
      dfsSiz8( "\nTotal bad sectors : ", bads * exft->ClustSize, ", have been reset\n");
   }
   else
   {
      TxPrint( "\nNo bad sectors present, no action required ...\n");
   }
   RETURN (rc);
}                                               // end 'dfsExftResetBadClus'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Search reference to specified cluster in FAT (cluster pointing to this one)
// Note: Dont use regular cache to avoid trashing that just by searching here
/*****************************************************************************/
ULONG dfsExftFindReference                       // RET   ref cluster or 0
(
   ULONG               cluster                  // IN    target cluster
)
{
   ULONG               rc = 0;                  // function return
   ULONG               cBase;                   // lowest cl in cache
   ULONG               cOffs;                   // offset cl in cache
   ULONG               cFirst;                  // first offset to check
   ULONG               cLast;                   // last offset to check
   DFSEXFTCACHE         cache;                   // allocated local cache

   ENTER();
   TRARGS(("Find cluster ref for: %8.8lx, in FAT at %8.8lx\n", cluster, exft->FatInUse));

   //- avoid use of FatValue() and use local cache to gain speed and avoid cache trashing
   if ((cache.Value = TxAlloc(DFSEXFT_CACHE, sizeof(ULONG))) != NULL)
   {
      for ( cBase  = 0;                         // Big steps, whole cache ...
           (cBase <= exft->MapClust) && (rc == 0) && !TxAbort();
            cBase += DFSEXFT_CACHE)
      {
         dfsExftGetFAT( cBase, &cache);          // cache needed part of FAT

         cFirst = (cBase == 0) ? 2 : 0;         // first valid cl is 2
         if ((cBase + DFSEXFT_CACHE) > exft->MapClust)
         {
            cLast = exft->MapClust % DFSEXFT_CACHE; // last valid offset
         }
         else
         {
            cLast = DFSEXFT_CACHE -1;            // last offset within cache
         }
         for (cOffs = cFirst; (cOffs <= cLast) && (rc == 0); cOffs++)
         {
            if (cache.Value[cOffs] == cluster)
            {
               rc = cBase + cOffs;
            }
         }
      }
      TxFreeMem( cache.Value);
   }
   RETURN (rc);
}                                               // end 'dfsExftFindReference'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Calulate 32-bit CRC over FAT-contents (to compare FAT1 and FAT2 perhaps :-)
/*****************************************************************************/
ULONG dfsExftCalculateFatCRC                     // RET   32-bit CRC
(
   ULONG               fatlsn                   // IN    first sector of FAT
)
{
   ULONG               rc = 0;                  // function return
   ULONG               cBase;                   // lowest cl in cache

   ENTER();
   TRARGS(("fatlsn: %8.8lx\n", fatlsn));

   #if defined (USEWINDOWING)
      if (txwIsWindow( TXHWND_DESKTOP))
      {
         dfsa->nextStatus = clock() + CLOCKS_PER_SEC; // only after 1 second ...
      }
   #endif
   dfsExftSetFatInUse( fatlsn);                  // flush cache and switch FAT

   for ( cBase  = 0;                            // Big steps, whole cache ...
        (cBase <= exft->MapClust) && !TxAbort();
         cBase += DFSEXFT_CACHE)
   {
      #if defined (USEWINDOWING)
         if (txwIsWindow( TXHWND_DESKTOP))
         {
            if (clock() > dfsa->nextStatus)
            {
               TXTM    status;
               sprintf( status, "Calculate CRC FAT%c, done %6lu of %lu",
                       (fatlsn == exft->Fat1) ? '1' :
                       (fatlsn == exft->Fat2) ? '2' : 'x',
                        cBase, exft->MapClust);
               txwSetSbviewStatus( status, cSchemeColor);
               dfsa->nextStatus = clock() + DFSP_PRC_INTERVAL;
            }
         }
      #endif

      dfsExftGetFAT(  cBase, &exft->CacheA);      // cache needed part of FAT
      rc ^= TxCrc32( exft->CacheA.Value, DFSEXFT_CACHE * sizeof(ULONG));
      TRACES(( "CRC value from cluster %8.8lx = %8.8lx\n", cBase, rc));
   }
   RETURN (rc);
}                                               // end 'dfsExftCalculateFatCRC'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set the active FAT LSN to be used for fat references using the cache
/*****************************************************************************/
void dfsExftSetFatInUse
(
   ULONG               fat2Use                  // IN    first sector of FAT
)
{
   ENTER();
   TRARGS(("fat2Use: %8.8lx\n", fat2Use));

   if (exft->FatInUse != fat2Use)
   {
      //- Flush changes to disk, BEFORE switching to other FAT disk area
      if (exft->CacheA.Dirty)
      {
         dfsExftFlushFAT( &exft->CacheA);
      }
      if (exft->CacheB.Dirty)
      {
         dfsExftFlushFAT( &exft->CacheB);
      }
      exft->FatInUse      = fat2Use;            // switch
      exft->CacheA.First  = EXFT_NO_CLUST;
      exft->CacheB.First  = EXFT_NO_CLUST;      // invalidate FAT caches
   }
   VRETURN();
}                                               // end 'dfsExftSetFatInUse'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Copy FAT-part from disk to specified FAT cache structure (32 bit entries)
/*****************************************************************************/
ULONG dfsExftGetFAT                             // RET   result
(
   ULONG               clust,                   // IN    cluster value to get
   DFSEXFTCACHE        *cache                   // INOUT cache structure used
)
{
   ULONG               rc  = NO_ERROR;
   LSN                 lsn;                     // LSN of 1st sector to read
   ULONG               sectors;                 // nr of FAT sectors to read
   BYTE               *dfat;                    // FAT sector buffer
   ULONG               clusters;                // cluster values in chunk

   ENTER();

   cache->First  = clust -  (clust % DFSEXFT_CACHE);

   if ((clusters = (exft->MapClust - cache->First)) > DFSEXFT_CACHE)
   {
      clusters = DFSEXFT_CACHE;
   }

   TRARGS(("FAT LSN:0x%8.8lx for cl:0x%8.8lx, first:0x%8.8lx, clusters:0x%4.4lx  cache:%c\n",
            exft->FatInUse, clust, cache->First, clusters, cache->Id));

   if ((exft->FatInUse != 0) && (clusters != 0))
   {
      dfat    = (BYTE   *) cache->Value;
      lsn     = cache->First   / (dfsGetSectorSize() / sizeof(ULONG));
      sectors = ((clusters -1) / (dfsGetSectorSize() / sizeof(ULONG))) +1;
      TRACES(( "Reading %lu FAT sectors, relative lsn:%8.8lx\n", sectors, lsn));

      lsn += exft->FatInUse;                     // add to start of FAT lsn
      if ((dfat != NULL) && (cache->Value != NULL))
      {
         //- make sure whole cache-buffer is ZERO to get reliable CRC
         //- values for partially filled ones (the last one)

         memset( cache->Value, 0, (DFSEXFT_CACHE * sizeof(ULONG)));

         if ((rc = dfsRead( dfstAreaSN2LSN( DFSTORE, lsn),
                             sectors, dfat)) == NO_ERROR)
         {
            if (++lastCacheStamp == 0)          // when wrapped, reset BOTH caches ages
            {
               exft->CacheA.Stamp = 0;
               exft->CacheB.Stamp = 0;
               lastCacheStamp = 1;              // will be assigned to current one
            }
            cache->Stamp = lastCacheStamp;
            cache->Dirty = FALSE;               // clean now

            TRHEXS(70,dfat, 128, "start of physical FAT");
            TRACES(("Converted to FAT cache, age is: 0x%8.8lx, contents:\n", lastCacheStamp));
            TRHEXS(70,cache->Value, 0x80, "start of FAT cache");
         }
         else
         {
            dfsLsx("\nFailed to read FAT sectors at LSN : ", lsn, CBR, "\n");
         }
         if ((ULONG *) dfat != cache->Value)    // temp buffer allocated ?
         {
            TxFreeMem(dfat);
         }
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   else
   {
      rc = DFS_BAD_STRUCTURE;
   }
   RETURN (rc);
}                                               // end 'dfsExftGetFAT'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Flush specified FAT cache (32 bit entries) back to disk FatInUse (+ other)
/*****************************************************************************/
ULONG dfsExftFlushFAT                            // RET   result
(
   DFSEXFTCACHE        *cache                    // INOUT cache structure to write
)
{
   ULONG               rc  = NO_ERROR;
   LSN                 lsn;                     // LSN of 1st sector to read
   ULONG               sectors;                 // nr of FAT sectors to read
   ULONG              *lfat;                    // logical FAT area, 32 bit
   BYTE               *dfat;                    // FAT sector buffer
   ULONG               clusters;                // cluster values in chunk

   ENTER();

   if ((clusters = (exft->MapClust - cache->First)) > DFSEXFT_CACHE)
   {
      clusters = DFSEXFT_CACHE;
   }
   TRARGS(("FAT LSN:0x%8.8lx first:0x%8.8lx, clusters:0x%8.8lx  cache:%c\n",
                  exft->Fat1, cache->First, clusters, cache->Id));

   if (clusters != 0)
   {
      dfat    = (BYTE   *) cache->Value;
      lsn     = cache->First   / (dfsGetSectorSize() / sizeof(ULONG));
      sectors = ((clusters -1) / (dfsGetSectorSize() / sizeof(ULONG))) +1;

      lfat = cache->Value;
      if ((dfat != NULL) && (lfat != NULL))
      {
         TRACES(( "Writing %lu FAT sectors, relative lsn:%8.8lx\n", sectors, lsn));
         //- write to first-fat area
         if ((rc = dfsWrite( dfstAreaSN2LSN( DFSTORE, lsn + exft->FatInUse),
                             sectors, dfat)) == NO_ERROR)
         {
            TRHEXS(70,cache->Value, 0x80, "start of FAT cache");
            TRACES(("Converted to physical FAT, contents:\n"));
            TRHEXS(70,dfat, 128, "start of physical FAT");

            //- write to second-fat area, IFF active FAT is Fat1 or Fat2 (not a temp one)
            if      ((exft->FatInUse == exft->Fat1) && (exft->Fat2 != 0)) // 1 used, 2 present
            {
               dfsWrite( dfstAreaSN2LSN( DFSTORE, lsn + exft->Fat2), sectors, dfat);
            }
            else if ((exft->FatInUse == exft->Fat2) && (exft->Fat1 != 0)) // 2 used, 1 present
            {
               dfsWrite( dfstAreaSN2LSN( DFSTORE, lsn + exft->Fat1), sectors, dfat);
            }
            if (++lastCacheStamp == 0)          // when wrapped, reset BOTH caches ages
            {
               exft->CacheA.Stamp = 0;
               exft->CacheB.Stamp = 0;
               lastCacheStamp = 1;              // will be assigned to current one
            }
            cache->Stamp = lastCacheStamp;
            cache->Dirty = FALSE;               // clean again now
         }
         else
         {
            dfsLsx("\nFailed to write FAT sectors at LSN : ", lsn + exft->FatInUse, CBR, "\n");
         }
         if ((ULONG *) dfat != cache->Value)   // temp buffer allocated ?
         {
            TxFreeMem(dfat);
         }
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   else
   {
      rc = DFS_BAD_STRUCTURE;
   }
   RETURN (rc);
}                                               // end 'dfsExftFlushFAT'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Show logical FAT
/*****************************************************************************/
void dfsExftShowFAT
(
   ULONG               size,                    // IN    nr of entries
   ULONG               start,                   // IN    first cl# to show
   char               *options                  // IN    display options
)
{
   ULONG               i;
   ULONG               value;
   TXTM                tbuf;
   TXLN                text;

   ENTER();

   strcpy( text, "");
   for (i = 0; (i < size) && !TxAbort() && exft->CacheA.Value; i++)
   {
      if ((i % 8) == 0)
      {
         TxPrint( "%s\n%s%08.8lX = %s", text, CBZ, start + i, CNN);
         strcpy( text, "");
      }
      value = dfsExftValue(start + i);
      if ((start + i) < 2)                      // Invalid cluster numbers
      {
         if ((start + i) == 0)
         {
            sprintf( tbuf, " <%sMedia-type%s %2.2x%s>  ",
                     CNC, CBC, (USHORT) (value & 0xff), CNN);
            strcat(  text, tbuf);
         }
      }
      else
      {
         if (value > EXFAT_MAXCLUST)
         {
            if (value >= EXFAT_MEDIA_CL)
            {
               sprintf( tbuf, " <%seof%s>   ", CNC, CNN);
            }
            else
            {
               sprintf( tbuf, " <%sbad%s>   ", CBR, CNN);
            }
         }
         else
         {
            if (value == 0)
            {
               sprintf( tbuf, " <free>  ");
            }
            else
            {
               sprintf( tbuf, "%8lX ", value);
            }
         }
         strcat(  text, tbuf);
      }
   }
   TxPrint("%s\n", text);
   VRETURN();
}                                               // end 'dfsExftShowFAT'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Count number of used Dir-entries for S_SPACE structure
/*****************************************************************************/
ULONG dfsExftDirUsed                            // RET   matching entries
(
   ULONG               chunks,                  // IN    nr of space entries
   S_SPACE            *space                    // IN    space allocation
)
{
   ULONG               rc;
   ULONG               used = 0;                // nr of used entries
   ULONG               chunk;                   // index in space-area
   ULONG               sect;                    // sectors to handle
   ULONG               entry;                   // index in fat directory
   USHORT              bps = dfsGetSectorSize();
   USHORT              entries = (bps / sizeof(S_EXFTDIR));
   BYTE               *dirbuf;
   S_EXFTDIR           *fatdir;                  // Fat directory sector

   ENTER();

   TRACES(("Read %lu chunks into %u entries of %u bytes\n",
            chunks, entries, sizeof(S_EXFTDIR)));

   if ((dirbuf = TxAlloc( 1, bps)) != NULL)     // allocate one sector
   {
      fatdir = (S_EXFTDIR *) dirbuf;

      for (chunk = 0; (chunk < chunks)  && (!TxAbort()); chunk++)
      {
         for (sect = 0; (sect < space[chunk].size)  && (!TxAbort()); sect++)
         {
            if ((rc = dfsRead(space[chunk].start + sect, 1, dirbuf)) == NO_ERROR)
            {
               for (entry = 0; entry < entries; entry++)
               {
                  if (fatdir[entry].EntryType & EXFAT_DIRBIT_IN_USE)
                  {
                     used++;
                  }
               }
            }
         }
      }
      TxFreeMem( dirbuf);
   }
   RETURN (used);
}                                               // end 'dfsExftDirUsed'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get label-string from the EFAT Root-directory
/*****************************************************************************/
BOOL dfsExftRootLabel                           // RET   Label found
(
   S_BOOTR            *boot,                    // IN    boot record for vol
   LSN                 offset,                  // IN    Offset for boot-rec
   char               *label                    // OUT   label string
)
{
   ULONG               rc = NO_ERROR;
   BOOL                found = FALSE;
   ULONG               size;                    // sectors in Root dir
   LSN                 root;                    // LSN of first root sector
   ULONG               sect;                    // sectors to handle
   ULONG               entry;                   // index in fat directory
   USHORT              bps = dfsGetSectorSize();
   USHORT              entries = (bps / sizeof(S_EXFTDIR));
   BYTE               *dirbuf;
   S_EXFTDIR          *fatdir;                  // exFat directory sector

   ENTER();

   //- Note: dfsa->boot is NOT valid when calling this (might be NULL!)

   if ((dirbuf = TxAlloc( 1, bps)) != NULL)     // allocate one sector
   {
      fatdir = (S_EXFTDIR *) dirbuf;

      size  = (1 << boot->ex.SpcShift);         // min size of Root-Dir, 1 Cl
      root  = (boot->ex.RootCluster -2) * size + boot->ex.ClHeapOffset;

      TRACES(("Root-Dir with %lu sectors at LSN %8.8lX, base: %8.8lX\n", size, root, offset));

      for (sect = 0; (sect < size) && (rc == NO_ERROR) && !found; sect++)
      {
         rc = dfsRead(offset + root + sect, 1, dirbuf);
         if (rc == NO_ERROR)
         {
            for (entry = 0; (entry < entries) && !found; entry++)
            {
               switch (fatdir[entry].EntryType)
               {
                  case EXFAT_DIR_VLABEL:             // volumelabel entry
                     TRACES(("Label found in Dir-entry %lu\n", entry));

                     TxUnic2Ascii( fatdir[entry].lb.Name, label, fatdir[entry].lb.Length);
                     found = TRUE;
                     break;

                  default:
                     break;
               }
            }
         }
      }
      TxFreeMem( dirbuf);
   }
   RETURN (found);
}                                               // end 'dfsExftRootLabel'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get starting cluster for the specified Bitmap file from root directory
/*****************************************************************************/
BOOL dfsExftRootGetBmCluster                    // RET   Bitmap cluster or 0
(
   S_BOOTR            *boot,                    // IN    boot record for vol
   LSN                 offset,                  // IN    Offset for boot-rec
   BYTE                mapnr,                   // IN    Bitmap 1st is 0
   ULONG              *cluster,                 // OUT   cluster number
   ULONG              *byteSize                 // OUT   size in bytes
)
{
   ULONG               rc = NO_ERROR;
   BOOL                found = FALSE;
   ULONG               size;                    // sectors in Root dir
   LSN                 root;                    // LSN of first root sector
   ULONG               sect;                    // sectors to handle
   ULONG               entry;                   // index in fat directory
   USHORT              bps = dfsGetSectorSize();
   USHORT              entries = (bps / sizeof(S_EXFTDIR));
   BYTE               *dirbuf;
   S_EXFTDIR          *fatdir;                  // exFat directory sector

   ENTER();

   //- Note: dfsa->boot is NOT valid when calling this (might be NULL!)

   if ((dirbuf = TxAlloc( 1, bps)) != NULL)     // allocate one sector
   {
      fatdir = (S_EXFTDIR *) dirbuf;

      size  = (1 << boot->ex.SpcShift);         // min size of Root-Dir, 1 Cl
      root  = (boot->ex.RootCluster -2) * size + boot->ex.ClHeapOffset;

      TRACES(("Root-Dir with %lu sectors at LSN %8.8lX, base: %8.8lX\n", size, root, offset));

      for (sect = 0; (sect < size) && (rc == NO_ERROR) && !found; sect++)
      {
         rc = dfsRead(offset + root + sect, 1, dirbuf);
         if (rc == NO_ERROR)
         {
            for (entry = 0; (entry < entries) && !found; entry++)
            {
               switch (fatdir[entry].EntryType)
               {
                  case EXFAT_DIR_BITMAP:        // bitmap entry
                     if ((fatdir[entry].bm.Flags & EXFAT_BITMAP_MASK) == mapnr)
                     {
                        TRACES(("Bitmap found in Dir-entry %lu\n", entry));

                        *cluster  =         fatdir[entry].bm.Cluster;
                        *byteSize = (ULONG) fatdir[entry].bm.Length;
                        found = TRUE;
                     }
                     break;

                  default:
                     break;
               }
            }
         }
      }
      TxFreeMem( dirbuf);

      TRACES(("BM cluster: %lx, Size: 0x%llx = %lld\n", *cluster, *byteSize, *byteSize));
   }
   RETURN (found);
}                                               // end 'dfsExftRootGetBmCluster'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create S_SPACE structure for an EFAT stream entry, upcase or bitmap file
/*****************************************************************************/
ULONG dfsExftAllocStream                        // RET   nr of clusters
(
   S_EXFTDIR          *entry,                   // Fat dir-entry ptr, STREAM
   ULONG              *errors,                  // INOUT error flags
   ULONG              *chunks,                  // OUT   nr of space entries
   S_SPACE           **space                    // OUT   space allocation
)
{
   ULONG               cl      = 0;             // nr of clusters
   S_SPACE            *sp      = NULL;
   ULONG               nr      = 0;

   ENTER();

   TRACES(("EntryType: %2.2hx\n", entry->EntryType));

   switch (entry->EntryType & ~EXFAT_DIRBIT_IN_USE) // normal + deleted
   {
      case EXFAT_DEC_BITMAP:
         cl = dfsExftAllocChain( dfsExftDir2Clust(entry),
                                 entry->bm.Length, errors, chunks, space);
         break;

      case EXFAT_DEC_UPCASE:
         cl = dfsExftAllocChain( dfsExftDir2Clust(entry),
                                 entry->up.Length, errors, chunks, space);
         break;

      case EXFAT_ANY_STREAM:
         TRACES(("GenFlags: 0x%4.4hx  Length:%lld\n", entry->st.GenFlags, entry->st.Length));
         if ((entry->st.GenFlags & EXFAT_CHAIN_INVALID) == 0)
         {
            cl = dfsExftAllocChain( dfsExftDir2Clust(entry),
                                    entry->st.Length, errors, chunks, space);
         }
         else
         {
            if (entry->st.Length > 0)
            {
               nr = 1;                                             //- nr of chunks
               if ((sp = (S_SPACE *) calloc((size_t) nr, sizeof(S_SPACE))) != NULL)
               {
                  ULONG      clustBytes = exft->ClustSize * dfsGetSectorSize();

                  cl = ((entry->st.Length - 1) / clustBytes) + 1;     //- number of clusters

                  sp->start = dfsExftClust2LSN( entry->st.Cluster);   //- LSN start of 1st cluster
                  sp->size  = cl * exft->ClustSize;                   //- size of chunk in sectors

                  TRACES(("cl:%lu  start:%8.8lx  size:%8.8lx\n", cl, sp->start, sp->size));
               }
            }
            *space  = sp;
            *chunks = nr;
         }
         break;

      default:                                  // others are invalid
         TRACES(("Invalid EntryType!"));
         break;
   }
   RETURN (cl);
}                                               // end 'dfsExftAllocStream'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create S_SPACE structure for a FAT allocation chain
/*****************************************************************************/
ULONG dfsExftAllocChain                         // RET   nr of clusters
(
   ULONG               cluster,                 // IN    starting cluster nr
   ULONG               maxsize,                 // IN    max file-size, or 0
   ULONG              *errors,                  // INOUT error flags
   ULONG              *chunks,                  // OUT   nr of space entries
   S_SPACE           **space                    // OUT   space allocation
)
{
   ULONG               cl;                      // nr of clusters
   ULONG               this    = cluster;       // current cluster
   ULONG               last    = 0;             // last handled cluster
   ULONG               size    = 0;             // sectors in this chunk
   ULONG               extents = 0;             // nr of extents
   ULONG               areas   = 0;             // nr of space-areas
   ULONG               climit  = exft->MapClust; // limit on nr of clusters
   S_SPACE            *sp      = NULL;

   ENTER();

   if (maxsize != 0)                            // explicit area-size given
   {                                            // limit == clusters +1 !!!
      climit = (maxsize / (exft->ClustSize * dfsGetSectorSize())) +2;
   }
   TRARGS(("Input cluster: %8.8lX, maxsize: %8.8lX  Cluster limit: %8.8lx\n",
                  cluster,         maxsize,                climit));

   for (cl = 1; (cl <= climit) && !TxAbort() && (cl != 0); cl++)
   {
      TRACES(( "IN-FOR cl: %lu this:%8.8lx last:%8.8lx extents: %lu size:%lu\n",
                       cl,     this,       last,       extents,     size));
      if (this == (last +1))                    // cluster in sequence ?
      {
         size += exft->ClustSize;
      }
      else                                      // create new chunk
      {
         if (areas <= extents)                  // areas used up ?
         {
            areas = areas * 8 + 16;
            TRACES(("(re)Allocate %lu space chunks, rel cl %lu\n", areas, cl))
            sp = (S_SPACE *) realloc( sp, (size_t) (areas+1) * sizeof(S_SPACE));
         }
         if (sp != NULL)
         {
            if (last != 0)                      // not first extent ?
            {
               sp[extents].size = size;         // set size of last extent
               extents++;                       // and move to next one
            }
            size = exft->ClustSize;              // start with 1 cluster
            sp[extents].size  = size;
            sp[extents].start = dfsExftClust2LSN(this);
         }
         else
         {
            cl = 0;
         }
      }
      if (this == EXFAT_EOFCLUST)               // EOF value
      {
         break;                                 // leave
      }
      else                                      // follow the chain
      {
         last = this;
         if ((last >= 2) && (last <= exft->MapClust)) // OK to follow ?
         {
            this = dfsExftValue(last);
            if ((this <= 1) || (last <= 1))     // free cluster, error
            {
               if (errors)
               {
                  *errors |= EF_ALLOC_CL_FREE;
               }
               break;
            }
            else if (this == EXFAT_BADCLUST)
            {
               if (errors)
               {
                  *errors |= EF_ALLOC_CL_BADS;
               }
               break;
            }
         }
         else
         {
            if (errors)
            {
               if (last == EXFAT_BADCLUST)
               {
                  *errors |= EF_ALLOC_CL_BADS;
               }
               else
               {
                  *errors |= EF_CLUSTER_RANGE;
               }
            }
            break;
         }
      }
      if (cl == climit)                         // maximum clusters reached
      {
         if (errors)
         {
            *errors |= EF_RUNAWAY_CHAIN;
         }
      }
   }
   TRACES(( "ENDFOR cl: %lu this:%8.8lx last:%8.8lx extents: %lu size:%lu\n",
                    cl,     this,       last,       extents,     size));
   if (sp != NULL)
   {
      TRACES(( "sp[extents -1].start: %8.8lx        .size: %lu\n",
                sp[extents -1].start, sp[extents -1].size));

      *space  = sp;
      *chunks = extents;
      TRINIT(70);
         dfsSspaceDisplay( SD_DEFAULT, extents, sp);
      TREXIT();
   }
   else
   {
      *space  = NULL;                           // dont return random value!
      *chunks = 0;
      cl = 1;                                   // signal (alloc) failure
   }
   if (errors)
   {
      TRACES(("error flags: 0x%8.8lx\n", *errors));
   }
   RETURN (cl -1);
}                                               // end 'dfsExftAllocChain'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Check if name is a valid EFAT filename
/*****************************************************************************/
BOOL dfsValidExftName
(
   const void         *fn,                      // IN    ptr to EFAT name
   USHORT              len                      // IN    length to check
)
{
   TXLN                name;
   USHORT              lok;

   memset(  name, 0,  TXMAXTM);
   strncpy( name, fn, len);

   lok = strcspn( name, DFS_EXFT_INVALIDS);

   TRACES(("FatName: '%*.*s', len: %u, strlen: %u,  length-ok: %u\n",
            len, len, name, len, strlen(name), lok));
   return (lok == len);
}                                               // end 'dfsValidExftName'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// Convert null-terminated name to a valid dir-entry name (multiple C1 entries)
/*****************************************************************************/
void dfsExftName2Entry
(
   char               *fn,                      // IN    ptr to FILE name
   S_EXFTDIR          *de                       // OUT   File, Stream + Name entries
)
{
   #if defined (NEVER)
   TXLN                local;
   char               *p;

   ENTER();
   strcpy( local, fn);

   //- to be refined, distribute given name over multiple C1 entries in Unicode
   //- must test for valid C0 (length byte) and C1 entries!
   TxStrToUpper( local);
   memset( de->Name, ' ', FAT_NSIZE + FAT_ESIZE);
   if ((p = strrchr(local, '.')) != NULL)       // extention present ?
   {
      *p++ = '\0';                              // terminate base-name
      memcpy(de->Ext, p, strlen(p));            // copy ext
   }
   memcpy( de->Name, local, strlen(local));     // copy base

   TRACES(("Converted '%s' to :\n", fn));
   TRHEXS(70, de, 64, "S_EXFTDIR directory entry");
   VRETURN();
   #endif
}                                               // end 'dfsExftName2Entry'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// Convert a valid dir-entry name (multiple C1) to valid null-terminated name
// Asumes there are 2 more sectors of DIR entries BEFORE and AFTER specified one
/*****************************************************************************/
void dfsExftEntry2Name
(
   S_EXFTDIR          *efile,                   // IN    File, Stream + Name entries
   char               *lfn                      // OUT   ptr to long FILE name
)
{
   S_EXFTDIR          *estream;                 // Stream entry
   S_EXFTDIR          *ename;                   // First name entry
   BYTE                pendingName = 0;
   USHORT              fragmentSize;            // Filename fragment size
   int                 sanity;                  // Max entries to move forward/backward

   ENTER();

   //- to be refined, may add more consistency checks, and perhaps 'ef' error flag
   switch (efile->EntryType & ~EXFAT_DIRBIT_IN_USE) // normal + deleted
   {
      case EXFAT_DEC_VLABEL:
         TxUnic2Ascii( efile->lb.Name, lfn, EXFAT_LABEL_SIZE);
         break;

      case EXFAT_DEC_BITMAP:
         strcpy( lfn, "ExFAT_alloc-bitmap.bin");
         break;

      case EXFAT_DEC_UPCASE:
         strcpy( lfn, "ExFAT_upcase-table.bin");
         break;

      case EXFAT_ANY_STREAM:
      case EXFAT_ANY_FILENAME:                  // need to search backwards to FILE entry
         sanity = 2 * dfsGetSectorSize() / sizeof(S_EXFTDIR);
         while (((efile->EntryType & ~EXFAT_DIRBIT_IN_USE) != EXFAT_DEC_FILE) && (sanity--))
         {
            efile--;                            // move backwards over 1 entry
         }
         if ((efile->EntryType & ~EXFAT_DIRBIT_IN_USE) != EXFAT_DEC_FILE)
         {
            strcpy( lfn, "-no-file-");          // no FILE found, invalid
            break;
         }
      case EXFAT_DEC_FILE:
         strcpy( lfn, "");                      // start with empty name
         estream = efile   + 1;                 // Stream entry
         ename   = estream + 1;                 // First name entry

         pendingName = estream->st.NameLength;

         sanity = 2 * dfsGetSectorSize() / sizeof(S_EXFTDIR);
         TRACES(("Estream name length: %hu, sanity:%d\n", pendingName, sanity));
         while ((pendingName > 0) && (sanity--))
         {
            TRACES(("Remaining name length: %hu, entry-type: 0x%2.2hx, sanity left:%d\n",
                     pendingName, ename->EntryType, sanity));
            if ((ename->EntryType & ~EXFAT_DIRBIT_IN_USE) == EXFAT_ANY_FILENAME)
            {
               if (pendingName > EXFAT_NAME_SIZE)
               {
                  fragmentSize = EXFAT_NAME_SIZE;
                  pendingName -= EXFAT_NAME_SIZE;
               }
               else                             // no more name fragments expected
               {
                  fragmentSize = pendingName;
                  pendingName  = 0;
               }
               TxUnicAppend( ename->nm.Name, lfn, fragmentSize);
            }
            else
            {
               //- may report some warning/error here
            }
            ename++;                            // to next dir name entry
         }
         break;

      default:                                  // others invalid, incl stream/name
         strcpy( lfn, "-invalid-");
         break;
   }
   TRHEXS(70, efile, 128, "S_EXFTDIR entry FILE+STREAM+NAME(s)");
   TRACES(("Converted to : '%s'\n", lfn));
   VRETURN();
}                                               // end 'dfsExftEntry2Name'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display all available info about a directory entry, including C0/C1 entries
/*****************************************************************************/
ULONG dfsExftShowDirectoryEntry
(
   LSN                 dirsect,                 // IN    lsn of FAT dir sector
   int                 entry                    // IN    File, Stream + Name entries
)
{
   ULONG               rc;
   BYTE               *sb  = NULL;              // sector buffer
   S_EXFTDIR          *efile;                   // DIR entry, FILE
   S_EXFTDIR          *stream;                  // DIR entry, STREAM
   TXLN                lfn;

   ENTER();
   TRACES(("dirsect: 0x%8.8lx, entry: 0x%2.2x\n", dirsect, entry));

   strcpy( lfn, "");                            // init first to not-present

   //- Need to allocate/read upto 5 sectors to allow reading LFN up to 255 characters
   if ((sb = TxAlloc(5, dfsGetSectorSize())) != NULL) // 2 extra sectors BEFORE and AFTER
   {
      if ((rc = dfsRead(dirsect - 2, 5, sb)) == NO_ERROR)
      {
         BYTE      *dirdata = sb + (2 * dfsGetSectorSize());
         int        sanity;

         switch (dfsIdentifySector(dirsect, 0, dirdata))
         {
            case ST_ROOTD:
            case ST_DIREC:
               efile   = &(((S_EXFTDIR *) dirdata)[entry]);

               TxPrint("\n");
               dfsExftDirHeader( "DirSec/entry", TRUE); // display header
               dfsLsn(" ", dirsect, CBC, " ");
               TxPrint("%s%2x%s ", CNC, entry, CNN);

               switch (efile->EntryType & ~EXFAT_DIRBIT_IN_USE) // normal + deleted
               {
                  case EXFAT_DEC_VLABEL:
                     TxUnic2Ascii( efile->lb.Name, lfn, EXFAT_LABEL_SIZE);
                     TxPrint( "%32.32s Label: %s\n", "", lfn);
                     break;

                  case EXFAT_DEC_BITMAP:
                     TxPrint( "%23.23s        alloc-bitmap %13llu %8lX\n\n",
                              "", efile->bm.Length, efile->bm.Cluster);

                     if (efile->bm.Length != 0)
                     {
                        dfsExftShowAlloc( efile);
                     }
                     break;

                  case EXFAT_DEC_UPCASE:
                     TxPrint( "%23.23s        upcase-table %13llu %8lX\n\n",
                              "", efile->up.Length, efile->up.Cluster);

                     if (efile->up.Length != 0)
                     {
                        dfsExftShowAlloc( efile);
                     }
                     break;

                  case EXFAT_ANY_STREAM:
                  case EXFAT_ANY_FILENAME:      // need to search backwards to FILE entry
                     sanity = 2 * dfsGetSectorSize() / sizeof(S_EXFTDIR);
                     while (((efile->EntryType & ~EXFAT_DIRBIT_IN_USE) != EXFAT_DEC_FILE) && (sanity--))
                     {
                        efile--;                // move backwards over 1 entry
                     }
                     if ((efile->EntryType & ~EXFAT_DIRBIT_IN_USE) != EXFAT_DEC_FILE)
                     {
                        break;                  // no FILE entry found, fail
                     }
                  case EXFAT_DEC_FILE:
                     stream = efile +1;        // advance to actual stream entry

                     dfsExftShowDirEntry( efile, stream, NORMAL);
                     dfsExftEntry2Name( efile, lfn);
                     TxPrint("%s%s%s\n\n", CBG, lfn, CNN);

                     if ((stream->st.Length != 0) || (efile->fl.Attrib & FATTR_DIRECTORY))
                     {
                        dfsExftShowAlloc( stream);
                     }
                     break;

                  default:                      // ignore others
                     break;
               }
               break;

            default:
               rc = DFS_PENDING;
               break;
         }
      }
      else if (dirsect < exft->Sect)             // should be there
      {
         dfsLsx("\nError reading sector : ", dirsect, CBR, "\n");
      }
      TxFreeMem(sb);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN( rc);
}                                               // end 'dfsExftShowDirectoryEntry'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// FAT filesystem, display DIR header/footer
/*****************************************************************************/
void dfsExftDirHeader
(
   char               *lead,                    // IN    lead text-string
   BOOL                head                     // IN    header (!footer)
)
{
   TXLN                line;


   sprintf(line, " ============ ====================== ======================== ======== ========\n");
   if (!head)
   {
      TxPrint( line);
   }
   TxPrint(                " %s Modified/Creation-Time Attrib/AccessTm/Filename Filesize  Cluster\n", lead);
   if (head)
   {
      TxPrint( line);
   }
}                                               // end 'dfsExftDirHeader'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display EFAT directory entry on two lines, all except filename
/*****************************************************************************/
void dfsExftShowDirEntry
(
   S_EXFTDIR          *entry,                   // IN    EFAT FILE entry
   S_EXFTDIR          *stream,                  // IN    EFAT STREAM entry
   int                 ac                       // IN    active color fg+bg
)
{
   ULONG               cluster = 0;
   TXTM                tbuf;
   TXLN                text;
   signed char         tzOffset;

   sprintf( text, " %4.4u-%2.2u-%2.2u %2.2u:%2.2u:%2.2u.%2.2u",
      entry->fl.DtModify.d.year + 1980, entry->fl.DtModify.d.month,   entry->fl.DtModify.d.day,
      entry->fl.TmModify.t.hours, entry->fl.TmModify.t.minutes, entry->fl.TmModify.t.twosecs * 2 +
                                  entry->fl.MsModify / 100,     entry->fl.MsModify % 100);

   if ((entry->fl.DtAccess.u != entry->fl.DtModify.u) ||
       (entry->fl.TmAccess.u != entry->fl.TmModify.u)  )
   {
      sprintf( tbuf, " %4.4u-%2.2u-%2.2u %2.2u:%2.2u:%2.2u",
         entry->fl.DtAccess.d.year + 1980, entry->fl.DtAccess.d.month, entry->fl.DtAccess.d.day,
         entry->fl.TmAccess.t.hours, entry->fl.TmAccess.t.minutes, entry->fl.TmAccess.t.twosecs * 2);
   }
   else
   {
      sprintf( tbuf, "%20.20s", "");
   }
   strcat( text, tbuf);

   sprintf( tbuf, " %13llu", stream->st.Length);
   strcat( text, tbuf);
   cluster = dfsExftDir2Clust( stream);
   if (cluster != 0)
   {
      sprintf( tbuf, " %8lX", cluster);         // can be 32-bit
      strcat( text, tbuf);
   }
   else                                         // must be an empty file/dir
   {
      strcat( text, " Empty 0");
   }
   tzOffset  =  (entry->fl.TzCreate & 0x7f) | ((entry->fl.TzCreate & 0x40) << 1); // sign-extend 7-bit to 8
   TxPrint( "%s%s\n %s%s%s utc%+2.2hd h", text, CGE,
            ansi[Ccol((CcM | CcI), Ccbg(ac))], (entry->EntryType & EXFAT_DIRBIT_IN_USE) ? "   " : "del", ansi[ ac],
            tzOffset / 4);

   if ((entry->fl.DtCreate.u != entry->fl.DtModify.u  ) ||
       (entry->fl.TmCreate.u != entry->fl.TmModify.u  ) ||
       (entry->fl.MsCreate   != entry->fl.MsModify    )  )
   {
      sprintf( text, " %4.4u-%2.2u-%2.2u %2.2u:%2.2u:%2.2u.%2.2u ",
         entry->fl.DtCreate.d.year + 1980, entry->fl.DtCreate.d.month, entry->fl.DtCreate.d.day,
         entry->fl.TmCreate.t.hours, entry->fl.TmCreate.t.minutes, entry->fl.TmCreate.t.twosecs * 2 +
                                     entry->fl.MsCreate / 100,     entry->fl.MsCreate % 100);
   }
   else
   {
      sprintf( text, "%24.24s", "");
   }
   dfstrFatAttrib( text, entry->fl.Attrib);

   TxPrint( "%s ", text);                       // will be followed by long filename ...

}                                               // end 'dfsExftShowDirEntry'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display FAT allocation chain from cluster
/*****************************************************************************/
void dfsExftShowAlloc
(
   S_EXFTDIR          *entry                    // IN    STREAM dir entry
)
{
   ULONG               rc = NO_ERROR;
   DFSISPACE           ispace;
   ULONG               ef = 0;                  // Initialize error flag!

   ENTER();
   TRARGS(("entryType: 0x%2.2hx  cl  %8.8lx\n", entry->EntryType, entry->st.Cluster));

   switch (entry->EntryType & ~EXFAT_DIRBIT_IN_USE)
   {
      case EXFAT_ANY_STREAM:
      case EXFAT_DEC_BITMAP:
      case EXFAT_DEC_UPCASE:
         dfsLsx("Space for cluster : ", entry->st.Cluster, CBM, "  ");
         TxPrint( "(%s)\n", (entry->st.GenFlags & EXFAT_CHAIN_INVALID) ? "Contiguous" : "FAT-chain");
         if (entry->st.GenFlags & EXFAT_CHAIN_INVALID)
         {
            ispace.chunks = 1;                  // return contiguous clusters
            if ((ispace.space = (S_SPACE *) calloc((size_t) ispace.chunks, sizeof(S_SPACE))) != NULL)
            {
               ULONG      clustBytes = exft->ClustSize * dfsGetSectorSize();

               ispace.space->start = dfsExftClust2LSN( entry->st.Cluster); // LSN start of 1st cluster
               ispace.space->size  = ((entry->st.Length + clustBytes - 1) // size of chunk in sectors
                                          / clustBytes) * exft->ClustSize;
            }
         }
         else                                   // valid FAT chain (fragmented)
         {
            dfsExftAllocChain( entry->st.Cluster, 0, &ef, &ispace.chunks, &ispace.space);
         }
         ispace.clsize = exft->ClustSize;
         if (ispace.space != NULL)
         {
            ULONG      sdo = SD_TOPLINE | SD_SUMMARY;
            ULONG      failedSectors = 0;

            TxPrint( " Allocation Check : ");
            if (dfsCheckAllocForSpace( &ispace, 0,
                    ((entry->EntryType & EXFAT_DIRBIT_IN_USE) != 0), &failedSectors) == NO_ERROR)
            {
               TxPrint("%sOK%s%s\n", CBG, CNN,
                       "         Saveto/Recover/Undelete of filedata is possible");
            }
            else
            {
               TxPrint("%sFAIL%s on %lu out of %lu Sectors\n", CBR, CNN, failedSectors,
                                 dfsSspaceSectors( TRUE, ispace.chunks, ispace.space));

            }
            if (!TxaOptUnSet('X'))              // unless -X- specified
            {
               sdo |= SD_NAVDOWN;               // next is data sector
            }
            rc = dfsSspaceDisplay( sdo, ispace.chunks, ispace.space);
            free( ispace.space);
         }
         if (ef != 0)
         {
            dfsExftDispError( ef, 0, "\nWarning: ", NULL);
         }
         break;

      default:
         break;
   }
   VRETURN();
}                                               // end 'dfsExftShowAlloc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Retrieve first-cluster number for specified Dir-sector-LSN + index (LsnInf)
/*****************************************************************************/
ULONG dfsExftLsnInf2Cluster                         // RET   First cluster or 0
(
   ULONG               lsn,                     // IN    Directory LSN
   ULONG               info                     // IN    directory entry
)
{
   ULONG               rc  = 0;
   S_EXFTDIR           direntry;                // Fat directory entry

   ENTER();

   if (dfsExftLsnInf2Entry( lsn, info, &direntry) == NO_ERROR)
   {
      rc = dfsExftDir2Clust( &direntry);
   }
   RETURN (rc);
}                                               // end 'dfsExftLsnInf2Cluster'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Retrieve directory-entry for specified Dir-sector-LSN + index (LsnInf)
/*****************************************************************************/
ULONG dfsExftLsnInf2Entry
(
   ULONG               lsn,                     // IN    Directory LSN
   ULONG               info,                    // IN    directory entry
   S_EXFTDIR          *direntry                 // OUT   directory entry
)
{
   ULONG               rc  = NO_ERROR;
   USHORT              bps = dfsGetSectorSize();
   BYTE               *dirbuf;
   S_EXFTDIR          *fatdir;                  // Fat directory sector

   ENTER();
   TRARGS(("Input LSN: %8.8lX, entry:%lu\n", lsn, DFSSNIGET( info)));

   if ((dirbuf = TxAlloc( 1, bps)) != NULL)     // allocate one sector
   {
      fatdir = (S_EXFTDIR *) dirbuf;

      rc = dfsRead( lsn, 1, dirbuf);
      if (rc == NO_ERROR)
      {
         *direntry = fatdir[ DFSSNIGET( info)];
      }
      TxFreeMem( dirbuf);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsExftLsnInf2Entry'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Retrieve STREAM directory-entry for specified Dir-sector-LSN + index (LsnInf)
// Input entry (lsn+inf) should point to a FILE, STREAM, BITMAP or UPCASE entry
/*****************************************************************************/
ULONG dfsExftLsnInf2Stream
(
   ULONG               lsn,                     // IN    Directory LSN
   ULONG               info,                    // IN    directory entry
   S_EXFTDIR          *direntry                 // OUT   directory entry (STREAM)
)
{
   ULONG               rc  = NO_ERROR;
   USHORT              bps = dfsGetSectorSize();
   BYTE               *dirbuf;
   S_EXFTDIR          *fatdir;                  // Fat directory sector
   S_EXFTDIR          *stream;                  // pointer to STREAM entry

   ENTER();
   TRARGS(("Input LSN: %8.8lX, entry:%lu\n", lsn, DFSSNIGET( info)));

   if ((dirbuf = TxAlloc( 2, bps)) != NULL)     // allocate two sectors
   {
      fatdir = (S_EXFTDIR *) dirbuf;

      rc = dfsRead( lsn, 2, dirbuf);
      if (rc == NO_ERROR)
      {
         stream = &fatdir[ DFSSNIGET( info)];
         if ((stream->EntryType & ~EXFAT_DIRBIT_IN_USE) == EXFAT_DEC_FILE)
         {
            stream++;                           // advance to stream entry
         }
         *direntry = *stream;                   // copy STREAM entry contents
      }
      TxFreeMem( dirbuf);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsExftLsnInf2Stream'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create SPACE allocation structure from a Directory LSN (root or other)
/*****************************************************************************/
ULONG dfsExftDirLsn2Space
(
   LSN                 lsn,                     // IN    lsn for DIR sector
   ULONG              *nr,                      // OUT   nr of space entries
   S_SPACE           **sp                       // OUT   space allocation
)
{
   ULONG               rc  = NO_ERROR;
   ULONG               chunks;                  // nr of space entries
   S_SPACE            *space = NULL;            // space allocation
   ULONG               ef = 0;                  // Initialize error flag!

   ENTER();
   TRARGS(("Input LSN: %8.8lX\n", lsn));

   //- Note: Function is only reliable for the ROOT directory since that is
   //-       the only one guaranteed to be in a FAT-chain, most other
   //-       directories will be one or more contiguous clusters, no chain
   //-
   //-       Could perhaps use the SLT (when available) to get to the parent
   //-       directory entry, and the invalid-FAT-chain bit, then construct
   //-       a one-chunk space for number of clusters

   if (dfsExftAllocChain( dfsExftLSN2Clust(lsn),
                         0, &ef, &chunks, &space) == 0)
   {
      if (ef != 0)                              // FAT area probably damaged
      {
         if ((ef & EF_ALLOC_CL_FREE) == 0)      // don't report ends-in-free ...
         {
            dfsExftDispError( ef, 0, "\nWarning: ", NULL);
         }

         chunks = 1;                            // just return first cluster
         if ((space = (S_SPACE *) calloc((size_t) chunks, sizeof(S_SPACE))) != NULL)
         {
            space->start = lsn;                 // start of cluster
            space->size  = exft->ClustSize;     // size of chunk
         }
         else
         {
            rc = DFS_ALLOC_ERROR;
         }
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   *sp = space;
   *nr = chunks;
   RETURN (rc);
}                                               // end 'dfsExftDirLsn2Space'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Iterate over directory-structures finding files, and execute callback
/*****************************************************************************/
ULONG dfsExftIterator
(
   ULONG               recurse,                 // IN    recurse subtrees
   ULONG               type,                    // IN    do 'D', 'f' or both
   char               *path,                    // IN    path spec, info only
   void               *p                        // IN    iterate parameters
)
{
   ULONG               rc  = NO_ERROR;
   DFS_PARAMS         *ip  = (DFS_PARAMS *) p;  // Iterator parameters
   S_EXFTDIR           de;                      // directory entry
   ULONG               nr;                      // Alloc chunks
   S_SPACE            *sp  = NULL;              // Alloc chunk array

   ENTER();

   //- Specified path translated to an LSN and optional entry-nr in params
   TRACES(("Lsn: %8.8lx  Entry: %lu\n", ip->Lsn, ip->Number));

   memset( &de, 0, sizeof( S_EXFTDIR));         // default to CL 0 ==> Root

   if (ip->Number & DFSSNINFO)                  // reference to a DIR entry
   {
      dfsExftLsnInf2Entry( ip->Lsn, ip->Number, &de);
   }
   if (dfsExftDirLsn2Space(dfsExftClust2LSN(dfsExftDir2Clust(&de)), &nr, &sp) == NO_ERROR)
   {
      iterationsDone = 0;                      // initialize static progress
      rc = dfsExftIterFiles( recurse, type, nr, sp, ip);
      free( sp);
   }
   RETURN (rc);
}                                               // end 'dfsExftIterator'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create valid FAT(32) boot record based on exft-> info from FatInit
/*****************************************************************************/
ULONG dfsExftMkBootRec
(
   BOOL                verbose,                 // IN    verbose and confirm
   ULONG               serialNr                 // IN    Vol-serialNr or 0
)
{
   ULONG               rc  = DFS_NO_CHANGE;
   #if defined (NEVER)
   ULONG               partSectors;
   S_BOOTR            *brt;
   TXTS                string;
   TXTS                osname;
   ULONG               serial = serialNr;
   #endif

   ENTER();

   TxPrint( "\nFixboot by creating from scratch is NOT IMPLEMENTED for EFAT yet.\n");

   #if defined (NEVER)
   if ((SINF->p != NULL) || ((exft->Fat1 != 0) && (exft->Fat2 > exft->Fat1)))
   {
      if (serial == 0)                          // create default
      {
         serial = dfstLSN2Psn( DFSTORE, 0) + 0xee000000;
      }
      if ((brt = TxAlloc(1, dfsGetSectorSize())) != NULL)
      {
         // dfsMakeFatOsString(   os, osname);
         // dfsInsertFatBootCode( os, brt);

         switch (dfstStoreType( DFSTORE))
         {
            case DFST_PHDISK:                   // possibly partitioned
            case DFST_VIRTUAL:
               if (SINF->p != NULL)             // partition info available
               {
                  partSectors             = SINF->p->sectors;
                  brt->ex.PartitionOffset = SINF->p->basePsn;
               }
               else                             // large floppy
               {
                  partSectors             = exft->Sect;
                  brt->ex.PartitionOffset = 0;
               }
               break;

            default:
               if (verbose)
               {
                  TxPrint( "\n%sNo partition-table info%s is available, this will "
                           "make the 'PartitionOffset'\nfield incorrect when this "
                           "is a primary partition.\n", CBR, CNN);
               }
               partSectors             = exft->Sect;
               brt->ex.PartitionOffset = 0;
               break;
         }
         //- Fill in the specific exfat or generic os structures ...
         //- to be refined

         brt->ex.NrOfFats         = 1;
         brt->ex.VolumeLength    = (LLONG) partSectors;

         brt->ex.RootCluster   = dfsExftLSN2Clust( exft->Root);
         brt->ex.FsRevMinor    = 0x00;
         brt->ex.FsRevMajor    = 0x10;
         brt->ex.BiosDrive     = 0x80;
         brt->ex.SerialNr      = serial;
         memcpy( brt->OpSys,    "EXFAT   ", 8);

         if (verbose)
         {
            TxPrint( "\nThe following new EFAT bootsector has been prepared:\n\n");
            dfsDisplaySector( dfstLSN2Psn( DFSTORE, LSN_BOOTR), ST_BOOTR, (BYTE   *) brt);
         }
         //- to be refined, different help item
         if ((dfsa->batch) || (verbose == FALSE) ||
             (TxConfirm( 5129, "A new EFAT bootsector has been prepared with "
                               "bootcode for OS:\n\n      %s\n\n"
                               "Do you want to replace the current "
                               "filesystem bootsector (sector 0)\n"
                               "by the newly created one ? [Y/N] : ", osname)))
         {
            if (DFSTORE_WRITE_ALLOWED)
            {
               rc = dfsWrite( LSN_BOOTR, 1, (BYTE   *) brt);
               if ((rc == NO_ERROR) && verbose)
               {
                  TxPrint("\nEFAT boot record successfully updated\n");
               }
            }
         }
         TxFreeMem( brt);
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   else
   {
      TxPrint("No partition info and no FAT positions available, "
              "create not possible\n");
   }
   #endif
   RETURN(rc);
}                                               // end 'dfsExftMkBootRec'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get/Set DIRTY bit in EFAT bootsector to specified value
/*****************************************************************************/
BOOL dfsExftDirtyStatusBootrec                  // RET   final dirty status
(                                               //
   BOOL                update,                  // IN    update value
   BOOL                dirty                    // IN    new value
)
{
   BOOL                rc = FALSE;
   S_BOOTR            *brt;
   USHORT             *FsysValue;

   ENTER();

   if ((brt = TxAlloc(1, dfsGetSectorSize())) != NULL)
   {
      if ((rc = dfsRead( LSN_BOOTR, 1, (BYTE   *) brt)) == NO_ERROR)
      {
         FsysValue = &brt->ex.VolumeFlags;
         if (update && (DFSTORE_WRITE_ALLOWED))
         {
            if (dirty)
            {
               *FsysValue |=  EXFAT_VOL_DIRTY;
            }
            else
            {
               *FsysValue &= ~EXFAT_VOL_DIRTY;
            }
            if ((dfsWrite( LSN_BOOTR, 1, (BYTE   *) brt)) == NO_ERROR)
            {
               TxPrint("\nEFAT boot record successfully updated with new DIRTY status\n");

               dfsExftCrcSyncBootAreas( LSN_BOOTR, 0); // sync to BOTH
            }
            else
            {
               TxPrint("\nEFAT boot record update for dirty-flag has failed!\n");
            }
         }
         rc = (BOOL) ((*FsysValue & EXFAT_VOL_DIRTY) ? TRUE : FALSE);
      }
      else
      {
         TxPrint("\nEFAT bootrecord read error!\n");
      }
      TxFreeMem( brt);
   }
   BRETURN(rc);
}                                               // end 'dfsExftDirtyStatusBootrec'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Recursively iterate over FAT directory and files, execute callback
/*****************************************************************************/
static ULONG dfsExftIterFiles
(
   ULONG               rs,                      // IN    recurse subtrees
   ULONG               type,                    // IN    do 'D', 'f' or both
   ULONG               chunks,                  // IN    nr of S_SPACE chunks
   S_SPACE            *space,                   // IN    space structure array
   DFS_PARAMS         *cp                       // IN    callback params
)
{
   ULONG               rc = NO_ERROR;
   ULONG               chunk;                   // index in space-area
   ULONG               sect;                    // sectors to handle
   USHORT              entry;                   // index in fat directory
   LSN                 dlsn;                    // Directory sector LSN
   ULONG               ef = 0;                  // Initialize error flag!
   ULONG               nr;                      // Alloc chunks
   S_SPACE            *sp = NULL;               // Alloc chunk array
   BYTE                id  = ST_UDATA;          // type of sector
   USHORT              bps = dfsGetSectorSize();
   USHORT              entries = (bps / sizeof(S_EXFTDIR));
   BYTE               *dirbuf;
   S_EXFTDIR          *efile;                   // DIR entry, FILE
   S_EXFTDIR          *stream;                  // DIR entry, STREAM
   BOOL                isDir;

   ENTER();
   TRARGS(("Recursion: %ld, chunks: %lu\n", rs, chunks));

   dirbuf = TxAlloc( 3, bps);                   // allocate three sectors
   if ((dirbuf != NULL) && (space != NULL))
   {
      for ( chunk = 0;
           (chunk < chunks) && (rc == NO_ERROR) && !TxAbort();
            chunk++)                            // walk all alloc chunks
      {
         TRACES(( "chunk size: %lu  start lsn: %8.8lx\n",
                   space[chunk].size, space[chunk].start));
         for ( sect = 0;
              (sect < space[chunk].size) && (rc == NO_ERROR) && !TxAbort();
               sect++)                          // each sector in chunk
         {
            if (cp->Flag == TRUE)               // progress in sectors
            {
               iterationsDone++;
            }
            dlsn = space[chunk].start + sect;
            TRACES(( "Read directory sector at: %8.8lx\n", dlsn));

            rc = dfsRead(dlsn, 3, dirbuf);

            switch (dfsIdentifySector( dlsn, 0, dirbuf))
            {
               case ST_ROOTD:
               case ST_DIREC:
                  for ( entry = 0;
                       (entry < entries) && (rc == NO_ERROR) && !TxAbort();
                        entry++)                // each dir-entry in sector
                  {
                     efile = &(((S_EXFTDIR *) dirbuf)[entry]);
                     isDir = ((efile->fl.Attrib & FATTR_DIRECTORY) != 0);

                     switch (efile->EntryType & ~EXFAT_DIRBIT_IN_USE) // normal + deleted
                     {
                        case EXFAT_DEC_BITMAP:
                        case EXFAT_DEC_UPCASE:  // length & cluster fields identical for these
                           if (efile->bm.Length != 0)
                           {
                              if (type != 'D')
                              {
                                 rc = (cp->Func)( dlsn, entry | DFSSNINFO, (char *) &id, cp); // execute callback
                              }
                           }
                           break;

                        case EXFAT_DEC_FILE:
                           stream = efile +1;   // advance to actual stream entry

                           if ((stream->st.Length != 0) || isDir)
                           {
                              if (((isDir == FALSE) && (type != 'D')) || //- type for this entry
                                  ((isDir == TRUE ) && (type != 'F'))  ) //- not excluded ?
                              {
                                 if (cp->Flag == FALSE) // progress in entries
                                 {
                                    iterationsDone++;
                                 }
                                 #if defined (USEWINDOWING)
                                    if (txwIsWindow( TXHWND_DESKTOP))
                                    {
                                       if (clock() > dfsa->nextStatus)
                                       {
                                          TXTM    status;

                                          if (cp->Flag == FALSE) // progress in entries
                                          {
                                             sprintf( status, "DIRsec: 0x%8.8lx, done:%6lu DIR entries",
                                                               dlsn, iterationsDone);
                                          }
                                          else
                                          {
                                             sprintf( status, "DIRsec: 0x%8.8lx, done:%6lu sectors",
                                                               dlsn, iterationsDone);
                                          }
                                          txwSetSbviewStatus( status, cSchemeColor);
                                          dfsa->nextStatus = clock() + DFSP_PRC_INTERVAL;
                                       }
                                    }
                                 #endif
                                 rc = (cp->Func)( dlsn, entry | DFSSNINFO, (char *) &id, cp); // execute callback
                              }
                              else
                              {
                                 TRACES(("Skipping entry: %u\n", entry));
                              }
                              if (isDir)
                              {
                                 if (rs != 1)   // recursion ?
                                 {
                                    ef = 0;
                                    dfsExftAllocChain( dfsExftDir2Clust( stream), 0,
                                                       &ef, &nr, &sp);
                                    if (sp != NULL)
                                    {
                                       rc = dfsExftIterFiles( rs -1, type, nr, sp, cp);
                                       free( sp); // re-alloc'ed no TxFreeMem !
                                    }
                                 }
                              }
                           }
                           break;

                        default:                // ignore others
                           break;
                     }
                  }
                  break;

               default:
                  break;
            }
         }
      }
      TxFreeMem( dirbuf);
   }
   RETURN( rc);
}                                               // end 'dfsExftIterFiles'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read allocation for ALL directory sectors into a single SPACE structure
/*****************************************************************************/
ULONG dfsExftAllDirs2Space
(
   BOOL                progressSectors,         // IN    progress in sectors, not DIRs
   ULONG              *nr,                      // OUT   nr of space entries
   S_SPACE           **sp                       // OUT   space allocation
)
{
   ULONG               rc = NO_ERROR;
   DFS_PARAMS          params;                  // generic function params
   ULONG               chunks = 0;              // chunks in space structure
   S_SPACE            *space  = NULL;           // space structure

   ENTER();

   //- Start with ROOT allocation itself (fixed chunk FAT16 or clusters on FAT32)
   if (dfsExftDirLsn2Space(exft->Root, &chunks, &space) == NO_ERROR)
   {
      params.Lsn    = exft->Root;               // start sector for iterator
      params.Number = 0;                        // not a specific entry
      params.Func   = dfsExftAddDir2Space;      // Iterator callback function
      params.Misc   = space;
      params.Count  = chunks;                   // Callback IN/OUT structure
      params.Flag   = progressSectors;

      //- Iterate over all directories recursively and add them to the SPACE
      rc = dfsExftIterator( -1, 'D', NULL, &params);
      if (rc == NO_ERROR)
      {
         chunks =             params.Count;
         space  = (S_SPACE *) params.Misc;
      }
   }
   *sp = space;
   *nr = chunks;
   RETURN (rc);
}                                               // end 'dfsExftAllDirs2Space'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add SPACE for specified directory to the accumulating total directory SPACE
/*****************************************************************************/
static ULONG dfsExftAddDir2Space
(
   ULONG               sn,                      // IN    sector-nr DIR sector
   ULONG               info,                    // IN    coded directory entry#
   char               *df,                      // IN    dummy, optional info
   void               *data                     // INOUT callback data, SPACE
)
{
   ULONG               rc      = NO_ERROR;
   DFS_PARAMS         *ip      = (DFS_PARAMS *) data;
   ULONG               extents =                ip->Count;
   S_SPACE            *sp      = (S_SPACE *)    ip->Misc;
   ULONG               chunks;
   S_SPACE            *space   = NULL;          // SPACE for this DIR
   S_EXFTDIR           stream;                  // DIR entry, STREAM

   ENTER();

   if ((rc = dfsExftLsnInf2Stream( sn, info, &stream)) == NO_ERROR)
   {
      if (dfsExftAllocStream( &stream, NULL, &chunks, &space) != 0)
      {
         if (space != NULL)
         {
            //- join with accumulator space, and free the old ones
            rc = dfsSspaceJoin( extents, sp, chunks, space, &extents, &sp);
            if (rc == NO_ERROR)
            {
               ip->Count = extents;
               ip->Misc  = sp;                  // pass back joined SPACE
            }
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsExftAddDir2Space'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Determine allocation-bit for specified LSN, ALLOCATED beyond last cluster!
/*****************************************************************************/
ULONG dfsExftAllocated                          // RET   LSN is allocated
(
   ULONG               lsn,                     // IN    LSN
   ULONG               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               asn = dfstLSN2AreaSN( DFSTORE, lsn); // Area aware sn
   ULONG               rc  = DFS_PSN_LIMIT;

   if ((exft->Bm.Size != 0) && (asn < exft->Bm.LimitLsn)) // inside the bitmap
   {
      if (asn < (exft->ClHeap + (exft->RawClust * exft->ClustSize))) // within valid cluster area
      {
         rc = (ULONG) dfsExftBitmapCache(dfsExftLSN2Clust( lsn), NULL);
      }
      else                                      // return ALLOCATED for any
      {                                         // lsn beyond last cluster
         rc = 1L;                               // to avoid CHECK errors
      }                                         // on non-standard bitmaps
   }
   //- TRACES(("ExftAllocated: lsn:0x%8.8lx, Bm.LimitLsn=0x%8.8lx, allocated:%lx\n", lsn, exft->Bm.LimitLsn, rc));
   return (rc);
}                                               // end 'dfsExftAllocated'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Align R/W cache to position for cluster, and get current allocation-bit
/*****************************************************************************/
BOOL dfsExftBitmapCache                         // RET   cluster is allocated
(
   ULONG               cluster,                 // IN    LCN
   ULONG              *bp                       // OUT   byte position in cache
)                                               //       or LSN_NULL if invalid
{
   ULONG               rc = NO_ERROR;
   ULONG               cl;
   BOOL                al = TRUE;               // default allocated
   ULONG               cs = exft->ClustSize;    // cluster size in sectors
   ULONG               byte;                    // byte pos in sector
   ULONG               rsn;                     // relative sector in Bitmap
   ULONG               bytepos = LSN_NULL;      // byte position in cluster
   BYTE                alloc_byte;

   if ((cluster >= 2) && (cs != 0))
   {
      cl  = cluster - 2;                        // first bit in bitmap is cluster 2
      rsn = cl / (8 * dfsGetSectorSize());      // find relative BM sector for cl

      if (((rsn <  exft->Bm.First) ||           // if rsn is not in current
           (rsn >= exft->Bm.First +             // cached range of rsn's
                   exft->Bm.Size)))             // read new range ...
      {
         if (exft->Bm.Dirty)                    // need to flush changes ?
         {
            rc = dfsExftBitmapFlush( FALSE);    // flush, but keep cache
         }
         if (rc == NO_ERROR)
         {
            exft->Bm.First = (rsn / cs) * cs;
            TRACES(("Read  new cache Bm.First: %8.8lx\n", exft->Bm.First));
            dfsSspaceReadFilePart( exft->Bm.Extents,
                                   exft->Bm.Space,
                                   exft->Bm.First,
                                   exft->Bm.Size,
                          (BYTE *) exft->Bm.Buffer);
         }
      }
      if (rc == NO_ERROR)
      {
         byte       = (cl % (8 * dfsGetSectorSize())) / 8; // byte index in sector
         bytepos    = ((rsn - exft->Bm.First) * dfsGetSectorSize()) + byte;

         if ((bytepos < (exft->Bm.Size * dfsGetSectorSize())) &&
                         exft->Bm.Buffer != NULL)
         {
            alloc_byte = exft->Bm.Buffer[ bytepos];
            if (((alloc_byte >> (cl & 0x07)) & 0x01) == 0) // LSB is lowest!
            {
               al = FALSE;
            }
         }
         else                                   // signal error condition
         {
            bytepos = LSN_NULL;
         }
      }
      if (bp != NULL)                           // return byte position too ?
      {
         *bp = bytepos;
      }
   }
   //- TRACES(("BitMapCache, cluster: 0x%8.8lx bytepos:0x%8.8lx al:%lu\n", cluster, bytepos, al));
   return (al);
}                                               // end 'dfsExftBitmapCache'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Flush BitMap cache when Dirty, and optional free its resources
/*****************************************************************************/
ULONG dfsExftBitmapFlush                        // RET   rc
(
   BOOL                terminate                // IN    terminate cache too
)
{
   ULONG               rc = NO_ERROR;           // rc

   ENTER();

   TRACES(("BitMap cache Bm.First: %8.8lx, ", exft->Bm.First));
   if (exft->Bm.Dirty)                          // need to flush changes ?
   {
      TRACES(( "dirty, flush it ...\n"));
      rc = dfsSspaceWriteFilePart( exft->Bm.Extents,
                                   exft->Bm.Space,
                                   exft->Bm.First,
                                   exft->Bm.Size,
                          (BYTE *) exft->Bm.Buffer);

      exft->Bm.Dirty = FALSE;                   // mark it clean again
   }
   else
   {
      TRACES(( "not dirty ...\n"));
   }
   if (terminate)
   {
      TxFreeMem( exft->Bm.Space);
      TxFreeMem( exft->Bm.Buffer);

      exft->Bm.Size     = 0;
      exft->Bm.LimitLsn = 0;
   }
   RETURN (rc);
}                                               // end 'dfsExftBitmapFlush'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set allocate status for LSN to specified value (beyond FS, to end of bitmap)
/*****************************************************************************/
ULONG dfsExftSetAlloc                           // RET   LSN allocation set
(
   ULONG               lsn,                     // IN    LSN
   ULONG               d2,                      // IN    dummy
   char               *value,                   // IN    NULL = not allocated
   void               *ref                      // IN    dummy (for EXFT)
)
{
   if ((exft->Bm.Size != 0) && (lsn < exft->Bm.LimitLsn)) // inside the bitmap
   {
      return( dfsExftSetBmCluster(dfsExftLSN2Clust( lsn), (value != NULL)));
   }
   else
   {
      return( DFS_PSN_LIMIT);
   }
}                                               // end 'dfsExftSetAlloc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set allocate status for CLUSTER to specified value
/*****************************************************************************/
ULONG dfsExftSetBmCluster                         // RET   CL  allocation set
(
   ULONG               cl,                      // IN    Cluster
   BOOL                value                    // IN    SET allocation bit
)
{
   ULONG               rc = NO_ERROR;
   ULONG               bytepos;
   BYTE                alloc_byte;
   BYTE                bit_mask = (1 << (cl & 0x07));

   dfsExftBitmapCache( cl, &bytepos);
   if (bytepos != LSN_NULL)                     // cluster position valid ?
   {
      alloc_byte = exft->Bm.Buffer[ bytepos];

      if (value)                                // set the allocation bit
      {
         exft->Bm.Buffer[ bytepos] = alloc_byte |  bit_mask;
      }
      else                                      // reset allocation bit
      {
         exft->Bm.Buffer[ bytepos] = alloc_byte & ~bit_mask;
      }
      TRACES(( "%s cache-byte at 0x%4.4lx = -%6.6lx- for cl:%8.8lx "
                  "from 0x%2.2hx to 0x%2.2hx\n",
                (value) ? "Set" : "Clr", bytepos, cl / 8, cl,
                alloc_byte, exft->Bm.Buffer[ bytepos]));

      exft->Bm.Dirty = TRUE;                    // mark cache dirty
   }
   else
   {
      rc = DFS_PSN_LIMIT;
   }
   return (rc);
}                                               // end 'dfsExftSetBmCluster'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read, calculate checksums and write primary and secondary boot area structures
/*****************************************************************************/
ULONG dfsExftCrcSyncBootAreas
(
   LSN                 source,                  // IN    LSN of source area
   int                 area                     // IN    destination area 0=both/1/2
)
{
   ULONG               rc = NO_ERROR;

   ENTER();
   TRACES(("source: 0x%8.8lx  area: %d\n", source, area));

   if (DFSTORE_WRITE_ALLOWED)
   {
      if ((rc = dfsRead( source, EXFAT_BOOTSIZE, brec)) == NO_ERROR)
      {
         S_BOOTR      *sd = (S_BOOTR *) brec;   // view as boot record

         if ((sd->ex.BpsShift >= 9) && (sd->ex.BpsShift <= 12))    //- 512..4096 bytes
         {
            ULONG      cs = 0;                  // checksum value
            ULONG      size;                    // size boot area in bytes
            ULONG      pos;                     // calculated position 1st CS
            ULONG      count;                   // calculated nr of Checksums
            ULONG     *refCs;                   // start of checksum sector
            ULONG      i;                       // byte position

            size = ((1 << sd->ex.BpsShift) * EXFAT_BOOTSIZE);
            cs   = dfsExftBootCheckSum( brec, size);               //- new checksum value

            pos    = (1 << sd->ex.BpsShift) * (EXFAT_BOOTSIZE -1);
            count  = (1 << sd->ex.BpsShift) / sizeof( ULONG);
            refCs  = (ULONG *) (&brec[ pos]);
            for (i = 0; i < count; i++)
            {
               refCs[i] = cs;                   // and update in last sector
            }

            if ((area == 0) || (area == 1))     // write primary
            {
               rc = dfsWrite( LSN_BOOTR, EXFAT_BOOTSIZE, brec);
            }
            if ((area == 0) || (area == 2))     // write secondary
            {
               rc = dfsWrite( EXFAT_BACKUPBOOT, EXFAT_BOOTSIZE, brec);
            }
         }
         else
         {
            rc = DFS_BAD_STRUCTURE;
         }
      }
   }
   RETURN(rc);
}                                               // end 'dfsExftCrcSyncBootAreas'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Calculate EFAT boot-area specific checksum, using published algorithm
/*****************************************************************************/
ULONG dfsExftBootCheckSum                       // RET   EFAT boot style crc
(
   BYTE               *data,                    // IN    Boot area (>= 11 sect)
   ULONG               size                     // IN    Size of data area
)
{
   ULONG               cs = 0;                  // checksum value
   ULONG               i;                       // checksum value
   S_BOOTR            *sd = (S_BOOTR *) data;   // view as boot record
   ULONG               limit = 0x10000;         // calculated size, upto 64 KiB

   ENTER();
   TRACES(("Data size: 0x%8.8lx\n", size));

   if ((sd->ex.BpsShift >= 9) && (sd->ex.BpsShift <= 12)) // 512 .. 4096 bytes
   {
      limit = (1 << sd->ex.BpsShift) * (EXFAT_BOOTSIZE -1);
   }
   if (limit > size)
   {
      limit = size;                             // but limit to specified data size
   }
   for (i = 0; i < limit; i++)
   {
      switch (i)
      {
         case 0x6A:                             // Volume flag bytes
         case 0x6B:                             // Incl. 'dirty' flag
         case 0x70:                             // Percent in-use byte
            continue;                           // skip
            break;

         default:
            cs  = (cs << 31) | (cs >> 1) + data[i];
            break;
      }
   }
   RETURN( cs);
}                                               // end 'dfsExftBootCheckSum'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read EFAT boot-area checksum reference from sector and check consistency
/*****************************************************************************/
ULONG dfsExftStoredChecksum                     // RET   EFAT boot style crc
(
   BYTE               *data,                    // IN    Boot area (>= 12 sect)
   ULONG               size                     // IN    Size of data area
)
{
   ULONG               cs = 0;                  // checksum value
   ULONG               i;                       // byte position
   S_BOOTR            *sd = (S_BOOTR *) data;   // view as boot record
   ULONG               pos;                     // calculated position 1st CS
   ULONG               count;                   // calculated nr of Checksums
   ULONG              *refCs;

   ENTER();

   if ((sd->ex.BpsShift >= 9) && (sd->ex.BpsShift <= 12))    //- 512..4096 bytes
   {
      if (size >= ((1 << sd->ex.BpsShift) * EXFAT_BOOTSIZE)) //- all within data
      {
         pos    = (1 << sd->ex.BpsShift) * (EXFAT_BOOTSIZE -1);
         count  = (1 << sd->ex.BpsShift) / sizeof( ULONG);
         refCs  = (ULONG *) (&data[ pos]);

         for (i = 1, cs = refCs[0]; i < count; i++)
         {
            if (refCs[i] != cs)                 // inconsistent ?
            {
               cs = 0;
               break;
            }
         }
      }
   }
   RETURN( cs);
}                                               // end 'dfsExftStoredChecksum'
/*---------------------------------------------------------------------------*/
