//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// EFAT utility functions
//
// Author: J. van Wijk
//
// JvW  19-06-2014   Initial version, derived from DFSUFAT
//
#ifndef    DFSUEFAT
   #define DFSUEFAT

#include <dfsefat.h>                            // EFAT structure definitions
#include <dfsaefat.h>                           // EFAT cache definitions

#define DFS_EFAT_INVALIDS ":;\"|\\=?*<>,+[]/"

// Display file allocation and path info for LSN
ULONG dfsEfatFileInfo                           // RET   result
(
   ULN64               lsn,                     // IN    Directory LSN
   ULN64               entry,                   // IN    directory entry
   char               *select,                  // IN    File select wildcard
   void               *lead                     // IN    leading text
);

// Replace sn-list by contents of a single Directory (DirLsn, entry pairs)
ULONG dfsEfatMakeBrowseList                     // RET   result
(
   ULN64               lsn,                     // IN    Directory LSN
   ULN64               info,                    // IN    dir entry, 0 for ROOT
   char               *str,                     // IN    unused
   void               *param                    // INOUT unused
);

// Get allocation information for file-DATA into integrated-SPACE structure
ULONG dfsEfatGetAllocSpace
(
   ULN64               lsn,                     // IN    Directory LSN
   ULN64               info,                    // IN    directory entry
   char               *str,                     // IN    unused
   void               *param                    // INOUT Integrated SPACE
);

// Initialize the Directory-Parent cache structure by walking the tree
ULONG dfsEfatInitDpCache
(
   ULN64               dummy1,                  // IN    not used
   ULN64               d2,                      // IN    dummy
   char               *dummy2,                  // IN    not used
   void               *param                    // IN    not used
);

// Search DP-cache for parent Lsn+index for specified directory sector number
ULONG dfsEfatSearchParent                  // RET   NOT_FOUND when fail
(
   ULN64               dirLsn,                  // IN    sector number
   ULN64              *parentLsn,               // OUT   parent dirsector LSN
   BYTE               *parentIdx                // OUT   parent index in sector
);

// Get long or short (8.3) name for given LSN and entry number for DIR sector
ULONG dfsEfatLsnInfo2Name                       // RET   result
(
   ULN64               lsn,                     // IN    Directory LSN
   ULONG               info,                    // IN    directory entry
   char               *name                     // OUT   directory entry name
);


// Find full PATH to Root-directory, starting at some dir LSN and entry info
ULONG dfsEfatLsnInfo2Path                       // RET   result und
(
   ULN64               start,                   // IN    starting lsn
   ULONG               entry,                   // IN    directory entry info
   char               *path                     // OUT   combined path string
);

// Determine type of whole directory cluster and cache it for later reuse
BYTE dfsEfatDirClusterType
(
   ULN64               lsn                      // IN    LSN of a (DIR) sector
);

#define dfsEfatClust2LSN(cl) (ULN64) dfsEfatCl2Lsn((cl),0,NULL,NULL)
// Translate Cluster-nr to LSN
ULN64 dfsEfatCl2Lsn                             // RET   LSN for this cluster
(
   ULN64               cluster,                 // IN    cluster number
   ULN64               d2,                      // IN    dummy
   char               *p1,                      // IN    dummy
   void               *p2                       // IN    dummy
);

#define dfsEfatLSN2Clust(sn) dfsEfatLsn2Cl((ULN64)(sn),0,NULL, NULL)
// Translate LSN to Cluster-nr
ULN64 dfsEfatLsn2Cl                             // RET   cluster for this LSN
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *p1,                      // IN    dummy
   void               *p2                       // IN    dummy
);

// Get cluster-value from a Directory entry structure
ULONG dfsEfatDir2Clust                          // RET   cluster value
(
   S_EFATDIR           *entry                    // IN    EFAT directory entry
);

// Set cluster-value in a Directory entry structure
void dfsEfatClust2Dir
(
   ULONG               cluster,                 // IN    cluster value to set
   S_EFATDIR          *entry                    // INOUT EFAT directory entry
);

// Get cluster-value from the Fat Cache structure (32 bit entries)
ULONG dfsEfatValue                              // RET   cluster value
(
   ULONG               clust                    // IN    cluster to get value
);

// Set cluster-value in the Fat Cache structure (32 bit entries)
ULONG dfsEfatSetCluster
(
   ULONG               clust,                   // IN    cluster to get value
   ULONG               value                    // IN    FAT-value to be set
);

// Reset bad-sector admin, change each BAD cluster in the FAT to FREE
ULONG dfsEfatResetBadClus                       // RET   ref cluster or 0
(
   void
);

// Search reference to specified cluster in FAT (cluster pointing to this one)
// Note: Dont use regular cache to avoid trashing that just by searching here
ULONG dfsEfatFindReference                      // RET   ref cluster or 0
(
   ULONG               cluster                  // IN    target cluster
);

// Check if cluster could be the 1st root-DIR sector (DIR but no '..' entry)
BOOL dfsEfatRootCandidate                       // RET   Could be a root DIR
(
   ULONG               cluster                  // IN    target cluster
);

// Calulate 32-bit CRC over FAT-contents (to compare FAT1 and FAT2 perhaps :-)
ULONG dfsEfatCalculateFatCRC                    // RET   32-bit CRC
(
   ULN64               fatlsn                   // IN    first sector of FAT
);

// Set the active FAT LSN to be used for fat references using the cache
void dfsEfatSetFatInUse
(
   ULN64               fat2Use                  // IN    first sector of FAT
);

// Copy FAT-part from disk to the DFS FAT cache structure (32 bit entries)
ULONG dfsEfatGetFAT                             // RET
(
   ULONG               clust,                   // IN    cluster value to get
   DFSEFATCACHE        *cache                   // INOUT cache structure used
);

// Flush specified FAT cache (32 bit entries) back to disk FatInUse (+ other)
ULONG dfsEfatFlushFAT                           // RET   result
(
   DFSEFATCACHE        *cache                   // INOUT cache structure to write
);

// Show logical FAT
void dfsEfatShowFAT
(
   ULONG               size,                    // IN    nr of entries
   ULONG               start,                   // IN    first cl# to show
   char               *options                  // IN    display options
);

// Count number of used Dir-entries for S_SPACE structure
ULONG dfsEfatDirUsed                            // RET   matching entries
(
   ULONG               chunks,                  // IN    nr of space entries
   S_SPACE            *space                    // IN    space allocation
);

// Get label-string from the EFAT Root-directory
BOOL dfsEfatRootLabel                           // RET   Label found
(
   S_BOOTR            *boot,                    // IN    boot record for vol
   ULN64               offset,                  // IN    Offset for boot-rec
   char               *label                    // OUT   label string
);

// Set label-string in the EFAT Root-directory
ULONG dfsEfatSetRootLabel
(
   char               *str                      // IN    label string
);

// Get starting cluster for the specified Bitmap file from root directory
BOOL dfsEfatRootGetBmCluster                    // RET   Bitmap cluster or 0
(
   S_BOOTR            *boot,                    // IN    boot record for vol
   ULN64               offset,                  // IN    Offset for boot-rec
   BYTE                mapnr,                   // IN    Bitmap 1st is 0
   ULONG              *cluster,                 // OUT   cluster number
   ULN64              *byteSize                 // OUT   size in bytes
);

// Create S_SPACE structure for an EFAT stream directory entry
ULONG dfsEfatAllocStream                        // RET   nr of clusters
(
   S_EFATDIR          *stream,                  // Fat dir-entry ptr, STREAM
   ULONG              *errors,                  // INOUT error flags
   ULONG              *chunks,                  // OUT   nr of space entries
   S_SPACE           **space                    // OUT   space allocation
);

// Create S_SPACE structure for a FAT allocation chain
ULONG dfsEfatAllocChain                         // RET   nr of clusters
(
   ULONG               cluster,                 // IN    starting cluster nr
   ULONG               maxsize,                 // IN    max file-size, or 0
   ULONG              *errors,                  // INOUT error flags (or NULL)
   ULONG              *chunks,                  // OUT   nr of space entries
   S_SPACE           **space                    // OUT   space allocation
);

// Check if name is a valid EFAT filename
BOOL dfsValidEfatName
(
   const void         *fn,                      // IN    ptr to EFAT name
   USHORT              len                      // IN    length to check
);

// Display all available info about a directory entry, including C0/C1 entries
ULONG dfsEfatShowDirectoryEntry
(
   ULN64               dirsect,                 // IN    lsn of FAT dir sector
   int                 entry                    // IN    File, Stream + Name entries
);

// FAT filesystem, display DIR header/footer
void dfsEfatDirHeader
(
   char               *lead,                    // IN    lead text-string
   ULONG               items                    // IN    DIR-lines shown (0 = TOP)
);

// Display EFAT directory entry on two lines, all except filename
void dfsEfatShowDirEntry
(
   S_EFATDIR          *entry,                   // IN    EFAT FILE entry
   S_EFATDIR          *stream,                  // IN    EFAT STREAM entry
   int                 ac                       // IN    active color fg+bg
);

// Display FAT allocation chain from STREAM directory entry
void dfsEfatShowAlloc
(
   S_EFATDIR          *entry                    // IN    STREAM dir entry
);

// Determine ExFAT alloc from entry/contiguous-clusters or chain, when valid
ULONG dfsEfatDirEntry2Space                     // RET   result
(
   S_EFATDIR          *entry,                   // IN    STREAM dir entry
   DFSISPACE          *isp,                     // OUT   SPACE alloc info
   ULONG              *eFlags                   // OUT   error flags, or NULL
);

// Convert null-terminated name to a valid dir-entry name (multiple C1 entries)
void dfsEfatName2Entry
(
   char               *fn,                      // IN    ptr to FILE name
   S_EFATDIR          *de                       // OUT   File, Stream + Name entries
);

// Convert a valid dir-entry name (multiple C1) to valid null-terminated name
// Asumes there are 2 more sectors of DIR entries BEFORE and AFTER specified one
void dfsEfatEntry2Name
(
   S_EFATDIR          *de,                      // IN    File, Stream + Name entries
   char               *fn                       // OUT   ptr to FILE name
);

// Retrieve first-cluster number for specified Dir-sector-LSN + index (LsnInf)
ULONG dfsEfatLsnInf2Cluster                         // RET   First cluster or 0
(
   ULN64               lsn,                     // IN    Directory LSN
   ULONG               info                     // IN    directory entry
);

// retrieve directory-entry for specified Dir-sector-LSN + index (LsnInf)
ULONG dfsEfatLsnInf2Entry
(
   ULN64               lsn,                     // IN    Directory LSN
   ULONG               info,                    // IN    directory entry
   S_EFATDIR          *direntry                 // OUT   directory entry
);

// Retrieve STREAM directory-entry for specified Dir-sector-LSN + index (LsnInf)
// Input entry (lsn+inf) should point to a FILE, STREAM, BITMAP or UPCASE entry
ULONG dfsEfatLsnInf2Stream
(
   ULN64               lsn,                     // IN    Directory LSN
   ULONG               info,                    // IN    directory entry index
   S_EFATDIR          *direntry                 // OUT   directory entry (STREAM)
);

// Create SPACE allocation structure from a Directory LSN (root or other)
ULONG dfsEfatDirLsn2Space
(
   ULN64               lsn,                     // IN    lsn for DIR sector
   ULONG              *nr,                      // OUT   nr of space entries
   S_SPACE           **sp                       // OUT   space allocation
);

// Iterate over directory-structures finding files, and execute callback
ULONG dfsEfatIterator
(
   ULN64               recurse,                 // IN    recurse subtrees
   ULN64               type,                    // IN    do 'D', 'f' or both
   char               *path,                    // IN    path specification
   void               *p                        // IN    iterate parameters
);

// Create valid FAT(32) bootrecord based on exft-> info from EfatInit
ULONG dfsEfatMkBootRec
(
   BOOL                verbose,                 // IN    verbose and confirm
   ULONG               serialNr                 // IN    Vol-serialNr or 0
);

// Get/Set DIRTY bit in EFAT bootsector to specified value
BOOL dfsEfatDirtyStatusBootrec                  // RET   final dirty status
(                                               //
   BOOL                update,                  // IN    update value
   BOOL                dirty                    // IN    new value
);

// Read allocation for ALL directory sectors into a single SPACE structure
ULONG dfsEfatAllDirs2Space
(
   BOOL                progressSectors,         // IN    progress in sectors, not DIRs
   ULONG              *nr,                      // OUT   nr of space entries
   S_SPACE           **sp                       // OUT   space allocation
);

// Determine allocation-bit for specified LSN, ALLOCATED beyond last cluster!
ULONG dfsEfatAllocated                          // RET   LSN is allocated
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);

// Align R/W cache to position for cluster, and get current allocation-bit
BOOL dfsEfatBitmapCache                         // RET   cluster is allocated
(
   ULONG               cluster,                 // IN    LCN
   ULONG              *bp                       // OUT   byte position in cache
);                                              //       or L64_NULL if invalid

// Flush BitMap cache when Dirty, and optional free its resources
ULONG dfsEfatBitmapFlush                        // RET   rc
(
   BOOL                terminate                // IN    terminate cache too
);

// Set allocate status for LSN to specified value (beyond FS, to end of bitmap)
ULONG dfsEfatSetAlloc                           // RET   LSN allocation set
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *value,                   // IN    NULL = not allocated
   void               *ref                      // IN    dummy (for EFAT)
);

// Set allocate status for CLUSTER to specified value
ULONG dfsEfatSetBmCluster                       // RET   CL  allocation set
(
   ULONG               cl,                      // IN    Cluster
   BOOL                value                    // IN    SET allocation bit
);

// Read, calculate checksums and write primary and secondary boot area structures
ULONG dfsEfatCrcSyncBootAreas
(
   ULN64               source,                  // IN    LSN of source area
   int                 area                     // IN    destination area 0=both/1/2
);

// Calculate EFAT boot-area specific checksum, using published algorithm
ULONG dfsEfatBootCheckSum                       // RET   EFAT boot style crc
(
   BYTE               *data,                    // IN    Boot area (>= 11 sect)
   ULONG               size                     // IN    Size of data area
);

// Read EFAT boot-area checksum reference from sector and check consistency
ULONG dfsEfatStoredChecksum                     // RET   EFAT boot style crc
(
   BYTE               *data,                    // IN    Boot area (>= 12 sect)
   ULONG               size                     // IN    Size of data area
);


#endif
