//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// EFAT utility functions
//
// Author: J. van Wijk
//
// JvW  19-06-2014   Initial version, derived from DFSUFAT
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsver.h>                             // DFS version info
#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfstable.h>                           // SLT utility functions

#include <dfsupart.h>                           // FDISK partition functions
#include <dfsafdsk.h>                           // FDISK display & analysis
#include <dfscfdsk.h>                           // FDISK util, FAT sectors

#include <dfsaefat.h>                           // EFAT display & analysis
#include <dfsuefat.h>                           // EFAT utility functions
#include <dfslefat.h>                           // EFAT SLT functions


// Put contents of a Directory SSPACE (DirLsn, entry pairs) into sn-list
static ULONG dfsEfatDirSpace2List
(
   ULONG               chunks,                  // IN    nr of space entries
   S_SPACE            *space                    // IN    space allocation
);

// Compare Dir-Parent cache elements for qsort, sort on cluster number
static int dfsEfatCompareDpItem
(
   const void         *one,
   const void         *two
);

// Add each clusternumber for an EFAT directory to the Directory-Parent cache
static ULONG dfsEfatAdd2DpCache
(
   ULONG               chunks,                  // IN    nr of S_SPACE chunks
   S_SPACE            *space,                   // IN    space structure array
   ULN64               parentLsn,               // IN    parent dirsector LSN
   BYTE                parentIdx                // IN    parent index in sector
);

// Recursively iterate over FAT directory and files, execute callback
static ULONG dfsEfatIterFiles
(
   ULONG               rs,                      // IN    recurse subtrees
   ULONG               type,                    // IN    do 'D', 'f' or both
   ULONG               chunks,                  // IN    nr of S_SPACE chunks
   S_SPACE            *space,                   // IN    space structure array
   DFS_PARAMS         *cp                       // IN    callback params
);

// Add SPACE for specified directory to the accumulating total directory SPACE
static ULONG dfsEfatAddDir2Space
(
   ULN64               sn,                      // IN    sector-nr DIR sector
   ULN64               info,                    // IN    coded directory entry#
   char               *df,                      // IN    dummy, optional info
   void               *data                     // INOUT callback data, SPACE
);


//- relative age of last read/write action to a cache structure
static ULONG        lastCacheStamp = 0;      // last age value issued

//- progress counter for FatIterator (is not always the sector-list!)
static ULONG        iterationsDone = 0;

/*****************************************************************************/
// Display file allocation and path info for LSN (list -f etc)
/*****************************************************************************/
ULONG dfsEfatFileInfo                           // RET   result
(
   ULN64               lsn,                     // IN    Directory LSN
   ULN64               info,                    // IN    directory entry
   char               *select,                  // IN    File select wildcard
   void               *param                    // INOUT leading text/shortpath
)
{
   ULONG               rc = NO_ERROR;
   BYTE               *sb = NULL;               // sector buffer
   S_EFATDIR          *efile;                   // Fat directory entry, FILE
   S_EFATDIR          *stream;                  // Fat directory entry, STREAM
   BYTE                st = 0;                  // sector type, default none
   ULN64               dirsect = lsn;
   USHORT              entry   = info & 0x0f;
   TXLN                shortpath;
   TXLN                text;
   TXTM                temp;                    // location/date text
   ULONG               percent   = 0;
   ULONG               threshold = 0;           // threshold percentage
   BOOL                isMinimum = TRUE;        // threshold is minimum value
   ULN64               size      = 0;           // nr of allocated sectors
   ULN64               location  = 0;           // location value; LSN/Cluster
   ULONG               minS      = 0;           // minimal size
   ULONG               maxS      = 0;           // maximal size
   char                itemType  = 0;           // type Dir, File, Browse, or ANY
   time_t              modTstamp = 0;           // Modify timestamp value, or 0
   char               *lead    = param;
   BOOL                verbose = (strcmp( lead, ":") != 0);
   BOOL                output;                  // no Printf, just return fpath
   BOOL                alcheck;                 // Execute a full allocation check

   ENTER();
   TRARGS(("LSN:0x%llX, info:0x%2.2x select:'%s', lead:'%s'\n", lsn, info, select, lead));

   if (strcmp( lead, "!") == 0)                 // Skip allocation check wanted
   {
      output  = FALSE;                          // no output
      alcheck = FALSE;                          // and no allocation check
      *lead   = 0;                              // discard 'garbage' in lead string
   }
   else                                         // Possible output/alloc-check wanted
   {
      output  = (*lead != 0);                   // output if any lead given (except '!')
      alcheck = TRUE;                           // but always an alloc check if no  '!'
   }
   TRACES(("verbose:%u output:%u alcheck:%u\n", verbose, output, alcheck));

   dfsParseFileSelection( select, text, &isMinimum, &threshold, &minS, &maxS, &modTstamp, &itemType);
   if ((info & DFSSNINFO) && (dirsect > 2))
   {
      if ((sb = TxAlloc(5, dfsGetSectorSize())) != NULL) // 2 extra sectors BEFORE and AFTER
      {
         if ((rc = dfsRead(dirsect - 2, 5, sb)) == NO_ERROR)
         {
            BYTE      *dirdata = sb + (2 * dfsGetSectorSize());
            int        sanity;

            switch (dfsIdentifySector(dirsect, 0, dirdata))
            {
               case ST_ROOTD:
               case ST_DIREC:
                  efile = &(((S_EFATDIR *) dirdata)[entry]);
                  st    = (efile->fl.Attrib & FATTR_DIRECTORY) ? 'D' : 'f';

                  switch (st)
                  {
                     case 'D':
                        if      (itemType == DFS_FS_ITEM_BROWSE) // no filtering on Directories, take all.
                        {
                           strcpy( text, "");   // no wildcard
                           isMinimum = TRUE; threshold = 0; // no percentage filter
                           modTstamp = 0;       // no timestamp filter
                           minS      = 0;    maxS      = 0; // no size filter
                        }
                        else if (itemType == DFS_FS_ITEM_FILES) // filter files, discard Directories
                        {
                           rc = DFS_ST_MISMATCH; // item-type mismatch
                           break;
                        }
                     case 'f':
                     default:
                        if      (itemType == DFS_FS_ITEM_DIRS) // filter dirs, discard files
                        {
                           rc = DFS_ST_MISMATCH; // item-type mismatch
                           break;
                        }
                        if ((efile->EntryType & ~EFAT_DIRBIT_IN_USE) == EFAT_DEC_VLABEL)
                        {
                           //- Create specific name string for a LABEL direcory entry
                           sprintf( shortpath, "%sLabel:%s ", CNN, CBG);
                           dfsEfatEntry2Name( efile, shortpath + strlen(shortpath));
                        }
                        else                    // non-label, (long) filename
                        {
                           //- Find path upto ROOT when SLT is available, or bare name otherwise
                           //- includes handling systemfiles (bitmap/upcase) with explicit name
                           dfsEfatLsnInfo2Path( lsn, info, shortpath);
                        }
                        if ((strlen(text) == 0) || (TxStrWicmp(shortpath, text) >= 0))
                        {
                           stream = efile;      // stream is same entry for bitmap/upcase
                           switch (efile->EntryType & ~EFAT_DIRBIT_IN_USE)
                           {
                              case EFAT_DEC_VLABEL:
                                 st = 'L';
                                 break;

                              case EFAT_ANY_STREAM:
                              case EFAT_ANY_FILENAME: // need to search backwards to FILE entry
                                 sanity = 2 * dfsGetSectorSize() / sizeof(S_EFATDIR);
                                 while (((efile->EntryType & ~EFAT_DIRBIT_IN_USE) != EFAT_DEC_FILE) && (sanity--))
                                 {
                                    efile--;    // move backwards over 1 entry
                                 }
                                 if ((efile->EntryType & ~EFAT_DIRBIT_IN_USE) != EFAT_DEC_FILE)
                                 {
                                    break;      // no FILE entry found, fail
                                 }
                              case EFAT_DEC_FILE:
                                 stream = efile +1; // advance to actual stream entry
                                 if ((efile->fl.Attrib & FATTR_DIRECTORY) != 0)
                                 {
                                    st   = 'D';
                                    strcat( shortpath, FS_PATH_STR);
                                 }
                                 else
                                 {
                                    st   = 'f';
                                 }
                                 if (stream->st.FileLength != 0)
                                 {
                                    size = ((stream->st.FileLength -1) / dfsGetSectorSize()) +1;
                                 }
                                 location = dfsEfatDir2Clust( stream);
                                 break;

                              case EFAT_DEC_BITMAP:
                                 st   = 'S';
                                 if (efile->bm.FileLength != 0)
                                 {
                                    size = ((efile->bm.FileLength -1) / dfsGetSectorSize()) +1;
                                 }
                                 location = dfsEfatDir2Clust( efile);
                                 break;

                              case EFAT_DEC_UPCASE:
                                 st   = 'S';
                                 if (efile->up.FileLength != 0)
                                 {
                                    size = ((efile->up.FileLength -1) / dfsGetSectorSize()) +1;
                                 }
                                 location = dfsEfatDir2Clust( efile);
                                 break;

                              default:
                                 break;
                           }

                           if ((verbose) && (location >= 2)) // check allocation (percentage)
                           {
                              DFSISPACE       is;
                              ULONG           ef   = 0; // error flags (not used here)
                              ULN64           size = 0; // allocated size in sectors
                              ULN64           bads = 0; // sectors with bad allocation

                              if (dfsEfatAllocStream( stream, &ef, &is.chunks, &is.space) != 0)
                              {
                                 is.clsize = efat->ClustSize;
                                 size      = dfsSspaceSectors( TRUE, is.chunks, is.space);

                                 if (alcheck)
                                 {
                                    #if defined (USEWINDOWING)
                                       if (size > DFSECT_BIGF)
                                       {
                                          txwSetSbviewStatus( "Checking allocation for a huge file ...", cSchemeColor);
                                       }
                                    #endif
                                    dfsCheckAllocForSpace( &is, 0,
                                      ((efile->EntryType & EFAT_DIRBIT_IN_USE) != 0), &bads);
                                    percent = dfsAllocationReliability( size, bads);
                                 }
                                 else
                                 {
                                    isMinimum = TRUE; threshold = 0; // no percentage filter
                                 }

                                 TxFreeMem( is.space); // free the SPACE
                              }
                              TRACES(( "percent:%u threshold:%u size:%u minS:%u maxS:%u\n",
                                        percent, threshold, size, minS, maxS));
                           }

                           if ((BOOL)((percent >= threshold)) == isMinimum)
                           {
                              time_t  modifiedTime = txOS2FileTime2t( &efile->fl.DtModify.u, &efile->fl.TmModify.u);

                              if ((modifiedTime > modTstamp) &&                        //- Timestamp match
                                  ((size >= minS) && ((size <= maxS) || (maxS == 0)))) //- Size match
                              {
                                 if (verbose)   // include alloc and size
                                 {
                                    strcpy( temp, "");
                                    if (!TxaOption('C') && (location != 0))
                                    {
                                       location = dfsEfatClust2LSN( location);
                                    }
                                    if (location != 0)
                                    {
                                       sprintf( temp, "%10llX", location);
                                    }
                                    if (alcheck)
                                    {
                                       sprintf(   text, "%s%c %s%3u%% %s%10.10s ", CBM, st,
                                                  (rc == NO_ERROR) ?  CBG : CBR, percent, CNN, temp);
                                    }
                                    else        // no reliability percentage
                                    {
                                       sprintf(   text, "%s%c      %s%10.10s ", CBM, st, CNN, temp);
                                    }
                                    dfstrSize( text, CBC, size, " ");
                                    dfstrBytes(text, CBZ, 0,    " "); // XATTR/EA-size
                                 }
                                 else
                                 {
                                    strcpy( text, ""); // remove any stray text, non-verbose
                                 }
                                 if (output)    // not totally silent?
                                 {
                                    strcat(    lead, text);
                                    TxPrint( "%s%s%s%s%s\n", lead, CBY, shortpath, CNN, CGE);
                                 }
                                 else           // return info in 'select'
                                 {
                                    sprintf( temp, "%4.4u-%2.2u-%2.2u %2.2u:%2.2u:%2.2u",
                                       efile->fl.DtModify.d.year + 1980, efile->fl.DtModify.d.month, efile->fl.DtModify.d.day,
                                       efile->fl.TmModify.t.hours, efile->fl.TmModify.t.minutes, efile->fl.TmModify.t.twosecs * 2 +
                                                                   efile->fl.MsModify / 100);
                                    TxStripAnsiCodes( text);
                                    sprintf( select, "%s %s", text, temp);
                                    dfstrUllDot20( select, "", stream->st.FileLength, "");
                                 }
                                 rc = NO_ERROR;
                              }
                              else
                              {
                                 TRACES(("file-size mismatch\n"));
                                 rc = DFS_ST_MISMATCH; // file-size mismatch
                              }
                           }
                           else
                           {
                              TRACES(("percentage mismatch\n"));
                              rc = DFS_ST_MISMATCH; // percentage mismatch
                           }
                        }
                        else
                        {
                           rc = DFS_ST_MISMATCH; // wildcard mismatch
                        }
                        strcpy(  param, shortpath); // return the found name
                        break;
                  }
                  break;

               default:
                  rc = DFS_PENDING;
                  break;
            }
         }
         else if (dirsect < efat->Sect)          // should be there
         {
            dfsX10("\nError reading sector : ", dirsect, CBR, "\n");
         }
         TxFreeMem(sb);
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   else if (dirsect == efat->Root)              // ROOT, fake 1 cluster dir
   {
      if (itemType != DFS_FS_ITEM_FILES)        // unless files-only
      {
         sprintf(   text, "%s%c%s      %8llX ",  CBM, 'D', CNN, dirsect);
         dfstrSize( text, CBC, efat->ClustSize, " ");
         if (output)                            // not totally silent?
         {
            strcat(    lead, text);
            TxPrint( "%s%s%s%s%s\n", lead, CBY, FS_PATH_STR, CNN, CGE);
         }
         else                                   // return info in 'select'
         {
            sprintf( temp, "01-12-2006 12:00:00"); // no date available for ROOT
            TxStripAnsiCodes( text);
            sprintf( select, "%s %s", text, temp);
         }
         strcpy(  param, FS_PATH_STR);          // return the found name
      }
      else                                      // Root does not match item type
      {
         rc = DFS_ST_MISMATCH;
      }
   }
   else
   {
      rc = DFS_ST_MISMATCH;                     // info is NOT a valid entrynr
   }
   RETURN(rc);
}                                               // end 'dfsEfatFileInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Replace sn-list by contents of a single Directory (DirLsn, entry pairs)
/*****************************************************************************/
ULONG dfsEfatMakeBrowseList                     // RET   result
(
   ULN64               lsn,                     // IN    Directory LSN
   ULN64               info,                    // IN    dir entry, 0 for ROOT
   char               *str,                     // IN    unused
   void               *param                    // INOUT unused
)
{
   ULONG               rc = NO_ERROR;
   BYTE               *sb = NULL;               // sector buffer
   S_EFATDIR          *efile;                   // Fat directory entry, FILE
   S_EFATDIR          *stream;                  // Fat directory entry, STREAM
   ULN64               dirsect = lsn;
   USHORT              entry   = info & 0x7f;
   ULONG               location  = 0;           // location value; LSN/Cluster
   DFSISPACE           is;
   ULONG               ef;                      // error flags (not used here)

   ENTER();
   TRARGS(("LSN:0x%llX, info:0x%2.2x\n", lsn, info));

   dfsInitList(0, "-f -P", "-d");               // optimal for menu file-recovery
   if ((info & DFSSNINFO) && (dirsect > 2))     // not the ROOT, and failsafe for lsn-2
   {
      ULN64            dotdotLsn;
      BYTE             dotdotInfo;

      if (dfsEfatSearchParent( dirsect, &dotdotLsn, &dotdotInfo) == NO_ERROR)
      {
         TRACES(("List marked as 1ST_PARENT\n"));
         DFSBR_SnlistFlag |= DFSBR_1ST_PARENT;
         dfsAddSI2List( dotdotLsn, dotdotInfo); // start the list with ..
      }
      if ((sb = TxAlloc(5, dfsGetSectorSize())) != NULL) // 2 extra sectors BEFORE and AFTER
      {
         if ((rc = dfsRead(dirsect - 2, 5, sb)) == NO_ERROR)
         {
            BYTE      *dirdata = sb + (2 * dfsGetSectorSize());
            int        sanity;

            switch (dfsIdentifySector(dirsect, 0, dirdata))
            {
               case ST_ROOTD:
               case ST_DIREC:
                  rc = DFS_BAD_STRUCTURE;       // error, unless proven otherwise
                  efile  = &(((S_EFATDIR *) dirdata)[entry]);
                  switch (efile->EntryType)
                  {
                     case EFAT_DIR_STREAM:
                     case EFAT_DIR_FILENAME:   // need to search backwards to FILE entry
                        sanity = 2 * dfsGetSectorSize() / sizeof(S_EFATDIR);
                        while ((efile->EntryType != EFAT_DIR_FILE) && (sanity--))
                        {
                           efile--;             // move backwards over 1 entry
                        }
                        if (efile->EntryType != EFAT_DIR_FILE)
                        {
                           break;               // no FILE entry found, fail
                        }
                     case EFAT_DIR_FILE:
                        stream = efile +1;      // advance to actual stream entry

                        if ((efile->fl.Attrib & FATTR_DIRECTORY) != 0) // must be a DIR
                        {
                           if ((location = dfsEfatDir2Clust( stream)) >= 2)
                           {
                              if (dfsEfatAllocStream( stream, &ef, &is.chunks, &is.space) != 0)
                              {
                                 rc = NO_ERROR;
                              }
                           }
                        }
                        break;

                     case EFAT_DIR_VLABEL:     // only want a directory (aka 'FILE')
                     case EFAT_DIR_BITMAP:     // so ignore anything else
                     case EFAT_DIR_UPCASE:
                     default:
                        break;
                  }
                  break;

               default:
                  rc = DFS_ST_MISMATCH;         // not a directory sector
                  break;
            }
         }
         TxFreeMem(sb);
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   else                                         // must be ROOT itself
   {
      rc = dfsEfatDirLsn2Space( lsn, &is.chunks, &is.space);
   }
   if ((rc == NO_ERROR) && (is.space))
   {
      rc = dfsEfatDirSpace2List( is.chunks, is.space);
      TxFreeMem( is.space);                     // free the SPACE
   }
   RETURN(rc);
}                                               // end 'dfsEfatMakeBrowseList'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get allocation information for file-DATA into integrated-SPACE structure
/*****************************************************************************/
ULONG dfsEfatGetAllocSpace
(
   ULN64               lsn,                     // IN    Directory LSN
   ULN64               info,                    // IN    directory entry
   char               *str,                     // IN    unused
   void               *param                    // INOUT Integrated SPACE
)
{
   ULONG               rc = NO_ERROR;
   DFSISPACE          *ispace = (DFSISPACE *) param;
   BYTE               *sb = NULL;               // sector buffer
   S_EFATDIR          *efile;                   // Fat directory entry, FILE
   S_EFATDIR          *stream;                  // Fat directory entry, STREAM
   USHORT              entry   = info & 0x7f;
   ULONG               location  = 0;           // location value; LSN/Cluster
   ULONG               ef;                      // error flags (not used here)

   ENTER();

   ispace->space = NULL;

   if ((info & DFSSNINFO) && ( lsn > 2))     // not the ROOT, and failsafe for lsn-2
   {
      if ((sb = TxAlloc(5, dfsGetSectorSize())) != NULL) // 2 extra sectors BEFORE and AFTER
      {
         if ((rc = dfsRead( lsn - 2, 5, sb)) == NO_ERROR)
         {
            BYTE      *dirdata = sb + (2 * dfsGetSectorSize());
            int        sanity;

            switch (dfsIdentifySector( lsn, 0, dirdata))
            {
               case ST_ROOTD:
               case ST_DIREC:
                  rc = DFS_BAD_STRUCTURE;       // error, unless proven otherwise
                  efile  = &(((S_EFATDIR *) dirdata)[entry]);
                  switch (efile->EntryType)
                  {
                     case EFAT_DIR_STREAM:
                     case EFAT_DIR_FILENAME:   // need to search backwards to FILE entry
                        sanity = 2 * dfsGetSectorSize() / sizeof(S_EFATDIR);
                        while ((efile->EntryType != EFAT_DIR_FILE) && (sanity--))
                        {
                           efile--;             // move backwards over 1 entry
                        }
                        if (efile->EntryType != EFAT_DIR_FILE)
                        {
                           break;               // no FILE entry found, fail
                        }
                     case EFAT_DIR_FILE:
                        stream = efile +1;      // advance to actual stream entry
                        if ((location = dfsEfatDir2Clust( stream)) >= 2)
                        {
                           if (dfsEfatAllocStream( stream, &ef, &ispace->chunks, &ispace->space) != 0)
                           {
                              rc = NO_ERROR;
                           }
                        }
                        break;

                     case EFAT_DIR_VLABEL:     // only want a directory (aka 'FILE')
                     case EFAT_DIR_BITMAP:     // so ignore anything else
                     case EFAT_DIR_UPCASE:
                     default:
                        break;
                  }
                  break;

               default:
                  rc = DFS_ST_MISMATCH;         // not a directory sector
                  break;
            }
         }
         TxFreeMem(sb);
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   else                                         // must be ROOT itself
   {
      rc = dfsEfatDirLsn2Space( lsn, &ispace->chunks, &ispace->space);
   }
   RETURN (rc);
}                                               // end 'dfsEfatGetAllocSpace'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Put contents of a Directory SSPACE (DirLsn, entry pairs) into sn-list
/*****************************************************************************/
static ULONG dfsEfatDirSpace2List
(
   ULONG               chunks,                  // IN    nr of space entries
   S_SPACE            *space                    // IN    space allocation
)
{
   ULONG               rc = NO_ERROR;
   ULONG               chunk;                   // index in space-area
   ULONG               sect;                    // sectors to handle
   ULONG               entry;                   // index in fat directory
   S_EFATDIR           prevEntryBuffer;         // Previous entry, contents
   S_EFATDIR          *prev;                    // Previous entry, pointer
   S_EFATDIR          *this;                    // Fat directory entry
   ULN64               dirsec;                  // current dir sector LSN
   ULN64               prevsec;                 // previous dir sector LSN
   USHORT              bps = dfsGetSectorSize();
   USHORT              entries = (bps / sizeof(S_EFATDIR));
   BYTE               *dirbuf;
   S_EFATDIR          *fatdir;                  // Fat directory sector

   ENTER();

   if ((dirbuf = TxAlloc( 1, bps)) != NULL)     // allocate one sector
   {
      fatdir = (S_EFATDIR *) dirbuf;
      prev   = &prevEntryBuffer;

      prevsec = space[0].start;                 // safe start value for prev

      for (chunk = 0; (chunk < chunks)  && !TxAbort(); chunk++)
      {
         for (sect = 0; (sect < space[chunk].size)  && !TxAbort(); sect++)
         {
            dirsec  = space[chunk].start + sect;
            if ((rc = dfsRead(dirsec, 1, dirbuf)) == NO_ERROR)
            {
               for (entry = 0; (entry < entries) && !TxAbort(); entry++)
               {
                  this = &(fatdir[entry]);
                  switch (this->EntryType)      // normal only, not deleted
                  {
                     case EFAT_DIR_BITMAP:      // to be refined, ignore or include?
                     case EFAT_DIR_UPCASE:      // could make it it optional (menu toggle)
                         //- dfsAddSI2List( dirsec, entry );
                        break;

                     case EFAT_DIR_FILE:
                        {
                           BOOL         filtered = FALSE;

                           if (dfsa->browseShowHidden == FALSE) // need to check hidden/system attribute
                           {
                              if ((this->fl.Attrib & FATTR_SYSTEM) ||
                                  (this->fl.Attrib & FATTR_HIDDEN)  )
                              {
                                 filtered = TRUE;
                              }
                           }
                           if (!filtered)
                           {
                              dfsAddSI2List( dirsec, entry);
                           }
                        }
                        break;

                     case EFAT_DIR_EODIR:       // stop at FIRST empty slot!
                        entry = entries;        // force entry-loop exit
                        break;

                     case EFAT_DIR_STREAM:
                     case EFAT_DIR_FILENAME:
                     case EFAT_DIR_VLABEL:
                     default:                   // others (deleted/renamed etc)
                        break;
                  }
                  prevEntryBuffer = *this;      // make a persistent copy!
                  prevsec = dirsec;             // and remember its LSN (STREAM to FILE backtracking)
               }
            }
         }
      }
      TxFreeMem( dirbuf);
   }
   RETURN (rc);
}                                               // end 'dfsEfatDirSpace2List'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Initialize the Directory-Parent cache structure by walking the tree
/*****************************************************************************/
ULONG dfsEfatInitDpCache
(
   ULN64               dummy1,                  // IN    not used
   ULN64               d2,                      // IN    dummy
   char               *dummy2,                  // IN    not used
   void               *param                    // IN    not used
)
{
   ULONG               rc  = NO_ERROR;
   S_SPACE            *root;                    // allocation for Root
   ULONG               rootchunks;              // nr of Root fragments

   ENTER();

   if (TxaOption( TXA_O_RESET))                 // rebuild the cache
   {
      if (efat->Dp != NULL)
      {
         efat->dpCacheAlloc = 0;                // make sure next Init will
         efat->dpCacheSize  = 0;                // start with an EMPTY cache
         TxFreeMem( efat->Dp);                  // free parent cache
      }
   }

   if (efat->Dp == NULL)                        // no parent cache yet
   {
      dfsProgressInit( 0, dfsa->FsLastInUse , 0, "Sector:", "searched", DFSP_STAT, 0);

      if ((efat->EfatFaked == FALSE) &&         // no faked values (no bootrec)
          (efat->CacheA.Value != NULL))         // valid fat structures
      {
         rc = dfsEfatDirLsn2Space( efat->Root, &rootchunks, &root);
         if (rc == NO_ERROR)
         {
            //- recursively add all directories that have subdirectories
            rc = dfsEfatAdd2DpCache( rootchunks, root, efat->Root, 0);
            if (rc == NO_ERROR)
            {
               TRACES(("DpCache SORT on cluster number\n"))
               qsort( efat->Dp, efat->dpCacheSize, // sort on Cluster number
                      sizeof(DFSEFATDPCACHE), dfsEfatCompareDpItem);
            }
         }
         TxFreeMem( root);
      }
      else
      {
         rc = DFS_ST_MISMATCH;                  // bad structures
      }
      dfsProgressTerm();
   }
   RETURN (rc);
}                                               // end 'dfsEfatInitDpCache'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Compare Dir-Parent cache elements for qsort, sort on cluster number
/*****************************************************************************/
static int dfsEfatCompareDpItem
(
   const void         *one,
   const void         *two
)
{
   DFSEFATDPCACHE     *p1  = (DFSEFATDPCACHE *) one;
   DFSEFATDPCACHE     *p2  = (DFSEFATDPCACHE *) two;
   int                 res = 0;

   ENTER();

   if (p1->Cluster != p2->Cluster)
   {
      res = (int) ((p1->Cluster < p2->Cluster) ? -1 : +1);
   }
   RETURN(res);
}                                               // end 'dfsEfatCompareDpItem'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add each clusternumber for an EFAT directory to the Directory-Parent cache
/*****************************************************************************/
static ULONG dfsEfatAdd2DpCache
(
   ULONG               chunks,                  // IN    nr of S_SPACE chunks
   S_SPACE            *space,                   // IN    space structure array
   ULN64               parentLsn,               // IN    parent dirsector LSN
   BYTE                parentIdx                // IN    parent index in sector
)
{
   ULONG               rc = NO_ERROR;
   ULONG               chunk;                   // index in space-area
   ULONG               sect = 0;                // sectors to handle
   USHORT              entry;                   // index in fat directory
   S_EFATDIR           prevEntryBuffer;         // Previous entry, contents
   S_EFATDIR          *prev;                    // Previous entry, pointer
   S_EFATDIR          *this;                    // Fat directory entry
   S_EFATDIR          *fatdir;                  // Fat directory sector
   ULN64               dlsn;                    // Directory sector LSN
   ULN64               prevFileLsn   = 0;       // Previous  LSN,   FILE entry
   USHORT              prevFileEntry = 0;       // Previous  entry, FILE
   S_SPACE            *sp = NULL;               // Alloc chunk array
   USHORT              fatEntries;              // entries per sector
   ULONG               lastCachedCluster = 0;   // last CL-number added to cache
   ULONG               currentCluster;          // CL-number for this entry
   ULONG               ef;                      // error flags (not used here)
   ULONG               nr;                      // Alloc chunks
   BOOL                debug = TxaOption( TXA_O_DEBUG);

   ENTER();

   prevEntryBuffer.EntryType = 0;               // satisfy GCC checking ...

   TRACES(("chunks: %u, parentLsn:0x%llX, parentIdx: %hhx\n", chunks, parentLsn, parentIdx));

   if ((fatdir = TxAlloc(1, dfsGetSectorSize())) != NULL)
   {
      prev   = &prevEntryBuffer;

      for ( chunk = 0;
           (chunk < chunks) && (rc == NO_ERROR) && !TxAbort();
            chunk++)                            // walk all alloc chunks
      {
         TRACES(("Start on chunk: %u, with %u sectors\n", chunks, sect < space[chunk].size));
         for ( sect = 0;
              (sect < space[chunk].size) && (rc == NO_ERROR) && !TxAbort();
               sect++)                          // each sector in chunk
         {
            fatEntries = dfsGetSectorSize() / sizeof(S_EFATDIR);
            dlsn = space[chunk].start + sect;
            rc = dfsRead( dlsn, 1, (BYTE   *) fatdir);
            for ( entry = 0;
                 (entry < fatEntries) && (rc == NO_ERROR) && !TxAbort();
                  entry++)                      // each dir-entry in sector
            {
               this = &(fatdir[entry]);

               if (this->EntryType)
               {
                  TRACES(("Chunk: %u, Sect: %u, entry %2u, type:0x%2.2hx\n", chunk, sect, entry, this->EntryType));
               }
               switch (this->EntryType)
               {
                  case EFAT_DIR_FILE:
                     prevFileLsn   = dlsn;    //- remember LSN and entry
                     prevFileEntry = entry;   //- for the STREAM processing
                     break;

                  case EFAT_DIR_STREAM:
                     if (prev->EntryType == EFAT_DIR_FILE)
                     {
                        if ((prev->fl.Attrib & FATTR_DIRECTORY) != 0)
                        {
                           TRACES(("Process subdir at LSN:0x%llx, entry: %hx\n", prevFileLsn, prevFileEntry));
                           dfsEfatAllocStream( this, &ef, &nr, &sp);
                           if (sp != NULL)
                           {
                              currentCluster = dfsEfatLSN2Clust( prevFileLsn);
                              if (currentCluster != lastCachedCluster)
                              {
                                 if (debug)
                                 {
                                    TxPrint("Add cluster %8.8x with parent 0x%llx-%hhx at entry %u\n",
                                             currentCluster, parentLsn, parentIdx & 0x7f, efat->dpCacheSize);
                                 }
                                 else
                                 {
                                    TRACES(("Add cluster %8.8x with parent 0x%llx-%hhx at entry %u\n",
                                             currentCluster, parentLsn, parentIdx & 0x7f, efat->dpCacheSize));
                                 }

                                 if (efat->dpCacheAlloc <= efat->dpCacheSize) // areas used up ?
                                 {
                                    efat->dpCacheAlloc = efat->dpCacheAlloc * 8 + 1024;
                                    TRACES(("(re)Allocate %u DP-cache entries\n", efat->dpCacheAlloc));
                                    efat->Dp = (DFSEFATDPCACHE *) realloc( efat->Dp,
                                      (size_t) (efat->dpCacheAlloc) * sizeof(DFSEFATDPCACHE));
                                 }
                                 efat->Dp[ efat->dpCacheSize].Cluster   = currentCluster;
                                 efat->Dp[ efat->dpCacheSize].ParentLsn = parentLsn;
                                 efat->Dp[ efat->dpCacheSize].ParentIdx = parentIdx;
                                 efat->dpCacheSize++;

                                 dfsProgressShow( dlsn, 0, efat->dpCacheSize, "Cached:");

                                 lastCachedCluster = currentCluster;
                              }
                              rc = dfsEfatAdd2DpCache( nr, sp, prevFileLsn, prevFileEntry | DFSSNINFO);
                              TxFreeMem( sp);
                           }
                        }
                     }
                     else
                     {
                        TRACES(( "EFAT stream, unexpected at this position!\n"));
                     }
                     break;

                  case EFAT_DIR_EODIR:          // stop at FIRST empty slot!
                     entry = fatEntries;        // force entry-loop exit
                     break;

                  case EFAT_DIR_BITMAP:
                  case EFAT_DIR_UPCASE:
                  default:                      // other types, deleted, name etc
                     //- no action
                     break;
               }
               prevEntryBuffer = *this;         // make persistent copy, each time!
            }
         }
      }
      TxFreeMem( fatdir);
   }
   RETURN( rc);
}                                               // end 'dfsEfatAdd2DpCache'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Search DP-cache for parent Lsn+index for specified directory sector number
/*****************************************************************************/
ULONG dfsEfatSearchParent                       // RET   NOT_FOUND when fail
(
   ULN64               dirLsn,                  // IN    DIR sector number
   ULN64              *parentLsn,               // OUT   parent dirsector LSN
   BYTE               *parentIdx                // OUT   parent index in sector
)
{
   ULONG               rc = DFS_NOT_FOUND;
   ULONG               cl = dfsEfatLSN2Clust( dirLsn); // DIR cluster, with subdir
   ULONG               i;

   ENTER();

   if (efat->Dp)                                // cache is available
   {
      for (i = 0; i < efat->dpCacheSize; i++)
      {
         TRACES(("Item:%3u cl:%8.8x Cluster:%8.8x\n", i, cl, efat->Dp[ i].Cluster));
         if       (cl == efat->Dp[ i].Cluster)
         {
            *parentLsn = efat->Dp[ i].ParentLsn;
            *parentIdx = efat->Dp[ i].ParentIdx;
            rc = NO_ERROR;
            break;                              // found parent
         }
         else if  (cl <  efat->Dp[ i].Cluster)
         {
            break;                              // past our cl number
         }                                      // in sorted array
      }
   }
   TRACES(("dirLsn:0x%llX = cl:%8.8x -> parent Lsn:0x%llx-%hhx\n",
            dirLsn,         cl,        *parentLsn,   *parentIdx));
   RETURN( rc);
}                                               // end 'dfsEfatSearchParent'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get long filename for given LSN and entry number for DIR sector
/*****************************************************************************/
ULONG dfsEfatLsnInfo2Name                        // RET   result
(
   ULN64               lsn,                     // IN    Directory LSN
   ULONG               info,                    // IN    directory entry info
   char               *name                     // OUT   directory entry name
)
{
   ULONG               rc = NO_ERROR;

   ENTER();
   TRARGS(("LSN:0x%llX, info:0x%2.2x\n", lsn, info));

   if (info & DFSSNINFO)
   {
      BYTE               *sb = NULL;            // sector buffer
      S_EFATDIR           *efile;               // Fat directory entry
      ULN64               dirsect = lsn;
      USHORT              entry   = info & ~DFSSNINFO;

      //- Need to allocate/read 5 sectors to allow reading LFN up to 255 characters
      if ((sb = TxAlloc(5, dfsGetSectorSize())) != NULL) // 2 extra sectors BEFORE and AFTER
      {
         if ((rc = dfsRead(dirsect - 2, 5, sb)) == NO_ERROR)
         {
            BYTE      *dirdata = sb + (2 * dfsGetSectorSize());

            switch (dfsIdentifySector(dirsect, 0, dirdata))
            {
               case ST_ROOTD:
               case ST_DIREC:
                  efile   = &(((S_EFATDIR *) dirdata)[entry]);

                  dfsEfatEntry2Name( efile, name); // get real or faked file name
                  break;

               default:
                  strcpy( name, "");
                  rc = DFS_ST_MISMATCH;
                  break;
            }
         }
         TxFreeMem(sb);
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   else
   {
      rc = DFS_ST_MISMATCH;                     // info is NOT a valid entrynr
   }
   RETURN(rc);
}                                               // end 'dfsEfatLsnInfo2Name'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Find full PATH to Root-directory, starting at some dir LSN and entry info
/*****************************************************************************/
ULONG dfsEfatLsnInfo2Path                        // RET   result und
(
   ULN64               start,                   // IN    starting lsn
   ULONG               entry,                   // IN    directory entry info
   char               *path                     // OUT   combined path string
)
{
   ULONG               rc = NO_ERROR;
   ULN64               ref = start;             // Reference/Parent LSN
   USHORT              info = entry;            // entry info from SLT
   BYTE                st;
   ULONG               sanity = 20;             // maximum iterations
   ULONG               sltIndex;                // dummy SLT location
   TX1K                fname;                   // shortname (+ path)

   ENTER();
   if (path)
   {
      *path = '\0';
   }

   rc = dfsEfatLsnInfo2Name( start, info, path); // get name for THIS dir entry

   if (rc == NO_ERROR)
   {
      if (dfsSlTableStatus(NULL) == SLT_READY) // there is an SLT
      {
         do
         {
            if (dfsSlTableFind( ref, &ref, &info, &st, &sltIndex))   //- Parent, next REF
            {
               if (ref == LSN_BOOTR)            // Root-DIR refrences to boot
               {
                  strcpy( fname, FS_PATH_STR);  // ROOT
                  strcat( fname, path);         // append existing path
                  strcpy( path, fname);         // and copy back
                  break;
               }
               else
               {
                  if ((rc = dfsEfatLsnInfo2Name( ref, info, fname)) == NO_ERROR)
                  {
                     TRACES(("'%s' + '%s'\n", fname, path));
                     strcat( fname, FS_PATH_STR);
                     strcat( fname, path);      // append existing path
                     strcpy( path, fname);      // and copy back
                  }
               }
            }
            else
            {
               rc = DFS_NOT_FOUND;
            }
            sanity--;
         } while ((rc == NO_ERROR) && (sanity) && !TxAbort());
      }
      else
      {
         rc = DFS_CMD_WARNING;
      }
   }
   TRACES(("Found path to root for 0x%llX+%2.2x:'%s'\n", start, entry, path));

   RETURN(rc);
}                                               // end 'dfsEfatLsnInfo2Path'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Determine type of whole directory cluster and cache it for later reuse
/*****************************************************************************/
BYTE dfsEfatDirClusterType                      // RET   DIR type or UDATA
(
   ULN64               lsn                      // IN    LSN of a (DIR) sector
)
{
   BYTE                rc = ST_UDATA;           // function return
   BYTE               *sb = NULL;               // sector buffer
   static ULN64        csFirst = 0;             // first in cached range
   static ULN64        csLast  = 0;             // last
   static BYTE         csType  = ST_UDATA;      // the associated type info
   ULN64               dirsect = csFirst;       // first sector of DIR cluster

   ENTER();
   TRACES(("lsn:0x%llx csF:0x%llx csL:%8.8x csT:%c\n", lsn, csFirst, csLast, csType));

   if ((lsn >= efat->ClHeap) && (lsn < efat->Sect)) // possible (sub) directory
   {
      if ((lsn >= csFirst) && (lsn <= csLast))  // in cached range ?
      {
         rc = csType;                           // use cached value
      }
      else                                      // determine and cache new type
      {
         if ((sb = TxAlloc( efat->ClustSize, dfsGetSectorSize())) != NULL)
         {
            dirsect = dfsEfatClust2LSN( dfsEfatLSN2Clust( lsn));
            if (dfsRead( dirsect, efat->ClustSize, sb) == NO_ERROR)
            {
               S_EFATDIR *sf;                   // entry, with short name
               S_EFATDIR *fatdir = (S_EFATDIR *) sb;
               BOOL       fdir   = TRUE;        // possible FAT directory
               ULONG      entry;                // index in fat directory
               S_EFATDIR  refdent;              // reference DIR-entry
               USHORT     fatEntries;           // entries per sector

               fatEntries = (dfsGetSectorSize() * efat->ClustSize) / sizeof(S_EFATDIR);
               for (entry = 0; (entry < fatEntries) && fdir; entry++)
               {
                  sf = &(fatdir[entry]);        // ptr to current entry

                  switch (sf->EntryType & ~EFAT_DIRBIT_IN_USE)
                  {
                     case EFAT_DEC_VLABEL:     // accept valid types, may do more checks
                        break;

                     case EFAT_DEC_BITMAP:     // like clustervalue < limit and not 0
                     case EFAT_DEC_UPCASE:
                        break;

                     case EFAT_DEC_FILE:
                        if      ((sf->fl.DtAccess.u == 0)         || //- no zero date (1-1-1980)
                                 (sf->fl.DtAccess.u > MAX_S_DATE) || //- allow valid date and
                                 (sf->fl.TmAccess.u > MAX_S_TIME)  ) //- valid time only
                        {
                           TRACEX(("dir-entry failed on invalid ACCESS date/time, entry: %u\n", entry));
                           fdir = FALSE;
                        }
                        else if ((sf->fl.DtModify.u == 0)         || //- no zero date (1-1-1980)
                                 (sf->fl.DtModify.u > MAX_S_DATE) || //- allow valid date and
                                 (sf->fl.TmModify.u > MAX_S_TIME) || //- valid time only
                                 (sf->fl.MsModify   > MAX_S_MS10)  ) //- valid 10-msecs only
                        {
                           TRACEX(("dir-entry failed on invalid MODIFY date/time, entry: %u\n", entry));
                           fdir = FALSE;
                        }
                        else if ((sf->fl.DtCreate.u == 0)         || //- no zero date (1-1-1980)
                                 (sf->fl.DtCreate.u > MAX_S_DATE) || //- allow valid date and
                                 (sf->fl.TmCreate.u > MAX_S_TIME) || //- valid time only
                                 (sf->fl.MsCreate   > MAX_S_MS10)  ) //- valid 10-msecs only
                        {
                           TRACEX(("dir-entry failed on invalid CREATE date/time, entry: %u\n", entry));
                           fdir = FALSE;
                        }
                        break;

                     case EFAT_ANY_STREAM:
                        //- to be refined, do some basic checks like valid cluster
                        break;

                     case EFAT_ANY_FILENAME:
                        //- no checks possible (any UNICODE string is valid)
                        break;

                     case EFAT_DIR_VOLGUID:
                     case EFAT_DIR_TEFATPAD:
                     case EFAT_DIR_WINCEACT:
                        //- to be refined
                        break;

                     case EFAT_DIR_EODIR:
                        memset(     &refdent,  0, sizeof(S_EFATDIR));
                        if (memcmp( sf, &refdent, sizeof(S_EFATDIR)) != 0)
                        {
                           fdir = FALSE;
                           TRACEX(("dir-entry failed on non-zero fields in entry: %u\n", entry));
                        }
                        break;

                     default:
                        if ((sf->EntryType & EFAT_DIRBITSUNUSED) != 0)
                        {
                           fdir = FALSE;
                           TRACEX(("dir-entry failed on invalid entrytype in entry: %u\n", entry));
                        }
                        break;
                  }
               }
               if (fdir)                        // valid DIR cluster
               {
                  rc = ST_DIREC;                // any kind of dir-sector
               }
               csType  = rc;                    // update type cache
               csFirst = dirsect;
               csLast  = dirsect + efat->ClustSize -1;
            }
            TxFreeMem( sb);
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsEfatDirClusterType'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate Cluster-nr to LSN
/*****************************************************************************/
ULN64 dfsEfatCl2Lsn                             // RET   LSN for this cluster
(
   ULN64               cluster,                 // IN    cluster number
   ULN64               d2,                      // IN    dummy
   char               *p1,                      // IN    dummy
   void               *p2                       // IN    dummy
)
{
   ULN64               lsn;

   if ((cluster < 2) && (efat->Root != 0))      // CL 0/1 and initialized ?
   {
      lsn = efat->Root;                         // allow cluster 0/1 in DIR
   }                                            // entries to point to root
   else                                         // real clusters in ClHeap
   {
      lsn = efat->ClHeap + ((cluster -2) * efat->ClustSize);
   }
   //- TRACES(("Cl:0x%llx => lsn:0x%llx\n", cluster, lsn));
   return (lsn);
}                                               // end 'dfsEfatCl2Lsn'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate LSN to Cluster-nr
/*****************************************************************************/
ULN64 dfsEfatLsn2Cl                             // RET   cluster for this LSN
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *p1,                      // IN    dummy
   void               *p2                       // IN    dummy
)
{
   ULN64               cluster = 2;             // minimal valid cluster

   if (lsn == DFS_MAX_PSN)                      // query max Cluster-nr
   {
      cluster = efat->RawClust - 1;             // highest is #clusters - 1
   }
   else if (lsn >= efat->ClHeap)
   {
      cluster = ((lsn - efat->ClHeap) / efat->ClustSize) + 2;
   }
   else if (lsn >= efat->FatOffset)
   {
      cluster = 1;                              // simulate, CL 1 is FAT area
   }
   else
   {
      cluster = 0;                              // simulate, CL 0, upto 1st FAT
   }
   //- TRACES(("lsn:0x%llx => Cl:0x%llx\n", lsn, cluster));
   return (cluster);
}                                               // end 'dfsEfatLsn2Cl'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get cluster-value from a Directory entry structure
/*****************************************************************************/
ULONG dfsEfatDir2Clust                          // RET   cluster value
(
   S_EFATDIR           *entry                   // IN    FAT directory entry
)
{
   ULONG               cluster = entry->st.Cluster;

   return (cluster);
}                                               // end 'dfsEfatDir2Clust'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set cluster-value in a Directory entry structure
/*****************************************************************************/
void dfsEfatClust2Dir
(
   ULONG               cluster,                 // IN    cluster value to set
   S_EFATDIR           *entry                   // INOUT FAT directory entry
)
{
   entry->st.Cluster = cluster;
}                                               // end 'dfsEfatClust2Dir'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Check if cluster could be the 1st root-DIR sector (DIR but no '..' entry)
/*****************************************************************************/
BOOL dfsEfatRootCandidate                       // RET   Could be a root DIR
(
   ULONG               cluster                  // IN    target cluster
)
{
   BOOL                rc = FALSE;              // function return
   BYTE               *sb = NULL;               // sector buffer
   ULONG               dirsect;

   ENTER();
   TRARGS(("cluster: %8.8x\n", cluster));

   if ((sb = TxAlloc(1, dfsGetSectorSize())) != NULL)
   {
      dirsect = dfsEfatClust2LSN( cluster);
      if (dfsRead( dirsect, 1, sb) == NO_ERROR)
      {
         switch (dfsIdentifySector( dirsect, 0, sb))
         {
            case ST_ROOTD:
            case ST_DIREC:
               rc = TRUE;
               break;

            default:
               break;
         }
      }
      TxFreeMem(sb);
   }
   BRETURN (rc);
}                                               // end 'dfsEfatRootCandidate'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get cluster-value from the Fat Cache structure (32 bit entries)
/*****************************************************************************/
ULONG dfsEfatValue                              // RET   cluster value
(
   ULONG               clust                    // IN    cluster to get value
)
{
   ULONG               value = 0;               // rc, cluster value
   DFSEFATCACHE        *this = &efat->CacheA;
   DFSEFATCACHE        *oldest;

   if ((clust < this->First) || (clust >= (this->First + DFSEFAT_CACHE)))
   {
      this = &efat->CacheB;
      if ((clust < this->First) || (clust >= (this->First + DFSEFAT_CACHE)))
      {
         //- Not in either cache, select a cache to be reloaded with needed info
         oldest = (this->Stamp < efat->CacheA.Stamp) ? this : &efat->CacheA;
         if (((this->Dirty == TRUE ) && (efat->CacheA.Dirty == TRUE )) ||
             ((this->Dirty == FALSE) && (efat->CacheA.Dirty == FALSE))  )
         {
            this = oldest;                      // same, use oldest, causing toggle
         }
         else                                   // reload the non-Dirty one, to allow
         {                                      // successive Rd/Wr without thrashing
            if (this->Dirty == TRUE)            // B is dirty
            {
               this = &efat->CacheA;            // so reload and usa A
            }
         }
         TRACES(("FatValue for 0x%8.8x, not cached, reload: cache-%c\n", clust, this->Id));
         if (this->Dirty == TRUE)               // selected cache is Dirty
         {
            //- Flush the dirty cache before reloading it
            dfsEfatFlushFAT( this);
         }
         dfsEfatGetFAT( clust, this);           // reload needed part of FAT
      }
   }
   value = this->Value[clust % DFSEFAT_CACHE];  // get cluster value from cache

   return (value);
}                                               // end 'dfsEfatValue'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set cluster-value in the Fat Cache structure (32 bit entries)
/*****************************************************************************/
ULONG dfsEfatSetCluster                         // RET   NO_CHANGE / NO_ERROR
(
   ULONG               clust,                   // IN    cluster to get value
   ULONG               value                    // IN    FAT-value to be set
)
{
   ULONG               rc = NO_ERROR;
   DFSEFATCACHE        *this = &efat->CacheA;
   DFSEFATCACHE        *oldest;

   if ((clust < this->First) || (clust >= (this->First + DFSEFAT_CACHE)))
   {
      this = &efat->CacheB;
      if ((clust < this->First) || (clust >= (this->First + DFSEFAT_CACHE)))
      {
         //- Not in either cache, select a cache to be reloaded with needed info
         oldest = (this->Stamp < efat->CacheA.Stamp) ? this : &efat->CacheA;
         if (((this->Dirty == TRUE ) && (efat->CacheA.Dirty == TRUE )) ||
             ((this->Dirty == FALSE) && (efat->CacheA.Dirty == FALSE))  )
         {
            this = oldest;                      // same, use oldest, causing toggle
         }
         else                                   // reload the Dirty one, to allow
         {                                      // successive Rd/Wr without thrashing
            if (this->Dirty == FALSE)           // B is clean
            {
               this = &efat->CacheA;            // so reload and use A
            }
         }
         TRACES(("SetCluster for 0x%8.8x, not cached, reload: cache-%c\n", clust, this->Id));
         if (this->Dirty == TRUE)               // selected cache is Dirty
         {
            //- Flush the dirty cache before reloading it
            dfsEfatFlushFAT( this);
         }
         rc = dfsEfatGetFAT( clust, this);      // reload needed part of FAT
      }
   }
   if (this->Value[clust % DFSEFAT_CACHE] != value)
   {
      if (this->Dirty == FALSE)
      {
         TRACES(("Cache %c now DIRTY on writing: 0x%8.8x(%7x)\n", this->Id, clust, value));
      }
      this->Value[clust % DFSEFAT_CACHE] = value; // set cluster value in cache
      this->Dirty = TRUE;
   }
   else
   {
      rc = DFS_NO_CHANGE;                       // value written same as existing
   }
   return (rc);
}                                               // end 'dfsEfatSetCluster'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Reset bad-sector admin, change each BAD cluster in the FAT to FREE
/*****************************************************************************/
ULONG dfsEfatResetBadClus                       // RET   ref cluster or 0
(
   void
)
{
   ULONG               rc = NO_ERROR;           // function return
   ULONG               cBase;                   // lowest cl in cache
   ULONG               cOffs;                   // offset cl in cache
   ULONG               cFirst;                  // first offset to check
   ULONG               cLast;                   // last offset to check
   ULONG               bads = 0;                // total bad clusters
   DFSEFATCACHE        *cache = &efat->CacheA;

   ENTER();
   TRACES(( "Fat1:0x%llx Fat2:0x%llx InUse:0x%llx\n", efat->Fat1, efat->Fat2, efat->FatInUse));

   TxPrint( "\nScanning FAT area for clusters marked 'BAD' ...\n");

   //- Since FAT areas might be changed, it is fine to use and invalidate the caches
   efat->CacheB.First  = EFAT_NO_CLUST;         // invalidate B cache

   for ( cBase  = 0;                            // Big steps, whole cache ...
        (cBase <= efat->MapClust) && (rc == NO_ERROR) && !TxAbort();
         cBase += DFSEFAT_CACHE)
   {
      dfsEfatGetFAT( cBase, cache);              // cache needed part of FAT

      cFirst = (cBase == 0) ? 2 : 0;            // first valid cl is 2
      if ((cBase + DFSEFAT_CACHE) > efat->MapClust)
      {
         cLast = efat->MapClust % DFSEFAT_CACHE;     // last valid offset
      }
      else
      {
         cLast = DFSEFAT_CACHE -1;               // last offset within cache
      }
      for (cOffs = cFirst; (cOffs <= cLast) && (rc == 0); cOffs++)
      {
         if (cache->Value[cOffs] == EFAT_BADCLUST)
         {
            TRACES(( "Reset BAD cluster nr: %8.8x\n", cBase + cOffs));
            cache->Value[cOffs] = EFAT_FREECLUS;
            bads++;
            cache->Dirty = TRUE;
         }
      }
      if (cache->Dirty)
      {
         rc = dfsEfatFlushFAT( cache);
      }
   }
   if (bads > 0)
   {
      dfsSz64( "\nTotal bad sectors : ", (ULN64) bads * efat->ClustSize, ", have been reset\n");
   }
   else
   {
      TxPrint( "\nNo bad sectors present, no action required ...\n");
   }
   RETURN (rc);
}                                               // end 'dfsEfatResetBadClus'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Search reference to specified cluster in FAT (cluster pointing to this one)
// Note: Dont use regular cache to avoid trashing that just by searching here
/*****************************************************************************/
ULONG dfsEfatFindReference                       // RET   ref cluster or 0
(
   ULONG               cluster                  // IN    target cluster
)
{
   ULONG               rc = 0;                  // function return
   ULONG               cBase;                   // lowest cl in cache
   ULONG               cOffs;                   // offset cl in cache
   ULONG               cFirst;                  // first offset to check
   ULONG               cLast;                   // last offset to check
   DFSEFATCACHE        cache;                   // allocated local cache

   ENTER();
   TRARGS(("Find cluster ref for: %8.8x, in FAT at 0x%llx\n", cluster, efat->FatInUse));

   //- avoid use of FatValue() and use local cache to gain speed and avoid cache trashing
   if ((cache.Value = TxAlloc(DFSEFAT_CACHE, sizeof(ULONG))) != NULL)
   {
      for ( cBase  = 0;                         // Big steps, whole cache ...
           (cBase <= efat->MapClust) && (rc == 0) && !TxAbort();
            cBase += DFSEFAT_CACHE)
      {
         dfsEfatGetFAT( cBase, &cache);          // cache needed part of FAT

         cFirst = (cBase == 0) ? 2 : 0;         // first valid cl is 2
         if ((cBase + DFSEFAT_CACHE) > efat->MapClust)
         {
            cLast = efat->MapClust % DFSEFAT_CACHE; // last valid offset
         }
         else
         {
            cLast = DFSEFAT_CACHE -1;            // last offset within cache
         }
         for (cOffs = cFirst; (cOffs <= cLast) && (rc == 0); cOffs++)
         {
            if (cache.Value[cOffs] == cluster)
            {
               rc = cBase + cOffs;
            }
         }
      }
      TxFreeMem( cache.Value);
   }
   RETURN (rc);
}                                               // end 'dfsEfatFindReference'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Calulate 32-bit CRC over FAT-contents (to compare FAT1 and FAT2 perhaps :-)
/*****************************************************************************/
ULONG dfsEfatCalculateFatCRC                     // RET   32-bit CRC
(
   ULN64               fatlsn                   // IN    first sector of FAT
)
{
   ULONG               rc = 0;                  // function return
   ULONG               cBase;                   // lowest cl in cache

   ENTER();
   TRARGS(("fatlsn:0x%llx\n", fatlsn));

   #if defined (USEWINDOWING)
      if (txwIsWindow( TXHWND_DESKTOP))
      {
         dfsa->statusTimer = TxTmrSetTimer( TMR_SEC( 1)); // only after 1 second ...
      }
   #endif
   dfsEfatSetFatInUse( fatlsn);                  // flush cache and switch FAT

   for ( cBase  = 0;                            // Big steps, whole cache ...
        (cBase <= efat->MapClust) && !TxAbort();
         cBase += DFSEFAT_CACHE)
   {
      #if defined (USEWINDOWING)
         if (txwIsWindow( TXHWND_DESKTOP))
         {
            if (TxTmrTimerExpired( dfsa->statusTimer))
            {
               TXTM    status;
               sprintf( status, "Calculate CRC FAT%c, done %9u of %u",
                       (fatlsn == efat->Fat1) ? '1' :
                       (fatlsn == efat->Fat2) ? '2' : 'x',
                        cBase, efat->MapClust);
               txwSetSbviewStatus( status, cSchemeColor);
               dfsa->statusTimer = TxTmrSetTimer( DFSP_STATUS_INTERVAL);
            }
         }
      #endif

      dfsEfatGetFAT(  cBase, &efat->CacheA);      // cache needed part of FAT
      rc ^= TxCrc32( efat->CacheA.Value, DFSEFAT_CACHE * sizeof(ULONG));
      TRACES(( "CRC value from cluster %8.8x = %8.8x\n", cBase, rc));
   }
   RETURN (rc);
}                                               // end 'dfsEfatCalculateFatCRC'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set the active FAT LSN to be used for fat references using the cache
/*****************************************************************************/
void dfsEfatSetFatInUse
(
   ULN64               fat2Use                  // IN    first sector of FAT
)
{
   ENTER();
   TRARGS(("fat2Use:0x%llx\n", fat2Use));

   if (efat->FatInUse != fat2Use)
   {
      //- Flush changes to disk, BEFORE switching to other FAT disk area
      if (efat->CacheA.Dirty)
      {
         dfsEfatFlushFAT( &efat->CacheA);
      }
      if (efat->CacheB.Dirty)
      {
         dfsEfatFlushFAT( &efat->CacheB);
      }
      efat->FatInUse      = fat2Use;            // switch
      efat->CacheA.First  = EFAT_NO_CLUST;
      efat->CacheB.First  = EFAT_NO_CLUST;      // invalidate FAT caches
   }
   VRETURN();
}                                               // end 'dfsEfatSetFatInUse'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Copy FAT-part from disk to specified FAT cache structure (32 bit entries)
/*****************************************************************************/
ULONG dfsEfatGetFAT                             // RET   result
(
   ULONG               clust,                   // IN    cluster value to get
   DFSEFATCACHE        *cache                   // INOUT cache structure used
)
{
   ULONG               rc  = NO_ERROR;
   ULN64               lsn;                     // LSN of 1st sector to read
   ULONG               sectors;                 // nr of FAT sectors to read
   BYTE               *dfat;                    // FAT sector buffer
   ULONG               clusters;                // cluster values in chunk

   ENTER();

   cache->First  = clust -  (clust % DFSEFAT_CACHE);

   if ((clusters = (efat->MapClust - cache->First)) > DFSEFAT_CACHE)
   {
      clusters = DFSEFAT_CACHE;
   }

   TRARGS(("FAT LSN:0x%llx for cl:0x%8.8x, first:0x%8.8x, clusters:0x%4.4x  cache:%c\n",
            efat->FatInUse, clust, cache->First, clusters, cache->Id));

   if ((efat->FatInUse != 0) && (clusters != 0))
   {
      dfat    = (BYTE   *) cache->Value;
      lsn     = cache->First   / (dfsGetSectorSize() / sizeof(ULONG));
      sectors = ((clusters -1) / (dfsGetSectorSize() / sizeof(ULONG))) +1;
      TRACES(( "Reading %u FAT sectors, relative lsn:0x%llx\n", sectors, lsn));

      lsn += efat->FatInUse;                     // add to start of FAT lsn
      if ((dfat != NULL) && (cache->Value != NULL))
      {
         //- make sure whole cache-buffer is ZERO to get reliable CRC
         //- values for partially filled ones (the last one)

         memset( cache->Value, 0, (DFSEFAT_CACHE * sizeof(ULONG)));

         if ((rc = dfsRead( dfstAreaP2Disk( DFSTORE, lsn), sectors, dfat)) == NO_ERROR)
         {
            if (++lastCacheStamp == 0)          // when wrapped, reset BOTH caches ages
            {
               efat->CacheA.Stamp = 0;
               efat->CacheB.Stamp = 0;
               lastCacheStamp = 1;              // will be assigned to current one
            }
            cache->Stamp = lastCacheStamp;
            cache->Dirty = FALSE;               // clean now

            TRHEXS(70,dfat, 128, "start of physical FAT");
            TRACES(("Converted to FAT cache, age is: 0x%8.8x, contents:\n", lastCacheStamp));
            TRHEXS(70,cache->Value, 0x80, "start of FAT cache");
         }
         else
         {
            dfsX10("\nFailed to read FAT sectors at LSN : ", lsn, CBR, "\n");
         }
         if ((ULONG *) dfat != cache->Value)    // temp buffer allocated ?
         {
            TxFreeMem(dfat);
         }
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   else
   {
      rc = DFS_BAD_STRUCTURE;
   }
   RETURN (rc);
}                                               // end 'dfsEfatGetFAT'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Flush specified FAT cache (32 bit entries) back to disk FatInUse (+ other)
/*****************************************************************************/
ULONG dfsEfatFlushFAT                            // RET   result
(
   DFSEFATCACHE        *cache                    // INOUT cache structure to write
)
{
   ULONG               rc  = NO_ERROR;
   ULN64               lsn;                     // LSN of 1st sector to read
   ULONG               sectors;                 // nr of FAT sectors to read
   ULONG              *lfat;                    // logical FAT area, 32 bit
   BYTE               *dfat;                    // FAT sector buffer
   ULONG               clusters;                // cluster values in chunk

   ENTER();

   if ((clusters = (efat->MapClust - cache->First)) > DFSEFAT_CACHE)
   {
      clusters = DFSEFAT_CACHE;
   }
   TRARGS(("FAT LSN:0x%llx first:0x%8.8x, clusters:0x%8.8x  cache:%c\n",
                  efat->Fat1, cache->First, clusters, cache->Id));

   if (clusters != 0)
   {
      dfat    = (BYTE   *) cache->Value;
      lsn     = cache->First   / (dfsGetSectorSize() / sizeof(ULONG));
      sectors = ((clusters -1) / (dfsGetSectorSize() / sizeof(ULONG))) +1;

      lfat = cache->Value;
      if ((dfat != NULL) && (lfat != NULL))
      {
         TRACES(( "Writing %u FAT sectors, relative lsn:0x%llx\n", sectors, lsn));
         //- write to first-fat area
         if ((rc = dfsWrite( dfstAreaP2Disk( DFSTORE, lsn + efat->FatInUse), sectors, dfat)) == NO_ERROR)
         {
            TRHEXS(70,cache->Value, 0x80, "start of FAT cache");
            TRACES(("Converted to physical FAT, contents:\n"));
            TRHEXS(70,dfat, 128, "start of physical FAT");

            //- write to second-fat area, IFF active FAT is Fat1 or Fat2 (not a temp one)
            if      ((efat->FatInUse == efat->Fat1) && (efat->Fat2 != 0)) // 1 used, 2 present
            {
               dfsWrite( dfstAreaP2Disk( DFSTORE, lsn + efat->Fat2), sectors, dfat);
            }
            else if ((efat->FatInUse == efat->Fat2) && (efat->Fat1 != 0)) // 2 used, 1 present
            {
               dfsWrite( dfstAreaP2Disk( DFSTORE, lsn + efat->Fat1), sectors, dfat);
            }
            if (++lastCacheStamp == 0)          // when wrapped, reset BOTH caches ages
            {
               efat->CacheA.Stamp = 0;
               efat->CacheB.Stamp = 0;
               lastCacheStamp = 1;              // will be assigned to current one
            }
            cache->Stamp = lastCacheStamp;
            cache->Dirty = FALSE;               // clean again now
         }
         else
         {
            dfsX10("\nFailed to write FAT sectors at LSN : ", lsn + efat->FatInUse, CBR, "\n");
         }
         if ((ULONG *) dfat != cache->Value)   // temp buffer allocated ?
         {
            TxFreeMem(dfat);
         }
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   else
   {
      rc = DFS_BAD_STRUCTURE;
   }
   RETURN (rc);
}                                               // end 'dfsEfatFlushFAT'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Show logical FAT
/*****************************************************************************/
void dfsEfatShowFAT
(
   ULONG               size,                    // IN    nr of entries
   ULONG               start,                   // IN    first cl# to show
   char               *options                  // IN    display options
)
{
   ULONG               i;
   ULONG               value;
   TXTM                tbuf;
   TXLN                text;

   ENTER();

   strcpy( text, "");
   for (i = 0; (i < size) && !TxAbort() && efat->CacheA.Value; i++)
   {
      if ((i % 8) == 0)
      {
         TxPrint( "%s\n%s%08.8X = %s", text, CBZ, start + i, CNN);
         strcpy( text, "");
      }
      value = dfsEfatValue(start + i);
      if ((start + i) < 2)                      // Invalid cluster numbers
      {
         if ((start + i) == 0)
         {
            sprintf( tbuf, " <%sMedia-type%s %2.2x%s>  ",
                     CNC, CBC, (USHORT) (value & 0xff), CNN);
            strcat(  text, tbuf);
         }
      }
      else
      {
         if (value > EFAT_MAXCLUST)
         {
            if (value >= EFAT_MEDIA_CL)
            {
               sprintf( tbuf, " <%seof%s>   ", CNC, CNN);
            }
            else
            {
               sprintf( tbuf, " <%sbad%s>   ", CBR, CNN);
            }
         }
         else
         {
            if (value == 0)
            {
               sprintf( tbuf, " <free>  ");
            }
            else
            {
               sprintf( tbuf, "%8X ", value);
            }
         }
         strcat(  text, tbuf);
      }
   }
   TxPrint("%s\n", text);
   VRETURN();
}                                               // end 'dfsEfatShowFAT'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Count number of used Dir-entries for S_SPACE structure
/*****************************************************************************/
ULONG dfsEfatDirUsed                            // RET   matching entries
(
   ULONG               chunks,                  // IN    nr of space entries
   S_SPACE            *space                    // IN    space allocation
)
{
   ULONG               rc;
   ULONG               used = 0;                // nr of used entries
   ULONG               chunk;                   // index in space-area
   ULONG               sect;                    // sectors to handle
   ULONG               entry;                   // index in fat directory
   USHORT              bps = dfsGetSectorSize();
   USHORT              entries = (bps / sizeof(S_EFATDIR));
   BYTE               *dirbuf;
   S_EFATDIR           *fatdir;                  // Fat directory sector

   ENTER();

   TRACES(("Read %u chunks into %u entries of %u bytes\n",
            chunks, entries, sizeof(S_EFATDIR)));

   if ((dirbuf = TxAlloc( 1, bps)) != NULL)     // allocate one sector
   {
      fatdir = (S_EFATDIR *) dirbuf;

      for (chunk = 0; (chunk < chunks)  && (!TxAbort()); chunk++)
      {
         for (sect = 0; (sect < space[chunk].size)  && (!TxAbort()); sect++)
         {
            if ((rc = dfsRead(space[chunk].start + sect, 1, dirbuf)) == NO_ERROR)
            {
               for (entry = 0; entry < entries; entry++)
               {
                  if (fatdir[entry].EntryType & EFAT_DIRBIT_IN_USE)
                  {
                     used++;
                  }
               }
            }
         }
      }
      TxFreeMem( dirbuf);
   }
   RETURN (used);
}                                               // end 'dfsEfatDirUsed'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get label-string from the EFAT Root-directory
/*****************************************************************************/
BOOL dfsEfatRootLabel                           // RET   Label found
(
   S_BOOTR            *boot,                    // IN    boot record for vol
   ULN64               offset,                  // IN    Offset for boot-rec
   char               *label                    // OUT   label string
)
{
   ULONG               rc = NO_ERROR;
   BOOL                found = FALSE;
   ULONG               size;                    // sectors in Root dir
   ULN64               root;                    // LSN of first root sector
   ULONG               sect;                    // sectors to handle
   ULONG               entry;                   // index in fat directory
   USHORT              bps = dfsGetSectorSize();
   USHORT              entries = (bps / sizeof(S_EFATDIR));
   BYTE               *dirbuf;
   S_EFATDIR          *fatdir;                  // Efat directory sector

   ENTER();

   //- Note: dfsa->boot is NOT valid when calling this (might be NULL!)

   if ((dirbuf = TxAlloc( 1, bps)) != NULL)     // allocate one sector
   {
      fatdir = (S_EFATDIR *) dirbuf;

      size  = (1 << boot->ex.SpcShift);         // min size of Root-Dir, 1 Cl
      root  = (boot->ex.RootCluster -2) * size + boot->ex.ClHeapOffset;

      TRACES(("Root-Dir with %u sectors at LSN %llX, base: %llX\n", size, root, offset));

      for (sect = 0; (sect < size) && (rc == NO_ERROR) && !found; sect++)
      {
         rc = dfsRead(offset + root + sect, 1, dirbuf);
         if (rc == NO_ERROR)
         {
            for (entry = 0; (entry < entries) && !found; entry++)
            {
               if (fatdir[entry].EntryType == EFAT_DIR_VLABEL)
               {
                  TRACES(("Label found in Dir-entry %u\n", entry));

                  TxUnic2Ascii( fatdir[entry].lb.Name, label, fatdir[entry].lb.Length);
                  found = TRUE;
               }
            }
         }
      }
      TxFreeMem( dirbuf);
   }
   RETURN (found);
}                                               // end 'dfsEfatRootLabel'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set label-string in the EFAT Root-directory
/*****************************************************************************/
ULONG dfsEfatSetRootLabel
(
   char               *str                      // IN    label string
)
{
   ULONG               rc = NO_ERROR;
   ULN64               found = 0;               // Directory sector with label
   ULONG               size;                    // sectors in Root dir
   ULN64               root;                    // LSN of first root sector
   ULONG               sect;                    // sectors to handle
   ULONG               entry;                   // index in fat directory
   USHORT              bps = dfsGetSectorSize();
   USHORT              entries = (bps / sizeof(S_EFATDIR));
   BYTE               *dirbuf;
   S_EFATDIR          *fatdir;                  // Efat directory sector
   TXTS                label;

   ENTER();

   //- Note: dfsa->boot IS valid when calling this (FS is initialized)

   if ((dirbuf = TxAlloc( 1, bps)) != NULL)     // allocate one sector
   {
      TxCopy( label, str, EFAT_LABEL_SIZE + 1); // clip/terminate

      fatdir = (S_EFATDIR *) dirbuf;

      size  = efat->Roots;                      // min size of Root-Dir, 1 Cl
      root  = efat->Root;                       // ROOT directory LSN

      TRACES(("Root-Dir with %u sectors at LSN %llX, base: %llX\n", size, root));

      for (sect = 0; (sect < size) && (rc == NO_ERROR) && !found; sect++)
      {
         rc = dfsRead( root + sect, 1, dirbuf);
         if (rc == NO_ERROR)
         {
            for (entry = 0; (entry < entries) && !found; entry++)
            {
               if (fatdir[entry].EntryType == EFAT_DIR_VLABEL)
               {
                  TRACES(("Set label in Existing Dir-entry %u\n", entry));

                  TxAscii2Unic( label, fatdir[entry].lb.Name, strlen( label));
                  fatdir[entry].lb.Length = strlen( label);
                  found = root + sect;
               }
            }
         }
      }
      if (!found)
      {
         for (sect = 0; (sect < size) && (rc == NO_ERROR) && !found; sect++)
         {
            rc = dfsRead( root + sect, 1, dirbuf);
            if (rc == NO_ERROR)
            {
               for (entry = 0; (entry < entries) && !found; entry++)
               {
                  if ((fatdir[entry].EntryType & EFAT_DIRBIT_IN_USE) == 0) // unused or deleted
                  {
                     TRACES(("Set label in Free/Deleted Dir-entry %u\n", entry));

                     TxAscii2Unic( label, fatdir[entry].lb.Name, strlen( label));
                     fatdir[entry].lb.Length = strlen( label);
                     found = root + sect;
                  }
               }
            }
         }
      }
      if (found)                                // found, write updated DIR-sector
      {
         rc = dfsWrite( found, 1, dirbuf);
         if (rc == NO_ERROR)
         {
            TxPrint( "New  Volume label : '%s' written to ROOT directory\n", str);
         }
      }
      else                                      // not found/updated at all
      {
         rc = DFS_CMD_FAILED;
      }
      TxFreeMem( dirbuf);
   }
   RETURN ( rc);
}                                               // end 'dfsEfatSetRootLabel'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get starting cluster for the specified Bitmap file from root directory
/*****************************************************************************/
BOOL dfsEfatRootGetBmCluster                    // RET   Bitmap cluster or 0
(
   S_BOOTR            *boot,                    // IN    boot record for vol
   ULN64               offset,                  // IN    Offset for boot-rec
   BYTE                mapnr,                   // IN    Bitmap 1st is 0
   ULONG              *cluster,                 // OUT   cluster number
   ULN64              *byteSize                 // OUT   size in bytes
)
{
   ULONG               rc = NO_ERROR;
   BOOL                found = FALSE;
   ULONG               size;                    // sectors in Root dir
   ULN64               root;                    // LSN of first root sector
   ULONG               sect;                    // sectors to handle
   ULONG               entry;                   // index in fat directory
   USHORT              bps = dfsGetSectorSize();
   USHORT              entries = (bps / sizeof(S_EFATDIR));
   BYTE               *dirbuf;
   S_EFATDIR          *fatdir;                  // Efat directory sector

   ENTER();

   //- Note: dfsa->boot is NOT valid when calling this (might be NULL!)

   if ((dirbuf = TxAlloc( 1, bps)) != NULL)     // allocate one sector
   {
      fatdir = (S_EFATDIR *) dirbuf;

      size  = (1 << boot->ex.SpcShift);         // min size of Root-Dir, 1 Cl
      root  = ((ULN64) boot->ex.RootCluster -2) * size + boot->ex.ClHeapOffset;

      TRACES(("Root-Dir with %u sectors at LSN 0x%llX, base: 0x%llX\n", size, root, offset));

      for (sect = 0; (sect < size) && (rc == NO_ERROR) && !found; sect++)
      {
         rc = dfsRead(offset + root + sect, 1, dirbuf);
         if (rc == NO_ERROR)
         {
            for (entry = 0; (entry < entries) && !found; entry++)
            {
               switch (fatdir[entry].EntryType)
               {
                  case EFAT_DIR_BITMAP:        // bitmap entry
                     if ((fatdir[entry].bm.Flags & EFAT_BITMAP_MASK) == mapnr)
                     {
                        TRACES(("Bitmap found in Dir-entry %u\n", entry));

                        *cluster  = fatdir[entry].bm.Cluster;
                        *byteSize = fatdir[entry].bm.FileLength;
                        found = TRUE;
                     }
                     break;

                  default:
                     break;
               }
            }
         }
      }
      TxFreeMem( dirbuf);

      TRACES(("BM cluster: 0x%x, Size: 0x%llx = %lld\n", *cluster, *byteSize, *byteSize));
   }
   RETURN (found);
}                                               // end 'dfsEfatRootGetBmCluster'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create S_SPACE structure for an EFAT stream entry, upcase or bitmap file
/*****************************************************************************/
ULONG dfsEfatAllocStream                        // RET   nr of clusters
(
   S_EFATDIR          *entry,                   // Fat dir-entry ptr, STREAM
   ULONG              *errors,                  // INOUT error flags
   ULONG              *chunks,                  // OUT   nr of space entries
   S_SPACE           **space                    // OUT   space allocation
)
{
   ULONG               cl      = 0;             // nr of clusters
   S_SPACE            *sp      = NULL;
   ULONG               nr      = 0;

   ENTER();

   TRACES(("EntryType: %2.2hx\n", entry->EntryType));

   switch (entry->EntryType & ~EFAT_DIRBIT_IN_USE) // normal + deleted
   {
      case EFAT_DEC_BITMAP:
         cl = dfsEfatAllocChain( dfsEfatDir2Clust(entry),
                                 entry->bm.FileLength, errors, chunks, space);
         break;

      case EFAT_DEC_UPCASE:
         cl = dfsEfatAllocChain( dfsEfatDir2Clust(entry),
                                 entry->up.FileLength, errors, chunks, space);
         break;

      case EFAT_ANY_STREAM:
         TRACES(("GenFlags: 0x%4.4hx  Length:%lld\n", entry->st.GenFlags, entry->st.FileLength));
         if ((entry->st.GenFlags & EFAT_CHAIN_INVALID) == 0)
         {
            cl = dfsEfatAllocChain( dfsEfatDir2Clust(entry),
                                    entry->st.FileLength, errors, chunks, space);
         }
         else
         {
            if (entry->st.FileLength > 0)
            {
               nr = 1;                                             //- nr of chunks
               if ((sp = (S_SPACE *) calloc((size_t) nr, sizeof(S_SPACE))) != NULL)
               {
                  ULONG      clustBytes = efat->ClustSize * dfsGetSectorSize();

                  cl = ((entry->st.FileLength - 1) / clustBytes) + 1; //- number of clusters

                  sp->start = dfsEfatClust2LSN( entry->st.Cluster);   //- LSN start of 1st cluster
                  sp->size  = cl * efat->ClustSize;                   //- size of chunk in sectors

                  TRACES(("cl:%u  start:0x%llx  size:0x%llx\n", cl, sp->start, sp->size));
               }
            }
            *space  = sp;
            *chunks = nr;
         }
         break;

      default:                                  // others are invalid
         TRACES(("Invalid EntryType!"));
         break;
   }
   RETURN (cl);
}                                               // end 'dfsEfatAllocStream'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create S_SPACE structure for a FAT allocation chain
// Note: Area aware to allow usage from FDISK mode too
/*****************************************************************************/
ULONG dfsEfatAllocChain                         // RET   nr of clusters
(
   ULONG               cluster,                 // IN    starting cluster nr
   ULONG               maxsize,                 // IN    max file-size, or 0
   ULONG              *errors,                  // INOUT error flags
   ULONG              *chunks,                  // OUT   nr of space entries
   S_SPACE           **space                    // OUT   space allocation
)
{
   ULONG               cl;                      // nr of clusters
   ULONG               this    = cluster;       // current cluster
   ULONG               last    = 0;             // last handled cluster
   ULONG               size    = 0;             // sectors in this chunk
   ULONG               extents = 0;             // nr of extents
   ULONG               areas   = 0;             // nr of space-areas
   ULONG               climit  = efat->MapClust; // limit on nr of clusters
   S_SPACE            *sp      = NULL;

   ENTER();

   if (maxsize != 0)                            // explicit area-size given
   {                                            // limit == clusters +1 !!!
      climit = (maxsize / (efat->ClustSize * dfsGetSectorSize())) +2;
   }
   TRARGS(("Input cluster: %8.8X, maxsize: %8.8X  Cluster limit: %8.8x  MapClust: %8.8x\n",
                  cluster,        maxsize,               climit,  efat->MapClust));

   //- Return 0 allocated extnets for cluster-value 0!
   for (cl = 1; (cl <= climit) && !TxAbort() && (cl != 0) && (cluster != 0); cl++)
   {
      TRACES(( "IN-FOR cl: %u this:%8.8x last:%8.8x extents: %u size:%u\n",
                       cl,    this,      last,      extents,    size));
      if (this == (last +1))                    // cluster in sequence ?
      {
         size += efat->ClustSize;
      }
      else                                      // create new chunk
      {
         if (areas <= extents)                  // areas used up ?
         {
            areas = areas * 8 + 16;
            TRACES(("(re)Allocate %u space chunks, rel cl %u\n", areas, cl))
            sp = (S_SPACE *) realloc( sp, (size_t) (areas+1) * sizeof(S_SPACE));
         }
         if (sp != NULL)
         {
            if (last != 0)                      // not first extent ?
            {
               sp[extents].size = size;         // set size of last extent
               extents++;                       // and move to next one
            }
            size = efat->ClustSize;              // start with 1 cluster
            sp[extents].size  = size;
            sp[extents].start = dfstAreaP2Disk( DFSTORE, dfsEfatClust2LSN(this));
         }
         else
         {
            cl = 0;
         }
      }
      if (this == EFAT_EOFCLUST)               // EOF value
      {
         break;                                 // leave
      }
      else                                      // follow the chain
      {
         last = this;
         if ((last >= 2) && (last <= efat->MapClust)) // OK to follow ?
         {
            this = dfsEfatValue(last);
            if ((this <= 1) || (last <= 1))     // free cluster, error
            {
               if (errors)
               {
                  *errors |= EF_ALLOC_CL_FREE;
               }
               break;
            }
            else if (this == EFAT_BADCLUST)
            {
               if (errors)
               {
                  *errors |= EF_ALLOC_CL_BADS;
               }
               break;
            }
         }
         else
         {
            TRACES(("Invalid cluster-value: %8.8x\n", last));
            if (errors)
            {
               if (last == EFAT_BADCLUST)
               {
                  *errors |= EF_ALLOC_CL_BADS;
               }
               else
               {
                  *errors |= EF_CLUSTER_RANGE;
               }
            }
            break;
         }
      }
      if (cl == climit)                         // maximum clusters reached
      {
         if (errors)
         {
            *errors |= EF_RUNAWAY_CHAIN;
         }
      }
   }
   TRACES(( "ENDFOR cl: %u this:%8.8x last:%8.8x extents: %u size:%u\n",
                    cl,    this,      last,      extents,    size));
   if (sp != NULL)
   {
      TRACES(( "sp[extents -1].start: 0x%llx        .size: %llu\n",
                sp[extents -1].start, sp[extents -1].size));

      *space  = sp;
      *chunks = extents;
      TRINIT(70);
         dfsSspaceDisplay( SD_DEFAULT, extents, sp);
      TREXIT();
   }
   else
   {
      *space  = NULL;                           // dont return random value!
      *chunks = 0;
      cl = 1;                                   // signal (alloc) failure
   }
   if (errors)
   {
      TRACES(("error flags: 0x%8.8x\n", *errors));
   }
   RETURN (cl -1);
}                                               // end 'dfsEfatAllocChain'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Check if name is a valid EFAT filename
/*****************************************************************************/
BOOL dfsValidEfatName
(
   const void         *fn,                      // IN    ptr to EFAT name
   USHORT              len                      // IN    length to check
)
{
   TXLN                name;
   USHORT              lok;

   memset(  name, 0,  TXMAXTM);
   strncpy( name, fn, len);

   lok = strcspn( name, DFS_EFAT_INVALIDS);

   TRACES(("FatName: '%*.*s', len: %u, strlen: %u,  length-ok: %u\n",
            len, len, name, len, strlen(name), lok));
   return (lok == len);
}                                               // end 'dfsValidEfatName'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Convert null-terminated name to a valid dir-entry name (multiple C1 entries)
/*****************************************************************************/
void dfsEfatName2Entry
(
   char               *fn,                      // IN    ptr to FILE name
   S_EFATDIR          *de                       // OUT   File, Stream + Name entries
)
{
   #if defined (NEVER)
   TXLN                local;
   char               *p;

   ENTER();
   strcpy( local, fn);

   //- to be refined, distribute given name over multiple C1 entries in Unicode
   //- must test for valid C0 (length byte) and C1 entries!
   TxStrToUpper( local);
   memset( de->Name, ' ', FAT_NSIZE + FAT_ESIZE);
   if ((p = strrchr(local, '.')) != NULL)       // extention present ?
   {
      *p++ = '\0';                              // terminate base-name
      memcpy(de->Ext, p, strlen(p));            // copy ext
   }
   memcpy( de->Name, local, strlen(local));     // copy base

   TRACES(("Converted '%s' to :\n", fn));
   TRHEXS(70, de, 64, "S_EFATDIR directory entry");
   VRETURN();
   #endif
}                                               // end 'dfsEfatName2Entry'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// Convert a valid dir-entry name (multiple C1) to valid null-terminated name
// Asumes there are 2 more sectors of DIR entries BEFORE and AFTER specified one
/*****************************************************************************/
void dfsEfatEntry2Name
(
   S_EFATDIR          *efile,                   // IN    File, Stream + Name entries
   char               *lfn                      // OUT   ptr to long FILE name
)
{
   S_EFATDIR          *estream;                 // Stream entry
   S_EFATDIR          *ename;                   // First name entry
   BYTE                pendingName = 0;
   USHORT              fragmentSize;            // Filename fragment size
   int                 sanity;                  // Max entries to move forward/backward

   ENTER();

   //- to be refined, may add more consistency checks, and perhaps 'ef' error flag
   switch (efile->EntryType & ~EFAT_DIRBIT_IN_USE) // normal + deleted
   {
      case EFAT_DEC_VLABEL:
         TxUnic2Ascii( efile->lb.Name, lfn, EFAT_LABEL_SIZE);
         break;

      case EFAT_DEC_BITMAP:
         strcpy( lfn, "EFAT_alloc-bitmap.bin");
         break;

      case EFAT_DEC_UPCASE:
         strcpy( lfn, "EFAT_upcase-table.bin");
         break;

      case EFAT_ANY_STREAM:
      case EFAT_ANY_FILENAME:                  // need to search backwards to FILE entry
         sanity = 2 * dfsGetSectorSize() / sizeof(S_EFATDIR);
         while (((efile->EntryType & ~EFAT_DIRBIT_IN_USE) != EFAT_DEC_FILE) && (sanity--))
         {
            efile--;                            // move backwards over 1 entry
         }
         if ((efile->EntryType & ~EFAT_DIRBIT_IN_USE) != EFAT_DEC_FILE)
         {
            strcpy( lfn, "-no-file-");          // no FILE found, invalid
            break;
         }
      case EFAT_DEC_FILE:
         strcpy( lfn, "");                      // start with empty name
         estream = efile   + 1;                 // Stream entry
         ename   = estream + 1;                 // First name entry

         pendingName = estream->st.NameLength;

         sanity = 2 * dfsGetSectorSize() / sizeof(S_EFATDIR);
         TRACES(("Estream name length: %hu, sanity:%d\n", pendingName, sanity));
         while ((pendingName > 0) && (sanity--))
         {
            TRACES(("Remaining name length: %hu, entry-type: 0x%2.2hx, sanity left:%d\n",
                     pendingName, ename->EntryType, sanity));
            if ((ename->EntryType & ~EFAT_DIRBIT_IN_USE) == EFAT_ANY_FILENAME)
            {
               if (pendingName > EFAT_NAME_SIZE)
               {
                  fragmentSize = EFAT_NAME_SIZE;
                  pendingName -= EFAT_NAME_SIZE;
               }
               else                             // no more name fragments expected
               {
                  fragmentSize = pendingName;
                  pendingName  = 0;
               }
               TxUnicAppend( ename->nm.Name, lfn, fragmentSize);
            }
            else
            {
               //- may report some warning/error here
            }
            ename++;                            // to next dir name entry
         }
         break;

      default:                                  // others invalid, incl stream/name
         strcpy( lfn, "-invalid-");
         break;
   }
   TRHEXS(70, efile, 128, "S_EFATDIR entry FILE+STREAM+NAME(s)");
   TRACES(("Converted to : '%s'\n", lfn));
   VRETURN();
}                                               // end 'dfsEfatEntry2Name'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display all available info about a directory entry, including C0/C1 entries
/*****************************************************************************/
ULONG dfsEfatShowDirectoryEntry
(
   ULN64               dirsect,                 // IN    lsn of FAT dir sector
   int                 entry                    // IN    File, Stream + Name entries
)
{
   ULONG               rc;
   BYTE               *sb  = NULL;              // sector buffer
   S_EFATDIR          *efile;                   // DIR entry, FILE
   S_EFATDIR          *stream;                  // DIR entry, STREAM
   TXLN                lfn;

   ENTER();
   TRACES(("dirsect: 0x%llx, entry: 0x%2.2x\n", dirsect, entry));

   strcpy( lfn, "");                            // init first to not-present

   //- Need to allocate/read upto 5 sectors to allow reading LFN up to 255 characters
   if ((sb = TxAlloc(5, dfsGetSectorSize())) != NULL) // 2 extra sectors BEFORE and AFTER
   {
      if ((rc = dfsRead(dirsect - 2, 5, sb)) == NO_ERROR)
      {
         BYTE      *dirdata = sb + (2 * dfsGetSectorSize());
         int        sanity;

         switch (dfsIdentifySector(dirsect, 0, dirdata))
         {
            case ST_ROOTD:
            case ST_DIREC:
               efile = &(((S_EFATDIR *) dirdata)[entry]);

               TxPrint("\n");
               dfsEfatDirHeader( "DirSector+entry ", 0); // display TOP header
               dfsX10(" ", dirsect, CBC, " e^ ");
               TxPrint("%s%2x%s",  CNC, entry, CNN);

               switch (efile->EntryType & ~EFAT_DIRBIT_IN_USE) // normal + deleted
               {
                  case EFAT_DEC_VLABEL:
                     TxUnic2Ascii( efile->lb.Name, lfn, EFAT_LABEL_SIZE);
                     TxPrint( "%34.34s Label: %s\n", "", lfn);
                     break;

                  case EFAT_DEC_BITMAP:
                     dfsUllDot20( "                               alloc-bitmap ", efile->bm.FileLength, " ");
                     TxPrint( "%s%8X%s\n", CNN, efile->bm.Cluster, CGE);

                     if (efile->bm.FileLength != 0)
                     {
                        dfsEfatShowAlloc( efile);
                     }
                     break;

                  case EFAT_DEC_UPCASE:
                     dfsUllDot20( "                               upcase-table ", efile->up.FileLength, " ");
                     TxPrint( "%s%8X%s\n", CNN, efile->up.Cluster, CGE);

                     if (efile->up.FileLength != 0)
                     {
                        dfsEfatShowAlloc( efile);
                     }
                     break;

                  case EFAT_ANY_STREAM:
                  case EFAT_ANY_FILENAME:      // need to search backwards to FILE entry
                     sanity = 2 * dfsGetSectorSize() / sizeof(S_EFATDIR);
                     while (((efile->EntryType & ~EFAT_DIRBIT_IN_USE) != EFAT_DEC_FILE) && (sanity--))
                     {
                        efile--;                // move backwards over 1 entry
                     }
                     if ((efile->EntryType & ~EFAT_DIRBIT_IN_USE) != EFAT_DEC_FILE)
                     {
                        break;                  // no FILE entry found, fail
                     }
                  case EFAT_DEC_FILE:
                     stream = efile +1;        // advance to actual stream entry

                     dfsEfatShowDirEntry( efile, stream, NORMAL);
                     dfsEfatEntry2Name( efile, lfn);
                     TxPrint("%s%s%s\n\n", CBY, lfn, CNN);

                     if ((stream->st.FileLength != 0) || (efile->fl.Attrib & FATTR_DIRECTORY))
                     {
                        dfsEfatShowAlloc( stream);
                     }
                     break;

                  default:                      // ignore others
                     break;
               }
               break;

            default:
               rc = DFS_PENDING;
               break;
         }
      }
      else if (dirsect < efat->Sect)             // should be there
      {
         dfsX10("\nError reading sector : ", dirsect, CBR, "\n");
      }
      TxFreeMem(sb);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN( rc);
}                                               // end 'dfsEfatShowDirectoryEntry'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// FAT filesystem, display DIR header/footer
/*****************************************************************************/
void dfsEfatDirHeader
(
   char               *lead,                    // IN    lead text-string
   ULONG               items                    // IN    DIR-lines shown (0 = TOP)
)
{
   TXLN                line;


   sprintf(line, " ================ ====================== ======================== =============== ========\n");
   if (items > 0)
   {
      TxPrint( line);
   }
   TxPrint( " %s Modified/Creation-Time Attrib/AccessTm/Filename        Filesize  Cluster\n", lead);
   if (items == 0)
   {
      TxPrint( line);
   }
}                                               // end 'dfsEfatDirHeader'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display EFAT directory entry on two lines, all except filename
/*****************************************************************************/
void dfsEfatShowDirEntry
(
   S_EFATDIR          *entry,                   // IN    EFAT FILE entry
   S_EFATDIR          *stream,                  // IN    EFAT STREAM entry
   int                 ac                       // IN    active color fg+bg
)
{
   ULONG               cluster = 0;
   TXTM                tbuf;
   TXLN                text;
   signed char         tzOffset;

   sprintf( text, " %4.4u-%2.2u-%2.2u %2.2u:%2.2u:%2.2u.%2.2u",
      entry->fl.DtModify.d.year + 1980, entry->fl.DtModify.d.month,   entry->fl.DtModify.d.day,
      entry->fl.TmModify.t.hours, entry->fl.TmModify.t.minutes, entry->fl.TmModify.t.twosecs * 2 +
                                  entry->fl.MsModify / 100,     entry->fl.MsModify % 100);

   if ((entry->fl.DtAccess.u != entry->fl.DtModify.u) ||
       (entry->fl.TmAccess.u != entry->fl.TmModify.u)  )
   {
      sprintf( tbuf, " %4.4u-%2.2u-%2.2u %2.2u:%2.2u:%2.2u",
         entry->fl.DtAccess.d.year + 1980, entry->fl.DtAccess.d.month, entry->fl.DtAccess.d.day,
         entry->fl.TmAccess.t.hours, entry->fl.TmAccess.t.minutes, entry->fl.TmAccess.t.twosecs * 2);
   }
   else
   {
      sprintf( tbuf, "%20.20s", "");
   }
   strcat( text, tbuf);

   dfstrUllDot20( text, " ", stream->st.FileLength, "");

   cluster = dfsEfatDir2Clust( stream);
   if (cluster != 0)
   {
      sprintf( tbuf, " %8X", cluster);          // can be 32-bit
      strcat( text, tbuf);
   }
   else                                         // must be an empty file/dir
   {
      strcat( text, " Empty 0");
   }
   tzOffset  =  (entry->fl.TzCreate & 0x7f) | ((entry->fl.TzCreate & 0x40) << 1); // sign-extend 7-bit to 8
   TxPrint( "%s%s\n   %s%s%s utc %+2.2hd hr", text, CGE, (ac == NORMAL) ? CNN : ansi[Ccol((CcM | CcI), Ccbg(ac))],
            (entry->EntryType & EFAT_DIRBIT_IN_USE) ? "   " : "del", ansi[ ac], tzOffset / 4);

   if ((entry->fl.DtCreate.u != entry->fl.DtModify.u  ) ||
       (entry->fl.TmCreate.u != entry->fl.TmModify.u  ) ||
       (entry->fl.MsCreate   != entry->fl.MsModify    )  )
   {
      sprintf( text, " %4.4u-%2.2u-%2.2u %2.2u:%2.2u:%2.2u.%2.2u ",
         entry->fl.DtCreate.d.year + 1980, entry->fl.DtCreate.d.month, entry->fl.DtCreate.d.day,
         entry->fl.TmCreate.t.hours, entry->fl.TmCreate.t.minutes, entry->fl.TmCreate.t.twosecs * 2 +
                                     entry->fl.MsCreate / 100,     entry->fl.MsCreate % 100);
   }
   else
   {
      sprintf( text, "%24.24s", "");
   }
   dfstrFatAttrib( text, entry->fl.Attrib);

   TxPrint( "%s ", text);                       // will be followed by long filename ...

}                                               // end 'dfsEfatShowDirEntry'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display FAT allocation chain from cluster
/*****************************************************************************/
void dfsEfatShowAlloc
(
   S_EFATDIR          *entry                    // IN    STREAM dir entry
)
{
   ULONG               rc = NO_ERROR;
   DFSISPACE           ispace;
   ULONG               ef = 0;                  // Initialize error flag!

   ENTER();
   TRARGS(("entryType: 0x%2.2hx  cl  %8.8x\n", entry->EntryType, entry->st.Cluster));

   if (dfsEfatDirEntry2Space( entry, &ispace, &ef) == NO_ERROR)
   {
      ULONG      sdo = SD_TOPLINE | SD_SUMMARY;
      ULN64      failedSectors = 0;

      dfsX10("Space for cluster : ", entry->st.Cluster, CBM, "  ");
      TxPrint( "(%s)\n", (entry->st.GenFlags & EFAT_CHAIN_INVALID) ? "Contiguous" : "FAT-chain");

      TxPrint( " Allocation Check : %s", TREOLN);
      if (dfsCheckAllocForSpace( &ispace, 0,
              ((entry->EntryType & EFAT_DIRBIT_IN_USE) != 0), &failedSectors) == NO_ERROR)
      {
         TxPrint("%sOK%s%s\n", CBG, CNN,
                 "         Saveto/Recover/Undelete of filedata is possible");
      }
      else
      {
         TxPrint("%sFAIL%s on %u out of %u Sectors\n", CBR, CNN, failedSectors,
                           dfsSspaceSectors( TRUE, ispace.chunks, ispace.space));

      }
      if (!TxaOptUnSet('X'))                    // unless -X- specified
      {
         sdo |= SD_NAVDOWN;                     // next is data sector

         //- Also fill the single directory-entry cache (next action probably display of that)
         efat->CachedDirEntry = *entry;
      }
      rc = dfsSspaceDisplay( sdo, ispace.chunks, ispace.space);
      free( ispace.space);

      if (ef != 0)
      {
         dfsEfatDispError( ef, 0, "\nWarning: ", NULL);
      }
   }
   VRETURN();
}                                               // end 'dfsEfatShowAlloc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Determine ExFAT alloc from entry/contiguous-clusters or chain, when valid
/*****************************************************************************/
ULONG dfsEfatDirEntry2Space                     // RET   result
(
   S_EFATDIR          *entry,                   // IN    STREAM dir entry
   DFSISPACE          *isp,                     // OUT   SPACE alloc info
   ULONG              *eFlags                   // OUT   error flags, or NULL
)
{
   ULONG               rc = NO_ERROR;

   ENTER();
   TRARGS(("entryType: 0x%2.2hx  cl  %8.8x\n", entry->EntryType, entry->st.Cluster));

   switch (entry->EntryType & ~EFAT_DIRBIT_IN_USE)
   {
      case EFAT_DEC_BITMAP:
         dfsEfatAllocChain( entry->bm.Cluster, 0, eFlags, &isp->chunks, &isp->space);
         break;

      case EFAT_DEC_UPCASE:
         dfsEfatAllocChain( entry->up.Cluster, 0, eFlags, &isp->chunks, &isp->space);
         break;

      case EFAT_ANY_STREAM:
         if (entry->st.GenFlags & EFAT_CHAIN_INVALID) // allocation is contiguous
         {
            isp->chunks = 1;                  // return contiguous clusters in ONE chunk
            if ((isp->space = (S_SPACE *) calloc((size_t) isp->chunks, sizeof(S_SPACE))) != NULL)
            {
               ULONG   clustBytes = efat->ClustSize * dfsGetSectorSize();

               isp->space->start = dfsEfatClust2LSN( entry->st.Cluster);      //- LSN start of 1st cluster
               isp->space->size  = ((entry->st.FileLength + clustBytes - 1)   //- size of chunk in sectors
                                          / clustBytes) * efat->ClustSize;
            }
            else
            {
               rc = DFS_ALLOC_ERROR;
            }
         }
         else                                   // valid FAT chain (fragmented)
         {
            dfsEfatAllocChain( entry->st.Cluster, 0, eFlags, &isp->chunks, &isp->space);
         }
         isp->clsize = efat->ClustSize;
         break;

      default:
         rc = DFS_ST_MISMATCH;
         break;
   }
   RETURN( rc);
}                                               // end 'dfsEfatDirEntry2Space'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Retrieve first-cluster number for specified Dir-sector-LSN + index (LsnInf)
/*****************************************************************************/
ULONG dfsEfatLsnInf2Cluster                         // RET   First cluster or 0
(
   ULN64               lsn,                     // IN    Directory LSN
   ULONG               info                     // IN    directory entry
)
{
   ULONG               rc  = 0;
   S_EFATDIR           direntry;                // Fat directory entry

   ENTER();

   if (dfsEfatLsnInf2Entry( lsn, info, &direntry) == NO_ERROR)
   {
      rc = dfsEfatDir2Clust( &direntry);
   }
   RETURN (rc);
}                                               // end 'dfsEfatLsnInf2Cluster'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Retrieve directory-entry for specified Dir-sector-LSN + index (LsnInf)
/*****************************************************************************/
ULONG dfsEfatLsnInf2Entry
(
   ULN64               lsn,                     // IN    Directory LSN
   ULONG               info,                    // IN    directory entry
   S_EFATDIR          *direntry                 // OUT   directory entry
)
{
   ULONG               rc  = NO_ERROR;
   USHORT              bps = dfsGetSectorSize();
   BYTE               *dirbuf;
   S_EFATDIR          *fatdir;                  // Fat directory sector

   ENTER();
   TRARGS(("Input LSN: 0x%llX, entry:%u\n", lsn, DFSSNIGET( info)));

   if ((dirbuf = TxAlloc( 1, bps)) != NULL)     // allocate one sector
   {
      fatdir = (S_EFATDIR *) dirbuf;

      rc = dfsRead( lsn, 1, dirbuf);
      if (rc == NO_ERROR)
      {
         *direntry = fatdir[ DFSSNIGET( info)];
      }
      TxFreeMem( dirbuf);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsEfatLsnInf2Entry'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Retrieve STREAM directory-entry for specified Dir-sector-LSN + index (LsnInf)
// Input entry (lsn+inf) should point to a FILE, STREAM, BITMAP or UPCASE entry
/*****************************************************************************/
ULONG dfsEfatLsnInf2Stream
(
   ULN64               lsn,                     // IN    Directory LSN
   ULONG               info,                    // IN    directory entry
   S_EFATDIR          *direntry                 // OUT   directory entry (STREAM)
)
{
   ULONG               rc  = NO_ERROR;
   USHORT              bps = dfsGetSectorSize();
   BYTE               *dirbuf;
   S_EFATDIR          *fatdir;                  // Fat directory sector
   S_EFATDIR          *stream;                  // pointer to STREAM entry

   ENTER();
   TRARGS(("Input LSN: 0x%llX, entry:%u\n", lsn, DFSSNIGET( info)));

   if ((dirbuf = TxAlloc( 2, bps)) != NULL)     // allocate two sectors
   {
      fatdir = (S_EFATDIR *) dirbuf;

      rc = dfsRead( lsn, 2, dirbuf);
      if (rc == NO_ERROR)
      {
         stream = &fatdir[ DFSSNIGET( info)];
         if ((stream->EntryType & ~EFAT_DIRBIT_IN_USE) == EFAT_DEC_FILE)
         {
            stream++;                           // advance to stream entry
         }
         *direntry = *stream;                   // copy STREAM entry contents
      }
      TxFreeMem( dirbuf);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsEfatLsnInf2Stream'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create SPACE allocation structure from a Directory LSN (root or other)
/*****************************************************************************/
ULONG dfsEfatDirLsn2Space
(
   ULN64               lsn,                     // IN    lsn for DIR sector
   ULONG              *nr,                      // OUT   nr of space entries
   S_SPACE           **sp                       // OUT   space allocation
)
{
   ULONG               rc  = NO_ERROR;
   ULONG               ef = 0;                  // Initialize error flag!
   DFSISPACE           ispace;                  // SPACE alloc info
   ULONG               cluster = dfsEfatLSN2Clust(lsn);

   ENTER();
   TRARGS(("Input LSN: 0x%llX == cluster:0x%x\n", lsn));

   ispace.space  = NULL;                        // set safe defaults
   ispace.chunks = 0;

   //- try to resolve using last cached DIR-entry first, to get the 'invalid-chain' bit
   if (((efat->CachedDirEntry.EntryType & EFAT_ANY_STREAM) == EFAT_ANY_STREAM) &&
        (efat->CachedDirEntry.st.Cluster == cluster)) // DIR stream-entry for this cluster
   {
      TRACES(("Resolve SPACE from cached directoryEntry\n"));
      rc = dfsEfatDirEntry2Space( &efat->CachedDirEntry, &ispace, &ef);
   }
   else                                         // just try the plain cluster-chain
   {
      //- Using the alloc-chain would be unreliable for all but the ROOT directory

      TRACES(("Force Single-cluster SPACE allocation\n"));
      ef = EF_ALLOC_CL_FREE;                    // force single-cluster SPACE allocation
   }
   if (ef != 0)                                 // FAT area damaged, or forced single ...
   {
      if ((ef & EF_ALLOC_CL_FREE) == 0)         // don't report ends-in-free/forced
      {
         dfsEfatDispError( ef, 0, "\nWarning: ", NULL);
      }
      ispace.chunks = 1;                        // just return first cluster
      if ((ispace.space = (S_SPACE *) calloc( 1, sizeof(S_SPACE))) != NULL)
      {
         ispace.space->start = lsn;             // start of cluster
         ispace.space->size  = efat->ClustSize; // size of chunk
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   *sp = ispace.space;
   *nr = ispace.chunks;
   RETURN (rc);
}                                               // end 'dfsEfatDirLsn2Space'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Iterate over directory-structures finding files, and execute callback
/*****************************************************************************/
ULONG dfsEfatIterator
(
   ULN64               recurse,                 // IN    recurse subtrees
   ULN64               type,                    // IN    do 'D', 'f' or both
   char               *path,                    // IN    path spec, info only
   void               *p                        // IN    iterate parameters
)
{
   ULONG               rc  = NO_ERROR;
   DFS_PARAMS         *ip  = (DFS_PARAMS *) p;  // Iterator parameters
   S_EFATDIR           de;                      // directory entry
   ULONG               nr;                      // Alloc chunks
   S_SPACE            *sp  = NULL;              // Alloc chunk array

   ENTER();

   //- Specified path translated to an LSN and optional entry-nr in params
   TRACES(("Lsn: 0x%llx  Entry: %u\n", ip->Lsn, ip->Number));

   memset( &de, 0, sizeof( S_EFATDIR));         // default to CL 0 ==> Root

   if (ip->Number & DFSSNINFO)                  // reference to a DIR entry
   {
      dfsEfatLsnInf2Entry( ip->Lsn, ip->Number, &de);
   }
   if (dfsEfatDirLsn2Space(dfsEfatClust2LSN(dfsEfatDir2Clust(&de)), &nr, &sp) == NO_ERROR)
   {
      iterationsDone = 0;                      // initialize static progress
      rc = dfsEfatIterFiles( recurse, type, nr, sp, ip);
      free( sp);
   }
   RETURN (rc);
}                                               // end 'dfsEfatIterator'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create valid FAT(32) boot record based on efat-> info from FatInit
/*****************************************************************************/
ULONG dfsEfatMkBootRec
(
   BOOL                verbose,                 // IN    verbose and confirm
   ULONG               serialNr                 // IN    Vol-serialNr or 0
)
{
   ULONG               rc  = DFS_NO_CHANGE;
   #if defined (NEVER)
   ULN64               partSectors;
   S_BOOTR            *brt;
   TXTS                string;
   TXTS                osname;
   ULONG               serial = serialNr;
   #endif

   ENTER();

   TxPrint( "\nFixboot by creating from scratch is NOT IMPLEMENTED for EFAT yet.\n");

   #if defined (NEVER)
   if ((SINF->p != NULL) || ((efat->Fat1 != 0) && (efat->Fat2 > efat->Fat1)))
   {
      if (serial == 0)                          // create default
      {
         serial = dfstLSN2Psn( DFSTORE, 0) + 0xee000000;
      }
      if ((brt = TxAlloc(1, dfsGetSectorSize())) != NULL)
      {
         // dfsMakeFatOsString(   os, osname);
         // dfsInsertFatBootCode( os, brt);

         switch (dfstStoreType( DFSTORE))
         {
            case DFST_PHDISK:                   // possibly partitioned
            case DFST_MEMORY:
               if (SINF->p != NULL)             // partition info available
               {
                  partSectors             = SINF->p->sectors;
                  brt->ex.PartitionOffset = SINF->p->basePsn;
               }
               else                             // large floppy
               {
                  partSectors             = efat->Sect;
                  brt->ex.PartitionOffset = 0;
               }
               break;

            default:
               if (verbose)
               {
                  TxPrint( "\n%sNo partition-table info%s is available, this will "
                           "make the 'PartitionOffset'\nfield incorrect when this "
                           "is a primary partition.\n", CBR, CNN);
               }
               partSectors             = efat->Sect;
               brt->ex.PartitionOffset = 0;
               break;
         }
         //- Fill in the specific efat or generic os structures ...
         //- to be refined

         brt->ex.NrOfFats      = 1;
         brt->ex.VolumeLength  = partSectors;

         brt->ex.RootCluster   = dfsEfatLSN2Clust( efat->Root);
         brt->ex.FsRevMinor    = 0x00;
         brt->ex.FsRevMajor    = 0x10;
         brt->ex.BiosDrive     = 0x80;
         brt->ex.SerialNr      = serial;
         memcpy( brt->OpSys,    "EFAT   ", 8);

         if (verbose)
         {
            TxPrint( "\nThe following new EFAT bootsector has been prepared:\n\n");
            dfsDisplaySector( dfstLSN2Psn( DFSTORE, LSN_BOOTR), ST_BOOTR, (BYTE   *) brt);
         }
         //- to be refined, different help item
         if ((dfsa->batch) || (verbose == FALSE) ||
             (TxConfirm( 5129, "A new EFAT bootsector has been prepared with "
                               "bootcode for OS:\n\n      %s\n\n"
                               "Do you want to replace the current "
                               "filesystem bootsector (sector 0)\n"
                               "by the newly created one ? [Y/N] : ", osname)))
         {
            if (DFSTORE_WRITE_ALLOWED)
            {
               rc = dfsWrite( LSN_BOOTR, 1, (BYTE   *) brt);
               if ((rc == NO_ERROR) && verbose)
               {
                  TxPrint("\nEFAT boot record successfully updated\n");
               }
            }
            else
            {
               rc = DFS_READ_ONLY;
            }
         }
         TxFreeMem( brt);
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   else
   {
      TxPrint("No partition info and no FAT positions available, "
              "create not possible\n");
   }
   #endif
   RETURN(rc);
}                                               // end 'dfsEfatMkBootRec'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get/Set DIRTY bit in EFAT bootsector to specified value
/*****************************************************************************/
BOOL dfsEfatDirtyStatusBootrec                  // RET   final dirty status
(                                               //
   BOOL                update,                  // IN    update value
   BOOL                dirty                    // IN    new value
)
{
   BOOL                rc = FALSE;
   S_BOOTR            *brt;
   USHORT             *FsysValue;

   ENTER();

   if ((brt = TxAlloc(1, dfsGetSectorSize())) != NULL)
   {
      if ((rc = dfsRead( LSN_BOOTR, 1, (BYTE   *) brt)) == NO_ERROR)
      {
         FsysValue = &brt->ex.VolumeFlags;
         if (update)
         {
            if (DFSTORE_WRITE_ALLOWED)
            {
               if (dirty)
               {
                  *FsysValue |=  EFAT_VOL_DIRTY;
               }
               else
               {
                  *FsysValue &= ~EFAT_VOL_DIRTY;
               }
               if ((dfsWrite( LSN_BOOTR, 1, (BYTE   *) brt)) == NO_ERROR)
               {
                  TxPrint("\nEFAT boot record successfully updated with new DIRTY status\n");

                  dfsEfatCrcSyncBootAreas( LSN_BOOTR, 0); // sync to BOTH
               }
               else
               {
                  TxPrint("\nEFAT boot record update for dirty-flag has failed!\n");
               }
            }
            else
            {
               rc = DFS_READ_ONLY;
            }
         }
         rc = (BOOL) ((*FsysValue & EFAT_VOL_DIRTY) ? TRUE : FALSE);
      }
      else
      {
         TxPrint("\nEFAT bootrecord read error!\n");
      }
      TxFreeMem( brt);
   }
   BRETURN(rc);
}                                               // end 'dfsEfatDirtyStatusBootrec'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Recursively iterate over FAT directory and files, execute callback
/*****************************************************************************/
static ULONG dfsEfatIterFiles
(
   ULONG               rs,                      // IN    recurse subtrees
   ULONG               type,                    // IN    do 'D', 'f' or both
   ULONG               chunks,                  // IN    nr of S_SPACE chunks
   S_SPACE            *space,                   // IN    space structure array
   DFS_PARAMS         *cp                       // IN    callback params
)
{
   ULONG               rc = NO_ERROR;
   ULONG               chunk;                   // index in space-area
   ULONG               sect;                    // sectors to handle
   USHORT              entry;                   // index in fat directory
   ULN64               dlsn;                    // Directory sector LSN
   ULONG               ef = 0;                  // Initialize error flag!
   ULONG               nr;                      // Alloc chunks
   S_SPACE            *sp = NULL;               // Alloc chunk array
   BYTE                id  = ST_UDATA;          // type of sector
   USHORT              bps = dfsGetSectorSize();
   USHORT              entries = (bps / sizeof(S_EFATDIR));
   BYTE               *dirbuf;
   S_EFATDIR          *efile;                   // DIR entry, FILE
   S_EFATDIR          *stream;                  // DIR entry, STREAM
   BOOL                isDir;

   ENTER();
   TRARGS(("Recursion: %d, chunks: %u\n", rs, chunks));

   dirbuf = TxAlloc( 3, bps);                   // allocate three sectors
   if ((dirbuf != NULL) && (space != NULL))
   {
      for ( chunk = 0;
           (chunk < chunks) && (rc == NO_ERROR) && !TxAbort();
            chunk++)                            // walk all alloc chunks
      {
         TRACES(( "chunk size: 0x%llx  start lsn: 0x%llx\n", space[chunk].size, space[chunk].start));
         for ( sect = 0;
              (sect < space[chunk].size) && (rc == NO_ERROR) && !TxAbort();
               sect++)                          // each sector in chunk
         {
            if (cp->Flag == TRUE)               // progress in sectors
            {
               iterationsDone++;
            }
            dlsn = space[chunk].start + sect;
            TRACES(( "Read directory sector at: 0x%llx\n", dlsn));

            rc = dfsRead(dlsn, 3, dirbuf);

            switch (dfsIdentifySector( dlsn, 0, dirbuf))
            {
               case ST_ROOTD:
               case ST_DIREC:
                  for ( entry = 0;
                       (entry < entries) && (rc == NO_ERROR) && !TxAbort();
                        entry++)                // each dir-entry in sector
                  {
                     efile = &(((S_EFATDIR *) dirbuf)[entry]);
                     isDir = ((efile->fl.Attrib & FATTR_DIRECTORY) != 0);

                     switch (efile->EntryType & ~EFAT_DIRBIT_IN_USE) // normal + deleted
                     {
                        case EFAT_DEC_BITMAP:
                        case EFAT_DEC_UPCASE:  // length & cluster fields identical for these
                           if (efile->bm.FileLength != 0)
                           {
                              if (type != 'D')
                              {
                                 rc = (cp->Func)( dlsn, entry | DFSSNINFO, (char *) &id, cp); // execute callback
                              }
                           }
                           break;

                        case EFAT_DEC_FILE:
                           stream = efile +1;   // advance to actual stream entry

                           if ((stream->st.FileLength != 0) || isDir)
                           {
                              if (((isDir == FALSE) && (type != 'D')) || //- type for this entry
                                  ((isDir == TRUE ) && (type != 'F'))  ) //- not excluded ?
                              {
                                 if (cp->Flag == FALSE) // progress in entries
                                 {
                                    iterationsDone++;
                                 }
                                 #if defined (USEWINDOWING)
                                    if (txwIsWindow( TXHWND_DESKTOP))
                                    {
                                       if (TxTmrTimerExpired( dfsa->statusTimer))
                                       {
                                          TXTM    status;

                                          if (cp->Flag == FALSE) // progress in entries
                                          {
                                             sprintf( status, "DIRsec: 0x%8.8llx, done:%6u DIR entries",
                                                               dlsn, iterationsDone);
                                          }
                                          else
                                          {
                                             sprintf( status, "DIRsec: 0x%8.8llx, done:%6u sectors",
                                                               dlsn, iterationsDone);
                                          }
                                          txwSetSbviewStatus( status, cSchemeColor);
                                          dfsa->statusTimer = TxTmrSetTimer( DFSP_STATUS_INTERVAL);
                                       }
                                    }
                                 #endif
                                 rc = (cp->Func)( dlsn, entry | DFSSNINFO, (char *) &id, cp); // execute callback
                              }
                              else
                              {
                                 TRACES(("Skipping entry: %u\n", entry));
                              }
                              if (isDir)
                              {
                                 if (rs != 1)   // recursion ?
                                 {
                                    ef = 0;
                                    dfsEfatAllocChain( dfsEfatDir2Clust( stream), 0, &ef, &nr, &sp);
                                    if (sp != NULL)
                                    {
                                       rc = dfsEfatIterFiles( rs -1, type, nr, sp, cp);
                                       TxFreeMem( sp);
                                    }
                                 }
                              }
                           }
                           break;

                        default:                // ignore others
                           break;
                     }
                  }
                  break;

               default:
                  break;
            }
         }
      }
      TxFreeMem( dirbuf);
   }
   RETURN( rc);
}                                               // end 'dfsEfatIterFiles'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read allocation for ALL directory sectors into a single SPACE structure
/*****************************************************************************/
ULONG dfsEfatAllDirs2Space
(
   BOOL                progressSectors,         // IN    progress in sectors, not DIRs
   ULONG              *nr,                      // OUT   nr of space entries
   S_SPACE           **sp                       // OUT   space allocation
)
{
   ULONG               rc = NO_ERROR;
   DFS_PARAMS          params;                  // generic function params
   ULONG               chunks = 0;              // chunks in space structure
   S_SPACE            *space  = NULL;           // space structure

   ENTER();

   //- Start with ROOT allocation itself (fixed chunk FAT16 or clusters on FAT32)
   if (dfsEfatDirLsn2Space( efat->Root, &chunks, &space) == NO_ERROR)
   {
      params.Lsn    = efat->Root;               // start sector for iterator
      params.Number = 0;                        // not a specific entry
      params.Func   = dfsEfatAddDir2Space;      // Iterator callback function
      params.Misc   = space;
      params.Count  = chunks;                   // Callback IN/OUT structure
      params.Flag   = progressSectors;

      //- Iterate over all directories recursively and add them to the SPACE
      rc = dfsEfatIterator( -1, 'D', NULL, &params);
      if (rc == NO_ERROR)
      {
         chunks =             params.Count;
         space  = (S_SPACE *) params.Misc;
      }
   }
   *sp = space;
   *nr = chunks;
   RETURN (rc);
}                                               // end 'dfsEfatAllDirs2Space'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Add SPACE for specified directory to the accumulating total directory SPACE
/*****************************************************************************/
static ULONG dfsEfatAddDir2Space
(
   ULN64               sn,                      // IN    sector-nr DIR sector
   ULN64               info,                    // IN    coded directory entry#
   char               *df,                      // IN    dummy, optional info
   void               *data                     // INOUT callback data, SPACE
)
{
   ULONG               rc      = NO_ERROR;
   DFS_PARAMS         *ip      = (DFS_PARAMS *) data;
   ULONG               extents =                ip->Count;
   S_SPACE            *sp      = (S_SPACE *)    ip->Misc;
   ULONG               chunks;
   S_SPACE            *space   = NULL;          // SPACE for this DIR
   S_EFATDIR           stream;                  // DIR entry, STREAM

   ENTER();

   if ((rc = dfsEfatLsnInf2Stream( sn, info, &stream)) == NO_ERROR)
   {
      if (dfsEfatAllocStream( &stream, NULL, &chunks, &space) != 0)
      {
         if (space != NULL)
         {
            //- join with accumulator space, and free the old ones
            rc = dfsSspaceJoin( extents, sp, chunks, space, &extents, &sp);
            if (rc == NO_ERROR)
            {
               ip->Count = extents;
               ip->Misc  = sp;                  // pass back joined SPACE
            }
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsEfatAddDir2Space'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Determine allocation-bit for specified LSN, ALLOCATED beyond last cluster!
/*****************************************************************************/
ULONG dfsEfatAllocated                          // RET   LSN is allocated
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               asn = dfstAreaD2Part( DFSTORE, lsn); // Area aware sn
   ULONG               rc  = DFS_PSN_LIMIT;

   if ((efat->Bm.Size != 0) && (asn < efat->Bm.LimitLsn)) // inside the bitmap
   {
      if (asn < (efat->ClHeap + (efat->RawClust * efat->ClustSize))) // within valid cluster area
      {
         rc = (ULONG) dfsEfatBitmapCache(dfsEfatLSN2Clust( asn), NULL);
      }
      else                                      // return ALLOCATED for any
      {                                         // lsn beyond last cluster
         rc = 1L;                               // to avoid CHECK errors
      }                                         // on non-standard bitmaps
   }
   //- TRACES(("EfatAllocated: lsn:0x%llx, Bm.LimitLsn=0x%llx, allocated:%x\n", lsn, efat->Bm.LimitLsn, rc));
   return (rc);
}                                               // end 'dfsEfatAllocated'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Align R/W cache to position for cluster, and get current allocation-bit
/*****************************************************************************/
BOOL dfsEfatBitmapCache                         // RET   cluster is allocated
(
   ULONG               cluster,                 // IN    LCN
   ULONG              *bp                       // OUT   byte position in cache
)                                               //       or L64_NULL if invalid
{
   ULONG               rc = NO_ERROR;
   ULONG               cl;
   BOOL                al = TRUE;               // default allocated
   ULONG               cs = efat->ClustSize;    // cluster size in sectors
   ULONG               byte;                    // byte pos in sector
   ULONG               rsn;                     // relative sector in Bitmap
   ULONG               bytepos = L32_NULL;      // byte position in cluster
   BYTE                alloc_byte;

   if ((cluster >= 2) && (cs != 0))
   {
      cl  = cluster - 2;                        // first bit in bitmap is cluster 2
      rsn = cl / (8 * dfsGetSectorSize());      // find relative BM sector for cl

      if (((rsn <  efat->Bm.First) ||           // if rsn is not in current
           (rsn >= efat->Bm.First +             // cached range of rsn's
                   efat->Bm.Size)))             // read new range ...
      {
         if (efat->Bm.Dirty)                    // need to flush changes ?
         {
            rc = dfsEfatBitmapFlush( FALSE);    // flush, but keep cache
         }
         if (rc == NO_ERROR)
         {
            efat->Bm.First = (rsn / cs) * cs;
            TRACES(("Read  new cache Bm.First: %8.8x\n", efat->Bm.First));
            dfsSspaceReadFilePart( efat->Bm.Extents,
                                   efat->Bm.Space,
                                   efat->Bm.First,
                                   efat->Bm.Size,
                          (BYTE *) efat->Bm.Buffer);
         }
      }
      if (rc == NO_ERROR)
      {
         byte       = (cl % (8 * dfsGetSectorSize())) / 8; // byte index in sector
         bytepos    = ((rsn - efat->Bm.First) * dfsGetSectorSize()) + byte;

         if ((bytepos < (efat->Bm.Size * dfsGetSectorSize())) &&
                         efat->Bm.Buffer != NULL)
         {
            alloc_byte = efat->Bm.Buffer[ bytepos];
            if (((alloc_byte >> (cl & 0x07)) & 0x01) == 0) // LSB is lowest!
            {
               al = FALSE;
            }
         }
         else                                   // signal error condition
         {
            bytepos = L32_NULL;
         }
      }
      if (bp != NULL)                           // return byte position too ?
      {
         *bp = bytepos;
      }
   }
   //- TRACES(("BitMapCache, cluster: 0x%8.8x bytepos:0x%8.8x al:%u\n", cluster, bytepos, al));
   return (al);
}                                               // end 'dfsEfatBitmapCache'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Flush BitMap cache when Dirty, and optional free its resources
/*****************************************************************************/
ULONG dfsEfatBitmapFlush                        // RET   rc
(
   BOOL                terminate                // IN    terminate cache too
)
{
   ULONG               rc = NO_ERROR;           // rc

   ENTER();

   TRACES(("BitMap cache Bm.First: %8.8x, ", efat->Bm.First));
   if (efat->Bm.Dirty)                          // need to flush changes ?
   {
      TRACES(( "dirty, flush it ...\n"));
      rc = dfsSspaceWriteFilePart( efat->Bm.Extents,
                                   efat->Bm.Space,
                                   efat->Bm.First,
                                   efat->Bm.Size,
                          (BYTE *) efat->Bm.Buffer);

      efat->Bm.Dirty = FALSE;                   // mark it clean again
   }
   else
   {
      TRACES(( "not dirty ...\n"));
   }
   if (terminate)
   {
      TxFreeMem( efat->Bm.Space);
      TxFreeMem( efat->Bm.Buffer);

      efat->Bm.Size     = 0;
      efat->Bm.LimitLsn = 0;
   }
   RETURN (rc);
}                                               // end 'dfsEfatBitmapFlush'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set allocate status for LSN to specified value (beyond FS, to end of bitmap)
/*****************************************************************************/
ULONG dfsEfatSetAlloc                           // RET   LSN allocation set
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *value,                   // IN    NULL = not allocated
   void               *ref                      // IN    dummy (for EFAT)
)
{
   if ((efat->Bm.Size != 0) && (lsn < efat->Bm.LimitLsn)) // inside the bitmap
   {
      return( dfsEfatSetBmCluster(dfsEfatLSN2Clust( lsn), (value != NULL)));
   }
   else
   {
      return( DFS_PSN_LIMIT);
   }
}                                               // end 'dfsEfatSetAlloc'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Set allocate status for CLUSTER to specified value
/*****************************************************************************/
ULONG dfsEfatSetBmCluster                         // RET   CL  allocation set
(
   ULONG               cl,                      // IN    Cluster
   BOOL                value                    // IN    SET allocation bit
)
{
   ULONG               rc = NO_ERROR;
   ULONG               bytepos;
   BYTE                alloc_byte;
   BYTE                bit_mask = (1 << (cl & 0x07));

   dfsEfatBitmapCache( cl, &bytepos);
   if (bytepos != L32_NULL)                     // cluster position valid ?
   {
      alloc_byte = efat->Bm.Buffer[ bytepos];

      if (value)                                // set the allocation bit
      {
         efat->Bm.Buffer[ bytepos] = alloc_byte |  bit_mask;
      }
      else                                      // reset allocation bit
      {
         efat->Bm.Buffer[ bytepos] = alloc_byte & ~bit_mask;
      }
      TRACES(( "%s cache-byte at 0x%4.4x = -%6.6x- for cl:%8.8x "
                  "from 0x%2.2hx to 0x%2.2hx\n",
                (value) ? "Set" : "Clr", bytepos, cl / 8, cl,
                alloc_byte, efat->Bm.Buffer[ bytepos]));

      efat->Bm.Dirty = TRUE;                    // mark cache dirty
   }
   else
   {
      rc = DFS_PSN_LIMIT;
   }
   return (rc);
}                                               // end 'dfsEfatSetBmCluster'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read, calculate checksums and write primary and secondary boot area structures
/*****************************************************************************/
ULONG dfsEfatCrcSyncBootAreas
(
   ULN64               source,                  // IN    LSN of source area
   int                 area                     // IN    destination area 0=both/1/2
)
{
   ULONG               rc = NO_ERROR;

   ENTER();
   TRACES(("source: 0x%llx  area: %d\n", source, area));

   if (DFSTORE_WRITE_ALLOWED)
   {
      if ((rc = dfsRead( source, EFAT_BOOTSIZE, brec)) == NO_ERROR)
      {
         S_BOOTR      *sd = (S_BOOTR *) brec;   // view as boot record

         if ((sd->ex.BpsShift >= 9) && (sd->ex.BpsShift <= 12))    //- 512..4096 bytes
         {
            ULONG      cs = 0;                  // checksum value
            ULONG      size;                    // size boot area in bytes
            ULONG      pos;                     // calculated position 1st CS
            ULONG      count;                   // calculated nr of Checksums
            ULONG     *refCs;                   // start of checksum sector
            ULONG      i;                       // byte position

            size = ((1 << sd->ex.BpsShift) * EFAT_BOOTSIZE);
            cs   = dfsEfatBootCheckSum( brec, size);               //- new checksum value

            pos    = (1 << sd->ex.BpsShift) * (EFAT_BOOTSIZE -1);
            count  = (1 << sd->ex.BpsShift) / sizeof( ULONG);
            refCs  = (ULONG *) (&brec[ pos]);
            for (i = 0; i < count; i++)
            {
               refCs[i] = cs;                   // and update in last sector
            }

            if ((area == 0) || (area == 1))     // write primary
            {
               rc = dfsWrite( LSN_BOOTR, EFAT_BOOTSIZE, brec);
            }
            if ((area == 0) || (area == 2))     // write secondary
            {
               rc = dfsWrite( EFAT_BACKUPBOOT, EFAT_BOOTSIZE, brec);
            }
         }
         else
         {
            rc = DFS_BAD_STRUCTURE;
         }
      }
   }
   else
   {
      rc = DFS_READ_ONLY;
   }
   RETURN(rc);
}                                               // end 'dfsEfatCrcSyncBootAreas'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Calculate EFAT boot-area specific checksum, using published algorithm
/*****************************************************************************/
ULONG dfsEfatBootCheckSum                       // RET   EFAT boot style crc
(
   BYTE               *data,                    // IN    Boot area (>= 11 sect)
   ULONG               size                     // IN    Size of data area
)
{
   ULONG               cs = 0;                  // checksum value
   ULONG               i;                       // checksum value
   S_BOOTR            *sd = (S_BOOTR *) data;   // view as boot record
   ULONG               limit = 0x10000;         // calculated size, upto 64 KiB

   ENTER();
   TRACES(("Data size: 0x%8.8x\n", size));

   if ((sd->ex.BpsShift >= 9) && (sd->ex.BpsShift <= 12)) // 512 .. 4096 bytes
   {
      limit = (1 << sd->ex.BpsShift) * (EFAT_BOOTSIZE -1);
   }
   if (limit > size)
   {
      limit = size;                             // but limit to specified data size
   }
   for (i = 0; i < limit; i++)
   {
      switch (i)
      {
         case 0x6A:                             // Volume flag bytes
         case 0x6B:                             // Incl. 'dirty' flag
         case 0x70:                             // Percent in-use byte
            continue;                           // skip
            break;

         default:
            cs  = ((cs << 31) | (cs >> 1)) + data[i];
            break;
      }
   }
   RETURN( cs);
}                                               // end 'dfsEfatBootCheckSum'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read EFAT boot-area checksum reference from sector and check consistency
/*****************************************************************************/
ULONG dfsEfatStoredChecksum                     // RET   EFAT boot style crc
(
   BYTE               *data,                    // IN    Boot area (>= 12 sect)
   ULONG               size                     // IN    Size of data area
)
{
   ULONG               cs = 0;                  // checksum value
   ULONG               i;                       // byte position
   S_BOOTR            *sd = (S_BOOTR *) data;   // view as boot record
   ULONG               pos;                     // calculated position 1st CS
   ULONG               count;                   // calculated nr of Checksums
   ULONG              *refCs;

   ENTER();

   if ((sd->ex.BpsShift >= 9) && (sd->ex.BpsShift <= 12))    //- 512..4096 bytes
   {
      if (size >= ((1 << sd->ex.BpsShift) * EFAT_BOOTSIZE)) //- all within data
      {
         pos    = (1 << sd->ex.BpsShift) * (EFAT_BOOTSIZE -1);
         count  = (1 << sd->ex.BpsShift) / sizeof( ULONG);
         refCs  = (ULONG *) (&data[ pos]);

         for (i = 1, cs = refCs[0]; i < count; i++)
         {
            if (refCs[i] != cs)                 // inconsistent ?
            {
               cs = 0;
               break;
            }
         }
      }
   }
   RETURN( cs);
}                                               // end 'dfsEfatStoredChecksum'
/*---------------------------------------------------------------------------*/
