//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// DFS  utility display services
//
// Author: J. van Wijk
//
// JvW  29-06-2000 Initial version, split off from DFSUTIL.C
// JvW  02-07-2000 Moved sector-display functions from dfs.c
//
// Note: Interface in dfsutil.h

#include <txlib.h>                              // ANSI controls and logging

#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfshpfs.h>                            // HPFS structure defs
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsupart.h>                           // Partition definitions
#include <dfsufdsk.h>                           // Fdisk & translation services
#include <dfsutil.h>                            // DFS utility & display functions
#include <dfsafdsk.h>                           // FDISK/LVM definitions

#if defined (WIN32)
   #include <dfsntreg.h>                        // Windows error texts
#endif

#if !defined (OEMSB)
#include <dfsspace.h>                           // DFS file-space interface
#include <dfstable.h>                           // SLT utility functions
#endif


/*****************************************************************************/
// Display single sectortype with type-char and colored description in ASCII
/*****************************************************************************/
void dfsSectorTypeInColor
(
   char              *lead,                     // IN    leading string
   BYTE               st                        // IN    sector type
)
{
   TXTM                text;
   BYTE                clean = (BYTE)(st & ~ST__INFO);

   dfsSectorTypeAsAscii( clean, text);
   if (TxPrintable( clean))
   {
      TxPrint( "%s  '%c'", lead, clean);
   }
   else
   {
      TxPrint( "%s  x%2.2hhx", lead, st);
   }
   TxPrint( " = %s%s%s", (st < ST__INFO) ? CBM : CNG, text, CNN);
}                                               // end 'dfsSectorTypeInColor'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Return specified sectortype in ASCII
/*****************************************************************************/
void dfsSectorTypeAsAscii
(
   BYTE                st,                      // IN    sector type
   char               *buf                      // OUT   text buffer
)
{
   ULONG               rc;

   rc = DFSFNCALL(dfsa->FsShowType, 0, 0, (char *) &st, buf);
   if (rc == DFS_PENDING)                       // not handled
   {
      switch (st)                               // searchable types
      {
         case ST_MASTR: sprintf(buf,"Master   Boot Rec"); break;
         case ST_EXTBR: sprintf(buf,"Extended Boot Rec"); break;
         case ST_BOOTR: sprintf(buf,"Fsys boot sector "); break;
         case ST_EMPTY: sprintf(buf,"ZERO 0x00 cleared"); break;
         case ST_BADFE: sprintf(buf,"Marked BADsec xFE"); break;
         case ST_FDISK: sprintf(buf,"FDISK xF6 cleared"); break;
         case ST_LVINF: sprintf(buf,"LVM DLAT info sec"); break;
         case ST_LVSIG: sprintf(buf,"LVM BBR signature"); break;
         case ST_LVREM: sprintf(buf,"LVM BBR V-deleted"); break;
         case ST_LVDLF: sprintf(buf,"LVM BBR drivelink"); break;
         case ST_LVDLT: sprintf(buf,"LVM BBR drv-table"); break;
         case ST_LVBBF: sprintf(buf,"LVM BBR badblocks"); break;
         case ST_LVBBT: sprintf(buf,"LVM BBR bad-table"); break;
         case ST_LUKSH: sprintf(buf,"LUKS Crypt Header"); break;
         case ST_CORES: sprintf(buf,"CoreStorageHeader"); break;
         case ST_MCDDM: sprintf(buf,"MAC DDM, APM  hdr"); break;
         case ST_MCDPM: sprintf(buf,"MAC DPM, part map"); break;
         case ST_GPTHD: sprintf(buf,"GPT header sector"); break;
         case ST_GPTAS: sprintf(buf,"GPT array sectors"); break;
         case ST_KNOWN: sprintf(buf,"any known type   "); break;
         case ST_WILDC: sprintf(buf,"any sector       "); break;
         case ST_FREES: sprintf(buf,"Free space       "); break;

                        //- Superblock generic FSMODE ones (bsfind)
         case ST_SEXT : sprintf(buf,"EXT  superblock *"); break;
         case ST_SHPFS: sprintf(buf,"HPFS superblock *"); break;
         case ST_SHFS : sprintf(buf,"HFS  superblock *"); break;
         case ST_SAPFS: sprintf(buf,"APFS superblock *"); break;
         case ST_SJFS : sprintf(buf,"JFS  superblock *"); break;
         case ST_SNTFS: sprintf(buf,"NTFS NTldr/Bmgr *"); break;
         case ST_SRSR : sprintf(buf,"RSR  superblock *"); break;
         case ST_SXFS : sprintf(buf,"XFS  superblock *"); break;
         default:
            switch (st | ST__INFO)              // non-searchable ones
            {
               case ST_BOOT2: sprintf(buf,"FAT32 2nd BootSec"); break;
               case ST_BOOT3: sprintf(buf,"FAT32 3rd BootSec"); break;
               case ST_BOOTX: sprintf(buf,"Non-std  Boot Rec"); break;
               case ST_RESRV: sprintf(buf,"Fsys reserved sec"); break;
               case ST_FINAL: sprintf(buf,"End of Part/Disk!"); break;
               case ST_HDATA: sprintf(buf,"Hex  format  data"); break;
               case ST_TDATA: sprintf(buf,"Text format  data"); break;
               case ST_FDATA: sprintf(buf,"File data        "); break;
               case ST_BADSC: sprintf(buf,"Bad sector area  "); break;
               case ST_ULZ16: sprintf(buf,"DFSee compress 16"); break;
               case ST_ULZ32: sprintf(buf,"DFSee compress 32"); break;
               case ST_MINIM: sprintf(buf,"No contents shown"); break;
               default:       sprintf(buf,"Unidentified data"); break;
            }
            break;
      }
   }
}                                              // end 'dfsSectorTypeAsAscii'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Format (byte/sect) size Dec KiB/MiB/GiB/TiB; append to string (8 positions)
/*****************************************************************************/
char *dfstrS4XiB                                // RET   resulting string
(
   char               *str,                     // OUT   resulting string
   char               *text,                    // IN    leading string
   ULONG               data,                    // IN    data (sectors or bytes)
   USHORT              bps,                     // IN    bytes per sector, or 1)
   char               *trail                    // IN    trailing text
)
{
   TXTS                form;
   double              value = TXS2KB( data, bps);

   strcat( str, text);

   if ((bps == 1) && (data <= 999999))          // decimal bytes (exact)
   {
      sprintf( form, "%6u B", data);            // 6 digits, Bytes
   }
   else if (value <= 99.9)
   {
      sprintf( form, "%4.1lf KiB", value);      // one decimal upto 100 KiB
   }
   else if (value <= 4999.0)
   {
      sprintf( form, "%4.0lf KiB", value);      // KiB upto 5 MiB
   }
   else
   {
      value /= 1024.0;                          // convert to Mib
      if (value <= 9999.0)                      // MiB upto 10 GiB
      {
         if      (value <= 9.99)
         {
            sprintf( form, "%4.2lf MiB", value);
         }
         else if (value <= 99.9)
         {
            sprintf( form, "%4.1lf MiB", value);
         }
         else
         {
            sprintf( form, "%4.0lf MiB", value);
         }
      }
      else
      {
         value /= 1024.0;                       // convert to Gib
         if (value <= 9999.0)                   // GiB upto 10 TiB
         {
            if      (value <= 9.99)
            {
               sprintf( form, "%4.2lf GiB", value);
            }
            else if (value <= 99.9)
            {
               sprintf( form, "%4.1lf GiB", value);
            }
            else
            {
               sprintf( form, "%4.0lf GiB", value);
            }
         }
         else
         {
            value /= 1024.0;                    // convert to Tib
            if (value <= 9999.0)                // TiB upto 10 EiB
            {
               if      (value <= 9.99)
               {
                  sprintf( form, "%4.2lf TiB", value);
               }
               else if (value <= 99.9)
               {
                  sprintf( form, "%4.1lf TiB", value);
               }
               else
               {
                  sprintf( form, "%4.0lf TiB", value);
               }
            }
            else
            {
               sprintf( form, "overflow");
            }
         }
      }
   }
   strcat( str, form);
   strcat( str, trail);
   return( str);
}                                               // end 'dfstrS4XiB'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display (sector) size in 8-hex digits and KiB/MiB/GiB; to TxPrint output
/*****************************************************************************/
void dfsSizeBps
(
   char               *text,                    // IN    leading string
   ULONG               data,                    // IN    data
   USHORT              bps,                     // IN    bytes per sector
   char               *trail                    // IN    trailing text
)
{
   TXLN                string;

   strcpy( string, "");
   TxPrint("%s", dfstrSizeBps( string, text, data, bps, trail));
}                                               // end 'dfsSizeBps'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Format (sector) size in 8-hex digits and KiB/MiB/GiB; append to string
/*****************************************************************************/
char *dfstrSizeBps                              // RET   resulting string
(
   char               *str,                     // OUT   resulting string
   char               *text,                    // IN    leading string
   ULONG               data,                    // IN    data
   USHORT              bps,                     // IN    bytes per sector
   char               *trail                    // IN    trailing text
)
{
   TXLN                form;

   sprintf(form, "%s0x%8.8X = ", text, data);
   dfstrSizeXiB( str, form, data, bps, trail);
   return( str);
}                                               // end 'dfstrSizeBps'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Format (sector) size in 7.1 value + KiB/MiB/GiB total 11; to TxPrint output
/*****************************************************************************/
void dfsSizeXiB
(
   char               *text,                    // IN    leading string
   ULONG               data,                    // IN    data
   USHORT              bps,                     // IN    bytes per sector
   char               *trail                    // IN    trailing text
)
{
   TXLN                string;

   strcpy( string, "");
   TxPrint("%s", dfstrSizeXiB( string, text, data, bps, trail));
}                                               // end 'dfsSizeXiB'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Format (sector) size in 7.1 value + KiB/MiB/GiB total 11; append to string
/*****************************************************************************/
char *dfstrSizeXiB                              // RET   resulting string
(
   char               *str,                     // OUT   resulting string
   char               *text,                    // IN    leading string
   ULONG               data,                    // IN    data
   USHORT              bps,                     // IN    bytes per sector
   char               *trail                    // IN    trailing text
)
{
   TXTS                form;

   strcat( str, text);
   if (bps != 0)
   {
      double           value = TXS2KB( data, bps);

      if      (value <= 4999.9)
      {
         sprintf(   form, "%7.1lf KiB", value);
      }
      else
      {
         value /= 1024.0;                       // convert to Mib
         if (value <= 99999.9)
         {
            sprintf(form, "%7.1lf MiB", value);
         }
         else if (value <= 9999999.9)           // Keep MiB as long as possible
         {                                      // works upto 10 terabyte ...
            sprintf(form, "%7.0lf MiB", (value - 0.1)); // avoid rounding up to much
         }
         else if (data != DFS32MAX)
         {                                      // when exceeding even that ...
            value /= 1024.0;                    // convert to Gib
            sprintf(form, "%7.1lf GiB", value);
         }
         else
         {
            sprintf(form, "--MAXIMUM--");
         }
      }
   }
   else
   {
      sprintf(form, "0 (bps = 0)");
   }
   strcat( str, form);
   strcat( str, trail);
   return( str);
}                                               // end 'dfstrSizeXiB'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Format (sect64) size in 7.1 value + KiB/MiB/GiB/TiB total 11 TxPrint output
/*****************************************************************************/
void dfsSz64XiB
(
   char               *text,                    // IN    leading string
   ULN64               data,                    // IN    data
   USHORT              bps,                     // IN    bytes per sector
   char               *trail                    // IN    trailing text
)
{
   TXLN                string;

   strcpy( string, "");
   TxPrint("%s", dfstrSz64XiB( string, text, data, bps, trail));
}                                               // end 'dfsSz64XiB'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Format (sect64) size in 7.1 value + KiB/MiB/GiB/TiB total 11 append to str
/*****************************************************************************/
char *dfstrSz64XiB                              // RET   resulting string
(
   char               *str,                     // INOUT resulting string
   char               *text,                    // IN    leading string
   ULN64               data,                    // IN    data
   USHORT              bps,                     // IN    bytes per sector
   char               *trail                    // IN    trailing text
)
{
   TXTS                form;

   strcat( str, text);
   if (bps != 0)
   {
      double           value = TXS2KB( data, bps);

      if      (value <= 4999.9)
      {
         sprintf(   form, "%7.1lf KiB", value);
      }
      else
      {
         value /= 1024.0;                       // convert to Mib
         if (value <= 99999.9)
         {
            sprintf(form, "%7.1lf MiB", value);
         }
         else
         {
            value /= 1024.0;                    // convert to Gib
            if (value <= 99999.9)
            {
               sprintf(form, "%7.1lf GiB", value );
            }
            else
            {                                   // when exceeding even that ...
               value /= 1024.0;                 // convert to Tib
               if (value <= 99999.9)
               {
                  sprintf(form, "%7.1lf TiB", value);
               }
               else
               {
                  value /= 1024.0;              // convert to Eib
                  if (value <= 99999.9)
                  {
                     sprintf(form, "%7.1lf EiB", value);
                  }
                  else
                  {
                     value /= 1024.0;           // convert to Pib
                     if (value <= 99999.9)
                     {
                        sprintf(form, "%7.1lf PiB", value);
                     }
                     else if (data != DFS_MAX_PSN)
                     {
                        sprintf(form, "%7.0lf PiB", (value - 0.1)); // avoid rounding up to much
                     }
                     else
                     {
                        sprintf(form, "--MAXIMUM--");
                     }
                  }
               }
            }
         }
      }
   }
   else
   {
      sprintf(form, "0 (bps = 0)");
   }
   strcat( str, form);
   strcat( str, trail);
   return( str);
}                                               // end 'dfstrSz64XiB'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display 64bit value with 0x+8-hex or 10 hex digits in color; to TxPrint output
/*****************************************************************************/
void dfsX10
(
   char               *text,                    // IN    leading string
   ULN64               data,                    // IN    data
   char               *color,                   // IN    color
   char               *trail                    // IN    trailing text
)
{
   TXLN                string;

   strcpy( string, "");
   TxPrint("%s", dfstrX10( string, text, data, color, trail));
}                                               // end 'dfsX10'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// Format 64bit value with 0x+8-hex or 10 hex digits in color; append to string
/*****************************************************************************/
char *dfstrX10                                  // RET   resulting string
(
   char               *str,                     // INOUT resulting string
   char               *text,                    // IN    leading string
   ULN64               data,                    // IN    data
   char               *color,                   // IN    color
   char               *trail                    // IN    trailing text
)
{
   TXTM                form;                    // for value + colors only!

   strcat( str, text);
   if (data <= (ULN64) DFS32MAX)                // fits in 8 hex digits
   {
      sprintf( form, "%s0x%s%8.8llX%s", CNZ, color, data, CNN);
   }
   else
   {
      sprintf( form, "%s%10llX%s",    color, data, CNN);
   }
   strcat( str, form);
   strcat( str, trail);
   return (str);
}                                               // end 'dfstrX10'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display 64bit value '-' / ' ' + 0x+8-hex / 10 hex digits; to TxPrint output
/*****************************************************************************/
void dfsSX11
(
   char               *text,                    // IN    leading string
   ULN64               data,                    // IN    data (as signed!)
   char               *color,                   // IN    (color) string
   char               *trail                    // IN    trailing text
)
{
   TXLN                string;

   strcpy( string, "");
   TxPrint("%s", dfstrSX11( string, text, data, color, trail));
}                                               // end 'dfsSX11'
/*---------------------------------------------------------------------------*/

/*****************************************************************************/
// Format 64bit value '-' / ' ' + 0x+8-hex / 10 hex digits; append to string
/*****************************************************************************/
char *dfstrSX11                                  // RET   resulting string
(
   char               *str,                     // INOUT resulting string
   char               *text,                    // IN    leading string
   ULN64               data,                    // IN    data (as signed!)
   char               *color,                   // IN    (color) string
   char               *trail                    // IN    trailing text
)
{
   ULN64               value;

   strcat( str, text);

   if ((data > 0x8000000000000000ULL) && (data != DFS_MAX_PSN)) // negative value
   {
      strcat( str, "-");                        // add explcit minus sign
      value = 0 - data;                         // and make value positive
   }
   else
   {
      strcat( str, " ");                        // just a space for positives
      value = data;
   }
   dfstrX10( str, "", value, color, trail);     // append 10 position value
   return (str);
}                                               // end 'dfstrSX11'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DFS show message to standard screen or to STDOUT if -G switch set
/*****************************************************************************/
void dfsGuiStdMessage
(
   char               *msg                      // IN    progress message
)
{
   if (dfsGuiStdOut())
   {
      printf( "%s\n", msg);
      fflush( stdout);
   }
   TxPrint( "%s\n", msg);                       // to regular output too
}                                               // end 'dfsGuiStdMessage'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// DFS test if important output should be sent to STDOUT if -g switch set
/*****************************************************************************/
BOOL dfsGuiStdOut
(
   void
)
{
   return ((dfsa->stdOutInterval != 0) && dfsa->nowindow);
}                                               // end 'dfsGuiStdOut'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get DFS return-code description in an (ANSI decorated) ASCII string
/*****************************************************************************/
char *dfsGetRcString                            // RET   RC description string
(
   ULONG               rc,                      // IN    DFS returncode
   TXLN                message                  // INOUT RC message string
)
{
   TXLN                msg;                     // base  of message line
   TXLN                psnstore;                // PSN & store description

   ENTER();

   sprintf( psnstore, " at Lsn:0x0%llx = PSN:0x0%llx", dfstPsn2LSN( DFSTORE, dfstLastPsn( DFSTORE)), dfstLastPsn( DFSTORE));

   #ifndef OEMSB
   if (!dfsGuiStdOut())                         // Normal operation
   {
      //- Show returncode as a SIGNED 16-bit value, mainly for (unix) -1 failure values
      sprintf( message,  "\nMsg: %s%-5hd%s = %04X : ", CBR, (short) (rc & ~TX_RCFLAG_MASK),
                                                       CNN,         (rc & ~TX_RCFLAG_MASK));
      strcat(  psnstore, "\n                    ");
      strcat(  psnstore, dfstStoreDesc1( DFSTORE) +10);
   }
   else
   {
      sprintf( message,    "Message%5hd : ", (short) (rc & ~TX_RCFLAG_MASK));
   }
   #else
      sprintf( message,  "\nMessage%5hd : ", (short) (rc & ~TX_RCFLAG_MASK));
   #endif
   switch (rc & ~TX_RCFLAG_MASK)                      // filter out the flags, like quit
   {
      case NO_ERROR:                   sprintf( msg,"No error");                                      break;

   #if defined (WIN32)
      case 1:                          sprintf( msg, "%s, %s", txNtLastError(), psnstore);
   #endif

   #if defined (DOS32)
      case 381:                                 // Low-level access DosBox
      case ERROR_NO_WRITE_ACCESS:      sprintf( msg,"Low-level read/write access not allowed");       break;
   #endif
   #if defined (UNIX)
      case TX_UNIX_ERROR:              sprintf( msg, "%s", strerror(errno));                          break;
      case ERROR_INVALID_DRIVE:        sprintf( msg,"Path has incorrect directory components");       break;
   #else
      case ERROR_INVALID_DRIVE:        sprintf( msg,"Drive is invalid");                              break;
   #endif
      case ERROR_DISK_CHANGE:
      case ERROR_WRONG_DISK:           sprintf( msg,"Wrong disk is in the drive");                    break;
      case TX_DISPLAY_CHANGE:          sprintf( msg,"Display mode has changed");                      break;
      case DFS_INV_MBR:                sprintf( msg,"Invalid master boot record detected");           break;
      case DFS_INV_BOOT:               sprintf( msg,"Invalid boot record detected");                  break;
      case DFS_OVERFLOW:               sprintf( msg,"Partition list overflow, too many partitions");  break;
      case DFS_REBOOT_REQ:             sprintf( msg,"Reboot required for safe operation ...");        break;
      case DFS_CMD_FAILED:             sprintf( msg,"Command completed with serious errors");         break;
      case DFS_CMD_WARNING:            sprintf( msg,"Command completed with warnings");               break;
      case DFS_READ_ONLY:              sprintf( msg,"Object is Read-Only, or can not be locked");     break;
      case DFS_BAD_STRUCTURE:          sprintf( msg,"Inconsistent structure");                        break;
      case DFS_ST_MISMATCH:            sprintf( msg,"Sector-type mismatch");                          break;
      case DFS_ALLOC_ERROR:            sprintf( msg,"Memory allocation error");                       break;
      case DFS_VALUE_ERROR:            sprintf( msg,"Specified value is not acceptable");             break;
      case DFS_PENDING:                sprintf( msg,"Function pending, or not implemented");          break;
      case TX_ABORTED:                 sprintf( msg,"Function, confirmation or message aborted");     break;
      case DFS_USER_ABORT:             sprintf( msg,"Function aborted by user, possibly incomplete"); break;
      case DFS_NO_CHANGE:              sprintf( msg,"Function aborted or skipped, no changes made");  break;
      case DFS_NO_DEVICE:              sprintf( msg,"No object opened (disk, partition ...)");        break;
      case DFS_PSN_LIMIT:              sprintf( msg,"SN exceeds current limit%s", psnstore);          break;
      case DFS_NOT_FOUND:              sprintf( msg,"Searched item not found");                       break;
      case DFS_LIST_EMPTY:             sprintf( msg,"Sector list is empty");                          break;
      case DFS_NO_SCRIPT:              sprintf( msg,"Script file could not be opened");               break;
      case TX_EXT_CMD_ERROR:           sprintf( msg,"Invalid external command or parameter");         break;
      case TX_NO_COMPRESS:             sprintf( msg,"TX library, no IMZ compression achieved");       break;
      case TX_NO_INITIALIZE:           sprintf( msg,"TX library, no IMZ compression opened");         break;
      case TX_BAD_OPTION_CHAR:         sprintf( msg,"TX library, option-character out of range");     break;
      case TX_TOO_MANY_ARGS:           sprintf( msg,"TX library, too many command arguments");        break;
      case TX_SYNTAX_ERROR:            sprintf( msg,"Syntax error in script or commandline");         break;
      case TX_ERROR:                   sprintf( msg,"Runtime error script or commandline");           break;
      case TX_FAILED:                  sprintf( msg,"TX library function failed");                    break;
      case TX_IGNORED:                 sprintf( msg,"TX library command ignored, empty");             break;
      case TX_INVALID_FILE:            sprintf( msg,"TX library invalid filename");                   break;
      case TX_INVALID_PATH:            sprintf( msg,"TX library invalid path");                       break;
      case TX_INVALID_DRIVE:           sprintf( msg,"TX library invalid drive");                      break;
      case TX_ACCESS_DENIED:           sprintf( msg,"TX library access denied");                      break;
      case ERROR_FILE_NOT_FOUND:       sprintf( msg,"File not found");                                break;
      case ERROR_PATH_NOT_FOUND:       sprintf( msg,"Path not found");                                break;
      case ERROR_ACCESS_DENIED:        sprintf( msg,"Access denied");                                 break;
      case ERROR_TOO_MANY_OPEN_FILES:  sprintf( msg,"Too many open files");                           break;
      case ERROR_INVALID_HANDLE:       sprintf( msg,"Invalid (file/disk) handle used (R/O?)");        break;
      case ERROR_SHARING_VIOLATION:    sprintf( msg,"Sharing violation, file is locked");             break;
      case ERROR_NO_MORE_FILES:        sprintf( msg,"No (more) matching files or directories found"); break;
      case ERROR_WRITE_PROTECT:        sprintf( msg,"Drive is write-protected, full or invalid");     break;
      case ERROR_NOT_READY:            sprintf( msg,"Drive/diskette not ready, or LSN out of range"); break;
      case ERROR_FILENAME_EXCED_RANGE: sprintf( msg,"Invalid filename, probably on FAT");             break;
      case ERROR_FILE_EXISTS:          sprintf( msg,"File exists with specified name");               break;
      case ERROR_CRC:                  sprintf( msg,"CRC error%s", psnstore);                         break;
      case ERROR_SEEK:                 sprintf( msg,"Seek error%s", psnstore);                        break;
      case ERROR_SECTOR_NOT_FOUND:     sprintf( msg,"Sector not found%s", psnstore);                  break;
      case ERROR_WRITE_FAULT:          sprintf( msg,"Write error%s", psnstore);                       break;
      case ERROR_READ_FAULT:           sprintf( msg,"Read error%s", psnstore);                        break;
      case ERROR_GEN_FAILURE:          sprintf( msg,"General failure%s", psnstore);                   break;
      case ERROR_INVALID_PARAMETER:    sprintf( msg,"Not implemented or invalid API parameter");      break;
      case ERROR_NOT_SUPPORTED:        sprintf( msg,"Request not supported (NET, IFS)");              break;
      case ERROR_LOCK_VIOLATION:       sprintf( msg,"Lock violation");                                break;
      case ERROR_DEVICE_IN_USE:        sprintf( msg,"Device in use");                                 break;
      case ERROR_DRIVE_LOCKED:         sprintf( msg,"Drive locked by another proccess (open files)"); break;
      case ERROR_OPEN_FAILED:          sprintf( msg,"Open failure");                                  break;
      case ERROR_BROKEN_PIPE:          sprintf( msg,"Broken pipe");                                   break;
      case ERROR_DISCARDED:            sprintf( msg,"Data was discarded");                            break;
      case ERROR_DISK_FULL:            sprintf( msg,"Disk is full");                                  break;
      case ERROR_NEGATIVE_SEEK:        sprintf( msg,"Seek error, possible 2GB limit%s", psnstore);    break;
      case ERROR_SEEK_ON_DEVICE:       sprintf( msg,"Seek error, no disk or partition selected");     break;
      case ERROR_INVALID_PATH:         sprintf( msg,"Invalid path, no wildcards allowed");            break;
      case ERROR_LOCKED_OUT:           sprintf( msg,"Physical disk is not accessible (locked)");      break;
      case ERROR_NOT_A_CMD:            sprintf( msg,"Not an executable program/batch/script file");   break;
      case ERROR_IO_FAILURE:           sprintf( msg,"Low level I/O failure, disk or controller");     break;
      default:                         sprintf( msg,"Unknown error%s", psnstore);                     break;
   }
   strcat( message, msg);
   if ((rc & DFS_QUIT) == DFS_QUIT)             // QUIT flag is set
   {
      strcat( message, " (and QUITing)");
   }

   #if defined (DARWIN)                         // MAC OSX extra warning to UNMOUNT for write
   switch (rc)
   {
      case DFS_READ_ONLY:
      case ERROR_INVALID_HANDLE:
         sprintf( msg,"\n%20.20sTo write to a disk, its volumes must be UNMOUNTED!  Use the menu:"
                      "\n%20.20sFile->Device and Volume management -> Unmount volumes from a disk",
                      "", "");
         strcat( message, msg);
         break;

      default:
         break;
   }
   #endif

   RETURN (message);
}                                               // end 'dfsGetRcString'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display DFS return-code
/*****************************************************************************/
void dfsExplainRC
(
   ULONG               rc                       // IN    DFS returncode
)
{
   if ((rc & ~DFS_QUIT) != NO_ERROR)
   {
      TXLN       errormsg;                      // complete message line

      dfsGetRcString( rc, errormsg);
      dfsGuiStdMessage( errormsg);              // TxPrint and possible STDOUT
      TxPrint("\n");
   }
}                                               // end 'dfsExplainRC'
/*---------------------------------------------------------------------------*/

