//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// DFS  utility display services
//
// Author: J. van Wijk
//
// JvW  29-06-2000 Initial version, split off from DFSUTIL.C
// JvW  02-07-2000 Moved sector-display functions from dfs.c
//
// Note: Interface in dfsutil.h

#include <txlib.h>                              // ANSI controls and logging

#include <dfsrgkey.h>                           // Registration interface
#include <dfsver.h>                             // DFS version info
#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfshpfs.h>                            // HPFS structure defs
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsupart.h>                           // Partition definitions
#include <dfsufdsk.h>                           // Fdisk & translation services
#include <dfsufgpt.h>                           // Fdisk GPT functions
#include <dfsutil.h>                            // DFS utility & display functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfstable.h>                           // SLT utility functions
#include <dfsafdsk.h>                           // FDISK/LVM definitions


/*****************************************************************************/
// Print several DFSee status-line items to the current TxPrint destination
/*****************************************************************************/
void dfsTxPrintStatusLine
(
   ULONG               rc                       // IN    dfs return-code
)
{
   TXLN                text;                    // text buffer
   TXLN                tbuf;                    // text buffer
   TXTM                sninfo;                  // text sninfo hint
   size_t              drwidth = 2;
   time_t              tt = time( &tt);         // current date/time

   ENTER();

   dfstGetReadOnly( DFSTORE, text);             // get readonly status text
   sprintf( tbuf, " Store %s%s%s : ", CBG, (DFSTORE) ? (DFSTORE == 1) ? "A" : "B" : "S", CNN);
   strcat(  text, tbuf);
   if (SINF->disknr)
   {
      sprintf( tbuf, "%sDisk : %s%u%s PartId %2.2hu = ",
                      dfsMediaTypeDescr( dfsDid2DiskType(SINF->disknr)),
                      CBC, SINF->disknr, CNN, SINF->partid);
      strcat(  text, tbuf);
   }
   else
   {
      if ((SINF->drive[1] == ':') && (strlen(SINF->drive) == 2))
      {
         strcat( text, "Volume : ");
      }
      else if (strncasecmp( SINF->drive, "/dev/", 5) == 0)
      {
         strcat( text, "Device : ");
      }
      else if ((SINF->drive[0] != '-') && (SINF->drive[0] != ' '))
      {
         strcat( text, "Image  : ");
      }
      else
      {
         strcat( text, "Unused : ");
      }
      drwidth = 18;
   }
   TxPrint("\n%s%s%-*s%s   mode=%s%-5.5s%s", text,
            CBG, drwidth, SINF->drive, CNN,
            CBM,          SINF->afsys, CNN);
   if (nav.this_sninfo & DFSSNINFO)             // sninfo in cyan
   {
      strcpy( sninfo, "^ ");                    // next line arrow hint
      TxPrint("%s%4hx %s", CBC, DFSSNIGET( nav.this_sninfo), CNN);
   }
   else
   {
      strcpy( sninfo, "  ");
      TxPrint("     ");
   }
   strftime( tbuf,  TXMAXTM, "%Y%m%d-%H%M%S ", localtime( &tt));
   if (dfsSlTableStatus( NULL) != SLT_EMPTY)
   {
      dfsSlTableStatusString( text);
   }
   else
   {
      strcpy( text, "");
   }
   TxPrint( "%s %s\n", tbuf, text);

   if (rc == 0)
   {
      sprintf( tbuf, "%s0%s      <Enter>", CBG,     CNN);
   }
   else
   {
      sprintf( tbuf, "%s%-6hd%s= 0x%-4X", CBR, (short) (rc & 0xffff), CNN, rc);
   }
   if (nav.down == L64_NULL)                    // build <Enter> prompt string
   {                                            // followed by the 'specials'
      strcpy( text, "SLT-NextPg   ");
   }
   else
   {
      strcpy(    text, "");
      dfstrX10(  text, "",      nav.down,                   CBG, "   ");
   }
   dfstrX10(     text, "up=",   nav.up,                     CBC, "  ");
   dfstrX10(     text, "this=", nav.this,                   CBM, sninfo);
   dfstrX10(     text, "Base=", dfstLSN2Psn( DFSTORE, 0),   CNG, "  ");
   dfstrX10(     text, "Xtra=", nav.xtra,                   CBY, "\n");
   TxPrint( "RC:%s : %s", tbuf, text);

   VRETURN ();
}                                               // end 'dfsTxPrintStatusLine'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display specified sectortype in ASCII
/*****************************************************************************/
void dfsShowSectorType
(
   BYTE               st                        // IN    sector type
)
{
   TXTM               ascii;

   dfsSectorTypeAsAscii( (BYTE)(st & ~ST__INFO), ascii);
   TxPrint("%s%s", ascii, CNN);
}                                               // end 'dfsShowSectorType'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display all known sectortypes with type-char and description in ASCII
/*****************************************************************************/
void dfsDisplaySectorTypes
(
   void
)
{
   int                 len;
   int                 i;

   ENTER();

   TxPrint("\nDFSee generic sector types, %ssearchable%s or %snot-searchable%s",
                                        CBM,       CNN,   CNG,           CNN);

   for (i = 0, len = strlen(dfsa->BsSectorTypes); i < len; i++)
   {
      dfsSectorTypeInColor(((i % 3) == 0) ? "\n" : "  ", dfsa->BsSectorTypes[i]);
   }

   TxPrint("\n\n%s specific sector types", SINF->afsys);
   for (i = 0, len = strlen(dfsa->FsSectorTypes); i < len; i++)
   {
      dfsSectorTypeInColor(((i % 3) == 0) ? "\n" : "  ", dfsa->FsSectorTypes[i]);
   }
   TxPrint( "\n");
   VRETURN();
}                                               // end 'dfsDisplaySectorTypes'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Create text-array with known sectortypes with type-char and description
/*****************************************************************************/
char **dfsSectorTypes2Text                      // RET   TXlib style text array
(                                               //       or NULL for alloc fail
   void
)
{
   char              **text = NULL;             // function return
   int                 line = 0;                // nr of lines done
   int                 lpsize;                  // nr of line pointers
   ULONG               rc = NO_ERROR;           // allocation result
   int                 len;
   int                 i;
   TXTM                description;             // type description for this type

   ENTER();

   //- allocate one line for each sectortype, plus 9 header and intermediates
   lpsize = strlen( dfsa->BsSectorTypes) + strlen( dfsa->FsSectorTypes) + 9;

   if ((text = TxAlloc( lpsize, sizeof(char *))) != NULL)
   {
      //- allocate one TXTM (64 positions) lines for all except last one
      for (line = 0; line < (lpsize - 1); line++)
      {
         if ((text[ line] = TxAlloc( 1, TXMAXTM + 1)) == NULL)
         {
            rc = TX_ALLOC_ERROR;
            break;
         }
      }
      if (rc == NO_ERROR)                       // text array allocated
      {
         line = 0;
         strcpy( text[ line++], "DFSee generic sector types, recognized in all modes:");
         strcpy( text[ line++], "====================================================");
         line++;                                // skip one line
         for (i = 0, len = strlen( dfsa->BsSectorTypes); i < len; i++, line++)
         {
            BYTE       st    = dfsa->BsSectorTypes[i];
            BYTE       clean = (BYTE)(st & ~ST__INFO);

            dfsSectorTypeAsAscii( clean, description);
            if (TxPrintable( clean))
            {
               sprintf( text[ line], "  '%c' = %s  %s", clean, description, (st < ST__INFO) ? "(searchable)" : "");
            }
            else
            {
               sprintf( text[ line], "  x%2.2hhx = %s  %s", st, description, (st < ST__INFO) ? "(searchable)" : "");
            }
         }

         if (strlen( dfsa->FsSectorTypes) > 0)
         {
            line++;                             // skip one line
            sprintf( text[ line++], "%6s specific sector types recognized:", SINF->afsys);
            sprintf( text[ line++], "========================================");
            line++;                             // skip one line
            for (i = 0, len = strlen( dfsa->FsSectorTypes); i < len; i++, line++)
            {
               BYTE       st    = dfsa->FsSectorTypes[i];
               BYTE       clean = (BYTE)(st & ~ST__INFO);

               dfsSectorTypeAsAscii( clean, description);
               if (TxPrintable( clean))
               {
                  sprintf( text[ line], "  '%c' = %s  %s", clean, description, (st < ST__INFO) ? "(searchable)" : "");
               }
               else
               {
                  sprintf( text[ line], "  x%2.2hhx = %s  %s", st, description, (st < ST__INFO) ? "(searchable)" : "");
               }
            }
         }
      }
      else
      {
         txFreeText( text);                     // free allocated memory
         text = NULL;                           // signal failure to caller
      }
   }
   RETURN (text);
}                                               // end 'dfsSectorTypes2Text'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read. analyse and display sector(s) using LSN addressing
/*****************************************************************************/
ULONG dfsReadAnDisplay
(
   ULN64               lsn,                     // IN    Start LSN
   ULN64               info,                    // IN    FAT entry/EXT Inode/NTFS parent
   BYTE               *stype                    // INOUT sector type found
)
{
   ULONG               rc;

   rc = DFSFNCALL(dfsa->FsDisplayLsnInfo, lsn, info, NULL, NULL);
   if (rc == DFS_PENDING)                       // no special display done
   {
      rc = dfsReadDisplayPhysical( dfstLSN2Psn( DFSTORE, lsn), stype);
   }
   nav.this = lsn;                              // keep it around!
   return (rc);
}                                               // end 'dfsReadAnDisplay'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read. analyse and display sector(s) based on Physical addressing
/*****************************************************************************/
ULONG dfsReadDisplayPhysical
(
   ULN64               psn,                     // IN    Start PSN
   BYTE               *stype                    // INOUT sector type found
)
{
   ULONG               dr  = NO_ERROR;
   BYTE                st  = *stype;
   ULN64               lsn = dfstPsn2LSN( DFSTORE, psn);
   TXTM                lead;

   ENTER();
   dr = dfstReadPsn( DFSTORE, psn, 1, rbuf);    // start with one sector
   if (dr == NO_ERROR)
   {
      nav.this = lsn;
      nav.down = lsn +1;
      st = dfsIdentifySector(lsn, 0, rbuf);
      if ((*stype != 0) && ((*stype | ST__INFO) != ST_MINIM))
      {
         TxPrint("Auto-detected as  : %s%c%s          = %s",
                            CBC, (st & ~ST__INFO), CNN, CNC);
         dfsShowSectorType((BYTE)(st & ~ST__INFO));
         TxPrint("%s\n", CNN);
         st = *stype;
      }
      sprintf( lead, "CRC=%8.8x; LSN : ", dfsCheckSum( rbuf));
      dfsDisplaySnType( lead, lsn, 0, (BYTE)(st & ~ST__INFO));
      if ((*stype | ST__INFO) != ST_MINIM)
      {
         dr = dfsDisplaySector( psn, st, rbuf);
      }
   }
   *stype = st;
   RETURN (dr);
}                                               // end 'dfsReadDisplayPhysical'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display sector(s) based on Physical addressing and known (forced) type
/*****************************************************************************/
ULONG dfsDisplaySector
(
   ULN64              psn,                      // IN    Start PSN
   BYTE               stype,                    // IN    sector/display type
   BYTE              *sect                      // IN    sector data
)
{
   ULONG              dr = NO_ERROR;
   BYTE               st = stype;

   ENTER();
   TRARGS(("psn:0x%llx  type:%2.2hx\n",  psn, stype));
   dr = DFSFNCALL(dfsa->FsDisplaySector, psn, 0, (char *) &st, sect);
   if (dr == DFS_PENDING)                       // not handled
   {
      dr = NO_ERROR;

      if ((dfsa->FsEntry != 0)   &&             // follow-up to bootsector set
          (dfstPsn2LSN( DFSTORE,                // and this is at a bootsector
                        psn) == 0))             // location (unformatted)
      {                                         // Set as DEFAULT now, even for empty
         nav.down = dfsa->FsEntry;              // next sector to visit
      }
      switch (st)                               // searchable ones
      {
         case ST_MASTR:
         case ST_EXTBR: (void) dfsPartitionRecord( sect, st, psn);   break;
         case ST_BOOTR: (void) dfsBootSector( sect);                 break;
         case ST_LVINF: (void) dfsFdskLvmInfo(sect);                 break;
         case ST_LVSIG: (void) dfsFdskLvmSign(sect);                 break;
         case ST_LVREM: (void) dfsFdskLvmSign(sect);                 break;
         case ST_LVDLF: (void) dfsFdskLvmDlf( sect);                 break;
         case ST_LVDLT: (void) dfsFdskLvmDlt( sect);                 break;
         case ST_LVBBF: (void) dfsFdskLvmBBf( sect);                 break;
         case ST_LVBBT: (void) dfsFdskLvmBBt( sect);                 break;
         case ST_LUKSH: (void) dfsFdskLuksHdr(sect);                 break;
         case ST_CORES: (void) dfsFdskCoreStg(sect);                 break;
         case ST_MCDDM: (void) dfsFdskMacDDM( sect);                 break;
         case ST_MCDPM: (void) dfsFdskMacDPM( sect);                 break;
         case ST_GPTHD: (void) dfsGptHdr( sect);                     break;

         case ST_FDISK:
            TxDisplHexDump( sect, SECTORSIZE/8);
            TxPrint( "\nThe partition is UNFORMATTED (cleared by FDISK/LVM)\n");
            break;

         case ST_EMPTY:
            TxDisplHexDump( sect, SECTORSIZE/8);
            TxPrint( "\nThe sector is EMPTY, contains 0x00 bytes only\n");
            break;

         case ST_BADFE:
            TxDisplHexDump( sect, SECTORSIZE/8);
            TxPrint( "\nThe sector is marked as a bad-sector, contains 0xFE bytes only\n");
            break;

         case ST_ULZ16:
         case ST_ULZ32:
            TxDisplHexDump( sect, SECTORSIZE/8);
            TxPrint( "\nStart of DFSee compressed data (IMZ)\n");
            break;

         default:
            switch ((st) | ST__INFO)            // non-searchable ones
            {
               case ST_MINIM:                   // no contents display!
                  break;

               case ST_BOOT2:                   // FAT32 secondary bootsectors
                  {
                     FAT32B2 *fat32 = (FAT32B2 *) sect;

                     TxPrint( "\nFAT32 free clust. : 0x%8.8X  ", fat32->FreeClusters);
                     dfsSiz8( "Freespace sectors : ",          ( fat32->FreeClusters * dfsGetClusterSize()), "");
                     TxPrint( "\n1st Cl to be used : 0x%8.8X  ", fat32->NextSearchCl);
                     dfsX10(  "for sector number : ",  dfsCl2Sn( fat32->NextSearchCl), CNN, "\n\n");
                  }
               case ST_BOOT3:
                  TxDisplHexDump( sect,         SECTORSIZE/16);
                  TxPrint( "..\n");
                  TxDisplHexDump( sect + 0x1e0, SECTORSIZE/16);
                  break;

               case ST_BOOTX:                   // Fsys boot or MBR/EBR
                  TxDisplHexDump( sect, SECTORSIZE);
                  TxPrint( "\nStandard BootRecord interpretation\n");
                  (void) dfsBootSector( sect);  // just show both
                  TxPrint("\nInterpretation as : %s", CNC);
                  dfsShowSectorType((BYTE)(ST_EXTBR));
                  TxPrint("%s\n", CNN);
                  (void) dfsPartitionRecord( sect, st, psn);
                  TxPrint("\nInterpretation as : %s", CNC);
                  dfsShowSectorType((BYTE)(ST_BOOTR));
                  TxPrint("%s\n", CNN);
                  (void) dfsBootSector( sect);  // just show both
                  break;

               case ST_TDATA:
                  TxDisplAscDump( "\n", sect, SECTORSIZE);
                  TxPrint("\n");
                  break;

               case ST_HDATA:
                  TxDisplHexDump( sect, SECTORSIZE/2);
                  break;

               default:
                  TxDisplHexDump( sect, dfsa->defDispSize);
                  break;
            }
            break;
      }
   }
   RETURN (dr);
}                                               // end 'dfsDisplaySector'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Show current sector contents as ascii text
/*****************************************************************************/
ULONG dfsSectorAsAscii
(
   ULN64               sn,                      // IN    lsn of sector
   ULONG               size                     // IN    Size to display
)
{
   ULONG               dr = 0;
   ULONG               done;
   ULN64               this;

   ENTER();
   dfsX10( "Start sector  LSN : ", sn, CNG, "    ");
   TxPrint("  Size to display : %u bytes\n\n", size);

   for ( done = 0,                   this = sn;
        (done < size) && (dr == NO_ERROR) && !TxAbort();
         done += dfsGetSectorSize(), this++)
   {
      dr = dfsRead( this, 1, rbuf);
      if (dr == NO_ERROR)
      {
         TxDisplAscDump( "", rbuf, (ULONG) dfsGetSectorSize());
      }
   }
   nav.down = this;                             // next sector to handle
   RETURN (dr);
}                                               // end 'dfsSectorAsAscii'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Show current sector contents hexadecimal
/*****************************************************************************/
ULONG dfsSectorAsHex
(
   ULN64               sn,                      // IN    lsn of sector
   ULONG               size,                    // IN    Size to display
   BOOL                asUlong                  // IN    format as ULONG
)
{
   ULONG               dr = 0;
   ULONG               done;
   ULN64               this;

   ENTER();
   dfsX10( "Start sector  LSN : ", sn, CNG, "    ");
   TxPrint("  Size to display : %u bytes\n\n", size);

   for ( done = 0,                   this = sn;
        (done < size) && (dr == NO_ERROR) && !TxAbort();
         done += dfsGetSectorSize(), this++)
   {
      if ((done != 0) && (((this - sn) %2) == 0) && !asUlong)
      {
         dfsSz64("Sector nr: ", this - sn, "\n");
      }
      dr = dfsRead( this, 1, rbuf);
      if (dr == NO_ERROR)
      {
         if (asUlong)
         {
            TxDispLongHex( rbuf, (ULONG) dfsGetSectorSize());
         }
         else
         {
            TxDisplayHex("", rbuf, min((ULONG)(size - done),
                         (ULONG) dfsGetSectorSize()), done);
         }
      }
   }
   nav.down = this;                             // next sector to handle
   RETURN (dr);
}                                               // end 'dfsSectorAsHex'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Format for Display attributes (FAT); append to string
/*****************************************************************************/
void dfstrFatAttrib
(
   char               *str,                     // INOUT text string
   BYTE                data                     // IN    data
)
{
   TXTS                attrib;

   if (data == FATTR_LABEL)                     // just a label
   {
      strcpy( attrib, " Label");
   }
   else
   {
      strcpy( attrib, "      ");
      if (data & FATTR_ARCHIVED)  attrib[1] = 'A';
      if (data & FATTR_SYSTEM)    attrib[2] = 'S';
      if (data & FATTR_HIDDEN)    attrib[3] = 'H';
      if (data & FATTR_READONLY)  attrib[4] = 'R';
      if (data & FATTR_DIRECTORY) attrib[5] = 'D';
   }
   strcat( str, attrib);
}                                               // end 'dfstrFatAttrib'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Check if 128-bit UUID or GUID value in 38 character string is valid
/*****************************************************************************/
BOOL dfsUidStringIsValid                        // RET   UUID/GUID string valid
(
   char               *uidStr                   // IN    UID String (38 chars)
)
{
   BOOL                rc = FALSE;
   char                ch;
   int                 i;

   ENTER();
   TRACES(("GUID in: '%s'\n", uidStr));

   if (strlen( uidStr) >= 38)
   {
      for (i = 0, rc = TRUE; (i < 38) && (rc == TRUE); i++)
      {
         ch = uidStr[ i];
         switch (i)
         {
            case  0: if (ch != '{') rc = FALSE; break;
            case  9: if (ch != '-') rc = FALSE; break;
            case 14: if (ch != '-') rc = FALSE; break;
            case 19: if (ch != '-') rc = FALSE; break;
            case 24: if (ch != '-') rc = FALSE; break;
            case 37: if (ch != '}') rc = FALSE; break;
            default:
               if (!(((ch >= '0') && (ch <= '9')) ||
                     ((ch >= 'A') && (ch <= 'F')) ||
                     ((ch >= 'a') && (ch <= 'f')) ))
               {
                  rc = FALSE;
               }
               break;
         }
      }
   }
   BRETURN( rc);
}                                               // end 'dfsUidStringIsValid'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Return (Linux) 128-bit UID value in 38 char (static) string, endian-swapped
/*****************************************************************************/
char *dfsFsUuidValueString                      // RET   Formatted UUID (static)
(
   BYTE               *uuid                     // IN    UUID data, 16 bytes
)
{
   static TXTM         ascii;

   ascii[0] = 0;

   dfstrFsUuidValue( ascii, uuid, TRUE);

   return (ascii);
}                                               // end 'dfsFsUuidValueString'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Return 128-bit UUID/GUID value in 38 char (static) string, no endian-swap
/*****************************************************************************/
char *dfsFsUuidValueNoSwap                      // RET   Formatted UUID (static)
(
   BYTE               *uuid                     // IN    UUID data, 16 bytes
)
{
   static TXTM         ascii;

   ascii[0] = 0;

   dfstrFsUuidValue( ascii, uuid, FALSE);

   return (ascii);
}                                               // end 'dfsFsUuidValueNoSwap'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Format (Linux) 128-bit UID value in 38 character string; append to string
// JvW: 22-03-2012, fixed display order of first 3 groups, little-endian swap
/*****************************************************************************/
void dfstrFsUuidValue
(
   char               *str,                     // INOUT UID text string, 38
   BYTE               *uuid,                    // IN    UID binary, 16 bytes
   BOOL                swap                     // IN    perform endian swaps
)
{
   int                 i;
   TXTS                word;
   BYTE                byte;

   strcat( str, "{");
   for (i = 0; i < FS_UUID_LENGTH; i++)
   {
      if ((i < 8) && swap)                      // first 8 may need swapping
      {
         switch (i)
         {
            default:
            case 0:   byte = uuid[3]; break;    // 1st group little-endian, swap
            case 1:   byte = uuid[2]; break;
            case 2:   byte = uuid[1]; break;
            case 3:   byte = uuid[0]; break;
            case 4:   byte = uuid[5]; break;    // 2nd group little-endian, swap
            case 5:   byte = uuid[4]; break;
            case 6:   byte = uuid[7]; break;    // 3rd group little-endian, swap
            case 7:   byte = uuid[6]; break;
         }
      }
      else                                      // no swap, straight copy
      {
         byte = uuid[i];
      }
      switch (i)
      {
         case  4: case  6: case  8: case  10:      strcat( str, "-");
         default: sprintf( word, "%2.2hhx", byte); strcat( str, word);
      }
   }
   strcat( str, "}");
}                                               // end 'dfstrFsUuidValue'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Convert UUID/GUID string (38) into 16 byte GUID, including byte-swaps
// JvW: 06-02-2018, selectable order first 3 groups, little-endian swap/no-swap
/*****************************************************************************/
ULONG dfsUidStringToBinary                      // RET   conversion result
(
   char               *str,                     // IN    UID text string, 38
   BYTE               *uuid,                    // OUT   UID binary, 16 bytes
   BOOL                swap                     // IN    perform endian swaps
)
{
   ULONG               rc = NO_ERROR;
   int                 i;
   char               *pos = str + 1;           // start position for convert
   TXTS                word;
   BYTE                byte;

   ENTER();

   if (dfsUidStringIsValid( str))
   {
      for (i = 0; i < FS_UUID_LENGTH; i++)
      {
         //- Read the two character corresponding to this byte index
         TxCopy( word, pos, 3);                 // 2 chars plus terminating 0
         sscanf( word, "%hhx", &byte);          // convert to binary

         if ((i < 8) && swap)                   // first 8 may need swapping
         {
            switch (i)
            {
               default:
               case 0:  uuid[3] = byte ; break; // 1st group little-endian, swap
               case 1:  uuid[2] = byte ; break;
               case 2:  uuid[1] = byte ; break;
               case 3:  uuid[0] = byte ; break;
               case 4:  uuid[5] = byte ; break; // 2nd group little-endian, swap
               case 5:  uuid[4] = byte ; break;
               case 6:  uuid[7] = byte ; break; // 3rd group little-endian, swap
               case 7:  uuid[6] = byte ; break;
            }
         }
         else                                   // no swap, straight copy
         {
            uuid[i] = byte;
         }
         switch (i)
         {
            case  3: case  5: case  7: case  9: pos += 3;  break;
            default:                            pos += 2;  break;
         }
      }
   }
   else
   {
      rc = DFS_VALUE_ERROR;
   }
   RETURN (rc);
}                                               // end 'dfsUidStringToBinary'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display ULONG size in 4-hex digits and Dec KiB/MiB/GiB; to TxPrint output
/*****************************************************************************/
void dfsUlSiz4
(
   char               *text,                    // IN    leading string
   ULONG               data,                    // IN    data
   char               *trail                    // IN    trailing text
)
{
   TXTM                string;

   strcpy( string, "");
   dfstrUlSiz4( string, text, data, trail);
   TxPrint( "%s", string);
}                                               // end 'dfsUlSiz4'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Format ULONG size in 4-hex digits and Dec KiB/MiB/GiB; append to string
/*****************************************************************************/
void dfstrUlSiz4
(
   char                *str,                    // OUT   resulting string
   char                *text,                   // IN    leading string
   ULONG                data,                   // IN    data
   char                *trail                   // IN    trailing text
)
{
   TXTM                this;

   if      (data == DFS32MAX)                   // infinity
   {
      sprintf( this, "**** = ");
   }
   else if (data > 0xffff)                      // allow upto 6 digits max
   {
      sprintf( this, "%-6X=",    data);
   }
   else
   {
      sprintf( this, "%4.4X = ", data);
   }
   strcat(    str, text);
   dfstrSize( str, this, data, trail);
}                                               // end 'dfstrUlSiz4'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Format byte size in 13-hex digits + KiB/MiB/GiB/TiB;  to TxPrint (27)
/*****************************************************************************/
void dfsSz64Byte
(
   char               *text,                    // IN    leading string
   ULN64               data,                    // IN    data
   char               *trail                    // IN    trailing text
)
{
   TXLN                string;

   strcpy( string, "");
   TxPrint("%s", dfstrSz64Byte( string, text, data, trail));
}                                               // end 'dfsSz64Byte'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Format byte size in 13-hex digits + KiB/MiB/GiB/TiB; append to string (27)
/*****************************************************************************/
char *dfstrSz64Byte                             // RET   resulting string
(
   char               *str,                     // OUT   resulting string
   char               *text,                    // IN    leading string
   ULN64               data,                    // IN    data
   char               *trail                    // IN    trailing text
)
{
   TXLN                form;

   if (data == DFS64MAX)                        // maximum 64bit value
   {
      sprintf( form, "%s--undefined-or-max-value---%s", text, trail);
      strcat(  str, form);
   }
   else
   {
      //- HEX representation, total 13 positions
      if (data > 0xffffffffffULL)               // larger than 10 digits
      {
         sprintf( form, "%s%13.13llX = ", text, data);
      }
      else                                      // use 0x with 8 HEX digits
      {
         sprintf(form, "%s0x%10.10llX = ", text, data);
      }
      dfstrSz64XiB( str, form, data, 1, trail); // 64bit 11 position format
   }
   return( str);
}                                               // end 'dfstrSz64Byte'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Format sector size in 10-hex digits + KiB/MiB/GiB/TiB;  to TxPrint (24)
/*****************************************************************************/
void dfsSz64Bps
(
   char               *text,                    // IN    leading string
   ULN64               data,                    // IN    data
   USHORT              bps,                     // IN    bytes per sector
   char               *trail                    // IN    trailing text
)
{
   TXLN                string;

   strcpy( string, "");
   TxPrint("%s", dfstrSz64Bps( string, text, data, bps, trail));
}                                               // end 'dfsSz64Bps'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Format sector size in 10-hex digits + KiB/MiB/GiB/TiB; append to string (24)
/*****************************************************************************/
char *dfstrSz64Bps                              // RET   resulting string
(
   char               *str,                     // OUT   resulting string
   char               *text,                    // IN    leading string
   ULN64               data,                    // IN    data
   USHORT              bps,                     // IN    bytes per sector
   char               *trail                    // IN    trailing text
)
{
   TXLN                form;

   if (data == DFS64MAX)                        // maximum 64bit value
   {
      sprintf( form, "%s--undefined-or-max-val--%s", text, trail);
      strcat(  str, form);
   }
   else
   {
      //- HEX representation, total 13 positions
      if (data > (ULN64) 0xffffffff)            // larger than 32-bit
      {
         sprintf( form, "%s%10.10llX = ", text, data);
      }
      else                                      // use 0x with 8 HEX digits
      {
         sprintf(form, "%s0x%8.8llX = ", text, data);
      }
      dfstrSz64XiB( str, form, data, bps, trail); // 64bit 11 position format
   }
   return( str);
}                                               // end 'dfstrSz64Bps'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display (sector) size in KiB/MiB/GiB/TiB; to TxPrint output (8 positions)
/*****************************************************************************/
void dfsSize
(
   char               *text,                    // IN    leading string
   ULONG               data,                    // IN    data (nr of sectors)
   char               *trail                    // IN    trailing text
)
{
   TXLN                string;

   strcpy( string, "");
   TxPrint("%s", dfstrS4XiB( string, text, data, dfsGetSectorSize(), trail));
}                                               // end 'dfsSize'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display value in 8-hex digits and decimal; to TxPrint output
/*****************************************************************************/
void dfsDec8
(
   char               *text,                    // IN    leading string
   ULONG               data,                    // IN    data
   char               *trail                    // IN    trailing text
)
{
   TxPrint("%s0x%08.8X = %9u%s", text, data, data, trail);
}                                               // end 'dfsDec8'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Format value in 8-hex digits and 10 decimals; 23 positions; append to string
/*****************************************************************************/
void dfstrDecX
(
   char               *str,                     // OUT   resulting string
   char               *text,                    // IN    leading string
   ULONG               data,                    // IN    data
   char               *trail                    // IN    trailing text
)
{
   TXTM                form;

   strcat(  str, text);
   sprintf( form, "0x%8.8X = %10u", data, data);
   strcat(  str, form);
   strcat(  str, trail);
}                                               // end 'dfstrDecX'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Format value in 8-hex digits and decimal; append to string
/*****************************************************************************/
void dfstrDec8
(
   char               *str,                     // OUT   resulting string
   char               *text,                    // IN    leading string
   ULONG               data,                    // IN    data
   char               *trail                    // IN    trailing text
)
{
   TXTM                form;

   strcat(  str, text);
   sprintf( form, "0x%8.8X = %9u", data, data);
   strcat( str, form);
   strcat( str, trail);
}                                               // end 'dfstrDec8'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display value in 4-hex digits and decimal; to TxPrint output
/*****************************************************************************/
void dfsDec4
(
   char               *text,                    // IN    leading string
   ULONG               data,                    // IN    data
   char               *trail                    // IN    trailing text
)
{
   TxPrint("%s%04.4X = %-5u%s", text, data, data, trail);
}                                               // end 'dfsDec4'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Format value in 4-hex digits and decimal; append to string
/*****************************************************************************/
void dfstrDec4
(
   char               *str,                     // OUT   resulting string
   char               *text,                    // IN    leading string
   ULONG               data,                    // IN    data
   char               *trail                    // IN    trailing text
)
{
   TXTM                form;

   strcat(  str, text);
   sprintf( form, "%4.4X = %-5u", data, data);
   strcat( str, form);
   strcat(  str, trail);
}                                               // end 'dfstrDec4'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Format 64-bit size in Byte/KiB/MiB/GiB/TiB; append to string (8 positions)
/*****************************************************************************/
char *dfstr64XiB                                // RET   resulting string
(
   char               *str,                     // OUT   resulting string
   char               *text,                    // IN    leading string
   ULN64               data,                    // IN    data (nr of bytes)
   char               *trail                    // IN    trailing text
)
{
   if (data < 0x7fffffff)
   {
      dfstrS4XiB( str, text, (ULONG) data, 1, trail);
   }
   else
   {
      dfstrS4XiB( str, text, (ULONG) (data >> 10), 1024, trail);
   }
   return( str);
}                                               // end 'dfstr64XiB'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display ULONG in decimal DOTTED format up to 4GB, to Tprintf (13 positions)
/*****************************************************************************/
void dfsUlDot13
(
   char               *text,                    // IN    leading string
   ULONG               data,                    // IN    data (nr of bytes)
   char               *trail                    // IN    trailing text
)
{
   TXLN                string;

   strcpy( string, "");
   TxPrint("%s", dfstrUlDot13( string, text, data, trail));
}                                               // end 'dfsUllDot20'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Format ULONG in DOTTED format up to 4GB; append to string (13 positions)
/*****************************************************************************/
char *dfstrUlDot13                              // RET   resulting string
(
   char               *str,                     // OUT   resulting string
   char               *text,                    // IN    leading string
   ULONG               data,                    // IN    data (nr of bytes)
   char               *trail                    // IN    trailing text
)
{
   TXLN                string;

   strcpy( string, "");
   dfstrUllDot20( string, "", (ULN64) data, trail); // 20 char value, plus trail
   strcat( str, text);                          // start with leader text
   strcat( str, string + 7);                    // skip 7 from 20 characters
   return (str);
}                                               // end 'dfstrUlDot13'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display ULL in decimal DOTTED format up to 999Tb, else raw (20 positions)
/*****************************************************************************/
void dfsUllDot20
(
   char               *text,                    // IN    leading string
   ULN64               data,                    // IN    data (nr of bytes)
   char               *trail                    // IN    trailing text
)
{
   TXLN                string;

   strcpy( string, "");
   TxPrint("%s", dfstrUllDot20( string, text, data, trail));
}                                               // end 'dfsUllDot20'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Format ULL in DOTTED format if < 999Tb else raw; app to str (20 positions)
/*****************************************************************************/
char *dfstrUllDot20                             // RET   resulting string
(
   char               *str,                     // OUT   resulting string
   char               *text,                    // IN    leading string
   ULN64               data,                    // IN    data (nr of bytes)
   char               *trail                    // IN    trailing text
)
{
   TXTT                rawnum;                  // decimal number string
   TXTT                dotnum;                  // dotted  number string
   int                 i;
   char               *s;
   BOOL                digit = FALSE;           // no digit done yet

   sprintf( rawnum, "%20llu", data);            // convert to number string

   TRACES(("LLDot20: raw: '%s'\n", rawnum));

   if (data > 999999999999999ULL)
   {
      strcpy( dotnum, rawnum);
   }
   else                                         // insert dots (max 15 digits)
   {
      for (i = 0, s = dotnum; i < 15; i++)      // possible available digits
      {
         if ((i % 3) == 0)                      // every 3 digits, after first digit
         {
            *s++ = (digit) ? '.' : ' ';         // DOT, when in digits
         }
         if (rawnum[i + 5] != ' ')              // handle next digit
         {
            digit = TRUE;
         }
         *s++ = rawnum[i + 5];
      }
      *s = 0;                                   // terminate the string
   }
   strcat( str, text);
   strcat( str, dotnum);
   strcat( str, trail);
   return( str);
}                                               // end 'dfstrUllDot20'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Format (sector) size in a compact floating-point MiB value when possible
/*****************************************************************************/
void dfsSizeMB
(
   USHORT              width,                   // IN    width of field
   ULONG               sectors,                 // IN    number of sectors
   USHORT              bps,                     // IN    bytes per sector
   char               *sb                       // OUT   string buffer
)                                               //       minimum 16 chars
{
   double              size = TXSMIB(sectors, bps);
   TXTS                fstr;

   strcpy( fstr, "");                             // default no contents!
   if (sectors != 0)
   {
      if      (width >= 6)
      {
         sprintf(fstr, "%-8.1lf", size);
      }
      else if (width >= 5)
      {
         sprintf(fstr, "%-7.1lf", size);
      }
      else if (width >= 4)
      {
         sprintf(fstr, "%-6.1lf", size);
      }
      else if ((width >= 3) && (sectors < (2*1024*100)))
      {
         sprintf(fstr, "%-3.1lf",   size);
      }
      else if ((width >= 2) && (sectors < (2*1024*10)))
      {
         sprintf(fstr, "%-2.1lf",   size);
      }
   }
   if (fstr[0] == '0')
   {
      strcpy( sb, &fstr[1]);                    // strip lead-0 on number < 1.0
   }
   else
   {
      if ((strlen(fstr) <= 4) && (fstr[strlen(fstr)-1] == '0'))
      {
         fstr[strlen(fstr)-2] = '\0';           // strip .0 on small numbers
      }
      strcpy( sb, fstr);
   }
}                                               // end 'dfsSizeMB'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display 32bit LSN value in 8-hex digits in color; to TxPrint output
/*****************************************************************************/
void dfsLX8
(
   char               *text,                    // IN    leading string
   ULONG               data,                    // IN    data
   char               *color,                   // IN    color
   char               *trail                    // IN    trailing text
)
{
   TxPrint("%s%s%08.8X%s%s", text, color, data, CNN, trail);
}                                               // end 'dfsLX8'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Format 32bit LSN value in 8-hex digits in color; append to string buffer
/*****************************************************************************/
void dfstrLX8
(
   char               *str,                     // OUT   resulting string
   char               *text,                    // IN    leading string
   ULONG               data,                    // IN    data
   char               *color,                   // IN    color
   char               *trail                    // IN    trailing text
)
{
   TXTM                form;

   sprintf( form, "%s%s%8.8X%s%s", text, color, data, CNN, trail);
   strcat( str, form);
}                                               // end 'dfstrLX8'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display alloc and SLT info for LSN, unless related LSN is same as last
/*****************************************************************************/
BOOL dfsDisplaySltIdInfo                        // OUT   type/contents found
(
   ULN64               sn,                      // IN    lsn to display for
   ULN64              *last                     // INOUT last (don't display
)                                               //             info if same)
{
   ULONG               rc     = NO_ERROR;
   ULN64               sl     = 0;              // sector LSN
   USHORT              sninfo = 0;              // info (FAT dir entry#)
   BYTE                st     = 0;              // sector type

   ENTER();



   dfsX10( "Alloc-bit for LSN : ", sn, CNN, " = ");
   TxPrint("%s\n", (DFSFNCALL(dfsa->FsLsnAllocated,sn,0,NULL,NULL)) ? "Allocated" : "Free");

   if (dfsSlTableStatus( NULL) == SLT_READY)
   {
      if (dfsSlTableFind( sn, &sl, &sninfo, &st, &nav.index)) // found
      {
         dfsX10("SLT searched  LSN : ", sn, CBG, " = ");
         dfsShowSectorType((BYTE)(st & ~ST__INFO));
         TxPrint("\n");
         if (sl != L64_NULL)
         {
            if (!TxaOptUnSet('v'))              // unless contents display suppressed
            {
               if (sl != *last)
               {
                  st = 0;                       // use autodetect
                  rc = dfsReadAnDisplay(sl, sninfo, &st);
               }
               else                             // used in bad-sector scan only!
               {
                  dfsX10("SLT reference LSN : ", sl, CBY, " = same as previous\n");
               }
            }
         }
         else
         {
            TxPrint("SLT reference LSN : none\n");
         }
         *last = sl;                            // remember for next time
      }
      else
      {
         dfsX10("Sector ", sn, "", " not found in SLT, it should be free space.\n");
      }
   }
   RETURN (rc);
}                                               // end 'dfsDisplaySltIdInfo'
/*---------------------------------------------------------------------------*/

