//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// DFS  utility display services, MAP and QUERY section
//
// Author: J. van Wijk
//
// JvW  29-06-2000 Initial version, split off from DFSUTIL.C
// JvW  02-07-2000 Moved sector-display functions from dfs.c
//
// Note: Interface in dfsutil.h

#include <txlib.h>                              // ANSI controls and logging

#include <dfsrgkey.h>                           // Registration interface
#include <dfsver.h>                             // DFS version info
#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfshpfs.h>                            // HPFS structure defs
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsupart.h>                           // Partition definitions
#include <dfsufdsk.h>                           // Fdisk & translation services
#include <dfsutil.h>                            // DFS utility & display functions
#include <dfsspace.h>                           // DFS S_SPACE interfaces
#include <dfsafdsk.h>                           // FDISK/LVM definitions
#include <dfsantfs.h>                           // NTFS interface (ntfs anchor)

#define DFS_REGSTR_MAG1 "for '"                 // Registration MAGIC 1
#define DFS_REGSTR_MAG2 "': "                   // Registration MAGIC 2
#define DFS_REGID_FILE  "dfsregid.txt"
#define DFS_REGID_WIDTH 36

                                                // Disk map definitions
#define DFSM_VERBOSE    1                       // Step size for Verbose
#define DFSM_COMPACT    2                       // Step size for Compact

#define DFSM_INFO       0x000                   // Map first regular info line
#define DFSM_P_ID       0x000                   // Map line type part-id
#define DFSM_SIZE       0x001                   // Map line type area size Mb
#define DFSM_DRIV       0x002                   // Map line type drive-letter
#define DFSM_FREE       0x003                   // Map line type freespace
#define DFSM_FORM       0x004                   // Map line type FS format
#define DFSM_TYPE       0x005                   // Map line type FS type
#define DFSM_LABL       0x006                   // Map line type label
#define DFSM_NORM       0x007                   // Std number of lines
#define DFSM_LDEV       0x007                   // Map line Linux devicename
#define DFSM_LAST       0x008                   // Max number of lines
#define DFSM_PTOP       0x100                   // Map primary top
#define DFSM_LTOP       0x101                   // Map logical top
#define DFSM_LBOT       0x111                   // Map logical bottom
#define DFSM_PBOT       0x110                   // Map primary bottom

#define DFS_RCHAR_LIN  72                       // minimum disk-map line chars
#define DFS_RCHAR_FREE  5                       // minimum char freespace
#define DFS_RCHAR_PART  4                       // minimum chars partitions
#define DFS_RCHAR_FIX   3                       // size of non-variable part


// Calculate relChars and relTotal values for disk map display usage
static void dfsCalcRelativeSizes
(
   void
);

// Show pseudo graphical map of 1 disk with partition and freespace info
static void dfsShowDiskMap
(
   USHORT              disk,                    // IN    disknr
   char                level                    // IN    detail level '-', '+'
);

// Show one line of pseudo graphical partition and freespace info
static void dfsShowDiskMapLine
(
   USHORT              disk,                    // IN    disknr
   USHORT              ml,                      // IN    line identifier
   char                level                    // IN    detail level '-', '+'
);

// Show one Map-line for a freespace area
static void dfsAddDiskMapFree
(
   char               *string,                  // INOUT assembly string
   DFSPARTINFO        *f,                       // IN    freespace area
   DFSDISKINFO        *d,                       // IN    disk information
   USHORT              ml,                      // IN    line identifier
   char                level,                   // IN    detail level '-', '+'
   char               *sb                       // IN    string buffer
);


//- negative RC value, but NOT an automatic quit
#define DFSQUERYUSAGE (((ULONG) -2) & ~DFS_QUIT)

static  char       *dfsQueryUsage[] =
{
   "Usage: QUERY info-specifier [disk | freespace-id | partition-id | driveletter]",
   "",
   " Note: All values are ROUNDED DOWN to fit a 16-bit integer value!",
   " Without parameters, a compact partition list is shown (like 'part -C -m')",
   "",
   " PArts  = # partitions   FRee   = free exists    DIsks  = # disks present",
   " PValid = P# valid = 0   FValid = F# valid = 0   DValid = Disk-nr valid = 0",
   " PLet   = Drive-letter   FLog   = log possible   DLog   = # logical partitions",
   " PPrim  = is a primary   FPrim  = pri possible   DPrim  = # primary partitions",
   " PType  = part type      FType  = freesp type    DTotal = # partitions on disk",
   " PCyls  = # cylinders    FCyls  = # cylinders    DCyls  = # cylinders  on disk",
   " PHeads = # heads        FHeads = # heads        DHeads = # heads",
   " PSects = # sectors      FSects = # sectors      DSects = # sectors/track",
   " PKiB   = P-size in KiB  FKiB   = freesp in KiB  DKiB   = Cylinder size in KiB",
   " PMiB   = P-size in MiB  FMiB   = freesp in MiB  DMiB   = Disk size in MiB",
   " PGiB   = P-size in GiB  FGiB   = freesp in GiB  DGiB   = Disk size in GiB",
   " PDisk  = Disk-number    FDisk  = Disk-number    DFree  = # freespace areas",
   " PFirst = First cyl.     FFirst = First cyl.     DFMiB  = freespace disk in MiB",
   " PEnd   = Last  cyl.     FEnd   = Last  cyl.     DFGiB  = freespace disk in GiB",
   " P-clus = sect/cluster   FId    = freesp id      DBmgr  = IBM BootMgr present",
   " PName  = LVM present    Lvm    = LVM present    DNames = #parts with LVM info",
   " P0mft  = MFT fragments                          DX     = #parts without LVM",
   " POk    = Status 0=clean, 1=unknown 2=dirty      DY     = Large Floppy Format",
   " PInfo  = 1st FS-letter                          DSTyle = Partitioning style",
   " PBtini = BOOT.INI OK=0"
#if defined (DEV32) || defined (WIN32)
                          "  PRemov = On removable   DRemov = Removable disk"
#endif
   , "",
   " MKiB   Minimum  GKiB   Grabable  UKiB         XKiB   Maximum    EKiB   Maximum",
   " MMiB = size of  GMiB = freed on  UMiB = Used  XMiB = freespace  EMiB = expand",
   " MGiB   filesys  GGiB   resizing  UGiB   size  XGiB   (defrag)   EGiB   size",
   "",
   " Cyls   = # of cylinders, current geometry     Version = DFSee major version nr",
   " Heads  = # of heads,     current geometry     VMinor  = DFSee minor version nr",
   " Sects  = # of sectors,   current geometry     RValid  = Registration valid = 0",
   "",
   " Bmgr   = LVM On-BMGR-menu flag (PID/letter)   Inst    = LVM Installable flag",
   NULL
};

#define USE_LAST_OR_P_LETTER            999

/*****************************************************************************/
// Display and return specified info item on disk, partition or freespace area
/*****************************************************************************/
ULONG dfsQueryValue                             // RET   queried value
(
   char               *query,                   // IN    item specification
   char               *param                    // IN    argument
)
{
   ULONG               rc = 0;                  // DOS rc
   DFSPARTINFO        *p  = NULL;               // ptr to partition info
   DFSPARTINFO        *f  = NULL;               // ptr to freespace info
   DFSDISKINFO        *d  = NULL;               // ptr to phys disk info
   USHORT              index = 1;               // index, default 1
   USHORT              disk  = 1;               // disk,  default 1
   USHORT              phDisks;                 // nr of physical disks
   USHORT              nrParts;                 // nr of partitions
   ULONG               value;
   ULONG               baseCyl;                 // first cylinder for part
   ULONG               lastCyl;                 // last  cylinder for part
   BYTE                xtype;                   // part/freespace type
   TXLN                string;                  // DFSee version string
   USHORT              param_value = 0;         // parameter validation value

   ENTER();

   dfsa->regconfirm = TRUE;                     // registration nag on exit

   sscanf( param,"%hu", &param_value);          // numeric value of parameter
   index = param_value;
   if ((index == 0) || (index > USE_LAST_OR_P_LETTER))
   {
      index = USE_LAST_OR_P_LETTER;             // default to last  disk/part
   }
   phDisks = dfsPartitionableDisks();           // read diskinfo en nr of disks
   nrParts = dfsPartitions();                   // get number of partitions

   TRACES(("query:'%s' param:'%s' index:%hu  param_value:%hu\n", query, param, index, param_value));

   switch (toupper(query[0]))
   {
      case 'D':                                 // disks
         if (index > phDisks)                   // auto-select last disk
         {
            index = phDisks;
         }
         disk = index;
         if ((d = dfsGetDiskInfo( disk)) != NULL)
         {
            switch (toupper(query[1]))
            {
               case 'V':                        // validate disk nr
                  if ((param_value > phDisks) || (param_value == 0))
                  {
                     rc = 1;                    // invalid
                  }
                  break;

               case 'F':                        // freespace query for disk
                  switch (toupper(query[2]))
                  {
                     case 'K': rc = (ULONG) (d->freeSects >>  1);             break;
                     case 'M': rc = (ULONG) (d->freeSects >> 11);             break;
                     case 'G': rc = (ULONG) (d->freeSects >> 21);             break;
                     default:  rc = (ULONG)  d->freeSpace;                    break;
                  }
                  break;

               case 'S':                        // Sectors / Style
                  switch (toupper(query[2]))
                  {
                     case 'T': rc = (ULONG)  d->pStyle[0];                    break;
                     default:  rc = (ULONG)  d->geoSecs;                      break;
                  }
                  break;

               case 'B': rc = (ULONG) (d->bmgrPsn  != L64_NULL);              break;
               case 'C': rc = (ULONG)  d->geoCyls;                            break;
               case 'G': rc = (ULONG) (d->sectors >> 21);                     break;
               case 'H': rc = (ULONG)  d->geoHeads;                           break;
               case 'K': rc = (ULONG) (d->cSC     >>  1);                     break;
               case 'L': rc = (ULONG) (d->totalpart - d->primaries);          break;
               case 'M': rc = (ULONG) (d->sectors >> 11);                     break;
               case 'N': rc = (ULONG)  d->lvmPartitions;                      break;
               case 'P': rc = (ULONG)  d->primaries;                          break;
               case 'R': rc = (ULONG)  d->Removable;                          break;
               case 'T': rc = (ULONG)  d->totalpart;                          break;
               case 'X': rc = (ULONG) (d->totalpart - d->lvmPartitions);      break;
               case 'Y': rc = (ULONG) (d->flags & DFS_F_FSYSTEMONLY) ? 1 : 0; break;
               default:  rc = (ULONG)  phDisks; index = 0;                    break;
            }
         }
         else                                   // no disks
         {
            if (toupper(query[1]) == 'V')       // validate disk nr
            {
               rc = 1;                          // invalid
            }
         }
         break;

      case 'P':                                 // partitions
         if (index == USE_LAST_OR_P_LETTER)
         {
            dfsParsePartSpec( param, FDSK_ANY, &p); // try to resolve PID or letter
            if (p != NULL)
            {
               index = p->id;                   // translate back from PID/letter
            }
         }
         if (p == NULL)
         {
            if (index > nrParts)                // auto-select last part
            {
               index = nrParts;
            }
            p = dfsGetPartInfo( index);         // direct PID to info
         }
         if (p != NULL)                         // valid partition now ?
         {
            disk    = p->disknr;
            value   = p->sectors;
            baseCyl = p->basePsn / p->cSC;
            lastCyl = p->lastPsn / p->cSC;
            xtype   = p->partent.PartitionType;

            switch (toupper(query[1]))
            {
               case 'V':                        // validate partition nr
                  if ((param_value > nrParts) || (param_value == 0))
                  {
                     rc = 1;                    // invalid
                  }
                  break;

               case 'O':                        // OK, FS status dirty/clean
                  sprintf( string, "part -q %hu", index);
                  dfsMultiCommand( string, 0, TRUE, FALSE, TRUE);

                  rc = dfsa->FsDirtyStatus;
                  break;

               case '0':                        // NTFS MFT-0 fragmentation
                  if (strncasecmp( p->fsform, "NTFS", 4) == 0)
                  {
                     sprintf(string, "part -q %hu", index);
                     dfsMultiCommand( string, 0, FALSE, FALSE, TRUE);

                     rc = ntfs->MftAll.chunks -1;
                  }
                  else
                  {
                     TxPrint( "Query P0 only supported on NTFS!\n");
                  }
                  break;

               case 'B':                        // WIN BOOTINI primary fat/ntfs
                  if ((p->primary) &&
                      ((strncasecmp( p->fsform, "FAT",  3) == 0) ||
                       (strncasecmp( p->fsform, "NTFS", 4) == 0) ))
                  {
                     sprintf( string, "part -q %hu #bootini", index);
                     rc = dfsMultiCommand( string, 0, TRUE, FALSE, TRUE);
                     if (rc == DFS_NOT_FOUND)
                     {
                        rc = NO_ERROR;          // BOOT.INI not found is OK
                     }
                  }
                  else
                  {
                     TxPrint( "Query PB only supported on primary FAT/NTFS!\n");
                  }
                  break;

               case 'R':                        // removable
                  if ((d = dfsGetDiskInfo( disk)) != NULL)
                  {
                     rc = (ULONG) d->Removable;
                  }
                  break;

               case '-': rc = (ULONG)  p->scluster;                    break;
               case 'C': rc = (ULONG) (lastCyl - baseCyl +1);          break;
               case 'D': rc = (ULONG)  disk;                           break;
               case 'E': rc = (ULONG)  lastCyl;                        break;
               case 'F': rc = (ULONG)  baseCyl;                        break;
               case 'G': rc = (ULONG) (value >> 21);                   break;
               case 'H': rc = (ULONG)  p->geoHeads;                    break;
               case 'I': rc = (ULONG) (toupper(p->fsform[0]));         break;
               case 'K': rc = (ULONG) (value >>  1);                   break;
               case 'L': rc = (ULONG)  p->drive[0];                    break;
               case 'M': rc = (ULONG) (value >> 11);                   break;
               case 'N': rc = (ULONG)  p->lvmPresent;                  break;
               case 'P': rc = (ULONG)  p->primary;                     break;
               case 'S': rc = (ULONG)  p->geoSecs;                     break;
               case 'T': rc = (ULONG)  xtype;                          break;
               default:  rc = (ULONG)  nrParts;             index = 0; break;
            }
         }
         else                                   // no partitions
         {
            if (toupper(query[1]) == 'V')       // validate partition nr
            {
               rc = 1;                          // invalid
            }
         }
         break;

      case 'M':                                 // Resize minimum sizes
      case 'G':                                 // Get/Grab from size
      case 'U':                                 // Used sectors
      case 'X':                                 // Maximum freespace
      case 'E':                                 // Maximum freespace
         if (index > nrParts)                   // auto-select last part
         {
            index = nrParts;
         }
         if ((p = dfsGetPartInfo( index)) != NULL)
         {
            sprintf(string, "part -q -a %hu", index); // select part, + ALLOC map
            dfsMultiCommand( string, 0, FALSE, FALSE, TRUE);

            switch (toupper(query[0]))
            {
               case 'E':                        // expandable size or 0
                  value = dfsa->FsExpandSize;
                  break;

               case 'M':
               case 'G':
                  if (dfsa->FsTruncPoint != 0)
                  {
                     value = dfsa->FsTruncPoint; // calculated minimum
                  }
                  else
                  {
                     value = p->sectors;        // current size
                  }
                  break;

               default:
                  value = dfsa->FsUnallocated;  // calculated freespace
                  break;
            }
            switch (toupper(query[0]))          // adjust to requested size
            {                                   // from inverted-value
               case 'G':
               case 'U':
                  value = p->sectors - value;
                  break;

               default:
                  break;
            }
            switch (toupper(query[1]))
            {
               case 'K': rc = (ULONG) (value >>  1);                   break;
               case 'G': rc = (ULONG) (value >> 21);                   break;
               case 'M':
               default:  rc = (ULONG) (value >> 11);                   break;
            }
         }
         break;

      case 'F':                                 // freespace areas
         if (index > nrParts)                   // auto-select last part
         {
            if (index > (nrParts + phDisks))    // auto-select last disk
            {
               d = dfsGetDiskInfo(phDisks);
            }
            else                                // use specified disk/area
            {
               d = dfsGetDiskInfo((USHORT) (index - nrParts));
            }
            if (d != NULL)
            {
               for (p = d->fspHead; p != NULL; p = p->fspChain)
               {
                  f = p;                        // remember last one ...
                  if (f != NULL)
                  {
                     index = f->id;             // and get the real id
                  }
               }
            }
         }
         else                                   // valid part number
         {
            if ((p = dfsGetPartInfo( index)) != NULL)
            {
               f = p->fspChain;
            }
         }
         if (f != NULL)
         {
            disk    = f->disknr;
            xtype   = f->partent.PartitionType;
            value   = f->sectors;
            baseCyl = f->basePsn / f->cSC;
            lastCyl = f->lastPsn / f->cSC;
            switch (toupper(query[1]))
            {
               case 'F': rc = (ULONG)  baseCyl;                     break;
               case 'E': rc = (ULONG)  lastCyl;                     break;
               case 'C': rc = (ULONG)  lastCyl - baseCyl +1;        break;
               case 'H': rc = (ULONG)  f->geoHeads;                 break;
               case 'S': rc = (ULONG)  f->geoSecs;                  break;
               case 'D': rc = (ULONG)  disk;                        break;
               case 'I': rc = (ULONG)  f->id;                       break;
               case 'T': rc = (ULONG)  xtype;                       break;
               case 'P': rc = (ULONG)((xtype & DFS_FSP_PRIMASK));   break;
               case 'L': rc = (ULONG)((xtype & DFS_FSP_LOGMASK));   break;
               case 'K': rc = (ULONG) (value >>  1);                break;
               case 'M': rc = (ULONG) (value >> 11);                break;
               case 'G': rc = (ULONG) (value >> 21);                break;
               default:  rc = (ULONG)  0;                           break;
            }
         }
         else
         {
            TxPrint( "There is no freespace area with id %02.2hu\n", index);
            if (toupper(query[1]) == 'V')       // when validating ...
            {
               rc = 1;                          // signal invalid
            }
         }
         break;

      case 'R':                                 // reg displayed at startup
         #if defined (REGISTRATION)
            if (dfsRegistrationValid( string, NULL) == FALSE)
            {
               if ((TxaExeSwitch('q')) &&       // quiet mode
                   !txwIsWindow(TXHWND_DESKTOP)) // and not windowed
               {
                  printf( "Search / Check registration, may take some time ...\n");
               }
               else
               {
                  TxPrint( "Search / Check registration, may take some time ...\n");
               }
               TxSleep(2000);                   // 2 seconds key-crack delay
               rc = 1;                          // not registred
            }
         #else
            strcpy( string, "OEM licensed version");
         #endif
         TxPrint( "%s\n", string);
         if (TxaOption(TXA_O_LABEL))            // generate reg-id label
         {
            TXTM       line1;
            TXTM       line2;
            char      *regid = TxaOptStr( TXA_O_LABEL, NULL, DFS_REGID_FILE);
            FILE      *fp;
            char      *s;

            sprintf( line1, "%-*.*s",  DFS_REGID_WIDTH -10,
                                       DFS_REGID_WIDTH -10, DFS_V);
            if ((s = strstr( string,   DFS_REGSTR_MAG1)) != NULL)
            {
               strcpy( line2, s + 5);
               if ((s = strstr( line2, DFS_REGSTR_MAG2)) != NULL)
               {
                  strcat( line1, s + 3);        // add 10-digit id to line1
                  *s = 0;                       // and terminate reg-name
                  line2[DFS_REGID_WIDTH] = 0;   // limit name to label width
               }
            }
            else
            {
               strcpy( line2, "No current registration!");
            }
            if ((fp = fopen( regid, "w")) != NULL)
            {
               fprintf( fp, "%s,%s\n", line1, line2);
               fclose(  fp);
            }
            if (dfsGuiStdOut())                 // to stdout too
            {
               printf( "%s,%s\n", line1, line2);
               fflush( stdout);
            }
         }
         index = 0;
         dfsa->regconfirm = FALSE;              // no nagging on exit
         break;

      case 'V':                                 // reg displayed at startup
         strcpy( string, DFS_V);
         string[5] = '\0';                      // just keep numeric part, upto xx.yy
         switch (toupper(query[1]))
         {
            case 'M': rc = (ULONG) atoi(string + 3);  break; // minor version
            default:  rc = (ULONG) atoi(string);      break; // major version
         }
         index = 0;
         dfsa->regconfirm = FALSE;              // no nagging on exit
         break;

      case 'B':                                 // LVM on BMGR menu flag
      case 'I':                                 // LVM Installable  flag
         dfsParsePartSpec( param, FDSK_ANY, &p);
         if (p != NULL)
         {
            switch (toupper(query[0]))
            {
               case 'B': rc = (ULONG) p->lvm.OnBmMenu;          break;
               case 'I': rc = (ULONG) p->lvm.Installable;       break;
            }
            index = p->id;                      // translate back from PID/letter
            disk  = p->disknr;
         }
         break;

      default:
         index = 0;
         switch (toupper(query[0]))
         {
            case 'C': rc = (ULONG) dfstGeoCylinders( DFSTORE);  break;
            case 'H': rc = (ULONG) dfstGeoHeads(     DFSTORE);  break;
            case 'S': rc = (ULONG) dfstGeoSectors(   DFSTORE);  break;
            case 'L': rc = (ULONG) dfsa->lvmPresent;            break;
            default:
               TxShowTxt( dfsQueryUsage);
               rc = DFSQUERYUSAGE;
               break;
         }
         break;
   }
   if (rc != DFSQUERYUSAGE)
   {
      if (rc > SHRT_MAX)                        // keep value within
      {                                         // 16 bit signed limit
         rc = SHRT_MAX;
      }
      TxPrint( "Value %s%11.11s%s : 0x%4.4hx = %s%4hu%s",
                        CBY, query, CNN, rc, CBC, rc, CNN);
      if (index != 0)
      {
         TxPrint( "  on %s %s%hu%s",
                  (toupper(query[0]) == 'D') ? "disk" :
                  (toupper(query[0]) == 'F') ? "freespace area" : "partition",
                   CBG, index, CNN);
         if (toupper(query[0]) != 'D')          // display disk for part/free
         {
            TxPrint( ", disknumber: %s%hu%s", CBM, disk, CNN);
         }
      }
   }
   TxPrint( "\n");
   RETURN (rc);
}                                               // end 'dfsQueryValue'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Calculate relChars and relTotal values for disk map display usage
/*****************************************************************************/
static void dfsCalcRelativeSizes
(
   void
)
{
   USHORT              phDisks;                 // nr of physical disks
   USHORT              nrParts;                 // nr of partitions
   USHORT              disk;                    // physical disk number
   DFSPARTINFO        *p;                       // ptr to partition info
   DFSPARTINFO        *e;                       // ptr to partition info
   DFSPARTINFO        *b;                       // ptr to biggest area
   DFSDISKINFO        *d;                       // ptr to phys disk info
   DFSDISKINFO        *r;                       // ptr to reference disk info
   USHORT              index;                   // index
   USHORT              chars = 1;               // nr of characters
   USHORT              distr;                   // chars to distribute
   USHORT              minim;                   // minimum chars
   BOOL                TrackZero;               // TrackZero freespace present

   ENTER();

   phDisks = dfsPartitionableDisks();           // read diskinfo en nr of disks
   nrParts = dfsPartitions();                   // get number of partitions

   //- Calculate relative, logarithmic, size for partitions and freespace
   for (disk = 1; disk <= phDisks; disk++)      // visit every physical disk
   {
      d = dfsGetDiskInfo( disk);
      if (d != NULL)
      {
         d->relTotal  = 0;
         d->relFrees  = 0;
         d->relParts  = 0;

         for (p = d->fspHead; p != NULL; p = p->fspChain)
         {
            d->relTotal += p->relSize;          // add each freespace area
            d->relFrees++;
         }

         for (index = 1; index <= nrParts; index++)
         {
            p = dfsGetPartInfo( index);
            if (p->disknr == disk)
            {
               d->relTotal += p->relSize;       // add each partition area
               d->relParts++;
            }
         }
      }
   }

   //- calculate maximum nr of chars for each disk, to show size-difference

   #if defined (LARGE_RELATIVE_DISK_SIZE_STEPS)
   if ((phDisks > 1) && (dfsGetDisplayMargin() > DFS_RCHAR_LIN))
   {
      chars = (dfsGetDisplayMargin() - DFS_RCHAR_LIN) / (phDisks -1);
   }
   #else
      chars = 2;                                // small step, so lager disk graphs
   #endif
   for (disk = 1; disk <= phDisks; disk++)      // visit every physical disk
   {
      d = dfsGetDiskInfo( disk);
      if (d != NULL)
      {
         d->relChars = dfsGetDisplayMargin();   // max value, screen size -1
         for (index = 1; index <= phDisks; index++)
         {
            r = dfsGetDiskInfo( index);
            if (r != NULL)
            {
               if (r->sectors > d->sectors)
               {
                  if (d->relChars > DFS_RCHAR_LIN)
                  {
                     d->relChars -= chars;
                  }
               }
            }
         }
      }
   }

   //- Calculate nr of chars for partition/freespace, and total for disk
   for (disk = 1; disk <= phDisks; disk++)      // visit every physical disk
   {
      d = dfsGetDiskInfo( disk);
      if (d != NULL)
      {
         b = NULL;

         TrackZero = ((d->relFrees) &&
                      (d->fspHead->partent.PartitionType == DFS_FSP_ZERO));

         chars = DFS_RCHAR_FIX;                 // total chars used in display
         minim = DFS_RCHAR_FIX +                // fixed
                (DFS_RCHAR_PART * d->relParts) + // partitions
                (DFS_RCHAR_FREE * d->relFrees); // freespace
         if (d->relParts > 99)                  // there are 3-digit numbers
         {
            minim += (d->relParts -99);         // extra space needed
         }
         if (TrackZero)
         {
            minim -= (DFS_RCHAR_FREE -1);       // correct for 1-char track-0
         }
         if (minim < d->relChars)               // spare-room to distribute
         {
            distr = d->relChars - minim;
         }
         else
         {
            distr = 0;                          // no spare room available
         }
         for (p = d->fspHead; p != NULL; p = p->fspChain)
         {
            if ((TrackZero) && (p == d->fspHead))
            {
               p->relChars = 1;
            }
            else
            {
               p->relChars = DFS_RCHAR_FREE;
               if (distr > 0)                   // room to distribute ?
               {
                  p->relChars += (USHORT) ((p->relSize * distr) / d->relTotal);
               }
            }
            chars  += p->relChars;              // total chars in use now
            if ((b == NULL) || (b->relSize < p->relSize))
            {
               b = p;                           // remember largest area
            }
         }

         for (index = 1; index <= nrParts; index++)
         {
            p = dfsGetPartInfo( index);
            if (p->disknr == disk)
            {
               p->relChars = DFS_RCHAR_PART;
               if (p->id > 99)                  // is a 3-digit number
               {
                  p->relChars++;
               }
               if (distr > 0)                   // room to distribute ?
               {
                  p->relChars += (USHORT) ((p->relSize * distr) / d->relTotal);
               }
               chars   += p->relChars;          // total chars in use now
               if ( (b == NULL) ||
                   ((b->lastPsn - b->basePsn) <
                    (p->lastPsn - p->basePsn)))
               {
                  b = p;                        // remember largest area
               }
            }
         }

         //- Expand relChars of largest area to fill the calculated line-length
         if ((b != NULL) && (chars < d->relChars))
         {
            b->relChars += (d->relChars - chars);
         }
         else
         {
            d->relChars = chars;                // real used size in chars
         }
      }
   }

   //- Calculate total relChars for the first Extended partition
   for (disk = 1; disk <= phDisks; disk++)      // visit every physical disk
   {
      d = dfsGetDiskInfo( disk);
      if (d != NULL)
      {
         if (d->ebrHead != NULL)
         {
            index = 0;
            d->ebrHead->relChars = 0;
            for (e = d->ebrHead; e; e = e->ebrChain)
            {
               if (e->id != 0)                  // linked to partition
               {
                  p = dfsGetPartInfo( e->id);
                  if ((p->fspChain != NULL) && (index != 0))
                  {
                     d->ebrHead->relChars += p->fspChain->relChars;
                  }
                  index++;
                  d->ebrHead->relChars += p->relChars;
               }
            }
         }
      }
   }
   VRETURN();
}                                               // end 'dfsCalcRelativeSizes'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Show pseudo graphical map of disk(s) with partition and freespace info
/*****************************************************************************/
void dfsShowMap
(
   USHORT              dnr,                     // IN    disknr, 0 == all disks
   char                level                    // IN    detail level '-', '+'
)
{
   USHORT              phDisks;                 // nr of physical disks
   USHORT              disk;

   ENTER();

   phDisks = dfsPartitionableDisks();           // read diskinfo en nr of disks
   dfsCalcRelativeSizes();
   if ((dnr == 0) || (dnr == FDSK_ANY))
   {
      for (disk = 1; disk <= phDisks; disk++)
      {
         dfsShowDiskMap( disk, level);
      }
   }
   else
   {
      if (dnr <= phDisks)
      {
         dfsShowDiskMap( dnr, level);
      }
      else
      {
         TxPrint("Invalid disk nr   : %u\n", dnr);
      }
   }
   VRETURN();
}                                               // end 'dfsShowMap'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Show pseudo graphical map of 1 disk with partition and freespace info
/*****************************************************************************/
static void dfsShowDiskMap
(
   USHORT              disk,                    // IN    disknr
   char                level                    // IN    detail level '-', '+'
)
{
   USHORT              step;
   USHORT              last;
   USHORT              line;                    // line index
   USHORT              width;
   DFSDISKINFO        *d;                       // ptr to phys disk info
   TX2K                str;                     // string assembly buffer

   ENTER();

   d = dfsGetDiskInfo( disk);
   if ((d != NULL) && (d->OpenError == FALSE))
   {
      TXTM     Device;
      USHORT   fixed;

      last = (level == '+') ? DFSM_LAST    : DFSM_NORM;
      step = (level == '+') ? DFSM_VERBOSE : DFSM_COMPACT;

      width = min( d->relChars, DFS_SCROLL_W -1) -2; // not wider than buffer

      TRACES(("disk nr: %hu  calculated width: %hu\n", disk, width));

      TxStrip( Device, d->UnixDeviceDisk, ' ', ' '); // copy and strip spaces
      fixed = strlen( d->DiskName) + strlen( Device) + 22;
      sprintf( str, "\n%s[%s%s %sdisk %2hu%s][%s%s%s][%s%s%s]",
               CBWnZ, CBMnZ, d->pStyle, CBCnZ, disk,      CBWnZ,
               CNYnZ, Device, CBWnZ, CBCnZ, d->DiskName,  CBWnZ);
      if (fixed < width)
      {
         dfsRstr( str, "", "", "", width - fixed, "", "");
      }
      TxPrint("%s%s\n", str, CNN);
      if (level != '-')
      {
         dfsShowDiskMapLine( disk, DFSM_PTOP, level);
         dfsShowDiskMapLine( disk, DFSM_LTOP, level);
      }
      for (line = DFSM_INFO; line < last; line += step)
      {
         dfsShowDiskMapLine( disk, line, level);
      }
      if (level != '-')
      {
         dfsShowDiskMapLine( disk, DFSM_LBOT, level);
         dfsShowDiskMapLine( disk, DFSM_PBOT, level);
      }
      strcpy(  str, "");
      dfsRstr( str, CBWnZ, "", "", width, "", CNN);
      TxPrint("%s\n", str);
   }
   VRETURN();
}                                               // end 'dfsShowDiskMap'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Show one line of pseudo graphical partition and freespace info
/*****************************************************************************/
static void dfsShowDiskMapLine
(
   USHORT              disk,                    // IN    disknr
   USHORT              ml,                      // IN    line identifier
   char                level                    // IN    detail level '-', '+'
)
{
   TX2K                s1;                      // string 1, screen-width + ANSI!
   TX2K                s2;                      // string 2, screen-width + ANSI!
   USHORT              nrParts;                 // nr of partitions
   USHORT              index;                   // partition index
   DFSDISKINFO        *d;                       // ptr to phys disk info
   DFSPARTINFO        *p;                       // ptr to partition info
   DFSPARTINFO        *f = NULL;                // related freespace area
   USHORT              partype;
   ULONG               sectors;
   BYTE                bg = CcZ;                // background color
   BYTE                fg = CcW;                // foreground color
   USHORT              width;
   USHORT              chars;                   // character width used sofar
   TX4K                str;                     // string assembly buffer
                                                // can hold 96 partitions, with much ANSI
   ENTER();

   nrParts = dfsPartitions();                   // get number of partitions
   d = dfsGetDiskInfo( disk);
   if (d != NULL)
   {
      BOOL             lFlop = ((d->flags & DFS_F_FSYSTEMONLY) != 0);
      BOOL             crypt = ((d->flags & DFS_F_CRYPTO_DISK) != 0);

      switch (ml)
      {
         case DFSM_P_ID:  strcpy( s1,
                            (lFlop) ? " " :
                            (crypt) ? "h" :
                                      "m");     break;
         case DFSM_DRIV:  strcpy( s1,
                            (crypt) ? "d" :
                                      "b");     break;
         case DFSM_FORM:  strcpy( s1, "r");     break;
         default:         strcpy( s1, " ");     break;
      }
      sprintf( str, "%s%s%s", CBWnZ, s1, CNN);
      for (index = 1, chars = 2; index <= nrParts; index++)
      {
         p = dfsGetPartInfo( index);
         if (p && (p->disknr == disk))          // partition on current disk
         {
            if (p->fspChain != NULL)            // leading freespace area
            {
               f = p->fspChain;                 // last shown freespace area
               if ((f->partent.PartitionType == DFS_FSP_LOG) &&
                   ((ml == DFSM_PTOP) || (ml == DFSM_PBOT)))
               {
                  dfsRstr( str, CBYnZ, "", "", f->relChars, "", CNN);
               }
               else
               {
                  dfsAddDiskMapFree( str, f, d, ml, level, s1);
               }
               chars += f->relChars;              //- chars used sofar
            }
            chars += p->relChars;                 //- chars used sofar

            TRACES(("PID: %hu  chars: %hu  strlen-str: %d\n", p->id, chars, strlen(str)));

            if ((chars > DFS_SCROLL_W   -  9) ||  //- will not fit in scrollbuffer
                (strlen(str) > (TXMAX4K -199) ))  //- or in the TX print buffer
            {                                     //- (at least 6 ANSI strings to come)
               break;                             //- just use assembled string
            }                                     //- upto this size ...
            sectors = p->sectors;
            partype = p->partent.PartitionType;
            if (partype < 0x20)                 // treat hidden as base-type
            {
               partype &= ~DFS_P_PHIDDEN;
            }
            switch (partype)
            {
               case DFS_P_FAT12     : bg = CcG;                     break;
               case DFS_P_XENIX_S   : bg = CcR;                     break;
               case DFS_P_XENIX_U   : bg = CcR;                     break;
               case DFS_P_FAT16     : bg = CcG;                     break;
               case DFS_P_FAT16X    : bg = CcG;                     break;
               case DFS_P_BIGDOS    : bg = CcG;                     break;
               case DFS_P_BOOTMGR   : bg = CcI;                     break;
               case DFS_P_FAT32     : bg = CcY;                     break;
               case DFS_P_FAT32X    : bg = CcY;                     break;
               case DFS_P_WARP_LVM  : bg = CcY;                     break;
               case DFS_P_SWAPSOLAR : bg = CcI;                     break;
               case DFS_P_LINUX_LVM : bg = CcY;                     break;
               case DFS_P_BEOS_FS   : bg = CcY;                     break;
               case DFS_P_EFI_ESP   : bg = CcY;                     break;
               case DFS_P_EFI_GPT   : bg = CcR;                     break;
               case DFS_P_WIN2XPLDM : bg = CcR;                     break;
               case DFS_P_HIBERNATE : bg = CcY;                     break;
               case DFS_P_BIGDOS_VS : bg = CcG;                     break;
               case DFS_P_IFS_MIRR  : bg = CcC;                     break;
               case DFS_P_BIGDOSCVS : bg = CcG;                     break;
               case DFS_P_IFS_DISAB : bg = CcC;                     break;
               case DFS_P_MACXHFSP  : bg = CcG;                     break;
               case DFS_P_MACXBOOT  : bg = CcR;                     break;
               case DFS_P_PS2SYST   : bg = CcY;                     break;

               case DFS_P_IFSHIDDEN :
               case DFS_P_INST_FS   :           // NTFS / JFS / HPFS, different colors
                  bg = (p->fsform[0] == 'N') ? CcR :
                       (p->fsform[0] == 'H') ? CcI : CcC;
                  break;

               case DFS_P_LINUXRAID  :           // Linux RAID
               case DFS_P_LINUXNATV  :           // Linux native, possibly LUKS encrypted
                  bg = (p->fsform[0] == 'E') ? CcR : CcC;
                  break;

               case DFS_P_MACXDATA  :           // Core-storage, HFS / FileVault, different colors
                  bg = (p->fsform[0] == 'H') ? CcG : CcC;
                  break;

               default:
                  bg = CcM;
                  break;
            }
            #if defined (USEWINDOWING) && defined (WIN32) // only on Windows, non-OEM
               if (txwIsWindow( TXHWND_DESKTOP) == FALSE) // when NOT windowed
               {
                  if (bg == CcI)                // don't use 'bright black' (undefined ANSI seq!)
                  {
                     bg = CcC;                  // translate to Cyan bg
                  }
               }
            #endif

            width = p->relChars -2;
            if (p->primary)
            {
               switch (ml)
               {
                  case DFSM_PTOP:
                     dfsRstr( str, CBYnW, "", "", width, "", CNN);
                     break;

                  case DFSM_PBOT:
                     dfsRstr( str, CBYnW, "", "", width, "", CNN);
                     break;

                  default:
                     fg = (ml & 0x001) ? CcY : CcW;
                     switch (ml)
                     {
                        case DFSM_P_ID:  sprintf(s1, "%u", p->id);          break;
                        case DFSM_FORM:  strcpy( s1, p->fsform);            break;
                        case DFSM_LABL:  strcpy( s1, p->plabel);            break;
                        case DFSM_DRIV:  strcpy( s1, p->drive);
                           if (d->Removable)
                           {
                              strcat( s1, "    Removable");
                           }
                           break;

                        case DFSM_FREE:
                           dfsSizeMB( width, p->secFree, p->bpsector, s1);
                           break;

                        case DFSM_SIZE:
                           dfsSizeMB( width, sectors, p->bpsector, s1);
                           break;

                        case DFSM_TYPE:
                           sprintf(s1, "%2.2x", p->partent.PartitionType);
                           break;

                        case DFSM_LDEV:         // Unix devicename
                           if (width >= strlen( p->UnixDevicePart))
                           {
                              strcpy( s1, p->UnixDevicePart);
                           }
                           else
                           {
                              strcpy( s1, p->UnixDevicePart + (strlen( p->UnixDevicePart) - width));
                           }
                           break;

                        default:
                           strcpy( s1, " ");
                           break;
                     }
                     sprintf( s2, "%s%s%s%-*.*s%s%s%s",
                               CBYnW, CNN, ansi[Ccol((fg|CcI),bg)],
                               width, width, s1, CNN, CBYnW, CNN);
                     strcat( str, s2);
                     TRACES(("s2 length: %u = '%s', length str: %u\n",
                              strlen( s2), s2, strlen(str)));
                     break;
               }
            }
            else
            {
               switch (ml)
               {
                  case DFSM_PTOP:
                  case DFSM_PBOT:
                     dfsRstr( str, CBYnZ, "",  "", p->relChars, "",  CNN);
                     break;

                  case DFSM_LTOP:
                     dfsRstr( str, CBYnW, "", "", width,       "", CNN);
                     break;

                  case DFSM_LBOT:
                     dfsRstr( str, CBYnW, "", "", width,       "", CNN);
                     break;

                  default:
                     fg = (ml & 0x001) ? CcY : CcW;
                     switch (ml)
                     {
                        case DFSM_P_ID:  sprintf(s1, "%u", p->id);          break;
                        case DFSM_FORM:  strcpy( s1, p->fsform);            break;
                        case DFSM_LABL:  strcpy( s1, p->plabel);            break;
                        case DFSM_DRIV:  strcpy( s1, p->drive);
                           if (d->Removable)
                           {
                              strcat( s1, "  Removable");
                           }
                           break;

                        case DFSM_FREE:
                           dfsSizeMB( width, p->secFree, p->bpsector, s1);
                           break;

                        case DFSM_SIZE:
                           dfsSizeMB( width, sectors,    p->bpsector, s1);
                           break;

                        case DFSM_TYPE:
                           sprintf(s1, "%2.2x", p->partent.PartitionType);
                           break;

                        case DFSM_LDEV:         // Linux devicename
                           if (width >= strlen( p->UnixDevicePart))
                           {
                              strcpy( s1, p->UnixDevicePart);
                           }
                           else
                           {
                              strcpy( s1, p->UnixDevicePart + (strlen( p->UnixDevicePart) - width));
                           }
                           break;

                        default:
                           strcpy( s1, " ");
                           break;
                     }
                     sprintf( s2, "%s%s%s%-*.*s%s%s%s",
                               CBYnW, CNN, ansi[Ccol((fg|CcI),bg)],
                               width, width, s1, CNN, CBYnW, CNN);
                     strcat( str, s2);
                     TRACES(("s2 length: %u = '%s', length str: %u\n", strlen( s2), s2, strlen(str)));
                     break;
               }
            }
         }
      }
      if (f != NULL)                            // at least one area shown
      {
         if (f->fspChain != NULL)               // not last freespace area
         {
            dfsAddDiskMapFree( str, f->fspChain, d, ml, level, s1);
         }
      }
      else if (d->fspHead != NULL)              // one and only (whole disk)
      {
         dfsAddDiskMapFree( str, d->fspHead, d, ml, level, s1);
      }
      TxPrint( "%s%s%s\n", str, CBW, CNN);     // print the assembled string
   }
   VRETURN();
}                                               // end 'dfsShowDiskMapLine'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Show one Map-line for a freespace area, or whole-disk large floppy format
/*****************************************************************************/
static void dfsAddDiskMapFree
(
   char               *string,                  // INOUT assembly string
   DFSPARTINFO        *f,                       // IN    freespace area
   DFSDISKINFO        *d,                       // IN    disk information
   USHORT              ml,                      // IN    line identifier
   char                level,                   // IN    detail level '-', '+'
   char               *sb                       // IN    string buffer
)
{
   USHORT              width = f->relChars -2;
   TX1K                s2;                      // 2nd assembly part
   BOOL                noMbr = ((d->pStyle[0] != DFSP_MBR) &&
                                (d->pStyle[0] != DFSP_GPT));

   strcpy( sb, "");
   switch (ml)
   {
      case DFSM_P_ID:                           // will be the first one called
         if      (d->pStyle[0] == DFSP_FSO)
         {
            strcpy( sb, "FileSystem ONLY, large floppy!");
         }
         else if (d->pStyle[0] == DFSP_APM)
         {
            strcpy( sb, "Apple Partition Map, possible disk Image");
         }
         else if (d->pStyle[0] == DFSP_CRP)
         {
            strcpy( sb, "Whole disk is encrypted!");
         }
         else
         {
            sprintf(sb, "%u", f->id);
         }
         break;

      case DFSM_FREE:
         if (noMbr)                             // no freespace
         {
            break;
         }
      case DFSM_SIZE:
         dfsSizeMB( width, f->lastPsn - f->basePsn +1, f->bpsector, sb);
         break;

      case DFSM_DRIV:
         if (noMbr)
         {
            strcpy( sb, d->ddrive);
            if (d->Removable)
            {
               strcat( sb, "    Removable");
            }
         }
         break;

      case DFSM_LABL:
         if (noMbr)
         {
            strcpy( sb, d->dlabel);
         }
         break;

      case DFSM_FORM:
         if (noMbr)
         {
            strcpy( sb, d->fsform);             // filesystem name
         }
         else
         {
            strcpy( sb, f->descr);              // "FreeSpace ..." or "MBR + GPT ..."
         }
         break;

      case DFSM_LBOT:                           // for Freespace, repeat from 7th char
         if ((!noMbr) && (strlen(f->descr) > 6) && (width < 6))
         {
            strcpy( sb, f->descr + 6);          // "ace ..." or "GPT ..."
         }
         break;

      case DFSM_PBOT:                           // for Freespace, repeat from 11th char
         if ((!noMbr) && (strlen(f->descr) > 10) && (width < 6))
         {
            strcpy( sb, f->descr + 10);         // "..." or "PTArray"
         }
         break;

      default:
         break;
   }
   sprintf( s2, "%s", CNWnM);
   strcat( string, s2);
   if (f->relChars > 1)
   {
      sprintf( s2, "%s%-*.*s%s", (ml & 0x001) ? (noMbr) ? CBWnG : CBWnM :
                                                 (noMbr) ? CBYnG : CBYnM,
                               width, width, sb, (noMbr) ? CNWnG : CNWnM);
      strcat( string, s2);
      TRACES(("s2 length: %u = '%s', length str: %u\n", strlen( s2), s2, strlen(string)));
   }
   strcat( string, CNN);
}                                               // end 'dfsAddDiskMapFree'
/*---------------------------------------------------------------------------*/

