//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
// DFS  utility display services, GEO section
//
// Author: J. van Wijk
//
// JvW  29-06-2000 Initial version, split off from DFSUTIL.C
// JvW  02-07-2000 Moved sector-display functions from dfs.c
//
// Note: Interface in dfsutil.h

#include <txlib.h>                              // ANSI controls and logging

#include <dfsrgkey.h>                           // Registration interface
#include <dfsver.h>                             // DFS version info
#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfsmedia.h>                           // Partitionable Media manager
#include <dfshpfs.h>                            // HPFS structure defs
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsupart.h>                           // Partition definitions
#include <dfsufdsk.h>                           // Fdisk & translation services
#include <dfsutil.h>                            // DFS utility & display functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfstable.h>                           // SLT utility functions
#include <dfsafdsk.h>                           // FDISK/LVM definitions


/*****************************************************************************/
// Translate LSN in 4 possible formats, using current GEO, 1 line display
/*****************************************************************************/
void dfsDisplayTransLsn
(
   char               *lead,                    // IN    leading string (20)
   ULN64               lsn                      // IN    LSN value
)
{
   dfsGeoDispTransLsn( lead, dfstGeoHeads(   DFSTORE),
                             dfstGeoSectors( DFSTORE), lsn);
}                                               // end 'dfsDisplayTransLsn'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate LSN in 4 possible formats, using given GEO, 1 line display
/*****************************************************************************/
void dfsGeoDispTransLsn
(
   char               *lead,                    // IN    leading string (20)
   ULONG               geoHeads,                // IN    geo heads
   ULONG               geoSecs,                 // IN    geo sectors
   ULN64               lsn                      // IN    LSN value
)
{
   dfsGeoDispTransPsn( lead, geoHeads, geoSecs, dfstLSN2Psn( DFSTORE, lsn));
}                                               // end 'dfsGeoDispTransLsn'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate PSN in 4 possible formats, using given GEO, 1 line display
// 20170602: Uses first character after 'lead' for minus sign, plus max 10 HEX
/*****************************************************************************/
void dfsGeoDispTransPsn
(
   char               *lead,                    // IN    leading string (20)
   ULONG               geoHeads,                // IN    geo heads
   ULONG               geoSecs,                 // IN    geo sectors
   ULN64               psn                      // IN    PSN value
)
{
   ULN64               lsn = dfstPsn2LSN( DFSTORE, psn); // can be negative!
   ULONG               c,h,s;                   // chs numbering
   USHORT              sc;                      // sector+cylinder value

   dfsSX11( lead, lsn, "", ", ");
   dfsGeoPsn2Chs( psn, geoHeads, geoSecs, &c, &h, &s);
   sc = DFSCOMBINE(s,c);                        // make combined sec-cyl value
   TxPrint("%sPte %2.2x %2.2x %2.2x%s =", CNC, h, LOBYTE(sc), HIBYTE(sc), CNN);
   TxPrint("%s CHS:%8u %3u %3u%s", CNY, c,  h,  s, CNN);
   dfsX10( " = ", psn, CNG, "\n");

}                                               // end 'dfsGeoDispTransPsn'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display single line with  LSN, determined sector-type, CHS and PSN
/*****************************************************************************/
void dfsDisplayLsnTypeChsPsn
(
   char               *lead,                    // IN    leading text
   ULN64               lsn,                     // IN    Start PSN
   USHORT              info                     // IN    additonal info (entry)
)
{
   dfsGeoDispLsnTypeChsPsn( lead, dfstGeoHeads(   DFSTORE),
                                  dfstGeoSectors( DFSTORE), lsn, info);
}                                               // end 'dfsDisplayLsnTypeChsPsn'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display single line with  LSN, determined sector-type, CHS and PSN, use GEO
/*****************************************************************************/
void dfsGeoDispLsnTypeChsPsn
(
   char               *lead,                    // IN    leading text
   ULONG               geoHeads,                // IN    geo heads
   ULONG               geoSecs,                 // IN    geo sectors
   ULN64               lsn,                     // IN    Start PSN
   USHORT              info                     // IN    additonal info (entry)
)
{
   ULONG               rc = NO_ERROR;
   BYTE                st = ST_UDATA;           // sector type
   BYTE               *sb;                      // sector buffer

   if ((sb = TxAlloc(1, dfsGetSectorSize())) != NULL)
   {
      rc = dfsRead( lsn, 1, sb);
      if (rc == NO_ERROR)
      {
         st = dfsIdentifySector( lsn, 0, sb);   // determine sector type
      }
      TxFreeMem(sb);
   }
   dfsGeoDispSnType( lead, dfstGeoHeads(   DFSTORE),
                           dfstGeoSectors( DFSTORE), lsn, info, st);
}                                               // end 'dfsGeoDispLsnTypeChsPsn'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display single line with  sector(s) based on Physical addressing
/*****************************************************************************/
void dfsDisplaySnType
(
   char               *lead,                    // IN    leading text
   ULN64               lsn,                     // IN    Start PSN
   USHORT              info,                    // IN    additonal info (entry)
   BYTE                st                       // IN    type of sector
)
{
   dfsGeoDispSnType( lead, dfstGeoHeads(   DFSTORE),
                           dfstGeoSectors( DFSTORE), lsn, info, st);
}                                               // end 'dfsDisplaySnType'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display single line with  sector(s) based on Physical addressing, using GEO
/*****************************************************************************/
void dfsGeoDispSnType
(
   char               *lead,                    // IN    leading text
   ULONG               geoHeads,                // IN    geo heads
   ULONG               geoSecs,                 // IN    geo sectors
   ULN64               lsn,                     // IN    Start PSN
   USHORT              info,                    // IN    additonal info (entry)
   BYTE                st                       // IN    type of sector
)
{
   ULN64               psn;
   ULONG               c,h,s;                   // chs numbering
   TXTM                tbuf;                    // temporary buffer
   TXLN                text;                    // accummulating text buffer
   USHORT              ClusterSize;             // current clustersize in sectors

   ENTER();

   strcpy(    text, "");
   dfstrX10(  text, lead, lsn, CBM,"");

   if (info & DFSSNINFO)                        // with extra info ?
   {
      sprintf( tbuf, "   info value 0x%s%4.4x", CBC, DFSSNIGET(info));
   }
   else
   {
      strcat( text, " = ");
      strcat( text, CBM);
      dfsSectorTypeAsAscii((BYTE)(st & ~ST__INFO), tbuf);
   }
   strcat( text, tbuf);
   strcat( text, CNN);

   psn = dfstLSN2Psn( DFSTORE, lsn);
   if (SINF->disknr)
   {
      dfsGeoPsn2Chs( psn, geoHeads, geoSecs, &c, &h, &s);
      sprintf( tbuf, " CHS%s%7u %3u %3u%s", CNY, c,  h,  s, CNN);
      strcat(  text, tbuf);
   }
   else
   {
      strcat(  text, "    absolute sector");
   }
   dfstrX10(   text, " ", psn, CNG, "");
   if ((ClusterSize = dfstGetClusterSize( DFSTORE)) != 1)
   {
      dfstrX10(   text, " Block:", dfsSn2Cl(lsn), CNM, "");
   }
   TxPrint( "%s\n", text);
   VRETURN();
}                                               // end 'dfsGeoDispSnType'
/*---------------------------------------------------------------------------*/

