//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// APFS (container) dump & display Analysis functions
//
// Author: J. van Wijk
//
// JvW  24-01-2018 Initial version, derived from High Performance File System
//
#ifndef    DFSUAPFS
   #define DFSUAPFS

#include <dfsaapfs.h>                           // required for internal typedefs

// Find Leaf-Node LSN+Index for specified path, starting at the root-directory
ULONG dfsApfsFindPath
(
   ULN64               loud,                    // IN    Show progress
   ULN64               d2,                      // IN    dummy
   char               *path,                    // IN    path specification
   void               *vp                       // OUT   found dir/file NODE+index
);

// Identify if FS structures for APFS are present (superblock in first sector)
ULONG dfsApfsFsIdentify
(
   S_BOOTR            *boot,                    // IN    Bootsector ref or NULL
   DFSPARTINFO        *p                        // INOUT partition info
);

// Display file allocation and path info for LSN + index
ULONG dfsApfsFileInfo                            // RET   LSN is valid NODE
(
   ULN64               leafLsn,                 // IN    Leaf-node sectornumber
   ULN64               leafIdx,                 // IN    Leaf-node record index
   char               *select,                  // IN    file selection string
   void               *param                    // INOUT leading text / path
);

// Replace sn-list by contents of a single Directory (InoLsn, InoIdx pairs)
ULONG dfsApfsMakeBrowseList                      // RET   LSN is valid INODE
(
   ULN64               leafLsn,                 // IN    Leaf-node sectornumber
   ULN64               leafIdx,                 // IN    Leaf-node record index
   char               *str,                     // IN    unused
   void               *param                    // INOUT unused
);

// Get allocation information for file-DATA into integrated-SPACE structure
ULONG dfsApfsGetAllocSpace
(
   ULN64               leafLsn,                 // IN    Leaf-node sectornumber
   ULN64               leafIdx,                 // IN    Leaf-node record index (DIR-REC)
   char               *str,                     // IN    unused
   void               *param                    // INOUT Integrated SPACE
);

// Find latest checkpoint, activate that or specified older one
ULONG dfsApfsFindCheckpoint
(
   ULONG               cpIndex,                 // IN   Checkpoint, 0 is latest
   ULONG               verbosity                // IN   Display CP details/summary
);

// Activate volume specified by its index in the current superblock
ULONG dfsApfsSelectVolume
(
   ULONG               vIndex,                  // IN   Volume index, 0 is first
   ULONG               verbosity                // IN   Display CP details/summary
);

// Read Object-Map structure, then recursively read the map tree into memory
ULONG dfsApfsCacheObjectMap
(
   ULN64               omapBlock,               // IN    Blocknr omap object
   S_BT_NODE         **omapTree                 // OUT   Allocated omap tree
);

// Free root and tree-node memory for an Object Map
void dfsApfsFreeObjectMap
(
   S_BT_NODE         **pOmap                    // IN    Ptr to Allocated omap
);

#define APFS_ANY_XID   MAXULN64
// Resolve physical block number for virtual Oid + Xid, using given Object Map
ULN64 dfsApfsVirt2Phys                          // RET   Physical block nr or 0
(
   ULN64               virt,                    // IN    Virtual block nr (Oid)
   ULN64               vxid,                    // IN    Transaction ID   (Xid)
   S_BT_NODE          *omap                     // IN    Object Map tree to use
);

// APFS read and check type for an APFS Btree Node from its sector number
ULONG dfsApfsReadChkCatNode
(
   ULN64               lsn,                     // IN    Node LSN
   S_BT_NODE         **btNode                   // OUT   Btree Node structure
);

// Read FS-tree structure, then recursively cache all next LEAF block numbers
// Frees old cache, rebuilds for CURRENT active volume (supplied tree block#)
ULONG dfsApfsCacheNextLeafs
(
   BOOL                virtual,                 // IN    Tree/Nodes are virtual
   ULONG               verbosity,               // IN    Display BM details/summary
   ULN64               rootBlock,               // IN    Block-number root node
   DFSAPFSLEAFCACHE   *nlc                      // INOUT Next-leaf cache
);

// Free allocated memory for a next-leaf btree cache
void dfsApfsLeafCacheFree
(
   DFSAPFSLEAFCACHE   *nlc                      // INOUT Next-leaf cache
);

// Get virtual/physical link value for the NEXT LEAF in sequence (LEAF iterator)
ULN64 dfsApfsNextLeafLink                       // RET   Next LEAF or 0
(
   ULN64               leaf,                    // IN    LEAF virtual/physBlock
   DFSAPFSLEAFCACHE   *nlc                      // IN    Next-leaf cache
);

// Determine allocation-bit for specified LSN
ULONG dfsApfsAllocated                          // RET   LSN is allocated
(
   ULN64               sn64,                    // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
);

// Align RO cache to position for block, and get current allocation-bit
BOOL dfsApfsBitmapCache                         // RET   block is allocated
(
   ULONG               bmId,                    // IN    Bitmap to use (SFQ_)
   ULN64               bl                       // IN    Block number
);

// Read SPACEMANAGER info, when not done yet, initialize BitMap cache
// Note: NOT Area aware to allow usage from FDISK mode at the moment!
ULONG dfsApfsBitMapInit
(
   ULONG               bmId,                    // IN    Bitmap to initialize (SFQ_)
   ULONG               verbosity                // IN    Display BM details/summary
);

// Free SPACEMANAGER info, including the BITMAP cache
void dfsApfsBitMapFree
(
   ULONG               bmId                     // IN    Bitmap to free (SFQ_)
);

// Search Inode-record in catalog B-tree, returning parent-Ino / item-name / isDir
ULONG dfsApfsIno2Parent                         // RET   result
(
   ULN64               ino,                     // IN    INO of item to search
   ULN64              *parent,                  // OUT   parent-INO        or NULL
   char               *name,                    // OUT   NAME of item, ""  or NULL
   BOOL               *isDir                    // OUT   item is DIR flag, or NULL
);

// Search catalog B-tree for specified key, get Leafnode + optional index/node
ULN64 dfsApfsSearchCatalog                      // RET   LEAF Node virtual-Id
(
   S_NODE_KEY          key,                     // IN    Search key
   USHORT             *record,                  // OUT   record-index,    or NULL
   S_BT_NODE         **leaf                     // OUT   ptr to leafnode, or NULL
);

// Find full PATH to Root-directory and optiona recType from Node LSN and index
ULONG dfsApfsLsnInfo2Path                       // RET   result
(
   ULN64               start,                   // IN    node lsn
   ULONG               index,                   // IN    node index
   int                *rType,                   // OUT   recType or NULL
   char               *path                     // OUT   combined path string
);

// Find related INO and optional recType for a (Node) LSN and index info value
ULN64 dfsApfsLsnInfo2Ino                       // RET   INO, or 0 when failed
(
   ULN64               nodeLsn,                 // IN    node lsn
   ULONG               index,                   // IN    node index
   int                *rType                    // OUT   recType or NULL
);

// Find catalog-record for a Node and index, get type, key and/or value info
int dfsApfsNodeIdx2Rec                          // RET   recordtype, or 0 when failed
(
   S_BT_NODE          *node,                    // IN    node structure
   ULONG               index,                   // IN    node index
   USHORT             *kLen,                    // OUT   length of Key   record, or NULL
   S_NODE_KEY         *kRec,                    // OUT   catalog   Key   record (pointer)
   USHORT             *vLen,                    // OUT   length of Value record, or NULL
   S_NODE_VAL         *vRec                     // OUT   catalog   Value record (pointer)
);

#define dfsApfsBlock2Lsn(bl) (ULN64) dfsApfsCl2Lsn((bl),0,NULL,NULL)
// Translate Block-nr to LSN, generic interface, Area aware
ULN64 dfsApfsCl2Lsn                             // RET   LSN for this block
(
   ULN64               block,                   // IN    block number
   ULN64               d2,                      // IN    dummy
   char               *p1,                      // IN    dummy
   void               *p2                       // IN    dummy
);

#define dfsApfsLsn2Block(sn) dfsApfsLsn2Cl((ULN64)(sn),0,NULL, NULL)
// Translate LSN to Block-nr, generic interface, Area aware
ULN64 dfsApfsLsn2Cl                             // RET   block for this LSN
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *p1,                      // IN    dummy
   void               *p2                       // IN    dummy
);

// Calculate 64-bit APFS-style checksum for given data area
// Should return 0 when executed against whole block, starting with checksum
// or the checksum value when starting at offset 8, and size blocksize - 8
ULN64 dfsApfsChecksum
(
   BYTE               *data,                    // IN    data area (APFS block)
   ULONG               size                     // IN    data area size, bytes
);

// Get single character sector-type from an APFS Block-type enumeration value
BYTE dfsApfsBlock2SecType                       // RET   Sector type character
(
   ULONG               bt                       // IN    APFS block type
);

// Convert APFS TIME (64 bit) to compiler time_t format (32 bit)
#define dfsApfsTime2Tm(at) ((time_t)(((ULN64)(at)) / 1000000000ULL))

// Convert APFS TIME (64 bit) to DFS standard date/time string (optional nsec)
// Returns 19 character standard date-time, plus (decimals +1) fraction if any
char *dfsApfsTime2str                           // RET   string value
(
   ULN64               at,                      // IN    APFS time value
   int                 decimals,                // IN    seconds #decimals 0..9
   char               *dtime                    // INOUT ptr to string buffer
);

// Calculate 64-bit Fletcher checksum for given data area (like Linux APFS)
ULN64 dfsApfsFletcher64
(
   ULONG              *data,                    // IN    data area (APFS block)
   ULONG               count,                   // IN    data area size, ULONGs
   ULN64               init                     // IN    initial checksum value
);

// Verify if given data block has a valid Fletcher64 checksum
BOOL dfsApfsValidBlock
(
   BYTE               *block,                   // IN    data area (APFS block)
   ULONG               size                     // IN    data area size, bytes
);

// APFS filesystem, display DIR-style header/footer
void dfsApfsDirHeader
(
   char               *lead,                    // IN    lead text-string     (20)
   ULONG               items                    // IN    DIR-lines shown (0 = TOP)
);

// Show contents of a directory/folder using 2 lines per entry
ULONG dfsShowFolderContents
(
   ULN64               folderIno,               // IN    inode number for folder
   ULN64               startLeaf                // IN    first leaf node to search
);

// Display APFS Dir/File information on two lines, in 'dfsee directory' format
ULN64 dfsApfsShowCatRecord                      // RET   DIR Inode number or 0
(
   ULN64               leafLsn,                 // IN    Leaf-node sectornumber
   USHORT              leafIndex,               // IN    Leaf-node record index
   BYTE                bg,                      // IN    current background
   char               *rKind                    // OUT   (f)ile, (D)ir or (S)ymlink
);                                              //       or NULL when  not needed

// Return description for APFS blocktype, subtype, or 'unknown'
char *dfsApfsBlockType
(
   USHORT              blocktype                // IN    type of APFS block
);

// Return description for APFS FS record type, or 'unknown'
char *dfsApfsFsRecType
(
   int                 recType                  // IN    FS record type
);

// Return single-char description for APFS DIR record type (DIR/FILE/LINK etc)
char *dfsApfsDirRecType                         // RET   single character str
(
   USHORT              drFlg                    // IN    Dir-Rec value Flags
);

// Return description for APFS XFIELD record type, or 'unknown'
char *dfsApfsXfieldType
(
   int                 xfType,                  // IN    Xfield record type
   BOOL                dir                      // IN    for a DIR-Record
);

// Return description for APFS Device/Bitmap (Single or Fusion drive main/tier2)
char *dfsApfsDeviceBmType
(
   int                 itemType                 // IN    type of APFS device/bitmap
);

// Return description for APFS spacemanager Free-Queue type
char *dfsApfsSpmSfqueType
(
   int                 sfqType                  // IN    type of APFS free queue
);

// Calculate combined filename-hash and length value (32-bit CRC32C based)
ULONG dfsApfsNameHash                           // RET   combined CRC+length
(
   BYTE               *name,                    // IN    filename string, UTF-8
   USHORT              length                   // IN    filename length
);

#endif
