//
//                     DFSee, Disk and Filesystem utility
//
//   Original code Copyright (c) 1994-2025 Fsys Software and Jan van Wijk
//
// ==========================================================================
//
//   DFSee, released under MIT License
//
//   Copyright (c) 1994-2025  Fsys Software and Jan Van Wijk
//
//   Permission is hereby granted, free of charge, to any person obtaining a copy
//   of this software and associated documentation files (the "Software"), to deal
//   in the Software without restriction, including without limitation the rights
//   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
//   copies of the Software, and to permit persons to whom the Software is
//   furnished to do so, subject to the following conditions:
//
//   The above copyright notice and this permission notice shall be included in all
//   copies or substantial portions of the Software.
//
//   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
//   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
//   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
//   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
//   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
//   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
//   SOFTWARE.
//
//
//   Questions on DFSee licensing can be directed to: jvw@dfsee.com
//
// ==========================================================================
//
//
// APFS (container) dump & display Analysis functions
//
// Author: J. van Wijk
//
// JvW  24-01-2018 Initial version, derived from High Performance File System
//

#include <txlib.h>                              // ANSI controls and logging

#include <dfsver.h>                             // DFS version info
#include <dfsdisk.h>                            // FS disk structure defs
#include <dfspart.h>                            // FS partition info manager
#include <dfstore.h>                            // Store and sector I/O
#include <dfs.h>                                // DFS navigation and defs
#include <dfsutil.h>                            // DFS utility functions
#include <dfsspace.h>                           // DFS file-space interface
#include <dfstable.h>                           // SLT utility functions
#include <dfsutf8.h>                            // UTF-8 Unicode normalizing

#include <dfsaapfs.h>                           // APFS analysis functions
#include <dfsuapfs.h>                           // APFS utility functions


// Read an Object-Map tree structure into memory from specified ROOT block
// Replaces blocknr links to child-nodes by node-pointers, with MAGIC flag
static ULONG dfsApfsCacheOmTree
(
   ULN64               treeBlock,               // IN    Blocknr omap tree ROOT
   S_BT_NODE         **omapTree                 // OUT   Allocated omap tree
);

// Compare Object-Map key to supplied Oid (primary) and Xid (secondary) values
// Shows PRIMARY key status (-1,0,+1) and SECONDARY (-2,+2) when primary is EQ
static int dfsApfsOmapKeyCompare                // RET   0=EQ; -1=smaller; 1=larger
(                                               //       2ndry -2=smaller; 2=larger
   S_OMAP_KEY         *eKey,                    // IN    entry key
   ULN64               oid,                     // IN    virtual object ID
   ULN64               xid                      // IN    related transction ID
);

// Iterate node records, recurse next-level for nodes, add to cache for leafs
static ULONG dfsApfsCacheNlNode
(
   BOOL                virtual,                 // IN    Tree/Nodes are virtual
   S_BT_NODE          *node,                    // IN    root/intermediate node
   DFSAPFSLEAFCACHE   *nlc                      // INOUT Next-leaf cache
);

// Build selection-list with selectable Volume entries for a container
static TXSELIST *dfsApfsVolumeSelist            // RET   selection list or NULL
(
                       void
);

// Compare catalog-record-key to supplied search key of specified type
// Note: All types share a 64-bit PRIMARY/SECONDARY key value (parent/ino/xid)
// The lower 60 bits are PRIMARY, upper 4 SECONDARY, with Tertiary parts optional
// Shows PRI/SEC key status (-1,0,+1) and TERTIARY wildcard (2) for pri/sec EQ
// and (-2) in case the tertiary key has a hash collision (continue searching)
static int dfsApfsCatalogKeyCompare             // RET   0=EQ;  -1=smaller; 1=larger
(                                               //       -2=collision 2=wildcard match
   S_NODE_KEY          rKey,                    // IN    record key in FS-tree
   S_NODE_KEY          sKey                     // IN    search key to compare
);


/*****************************************************************************/
// Find Leaf-Node LSN+Index for specified path, starting at the root-directory
/*****************************************************************************/
ULONG dfsApfsFindPath
(
   ULN64               loud,                    // IN    Show progress
   ULN64               d2,                      // IN    dummy
   char               *path,                    // IN    path specification
   void               *vp                       // OUT   found dir/file NODE+index
)
{
   ULONG               rc  = NO_ERROR;
   TXLN                part;
   char               *p   = path;
   int                 l;
   ULN64               curIno  = APFS_ROOT_DIR_INODE;
   USHORT              nodeIdx = 1;             // ROOT is 2nd record (to be refined, may search for 'root' with parent 1)
   ULN64               leafId  = dfsApfsNextLeafLink( 0, &apfs->vCatTree); // virt link to first leaf
   S_BT_NODE          *node = NULL;
   DFS_PARAMS         *parm = (DFS_PARAMS *) vp;
   TXLN                keySpace;                // key storage space
   S_NODE_KEY          key;                     // universal key pointer (union)

   ENTER();

   key.data = keySpace;                         // storage large enough for a full filename

   if (loud)
   {
      dfsX10( "Root   folder INO : ", curIno, CNN, "   ");
      TxPrint("find path: '%s'\n", path);
   }
   parm->Lsn    = dfsApfsBlock2Lsn( dfsApfsVirt2Phys( leafId, APFS_ANY_XID, apfs->vOmap)); // initial values, for ROOT (empty path)
   parm->Number = (ULONG) nodeIdx | DFSSNINFO;
   while ((rc == NO_ERROR) && strlen(p) && !TxAbort())
   {
      if ((l = strcspn(p, FS_PATH_STR)) != 0)
      {
         strncpy(part, p, l);                   // isolate part
         part[l] = '\0';
         p += l;                                // skip part
         if (*p == FS_PATH_SEP)
         {
            p++;                                // skip '\'
         }
      }
      if (strlen(part))
      {
         if (loud)
         {
            dfsX10( "Search folder INO : ", curIno, CNN, "");
         }
         key.dh->fsId = APFSmkFsId( curIno, APFST_DIR_RECORD); // hybrid primary key value
         key.dh->drLengthHash = dfsApfsNameHash( (BYTE *) part, strlen( part) + 1);
         strcpy((char *) key.dh->drName, part);   // copy name itself too (hash-collission-detect)
         leafId = dfsApfsSearchCatalog( key, &nodeIdx, &node);
         if ((leafId != 0) && ((leafId != L64_NULL)))
         {
            S_NODE_VAL catData;
            int        recType;

            recType = dfsApfsNodeIdx2Rec( node, nodeIdx, NULL, NULL, NULL, &catData);
            if (recType == APFST_DIR_RECORD)
            {
               curIno = catData.dr->fileId;
               if (loud)
               {
                  dfsX10( " - INO : ", curIno, CNN, " ");
                  TxPrint("for '%s'\n", part);
               }
               if (*p == '\0')                  // end of string, found!
               {
                  parm->Lsn    = dfsApfsBlock2Lsn( dfsApfsVirt2Phys( leafId, APFS_ANY_XID, apfs->vOmap));
                  parm->Number = (ULONG) nodeIdx | DFSSNINFO;
                  parm->Flag   = FALSE;         // Size NOT available (from DIR-entry)

                  if (loud)
                  {
                     dfsX10( "Found virtual Blk : ", leafId, CBG, "   ");
                     TxPrint("entry number : %hu\n", nodeIdx);
                  }
                  if (catData.dr->drFlags == DRF_REG) // regular file
                  {
                     //- to be refined, could get filesize (SearchCatalog for the inode) if it is a file
                     //- parm->byteSize = ...
                  }
               }
            }
            else
            {
               rc = DFS_BAD_STRUCTURE;
            }
            TxFreeMem( node);
         }
         else
         {
            rc = ERROR_PATH_NOT_FOUND;
         }
         if (rc != NO_ERROR)
         {
            if (loud)
            {
               TxPrint(" - Search failed  for '%s'\n", part);
            }
         }
      }
      else
      {
         rc = ERROR_PATH_NOT_FOUND;
      }
   }
   RETURN (rc);
}                                               // end 'dfsApfsFindPath'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Identify if FS structures for APFS are present (superblock in first sector)
// Note that the sectorsize may be different between partition/disk open!
/*****************************************************************************/
ULONG dfsApfsFsIdentify
(
   S_BOOTR            *boot,                    // IN    Bootsector ref or NULL
   DFSPARTINFO        *p                        // INOUT partition info
)
{
   ULONG               rc = DFS_FS_UNKNOWN;     // function return

   ENTER();

   if (dfsApfsIsSuperBlock( (BYTE *) boot))
   {
      rc = NO_ERROR;                            // it IS APFS, now try for a label ...
      strcpy( p->fsform, "APFS");

      if (p != NULL)                            // try to read 1st volumename as label
      {
         BYTE               *block = NULL;
         S_C_SUPER          *sup   = (S_C_SUPER *) boot; // Master superblock, block 0, 512 bytes!
         ULN64               lsn;
         ULONG               spb   = 4096 / dfsGetSectorSize(); // sectors per block

         strcpy( p->plabel, "(!mounted!)"); // Set default label, remains on failure (most likely mounted FS)

         TRACES(("p->basePsn: 0x%llx  spb:%u\n", p->basePsn, spb));

         if ((block = TxAlloc( spb, dfsGetSectorSize())) != NULL)
         {
            S_B_HEADER      *hdr    = (S_B_HEADER *) block; // APFS struct header alias
            S_V_SUPER       *volsup = (S_V_SUPER  *) block; // volume superblock  alias
            S_BT_NODE       *sd     = (S_BT_NODE  *) block; // generic tree node  alias
            S_BT_FIXED      *fixToc = (S_BT_FIXED *) (sd->btData + sd->btNode.btToc.Offset);

            lsn = sup->csObjectMapBlock * spb;
            if ((dfsRead( p->basePsn + lsn, spb, block)) == NO_ERROR) // object-map header
            {
               TRACES(( "checksum:%llx transactId:%llx hdrType:%x\n", hdr->checksum, hdr->transactId, APFShdrType( hdr)));

               //- only find latest checkpoint when needed (on a mounted and unstable FS) to save time!
               if ((hdr->checksum == 0) || (hdr->transactId == 0) || (APFShdrType( hdr) != BT_OBJECT_MAP))
               {
                  ULN64      latestXid = 0;    //- Xid   of latest  CP found sofar
                  ULONG      thisCpi;          //- Index of current CP checked
                  ULONG      cpSbCpi   = 0;    //- Index of matched SB
                  ULONG      size      = sup->csSbDescBlockCount; // Nr of blocks to check
                  ULN64      baseBlock = sup->csSbDescBaseBlock; // allow 'sup' block to be overwritten

                  TRACES(("start search for latest c-superblock at block: 0x%llx\n", baseBlock));
                  if ((baseBlock) && (baseBlock < sup->csTotalBlocks) && (size < 0xffff)) // sanity check
                  {
                     for (thisCpi = 0, lsn = baseBlock * spb; thisCpi < size; thisCpi++, lsn += spb)
                     {
                        if (dfsRead( p->basePsn + lsn, spb, block) == NO_ERROR)
                        {
                           if ((hdr->checksum) && (hdr->checksum != DFS64MAX) && (hdr->transactId))
                           {
                              if      (APFShdrType( hdr) == BT_CHECKP_MAP)
                              {
                                 if (hdr->transactId > latestXid)
                                 {
                                    latestXid = hdr->transactId;
                                 }
                              }
                              else if (APFShdrType( hdr) == BT_C_SUPERBLOCK)
                              {
                                 if (hdr->transactId == latestXid) // matching SB to previous MAP block
                                 {
                                    cpSbCpi = thisCpi; // remember index for matching SB
                                 }
                              }
                           }
                        }
                     }
                  }
                  if (cpSbCpi != 0)             // found valid latest checkpoint SB
                  {
                     lsn = (baseBlock + cpSbCpi) * spb;

                     TRACES(("latestXid: 0x0%llx  cpSbCpi: %u thisCpi: %u\n", latestXid, cpSbCpi, thisCpi));

                     if (dfsRead( p->basePsn + lsn, spb, block) == NO_ERROR) // re-read that C-superblock
                     {
                        sup = (S_C_SUPER *) block; // now points to LATEST superblock found

                        lsn = sup->csObjectMapBlock * spb;
                        dfsRead( p->basePsn + lsn, spb, block); // read associate object-map header block
                     }
                  }
               }
               if ((hdr->checksum)   && (hdr->checksum     != DFS64MAX)    &&
                   (hdr->transactId) && (APFShdrType( hdr) == BT_OBJECT_MAP))
               {
                  lsn = (((S_OBJECT_MAP *) block)->omTreeOid) * spb;
                  if ((dfsRead( p->basePsn + lsn, spb, block)) == NO_ERROR) // object-map TREE root
                  {
                     if ((hdr->checksum)   && (hdr->checksum     != DFS64MAX)    &&
                         (hdr->transactId) && (APFShdrType( hdr) == BT_BTREE_ROOT))
                     {
                        ULONG  valOffs = APFS_BLOCKSIZE - sizeof( S_BTREE_INFO);
                        ULN64 *p64;             // pointer to 64-bit key or value

                        p64 = (ULN64 *) (block + valOffs - fixToc[ 0].nVoffset); // first entry is 1st volume super
                        p64++;                  // but it is the SECOND 64-bit value in this value record
                        lsn = (*p64) * spb;
                        if ((dfsRead( p->basePsn + lsn, spb, block)) == NO_ERROR) // volume superblock
                        {
                           if ((hdr->checksum)   && (hdr->checksum     != DFS64MAX)      &&
                               (hdr->transactId) && (APFShdrType( hdr) == BT_V_SUPERBLOCK))
                           {
                              TxCopy( p->plabel, volsup->vsVolumeName, TXMAXTS); // limit to available size in partinfo
                           }
                        }
                     }
                  }
               }
            }
            TxFreeMem( block);
         }
      }
   }
   RETURN (rc);
}                                               // end 'dfsApfsFsIdentify'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display file allocation and path info for LSN + index
/*****************************************************************************/
ULONG dfsApfsFileInfo                            // RET   LSN is valid NODE
(
   ULN64               leafLsn,                 // IN    Leaf-node sectornumber
   ULN64               leafIdx,                 // IN    Leaf-node record index
   char               *select,                  // IN    file selection string
   void               *param                    // INOUT leading text / path
)
{
   ULONG               rc = NO_ERROR;
   TXLN                fpath;
   ULN64               fsize     = 0;           // filesize in bytes
   ULN64               size      = 0;           // allocated size, sectors
   ULN64               bads      = 0;           // bad allocated sectors
   TXLN                text;
   ULONG               percent   = 100;
   ULONG               threshold = 0;           // threshold percentage
   BOOL                isMinimum = TRUE;        // threshold is minimum value
   ULN64               location  = 0;
   TXLN                temp;                    // location/size/date/symlink text
   ULONG               minS      = 0;           // minimal size
   ULONG               maxS      = 0;           // maximal size
   char                itemType  = 0;           // type Dir, File, Browse, or ANY
   time_t              modTstamp = 0;           // Modify timestamp value, or 0
   char               *lead    = param;
   BOOL                verbose = (strcmp( lead, ":") != 0);
   BOOL                output;                  // no Printf, just return fpath
   BOOL                alcheck;                 // Execute a full allocation check
   S_BT_NODE          *node = NULL;             // leaf node for file
   char                recKind = 0;             // record: (f)ile, (D)ir or (S)ymlink

   ENTER();
   TRARGS(("LSN: 0x%llX, index: %llX  select: '%s', lead: '%s'\n", leafLsn, leafIdx, select, lead));

   if (strcmp( lead, "!") == 0)                 // Skip allocation check wanted
   {
      output  = FALSE;                          // no output
      alcheck = FALSE;                          // and no allocation check
      *lead   = 0;                              // discard 'garbage' in lead string
   }
   else                                         // Possible output/alloc-check wanted
   {
      output  = (*lead != 0);                   // output if any lead given (except '!')
      alcheck = TRUE;                           // but always an alloc check if no  '!'
   }
   TRACES(("verbose:%u output:%u alcheck:%u\n", verbose, output, alcheck));

   if ((leafIdx & DFSSNINFO) && (dfsApfsReadChkCatNode( leafLsn, &node) == NO_ERROR) && (node->btNode.btFlags & BNF_LEAF))
   {
      S_FILE_EXT_KEY   fexKey;                  // key to search for the file-extent
      S_XATTR_KEY      xatKey;                  // key to search for an XATTR record
      S_INODE_KEY      inoKey;                  // key to search for the INODE
      S_NODE_KEY       nodeKey;                 // generic record key   (pointer)
      S_NODE_VAL       nodeVal;                 // generic record value (pointer)
      USHORT           vLength;                 // length of record value
      ULN64            inoRefCount;             // nr of files/links     in Inode
      ULN64            inoModTime;              // last-modify timestamp in Inode
      ULN64            ino = 0;
      ULN64            leafId;                  // virtual leaf ID for found leaf
      USHORT           index;                   // record index for INODE in found leaf
      int              recType;
      int              i;

      recType = dfsApfsNodeIdx2Rec( node, DFSSNIGET( leafIdx), NULL, &nodeKey, &vLength, &nodeVal);
      switch (recType)
      {
         case APFST_DIR_RECORD:
            strcpy( temp, (char *) nodeKey.dh->drName); // remember original DIR-RECORD fname
         case APFST_DIR_STATS:
         case APFST_FILE_EXTENT:
         case APFST_DSTREAM_ID:
         case APFST_XATTR:
         case APFST_SIBLING_LINK:
            if (recType == APFST_DIR_RECORD)    // INO number is in the record VALUE
            {
               ino = nodeVal.dr->fileId;
            }
            else                                // INO number is in the record KEY
            {                                   // Inode record must be searched/read!
               ino = APFSidPrim( nodeKey.ds->fsId);
            }
            TxFreeMem( node);                   // free existing node record
            nodeKey.in = &inoKey;               // create the key for Btree search
            inoKey.fsId = APFSmkFsId( ino, APFST_INODE); // hybrid primary/secondary key value
            leafId = dfsApfsSearchCatalog( nodeKey, &index, &node);
            if ((leafId == 0) || (leafId == L64_NULL) ||
                (dfsApfsNodeIdx2Rec( node, index, NULL, &nodeKey, &vLength, &nodeVal) != APFST_INODE))
            {
               TRACES(("Failed to locate INODE record!\n"));
               if ((output == FALSE) && (recType == APFST_DIR_RECORD)) // most likely, so partial info available (name)
               {
                  strcpy( param, temp);         // return name from original DIR-record
                  strcpy( select, "!Error getting Inode from DIR for: ");
                  rc = DFS_NOT_FOUND;           // signal partial success, but no full info
               }
               else
               {
                  rc = DFS_BAD_STRUCTURE;
               }
            }
            break;

         case APFST_INODE:
            ino = APFSidPrim( nodeKey.in->fsId); // make INO value available for search ATTR/ALLOC
            break;

         default:
            rc = DFS_PENDING;                   // no fileinfo available
            break;
      }
      if (rc == NO_ERROR)                      // we have a valid INODE record now, to get all info
      {
         recKind   = ((nodeVal.in->inMode & S_IFLNK) == S_IFLNK) ? 's' : S_ISDIR( nodeVal.in->inMode)  ? 'D' : 'f';
         inoRefCount = nodeVal.in->inRefCount;  // remember refcount (record/nodeVal is reused!)
         inoModTime  = nodeVal.in->inModTime;

         dfsParseFileSelection( select, text, &isMinimum, &threshold, &minS, &maxS, &modTstamp, &itemType);

         #if defined (HAVEALLOC)
         if (verbose)
         {
            //- Allow skipping the allocation-check when not needed (Speedup BROWSE display)
            //- rc = dfsApfsCatCheckAlloc( nodeVal, (alcheck) ? "" : "s", &fsize, &size, &bads, &location);
         }
         else                                   // no size/percentage info
         #endif
         {
            isMinimum = TRUE; threshold = 0;    // no percentage filter
            minS      = 0;    maxS      = 0;    // no size filter
         }

         TRACES(("item Kind: %c\n", recKind));
         switch (recKind)
         {
            case 'D':                           // normal dir
               if      (itemType == DFS_FS_ITEM_BROWSE) // no filtering on Directories, take all.
               {
                  strcpy( text, "");            // no wildcard
                  isMinimum = TRUE; threshold = 0; // no percentage filter
                  modTstamp = 0;                // no timestamp filter
                  minS      = 0;    maxS      = 0; // no size filter
               }
               else if (itemType == DFS_FS_ITEM_FILES) // filter files, discard Directories
               {
                  rc = DFS_ST_MISMATCH;         // item-type mismatch
                  break;
               }
            case 'f':                           // normal file
            case 's':                           // symbolic link
               if      (itemType == DFS_FS_ITEM_DIRS) // filter dirs, discard files
               {
                  rc = DFS_ST_MISMATCH;         // item-type mismatch
                  break;
               }
               strcpy( fpath, "");
               dfsApfsLsnInfo2Path( leafLsn, DFSSNIGET( leafIdx), NULL, fpath);
               if ((strlen(text) == 0) || (TxStrWicmp( fpath, text) >= 0))
               {
                  if (recKind == 'f') // get byteSize and allocated size, and location when possible
                  {
                     if (vLength > sizeof( S_INODE_VAL)) // there are Xfields
                     {
                        S_XF_BLOB *xfb  = (S_XF_BLOB *) (nodeVal.in->inXfields);
                        BYTE      *data = ((BYTE *) xfb->xfData) + (xfb->xfExtents * sizeof( S_XF_FIELD));

                        for (i = 0; i < xfb->xfExtents; i++)
                        {
                           if (xfb->xfData[ i].xfType == XFT_INO_DATASTREAM)
                           {
                              fsize = ((S_DSTREAM *) data)->dsByteSize;
                              size  = ((S_DSTREAM *) data)->dsAllocSize / dfsGetSectorSize();
                              break;
                           }
                           data += ((xfb->xfData[ i].xfSize + 7) & 0xFFF8); //  next Xfield (8 byte size align)
                        }
                     }
                     TxFreeMem( node);          // free existing node record
                     nodeKey.fe = &fexKey;      // create the key for Btree search
                     fexKey.fsId = APFSmkFsId( ino, APFST_FILE_EXTENT); // hybrid primary/secondary key value
                     fexKey.feLogAddress = 0;   // first extent, relative address 0
                     leafId = dfsApfsSearchCatalog( nodeKey, &index, &node);
                     if ((leafId != 0) && ((leafId != L64_NULL)))
                     {
                        if (dfsApfsNodeIdx2Rec( node, index, NULL, &nodeKey, &vLength, &nodeVal) == APFST_FILE_EXTENT)
                        {
                           location = dfsApfsBlock2Lsn( nodeVal.fe->feBlockNr); // LSN of first datablock
                        }
                     }
                  }
                  if (verbose)                  // include alloc and size
                  {
                     if (location == 0)
                     {
                        if (fsize == 0)
                        {
                           sprintf(temp, "%s     %s%s ", CBZ, (recKind == 's') ? "     " : (recKind == 'D') ? "  dir" : "empty", CNN);
                           rc = NO_ERROR;       // allocation always OK
                        }
                        else                    // there IS something, perhaps resource fork
                        {
                           //- to be refined
                           strcpy(  temp, "           ");
                           rc = NO_ERROR;       // allocation always OK
                        }
                     }
                     else
                     {
                        sprintf( temp, "%10llX ", location);
                     }
                     percent = dfsAllocationReliability( size, bads);

                     TRACES(( "percent:%u threshold:%u size:%u minS:%u maxS:%u\n",
                               percent, threshold, size, minS, maxS));
                  }
                  if ((BOOL)((percent >= threshold)) == isMinimum)
                  {
                     if ((dfsApfsTime2Tm( inoModTime) > modTstamp) &&          //- Timestamp match time_t
                         ((size >= minS) && ((size <= maxS) || (maxS == 0))))  //- Size match
                     {
                        if (verbose)            // include alloc and size
                        {
                           TXTS         recIndicator;

                           if (recType == APFST_DIR_RECORD) // the 'normal' case, no indicator
                           {
                              strcpy( recIndicator, " ");
                           }
                           else
                           {
                              strcpy( recIndicator, dfsApfsFsRecType( recType));
                           }

                           if (alcheck)
                           {
                              sprintf( text, "%s%c%s%c%s%3u%%%s ", CBM, recKind, CBZ, recIndicator[0], (rc == NO_ERROR) ? CBG : CBR, percent, CNN);
                           }
                           else                 // no reliability percentage
                           {
                              sprintf( text, "%s%c%s%c     ", CBM, recKind, CBZ, recIndicator[0]);
                           }
                           strcat(     text, temp);

                           if (recKind == 'D')
                           {
                              sprintf( temp, "%s #%s%6llu ", CBZ, CNN, inoRefCount);
                           }
                           else                 // report size in sectors, KiB..TiB
                           {
                              strcpy( temp, "");
                              dfstrSize( temp, CBC, size, " ");
                           }
                           dfstrBytes(temp, CBZ, 0,    " "); // XATTR (to be refined, get XATTR record(s) ?)
                           strcat( text, temp);
                        }
                        else
                        {
                           strcpy( text, "");   // remove any stray text, non-verbose
                        }
                        if (recKind == 's')     // symlink
                        {
                           TxFreeMem( node);    // free existing node record
                           nodeKey.xa = &xatKey; // create the key for Btree search
                           xatKey.fsId = APFSmkFsId( ino, APFST_XATTR); // hybrid primary/secondary key value
                           xatKey.xaNameLength = strlen( XA_SYMLINK_NAME) + 1; // includes the termination char
                           strcpy( (char *) xatKey.xaName, XA_SYMLINK_NAME);
                           leafId = dfsApfsSearchCatalog( nodeKey, &index, &node);
                           if (((leafId != 0) && ((leafId != L64_NULL))) &&
                               (dfsApfsNodeIdx2Rec( node, index, NULL, &nodeKey, &vLength, &nodeVal) == APFST_XATTR))
                           {
                              strcpy( temp, (char *) nodeVal.xa->xaData); // copy symlink name
                           }
                           else
                           {
                              strcpy( temp, "?"); // signal failure to find the link name
                           }
                        }
                        if (output)             // not totally silent?
                        {
                           strcat( lead, text);
                           TxPrint( "%s%s%s", lead, CBY, fpath);
                           if (recKind == 's')    // symlink
                           {
                              if (strlen( fpath) < 25)
                              {
                                 TxPrint( "%*.*s", (25 - strlen( fpath)), (25 - strlen( fpath)), "");
                              }
                              TxPrint( "%s%s%s%s", CNN, DFS_SH_LINK_STR, CBC, temp);
                           }
                           TxPrint( "%s%s\n", CNN, CGE);
                        }
                        else                    // return info in 'select' and 'param' (Browse)
                        {
                           if (recKind == 's') // symlink
                           {
                              strcat( fpath, DFS_SH_LINK_STR); // indicator for symbolic link in BROWSE
                              strcat( fpath, temp);
                           }
                           dfsApfsTime2str( inoModTime, 0, temp);
                           TxStripAnsiCodes( text);
                           sprintf( select, "%s %s", text, temp);
                           dfstrUllDot20( select, "", fsize, "");
                        }
                        rc = NO_ERROR;
                     }
                     else
                     {
                        rc = DFS_ST_MISMATCH;   // file-size or mod-time mismatch
                     }
                  }
                  else
                  {
                     rc = DFS_ST_MISMATCH;      // percentage mismatch
                  }
               }
               else
               {
                  rc = DFS_ST_MISMATCH;         // wildcard mismatch
               }
               strcpy( param, fpath);           // return the found name
               break;

            default:
               rc = DFS_PENDING;                // unsupported record types
               break;
         }
      }
      TxFreeMem( node);
   }
   else
   {
      rc = DFS_PENDING;                         // not a catalog node with index
   }
   TRACES(("select OUT: '%s'\n", select));
   TRACES(("param  OUT: '%s'\n", lead));
   RETURN(rc);
}                                               // end 'dfsApfsFileInfo'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Replace sn-list by contents of a single Directory (InoLsn, InoIdx pairs)
/*****************************************************************************/
ULONG dfsApfsMakeBrowseList                     // RET   LSN is valid INODE
(
   ULN64               leafLsn,                 // IN    Leaf-node sectornumber
   ULN64               leafIdx,                 // IN    Leaf-node record index
   char               *str,                     // IN    unused
   void               *param                    // INOUT unused
)
{
   ULONG               rc = NO_ERROR;
   ULN64               ino;                    // cnid of folder to browse
   ULN64               parent;                  // cnid of its parent, if any
   BOOL                isDirectory;
   ULN64               thisNodeNr;              // node nr  to be handled next (auto)
   ULN64               thisNodeLsn;             // node LSN to be handled next (auto)
   ULN64               entryLsn;
   USHORT              entryIdx;
   TXLN                parentName;              // name of parent directory
   ULN64               parentParent;            // parent of parent (to get lsn/index)
   S_HDREC_KEY         dhKey;                   // hashed-DIR record key
   S_NODE_KEY          nodeKey;                 // generic record key   (pointer)
   S_BT_NODE          *node = NULL;             // leaf node

   ENTER();
   TRARGS(("LSN: 0x%llX, index: %llX\n", leafLsn, leafIdx));

   dfsInitList(0, "-f -P", "-d");               // optimal for menu file-recovery

   if ((apfs->vTransactions > APFS_MOUNT_TR_LIMIT) && (apfs->vMountWarning == FALSE))
   {
      apfs->vMountWarning = TRUE;               // only give the warning once
      TxNamedMessage( !dfsa->batch, 5944, " WARNING: MOUNTED/UNSTABLE filesystem ",
         "APFS volume seems mounted, with %llu transactions since last (un)mount\n"
         "Browsing might be unreliable, with missing files/directory information,\n"
         "failures to view or recover data from them, or even a program crash!\n\n"
         "That said, it might just work for what you need to do ...\n\n"
         "To minimize the risc, an automatic SYNC to the latest checkpoint will be\n"
         "executed at selection of any directory, taking some extra time (seconds)", apfs->vTransactions);
   }
   if ((TxaOption( TXA_O_REFR)) || ((apfs->vTransactions > APFS_MOUNT_TR_LIMIT) && (!TxaOptUnSet( TXA_O_REFR))))
   {
      ULONG             savedVerbosity = dfsa->verbosity;
      dfsa->verbosity = TXAO_SILENT;            // needed for the SB display functions!
      dfsApfsFindCheckpoint( 0, TXAO_SILENT);   // silently sync to latest checkpoint
      dfsa->verbosity = savedVerbosity;
   }
   ino = dfsApfsLsnInfo2Ino( leafLsn, DFSSNIGET( leafIdx), NULL);
   if (ino != 0)
   {
      rc = dfsApfsIno2Parent( ino, &parent, NULL, &isDirectory);
      if ((rc == NO_ERROR) && (isDirectory))
      {
         nodeKey.dh = &dhKey;                   // key for Btree search
         if (ino > APFS_ROOT_DIR_INODE)         // add .. for parent folder
         {
            //- get leaf-Lsn and index for the parent Folder entry
            rc = dfsApfsIno2Parent( parent, &parentParent, parentName, &isDirectory);
            if (rc == NO_ERROR)
            {
               TRACES(("parentParent INO %llx name '%s'\n", parentParent, parentName));

               dhKey.fsId = APFSmkFsId( parentParent, APFST_DIR_RECORD); // hybrid primary/secondary key value
               dhKey.drLengthHash = dfsApfsNameHash( (BYTE *) parentName, strlen( parentName) + 1);
               strcpy((char *) dhKey.drName, parentName); // copy name itself too (hash-collission-detect)

               thisNodeNr = dfsApfsSearchCatalog( nodeKey, &entryIdx, NULL);
               if ((thisNodeNr != 0) && (thisNodeNr != L64_NULL))
               {
                  TRACES(("List marked as 1ST_PARENT\n"));
                  DFSBR_SnlistFlag |= DFSBR_1ST_PARENT;

                  entryLsn = dfsApfsBlock2Lsn( dfsApfsVirt2Phys( thisNodeNr, APFS_ANY_XID, apfs->vOmap));

                  rc = dfsAddSI2List( entryLsn, entryIdx);
               }
            }
         }
         if (rc == NO_ERROR)
         {
            ULN64            parentId = 0;      // last parent-ID evaluated
            ULONG            entry;
            S_BT_ENTRY      *varToc;
            ULONG            keyOffs;

            if ((node = TxAlloc( apfs->SectorsPerBlock, dfsGetSectorSize())) != NULL)
            {
               BYTE         *block = (BYTE *) node;

               //- determine first leaf-node for the folder to browse
               dhKey.fsId = APFSmkFsId( ino, APFST_DIR_RECORD);   //- hybrid primary/secondary key value
               dhKey.drLengthHash = 0;                            //- tertiary key, 0 will find FIRST one
               thisNodeNr = dfsApfsSearchCatalog( nodeKey, NULL, NULL);
               if ((thisNodeNr != 0) && (thisNodeNr != L64_NULL))
               {
                  thisNodeLsn = dfsApfsBlock2Lsn( dfsApfsVirt2Phys( thisNodeNr, APFS_ANY_XID, apfs->vOmap));

                  TRACES(("Traverse leaf nodes from node %llx, for entries with parent INO %llx\n", thisNodeNr, ino));
                  do                            // iterate over leaf-nodes, collect entries for directory ino
                  {
                     if ((rc = dfsRead( thisNodeLsn, apfs->SectorsPerBlock, block)) == NO_ERROR)
                     {
                        if (dfsApfsValidBlock( block, apfs->BlockSize))
                        {
                           varToc  = (S_BT_ENTRY *) (node->btData + node->btNode.btToc.Offset);
                           keyOffs = sizeof( S_B_HEADER) + sizeof( S_BTREE_NODE) + node->btNode.btToc.Offset + node->btNode.btToc.Length;
                           for (entry = 0; entry < node->btNode.btCount; entry++)
                           {
                              nodeKey.data = (block + keyOffs + varToc[ entry].nKoffset);

                              TRACES(("entry:%2u rec-fsId:%llx\n", entry, nodeKey.dh->fsId));

                              parentId = APFSidPrim( nodeKey.dh->fsId); // parent is primary key for DIR-RECORD
                              if ((parentId == ino) && (APFSidType( nodeKey.dh->fsId) == APFST_DIR_RECORD))
                              {
                                 BOOL       filtered = FALSE;

                                 if (dfsa->browseShowHidden == FALSE)
                                 {
                                    if (nodeKey.dh->drName[0] == '.') // name starts with a DOT, hidden
                                    {
                                       filtered = TRUE;
                                    }
                                 }
                                 if (!filtered)
                                 {
                                    rc = dfsAddSI2List( thisNodeLsn, entry); // add this entry to list
                                 }
                              }
                              else if (parentId > ino) // rest is larger, quit linear search
                              {
                                 TRACES(("Rest is larger, quit search ...\n"));
                                 break;
                              }
                           }
                        }
                        else
                        {
                           rc = DFS_BAD_STRUCTURE;
                        }
                     }
                     if (parentId == ino)      // still matching parent-ID, next leaf node
                     {
                        thisNodeNr  = dfsApfsNextLeafLink( thisNodeNr, &apfs->vCatTree);
                        thisNodeLsn = dfsApfsBlock2Lsn( dfsApfsVirt2Phys( thisNodeNr, APFS_ANY_XID, apfs->vOmap));
                     }
                     else                       // last Node (for this parent) handled
                     {
                        thisNodeNr  = 0;        // terminate ...
                     }
                  } while ((rc == NO_ERROR) && (thisNodeNr != 0) && !TxAbort());
               }
               TxFreeMem( node);
            }
            else
            {
               rc = DFS_ALLOC_ERROR;
            }
         }
      }
      else
      {
         rc = DFS_ST_MISMATCH;                  // not a directory
      }
   }
   RETURN(rc);
}                                               // end 'dfsApfsMakeBrowseList'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get allocation information for file-DATA into integrated-SPACE structure
// Supports 'in-memory' allocation for a symlink, containing the link-name
// to be refined, add allocation for XATTR (resource fork?) stream as well
/*****************************************************************************/
ULONG dfsApfsGetAllocSpace
(
   ULN64               leafLsn,                 // IN    Leaf-node sectornumber
   ULN64               leafIdx,                 // IN    Leaf-node record index (DIR-REC)
   char               *str,                     // IN    unused
   void               *param                    // INOUT Integrated SPACE
)
{
   ULONG               rc = NO_ERROR;
   DFSISPACE          *ispace = (DFSISPACE *) param;
   S_BT_NODE          *node = NULL;             // leaf node for file

   ENTER();
   TRARGS(("LSN: 0x%llX, index: %llX\n", leafLsn, leafIdx));

   ispace->space    = NULL;                     // init to empty object
   ispace->chunks   = 0;
   ispace->byteSize = 0;

   if ((dfsApfsReadChkCatNode( leafLsn, &node) == NO_ERROR) && (node->btNode.btFlags & BNF_LEAF))
   {
      S_INODE_KEY      inoKey;                  // key to search for the INODE
      S_FILE_EXT_KEY   fexKey;                  // key to search for the file-extent
      S_XATTR_KEY      xatKey;                  // key to search for an XATTR record
      S_NODE_KEY       nodeKey;                 // generic record key   (pointer)
      S_NODE_VAL       nodeVal;                 // generic record value (pointer)
      USHORT           vLength;                 // length of record value
      ULN64            ino = 0;
      ULN64            leafId;                  // virtual leaf ID for found leaf
      USHORT           index;                   // record index for INODE in found leaf
      int              recType;
      char             recKind = 0;             // record: (f)ile, (D)ir or (S)ymlink
      ULN64            rOffset = 0;             // relative offset of extent in file
      ULONG            extents = 0;             // nr of extents
      ULONG            areas   = 0;             // nr of space-areas
      S_SPACE         *sp      = NULL;          // SSPACE structure, growing as needed
      int              i;

      recType = dfsApfsNodeIdx2Rec( node, DFSSNIGET( leafIdx), NULL, &nodeKey, &vLength, &nodeVal);
      switch (recType)
      {
         case APFST_DIR_RECORD:
         case APFST_DIR_STATS:
         case APFST_FILE_EXTENT:
         case APFST_DSTREAM_ID:
         case APFST_XATTR:
         case APFST_SIBLING_LINK:
            if (recType == APFST_DIR_RECORD)    // INO number is in the record VALUE
            {
               ino = nodeVal.dr->fileId;
            }
            else                                // INO number is in the record KEY
            {                                   // Inode record must be searched/read!
               ino = APFSidPrim( nodeKey.ds->fsId);
            }
            TxFreeMem( node);                   // free existing node record
            nodeKey.in = &inoKey;               // create the key for Btree search
            inoKey.fsId = APFSmkFsId( ino, APFST_INODE); // hybrid primary/secondary key value
            leafId = dfsApfsSearchCatalog( nodeKey, &index, &node);
            if ((leafId == 0) || (leafId == L64_NULL) ||
                (dfsApfsNodeIdx2Rec( node, index, NULL, &nodeKey, &vLength, &nodeVal) != APFST_INODE))
            {
               TRACES(("Failed to locate INODE record!\n"));
               rc = DFS_BAD_STRUCTURE;
            }
            break;

         case APFST_INODE:
            ino = APFSidPrim( nodeKey.in->fsId); // make INO value available for search EXTENTS
            break;

         default:
            rc = DFS_ST_MISMATCH;               // SN/index not for a valid record-type
            break;
      }

      if (rc == NO_ERROR)                       // we have a valid Inode now
      {
         recKind = ((nodeVal.in->inMode & S_IFLNK) == S_IFLNK) ? 's' : S_ISDIR( nodeVal.in->inMode)  ? 'D' : 'f';
         if (recKind == 'f')                    // normal file, create SSPACE extents
         {                                      // get byte-size from Inode Xfield
            if (vLength > sizeof( S_INODE_VAL)) // there are Xfields
            {
               S_XF_BLOB *xfb  = (S_XF_BLOB *) (nodeVal.in->inXfields);
               BYTE      *data = ((BYTE *) xfb->xfData) + (xfb->xfExtents * sizeof( S_XF_FIELD));

               for (i = 0; i < xfb->xfExtents; i++)
               {
                  if (xfb->xfData[ i].xfType == XFT_INO_DATASTREAM)
                  {
                     ispace->byteSize = ((S_DSTREAM *) data)->dsByteSize;
                     break;
                  }
                  data += ((xfb->xfData[ i].xfSize + 7) & 0xFFF8); //  next Xfield (8 byte size align)
               }
            }
            while (rc == NO_ERROR)              // valid INODE (number), and no errors
            {
               TxFreeMem( node);                // free existing node record
               nodeKey.fe = &fexKey;            // create the key for Btree search
               fexKey.fsId = APFSmkFsId( ino, APFST_FILE_EXTENT); // hybrid primary/secondary key value
               fexKey.feLogAddress = rOffset;   // secondary key is relative offset in the file
               leafId = dfsApfsSearchCatalog( nodeKey, &index, &node);
               if ((leafId != 0) && (leafId != L64_NULL) &&
                   (dfsApfsNodeIdx2Rec( node, index, NULL, NULL, NULL, &nodeVal) == APFST_FILE_EXTENT))
               {
                  //- Add this extent to the SSPACE structure
                  if (areas <= extents)         // areas used up ?
                  {
                     areas = areas * 8 + 16;
                     TRACES(("(re)Allocate %u space chunks, at rel-offset %llx\n", areas, rOffset))
                     sp = (S_SPACE *) realloc( sp, (size_t) (areas+1) * sizeof(S_SPACE));
                  }
                  if (sp != NULL)
                  {
                     sp[ extents].size  = (nodeVal.fe->feLengthFlags & APFSFL_MASK) / dfsGetSectorSize();
                     sp[ extents].start = dfsApfsBlock2Lsn( nodeVal.fe->feBlockNr);

                     rOffset += (nodeVal.fe->feLengthFlags & APFSFL_MASK); // next offset to search for
                     extents++;                 // and move to next extent in SSPACE
                  }
                  else
                  {
                     rc = DFS_ALLOC_ERROR;
                  }
               }
               else
               {
                  TRACES(("No FILE_EXTENT record for offset: %llx\n", rOffset));
                  break;
               }
            }
         }
         else if (recKind == 's')               // symlink, create single in-memory area
         {
            TxFreeMem( node);                   // free existing node record
            nodeKey.xa = &xatKey;               // create the key for Btree search
            xatKey.fsId = APFSmkFsId( ino, APFST_XATTR); // hybrid primary/secondary key value
            xatKey.xaNameLength = strlen( XA_SYMLINK_NAME) + 1; // includes the termination char
            strcpy( (char *) xatKey.xaName, XA_SYMLINK_NAME);
            leafId = dfsApfsSearchCatalog( nodeKey, &index, &node);
            if (((leafId != 0) && ((leafId != L64_NULL))) &&
                (dfsApfsNodeIdx2Rec( node, index, NULL, &nodeKey, &vLength, &nodeVal) == APFST_XATTR))
            {
               ispace->byteSize = strlen( (char *) nodeVal.xa->xaData) + 1;
               if ((sp = TxAlloc( 1, ispace->byteSize)) != NULL)
               {
                  strcpy( (char *) sp, (char *) nodeVal.xa->xaData);
               }
               else
               {
                  rc = DFS_ALLOC_ERROR;
               }
            }
            else
            {
               //- Links with missing XATTR linkname will be recovered as empty files!
               TRACES(("Symlink with missing XATTR linkname!\n"));
            }
         }
         else
         {
            //- Directories will result in an empty allocation
            TRACES(("Attempt to get allocation for a Directory!\n"));
         }
      }
      ispace->space  = sp;
      ispace->chunks = extents;
   }
   else
   {
      rc = DFS_ST_MISMATCH;                     // SN/index not for a valid LEAF node
   }
   RETURN (rc);
}                                               // end 'dfsApfsGetAllocSpace'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Find latest checkpoint, activate that or specified older one
/*****************************************************************************/
ULONG dfsApfsFindCheckpoint
(
   ULONG               cpIndex,                 // IN    Checkpoint, 0 is latest
   ULONG               verbosity                // IN    Display CP details/summary
)
{
   ULONG               rc = NO_ERROR;           // function return
   ULN64               latestXid = 0;           // Xid   of latest  CP found sofar
   ULONG               thisCpi;                 // Index of current CP checked
   ULONG               cpSbCpi = 0;             // Index of matched SB
   ULN64               cpLsn;                   // LSN for checkpoint (map or SB)
   S_B_HEADER         *cpHdr = (S_B_HEADER *) apfs->cpSup;    //- buffer alias, object header
   BYTE               *cpBlk = (BYTE *)       apfs->cpSup;    //- buffer alias, read/validate
   ULONG               size  = apfs->sup->csSbDescBlockCount; //- Nr of blocks to check

   ENTER();
   TRACES(("Activate checkpoint# : %u  verbosity: %u\n", cpIndex, verbosity));

   //- Assumes that there is ONE CPMAP and one SB block per checkpoint!
   //- Note that the FIRST block in the desc-area can be an SB or a CheckpointMap!
   //- So need to match a CPM with the CP in the next block to know it is valid

   txwAllowUserStatusMessages( TRUE);           // only our own messages now
   txwSetSbviewStatus( "Locating container superblock and object map  ...", cSchemeColor);

   TRACES(("start search for latest c-superblock at block: 0x%llx\n", apfs->sup->csSbDescBaseBlock));

   if ((apfs->sup->csSbDescBaseBlock  != 0) &&  // minimal sanity check
       (apfs->sup->csSbDescBaseBlock  < apfs->sup->csTotalBlocks) &&
       (apfs->sup->csSbDescBlockCount < 0xffff))
   {
      for ( thisCpi = 0,        cpLsn = dfsApfsBlock2Lsn( apfs->sup->csSbDescBaseBlock);
           (thisCpi < size) && (rc == NO_ERROR);
            thisCpi++,          cpLsn += apfs->SectorsPerBlock)
      {
         if ((rc = dfsRead( cpLsn, apfs->SectorsPerBlock, cpBlk)) == NO_ERROR)
         {
            if (dfsApfsValidBlock( cpBlk, apfs->BlockSize))
            {
               TRACES(("Check block type 0x%8.8x  Xid: 0x%16.16llx\n", cpHdr->objectType, cpHdr->transactId));

               switch (APFShdrType( cpHdr))
               {
                  case BT_CHECKP_MAP:
                     if (cpHdr->transactId > latestXid)
                     {
                        latestXid = cpHdr->transactId;
                     }
                     break;

                  case BT_C_SUPERBLOCK:
                     if (cpHdr->transactId == latestXid) // matching SB to previous MAP block
                     {
                        cpSbCpi = thisCpi;      // remember index for matching SB
                     }
                     break;

                  default:
                     //- ignore other type of blocks (should not be there anyway)
                     break;
               }
            }
         }
      }
   }
   if (cpSbCpi != 0)                            // found a valid latest checkpoint SB
   {                                            // calculate desired index
      thisCpi = (size + cpSbCpi - cpIndex - cpIndex) % size; // desired SB index

      TRACES(("latestXid: 0x0%llx  cpSbCpi: %u thisCpi: %u\n", latestXid, cpSbCpi, thisCpi));

      cpLsn = dfsApfsBlock2Lsn( apfs->sup->csSbDescBaseBlock + thisCpi);
      if ((rc = dfsRead( cpLsn, apfs->SectorsPerBlock, (BYTE *) apfs->cpSup)) == NO_ERROR)
      {
         //- We KNOW that the matching checkpoint-MAP is in the previous block!
         rc = dfsRead( cpLsn - apfs->SectorsPerBlock, apfs->SectorsPerBlock, (BYTE *) apfs->cpMap);
         if (rc == NO_ERROR)
         {
            if ((APFSobjType( apfs->cpMap) == BT_CHECKP_MAP) &&
                (APFSobjType( apfs->cpSup) == BT_C_SUPERBLOCK))
            {
               apfs->cpLsn = cpLsn;
               if (verbosity >= TXAO_VERBOSE)
               {
                  TxPrint("Active Checkpoint : %4u = SB index %4u with transaction - Xid : 0x%16.16llx\n",
                                               cpIndex, thisCpi, apfs->cpSup->blockHdr.transactId);
                  if (cpIndex != 0)
                  {
                     TxPrint("Latest Checkpoint :    0 = SB index %4u with transaction - Xid : 0x%16.16llx\n", cpSbCpi, latestXid);
                     TxPrint("                       You are looking at an OLDER state of the filesystem!\n");
                  }
               }
               apfs->cTransactions = latestXid - apfs->sup->blockHdr.transactId;
               if (verbosity > TXAO_SILENT)
               {
                  TxPrint("Filesystem status : APFS container seems to be ");
                  if      (apfs->cTransactions == 0)
                  {
                     TxPrint("%sUNMOUNTED%s, and %sSTABLE", CBG, CNN, CBG);
                  }
                  else if (apfs->cTransactions <= APFS_MOUNT_TR_LIMIT)
                  {
                     TxPrint("%sMOUNTED%s, but %sSTABLE", CBY, CNN, CBG);
                  }
                  else
                  {
                     TxPrint("%sMOUNTED%s, and %sVOLATILE!", CBR, CNN, CBR);
                  }
                  TxPrint("%s  (#transactions:%llu)\n\n", CNN, apfs->cTransactions);
               }

               dfsApfsFreeObjectMap( &apfs->cOmap); // free existing map, if any
               rc = dfsApfsCacheObjectMap( apfs->cpSup->csObjectMapBlock, &apfs->cOmap);

               if (verbosity > TXAO_QUIET)
               {
                  TxPrint("Superblock  index : %-4u %s   ", thisCpi,
                                    (cpIndex == 0) ? "is last checkpoint" : "has been selected ");
                  dfsX10( "BlockNr  : ", apfs->sup->csSbDescBaseBlock + thisCpi, CNM, "\n");
               }
               dfsApfsDisplaySuperBlock( (BYTE *) apfs->cpSup); // needs the Objectmap for proper display!
               nav.this = apfs->cpLsn;

               if (rc == NO_ERROR)              // result from CacheObjectMap
               {
                  if (verbosity > TXAO_QUIET)
                  {
                     TxPrint( "\n");            // separate from superblock output
                  }

                  dfsApfsBitMapFree( SPD_MAIN); // Free existing bitmap cache
                  dfsApfsBitMapInit( SPD_MAIN, verbosity);
                  if ((apfs->SpMgr != NULL) && (apfs->SpMgr->spDev[ SPD_TIER2].spdBlockCount != 0))
                  {
                     dfsApfsBitMapFree( SPD_TIER2); // Free existing bitmap cache
                     dfsApfsBitMapInit( SPD_TIER2, verbosity);
                  }
                  if (TxaOption('a'))           // show NO allocation by default
                  {
                     dfsApfsAllocMap( 0, 0, "@", NULL); // get/show alloc and minimum size
                  }
                  txwSetSbviewStatus( "Building VOLUME list and virtual object cache ...", cSchemeColor);
                  dfsa->FsModeSelist = dfsApfsVolumeSelist();         //- create selection list for user select

                  rc = dfsApfsSelectVolume( apfs->vIndex, verbosity); //- activate the same volume
               }
               else
               {
                  TxPrint( "\nSelected checkpoint/superblock has invalid ObjectMap! (FS mounted ?)\n");
                  rc = NO_ERROR;                // no vague other error explanations ...
               }
            }
            else
            {
               rc = DFS_ST_MISMATCH;
            }
         }
      }
   }
   txwSetSbviewStatus( "", cSchemeColor);       // clear CP progress text
   txwAllowUserStatusMessages( FALSE);          // resume default status text
   RETURN (rc);
}                                               // end 'dfsApfsFindCheckpoint'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Activate volume specified by its index in the current superblock
/*****************************************************************************/
ULONG dfsApfsSelectVolume
(
   ULONG               vIndex,                  // IN    Volume index, 0 is first
   ULONG               verbosity                // IN    Display CP details/summary
)
{
   ULONG               rc = NO_ERROR;           // function return
   ULN64               vsBlockNr;

   ENTER();
   TRACES(("Activate volume# : %u\n", vIndex));

   if ((vIndex < apfs->cpSup->csMaxVolCount) && (apfs->cpSup->csVolumeOid[ vIndex] != 0))
   {
      if ((vsBlockNr   = dfsApfsVirt2Phys( apfs->cpSup->csVolumeOid[ vIndex], APFS_ANY_XID, apfs->cOmap)) != 0)
      {
         apfs->vsLsn = dfsApfsBlock2Lsn( vsBlockNr);
         if ((rc = dfsRead( apfs->vsLsn, apfs->SectorsPerBlock, (BYTE *) apfs->vsBlk)) == NO_ERROR)
         {
            if (APFSobjType( apfs->vsBlk) == BT_V_SUPERBLOCK)
            {
               dfsApfsFreeObjectMap( &apfs->vOmap); // free existing map, if any
               rc = dfsApfsCacheObjectMap( apfs->vsBlk->vsObjectMapOid, &apfs->vOmap);
               if (rc == NO_ERROR)
               {
                  if (verbosity >= TXAO_VERBOSE)
                  {
                     TxPrint( "\n");            // separate NL cache info from Container stuff ...
                  }
                  txwSetSbviewStatus( "Building CATALOG and EXTENT tree leaf caches ...", cSchemeColor);
                  dfsApfsCacheNextLeafs( TRUE,  verbosity, apfs->vsBlk->vsRootTreeOid,     &apfs->vCatTree);
                  dfsApfsCacheNextLeafs( FALSE, verbosity, apfs->vsBlk->vsExtentsTreeOid,  &apfs->vExtTree);
                  dfsApfsCacheNextLeafs( FALSE, verbosity, apfs->vsBlk->vsSnapMetaTreeOid, &apfs->vSnapTree);

                  if (verbosity > TXAO_QUIET)
                  {
                     TxPrint("Volume with index : %-2u loaded and activated   ", vIndex);
                     dfsX10( "BlockNr  : ", vsBlockNr, CNM, "\n");
                  }
                  dfsApfsBlockVsuper( (BYTE *) apfs->vsBlk);
                  nav.this = apfs->vsLsn;
               }
            }
            else
            {
               rc = DFS_ST_MISMATCH;
            }
         }
      }
      else
      {
         TxPrint( "\nSpecified volume index : %u has no valid Superblock, %sinaccessible!%s\n", vIndex, CBR, CNN);
      }
   }
   else if (verbosity > TXAO_QUIET)
   {
      TxPrint( "\nSpecified volume index : %u is %sinvalid!%s\n", vIndex, CBR, CNN);
   }
   RETURN (rc);
}                                               // end 'dfsApfsSelectVolume'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read Object-Map structure, then recursively read the map tree into memory
/*****************************************************************************/
ULONG dfsApfsCacheObjectMap
(
   ULN64               omapBlock,               // IN    Blocknr omap object
   S_BT_NODE         **omapTree                 // OUT   Allocated omap tree
)
{
   ULONG               rc = NO_ERROR;           // function return
   S_OBJECT_MAP       *map;                     // Object-Map object

   ENTER();
   TRACES(("Object-Map   block: 0x0%llx\n", omapBlock));

   if ((map = TxAlloc( 1, apfs->BlockSize)) != NULL)
   {
      rc = dfsRead( dfsApfsBlock2Lsn( omapBlock), (apfs->BlockSize / dfsGetSectorSize()), (BYTE *) map);
      if (rc == NO_ERROR)
      {
         if (APFSobjType( map) == BT_OBJECT_MAP)
         {
            rc = dfsApfsCacheOmTree( map->omTreeOid, omapTree);
         }
         else
         {
            rc = DFS_ST_MISMATCH;
         }
      }
      TxFreeMem( map);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   TRACES(("Omap root at: %p\n", *omapTree));
   RETURN (rc);
}                                               // end 'dfsApfsCacheObjectMap'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read an Object-Map tree structure into memory from specified ROOT block
// Replaces blocknr links to child-nodes by node-pointers, with MAGIC flag
/*****************************************************************************/
static ULONG dfsApfsCacheOmTree
(
   ULN64               treeBlock,               // IN    Blocknr omap tree ROOT
   S_BT_NODE         **omapTree                 // OUT   Allocated omap tree
)
{
   ULONG               rc = NO_ERROR;           // function return
   S_BT_NODE          *omn;                     // (sub) tree node

   ENTER();
   TRACES(("Add tree node block: 0x0%llx\n", treeBlock));

   if ((omn = TxAlloc( 1, apfs->BlockSize)) != NULL)
   {
      BYTE            *block = (BYTE *) omn;

      rc = dfsRead( dfsApfsBlock2Lsn( treeBlock), (apfs->BlockSize / dfsGetSectorSize()), (BYTE *) omn);
      if (rc == NO_ERROR)
      {
         if (((omn->btNode.btFlags & BNF_LEAF)       == 0) && // it is not a leaf node
             ((omn->btNode.btFlags & BNF_FIXED_SIZE) != 0)  ) // and it is fixed size
         {
            ULONG            entry;
            ULONG            valOffs = APFS_BLOCKSIZE;
            S_BT_FIXED      *fixToc  = (S_BT_FIXED *) (omn->btData + omn->btNode.btToc.Offset);
            S_OMAP_VLINK    *vLink;

            TRDUMP(70, "Object-Map block, TOC, upto key-area:\n", block, 0x100, 0);
            TRDUMP(70, "Data area at end, initial    (vLink):\n", block + APFS_BLOCKSIZE - 0x100, 0x100, 0);

            if (APFSobjType( omn) == BT_BTREE_ROOT)
            {
               valOffs -= sizeof( S_BTREE_INFO); // values start just before the INFO struct
            }

            //- iterate over entries, and add the child-node links, as allocated nodes
            //- Note that the VALUEs in a non-LEAF block are ALWAYS size 8!
            for (entry = 0; (entry < omn->btNode.btCount) && (rc == NO_ERROR); entry++)
            {
               vLink = (S_OMAP_VLINK *) (block + valOffs - fixToc[ entry].nVoffset);

               TRACES(("vLink @: 0x%4.4x  ovPaddr:%16.16llx\n", valOffs - fixToc[ entry].nVoffset, vLink->ovPaddr));

               rc = dfsApfsCacheOmTree( vLink->ovPaddr, (S_BT_NODE **) &vLink->ovLnode);

               TRDUMP(70, "Data area at end, updated    (vLink):\n", block + APFS_BLOCKSIZE - 0x100, 0x100, 0);
            }
         }
      }
      *omapTree = omn;                          // attach the (sub) tree
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   TRACES(("Omap node at: %p\n", *omapTree));
   RETURN (rc);
}                                               // end 'dfsApfsCacheOmTree'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Free root and tree-node memory for an ALLOCATED Object Map
/*****************************************************************************/
void dfsApfsFreeObjectMap
(
   S_BT_NODE         **pOmap                    // IN    Ptr to Allocated omap
)
{
   S_BT_NODE          *omn   = *pOmap;
   BYTE               *block = (BYTE *) omn;
   ENTER();

   TRACES(("Free node: %p\n", omn));

   if (omn != NULL)
   {
      if (((omn->btNode.btFlags & BNF_LEAF)       == 0) && // it is not a leaf node
          ((omn->btNode.btFlags & BNF_FIXED_SIZE) != 0)  ) // and it is fixed size
      {
         ULONG            entry;
         ULONG            valOffs = APFS_BLOCKSIZE;
         S_BT_FIXED      *fixToc  = (S_BT_FIXED *) (omn->btData + omn->btNode.btToc.Offset);
         S_OMAP_VLINK    *vLink;

         if (APFSobjType( omn) == BT_BTREE_ROOT)
         {
            valOffs -= sizeof( S_BTREE_INFO);   // values start just before the INFO struct
         }

         //- iterate over entries, and free the child-node links
         for (entry = 0; entry < omn->btNode.btCount; entry++)
         {
            vLink = (S_OMAP_VLINK *) (block + valOffs - fixToc[ entry].nVoffset);
            dfsApfsFreeObjectMap( (S_BT_NODE **) &vLink->ovLnode); // free this child (subtree)
         }
      }
      TxFreeMem( *pOmap);                       // free the node itself, and set to NULL
   }
   VRETURN ();
}                                               // end 'dfsApfsFreeObjectMap'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Resolve physical block number for virtual Oid + Xid, using given Object Map
/*****************************************************************************/
ULN64 dfsApfsVirt2Phys                          // RET   Physical block nr or 0
(
   ULN64               virt,                    // IN    Virtual block nr (Oid)
   ULN64               vxid,                    // IN    Transaction ID   (Xid)
   S_BT_NODE          *omap                     // IN    Object Map tree to use
)
{
   ULN64               rc = 0;                  // function return
   ULONG               sanity = 10;             // limit levels of nodes to search
   S_BT_NODE          *omn    = omap;           // object map node being examined

   ENTER();
   TRACES(("virt:%llx vxid:%llx\n", virt, vxid));

   if ((omap != NULL) && (APFSobjType( omap) == BT_BTREE_ROOT))
   {
      do
      {
         TRLEVX(200,("node: %p\n", omn));
         if ((omn->btNode.btFlags & BNF_LEAF) == 0)
         {
            BYTE            *block   = (BYTE *) omn;
            ULONG            entry;
            ULONG            keyOffs = sizeof( S_B_HEADER) + sizeof( S_BTREE_NODE) + omn->btNode.btToc.Offset + omn->btNode.btToc.Length;
            ULONG            valOffs = APFS_BLOCKSIZE;
            S_BT_FIXED      *fixToc  = (S_BT_FIXED *) (omn->btData + omn->btNode.btToc.Offset);
            S_OMAP_VLINK    *vLink;
            S_OMAP_KEY      *key;
            S_BT_NODE       *candidate = NULL;  // candidate next level node
            int              keyComp;           // key compare result

            if (APFSobjType( omn) == BT_BTREE_ROOT)
            {
               valOffs -= sizeof( S_BTREE_INFO); // values start just before the INFO struct
            }

            //- iterate over index-entries; find LARGEST key less or equal to reference
            for (entry = 0; entry < omn->btNode.btCount; entry++)
            {
               key = (S_OMAP_KEY *) (block + keyOffs + fixToc[ entry].nKoffset);
               keyComp = dfsApfsOmapKeyCompare( key, virt, vxid);

               if (keyComp <= 0)                // entry-key is less or equal, node is candidate
               {
                  vLink = (S_OMAP_VLINK *) (block + valOffs - fixToc[ entry].nVoffset);
                  candidate = (S_BT_NODE *)  vLink->ovLnode;
               }
               else                             // record key was larger
               {
                  TRLEVX(200,("next level node: %p\n", candidate));
                  omn = candidate;
                  break;                        // candidate is the next branch to follow
               }
            }
         }
         else
         {
            TRLEVX(200,("Exit index-searching on (leaf) node: %p\n", omn));
         }
      } while (sanity-- && (omn != NULL) && ((omn->btNode.btFlags & BNF_LEAF) == 0));

      if ((omn != NULL) && (omn->btNode.btFlags & BNF_LEAF)) // found the matching leaf node
      {
         BYTE            *block   = (BYTE *) omn;
         ULONG            entry;
         ULONG            keyOffs = sizeof( S_B_HEADER) + sizeof( S_BTREE_NODE) + omn->btNode.btToc.Offset + omn->btNode.btToc.Length;
         ULONG            valOffs = APFS_BLOCKSIZE;
         S_BT_FIXED      *fixToc  = (S_BT_FIXED *) (omn->btData + omn->btNode.btToc.Offset);
         S_OMAP_VALUE    *value   = NULL;       // candidate indicator and final found value
         S_OMAP_KEY      *key;
         int              keyComp;              // key compare result

         if (APFSobjType( omn) == BT_BTREE_ROOT)
         {
            valOffs -= sizeof( S_BTREE_INFO);   // values start just before the INFO struct
         }

         //- iterate over leaf-entries until key(mis)match
         for (entry = 0; entry < omn->btNode.btCount; entry++)
         {
            key = (S_OMAP_KEY *) (block + keyOffs + fixToc[ entry].nKoffset);
            keyComp = dfsApfsOmapKeyCompare( key, virt, vxid);

            if (keyComp == 0)                   // exact key match found
            {
               TRLEVX(200,("Primary match with equal Xid\n"));
               value = (S_OMAP_VALUE *) (block + valOffs - fixToc[ entry].nVoffset);
               break;
            }
            else if (keyComp == -2)             // primary match, secondary smaller
            {                                   // remember value as best-match candidate
               value = (S_OMAP_VALUE *) (block + valOffs - fixToc[ entry].nVoffset);
            }
            else if (keyComp > 0)               // record key was larger, abort
            {
               if (value != NULL)               // there is a candidate
               {
                  TRLEVX(200,("Primary match with smaller Xid\n"));
               }
               TRLEVX(200,("Leaf node abort search on record: %u\n", entry));
               break;                           // wanted key is not present
            }
         }
         if (value != NULL)                     // exact match or best candidate ...
         {
            rc = value->ovPaddr;                // return physical address (block number)
         }
      }
   }
   RETN64 (rc);
}                                               // end 'dfsApfsVirt2Phys'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Compare Object-Map key to supplied Oid (primary) and Xid (secondary) values
// Shows PRIMARY key status (-1,0,+1) and SECONDARY (-2,+2) when primary is EQ
/*****************************************************************************/
static int dfsApfsOmapKeyCompare                // RET   0=EQ; -1=smaller; 1=larger
(                                               //       2ndry -2=smaller; 2=larger
   S_OMAP_KEY         *eKey,                    // IN    entry key
   ULN64               oid,                     // IN    virtual object ID
   ULN64               xid                      // IN    related transction ID
)
{
   int                 rc = 0;                  // function return, default to 'match'

   if (eKey->okOid == oid)                      // match for primary key part ?
   {
      if (eKey->okXid == xid)                   // match for secondary key part ?
      {
         rc = 0;                                // complete match
      }
      else if (eKey->okXid > xid)
      {
         rc =  2;                               // entryKey larger  than reference, but primary EQUAL
      }
      else
      {
         rc = -2;                               // entryKey smaller than reference, but primary EQUAL
      }
   }
   else if (eKey->okOid > oid)
   {
      rc =  1;                                  // entryKey larger  than reference
   }
   else
   {
      rc = -1;                                  // entryKey smaller than reference
   }
   TRLEVX(200,("OmapKeyCompare RC:%+d for kOid:%llx oid:%llx  kXid:%llx xid:%llx\n", rc, eKey->okOid, oid, eKey->okXid, xid));
   return (rc);
}                                               // end 'dfsApfsOmapKeyCompare'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// APFS read and check type for an APFS Btree Node from its sector number
/*****************************************************************************/
ULONG dfsApfsReadChkCatNode
(
   ULN64               lsn,                     // IN    Node LSN
   S_BT_NODE         **btNode                   // OUT   Btree Node structure
)
{
   ULONG               rc = NO_ERROR;
   BYTE                st = ST_UDATA;
   BYTE               *node;

   ENTER();
   TRARGS(("Inode LSN: %llX\n", lsn));

   if ((node = TxAlloc( apfs->SectorsPerBlock, dfsGetSectorSize())) != NULL)
   {
      if ((rc = dfsRead( lsn, apfs->SectorsPerBlock, node)) == NO_ERROR)
      {
         if (dfsApfsValidBlock( node, apfs->BlockSize))
         {
            st = dfsApfsBlock2SecType( ((S_B_HEADER *) node)->objectType);
         }
         switch (st)
         {
            case ST_BTREE:
            case ST_ANODE:                      // any node type is OK
               break;

            default:
               rc = DFS_ST_MISMATCH;
               break;
         }
      }
      if (rc != NO_ERROR)
      {
         TxFreeMem( node);
      }
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   *btNode = (S_BT_NODE *) node;
   RETURN(rc);
}                                               // end 'dfsApfsReadChkCatNode'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read FS-tree structure, then recursively cache all next LEAF block numbers
// Frees old cache, rebuilds for CURRENT active volume (supplied tree block#)
/*****************************************************************************/
ULONG dfsApfsCacheNextLeafs
(
   BOOL                virtual,                 // IN    Tree/Nodes are virtual
   ULONG               verbosity,               // IN    Display BM details/summary
   ULN64               tree,                    // IN    Tree root OID
   DFSAPFSLEAFCACHE   *nlc                      // INOUT Next-leaf cache
)
{
   ULONG               rc = NO_ERROR;           // function return
   S_BT_NODE          *root = NULL;             // root node block
   ULN64               rootBlock;

   ENTER();

   dfsApfsLeafCacheFree( nlc);                  // free old cache, if any

   if (virtual)                                 // tree is a virtual OID
   {
      rootBlock = dfsApfsVirt2Phys( tree, APFS_ANY_XID, apfs->vOmap);
   }
   else                                         // tree is a physical OID
   {
      rootBlock = tree;
   }

   if ((root = TxAlloc( 1, apfs->BlockSize)) != NULL)
   {
      rc = dfsRead( dfsApfsBlock2Lsn( rootBlock), (apfs->BlockSize / dfsGetSectorSize()), (BYTE *) root);
      if (rc == NO_ERROR)
      {
         if (APFSobjType( root) == BT_BTREE_ROOT)
         {
            nlc->vNodeCount    = root->btInfo.bfNodeCount;
            if ((nlc->vNlCache = TxAlloc( nlc->vNodeCount + 2, sizeof( ULN64))) != NULL)
            {
               nlc->vNlCount     = 1;           // first entry stays EMPTY (0 -> first LEAF)
               if ((root->btNode.btFlags & BNF_LEAF) != 0) // root is a leaf too ?
               {
                  nlc->vNlCache[ nlc->vNlCount++] = tree; // one and only leaf
                  TRACES(("Cache one-and-only (single node tree) : 0x0%llx\n", tree));
               }
               else                             // add next level(s) recursively
               {
                  rc = dfsApfsCacheNlNode( virtual, root, nlc);
               }
               if (verbosity >= TXAO_VERBOSE)
               {
                  TxPrint( "Vol %-8.8s Tree : %-9llu LEAF %s links cached\n",
                            nlc->vTreeName, nlc->vNlCount - 1, (virtual) ? "Virtual" : "PhysBlk");
               }
            }
            else
            {
               rc = DFS_ALLOC_ERROR;
            }
         }
         else
         {
            rc = DFS_ST_MISMATCH;
         }
      }
      TxFreeMem( root);
   }
   else
   {
      rc = DFS_ALLOC_ERROR;
   }
   TRACES(("nlc->vNlCount: %llu on tree: 0x0%llx\n", nlc->vNlCount, tree));
   RETURN (rc);
}                                               // end 'dfsApfsCacheNextLeafs'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Iterate node records, recurse next-level for nodes, add to cache for leafs
/*****************************************************************************/
static ULONG dfsApfsCacheNlNode
(
   BOOL                virtual,                 // IN    Tree/Nodes are virtual
   S_BT_NODE          *node,                    // IN    root/intermediate node
   DFSAPFSLEAFCACHE   *nlc                      // INOUT Next-leaf cache
)
{
   ULONG               rc = NO_ERROR;           // function return
   ULONG               valOffs = APFS_BLOCKSIZE;
   ULONG               entry;
   ULN64              *p64;                     // pointer to 64-bit key or value
   ULN64               nodeBlock;               // blocknumber (next) node
   S_BT_NODE          *linkNode;                // linked node to visit (recurse)

   ENTER();

   if (((node->btNode.btFlags & BNF_FIXED_SIZE) == 0) && // Variable size records
       ((node->btNode.btFlags & BNF_LEAF)       == 0)  ) // and not a leaf node
   {
      S_BT_ENTRY   *varToc  = (S_BT_ENTRY *) (node->btData + node->btNode.btToc.Offset);
      S_NODE_VAL    nodeVal;                    // pointer union to various known types

      if (APFSobjType( node) == BT_BTREE_ROOT)
      {
         valOffs -= sizeof( S_BTREE_INFO);      // values start just before the INFO struct
      }

      for (entry = 0; (entry < node->btNode.btCount) && (rc == NO_ERROR); entry++)
      {
         nodeVal.data = (((BYTE *) node) + valOffs - varToc[ entry].nVoffset);
         p64 = (ULN64 *) nodeVal.data;          // ptr to link to child node or leaf
         if (virtual)                           // link is a virtual OID
         {
            nodeBlock = dfsApfsVirt2Phys( *p64, APFS_ANY_XID, apfs->vOmap);
         }
         else                                   // link is a physical OID
         {
            nodeBlock = *p64;
         }
         if (nodeBlock != 0)                    // only add when found, seems there are 'dead' links!
         {
            if (node->btNode.btLevel > 1)       // more intermediate nodes, recurse
            {
               if ((linkNode = TxAlloc( 1, apfs->BlockSize)) != NULL)
               {
                  rc = dfsRead( dfsApfsBlock2Lsn( nodeBlock), (apfs->BlockSize / dfsGetSectorSize()), (BYTE *) linkNode);
                  if (rc == NO_ERROR)
                  {
                     if (APFSobjType( linkNode) == BT_BTREE_NODE) // should not be a root :)
                     {
                        rc = dfsApfsCacheNlNode( virtual, linkNode, nlc); // recurse into next-level node
                     }
                     else
                     {
                        rc = DFS_ST_MISMATCH;
                     }
                  }
                  TxFreeMem( linkNode);
               }
               else
               {
                  rc = DFS_ALLOC_ERROR;
               }
            }
            else                                // next level are leaves, add link to the cache
            {
               nlc->vNlCache[ nlc->vNlCount++] = *p64; // store (virtual) link and advance to next slot

               TRLEVX(200,("Cache node link : 0x0%llx\n", *p64));
            }
         }
         else                                   // invalid (virtual) link encountered (like '0x12345678-dead')
         {
            //- rc = DFS_BAD_STRUCTURE;
         }                                      // just continue, adding the found ones only
      }
   }
   else                                         // fixed-size nodes currently not supported
   {                                            // Fs-Tree, Extent and Snapshot are all variable
      rc = DFS_ST_MISMATCH;
   }
   TRACES(("nlc->vNlCount: %llu on node: 0x0%llx\n", nlc->vNlCount, node));
   RETURN (rc);
}                                               // end 'dfsApfsCacheNlNode'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Free allocated memory for a next-leaf btree cache
/*****************************************************************************/
void dfsApfsLeafCacheFree
(
   DFSAPFSLEAFCACHE   *nlc                      // INOUT Next-leaf cache
)
{
   ENTER();

   nlc->vNodeCount = 0;
   nlc->vNlCount   = 0;
   nlc->vLastIndex = 0;
   TxFreeMem( nlc->vNlCache);

   VRETURN ();
}                                               // end 'dfsApfsFreeObjectMap'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get virtual/physical link value for the NEXT LEAF in sequence (LEAF iterator)
/*****************************************************************************/
ULN64 dfsApfsNextLeafLink                       // RET   Next LEAF or 0
(
   ULN64               leaf,                    // IN    LEAF virtual/physBlock
   DFSAPFSLEAFCACHE   *nlc                      // IN    Next-leaf cache
)
{
   ULN64               rc = 0;                  // function return
   ULN64               i  = nlc->vLastIndex;

   ENTER();
   TRACES(("leaf:0x%llx  (last was: 0x%llx)\n", leaf, nlc->vNlCache[ i]));

   //- linear search for current LEAF (may use qsort and bsearch when too slow)
   //- but that will require doubling the size of the lookup table index->next

   if ((leaf == 0) || (nlc->vNlCache[ i] != leaf)) // It is NOT the previous one
   {
      i = 0;                                    // so start at the beginning ...
   }
   while (i < nlc->vNlCount)
   {
      if (nlc->vNlCache[ i] == leaf)
      {
         nlc->vLastIndex = i + 1;
         rc = nlc->vNlCache[ nlc->vLastIndex];
         break;
      }
      i++;
   }
   RETN64 (rc);
}                                               // end 'dfsApfsNextLeafLink'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Build selection-list with selectable Volume entries for a container
/*****************************************************************************/
static TXSELIST *dfsApfsVolumeSelist            // RET   selection list or NULL
(
                       void
)
{
   TXSELIST           *list  = NULL;            // total list
   int                 i;

#if defined (USEWINDOWING)
   TXS_ITEM           *item;                    // single item
   ULONG               lsize;                   // list-size
   ULN64               vsBlockNr;
#endif
   BYTE               *block;                   // one sector/block buffer
   S_V_SUPER          *vs;                      // buffer alias, volume superblock

   ENTER();

   if ((block = TxAlloc( 1, apfs->BlockSize)) != NULL) // to read volume-superblocks
   {
      vs = (S_V_SUPER *) block;                 // set the alias

      apfs->vCount = 0;                         // volumes present in superblock
      for (i = 0; i < apfs->cpSup->csMaxVolCount; i++)
      {
         if (apfs->cpSup->csVolumeOid[ i] != 0)
         {
            apfs->vCount++;
         }
      }
      TRACES(( "Volume count: %u\n", apfs->vCount));

      #if defined (USEWINDOWING)

      if (apfs->vCount != 0)                    // some Volumes present
      {
         lsize = apfs->vCount;
         if (TxSelCreate( lsize, lsize, lsize, TXS_AS_NOSTATIC, FALSE, NULL, &list) == NO_ERROR)
         {
            char           *listdescr;          // list level description

            list->astatus = TXS_AS_NOSTATIC      | // all dynamic allocated
                            TXS_LST_DESC_PRESENT | // with list description
                            TXS_LST_DYN_CONTENTS;

            if ((listdescr  = TxAlloc( 1, TXMAXTM)) != NULL)
            {
               sprintf( listdescr, "Volumes present in the opened APFS Container");

               list->miscInfo = listdescr;
            }
            list->selected = apfs->vIndex;      // current select volume is default

            for (i = 0; i < lsize; i++)         // all list entries
            {
               TRACES(("format list entry: %d for total of %u\n", i, lsize));
               if ((item  = TxAlloc( 1, sizeof(TXS_ITEM))) != NULL)
               {
                  list->count    = i +1;        // actual items in list
                  list->items[i] = item;        // attach item to list

                  item->helpid = TXWH_USE_OWNER_HELP; // from owner-menu-item

                  vsBlockNr = dfsApfsVirt2Phys( apfs->cpSup->csVolumeOid[ i], APFS_ANY_XID, apfs->cOmap);
                  if (dfsRead( dfsApfsBlock2Lsn( vsBlockNr), apfs->SectorsPerBlock, block) == NO_ERROR) // read Vsuper
                  {
                     if (((item->text = TxAlloc( 1, TXMAXTM)) != NULL) &&
                         ((item->desc = TxAlloc( 1, TXMAXTM)) != NULL)  )
                     {
                        TXTM            text;

                        item->value   = TXDID_MAX + i; // i is the vIndex

                        TxCopy(  text, vs->vsVolumeName, TXMAXTM - 5);
                        sprintf( item->text, "%2u  %s", i,  text);

                        dfsApfsTime2str( vs->vsFormattedBy.vsIdTimestamp, 0, text);
                        sprintf( item->desc, "Formatted %s UUID %s", text, dfsFsUuidValueNoSwap( vs->vsVolumeGuid));

                        //- get unique SelectChar starting at text[1] ...
                        item->index = TxSelGetCharSelect( list, i, 1);

                        TRACES(("text: %d '%s'\n", strlen(item->text), item->text));
                        TRACES(("desc: %d '%s'\n", strlen(item->desc), item->desc));
                     }
                  }
               }
            }
         }
      }
      #endif

      TxFreeMem( block);
   }
   RETURN( list);
}                                               // end 'dfsApfsVolumeSelist'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Determine allocation-bit for specified LSN, ALLOCATED beyond last block!
/*****************************************************************************/
ULONG dfsApfsAllocated                          // RET   LSN is allocated
(
   ULN64               lsn,                     // IN    LSN (can be PSN if Area!)
   ULN64               d2,                      // IN    dummy
   char               *dc,                      // IN    dummy
   void               *data                     // INOUT dummy
)
{
   ULONG               asn = dfstAreaD2Part( DFSTORE, lsn); // Area aware sn

   if (asn == lsn)                              // opened as partition or volume
   {
      if (asn < (apfs->Bm[ SPD_MAIN].blockCount * apfs->SectorsPerBlock)) // inside MAIN bitmap
      {
         if (asn < (apfs->Block * apfs->SectorsPerBlock)) // within valid block area
         {
            return((ULONG) dfsApfsBitmapCache( SPD_MAIN, dfsApfsLsn2Block( asn)));
         }
         else                                   // return ALLOCATED for any
         {                                      // lsn beyond last block
            return( 1);                         // to avoid CHECK errors
         }                                      // on non-standard bitmaps
      }
      else
      {
         return(DFS_PSN_LIMIT);
      }
   }
   else                                         // opened as area from FDISK
   {                                            // assume ALLOCATED!
      return( 1);                               // no SMART at disk level for now
   }
}                                               // end 'dfsApfsAllocated'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Align RO cache to position for block, and get current allocation-bit
// Note: supperts a SINGLE level of CAB nodes (limits FS size to 1500 TB)
/*****************************************************************************/
BOOL dfsApfsBitmapCache                         // RET   block is allocated
(
   ULONG               bmId,                    // IN    Bitmap to use (SFQ_)
   ULN64               bl                       // IN    Block number
)
{
   ULONG               rc = NO_ERROR;
   ULN64               bmRelBlock;              // relative block in Bitmap
   DFSAPFSBMCACHE     *cache = &apfs->Bm[ bmId];

   //- remove enter/ret tracing when it works, for better performance

   if ((apfs->SpMgr != NULL) && (apfs->SpMgr->spBlocksPerChunk != 0)) // avoid access/divide traps
   {
      bmRelBlock = bl / apfs->SpMgr->spBlocksPerChunk; // find relative BM block for bl

      if      (bmRelBlock == cache->bmSparse)   // BM block not present, block is FREE!
      {
         return FALSE;                          // fastest return possible no cleanup needed
      }
      else if (bmRelBlock != cache->bmBlock)    // needed block not cached
      {
         ULONG               cibIndex;          // cib link to use (in spmgr or CAB)
         ULONG               bmpIndex;          // Bitmap block index in CIB to use
         ULN64               bmpBlock;          // LEAF level bitmap block to be read
         S_SP_CIB_BLOCK     *cib = NULL;

         if (cache->bmType == BT_SPACEMAN_CIB)  // top level CIB (disks up to about 3TB)
         {
            cibIndex = bmRelBlock / apfs->SpMgr->spChunksPerCib; // CIB# to use within top-level
            bmpIndex = bmRelBlock % apfs->SpMgr->spChunksPerCib; // BMP# to use within CIB

            if (cibIndex < cache->bmCount)
            {
               cib  = cache->bmTopLevel[ cibIndex].cib;
               bmpBlock = cib->cibEntry[ bmpIndex].cieBmAddr;
            }
            else
            {
               rc = DFS_BAD_STRUCTURE;
            }
         }
         else                                   // top level CAB (huge disks up to 1500 TB)
         {
            ULN64            blocksPerCab  = apfs->SpMgr->spChunksPerCib * apfs->SpMgr->spCibsPerCab;
            ULN64            cabBlkNr;
            ULONG            cabIndex;
            S_SP_CAB_BLOCK  *cab;
            ULN64            cibBlock;          // blocknumber of CIB to read from CAB

            cabIndex = bmRelBlock / blocksPerCab; // CAB# to use within top-level
            cabBlkNr = bmRelBlock % blocksPerCab; // relative blocknr within CAB

            cibIndex = cabBlkNr / apfs->SpMgr->spChunksPerCib; // CIB# to use within CAB
            bmpIndex = cabBlkNr % apfs->SpMgr->spChunksPerCib; // BMP# to use within CIB

            if (cabIndex < cache->bmCount)
            {
               cab = cache->bmTopLevel[ cibIndex].cab;
               if (cibIndex <= cab->cabCibCount)
               {
                  if ((cib = TxAlloc( 1, apfs->BlockSize)) != NULL)
                  {
                     cibBlock = cab->cabCibBlock[ cibIndex];
                     rc = dfsRead( dfsApfsBlock2Lsn( cibBlock), apfs->SectorsPerBlock, (BYTE *) cib);
                     if (rc == NO_ERROR)
                     {
                        TRACES(("Read  CIB %u from block: %llx\n", cibIndex, cibBlock));
                        bmpBlock = cib->cibEntry[ bmpIndex].cieBmAddr;
                     }
                     TxFreeMem( cib);           // free temporary allocated CIB
                  }
                  else
                  {
                     rc = DFS_ALLOC_ERROR;
                  }
               }
               else
               {
                  rc = DFS_BAD_STRUCTURE;
               }
            }
            else
            {
               rc = DFS_BAD_STRUCTURE;
            }
         }

         if (rc == NO_ERROR)
         {
            if (bmpBlock == 0)                  // not present yet (sparse) so block is FREE
            {
               TRACES(("BitMapCache: Sparse rel: %llx for bl:%llx\n", bmRelBlock, bl));
               cache->bmSparse = bmRelBlock;    // cache this value, for speed-up
               return FALSE;                    // fast resturn
            }
            else
            {
               //- read the calculated bitmap block into the cache buffer
               rc = dfsRead( dfsApfsBlock2Lsn( bmpBlock), apfs->SectorsPerBlock, cache->Buffer);
               if (rc == NO_ERROR)
               {
                  cache->bmBlock = bmRelBlock;
                  TRACES(("BitMapCache: Read rel: %llx from block# %llx for bl:%llx\n", bmRelBlock, bmpBlock, bl));
               }
            }
         }
      }
      if ((rc == NO_ERROR) && (cache->Buffer != NULL))
      {
         ULONG bytepos    = (bl % (apfs->SpMgr->spBlocksPerChunk)) / 8; // byte index in sector
         BYTE  alloc_byte = cache->Buffer[ bytepos];
         if (((alloc_byte >> (bl & 0x07)) & 0x01) == 0) // LSB is lowest!
         {
            return FALSE;
         }
      }
   }
   return TRUE;                                 // block must be allocated
}                                               // end 'dfsApfsBitmapCache'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Read SPACEMANAGER info, when not done yet, initialize BitMap cache
// Note: NOT Area aware to allow usage from FDISK mode at the moment!
/*****************************************************************************/
ULONG dfsApfsBitMapInit
(
   ULONG               bmId,                    // IN    Bitmap to initialize (SPD_)
   ULONG               verbosity                // IN    Display BM details/summary
)
{
   ULONG               rc = NO_ERROR;
   DFSAPFSBMCACHE     *cache = &apfs->Bm[ bmId];
   ULN64               lsn;
   ULN64              *links;                   // array of CAB/CIB block numbers
   ULONG               i;

   ENTER();
   TRACES(("bmId: %u, verbosity: %u\n", bmId, verbosity));

   if (apfs->SpMgr == NULL)                  // Spacemanager not read yet ...
   {
      if ((apfs->cpMap->cpm_map[ CPM_SPMGR_INDEX].cpm_type & HDR_TYPE_VALUE_MASK) == BT_SPACEMANAGER)
      {
         if ((apfs->SpMgr = TxAlloc( 1, apfs->BlockSize)) != NULL)
         {
            lsn = dfsApfsBlock2Lsn( apfs->cpMap->cpm_map[ CPM_SPMGR_INDEX].cpm_paddr);
            rc = dfsRead( lsn, apfs->SectorsPerBlock, (BYTE *) apfs->SpMgr);

            if (APFSobjType( apfs->SpMgr) != BT_SPACEMANAGER)
            {
               TRACES(("SPMGR block from Checkpoint does NOT contain a spacemanager object!\n"));
               TxFreeMem( apfs->SpMgr);
               rc = DFS_ST_MISMATCH;
            }
         }
         else
         {
            rc = DFS_ALLOC_ERROR;
         }
      }
      else                                      // first item in checkpoint map is NOT a spmgr
      {
         TRACES(("Checkpoint first item is NOT a spacemanager!\n"));
         rc = DFS_ST_MISMATCH;
      }
   }

   if (rc == NO_ERROR)                          // so far so good, initialize the specified cache
   {
      if (verbosity > TXAO_QUIET)
      {
         TxPrint( "BitMap - %s : ", dfsApfsDeviceBmType( bmId));
      }
      if ((cache->Buffer = (BYTE *) TxAlloc( 1, apfs->BlockSize)) != NULL)
      {
         S_SPMGR_DEVICE *dev = &apfs->SpMgr->spDev[ bmId];
         cache->cacheId      = bmId;            // set the selected bitmap ID
         cache->bmSparse     = BMB_NONE;        // no BM sparse cached yet
         cache->bmBlock      = BMB_NONE;        // no BM block cached yet

         if (verbosity > TXAO_QUIET)
         {
            ULN64   usedBlocks = dev->spdBlockCount - dev->spdFreeCount;
            double  percentage = (((double) usedBlocks) * 100.0) / dev->spdBlockCount;

            dfsSz64("Free: ", dev->spdFreeCount * apfs->SectorsPerBlock, " ");
            dfsSz64("Used: ", usedBlocks        * apfs->SectorsPerBlock, " = ");
            if (dev->spdBlockCount != 0)
            {
               TxPrint( "%5.1lf%%", percentage);
            }
            TxPrint( "\n");
         }
         if (dev->spdCabCount != 0)             // there are CAB levels present
         {
            cache->bmType  = BT_SPACEMAN_CAB;
            cache->bmCount = dev->spdCabCount;
         }
         else
         {
            cache->bmType  = BT_SPACEMAN_CIB;
            cache->bmCount = dev->spdCibCount;
         }
         cache->blockCount = dev->spdBlockCount;
         links = (ULN64 *) (((BYTE *) apfs->SpMgr) + dev->spdIndexOffset);

         TRACES(("Nr of links: %u, of type: %s\n", cache->bmCount, dfsApfsBlockType(cache->bmType)));
         if ((cache->bmTopLevel = TxAlloc( cache->bmCount, sizeof( DFSBMTOPLEVEL))) != NULL)
         {
            for (i = 0; i < cache->bmCount; i++) // alloc and read all top-level blocks (CIB/CAB)
            {
               if ((cache->bmTopLevel[ i].cib = TxAlloc( 1, apfs->BlockSize)) != NULL)
               {
                  lsn = dfsApfsBlock2Lsn( links[ i]); // array of blocknumbers
                  rc  = dfsRead( lsn, apfs->SectorsPerBlock, (BYTE *) cache->bmTopLevel[ i].cib);
                  if (rc == NO_ERROR)
                  {
                     if ((cache->bmTopLevel[ i].cib->blockHdr.objectType & HDR_TYPE_VALUE_MASK) != cache->bmType)
                     {
                        TRACES(("Top-level link %u does NOT contain correct CIB/CAB object!\n", i));
                        rc = DFS_ST_MISMATCH;
                     }
                     else
                     {
                        if (verbosity >= TXAO_VERBOSE)
                        {
                           TxPrint( " BM top-level %3u : %s with %3u entries\n",
                                     cache->bmTopLevel[ i].cib->cibIndex,
                                    (cache->bmType == BT_SPACEMAN_CIB) ? "CIB" : "CAB",
                                     cache->bmTopLevel[ i].cib->cibEntryCount);
                        }
                        else
                        {
                           TRACES(("Top-level BM index: %3u with %3u entries\n",
                              cache->bmTopLevel[ i].cib->cibIndex, cache->bmTopLevel[ i].cib->cibEntryCount));
                        }
                     }
                  }
               }
               else
               {
                  rc = DFS_ALLOC_ERROR;
                  break;
               }
            }
         }
         else
         {
            rc = DFS_ALLOC_ERROR;
         }
      }
      else
      {
         rc = DFS_ALLOC_ERROR;
      }
   }
   RETURN (rc);
}                                               // end 'dfsApfsBitMapInit'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Free SPACEMANAGER info, including the BITMAP cache
/*****************************************************************************/
void dfsApfsBitMapFree
(
   ULONG               bmId                     // IN    Bitmap to free (SFQ_)
)
{
   DFSAPFSBMCACHE     *cache = &apfs->Bm[ bmId];
   ULONG               i;

   ENTER();
   TRACES(("bmId: %u\n", bmId));

   for (i = 0; i < cache->bmCount; i++)         // alloc and read all top-level blocks (CIB/CAB)
   {
      TxFreeMem( cache->bmTopLevel[ i].cib);    // free each link
   }
   TxFreeMem( cache->bmTopLevel);               // free the link array
   TxFreeMem( cache->Buffer);                   // free the BM buffer

   if (bmId == SPD_MAIN)
   {
      TxFreeMem( apfs->SpMgr);                  // free the SPMGR itself
   }
   cache->bmCount = 0;

   VRETURN();
}                                               // end 'dfsApfsBitMapFree'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Search Inode-record in catalog B-tree, returning parent-Ino / item-name / isDir
/*****************************************************************************/
ULONG dfsApfsIno2Parent                         // RET   result
(
   ULN64               ino,                     // IN    INO of item to search
   ULN64              *parent,                  // OUT   parent-INO        or NULL
   char               *name,                    // OUT   NAME of item, ""  or NULL
   BOOL               *isDir                    // OUT   item is DIR flag, or NULL
)
{
   ULONG               rc = DFS_NOT_FOUND;      // function return, default not found
   ULN64               leafId;
   USHORT              index;
   S_BT_NODE          *node;
   S_INODE_KEY         inKey;
   S_NODE_KEY          key;
   int                 i;

   ENTER();

   key.in = &inKey;
   key.in->fsId = APFSmkFsId( ino, APFST_INODE); // hybrid primary/secondary key value
   leafId = dfsApfsSearchCatalog( key, &index, &node);
   if ((leafId != 0) && ((leafId != L64_NULL)))
   {
      S_NODE_VAL nodeVal;
      USHORT     vLength;
      int        recType;

      recType = dfsApfsNodeIdx2Rec( node, index, NULL, NULL, &vLength, &nodeVal);
      if (recType == APFST_INODE)
      {
         if (name != NULL)
         {
            strcpy( name, "");                  // start empty
            if (vLength > sizeof( S_INODE_VAL)) // there are Xfields
            {
               S_XF_BLOB *xfb  = (S_XF_BLOB *) (nodeVal.in->inXfields);
               BYTE      *data = ((BYTE *) xfb->xfData) + (xfb->xfExtents * sizeof( S_XF_FIELD));

               for (i = 0; i < xfb->xfExtents; i++)
               {
                  if (xfb->xfData[ i].xfType == XFT_INO_FILENAME)
                  {
                     strcpy( name, (char *) data); // filename in Xfield is null-terminated
                     break;
                  }
                  data += ((xfb->xfData[ i].xfSize + 7) & 0xFFF8); //  next Xfield (8 byte size align)
               }
            }
         }
         if (isDir != NULL)
         {
            *isDir = (S_ISDIR( nodeVal.in->inMode));
         }
         if (parent != NULL)
         {
            *parent = nodeVal.in->inParentId;
         }
         rc = NO_ERROR;
      }
   }
   TRACES(("parent: %llx for ino: %llx = '%s' isDir:%s\n", (parent) ? *parent : 0, ino, (name) ? name : "-",
                                                                    (isDir) ? (*isDir) ? "yes" : "no" : "-"));
   RETURN (rc);
}                                               // end 'dfsApfsIno2Parent'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Search catalog B-tree for specified key, get Leafnode + optional index/node
/*****************************************************************************/
ULN64 dfsApfsSearchCatalog                      // RET   LEAF Node virtual-Id
(
   S_NODE_KEY          key,                     // IN    Search key
   USHORT             *record,                  // OUT   record-index,    or NULL
   S_BT_NODE         **leaf                     // OUT   ptr to leafnode, or NULL
)
{
   ULN64               rc = 0;                  // function return, default not found
   BYTE               *block;                   // BYTE alias for current node
   S_BT_NODE          *node;                    // current node
   ULN64               nodeNr;                  // virtual block number
   ULN64               blockNr;                 // Physical block number
   ULONG               sanity = BT_SANE_LEVELS; // limit levels of Indexes to search
   int                 keyComp;                 // key compare result
   ULONG               entry;
   S_NODE_KEY          nodeKey;
   S_NODE_VAL          nodeVal;                 // pointer union to various known types
   S_BT_ENTRY         *varToc;
   ULONG               keyOffs;
   ULONG               valOffs;
   ULONG               maxLeafRetries = 1;      // try in ONE next leaf, when not found in 1st

   ENTER();
   TRACES(("primary key: 0x%16.16llx\n", key.in->fsId));

   if ((nodeNr = apfs->vsBlk->vsRootTreeOid) != 0)
   {
      if ((block = TxAlloc( apfs->SectorsPerBlock, dfsGetSectorSize())) != NULL)
      {
         node = (S_BT_NODE *) block;
         do
         {
            if ((blockNr = dfsApfsVirt2Phys( nodeNr, APFS_ANY_XID, apfs->vOmap)) != 0)
            {
               if (dfsRead( dfsApfsBlock2Lsn( blockNr), apfs->SectorsPerBlock, block) == NO_ERROR)
               {
                  varToc  = (S_BT_ENTRY *) (node->btData + node->btNode.btToc.Offset);
                  keyOffs = sizeof( S_B_HEADER) + sizeof( S_BTREE_NODE) + node->btNode.btToc.Offset + node->btNode.btToc.Length;
                  valOffs = APFS_BLOCKSIZE;
                  if (APFSobjType( node) == BT_BTREE_ROOT)
                  {
                     valOffs -= sizeof( S_BTREE_INFO); // values start just before the INFO struct
                  }
                  if ((node->btNode.btFlags & BNF_LEAF) == 0) // no leaf, so must be root/index node
                  {
                     //- iterate over index-records; find LARGEST key less or equal to reference
                     for (entry = 0; entry < node->btNode.btCount; entry++)
                     {
                        nodeKey.data = (block + keyOffs + varToc[ entry].nKoffset);
                        nodeVal.data = (block + valOffs - varToc[ entry].nVoffset);

                        keyComp = dfsApfsCatalogKeyCompare( nodeKey, key);

                        if (keyComp <= 0)       // record-key is less or equal, node is candidate
                        {
                           nodeNr = *( nodeVal.p64); // virtual link to next level node
                        }
                        else                    // record key was larger
                        {
                           TRACES(("next level node: 0x0%llx\n", nodeNr));
                           break;               // nodeNr is the branch to follow
                        }
                     }
                  }
                  else
                  {
                     TRACES(("Exit index-searching on (leaf) nodeNr: 0x0%llx\n", nodeNr));
                  }
               }
            }
         } while (sanity-- && ((node->btNode.btFlags & BNF_LEAF) == 0));

         do                                     // now search the found leaf node(s)
         {
            if ((node->btNode.btFlags & BNF_LEAF) == BNF_LEAF)
            {
               //- iterate over leaf-records until key(mis)match
               for (entry = 0; entry < node->btNode.btCount; entry++)
               {
                  nodeKey.data = (block + keyOffs + varToc[ entry].nKoffset);
                  nodeVal.data = (block + valOffs - varToc[ entry].nVoffset);

                  keyComp = dfsApfsCatalogKeyCompare( nodeKey, key);

                  if ((keyComp == 0) || (keyComp == 2)) // key exact or tertiary wildcard match found
                  {
                     //- to be refined, for DIR-RECORD needs a hash-collision-detect,
                     //- and do an additional compare on the real name as well!
                     //- adjusting keyComp (this could be integrated in the compare function)

                     TRACES(("Leaf record found: %u\n", entry));

                     if (record != NULL)
                     {
                        *record = entry;
                     }
                     if (leaf != NULL)
                     {
                        *leaf = node;
                     }
                     rc = nodeNr;               // return virtual leaf node number
                     break;
                  }
                  else if (keyComp > 0)         // record key was larger, abort
                  {
                     TRACES(("Leaf node abort search on record: %u\n", entry));

                     rc = L64_NULL;             // signal search failed ...
                     break;                     // wanted key is not present
                  }
               }
            }
            if ((rc == 0) && (maxLeafRetries))  // not found, but at end of leaf, try next one
            {                                   // (needed in case of HASH collision for example)
               if ((nodeNr = dfsApfsNextLeafLink( nodeNr, &apfs->vCatTree)) != 0)
               {
                  if (((blockNr = dfsApfsVirt2Phys( nodeNr, APFS_ANY_XID, apfs->vOmap))    != 0)      &&
                       (dfsRead( dfsApfsBlock2Lsn( blockNr), apfs->SectorsPerBlock, block) == NO_ERROR))
                  {
                     TRACES(("Continue search in NEXT LEAF nodeNr: 0x0%llx\n", nodeNr));
                  }
                  else
                  {
                     nodeNr = 0;               // failure to continue in next leaf, no retry, abort
                  }
               }
            }
         } while ((rc == 0) && (nodeNr != 0) && (maxLeafRetries--)); // not found yet, try next until maximum

         if ((leaf == NULL) || (*leaf != node)) // node not returned to caller
         {
            TxFreeMem( node);
         }
      }
   }
   RETN64 (rc);
}                                               // end 'dfsApfsSearchCatalog'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Compare catalog-record-key to supplied search key of specified type
// Note: All types share a 64-bit PRIMARY/SECONDARY key value (parent/ino/xid)
// The lower 60 bits are PRIMARY, upper 4 SECONDARY, with Tertiary parts optional
// Shows PRI/SEC key status (-1,0,+1) and TERTIARY wildcard (2) for pri/sec EQ
// and (-2) in case the tertiary key has a hash collision (continue searching)
/*****************************************************************************/
static int dfsApfsCatalogKeyCompare             // RET   0=EQ;  -1=smaller; 1=larger
(                                               //       -2=collision 2=wildcard match
   S_NODE_KEY          rKey,                    // IN    record key in FS-tree
   S_NODE_KEY          sKey                     // IN    search key to compare
)
{
   int                 rc = 0;                  // default match
   int                 rType = APFSidType( rKey.dr->fsId); // get type from RECORD key
   int                 sType = APFSidType( sKey.dr->fsId); // get type from SEARCH key
   ULN64               rPrim = APFSidPrim( rKey.in->fsId); // primary part  RECORD key
   ULN64               sPrim = APFSidPrim( sKey.in->fsId); // primary part  SEARCH key
   BOOL                caseInsens = ((apfs->vsBlk->vsInCompatible & VFI_CASE_INSENS) != 0);

   ENTER();
   TRACES(("caseInsens:%u sKey:%llx rKey:%llx\n", caseInsens, sKey.in->fsId, rKey.in->fsId));

   if (rPrim == sPrim)                          // Match for primary key ?
   {
      if (rType == sType)                       // Matching types too (secondary key)
      {
         TRACES(("Rhash: 0x%8.8x  Shash: 0x%8.8x\n", rKey.dh->drLengthHash, sKey.dh->drLengthHash));
         switch (sType)                         // check tertiary key, if any
         {
            //- Note: Does NOT support DIR_RECORDS without the namehash at the moment  (undetectable here)
            case APFST_DIR_RECORD:
               if (sKey.dh->drLengthHash != 0)  // tertiary hash-value supplied, compare against record
               {
                  rc = (rKey.dh->drLengthHash == sKey.dh->drLengthHash) ? 0 :
                       (rKey.dh->drLengthHash >  sKey.dh->drLengthHash) ? 1 : -1;

                  if (rc == 0)                  // hash is equal, test for collision (name mismatch)
                  {
                     if (dfsUtf8Cmp( rKey.dh->drName, sKey.dh->drName, caseInsens) != 0)
                     {
                        rc = -2;                // signal SMALLER so search will continue with next ...
                     }
                  }
               }
               else                             // search key tertiary 0 (wildcard, any value)
               {
                  rc =  2;                      // recordKey tertiary ignored, tertiary WILDCARD match
               }                                // (supports searching for the FIRST pri/sec match)
               break;

            case APFST_FILE_EXTENT:  rc = (rKey.fe->feLogAddress == sKey.fe->feLogAddress) ? 0 :
                                          (rKey.fe->feLogAddress >  sKey.fe->feLogAddress) ? 1 : -1;  break;
            case APFST_SIBLING_LINK: rc = (rKey.sl->slSibId      == sKey.sl->slSibId)      ? 0 :
                                          (rKey.sl->slSibId      >  sKey.sl->slSibId)      ? 1 : -1;  break;
            case APFST_XATTR:        rc = dfsUtf8Cmp( rKey.xa->xaName, sKey.xa->xaName, caseInsens);  break;
            case APFST_SNAP_NAME:    rc = dfsUtf8Cmp( rKey.sn->snName, sKey.sn->snName, caseInsens);  break;
            default:                 rc = 0;                                                          break;
         }
         TRACES(("Prim Key and Type match, Tertiary result: %d\n", rc));
      }
      else                                      // order depends on the type values
      {
         rc = (rType > sType) ? 1 : -1;
         TRACES(("Prim Key same, Type result: %d\n", rc));
      }
   }
   else
   {
      rc = (rPrim > sPrim) ? 1 : -1;
   }
   RETURN (rc);
}                                               // end 'dfsApfsCatalogKeyCompare'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Find full PATH to Root-directory and optiona recType from Node LSN and index
/*****************************************************************************/
ULONG dfsApfsLsnInfo2Path                       // RET   result
(
   ULN64               start,                   // IN    node lsn
   ULONG               index,                   // IN    node index
   int                *rType,                   // OUT   recType or NULL
   char               *path                     // OUT   combined path string
)
{
   ULONG               rc = NO_ERROR;
   ULN64               ino;
   ULN64               parent;
   TXLN                thisName;                // name for current component
   BOOL                isDirectory;
   ULONG               sanity = 100;

   ENTER();
   if (path)
   {
      *path = '\0';
   }

   ino = dfsApfsLsnInfo2Ino( start, index, rType);
   while ((ino != 0) && (rc == NO_ERROR) && sanity--)
   {
      if (ino <= 2)                             // root, or worse :)
      {
         strcpy( thisName, "");                 // don't want the volumename!
         isDirectory = TRUE;
         parent = 0;                            // terminate after this ...
      }
      else
      {
         rc = dfsApfsIno2Parent( ino, &parent, thisName, &isDirectory);
      }
      if (rc == NO_ERROR)
      {
         if (isDirectory)
         {
            strcat( thisName, FS_PATH_STR);
         }
         strcat( thisName, path);               // append previous parts
         strcpy( path, thisName);               // and copy back ...

         ino = parent;                          // to next path component
      }
   }
   TRACES(("Found path to root for 0x%llX+%4.4x:'%s'\n", start, index, path));
   RETURN(rc);
}                                               // end 'dfsApfsLsnInfo2Path'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Find related INO and optional recType for a (Node) LSN and index info value
/*****************************************************************************/
ULN64 dfsApfsLsnInfo2Ino                       // RET   INO, or 0 when failed
(
   ULN64               nodeLsn,                 // IN    node lsn
   ULONG               index,                   // IN    node index
   int                *rType                    // OUT   recType or NULL
)
{
   ULN64               rc = 0;
   S_BT_NODE          *node;
   S_NODE_VAL          nodeVal;
   S_NODE_KEY          nodeKey;
   int                 recType = 0;

   ENTER();
   TRACES(("Node-Lsn: 0x0%llx index: 0x%x\n", nodeLsn, index));

   if (dfsApfsReadChkCatNode( nodeLsn, &node) == NO_ERROR)
   {
      recType = dfsApfsNodeIdx2Rec( node, index, NULL, &nodeKey, NULL, &nodeVal);
      switch (recType)
      {
         case APFST_DIR_RECORD:    rc = nodeVal.dr->fileId;             break;  //- from value
         case APFST_DIR_STATS:
         case APFST_FILE_EXTENT:
         case APFST_DSTREAM_ID:
         case APFST_XATTR:
         case APFST_SIBLING_LINK:
         case APFST_INODE:         rc = APFSidPrim( nodeKey.in->fsId);  break;  //- from pri key
         default:                                                       break;
      }
      TxFreeMem( node);
      if (rType != NULL)
      {
         *rType = recType;
      }
   }
   RETN64(rc);
}                                               // end 'dfsApfsLsnInfo2Ino'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Find catalog-record for a Node and index, get type, key and/or value info
/*****************************************************************************/
int dfsApfsNodeIdx2Rec                          // RET   recordtype, or 0 when failed
(
   S_BT_NODE          *node,                    // IN    node structure
   ULONG               index,                   // IN    node index
   USHORT             *kLen,                    // OUT   length of Key   record, or NULL
   S_NODE_KEY         *kRec,                    // OUT   catalog   Key   record (pointer)
   USHORT             *vLen,                    // OUT   length of Value record, or NULL
   S_NODE_VAL         *vRec                     // OUT   catalog   Value record (pointer)
)
{
   int                 rc = 0;
   S_NODE_KEY          nodeKey;
   S_NODE_VAL          nodeVal;
   S_BT_ENTRY         *varToc;
   ULONG               keyOffs;
   ULONG               valOffs;

   ENTER();
   TRACES(("node-ID: 0x%llx  index: %u  count:%u\n", node->blockHdr.objectId, index, node->btNode.btCount));

   if (index < node->btNode.btCount)
   {
      varToc  = (S_BT_ENTRY *) (node->btData + node->btNode.btToc.Offset);
      keyOffs = sizeof( S_B_HEADER) + sizeof( S_BTREE_NODE) + node->btNode.btToc.Offset + node->btNode.btToc.Length;
      valOffs = APFS_BLOCKSIZE;
      if (APFSobjType( node) == BT_BTREE_ROOT)
      {
         valOffs -= sizeof( S_BTREE_INFO);      // values start just before the INFO struct
      }
      nodeVal.data = (((BYTE *) node) + valOffs - varToc[ index].nVoffset);
      nodeKey.data = (((BYTE *) node) + keyOffs + varToc[ index].nKoffset);

      if (kLen)
      {
         *kLen = varToc[ index].nKlength;
      }
      if (kRec)
      {
         *kRec = nodeKey;
      }
      if (vLen)
      {
         *vLen = varToc[ index].nVlength;
      }
      if (vRec)
      {
         *vRec = nodeVal;
      }
      rc = APFSidType( nodeKey.dr->fsId);
   }
   RETURN(rc);
}                                               // end 'dfsApfsNodeIdx2Rec'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate Block-nr to LSN, generic interface, Area aware
/*****************************************************************************/
ULN64 dfsApfsCl2Lsn                             // RET   LSN for this block
(
   ULN64               block,                   // IN    block number
   ULN64               d2,                      // IN    dummy
   char               *p1,                      // IN    dummy
   void               *p2                       // IN    dummy
)
{
   ULN64               lsn = block;

   lsn = dfstAreaP2Disk( DFSTORE, (block * apfs->SectorsPerBlock));
   return (lsn);
}                                               // end 'dfsApfsCl2Lsn'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Translate LSN to Block-nr, generic interface, Area aware
/*****************************************************************************/
ULN64 dfsApfsLsn2Cl                             // RET   block for this LSN
(
   ULN64               lsn,                     // IN    LSN
   ULN64               d2,                      // IN    dummy
   char               *p1,                      // IN    dummy
   void               *p2                       // IN    dummy
)
{
   ULN64               block = 0;

   block = dfstAreaD2Part( DFSTORE, lsn) / apfs->SectorsPerBlock;
   return (block);
}                                               // end 'dfsApfsLsn2Cl'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Get single character sector-type from an APFS Block-type enumeration value
/*****************************************************************************/
BYTE dfsApfsBlock2SecType                       // RET   Sector type character
(
   ULONG               bt                       // IN    APFS block type
)
{
   BYTE                rc;                      // function return

   switch (bt & HDR_TYPE_VALUE_MASK)
   {
      case BT_C_SUPERBLOCK         : rc = ST_C_SUP;    break;
      case BT_BTREE_ROOT           : rc = ST_BTREE;    break;
      case BT_BTREE_NODE           : rc = ST_ANODE;    break;
      case BT_SPACEMANAGER         : rc = ST_SPMGR;    break;
      case BT_SPACEMAN_CAB         : rc = ST_SPCAB;    break;
      case BT_SPACEMAN_CIB         : rc = ST_SPCIB;    break;
      case BT_SPACEMANBMAP         : rc = ST_SPBMP;    break;
      case BT_OBJECT_MAP           : rc = ST_OBMAP;    break;
      case BT_CHECKP_MAP           : rc = ST_CHKPT;    break;
      case BT_V_SUPERBLOCK         : rc = ST_V_SUP;    break;
      case BT_REAPER               : rc = ST_REAPR;    break;
      case BT_REAP_LIST            : rc = ST_REAPL;    break;
      case BT_EFI_JUMPSTART        : rc = ST_EFIBT;    break;
      case BT_FUSION_WRBACK_CACHE  : rc = ST_FUWBC;    break;
      case BT_FUSION_WRBACK_LIST   : rc = ST_FUWBL;    break;
      case BT_ENCR_ROLLING_STATE   : rc = ST_ENCRS;    break;
      case BT_GENERIC_BITMAP       : rc = ST_GBMAP;    break;
      case BT_GENERIC_BITMAP_BLK   : rc = ST_GBMBL;    break;
      default:                       rc = ST_UDATA;    break;
   }
   return (rc);
}                                               // end 'dfsApfsBlock2SecType'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Convert APFS TIME (64 bit) to DFS standard date/time string (optional nsec)
// Returns 19 character standard date-time, plus (decimals +1) fraction if any
/*****************************************************************************/
char *dfsApfsTime2str                           // RET   string value
(
   ULN64               at,                      // IN    APFS time value
   int                 decimals,                // IN    seconds #decimals 0..9
   char               *dtime                    // INOUT ptr to string buffer
)
{
   time_t              tm;                      // time in compiler format
   struct tm          *gm;
   TXTS                secdec;

   ENTER();

   tm = dfsApfsTime2Tm(at);                     // seconds since 1-1-1970
   TRACES(("at:%16.16llx  tm:%8.8x\n", at, tm));

   if ((at != 0) && ((gm = gmtime( &tm)) != NULL))
   {
      strftime( dtime, TXMAXTM, "%Y-%m-%d %H:%M:%S", gmtime( &tm));
   }
   else                                         // invalid, before 1-1-1970
   {
      sprintf(  dtime, "-%16.16llx--", at);
   }
   if (decimals)
   {
      sprintf( secdec, ".%9.9llu", at % 1000000000);
      secdec[  decimals + 1] = 0;
      strcat(  dtime, secdec);
   }
   TRACES(("Formatted date/time: %s\n", dtime));

   RETURN( dtime);
}                                               // end 'dfsApfsTime2str'
/*---------------------------------------------------------------------------*/


#define MAX32BIT64 0xffffffffULL
#define MODVALUE64 0xffffffffULL
/*****************************************************************************/
// Calculate 64-bit Fletcher checksum for given data area (like Linux APFS)
// Modified (by APPLE) from regular Fletcher algorithm, modulo AFTER loop
/*****************************************************************************/
ULN64 dfsApfsFletcher64
(
   ULONG              *data,                    // IN    data area (APFS block)
   ULONG               count,                   // IN    data area size, ULONGs
   ULN64               init                     // IN    initial checksum value
)
{
   ULN64               rc = 0;                  // function return
   ULONG               i;
   ULN64               sum1 = init & MAX32BIT64;
   ULN64               sum2 = init >> 32;

   TRLEVX(500,("data: %p  count: %5lu  init: 0x%16.16llx\n", data, count, init));

   TRLEVX(500,("sum1: 0x%16.16llx  sum2: 0x%16.16llx  init: 0x%16.16llx\n", sum1, sum2, init));

   for (i = 0; i < count; i++)
   {
      sum1 = sum1 + data[ i];
      sum2 = sum2 + sum1;
   }
   sum1 %= MODVALUE64;
   sum2 %= MODVALUE64;

   TRLEVX(500,("Fletcher64 sum2: 0x%16.16llx  sum1: 0x%16.16llx\n", sum2, sum1));

   rc = (sum2 << 32) | sum1;

   return (rc);
}                                               // end 'dfsApfsFletcher64'
/*---------------------------------------------------------------------------*/
#undef  MAX32BIT64
#undef  MODVALUE64

/*****************************************************************************/
// Verify if given data block has a valid Fletcher64 checksum
/*****************************************************************************/
BOOL dfsApfsValidBlock
(
   BYTE               *block,                   // IN    data area (APFS block)
   ULONG               size                     // IN    data area size, bytes
)
{
   BOOL                rc = FALSE;              // function return
   ULN64               cs;
   ULONG              *data = (ULONG *) block;

   ENTER();

   cs = *((ULN64 *) block);
   TRACES(("cs blk: 0x%16.16llx\n", cs));

   if ((cs != 0) && (cs != 0xffffffffffffffffULL))
   {
      size /= 4;                                // number of ULONG's in data

      cs = dfsApfsFletcher64( data + 2, size - 2, 0 ); // data beyond checksum
      cs = dfsApfsFletcher64( data    ,        2, cs); // add checksum to get 0

      rc = (cs == 0);
   }
   BRETURN (rc);
}                                               // end 'dfsApfsValidBlock'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// APFS filesystem, display DIR-style header/footer
/*****************************************************************************/
void dfsApfsDirHeader
(
   char               *lead,                    // IN    lead text-string     (20)
   ULONG               items                    // IN    DIR-lines shown (0 = TOP)
)
{
   TXLN                line;

   sprintf( line, " ==================== =================== ==== =================== ==================== ====================\n");
   if (items > 0)
   {
      TxPrint( line);
   }
   TxPrint(                         " %s Creation/LastAccess Attr Modified / Filename -> SymLink  Filesize   Resource-fork size\n", lead);
   if (items == 0)
   {
      TxPrint( line);
   }
}                                               // end 'dfsApfsDirHeader'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Show contents of a directory/folder using 2 lines per entry
/*****************************************************************************/
ULONG dfsShowFolderContents
(
   ULN64               folderIno,               // IN    inode number for folder
   ULN64               startLeaf                // IN    first leaf node to search
)
{
   ULONG               rc = NO_ERROR;           // function return
   ULN64               leafId;
   ULN64               leafLsn;
   BYTE               *block;                   // alias for node pointer
   S_BT_NODE          *node = NULL;             // leaf node for file
   S_NODE_KEY          nodeKey;                 // generic record key   (pointer)
   S_BT_ENTRY         *varToc;
   ULONG               keyOffs;
   ULONG               entry;
   ULN64               parentIno = 0;           // parent ID in DIR-RECORD being checked
   ULONG               listed = 0;              // number of records added to snlist
   int                 ac;                      // alternate color fg+bg

   ENTER();

   if (!TxaOptUnSet('l'))                       // create a new list for DIR
   {
      dfsInitList(0, "-f -P", "-d");            // optimal for menu file-recovery
   }
   dfsApfsDirHeader( "Nr      Dir/File INO", listed);

   for ( leafId = startLeaf;
        (leafId != 0) && (rc == NO_ERROR) && (parentIno <= folderIno);
         leafId = dfsApfsNextLeafLink( leafId, &apfs->vCatTree))
   {
      leafLsn = dfsApfsBlock2Lsn( dfsApfsVirt2Phys( leafId, APFS_ANY_XID, apfs->vOmap));
      if ((dfsApfsReadChkCatNode( leafLsn, &node) == NO_ERROR) && (node->btNode.btFlags & BNF_LEAF))
      {
         block   = (BYTE *) node;
         varToc  = (S_BT_ENTRY *) (node->btData + node->btNode.btToc.Offset);
         keyOffs = sizeof( S_B_HEADER) + sizeof( S_BTREE_NODE) + node->btNode.btToc.Offset + node->btNode.btToc.Length;

         //- iterate over index-records; show DIR-Records with matching parent folder INO value
         for (entry = 0; (entry < node->btNode.btCount) && !TxAbort(); entry++)
         {
            nodeKey.data = (block + keyOffs + varToc[ entry].nKoffset);

            TRACES(("Leaf:0x%llx  entry:%3u  pri/sec key: %llx\n", leafId, entry, nodeKey.dh->fsId));
            if (APFSidType( nodeKey.dh->fsId) == APFST_DIR_RECORD)
            {
               parentIno = APFSidPrim( nodeKey.dh->fsId);

               if (parentIno == folderIno)      // matching directory inode number
               {
                  ac = (listed % 2) ? TXaBWnC : TXaNWnZ; // alternate color per listed entry
                  TxPrint( "%s.%07.7u r%3x        ",    ansi[ac], listed, entry);
                  dfsApfsShowCatRecord( leafLsn, entry, Ccbg(ac), NULL);

                  if (!TxaOptUnSet('l'))        // create a new list for DIR
                  {
                     rc = dfsAddSI2List( leafLsn, entry);
                  }
                  listed++;
               }
               else if (parentIno > folderIno)  // we are past the records for this folder ...
               {
                  break;                        // terminate inner entry loop
               }
            }
         }
         TxFreeMem( node);
      }
      else
      {
         TxPrint( "\n\nLsn 0x%llx is NOT a valid FS-tree LEAF node\n", leafLsn);
         rc = DFS_ST_MISMATCH;
      }
   }
   if (rc == NO_ERROR)
   {
      if (listed > 12)
      {
         dfsApfsDirHeader( "Nr      Dir/File INO", listed);
      }
      else if (listed == 0)
      {
         TxPrint( "\nNo FS-tree records matching parent folder-ID %llx\n", folderIno);
      }
   }
   RETURN (rc);
}                                               // end 'dfsShowFolderContents'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Display APFS Dir/File information on two lines, in 'dfsee directory' format
/*****************************************************************************/
ULN64 dfsApfsShowCatRecord                      // RET   DIR Inode number or 0
(
   ULN64               leafLsn,                 // IN    Leaf-node sectornumber
   USHORT              leafIndex,               // IN    Leaf-node record index
   BYTE                bg,                      // IN    current background
   char               *rKind                    // OUT   (f)ile, (D)ir or (S)ymlink
)                                               //       or NULL when  not needed
{
   ULN64               ino = 0;                 // Inode number for this record
   S_BT_NODE          *node = NULL;             // leaf node for file
   char                recKind = 0;             // record: (f)ile, (D)ir or (S)ymlink
   USHORT              fileMode;
   ULN64               fsize = 0;               // filesize in bytes
   TXTM                tbuf;
   TXLN                text;
   TXLN                fname;

   ENTER();
   TRARGS(("LSN: 0x%llX, index: %X\n", leafLsn, leafIndex));

   if ((dfsApfsReadChkCatNode( leafLsn, &node) == NO_ERROR) && (node->btNode.btFlags & BNF_LEAF))
   {
      S_XATTR_KEY      xatKey;                  // key to search for an XATTR record
      S_INODE_KEY      inoKey;                  // key to search for the INODE
      S_NODE_KEY       nodeKey;                 // generic record key   (pointer)
      S_NODE_VAL       nodeVal;                 // generic record value (pointer)
      USHORT           vLength;                 // length of record value
      ULN64            inoRefCount;             // nr of files/links     in Inode
      ULN64            inoModTime;              // last-modify timestamp in Inode
      ULN64            leafId;                  // virtual leaf ID for found leaf
      USHORT           index;                   // record index for INODE in found leaf
      int              recType;
      int              i;

      recType = dfsApfsNodeIdx2Rec( node, leafIndex, NULL, &nodeKey, &vLength, &nodeVal);
      switch (recType)
      {
         case APFST_DIR_RECORD:
            strcpy( fname, (char *) nodeKey.dh->drName); // remember original DIR-RECORD fname
         case APFST_DIR_STATS:
         case APFST_FILE_EXTENT:
         case APFST_DSTREAM_ID:
         case APFST_XATTR:
         case APFST_SIBLING_LINK:
            if (recType == APFST_DIR_RECORD)    // INO number is in the record VALUE
            {
               ino = nodeVal.dr->fileId;
            }
            else                                // INO number is in the record KEY
            {                                   // Inode record must be searched/read!
               ino = APFSidPrim( nodeKey.ds->fsId);
            }
            TxFreeMem( node);                   // free existing node record
            nodeKey.in = &inoKey;               // create the key for Btree search
            inoKey.fsId = APFSmkFsId( ino, APFST_INODE); // hybrid primary/secondary key value
            leafId = dfsApfsSearchCatalog( nodeKey, &index, &node); // overwrites node contents (name)!
            if ((leafId == 0) || (leafId == L64_NULL) ||
                (dfsApfsNodeIdx2Rec( node, index, NULL, &nodeKey, &vLength, &nodeVal) != APFST_INODE))
            {
               TRACES(("Failed to locate INODE record!\n"));
               if (recType == APFST_DIR_RECORD)
               {
                  TxPrint(" No INO record found for: %s%s", ansi[Ccol((CcR | CcI), bg)], fname);
                  TxPrint( "%s\n", CGE);        // clear (in bg color) to end of line
                  TxPrint( "%s inode %14llx", ansi[Ccol((CcW | CcI), bg)], ino);
                  TxPrint( "%s%s\n", CGE, CNN); // clear (in bg color) to end of line
               }
               else
               {
                  TxPrint(" Failed to locate INODE record!\n");
               }
               ino = 0;
            }
            break;

         case APFST_INODE:
            ino = APFSidPrim( nodeKey.in->fsId); // make INO value available for search ATTR/ALLOC
            break;

         default:
            break;
      }
      if (ino != 0)                           // we have a valid INODE record now, to show all info
      {
         recKind   = ((nodeVal.in->inMode & S_IFLNK) == S_IFLNK) ? 's' : S_ISDIR( nodeVal.in->inMode)  ? 'D' : 'f';
         inoRefCount = nodeVal.in->inRefCount;  // remember refcount (record/nodeVal is reused!)
         inoModTime  = nodeVal.in->inModTime;
         strcpy( fname, "");                    // make sure to start empty

         //- first line
         strcpy( text, " ");
         strcat( text, dfsApfsTime2str( nodeVal.in->inCreTime, 0, tbuf));

         if (recKind == 's')                    // File attributes, show symlink
         {
            strcat( text, " Slnk");
         }
         else                                   // File/Dir attributes, to be refined
         {
            if (recKind == 'D')                 // It is a directory, show that!
            {
               strcat( text, "    D");
            }
            else                                // File/Dir attributes, to be refined
            {
               strcat( text, "     ");          // to be refined (FAT-like) attributes
            }
         }

         if ((nodeVal.in->inModTime != 0) && (nodeVal.in->inModTime != nodeVal.in->inCreTime))
         {
            strcat( text, " ");
            strcat( text, dfsApfsTime2str( nodeVal.in->inModTime, 0, tbuf));
         }
         else
         {
            strcat( text, "                    ");
         }

         if (vLength > sizeof( S_INODE_VAL))    // there are Xfields
         {
            S_XF_BLOB *xfb  = (S_XF_BLOB *) (nodeVal.in->inXfields);
            BYTE      *data = ((BYTE *) xfb->xfData) + (xfb->xfExtents * sizeof( S_XF_FIELD));

            TRACES(("There are %hu Xfields\n", xfb->xfExtents));

            for (i = 0; i < xfb->xfExtents; i++)
            {
               switch (xfb->xfData[ i].xfType)
               {
                  case XFT_INO_DATASTREAM:
                     fsize = ((S_DSTREAM *) data)->dsByteSize;
                     TRACES(("Xfield %u DATASTREAM fsize: 0x%llx\n", fsize));
                     break;

                  case XFT_INO_FILENAME:
                     strcpy( fname, (char *) data); // get the filename from INO Xfield
                     TRACES(("Xfield %u FILENAME '%s'\n", fname));
                     break;

                  default:
                     break;
               }
               data += ((xfb->xfData[ i].xfSize + 7) & 0xFFF8); //  next Xfield (8 byte size align)
            }
         }
         if (recKind == 'f')
         {
            dfstrUllDot20( text, " ", fsize, "");

            #if defined (HFS)
               if ((forkSize = TxBE64( file->crRsrcFork.logicalSize)) != 0)
               {
                  dfstrUllDot20( text, " ", forkSize, "");
               }
            #endif
         }
         sprintf( tbuf, "%s\n", CGE);
         strcat(  text, tbuf);

         //- second line
         sprintf( tbuf, " inode %14llx ", ino);
         strcat(  text, tbuf);

         if ((nodeVal.in->inAccTime != 0) && (nodeVal.in->inAccTime != nodeVal.in->inCreTime))
         {
            strcat( text, dfsApfsTime2str( nodeVal.in->inAccTime, 0, tbuf));
         }
         else
         {
            strcat( text, "                   ");
         }

         if ((fileMode = nodeVal.in->inMode & 0777) != 0)
         {
            sprintf( tbuf, "  %3o ", fileMode);
            strcat(  text, tbuf);
         }
         else
         {
            strcat(  text, "      ");
         }

         strcat(  text, ansi[Ccol((CcY | CcI), bg)]); // Yellow
         TxPrint( "%s%s", text, fname);

         if (recKind == 's')
         {
            TxFreeMem( node);                   // free existing node record
            nodeKey.xa = &xatKey;               // create the key for Btree search
            xatKey.fsId = APFSmkFsId( ino, APFST_XATTR); // hybrid primary/secondary key value
            xatKey.xaNameLength = strlen( XA_SYMLINK_NAME) + 1; // includes the termination char
            strcpy( (char *) xatKey.xaName, XA_SYMLINK_NAME);
            leafId = dfsApfsSearchCatalog( nodeKey, &index, &node);
            if (((leafId != 0) && ((leafId != L64_NULL))) &&
               (dfsApfsNodeIdx2Rec( node, index, NULL, &nodeKey, &vLength, &nodeVal) == APFST_XATTR))
            {
               if (strlen( fname) < 25)         // align symlink arrows for shorter filenames
               {
                  sprintf( text, "%*.*s", (int)(25 - strlen( fname)), (int)(25 - strlen( fname)), "");
               }
               else
               {
                  strcpy(  text, "");
               }
               TxPrint( "%s%s%s%s%s%s", text, ansi[Ccol( CcW,  bg)], DFS_SH_LINK_STR,
                                              ansi[Ccol((CcC | CcI), bg)],
                                              ansi[Ccol( CcW,  bg)], (char *) nodeVal.xa->xaData);
            }
            else                                // symlink XATTR not found!
            {
               TRACES(("Failed to locate SYMLINK XATTR for ino: %llx\n", ino));
            }
         }
         TxPrint( "%s%s\n", CGE, CNN);          // clear (color) to end of line
         if (recKind != 'D')
         {
            ino = 0;                            // only return ino value for a DIR record
         }
         if (rKind != NULL)
         {
            *rKind = recKind;                   // return type of record, when requested
         }
         TxFreeMem( node);
      }
   }
   else
   {
      TxPrint( "\n\nLsn 0x%llx and index %x do NOT lead to a valid FS-tree record\n", leafLsn, leafIndex);
   }
   RETN64( ino);
}                                               // end 'dfsApfsShowCatRecord'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Return description for APFS blocktype, subtype, or 'unknown'
/*****************************************************************************/
char *dfsApfsBlockType
(
   USHORT              blocktype                // IN    type of APFS block
)
{
   switch (blocktype & HDR_TYPE_VALUE_MASK)
   {
      case BT_NO_TYPE             : return "None";
      case BT_C_SUPERBLOCK        : return "Container Superblock";
      case BT_BTREE_ROOT          : return "Btree-Root";
      case BT_BTREE_NODE          : return "Btree-Node";
      case BT_SPACEMANAGER        : return "Spacemanager";
      case BT_SPACEMAN_CAB        : return "Spm CAB level2 Index";
      case BT_SPACEMAN_CIB        : return "Spm CIB level1 Index";
      case BT_SPACEMANBMAP        : return "Spm Bitmap";
      case BT_SPACEMANFREE        : return "Spm Free Queue";
      case BT_EXTENT_TREE         : return "Extent list";
      case BT_OBJECT_MAP          : return "Object Map";
      case BT_CHECKP_MAP          : return "CheckPoint Map";
      case BT_V_SUPERBLOCK        : return "Volume Superblock";
      case BT_V_FS_TREE           : return "FS tree";
      case BT_V_BLOCKREFTREE      : return "Block-Extent-ref";
      case BT_V_SNAPMETATREE      : return "Snapshot meta tree";
      case BT_REAPER              : return "Reaper object";
      case BT_REAP_LIST           : return "Reaper list";
      case BT_OMAP_SNAPSHOT       : return "Map Snapshot";
      case BT_EFI_JUMPSTART       : return "EFI JumpStart, boot";
      case BT_FUSION_MIDDLE_TREE  : return "Fusion Cached";
      case BT_FUSION_WRBACK_CACHE : return "Fusion WrBack Cache";
      case BT_FUSION_WRBACK_LIST  : return "Fusion WrBack List";
      case BT_ENCR_ROLLING_STATE  : return "Encr Rolling State";
      case BT_GENERIC_BITMAP      : return "Gen Purpose Bitmap";
      case BT_GENERIC_BITMAP_TREE : return "Gp Bitmap Tree";
      case BT_GENERIC_BITMAP_BLK  : return "Gp Bitmap Block";
      case BT_UNIVERSAL_TEST_TYPE : return "Test object!";
      default:                      return "Unknown block type!";
   }
}                                               // end 'dfsApfsBlockType'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Return description for APFS FS record type, or 'unknown'
/*****************************************************************************/
char *dfsApfsFsRecType
(
   int                 recType                  // IN    FS record type
)
{
   switch (recType)
   {
      case APFST_FIRST            : return "None      ";
      case APFST_SNAP_METADATA    : return "SnapMeta  ";
      case APFST_PHYS_EXTENT      : return "PhysExtent";
      case APFST_INODE            : return "Inode     ";
      case APFST_XATTR            : return "Xattr     ";
      case APFST_SIBLING_LINK     : return "SiblingLnk";
      case APFST_DSTREAM_ID       : return "DstreamId ";
      case APFST_CRYPTO_STATE     : return "CryptState";
      case APFST_FILE_EXTENT      : return "FileExtent";
      case APFST_DIR_RECORD       : return "DirEntry  ";
      case APFST_DIR_STATS        : return "StatDirect";
      case APFST_SNAP_NAME        : return "SnapName  ";
      case APFST_SIBLING_MAP      : return "MapSibling";
      default:                      return "BadFsType!";
   }
}                                               // end 'dfsApfsFsRecType'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Return description for APFS XFIELD record type, or 'unknown'
/*****************************************************************************/
char *dfsApfsXfieldType
(
   int                 xfType,                  // IN    Xfield record type
   BOOL                dirRecord                // IN    for a DIR-Record
)
{
   if (dirRecord)
   {
      switch (xfType)
      {
         case XFT_FIRST           : return "None      ";
         case XFT_DREC_SIBLING_ID : return "Sibling-ID";
         default:                   return "BadXfType!";
      }
   }
   else
   {
      switch (xfType)
      {
         case XFT_FIRST           : return "None      ";
         case XFT_DREC_SIBLING_ID : return "SnapshotID";
         case XFT_INO_DELTATREEID : return "DeltaTrOID";
         case XFT_INO_DOCUMENT_ID : return "DocumentID";
         case XFT_INO_FILENAME    : return "FileName  ";
         case XFT_INO_PREV_FSIZE  : return "PrevFsize ";
         case XFT_INO_RESERVED_6  : return "Reserved-6";
         case XFT_INO_FINDERINFO  : return "FinderInfo";
         case XFT_INO_DATASTREAM  : return "DataStream";
         case XFT_INO_RESERVED_9  : return "Reserved-9";
         case XFT_INO_DIRSTAT_KEY : return "DirStatKey";
         case XFT_INO_FS_UUID     : return "FsMnt-UUID";
         case XFT_INO_RESERVED_C  : return "Reserved-C";
         case XFT_INO_SPARSEBYTES : return "SparsBytes";
         case XFT_INO_RDEVICE_ID  : return "Rdevice-ID";
         default:                   return "BadXfType!";
      }
   }
}                                               // end 'dfsApfsXfieldType'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Return single-char description for APFS DIR record type (DIR/FILE/LINK etc)
/*****************************************************************************/
char *dfsApfsDirRecType                         // RET   single character str
(
   USHORT              drFlg                    // IN    Dir-Rec value Flags
)
{
   switch (drFlg)
   {
      case DRF_FIFO    : return "P";
      case DRF_CHR     : return "C";
      case DRF_DIR     : return "D";
      case DRF_BLK     : return "B";
      case DRF_REG     : return "f";
      case DRF_LNK     : return "L";
      case DRF_SOCK    : return "S";
      case DRF_WHT     : return "W";
      default:           return "!";
   }
}                                               // end 'dfsApfsDirRecType'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Return description for APFS Device/Bitmap (Single or Fusion drive main/tier2)
/*****************************************************************************/
char *dfsApfsDeviceBmType
(
   int                 itemType                 // IN    type of APFS device/bitmap
)
{
   switch (itemType)
   {
      case SPD_MAIN  : return "MAIN-dsk";
      case SPD_TIER2 : return "TIER2dsk";
      default:         return "Unknown!";
   }
}                                               // end 'dfsApfsDeviceBmType'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Return description for APFS spacemanager Free-Queue type
/*****************************************************************************/
char *dfsApfsSpmSfqueType
(
   int                 sfqType                  // IN    type of APFS free queue
)
{
   switch (sfqType)
   {
      case SFQ_IPOOL : return "Int-Pool";  break;
      case SFQ_MAIN  : return "MAIN-dsk";  break;
      case SFQ_TIER2 : return "TIER2dsk";  break;
      default:         return "Unknown!";  break;
   }
}                                               // end 'dfsApfsSpmSfqueType'
/*---------------------------------------------------------------------------*/


/*****************************************************************************/
// Calculate combined filename-hash and length value (32-bit CRC32C based)
/*****************************************************************************/
ULONG dfsApfsNameHash                           // RET   combined CRC+length
(
   BYTE               *name,                    // IN    filename string, UTF-8
   USHORT              length                   // IN    filename length
)
{
   ULONG               rc = 0;                  // function return
   ULONG               utf32Length;
   ULONG               utf32Name[ TXMAXLN];     // Enough room for TXLN utf8 name
   ULONG               normLength;
   ULONG              *normUtf32Name = NULL;    // allocated normalized name
   BOOL                caseIns = ((apfs->vsBlk->vsInCompatible &  VFI_CASE_INSENS) != 0);

   ENTER();
   TRACES(("length: %hu  for '%s'\n", length, name));

   //- convert to UTF-32 and normalize/case-fold the supplied UTF-8 name
   utf32Length = dfsUtf8ToUtf32(        name,    TXMAXLN,     utf32Name);
   normLength  = dfsUtfNormCaseFoldStr( caseIns, utf32Length, utf32Name, &normUtf32Name);

   if (normLength != 0)                         // then calculate the CRC32C hash + length
   {
      rc = TxCalcCrc32C( 0x1EDC6F41, 0xFFFFFFFF, normLength * sizeof( ULONG), (BYTE *) normUtf32Name);

      rc = ((rc & 0x3FFFFF) << 10) | (length & 0x3FF);
   }
   TxFreeMem( normUtf32Name);

   RETURN (rc);
}                                               // end 'dfsApfsNameHash'
/*---------------------------------------------------------------------------*/

